MODULE bdy_twolayers
   !!========================================================================
   !!                       ***  MODULE  bdytra  ***
   !! Two-layers Boundary condition (e.g. Bosphorus channel)
   !! References:  Beckers & Stanev (1999)
   !!========================================================================
   !! History :  1.0  !  2018-06  (B. Lemieux, L. Vandenbulcke) original code
   !!========================================================================
   USE oce             ! ocean dynamics and tracers variables
   USE dom_oce         ! ocean space and time domain variables
   USE bdy_oce         ! ocean open boundary conditions
   USE bdylib          ! for orlanski library routines
  !USE bdydta, ONLY:bf ! LV 20220119: not required ?
   USE lbclnk          ! ocean lateral boundary conditions (or mpp link)
   USE restart         ! restart needed to read/write total_salt_initial
   USE iom             ! iom I/O
   USE in_out_manager  ! I/O manager
   use lib_fortran
   use sbc_oce, only:  rnf, emp
   USE bdy_twolayers_public

   IMPLICIT NONE
   PRIVATE

   PUBLIC bdy_tra_twolayers
   PUBLIC bdy_dyn2d_twolayers
   PUBLIC bdy_dyn3d_twolayers
   PUBLIC bdy_ssh_twolayers

   REAL(wp)              :: ztilde, Q_out
   REAL(wp)              :: Q_data, v_barotropic, v_offset

   !! * Substitutions
!#  include "do_loop_substitute.h90"
#  include "domzgr_substitute.h90"

   
CONTAINS

  SUBROUTINE bdy_twolayers_main ( idx , dta , kbb )
      TYPE(OBC_INDEX), INTENT(in) ::   idx  ! OBC indices
      TYPE(OBC_DATA),  INTENT(in) ::   dta  ! OBC external data
      INTEGER,         INTENT(in) ::   kbb
      !!
      INTEGER  ::   jb, igrd         ! dummy loop indices
      INTEGER  ::   ii, ij, jk       ! 2D addresses
      REAL(wp) ::   Q_desired, speedprofile(jpk)

      ! find out depth of Bosphorus channel at each time step ( ln_linssh may be .false. )
      igrd = 1
      DO jb = 1, idx%nblenrim(igrd)
         ii   = idx%nbi(jb,igrd)
         ij   = idx%nbj(jb,igrd)
         if (tmask(ii,ij,1).eq.1) then
           jk=1; TotDepth=0.0
           do while (vmask(ii,ij,jk).eq.1.and.jk.le.jpkm1)
              TotDepth = TotDepth + e3v(ii,ij,jk,kbb)
              jk = jk+1
           END DO
        end if
      END DO

      ! compute incoming velocity
      igrd = 3      ! v-velocity grid
      DO jb = 1, idx%nblenrim(igrd)
         ii   = idx%nbi(jb,igrd)
         ij   = idx%nbj(jb,igrd)

         ! load "typical" speed profile, compute resulting barotropic flux, at every time step as e3v may evolve
         Q_data=0.0
         if (vmask(ii,ij,1).eq.1) then
            do jk = 1,jpkm1
               speedprofile(jk) = dta%v3d(jb,jk)                                       ! [  m/s ]
               Q_data = Q_data + e3v(ii,ij,jk,kbb) * speedprofile(jk) * vmask(ii,ij,jk)  ! [ m2/s ]
            end do
            !Q_data=Q_data*e1v(ii,ij)                                                  ! [ m3/s ], but later we need just m2/s
            
            ! compute barotropic flux using an empirical law based on SSH
            ztilde=ssh(ii,ij+1,kbb)+0.175
            Q_desired  = -26000*((ztilde-0.45)/(0.175-0.45)) + 26000  ! positive outward
            v_barotropic= -1 * Q_desired / e1v(ii,ij) / TotDepth

            ! compute offset to apply to velocities, in order to obtain desired barotropic flux
            v_offset= ( -Q_desired / e1v(ii,ij) - Q_data ) / TotDepth
            !write(77,*) ssh(ii,ij:ij+1,kbb),v_barotropic
            
            ! compute the new baroclinic velocity (including the baroclinic component)
            DO jk = 1, jpkm1
               v_baroclinic(jk) = (speedprofile(jk)+v_offset) * vmask(ii,ij,jk)
            END DO
         end if
      END DO
      ! when we iterate the 1st and 3d points, we are in the land and don't modify v_barotropic and v_baroclinic                                                                                                        
  END SUBROUTINE bdy_twolayers_main
  

  SUBROUTINE bdy_tra_twolayers ( idx, dta, pts, kbb, kaa,  kt )
      !!----------------------------------------------------------------------
      !!                 ***  SUBROUTINE bdy_tra_twolayers  ***
      !!
      !! ** Purpose : Apply a specified value for tracers at open boundaries.
      !!   This routine is called only once (instead of once for T and once for S)
      !!
      !!----------------------------------------------------------------------
      INTEGER,         INTENT(in) ::   kt, kbb, kaa
      TYPE(OBC_INDEX), INTENT(in) ::   idx  ! OBC indices
      TYPE(OBC_DATA),  INTENT(in) ::   dta  ! OBC external data
      REAL(wp), DIMENSION(jpi,jpj,jpk,jpts,jpt), INTENT(inout) ::   pts ! tracer
      INTEGER                     ::   igrd, jb,ii,ij,jk
      REAL(wp)  ::   z2dt, zs2d(jpi,jpj)
      REAL(wp)  ::   QS_out, Q_in, salinity_incoming
      REAL(wp)  ::   total_salt_now, tempo, maxsal
      REAL(wp)  ::   mmean_steps, mmean_steps_temp
      
      IF( l_1st_euler .AND. kt == nn_it000 ) THEN   ;   z2dt = rdt
      ELSE                                        ;   z2dt = 2. * rdt
      ENDIF
      mmean_steps = 90.0*86400.0/z2dt  ! moving average over 90 days (Stanev JMS 1997)
      mmean_steps_temp=min(dble(kt),mmean_steps)
     
      ! if bdy_tra is called first, call the main twolayer computations
      if (.not.ln_dynspg_ts) call bdy_twolayers_main(idx,dta,kbb)

      ! compute "moving average" of total salt in the whole domain
      zs2d(:,:) = 0._wp
      DO jk=1,jpkm1
        zs2d(:,:) = zs2d(:,:) + (pts(:,:,jk,jp_sal,kbb)*tmask(:,:,jk)*e3t(:,:,jk,kbb))
      ENDDO
      zs2d=zs2d*e1e2t
      total_salt_now = glob_sum('bdy_tra_twolayers',zs2d)
      total_salt_mmean= ( (mmean_steps_temp-1)*total_salt_mmean + total_salt_now) / mmean_steps_temp

      ! Handling salt content when reinitializing NEMO
      IF ( kt == nn_it000 ) THEN
        if (lwp) write(numout,*) "BOSPORUS BDY: WATER FLUX: LINEAR PARAMETERIZATION FUNCTION OF SSH"
        if (lwp) write(numout,*) "BOSPORUS BDY: SALT FLUX: TIME-LAGGED RELAXATION TO CONSERVE TOTAL S IN BASIN"
        ! If there is a restart with "total_salt_initial" and total_fw_flux_mmean, then we read it
        IF( ln_rstart .AND. iom_varid( numror, 'total_salt_initial', ldstop = .FALSE. ) > 0 ) THEN
            CALL iom_get( numror, 'total_salt_initial', total_salt_initial )
            CALL iom_get( numror, 'total_salt_mmean', total_salt_mmean )
            CALL iom_get( numror, 'total_fw_flux_mmean', total_fw_flux_mmean )
            ! If not we set the initial salt to the above calculated total_salt_now
        ELSE
            total_salt_initial=total_salt_now
        ENDIF
      ENDIF
     
      ! compute the salinity of the incoming water
      igrd = 1                       ! Everything is at T-points here
      QS_out=0.0 ; Q_in=0.0 ; Q_out=0.0
      DO jb = 1, idx%nblenrim(igrd)
         ii = idx%nbi(jb,igrd)
         ij = idx%nbj(jb,igrd)
         if (tmask(ii,ij,1).eq.1) then
            ! outgoing salt flux, incoming water flux ( v_baroclinic was computed before, during bdy_twolayers_main )
            DO jk = 1, jpkm1
                tempo=v_baroclinic(jk)*e1v(ii,ij)*e3v(ii,ij,jk,kbb)*vmask(ii,ij,jk)
                if (v_baroclinic(jk)<0) then     ! qs_out positive outward
                   QS_out=QS_out - pts(ii,ij+1,jk,jp_sal,kbb)*tempo
                   Q_out=Q_out + tempo
                else
                   Q_in=Q_in + tempo
                end if
            END DO
            if (Q_in>0) then
              salinity_incoming = ( (total_salt_initial-total_salt_mmean)/z2dt + QS_out ) / Q_in  !!! up to Nemo 4.0.6 I used total_salt_now instead of total_salt_mmean
            else
              salinity_incoming = 38.0_wp
            end if

            maxsal=40.0_wp ; ! if (total_salt_mmean .lt. (0.9*total_salt_initial)) maxsal=48.0_wp
            DO jk = 1, jpkm1
               ! temperature: we use zero gradient for temperature tsa(ii,ij,jk,jp_tem)=tsa(ii,ij+1,jk,jp_tem)
               ! when using this "zero gradient", we need to modify bdytra to activate lbc_lnk exchanges
               pts(ii,ij,jk,jp_tem,kaa) = pts(ii,ij+1,jk,jp_tem,kaa)
                    ! previously we used 13.0 , following Falina et al 2017, Chiggiato et al 2011, Ilicak et al 2021
               ! salinity
               if (v_baroclinic(jk)>0) then
                  pts(ii,ij,jk,jp_sal,kaa) = max(min(salinity_incoming,maxsal),18.0_wp)
               else
                  pts(ii,ij,jk,jp_sal,kaa) = pts(ii,ij+1,jk,jp_sal,kaa)
               end if
                  ! previously we used 36.3 ! same references , using 36.3 as 37 (max value) is attenuated due to brackish water in the upper layer
            end do
         else ! in the land
            pts(ii,ij,:,:,kaa)=0.0_wp
         end if
      END DO

      !save some stuff with Xios
      if (iom_use("fw_flux")) then
         tempo=total_fw_flux_mmean * 86400*365/1.0e9   ;                                                              CALL iom_put( 'fw_flux', tempo )
      end if
      if (iom_use("Qout")) then
         tempo=Q_out * 86400*365/1.0e9                 ; if( lk_mpp ) CALL mpp_min( 'bdy_tra_twolayers', tempo )   ;  CALL iom_put( 'Qout' ,   tempo )
      end if
      if (iom_use("Qin")) then
         tempo= Q_in * 86400*365/1.0e9                 ; if( lk_mpp ) CALL mpp_max( 'bdy_tra_twolayers', tempo )   ;  CALL iom_put( 'Qin'  ,   tempo )
      end if
      if (iom_use("Stot")) then
         tempo= total_salt_mmean                       ;                                                              CALL iom_put( 'Stot' ,   tempo )
      end if
      if (iom_use("Sin")) then
         tempo= salinity_incoming                      ; if( lk_mpp ) CALL mpp_max( 'bdy_tra_twolayers', tempo )   ;  CALL iom_put( 'Sin'  ,   tempo )
      end if

   END SUBROUTINE bdy_tra_twolayers





   SUBROUTINE bdy_dyn2d_twolayers( kbb, idx, dta, ib_bdy, pva2d, pssh)
      !!----------------------------------------------------------------------
      !!                 ***  SUBROUTINE bdy_dyn2d_twolayer ***
      !!----------------------------------------------------------------------
      INTEGER,                      INTENT(in)    ::   kbb
      TYPE(OBC_INDEX),              INTENT(in)    ::   idx  ! OBC indices
      TYPE(OBC_DATA),               INTENT(in)    ::   dta  ! OBC external data
      INTEGER,                      INTENT(in)    ::   ib_bdy  ! number of current open boundary set
      REAL(wp), DIMENSION(:,:),     INTENT(inout) ::   pva2d
      REAL(wp), DIMENSION(:,:),     INTENT(in   ) ::   pssh
      INTEGER  ::   igrd,  ii, ij, jb
      !!----------------------------------------------------------------------

      ! if bdy_dyn2d is called first, call the main twolayer computations
      if (ln_dynspg_ts) call bdy_twolayers_main(idx,dta,kbb)

      igrd = 3      ! v-velocity grid
      DO jb = 1, idx%nblenrim(igrd)

         ii   = idx%nbi(jb,igrd)
         ij   = idx%nbj(jb,igrd)

         pva2d(ii,ij) = v_barotropic * vmask(ii,ij,1)

      end do
      !CALL lbc_bdy_lnk( 'bdy_dyn2d_twolayers', pva2d, 'V', -1., ib_bdy )  !  Nemo 4.0, this has moved to bdy_dyn2d

   END SUBROUTINE bdy_dyn2d_twolayers


   

   SUBROUTINE bdy_dyn3d_twolayers( pvv, kaa, idx,  ib_bdy )
      !!----------------------------------------------------------------------
      !!                  ***  SUBROUTINE bdy_dyn3d_towlayers  ***
      !!----------------------------------------------------------------------
      REAL(wp), DIMENSION(jpi,jpj,jpk,jpt), INTENT( inout ) ::   pvv     ! Ocean V-velocity (to be updated at open boundaries)
      INTEGER                             , INTENT( in    ) ::   Kaa     ! Time level index
      TYPE(OBC_INDEX)                     , INTENT( in    ) ::   idx     ! OBC indices
      INTEGER                             , INTENT( in    ) ::   ib_bdy  ! BDY set index

      INTEGER  ::   jb, jk            ! dummy loop indices
      INTEGER  ::   ii, ij, igrd      ! local integers
      !!----------------------------------------------------------------------
      !
      igrd = 3

      DO jb = 1, idx%nblenrim(igrd)

         ii   = idx%nbi(jb,igrd)
         ij   = idx%nbj(jb,igrd)

         if (vmask(ii,ij,1).eq.1) then
            DO jk = 1, jpkm1
               pvv(ii,ij,jk,kaa) = (v_baroclinic(jk)-v_barotropic) * vmask(ii,ij,jk)
            END DO
         end if

      END DO
      
      !CALL lbc_bdy_lnk( 'bdy_dyn3d_twolayers' , ua, 'U', -1., ib_bdy )   ! Boundary points should be updated ; moved to bdy_dyn3d in Nemo 4.0.6
      !CALL lbc_bdy_lnk( 'bdy_dyn3d_twolayers' , va, 'V', -1., ib_bdy )

   END SUBROUTINE bdy_dyn3d_twolayers


   SUBROUTINE bdy_ssh_twolayers (zssh,idx)
      !!----------------------------------------------------------------------
      !!                  ***  SUBROUTINE bdy_ssh_towlayers  ***
      !!   the ssh outside the interior domain is simply set to zero         
      !!----------------------------------------------------------------------
      REAL(wp), DIMENSION(jpi,jpj,1), INTENT(inout) ::   zssh
      TYPE(OBC_INDEX), INTENT(in) ::   idx     ! OBC indices
      integer :: igrd,jb,ii,ij

      igrd = 1      ! T grid used for SSH
      DO jb = 1, idx%nblenrim(igrd)

         ii   = idx%nbi(jb,igrd)
         ij   = idx%nbj(jb,igrd)
      
         zssh(ii,ij,1)=0.0_wp
      end do

   END SUBROUTINE bdy_ssh_twolayers

   
END MODULE bdy_twolayers

