MODULE trcwbstress
  !!======================================================================
  !!                         ***  MODULE trcwbstress  ***
  !! TOP :   initialisation of the wave bottom stress effect on tracers
  !!======================================================================
  !! History :      !  2017  (T. Lovato) Create benthic infrastructure
  !!                   2024 (L.Vandenbulcke, M.Choblet) added wavestress at bottom
  !!----------------------------------------------------------------------
  USE par_trc         ! TOP parameters
  USE oce_trc
  USE trc
  USE par_ben
  USE trcnam_ben      ! Benthic namelist
  USE iom
  USE fldread

  IMPLICIT NONE
  PRIVATE

  PUBLIC   trc_ini_wbstress   ! called by trcini_my_trc.F90 module
  PUBLIC   trc_ini_curstress   ! called by trcini_my_trc.F90 module 
  PUBLIC   trc_wbstress       ! called by trcsms_my_trc.F90 module
  !
  REAL(wp),  PUBLIC, ALLOCATABLE, DIMENSION(:,:)  :: wavestress
  REAL(wp),  PUBLIC, ALLOCATABLE, DIMENSION(:,:)  :: wave_direction
  TYPE(FLD), PUBLIC, ALLOCATABLE, DIMENSION(:)    :: sf_wbstress   ! field structure of input (file informations, fields read)
  INTEGER,   PUBLIC                               :: waves_botstr  ! Select type of stress
  INTEGER,   PUBLIC                               :: cur_botstr  ! Select type of stress

  !!----------------------------------------------------------------------
  !! NEMO/TOP 4.0 , NEMO Consortium (2017)
  !! $Id$
  !! Software governed by the CeCILL licence     (NEMOGCM/NEMO_CeCILL.txt)
  !!----------------------------------------------------------------------

#include "do_loop_substitute.h90"
#include "domzgr_substitute.h90"

CONTAINS

  SUBROUTINE trc_ini_curstress
    !!----------------------------------------------------------------------
    !!                     ***  trc_ini_curstress  ***
    !!
    !! ** Purpose :   Read settings for bottom current stress
    !!                Option 0: Computed with logarithmic, grain size dependent formula
    !!                Option 1: Computed as in NEMO (9.4.2 in Nemo manual)
    !!
    !!                Added by Mathurin Choblet in 2023/2024
    !!----------------------------------------------------------------------
    use iom
    !
    !
    NAMELIST/namben_curstress/ cur_botstr
    !!---------------------------------------------------------
    !
    ! set default to (old) bamhbi formulation
    cur_botstr = 0
    !
    !
    READ  ( numnat_cfg, namben_curstress)
    !
    IF(lwp) WRITE(numout,*)
    IF(lwp) WRITE(numout,*) ' trc_ini_curstress : Read namelist namben_curstress '
    IF(lwp) WRITE(numout,*) '    Type of current bottom stress :   '
    !
    SELECT CASE (cur_botstr)
    CASE (0)
       IF(lwp) WRITE(numout,*) '    -> Compute bottom stress using bamhbi formulation (!= the bottom friction in Nemo) (cur_botstr=0)'
    CASE (1)
       IF(lwp) WRITE(numout,*) '    -> Compute bottom current stress as in Nemo (cur_botstr=1)'
    END SELECT
    !
    IF(lwp) WRITE(numout,*) ' ~~~~~~~~~~~~~~'


  END SUBROUTINE trc_ini_curstress

  SUBROUTINE trc_ini_wbstress
    !!----------------------------------------------------------------------
    !!                     ***  trc_ini_wbstress  ***
    !!
    !! ** Purpose :   Read settings for wave bottom stress effect on tracers
    !!
    !!----------------------------------------------------------------------
    use iom
    USE fldread       !  read input fields
    !
    ! arrays for data I/O
    INTEGER                                         :: jp_wbs = 4    ! fields used for wave-induced bottom stress
    TYPE(FLD_N),         ALLOCATABLE, DIMENSION(:)  :: sn_wbstress   ! namelist structure of input (file informations, fields read)
    TYPE(FLD_N),         ALLOCATABLE, DIMENSION(:)  :: slf_i         ! structure of input for fldread (file informations, fields read)
    !
    INTEGER :: jn, jl, ierr1, ierr2, ierr3, ierr4
    CHARACTER(len=100) :: cn_dir
    !
    NAMELIST/namben_wbstress/ waves_botstr, sn_wbstress, cn_dir
    !!---------------------------------------------------------
    ierr1 = 0  ;  ierr2 = 0
    !
    ALLOCATE( wavestress(jpi,jpj), STAT=ierr1 )
    wavestress=0.0_wp
    ALLOCATE( wave_direction(jpi,jpj), STAT=ierr2 )
    wave_direction=0.0_wp
    !
    ALLOCATE( sn_wbstress(jp_wbs), STAT=ierr3 )
    IF( ierr1 + ierr2 + ierr3 > 0 ) THEN
       CALL ctl_stop( 'trc_dta_ben: unable to allocate namelist arrays' )   ;   RETURN
    ENDIF
    ! dummy values of namelist
    cn_dir  = './'
    ! set default no stress
    waves_botstr = 0
    !
    !                  ! filename ! freq !  title   ! interp !  clim  ! 1m/1y ! wgt ! rot ! SoL !
    sn_wbstress = FLD_N( 'NONAME' ,  0   , 'NOTITLE', .false., .false.,  ''   ,  '' ,  '' , ''  )
    !
    READ  ( numnat_cfg, namben_wbstress)
    !
    IF(lwp) WRITE(numout,*)
    IF(lwp) WRITE(numout,*) ' trc_ini_wbstress : Read namelist namben_wbstress '
    IF(lwp) WRITE(numout,*) '    Type of wave bottom stress :   '
    !
    wavestress = 0._wp ! make sure wavestress is initialized

    SELECT CASE (waves_botstr)
    CASE (0)
       IF(lwp) WRITE(numout,*) '    -> No effect of bottom stress due to waves (waves_botstr=0)'
       !wavestress = 0._wp
    CASE (1)
       IF(lwp) WRITE(numout,*) '    -> Read bottom stress due to waves (waves_botstr=1)'
    CASE (2)
       IF(lwp) WRITE(numout,*) '    -> Compute bottom stress due to waves (waves_botstr=2)'
    !!! M.C. 07.02.2024 Add wave direction
    CASE (3)
       IF(lwp) WRITE(numout,*) '    -> Compute bottom stress due to waves and take into account wave and current direction(waves_botstr=3)'
    END SELECT
    !
    IF(lwp) WRITE(numout,*) ' ~~~~~~~~~~~~~~'


    ! Fill data structure
    IF ( waves_botstr .gt. 0) THEN
       ALLOCATE( sf_wbstress(waves_botstr), STAT=ierr1 )
       IF( ierr1 > 0 ) THEN
          CALL ctl_stop( 'trc_ini_wbstress: unable to allocate sf_wbstress structure' )   ;   RETURN
       ENDIF

       SELECT CASE (waves_botstr)
       CASE (1)
          jn = 1
          ALLOCATE( sf_wbstress(jn)%fnow(jpi,jpj,1)   , STAT=ierr2 )
          IF( sn_wbstress(jn)%ln_tint )  ALLOCATE( sf_wbstress(jn)%fdta(jpi,jpj,1,2) , STAT=ierr4 )
          IF( ierr2 + ierr4 > 0 ) THEN
             CALL ctl_stop( 'trc_ini_wbstress : unable to allocate {sf,sn}_wbstress structure' )   ;   RETURN
          ENDIF
          CALL fld_fill( sf_wbstress, sn_wbstress, cn_dir, 'trc_ini_wbstress', 'Wave Bottom Stress', 'namben_wbstress' )

       CASE (2)
          ALLOCATE( slf_i(waves_botstr), STAT=ierr1 )
          IF( ierr1 > 0 ) THEN
             CALL ctl_stop( 'trc_ini_wbstress: unable to allocate slf_i structure' )   ;   RETURN
          ENDIF
          DO jn = 2 , 3
             jl = jn - 1
             slf_i(jl)       = sn_wbstress(jn)
             ALLOCATE( sf_wbstress(jl)%fnow(jpi,jpj,1)   , STAT=ierr2 )
             IF( sn_wbstress(jn)%ln_tint )  ALLOCATE( sf_wbstress(jl)%fdta(jpi,jpj,1,2) , STAT=ierr4 )
             IF( ierr2 + ierr4 > 0 ) THEN
                CALL ctl_stop( 'trc_ini_wbstress : unable to allocate {sf,sn}_wbstress structure' )   ;   RETURN
             ENDIF
             !
          ENDDO
          CALL fld_fill( sf_wbstress, slf_i, cn_dir, 'trc_ini_wbstress', 'Wave Bottom Stress', 'namben_wbstress' )

       CASE (3)
          ALLOCATE( slf_i(waves_botstr), STAT=ierr1 )
          IF( ierr1 > 0 ) THEN
             CALL ctl_stop( 'trc_ini_wbstress: unable to allocate slf_i structure' )   ;   RETURN
          ENDIF
          DO jn = 2 , 4
             jl = jn - 1
             slf_i(jl)       = sn_wbstress(jn)
             ALLOCATE( sf_wbstress(jl)%fnow(jpi,jpj,1)   , STAT=ierr2 )
             IF( sn_wbstress(jn)%ln_tint )  ALLOCATE( sf_wbstress(jl)%fdta(jpi,jpj,1,2) , STAT=ierr4 )
             IF( ierr2 + ierr4 > 0 ) THEN
                CALL ctl_stop( 'trc_ini_wbstress : unable to allocate {sf,sn}_wbstress structure' )   ;   RETURN
             ENDIF
             !
          ENDDO
          CALL fld_fill( sf_wbstress, slf_i, cn_dir, 'trc_ini_wbstress', 'Wave Bottom Stress', 'namben_wbstress' )
       END SELECT
    ENDIF

  END SUBROUTINE trc_ini_wbstress
 
  !
  SUBROUTINE trc_wbstress(kt,kmm)
    !!----------------------------------------------------------------------
    !!                     ***  trc_ini_wbstress  ***
    !!
    !! ** Purpose :  Compute wave bottom stress effect to be used in bottom stress
    !!
    !!----------------------------------------------------------------------
    use oce,     ONLY : rhop
    use phycst,  ONLY : rpi
    !
    INTEGER, INTENT( in ) ::   kt,kmm              ! ocean time-step index
    !
    INTEGER :: ji, jj
    !
    REAL(wp) :: ks, T, H, d , x, b, y, Uw, A, Rw , n, bottomRho
    REAL(wp) :: fw, fw_rough, fw_smooth

    SELECT CASE (waves_botstr)
    CASE (0)
       !wavestress = 0._wp

    CASE (1)
       CALL fld_read(kt=kt, kn_fsbc=1, sd=sf_wbstress)
       wavestress(:,:)=sf_wbstress(1)%fnow(:,:,1) ! stress from file
       
    CASE (2,3)
       CALL fld_read(kt=kt, kn_fsbc=1, sd=sf_wbstress)
       ! COMPUTE bottom STRESS due to WAVES
       ! using Hunt's method (1979), see https://eprints.hrwallingford.com/588/1/TR155.pdf 
       ! input: wave period and significant height, data from namben_wbstress
       ks = 2.5 * 250E-06
       DO_2D(1,1,1,1)
             IF ( tmask(ji,jj,mbkt(ji,jj)) .eq. 1 ) THEN
             d=gdept(ji,jj,mbkt(ji,jj),Kmm)
             IF ( d .lt. 200.0_wp ) THEN

                T = sf_wbstress(1)%fnow(ji,jj,1) ! peak period
                H = sf_wbstress(2)%fnow(ji,jj,1) ! significant height

                if (T.eq.0.0 .or. H.eq.0) then
                   wavestress(ji,jj)=0.0

                else
                  x=((2*rpi/T)**2)*d/9.81
                  b=1/(1+x*(0.66667+x*(0.35550+x*(0.16084+x*(0.06320+x*(0.02174+x*(0.00654+x*(0.00171+x*(0.00039+x*0.00011)))))))))
                  y=sqrt(x**2+b*x)
                  Uw=H/2*sqrt(2*9.81*y/(d*sinh(2*y)))

                  A=Uw*T/(2*rpi)
                  fw_rough=0.237*(A/ks)**(-0.52)
                  Rw=Uw*A/0.0000012  ! kinematic viscosity nu=0.0000012 [m2/s]
                  if (Rw<=5e5) then  ! laminar
                     B=2      ; N=0.5
                  else               ! turbulent
                     B=0.0521 ; N=0.187
                  end if
                  fw_smooth=B*Rw**(-N)
                  fw=max(fw_rough,fw_smooth)
                  bottomRho=rhop(ji,jj,mbkt(ji,jj))

                  wavestress(ji,jj)=0.5*bottomRho*fw*(Uw**2)
                end if
               
             END IF
             END IF
       END_2D
    END SELECT

    IF (waves_botstr .eq. 3) THEN 
       wave_direction = sf_wbstress(3)%fnow(ji,jj,1)  
    END IF 

  END SUBROUTINE trc_wbstress


  !!======================================================================
END MODULE trcwbstress
