MODULE trcrst_ben
   !!======================================================================
   !!                         ***  MODULE trcrst_ben  ***
   !! TOP :   Manage the behtic tracer restart
   !!======================================================================
   !! History :    -   !  1991-03  ()  original code
   !!             1.0  !  2005-03 (O. Aumont, A. El Moussaoui) F90
   !!              -   !  2005-10 (C. Ethe) print control
   !!             2.0  !  2005-10 (C. Ethe, G. Madec) revised architecture
   !!----------------------------------------------------------------------
   !!   ben_rst :   Restart for benthic tracer
   !!----------------------------------------------------------------------
   !!   ben_rst_opn    : open  restart file
   !!   ben_rst_read   : read  restart file
   !!   ben_rst_wri    : write restart file
   !!----------------------------------------------------------------------
   USE par_ben
   USE oce_trc
   USE trc
   USE iom
   USE daymod
   USE lib_mpp
   
   IMPLICIT NONE
   PRIVATE

   PUBLIC   ben_rst_opn       ! called by trcstp
   PUBLIC   ben_rst_read      ! called by trc_ini_ben 
   PUBLIC   ben_rst_wri       ! called by trcstp
   PUBLIC   ben_rst_cal

   INTEGER, PUBLIC ::   numrbr, numrbw   !: logical unit for trc restart (read and write)

CONTAINS
   
   SUBROUTINE ben_rst_opn( kt )
      !!----------------------------------------------------------------------
      !!                    ***  ben_rst_opn  ***
      !!
      !! ** purpose  :   output of ben-trc variable in a netcdf file
      !!----------------------------------------------------------------------
      INTEGER, INTENT(in) ::   kt       ! number of iteration, time step
      !
      CHARACTER(LEN=20)   ::   clkt     ! ocean time-step define as a character
      CHARACTER(LEN=256)  ::   clpath
      CHARACTER(LEN=50)   ::   clname   ! ben output restart file name
      CHARACTER(LEN=50)   ::   clpname  ! ben output restart file name including AGRIF
     !!----------------------------------------------------------------------
      !
      IF( l_offline ) THEN
         IF( kt == nittrc000 ) THEN
            lrst_trc = .FALSE.
            IF( ln_rst_list ) THEN
               nrst_lst = 1
               nitrst = nn_stocklist( nrst_lst )
            ELSE
               nitrst = nitend
            ENDIF
         ENDIF


         IF( .NOT. ln_rst_list .AND. MOD( kt - 1, nn_stock ) == 0 ) THEN
            ! we use kt - 1 and not kt - nittrc000 to keep the same periodicity from the beginning of the experiment
            nitrst = kt + nn_stock - 1                  ! define the next value of nitrst for restart writing
            IF( nitrst > nitend )   nitrst = nitend   ! make sure we write a restart at the end of the run
         ENDIF
      ELSE
         IF( kt == nittrc000 ) lrst_trc = .FALSE.
      ENDIF

      IF( .NOT. ln_rst_list .AND. nn_stock == -1 )   RETURN   ! we will never do any restart


     IF( kt == nitrst - 1 .OR. nn_stock == 1 .OR. ( kt == nitend - 1 .AND. .NOT. lrst_trc ) ) THEN
         ! beware of the format used to write kt (default is i8.8, that should be large enough)
         IF( nitrst > 1.0e9 ) THEN   ;   WRITE(clkt,*       ) nitrst
         ELSE                        ;   WRITE(clkt,'(i8.8)') nitrst
         ENDIF
         ! create the file         
         IF(lwp) WRITE(numout,*)
         clname = TRIM(cexper)//"_"//TRIM(ADJUSTL(clkt))//"_"//TRIM(cn_benrst_out)
         clpath = TRIM(cn_trcrst_outdir)
         IF( clpath(LEN_TRIM(clpath):) /= '/' ) clpath = TRIM(clpath) // '/'
         IF(lwp) WRITE(numout,*) &
             '             open ben restart.output NetCDF file: ',TRIM(clpath)//clname
         IF(.NOT.lwxios) THEN
            CALL iom_open( TRIM(clpath)//TRIM(clname), numrbw, ldwrt = .TRUE. , kdlev=jp_benlvl, cdcomp="SED")
         ELSE
#if defined key_xios
            cw_sedrst_cxt = "rstwt_"//TRIM(ADJUSTL(clkt))
            IF( TRIM(Agrif_CFixed()) == '0' ) THEN
               clpname = clname
            ELSE
               clpname = TRIM(Agrif_CFixed())//"_"//clname
            ENDIF
            numrbw = iom_xios_setid(TRIM(clpath)//TRIM(clpname))
            CALL iom_init( cw_sedrst_cxt, kdid = numrbw, ld_closedef = .FALSE. )
#else
            CALL ctl_stop( 'Can not use XIOS in ben_rst_opn' )
#endif
         ENDIF
         lrst_trc = .TRUE.
      ENDIF
      !
   END SUBROUTINE ben_rst_opn

   SUBROUTINE ben_rst_read ( Kbb, Kmm )
      !!----------------------------------------------------------------------
      !!                    ***  ben_rst_read  ***
      !!
      !! ** purpose  :   read Benthic tracer fields in restart files
      !!----------------------------------------------------------------------
      INTEGER, INTENT( in ) ::   Kbb, Kmm  ! time level indices
      INTEGER  ::  jn     

      !!----------------------------------------------------------------------
      !
      IF(lwp) WRITE(numout,*)
      IF(lwp) WRITE(numout,*) 'ben_rst_read : read data in the benthic restart file'
      IF(lwp) WRITE(numout,*) '~~~~~~~~~~~~'

      ! READ prognostic variables and computes diagnostic variable

      DO jn = 1, jp_ben
         CALL iom_get( numrbr, jpdom_auto, 'BT'//cbennm(jn), bt(:,:,:,jn,Kbb) )
      END DO
      bt(:,:,:,:,Kmm)=bt(:,:,:,:,Kbb)
      
      CALL iom_close( numrbr )
      !
   END SUBROUTINE ben_rst_read

   SUBROUTINE ben_rst_wri( kt , Kbb , Kmm, Krhs )
      IMPLICIT NONE
      !!----------------------------------------------------------------------
      !!                    ***  ben_rst_wri  ***
      !!
      !! ** purpose  :   write Benthic tracer fields in restart files
      !!----------------------------------------------------------------------
      INTEGER, INTENT( in ) ::   kt    ! ocean time-step index
      INTEGER, INTENT( in ) ::   Kbb, Kmm, Krhs  ! time level indices
      !!
      INTEGER  :: jn
      REAL(wp) :: zarak0
      !!----------------------------------------------------------------------
      !
      !CALL ben_rst_cal( kt, 'WRITE' )   ! calendar is called already in trcstp
      CALL iom_rstput( kt, nitrst, numrbw, 'rdttrc1', rn_Dt )   ! passive tracer time step (=ocean time step)
      ! prognostic variables 
      ! -------------------- 
      !DO jn = 1, jp_ben
      !   CALL iom_rstput( kt, nitrst, numrbw, 'BTN'//cbennm(jn), btn(:,:,:,jn), plvl=ben_depths)
      !END DO

      DO jn = 1, jp_ben
         CALL iom_rstput( kt, nitrst, numrbw, 'BT'//cbennm(jn), bt(:,:,:,jn,Kbb)) !, plvl=ben_depths
      END DO

      IF( kt == nitrst ) THEN
          CALL iom_close( numrbw )     ! close the restart file (only at last time step)
          lrst_trc = .FALSE.
      ENDIF
      !
   END SUBROUTINE ben_rst_wri 


   SUBROUTINE ben_rst_cal( kt, cdrw )
      !!---------------------------------------------------------------------
      !!                   ***  ROUTINE ben_rst_cal  ***
      !!
      !!  ** Purpose : Read or write calendar in restart file:
      !!
      !!  WRITE(READ) mode:
      !!       kt        : number of time step since the begining of the experiment at the
      !!                   end of the current(previous) run
      !!       adatrj(0) : number of elapsed days since the begining of the experiment at the
      !!                   end of the current(previous) run (REAL -> keep fractions of day)
      !!       ndastp    : date at the end of the current(previous) run (coded as yyyymmdd integer)
      !!
      !!   According to namelist parameter nrstdt,
      !!       nn_rsttr = 0  no control on the date (nittrc000 is  arbitrary).
      !!       nn_rsttr = 1  we verify that nittrc000 is equal to the last
      !!                   time step of previous run + 1.
      !!       In both those options, the  exact duration of the experiment
      !!       since the beginning (cumulated duration of all previous restart runs)
      !!       is not stored in the restart and is assumed to be (nittrc000-1)*rdt.
      !!       This is valid is the time step has remained constant.
      !!
      !!       nn_rsttr = 2  the duration of the experiment in days (adatrj)
      !!                    has been stored in the restart file.
      !!----------------------------------------------------------------------
      INTEGER         , INTENT(in) ::   kt         ! ocean time-step
      CHARACTER(len=*), INTENT(in) ::   cdrw       ! "READ"/"WRITE" flag
      !
      LOGICAL  ::  llok
      REAL(wp) ::  zkt, zrdttrc1
      REAL(wp) ::  zndastp

      ! Time domain : restart
      ! ---------------------

      IF( TRIM(cdrw) == 'READ' ) THEN

         IF(lwp) WRITE(numout,*)
         IF(lwp) WRITE(numout,*) 'ben_rst_cal : read the benthic restart file for calendar'
         IF(lwp) WRITE(numout,*) '~~~~~~~~~~~~'

         CALL iom_open( cn_benrst_in, numrbr, kdlev = jp_benlvl)

         CALL iom_get ( numrbr, 'kt', zkt )   ! last time-step of previous run
         IF(lwp) THEN
            WRITE(numout,*) ' *** Info read in benthic restart : '
            WRITE(numout,*) '   previous time-step                               : ', NINT( zkt )
            WRITE(numout,*) ' *** restart option'
            SELECT CASE ( nn_rsttr )
            CASE ( 0 )   ;   WRITE(numout,*) ' nn_rsttr = 0 : no control of nittrc000'
            CASE ( 1 )   ;   WRITE(numout,*) ' nn_rsttr = 1 : no control the date at nittrc000 (use ndate0 read in the namelist)'
            CASE ( 2 )   ;   WRITE(numout,*) ' nn_rsttr = 2 : calendar parameters read in restart'
            END SELECT
            WRITE(numout,*)
         ENDIF
         ! Control of date 
         IF( nittrc000  - NINT( zkt ) /= 1 .AND.  nn_rsttr /= 0 )                                  &
            &   CALL ctl_stop( ' ===>>>> : problem with nittrc000 for the restart',                 &
            &                  ' verify the restart file or rerun with nn_rsttr = 0 (namelist)' )
         IF( l_offline ) THEN      ! set the date in offline mode
            ! Check dynamics and tracer time-step consistency and force Euler restart if changed
            IF( iom_varid( numrbr, 'rdttrc1', ldstop = .FALSE. ) > 0 )   THEN
               CALL iom_get( numrbr, 'rdttrc1', zrdttrc1 )
               IF( zrdttrc1 /= rn_Dt )   l_1st_euler = .false.
            ENDIF
            !                          ! define ndastp and adatrj
            IF ( nn_rsttr == 2 ) THEN
               CALL iom_get( numrbr, 'ndastp', zndastp ) 
               ndastp = NINT( zndastp )
               CALL iom_get( numrbr, 'adatrj', adatrj  )
            ELSE
               ndastp = ndate0 - 1     ! ndate0 read in the namelist in dom_nam
               adatrj = ( REAL( nittrc000-1, wp ) * rn_Dt ) / rday
               ! note this is wrong if time step has changed during run
            ENDIF
            !
            IF(lwp) THEN
              WRITE(numout,*) ' *** Info used values : '
              WRITE(numout,*) '   date ndastp                                      : ', ndastp
              WRITE(numout,*) '   number of elapsed days since the begining of run : ', adatrj
              WRITE(numout,*)
            ENDIF
            !
            CALL day_init          ! compute calendar
            !
         ENDIF
         !
      ELSEIF( TRIM(cdrw) == 'WRITE' ) THEN
         !
         IF(  kt == nitrst ) THEN
            IF(lwp) WRITE(numout,*)
            IF(lwp) WRITE(numout,*) 'ben_rst_cal : write the benthic restart file (NetCDF) at it= ', kt, ' date= ', ndastp
            IF(lwp) WRITE(numout,*) '~~~~~~~'
         ENDIF
         CALL iom_rstput( kt, nitrst, numrbw, 'kt'     , REAL( kt    , wp) )   ! time-step
         CALL iom_rstput( kt, nitrst, numrbw, 'ndastp' , REAL( ndastp, wp) )   ! date
         CALL iom_rstput( kt, nitrst, numrbw, 'adatrj' , adatrj            )   ! number of elapsed days since
         !                                                                     ! the begining of the run [s]
      ENDIF

   END SUBROUTINE ben_rst_cal

   !!----------------------------------------------------------------------
   !! NEMO/TOP 4.0 , NEMO Consortium (2017)
   !! $Id$ 
   !! Software governed by the CeCILL licence (NEMOGCM/NEMO_CeCILL.txt)
   !!======================================================================
END MODULE trcrst_ben
