module bamhbi_sinking

#include "bamhbi.h90"
#ifdef forcingmodel_nemo
  use bamhbi_driver_nemo
  use bamhbi
  use bamhbi_params
#include "domzgr_substitute.h90"
#endif
#ifdef forcingmodel_deSolve
  use bamhbi_driver_R
#endif
#include "do_loop_substitute.h90"
  
  IMPLICIT NONE

  contains  
!-----------------------------------------------------------------------
!
! !ROUTINE: trc_sink.F90
!
! !INTERFACE:
   subroutine trc_sink(kbb,kmm,krhs,m,ws_in,ws_scalar)  ! w is positive upward (so here it is probably mostly negative)
!
! !DESCRIPTION:
!
!  This subroutine is based on trc_sink_bfm, with the added possibility that the vertical velocity
!  can be scalar instead of a 3D field
!     
!  This subroutine computes the RHS of a one-dimensional advection equation
!  for the sinking of biogeochemical components. 
!  Conservative advection has to be applied when settling of sediment or
!  rising of phytoplankton is considered. 
!
!  For BAMHBI, fluxes are defined at grid center (as is the variable)
!  The (input) vertical velocity "ws" is positive upward (so it should be negative)
!
! !INPUT PARAMETERS:
   integer,  INTENT(IN)            :: krhs, kbb, kmm   ! time steps
   integer,  INTENT(IN)            :: m                ! index of the scalar to sink
   REAL(wp), optional, INTENT(IN)  :: ws_in(jpi,jpj,jpk)
   REAL(wp), optional, INTENT(IN)  :: ws_scalar
!
! !REVISION HISTORY:
!  Original author(s): Lars Umlauf (GOTM team)
!  BFM-NEMO adaptation: Marcello Vichi (CMCC-INGV)
!  BAMHBI-NEMO adaptation: Luc Vandenbulcke (MAST/ULg)
!
! !LOCAL VARIABLES:
   integer   :: ji,jj,jk
   real(wp)  :: ws(jpi,jpj,jpk),cu(jpi,jpj,jpk)
   real(wp)  :: locflux
!
!-----------------------------------------------------------------------

!  initialize fluxes at grid center with zero
   cu   = 0.0_wp

!  put sinking velocity in variable ws (positive upward, so usually negative)
   ws(:,:,:) = 0.0_wp
   if (present(ws_in)) then
      ws = ws_in * tmask
   elseif (present(ws_scalar)) then
      ws = ws_scalar * tmask
   else
     write(*,*) 'CRITICAL ERROR... NO SINKING VELOCITY PROVIDED TO TRC_SINK FOR SCALAR ',m
   endif

   ! Limit sinking speed so that CFL condition is met --> needs to be stricter, use 0.75% ?!?
   DO_2D(0,0,0,0)
         do jk = 1, mbkt(ji,jj)
            IF (ws(ji,jj,jk).ge.0) THEN
               ws(ji,jj,jk) = MIN( ws(ji,jj,jk) ,  e3t(ji,jj,jk,Kmm)/rDt_trc *0.8_wp )
            ELSE
               ws(ji,jj,jk) = MAX( ws(ji,jj,jk) , -e3t(ji,jj,jk,Kmm)/rDt_trc *0.8_wp )
            END IF
         end do
   END_2D

   DO_2D(0,0,0,0)
         do jk = 1, mbkt(ji,jj)
            cu(ji,jj,jk) = ws(ji,jj,jk) * tr(ji,jj,jk,m,Kbb)    ! mmol/m3 * m/s == mmol/(m2*s)
         end do
         ! forbid sinking beyond the lower cell (which is handled by the benthic system if enabled)
	 ! i.e. the flux cu at the lowest cell can only be positive (upward)
         jk=mbkt(ji,jj)
	 cu(ji,jj,jk)= max(cu(ji,jj,jk),0.0_wp)
   END_2D
      
   DO_2D(0,0,0,0)
         if (mbkt(ji,jj).gt.1) then                                     ! if there's only 1 layer, there's no sinking, dummy !
         jk = 1
            locflux = -abs(cu(ji,jj,jk))                                ! whether it goes up or down, it goes out of the cell, so negative sign
            if (cu(ji,jj,jk+1).gt.0) locflux = locflux + cu(ji,jj,jk+1) ! add stuff coming in from below
            tr(ji,jj,jk,m,Krhs) = tr(ji,jj,jk,m,Krhs) + tmask(ji,jj,jk) * locflux / e3t(ji,jj,jk,Kmm)
         jk = mbkt(ji,jj)                                               ! in the bottom cell, consider only exchange with the upper neighbour. Benthic model will take care of the rest
            locflux = -abs(cu(ji,jj,jk))
            if (cu(ji,jj,jk-1).lt.0) locflux = locflux - cu(ji,jj,jk-1)
            tr(ji,jj,jk,m,Krhs) = tr(ji,jj,jk,m,Krhs) + tmask(ji,jj,jk) * locflux / e3t(ji,jj,jk,Kmm)
         end if
         do jk = 2, mbkt(ji,jj)-1
            locflux = -abs(cu(ji,jj,jk))
            if (cu(ji,jj,jk-1).lt.0) locflux = locflux - cu(ji,jj,jk-1) ! substract the negative flux, so a positive contribution (entering the cell)
            if (cu(ji,jj,jk+1).gt.0) locflux = locflux + cu(ji,jj,jk+1)
            tr(ji,jj,jk,m,Krhs) = tr(ji,jj,jk,m,Krhs) + tmask(ji,jj,jk) * locflux / e3t(ji,jj,jk,Kmm)
         end do
   END_2D

   return
   end subroutine trc_sink

   !---------------------------------
   
#ifdef redistributeSinking
   subroutine redistribute_sinking(kbb,kmm)
   integer, intent(in)  :: kbb, kmm      ! time steps
   integer              :: k_sinkdist    ! index of the layer below which everything that sinks will be redistributed close to the surface
   integer              :: kk            ! index of the lowest surface layer where we redistribute
   real(wp)             :: dzsurf
   integer              :: ji,jj
   real(wp)             :: sunkenCDI, sunkenNDI, sunkenPOC, sunkenPON, sunkenSID
   
   DO_2D(0,0,0,0)
       k_sinkdist = min(mbkt(ji,jj), int(k_redistribute_bottom))
       sunkenCDI = - WDIA(ji,jj,k_sinkdist)*tr(ji,jj,k_sinkdist,CDI,kbb)   ! WDIA is negative, sunkenXXX is positive [ m/s * mmol/m3]
       sunkenNDI = - WDIA(ji,jj,k_sinkdist)*tr(ji,jj,k_sinkdist,NDI,kbb)      ! (WDIA is defined (computed) at the T points)
       sunkenPOC = - WPOM(ji,jj,k_sinkdist)*tr(ji,jj,k_sinkdist,POC,kbb)
       sunkenPON = - WPOM(ji,jj,k_sinkdist)*tr(ji,jj,k_sinkdist,PON,kbb)
       sunkenSID = - (-vsinkingrate_silicious_detritus*tr(ji,jj,k_sinkdist,SID,kbb))

       kk=minloc(abs(gdept(ji,jj,:,Kmm)-dz_redistribute_top),dim=1)            ! find k closest to redistribute_top_dz (e.g. 5 meter)
       dzsurf=sum(e3t(ji,jj,1:kk,kmm))
       dpDIC(ji,jj,1:kk)=dpDIC(ji,jj,1:kk) + 1/dzsurf*(sunkenCDI+sunkenPOC)    !  [ mmol/m3 /s ], will be integrated in time by Nemo
       dpNOS(ji,jj,1:kk)=dpNOS(ji,jj,1:kk) + 1/dzsurf*(sunkenNDI+sunkenPON)
       dpSIO(ji,jj,1:kk)=dpSIO(ji,jj,1:kk) + 1/dzsurf*(sunkenSID+sunkenNDI*SiNrDiatoms)
       dpPHO(ji,jj,1:kk)=dpPHO(ji,jj,1:kk) + 1/dzsurf*((sunkenNDI+sunkenPON)*PNRedfield)
       
       ! Here we consiburialC(1,1)*NCrSED(1,1,1)*PNRedfieldder that sinking material redistributed in the surface layers should NOT also accumulate at depth (i.e. NO duplication)
       ! REMOVE the next 13 lines to enable such tracer duplication
       if (k_sinkdist.ge.mbkt(ji,jj)) then ! sinking/redistribution is taken from the lower cell, we have to remove sinking material from there
          ddCDI(ji,jj,mbkt(ji,jj)) = ddCDI(ji,jj,mbkt(ji,jj)) + sunkenCDI/e3t(ji,jj,mbkt(ji,jj),kmm)
          ddNDI(ji,jj,mbkt(ji,jj)) = ddNDI(ji,jj,mbkt(ji,jj)) + sunkenNDI/e3t(ji,jj,mbkt(ji,jj),kmm)
          ddPOC(ji,jj,mbkt(ji,jj)) = ddPOC(ji,jj,mbkt(ji,jj)) + sunkenPOC/e3t(ji,jj,mbkt(ji,jj),kmm)
          ddPON(ji,jj,mbkt(ji,jj)) = ddPON(ji,jj,mbkt(ji,jj)) + sunkenPON/e3t(ji,jj,mbkt(ji,jj),kmm)
          ddSID(ji,jj,mbkt(ji,jj)) = ddSID(ji,jj,mbkt(ji,jj)) + sunkenSID/e3t(ji,jj,mbkt(ji,jj),kmm)
       else
          ddCDI(ji,jj,k_sinkdist+1) = ddCDI(ji,jj,k_sinkdist+1) + sunkenCDI/e3t(ji,jj,mbkt(ji,jj),kmm)
          ddNDI(ji,jj,k_sinkdist+1) = ddNDI(ji,jj,k_sinkdist+1) + sunkenNDI/e3t(ji,jj,mbkt(ji,jj),kmm)
          ddPOC(ji,jj,k_sinkdist+1) = ddPOC(ji,jj,k_sinkdist+1) + sunkenPOC/e3t(ji,jj,mbkt(ji,jj),kmm)
          ddPON(ji,jj,k_sinkdist+1) = ddPON(ji,jj,k_sinkdist+1) + sunkenPON/e3t(ji,jj,mbkt(ji,jj),kmm)
          ddSID(ji,jj,k_sinkdist+1) = ddSIO(ji,jj,k_sinkdist+1) + sunkenSID/e3t(ji,jj,mbkt(ji,jj),kmm)
       endif
   END_2D
   end subroutine redistribute_sinking
#endif

#ifdef redistributeLoss
   subroutine redistribute_loss(kbb,kmm)
   integer  :: ji,jj,kk,kbb,kmm
   real(wp) :: dzsurf,n2loss
   DO_2D(0,0,0,0)
       kk=minloc(abs(gdept(ji,jj,:,Kmm)-dz_redistribute_top),dim=1)
       dzsurf=sum(e3t(ji,jj,1:kk,kmm))
       n2loss= denitrificationIntegrated(ji,jj) + ANAMMOXIntegrated(ji,jj) + Oxidation_by_nitrateIntegrated(ji,jj)
       dpNOS(ji,jj,1:kk) = dpNOS(ji,jj,1:kk) + N2loss / dzsurf
#ifdef benthic
       dpDIC(ji,jj,1:kk) = dpDIC(ji,jj,1:kk) + burialC(ji,jj)
       dpNOS(ji,jj,1:kk) = dpNOS(ji,jj,1:kk) + burialC(ji,jj)*bt(ji,jj,1,NCrSED,kbb) + denitinsed(ji,jj)
       dpSIO(ji,jj,1:kk) = dpSIO(ji,jj,1:kk) + burialS(ji,jj)
       dpPHO(ji,jj,1:kk) = dpPHO(ji,jj,1:kk) + burialC(ji,jj)*bt(ji,jj,1,NCrSED,kbb)*PNRedfield
#endif
   END_2D
   end subroutine redistribute_loss
#endif


end module bamhbi_sinking

