  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!
  !                BAMHBI biogeochemical model module                     !
  !                                                                       !
  ! v0.1 created and included in the GOTM model       (M. Gregoire, 200X) !
  ! v0.2 improved and moved to the GHER 3D model         (A. Capet, 2013) !
  ! v0.3 adapted and optimized for the NEMO 3.X model                     !
  !                                    (L. Vandenbulcke, T. Lovato, 2017) !
  ! v1.0 adapted for different models (but specifically NEMO 4.2)         !
  !                              (M. Gregoire and MAST lab, ULiege, 2025) !
  !!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!!

  !------------------------------------------------------------------------------!
  ! Work flow when coupled to NEMO TOP module (my_trc)
  !
  ! Using my_trc, the # of tracers is set in par_trc, variable jp_my_trc
  ! The variables are listed in namelist_top (in the EXP directory), and in
  ! XIOS's iodef.xml for saving to netcdf.
  !
  ! Initialisation is done in subroutine trc_init (file trcini.F90), which reads
  ! the bio namelist, then calls trc_ini_my_trc, which calls trc_sms_my_trc_alloc
  ! in file trcsms_my_trc.F90, which itself calls "bioinit" (in this module)
  ! trc_init also reads the restart file (if needed)
  !
  ! The general workflow for the bio model is given in subroutine trc_sms_my_trc
  ! (file trcsms_my_trc.F90), which calls
  ! prebio, CalculateFlagellates, Calculate*, UpdateBoundaryFluxes ...
  ! All these routines are in this module (bamhbi)
  ! The routines update production and destruction terms, which are then
  ! added to the model state variable tr(x,y,z,variable,time) by trcsms_my_trc.
  ! 
  !------------------------------------------------------------------------------!

module bamhbi
#include "bamhbi.h90"

#ifdef forcingmodel_nemo
  use bamhbi_driver_nemo
  use bamhbi_params
#include "domzgr_substitute.h90"
#endif
#ifdef forcingmodel_deSolve
  use bamhbi_driver_R ! contains also declaration of bamhbi params
#endif

#ifdef alkalinity_system
  use bamhbi_alkalinity_vars
#endif
  use benthic_metamodels
  
  real(wp) :: dt
  
  ! some variables (temporary, outputs, diagnostics etc) will only be allocated if some compilation flags are defined (biodiagrates etc)
  real(wp),allocatable ::   CLIPC(:,:,:),  CLIPN(:,:,:), CLIPSi(:,:,:), CLIPChl(:,:,:)
  real(wp),allocatable ::   PAR(:,:,:)
  real(wp),allocatable ::   WPOM(:,:,:),WDIA(:,:,:)
#ifdef aggregation 
  real(wp),allocatable ::  WAGG(:,:,:)
#endif
  real(wp),allocatable ::   WMIGSID(:,:,:), WMIGPOM(:,:,:)
  real(wp),allocatable ::   CONSC(:,:,:)
  real(wp),allocatable ::   CONSN(:,:,:)
  real(wp),allocatable ::   CONSSi(:,:,:)
  real(wp),allocatable ::   CONSOxy(:,:,:)
  real(wp),allocatable ::   conspho(:,:,:)
  real(wp),allocatable ::   BudgetC(:,:,:)
  real(wp),allocatable ::   BudgetN(:,:,:)
  real(wp),allocatable ::   BudgetSi(:,:,:)
  real*8,allocatable   ::   integrated_budgetC, integrated_budgetN, integrated_budgetSi
#ifdef chlorophyll_geider
  real(wp),allocatable ::   PphotoC_DIAT(:,:,:)
  real(wp),allocatable ::   RespChl_DIAT(:,:,:)
  real(wp),allocatable ::   MortChl_DIAT(:,:,:)
  real(wp),allocatable ::   growthCHD_DIAT(:,:,:)
  real(wp),allocatable ::   PmaxC_DIAT(:,:,:)
  real(wp),allocatable ::   RhoChl_DIAT(:,:,:)
  real(wp),allocatable ::   ChlorosynthesisCHD_DIAT(:,:,:)
  real(wp),allocatable ::   VNC_DIAT(:,:,:)
  real(wp),allocatable ::   LightLimCHD_DIAT(:,:,:) 
  real(wp),allocatable ::   Growth_DIAT(:,:,:)
  real(wp),allocatable ::   tf_DIAT(:,:,:)
  real(wp),allocatable ::   tf_SILICATE(:,:,:)
  real(wp),allocatable ::   ChlCr_DIAT(:,:,:)
  real(wp),allocatable ::   ChlCr_DIAT_borne(:,:,:)
  real(wp),allocatable ::   ChlCr_DIAT_bamhbi(:,:,:)
  real(wp),allocatable ::   Nitrate_uptake_DIAT(:,:,:)
  real(wp),allocatable ::   Ammonium_uptake_DIAT(:,:,:)
  real(wp),allocatable ::   Phosphate_uptake_DIAT(:,:,:)
  real(wp),allocatable ::   NCr_DIAT(:,:,:)
  real(wp),allocatable ::   NCr_DIAT_borne(:,:,:)
  real(wp),allocatable ::   DOC_extra_excr_DIAT(:,:,:)
  real(wp),allocatable ::   CHL_geider(:,:,:),CHL_GeiderIntegrated(:,:)
#endif
  
  real(wp),allocatable ::   dPNFL(:,:,:),dDNFL(:,:,:),  dPCFL(:,:,:),dDCFL(:,:,:),  dPNDI(:,:,:),dDNDI(:,:,:),  dPCDI(:,:,:),dDCDI(:,:,:)
  real(wp),allocatable ::   dPNEM(:,:,:),dDNEM(:,:,:),  dPCEM(:,:,:),dDCEM(:,:,:),  dPMIC(:,:,:),dDMIC(:,:,:),  dPMES(:,:,:),dDMES(:,:,:)
  real(wp),allocatable ::   dPPOC(:,:,:),dDPOC(:,:,:)
  real(wp),allocatable ::   dPPON(:,:,:),dDPON(:,:,:),  dPDNL(:,:,:),dDDNL(:,:,:),  dPDCL(:,:,:),dDDCL(:,:,:),  dPDNS(:,:,:),dDDNS(:,:,:)
  real(wp),allocatable ::   dPDCS(:,:,:),dDDCS(:,:,:),  dPNOS(:,:,:),dDNOS(:,:,:),  dPNHS(:,:,:),dDNHS(:,:,:)
  
#ifdef aggregation
  real(wp),allocatable ::   dPAGG(:,:,:),dDAGG(:,:,:)
#endif
  real(wp),allocatable ::   dPDOX(:,:,:),dDDOX(:,:,:),  dPDIC(:,:,:),dDDIC(:,:,:),  dPODU(:,:,:),dDODU(:,:,:),  dPBAC(:,:,:),dDBAC(:,:,:)
  real(wp),allocatable ::   dPSIO(:,:,:),dDSIO(:,:,:),  dPSID(:,:,:),dDSID(:,:,:)
#ifdef gelatinous
  real(wp),allocatable ::   dPGEL(:,:,:),dDGEL(:,:,:),  dPNOC(:,:,:),dDNOC(:,:,:)
#endif
  real(wp),allocatable ::   dPPHO(:,:,:),dDPHO(:,:,:)
#ifdef chlorophyll_geider
  real(wp),allocatable ::   dPCHD(:,:,:),dDCHD(:,:,:),  dPCHE(:,:,:),dDCHE(:,:,:),  dPCHF(:,:,:),dDCHF(:,:,:)
#endif

#ifdef temp_clip
  real(wp),allocatable ::   unclipped_var(:,:,:,:)
#endif

  real(wp),allocatable ::   Airseaoxygenflux(:,:)  !in mmolC/m2/sec airseaCO2flux positive if downward as it is computed in the routine DIC flux: please check if it is in agreement with the convention  made in the model (otherwise reverse)
  real(wp)       :: SinkingRateDiatoms , NCrDiatom, SinkingRatePOM,SinkingRateAGG

  !! sediments
  real(wp),allocatable :: tauxb(:,:),tauyb(:,:),botstress(:,:),curstress(:,:)  
  real(wp),allocatable :: resuspcounter_f(:,:),resuspcounter_s(:,:) !  19.12.23 Resuspensioncounter added 
  real(wp),allocatable,save :: botcon(:,:,:)
  real(wp),allocatable,save :: FluxBottotN(:,:),FluxBottotC(:,:),FluxBottotS(:,:),FluxBotodu(:,:),fluxsurfdox(:,:)
  real(wp),allocatable,save :: FluxBotnhs(:,:) ,Fluxbotnos(:,:) ,FluxBotsio(:,:), FluxBotdic(:,:),FluxBotdox(:,:)
  real(wp),allocatable,save :: FluxBotPHO(:,:)
  
  real(wp),allocatable ::   IronLimitation(:)

#ifdef aggregation
  real(wp),allocatable,save :: FluxBotagg(:,:)
#endif
  real(wp),allocatable,save :: FluxBotcdi(:,:),fluxbotndi(:,:)
  real(wp),allocatable,save :: FluxBotpon(:,:),fluxbotpoc(:,:),fluxbotsid(:,:)
#ifdef sedparamdiag
  real(wp),allocatable,save :: pnit2D(:,:),pdenit2D(:,:),panox2D(:,:)
  real(wp),allocatable,save :: burialC(:,:),burialS(:,:),denitinsed(:,:)
  real(wp),allocatable,save :: fluxbotpoc_resusp(:,:), fluxbotpoc_depo(:,:),fluxbotsid_resusp(:,:),fluxbotsid_depo(:,:),fluxbotpon_resusp(:,:),fluxbotpon_depo(:,:)
#endif
  real(wp),allocatable,save :: c_db(:,:)
  integer                   ::  fCSED, sCSED, sSSED, fSSED, NCrSED
#ifdef testcons2
  real(wp), pointer, dimension(:,:), save  :: ben_budgetC, ben_budgetN, ben_budgetSi
  real*8, pointer, save                    :: PB_budget_C, PB_budget_N, PB_budget_Si
#endif
  
  real(wp)  ::    pdenitMult                  !- facteur multiplicatif pour pdenit
  
  real(wp),allocatable :: Cmin(:,:),Nmin(:,:),Smin(:,:)
  real(wp),allocatable :: zun(:,:,:),zvn(:,:,:) ! to compute shear stress in AGG module
  
  real(wp),allocatable ::  kbio_short(:,:,:), kbio_long(:,:,:)

  ! More diagnostic variables
#ifdef primaryprod
  REAL(WP),allocatable ::  GPP(:,:,:), NPP(:,:,:), NPPintegrated(:,:) ! primary production
#endif

#ifdef biodiagrates
! Unless stated otherwise, all respirations are in units of Carbon, not Oxygen
! (especially important when closing the oxygen budget with Zooplanktonrespiration
  real(wp),allocatable ::         Denitrification(:,:,:)           ! mmolN/m3/day  DEnitrification rate
  real(wp),allocatable ::             DOCLuptake_BAC(:,:,:)           ! mmolC/m3/day  Uptake of labile DOC
  real(wp),allocatable ::    Totalrespiration_zoo(:,:,:)           ! mmolC/m3/day
  real(wp),allocatable ::    Bacteria_respiration(:,:,:)           ! mmolC/m3/day  Respiration of bacteria
  real(wp),allocatable ::    TotalRespiration_Gel(:,:,:)           ! mmolC/m3/day
  real(wp),allocatable ::         Denitrificationintegrated(:,:)   ! mmolN/m2/day  DEnitrification rate
  real(wp),allocatable ::             DOCLuptake_BACIntegrated(:,:)   ! mmolC/m2/day  Uptake of labile DOC
  real(wp),allocatable ::    Bacteria_respirationIntegrated(:,:)   ! mmolC/m2/day  Respiration of bacteria
  real(wp),allocatable ::    TotalRespiration_ZOOIntegrated(:,:)   !
  real(wp),allocatable ::    TotalRespiration_GelIntegrated(:,:)   !

  REAL(WP),allocatable :: PHYTONITRATEREDUCTION(:,:,:)             ! ?
  REAL(WP),allocatable ::         Nitrification(:,:,:)             ! Ammonium to Nitrate 
  REAL(WP),allocatable ::               ANAMMOX(:,:,:)             ! NHS and NOS -> N2

  REAL(WP),allocatable :: PHYTONITRATEREDUCTIONIntegrated(:,:)     !
  REAL(WP),allocatable ::    NitrificationIntegrated(:,:)          !
  REAL(WP),allocatable ::          ANAMMOXIntegrated(:,:)          !

  real(wp),allocatable ::  C_Phymort_output(:,:,:)                 ! Mortality of phytoplankton in Carbon units
  real(wp),allocatable ::  N_Phymort_output(:,:,:)                 ! Mortality of phytoplankton in Nitrogen units
  real(wp),allocatable ::  NHSuptake_BAC(:,:,:)                    ! Uptake of NHS by BACTERIA
  real(wp),allocatable ::  BACExcretion(:,:,:)                     ! Excretion by Bacteria in Nitrogen units
  real(wp),allocatable ::  DOXprod_PHY_nutuptake(:,:,:)            ! DOX production by phytoplankton nutrient uptake
  real(wp),allocatable ::  DOXprod_PHY_growth(:,:,:)               ! DOXproduction by phytoplankton due to growth
  real(wp),allocatable ::  SID_dissolution(:,:,:)                  ! SID dissolves and becomes SIO 
  real(wp),allocatable ::  SIO_uptake_diatoms(:,:,:)               ! Sil-uptake by diatoms
  real(wp),allocatable ::  SID_creation(:,:,:)                     ! Creation of SID due to Diatom mortality and MesoZoograzing
  real(wp),allocatable ::  PHY_NHS_excr(:,:,:)                     ! Ammonium (and Phosphate-> Redfield ratio) production byPHY
  real(wp),allocatable ::  PHY_N_uptake(:,:,:)                     ! Nutrient uptake of phytoplankton (NHS and NOS)
  real(wp),allocatable ::  zoo_n_excretion(:,:,:)                  ! Excrection of Zooplankton in Nitrogen

  real(wp),allocatable ::   TotalRespirationFlagellates(:,:,:)    
  real(wp),allocatable ::   Carbon_UptakeFlagellates(:,:,:)
  real(wp),allocatable ::   Nitrogen_Uptake_Flagellates(:,:,:)
  real(wp),allocatable ::   TotalRespirationDiatoms(:,:,:)
  real(wp),allocatable ::   Carbon_UptakeDiatoms(:,:,:)
  real(wp),allocatable ::   Nitrogen_Uptake_Diatoms(:,:,:)
  real(wp),allocatable ::   TotalRespirationEmiliana(:,:,:)
  real(wp),allocatable ::   Carbon_UptakeEmiliana(:,:,:)
  real(wp),allocatable ::   Nitrogen_Uptake_Emiliana(:,:,:)
  real(wp),allocatable ::   Silicate_upDiatoms(:,:,:)

  real(wp),allocatable ::   Oxidation_by_nitrate(:,:,:)               ! Destruction of NOS for Oxidation of ODU and NOS (unit: O)
  real(wp),allocatable ::   Oxidation_by_oxygen(:,:,:)                ! Destruction of Oxygen by ODU and NHS oxidation  (unit: 0)

  real(wp),allocatable ::   bacteria_anoxrem(:,:,:)                   ! ODU production process 
  real(wp),allocatable ::   bacteria_oxygenconsumption(:,:,:)         ! Bacteria Oxygen consumption

  real(wp),allocatable ::  DOCSLHYDR_output(:,:,:)                    ! Hydrolisis of semi-labile DOC
  real(wp),allocatable ::  DONSLHYDR_output(:,:,:)                    ! Hydrolisis of semi-labile DON
  real(wp),allocatable ::   POCHYDR_output(:,:,:)                     ! Hydrolisis of POC
  real(wp),allocatable ::   PONHYDR_output(:,:,:)                     ! Hydrolisis of PON
  real(wp),allocatable ::  BAC_GROWTH(:,:,:)                          ! Growth of Bacteria in N
  real(wp),allocatable ::  DONLuptake_BAC(:,:,:)                      ! Uptake of DONL by Bacteria
  real(wp),allocatable ::  C_BACMort_output(:,:,:)                    ! Mortality of Bacteria in C
  real(wp),allocatable ::  N_BACMort_output(:,:,:)                    ! Mortality of Bacteria in N

!Vertically integrated variants of diagnostics
  real(wp),allocatable ::   TotalRespirationDiatomsIntegrated(:,:)
  real(wp),allocatable ::   Carbon_UptakeDiatomsIntegrated(:,:)
  real(wp),allocatable ::   Nitrogen_Uptake_DiatomsIntegrated(:,:)
  real(wp),allocatable ::   TotalRespirationFlagellatesIntegrated(:,:)
  real(wp),allocatable ::   Carbon_UptakeFlagellatesIntegrated(:,:)
  real(wp),allocatable ::   Nitrogen_Uptake_EmilianaIntegrated(:,:)
  real(wp),allocatable ::   Silicate_upDiatomsIntegrated(:,:)
  real(wp),allocatable ::   bacteria_anoxremIntegrated(:,:)
  real(wp),allocatable ::   Oxidation_by_nitrateIntegrated(:,:)
  real(wp),allocatable ::   Oxidation_by_oxygenIntegrated(:,:)
  real(wp),allocatable ::   bacteria_oxygenconsumptionIntegrated(:,:)
  REAL(WP),allocatable ::   Nitrogen_Uptake_FlagellatesIntegrated(:,:)
  REAL(WP),allocatable ::   TotalRespirationEmilianaIntegrated(:,:)
  REAL(WP),allocatable ::   Carbon_UptakeEmilianaIntegrated(:,:)      
#endif
  
#ifdef biodiagtrophic
  REAL(WP),allocatable ::  Phy_To_ZOO(:,:,:)               ! Grazing of Phyto by ZOO 
  REAL(WP),allocatable ::  Bac_To_ZOO(:,:,:)               ! Grazing of Bacteria by ZOO
  REAL(WP),allocatable ::  Poc_To_ZOO(:,:,:)               ! Grazing of POC by ZOO
  REAL(WP),allocatable ::  PON_To_ZOO(:,:,:)               ! Grazing of PON by ZOO
  REAL(WP),allocatable ::  phy_to_ZOOIntegrated(:,:)
  REAL(WP),allocatable ::  bac_to_ZOOIntegrated(:,:)
  REAL(WP),allocatable ::  poc_to_ZOOIntegrated(:,:)
  REAL(WP),allocatable ::  PON_To_ZOOIntegrated(:,:)
  real(wp),allocatable ::  ZOO_TO_POC(:,:,:)               ! Mortality and excretion that becomes POC (unit: C)
  real(wp),allocatable ::  ZOO_TO_PON(:,:,:)               ! Mortality and excretion that becomes PON (unit: N)
  real(wp),allocatable ::  DOC_leakage_output(:,:,:)       ! Leakage of DOC during PHY carbon uptake (photosynth)
  real(wp),allocatable ::  DON_leakage_output(:,:,:)       ! Leakage of DON during PHY nutrient uptake
  real(wp),allocatable ::  DOC_extra_excr_output(:,:,:)    ! Extra excretion of DOC by PHY
  real(wp),allocatable ::  C_ZOOMessyfeeding_output(:,:,:) !Messy feeding loss of ZOO grazing on C
  real(wp),allocatable ::  N_ZOOMessyfeeding_output(:,:,:) !Messy feeding loss of ZOO grazing on N
#endif

#ifdef agg_diag
  real(wp),allocatable ::  shearrate3D(:,:,:)
#endif
    
  real(wp), parameter  :: small_concentration = 1.0e-12 !0 leads to nans

#include "do_loop_substitute.h90"

contains

  !******************************************************************************************************
  !  INITIALISATION (ALLOCATIONS) : when coupled to NEMO,
  !  bioinit is called from trc_sms_my_trc_alloc function in trcsms_my_trc.F90
  !  This function is itself called from trc_ini_my_trc subroutine in trcini_my_trc.F90
  !  This last subroutine is itself called from trc_init in trcini.F90, which is called from nemogcm.f90
  !******************************************************************************************************
  subroutine bioinit(alloc_ok,kbb,kmm)
  implicit none
  integer, intent(out) :: alloc_ok
  integer, intent(in) :: kbb,kmm
  character*256 :: fname
  integer :: ji,jj,jk,jn,ikt,istat

  CFL=1
  NFL=2
  CEM=3
  NEM=4
  CDI=5
  NDI=6
  MIC=7
  MES=8
  BAC=9
  DCL=10
  DNL=11
  DCS=12
  DNS=13
  NOS=14
  NHS=15
  SIO=16
  DOX=17
  DIC=18
  ODU=19
  POC=20
  PON=21
  SID=22
  PHO=23
  jn=24
#ifdef aggregation
  AGG=jn; jn=jn+1
#endif
#ifdef gelatinous
  GEL=jn; jn=jn+1
  NOC=jn; jn=jn+1
#endif
#ifdef chlorophyll_geider
  CHD=jn; jn=jn+1
  CHE=jn; jn=jn+1
  CHF=jn; jn=jn+1
#endif
#ifdef alkalinity_system
  CHA=jn; jn=jn+1
#endif

! check if there are as many state variables in namelist_top_cfg as in bamhbi.?90
  alloc_ok=0
  allocate(dPNFL(jpi,jpj,jpk),dDNFL(jpi,jpj,jpk),  dPCFL(jpi,jpj,jpk),dDCFL(jpi,jpj,jpk),  dPNDI(jpi,jpj,jpk),dDNDI(jpi,jpj,jpk),  dPCDI(jpi,jpj,jpk),dDCDI(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(dPNEM(jpi,jpj,jpk),dDNEM(jpi,jpj,jpk),  dPCEM(jpi,jpj,jpk),dDCEM(jpi,jpj,jpk),  dPMIC(jpi,jpj,jpk),dDMIC(jpi,jpj,jpk),  dPMES(jpi,jpj,jpk),dDMES(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(dPPOC(jpi,jpj,jpk),dDPOC(jpi,jpj,jpk),  stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(dPPON(jpi,jpj,jpk),dDPON(jpi,jpj,jpk),  dPDNL(jpi,jpj,jpk),dDDNL(jpi,jpj,jpk),  dPDCL(jpi,jpj,jpk),dDDCL(jpi,jpj,jpk),  dPDNS(jpi,jpj,jpk),dDDNS(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(dPDCS(jpi,jpj,jpk),dDDCS(jpi,jpj,jpk),  dPNOS(jpi,jpj,jpk),dDNOS(jpi,jpj,jpk),  dPNHS(jpi,jpj,jpk),dDNHS(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(dPDOX(jpi,jpj,jpk),dDDOX(jpi,jpj,jpk),  dPDIC(jpi,jpj,jpk),dDDIC(jpi,jpj,jpk),  dPODU(jpi,jpj,jpk),dDODU(jpi,jpj,jpk),  dPBAC(jpi,jpj,jpk),dDBAC(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(dPSIO(jpi,jpj,jpk),dDSIO(jpi,jpj,jpk),  dPSID(jpi,jpj,jpk),dDSID(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#ifdef gelatinous
  allocate(dPGEL(jpi,jpj,jpk),dDGEL(jpi,jpj,jpk),  dPNOC(jpi,jpj,jpk),dDNOC(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#endif
#ifdef aggregation
  allocate(  dPAGG(jpi,jpj,jpk),dDAGG(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#endif
  allocate(dPPHO(jpi,jpj,jpk),dDPHO(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#ifdef chlorophyll_geider
  allocate(dPCHD(jpi,jpj,jpk),dDCHD(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(dPCHE(jpi,jpj,jpk),dDCHE(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(dPCHF(jpi,jpj,jpk),dDCHF(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#endif
  
  dPCFL=0. ; dPNFL=0. ; dPCEM=0. ; dPNEM=0. ; dPCDI=0. ; dPNDI=0. ; dPMIC=0. ; dPMES=0. ; dPBAC=0. ; dPDCL=0. ; dPDNL=0. ; dPDCS=0. ; dPDNS=0. ;
  dPNOS=0. ; dPNHS=0. ; dPSIO=0. ; dPDOX=0. ; dPDIC=0. ; dPODU=0. ; dPPOC=0. ; dPPON=0. ; dPSID=0. ; dPPHO=0. ;
  dDCFL=0. ; dDNFL=0. ; dDCEM=0. ; dDNEM=0. ; dDCDI=0. ; dDNDI=0. ; dDMIC=0. ; dDMES=0. ; dDBAC=0. ; dDDCL=0. ; dDDNL=0. ; dDDCS=0. ; dDDNS=0. ;
  dDNOS=0. ; dDNHS=0. ; dDSIO=0. ; dDDOX=0. ; dDDIC=0. ; dDODU=0. ; dDPOC=0. ; dDPON=0. ; dDSID=0. ; dDPHO=0. ;
#ifdef gelatinous
  dPGEL=0. ; dDGEL=0. ; dPNOC=0. ; dDNOC=0.
#endif
#ifdef aggregation
  dPAGG=0. ; dDAGG=0.
#endif
#ifdef chlorophyll_geider
  dPCHD=0. ; dDCHD=0.; dPCHE=0. ; dDCHE=0. ; dPCHF=0. ; dDCHF=0.
#endif

  allocate( AirSeaOxygenFlux(jpi,jpj), stat=istat); alloc_ok=alloc_ok + istat
  allocate( IronLimitation(jpk), stat=istat); alloc_ok=alloc_ok + istat
#ifdef temp_clip
  allocate( unclipped_var(jpi,jpj,jpk,jp_bgc), stat=istat ) ; alloc_ok=alloc_ok + istat
#endif
  allocate( CLIPC(jpi,jpj,jpk),CLIPN(jpi,jpj,jpk),CLIPSi(jpi,jpj,jpk),CLIPChl(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   Chlorophyll(jpi,jpj,jpk,3), stat=istat) ; alloc_ok=alloc_ok + istat ! 3 phyto groups
#ifdef chlorophyll_geider
  allocate(  CHL_Geider(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(  CHL_GeiderIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   PphotoC_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   RespChl_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   MortChl_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   growthCHD_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   PmaxC_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   RhoChl_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   ChlorosynthesisCHD_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   VNC_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   LightLimCHD_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat 
  allocate(   Growth_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   tf_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   tf_SILICATE(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   ChlCr_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   ChlCr_DIAT_borne(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   ChlCr_DIAT_bamhbi(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   Nitrate_uptake_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   Ammonium_uptake_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   Phosphate_uptake_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   NCr_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   NCr_DIAT_borne(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   DOC_extra_excr_DIAT(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#endif
  
  allocate(   PAR(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#ifdef testcons
  allocate(   CONSC(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   CONSN(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   CONSSi(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   CONSOxy(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   conspho(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#endif
#ifdef testcons2
  allocate(   BudgetC(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   BudgetN(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   BudgetSi(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   integrated_budgetC, integrated_budgetN, integrated_budgetSi, stat=istat) ; alloc_ok=alloc_ok + istat
#endif

#ifdef fulloptics
    allocate( kbio_short(jpi,jpj,jpk), kbio_long(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#endif
  
#ifdef primaryprod
  allocate( GPP(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( NPP(jpi,jpj,jpk), NPPintegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  GPP=0.0_wp ; NPP=0.0_wp ; NPPintegrated=0.0_wp
#endif

#ifdef biodiagrates
  allocate(      Denitrification(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(          DOCLuptake_BAC(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Bacteria_respiration(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Totalrespiration_ZOO(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( TotalRespiration_Gel(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(PHYTONITRATEREDUCTION(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(        Nitrification(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(              ANAMMOX(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(      Denitrificationintegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(          DOCLuptake_BACIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Bacteria_respirationIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( TotalRespiration_ZOOIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( TotalRespiration_GelIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(PHYTONITRATEREDUCTIONIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(   NitrificationIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(         ANAMMOXIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(C_Phymort_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( N_Phymort_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(NHSuptake_BAC(jpi,jpj,jpk), BACExcretion(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( DOXprod_PHY_nutuptake(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(DOXprod_PHY_growth(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( SID_dissolution(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(SIO_uptake_diatoms(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(SID_creation(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( PHY_NHS_excr(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( PHY_N_uptake(jpi,jpj,jpk),  stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(zoo_n_excretion(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     TotalRespirationFlagellates(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(        Carbon_UptakeFlagellates(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     Nitrogen_Uptake_Flagellates(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     TotalRespirationDiatoms(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(        Carbon_UptakeDiatoms(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     Nitrogen_Uptake_Diatoms(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     TotalRespirationEmiliana(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(        Carbon_UptakeEmiliana(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     Nitrogen_Uptake_Emiliana(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     Silicate_upDiatoms(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     Oxidation_by_nitrate(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     Oxidation_by_oxygen(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     bacteria_anoxrem(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(     bacteria_oxygenconsumption(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( TotalRespirationDiatomsIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Carbon_UptakeDiatomsIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Nitrogen_Uptake_DiatomsIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( TotalRespirationFlagellatesIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Carbon_UptakeFlagellatesIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Nitrogen_Uptake_FlagellatesIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( TotalRespirationEmilianaIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Carbon_UptakeEmilianaIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Nitrogen_Uptake_EmilianaIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( Silicate_upDiatomsIntegrated(jpi,jpj), stat=istat) ;  alloc_ok=alloc_ok + istat
  allocate(    bacteria_anoxremIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(    Oxidation_by_nitrateIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(    Oxidation_by_oxygenIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(    bacteria_oxygenconsumptionIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( DOCSLHYDR_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( DONSLHYDR_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(POCHYDR_output(jpi,jpj,jpk),  stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(PONHYDR_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(BAC_GROWTH(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(DONLuptake_BAC(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(C_BACMort_output(jpi,jpj,jpk),  stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(N_BACMort_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#endif
#ifdef biodiagtrophic
  allocate(PHY_TO_ZOO(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(BAC_TO_ZOO(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(POC_TO_ZOO(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(PON_To_ZOO(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(PON_To_ZOOIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(PHY_TO_ZOOIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(BAC_TO_ZOOIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(POC_TO_ZOOIntegrated(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(ZOO_TO_POC(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(ZOO_TO_PON(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(DOC_leakage_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(DON_leakage_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(DOC_extra_excr_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(C_ZooMessyfeeding_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate(N_ZOOMessyfeeding_output(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat
#endif


#ifdef agg_diag
  allocate( shearrate3D(jpi,jpj,jpk),stat=istat ) ; alloc_ok=alloc_ok + istat
  shearrate3D=0.0_wp
#endif
  
  !allocate(LightAbsFrac(jpi,jpj,jpk), stat=istat) ; alloc_ok=alloc_ok + istat

  allocate( botcon(jpi,jpj,jp_my_trc), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( FluxBottotN(jpi,jpj),FluxBottotC(jpi,jpj),FluxBottotS(jpi,jpj),FluxBotodu(jpi,jpj),fluxsurfdox(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( FluxBotnhs(jpi,jpj) ,Fluxbotnos(jpi,jpj) ,FluxBotsio(jpi,jpj), FluxBotdic(jpi,jpj),FluxBotdox(jpi,jpj),  stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( FluxBotPHO(jpi,jpj),  stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( FluxBotcdi(jpi,jpj),fluxbotndi(jpi,jpj),  stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( FluxBotpon(jpi,jpj),fluxbotpoc(jpi,jpj),fluxbotsid(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
#ifdef aggregation
  allocate( FluxBotagg(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
#endif
  allocate(Cmin(jpi,jpj),Nmin(jpi,jpj),Smin(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  
#ifdef sedparamdiag
  allocate( pnit2D(jpi,jpj) ,pdenit2D(jpi,jpj),panox2D(jpi,jpj),    stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( burialC(jpi,jpj),burialS(jpi,jpj) ,denitinsed(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  allocate( fluxbotpoc_resusp(jpi,jpj),fluxbotpoc_depo(jpi,jpj), fluxbotpon_resusp(jpi,jpj), fluxbotpon_depo(jpi,jpj),fluxbotsid_resusp(jpi,jpj), fluxbotsid_depo(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  pnit2d=0.0_wp
  pdenit2d=0.0_wp
  panox2d=0.0_wp
  burialC=0.0_wp
  burialS=0.0_wp
  denitinsed=0.0_wp
#endif
  allocate( tauxb(jpi,jpj),tauyb(jpi,jpj),botstress(jpi,jpj),c_db(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  tauxb=0.0_wp
  tauyb=0.0_wp
  botstress=0.0_wp
  allocate( resuspcounter_f(jpi,jpj),resuspcounter_s(jpi,jpj), stat=istat) ; alloc_ok=alloc_ok + istat
  resuspcounter_f=0.0_wp
  resuspcounter_s=0.0_wp
#ifdef botstressdiag
  allocate( curstress(jpi,jpj),stat=istat) ; alloc_ok=alloc_ok + istat
  curstress=0.0_wp
#endif
  allocate( WPOM(jpi,jpj,jpk),WDIA(jpi,jpj,jpk),stat=istat) ; alloc_ok=alloc_ok + istat

#ifdef aggregation
  ALLOCATE(WAGG(jpi,jpj,jpk), zun(jpi,jpj,jpk),zvn(jpi,jpj,jpk))
#endif
  
  chlorophyll=0.0_wp
#ifdef chlorophyll_geider
  CHL_Geider=0.0_wp
  clipchl=0.0_wp
  RangeChlNrPHY=MaxChlNrPHY-MinChlNrPHY
#endif
  MaxSiCrDiatoms=MaxNCrDiatoms*SiNrDiatoms  ! computed once and for all
  MinSiCrDiatoms=MinNCrDiatoms*SiNrDiatoms
  clipC=0. ; clipN=0.; clipSi=0.
  par=0.
  AirSeaOxygenFlux=0.

#ifdef testcons
  consC=0. ; consN=0. ; consSi=0. ; consOxy=0.
#endif
#ifdef testcons2
  budgetC=0. ; budgetN=0. ; budgetSi=0.
#endif

#ifdef alkalinity_system
  call alka_init(istat)  ! allocate variables of the alka module ; if using R, also initialize pH and pCO2_air
  alloc_ok=alloc_ok+istat
#endif

  !do jn=jp_myt0,jp_myt1
  !  DO_3D(0,0,0,0,1,jpk)
  !    if (tmask(ji,jj,jk).eq.1) then
  !       if (tr(ji,jj,jk,jn,Kbb).lt.small_concentration) tr(ji,jj,jk,jn,Kbb)=small_concentration  ! NOTE: to be avoided by having correct initial condition files
  !       if (tr(ji,jj,jk,jn,Kmm).lt.small_concentration) tr(ji,jj,jk,jn,Kmm)=small_concentration
  !    end if
  !  END_3D
  !end do
  
end subroutine bioinit



subroutine benthic_init (Kbb, Kmm)
  implicit none
  integer :: Kbb, Kmm
  integer :: istat
  integer :: ji,jj, ikt
  integer :: benthiccounter
 
  !benthic vars
  fCSED=1
  sCSED=2
  sSSED=3
  fSSED=4
  NCrSED=5
  benthiccounter=5

!first total number check
IF (jp_ben /= benthiccounter) THEN
    WRITE(ctmp1,*) 'BAMHBI BENTHIC TRACER CHECK: Wrong number of tracers - Namelist jp_ben = ', jp_ben, 'Model configuration (bamhbi.h90) requires = ', benthiccounter
    CALL ctl_stop(ctmp1)
ENDIF


  DO_2D(0,0,0,0)
        if ( tmask(ji,jj,1) .eq. 0 ) then
           bt(ji,jj,:,fCSED,Kbb)  = 0.
           bt(ji,jj,:,sCSED,Kbb)  = 0.
           bt(ji,jj,:,sSSED,Kbb)  = 0.
           bt(ji,jj,:,sCSED,Kbb)  = 0.
           bt(ji,jj,:,NCrSED,Kbb) = 0.
        endif
  END_2D

#ifdef testcons2
  allocate(ben_budgetC(jpi,jpj),ben_budgetN(jpi,jpj),ben_budgetSi(jpi,jpj),PB_budget_C,PB_budget_N,PB_budget_Si,stat=istat)
  if (istat/=0) call ctl_stop( 'CRITICAL: benthic_init: unable to allocate benthic variables' )
#endif

  tauxb=0. ; tauyb=0.
  if (cur_botstr.eq.0) then
    DO_2D(0,0,0,0)
        ikt=mbkt(ji,jj)
        if (tmask(ji,jj,ikt)==1) then
           c_db(ji,jj)=((1+log(2.083E-5/e3t(ji,jj,ikt,Kmm)))/0.396)**(-2) ! following Stanev rather than Capet code ?
           !c_db(ji,jj)=max(min(dz(mbathy(ji,jj)),30.),0.2) ! ARBITRAIRE ?
        end if
    END_2D
  end if
  
end subroutine benthic_init









!**********************************************************************
subroutine pre_bio ( kt , kbb, kmm, krhs )
  implicit none
  integer, intent(in) :: kt, kbb, kmm, krhs
  integer      :: ji,jj,jk,l
  integer*4    :: nan_count,correction_count
  real(wp)     :: local_correction
  logical      :: isnan
  real(wp)     :: tempClipC, tempClipN, tempclipSi, tempclipChl
  real(wp)     :: tempooo

  ! set time step size (Euler/Leapfrog) from trcstp
  dt = rDt_trc

  ! NANCOUNT and CLIPPING
  do l=jp_myt0,jp_myt1
     nan_count=0

     DO_3D(0,0,0,0,1,jpkm1)
              if (tmask(ji,jj,jk).eq.1) then

                 if (isnan(tr(ji,jj,jk,l,Kbb))) then
                    nan_count=nan_count+1
                    tr(ji,jj,jk,l,Kbb)=small_concentration
#ifdef clipping
                 elseif (tr(ji,jj,jk,l,Kbb).LT.small_concentration) then
                    local_correction=abs(tr(ji,jj,jk,l,Kbb)-small_concentration)*cvol(ji,jj,jk)  ! for inventory cvol should be used rather than e3t !

                    if ((l.eq.CEM).or.(l.eq.CFL).or.(l.eq.CDI).or.(l.eq.BAC).or.(l.eq.DCL).or.(l.eq.DCS).or.(l.eq.MES).or.(l.eq.MIC).or.(l.eq.DIC) &
#ifdef gelatinous
                         .or.(l.eq.NOC).or.(l.eq.GEL) &
#endif
                         ) then
                       ! going to write in Carbon clip
                       clipC(ji,jj,jk)=clipC(ji,jj,jk)+local_correction   !abs(trb(ji,jj,jk,l)-small_concentration)*e3t_b(ji,jj,jk)
                    elseif ((l.eq.NEM).or.(l.eq.NFL).or.(l.eq.NDI).or.(l.eq.DNL).or.(l.eq.DNS).or.(l.eq.NOS).or.(l.eq.NHS)) then
                       ! going to write in nitrogen clip
                       clipN(ji,jj,jk)=clipN(ji,jj,jk)+local_correction   !abs(trb(ji,jj,jk,l)-small_concentration)*e3t_b(ji,jj,jk)
                    elseif ((l.eq.SID).or.(l.eq.SIO)) then
                       ! going to write in silicious clip
                       clipSi(ji,jj,jk)=clipSi(ji,jj,jk)+local_correction   !abs(trb(ji,jj,jk,l)-small_concentration)*e3t_b(ji,jj,jk)
#ifdef chlorophyll_geider
                    elseif ((l.eq.CHD).or.(l.eq.CHE).or.(l.eq.CHF)) then
                       clipChl(ji,jj,jk)=clipChl(ji,jj,jk)+local_correction
#endif
                    endif
                    tr(ji,jj,jk,l,Kbb)=small_concentration
#endif
                 endif
              endif
     END_3D

     if (nan_count.ne.0) write(numout,*) 'NaNs',kt,l,nan_count
  end do

#ifdef clipping
  tempclipC=glob_sum('pre_bio',clipC) ; tempclipN=glob_sum('pre_bio',clipN) ; tempclipSi=glob_sum('pre_bio',clipSi)
#ifdef chlorophyll_geider
  tempclipChl=glob_sum('pre_bio',clipchl
  if( lwp )  write(numout,*) 'clips :',tempClipC, tempClipN, tempClipSi, tempClipChl
#else
  if( lwp )  write(numout,*) 'clips :',tempClipC, tempClipN, tempClipSi
#endif
#ifdef temp_clip
  clipC=0.0_wp ; clipN=0.0_wp ; clipSi=0.0_wp ; clipChl=0.0_wp  ! at next time step, clip budget starts with 0.0 as we didn't actually really clip anything
#endif
#endif

  where (chlorophyll<0.) chlorophyll=0.
  call UpdateLight(kt,kbb,kmm) !  will use chlorophyll or chl_geider

  ! TEST CONSERVATION  
  DO_3D(0,0,0,0,1,jpk)   ! we should loop without the halo here !
        if (tmask(ji,jj,jk).eq.1) then
           Chlorophyll(ji,jj,jk,:)=0._wp
#ifdef chlorophyll_geider
           chl_geider(ji,jj,jk)=0.0_wp
#endif
#ifdef biodiagrates
              Totalrespiration_ZOO=0._wp
#endif
              !Test biological model conservativity
#ifdef testcons
              CONSC(ji,jj,jk) =(dPCFL(ji,jj,jk) + dPCDI(ji,jj,jk) + dPCEM(ji,jj,jk) + dPMIC(ji,jj,jk) + dPMES(ji,jj,jk) + dPBAC(ji,jj,jk) &
                           + dPPOC(ji,jj,jk) + dPDCL(ji,jj,jk) + dPDCS(ji,jj,jk) + dPDIC(ji,jj,jk) &
                           - dDCFL(ji,jj,jk) - dDCDI(ji,jj,jk) - dDCEM(ji,jj,jk) - dDMIC(ji,jj,jk) - dDMES(ji,jj,jk) - dDBAC(ji,jj,jk) &
                           - dDPOC(ji,jj,jk) - dDDCL(ji,jj,jk) - dDDCS(ji,jj,jk) - dDDIC(ji,jj,jk) &
#ifdef gelatinous
                           + dPGEL(ji,jj,jk) + dPNOC(ji,jj,jk) - dDGEL(ji,jj,jk) - dDNOC(ji,jj,jk) &
#endif
                            )
              CONSN(ji,jj,jk) = dPNFL(ji,jj,jk) + dPNDI(ji,jj,jk) + dPNEM(ji,jj,jk)+ &
                   NCrMicroZoo*dPMIC(ji,jj,jk)+ NCrMesoZoo*dPMES(ji,jj,jk) + NCrBAC*dPBAC(ji,jj,jk) + dPPON(ji,jj,jk)+ dPDNL(ji,jj,jk)+ &
                   dPDNS(ji,jj,jk) + dPNOS(ji,jj,jk)+ dPNHS(ji,jj,jk) &
                   - dDNFL(ji,jj,jk) - dDNDI(ji,jj,jk) - dDNEM(ji,jj,jk)- NCrMicroZoo*dDMIC(ji,jj,jk)- &
                   NCrMesoZoo*dDMES(ji,jj,jk) - NCrBAC*dDBAC(ji,jj,jk) - dDPON(ji,jj,jk) - dDDNL(ji,jj,jk) - dDDNS(ji,jj,jk) &
                   -       dDNOS(ji,jj,jk) - dDNHS(ji,jj,jk) &
#ifdef gelatinous
                   +  NCrGelatinous*dPGEL(ji,jj,jk)+NCrNoctiluca*dPNOC(ji,jj,jk) -  NCrGelatinous*dDgel(ji,jj,jk) - NCrNoctiluca*dDNOC(ji,jj,jk)
#endif

              CONSSi(ji,jj,jk)=SiNrDiatoms*dPNDI(ji,jj,jk) + dPSIO(ji,jj,jk)+ dPSID(ji,jj,jk) &
                   - SiNrDiatoms*dDNDI(ji,jj,jk) - dDSIO(ji,jj,jk) -   dDSID(ji,jj,jk)
              CONSOXY(ji,jj,jk)= dPDOX(ji,jj,jk)-dDDOX(ji,jj,jk)
              CONSPHO(ji,jj,jk)=dPPHO(ji,jj,jk)-dDPHO(ji,jj,jk)+Pnredfield*(dPNFL(ji,jj,jk) + dPNDI(ji,jj,jk) + dPNEM(ji,jj,jk)+ &
                   NCrMicroZoo*dPMIC(ji,jj,jk)+ NCrMesoZoo*dPMES(ji,jj,jk) + NCrBAC*dPBAC(ji,jj,jk) + dPPON(ji,jj,jk)+ dPDNL(ji,jj,jk)+ &
                   dPDNS(ji,jj,jk) &
                   -dDNFL(ji,jj,jk) - dDNDI(ji,jj,jk) - dDNEM(ji,jj,jk) -&
                   NCrMicroZoo*dDMIC(ji,jj,jk)- NCrMesoZoo*dDMES(ji,jj,jk) - NCrBAC*dDBAC(ji,jj,jk) - dDPON(ji,jj,jk) - dDDNL(ji,jj,jk)-&
                   dDDNS(ji,jj,jk) &
#ifdef gelatinous
                   +  NCrGelatinous*dPGEL(ji,jj,jk)+NCrNoctiluca*dPNOC(ji,jj,jk) -  NCrGelatinous*dDgel(ji,jj,jk) - NCrNoctiluca*dDNOC(ji,jj,jk) &
#endif
              )
#endif
           endif
  END_3D
  
#ifdef testcons2
  ! cvol is defined in module TRC, cvol = tmask * e1 * e2 * e3
  BudgetC= cvol * (tr(:,:,:,CFL,Kbb)+tr(:,:,:,CDI,Kbb)+tr(:,:,:,CEM,Kbb)+tr(:,:,:,MES,Kbb)+tr(:,:,:,MIC,Kbb)+tr(:,:,:,BAC,Kbb) &
                      +tr(:,:,:,POC,Kbb)+tr(:,:,:,DCL,Kbb)+tr(:,:,:,DCS,Kbb)+tr(:,:,:,DIC,Kbb) &
#ifdef gelatinous
                      +tr(:,:,:,NOC,Kbb)+tr(:,:,:,GEL,Kbb) &
#endif
                  )
  BudgetN= cvol * (tr(:,:,:,NFL,Kbb)+tr(:,:,:,NDI,Kbb)+tr(:,:,:,NEM,Kbb)+ NCrMesoZoo*tr(:,:,:,MES,Kbb) + NCrMicroZoo*tr(:,:,:,MIC,Kbb) + &
                   NCrBAC*tr(:,:,:,BAC,Kbb) + tr(:,:,:,PON,Kbb)+tr(:,:,:,DNL,Kbb)+tr(:,:,:,DNS,Kbb)+tr(:,:,:,NOS,Kbb)+tr(:,:,:,NHS,Kbb) &
#ifdef gelatinous
                   + NCrNoctiluca*tr(:,:,:,NOC,Kbb) + NCrGelatinous*tr(:,:,:,GEL,Kbb) &
#endif
                  )
  BudgetSi= cvol *  (SiNrDiatoms*tr(:,:,:,NDI,Kbb) + tr(:,:,:,SIO,Kbb) + tr(:,:,:,SID,Kbb))

  integrated_budgetC=glob_sum('prebio_testcons2',budgetC)
  integrated_budgetN=glob_sum('prebio_testcons2',budgetN)
  integrated_budgetSi=glob_sum('prebio_testcons2',budgetSi)

#endif

  ! remise a zero
  dPCFL=0._wp ; dPNFL=0._wp ; dPCEM=0._wp ; dPNEM=0._wp ; dPCDI=0._wp ; dPNDI=0._wp ; dPMIC=0._wp ; dPMES=0._wp ; dPBAC=0._wp ; dPDCL=0._wp ; dPDNL=0._wp ; dPDCS=0._wp ; dPDNS=0._wp ;
  dPNOS=0._wp ; dPNHS=0._wp ; dPSIO=0._wp ; dPDOX=0._wp ; dPDIC=0._wp ; dPODU=0._wp ; dPPOC=0._wp ; dPPON=0._wp ; dPSID=0._wp ; dPGEL=0._wp ; dPNOC=0._wp ; dPPHO=0._wp ;
  dDCFL=0._wp ; dDNFL=0._wp ; dDCEM=0._wp ; dDNEM=0._wp ; dDCDI=0._wp ; dDNDI=0._wp ; dDMIC=0._wp ; dDMES=0._wp ; dDBAC=0._wp ; dDDCL=0._wp ; dDDNL=0._wp ; dDDCS=0._wp ; dDDNS=0._wp ;
  dDNOS=0._wp ; dDNHS=0._wp ; dDSIO=0._wp ; dDDOX=0._wp ; dDDIC=0._wp ; dDODU=0._wp ; dDPOC=0._wp ; dDPON=0._wp ; dDSID=0._wp ; dDGEL=0._wp ; dDNOC=0._wp ; dDPHO=0._wp ;
#ifdef aggregation
  dPAGG=0._wp ; dDAGG=0._wp
#endif
#ifdef chlorophyll_geider
  dPCHD=0._wp ; dDCHD=0._wp ;  dPCHE=0._wp ; dDCHE=0._wp ;  dPCHF=0._wp ; dDCHF=0._wp
#endif
#ifdef alkalinity_system
  dPCHA=0._wp ; dDCHA=0._wp
#endif

  AirSeaOxygenFlux=0._wp
#ifdef alkalinity_system
  AirseaDICFlux=0._wp
#endif  

#ifdef benthic
  ! remise a zero benthic trend
  fluxbotcdi=0._wp ; fluxbotndi=0._wp ; fluxbotnos=0._wp ; fluxbotnhs=0._wp ; fluxbotsio=0._wp ; fluxbotdox=0._wp ; fluxbotdic=0._wp
  fluxbotodu=0._wp ; fluxbotpoc=0._wp ; fluxbotpon=0._wp ; fluxbotsid=0._wp ; fluxbotpho=0._wp
#ifdef aggregation
  fluxbotagg=0._wp
#endif
#endif  
 
#ifdef biodiagrates
Denitrification = 0._wp ; DOCLuptake_BAC = 0._wp ; Bacteria_respiration = 0._wp ; Totalrespiration_ZOO = 0._wp ;
TotalRespiration_Gel = 0._wp ; PHYTONITRATEREDUCTION = 0._wp ; Nitrification = 0._wp ; ANAMMOX = 0._wp ; Denitrificationintegrated = 0.
DOCLuptake_BACIntegrated = 0._wp ; Bacteria_respirationIntegrated = 0._wp ; TotalRespiration_ZOOIntegrated = 0._wp ; TotalRespiration_GelIntegrated = 0._wp ; PHYTONITRATEREDUCTIONIntegrated = 0.
NitrificationIntegrated = 0._wp ; ANAMMOXIntegrated = 0._wp ; C_Phymort_output = 0._wp ; N_Phymort_output = 0._wp ; NHSuptake_BAC = 0.
BACExcretion = 0._wp ; DOXprod_PHY_nutuptake = 0._wp ; DOXprod_PHY_growth = 0._wp ; SID_dissolution = 0._wp ; SIO_uptake_diatoms = 0.
SID_creation = 0._wp ; PHY_NHS_excr = 0._wp ; PHY_N_uptake = 0._wp ; zoo_n_excretion = 0._wp
TotalRespirationFlagellates = 0._wp; Carbon_UptakeFlagellates = 0._wp; Nitrogen_Uptake_Flagellates = 0._wp; TotalRespirationDiatoms = 0._wp; Carbon_UptakeDiatoms = 0._wp
Nitrogen_Uptake_Diatoms = 0._wp; TotalRespirationEmiliana = 0._wp; Carbon_UptakeEmiliana = 0._wp; Nitrogen_Uptake_Emiliana = 0._wp; Silicate_upDiatoms = 0._wp
Oxidation_by_nitrate = 0._wp; Oxidation_by_oxygen = 0._wp; bacteria_anoxrem = 0._wp; bacteria_oxygenconsumption = 0._wp; TotalRespirationDiatomsIntegrated = 0._wp
Carbon_UptakeDiatomsIntegrated = 0._wp; Nitrogen_Uptake_DiatomsIntegrated = 0._wp; TotalRespirationFlagellatesIntegrated = 0._wp; Carbon_UptakeFlagellatesIntegrated = 0._wp; Nitrogen_Uptake_FlagellatesIntegrated = 0._wp
TotalRespirationEmilianaIntegrated = 0._wp; Carbon_UptakeEmilianaIntegrated = 0._wp; Nitrogen_Uptake_EmilianaIntegrated = 0._wp; Silicate_upDiatomsIntegrated = 0._wp; bacteria_anoxremIntegrated = 0._wp
Oxidation_by_nitrateIntegrated = 0._wp; Oxidation_by_oxygenIntegrated = 0._wp; bacteria_oxygenconsumptionIntegrated = 0._wp; DOCSLHYDR_output = 0._wp; DONSLHYDR_output = 0._wp
POCHYDR_output = 0._wp; PONHYDR_output = 0._wp; BAC_GROWTH = 0._wp; DONLuptake_BAC = 0._wp; C_BACMort_output = 0._wp; N_BACMort_output = 0._wp
#endif

#ifdef biodiagtrophic
PHY_TO_ZOO = 0._wp; BAC_TO_ZOO = 0._wp; POC_TO_ZOO = 0._wp; PON_TO_ZOO = 0._wp; PON_TO_ZOOIntegrated = 0._wp;
PHY_TO_ZOOIntegrated = 0._wp; BAC_TO_ZOOIntegrated = 0._wp; POC_TO_ZOOIntegrated = 0._wp; ZOO_TO_POC = 0._wp; ZOO_TO_PON = 0._wp;
DOC_leakage_output = 0._wp; DON_leakage_output = 0._wp; DOC_extra_excr_output = 0._wp; C_ZooMessyfeeding_output = 0._wp; N_ZOOMessyfeeding_output = 0._wp;
#endif
 
end subroutine pre_bio







!###########################################################################################################################################################################################################################################################################
!
!                                                 P E L A G I C    M O D E L
!
!###########################################################################################################################################################################################################################################################################




#ifdef aggregation
!#########################################################################################
!                              3DAGGREG.F90
! 1-D ecosystem model - Biological model of POM
! Kriest and Evans, 2000, Proc> Indian Acad. Sci. 109, No 4, pp. 453 - 469.
! Kriest and Evans, Deep Sea research, I, 46, 1999, pp 1841 - 1859.
! Implementation: Marilaure Gregoire, NIOO-CEME
! Translation into NEMO : Luc Vandenbulcke
!######################################################################
SUBROUTINE CalculateAggregation(Kbb,Kmm)
  IMPLICIT NONE

  integer, intent(in) :: Kbb, Kmm
  real(wp)  :: &
       vsedcell,              & ! sedimentation velocity of a unique cell of size m in m/day
       vsedmax,               & ! max. sedimentation velocity in m/day
       FM,                    & ! the fraction of particles being larger than the upper bound M
       GM,                    & ! the fraction of mass of particles being larger than the upper bound M
       epsilon,               & ! size distribution coefficient
       SM,                    & ! ratio of sinking rate of large particles and one single cell
       aggrsett,              & ! number of particles colliding due to settlement
       aggrshear,             & ! number of particles colliding due to shear forces
       eshearrate,            & ! shear rate ! computed, used to be defined in the mixed layer and reduced by a factor of 100 below it
       apprdiv ,              & ! small constant to prevent a division by zero
       sizeaggregate,         & ! m averaged size of the aggregate
       Aggregationrate,       & ! number of particles/m3/day  rate of aggregation by settling and shear
       NumbOfCellsInAggregate,& ! - number of cells per aggregate
       NumbOfLargeAggregate,  & ! - number of large aggregates (size>0.5mm)
       !Aggr_Settelment ,      & ! number of particles/m3/day
       !Aggr_Shearrate,        & ! number of particles/m3/day
       GRU,GRV                  ! gradient verticaux des vitessses horizontales
  integer :: ji,jj,jk

#ifdef nanquest
  logical      :: isnan
#endif

  IF( l_offline ) THEN      ! When off-line un & vn contain transport --> recompute velocity
    DO jk = 1, jpkm1
      zun(:,:,jk) = uu(:,:,jk,Kmm) / e2u(:,:) / e3u(:,:,jk,Kmm)
      zvn(:,:,jk) = vv(:,:,jk,Kmm) / e1v(:,:) / e3v(:,:,jk,Kmm)
    END DO
  ELSE                      ! when online, copy uu,vv to zun,zvn
    zun(:,:,:)=uu(1:,1:,:,Kmm)
    zvn(:,:,:)=vv(1:,1:,:,Kmm)   ! in Nemo, the indexes go from 1:jpi, 1:jpj ; but with deSolve, they go from 0:1,0:1, so we select just 1,1
  END IF

  ! CALCULATING FLUXES
  ! NumbOfLargeAggregate=0._wp

  DO_2D(0,0,0,0)  ! row 1 is the overlap region, NEMO takes care of it
     do jk=1,jpkm1
        if (tmask(ji,jj,jk).eq.1) then
          CALL SIZE_DISTRIBUTION(tr(JI,JJ,JK,PON,Kbb),tr(JI,JJ,JK,AGG,Kbb),dzetabio,C,m,etabio,epsilon)
          ! Compute the sedimentation velocity of POM and POMNOS
          vsedcell = B*m**etabio
          vsedmax = B*Maxsize**etabio
          if (epsilon.gt.1.) then
             GM = (Maxsize/m)**(1.0 + dzetabio - epsilon)
             FM = (Maxsize/m)**(1.0 - epsilon)
          else
             FM=Maxsize/m
             GM=Maxsize/m
          endif
          SM=(Maxsize/m)**etabio

          ! Auxillary variable used to calculate the number of large aggregate (not necessary for the computation of shearrate and sinking speed)
          !          sizeaggregate=(PONI(JI,JJ,JK)/AGGI(JI,JJ,JK))/(C*(m**dzetabio))*m
          !       NumbOfCellsInAggregate=(PONI(JI,JJ,JK)/AGGI(JI,JJ,JK))/(C*(m**dzetabio))
          !          if (sizeaggregate.ge.(0.0005)) then
          !       NumbOfLargeAggregate=NumbOfLargeAggregate+1
          !          endif
          !End of computation of auxillary variable

          ! Shear
          ! All levels
          GRU=  zun(ji,jj,jk)-zun(ji,jj,jk+1)
          GRV=  zvn(ji,jj,jk)-zvn(ji,jj,jk+1)
          if (ji.gt.1) GRU=( GRU + (zun(ji-1,jj,jk)-zun(ji-1,jj,jk+1)) )/ 2.0_wp
          if (jj.gt.1) GRV=( GRV + (zvn(ji,jj-1,jk)-zvn(ji,jj-1,jk+1)) )/ 2.0_wp
          eshearrate=sqrt(GRU*GRU+GRV*GRV) / e3w(ji,jj,jk+1,Kmm)
          ! All but surface levels
          IF(jk.GT.1) THEN
             GRU=  zun(ji,jj,jk)-zun(ji,jj,jk-1)
             GRV=  zvn(ji,jj,jk)-zvn(ji,jj,jk-1)
             if (ji.gt.1) GRU= ( GRU + (zun(ji-1,jj,jk)-zun(ji-1,jj,jk-1)) )/ 2.0_wp
             if (jj.gt.1) GRV= ( GRV + (zvn(ji,jj-1,jk)-zvn(ji,jj-1,jk-1)) )/ 2.0_wp
             !Approximation; it should be a weighted average since T-points are not set half way between w-points
             eshearrate= ( eshearrate + sqrt(GRU*GRU+GRV*GRV)/ e3w(ji,jj,jk,Kmm) )/2.
          ENDIF

#ifdef nanquest
          if (isnan(eshearrate)) then
            write (*,*) '** NAN QUEST ** in Calcshear',ji,jj,jk ; call flush(6) ; stop
          endif
#endif
#ifdef agg_diag
          shearrate3D(ji,jj,jk)=eshearrate
#endif
          
          CALL AGGREGATION(tr(JI,JJ,JK,AGG,Kbb),eshearrate,Maxsize,m,stick,etabio,epsilon,FM,GM,SM,vsedcell, &
               aggrsett,aggrshear,aggregationrate)
          dDAGG(JI,JJ,JK) = dDAGG(JI,JJ,JK) + aggregationrate

#ifdef nanquest
          if (isnan(dDAGG(JI,JJ,JK))) then
            write (*,*) '** NAN QUEST ** in CalcAGG',ji,jj,jk ;  call flush(6) ; stop
          endif
#endif

        end if
     end do
  END_2D

END SUBROUTINE       CalculateAggregation
#endif


!#########################################################################################
!                              3DBACTERIA.F90
!
! 1-D ecosystem model - Biological model of bacteria
!
! Anderson, 1992,  Modelling the influence of food C:N ratio, and respiration on growth
! and nitrogen excretion in marine zooplankton and bacteria,
! Journal of Plankton Research, vol. 14, n 12, pp. 1645-1671, 1992
!
! Anderson and Williams,Modelling the Seasonal
! Cycle of Dissolved Organic Carbonj at Station E1 in the English Channel, Estuarine, Coastal
! and Shelf Science 1998, 46, 93-109.
!
! Anderson amd Pondhaven (2003),Non-redfield carbon and nitrogen cycling in the Sarasso Sea :
! pelagic imbalances and export flux, in press in DSR
!
! Implementation: Marilaure Gregoire,       NIOO-CEME
! Translation into NEMO : Luc Vandenbulcke                                                                                                                                                                                                                                    !--------------------------------------------------------------------
! Contains the pelagic submodel, for bacteria described in Anderson (1992),JPR and also
! in Anderson and Ponhaven (2003), DSR I. This model is a nitrogen-carbon balanced model.
!It is assumed that the organic matter is composed of nitrogenous (proteins, amino acids)
! and non nitrogenous compounds (carbohydrate, lipids). Fixed C/N ratio is aasigned to bacteria
! The cycling of C and N by bacteria is described by elemental stoichiometry (Anderson,1992, Anderson and Williams (1998).
! Bacteria act as either remineralizers or consumers of ammonium, depending on the relative imbalance
! in the C/N ratios of their biomass relative to the DOM they consume, mediated by the C gross growth efficiency of utilization
! The hypothesis is that bacteria preferentially use nitrogeneous compounds for growth and carbon compounds for
! respiration (nore energy in non nitrogenous substrats). Bacteria growth, excretion and respiration are calculated
! from elemental stoichiometry. This method assumes that labile DOC amd DON are the primary growth substrates with ammonium supplementing
! DOM when the C/N ratio of DOM is high.
!######################################################################
SUBROUTINE CalculateBacteria(Kbb, Kmm)
  IMPLICIT NONE

  integer, intent(in) :: Kbb, Kmm
  real(wp)  :: &
    tf,                              &  ! Temperature function
    C_BACMort,N_BACMort,   &  ! bacteria mortality,  mmol/m3/day
    BACResp,BACExcr,       &  ! Zooplankton respiration, mmol/m3/day and excretion
    BACGrowth,denitrif,bacteria_oxygenconsumption_local, bacteria_anoxrem_local,Uptake_DOCL_local, &                !Bacterial growth rate
    Uptake_DONL_local,NHSuptake_BAC_local
  real(wp) :: Iron, FeMnLIM
  INTEGER  :: ji,jj,jk, jk_addpho
  real(wp) :: prevLim, PHO_precipitated, PHO_precipitated_sum
  logical  :: LimIncreasing, LimDecreasing
  ! calculate iron limitation depending only on depth (as depth can vary in time)  -- neglect the z partial step problem in order to avoir to compute this in 3D
  do jk=1,jpk
     Iron=IronCsurf+Param1IronCurve/(Param2IronCurve*sqrt(2.*3.14159265359))*exp(-(gdept_1d(jk)-275.0_wp)**2/(2.*Param2IronCurve**2))  ! iron concentration
     IronLimitation(jk) = Iron/(Iron+Halfsaturation_Iron)
  end do
  
  ! CALCULATE FLUXES
  DO_2D(0,0,0,0)
     do jk=1,jpkm1
        if (tmask(ji,jj,jk).eq.1) then
          !TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor(ts(ji,jj,jk,jp_tem,Kmm),Q10bac)

          ! BACTERIA
          ! Growth rate of bacteria (BACGROWTH, in mmolC/m3/day)
          CALL BAC_GROWTH_RATE(maxgrowthbac,tf,csatdocl,csatamm,bactgrowtheff,NCrBac,&
            tr(ji,jj,jk,pho,Kbb),csatpo4,      &
            tr(ji,jj,jk,bac,Kbb),              &
            Uptake_DOCL_local,Uptake_DONL_local,NHSuptake_BAC_local,&
            BACGrowth,BACResp,BACExcr,      &
            tr(ji,jj,jk,dcl,Kbb),tr(ji,jj,jk,dnl,Kbb),tr(ji,jj,jk,nhs,Kbb))

          ! Bacteria mortality rate (C_BACMort,N_BACMort, /day);  attention: do not forget to add OXYGEN
          C_BACMort = mortbac*tf*tr(ji,jj,jk,bac,Kbb)
          ! Mortality in nitrogen units
          N_BACMort  = mortbac*tf*tr(ji,jj,jk,bac,Kbb)*NCrBac

          ! ADJUSTING THE RATE OF CHANGE
          ! Bacteria C increases by intake of DOCl
          ! it decreases by mortality,predation (see zooplankton.f)
          dPBAC(ji,jj,jk) =dPBAC(ji,jj,jk) + Bacgrowth
          dDBAC(ji,jj,jk) =dDBAC(ji,jj,jk) + C_BACMort

          ! Ammonium is excreyed by bacteria and can be taken up by bacteria
          dPNHS(ji,jj,jk) = dPNHS(ji,jj,jk) + BACExcr
          dDNHS(ji,jj,jk) = dDNHS(ji,jj,jk) + NHSuptake_BAC_local

          ! phosphore    ! M.G.-L.V. 28/01/2024: replaced the original line by the one below, to emulate PHO precipitation by iron-3 (e.g. Dijkstra et al, 2014) and Mn (Yakushev et al)
          dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk) + BACExcr*PNRedfield
          !dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk) + BACExcr*PNRedfield * (1.0_wp - FeMnLIM * ODU_solid)
          ! this trick removes a part of the PHO produced by bacteries, but does not impact the PHO that's upwelled by internal waves
          dDPHO(ji,jj,jk) = dDPHO(ji,jj,jk) + NHSuptake_BAC_local*PNRedfield

          ! alternatively (but less realistic?) Kostas removes 25% PHO per day in the surface layer (priv. comm.)
          ! if (jk.lt.10) dDPHO(ji,jj,jk) = dDPHO(ji,jj,jk) + 0.25/86400*tr(ji,jj,jk,pho,kbb)
          ! Now, let's remove phosphate at the Iron-Mangase depth (or better density), not dependend on bacteria
          if (jk.eq.1) then ; PrevLim=10.0e-10_wp ; Limincreasing=.false. ; Limdecreasing=.false. ; Pho_precipitated_sum=0.0_wp ; jk_addpho=0 ; endif
          FeMnLim = limitation_by_ironmanganese(gdept(ji,jj,jk,Kmm),rhop(ji,jj,jk))
          if (FeMnLim.gt.prevlim) LimIncreasing=.true.
          if (FeMnLim.lt.prevlim.and.LimIncreasing) LimDecreasing=.true.
          if (FeMnLim.lt.1.0e-8.and.LimDecreasing) then ; jk_addpho=jk+1 ; LimIncreasing=.false. ; LimDecreasing=.false. ; endif
          prevLim=FeMnLim
          PHO_precipitated= FeMnLIM * Pho_precip_fac * tr(ji,jj,jk,pho,kbb)  ! this contributes only around sigma=16.1, where FeMnLim is not zero
          PHO_precipitated_sum = PHO_precipitated_sum + PHO_precipitated * e3t(ji,jj,jk,Kmm)
          dDPHO(ji,jj,jk) = dDPHO(ji,jj,jk) + PHO_precipitated
          ! and now add back this phosphate below the phosphatecline. At this depth, avt is really small, so it will take time to diffuse vertically...
          !! if (jk.eq.jk_addpho) dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk) + PHO_precipitated_sum / e3t(ji,jj,jk,Kmm) !! re-adding creates a PHO peak !

          ! as a result of respiration. It increases as a result of mortality of bacteria which is fractionned beween the labile and semi labile pool
          ! with a coefficient labilefractionDON
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + labilefraction*C_BACMort
          dDDCL(ji,jj,jk) = dDDCL(ji,jj,jk) + BACGrowth + BACResp
          dPDNL(ji,jj,jk) = dPDNL(ji,jj,jk) + labilefraction*N_BACMort
          dDDNL(ji,jj,jk) = dDDNL(ji,jj,jk) + Uptake_DONL_local

          dPDCs(ji,jj,jk) = dPDCs(ji,jj,jk) + (1.0 - labilefraction)*C_BACMort
          dPDNs(ji,jj,jk) = dPDNs(ji,jj,jk) + (1.0 - labilefraction)*N_BACMort

          denitrif = BACResp*tr(ji,jj,jk,nos,Kbb)/(tr(ji,jj,jk,nos,Kbb)+ksdeninos)*(kindenidox/(tr(ji,jj,jk,dox,Kbb)+kindenidox))*NCr
#ifdef testcons
          denitrif=0
#endif
          
          bacteria_oxygenconsumption_local=BACResp*tr(ji,jj,jk,dox,Kbb)/(tr(ji,jj,jk,dox,Kbb)+ksremindox) * OCr
          bacteria_anoxrem_local= BACResp*(kinanoxremnos/(tr(ji,jj,jk,nos,Kbb)+kinanoxremnos))* (kinanoxremdox/(tr(ji,jj,jk,dox,Kbb)+kinanoxremdox))*ODUCr

          ! Oxygen decreases due to bacterial respiration
          dDDOX(ji,jj,jk)=dDDOX(ji,jj,jk) + bacteria_oxygenconsumption_local

          !NOs decreases due to bacterial respiration
          dDNOS(ji,jj,jk)=dDNOS(ji,jj,jk) + denitrif
          dPDIC(ji,jj,jk)=dPDIC(ji,jj,jk) + BACResp

          !ODU increases due to bacterial anoxic remineralisation
          dPODU(ji,jj,jk)=dPODU(ji,jj,jk) + bacteria_anoxrem_local*(1. - IronLimitation(jk) * ODU_solid)

          ! CO2 production and consumption
          ! dDIC=dDIC+BACResp
          ! CALL UpdateCO2(I, - GrowthPHY)

          !Diagnostics
#ifdef biodiagrates
          DOCLuptake_BAC(ji,jj,jk)=Uptake_DOCL_local
          Bacteria_Respiration(ji,jj,jk)=BACResp
          denitrification(ji,jj,jk)=denitrif
          NHSuptake_BAC(ji,jj,jk)= NHSuptake_BAC_local
	  BACExcretion(ji,jj,jk) = BACExcr
          bacteria_anoxrem(ji,jj,jk)=bacteria_anoxrem_local
          bacteria_oxygenconsumption(ji,jj,jk)= bacteria_oxygenconsumption_local
          BAC_GROWTH(ji,jj,jk)= BACGrowth
          DONLuptake_BAC(ji,jj,jk)= Uptake_DONL_local
          C_BACMort_output(ji,jj,jk)= C_BACMort
	  N_BACMort_output(ji,jj,jk) = N_BACMort
#endif

        endif
     end do
  END_2D



#ifdef biodiagrates
  call IntegrateVertical(DOCLuptake_BAC,DOCLuptake_BACIntegrated,kmm)
  call IntegrateVertical(Bacteria_respiration,Bacteria_respirationIntegrated,kmm)
  call IntegrateVertical(denitrification,DenitrificationIntegrated,kmm )
  call IntegrateVertical(bacteria_oxygenconsumption,bacteria_oxygenconsumptionIntegrated,kmm)
  call IntegrateVertical(bacteria_anoxrem,bacteria_anoxremIntegrated,kmm)
#endif

END SUBROUTINE CalculateBacteria



!#########################################################################################
!                              3DSinkingF90
!
! 1-D ecosystem model - Biological model of POM
!
! Kriest and Evans, 2000, Proc> Indian Acad. Sci. 109, No 4, pp. 453 - 469.
! Kriest and Evans, Deep Sea research, I, 46, 1999, pp 1841 - 1859.
!
! Implementation: Marilaure Gregoire,           NIOO-CEME
! Translation into NEMO: L. Vandenbulcke,   ULg / MAST
!
!######################################################################
SUBROUTINE CalculateSinking(PartNitro,Aggregate,SinkingPOM,SinkingAGG)
  IMPLICIT NONE
  real(wp), INTENT(IN) ::      PartNitro,Aggregate
  real(wp)  :: &
       vsedcell, & ! sedimentation velocity of a unique cell of size m in m/sec
       vsedmax, &  ! max. sedimentation velocity in m/sec
       FM,      &  ! the fraction of particles being larger than the upper bound M
       GM,      &  ! the fraction of mass of particles being larger than the upper bound M
       epsilon, &  ! size distribution coefficient
       SM          ! ratio of sinking rate of large particles and one single cell
  real(wp), INTENT(OUT) ::   SinkingPOM,SinkingAGG

  CALL SIZE_DISTRIBUTION(PartNitro,Aggregate,dzetabio,C,m,etabio,epsilon) ! computes epsilon based on PON and AGG

  ! Compute the sedimentation velocity of POM and POMNOS
  vsedcell = B*m**etabio
  vsedmax = B*Maxsize**etabio

  If (epsilon.gt.1.) THEN
     FM = (Maxsize/m)**(1.0 - epsilon)
     SinkingAGG = ((epsilon - 1.0)*vsedcell-FM*etabio*vsedmax)/(epsilon - etabio - 1.0)
     GM = (Maxsize/m)**(1.0 + dzetabio - epsilon)
     SinkingPOM= ((epsilon - dzetabio - 1.0)*vsedcell - GM*etabio*vsedmax)/(epsilon - dzetabio - etabio - 1.0)
  ELSE
     FM=Maxsize/m
     SinkingAGG=vsedcell
     SinkingPOM=vsedcell
  ENDIF
  
END SUBROUTINE CalculateSinking



!#########################################################################################
!                              3Dchemicals.F90
!
! 1-D ecosystem model - Chemical Processus
!
! Chemical equations of the suboxic and anoxic layers. All the reduced substances of the anoxic layer (Fe2+,Mn2+
! HS-) are gathered in a state variable called ODU (Oxygen demand units) as in Soetaert et al., 1996.
! Oxydants are considered to be not limiting. It is assumed that everywhere in the water columm,we will have an oxidant of particulate prganic matter
! (DOX,NO3,iron oxide, manganese oxyde,sulfate). This reaction produces ODU and ODU diffuses upwards to be oxydized by either DOX or NOs
! In reality, ODU can be also oxidyzed by MNO2 or FEOOH but these reactions are very fast and produce FE2+ and Mn2+ which are then oxydized by
! nitrate. The rate of the first eraction 9oxydation by mno2 or feooh is very high (about one order of magnitude higher than thge oxidation by nitrate and oxygen
! and we can consider that it is almost instantaneous and short circuiting this reaction. Alos, the oxidation of ODU will be considered as a direct
! oxidation by NOs and oxygen producing oxydant that are modelled. Heterotrophic bacteria are assumed to respire either oxygen,nitrate or ODU.
! according to environmental chemical conditions. The respiration of oxygen is computed by multiplying the
! respiration rate  of carbon by an oxygen li;itation function (Monod), the respiration of nitrate is computed
! by multiplying the respiration of carbon bu a nitrate limit funct (Monod) and a inhibition funvtion of
! oxygen concentration (inverse monod), the respiration of other oxydant is not computed since these oxydants are not
! modelled but the production of ODU is computed.
! Ammonium can be oxidized by oxygen producing nitrate : this is the nitrification rate. Nitrification depends
! on the oxygen concentration according to a Monod kinetics. The ammonium diffusing upwrds from the deep anoxic layer
! is oxidized by nitrate producing nitrogen gas that escapes to the atmosphere.
! Implementation: Marilaure Gregoire,                NIOO-CEME
! Translation into NEMO : Luc Vandenbulcke
!######################################################################
SUBROUTINE ChemicalDynamics(Kbb,Kmm)
  IMPLICIT NONE

  integer, intent(in) :: Kbb, Kmm
  real(wp)  ::       tf,   &  ! Temperature function
       Nitrification_Rate,Ammonium_Oxidation_rate_by_Nitrate,   &     ! Nitrification rate (/day) and Ammonium oxidation rate by nitrate (/day)
       ODU_Oxidation_Rate_by_oxygen,ODU_Oxidation_Rate_by_nitrate, &  ! hODU oxidation rate by oxygen, by nitrate (/day)
       lim,inhib ! results of limitation or inhibition function
  INTEGER :: ji,jj,jk

  ! CALCULATING FLUXES
  DO_3D(0,0,0,0,1,jpk)
        if (tmask(ji,jj,jk).eq.1) then
          ! TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor (ts(ji,jj,jk,jp_tem,Kmm),Q10chem)

          ! Compute ammonium oxidation by oxygen
          lim = michaelis(tr(ji,jj,jk,dox,Kbb),ksoxnhsdox)
          Nitrification_Rate = tf*Roxnhs*lim

          ! Compute Ammonium oxidation by nitrate
          !lim   = inhibition(tr(ji,jj,jk,odu,Kbb),kinoxnhsodu)
          lim   = michaelis(tr(ji,jj,jk,nos,Kbb),0.3_wp)
          inhib = inhibition(tr(ji,jj,jk,dox,Kbb),kinoxnhsdox)
          Ammonium_Oxidation_rate_by_Nitrate = tf*Roxnhsnos*inhib*lim*inhibition(tr(ji,jj,jk,odu,Kbb),kinoxnhsodu)

          ! Compute ODU oxidation by oxygen
          lim = michaelis(tr(ji,jj,jk,dox,Kbb),ksoxodudox)
          ODU_Oxidation_Rate_by_oxygen = tf*Roxodu*lim

          ! Compute ODU oxidation by nitrate
          lim = michaelis(tr(ji,jj,jk,nos,Kbb),ksoxodunos)
          inhib = inhibition(tr(ji,jj,jk,dox,Kbb),kinoxodudox)
          ODU_Oxidation_Rate_by_nitrate = tf*Roxodunos*inhib*lim

          ! ADJUSTING THE RATE OF CHANGE
          ! oxidation of Ammonium by oxygen consumes oxygen and ammonium and produces nitrate
          dPNOS(ji,jj,jk) = dPNOS(ji,jj,jk) + Nitrification_Rate*tr(ji,jj,jk,nhs,Kbb)
          dDNHS(ji,jj,jk) = dDNHS(ji,jj,jk) + Nitrification_Rate*tr(ji,jj,jk,nhs,Kbb)
          dDDOX(ji,jj,jk) = dDDOX(ji,jj,jk) + Nitrification_Rate*tr(ji,jj,jk,nhs,Kbb)*ONoxnhsr

          ! Oxidation of ammonium by nitrate consumes nitrate and ammonium and produces N2 which is definitely lost for the system
#ifdef testcons
          Ammonium_Oxidation_rate_by_Nitrate=0.
#endif
          dDNOS(ji,jj,jk) = dDNOS(ji,jj,jk) + Ammonium_Oxidation_rate_by_Nitrate*tr(ji,jj,jk,nhs,Kbb)*NOsNHsr
          dDNHS(ji,jj,jk) = dDNHS(ji,jj,jk) + Ammonium_Oxidation_rate_by_Nitrate*tr(ji,jj,jk,nhs,Kbb)

          !oxidation of ODU by oxygen consumes oxygen and ODU and produces oxydant not modelled
          dDDOX(ji,jj,jk) = dDDOX(ji,jj,jk) + ODU_Oxidation_Rate_by_oxygen*tr(ji,jj,jk,odu,Kbb)*OODUr
          dDODU(ji,jj,jk) = dDODU(ji,jj,jk) + ODU_Oxidation_Rate_by_oxygen*tr(ji,jj,jk,odu,Kbb)

          ! oxidation of ODU by nitrate consumes nitrate and ODU and produces oxydant not modelled
#ifdef testcons
          ODU_Oxidation_Rate_by_nitrate=0.
#endif
          dDODU(ji,jj,jk) = dDODU(ji,jj,jk) + ODU_Oxidation_Rate_by_nitrate*tr(ji,jj,jk,odu,Kbb)
          dDNOS(ji,jj,jk) = dDNOS(ji,jj,jk) + ODU_Oxidation_Rate_by_nitrate*tr(ji,jj,jk,odu,Kbb)*NODUr

          ! Diagnostics Compute auxillary variables

#ifdef biodiagrates
          Oxidation_by_oxygen(ji,jj,jk) =  ODU_Oxidation_Rate_by_oxygen* tr(ji,jj,jk,odu,Kbb)*OODUr + Nitrification_Rate*tr(ji,jj,jk,nhs,Kbb)*ONoxnhsr
          Oxidation_by_nitrate(ji,jj,jk) = ODU_Oxidation_Rate_by_nitrate*tr(ji,jj,jk,odu,Kbb)*NODUr + Ammonium_Oxidation_rate_by_Nitrate*tr(ji,jj,jk,nhs,Kbb)*NOSNHSr
          Nitrification(ji,jj,jk)=Nitrification_Rate*tr(ji,jj,jk,nhs,Kbb)
          ANAMMOX(ji,jj,jk)=Ammonium_Oxidation_rate_by_Nitrate*tr(ji,jj,jk,nhs,Kbb)
#endif
        end if
  END_3D


  ! OUTPUT VARIABLES
#ifdef biodiagrates
  call IntegrateVertical(Oxidation_by_nitrate,Oxidation_by_nitrateIntegrated,kmm)
  call IntegrateVertical(Oxidation_by_oxygen,Oxidation_by_oxygenIntegrated,kmm)
  call IntegrateVertical(Nitrification,NitrificationIntegrated,kmm)
  call IntegrateVertical(ANAMMOX,ANAMMOXIntegrated,kmm)
#endif

  ! Diagnostics Averaged over entire water column

END SUBROUTINE ChemicalDynamics






!#########################################################################################
!                              3Ddiatoms.F90
!
! 1-D ecosystem model - Biological model of Tett
!
! Tett, P., 1998. Parameterising a microplankton model.
! Department of Biological Sciences, Napier University,
! Report ISBN 0 902703 60 9, 60 pp.
!
! Sharples & Tett (1994). Modeling the effect of physical
! variability on the midwater chlorophyll maximum.
! Journal of marine research 52: 219-238
!
! Implementation: Marilaure Gregoire,                 NIOO-CEME
! Translation in NEMO: Luc Vandenbulcke,   Universite de Liege, MAST
! Geider: Catherine Meulders, Universite de Liege, MAST
!
!######################################################################
SUBROUTINE CalculateDiatoms(Kbb,Kmm)
  !--------------------------------------------------------------------
  ! Contains the pelagic submodel, as used in Soetaert et al., 2001.
  ! References to the microplankton model:
  ! Tett, P., 1998. Parameterising a microplankton model.
  ! Department of Biological Sciences, Napier University,
  ! Report ISBN 0 902703 60 9, 60 pp.
  !
  ! Sharples & Tett (1994).  Modeling the effect of physical
  ! variability on the midwater chlorophyll maximum.
  ! Journal of marine research 52: 219-238
  !
  !--------------------------------------------------------------------*
  implicit none
  integer, intent(in) :: Kbb, Kmm
  real(wp)  :: &
       NCrDiat, &
       ChlCrDiatBounded, &
       tf, tfsilicate,   &                             ! Temperature function
       SiCrDiat, &                                                    ! si/c ratio of diatoms
       PHYMort, NetNGrowthDiatoms,Silicate_upDia,  &                     ! Phytoplankton mortality rate /day
       C_PHYMort, N_PHYMort,       &                                     ! Phytoplankton mortality flux, mmolC/day, mmolN/day
       Nitrate_UpPHY, Ammonium_UpPHY,nutrient_UpPHY, Nitrogen_UpPHY,  &  ! Nitrate, ammonium, nitrogen,nutrient uptake rate of phytoplankton, mmolN/m3/day
       GrowthPHY,Carbon_UptakePHY,TotalRespirationPHY,  &                ! Assimil-Resp,C assimilation rate of phytoplankton,total repsiration(basal+activity) mmolC/m3/day
       DOC_leakage,DON_leakage,         &                                ! DOC and DON leakage by phytoplankton in mmol/day
       DOC_extra_excr,&                                                  ! Phytoplankton extra DOC excretion, in mmolC/day
       Phosphate_upDiatoms, &                                            !Diatoms phosphate uptake
       LightLimitationDiatoms,NutrientLimitationDiatoms
#ifdef chlorophyll_geider
  real(wp) :: & 
       NCrDiatBounded, &
       ChlNrDiat,  &
       ChlCrDiat, &
       PmaxC, PphotoC, RhoChl, RespChl, VNC, MortChl, &
       ChloroSynthesisCHD, IkCHD, ChlCratD, growthCHD,&
       LightLimCHD
#endif  
  integer :: ji,jj,jk
#ifdef nanquest
  logical      :: isnan
#endif
#ifdef chlorophyll_geider
  real(wp) :: RangeNCrPHY, RatioNCrPHY
  RangeNCrPHY = MaxNCrDiatoms - MinNCrDiatoms
  RatioNCrPHY = MinNCrDiatoms / MaxNCrDiatoms
#endif
  
  ! CALCULATING FLUXES
  DO_3D(0,0,0,0,1,jpk)
        if (tmask(ji,jj,jk).eq.1) then

          ! TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor (ts(ji,jj,jk,jp_tem,Kmm),Q10PhyDiatoms)
          tfsilicate = Q10Factor (ts(ji,jj,jk,jp_tem,Kmm),Q10SilicateDiss)

          ! PHYTOPLANKTON
          ! N/C ratio
          NCrDiat  = Ratio(tr(ji,jj,jk,ndi,Kbb),tr(ji,jj,jk,cdi,Kbb))

#ifdef chlorophyll_geider
          ! Chlorophyll to nitrogen ratio (ChlNrPHY, mg Chl/mol N)
          ! ChlNrDiatoms = Ratio (trb(i,j,k,chd),trb(i,j,k,ndi)) ! not used
          ChlCrDiat = Ratio(tr(ji,jj,jk,chd,kbb),tr(ji,jj,jk,cdi,kbb))
          ChlNrDiat = Ratio(tr(ji, jj, jk, chd, kbb), tr(ji,jj,jk,ndi,kbb))          
 
          ! COMPUTE RATIO RANGES
	  CALL MIN_MAX_RATIO(NCrDiat,MaxNCrdiatoms,MinNCrdiatoms, NCrDiatBounded)
          CALL MIN_MAX_RATIO(ChlCrDiat, MaxNCrdiatoms*MaxChlNrDiatoms, MinNCrdiatoms*MinChlNrDiatoms, ChlCrDiatBounded)
		  
	  ! Si/C ratio
          SiCrDiat = NCrDiat*SiNrDiatoms

          ! Nitrate uptake rates of phytoplankton (NO3_upPHY, mmol N/m3/day)
          CALL NO_UPTAKE_RATE(NCrDiat,tr(ji,jj,jk,nos,kbb),tr(ji,jj,jk,nhs,kbb),tr(ji,jj,jk,cdi,kbb),  &
                     tf,MaxNCrDiatoms,NosMaxUptakeDiatoms,ksNOsDiatoms,kinNHsPhy, Nitrate_upPHY)

          ! Ammonium uptake rates of phytoplankton (NH3_upPHY, mmolN /m3/day)
          ! (and excretion if NC ratio too high)
          CALL NUT_UPTAKE_RATE(NCrDiat,tr(ji,jj,jk,nhs,kbb),tr(ji,jj,jk,cdi,kbb),tf,MaxNCrDiatoms,         &
                   NHsMaxUptakeDiatoms,ksNHsDiatoms,NHSNoUptakeDiatoms,  Ammonium_upPHY)

          ! Compute Silicate uptake
          CALL NUT_UPTAKE_RATE(SiCrDiat,tr(ji,jj,jk,sio,kbb),tr(ji,jj,jk,cdi,kbb),                     &
                   tf,MaxSiCrDiatoms,SiMaxUptakeDiatoms,ksSiDiatoms,0.0_wp, Silicate_upDia)

          CALL NUT_UPTAKE_RATE(NCrDiat,tr(ji,jj,jk,pho,kbb),tr(ji,jj,jk,cdi,kbb),                      &
                   tf,MaxNCrDiatoms,PO4MaxUptakeDiatoms,ksPO4Diatoms,PO4NoUptakeDiatoms, Phosphate_upDiatoms)

          Nitrogen_UpPHY=Ammonium_upPHY + Nitrate_upPHY
          Nutrient_UpPHY=min(Nitrogen_UpPHY,Silicate_upDia/SiNrDiatoms,Phosphate_upDiatoms/PNRedfield)
          
          chlorophyll(ji,jj,jk,2) = tr(ji,jj,jk,chd,kbb)
          
          CALL CHL_COMPUTATION(MuMaxDiatoms, ChlCrDiatBounded, NCrDiatBounded, MinNCrDiatoms, RangeNCrPHY,   &
               tf, alphaPIDiatoms, PAR(ji,jj,jk), WattToPhotons, MaxChlNrDiatoms, tr(ji,jj,jk,cdi,kbb), Nutrient_UpPHY, &
               RespirationDiatoms, MortalityDiatoms, tr(ji,jj,jk,chd,kbb), PphotoC, RespChl,MortChl, &
               growthCHD, PmaxC, RhoChl, ChlorosynthesisCHD, VNC, LightLimCHD )

          ! Compute the diatoms growth taking into account the potential limitimg effect of silicate
          CALL GROWTH_RATE(MinNCrDiatoms, NCrDiatBounded, RatioNCrPHY, ChlCrDiatBounded,  &
               alphaPIDiatoms, par(ji,jj,jk), WattToPhotons, PphotoC, tr(ji,jj,jk,cdi,kbb),         &
	       MuMaxDiatoms,RespirationDiatoms,GrowthRespDiatoms, tf, extradocphyexcr,       &
	       GrowthPHY,Carbon_UptakePHY,TotalRespirationPHY,DOC_extra_excr)
#else
          ! Chlorophyll to carbon ratio (ChlCrPHY, mg Chl/mol C)
          CALL CHL_C_RATIO(NCrDiat,MaxNCrDiatoms,MinNCrDiatoms,MinChlNrDiatoms,MaxChlNrDiatoms,ChlCrDiatBounded)
          chlorophyll(ji,jj,jk,2) = ChlCrDiatBounded * tr(ji,jj,jk,cdi,Kbb)

          ! Nitrate uptake rates of phytoplankton (NO3_upPHY, mmol N/m3/day)
          CALL NO_UPTAKE_RATE(NCrDiat,tr(ji,jj,jk,nos,Kbb),tr(ji,jj,jk,nhs,Kbb),     &
                   tr(ji,jj,jk,cdi,Kbb),tf,MaxNCrDiatoms,              &
                   NosMaxUptakeDiatoms,ksNOsDiatoms,kinNHsPhy, Nitrate_upPHY)

          ! Ammonium uptake rates of phytoplankton (NH3_upPHY, mmolN /m3/day)
          ! (and excretion if NC ratio too high)
          CALL NUT_UPTAKE_RATE(NCrDiat,tr(ji,jj,jk,nhs,Kbb),        &
                  tr(ji,jj,jk,cdi,Kbb),tf,MaxNCrDiatoms,              &
                  NHsMaxUptakeDiatoms,ksNHsDiatoms,0.0_wp,  Ammonium_upPHY)

          !Compute Silicate uptake
          SiCrDiat = NCrDiat*SiNrDiatoms
          CALL NUT_UPTAKE_RATE(SiCrDiat,tr(ji,jj,jk,sio,Kbb), &
              tr(ji,jj,jk,cdi,Kbb),tf,MaxSiCrDiatoms,              &
              SiMaxUptakeDiatoms,ksSiDiatoms,0.0_wp, Silicate_upDia)

          CALL NUT_UPTAKE_RATE(NCrDiat,tr(ji,jj,jk,pho,Kbb),    &
             tr(ji,jj,jk,cdi,Kbb),tf,MaxNCrDiatoms,                &
             PO4MaxUptakeDiatoms,ksPO4Diatoms,0.0_wp, Phosphate_upDiatoms)

          Nitrogen_UpPHY=Ammonium_upPHY + Nitrate_upPHY
          Nutrient_UpPHY=min(Nitrogen_UpPHY,Silicate_upDia/SiNrDiatoms,Phosphate_upDiatoms/PNRedfield)

          ! Compute the diatoms growth taking into account the potential limitimg effect of silicate
          CALL GROWTH_RATE(QuantumYieldDiatoms,alphaPIDiatoms, par(ji,jj,jk), WattToPhotons, LightLimitationDiatoms,NutrientLimitationDiatoms,         &
                   MaxNCrDiatoms,MinNCrDiatoms,NCrDiat,           &
                   MaxSiCrDiatoms,MinSiCrDiatoms,SiCrDiat,  &
                   MuMaxDiatoms,RespirationDiatoms,GrowthRespDiatoms,extradocphyexcr,    &
                   tf, tr(ji,jj,jk,cdi,Kbb),GrowthPHY,Carbon_UptakePHY,TotalRespirationPHY,DOC_extra_excr)
#endif
          
          !Compute the leakage and extra DOC excretion, DOC_extra_excr
          DOC_leakage = leakagephy*Carbon_UptakePHY
          DON_leakage  =leakagephy*abs(Nutrient_UpPHY)

          ! Phytoplankton mortality rate (PHYMort, /day)
          CALL PHYMORT_RATE(MortalityDiatoms,tf, PHYmort)

          ! Phytoplankton mortality flux C_PHYmort,N_PHYMort (in mmol C/m3/day or mmol N/m3/day)
          C_PHYmort  = PHYMort * tr(ji,jj,jk,cdi,Kbb)
          N_PHYMort  = PHYMort * tr(ji,jj,jk,ndi,Kbb)

          ! ADJUSTING THE RATE OF CHANGE

          ! phytoplankton C increases by growth,
          ! it decreases by zooplankton grazing (see the zooplankton subroutine) and phytoplankton mortality
          dPCDI(ji,jj,jk) = dPCDI(ji,jj,jk) +  GrowthPHY
          dDCDI(ji,jj,jk) = dDCDI(ji,jj,jk) + C_PhyMort+DOC_leakage

          ! phytoplankton N increases by N uptake,
          ! it decreases by zooplankton grazing and phytoplankton mortality and leakage
          ! NetNGrowthDiatoms=  Nutrient_UpPHY - N_PHYMort-DON_leakage ! luc commented this, it doesn't seem to be used
          dPNDI(ji,jj,jk) = dPNDI(ji,jj,jk)  +  Nutrient_UpPHY
          dDNDI(ji,jj,jk) = dDNDI(ji,jj,jk)  +  N_PHYMort+DON_leakage

          ! IF CN ratio of phytoplankton not lower than CNmin, than nitrogen is taken up
          ! Nitrate is taken up by phytoplankton
          IF (Nutrient_UpPHY.gt.0) THEN
            dDNOs(ji,jj,jk) = dDNOs(ji,jj,jk) + Nutrient_UpPHY*Nitrate_upPHY/Nitrogen_UpPHY
            dPDOX(ji,jj,jk) = dPDOX(ji,jj,jk) + Nutrient_UpPHY*Nitrate_upPHY/Nitrogen_UpPHY*ONoxnhsr
            ! Ammonium is taken up by phytoplankton
            dDNHs(ji,jj,jk) = dDNHs(ji,jj,jk) + Nutrient_UpPHY*Ammonium_upPHY/Nitrogen_UpPHY
            dDPHO(ji,jj,jk) = dDPHO(ji,jj,jk) + Nutrient_UpPHY*PNRedfield
#ifdef biodiagrates
            PHY_N_uptake(ji,jj,jk) = PHY_N_uptake(ji,jj,jk) + Nutrient_UpPHY*(Ammonium_upPHY+Nitrate_upPHY)/Nitrogen_UpPHY
#endif
          ELSE
            ! IF CN ratio of phytoplankton is lower than CNmin, than ammonium is excreted
            dPNHs(ji,jj,jk) = dPNHs(ji,jj,jk) - Nutrient_UpPHY
            dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk) - Nutrient_UpPHY*PNRedfield
#ifdef biodiagrates
           PHY_NHS_excr(ji,jj,jk) = PHY_NHS_excr(ji,jj,jk) + Nutrient_UpPhy
#endif
          ENDIF

          ! UPdate Silicate concentration
          Silicate_upDia= (Nutrient_UpPHY - DON_leakage)*SiNrDiatoms
          dDSIO(ji,jj,jk) = dDSIO(ji,jj,jk) + Silicate_upDia
          dPSIO(ji,jj,jk) = dPSIO(ji,jj,jk) + tfsilicate*kdis_Silicious_Detritus*tr(ji,jj,jk,sid,Kbb)
          dDSID(ji,jj,jk) = dDSID(ji,jj,jk) + tfsilicate*kdis_Silicious_Detritus*tr(ji,jj,jk,sid,Kbb)
          dPSID(ji,jj,jk) = dPSID(ji,jj,jk) + N_PhyMort*SiNrDiatoms

          ! As in Anderson and Pondhaven (2003), the phytoplanton mortality increases the pool of POM and DOM with a coefficient of mortdom
          ! for the DOM pool a part is considered as labile (labilefraction) and another part as semi labile (1-labilefraction)
          dPPOC(ji,jj,jk) = dPPOC(ji,jj,jk) + (1.0 - mortphydom)*C_PhyMort
          dPPON(ji,jj,jk) = dPPON(ji,jj,jk) + (1.0 - mortphydom)*N_PhyMort
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + mortphydom*C_PhyMort*labilefraction
          dPDNL(ji,jj,jk) = dPDNL(ji,jj,jk) + mortphydom*N_PhyMort*labilefraction
          dPDCS(ji,jj,jk) = dPDCS(ji,jj,jk) + mortphydom*C_PhyMort*(1.0 - labilefraction)
          dPDNS(ji,jj,jk) = dPDNS(ji,jj,jk) + mortphydom*N_PhyMort*(1.0 - labilefraction)

          ! As in Anderson and Pondhaven (2003), the DCLI and DNLI concentration increases also due to phytoplankton leakage which is considered
          ! to peoduced only labile DOC and DON
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + DOC_leakage
          dPDNL(ji,jj,jk) = dPDNL(ji,jj,jk) + DON_leakage

          ! As in Anderson and Pondhaven (2003), the phytoplankton extra DOC excretion, DOC_extra_excr, increases the pool of DOC (only DOC and not DON)
          ! this extra-excretion is composed of a part (leakage) which is assimilated to leakage and is thus only labile
          ! the remaining part (1-leakage) is composed of a labile (labileextradocexcr) and semi labile part (1-labileextradocexcr)
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + leakagephy*DOC_extra_excr + labileextradocphyexcr*(1.0 - leakagephy)*DOC_extra_excr
          dPDCS(ji,jj,jk) = dPDCS(ji,jj,jk) + (1.0 - labileextradocphyexcr)*(1.0 - leakagephy)*DOC_extra_excr

#ifdef chlorophyll_geider
          ! Adjusting chlorophyll 
          dPCHD(ji,jj,jk) = dPCHD(ji,jj,jk) + growthCHD       
          dDCHD(ji,jj,jk) = dDCHD(ji,jj,jk) + RespChl + MortChl

#endif
#ifdef aggregation
            ! The number of aggregates, POMNOS including PON increases and decreases with PON
            !     dPAGGI(ji,jj,jk) = dPAGGI(ji,jj,jk) + (1.0 - mortphydom)*N_PhyMort*AGGI(ji,jj,jk)/(PONI(ji,jj,jk) +PNSI(ji,jj,jk))
            dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) + (1.0 - mortphydom)*N_PhyMort*tr(ji,jj,jk,agg,Kbb)/(tr(ji,jj,jk,pon,Kbb))
#ifdef nanquest
            if (isnan(dPAGG(JI,JJ,JK))) then
              write (*,*) '** NAN QUEST ** in CalcDIATOMS'
              write (*,*) 'ji,jj,jk,PON,AGG=',ji,jj,jk,TR(JI,JJ,JK,PON,Kbb),tr(JI,JJ,JK,AGG,Kbb)
              call flush(6)
              stop
            endif
#endif
#endif
          ! The oxygen concentration increases due to photosynthesis
          dPDOX(ji,jj,jk) = dPDOX(ji,jj,jk) + (GrowthPHY + DOC_extra_excr)*OCr

          ! CO2 production and consumption
          dDDIC(ji,jj,jk) = dDDIC(ji,jj,jk) +GrowthPHY+DOC_extra_excr
          !Store Diagnostics the fluxes

#ifdef primaryprod
          GPP(ji,jj,jk) = GPP(ji,jj,jk) + Carbon_UptakePHY
          NPP(ji,jj,jk) = NPP(ji,jj,jk) + Carbon_UptakePHY - TotalRespirationPHY
#endif

#ifdef chlorophyll_geider
          PphotoC_DIAT(ji,jj,jk) = PphotoC
	  RespChl_DIAT(ji,jj,jk) =  RespChl 
	  MortChl_DIAT(ji,jj,jk) =  MortChl
	  growthCHD_DIAT(ji,jj,jk) =  growthCHD 
	  PmaxC_DIAT(ji,jj,jk) =  PmaxC
	  RhoChl_DIAT(ji,jj,jk) =  RhoChl
	  ChlorosynthesisCHD_DIAT(ji,jj,jk) =  ChlorosynthesisCHD
	  VNC_DIAT(ji,jj,jk) =  VNC
	  LightLimCHD_DIAT(ji,jj,jk) =   LightLimCHD 
	  Growth_DIAT(ji,jj,jk) =  GrowthPHY 
	  tf_DIAT(ji,jj,jk) =   tf
	  tf_SILICATE(ji,jj,jk) = tfsilicate
	  ChlCr_DIAT(ji,jj,jk) =   ChlCrDiat
	  ChlCr_DIAT_borne(ji,jj,jk) =   ChlCrDiatBounded
	  ChlCr_DIAT_bamhbi(ji,jj,jk) =   ChlCrDiat
	  Nitrate_uptake_DIAT(ji,jj,jk) =  Nitrate_upPHY 
	  Ammonium_uptake_DIAT(ji,jj,jk) =  Ammonium_upPHY 
	  Phosphate_uptake_DIAT(ji,jj,jk) = Phosphate_upDiatoms  
	  NCr_DIAT(ji,jj,jk) =  NCrDiat 
	  NCr_DIAT_borne(ji,jj,jk) =  NCrDiatBounded 
	  DOC_extra_excr_DIAT(ji,jj,jk) =   DOC_extra_excr
#endif
          
#ifdef biodiagrates
          Nitrogen_Uptake_Diatoms(ji,jj,jk) = Nutrient_UpPHY
          Carbon_UptakeDiatoms(ji,jj,jk)    = Carbon_UptakePHY
          TotalRespirationDiatoms(ji,jj,jk) = TotalRespirationPHY
          Silicate_upDiatoms(ji,jj,jk)      = Silicate_upDia
          PhytoNitrateReduction(ji,jj,jk)   = PhytoNitrateReduction(ji,jj,jk)+Nutrient_UpPHY*ratio(Nitrate_upPHY,Nitrogen_UpPHY)*ONoxnhsr
          C_Phymort_output(ji,jj,jk)= C_Phymort_output(ji,jj,jk)+ C_Phymort
	  N_Phymort_output(ji,jj,jk)= N_Phymort_output(ji,jj,jk)+ N_Phymort
	  DOXprod_PHY_nutuptake(ji,jj,jk)= DOXprod_PHY_nutuptake(ji,jj,jk)+ Nutrient_UpPHY*Nitrate_upPHY/Nitrogen_UpPHY*ONoxnhsr
	  DOXprod_PHY_growth(ji,jj,jk)= DOXprod_PHY_growth(ji,jj,jk)+ (GrowthPHY + DOC_extra_excr)*OCr
	  SID_dissolution(ji,jj,jk)= tfsilicate*kdis_Silicious_Detritus*tr(ji,jj,jk,sid,Kbb)
	  SIO_uptake_diatoms(ji,jj,jk)= Silicate_upDia
          SID_creation(ji,jj,jk)=  N_PhyMort*SiNrDiatoms
#endif

#ifdef biodiagtrophic
	  DOC_leakage_output(ji,jj,jk)= DOC_leakage_output(ji,jj,jk)+ DOC_leakage
	  DON_leakage_output(ji,jj,jk)= DON_leakage_output(ji,jj,jk)+ DON_leakage
	  DOC_extra_excr_output(ji,jj,jk)= DOC_extra_excr_output(ji,jj,jk)+ DOC_extra_excr
#endif
        end if
  END_3D

  ! OUTPUT VARIABLES
  ! Averaged over entire water column Diagnostics
#ifdef biodiagrates
  call IntegrateVertical(Nitrogen_Uptake_Diatoms,Nitrogen_Uptake_DiatomsIntegrated,kmm)
  call IntegrateVertical(Silicate_upDiatoms,Silicate_upDiatomsIntegrated,kmm)
  call IntegrateVertical(Carbon_UptakeDiatoms,Carbon_UptakeDiatomsIntegrated,kmm)
  call IntegrateVertical(TotalRespirationDiatoms,TotalRespirationDiatomsIntegrated,kmm)
#endif

END SUBROUTINE CalculateDiatoms




!#########################################################################################
!                              3Ddom.F90
!
! 1-D ecosystem model - Biological model of DOM
!
! Anderson, 1992,  Modelling the influence of food C:N ratio, and respiration on growth
! and nitrogen excretion in marine zooplankton and bacteria,
! Journal of Plankton Research, vol. 14, n 12, pp. 1645-1671, 1992
!
! Anderson and Williams,Modelling the Seasonal
! Cycle of Dissolved Organic Carbonj at Station E1 in the English Channel, Estuarine, Coastal
! and Shelf Science 1998, 46, 93-109.
!
! Anderson amd Pondhaven (2003),Non-redfield carbon and nitrogen cycling in the Sarasso Sea :
! pelagic imbalances and export flux, in press in DSR
!
! Implementation: Marilaure Gregoire,           NIOO-CEME
! Translation into NEMO : Luc Vandenbulcke,     ULg / MAST
!
!--------------------------------------------------------------------
! Contains the pelagic submodel, for DOM described in Anderson and Williams(1998),ECS and also
! in Anderson and Ponhaven (2003), DSR I. The DOM pool is divided between labile (DOCL, DONL)
! and semi-labile (DOCSL, DONSL) pools. each with dynamic C/N. The labile pool (turnover of hours to days)
! represents molecules which can be rapidly taken up by bacteria, whereas semi-labile pool (turnover of months)
!is characterized by complex bounding structures and must first be hydrolysed by exoenzymes before uptake by bacteria can occur
! (Anderson and Williams, 1998, 1999). DOM isproduced by messy feeding, phytoplankton and bacteria mortality (these terms are
! computed in the zopoplankton, phytoplankton and bacteria subroutines)
! and detrital turnover>
!--------------------------------------------------------------------
SUBROUTINE CalculateDOM(Kbb,Kmm)
  IMPLICIT NONE
  integer, intent(in) :: Kbb, Kmm
  real(wp)  :: &
       tf,                              &  ! Temperature function
       hydroPOC,hydroPON, &                ! Hydrolysis rate
       ! hydroPCS,hydroPNS, &              ! Hydrolysis rate
       POCHYDR,PONHYDR,   &                ! hydrolysis of POC (mmolC/m3/day) and PON (mmolN/m3/day)
       ! PCSHYDR,PNSHYDR,   &              ! hydrolysis of PCS (mmolC/m3/day) and PNS (mmolN/m3/day)
       DOCSLHYDR,DONSLHYDR,hydroPOMLim     ! hydrolysis of DOCSL (mmolC/m3/day) and DONSL (mmolN/m3/day)
  integer :: ji,jj,jk
#ifdef nanquest
  logical      :: isnan
#endif

  ! CALCULATING FLUXES
  DO_3D(0,0,0,0,1,jpk)
        if (tmask(ji,jj,jk).eq.1) then

          ! TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor(ts(ji,jj,jk,jp_tem,Kmm),Q10chem)

          ! Dissolved Organic matter divided in a labile part DOC and DON and a semi-labile part DOCSL and DONSL
          ! the C:N ratio of the DOM is not constant

          ! The DOM increases as a result of hydrolysis of the POM (POCHYDR,PONHYDR). This hydrolysis produced labile DOM with a coefficient
          ! labilefrac and a semi-labile part with a coefficient (1-labilefrac)
          ! The rate of hydrolysis is assumed to be proportional to the POC and PON concentration and the rate of hydrolysis is constant
          ! but different for the POC and PON. It is assumed as in Anderson and Williams that the PON is faster decomposed
          ! We have to consider that the decomposition of POM is not so active in anoxic conditions (Devol et al, 2001, L&O,
          ! Van Mooy et al., 2002 Geochimica et Cosmochimica)
          hydroPOMLim=(tr(ji,jj,jk,dox,Kbb)+1.0)/(tr(ji,jj,jk,dox,Kbb)+ksatOxygenHydrolysis +1.0)
          hydroPOC=hydroPOMLim*hydPOCmax
          hydroPON=hydroPOMLim*hydPONmax
          POCHYDR = tf*hydroPOC*tr(ji,jj,jk,poc,Kbb)
          PONHYDR = tf*hydroPON*tr(ji,jj,jk,pon,Kbb)
          !hydroPCS=hydroPOMLim*hydPCSmax
          !hydroPNS=hydroPOMLim*hydPNSmax
          !PCSHYDR = tf*hydroPCS*PCSI(ji,jj,jk)
          !PNSHYDR = tf*hydroPNS*PNSI(ji,jj,jk)

          ! The semi-labile DOM is decomposed into labile dom according to the model proposed in Anderson and Williams (1998)
          ! and Anderson and Pondhaven (2003).
          DOCSLHYDR = tf*maxhydrDOCSL*tr(ji,jj,jk,DCS,Kbb)/(tr(ji,jj,jk,DCS,Kbb) + csatdocsl)*tr(ji,jj,jk,bac,Kbb)
          DONSLHYDR = tf*maxhydrDOCSL*tr(ji,jj,jk,DCS,Kbb)/(tr(ji,jj,jk,DCS,Kbb) + csatdocsl)*tr(ji,jj,jk,bac,Kbb)*tr(ji,jj,jk,DNS,Kbb)/tr(ji,jj,jk,DCS,Kbb)

          ! ADJUSTING THE RATE OF CHANGE
          ! The hydrolysis of POM increases the pool of DOm and decreases the pool of POM
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + labilefraction*POCHYDR ! + labilefraction_fromS*PCSHYDR
          dPDNL(ji,jj,jk) = dPDNL(ji,jj,jk) + labilefraction*PONHYDR ! + labilefraction_fromS*PNSHYDR
          dPDCs(ji,jj,jk) = dPDCs(ji,jj,jk) + (1.0 - labilefraction)*POCHYDR !  +  (1.0 - labilefraction_fromS)*PCSHYDR
          dPDNs(ji,jj,jk) = dPDNs(ji,jj,jk) + (1.0 - labilefraction)*PONHYDR !  +  (1.0 - labilefraction_fromS)*PNSHYDR
          dDPOC(ji,jj,jk) = dDPOC(ji,jj,jk) + POCHYDR
          dDPON(ji,jj,jk) = dDPON(ji,jj,jk) + PONHYDR
         !dDPCS(ji,jj,jk) = dDPCS(ji,jj,jk) + PCSHYDR
         !dDPNS(ji,jj,jk) = dDPNS(ji,jj,jk) + PNSHYDR
#ifdef aggregation
          ! The hydrolysis of the POM decreases the number of aggregates POMNOS
          ! If diatoms can form aggregates :
           !dDAGG(ji,jj,jk) = dDAGG(ji,jj,jk) + (PONHYDR+PNSHYDR)*trb(ji,jj,jk,AGG,Kbb)/(tr(ji,jj,jk,pon,Kbb)+PNS(ji,jj,jk))
            dDAGG(ji,jj,jk) = dDAGG(ji,jj,jk) + (PONHYDR)*tr(ji,jj,jk,agg,Kbb)/(tr(ji,jj,jk,pon,Kbb))
#ifdef nanquest
            if (isnan(dDAGG(JI,JJ,JK))) then
              write (*,*) '** NAN QUEST ** in CalcDOM'
              write (*,*) 'ji,jj,jk,PON(JI,JJ,JK),AGG(JI,JJ,JK)',ji,jj,jk,TR(JI,JJ,JK,PON?kbb),tr(JI,JJ,JK,AGG,Kbb)
              write (*,*) 'PONHYDR',PONHYDR
              call flush(6)
              stop
            endif
#endif
#endif
          ! The hydrolysis of the semi labile DOM increases the pool of labile DOM and decreases the pool of semi-labile DOM
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + DOCSLHYDR
          dPDNL(ji,jj,jk) = dPDNL(ji,jj,jk) + DONSLHYDR
          dDDCs(ji,jj,jk) = dDDCs(ji,jj,jk) + DOCSLHYDR
          dDDNs(ji,jj,jk) = dDDNs(ji,jj,jk) + DONSLHYDR
          
#ifdef biodiagrates
          DOCSLHYDR_output(ji,jj,jk)= DOCSLHYDR
          DONSLHYDR_output(ji,jj,jk)= DONSLHYDR
          POCHYDR_output(ji,jj,jk)= POCHYDR
	  PONHYDR_output(ji,jj,jk)= PONHYDR
#endif
          
        end if
  END_3D

  ! OUTPUT VARIABLES
END SUBROUTINE CalculateDOM

       



!#########################################################################################
!                              3DemilianaF90
!
! 1-D ecosystem model - Biological model of Tett
!
! Tett, P., 1998. Parameterising a microplankton model.
! Department of Biological Sciences, Napier University,
! Report ISBN 0 902703 60 9, 60 pp.
!
! Sharples & Tett (1994). Modeling the effect of physical
! variability on the midwater chlorophyll maximum.
! Journal of marine research 52: 219-238
!
! Implementation: Marilaure Gregoire,             NIOO-CEME
! Translation into NEMO: L. Vandenbulcke, ULg / MAST
!
! Contains the pelagic submodel, as used in Soetaert et al., 2001.
! References to the microplankton model:
! Tett, P., 1998. Parameterising a microplankton model.
! Department of Biological Sciences, Napier University,
! Report ISBN 0 902703 60 9, 60 pp.
!
! Sharples & Tett (1994).  Modeling the effect of physical
! variability on the midwater chlorophyll maximum.
! Journal of marine research 52: 219-238
!--------------------------------------------------------------------*
SUBROUTINE CalculateEmiliana(Kbb,Kmm)
  IMPLICIT NONE
  integer, intent(in) :: Kbb, Kmm
  real(wp)  :: &
    NCrEm, &
    ChlCrEmBounded, &
    tf,            &                                    ! Temperature function
    PHYMort,                  &           ! Phytoplankton mortality rate /day
    C_PHYMort, N_PHYMort,       &                       ! Phytoplankton mortality flux, mmolC/day, mmolN/day
    Nitrate_UpPHY, Ammonium_UpPHY ,  &                  ! Nitrate, ammonium uptake rate of phytoplankton, mmolN/m3/day
    GrowthPHY,Carbon_UptakePHY,TotalRespirationPHY,  &  ! Assimil-Resp,C assimilation rate of phytoplankton,total repsiration(basal+activity) mmolC/m3/day
    DOC_leakage,DON_leakage,         &                  ! DOC and DON leakage by phytoplankton in mmol/day
    DOC_extra_excr, &                                   ! Phytoplankton extra DOC excretion, in mmolC/day
    Phosphate_upEmiliana,            &                  ! Emiliana phosphate uptake
    Nitrogen_UpPHY,Nutrient_UpPHY, &
    LightLimitationEmiliana,NutrientLimitationEmiliana
#ifdef chlorophyll_geider
  real(wp) :: & 
    NCrEmBounded, &
    ChlNrEm, &
    ChlCrEm, &
    PmaxC, PphotoC, RhoChl, RespChl, VNC, MortChl, &
    ChloroSynthesisCHE, IkCHE, ChlCratE, growthCHE,LightLimCHE
#endif
  integer :: ji,jj,jk
#ifdef nanquest
  logical      :: isnan
#endif
#ifdef chlorophyll_geider
  real(wp) :: RangeNCrPHY,RatioNCrPHY
  RangeNCrPHY = MaxNCrEmiliana - MinNCrEmiliana
  RatioNCrPHY = MinNCrEmiliana / MaxNCrEmiliana
#endif

  ! CALCULATING FLUXES
  DO_3D(0,0,0,0,1,jpk)
        if (tmask(ji,jj,jk).eq.1) then
          ! TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor(ts(ji,jj,jk,jp_tem,Kmm),Q10Phy)

          ! PHYTOPLANKTON
          ! N/C ratio
          NCrEm  = Ratio(tr(ji,jj,jk,nem,Kbb),tr(ji,jj,jk,cem,Kbb))

#ifdef chlorophyll_geider

          ChlCrEm = Ratio(tr(ji,jj,jk,che,kbb),tr(ji,jj,jk,cem,kbb))
          ChlNrEm = Ratio(tr(ji,jj,jk,che,kbb), tr(ji,jj,jk,nem,kbb))          
          
	  CALL MIN_MAX_RATIO(NCrEm,MaxNCrdiatoms,MinNCrdiatoms, NCrEmBounded)
          CALL MIN_MAX_RATIO(ChlCrEm, MaxNCrEmiliana*MaxChlNrEmiliana, MinNCrEmiliana*MinChlNrEmiliana, ChlCrEmBounded)

          ! Nitrate uptake rates of phytoplankton (NO3_upPHY, mmol N/m3/day)
          CALL NO_UPTAKE_RATE(NCrEm,tr(ji,jj,jk,nos,kbb),tr(ji,jj,jk,nhs,kbb),     &
               tr(ji,jj,jk,cem,kbb),tf,MaxNCrEmiliana,NosMaxUptakeEmiliana,ksNOsEmiliana,kinNHsPhy, Nitrate_upPHY)

          ! Ammonium uptake rates of phytoplankton (NH3_upPHY, mmolN /m3/day)
          ! (and excretion if NC ratio too high)
          CALL NUT_UPTAKE_RATE(NCrEm,tr(ji,jj,jk,nhs,kbb),tr(ji,jj,jk,cem,kbb),tf,MaxNCrEmiliana,  &
                   NHsMaxUptakeEmiliana,ksNHsEmiliana,NHSNoUptakeEmiliana, Ammonium_upPHY)

          CALL NUT_UPTAKE_RATE(NCrEm,tr(ji,jj,jk,pho,kbb),tr(ji,jj,jk,cem,kbb),tf,MaxNCrEmiliana,  &
                   PO4MaxUptakeEmiliana,ksPO4Emiliana,PO4NoUptakeEmiliana, Phosphate_upEmiliana)

          ! Potential Nitrogen uptake
          Nitrogen_UpPHY=Ammonium_upPHY + Nitrate_upPHY
          ! Nutrient uptake
          Nutrient_UpPHY=min(Nitrogen_UpPHY,Phosphate_upEmiliana/PNRedfield)
          !Nutrient_UpPHY=max(Nutrient_UpPHY,0)
          
          Chlorophyll(ji,jj,jk,3) = tr(ji,jj,jk,che,kbb)
          
          CALL CHL_COMPUTATION(MuMaxEmiliana, ChlCrEmBounded, NCrEmBounded, MinNCrEmiliana, RangeNCrPHY, tf, &
				alphaPIEmiliana, PAR(ji,jj,jk), WattToPhotons, MaxChlNrEmiliana,  &
				tr(ji,jj,jk,cem,kbb), Nutrient_UpPHY, RespirationEmiliana, &
				MortalityEmiliana, tr(ji,jj,jk,che,kbb), PphotoC, &
				RespChl, MortChl,  &
				growthCHE, PmaxC, RhoChl, ChlorosynthesisCHE, VNC, LightLimCHE)
				
          CALL GROWTH_RATE(MinNCrEmiliana, NCrEmBounded, RatioNCrPHY, ChlCrEmBounded,     &
				alphaPIEmiliana, par(ji,jj,jk), WattToPhotons, PphotoC, tr(ji,jj,jk,cem,kbb),         &
				MuMaxEmiliana,RespirationEmiliana,GrowthRespEmiliana, tf, extradocphyexcr,    &
				GrowthPHY,Carbon_UptakePHY,TotalRespirationPHY,DOC_extra_excr)
#else
          ! Chlorophyll to carbon ratio (ChlCrPHY, mg Chl/mol C)
          CALL CHL_C_RATIO(NCrEm,MaxNCrEmiliana,MinNCrEmiliana,           &
                   MinChlNrEmiliana,MaxChlNrEmiliana,ChlCrEmBounded)
          Chlorophyll(ji,jj,jk,3) = ChlCrEmBounded * tr(ji,jj,jk,cem,Kbb)

          ! Nitrate uptake rates of phytoplankton (NO3_upPHY, mmol N/m3/day)
          CALL NO_UPTAKE_RATE(NCrEm,tr(ji,jj,jk,nos,Kbb),tr(ji,jj,jk,nhs,Kbb),     &
                   tr(ji,jj,jk,cem,Kbb),tf,MaxNCrEmiliana,                               &
                   NosMaxUptakeEmiliana,ksNOsEmiliana,kinNHsPhy, Nitrate_upPHY)

          ! Ammonium uptake rates of phytoplankton (NH3_upPHY, mmolN /m3/day)
          ! (and excretion if NC ratio too high)
          CALL NUT_UPTAKE_RATE(NCrEm,tr(ji,jj,jk,nhs,Kbb),        &
                   tr(ji,jj,jk,cem,Kbb),tf,MaxNCrEmiliana,              &
                   NHsMaxUptakeEmiliana,ksNHsEmiliana,0.0_wp, Ammonium_upPHY)

          CALL NUT_UPTAKE_RATE(NCrEm,tr(ji,jj,jk,pho,Kbb),        &
                   tr(ji,jj,jk,cem,Kbb),tf,MaxNCrEmiliana,              &
                   PO4MaxUptakeEmiliana,ksPO4Emiliana,0.0_wp, Phosphate_upEmiliana)

          ! Potential Nitrogen uptake
          Nitrogen_UpPHY=Ammonium_upPHY + Nitrate_upPHY
          ! Nutrient uptake
          Nutrient_UpPHY=min(Nitrogen_UpPHY,Phosphate_upEmiliana/PNRedfield)
          !Nutrient_UpPHY=max(Nutrient_UpPHY,0)

          CALL GROWTH_RATE(QuantumYieldEmiliana,alphaPIEmiliana,par(ji,jj,jk), WattToPhotons,LightLimitationEmiliana,NutrientLimitationEmiliana,        &
                   MaxNCrEmiliana,MinNCrEmiliana,NCrEm,           &
                   1.0_wp,0.0_wp,1.0_wp,  &
                   MuMaxEmiliana,RespirationEmiliana,GrowthRespEmiliana,extradocphyexcr,           &
                   tf,tr(ji,jj,jk,cem,Kbb),GrowthPHY,Carbon_UptakePHY,TotalRespirationPHY,DOC_EXTRA_EXCR)
#endif    
          !Compute the leakage    and extra DOC excretion, DOC_extra_excr
          DOC_leakage = leakagephy*Carbon_UptakePHY
          DON_leakage  =leakagephy*abs(Nutrient_upPHY)

          ! Phytoplankton mortality rate (PHYMort, /day)
          CALL PHYMORT_RATE(MortalityEmiliana,tf, PHYmort)

          ! Phytoplankton mortality flux C_PHYmort,N_PHYMort (in mmol C/m3/day or mmol N/m3/day)
          C_PHYmort  = PHYMort * tr(ji,jj,jk,cem,Kbb)
          N_PHYMort  = PHYMort * tr(ji,jj,jk,nem,Kbb)

          ! ADJUSTING THE RATE OF CHANGE
          ! phytoplankton C increases by growth,
          ! it decreases by zooplankton grazing (see the zooplankton subroutine) and phytoplankton mortality
          dPCEM(ji,jj,jk) = dPCEM(ji,jj,jk) + GrowthPHY
          dDCEM(ji,jj,jk) = dDCEM(ji,jj,jk) + C_PhyMort+DOC_leakage

          ! phytoplankton N increases by N uptake,
          ! it decreases by zooplankton grazing and phytoplankton mortality
          dPNEM(ji,jj,jk) = dPNEM(ji,jj,jk) + Nutrient_UpPHY
          dDNEM(ji,jj,jk) = dDNEM(ji,jj,jk) + N_PHYMort+DON_leakage

          ! IF CN ratio of phytoplankton not lower than CNmin, than nitrogen is taken up
          ! Nitrate is taken up by phytoplankton
          IF (Nutrient_UpPHY.gt.0) THEN
            dDNOs(ji,jj,jk) = dDNOs(ji,jj,jk) + Nutrient_UpPHY*Nitrate_upPHY/Nitrogen_UpPHY
            dPDOX(ji,jj,jk) = dPDOX(ji,jj,jk) + Nutrient_UpPHY*Nitrate_upPHY/Nitrogen_UpPHY*ONoxnhsr

            ! Ammonium is taken up by phytoplankton
            dDNHs(ji,jj,jk) = dDNHs(ji,jj,jk) + Nutrient_UpPHY*Ammonium_upPHY/Nitrogen_UpPHY
            dDPHO(ji,jj,jk) = dDPHO(ji,jj,jk) + Nutrient_UpPHY*PNRedfield
            
#ifdef biodiagrates
            phy_n_uptake(ji,jj,jk) = phy_n_uptake(ji,jj,jk) + Nutrient_UpPHY*(Nitrate_upPHY+Ammonium_upPHY)/Nitrogen_UpPHY
#endif
          ELSE
            ! IF CN ratio of phytoplankton is lower than CNmin, than ammonium is excreted
            dPNHs(ji,jj,jk) = dPNHs(ji,jj,jk) - Nutrient_UpPHY
            dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk) - Nutrient_UpPHY*PNRedfield

#ifdef biodiagrates
           PHY_NHS_excr(ji,jj,jk) = PHY_NHS_excr(ji,jj,jk) + Nutrient_UpPhy
#endif
          END IF

          ! As in Anderson and Pondhaven (2003), the phytoplanton mortality increases the pool of POM and DOM with a coefficient of mortdom
          ! for the DOM pool a part is considered as labile (labilefraction) and another part as semi labile (1-labilefraction)
          dPPOC(ji,jj,jk) = dPPOC(ji,jj,jk) + (1.0 - mortphydom)*C_PhyMort
          dPPON(ji,jj,jk) = dPPON(ji,jj,jk) + (1.0 - mortphydom)*N_PhyMort
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + mortphydom*C_PhyMort*labilefraction
          dPDNL(ji,jj,jk) = dPDNL(ji,jj,jk) + mortphydom*N_PhyMort*labilefraction
          dPDCS(ji,jj,jk) = dPDCS(ji,jj,jk) + mortphydom*C_PhyMort*(1.0 - labilefraction)
          dPDNS(ji,jj,jk) = dPDNS(ji,jj,jk) + mortphydom*N_PhyMort*(1.0 - labilefraction)

          ! As in Anderson and Pondhaven (2003), the DOCL and DONL concentration increases also due to phytoplankton leakage which is considered
          ! to peoduced only labile DOC and DON
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + DOC_leakage
          dPDNL(ji,jj,jk) = dPDNL(ji,jj,jk) + DON_leakage

          ! As in Anderson and Pondhaven (2003), the phytoplankton extra DOC excretion, DOC_extra_excr, increases the pool of DOC (only DOC and not DON)
          ! this extra-excretion is composed of a part (leakage) which is assimilated to leakage and is thus only labile
          ! the remaining part (1-leakage) is composed of a labile (labileextradocexcr) and semi labile part (1-labileextradocexcr)
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + leakagephy*DOC_extra_excr + labileextradocphyexcr*(1.0 - leakagephy)*DOC_extra_excr
          dPDCS(ji,jj,jk) = dPDCS(ji,jj,jk) + (1.0 - labileextradocphyexcr)*(1.0 - leakagephy)*DOC_extra_excr

#ifdef chlorophyll_geider
	  ! Adjusting chlorophyll 
          dPCHE(ji,jj,jk) = dPCHE(ji,jj,jk) + growthCHE
          dDCHE(ji,jj,jk) = dDCHE(ji,jj,jk) + RespChl + MortChl
          
#endif
#ifdef aggregation
            ! The number of aggregates, POMNOS including  PON increases and decreases with PON
            !      dPAGGI(ji,jj,jk) = dPAGGI(ji,jj,jk) + (1.0 - mortphydom)*N_PhyMort*AGGI(ji,jj,jk)/(PONI(ji,jj,jk)+PNSI(ji,jj,jk))
            dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) + (1.0 - mortphydom)*N_PhyMort*tr(ji,jj,jk,agg,Kbb)/(tr(ji,jj,jk,pon,Kbb))

#ifdef nanquest
            if (isnan(dPAGG(JI,JJ,JK))) then
              write (*,*) '** NAN QUEST ** in CalcEMILIANNA'
              write (*,*) 'ji,jj,jk,PON(JI,JJ,JK),trb(JI,JJ,JK,AGG)'
              write (*,*) ji,jj,jk,TR(JI,JJ,JK,PON,Kbb),tr(JI,JJ,JK,AGG,Kbb)
              call flush(6)
              stop
            endif
#endif
#endif
          ! The oxygen concentration increases due to photosynthesis
          dPDOX(ji,jj,jk) = dPDOX(ji,jj,jk) + (GrowthPHY + DOC_extra_excr)*OCr

          ! CO2 production and consumption
          dDDIC(ji,jj,jk) = dDDIC(ji,jj,jk) + GrowthPHY + DOC_extra_excr
          !Diagnostics Store the fluxes

#ifdef primaryprod
          GPP(ji,jj,jk) = GPP(ji,jj,jk) + Carbon_UptakePHY
          NPP(ji,jj,jk) = NPP(ji,jj,jk) + Carbon_UptakePHY - TotalRespirationPHY
#endif
#ifdef biodiagrates
          Nitrogen_Uptake_Emiliana(ji,jj,jk) = Ammonium_upPHY + Nitrate_upPHY
          Carbon_UptakeEmiliana(ji,jj,jk)    = Carbon_UptakePHY
          TotalRespirationEmiliana(ji,jj,jk) = TotalRespirationPHY
          PhytoNitrateReduction(ji,jj,jk) = PhytoNitrateReduction(ji,jj,jk)+Nutrient_UpPHY*ratio(Nitrate_upPHY,Nitrogen_UpPHY)*ONoxnhsr
          C_Phymort_output(ji,jj,jk)= C_Phymort_output(ji,jj,jk)+ C_Phymort
          N_Phymort_output(ji,jj,jk)= N_Phymort_output(ji,jj,jk)+ N_Phymort
          DOXprod_PHY_nutuptake(ji,jj,jk)= DOXprod_PHY_nutuptake(ji,jj,jk)+ Nutrient_UpPHY*Nitrate_upPHY/Nitrogen_UpPHY*ONoxnhsr
	  DOXprod_PHY_growth(ji,jj,jk)= DOXprod_PHY_growth(ji,jj,jk)+ (GrowthPHY + DOC_extra_excr)*OCr
#endif
#ifdef biodiagtrophic
	  DOC_leakage_output(ji,jj,jk)= DOC_leakage_output(ji,jj,jk)+ DOC_leakage
	  DON_leakage_output(ji,jj,jk)= DON_leakage_output(ji,jj,jk)+ DON_leakage
	  DOC_extra_excr_output(ji,jj,jk)= DOC_extra_excr_output(ji,jj,jk)+ DOC_extra_excr
#endif
        end if
  END_3D

  ! OUTPUT VARIABLES
  ! Averaged over entire water column Diagnostics
#ifdef biodiagrates
   call IntegrateVertical(Nitrogen_Uptake_Emiliana,Nitrogen_Uptake_EmilianaIntegrated,kmm)
   call IntegrateVertical(Carbon_UptakeEmiliana,Carbon_UptakeEmilianaIntegrated,kmm)
   call IntegrateVertical(TotalRespirationEmiliana,TotalRespirationEmilianaIntegrated,kmm)
   call Integratevertical(PhytoNitrateReduction,PhytoNitrateReductionIntegrated,kmm)
#endif
#ifdef primaryprod
   ! integratevertical is called here because Emiliana is the last phyto group to be called
   call Integratevertical(NPP,NPPIntegrated,kmm)
#endif

END SUBROUTINE CalculateEmiliana
   



!#########################################################################################
!                              3DflagellatesF90
!
! 1-D ecosystem model - Biological model of Tett
!
! Tett, P., 1998. Parameterising a microplankton model.
! Department of Biological Sciences, Napier University,
! Report ISBN 0 902703 60 9, 60 pp.
!
! Sharples & Tett (1994). Modeling the effect of physical
! variability on the midwater chlorophyll maximum.
! Journal of marine research 52: 219-238
!
! Implementation: Marilaure Gregoire,                 NIOO-CEME
! Translation into NEMO : Luc Vandenbulcke, ULg / MAST
!
!--------------------------------------------------------------------
! Contains the pelagic submodel, as used in Soetaert et al., 2001.
! References to the microplankton model:
! Tett, P., 1998. Parameterising a microplankton model.
! Department of Biological Sciences, Napier University,
! Report ISBN 0 902703 60 9, 60 pp.                                                                                                                                                                                                                                           !
! Sharples & Tett (1994).  Modeling the effect of physical
! variability on the midwater chlorophyll maximum.
! Journal of marine research 52: 219-238
!
!--------------------------------------------------------------------*
SUBROUTINE CalculateFlagellates(Kbb,Kmm)
  IMPLICIT NONE
  integer, intent(in) :: Kbb, Kmm
  real(wp)  ::                              &
       NCrFlag, & 
       ChlCrFlagBounded,         &
       tf,                              &               ! Temperature function
       PHYMort,                        &               ! Phytoplankton mortality rate /day
       C_PHYMort, N_PHYMort,            &               ! Phytoplankton mortality flux, mmolC/day, mmolN/day
       Nitrate_UpPHY, Ammonium_UpPHY ,  &               ! Nitrate, ammonium uptake rate of phytoplankton, mmolN/m3/day
       GrowthPHY,Carbon_UptakePHY,TotalRespirationPHY,& ! Assimil-Resp,C assimilation rate of phytoplankton,total repsiration(basal+activity) mmolC/m3/day
       DOC_leakage,DON_leakage,LightLimitationFlagellates,NutrientLimitationFlagellates,         & ! DOC and DON leakage by phytoplankton in mmol/day
       Phosphate_upFlagellates,         &               ! Flagellates phosphate uptake
       Nitrogen_UpPHY,Nutrient_UpPHY,   &               ! Flagellates phosphate uptake
       DOC_extra_excr                                   ! Phytoplankton extra DOC excretion, in mmolC/day
#ifdef chlorophyll_geider
  real(wp) :: &
       NCrFlagBounded, &
       ChlNrFlag, &
       ChlCrFlag, &
       PmaxC, PphotoC, RhoChl, RespChl, VNC, MortChl,  &
       ChloroSynthesisCHF, IkCHF, ChlCratF, growthCHF, LightlimCHF
#endif
  integer :: ji,jj,jk
#ifdef nanquest
  logical :: isnan
#endif
#ifdef chlorophyll_geider
  real(wp) :: RangeNCrPHY,RatioNCrPHY
  RangeNCrPHY = MaxNCrFlagellates - MinNCrFlagellates
  RatioNCrPHY = MinNCrFlagellates / MaxNCrFlagellates
#endif

  ! CALCULATING FLUXES
  DO_3D(0,0,0,0,1,jpk)
        if (tmask(ji,jj,jk).eq.1) then
          ! TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor(ts(ji,jj,jk,jp_tem,Kmm),Q10Phy)

          ! PHYTOPLANKTON

          ! N/C ratio
          NCrFlag  = Ratio(tr(ji,jj,jk,nfl,Kbb),tr(ji,jj,jk,cfl,Kbb))

#ifdef chlorophyll_geider
          ! Chlorophyll to nitrogen ratio (ChlNrPHY, mg Chl/mol N)
			! ChlNrFlagellates = Ratio (trb(i,j,k,chf),trb(i,j,k,nfl)) 
          ChlCrFlag = Ratio(tr(ji,jj,jk,chf,kbb), tr(ji,jj,jk,cfl,kbb))
          ChlNrFlag = Ratio(tr(ji,jj,jk,chf,kbb), tr(ji,jj,jk,nfl,kbb)) 
 
          CALL MIN_MAX_RATIO(NCrFlag,MaxNCrdiatoms,MinNCrdiatoms, NCrFlagBounded)
          CALL MIN_MAX_RATIO(ChlCrFlag, MaxNCrFlagellates*MaxChlNrFlagellates, MinNCrFlagellates*MinChlNrFlagellates, ChlCrFlagBounded)
		  

          ! Nitrate uptake rates of phytoplankton (NO3_upPHY, mmol N/m3/day)
          CALL NO_UPTAKE_RATE(NCrFlag,tr(ji,jj,jk,nos,kbb),tr(ji,jj,jk,nhs,kbb),     &
                   tr(ji,jj,jk,cfl,kbb),tf,MaxNCrFlagellates,              &
                   NosMaxUptakeFlagellates,ksNOsFlagellates,kinNHsPhy, Nitrate_upPHY)

          ! Ammonium uptake rates of phytoplankton (NH3_upPHY, mmolN /m3/day)
          ! (and excretion if NC ratio too high)
          CALL NUT_UPTAKE_RATE(NCrFlag,tr(ji,jj,jk,NHS,kbb),tr(ji,jj,jk,cfl,kbb),tf,MaxNCrFlagellates,              &
                   NHsMaxUptakeFlagellates,ksNHsFlagellates,NHSNoUptakeFlagellates, Ammonium_upPHY)

          CALL NUT_UPTAKE_RATE(NCrFlag,tr(ji,jj,jk,pho,kbb),tr(ji,jj,jk,cfl,kbb),tf,MaxNCrFlagellates,              &
              PO4MaxUptakeFlagellates,ksPO4Flagellates,PO4NoUptakeFlagellates, Phosphate_upFlagellates)
         
          ! Potential Nitrogen uptake
          Nitrogen_UpPHY=Ammonium_upPHY + Nitrate_upPHY
          ! Nutrient uptake
          Nutrient_UpPHY=min(Nitrogen_UpPHY,Phosphate_upFlagellates/PNRedfield)
          ! Nutrient_UpPHY=max(Nutrient_UpPHY,0)              
          
          chlorophyll(ji,jj,jk,1) = tr(ji,jj,jk,chf,kbb)
          
          CALL CHL_COMPUTATION(MuMaxFlagellates, ChlCrFlagBounded, NCrFlagBounded, MinNCrFlagellates, RangeNCrPHY,   &
          tf, alphaPIFlagellates, PAR(ji,jj,jk), WattToPhotons, MaxChlNrFlagellates, tr(ji,jj,jk,cfl,kbb), Nutrient_UpPHY, &
          RespirationFlagellates, MortalityFlagellates, tr(ji,jj,jk,chf,kbb), PphotoC, RespChl, MortChl, &
          growthCHF, PmaxC, RhoChl, ChlorosynthesisCHF, VNC, LightLimCHF)                                                                                                                                                                                                                    
          CALL GROWTH_RATE(MinNCrFlagellates, NCrFlagBounded, RatioNCrPHY, ChlCrFlagBounded,     &
                        alphaPIFlagellates, par(ji,jj,jk), WattToPhotons, PphotoC, tr(ji,jj,jk,cfl,kbb),        &
                        MuMaxFlagellates,RespirationFlagellates,GrowthRespFlagellates, tf, extradocphyexcr,    &
                        GrowthPHY,Carbon_UptakePHY,TotalRespirationPHY,DOC_extra_excr)
#else
          ! Chlorophyll to carbon ratio (ChlCrPHY, mg Chl/mol C)
          CALL CHL_C_RATIO(NCrFlag,MaxNCrFlagellates,MinNCrFlagellates,           &
                 MinChlNrFlagellates,MaxChlNrFlagellates,ChlCrFlagBounded)
          chlorophyll(ji,jj,jk,1) = ChlCrFlagBounded * tr(ji,jj,jk,cfl,Kbb)

          ! Nitrate uptake rates of phytoplankton (NO3_upPHY, mmol N/m3/day)
          CALL NO_UPTAKE_RATE(NCrFlag,tr(ji,jj,jk,nos,Kbb),tr(ji,jj,jk,nhs,Kbb),     &
                   tr(ji,jj,jk,cfl,Kbb),tf,MaxNCrFlagellates,              &
                   NosMaxUptakeFlagellates,ksNOsFlagellates,kinNHsPhy, Nitrate_upPHY)

          ! Ammonium uptake rates of phytoplankton (NH3_upPHY, mmolN /m3/day)
          ! (and excretion if NC ratio too high)
          CALL NUT_UPTAKE_RATE(NCrFlag,tr(ji,jj,jk,NHS,Kbb),tr(ji,jj,jk,cfl,Kbb),tf,MaxNCrFlagellates,              &
                   NHsMaxUptakeFlagellates,ksNHsFlagellates,0.0_wp, Ammonium_upPHY)

          CALL NUT_UPTAKE_RATE(NCrFlag,tr(ji,jj,jk,pho,Kbb),tr(ji,jj,jk,cfl,Kbb),tf,MaxNCrFlagellates,              &
              PO4MaxUptakeFlagellates,ksPO4Flagellates,0.0_wp, Phosphate_upFlagellates)

          ! Potential Nitrogen uptake
          Nitrogen_UpPHY=Ammonium_upPHY + Nitrate_upPHY
          ! Nutrient uptake
          Nutrient_UpPHY=min(Nitrogen_UpPHY,Phosphate_upFlagellates/PNRedfield)
          ! Nutrient_UpPHY=max(Nutrient_UpPHY,0)                                                                                                                                                                                                                        
          CALL GROWTH_RATE(QuantumYieldFlagellates,alphaPIFlagellates,PAR(ji,jj,jk), WattToPhotons,LightLimitationFlagellates,NutrientLimitationFlagellates,          &
              MaxNCrFlagellates,MinNCrFlagellates,NCrFlag,           &
              1.0_wp,0.0_wp,1.0_wp,  &
              MuMaxFlagellates,RespirationFlagellates,GrowthRespFlagellates,extradocphyexcr,           &
              tf, tr(ji,jj,jk,cfl,Kbb),GrowthPHY,Carbon_UptakePHY,TotalRespirationPHY,DOC_EXTRA_EXCR)
#endif
          
          !Compute the leakage    and extra DOC excretion, DOC_extra_excr
          DOC_leakage = leakagephy*Carbon_UptakePHY
          DON_leakage  =leakagephy*abs(Nutrient_upPHY)

          ! Phytoplankton mortality rate (PHYMort, /day)
          CALL PHYMORT_RATE(MortalityFlagellates,tf, PHYmort)

          ! Phytoplankton mortality flux C_PHYmort,N_PHYMort (in mmol C/m3/day or mmol N/m3/day)
          C_PHYmort  = PHYMort * tr(ji,jj,jk,cfl,Kbb)
          N_PHYMort  = PHYMort * tr(ji,jj,jk,nfl,Kbb)

          ! ADJUSTING THE RATE OF CHANGE
          ! phytoplankton C increases by growth,
          ! it decreases by zooplankton grazing (see the zooplankton subroutine) and phytoplankton mortality
          dPCFL(ji,jj,jk) =dPCFL(ji,jj,jk) + GrowthPHY
          dDCFL(ji,jj,jk) =dDCFL(ji,jj,jk) + C_PhyMort + DOC_leakage

          ! phytoplankton N increases by N uptake,
          ! it decreases by zooplankton grazing and phytoplankton mortality
          dPNFL(ji,jj,jk) =dPNFL(ji,jj,jk) + Nutrient_UpPHY
          dDNFL(ji,jj,jk) =dDNFL(ji,jj,jk) + N_PHYMort +  DON_leakage

          ! IF CN ratio of phytoplankton not lower than CNmin, than nitrogen is taken up
          ! Nitrate is taken up by phytoplankton
          IF (Nutrient_UpPHY.gt.0) THEN
            dDNOs(ji,jj,jk) = dDNOs(ji,jj,jk) + Nutrient_UpPHY*Nitrate_upPHY/Nitrogen_UpPHY
            dPDOX(ji,jj,jk) = dPDOX(ji,jj,jk) + Nutrient_UpPHY*Nitrate_upPHY/Nitrogen_UpPHY*ONoxnhsr
            ! Ammonium is taken up by phytoplankton
            dDNHs(ji,jj,jk) = dDNHs(ji,jj,jk) + Nutrient_UpPHY*Ammonium_upPHY/Nitrogen_UpPHY
            dDPHO(ji,jj,jk) = dDPHO(ji,jj,jk) + Nutrient_UpPHY*PNRedfield
#ifdef biodiagrates
            phy_n_uptake(ji,jj,jk) = phy_n_uptake(ji,jj,jk) + Nutrient_UpPHY*(Nitrate_upPHY+Ammonium_upPHY)/Nitrogen_UpPHY
#endif
          ELSE
            ! IF CN ratio of phytoplankton is lower than CNmin, than ammonium is excreted
            dPNHs(ji,jj,jk) = dPNHs(ji,jj,jk) - Nutrient_UpPHY
            dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk) - Nutrient_UpPHY*PNRedfield
#ifdef biodiagrates
           PHY_NHS_excr(ji,jj,jk) = Nutrient_UpPhy
#endif
          ENDIF

          ! As in Anderson and Pondhaven (2003), the phytoplanton mortality increases the pool of POM and DOM with a coefficient of mortdom
          ! for the DOM pool a part is considered as labile (labilefraction) and another part as semi labile (1-labilefraction)
          dPPOC(JI,JJ,JK) = dPPOC(JI,JJ,JK) + (1.0 - mortphydom)*C_PhyMort
          dPPON(JI,JJ,JK) = dPPON(JI,JJ,JK) + (1.0 - mortphydom)*N_PhyMort
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + mortphydom*C_PhyMort*labilefraction
          dPDNL(JI,JJ,JK) = dPDNL(JI,JJ,JK) + mortphydom*N_PhyMort*labilefraction
          dPDCS(JI,JJ,JK) = dPDCS(JI,JJ,JK) + mortphydom*C_PhyMort*(1.0 - labilefraction)
          dPDNS(JI,JJ,JK) = dPDNS(JI,JJ,JK) + mortphydom*N_PhyMort*(1.0 - labilefraction)

          ! As in Anderson and Pondhaven (2003), the DOCL and DONL concentration increases also due to phytoplankton leakage which is considered
          ! to produce only labile DOC and DON
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + DOC_leakage
          dPDNL(JI,JJ,JK) = dPDNL(JI,JJ,JK) + DON_leakage

          ! As in Anderson and Pondhaven (2003), the phytoplankton extra DOC excretion, DOC_extra_excr, increases the pool of DOC (only DOC and not DON)
          ! this extra-excretion is composed of a part (leakage) which is assimilated to leakage and is thus only labile
          ! the remaining part (1-leakage) is composed of a labile (labileextradocexcr) and semi labile part (1-labileextradocexcr)
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + leakagephy*DOC_extra_excr + labileextradocphyexcr*(1.0 - leakagephy)*DOC_extra_excr
          dPDCS(JI,JJ,JK) = dPDCS(JI,JJ,JK) + (1.0 - labileextradocphyexcr)*(1.0 - leakagephy)*DOC_extra_excr

#ifdef chlorophyll_geider
	  ! Adjusting chlorophyll 
          dPCHF(ji,jj,jk) = dPCHF(ji,jj,jk) + growthCHF       
          dDCHF(ji,jj,jk) = dDCHF(ji,jj,jk) + RespChl + MortChl
#endif
#ifdef aggregation
            ! The number of aggregates, POMNOS including  PON increases and decreases with PON
            !      dPAGGI(ji,jj,jk) = dPAGGI(ji,jj,jk) + (1.0 - mortphydom)*N_PhyMort*AGGI(ji,jj,jk)/(PONI(JI,JJ,JK)+PNSI(JI,JJ,JK))
             dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) + (1.0 - mortphydom)*N_PhyMort*tr(ji,jj,jk,agg,Kbb)/(TR(JI,JJ,JK,PON,Kbb))
#ifdef nanquest
            if (isnan(dPAGG(JI,JJ,JK))) then
              write (*,*) '** NAN QUEST ** in Calcflagellates'
              write (*,*) 'ji,jj,jk,PON(JI,JJ,JK),AGG(JI,JJ,JK)',ji,jj,jk,TR(JI,JJ,JK,PON,Kbb),tr(JI,JJ,JK,AGG,Kbb)
              stop
            endif
#endif
#endif
          ! The oxygen concentration increases due to photosynthesis
          dPDOX(ji,jj,jk) = dPDOX(ji,jj,jk) + (GrowthPHY + DOC_extra_excr)*OCr

          ! CO2 production and consumption
          dDDIC(ji,jj,jk) = dDDIC(ji,jj,jk) + GrowthPHY + DOC_extra_excr

          !Diagnostics Store the fluxes
#ifdef primaryprod
          ! initialized here because FL is called first ...
          GPP(ji,jj,jk) = Carbon_UptakePHY
          NPP(ji,jj,jk) = Carbon_UptakePHY - TotalRespirationPHY
#endif

#ifdef biodiagrates
          Nitrogen_Uptake_Flagellates(ji,jj,jk) = Ammonium_upPHY + Nitrate_upPHY
          Carbon_UptakeFlagellates(ji,jj,jk)=Carbon_UptakePHY
          TotalRespirationFlagellates(ji,jj,jk)=TotalRespirationPHY
          PhytoNitrateReduction(ji,jj,jk)=Nutrient_UpPHY*ratio(Nitrate_upPHY,Nitrogen_UpPHY)*ONoxnhsr
          C_Phymort_output(ji,jj,jk)=  C_Phymort
          N_Phymort_output(ji,jj,jk)=  N_Phymort
          DOXprod_PHY_nutuptake(ji,jj,jk)= Nutrient_UpPHY*Nitrate_upPHY/Nitrogen_UpPHY*ONoxnhsr
	  DOXprod_PHY_growth(ji,jj,jk)= (GrowthPHY + DOC_extra_excr)*OCr
#endif
#ifdef biodiagtrophic
	  DOC_leakage_output(ji,jj,jk)=  DOC_leakage
	  DON_leakage_output(ji,jj,jk)=  DON_leakage
	  DOC_extra_excr_output(ji,jj,jk)=DOC_extra_excr
#endif
        endif
  END_3D

  ! OUTPUT VARIABLES
  ! Diagnostics Averaged over entire water column
#ifdef biodiagrates
  call IntegrateVertical(Nitrogen_Uptake_Flagellates,Nitrogen_Uptake_FlagellatesIntegrated,kmm)
  call IntegrateVertical(Carbon_UptakeFlagellates,Carbon_UptakeFlagellatesIntegrated,kmm)
  call IntegrateVertical(TotalRespirationFlagellates,TotalRespirationFlagellatesIntegrated,kmm)
#endif

END SUBROUTINE CalculateFlagellates



!#########################################################################################
!                              3DmesozooF90
!
! 1-D ecosystem model - Biological model of zooplankton
!
! Anderson, 1992,  Modelling the influence of food C:N ratio, and respiration on growth
! and nitrogen excretion in marine zooplankton and bacteria,
! Journal of Plankton Research, vol. 14, n 12, pp. 1645-1671, 1992
!
! Anderson and Hessen (1995), Carbon and Nitrogen limitation in marine copepods,
! Journal of Plankton Research, vol 17, n 2, pp. 317 - 331.
!
! Anderson amd Pondhaven (2003),Non-redfield carbon and nitrogen cycling in the Sarasso Sea :
! pelagic imbalances and export flux, in press in DSR
!
! Implementation: Marilaure Gregoire,                 NIOO-CEME
! Translation into NEMO: Luc Vandenbulcke, ULg / MAST
!
!--------------------------------------------------------------------
! Contains the pelagic submodel, for zooplankton (micro- and meso- zooplankton)
! This model is based on the Anderson and Hensen (1995) model described in JPR and also
! in Anderson and Ponhaven described in DSR I. This model is a nitrogen-carbon balanced model.
! It is assumed that the organic matter is composed of nitrogenous (proteins, amino acids)
! and non nitrogenous compounds (carbohydrate, lipids).
! The hypothesis is that zooplankton preferentially used nitrogeneous compounds for growth and carbon compounds for
! respiration (nore energy in non nitrogenous substrate). Based on this hypothesis and also on the
!fact that nitrogen and carbon have different assimilation efficiencies, the model estimates the zooplankton
!growth, respiration and excretion so as to maintain their internal ratio.
!--------------------------------------------------------------------*
SUBROUTINE CalculateMesozoo(Kbb,Kmm)
  IMPLICIT NONE
  integer, intent(in) :: Kbb, Kmm
  real(wp)  :: &
       tf,                       &  ! Temperature function
       C_ZOOMessyfeeding,N_ZOOMessyfeeding,  &  !Zooplankton messy feeding to the DOM
       grazing_nitrogenZOO,grazing_carbonMesoZoo,      & !Grazing on nitrogen
       C_ZOOIntake,N_ZOOIntake,  & !ZOOPLANKTON CARBON AND NITROGEN INTAKE
       C_ZOOMort,N_ZOOMort, NCrfoodMesoZoo, &  ! Zooplankton mortality,  mmol/m3/day
       C_ZOOEgest,N_ZOOEgest,   &  ! Zooplankton POM Egestion, mmol/m3/day
       C_ZOOResp,N_ZOOExcr,  &  ! Zooplankton respiration, mmol/m3/day and excretion
       ZOOGrowth,    & !Zooplankton growth rate
       FluxPrey_carbon    ! Flux of ingested preys in carbon units (mmolC/m3)
  INTEGER                    :: JI,JJ,JK,S
#ifdef nanquest
  logical      :: isnan
#endif

  ! CALCULATING FLUXES
  DO_3D(0,0,0,0,1,jpk)
        if (tmask(ji,jj,jk).eq.1) then

          ! TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor(ts(ji,jj,jk,jp_tem,Kmm),Q10Zoo)

          ! ZOOPLANKTON
          ! Grazing rate of zooplankton (grazing_carbonZoo(I),NCrfoodZoo(I),mmolC/day)
          CALL GRAZING_RATE(tf,MaxgrazingrateMesoZoo,Half_Saturation_MesoZoo,   &
               Capt_eff_MesoZoo_Flagellates, Capt_eff_MesoZoo_Emiliana,   &
               Capt_eff_MesoZoo_Diatoms,Capt_eff_MesoZoo_MicroZoo,Capt_eff_MesoZoo_MesoZoo, &
               Capt_eff_MesoZoo_pom,Capt_eff_MesoZoo_bac,&
               tr(ji,jj,jk,mes,Kbb),              &
               grazing_carbonMesoZoo,NCrfoodMesoZoo,FluxPrey_carbon, &
               ji,jj,jk,Kbb)

          ! Grazing rate of zooplankton on nitrogen(grazing_nitrogenZoo,NCrfoodZoo(I),mmolN/day)
          grazing_nitrogenZOO = grazing_carbonMesoZoo*NCrfoodMesoZoo

          ! Ingestion rate of zooplankton C_ZOOIntake (mmolC/m3/day),N_ZOOIntake mmolN/m3/day)
          ! Intake of N and C, N_ZOOIntake (mmolN/m3/day), C_ZOOIntake (mmolC/m3/day) are the sum
          ! of grazing on phytoplanktonm zooplankton, detritus and bacteria, less messy feeding losses
          ! altrhought in fact some of these losses occur after passage through the gut.
          C_ZOOIntake = grazing_carbonMesoZoo*(1. - Messy_feeding_MesoZoo)
          N_ZOOIntake = grazing_nitrogenZoo*(1. - Messy_feeding_MesoZoo)

          !Zooplankton messy feeding C_ZOOMessyfeeding,N_ZOOMessyfeeding
          C_ZOOMessyfeeding = grazing_carbonMesoZoo*Messy_feeding_MesoZoo
          N_ZOOMessyfeeding = grazing_nitrogenZoo*Messy_feeding_MesoZoo

          ! Growth rate of zooplankton computed according to Anderson and Hensen
          CALL ZOO_GROWTH_RATE(Ass_Eff_OnNitrogen,Ass_Eff_OnCarbon,NCrMesozoo, &
               efficiency_growth_MesoZoo,NCrfoodMesoZoo, &
               C_ZOOIntake,N_ZOOIntake,N_ZOOExcr,Zoogrowth)

          ! Zooplankton respiration(C_ZOOResp, mmolC/m3/day)
          C_ZOOResp=Ass_Eff_OnCarbon*C_ZOOIntake-Zoogrowth

          ! Egestion rate of zooplankton(C_ZOOEgest,N_ZOOEgest,mmol/day)
          C_ZOOEgest= (1-Ass_Eff_OnCarbon)*C_ZOOIntake
          N_ZOOEgest = (1-Ass_Eff_OnNitrogen)*N_ZOOIntake

          ! Zooplankton mortality rate (C_ZOOMort,N_ZOOMort, /day)
          ! Attention do not forget to add OXYGEN
          CALL MORTALITY_RATE(HalfSatMort_MesoZoo,NLin_Mort_MesoZoo,expmortMesoZoo,&
               DOXsatmort,mortanoxic,tf,tr(ji,jj,jk,mes,Kbb),tr(ji,jj,jk,dox,Kbb),C_ZOOMort)

          ! Mortality in nitrogen units
          N_ZOOMort  = C_ZOOMort * NCrMesoZoo

          ! ADJUSTING THE RATE OF CHANGE
          ! zoooplankton C increases by intake of preys,
          ! it decreases by egestion,respiration,mortality,predation (computed below) with  Zoogrowth =Intake -respiration - egestion
          !and intake = grazing - messyfeeding
          dPmes(ji,jj,jk) = dPmes(ji,jj,jk) + Zoogrowth
          dDmes(ji,jj,jk) = dDmes(ji,jj,jk) + C_ZOOMort

          ! Detritus is formed by the non-assimilated zooplankton grazing,
          ! when phytoplankton dies (see the phytoplankton subroutine), when zooplankton dies
          dPPOC(ji,jj,jk) = dPPOC(ji,jj,jk) + C_ZooEgest+C_ZOOMort
          dPPON(ji,jj,jk) = dPPON(ji,jj,jk) + N_ZooEgest+N_ZOOMort
#ifdef aggregation
          ! The number of aggregates increases with PON
          ! if diatoms aggregate
            !dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) + (N_ZooEgest+N_ZOOMort)*AGG(ji,jj,jk)/(trb(ji,jj,jk,pon)+PNS(ji,jj,jk) )
            dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) + (N_ZooEgest+N_ZOOMort)*tr(ji,jj,jk,agg,Kbb)/(tr(ji,jj,jk,pon,Kbb))
#ifdef nanquest
            if (isnan(dPAGG(JI,JJ,JK))) then
              write(*,*) '** NAN QUEST ** in Calcmesozoo'
              write(*,*) 'ji,jj,jk,PONI(JI,JJ,JK),AGGI(JI,JJ,JK)',ji,jj,jk,TR(JI,JJ,JK,PON,Kbb),tr(JI,JJ,JK,AGG,Kbb)
              call flush(6)
              stop
            endif
#endif
#endif
          ! Dissolved orgaqnic matter is formed by messy feeding, a part (labilefraction) increases the labile pool
          ! while the remaining (1 -labilefrac) increases the semi-labile fraction.
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + labilefraction*C_ZOOMessyfeeding
          dPDNL(ji,jj,jk) = dPDNL(ji,jj,jk) + labilefraction*N_ZOOMessyfeeding
          dPDCS(ji,jj,jk) = dPDCS(ji,jj,jk) + (1.0 - labilefraction)*C_ZOOMessyfeeding
          dPDNS(ji,jj,jk) = dPDNS(ji,jj,jk) + (1.0 - labilefraction)*N_ZOOMessyfeeding

          ! Ammonium is excreyed by zooplankton
          dPNhS(ji,jj,jk) = dPNhS(ji,jj,jk)  + N_ZOOExcr
          dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk)  + N_ZOOExcr*PNRedfield

#ifdef biodiagrates
          zoo_n_excretion(ji,jj,jk) = zoo_n_excretion(ji,jj,jk) + N_ZOOExcr
#endif
          ! Grazing on phytoplankton
          dDCFL(ji,jj,jk) = dDCFL(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Flagellates*tr(ji,jj,jk,cfl,Kbb)
          dDCem(ji,jj,jk) = dDCem(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Emiliana   *tr(ji,jj,jk,cem,Kbb)
          dDCdi(ji,jj,jk) = dDCdi(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Diatoms    *tr(ji,jj,jk,cdi,Kbb)

          dDNFL(ji,jj,jk) = dDNFL(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Flagellates*tr(ji,jj,jk,nfl,Kbb)
          dDNem(ji,jj,jk) = dDNem(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Emiliana   *tr(ji,jj,jk,nem,Kbb)
          dDNdi(ji,jj,jk) = dDNdi(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Diatoms    *tr(ji,jj,jk,ndi,Kbb)

#ifdef chlorophyll_geider
          dDCHF(ji,jj,jk) = dDCHF(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Flagellates*tr(ji,jj,jk,CHF,kbb)
          dDCHE(ji,jj,jk) = dDCHE(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Emiliana   *tr(ji,jj,jk,CHE,kbb)
          dDCHD(ji,jj,jk) = dDCHD(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Diatoms    *tr(ji,jj,jk,CHD,kbb)

#endif
          !When eating diatoms, mesozoo ejects silicate as silicious_Detritus
          dPSID(ji,jj,jk) = dPSID(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Diatoms*tr(ji,jj,jk,ndi,Kbb)*SiNrDiatoms

          ! Grazing on zoooplankton
          dDmic(ji,jj,jk) = dDmic(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_MicroZoo*tr(ji,jj,jk,mic,Kbb)

          !  Grazing on detritus
          dDPOC(ji,jj,jk) = dDPOC(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_POM*tr(ji,jj,jk,poc,Kbb)
          dDPON(ji,jj,jk) = dDPON(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_POM*tr(ji,jj,jk,pon,Kbb)

          ! Grazing on Bacteria
          dDBAC(ji,jj,jk) = dDBAC(ji,jj,jk) + grazing_carbonMesoZOO/FluxPrey_carbon*Capt_eff_MesoZoo_BAC*tr(ji,jj,jk,bac,Kbb)

          ! POMNOS decreases by the grazing
          ! if diatoms form aggregate
#ifdef aggregation
            !dDAGG(ji,jj,jk) = dDAGG(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_POM*PONI(ji,jj,jk)*AGG(ji,jj,jk)/(PON(ji,jj,jk)+PNS(ji,jj,jk))
            dDAGG(ji,jj,jk) = dDAGG(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_POM*tr(ji,jj,jk,AGG,Kbb)
#ifdef nanquest
            if (isnan(dDAGG(JI,JJ,JK))) then
              write (*,*) '** NAN QUEST ** in CalcMESO'
              write (*,*) 'ji,jj,jk,PONI(JI,JJ,JK),AGGI(JI,JJ,JK)',ji,jj,jk,TR(JI,JJ,JK,PON,Kbb),tr(JI,JJ,JK,AGG,Kbb)
              call flush(6)
              stop
             endif
#endif
#endif

          ! DOX decreases due to respiration of zooplankton
          dDDOX(ji,jj,jk) = dDDOX(ji,jj,jk) + C_ZOOResp*OCr
          dPDIC(ji,jj,jk) = dPDIC(ji,jj,jk)+ C_ZOOResp

          ! CO2 production and consumption
          ! dDIC=dDIC+C_ZooResp
          ! CALL UpdateCO2(I, - GrowthPHY)

#ifdef biodiagrates
          !  Diagnostics Store the fluxes
          Totalrespiration_ZOO(ji,jj,jk) = Totalrespiration_ZOO(ji,jj,jk) + C_ZOORESP
          SID_creation(ji,jj,jk) = SID_creation(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Diatoms*tr(ji,jj,jk,ndi,Kbb)*SiNrDiatoms
#endif
#ifdef biodiagtrophic
          !  Diagnostics Store the fluxes
          PHY_TO_ZOO(ji,jj,jk) = phy_to_Zoo(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Flagellates*tr(ji,jj,jk,cfl,Kbb) &
               + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Emiliana   *tr(ji,jj,jk,cem,Kbb)                                        &
               + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_Diatoms    *tr(ji,jj,jk,cdi,Kbb)
          bac_to_Zoo(ji,jj,jk) = bac_to_Zoo(ji,jj,jk) + grazing_carbonMesoZOO/FluxPrey_carbon*Capt_eff_MesoZoo_BAC * tr(ji,jj,jk,bac,Kbb)
          POC_to_ZOO(ji,jj,jk) = POC_to_ZOO(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_POM * tr(ji,jj,jk,poc,Kbb)
          PON_to_ZOO(ji,jj,jk) = PON_to_ZOO(ji,jj,jk) + grazing_carbonMesoZoo/FluxPrey_carbon*Capt_eff_MesoZoo_POM * tr(ji,jj,jk,pon,Kbb)
          ZOO_TO_POC(ji,jj,jk)= ZOO_TO_POC(ji,jj,jk) + C_ZooEgest + C_ZOOMort
          ZOO_TO_PON(ji,jj,jk)= ZOO_TO_PON(ji,jj,jk) + N_ZooEgest + N_ZOOMort
          C_ZOOMessyfeeding_output(ji,jj,jk)= C_ZOOMessyfeeding_output(ji,jj,jk) + C_ZOOMessyfeeding
          N_ZOOMessyfeeding_output(ji,jj,jk)= N_ZOOMessyfeeding_output(ji,jj,jk) + N_ZOOMessyfeeding
#endif
        end if
  END_3D

  ! OUTPUT VARIABLES
#ifdef biodiagrates
  call IntegrateVertical(Totalrespiration_ZOO,Totalrespiration_zooIntegrated,kmm)
#endif
#ifdef biodiagtrophic
#ifndef gelatinous
  call IntegrateVertical(phy_to_ZOO,phy_to_ZOOIntegrated,kmm)
  call IntegrateVertical(bac_to_ZOO,bac_to_ZOOIntegrated,kmm)
  call IntegrateVertical(poc_to_ZOO,poc_to_ZOOIntegrated,kmm)
#endif
#endif

END SUBROUTINE CalculateMesozoo



!#########################################################################################
!                              3DmicrozooF90
!
! 1-D ecosystem model - Biological model of zooplankton
!
! Anderson, 1992,  Modelling the influence of food C:N ratio, and respiration on growth
! and nitrogen excretion in marine zooplankton and bacteria,
! Journal of Plankton Research, vol. 14, n 12, pp. 1645-1671, 1992
!
! Anderson and Hessen (1995), Carbon and Nitrogen limitation in marine copepods,
! Journal of Plankton Research, vol 17, n 2, pp. 317 - 331.
!
! Anderson amd Pondhaven (2003),Non-redfield carbon and nitrogen cycling in the Sarasso Sea :
! pelagic imbalances and export flux, in press in DSR
!
! Implementation: Marilaure Gregoire,                 NIOO-CEME
!
! Contains the pelagic submodel, for zooplankton (micro- and meso- zooplankton)
! This model is based on the Anderson and Hensen (1995) model described in JPR and also
! in Anderson and Ponhaven described in DSR I. This model is a nitrogen-carbon balanced model.
! It is assumed that the organic matter is composed of nitrogenous (proteins, amino acids)
! and non nitrogenous compounds (carbohydrate, lipids).
! The hypothesis is that zooplankton preferentially used nitrogeneous compounds for growth and carbon compounds for
! respiration (nore energy in non nitrogenous substrate). Based on this hypothesis and also on the
! fact that nitrogen and carbon have different assimilation efficiencies, the model estimates the zooplankton
! growth, respiration and excretion so as to maintain their internal ratio.
!--------------------------------------------------------------------*
SUBROUTINE CalculateMicroZoo(Kbb,Kmm)
  IMPLICIT NONE
  integer, intent(in) :: Kbb, Kmm
  real(wp)  :: &
       tf,                       &  ! Temperature function
       C_ZOOMessyfeeding,N_ZOOMessyfeeding,  &  !Zooplankton messy feeding to the DOM
       grazing_nitrogenZOO,NCrfoodMicroZoo,grazing_carbonMicroZoo,     & !Grazing on nitrogen
       C_ZOOIntake,N_ZOOIntake,  & !ZOOPLANKTON CARBON AND NITROGEN INTAKE
       C_ZOOMort,N_ZOOMort,  &  ! Zooplankton mortality,  mmol/m3/day
       C_ZOOEgest,N_ZOOEgest,   &  ! Zooplankton POM Egestion, mmol/m3/day
       C_ZOOResp,N_ZOOExcr,  &  ! Zooplankton respiration, mmol/m3/day and excretion
       ZOOGrowth,    & !Zooplankton growth rate
       FluxPrey_carbon     ! Flux of ingested preys in carbon units (mmolC/m3)
  INTEGER                    :: JI,JJ,JK,S
#ifdef nanquest
  logical      :: isnan
#endif

  ! CALCULATING FLUXES
  DO_3D(0,0,0,0,1,jpk)
        if (tmask(ji,jj,jk).eq.1) then

          ! TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor(ts(ji,jj,jk,jp_tem,Kmm),Q10Zoo)

          ! ZOOPLANKTON
          ! Grazing rate of zooplankton (grazing_carbonZoo(I),NCrfoodZoo(I),mmolC/day)
          CALL GRAZING_RATE(tf,MaxgrazingrateMicroZoo,Half_Saturation_MicroZoo,   &
                   Capt_eff_MicroZoo_Flagellates, Capt_eff_MicroZoo_Emiliana,   &
                   Capt_eff_MicroZoo_Diatoms,Capt_eff_MicroZoo_MicroZoo,Capt_eff_MicroZoo_MesoZoo, &
                   Capt_eff_MicroZoo_pom,Capt_eff_MicroZoo_bac,&
                   tr(ji,jj,jk,mic,Kbb),              &
                   grazing_carbonMicroZoo,NCrfoodMicroZoo,FluxPrey_carbon, &
                   ji,jj,jk,Kbb)

          ! Grazing rate of zooplankton on nitrogen(grazing_nitrogenZoo,NCrfoodZoo(I),mmolN/day)
          grazing_nitrogenZOO = grazing_carbonMicroZoo*NCrfoodMicroZoo

          ! Ingestion rate of zooplankton C_ZOOIntake (mmolC/m3/day),N_ZOOIntake mmolN/m3/day)
          ! Intake of N and C, N_ZOOIntake (mmolN/m3/day), C_ZOOIntake (mmolC/m3/day) are the sum
          ! of grazing on phytoplanktonm zooplankton, detritus and bacteria, less messy feeding losses
          ! altrhought in fact some of these losses occur after passage through the gut.
          C_ZOOIntake = grazing_carbonMicroZoo*(1. - Messy_feeding_MicroZoo)
          N_ZOOIntake = grazing_nitrogenZoo*(1. - Messy_feeding_MicroZoo)

          !Zooplankton messy feeding C_ZOOMessyfeeding,N_ZOOMessyfeeding
          C_ZOOMessyfeeding = grazing_carbonMicroZoo*Messy_feeding_MicroZoo
          N_ZOOMessyfeeding = grazing_nitrogenZoo*Messy_feeding_MicroZoo

          ! Growth rate of zooplankton computed according to Anderson and Hensen
          CALL ZOO_GROWTH_RATE(Ass_Eff_OnNitrogen,Ass_Eff_OnCarbon,NCrMicroZoo, &
               efficiency_growth_MicroZoo,NCrfoodMicroZoo, &
               C_ZOOIntake,N_ZOOIntake,N_ZOOExcr,Zoogrowth)

          ! Zooplankton respiration(C_ZOOResp, mmolC/m3/day)
          C_ZOOResp=Ass_Eff_OnCarbon*C_ZOOIntake-Zoogrowth

          ! Egestion rate of zooplankton(C_ZOOEgest,N_ZOOEgest,mmol/day)
          C_ZOOEgest= (1-Ass_Eff_OnCarbon)*C_ZOOIntake
          N_ZOOEgest = (1-Ass_Eff_OnNitrogen)*N_ZOOIntake

          ! Zooplankton mortality rate (C_ZOOMort,N_ZOOMort, /day)
          ! Attention do not forget to add OXYGEN
          CALL MORTALITY_RATE(HalfSatMort_MicroZoo,NLin_Mort_MicroZoo,expmortMicroZoo,&
                   DOXsatmort,mortanoxic,tf,tr(ji,jj,jk,mic,Kbb),tr(ji,jj,jk,dox,Kbb),C_ZOOMort)

          ! Mortality in nitrogen units
          N_ZOOMort  = C_ZOOMort * NCrMicroZoo

          ! ADJUSTING THE RATE OF CHANGE
          ! zoooplankton C increases by intake of preys,
          ! it decreases by egestion,respiration,mortality,predation (computed below) with  Zoogrowth =Intake -respiration - egestion
          !and intake = grazing - messyfeeding
          dPMIC(ji,jj,jk) = dPMIC(ji,jj,jk) + Zoogrowth
          dDMIC(ji,jj,jk) = dDMIC(ji,jj,jk) + C_ZOOMort

          ! Detritus is formed by the non-assimilated zooplankton grazing,
          ! when phytoplankton dies (see the phytoplankton subroutine), when zooplankton dies
          dPPOC(ji,jj,jk) = dPPOC(ji,jj,jk) + C_ZooEgest+C_ZOOMort
          dPPON(ji,jj,jk) = dPPON(ji,jj,jk) + N_ZooEgest+N_ZOOMort
#ifdef aggregation
          ! The number of aggregates increases with PON
          ! if diatoms aggregate
            !dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) + (N_ZooEgest+N_ZOOMort)*AGG(ji,jj,jk)/(PON(ji,jj,jk)+PNS(ji,jj,jk))
            dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) + (N_ZooEgest+N_ZOOMort)*tr(ji,jj,jk,agg,Kbb)/(tr(ji,jj,jk,pon,Kbb))
#ifdef nanquest
            if (isnan(dPAGG(JI,JJ,JK))) then
              write (*,*) '** NAN QUEST ** in Calcmicrozoo'
              write (*,*) 'ji,jj,jk,PON(JI,JJ,JK),AGG(JI,JJ,JK)',ji,jj,jk,TR(JI,JJ,JK,PON,Kbb),tr(JI,JJ,JK,AGG,Kbb)
              call flush(6)
              stop
            endif
#endif
#endif
          ! Dissolved orgaqnic matter is formed by messy feeding, a part (labilefraction) increases the labile pool
          ! while the remaining (1 -labilefrac) increases the semi-labile fraction.
          dPDCL(ji,jj,jk) = dPDCL(ji,jj,jk) + labilefraction*C_ZOOMessyfeeding
          dPDNL(ji,jj,jk) = dPDNL(ji,jj,jk) + labilefraction*N_ZOOMessyfeeding
          dPDCS(ji,jj,jk) = dPDCS(ji,jj,jk) + (1.0 - labilefraction)*C_ZOOMessyfeeding
          dPDNS(ji,jj,jk) = dPDNS(ji,jj,jk) + (1.0 - labilefraction)*N_ZOOMessyfeeding

          ! Ammonium is excreyed by zooplankton
          dPNHS(ji,jj,jk) = dPNHS(ji,jj,jk)  + N_ZOOExcr
          dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk)  + N_ZOOExcr*PNRedfield
#ifdef biodiagrates
          zoo_n_excretion(ji,jj,jk) =   zoo_n_excretion(ji,jj,jk) + N_ZOOExcr
#endif

          ! Grazing on phytoplankton
          dDCFL(ji,jj,jk) = dDCFL(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Flagellates*tr(ji,jj,jk,cfl,Kbb)
          dDCEM(ji,jj,jk) = dDCEM(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Emiliana   *tr(ji,jj,jk,cem,Kbb)
          dDCdi(ji,jj,jk) = dDCdi(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Diatoms    *tr(ji,jj,jk,cdi,Kbb)

          dDnfl(ji,jj,jk) = dDnfl(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Flagellates*tr(ji,jj,jk,nfl,Kbb)
          dDnem(ji,jj,jk) = dDnem(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Emiliana   *tr(ji,jj,jk,nem,Kbb)
          dDndi(ji,jj,jk) = dDndi(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Diatoms    *tr(ji,jj,jk,ndi,Kbb)

#ifdef chlorophyll_geider
          dDCHF(ji,jj,jk) = dDCHF(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Flagellates*tr(ji,jj,jk,CHF,kbb)
          dDCHE(ji,jj,jk) = dDCHE(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Emiliana   *tr(ji,jj,jk,CHE,kbb)
          dDCHD(ji,jj,jk) = dDCHD(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Diatoms    *tr(ji,jj,jk,CHD,kbb)

#endif
          !When eating diatoms, micro ejects silicate as silicious_Detritus
          dPSID(ji,jj,jk) = dPSID(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Diatoms*tr(ji,jj,jk,ndi,Kbb)*SiNrDiatoms

          !  Grazing on detritus
          dDPOC(ji,jj,jk) = dDPOC(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_POM*tr(ji,jj,jk,poc,Kbb)
          dDPON(ji,jj,jk) = dDPON(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_POM*tr(ji,jj,jk,pon,Kbb)

          ! Grazing on Bacteria
          dDBAC(ji,jj,jk) = dDBAC(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_BAC*tr(ji,jj,jk,bac,Kbb)
#ifdef aggregation
          ! POMNOS decreases by the grazing
            !dDAGG(ji,jj,jk) = dDAGG(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_POM*PON(ji,jj,jk)*AGG(ji,jj,jk)/(PON(ji,jj,jk)+PNS(ji,jj,jk))
            dDAGG(ji,jj,jk) = dDAGG(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_POM*tr(ji,jj,jk,AGG,Kbb)
#ifdef nanquest
            if (isnan(dDAGG(JI,JJ,JK))) then
              write (*,*) '** NAN QUEST ** in CalcMIC'
              write (*,*) 'ji,jj,jk,PONI(JI,JJ,JK),AGGI(JI,JJ,JK)',ji,jj,jk,TR(JI,JJ,JK,PON,Kbb),tr(JI,JJ,JK,AGG,Kbb)
              call flush(6)
              stop
            endif
#endif
#endif
          ! DOX decreases due to respiration of zooplankton
          dDDOX(ji,jj,jk) = dDDOX(ji,jj,jk) + C_ZOOResp*OCr
          dPDIC(ji,jj,jk) = dPDIC(ji,jj,jk) + C_ZOOResp

          ! diagnostics
#ifdef biodiagrates
          !mic est appele d abord par updatebioflux,donc on initialise ici ce diagnostique cumulatif pour les zoo
          Totalrespiration_ZOO(ji,jj,jk) = C_ZOORESP
          SID_creation(ji,jj,jk) = SID_creation(ji,jj,jk) + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Diatoms*tr(ji,jj,jk,ndi,Kbb)*SiNrDiatoms
#endif
#ifdef biodiagtrophic
          !  Diagnostics Store the fluxes
          phy_to_Zoo(ji,jj,jk) = grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Flagellates*tr(ji,jj,jk,cfl,Kbb)&
               + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Emiliana   *tr(ji,jj,jk,cem,Kbb)&
               + grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_Diatoms    *tr(ji,jj,jk,cdi,Kbb)
          bac_to_Zoo(ji,jj,jk) = grazing_carbonMicroZOO/FluxPrey_carbon*Capt_eff_MicroZoo_BAC * tr(ji,jj,jk,bac,Kbb)
          POC_to_ZOO(ji,jj,jk) = grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_POM * tr(ji,jj,jk,poc,Kbb)
          PON_to_ZOO(ji,jj,jk) = grazing_carbonMicroZoo/FluxPrey_carbon*Capt_eff_MicroZoo_POM * tr(ji,jj,jk,pon,Kbb)
          ZOO_TO_POC(ji,jj,jk)= C_ZooEgest + C_ZOOMort
          ZOO_TO_PON(ji,jj,jk)= N_ZooEgest + N_ZOOMort
          C_ZOOMessyfeeding_output(ji,jj,jk)= C_ZooMessyfeeding
          N_ZOOMessyfeeding_output(ji,jj,jk)= N_ZOOMessyfeeding
#endif

        end if
  END_3D

END SUBROUTINE CalculateMicroZoo



!#########################################################################################
!                              3DGELATINOUS
!
! Contains the pelagic submodel, for gelatinous (noctiluca, aurelia, mnemiopsis) based on
! the model of Lancelot et al. (2002) published in ESCS. Parameters have to be adjusted.
! The gelatinous model differs from the classic zooplankton model essentially by the term of grazing.
! The grazing rate increases linearly with the prey concentration as in the litterature.
! Otherwise, the mortality is a non-linear term as fro the zooplankton. This term will have to be thoroughly calibrated since it is the
! closure of the model. The egestion is assumed to be the non-assimilated food which directly goes to POM. Respiration
! is considered to be the part of the assimilated food which is not used for the growth (as for the zooplankton).
! as in Lancelot et al. (2002), each gelatinous group has no predator. Noctiluca feeds on the three phytoplnakton group,
! the microzooplaqnkton, bacteria and the two POMS. Aurelia and Mnemiopsis feed both only on mesozoo.
! A flux of adjustment considered as an extra-respiration or an excretion is computed to maimtain constant the N:C ratio.
! This part has to be improved on the model of the zooplankton
!--------------------------------------------------------------------*
#ifdef gelatinous
SUBROUTINE CalculateGelatinous(Kbb,Kmm)
  IMPLICIT NONE
  integer, intent(in) :: Kbb, Kmm
  real(wp) ::  &
       tf,                                         & ! Temperature function
       C_ZOOMort,N_ZOOMort,                        & ! Zooplankton mortality,  mmol/m3/day
       C_ZOOEgest,N_ZOOEgest,                      & ! Zooplankton POM Egestion, mmol/m3/day
       C_ZOOResp,                                  & ! Zooplankton respiration, mmol/m3/day
       C_ZOOPred,N_ZOOPred,                        & ! Predation on ZOO, mmol/m3/day
       FluxPrey_carbon, NCrfoodGelatinous,         & ! Flux of ingested preys in carbon units (mmolC/m3)
       grazing_nitrogen, grazing_carbonGelatinous, & !grazing in nitrogen
       NCrzootest,                                 & ! N:C ratio of the zooplankton before adjustment
       C_ZOOAdjust,N_ZOOAdjust                       ! Flux of adjustment to maintain constant the ZOO NCr  GrGelMeso
  INTEGER                    :: JI,JJ,JK,S

  ! CALCULATING FLUXES
  DO_3D(0,0,0,0,1,jpk)  
        if (tmask(ji,jj,jk).eq.1) then

          ! TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor(ts(ji,jj,jk,jp_tem,Kmm),Q10Gelatinous)

          ! ZOOPLANKTON
          ! Grazing rate of zooplankton (grazing_carbonZoo(I),NCrfoodZoo(I),mmolC/day)
          CALL GELATINOUS_GRAZING_RATE(tf,MaxgrazingrateGelatinous,threshold_feeding_Gelatinous,   &
                   Capt_eff_Gelatinous_Flagellates,Capt_eff_Gelatinous_Emiliana,Capt_eff_Gelatinous_Diatoms,&
                   Capt_eff_Gelatinous_MicroZoo,Capt_eff_Gelatinous_MesoZoo, &
                   Capt_eff_Gelatinous_pom, &
                   tr(ji,jj,jk,GEL,Kbb),              &
                   grazing_carbonGelatinous,NCrfoodGelatinous,FluxPrey_carbon, &
                   ji,jj,jk,Kbb)

          grazing_nitrogen=grazing_carbonGelatinous*NCrfoodGelatinous

          ! Egestion rate of zooplankton(C_ZOOEgest,N_ZOOEgest,mmol/day)
          CALL GELATINOUS_EGESTION_RATE(Ass_Eff_Gelatinous,grazing_CarbonGelatinous,C_ZOOEgest)

          N_ZOOEgest = C_ZOOEgest*NCrfoodGelatinous

          ! Zooplankton respiration(C_ZOOResp, mmolC/m3/day)
          CALL GELATINOUS_RESPIRATION_RATE(tf,Ass_Eff_Gelatinous,efficiency_growth_Gelatinous,&
                   basal_Resp_Gelatinous,tr(JI,JJ,JK,GEL,Kbb),        &
                   grazing_carbonGelatinous,           &
                   C_ZOOResp)

          ! Zooplankton mortality rate (C_ZOOMort,N_ZOOMort, /day)
          ! Attention do not forget to add OXYGEN
          CALL GELATINOUS_MORTALITY_RATE(HalfSatMort_Gelatinous,NLin_Mort_Gelatinous,expmortGelatinous, &
                   DOXsatmort,mortanoxic,tf,tr(ji,jj,jk,GEL,Kbb),tr(ji,jj,jk,dox,Kbb),C_ZOOMort)


          ! Mortality in nitrogen units
          N_ZOOMort  = C_ZOOMort * NCrGelatinous

          !Computes the nitrogen/carbon fluxes necessary to conserve the NCrzoo (C_ZOOAdjust,N_ZOOAdjust)
          !The adjustment is realised through an extra-respiration or an excretion term on the model of the zooplankton in ERSEM
          ! in this case excretion can be zero. We have to compute the N:C ratio of zooplankton with the computed fluxes
          NCrzootest=(grazing_carbonGelatinous*NCrfoodGelatinous-N_ZooEgest)/&
                   (grazing_carbonGelatinous-C_ZOOEgest-C_ZooResp)

          if (NCrzootest> NCrGelatinous) then
              C_ZOOAdjust=0
              N_ZOOAdjust=(grazing_carbonGelatinous*NCrfoodGelatinous-N_ZooEgest)-&
                   (grazing_carbonGelatinous-C_ZOOEgest-C_ZooResp)*NCrGelatinous
          else
              N_ZOOAdjust=0
              C_ZOOAdjust=(grazing_carbonGelatinous-C_ZOOEgest-C_ZooResp)- &
                   (grazing_carbonGelatinous*NCrfoodGelatinous-N_ZooEgest)/NcrGelatinous
          endif

          ! ADJUSTING THE RATE OF CHANGE
          ! zoooplankton C increases by intake of preys,
          ! it decreases by egestion,respiration,mortality,predation,adjustement
          dPGEL(ji,jj,jk) = dPGEL(ji,jj,jk) + grazing_carbonGelatinous
          dDGEL(ji,jj,jk) = dDGEL(ji,jj,jk) + C_ZooEgest + C_ZooResp + C_ZOOMort + C_ZooAdjust

          ! Detritus is formed by the non-assimilated zooplankton grazing,
          ! when phytoplankton dies, when zooplankton dies
          dPPOC(ji,jj,jk) = dPPOC(ji,jj,jk) + C_ZooEgest+C_ZOOMort
          dPPON(ji,jj,jk) = dPPON(ji,jj,jk) + N_ZooEgest+N_ZOOMort
#ifdef aggregation
          ! The number of aggregates increases with PON
          ! if diatoms can form aggregate
            !dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) + (N_ZooEgest+N_ZOOMort)*AGG(ji,jj,jk)/(PON(ji,jj,jk)+PNS(ji,jj,jk))
            dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) + (N_ZooEgest+N_ZOOMort)*tr(ji,jj,jk,agg,Kbb)/(tr(ji,jj,jk,pon,Kbb))
#endif
          ! Ammonium is excreyed by zooplankton
          dPNHS(ji,jj,jk) = dPNHS(ji,jj,jk) + N_ZOOAdjust
          dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk)  + N_ZOOAdjust*PNRedfield
          ! Grazing on zoooplankton
          dDMIC(ji,jj,jk) = dDMIC(ji,jj,jk) + grazing_carbonGelatinous/FluxPrey_carbon*Capt_eff_Gelatinous_microzoo*tr(ji,jj,jk,mic,Kbb)
          dDMES(ji,jj,jk) = dDMES(ji,jj,jk) + grazing_carbonGelatinous/FluxPrey_carbon*Capt_eff_Gelatinous_mesozoo *tr(ji,jj,jk,mes,Kbb)
          ! Grazing on detritus
          dDPOC(ji,jj,jk) = dDPOC(ji,jj,jk) + grazing_carbonGelatinous/FluxPrey_carbon*Capt_eff_Gelatinous_POM*tr(ji,jj,jk,poc,Kbb)
          dDPON(ji,jj,jk) = dDPON(ji,jj,jk) + grazing_carbonGelatinous/FluxPrey_carbon*Capt_eff_Gelatinous_pom*tr(ji,jj,jk,pon,Kbb)
#ifdef aggregation
          ! POMNOS decreases by the grazing
          ! If diatoms can form aggregate
            !dDAGG(ji,jj,jk)=dDAGG(ji,jj,jk) + grazing_carbonGelatinous/FluxPrey_carbon*Capt_eff_Gelatinous_pom*PON(ji,jj,jk)*AGG(ji,jj,jk)/(PON(ji,jj,jk)+PNS(ji,jj,jk))
            dDAGG(ji,jj,jk)=dDAGG(ji,jj,jk) + grazing_carbonGelatinous/FluxPrey_carbon*Capt_eff_Gelatinous_pom*tr(ji,jj,jk,AGG,Kbb)
#endif
          ! DOX decreases due to respiration of zooplankton
          dDDOX(ji,jj,jk) = dDDOX(ji,jj,jk) + (C_ZOOResp + C_ZooAdjust)*OCr
          dPDIC(ji,jj,jk) = dPDIC(ji,jj,jk) + (C_ZOOResp + C_ZooAdjust)

          ! diagnostics
#ifdef biodiagrates
          TotalRespiration_Gel(ji,jj,jk)=C_ZOOResp + C_ZooAdjust
#endif
#ifdef biodiagtrophic
      !  Diagnostics Store the fluxes
      POC_to_ZOO(ji,jj,jk) = POC_to_ZOO(ji,jj,jk) + grazing_carbonGelatinous/FluxPrey_carbon*Capt_eff_Gelatinous_POM*tr(ji,jj,jk,poc,Kbb)
      PON_to_ZOO(ji,jj,jk) = PON_to_ZOO(ji,jj,jk) + grazing_carbonGelatinous/FluxPrey_carbon*Capt_eff_Gelatinous_POM*tr(ji,jj,jk,pon,Kbb)
#endif


        end if
  END_3D

END SUBROUTINE CalculateGelatinous
#endif


!#########################################################################################
!                              3DNOCTILUCA
!
! Contains the pelagic submodel, for gelatinous (noctiluca, aurelia, mnemiopsis) based on
! the model of Lancelot et al. (2002) published in ESCS. Parameters have to be adjusted.
! The gelatinous model differs from the classic zooplankton model essentially by the term of grazing.
! The grazing rate increases linearly with the prey concentration as in the litterature.
! Otherwise, the mortality is a non-linear term as fro the zooplankton. This term will have to be thoroughly calibrated since it is the
! closure of the model. The egestion is assumed to be the non-assimilated food which directly goes to POM. Respiration
! is considered to be the part of the assimilated food which is not used for the growth (as for the zooplankton).
! as in Lancelot et al. (2002), each gelatinous group has no predator. Noctiluca feeds on the three phytoplnakton group,
! the microzooplaqnkton, bacteria and the two POMS. Aurelia and Mnemiopsis feed both only on mesozoo.
! A flux of adjustment considered as an extra-respiration or an excretion is computed to maimtain constant the N:C ratio.
! This part has to be improved on the model of the zooplankton
!--------------------------------------------------------------------*
#ifdef gelatinous
SUBROUTINE CalculateNoctiluca(Kbb,Kmm)
  IMPLICIT NONE
  integer, intent(in) :: Kbb, Kmm
  real(wp) ::  &
       tf,                                         &  ! Temperature function
       C_ZOOMort,N_ZOOMort,                        &  ! Zooplankton mortality,  mmol/m3/day
       C_ZOOEgest,N_ZOOEgest, NCrfoodNoctiluca,    &  ! Zooplankton POM Egestion, mmol/m3/day
       C_ZOOResp, grazing_carbonNoctiluca, &  ! Zooplankton respiration, mmol/m3/day
       C_ZOOPred,N_ZOOPred,                &  ! Predation on ZOO, mmol/m3/day
       FluxPrey_carbon,                    &  ! Flux of ingested preys in carbon units (mmolC/m3)
       NCrzootest,                         &  ! N:C ratio of the zooplankton before adjustment
       grazing_nitrogen,                          &  !grazing in nitrogen
       C_ZOOAdjust,N_ZOOAdjust                ! Flux of adjustment to maintain constant the ZOO NCr
  INTEGER  :: JI,JJ,JK

  ! CALCULATING FLUXES
  DO_3D(0,0,0,0,1,jpk)  
        if (tmask(ji,jj,jk).eq.1) then

          ! TEMPERATURE EFFECT on rates (all rates are defined at 20 dg C)
          tf = Q10Factor (ts(ji,jj,jk,jp_tem,Kmm),Q10Zoo)

          ! ZOOPLANKTON
          ! Grazing rate of zooplankton (grazing_carbonZoo(I),NCrfoodZoo(I),mmolC/day)
          CALL GELATINOUS_GRAZING_RATE(tf,MaxgrazingrateNoctiluca,threshold_feeding_Noctiluca,   &
                   Capt_eff_Noctiluca_Flagellates,Capt_eff_Noctiluca_Emiliana,Capt_eff_Noctiluca_Diatoms,&
                   Capt_eff_Noctiluca_MicroZoo,Capt_eff_Noctiluca_MesoZoo, &
                   Capt_eff_Noctiluca_pom, &
                   tr(ji,jj,jk,NOC,Kbb),              &
                   grazing_carbonNoctiluca,NCrfoodNoctiluca,FluxPrey_carbon, &
                   ji,jj,jk,Kbb)

          grazing_nitrogen=grazing_carbonNoctiluca*NCrfoodNoctiluca

          ! Egestion rate of zooplankton(C_ZOOEgest,N_ZOOEgest,mmol/day)
          CALL GELATINOUS_EGESTION_RATE(Ass_Eff_Noctiluca,grazing_CarbonNoctiluca,C_ZOOEgest)
          N_ZOOEgest = C_ZOOEgest*NCrfoodNoctiluca
          
          ! Zooplankton respiration(C_ZOOResp, mmolC/m3/day)
          CALL GELATINOUS_RESPIRATION_RATE(tf,Ass_Eff_Noctiluca,efficiency_growth_Noctiluca,&
               basal_Resp_Noctiluca,tr(ji,jj,jk,NOC,Kbb),        &
               grazing_carbonNoctiluca,           &
               C_ZOOResp)

          ! Zooplankton mortality rate (C_ZOOMort,N_ZOOMort, /day)
          ! Attention do not forget to add OXYGEN
          CALL GELATINOUS_MORTALITY_RATE(HalfSatMort_Noctiluca,NLin_Mort_Noctiluca,expmortNoctiluca, &
                   DOXsatmort,mortanoxic,tf,tr(ji,jj,jk,NOC,Kbb),tr(ji,jj,jk,dox,Kbb),C_ZOOMort)

          ! Mortality in nitrogen units
          N_ZOOMort  = C_ZOOMort* NCrNoctiluca

          !Computes the nitrogen/carbon fluxes necessary to conserve the NCrzoo (C_ZOOAdjust,N_ZOOAdjust)
          !The adjustment is realised through an extra-respiration or an excretion term on the model of the zooplankton in ERSEM
          ! in this case excretion can be zero. We have to compute the N:C ratio of zooplankton with the computed fluxes
          NCrzootest=(grazing_carbonNoctiluca*NCrfoodNoctiluca-N_ZooEgest)/&
                   (grazing_carbonNoctiluca-C_ZOOEgest-C_ZooResp)

          if (NCrzootest> NCrNoctiluca) then
             C_ZOOAdjust=0
             N_ZOOAdjust=(grazing_carbonNoctiluca*NCrfoodNoctiluca-N_ZooEgest)-&
                  (grazing_carbonNoctiluca-C_ZOOEgest-C_ZooResp)*NCrNoctiluca
          else
             N_ZOOAdjust=0
             C_ZOOAdjust=(grazing_carbonNoctiluca-C_ZOOEgest-C_ZooResp)- &
                  (grazing_carbonNoctiluca*NCrfoodNoctiluca-N_ZooEgest)/NcrNoctiluca
          endif

          ! ADJUSTING THE RATE OF CHANGE

          ! zoooplankton C increases by intake of preys,
          ! it decreases by egestion,respiration,mortality,predation,adjustement
          dPNOC(ji,jj,jk) = dPNOC(ji,jj,jk) + grazing_carbonNoctiluca
          dDNOC(ji,jj,jk) = dDNOC(ji,jj,jk) + C_ZooEgest + C_ZooResp + C_ZOOMort + C_ZooAdjust

          ! Detritus is formed by the non-assimilated zooplankton grazing,
          ! when phytoplankton dies, when zooplankton dies
          dPPOC(ji,jj,jk) = dPPOC(ji,jj,jk) + C_ZooEgest+C_ZOOMort
          dPPON(ji,jj,jk) = dPPON(ji,jj,jk) + N_ZooEgest+N_ZOOMort

          ! The number of aggregates increases with PON
          ! if diatoms can form aggregate
#ifdef aggregation
            !dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) +(N_ZooEgest+N_ZOOMort)*AGG(ji,jj,jk)/(PON(ji,jj,jk) +PNS(ji,jj,jk))
            dPAGG(ji,jj,jk) = dPAGG(ji,jj,jk) +(N_ZooEgest+N_ZOOMort)*tr(ji,jj,jk,agg,Kbb)/(tr(ji,jj,jk,pon,Kbb))
#endif
          ! Ammonium is excreted by zooplankton
          dPNHS(ji,jj,jk) = dPNHS(ji,jj,jk) + N_ZOOAdjust
          dPPHO(ji,jj,jk) = dPPHO(ji,jj,jk) + N_ZOOAdjust*PNRedfield

          ! Grazing on zoooplankton
          dDMIC(ji,jj,jk) = dDMIC(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_microzoo*tr(ji,jj,jk,mic,Kbb)
          dDMES(ji,jj,jk) = dDMES(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_mesozoo *tr(ji,jj,jk,mes,Kbb)
          !  Grazing on detritus
          dDPOC(ji,jj,jk) = dDPOC(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_POM*tr(ji,jj,jk,poc,Kbb)
          dDPON(ji,jj,jk) = dDPON(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_pom*tr(ji,jj,jk,pon,Kbb)
          ! Grazing on phytoplankton
          dDCFL(ji,jj,jk) = dDCFL(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Flagellates*tr(ji,jj,jk,cfl,Kbb)
          dDCEM(ji,jj,jk) = dDCEM(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Emiliana   *tr(ji,jj,jk,cem,Kbb)
          dDCdi(ji,jj,jk) = dDCdi(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Diatoms    *tr(ji,jj,jk,cdi,Kbb)

          dDnfl(ji,jj,jk) = dDnfl(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Flagellates*tr(ji,jj,jk,nfl,Kbb)
          dDnem(ji,jj,jk) = dDnem(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Emiliana   *tr(ji,jj,jk,nem,Kbb)
          dDndi(ji,jj,jk) = dDndi(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Diatoms    *tr(ji,jj,jk,ndi,Kbb)

#ifdef chlorophyll_geider
          dDCHF(ji,jj,jk) = dDCHF(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Flagellates*tr(ji,jj,jk,CHF,kbb)
          dDCHE(ji,jj,jk) = dDCHE(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Emiliana   *tr(ji,jj,jk,CHE,kbb)
          dDCHD(ji,jj,jk) = dDCHD(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Diatoms    *tr(ji,jj,jk,CHD,kbb)

#endif
          !When eating diatoms, micro ejects silicate as silicious_Detritus
          dPSID(ji,jj,jk) = dPSID(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Diatoms*tr(ji,jj,jk,ndi,Kbb)*SiNrDiatoms
#ifdef aggregation
          ! POMNOS decreases by the grazing
          ! If diatoms can form aggregate
            !dDAGG(ji,jj,jk) = dDAGG(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_pom*PON(ji,jj,jk)*AGG(ji,jj,jk)/(PON(ji,jj,jk)+PNS(ji,jj,jk))
             dDAGG(ji,jj,jk) = dDAGG(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_pom*tr(ji,jj,jk,agg,Kbb)
#endif
          ! DOX decreases due to respiration of zooplankton
          dDDOX(ji,jj,jk) = dDDOX(ji,jj,jk) + (C_ZOOResp + C_ZooAdjust)*OCr
          dPDIC(ji,jj,jk) = dPDIC(ji,jj,jk) + (C_ZOOResp + C_ZooAdjust)

          ! diagnostics
#ifdef biodiagrates
          TotalRespiration_Gel(ji,jj,jk) = TotalRespiration_Gel(ji,jj,jk) + C_ZOOResp + C_ZooAdjust
          SID_creation(ji,jj,jk) = SID_creation(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Diatoms*tr(ji,jj,jk,ndi,Kbb)*SiNrDiatoms
#endif

#ifdef biodiagtrophic
      !  Diagnostics Store the fluxes
      phy_to_Zoo(ji,jj,jk) = phy_to_Zoo(ji,jj,jk)+   grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Flagellates*tr(ji,jj,jk,cfl,Kbb)&
      +  grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Emiliana   *tr(ji,jj,jk,cem,Kbb) &
      + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_Diatoms*tr(ji,jj,jk,cdi,Kbb)
      POC_to_ZOO(ji,jj,jk) = POC_to_ZOO(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_POM*tr(ji,jj,jk,poc,Kbb)
      PON_to_ZOO(ji,jj,jk) = PON_to_ZOO(ji,jj,jk) + grazing_carbonNoctiluca/FluxPrey_carbon*Capt_eff_Noctiluca_POM*tr(ji,jj,jk,pon,Kbb)
      ZOO_TO_POC(ji,jj,jk)= ZOO_TO_POC(ji,jj,jk) + C_ZooEgest + C_ZOOMort
      ZOO_TO_PON(ji,jj,jk)= ZOO_TO_PON(ji,jj,jk) + N_ZooEgest + N_ZOOMort
#endif
       end if
  END_3D
  
  ! OUTPUT VARIABLES
  !********************************************************************
  ! Averaged over entire water column
#ifdef biodiagrates
  call IntegrateVertical(TotalRespiration_Gel,TotalRespiration_GelIntegrated,kmm)
#endif


#ifdef biodiagtrophic
  call IntegrateVertical(phy_to_ZOO,phy_to_ZOOIntegrated,kmm)
  call IntegrateVertical(bac_to_ZOO,bac_to_ZOOIntegrated,kmm)
  call IntegrateVertical(poc_to_ZOO,poc_to_ZOOIntegrated,kmm)
  call IntegrateVertical(pon_to_ZOO,pon_to_ZOOIntegrated,kmm)
#endif



END SUBROUTINE CalculateNoctiluca
#endif

#ifdef gelatinous
!**************************************************************
!   GELATINOUS GRAZING RATE
!**************************************************************
SUBROUTINE GELATINOUS_GRAZING_RATE(tf,maxgrazingrate,thresh,   &
     effprey1,effprey2,effprey3,effprey4,effprey5, &
     effprey6, &
     ZOO,              &
     grazing_carbon,NCrfood,FluxPrey_carbon, &
     ji,jj,jk, Kbb)
  !------------------------------------------------------------*
  ! Calculates the gelatinous grazing rate, as a linear function of   *
  ! the total flux of prey. Capture efficiencies are assumed to be 1*
  ! A total of 11 potential preys is considered (3phytos,5zoos, 2 poms, bacteria). *
  ! Capture efficiencies are assumed to be zero for some of these preys.*
  ! Return values in grazing_carbon (mmolC/day),NCrfood (mmolN/mmolC)
  !------------------------------------------------------------*
  IMPLICIT NONE
  INTEGER, INTENT(IN) :: ji,jj,jk,Kbb
  real(wp), INTENT(IN) ::        &
       Maxgrazingrate,               & ! ingestion coefficient (day mmolC/m3)-1
       Zoo,                          & ! Zooplankton carbon concentration
       thresh,                       & ! feeding threshold for the grazing (mmolC/m3)
       effprey1,effprey2,effprey3,   & ! Capture efficiencies on the different preys
       effprey4,effprey5,effprey6,   & ! Capture efficiencies on the different preys
       tf                              ! temperature factor
  real(wp), INTENT(OUT) ::    grazing_carbon,NCrfood,FluxPrey_carbon
  real(wp) :: FluxPrey_nitrogen

  ! Flux of consummed preys in carbon
  FluxPrey_carbon=effprey1*tr(ji,jj,jk,cfl,Kbb)+effprey2*tr(ji,jj,jk,cem,Kbb)+effprey3*tr(ji,jj,jk,cdi,Kbb)+ &
          effprey4*tr(ji,jj,jk,mic,Kbb)+effprey5*tr(ji,jj,jk,mes,Kbb) &
          +effprey6*tr(ji,jj,jk,poc,Kbb)

  ! Flux of consummed preys in nitrogen
  FluxPrey_nitrogen=effprey1*tr(ji,jj,jk,nfl,Kbb)+effprey2*tr(ji,jj,jk,nem,Kbb)+effprey3*tr(ji,jj,jk,ndi,Kbb)+ &
          effprey4*tr(ji,jj,jk,mic,Kbb)*NCrMicroZoo + effprey5*tr(ji,jj,jk,mes,Kbb)*NCrMesoZoo+ &
          effprey6*tr(ji,jj,jk,pon,Kbb)

  ! Grazing rate in carbon
  if (FluxPrey_carbon>thresh) then
     grazing_carbon = tf*maxgrazingrate*(FluxPrey_carbon-thresh)*Zoo
  else
     grazing_carbon = 0
  endif

  ! N:C molar ratio of the consumed food
  NCrfood=FluxPrey_nitrogen/FluxPrey_carbon

END SUBROUTINE GELATINOUS_GRAZING_RATE

SUBROUTINE GELATINOUS_RESPIRATION_RATE(tf,Asszoo,Effzoo,bzoo,ZOO,grazing_carbonZoo,C_ZOORespiration)
  !------------------------------------------------------------*
  ! Calculates the respiration rate of zooplankton as a sum of a basal respiration  *
  ! and an activity respiration. The basal respiration is computed as in the ERSEM model *
  ! as a linear function of zooplankton concentration depending on the temperature  .
  ! The activity respiration is a constant fraction of the carbon intake rate.*
  ! Return values in C_ZOOResp (mmolC/m3)                  *
  !------------------------------------------------------------*
  IMPLICIT NONE
  real(wp), INTENT(IN) ::     &
       Asszoo,                     & ! Assimilation efficiency
       grazing_carbonZoo,          & ! Intake of carbon
       EffZOO,                     & ! Part of the Assimilated food of zooplankton used for growth
       Zoo,                        & ! Zooplankton carbon concentration
       bzoo,                       & ! Basal respiration rate
       tf                            ! Temperature factor
  real(wp), INTENT(OUT) ::  C_ZOORespiration
  real(wp) :: basalrespiration,activityrespiration

  ! Basal respiration
  basalrespiration=bzoo*tf*Zoo

  ! Activity respiration
  activityrespiration=(1.-Effzoo)*Asszoo*grazing_carbonZoo

  !Total respiration
  C_ZOORespiration = basalrespiration + activityrespiration
END SUBROUTINE GELATINOUS_RESPIRATION_RATE

SUBROUTINE GELATINOUS_EGESTION_RATE(Asszoo,grazing_carbonZoo,C_ZOOEgest)
  !------------------------------------------------------------*
  ! Calculates the egestion rates in carbon (C_ZOOEgest) *
  ! as a constant fraction of the total intake (1-Asszoo), assuming that all the ingested materials*
  ! is in the form of particulate matter and considering that the N:C ration of the ingested food*
  ! equals the N:C ratio of the food taken up by zooplankton*
  ! Return values in C_ZOOEgest (mmol/day)                    *
  !------------------------------------------------------------*
  IMPLICIT NONE
  real(wp), INTENT (IN) ::       &
       Asszoo,                   & ! Assimilation efficiency
       grazing_carbonZoo           ! Intake of carbon
  real(wp), INTENT(OUT) :: C_ZOOEgest

  C_ZOOEgest = (1-Asszoo)*grazing_carbonZoo
END SUBROUTINE GELATINOUS_EGESTION_RATE

SUBROUTINE GELATINOUS_MORTALITY_RATE(HalfSatMort_Zoo,mortnonlinzoo,expmortzoo,&
     DOXsatmortality,mortalityanoxic,tf,ZOO,DOX,C_ZOOMortality)
  !------------------------------------------------------------*
  ! Calculates the mortality rate (per time and unit carbon biomass) of zooplankton  *
  ! Mortality is modeled as a sum of a linear and non-linear function of the zooplankton biomass*
  ! The order of the non-linearity is set in expmortzoo.*
  ! The mortality rate depends on the temperature and oxygen concentration*
  ! Return values in C_ZOOMort (mmolC/m3/day )                            *
  !------------------------------------------------------------*
  IMPLICIT NONE
  real(wp), INTENT(IN)  ::              &
       HalfSatMort_Zoo,                 & ! HAlf saturation constant of mortalirty
       mortnonlinzoo,                   & ! Non-linear mortality rate
       expmortzoo,                      & ! Order of the non-linearity of mortality rate
       DOXsatmortality,mortalityanoxic, & !Anoxic mortality of zooplankton
       ZOO, DOX,                        & ! Zooplankton biomass in carbon, oxygen concentration in mmolO/m3
       tf                                 ! Temperature factor
  real(wp), INTENT(OUT) :: C_ZOOMortality
  real(wp) ::  foxygen

  ! Oxygen limitation function
  foxygen=DOX/(DOX+DOXsatmortality)
  !Mortality rate : sum of a linear mortality, a quadratic mortality and an oxygen dependent term
  !          C_ZOOMortality=tf*(mortnonlinzoo*ZOO**expmortzoo/(ZOO+HalfSatMort_Zoo)+(1-foxygen)*mortalityanoxic*ZOO)
  C_ZOOMortality=tf*(mortnonlinzoo*ZOO+(1-foxygen)*mortalityanoxic*ZOO)
END SUBROUTINE GELATINOUS_MORTALITY_RATE
#endif
        ! gelatinous





!#########################################################################################
!                              3DZooplankton_computation.F90
!
! 1-D ecosystem model - Biological model of zooplankton
!
! Anderson, 1992,  Modelling the influence of food C:N ratio, and respiration on growth
! and nitrogen excretion in marine zooplankton and bacteria,
! Journal of Plankton Research, vol. 14, n 12, pp. 1645-1671, 1992
!
! Anderson and Hessen (1995), Carbon and Nitrogen limitation in marine copepods,
! Journal of Plankton Research, vol 17, n 2, pp. 317 - 331.
!
! Anderson amd Pondhaven (2003),Non-redfield carbon and nitrogen cycling in the Sarasso Sea :
! pelagic imbalances and export flux, in press in DSR
!
! Calculates the zooplankton grazing rate, as a michaelis Menten function   *
! depending on the total flux of prey. Capture efficiencies are assumed to be *
! constant. A total of 10 potential preys is considered (3phytos,5zoos, 1 pom, bacteria). *
! Capture efficiencies are assumed to be zero for some of these preys.*
! Return values in grazing_carbon (mmolC/m3/day),NCrfood (mmolN/mmolC)
!------------------------------------------------------------*
  SUBROUTINE GRAZING_RATE(tf,maxgrazingrate,csat,   &
     effprey1,effprey2,effprey3,effprey4,effprey5, &
     effprey6,effprey7, &
     ZOO,              &
     grazing_carbon,NCrfood,FluxPrey_carbon,JI,JJ,JK,Kbb)
  IMPLICIT NONE
  real(wp), INTENT(IN) ::          &
       Maxgrazingrate,             & ! max. specific grazing rate (day-1)
       Zoo,                        & ! Zooplankton carbon concentration
       Csat,                       & ! Half saturation constant for the grazing
       effprey1,effprey2,effprey3, & ! Capture efficiencies on the different preys
       effprey4,effprey5,effprey6, & ! Capture efficiencies on the different preys
       effprey7,                   & ! Capture efficiencies on the different preys
       tf                            ! temperature factor
  real(wp), INTENT(OUT) ::    grazing_carbon,NCrfood,FluxPrey_carbon
  real(wp) :: FluxPrey_nitrogen
  INTEGER, INTENT(IN) :: JI,JJ,JK, Kbb
  ! Flux of consummed preys in carbon
  FluxPrey_carbon=effprey1*TR(JI,JJ,JK,CFL,Kbb)+effprey2*TR(JI,JJ,JK,CEM,Kbb)+effprey3*TR(JI,JJ,JK,CDI,Kbb)+ &
       effprey4*TR(JI,JJ,JK,MIC,Kbb)+effprey5*TR(JI,JJ,JK,MES,Kbb) &
       + effprey6*TR(JI,JJ,JK,POC,Kbb)+effprey7*TR(JI,JJ,JK,BAC,Kbb)
  ! Flux of consummed preys in nitrogen
  FluxPrey_nitrogen=effprey1*TR(JI,JJ,JK,NFL,Kbb)+effprey2*TR(JI,JJ,JK,NEM,Kbb)+effprey3*TR(JI,JJ,JK,NDI,Kbb)+ &
       effprey4*TR(JI,JJ,JK,MIC,Kbb)*NCrMicroZoo + effprey5*TR(JI,JJ,JK,MES,Kbb)*NCrMesoZoo+ &
       effprey6*TR(JI,JJ,JK,PON,Kbb)+effprey7*TR(JI,JJ,JK,BAC,Kbb)*NCrBAC
  ! Grazing rate in carbon
  grazing_carbon= tf*maxgrazingrate*(FluxPrey_carbon/(FluxPrey_carbon+csat))*Zoo
  ! N:C molar ratio of the consumed food
  NCrfood=FluxPrey_nitrogen/FluxPrey_carbon
  END SUBROUTINE GRAZING_RATE


! **************************************************************
!   ZOOPLANKTON GROWTH RATE
!  Zooplankton production, ZOOgrowth (mmolC/m3/day) is calculated according to
!the stoichiometric model of Anderson and Henssen (1995)
! This model operates on the basis of a food threshold elemental ratio NCrfoodZooref (molN/molC)
!above which C limits growth and below
! which N limits:
!**************************************************************
SUBROUTINE ZOO_GROWTH_RATE(AssimNZoo,AssimCZoo,NCrzooplankton,Zoogrowthefficiency,NCrfood, &
     C_ZOOIntake,N_ZOOIntake,N_ZOOExcretion,Zoogrowthrate)
  IMPLICIT NONE
  real(wp), INTENT(IN) ::      &
       AssimNZoo,AssimCZoo,   & ! zooplankton assimilation efficiencies on nitrogen and carbon
       NCrzooplankton,  & ! Zooplankton N:C molar ratio
       Zoogrowthefficiency,  & ! C net production efficiency, fraction assimilated C allocated to production, remainder respired
       NCrfood, & ! N:C molar ratio of the grazed food
       C_ZOOIntake,N_ZOOIntake  ! Intake of nitrogen and carbon (grazing less messy feeding)
  real(wp), INTENT(OUT) ::    N_ZOOExcretion,Zoogrowthrate
  real(wp) :: NCrfoodZooref
  ! Compute the food threshold elemental ratio NCrfoodZooref (molN/molC)
  NCrfoodZooref = AssimCZoo*Zoogrowthefficiency/AssimNZoo*NCrzooplankton
  ! if the N:C ratio of ingested food NCrfood is less than NCrfoodZooref then N limits production, excretion of N
  ! N_ZOOExcr is zero and Zoogrowth equals :
  if (NCrfood< NCrfoodZooref) then
     Zoogrowthrate=AssimNZoo*N_ZOOIntake/NCrzooplankton
     N_ZOOExcretion=0
  else
     Zoogrowthrate = AssimCZoo*Zoogrowthefficiency*C_ZOOIntake
     N_ZOOExcretion=C_ZOOIntake*(AssimNZoo*NCrfood-AssimCZoo*Zoogrowthefficiency*NCrzooplankton)
  endif
END SUBROUTINE ZOO_GROWTH_RATE


!**************************************************************
!     ZOOPLANKTON MORTALITY
!**************************************************************
SUBROUTINE MORTALITY_RATE(HalfSatMort_Zoo,mortnonlinzoo,expmortzoo,DOXsatmortality,&
     mortalityanoxic,tf,ZOO,Oxy,C_ZOOMortality)
  !------------------------------------------------------------*
  ! Calculates the mortality rate (per time and unit carbon biomass) of zooplankton  *
  ! Mortality is modeled as a sum of a linear and non-linear function of the zooplankton biomass*
  ! The order of the non-linearity is set in expmortzoo.*
  ! The mortality rate depends on the temperature and oxygen concentration*
  ! Return values in C_ZOOMort (mmolC/m3/day )                            *
  !------------------------------------------------------------*
  IMPLICIT NONE
  real(wp), INTENT(IN)  ::   &
       HalfSatMort_Zoo,  & ! HAlf saturation constant of mortalirty
       mortnonlinzoo, & ! Non-linear mortality rate
       expmortzoo,    & ! Order of the non-linearity of mortality rate
       DOXsatmortality,mortalityanoxic, & !Anoxic mortality of zooplankton
       ZOO, Oxy,   &! Zooplankton biomass in carbon, oxygen concentration in mmolO/m3
       tf            ! Temperature factor
  real(wp), INTENT(OUT) :: C_ZOOMortality
  real(wp) ::  foxygen
  ! Oxygen limitation function
  foxygen=Oxy/(Oxy+DOXsatmortality)
  !Mortality rate : sum of a linear mortality, a quadratic mortality and an oxygen dependent term
  C_ZOOMortality=tf*(mortnonlinzoo*ZOO**expmortzoo/(ZOO+HalfSatMort_Zoo)+(1-foxygen)*mortalityanoxic*ZOO)
  !C_ZOOMortality=tf*(mortnonlinzoo*ZOO+(1-foxygen)*mortalityanoxic*ZOO)
  !          **expmortzoo/(ZOO+HalfSatMort_Zoo)+(1-foxygen)*mortalityanoxic*ZOO)
END SUBROUTINE MORTALITY_RATE


!**************************************************************
!   MIN MAX RATIO
!**************************************************************
SUBROUTINE MIN_MAX_RATIO(Ratio,MaxRatio,MinRatio,Out_Ratio)
IMPLICIT NONE
  real(wp), INTENT(IN)   ::     &
    Ratio,			&  ! actual ratio
    MaxRatio,MinRatio     	   ! Max, MIn, ratio
  real(wp), INTENT (OUT) :: Out_Ratio
  
  IF(Ratio > MaxRatio) THEN
     Out_Ratio = MaxRatio
  ELSEIF (Ratio < MinRatio) THEN
     Out_Ratio = MinRatio
  ELSE
     Out_Ratio = Ratio
  ENDIF
END SUBROUTINE MIN_MAX_RATIO


#ifdef chlorophyll_geider
!**************************************************************
!   CHLOROPHYLL COMPUTATION
!**************************************************************
 SUBROUTINE CHL_COMPUTATION(Mumax, ChlCratio, NCratio, MinNCr, RangeNCr, tf, alphaPI, light,WattToPhot, MaxChlNr, PhytoC, &
    Nutrient_Uptake,  RespirationPHY, MortPHY, CHL, PphotoC,  RespChla, MortChla, growthCHL, PmaxC, RhoChla, Chlorosynthesis, VNC, LightLim) 
			
  IMPLICIT NONE
  real(wp), INTENT(IN)   ::     &
    RangeNCr,MinNCr,NCratio,     & ! Max, Min, actual N:C ratio
    MaxChlNr, ChlCratio,        &  ! Min Chl:N ratio and range
    WattToPhot,	PhytoC,						&	! 
    Mumax, tf, alphaPI, light,RespirationPHY, &
    CHL, MortPHY, Nutrient_Uptake
  real(wp), INTENT (OUT) ::  PphotoC,  RespChla, MortChla,  growthCHL, PmaxC, RhoChla, Chlorosynthesis, VNC, LightLim      
    
  ! Compute the photosynthesis rates at light saturation (PmaxC) and at light E0 (PphotoC)
  PmaxC = Mumax*(NCratio-MinNCr)/(RangeNCr)*tf    ! C-specific ligth saturated rate of photosynthesis, in molC/molC/day (Geider et al., 1998)
  LightLim = AlphaPI*ChlCratio*WattToPhot*light
  
  IF (abs(PmaxC) < 1.0E-18) THEN
    PphotoC = 0 
  ELSE
    PphotoC = PmaxC*(1-exp(-LightLim/PmaxC))    ! instantaneaous carbon-specific, light-limited photosynthetic rate, in gC/gC/day (Geider et al., 1998)
   !PphotoC = PmaxC*(1-exp(-AlphaPI*light/(PmaxC)))
  ENDIF
  
  ! Compute the nutrients uptake  Instantaneous Carbon-Specific Nutrient uptake in molN/molC/day is a Michaelis-Menten function of nutrient concentration
  IF (abs(PhytoC) < 1E-18) THEN
    VNC = 0.
  ELSEIF (Nutrient_Uptake < 0.) THEN
    VNC = 0.
  ELSE
    VNC = Nutrient_Uptake/PhytoC
  ENDIF
 
  ! Compute the chlorophyll synthesis
  IF ((light==0) .OR. abs(LightLim) < 1E-18) THEN
    RhoChla = 0
  ELSE
    RhoChla = maxChlNr*PphotoC/(LightLim)
  ENDIF
  
  ! Compute the light-saturation parameter for photosynthesis (µmol  photons /s/m2)
 
  ! Compute the degradation of chlorophyll 
  RespChla = RespirationPHY*CHL*tf    ! RespChla Chla degradation rate gChla/day
  MortChla = MortPHY*CHL*tf !     Mortality rate

  !Ik = PmaxC/(AlphaPI*ChlCratio)
  ChloroSynthesis = RhoChla*VNC 
  growthCHL = ChloroSynthesis/ChlCratio*CHL 
	
END SUBROUTINE CHL_COMPUTATION
#endif


!**************************************************************
!   CHLOROPHYLL TO CARBON RATIO
!**************************************************************
!------------------------------------------------------------*
! Calculates the Chlorophyll to carbon ratio of the phyto-   *
! plankton (mg Chl/mol C), based on the N / C content
! chla/N ratio varies linearly with the NC ratio
! chla/c ratio varies non-linearly with the NC ratio   *
! The higher the N/C content, the higher the chlor/C ratio   *
! The chlorophyll to carbon content must be larger than some *
! value (min_chl_n * MinNCr) and smaller than                *
! (Min_chl_n+range)* MaxNCr.                                 *
! Return values in ChlCrPHY                                  *
!------------------------------------------------------------*
#ifdef chlorophyll_geider
SUBROUTINE CHL_C_RATIO(MaxNCr, MinNCr, NCratio, RangeNCr, MinChlNr, MinChlCr,MaxChlCr, rangeChl_n, ChlCrPHY)
  IMPLICIT NONE
  real(wp), INTENT(IN)   ::     &
    MaxNCr, MinNCr, NCratio, RangeNCr,    & ! Max, MIn, actual N:C ratio
    MinChlNr, MinChlCr, MaxChlCr, rangeChl_n          ! Min Chl:N ratio and range
  real(wp), INTENT (OUT) :: ChlCrPHY
 ! real(wp) :: RangeChl_N
  !------------------------------------------------------------------
  ! chlCrPHY has to be in mgChla/mmolC
  IF      (NCratio >= MaxNCr) THEN
     ChlCrPHY= MaxChlCr
  ELSEIF (NCratio <= MinNCr) THEN
     ChlCrPHY= MinChlCr
  ELSE
 !    RangeChl_N = MaxChlNr - MinChlNr
     ChlCrPHY= NCratio* (MinChlNr +                          &
          rangeChl_n * (NCratio- MinNCr)/     &
          (RangeNCr))
  ENDIF
END SUBROUTINE CHL_C_RATIO
#else
SUBROUTINE CHL_C_RATIO (NCratio,MaxNCr,MinNCr,MinChlNr,MaxChlNr,ChlCrPHY)
  IMPLICIT NONE
  real(wp), INTENT(IN)   ::     &
    MaxNCr,MinNCr,NCratio,     & ! Max, MIn, actual N:C ratio
    MinChlNr, MaxChlNr           ! Min Chl:N ratio and range
  real(wp), INTENT (OUT) :: ChlCrPHY
  real(wp) :: RangeChl_N
  !------------------------------------------------------------------
  ! chlCrPHY has to be in mgChla/mmolC
  IF      (NCratio >= MaxNCr) THEN
     ChlCrPHY= MaxNCr  * MaxChlNr
  ELSEIF (NCratio <= MinNCr) THEN
     ChlCrPHY= MinNCr  * MinChlNr
  ELSE
     RangeChl_N = MaxChlNr - MinChlNr
     ChlCrPHY= NCratio* (MinChlNr +                          &
          rangeChl_n * (NCratio- MinNCr)/     &
          (MaxNCr-MinNCr))
  ENDIF
END SUBROUTINE CHL_C_RATIO
#endif

  
!**************************************************************
!   PHYTOPLANKTON GROWTH RATE
!**************************************************************
#ifdef chlorophyll_geider
SUBROUTINE GROWTH_RATE(MinNCr, RatioNC, RatioNCMinMax, RatioChlC,alphaPI,light, WattToPhot, PphotoC, PhytoplanktonC,  &
	mumax, Resp,ProdResp, tf, extradocphyexc,Growth, Carbon_Uptake,Respiration,DOC_ex_excr)
!------------------------------------------------------------*
! Calculates the phytoplankton growth rate, as a minimum of  *
! nutrient limitated or light limited growth.                *
! Nutrient limitation considers internal plankton nutrients  *
! either nitrogen or silicate, see the report of Tett (1998) *
! Light limitation as in Sharples & Tett (1994)              *
! Return values in growth (mmol C/m3/day)                     *
!------------------------------------------------------------*
  IMPLICIT NONE
  real(wp), INTENT(IN) ::      &
     MinNCr, RatioNC,  RatioNCMinMax,   & ! Min N:C ratio, actual N:C ratio, Min/Max ratio
     RatioChlC, & !(Chl:C) ratio needed  for the P-I curve initial slope
     MuMax,                      & ! max. specific growth rate
     PhytoplanktonC,             & ! Phytoplankton carbon concentration
     alphaPI,  Light, WattToPhot,  PphotoC,  & ! Half saturation light intensity
     Resp, ProdResp,             & ! Maintenance and growth respiration
     extradocphyexc,   tf          ! Light, fraction of extra DOC excretion, temperature factor
  real(wp), INTENT(OUT) ::    Growth,Carbon_Uptake,Respiration,DOC_ex_excr
  real(wp) :: Mu_Light, Mu_Nitrogen
 
  ! Van der mersch et al, 2004, model, limnology and oceanography
  Mu_Nitrogen = 1. - MinNCr/RatioNC ! actual mu nutrient
 !Mu_light = 1.-exp(-alphaPI*WattToPhot*Light/mumax)
  Mu_light = 1.-exp(-alphaPI*RatioChlC*WattToPhot*Light/mumax)

  Carbon_Uptake=PphotoC*PhytoplanktonC
  Respiration=Carbon_Uptake*ProdResp + Resp*PhytoplanktonC*tf
  Growth=Carbon_Uptake-Respiration

  ! Compute the extra DOC excretion as a fraction (extradocphyexcr) of the difference of growth in nutrient limited (actual NC ratio) and nutrient
  ! saturated (max NC ratio) as in VDM et al 2004 L&O
  DOC_ex_excr=abs(extradocphyexc*mumax*PhytoplanktonC*tf*Mu_Light*((1. - RatioNCMinMax)- (Mu_nitrogen)))
END SUBROUTINE GROWTH_RATE
#else
SUBROUTINE GROWTH_RATE(QuantumYield,alphaPI,light, WattToPhot, LightLimitation,NutrientLimitation,       &
     MaxNCr,MinNCr,NCratio,          &
     MaxSiCr,MinSiCr,SiCratio,       &
     mumax,Resp,ProdResp,extradocphyexc,           &
     tf, PhytoplanktonC,Growth,Carbon_Uptake,Respiration,DOC_ex_excr)
!------------------------------------------------------------*
! Calculates the phytoplankton growth rate, as a minimum of  *
! nutrient limitated or light limited growth.                *
! Nutrient limitation considers internal plankton nutrients  *
! either nitrogen or silicate, see the report of Tett (1998) *
! Light limitation as in Sharples & Tett (1994)              *
! Return values in growth (mmol C/m3/day)                     *
!------------------------------------------------------------*
  IMPLICIT NONE
  real(wp), INTENT(IN) ::        &
     MaxNCr, MinNCr, NCratio,    & ! Max, Min, actual N:C ratio
     MaxSiCr, MinSiCr, SiCratio, & ! Max, Min, actual Si:C ratio
     MuMax,                      & ! max. specific growth rate
     QuantumYield,               &  ! Maximum quantun yield
     PhytoplanktonC,             & ! Phytoplankton carbon concentration
     alphaPI,                    & ! Half saturation light intensity
     Resp, ProdResp,             & ! Maintenance and growth respiration
     Light, WattToPhot, extradocphyexc,     & ! Light, fraction of extra DOC excretion
     tf                            ! temperature factor
  real(wp), INTENT(OUT)   ::     Growth,Carbon_Uptake,Respiration,DOC_ex_excr
  real(wp), INTENT(INOUT) ::     LightLimitation,NutrientLimitation
  real(wp) :: Mu, Mu_Nutrient, Mu_Light, Mu_Silicate, Mu_Nitrogen
  real(wp) :: NCrat,SiCrat

  !-----------
  ! Potential nutrient controlled growth rate; due to droop kinetics,
  ! the actual NC ratio can slightly surpass the maximal or minimal ratio
  IF(NCratio > MaxNCr) THEN
     NCrat = MaxNCr
  ELSEIF (NCratio < MinNCr) THEN
     NCrat = MinNCr
  ELSE
     NCrat = NCratio
  ENDIF
  ! the actual SiC ratio can slightly surpass the maximal or minimal ratio
  IF(SiCratio > MaxSiCr) THEN
     SiCrat = MaxSiCr
  ELSEIF (SiCratio < MinSiCr) THEN
     SiCrat = MinSiCr
  ELSE
     SiCrat = SiCratio
  ENDIF
  ! Tett model
  !      Mu_Nutrient = MuMax * (1. - (MinNCr  / NCrat) )
  !      Mu_Silicate = MuMax * (1. - (MinSiCr / SiCrat) )
  !      Mu_Light = (QuantumYield*Light*ChlCratio - Resp)*(1.-ProdResp)
  ! Decide which factor is in control
  !     Mu = MIN(mu_Nutrient, mu_light, mu_Silicate)

  ! Temperature dependence, and convert to the model's time unit
  !     Growth = mu *PhytoplanktonC * tf
  ! Van der mersch et al, 2004, model, limnology and oceanography
  Mu_Nitrogen = 1. - MinNCr  / NCrat
  Mu_Silicate = 1. - MinSiCr / SiCrat
  Mu_Nutrient = min(Mu_Nitrogen,Mu_Silicate)
  Mu_light = 1.-exp(-alphaPI*WattToPhot*Light/mumax)
  LightLimitation=Mu_light
  NutrientLimitation = Mu_Nutrient

  Carbon_Uptake=mumax*Mu_Light*Mu_Nutrient*PhytoplanktonC*tf

  Respiration=Carbon_Uptake*ProdResp + Resp*PhytoplanktonC*tf
  Growth=Carbon_Uptake-Respiration

  ! Compute the extra DOC excretion as a fraction (extradocphyexcr) of the difference of growth in nutrient limited (actual NC ratio) and nutrient
  ! saturated (max NC ratio) as in VDM et al 2004 L&O
  ! DOC_ex_excr=extradocphyexc*mumax*Mu_Light*(MinNCr*(1./NCrat - 1./MaxNCr))*PhytoplanktonC*tf
  DOC_ex_excr=abs(extradocphyexc*mumax*PhytoplanktonC*tf*Mu_Light*((1. - MinNCr  / MaxNCr)- (1. - MinNCr  / NCrat)))
  !First formulation of Anderson and Williams
  ! DOC_ex_excr=extradocphyexc*Carbon_Uptake
END SUBROUTINE GROWTH_RATE
#endif

  
!**************************************************************
!   PHYTOPLANKTON UPTAKE RATE OF NO3
!**************************************************************
SUBROUTINE NO_UPTAKE_RATE(NCratio,nitrate,ammonium,PhytoplanktonC,tf,MaxNCr,umax,ks,half_inhib_amm,NOuptake)
!------------------------------------------------------------*
! Calculates the uptake rate of nitrate by the phytoplankton *
! Nitrate uptake is inhibited by ammonium (1 - Monod)        *
! and limited by nitrate concentrations (Monod)              *
! The amount depends on the actual NC ratio                  *
! It is first-order with respect to the phytoplankton N      *
! (if NC ratio = maximal ratio: no uptake)                   *
! Return values in NOuptake (mmol N/m3/day)                     *
!------------------------------------------------------------*
  IMPLICIT NONE
  real(wp), INTENT (IN) ::     &
       NCratio, MaxNCr,    & ! actual and maximal N:C ratio
       nitrate, ammonium,  & ! nutrient concentrations
       ks, half_inhib_amm, & ! NO3 half-saturation and NH3 inhibition coeff.
       tf,                 & ! Temperature factor
       PhytoplanktonC,     & ! Phytoplankton concentration
       umax                  ! Maximal NO3 uptake rate
  real(wp), INTENT(OUT) :: NOuptake

  IF(NCratio >=  MaxNCr) THEN
     NOuptake=0.
  ELSE
     NOuptake= umax*tf* (1.-(NCratio/MaxNCr))            &
          * (nitrate/(ks+nitrate))                      &
          * (1. / (1. + (ammonium/half_inhib_amm))) &
          * PhytoplanktonC
  ENDIF
END SUBROUTINE NO_UPTAKE_RATE


  
!**************************************************************
!   PHYTOPLANKTON UPTAKE RATE OF NH4 (or other nutrient)
!**************************************************************
SUBROUTINE NUT_UPTAKE_RATE(NutCratio,Nutrient,PhytoplanktonC, &
       tf,MaxNutCr,umax,ks,Nut0,          &
       NUTuptake)
!------------------------------------------------------------*
! Calculates the uptake rate of a nutrient by phytoplankton  *
! Nutrient uptake is limited by Nutrient conc. (Monod)       *
! The amount depends on the actual Nutrient/C ratio of       *
! phytoplankton (up to now, we only have nitrogen)           *
! It is first-order with respect to the phytoplankton N      *
! (if NutC ratio = maximal ratio: no uptake)                 *
! if NutC ratio larger than maximal ratio: Nut excretion !   *
! Return values in NUTuptake (mM Nut/m3/day)                 *
!------------------------------------------------------------*
  IMPLICIT NONE
  real(wp), INTENT(IN) ::     &
         NutCratio, MaxNutCr, & ! actual and maximal Nut:C ratio
         Nutrient,            & ! nutrient concentration
         NUT0,                & ! nutrient concentration below which no uptake
         ks,                  & ! Nutrient half-saturation coeff
         tf,                  & ! Temperature factor
         PhytoplanktonC,      & ! Phytoplankton concentration
         umax                   ! Maximal nutrient uptake rate in day-1
  real(wp), INTENT(OUT) ::  NUTuptake
  !-----------
  IF(NutCratio > MaxNutCr) THEN
     ! excretion of ammonium (negative uptake).If we have silicate, silicate excretion is  allowed.
     NUTuptake =  umax*tf                                  &
          * (1.-(NutCratio / MaxNutCr))       &
          * PhytoplanktonC
  ELSEIF (Nutrient > Nut0) THEN
     NUTuptake =  umax*tf                                  &
          * (1.-(NutCratio /MaxNutCr))        &
          * (Nutrient-Nut0)/(ks+Nutrient-Nut0)  &
          * PhytoplanktonC
  ELSE
     NUTuptake = 0.
  ENDIF
END SUBROUTINE NUT_UPTAKE_RATE


!**************************************************************
!     phytoPLANKTON MORTALITY
!**************************************************************
SUBROUTINE PHYMORT_RATE(Mortality,tf, PHYMORT)
!------------------------------------------------------------*
! Calculates the mortality rate (per time) of phytoplankton  *
! Mortality is modeled as a linear function dependent of the temperature *
! Return values in PhyMort (/day )                            *
!------------------------------------------------------------*
  IMPLICIT NONE
  real(wp), INTENT(IN)  ::   &
       Mortality,  & ! First-order mortality rate
       tf            ! Temperature factor
  real(wp), INTENT(OUT) :: PHYMort
  !-----------
  PHYMort = Mortality * tf
END SUBROUTINE PHYMORT_RATE

  
!**************************************************************
!   Sedimentation of dinoflagellates and Diatoms interpolated between a max and min values
! according to the NCratio
!**************************************************************
SUBROUTINE Phytoplankton_Sedimentation(NCrPhy,MaxNCrPHY,MinNCrPHY,SinkingRatemin,SinkingRatemax,SinkingRate)
! As in Soetaert et al., (2001), the sinking velocity of dinoflagellates and diatoms is variable
  IMPLICIT NONE
  real(wp), INTENT(IN) ::     &
    NCrPhy,MaxNCrPHY,MinNCrPHY, & !Max, Min, actual N:C ratio
    SinkingRatemin, & ! Minimum sinking velocity
    SinkingRatemax  !Maximum sinking velocity
  real(wp), INTENT (OUT) :: SinkingRate
  ! In case of nitrogen limitation (NCratio = to NC min) the sinking speed increases. This is
  ! to simulate the increase of the sinking velocity at the end of bloom when nitrogen is limiting
  SinkingRate = SinkingRatemin + (SinkingRatemax - SinkingRatemin)*(MaxNCrPHY - NCrPHY)/&
       (MaxNCrPHY - MinNCrPHY)
  SinkingRate = min(max(Sinkingrate,SinkingRatemin),SinkingRatemax)
END SUBROUTINE Phytoplankton_Sedimentation






SUBROUTINE UpdateLight(kt,kbb,kmm)
!-------------------------------------------------------------------------*
! Compute the PAR influenced by phyto and detritus by one of 3 methods    *
!         * elementary algorithm                                          *  
!         * MAST full optics (2 wavebands)                                *
!-------------------------------------------------------------------------*
  IMPLICIT NONE
  integer, intent(in) :: kt,Kbb,Kmm
  logical             :: isnan
  INTEGER             :: JI,JJ,JK,L,M,v
  real(wp)            :: cequireste1,cequireste2,cequireste3,l1,l2,l3,P1,P2,PAR0,PARATINTERFACE
#ifdef fulloptics
  real(wp)            :: local_chl
  real(wp)            :: abstot_short,bbtot_short,abstot_long,bbtot_long
  real(wp)            :: cdomdepth
#endif
  
#ifdef fulloptics
  ! FIRST STEP : compute the attenuation coefficient
  DO_3D(0,0,0,0,1,jpk)
        if (tmask(ji,jj,jk).eq.1) then
          local_chl=sum(chlorophyll(ji,jj,jk,:))

          cdomdepth=1 ; if (gdept(ji,jj,jk,Kmm).gt.120) cdomdepth=0

          ! computation of Kd_short=(a_tot+bb_tot)/mu_d ! -> NO changed toward lee2005
          abstot_short=  a_w_short & ! Sea WATER
            + (a_chl_A_short*Local_Chl**a_chl_B_short) & ! chlorophyll
            + a_poc_short*(TR(JI,JJ,JK,POC,Kbb))*12.0/1000.0 & ! suspended minerals (in g/m3)
            + cdomdepth * ( a_cdom_intercept_short + a_cdom_slope_short*min(max(ts(ji,jj,1,jp_sal,Kmm),10.0),18.5) ) ! CDOM
          bbtot_short=  bb_w_short & ! Sea WATER
            + bb_dia_short*TR(JI,JJ,JK,CDI,Kbb)*12.0+ bb_fla_short*TR(JI,JJ,JK,CFL,Kbb)*12.0+bb_emi_short*TR(JI,JJ,JK,CEM,Kbb)*12.0 &! phyto
            + bb_poc_short*(TR(JI,JJ,JK,POC,Kbb))*12.0/1000.0 ! suspended minerals (in g/m3)
          KBIO_SHORT(ji,jj,jk)=(1+0.005*zendeg(ji,jj))*abstot_short + 4.18*(1-0.52*exp(-10.8*abstot_short))*bbtot_short
#ifdef nanquest
          if (isnan(kbio_short(ji,jj,jk))) write(numout,*) 'KBIOS==NaN for ji,jj,jk=',ji,jj,jk,Local_Chl,tr(JI,JJ,JK,smi,Kbb),TR(JI,JJ,JK,POC,Kbb),min(max(ts(ji,jj,1,jp_sal.kmm),10.0),18.5),TR(JI,JJ,JK,CDI,Kbb),TR(JI,JJ,JK,CFL,Kbb),TR(JI,JJ,JK,CEM,Kbb)
#endif
          ! computation of Kd_long=(a_tot+bb_tot)/mu_d
          abstot_long=  a_w_long & ! Sea WATER
            + (a_chl_A_long*Local_Chl**a_chl_B_long) & ! chlorophyll
            + a_poc_long*(TR(JI,JJ,JK,POC,Kbb))*12.0/1000.0 & ! suspended minerals (in g/m3)
            + cdomdepth * ( a_cdom_intercept_long + a_cdom_slope_long*min(max(ts(ji,jj,1,jp_sal,Kmm),10.0),18.5) ) ! CDOM proxy
          bbtot_long=  bb_w_long & ! Sea WATER
            + bb_dia_long*TR(JI,JJ,JK,CDI,Kbb)*12.0+ bb_fla_long*TR(JI,JJ,JK,CFL,Kbb)*12.0+bb_emi_long*TR(JI,JJ,JK,CEM,Kbb)*12.0 &! phyto
            + bb_poc_long*(TR(JI,JJ,JK,POC,Kbb))*12.0/1000.0 ! suspended minerals (in g/m3)
          KBIO_long(ji,jj,jk)=(1+0.005*zendeg(ji,jj))*abstot_long+4.18*(1-0.52*exp(-10.8*abstot_long))*bbtot_long
#ifdef nanquest
          if (isnan(kbio_long(ji,jj,jk))) write(numout,*) 'KBIOL==NaN for ji,jj,jk=',ji,jj,jk,Local_Chl,tr(JI,JJ,JK,smi,Kbb),TR(JI,JJ,JK,POC,Kbb),min(max(ts(ji,jj,1,jp_sal,Kmm),10.0),18.5),TR(JI,JJ,JK,CDI,Kbb),TR(JI,JJ,JK,CFL,Kbb),TR(JI,JJ,JK,CEM,Kbb)
#endif
        end if
  END_3D
#endif
#ifdef simpleoptics
  DO_3D(0,0,0,0,1,jpk)  
        if (tmask(ji,jj,jk).eq.1) then
          local_chl=sum(chlorophyll(ji,jj,jk,:))

          kbio_SHORT(ji,jj,jk)=LightAbsvisS+kBIOABSCHL*Local_Chl+kBIOABSPOC*TR(JI,JJ,JK,POC,Kbb)  ! +kBIOABSPCS*PCSI(JI,JJ,JK)
          kbio_LONG(ji,jj,jk) =LightAbsvisL+kBIOABSCHL*Local_Chl+kBIOABSPOC*TR(JI,JJ,JK,POC,Kbb)  ! +kBIOABSPCS*PCSI(JI,JJ,JK)
        endif
  END_3D
#endif

#if defined fulloptics || defined simpleoptics
  DO_2D(0,0,0,0)
    
      P1=1 ; P2=1
      PAR0=(1-LightAbsA)*qsr(ji,jj) ! visible light [W/m2]

      do jk=1,jpk
        if (tmask(ji,jj,jk).eq.1) then
          PAR(ji,jj,jk)= PAR0/e3t(ji,jj,jk,Kmm)* &
             (     LightAbsB*P1/(kbio_long (ji,jj,jk) )*(1-exp(-(kbio_long (ji,jj,jk))*e3t(ji,jj,jk,Kmm))) +  &
               (1-LightAbsB)*P2/(kbio_short(ji,jj,jk) )*(1-exp(-(kbio_short(ji,jj,jk))*e3t(ji,jj,jk,Kmm)))  )
          P1=P1*exp(-(kbio_long (ji,jj,jk))*e3t(ji,jj,jk,Kmm))
          P2=P2*exp(-(kbio_short(ji,jj,jk))*e3t(ji,jj,jk,Kmm))
        end if
      end do
  END_2D
#endif
  
END SUBROUTINE UpdateLight

  
!**************************************************************
!    SIZE_DISTRIBUTION
!**************************************************************
SUBROUTINE SIZE_DISTRIBUTION(DetritusN,NumberaggregateN,  &
       Ncontexp,Ncontcoef,cellsize,sinkexp, epsilon)
  IMPLICIT NONE
  real(wp), INTENT(IN) ::      &
     DetritusN,        & ! Nitrogeneous detritus concentration
     NumberaggregateN, & ! number of aggregates of PON and NDiatoms
     cellsize,         & ! minimum cell size
     Ncontexp,sinkexp, & ! Nitrogen content exponent
     Ncontcoef           ! C, nitrogen content coefficient
  real(wp), INTENT(OUT) ::    epsilon
  real(wp) :: CellsPerAggregate,Cellbiomass,apprdiv
  apprdiv = 1.e-6

  ! biomass of one algal cell, nmolN/cell
  CellBiomass       = Ncontcoef * CellSize**Ncontexp

  IF (NumberaggregateN >0.) THEN
    CellsPerAggregate = DetritusN/NumberaggregateN/Cellbiomass
  ELSE
    CellsPerAggregate = 1.
  ENDIF

  IF (CellsPerAggregate > 1.) THEN
     epsilon = (CellsPerAggregate *(1.+Ncontexp)-1.) / (CellsPerAggregate - 1.)
  ELSE
     epsilon = 100.
  ENDIF

  ! Prevent sizeexponent from becoming one of the values substracted in the denominator in AGGREGATION
  IF ( (epsilon == 1+SinkExp+Ncontexp ) .OR. (epsilon == 1+SinkExp ) .OR. (epsilon == 3+SinkExp ) ) epsilon = epsilon + apprdiv
  IF ( (epsilon == 2) .OR.  (epsilon == 3) .OR. (epsilon == 4) ) epsilon = epsilon + apprdiv
END SUBROUTINE SIZE_DISTRIBUTION

  

!**************************************************************
!    AGGREGATION
!**************************************************************
SUBROUTINE AGGREGATION(NumberaggregateN,eshearrate,Maxcellsize,cellsize,stickness,sinkexp,epsilon,FM,GM,SM,&
       vsedcell,aggrsett,aggrshear,AGGR)
  IMPLICIT NONE

  real(wp), INTENT(IN) ::      &
     NumberaggregateN, & ! number of aggregates of PON and NDiatoms
     eshearrate,   &
     sinkexp,      &
     stickness,    &
     cellsize,     &
     Maxcellsize,  &
     vsedcell,     & ! sedimentation velocity of a unique cell of size m
     FM,           & !the fraction of particles being larger than the upper bound M
     GM,           & ! the fraction of mass of particles being larger than the upper bound M
     epsilon,      & ! size distribution coefficient
     SM              ! ratio of sinking rate of large particles and one single cell
  real(wp), INTENT(OUT) :: aggrsett,aggrshear,AGGR
  real(wp)  :: e1,e2,e3,e4,es1,es3, max2,max3,min2,min3, &
           INTSETTL1,INTSETTL2, &          ! different integrals defined as I1, I2 and I4 in Kriest and Evans for the settlement
           INTSHEAR1,INTSHEAR2, INTSHEAR4  ! different integrals defined as I1, I2 in Kriest and Evans for the shear

  e1 = epsilon - 1.0
  e2 = epsilon - 2.0
  e3 = epsilon - 3.0
  e4 = epsilon - 4.0
  es1 = e1 - sinkexp
  es3 = e3 - sinkexp
  Max2 = Maxcellsize**2
  Max3 = Maxcellsize**3
  Min2 = cellsize**2
  Min3 = cellsize**3

  ! Compute number of particles colliding due to shear stress, aggrshear
  INTSHEAR1=2.0*((FM - 1.0)*(FM*Max3 - Min3)*e1/e4 + 3.0*(FM*Maxcellsize - cellsize)*(FM*Max2 - Min2)* e1*e1/(e2*e3))
  INTSHEAR2= FM*((Max3 + 3.0*(Max2*cellsize*e1/e2 +  Maxcellsize*Min2*e1/e3) + Min3*e1/e4) - FM*Max3*(1.0 +3.0*(e1/e2 + e1/e3 ) + e1/e4))
  INTSHEAR4 = FM*FM*8.0*Max3
  aggrshear = 0.163*eshearrate*NumberaggregateN**2*(INTSHEAR1 + 2.0*INTSHEAR2 + INTSHEAR4)

  ! Compute number of particles colliding due to differential settlement, aggrsett
  if (e1 == 0.) THEN
     INTSETTL1=0.
  ELSE
     INTSETTL1 = 2.0*e1* (Min2*((1.0 - FM*SM)/(e3*es1) - (1.0 - FM)/(es3*e1)) - (FM*FM*SM*Max2 - Min2)*sinkexp/(es3*e3*(2.0*epsilon - sinkexp -4.0)))
  ENDIF
  INTSETTL2 = FM*(SM*(Min2 - FM*Max2)/e3 - (Min2 - FM*Max2*SM)/es3)
  aggrsett = rPI*0.125*e1*vsedcell*NumberaggregateN**2*(INTSETTL1 + 2.0*INTSETTL2)

  ! Use TEP equation see ruiz et al stickiness= TEP/(TEP+ksTEP)
  AGGR = 0.5*stickness*(aggrsett + aggrshear)

end subroutine  AGGREGATION
  



  
!**************************************************************
!   Bacteria GROWTH RATE
!**************************************************************
SUBROUTINE BAC_GROWTH_RATE( &
! Calculates the bacterial growth rate production, BACgrowth (mmolC/m3/day) is calculated according to
! the stoichiometric model of Anderson  (1992) and Anderson and Williams (1998) and Andersoin and Pondhaven (2003)
  maxgrowthratebac,tf,ksatdocl,                     &
  ksatammonium,bactgrowthefficiency,NCrBacteria,    &
  Phosphate,ksatphosphate,                          &
  BacteriaC,                                        &
  Uptake_DOCLabile,Uptake_DONLabile,Uptake_AMM,     &
  BACGrowth,BACResp,BACExcr,                        &
  DOCLabile,DONLabile,Ammonium)

  IMPLICIT NONE

  real(wp), INTENT(IN) ::      &
     tf ,                      & ! temperature dependence for bacteria
     maxgrowthratebac,         & ! max.labile DOC or ammonium uptake /day
     BacteriaC,                & ! Bacteria carbon concentration
     ksatdocl,                 & ! half saturation for labile DOC uptake
     ksatammonium,             & ! mmolN/m3 half saturation for ammonium uptake
     bactgrowthefficiency,     & ! bacterial gross growth efficiency on carbon
     Phosphate,ksatphosphate,  & !Parameters for phosphate uptake
     NCrBacteria,DOCLabile,DONLabile,Ammonium             ! mmolN/mmolC   Goldman ratio for Bacteria

  real(wp), INTENT(OUT) ::    Uptake_DOCLabile,Uptake_DONLabile,Uptake_AMM,BACGrowth,BACResp,BACExcr

  real(wp) :: &
       testratio,    &       ! ratio allowing to deterimined if we are faced to a nitrogen or carbon limitation
       Nutlim,&              ! Parameters for phosphate uptake
       Uptake_Potential_NHS  ! Potential uptake of ammonium, in mmolN/m3/day

  Uptake_DOCLabile = tf*maxgrowthratebac*DOCLabile/(DOCLabile+ksatdocl)*BacteriaC
  Uptake_DONLabile = Uptake_DOCLabile*(DONLabile/DOCLabile)

  NutLim= min(Ammonium/(ksatammonium+ Ammonium),Phosphate/(ksatphosphate+Phosphate))
  Uptake_Potential_NHS = maxgrowthratebac*NutLim*BacteriaC*NCrBacteria

  ! Test if bacterial growth is limited by carbon (DOClabile) or nitrogen (ammonium + DONLabile)
  ! Define a threshold value and compare the Uptake_Potential_NHS, to this value
  testratio = Uptake_DOCLabile*(Uptake_DONLabile/Uptake_DOCLabile - bactgrowthefficiency*NCrBacteria)

  if (Uptake_Potential_NHS > (-testratio)) then
   ! In this case we are in a situation of carbon limitation
   BACGrowth = bactgrowthefficiency*Uptake_DOCLabile

   ! Growth rate computed taking into account the iron limitation
   BACGrowth=BACGrowth

   BACResp = Uptake_DOCLabile*(1.0 - bactgrowthefficiency)

   ! We have to test now, if NHS uptake is necessary to use all the DOCl
   if (testratio > 0) then
     ! We are in case of remineralisation of ammonium through bacteria excretion and no net uptake of ammonium is necessary to use all the DOC
     Uptake_AMM = 0
     BACExcr = testratio
   else
     Uptake_AMM = -testratio
     BACExcr = 0
   endif

  else
    ! if we are in case of nitrogen limitation,it means that all the DON and the potential uptake of NHS is not
    ! sufficient to consume all the DOC
    Uptake_AMM = Uptake_Potential_NHS
    BACGrowth = (Uptake_Potential_NHS + Uptake_DONLabile)/NCrBacteria
    ! Growth rate computed taking into account the iron limitation
    BACGrowth=BACGrowth
    BACExcr = 0
    BACResp = BACGrowth*(1.0/bactgrowthefficiency  - 1.0)
  end if

END SUBROUTINE BAC_GROWTH_RATE



!********************************************************************
real(wp) FUNCTION limitation_by_ironmanganese(zdepth,sigma)
  ! this is a function that's zero everywhere, except between 15.8 and 16.3
  ! it is 1.0 at sigma=16.05
  ! it represents the normalized concentration of Fe3 and Mn3, which can precipitate phosphate
  implicit none
  real(wp), intent(in) :: zdepth,sigma
  limitation_by_ironmanganese=exp(-((sigma-1000.0-Fe3Mn3_density)*8.0_wp)**2.0_wp)                                  ! a peak around 16.1 kg/m3
  limitation_by_ironmanganese=limitation_by_ironmanganese * (1.0-1./(1+exp(-2.0*(zdepth-190.0)/5.0)))   ! also limit to depths above ~200 meter
end FUNCTION limitation_by_ironmanganese



!********************************************************************
real(wp) FUNCTION Q10Factor (Temperature, Q10)
  !---------------------------------------------------------------------
  ! temperature dependence according to a Q10 formulation
  !---------------------------------------------------------------------
  IMPLICIT NONE
  real(wp), INTENT (IN) :: Temperature, Q10

  IF (Q10 <= 0.) THEN
     Q10Factor = 0.
  ELSE
     Q10Factor = EXP (LOG(Q10)*(Temperature-20.)/10.)
  ENDIF
END FUNCTION Q10Factor





!*********************************************************************
SUBROUTINE DynamicsOxygen(Kbb,Kmm)

  IMPLICIT NONE
  integer, intent(in) :: Kbb, Kmm
  real(wp) :: SurfaceTemp,SurfaceSal,doxsurface,Windspeed
  real(wp) :: refdens, SaturatedOxygen, Piston
  integer :: ji,jj
  !--------------------------------------------------------------
  ! Oxygen production/consumption due to biogeochemical processes
  ! Oxygen is produced by photosynthesis and consumed by respiration
  ! Oxygen is consumed in the nitrification process. Nitrate uptake by
  ! the phytoplankton changes the ON ratio of the phytoplankton.
  ! This is also adjusted with the external oxygen concentration
  ! (taken care of elsewhere)

  ! reaeration (mmol O2/m2/sec)
  DO_2D(0,0,0,0)
        if (tmask(ji,jj,1).eq.1) then

           ! inputs
           SurfaceTemp = max(ts(ji,jj,1,jp_tem,Kmm),-1.0_wp)  ! don't consider temperatures smaller than -1°C
           SurfaceSal  = ts(ji,jj,1,jp_sal,Kmm)
           doxsurface  = tr(ji,jj,1,dox,Kbb)
           windspeed   = wndm(ji,jj)

           ! computations
           !RefDens=6.793952E-2*SurfaceTemp-9.095290E-3*(SurfaceTemp)**2 &
           !  + 1.001685E-4*(SurfaceTemp)**3-1.120083E-6*(SurfaceTemp)**4  &
           !  + 6.536336E-9*(SurfaceTemp)**5  &
           !  +(8.24493E-1 - 4.0899E-3*SurfaceTemp &
           !  +7.6438E-5*(SurfaceTemp)**2 - 8.2467E-7*(SurfaceTemp**3)  &
           !  +5.3875E-9*(SurfaceTemp)**4)*SurfaceSal+(-5.72466E-3+1.0227E-4*SurfaceTemp &
           !  -1.6546E-6*(SurfaceTemp)**2)*(SurfaceSal)**1.5 +4.8314E-4*(SurfaceSal)**2+999.842594
           refdens=rhop(ji,jj,1)
           SaturatedOxygen        = O2_SATURATION (SurfaceTemp,SurfaceSal,RefDens)
           Piston                 = Piston_Wanninkhof(SurfaceTemp,Windspeed)
           AirSeaOxygenFlux(ji,jj)  = Piston * (SaturatedOxygen - doxsurface)
           
#ifdef csystemcheck
           O2satout(ji,jj) = SaturatedOxygen
#endif

        end if
  END_2D

END SUBROUTINE DynamicsOxygen


!***********************************************************************
! PISTON VELOCITY m/s
real(wp) FUNCTION Piston_Wanninkhof (temp, wind)

  IMPLICIT NONE

  real(wp) :: Temp,     &  ! temperature (dg C)
              Wind         ! the wind speed (m/s)
  real(wp) :: Schmidt      ! The Schmidt number

  ! 1. The Schmidt number
  Schmidt = 1953.4-128.00*Temp+3.9918*Temp*Temp-0.050091*Temp*Temp*Temp

  ! 2. The piston veloicty, in cm/hour according to Wanninkhof (1992)
  ! 2a. Journ. Geophys. Res. 97: 7327-7382:  Piston_Wanninkhof = 0.31*wind*wind /SQRT(Schmidt/660)
  ! 2b. Ho et al. Air-sea exchange at high wind speed,Geophysical research letters, VOL 33 (2006):
  Piston_Wanninkhof = 0.266*wind*wind/SQRT(Schmidt/600)

  ! 3. convert from cm/hour to [NEMO] m/sec : *0.01/3600 , or [R] m/d
  Piston_Wanninkhof = Piston_Wanninkhof/360000*PWUnitConv

END FUNCTION Piston_Wanninkhof


!**************************************************************
!     OXYGEN SATURATION                                       *
! Calculates the saturated oxygen concentration as a          *
! function of salinity and temperature                        *
real(wp) FUNCTION O2_SATURATION(Temp, Sal, Rho0)

  IMPLICIT NONE
  real(wp)   &
       Sal,      &    ! Salinity, PSU
       Temp,     &    ! Temperature, dg C
       SatOx,    &    ! Saturated concentration, µmol/l
       Rho0           ! Seawater density, kg/m3

  real(wp) :: T, Tdiv100

  real(wp), PARAMETER :: &
       a1 = -173.9894,      &
       a2 = 255.5907,       &
       a3 = 146.4813,       &
       a4 = -22.204,        &
       b1 = -0.037362,      &
       b2 = 0.016504,       &
       b3 = -0.0020564

  ! Temperature in Kelvin
  T = Temp+273.15

  Tdiv100 = T/100.

  ! Saturated oxygen concentration, in umol/kg
  SatOx = EXP(a1 + a2*100./T + a3*LOG(Tdiv100)+a4*Tdiv100+     &
       Sal*(b1+b2*Tdiv100+b3*Tdiv100*Tdiv100))
  ! in umol/dm3
  O2_SATURATION = SatOx* Rho0 * .001

END FUNCTION O2_SATURATION








!********************************************************************
real(wp) FUNCTION Ratio (X,Y)
  !---------------------------------------------------------------------
  ! Calculates the ratio of two numbers; check for dividing by 0
  !---------------------------------------------------------------------
  IMPLICIT NONE
  real(wp), INTENT(IN) :: X, Y
 !IF (Y > 1.0e-19) THEN
  IF (Y /= 0.) THEN
     Ratio = X / Y
  ELSE
     Ratio = 0.
  ENDIF
END FUNCTION Ratio

!********************************************************************
subroutine Integratevertical(component,componentintegrated,kmm,k_redist)
  !---------------------------------------------------------------------
  ! Integrate over the vertical (vertical_thickness) a field vector
  !---------------------------------------------------------------------
  IMPLICIT NONE
  INTEGER :: ji,jj,jk,kk,kmm
  integer,  intent (in), optional :: k_redist
  real(wp), INTENT (IN)  :: component(jpi,jpj,jpk)
  real(wp), INTENT (OUT) :: componentintegrated(jpi,jpj)

  componentintegrated=0.0_wp
  kk=jpk ; if (present(k_redist)) kk=k_redist
  DO_2D(0,0,0,0)
        do jk=1,kk
           componentintegrated(ji,jj) = componentintegrated(ji,jj) + tmask(ji,jj,jk)*component(ji,jj,jk)*e3t(ji,jj,jk,Kmm)  ! tmask is included in e3t when using VVL (qco)
        end do
  END_2D
end subroutine Integratevertical

!********************************************************************
real(wp) FUNCTION Michaelis (limitingelement,halfsaturationconstant)
  !------------------------------------------------------------------------
  ! Limiting michaelis menten function
  !-------------------------------------------------------------------------
  IMPLICIT NONE
  real(wp), INTENT (IN) :: limitingelement,halfsaturationconstant
  Michaelis = limitingelement/(limitingelement + halfsaturationconstant)
END FUNCTION Michaelis

real(wp) FUNCTION Inhibition (inhibitingelement,Inhibitionconstant)
  !------------------------------------------------------------------------
  ! Inibiting function computed as 1- michaelis menten function
  !-------------------------------------------------------------------------
  IMPLICIT NONE
  real(wp), INTENT (IN) :: inhibitingelement,Inhibitionconstant
  Inhibition = Inhibitionconstant/(inhibitingelement + Inhibitionconstant)
END FUNCTION Inhibition


!********************************************************************
subroutine SinkingVelocities(Kbb,Kmm)
  implicit none
  integer, intent(in) :: Kbb, Kmm
  integer             :: ji,jj,jk
  real(wp) :: max_sinking_speed ! L.V. 17-8-2023

  max_sinking_speed=2.0_wp/86400 ! max 2.0 m/day ; see Kubryakov & Satnichny, JMS2022 https://doi.org/10.1016/j.jmarsys.2021.103695
                                                    ! see also https://doi.org/10.1038/s41598-020-60424-5
  
  WDIA=0.0  ; WPOM=0.0
#ifdef aggregation
 WAGG=0.0
#endif
#ifdef testcons
  vsinkingrate_silicious_detritus=0.0
#else
  DO_3D(0,0,0,0,1,jpk)
        if (tmask(ji,jj,jk).eq.1) then
          ! POM
#ifdef aggregation
          call CalculateSinking(tr(ji,jj,jk,pon,Kbb),tr(ji,jj,jk,agg,Kbb),SinkingRatePOM,SinkingRateAGG)
#else
          SinkingRatePOM=SinkingRatePOM_Constant
          SinkingRateAGG=SinkingRatePOM_Constant
#endif
          ! artificial limit of WPOM and WAGG to a maximum value, when above 100m depth
          if (gdept(ji,jj,jk,Kmm).lt.100.0) then
            WPOM(ji,jj,jk)=min(SinkingRatePOM,max_sinking_speed)
#ifdef aggregation
            WAGG(ji,jj,jk)=min(SinkingRateAGG,max_sinking_speed)
#endif
          else
            WPOM(ji,jj,jk)=SinkingRatePOM
#ifdef aggregation
            WAGG(ji,jj,jk)=SinkingRateAGG
#endif
          end if
          ! SinkingRatePOM has been computed positive (downward) ; WPOM needs to be negative (downward) following NEMO convention
          WPOM(ji,jj,jk)=-WPOM(ji,jj,jk)
#ifdef aggregation
          WAGG(ji,jj,jk)=-WAGG(ji,jj,jk)
#endif
          ! SinkingRateDIA depends also on N:C ratio ; it is already limited during Phytoplankton_sedimentation subroutine
          NCrDiatom=ratio(tr(ji,jj,jk,ndi,Kbb),tr(ji,jj,jk,cdi,Kbb))
          CALL Phytoplankton_Sedimentation(NCrDiatom,MaxNCrDiatoms,MinNCrDiatoms,SinkingRateDiatomsmin,SinkingRateDiatomsmax,SinkingRateDiatoms)
          WDIA(ji,jj,jk)=-SinkingRateDiatoms 
        end if
  END_3D
#endif
end subroutine SinkingVelocities

#ifdef aggregation
subroutine agg_init(Kbb,Kmm)
  implicit none
  integer, intent(in) :: Kbb, Kmm
  integer             :: ji,jj,jk
  real(wp) :: globtotal

  tr(:,:,:,AGG,:)=0.0_wp
  DO_3D(0,0,0,0,1,jpk)
    if (tmask(ji,jj,jk).eq.1) tr(ji,jj,jk,AGG,Kbb) = (epsini - 1.0 - dzetabio)*(tr(ji,jj,jk,PON,Kbb))/((epsini - 1.0)*C*m**dzetabio) ! Kriest et al 2002
  END_3D
  tr(:,:,:,AGG,Kmm)=tr(:,:,:,AGG,Kbb)
  globtotal = glob_sum( 'aggini', tr(:,:,:,AGG,Kbb) * cvol(:,:,:)   )
  if (lwp) then
     write(numout,*)
     write(numout,*) 'BAMHBI AGG initialization from PON'
     write(numout,*) '  total initial content :  ',globtotal
  end if  
end subroutine agg_init

 
subroutine limit_dAGG(Krhs)
  implicit none
  integer             :: ji,jj,jk
  integer, intent(in) :: Krhs
  real(wp)            :: limit_per_timestep
  ! L.V. july 2017: dirty limitation of dPAGG and dDAGG to 1E9 / day, due to too large increases due to too small PON and/or too large WAGG and/or flux from benthic module
  limit_per_timestep = dAGG_limit*dt
  DO_3D(0,0,0,0,1,jpk)
          if (tmask(ji,jj,jk).eq.1) tr(ji,jj,jk,AGG,Krhs)=max(min(tr(ji,jj,jk,AGG,Krhs),limit_per_timestep),-limit_per_timestep)
  END_3D
end subroutine limit_dAGG
#endif

!###########################################################################################################################################################################################################################################################################
!
!                                                 B E N T H I C    M O D E L
!
!###########################################################################################################################################################################################################################################################################

subroutine bamhbi_benthic(Kbb,Kmm,Krhs)
    implicit none
    integer, intent(in) :: Kbb, Kmm,Krhs
    real(wp)    :: Cmin(jpi,jpj),Nmin(jpi,jpj),Smin(jpi,jpj)
    real(wp)    :: CFLUXPERHOUR,tfCdeg,pdenit,panox,pnit,seuilpanox,seuilpdenit,BDOX,BNOS,BNHS
    real(wp)    :: fluxbotDOXMAX, fluxbotNOSMAX, fluxbotNHSMAX
    real(wp)    :: resusp_fS,resusp_sS, resusp_fS_max
    real(wp)    :: resusp_fC,resusp_sC, resusp_fC_max
    real(wp)    :: fRESUSP_ero_f,fresusp_ero_s,fresusp_dep ! fresusp= t_bottom / t_crit
#ifdef bflbounds
    logical     :: DOXoverconsumption, NOSoverconsumption, NHSoverconsumption
    real(wp)    :: oldpdenit, oldpnit
#endif
    real(wp)    :: tf
    real(wp)    :: dt_ben
    integer     :: ji,jj,jk,ikt
    
    dt_ben=dt

    bt(:,:,:,:,Krhs) = 0._wp
    
#ifdef testcons2
    ! conservation -- we first vertically integrate the (3D) benthic stocks to 2D variables
    ben_budgetC=0. ; ben_budgetN=0. ; ben_budgetSi=0.
    do jk=1,jp_benlvl
      DO_2D(0,0,0,0)
             ikt=mbkt(ji,jj)
             ben_budgetC(ji,jj) = ben_budgetC(ji,jj) + tmask(ji,jj,ikt)*e1t(ji,jj)*e2t(ji,jj)*(bt(ji,jj,jk,fCSED,Kbb)+bt(ji,jj,jk,sCSED,Kbb))
             ben_budgetN(ji,jj) = ben_budgetN(ji,jj) + tmask(ji,jj,ikt)*e1t(ji,jj)*e2t(ji,jj)*(bt(ji,jj,jk,fCSED,Kbb)+bt(ji,jj,jk,sCSED,Kbb))*bt(ji,jj,jk,NCrSED,Kbb)
            !ben_budgetSi(ji,jj)= ben_budgetSi(ji,jj)+ tmask(ji,jj,ikt)*e1t(ji,jj)*e2t(ji,jj)*(fSSED(ji,jj,jk)+sSSED(ji,jj,jk))
      END_2D
    end do          ! and then nemo sums them horizontally ; we add them to pelagic budget
    PB_budget_C = glob_sum('testcons2',ben_budgetC) ;  ! PB_budget_C = PB_budget_C + integrated_budgetC
    PB_budget_N = glob_sum('testcons2',ben_budgetN) ;  ! PB_budget_N = PB_budget_N + integrated_budgetN
    ! PB_budget_Si= glob_sum(ben_budgetSi);  ! PB_budget_Si= PB_budget_Si+ integrated_budgetSi

    if ( lwp ) then
       write(numout,*) 'C: ',integrated_budgetC,  PB_budget_C, integrated_budgetC + PB_budget_C
       write(numout,*) 'N: ',integrated_budgetN,  PB_budget_N, integrated_budgetN + PB_budget_N
       !write(numout,*) 'Si: ',integrated_budgetSi ! , PB_budget_Si
       write(numout,*)
    end if
#endif
    
#ifdef benthic
    ! if benthic is not defined, all fluxes == 0
    
    ! 0.! bottom stress from waves and current
    call bottomstress(Kmm)

    do jk=1,jp_benlvl
      DO_2D(0,0,0,0)
        ikt=mbkt(ji,jj)  
        if (tmask(ji,jj,ikt)==1) then    ! we have at least 1 layer of water
#ifdef bflbounds
          DOXoverconsumption=.false.
          NOSoverconsumption=.false.
          NHSoverconsumption=.false.
#endif

          fRESUSP_ero_f   = botstress(ji,jj) / criticalstress_ero_f
          fresusp_ero_s   = botstress(ji,jj) / criticalstress_ero_S
          fresusp_dep     = botstress(ji,jj) / criticalstress_dep

          ! 1.! compute remineralised quantities (sediments that become dissolved), burial, and stock trends
          tf = Q10Factor(ts(ji,jj,ikt,1,Kmm),Q10CDEG)

          bt(ji,jj,jk,fCSED,Krhs)=-fCdegrate*tf*bt(ji,jj,jk,fcsed,Kbb)
          bt(ji,jj,jk,sCSED,Krhs)=-(sCdegrate*tf*bt(ji,jj,jk,scsed,Kbb))-bt(ji,jj,jk,scsed,Kbb)*sCburialrate
          bt(ji,jj,jk,fSSED,Krhs)=-(fSdisrate*tf*bt(ji,jj,jk,fssed,Kbb))
          bt(ji,jj,jk,sSSED,Krhs)=-(sSdisrate*tf*bt(ji,jj,jk,sssed,Kbb))-bt(ji,jj,jk,sssed,Kbb)*sSburialrate

          Smin(ji,jj)= fSdisrate*tf*bt(ji,jj,jk,fssed,Kbb)+ sSdisrate*tf*bt(ji,jj,jk,sssed,Kbb)
          Cmin(ji,jj)= fCdegrate*tf*bt(ji,jj,jk,fcsed,Kbb)+ sCdegrate*tf*bt(ji,jj,jk,scsed,Kbb)
          Nmin(ji,jj)= Cmin(ji,jj) *bt(ji,jj,jk,ncrsed,Kbb)


          ! 2.! resusp and sed flux following fRESUP
          if (fRESUSP_dep.le.1.0) THEN  ! the bottom stress is weak --> we have deposition
            fluxbotcdi(ji,jj) =(1-fRESUSP_dep)*tr(ji,jj,ikt,CDI,Kbb)*WDIA(ji,jj,ikt)  ! fluxbotcdi is negative and downward (as WDIA is downward and negative)
            fluxbotpoc(ji,jj) =(1-fRESUSP_dep)*tr(ji,jj,ikt,POC,Kbb)*WPOM(ji,jj,ikt)
            fluxbottotC(ji,jj)=fluxbotpoc(ji,jj)+fluxbotcdi(ji,jj)

            fluxbotndi(ji,jj) =(1-fRESUSP_dep)*tr(ji,jj,ikt,NDI,Kbb)*WDIA(ji,jj,ikt)
            fluxbotpon(ji,jj) =(1-fRESUSP_dep)*tr(ji,jj,ikt,PON,Kbb)*WPOM(ji,jj,ikt)
            fluxbottotN(ji,jj)=fluxbotpon(ji,jj)+fluxbotndi(ji,jj)

#ifdef aggregation
            fluxbotAGG(ji,jj) =(1-fRESUSP_dep)*tr(ji,jj,ikt,AGG,Kbb)*WAGG(ji,jj,ikt)
#endif 
            fluxbotsid(ji,jj) =(1-fRESUSP_dep)*tr(ji,jj,ikt,SID,Kbb)*(-vsinkingrate_Silicious_Detritus)
            fluxbottotS(ji,jj)=fluxbotsid(ji,jj)+SiNrDiatoms*fluxbotndi(ji,jj)

            resusp_fS=0.0 ; resusp_sS=0.0 ; resusp_fC=0.0 ; resusp_sC=0.0

#ifdef sedparamdiag
           fluxbotpoc_depo(ji,jj)=fluxbotpoc(ji,jj)
           fluxbotpon_depo(ji,jj)=fluxbotpon(ji,jj)
           fluxbotsid_depo(ji,jj)=fluxbotsid(ji,jj)
#endif
          else                        ! too much friction -> erosion
            fluxbotcdi(ji,jj) = 0.0
            fluxbotpoc(ji,jj) = 0.0
            fluxbottotC(ji,jj)= 0.0

            fluxbotndi(ji,jj) = 0.0
            fluxbotpon(ji,jj) = 0.0
            fluxbottotN(ji,jj)= 0.0

#ifdef aggregation
            fluxbotAGG(ji,jj) = 0.0
#endif

            fluxbotsid(ji,jj) = 0.0
            fluxbottotS(ji,jj)= 0.0

            resusp_fC_max= alphaRESUSP*Me_fC*(fRESUSP_ero_f-1.0)
            resusp_fC    = min( resusp_fC_max , 0.99*bt(ji,jj,jk,fcsed,Kbb)/dt_ben )
            resusp_fS_max= alphaRESUSP*Me_fs*(fRESUSP_ero_f-1.0)
            resusp_fS    = min( resusp_fS_max , 0.99*bt(ji,jj,jk,fssed,Kbb)/dt_ben )

            if (resusp_fS.le.1e-10) resusp_fS=0.
            if (resusp_fC.le.1e-10) resusp_fC=0.

            resusp_sC=min( alphaRESUSP*Me_sC*(fRESUSP_ero_s-1.0)*(1.0-resusp_fC/resusp_fC_max), 0.99*bt(ji,jj,jk,scsed,Kbb)/dt_ben)
            resusp_sS=min( alphaRESUSP*Me_sS*(fRESUSP_ero_s-1.0)*(1.0-resusp_fS/resusp_fS_max), 0.99*bt(ji,jj,jk,sssed,Kbb)/dt_ben)

            if (resusp_sS.le.1e-10) resusp_sS=0.
            if (resusp_sC.le.1e-10) resusp_sC=0.
          endif

          ! 3.! update of stock trend (the solid part; mineral part is done in 1.)
          bt(ji,jj,jk,fCSED,Krhs) = bt(ji,jj,jk,fCSED,Krhs) - (   pfCSED  *(fluxbottotC(ji,jj)) + resusp_fC )
          bt(ji,jj,jk,sCSED,Krhs) = bt(ji,jj,jk,sCSED,Krhs) - ( (1-pfCSED)*(fluxbottotC(ji,jj)) + resusp_sC )
          bt(ji,jj,jk,fSSED,Krhs) = bt(ji,jj,jk,fSSED,Krhs) - (   pfSSED  *fluxbottotS(ji,jj) + resusp_fS   )
          bt(ji,jj,jk,sSSED,Krhs) = bt(ji,jj,jk,sSSED,Krhs) - ( (1-pfSSED)*fluxbottotS(ji,jj) + resusp_sS   )

          ! 4.! in case of erosion (resuspension), update bottom flux and diagnostics
          if (fRESUSP_dep.GT.1.0) THEN
            fluxbotPOC(ji,jj) = resusp_fC+resusp_sC                ! here fluxbotPOC > 0 and goes upward
            fluxbotSID(ji,jj) = resusp_fS+resusp_sS
            fluxbotPON(ji,jj) = (resusp_fC+resusp_sC)*bt(ji,jj,jk,ncrsed,Kbb)
            fluxbottotC(ji,jj)= fluxbotPOC(ji,jj)
            fluxbottotS(ji,jj)= fluxbotSID(ji,jj)
            fluxbottotN(ji,jj)= fluxbotPON(ji,jj)
            ! pour la variable aggregat je raisonne comme suit : un flux de resuspension de POC revient a une wPOC vers le haut
            ! pour que le rapport POC sur AGG, soit la masse des aggregats, soit conservé, il faut que WAGG = WPOC (cf cas eps<1 dans calculatesink)
            ! on calcule donc la pseudo vitesse WPOC et on l'impose pour AGG
#ifdef aggregation
            fluxbotAGG(ji,jj) = ( fluxbotPON(ji,jj) / tr(ji,jj,ikt,PON,Kbb) ) * tr(ji,jj,ikt,AGG,Kbb)
#endif
#ifdef sedparamdiag
           fluxbotpoc_resusp(ji,jj)=fluxbotpoc(ji,jj)
           fluxbotpon_resusp(ji,jj)=fluxbotpon(ji,jj)
           fluxbotsid_resusp(ji,jj)=fluxbotsid(ji,jj)
#endif

          endif

          ! 6.! computation of remineralized fluxes
          ! the computation done here for sediment model is based on the work of Marie Suleau
          ! finding function approaching at the best the sediment model of Wijssman
!#ifdef forcingmodel_deSolve
!          CFLUXPERHOUR= max(Cmin(ji,jj)/24,0.05/100/24)
!          CFLUXPERHOUR= min(CFLUXPERHOUR,4000.0/100/24)
!#endif
          CFLUXPERHOUR=max(Cmin(ji,jj)*CfluxUnitConv(1),0.05*CfluxUnitConv(2))
          CFLUXPERHOUR=min(CFLUXPERHOUR,2.*CfluxUnitConv(3))
          
          BDOX=max(tr(ji,jj,ikt,DOX,Kbb),10.0)
          BNOS=min(max(tr(ji,jj,ikt,NOS,Kbb),1.0),30.0)
          BNHS=min(max(tr(ji,jj,ikt,NHS,Kbb),0.001),5.0)
          
          call Pnitfun(CFLUXPERHOUR,CFLUXPERHOUR*bt(ji,jj,jk,ncrsed,Kbb),BDOX,BNOS,BNHS,pnit)

          if (tr(ji,jj,ikt,DOX,Kbb).le.2.0)   pnit=0.0

          call Panoxfun(CFLUXPERHOUR,CFLUXPERHOUR*bt(ji,jj,jk,ncrsed,Kbb),BDOX,BNOS,BNHS,panox)

          call Pdenitfun(CFLUXPERHOUR,CFLUXPERHOUR*bt(ji,jj,jk,ncrsed,Kbb),BDOX,BNOS,BNHS,pdenit)

          if (tr(ji,jj,ikt,NOS,Kbb).le.0.01) pdenit=0.0

          !***compute mineralisation and release***
          fluxbotODU(ji,jj)=0.
          fluxbotNHSmax=-tr(ji,jj,ikt,NHS,Kbb) * e3t(ji,jj,ikt,Kmm) / DT_BEN
          fluxbotNOSmax=-tr(ji,jj,ikt,NOS,Kbb) * e3t(ji,jj,ikt,Kmm) / DT_BEN
          fluxbotDOXmax=-tr(ji,jj,ikt,DOX,Kbb) * e3t(ji,jj,ikt,Kmm) / DT_BEN

#ifdef bflbounds
          oldpnit=pnit
          oldpdenit=pdenit
#endif

          fluxbotNHS(ji,jj)=Nmin(ji,jj)*(1.0-pnit)

          if (fluxbotNHS(ji,jj).lt.fluxbotNHSmax) then
            pnit=min(pnit,1.0)
            !autre option ; pnit=min(pnit,1-(fluxbotNHSmax/nmin(ji,jj))) ! mais ça amene d'office zéro ..
            fluxbotNHS(ji,jj)=Nmin(ji,jj)*(1.0-pnit)
#ifdef bflbounds
            NHSoverconsumption=.TRUE.
#endif
          endif

          fluxbotNOS(ji,jj)=Nmin(ji,jj)*pnit-Cmin(ji,jj)*pdenit*0.8
          if (fluxbotNOS(ji,jj).lt.fluxbotNOSmax) Then
            pdenit=min(pdenit,bt(ji,jj,jk,ncrsed,Kbb)*pnit/0.8)
            fluxbotNOS(ji,jj)=Nmin(ji,jj)*pnit-Cmin(ji,jj)*pdenit*0.8
#ifdef bflbounds
            NOSoverconsumption=.TRUE.
#endif
          endif

          fluxbotDOX(ji,jj)=-Cmin(ji,jj)*(1.0-pdenit-panox*psoliddepo)*OCrdegrad- Nmin(ji,jj)*pnit*ONrnitrif

          if (fluxbotDOX(ji,jj).lt.fluxbotDOXmax) Then
            panox=1
            fluxbotDOX(ji,jj)=-Cmin(ji,jj)*(1.0-pdenit-panox*psoliddepo)*OCrdegrad- Nmin(ji,jj)*pnit*ONrnitrif
            fluxbotODU(ji,jj)=fluxbotDOXmax-fluxbotDOX(ji,jj)
            fluxbotDOX(ji,jj)=fluxbotDOXmax
#ifdef bflbounds
            DOXoverconsumption=.TRUE.
#endif
          endif

          Fluxbotpho(ji,jj)=Nmin(ji,jj)*PNRedfield
          fluxbotSIO(ji,jj)=Smin(ji,jj)
          fluxbotDIC(ji,jj)=Cmin(ji,jj)

#ifdef bflbounds
          if (DOXoverconsumption.or.NOSoverconsumption.or.NHSoverconsumption) then
            write(*,*) "***********", ji,"  ",jj,"  ",jk,"************************"
            write(*,*) "DOXoverconsumption NOSoverconsumption NHSoverconsumption"
            write(*,*) DOXoverconsumption, NOSoverconsumption, NHSoverconsumption
            write(*,*) "pnit"  , pnit ,"oldpnit", oldpnit
            write(*,*) "panox" , panox
            write(*,*) "pdenit", pdenit,"oldpdenit", oldpdenit
            write(*,*) "BOTCON(ji,jj,dox)",BOTCON(ji,jj,dox)
            write(*,*) "bflDOX", fluxbotdox(ji,jj), "available", fluxbotdoxmax
            write(*,*) "BOTCON(ji,jj,nhs)",BOTCON(ji,jj,nhs)
            write(*,*) "bflNHS",fluxbotnhs(ji,jj),"available",   fluxbotnhsmax
            write(*,*) "BOTCON(ji,jj,nos)",BOTCON(ji,jj,nos)
            write(*,*) "bflNOS",fluxbotnos(ji,jj)," available",  fluxbotdoxmax
            write(*,*) "bflODU", fluxbotODU(ji,jj)
            write(*,*) '***********************************************************'
          endif
#endif

        
#ifdef sedparamdiag
          burialC(ji,jj)    = bt(ji,jj,jk,scsed,Kbb)*sCburialrate
          burialS(ji,jj)    = bt(ji,jj,jk,sssed,Kbb)*sSburialrate
          denitinsed(ji,jj) = Cmin(ji,jj)*pdenit*0.8
          pdenit2D(ji,jj) = pdenit
          pnit2D(ji,jj)   = pnit
          panox2D(ji,jj)  = panox
#endif
          ! 5.! update of NCrSED
          if ((bt(ji,jj,jk,fcsed,Kbb)+bt(ji,jj,jk,scsed,Kbb)).lt.0.05) then
            bt(ji,jj,jk,NCrSED,Krhs)=0.
          else
            bt(ji,jj,jk,NCrSED,Krhs)= (1/(bt(ji,jj,jk,fcsed,Kbb)+bt(ji,jj,jk,scsed,Kbb)) ) * ( (-fluxbottotN(ji,jj)-Nmin(ji,jj)) -bt(ji,jj,jk,ncrsed,Kbb)*(-fluxbottotC(ji,jj)-Cmin(ji,jj)) )
          end if
        end if ! tmask==1
    END_2D
  end do     ! jk benthic levels

#endif
  ! benthic module
  
end subroutine bamhbi_benthic



!*************************************************!
subroutine BOTTOMSTRESS(Kmm)
  IMPLICIT NONE
  integer, intent(in) :: Kmm
  !local variables
  integer  :: ji,jj
  real(wp) :: u2,v2,sqrtu2
  real(wp) :: epsilon=1.E-30
  real(wp) :: currentstress
  real(wp) :: tm
  real(wp) :: vec_sum
  real(wp) :: phi !the vector between currents and waves
  real(wp) :: phi_c !angle currents
  real(wp) :: phi_w !angle waves

  ! M.C 05/2024 :additions for computing currentstress as in nemo (namben_curstress)
  ! code follows src/OCE/DIA/diawri.F90
  real(wp):: zztmp, zztmp2
  REAL(wp), DIMENSION(A2D(     0))     ::   z2d
  
  ! BOTTOM STRESS FROM WAVES is computed in trc_wbstress.F90 
  if (cur_botstr.eq.1) then
        zztmp = rho0 * 0.25_wp
        z2d(:,:) = 0._wp
  end if

  ! BOTTOM STRESS FROM CURRENT, AND TOTAL STRESS
  DO_2D(0,0,0,0)
      if (tmask(ji,jj,mbkt(ji,jj)).eq.1) then    ! we have at least 1 layer of water
        u2=0.0 ; v2=0.0
         if (cur_botstr.eq.0) then
             if (umask(ji,jj,mbku(ji,jj)).eq.1)        u2=uu(ji,jj,mbku(ji,jj),Kmm)
             if (ji.gt.1) then                                                                       ! this will never be the case in R/deSolve
                 if (umask(ji-1,jj,mbku(ji-1,jj)).eq.1) u2=(u2+uu(ji-1,jj,mbku(ji-1,jj),Kmm))/2
             end if
             if (vmask(ji,jj,mbkv(ji,jj)).eq.1)        v2=vv(ji,jj,mbkv(ji,jj),Kmm)
             if (jj.gt.1) then
                 if (vmask(ji,jj-1,mbkv(ji,jj-1)).eq.1) v2=(v2+vv(ji,jj-1,mbkv(ji,jj-1),Kmm))/2
             end if
             sqrtu2=sqrt(u2**2 + v2**2 + epsilon)
             tauxb(ji,jj)=c_db(ji,jj)*sqrtu2*u2
             tauyb(ji,jj)=c_db(ji,jj)*sqrtu2*v2
             currentstress=rhop(ji,jj,mbkt(ji,jj)) * sqrt((tauxb(ji,jj))**2 + (tauyb(ji,jj))**2)
         elseif (cur_botstr.eq.1) then
             !code follows src/OCE/DIA/diawri.F90
              zztmp2 = (  ( rCdU_bot(ji+1,jj)+rCdU_bot(ji  ,jj) ) * uu(ji  ,jj,mbku(ji  ,jj),Kmm)  )**2   &
              &   + (  ( rCdU_bot(ji  ,jj)+rCdU_bot(ji-1,jj) ) * uu(ji-1,jj,mbku(ji-1,jj),Kmm)  )**2   &
              &   + (  ( rCdU_bot(ji,jj+1)+rCdU_bot(ji,jj  ) ) * vv(ji,jj  ,mbkv(ji,jj  ),Kmm)  )**2   &
              &   + (  ( rCdU_bot(ji,jj  )+rCdU_bot(ji,jj-1) ) * vv(ji,jj-1,mbkv(ji,jj-1),Kmm)  )**2
              z2d(ji,jj) = zztmp * SQRT( zztmp2 ) * tmask(ji,jj,1)
              currentstress=z2d(ji,jj)
         end if

        ! TOTAL BOTTOM STRESS
        if (waves_botstr .eq. 3) then
            ! sum of vectors
            phi_c = ATAN2(v2,u2)
            phi_w = wave_direction(ji,jj)        ! angle provided
            ! convert phi_w to the same coordinate system as phi_c 
            ! 1. Substract 90 degrees 2. 0-360 to (-180,180) conversion 3. invert angle (from clockwise to counterclockwise)
            ! 4. Convert to radians
            phi_w = - (MODULO((phi_w-90+180),360.)+180)/180*rpi
            ! nonlinear enhancement,  following R. Soulsby, dynamics of marine sands or also Gayer 2006
            ! (https://link.springer.com/article/10.1007/s10236-006-0070-5)
            tm = currentstress*(1 + 1.2 * (wavestress(ji,jj)/(currentstress+wavestress(ji,jj)))**(3.2))
            ! difference between to angles
            phi=ABS((MODULO(phi_c-phi_w+rpi,2*rpi)-rpi))
            vec_sum = SQRT((tm + wavestress(ji,jj)*COS(phi))**2 + (wavestress(ji,jj)*SIN(phi))**2)
            botstress(ji,jj)=min(1.0, vec_sum) !saveguard against too big wavestresses
        else
           botstress(ji,jj)=min(1.0, currentstress+wavestress(ji,jj)) !saveguard against too big wavestresses
        end if
#ifdef botstressdiag
        curstress(ji,jj)=currentstress
        !resuspension counter for fast and slow decaying stock (1==resuspension, 0==deposition) 
        if (botstress(ji,jj).gt. criticalstress_ero_s) then
            resuspcounter_s(ji,jj) = 1
        else
            resuspcounter_s(ji,jj) = 0
        endif

        if (botstress(ji,jj).gt. criticalstress_ero_f) then
            resuspcounter_f(ji,jj) = 1
        else
            resuspcounter_f(ji,jj) = 0
        endif
#endif
      end if
  END_2D
  
end subroutine bottomstress

end module bamhbi 

