package fieldextension

import (
	"testing"

	"github.com/consensys/gnark-crypto/field/babybear"
	"github.com/consensys/gnark/frontend"
	"github.com/consensys/gnark/test"
)

type ReduceCircut struct {
	Input   []frontend.Variable
	Reduced []frontend.Variable
}

func (c *ReduceCircut) Define(api frontend.API) error {
	e, err := NewExtension(api)
	if err != nil {
		return err
	}
	res := e.Reduce(c.Input)
	e.AssertIsEqual(c.Reduced, res)
	return nil
}

func TestReduce(t *testing.T) {
	assert := test.NewAssert(t)
	for _, tc := range []struct {
		input, reduced []int
	}{
		{[]int{1467980320, 1137445292}, []int{1467980320, 1137445292, 0, 0, 0, 0, 0, 0}},
		{[]int{1906025257, 900972831, 355994451}, []int{1906025257, 900972831, 355994451, 0, 0, 0, 0, 0}},
		{[]int{1315269736, 1305411155, 1484949641, 1487157818}, []int{1315269736, 1305411155, 1484949641, 1487157818, 0, 0, 0, 0}},
		{[]int{930655562, 191916507, 245232235, 249903878, 1688769114}, []int{930655562, 191916507, 245232235, 249903878, 1688769114, 0, 0, 0}},
		{[]int{1900558240, 1034669852, 62012066, 1636768938, 1951223124, 800157949}, []int{1900558240, 1034669852, 62012066, 1636768938, 1951223124, 800157949, 0, 0}},
		{[]int{1506768621, 1188015241, 521233244, 464809937, 288133325, 339109914, 1107846641}, []int{1506768621, 1188015241, 521233244, 464809937, 288133325, 339109914, 1107846641, 0}},

		{[]int{1467980320, 1137445292}, []int{1467980320, 1137445292}},
		{[]int{1906025257, 900972831, 355994451}, []int{1906025257, 900972831, 355994451}},
		{[]int{1315269736, 1305411155, 1484949641, 1487157818}, []int{1315269736, 1305411155, 1484949641, 1487157818}},
		{[]int{930655562, 191916507, 245232235, 249903878, 1688769114}, []int{930655562, 191916507, 245232235, 249903878, 1688769114}},
		{[]int{1900558240, 1034669852, 62012066, 1636768938, 1951223124, 800157949}, []int{1900558240, 1034669852, 62012066, 1636768938, 1951223124, 800157949}},
		{[]int{1506768621, 1188015241, 521233244, 464809937, 288133325, 339109914, 1107846641}, []int{1506768621, 1188015241, 521233244, 464809937, 288133325, 339109914, 1107846641}},

		{[]int{1200147517, 527805146, 1459729161, 298883860, 1301476489, 186161068, 997795829, 257063407}, []int{1200147517, 527805146, 1459729161, 298883860, 1301476489, 186161068, 997795829, 257063407}},
		{[]int{1353990425, 388912686, 1299455585, 514865345, 286702144, 1363798779, 1209821622, 492855042, 1874476453}, []int{1840572198, 388912686, 1299455585, 514865345, 286702144, 1363798779, 1209821622, 492855042}},
		{[]int{953116276, 1677525413, 1330847726, 935325903, 367765685, 666819005, 1259969643, 141562180, 860612033, 1047773391}, []int{353519034, 1123437188, 1330847726, 935325903, 367765685, 666819005, 1259969643, 141562180}},
		{[]int{752482022, 867506333, 1723423219, 361328340, 1241112226, 476919145, 182725336, 1468842972, 551661607, 617211228, 590726493}, []int{780961936, 1617032078, 168350958, 361328340, 1241112226, 476919145, 182725336, 1468842972}},
		{[]int{1013674053, 1587348044, 1207155881, 1116555932, 478056632, 1288268012, 1451373934, 1796131301, 1248869310, 1814778483, 275039764, 1209127427}, []int{658375016, 1417252147, 206061443, 324096182, 478056632, 1288268012, 1451373934, 1796131301}},
		{[]int{1498948171, 1003564318, 56559900, 1147491866, 1124826785, 1729654757, 819256679, 1503546020, 1907968762, 427148195, 1043561808, 666209549, 911979384}, []int{340679422, 1675662621, 1469410183, 422733221, 1090270404, 1729654757, 819256679, 1503546020}},
		{[]int{681684962, 586522244, 2004199348, 221839431, 1345587360, 872049662, 1613021061, 1424383966, 558639729, 930888084, 1820774825, 932126772, 391304517, 456497888}, []int{786924218, 759961563, 1900063213, 408904318, 1623405205, 1866994588, 1613021061, 1424383966}},
		{[]int{1727523785, 1360879540, 1735135715, 148864715, 863920986, 1616761360, 984599128, 940289252, 1438501308, 1482499975, 491492725, 1154531434, 1826009304, 581938247, 1777373046}, []int{1444910805, 1562251897, 1101757927, 769114963, 817364120, 1978284314, 403043424, 940289252}},
		{[]int{877033825, 65655943, 178692522, 262720537, 1210970053, 1422087058, 1186945353, 1031788918, 1005627592, 1704405255, 1402192569, 1123357903, 932685461, 1556434039, 1727137317, 1394736528}, []int{1872607732, 694720459, 1509949334, 540061944, 1404180519, 423468198, 52796630, 267763358}},
		{[]int{1134977302, 1648475213, 477296831, 768465845, 11704608, 542135288, 1210705842, 1659979380, 1241159085, 1275966185, 1014704952, 526651747, 1562882049, 1877460597, 1053879427, 1280810580, 1520192977}, []int{1431017196, 1591241801, 1572721698, 521837299, 1097279779, 1061542645, 723784013, 1656034313}},
		{[]int{1555011839, 787815769, 182103774, 781857749, 471054316, 962296659, 1274815353, 1018400075, 1523982888, 1230566410, 1442405474, 639520646, 1147075418, 1454417950, 191443111, 424358887, 598662772, 1646007563}, []int{160052574, 84773776, 1955702541, 1776787092, 1009288388, 854766741, 1367423653, 1659815990}},
		{[]int{1779267173, 1543263614, 1661719435, 1900359777, 893592452, 1006662846, 1442338151, 1005387230, 1302908503, 1159753495, 1351425996, 1816874489, 1790280276, 1100455927, 955172652, 726156767, 566008880, 282490572, 1152386475}, []int{41166504, 163529167, 944692949, 1753319946, 454016278, 1032082517, 1882907718, 940047983}},
		{[]int{1002186586, 435254065, 1396145151, 1622293879, 174136851, 1037681320, 768511646, 675947902, 188213580, 1414683255, 1986089618, 1203380915, 1567490308, 36246439, 1651769094, 105294241, 1140394104, 1184467882, 986378799, 1295577854}, []int{131608080, 269375833, 1666351158, 496800993, 1310402871, 1436392149, 818578391, 1834184553}},
	} {
		bb8 := make([]frontend.Variable, len(tc.input))
		for i := range tc.input {
			bb8[i] = frontend.Variable(tc.input[i])
		}
		bb8red := make([]frontend.Variable, len(tc.reduced))
		for i := range tc.reduced {
			bb8red[i] = frontend.Variable(tc.reduced[i])
		}
		err := test.IsSolved(&ReduceCircut{Input: make([]frontend.Variable, len(bb8)), Reduced: make([]frontend.Variable, len(bb8red))},
			&ReduceCircut{Input: bb8, Reduced: bb8red}, babybear.Modulus())
		assert.NoError(err)
	}
}

type AddCircuit struct {
	A, B, C []frontend.Variable
}

func (c *AddCircuit) Define(api frontend.API) error {
	e, err := NewExtension(api)
	if err != nil {
		return err
	}
	res := e.Add(c.A, c.B)
	e.AssertIsEqual(c.C, res)
	return nil
}

func TestAdd(t *testing.T) {
	assert := test.NewAssert(t)
	for _, tc := range []struct {
		a, b, c []int
	}{
		{[]int{1504941483, 528713979, 1590716977, 1030723568, 691448958, 45161890, 558331570, 1584182780, 884750304, 1178012232, 1236551897, 1743822194, 1102524691, 949136580, 968686988, 1807636110, 1419005839}, []int{1451698632, 267757499, 1153206782, 291258043, 1014114345, 588561574, 161218185, 1655775873, 115681370, 24609626, 1495418674}, []int{943374194, 796471478, 730657838, 1321981611, 1705563303, 633723464, 719549755, 1226692732, 1000431674, 1202621858, 718704650, 1743822194, 1102524691, 949136580, 968686988, 1807636110, 1419005839}},
		{[]int{1874099641, 1982653637, 1187310579, 475561226, 1692092055}, []int{1230758452, 1959413289, 1645368110, 432360750, 1418838351, 687464797, 1234750833, 1203209996, 888838337, 852882006, 356386082, 916503764, 1792596122, 1102186785, 1444663299}, []int{1091592172, 1928801005, 819412768, 907921976, 1097664485, 687464797, 1234750833, 1203209996, 888838337, 852882006, 356386082, 916503764, 1792596122, 1102186785, 1444663299}},
		{[]int{1795185782, 1766445854, 504379178, 1820376092, 137151794, 1064960087, 1759175291, 585123542, 1604030370, 1511659175, 916198528, 1166864589, 1699685308}, []int{805729528, 430124370, 1260617837, 297604025, 613457793, 20971739, 105513811}, []int{587649389, 183304303, 1764997015, 104714196, 750609587, 1085931826, 1864689102, 585123542, 1604030370, 1511659175, 916198528, 1166864589, 1699685308}},
		{[]int{370766974, 1556330301, 1310468525, 1225434398, 928378540, 435540789, 361405873, 1035503425, 545600368, 120758801, 1022518983, 1758884239, 1312473265, 1134254141}, []int{1297416854, 637666735, 325808988, 824671410}, []int{1668183828, 180731115, 1636277513, 36839887, 928378540, 435540789, 361405873, 1035503425, 545600368, 120758801, 1022518983, 1758884239, 1312473265, 1134254141}},
		{[]int{1891025422, 2009813156, 1706954798, 1389626918, 1029725850, 1487402244, 717521687, 10632936, 73787955, 744460996, 1457784272, 1484874357, 811933684, 1652886077, 1531756184, 753745186, 1714652400}, []int{457901568, 944951453, 169650164, 315210583, 1580068898, 1204321039, 648114211, 1202582296, 197451510, 734577008, 1745397260, 1991793135, 1515312634, 736548227, 1072265360, 1703339801, 5096947, 881796514}, []int{335661069, 941498688, 1876604962, 1704837501, 596528827, 678457362, 1365635898, 1213215232, 271239465, 1479038004, 1189915611, 1463401571, 313980397, 376168383, 590755623, 443819066, 1719749347, 881796514}},
		{[]int{1621710603, 226525544, 1202575715}, []int{1412522468, 178072249, 1954193329, 164698463, 2004081065, 1337457847, 1308872918}, []int{1020967150, 404597793, 1143503123, 164698463, 2004081065, 1337457847, 1308872918}},
		{[]int{409456482, 1543428783, 135589462, 1688687654, 1313059883, 348554791, 299198720, 1323721072, 1389838688, 822515643, 927970864, 1040608757, 1776611271, 1797713807, 712571504, 775475735, 1363147356, 787062335, 734743186, 334849816}, []int{1779280006, 450050841, 889363814, 1440765181, 1194153487, 1482798286, 28525033, 743091086, 1967868359, 423958824, 259288007, 640076739, 873173657, 1402881862, 627946497, 315209236, 676276018, 1482056562, 107131096, 295273407}, []int{175470567, 1993479624, 1024953276, 1116186914, 493947449, 1831353077, 327723753, 53546237, 1344441126, 1246474467, 1187258871, 1680685496, 636519007, 1187329748, 1340518001, 1090684971, 26157453, 255852976, 841874282, 630123223}},
		{[]int{439132273, 1435362348, 652986404, 595027578, 50394610, 1163471868, 1350110751, 1387888121, 1541711601, 1311011531, 629723242, 332422020, 1846595946, 1630183415, 892729502, 29895452, 1044010203}, []int{1813396452, 114068876, 1327268679, 1868447085, 184894747, 1182003852}, []int{239262804, 1549431224, 1980255083, 450208742, 235289357, 332209799, 1350110751, 1387888121, 1541711601, 1311011531, 629723242, 332422020, 1846595946, 1630183415, 892729502, 29895452, 1044010203}},
		{[]int{166180982, 764991101, 689087390, 429838129, 645158827, 1453030567, 1933567468, 1814820989, 457070860, 1832972348, 222162489, 312570738, 1353658637, 97753143, 1606729033, 596918423, 1097411730}, []int{682845415, 1075084129, 166827081, 1149467700, 750197496, 1980081828, 137604657, 584718339, 1309044764, 1639753374, 1544780495, 1889342289}, []int{849026397, 1840075230, 855914471, 1579305829, 1395356323, 1419846474, 57906204, 386273407, 1766115624, 1459459801, 1766942984, 188647106, 1353658637, 97753143, 1606729033, 596918423, 1097411730}},
	} {
		bb8a := make([]frontend.Variable, len(tc.a))
		for i := range tc.a {
			bb8a[i] = frontend.Variable(tc.a[i])
		}
		bb8b := make([]frontend.Variable, len(tc.b))
		for i := range tc.b {
			bb8b[i] = frontend.Variable(tc.b[i])
		}
		bb8c := make([]frontend.Variable, len(tc.c))
		for i := range tc.c {
			bb8c[i] = frontend.Variable(tc.c[i])
		}
		err := test.IsSolved(&AddCircuit{A: make([]frontend.Variable, len(bb8a)), B: make([]frontend.Variable, len(bb8b)), C: make([]frontend.Variable, len(bb8c))},
			&AddCircuit{A: bb8a, B: bb8b, C: bb8c}, babybear.Modulus())
		assert.NoError(err)
	}
}

type SubCircuit struct {
	A, B, C []frontend.Variable
}

func (c *SubCircuit) Define(api frontend.API) error {
	e, err := NewExtension(api)
	if err != nil {
		return err
	}
	res := e.Sub(c.A, c.B)
	e.AssertIsEqual(c.C, res)
	return nil
}

func TestSub(t *testing.T) {
	assert := test.NewAssert(t)
	for _, tc := range []struct {
		a, b, c []int
	}{
		{[]int{1146194893, 161636653, 1838869339, 53943494, 240077858, 1545249092, 1809326915, 1715283441, 1371628, 294589792, 350818866, 391858895, 1629176799, 601342455, 1570046548, 1407018614, 116964098}, []int{1358850047, 1241999865, 899127662}, []int{1800610767, 932902709, 939741677, 53943494, 240077858, 1545249092, 1809326915, 1715283441, 1371628, 294589792, 350818866, 391858895, 1629176799, 601342455, 1570046548, 1407018614, 116964098}},
		{[]int{1930372684, 1864892085, 1136595379, 1655262918, 778003842, 1395703951, 674238279, 303428310, 1869785911, 1465648550, 1654265669, 601993522, 1573728473, 678122861}, []int{758288169, 655811754, 1808890303}, []int{1172084515, 1209080331, 1340970997, 1655262918, 778003842, 1395703951, 674238279, 303428310, 1869785911, 1465648550, 1654265669, 601993522, 1573728473, 678122861}},
		{[]int{198803940, 683262254, 1171724940, 220582, 1436309010, 1011767254, 1619789563, 984205254, 1230618647, 661342751, 1574746193, 850095862, 1888386567}, []int{29202675, 1965459445, 1226138134, 614755, 823163111, 1965257586, 570492890, 714310672, 1863719043, 316112110, 751275028, 1305876957, 76087403, 289554855, 543603956, 1343584811}, []int{169601265, 731068730, 1958852727, 2012871748, 613145899, 1059775589, 1049296673, 269894582, 1380165525, 345230641, 823471165, 1557484826, 1812299164, 1723711066, 1469661965, 669681110}},
		{[]int{867771133, 1674871834, 173849765, 1667039402, 1926702105, 192555144}, []int{793120622, 876063077, 577433800, 1846006825, 1905707677, 1851151225}, []int{74650511, 798808757, 1609681886, 1834298498, 20994428, 354669840}},
		{[]int{56091872, 813716739, 362113363, 1053599731, 178619716, 1801257436, 864815551, 1305284265, 340955220, 1066690326, 674386095, 370881527, 1974134341, 167570042, 1480417387, 190897437}, []int{414748620, 1946157966, 678505871, 1157487387, 1854184016, 438292057, 1226900614, 2009898878, 557555644, 1058000961, 951280428, 1740323340, 1389148174, 315149809, 1822366716, 1274014418, 1803141600, 27865225}, []int{1654609173, 880824694, 1696873413, 1909378265, 337701621, 1362965379, 1651180858, 1308651308, 1796665497, 8689365, 1736371588, 643824108, 584986167, 1865686154, 1671316592, 930148940, 210124321, 1985400696}},
		{[]int{894252018, 1208416601, 802813920, 406175937, 1248756763, 2010718340, 132883210, 808520913}, []int{521827370, 428787881, 1443028395, 248442971, 1526599792, 1784112161, 1259960262, 1432566078, 234210554, 377567478, 1616559930, 1457879671, 1783692545, 1166700134, 63192557, 238060092, 1077493263}, []int{372424648, 779628720, 1373051446, 157732966, 1735422892, 226606179, 886188869, 1389220756, 1779055367, 1635698443, 396705991, 555386250, 229573376, 846565787, 1950073364, 1775205829, 935772658}},
		{[]int{1222756305, 1532801094, 1965391915, 1635685881, 1432129702, 1842258559, 818133559, 126161692, 1872764052, 1885587202, 388899896, 1271969485, 1753820414, 551808295, 272431669, 879739774, 672550552}, []int{296928275, 1436034937, 1801721783, 1498823779, 841763593, 248672479, 124418116, 1495721918, 555622041, 962101046, 1267239367, 1607045139, 1006652808, 369825252, 1129445804}, []int{925828030, 96766157, 163670132, 136862102, 590366109, 1593586080, 693715443, 643705695, 1317142011, 923486156, 1134926450, 1678190267, 747167606, 181983043, 1156251786, 879739774, 672550552}},
		{[]int{1731622955, 615410865, 1558496679, 195832953, 78170750, 61301540, 424972314, 1058412714}, []int{1077167652, 248376566, 1905047628, 1483682839, 135881338, 1082317338, 975917104, 914666340}, []int{654455303, 367034299, 1666714972, 725416035, 1955555333, 992250123, 1462321131, 143746374}},
		{[]int{1362372527, 1429758972, 1923199203, 808799268, 908434557, 22885471, 289022981, 655969201, 944182779, 947702885}, []int{1640391773, 1285351917, 1033611649, 157640943, 584694384}, []int{1735246675, 144407055, 889587554, 651158325, 323740173, 22885471, 289022981, 655969201, 944182779, 947702885}},
	} {
		bb8a := make([]frontend.Variable, len(tc.a))
		for i := range tc.a {
			bb8a[i] = frontend.Variable(tc.a[i])
		}
		bb8b := make([]frontend.Variable, len(tc.b))
		for i := range tc.b {
			bb8b[i] = frontend.Variable(tc.b[i])
		}
		bb8c := make([]frontend.Variable, len(tc.c))
		for i := range tc.c {
			bb8c[i] = frontend.Variable(tc.c[i])
		}
		err := test.IsSolved(&SubCircuit{A: make([]frontend.Variable, len(bb8a)), B: make([]frontend.Variable, len(bb8b)), C: make([]frontend.Variable, len(bb8c))},
			&SubCircuit{A: bb8a, B: bb8b, C: bb8c}, babybear.Modulus())
		assert.NoError(err)
	}
}

type MulCircuit struct {
	A, B, C []frontend.Variable
}

func (c *MulCircuit) Define(api frontend.API) error {
	e, err := NewExtension(api)
	if err != nil {
		return err
	}
	res := e.Mul(c.A, c.B)
	e.AssertIsEqual(c.C, res)
	return nil
}

func TestMul(t *testing.T) {
	assert := test.NewAssert(t)
	for _, tc := range []struct {
		a, b, c []int
	}{
		{[]int{234968604, 1416371157, 1226800682, 893689929, 1778035510, 146580532, 280014629, 1865717137, 982812264, 531104756, 624717176}, []int{1870372928, 89929324, 1716676259}, []int{1906632739, 115903316, 672362298, 305415989, 834985591, 1605817228, 1210941820, 985790928}},
		{[]int{1087077945, 320581995, 1629282702, 1741108544, 1040857706, 1916768501, 1565495085, 823889356, 1417428004, 1583630854, 1114754081, 1910869750, 187917565, 1438312600}, []int{1563899835, 168797949, 1371079710, 2987340, 1026622935, 1246885219, 506032556, 1788593166, 237013976, 1824355399, 625048497, 68448670, 1607339381, 951954832, 885388282, 683432779, 1575631187}, []int{720142330, 1056607600, 416577423, 1478261035, 1220299325, 903507263, 959938193, 355726286}},
		{[]int{1270288480, 1120584133, 721331187, 1421659182, 1094444484, 359616929, 969570910, 1882596876, 1297123805, 1881461151, 97448081}, []int{2003818742, 1858628164, 1023684969, 1085350554, 781453742, 1116677995, 1468065106, 1335317024, 1486544729, 1673869660, 144423861}, []int{426591007, 224682575, 778802683, 1271911177, 251644533, 207528538, 1964476679, 1876339154}},
		{[]int{1853783118, 1380960591, 964095257, 695279244, 315564693, 1867490771, 53851649, 1343775624, 653780889, 1583674803}, []int{1969070769, 1769394471, 414599120, 647597532, 1788546055, 224442741, 1412932412, 680401167, 298718932, 1146328071, 1478899454, 1909103677, 1428990649, 1439633502, 54662272, 596249162, 461878709, 563248862, 1000459500, 1645847614}, []int{504670415, 227198315, 1349561269, 501560516, 894895922, 1993202942, 1850127592, 1108750151}},
		{[]int{414037409, 953085481, 1924772772, 1517340116, 1237653110, 133837088, 1315588440, 238864701}, []int{1304959022, 1925100119, 978981709, 1918377397, 1207231558, 281134995, 502889770}, []int{142108887, 1889729586, 1250323514, 594024056, 1230999660, 1787836861, 1534177366, 801580624}},
		{[]int{947344343, 423823149, 344707902, 700248832, 566581327, 1849547514, 399209144, 1091850846, 1364174972, 1803614392, 1634840199, 1026184357, 1838704001, 203731055, 743992513, 251080705, 1036651012, 759652320, 577883317, 1716209722, 1529813228}, []int{886233417, 796245045, 174590451, 434936528, 626331990}, []int{1530952022, 551906495, 1612686281, 1586062699, 971563806, 686771089, 907205414, 1181573107}},
		{[]int{7027166, 463591444, 1803846561, 1505438619, 2012281334, 1039204555, 1439978503, 1620975569, 870977727, 746630744, 1686836478, 1924796057}, []int{819644881, 463949651, 522020012, 1377665054, 831007978, 1954765014, 1976440214, 1392258642, 52259004, 1536634317, 1591661847, 628460335, 150161825, 1915169606, 671751539, 196434398, 1160799204, 1385730435, 583362563}, []int{128480043, 456509411, 299825145, 1189434742, 1790453058, 918340297, 1075473370, 992322875}},
		{[]int{1711268919, 1677353510}, []int{315387358, 877475853, 1779977986, 1816170934, 1740575889, 1377265373, 2007938566, 486612909, 953317838, 150150087, 1034065308, 130344828, 1720755480, 1194766973, 74573519, 511551933, 1766944307, 214027799, 226716130, 1055958601, 1902536491}, []int{1633902657, 998090610, 1658849111, 175196657, 889509699, 1868699941, 1847876682, 1348099822}},
		{[]int{1926641336, 233517016, 1382898361, 516240145, 730324703, 196139649, 1751487986, 1718388392, 93866265, 234489342, 1447664327, 978489786, 629636261}, []int{1812584753, 1146117727, 185071390}, []int{1032160623, 1983522902, 1373330365, 1300425158, 649650457, 1711009205, 567980350, 546199098}},
	} {
		bb8a := make([]frontend.Variable, len(tc.a))
		for i := range tc.a {
			bb8a[i] = frontend.Variable(tc.a[i])
		}
		bb8b := make([]frontend.Variable, len(tc.b))
		for i := range tc.b {
			bb8b[i] = frontend.Variable(tc.b[i])
		}
		bb8c := make([]frontend.Variable, len(tc.c))
		for i := range tc.c {
			bb8c[i] = frontend.Variable(tc.c[i])
		}
		err := test.IsSolved(&MulCircuit{A: make([]frontend.Variable, len(bb8a)), B: make([]frontend.Variable, len(bb8b)), C: make([]frontend.Variable, len(bb8c))},
			&MulCircuit{A: bb8a, B: bb8b, C: bb8c}, babybear.Modulus())
		assert.NoError(err)
	}
}
