// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by gnark DO NOT EDIT

package gkr

import (
	"encoding/json"
	"fmt"
	"hash"
	"os"
	"path/filepath"
	"reflect"

	"github.com/consensys/bavard"
	fiatshamir "github.com/consensys/gnark-crypto/fiat-shamir"
	"github.com/consensys/gnark/internal/gkr/gkrtesting"
	"github.com/consensys/gnark/internal/gkr/gkrtypes"
	"github.com/consensys/gnark/internal/small_rational"
	"github.com/consensys/gnark/internal/small_rational/polynomial"
	"github.com/consensys/gnark/internal/utils"
)

func GenerateVectors() error {
	testDirPath, err := filepath.Abs("../../gkr/test_vectors")
	if err != nil {
		return err
	}

	fmt.Printf("generating GKR test cases: scanning directory %s for test specs\n", testDirPath)

	dirEntries, err := os.ReadDir(testDirPath)
	if err != nil {
		return err
	}
	for _, dirEntry := range dirEntries {
		if !dirEntry.IsDir() {

			if filepath.Ext(dirEntry.Name()) == ".json" {
				path := filepath.Join(testDirPath, dirEntry.Name())
				if !bavard.ShouldGenerate(path) {
					continue
				}
				fmt.Println("\tprocessing", dirEntry.Name())
				if err = run(path); err != nil {
					return err
				}
			}
		}
	}

	return nil
}

func run(absPath string) error {
	testCase, err := newTestCase(absPath)
	if err != nil {
		return err
	}

	transcriptSetting := fiatshamir.WithHash(testCase.Hash)

	var proof Proof
	proof, err = Prove(testCase.Circuit, testCase.FullAssignment, transcriptSetting)
	if err != nil {
		return err
	}

	if testCase.Info.Proof, err = toPrintableProof(proof); err != nil {
		return err
	}
	var outBytes []byte
	if outBytes, err = json.MarshalIndent(testCase.Info, "", "\t"); err == nil {
		if err = os.WriteFile(absPath, outBytes, 0); err != nil {
			return err
		}
	} else {
		return err
	}

	testCase, err = newTestCase(absPath)
	if err != nil {
		return err
	}

	err = Verify(testCase.Circuit, testCase.InOutAssignment, proof, transcriptSetting)
	if err != nil {
		return err
	}

	testCase, err = newTestCase(absPath)
	if err != nil {
		return err
	}

	err = Verify(testCase.Circuit, testCase.InOutAssignment, proof, fiatshamir.WithHash(newMessageCounter(2, 0)))
	if err == nil {
		return fmt.Errorf("bad proof accepted")
	}
	return nil
}

func toPrintableProof(proof Proof) (gkrtesting.PrintableProof, error) {
	res := make(gkrtesting.PrintableProof, len(proof))

	for i := range proof {

		partialSumPolys := make([][]interface{}, len(proof[i].partialSumPolys))
		for k, partialK := range proof[i].partialSumPolys {
			partialSumPolys[k] = elementSliceToInterfaceSlice(partialK)
		}

		res[i] = gkrtesting.PrintableSumcheckProof{
			FinalEvalProof:  elementSliceToInterfaceSlice(proof[i].finalEvalProof),
			PartialSumPolys: partialSumPolys,
		}
	}
	return res, nil
}

func elementToInterface(x *small_rational.SmallRational) interface{} {
	if i := x.BigInt(nil); i != nil {
		return i
	}
	return x.Text(10)
}

func elementSliceToInterfaceSlice(x interface{}) []interface{} {
	if x == nil {
		return nil
	}

	X := reflect.ValueOf(x)

	res := make([]interface{}, X.Len())
	for i := range res {
		xI := X.Index(i).Interface().(small_rational.SmallRational)
		res[i] = elementToInterface(&xI)
	}
	return res
}

func elementSliceSliceToInterfaceSliceSlice(x interface{}) [][]interface{} {
	if x == nil {
		return nil
	}

	X := reflect.ValueOf(x)

	res := make([][]interface{}, X.Len())
	for i := range res {
		res[i] = elementSliceToInterfaceSlice(X.Index(i).Interface())
	}

	return res
}

func unmarshalProof(printable gkrtesting.PrintableProof) (Proof, error) {
	proof := make(Proof, len(printable))
	for i := range printable {
		finalEvalProof := []small_rational.SmallRational(nil)

		if printable[i].FinalEvalProof != nil {
			finalEvalSlice := reflect.ValueOf(printable[i].FinalEvalProof)
			finalEvalProof = make([]small_rational.SmallRational, finalEvalSlice.Len())
			for k := range finalEvalProof {
				if _, err := finalEvalProof[k].SetInterface(finalEvalSlice.Index(k).Interface()); err != nil {
					return nil, err
				}
			}
		}

		proof[i] = sumcheckProof{
			partialSumPolys: make([]polynomial.Polynomial, len(printable[i].PartialSumPolys)),
			finalEvalProof:  finalEvalProof,
		}
		for k := range printable[i].PartialSumPolys {
			var err error
			if proof[i].partialSumPolys[k], err = sliceToElementSlice(printable[i].PartialSumPolys[k]); err != nil {
				return nil, err
			}
		}
	}
	return proof, nil
}

type TestCase struct {
	Circuit         gkrtypes.Circuit
	Hash            hash.Hash
	Proof           Proof
	FullAssignment  WireAssignment
	InOutAssignment WireAssignment
	Info            gkrtesting.TestCaseInfo // we are generating the test vectors, so we need to keep the circuit instance info to ADD the proof to it and resave it
}

var (
	testCases = make(map[string]*TestCase)
	cache     = gkrtesting.NewCache()
)

func newTestCase(path string) (*TestCase, error) {
	path, err := filepath.Abs(path)
	if err != nil {
		return nil, err
	}
	dir := filepath.Dir(path)

	tCase, ok := testCases[path]
	if ok {
		return tCase, nil
	}

	info, err := cache.ReadTestCaseInfo(path)
	if err != nil {
		return nil, err
	}

	circuit := cache.GetCircuit(filepath.Join(dir, info.Circuit))
	var _hash hash.Hash
	if _hash, err = hashFromDescription(info.Hash); err != nil {
		return nil, err
	}
	var proof Proof
	if proof, err = unmarshalProof(info.Proof); err != nil {
		return nil, err
	}

	fullAssignment := make(WireAssignment, len(circuit))
	inOutAssignment := make(WireAssignment, len(circuit))

	sorted := circuit.TopologicalSort()

	inI, outI := 0, 0
	for i, w := range sorted {
		var assignmentRaw []interface{}
		if w.IsInput() {
			if inI == len(info.Input) {
				return nil, fmt.Errorf("fewer input in vector than in circuit")
			}
			assignmentRaw = info.Input[inI]
			inI++
		} else if w.IsOutput() {
			if outI == len(info.Output) {
				return nil, fmt.Errorf("fewer output in vector than in circuit")
			}
			assignmentRaw = info.Output[outI]
			outI++
		}
		if assignmentRaw != nil {
			var wireAssignment []small_rational.SmallRational
			if wireAssignment, err = sliceToElementSlice(assignmentRaw); err != nil {
				return nil, err
			}

			fullAssignment[i] = wireAssignment
			inOutAssignment[i] = wireAssignment
		}
	}

	fullAssignment.Complete(utils.References(circuit))

	for i, w := range sorted {
		if w.IsOutput() {
			if err = sliceEquals(inOutAssignment[i], fullAssignment[i]); err != nil {
				return nil, fmt.Errorf("assignment mismatch: %v", err)
			}
		}
	}

	tCase = &TestCase{
		FullAssignment:  fullAssignment,
		InOutAssignment: inOutAssignment,
		Proof:           proof,
		Hash:            _hash,
		Circuit:         circuit,
		Info:            info,
	}

	testCases[path] = tCase

	return tCase, nil
}
