// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by gnark DO NOT EDIT

package gkr

import (
	"fmt"
	"math/big"

	"github.com/consensys/gnark-crypto/hash"
	"github.com/consensys/gnark-crypto/utils"
	hint "github.com/consensys/gnark/constraint/solver"
	"github.com/consensys/gnark/frontend"
	"github.com/consensys/gnark/internal/gkr/gkrtypes"
	algo_utils "github.com/consensys/gnark/internal/utils"

	"github.com/consensys/gnark-crypto/ecc/bw6-761/fr"
	fiatshamir "github.com/consensys/gnark-crypto/fiat-shamir"
)

type SolvingData struct {
	assignment WireAssignment
	circuit    gkrtypes.Circuit
	workers    *utils.WorkerPool
}

func (d *SolvingData) init(info gkrtypes.SolvingInfo) {
	d.workers = utils.NewWorkerPool()
	d.circuit = info.Circuit
	d.circuit.SetNbUniqueOutputs()

	d.assignment = make(WireAssignment, len(d.circuit))
	for i := range d.assignment {
		d.assignment[i] = make([]fr.Element, info.NbInstances)
	}
}

// this module assumes that wire and instance indexes respect dependencies

func setOuts(a WireAssignment, circuit gkrtypes.Circuit, outs []*big.Int) {
	outsI := 0
	for i := range circuit {
		if circuit[i].IsOutput() {
			for j := range a[i] {
				a[i][j].BigInt(outs[outsI])
				outsI++
			}
		}
	}
	// Check if outsI == len(outs)?
}

func SolveHint(info gkrtypes.SolvingInfo, data *SolvingData) hint.Hint {
	return func(_ *big.Int, ins, outs []*big.Int) error {
		// assumes assignmentVector is arranged wire first, instance second in order of solution
		offsets := info.AssignmentOffsets()
		data.init(info)
		maxNIn := data.circuit.MaxGateNbIn()

		chunks := info.Chunks()

		solveTask := func(chunkOffset int) utils.Task {
			return func(startInChunk, endInChunk int) {
				start := startInChunk + chunkOffset
				end := endInChunk + chunkOffset
				inputs := make([]frontend.Variable, maxNIn)
				dependencyHeads := make([]int, len(data.circuit)) // for each wire, which of its dependencies we would look at next
				for wI := range data.circuit {                    // skip instances that are not relevant (related to instances before the current task)
					deps := info.Dependencies[wI]
					dependencyHeads[wI] = algo_utils.BinarySearchFunc(func(i int) int {
						return deps[i].InputInstance
					}, len(deps), start)
				}

				for instanceI := start; instanceI < end; instanceI++ {
					for wireI := range data.circuit {
						wire := &data.circuit[wireI]
						deps := info.Dependencies[wireI]
						if wire.IsInput() {
							if dependencyHeads[wireI] < len(deps) && instanceI == deps[dependencyHeads[wireI]].InputInstance {
								dep := deps[dependencyHeads[wireI]]
								data.assignment[wireI][instanceI].Set(&data.assignment[dep.OutputWire][dep.OutputInstance])
								dependencyHeads[wireI]++
							} else {
								data.assignment[wireI][instanceI].SetBigInt(ins[offsets[wireI]+instanceI-dependencyHeads[wireI]])
							}
						} else {
							// assemble the inputs
							inputIndexes := info.Circuit[wireI].Inputs
							for i, inputI := range inputIndexes {
								inputs[i] = &data.assignment[inputI][instanceI]
							}
							gate := data.circuit[wireI].Gate
							data.assignment[wireI][instanceI].Set(gate.Evaluate(api, inputs[:len(inputIndexes)]...).(*fr.Element))
						}
					}
				}
			}
		}

		start := 0
		for _, end := range chunks {
			data.workers.Submit(end-start, solveTask(start), 1024).Wait()
			start = end
		}

		for _, p := range info.Prints {
			serializable := make([]any, len(p.Values))
			for i, v := range p.Values {
				if p.IsGkrVar[i] { // serializer stores uint32 in slices as uint64
					serializable[i] = data.assignment[algo_utils.ForceUint32(v)][p.Instance].String()
				} else {
					serializable[i] = v
				}
			}
			fmt.Println(serializable...)
		}

		setOuts(data.assignment, info.Circuit, outs)

		return nil
	}
}

func ProveHint(hashName string, data *SolvingData) hint.Hint {

	return func(_ *big.Int, ins, outs []*big.Int) error {
		insBytes := algo_utils.Map(ins[1:], func(i *big.Int) []byte { // the first input is dummy, just to ensure the solver's work is done before the prover is called
			b := make([]byte, fr.Bytes)
			i.FillBytes(b)
			return b[:]
		})

		hsh := hash.NewHash(hashName + "_BW6_761")

		proof, err := Prove(data.circuit, data.assignment, fiatshamir.WithHash(hsh, insBytes...), WithWorkers(data.workers))
		if err != nil {
			return err
		}

		return proof.SerializeToBigInts(outs)

	}
}
