// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by gnark DO NOT EDIT

package gkr

import (
	"fmt"

	"github.com/consensys/gnark-crypto/ecc"
	"github.com/consensys/gnark/internal/gkr/gkrtypes"

	"slices"

	"github.com/consensys/gnark-crypto/ecc/bls24-317/fr"
	"github.com/consensys/gnark-crypto/ecc/bls24-317/fr/fft"
	"github.com/consensys/gnark-crypto/ecc/bls24-317/fr/polynomial"
	"github.com/consensys/gnark/std/gkrapi/gkr"
)

// IsGateFunctionAdditive returns whether x_i occurs only in a monomial of total degree 1 in f
func IsGateFunctionAdditive(f gkr.GateFunction, i, nbIn int) bool {
	fWrapped := api.convertFunc(f)

	// fix all variables except the i-th one at random points
	// pick random value x1 for the i-th variable
	// check if f(-, 0, -) + f(-, 2*x1, -) = 2*f(-, x1, -)
	x := make(fr.Vector, nbIn)
	x.MustSetRandom()
	x0 := x[i]
	x[i].SetZero()
	in := slices.Clone(x)
	y0 := fWrapped(in...)

	x[i] = x0
	copy(in, x)
	y1 := fWrapped(in...)

	x[i].Double(&x[i])
	copy(in, x)
	y2 := fWrapped(in...)

	y2.Sub(y2, y1)
	y1.Sub(y1, y0)

	if !y2.Equal(y1) {
		return false // not linear
	}

	// check if the coefficient of x_i is nonzero and independent of the other variables (so that we know it is ALWAYS nonzero)
	if y1.IsZero() { // f(-, x1, -) = f(-, 0, -), so the coefficient of x_i is 0
		return false
	}

	// compute the slope with another assignment for the other variables
	x.MustSetRandom()
	x[i].SetZero()
	copy(in, x)
	y0 = fWrapped(in...)

	x[i] = x0
	copy(in, x)
	y1 = fWrapped(in...)

	y1.Sub(y1, y0)

	return y1.Equal(y2)
}

// fitPoly tries to fit a polynomial of degree less than degreeBound to f.
// degreeBound must be a power of 2.
// It returns the polynomial if successful, nil otherwise
func (f gateFunctionFr) fitPoly(nbIn int, degreeBound uint64) polynomial.Polynomial {
	// turn f univariate by defining p(x) as f(x, rx, ..., sx)
	// where r, s, ... are random constants
	fIn := make([]fr.Element, nbIn)
	consts := make(fr.Vector, nbIn-1)
	consts.MustSetRandom()

	p := make(polynomial.Polynomial, degreeBound)
	domain := fft.NewDomain(degreeBound)
	// evaluate p on the unit circle (first filling p with evaluations rather than coefficients)
	x := fr.One()
	for i := range p {
		fIn[0] = x
		for j := range consts {
			fIn[j+1].Mul(&x, &consts[j])
		}
		p[i].Set(f(fIn...))

		x.Mul(&x, &domain.Generator)
	}

	// obtain p's coefficients
	domain.FFTInverse(p, fft.DIF)
	fft.BitReverse(p)

	// check if p is equal to f. This not being the case means that f is of a degree higher than degreeBound
	fIn[0].MustSetRandom()
	for i := range consts {
		fIn[i+1].Mul(&fIn[0], &consts[i])
	}
	pAt := p.Eval(&fIn[0])
	fAt := f(fIn...)
	if !pAt.Equal(fAt) {
		return nil
	}

	// trim p
	lastNonZero := len(p) - 1
	for lastNonZero >= 0 && p[lastNonZero].IsZero() {
		lastNonZero--
	}
	return p[:lastNonZero+1]
}

// FindGateFunctionDegree returns the degree of the gate function, or -1 if it fails.
// Failure could be due to the degree being higher than max or the function not being a polynomial at all.
func FindGateFunctionDegree(f gkr.GateFunction, max, nbIn int) (int, error) {
	fFr := api.convertFunc(f)
	bound := uint64(max) + 1
	for degreeBound := uint64(4); degreeBound <= bound; degreeBound *= 8 {
		if p := fFr.fitPoly(nbIn, degreeBound); p != nil {
			if len(p) == 0 {
				return -1, gkrtypes.ErrZeroFunction
			}
			return len(p) - 1, nil
		}
	}
	return -1, fmt.Errorf("could not find a degree: tried up to %d", max)
}

func VerifyGateFunctionDegree(f gkr.GateFunction, claimedDegree, nbIn int) error {
	fFr := api.convertFunc(f)
	if p := fFr.fitPoly(nbIn, ecc.NextPowerOfTwo(uint64(claimedDegree)+1)); p == nil {
		return fmt.Errorf("detected a higher degree than %d", claimedDegree)
	} else if len(p) == 0 {
		return gkrtypes.ErrZeroFunction
	} else if len(p)-1 != claimedDegree {
		return fmt.Errorf("detected degree %d, claimed %d", len(p)-1, claimedDegree)
	}
	return nil
}
