// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by gnark DO NOT EDIT

package cs

import (
	"encoding/binary"
	"errors"
	"math/big"

	"github.com/consensys/gnark/constraint"
	"github.com/consensys/gnark/internal/utils"

	fr "github.com/consensys/gnark/internal/smallfields/tinyfield"
)

// CoeffTable ensure we store unique coefficients in the constraint system
type CoeffTable struct {
	Coefficients []fr.Element
	mCoeffs      map[fr.Element]uint32 // maps coefficient to coeffID
}

func newCoeffTable(capacity int) CoeffTable {
	r := CoeffTable{
		Coefficients: make([]fr.Element, 5, 5+capacity),
		mCoeffs:      make(map[fr.Element]uint32, capacity),
	}

	r.Coefficients[constraint.CoeffIdZero].SetUint64(0)
	r.Coefficients[constraint.CoeffIdOne].SetOne()
	r.Coefficients[constraint.CoeffIdTwo].SetUint64(2)
	r.Coefficients[constraint.CoeffIdMinusOne].SetInt64(-1)
	r.Coefficients[constraint.CoeffIdMinusTwo].SetInt64(-2)

	return r

}

func (ct *CoeffTable) toBytes() []byte {
	buf := make([]byte, 0, 8+len(ct.Coefficients)*fr.Bytes)
	ctLen := uint64(len(ct.Coefficients))

	buf = binary.LittleEndian.AppendUint64(buf, ctLen)
	for _, c := range ct.Coefficients {
		for _, w := range c {
			buf = binary.LittleEndian.AppendUint32(buf, w)
		}
	}

	return buf
}

func (ct *CoeffTable) fromBytes(buf []byte) error {
	if len(buf) < 8 {
		return errors.New("invalid buffer size")
	}
	ctLen := binary.LittleEndian.Uint64(buf[:8])
	buf = buf[8:]

	if uint64(len(buf)) < ctLen*fr.Bytes {
		return errors.New("invalid buffer size")
	}
	ct.Coefficients = make([]fr.Element, ctLen)
	for i := uint64(0); i < ctLen; i++ {
		var c fr.Element
		k := int(i) * fr.Bytes
		for j := 0; j < fr.Limbs; j++ {
			c[j] = binary.LittleEndian.Uint32(buf[k+j*4 : k+(j+1)*4])
		}
		ct.Coefficients[i] = c
	}
	return nil
}

func (ct *CoeffTable) AddCoeff(coeff constraint.U32) uint32 {
	c := (*fr.Element)(coeff[:])
	var cID uint32
	if c.IsZero() {
		cID = constraint.CoeffIdZero
	} else if c.IsOne() {
		cID = constraint.CoeffIdOne
	} else if c.Equal(&two) {
		cID = constraint.CoeffIdTwo
	} else if c.Equal(&minusOne) {
		cID = constraint.CoeffIdMinusOne
	} else if c.Equal(&minusTwo) {
		cID = constraint.CoeffIdMinusTwo
	} else {
		cc := *c
		if id, ok := ct.mCoeffs[cc]; ok {
			cID = id
		} else {
			cID = uint32(len(ct.Coefficients))
			ct.Coefficients = append(ct.Coefficients, cc)
			ct.mCoeffs[cc] = cID
		}
	}
	return cID
}

func (ct *CoeffTable) MakeTerm(coeff constraint.U32, variableID int) constraint.Term {
	cID := ct.AddCoeff(coeff)
	return constraint.Term{VID: uint32(variableID), CID: cID}
}

// CoeffToString implements constraint.Resolver
func (ct *CoeffTable) CoeffToString(cID int) string {
	return ct.Coefficients[cID].String()
}

// implements constraint.Field
type field struct{}

var _ constraint.Field[constraint.U32] = &field{}

var (
	two      fr.Element
	minusOne fr.Element
	minusTwo fr.Element
)

func init() {
	minusOne.SetOne()
	minusOne.Neg(&minusOne)
	two.SetOne()
	two.Double(&two)
	minusTwo.Neg(&two)
}

func (engine *field) FromInterface(i interface{}) constraint.U32 {
	var e fr.Element
	if _, err := e.SetInterface(i); err != nil {
		// need to clean that --> some code path are dissimilar
		// for example setting a fr.Element from an fp.Element
		// fails with the above but succeeds through big int... (2-chains)
		b := utils.FromInterface(i)
		e.SetBigInt(&b)
	}
	var r constraint.U32
	copy(r[:], e[:])
	return r
}
func (engine *field) ToBigInt(c constraint.U32) *big.Int {
	e := (*fr.Element)(c[:])
	r := new(big.Int)
	e.BigInt(r)
	return r

}
func (engine *field) Mul(a, b constraint.U32) constraint.U32 {
	_a := (*fr.Element)(a[:])
	_b := (*fr.Element)(b[:])
	_a.Mul(_a, _b)
	return a
}

func (engine *field) Add(a, b constraint.U32) constraint.U32 {
	_a := (*fr.Element)(a[:])
	_b := (*fr.Element)(b[:])
	_a.Add(_a, _b)
	return a
}
func (engine *field) Sub(a, b constraint.U32) constraint.U32 {
	_a := (*fr.Element)(a[:])
	_b := (*fr.Element)(b[:])
	_a.Sub(_a, _b)
	return a
}
func (engine *field) Neg(a constraint.U32) constraint.U32 {
	e := (*fr.Element)(a[:])
	e.Neg(e)
	return a

}
func (engine *field) Inverse(a constraint.U32) (constraint.U32, bool) {
	if a.IsZero() {
		return a, false
	}
	e := (*fr.Element)(a[:])
	if e.IsZero() {
		return a, false
	} else if e.IsOne() {
		return a, true
	}
	var t fr.Element
	t.Neg(e)
	if t.IsOne() {
		return a, true
	}

	e.Inverse(e)
	return a, true
}

func (engine *field) IsOne(a constraint.U32) bool {
	e := (*fr.Element)(a[:])
	return e.IsOne()
}

func (engine *field) One() constraint.U32 {
	e := fr.One()
	var r constraint.U32
	copy(r[:], e[:])
	return r
}

func (engine *field) String(a constraint.U32) string {
	e := (*fr.Element)(a[:])
	return e.String()
}

func (engine *field) Uint64(a constraint.U32) (uint64, bool) {
	e := (*fr.Element)(a[:])
	if !e.IsUint64() {
		return 0, false
	}
	return e.Uint64(), true
}
