// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by gnark DO NOT EDIT

package groth16

import (
	"bytes"
	"crypto/sha256"
	"errors"
	"fmt"
	"io"
	"math/big"
	"text/template"
	"time"

	"github.com/consensys/gnark-crypto/ecc/bn254/fp"
	"golang.org/x/crypto/sha3"

	"github.com/consensys/gnark-crypto/ecc"
	curve "github.com/consensys/gnark-crypto/ecc/bn254"
	"github.com/consensys/gnark-crypto/ecc/bn254/fr"
	"github.com/consensys/gnark-crypto/ecc/bn254/fr/hash_to_field"
	"github.com/consensys/gnark-crypto/ecc/bn254/fr/pedersen"
	"github.com/consensys/gnark/backend"
	"github.com/consensys/gnark/backend/solidity"
	"github.com/consensys/gnark/constraint"
	"github.com/consensys/gnark/logger"
)

var (
	errPairingCheckFailed         = errors.New("pairing doesn't match")
	errCorrectSubgroupCheckFailed = errors.New("points in the proof are not in the correct subgroup")
)

// Verify verifies a proof with given VerifyingKey and publicWitness
func Verify(proof *Proof, vk *VerifyingKey, publicWitness fr.Vector, opts ...backend.VerifierOption) error {
	opt, err := backend.NewVerifierConfig(opts...)
	if err != nil {
		return fmt.Errorf("new verifier config: %w", err)
	}
	if opt.HashToFieldFn == nil {
		opt.HashToFieldFn = hash_to_field.New([]byte(constraint.CommitmentDst))
	}

	nbPublicVars := len(vk.G1.K) - len(vk.PublicAndCommitmentCommitted)

	if len(publicWitness) != nbPublicVars-1 {
		return fmt.Errorf("invalid witness size, got %d, expected %d (public - ONE_WIRE)", len(publicWitness), len(vk.G1.K)-1)
	}
	log := logger.Logger().With().Str("curve", vk.CurveID().String()).Str("backend", "groth16").Logger()
	start := time.Now()

	// check that the points in the proof are in the correct subgroup
	if !proof.isValid() {
		return errCorrectSubgroupCheckFailed
	}

	var doubleML curve.GT
	chDone := make(chan error, 1)

	// compute (eKrsδ, eArBs)
	go func() {
		var errML error
		doubleML, errML = curve.MillerLoop([]curve.G1Affine{proof.Krs, proof.Ar}, []curve.G2Affine{vk.G2.deltaNeg, proof.Bs})
		chDone <- errML
		close(chDone)
	}()

	maxNbPublicCommitted := 0
	for _, s := range vk.PublicAndCommitmentCommitted { // iterate over commitments
		maxNbPublicCommitted = max(maxNbPublicCommitted, len(s))
	}
	commitmentsSerialized := make([]byte, len(vk.PublicAndCommitmentCommitted)*fr.Bytes)
	commitmentPrehashSerialized := make([]byte, curve.SizeOfG1AffineUncompressed+maxNbPublicCommitted*fr.Bytes)
	for i := range vk.PublicAndCommitmentCommitted { // solveCommitmentWire
		copy(commitmentPrehashSerialized, proof.Commitments[i].Marshal())
		offset := curve.SizeOfG1AffineUncompressed
		for j := range vk.PublicAndCommitmentCommitted[i] {
			copy(commitmentPrehashSerialized[offset:], publicWitness[vk.PublicAndCommitmentCommitted[i][j]-1].Marshal())
			offset += fr.Bytes
		}
		opt.HashToFieldFn.Write(commitmentPrehashSerialized[:offset])
		hashBts := opt.HashToFieldFn.Sum(nil)
		opt.HashToFieldFn.Reset()
		nbBuf := fr.Bytes
		if opt.HashToFieldFn.Size() < fr.Bytes {
			nbBuf = opt.HashToFieldFn.Size()
		}
		var res fr.Element
		res.SetBytes(hashBts[:nbBuf])
		publicWitness = append(publicWitness, res)
		copy(commitmentsSerialized[i*fr.Bytes:], res.Marshal())
	}
	if len(vk.CommitmentKeys) > 0 {
		challenge, err := fr.Hash(commitmentsSerialized, []byte("G16-BSB22"), 1)
		if err != nil {
			return err
		}
		if err = pedersen.BatchVerifyMultiVk(vk.CommitmentKeys, proof.Commitments, []curve.G1Affine{proof.CommitmentPok}, challenge[0]); err != nil {
			return err
		}
	}

	// compute e(Σx.[Kvk(t)]1, -[γ]2)
	var kSum curve.G1Jac
	if _, err := kSum.MultiExp(vk.G1.K[1:], publicWitness, ecc.MultiExpConfig{}); err != nil {
		return err
	}
	kSum.AddMixed(&vk.G1.K[0])

	for i := range proof.Commitments {
		kSum.AddMixed(&proof.Commitments[i])
	}

	var kSumAff curve.G1Affine
	kSumAff.FromJacobian(&kSum)

	right, err := curve.MillerLoop([]curve.G1Affine{kSumAff}, []curve.G2Affine{vk.G2.gammaNeg})
	if err != nil {
		return err
	}

	// wait for (eKrsδ, eArBs)
	if err := <-chDone; err != nil {
		return err
	}

	right = curve.FinalExponentiation(&right, &doubleML)
	if !vk.e.Equal(&right) {
		return errPairingCheckFailed
	}

	log.Debug().Dur("took", time.Since(start)).Msg("verifier done")
	return nil
}

// ExportSolidity writes a solidity Verifier contract on provided writer.
// This is an experimental feature and gnark solidity generator as not been thoroughly tested.
//
// See https://github.com/Consensys/gnark-tests for example usage.
func (vk *VerifyingKey) ExportSolidity(w io.Writer, exportOpts ...solidity.ExportOption) error {
	cfg, err := solidity.NewExportConfig(exportOpts...)
	log := logger.Logger()
	if err != nil {
		return err
	}
	if cfg.HashToFieldFn == nil {
		// set the target hash function to legacy keccak256 as it is the default for `solidity.WithTargetSolidityVerifier``
		cfg.HashToFieldFn = sha3.NewLegacyKeccak256()
		log.Debug().Msg("hash to field function not set, using keccak256 as default")
	}
	// a bit hacky way to understand what hash function is provided. We already
	// receive instance of hash function but it is difficult to compare it with
	// sha256.New() or sha3.NewLegacyKeccak256() directly.
	//
	// So, we hash an empty input and compare the outputs.
	cfg.HashToFieldFn.Reset()
	hashBts := cfg.HashToFieldFn.Sum(nil)
	var hashFnName string
	if bytes.Equal(hashBts, sha256.New().Sum(nil)) {
		hashFnName = "sha256"
	} else if bytes.Equal(hashBts, sha3.NewLegacyKeccak256().Sum(nil)) {
		hashFnName = "keccak256"
	} else {
		return fmt.Errorf("unsupported hash function used, only supported sha256 and legacy keccak256")
	}
	cfg.HashToFieldFn.Reset()
	helpers := template.FuncMap{
		"sum": func(a, b int) int {
			return a + b
		},
		"sub": func(a, b int) int {
			return a - b
		},
		"mul": func(a, b int) int {
			return a * b
		},
		"intRange": func(max int) []int {
			out := make([]int, max)
			for i := 0; i < max; i++ {
				out[i] = i
			}
			return out
		},
		"fpstr": func(x fp.Element) string {
			bv := new(big.Int)
			x.BigInt(bv)
			return bv.String()
		},
		"hashFnName": func() string {
			return hashFnName
		},
	}

	if len(vk.PublicAndCommitmentCommitted) > 1 {
		log.Warn().Msg("exporting solidity verifier with more than one commitment is not supported")
	} else if len(vk.PublicAndCommitmentCommitted) == 1 {
		log.Warn().Msg("exporting solidity verifier only supports `sha256` as `HashToField`. The generated contract may not work for proofs generated with other hash functions.")
	}

	tmpl, err := template.New("").Funcs(helpers).Parse(solidityTemplate)
	if err != nil {
		return err
	}

	// negate Beta, Gamma and Delta, to avoid negating proof elements in the verifier
	var betaNeg curve.G2Affine
	betaNeg.Neg(&vk.G2.Beta)
	beta := vk.G2.Beta
	vk.G2.Beta = betaNeg
	vk.G2.Gamma, vk.G2.gammaNeg = vk.G2.gammaNeg, vk.G2.Gamma
	vk.G2.Delta, vk.G2.deltaNeg = vk.G2.deltaNeg, vk.G2.Delta

	// execute template
	err = tmpl.Execute(w, struct {
		Cfg solidity.ExportConfig
		Vk  VerifyingKey
	}{
		Cfg: cfg,
		Vk:  *vk,
	})

	// restore Beta, Gamma and Delta
	vk.G2.Beta = beta
	vk.G2.Gamma, vk.G2.gammaNeg = vk.G2.gammaNeg, vk.G2.Gamma
	vk.G2.Delta, vk.G2.deltaNeg = vk.G2.deltaNeg, vk.G2.Delta

	return err
}
