// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by gnark DO NOT EDIT

package groth16

import (
	curve "github.com/consensys/gnark-crypto/ecc/bls12-377"

	"github.com/consensys/gnark-crypto/ecc/bls12-377/fr/fft"

	"github.com/consensys/gnark-crypto/ecc/bls12-377/fr/pedersen"
	"github.com/consensys/gnark/backend/groth16/internal/test_utils"
	"github.com/consensys/gnark/io"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"

	"math/big"

	"github.com/leanovate/gopter"
	"github.com/leanovate/gopter/gen"
	"github.com/leanovate/gopter/prop"

	"testing"
)

func TestProofSerialization(t *testing.T) {
	parameters := gopter.DefaultTestParameters()
	parameters.MinSuccessfulTests = 10

	properties := gopter.NewProperties(parameters)

	properties.Property("Proof -> writer -> reader -> Proof should stay constant", prop.ForAll(
		func(ar, krs curve.G1Affine, bs curve.G2Affine) bool {
			var proof Proof

			// create a random proof
			proof.Ar = ar
			proof.Krs = krs
			proof.Bs = bs
			proof.Commitments = make([]curve.G1Affine, 10)
			for i := range proof.Commitments {
				proof.Commitments[i] = ar
			}
			proof.CommitmentPok = ar

			err := io.RoundTripCheck(&proof, func() any { return new(Proof) })
			return err == nil
		},
		GenG1(),
		GenG1(),
		GenG2(),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestVerifyingKeySerialization(t *testing.T) {

	roundTrip := func(withCommitment bool) func(curve.G1Affine, curve.G2Affine) bool {
		return func(p1 curve.G1Affine, p2 curve.G2Affine) bool {
			var vk VerifyingKey

			// create a random vk
			nbWires := 6

			vk.G1.Alpha = p1
			vk.G1.Beta = p1
			vk.G1.Delta = p1

			vk.G2.Gamma = p2
			vk.G2.Beta = p2
			vk.G2.Delta = p2

			var err error
			vk.e, err = curve.Pair([]curve.G1Affine{vk.G1.Alpha}, []curve.G2Affine{vk.G2.Beta})
			if err != nil {
				t.Fatal(err)
				return false
			}
			vk.G2.deltaNeg.Neg(&vk.G2.Delta)
			vk.G2.gammaNeg.Neg(&vk.G2.Gamma)

			vk.G1.K = make([]curve.G1Affine, nbWires)
			for i := 0; i < nbWires; i++ {
				vk.G1.K[i] = p1
			}

			if withCommitment {
				vk.PublicAndCommitmentCommitted = test_utils.Random2DIntSlice(5, 10) // TODO: Use gopter randomization
				bases := make([][]curve.G1Affine, len(vk.PublicAndCommitmentCommitted))
				elem := p1
				for i := 0; i < len(vk.PublicAndCommitmentCommitted); i++ {
					bases[i] = make([]curve.G1Affine, len(vk.PublicAndCommitmentCommitted[i]))
					for j := range bases[i] {
						bases[i][j] = elem
						elem.Add(&elem, &p1)
						vk.CommitmentKeys = append(vk.CommitmentKeys, pedersen.VerifyingKey{G: p2, GSigmaNeg: p2})
					}
				}
				assert.NoError(t, err)
			}

			err = io.RoundTripCheck(&vk, func() any { return new(VerifyingKey) })
			return err == nil
		}
	}

	parameters := gopter.DefaultTestParameters()
	parameters.MinSuccessfulTests = 10

	properties := gopter.NewProperties(parameters)

	properties.Property("VerifyingKey -> writer -> reader -> VerifyingKey should stay constant", prop.ForAll(
		roundTrip(false),
		GenG1(),
		GenG2(),
	))

	properties.Property("VerifyingKey (with commitments) -> writer -> reader -> VerifyingKey should stay constant", prop.ForAll(
		roundTrip(true),
		GenG1(),
		GenG2(),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func TestProvingKeySerialization(t *testing.T) {
	parameters := gopter.DefaultTestParameters()
	parameters.MinSuccessfulTests = 10

	properties := gopter.NewProperties(parameters)

	properties.Property("ProvingKey -> writer -> reader -> ProvingKey should stay constant", prop.ForAll(
		func(p1 curve.G1Affine, p2 curve.G2Affine, nbCommitment int) bool {
			var pk ProvingKey

			// create a random pk
			domain := fft.NewDomain(8)
			pk.Domain = *domain

			nbWires := 6
			nbPrivateWires := 4

			// allocate our slices
			pk.G1.A = make([]curve.G1Affine, nbWires)
			pk.G1.B = make([]curve.G1Affine, nbWires)
			pk.G1.K = make([]curve.G1Affine, nbPrivateWires)
			pk.G1.Z = make([]curve.G1Affine, pk.Domain.Cardinality)
			pk.G2.B = make([]curve.G2Affine, nbWires)

			pk.G1.Alpha = p1
			pk.G2.Beta = p2
			pk.G1.K[1] = p1
			pk.G1.B[0] = p1
			pk.G2.B[0] = p2

			// infinity flags
			pk.NbInfinityA = 1
			pk.InfinityA = make([]bool, nbWires)
			pk.InfinityB = make([]bool, nbWires)
			pk.InfinityA[2] = true

			if nbCommitment > 0 {
				pedersenBasis := make([]curve.G1Affine, nbCommitment)
				pedersenBases := make([][]curve.G1Affine, nbCommitment)
				for i := range pedersenBasis {
					pedersenBasis[i] = p1
					pedersenBases[i] = pedersenBasis[:i+1]
				}
				{
					var err error
					pk.CommitmentKeys, _, err = pedersen.Setup(pedersenBases)
					require.NoError(t, err)
				}
			}

			if err := io.RoundTripCheck(&pk, func() any { return new(ProvingKey) }); err != nil {
				t.Log(err)
				return false
			}

			if err := io.DumpRoundTripCheck(&pk, func() any { return new(ProvingKey) }); err != nil {
				t.Log(err)
				return false
			}
			return true
		},
		GenG1(),
		GenG2(),
		gen.IntRange(0, 2),
	))

	properties.TestingRun(t, gopter.ConsoleReporter(false))
}

func GenG1() gopter.Gen {
	_, _, g1GenAff, _ := curve.Generators()
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		var scalar big.Int
		scalar.SetUint64(genParams.NextUint64())

		var g1 curve.G1Affine
		g1.ScalarMultiplication(&g1GenAff, &scalar)

		genResult := gopter.NewGenResult(g1, gopter.NoShrinker)
		return genResult
	}
}

func GenG2() gopter.Gen {
	_, _, _, g2GenAff := curve.Generators()
	return func(genParams *gopter.GenParameters) *gopter.GenResult {
		var scalar big.Int
		scalar.SetUint64(genParams.NextUint64())

		var g2 curve.G2Affine
		g2.ScalarMultiplication(&g2GenAff, &scalar)

		genResult := gopter.NewGenResult(g2, gopter.NoShrinker)
		return genResult
	}
}
