// SPDX-FileCopyrightText: © 2025 PRISMS Center at the University of Michigan
// SPDX-License-Identifier: GNU Lesser General Public Version 2.1

#include "custom_pde.h"

#include <prismspf/core/parse_cmd_options.h>
#include <prismspf/core/pde_problem.h>
#include <prismspf/core/variable_attribute_loader.h>
#include <prismspf/core/variable_attributes.h>

#include <prismspf/user_inputs/input_file_reader.h>
#include <prismspf/user_inputs/user_input_parameters.h>

#include <prismspf/config.h>

#ifdef PRISMS_PF_WITH_CALIPER
#  include <caliper/cali-manager.h>
#  include <caliper/cali.h>
#endif

int
main(int argc, char *argv[])
{
  try
    {
      // Initialize MPI
      dealii::Utilities::MPI::MPI_InitFinalize
        mpi_init(argc, argv, dealii::numbers::invalid_unsigned_int);

      // Parse the command line options (if there are any) to get the name of the input
      // file
      prisms::ParseCMDOptions cli_options(argc, argv);
      std::string             parameters_filename = cli_options.get_parameters_filename();

      // Caliper config manager initialization
#ifdef PRISMS_PF_WITH_CALIPER
      cali::ConfigManager mgr;
      mgr.add(cli_options.get_caliper_configuration().c_str());

      // Check for configuration errors
      if (mgr.error())
        {
          std::cerr << "Caliper error: " << mgr.error_msg() << std::endl;
        }

      // Start configured performance measurements, if any
      mgr.start();
#endif

      // Restrict deal.II console printing
      dealii::deallog.depth_console(0);

      // Before fully parsing the parameter file, we need to know how many field
      // variables there are and whether they are scalars or vectors, how many
      // postprocessing variables there are, how many sets of elastic constants
      // there are, and how many user-defined constants there are.
      //
      // This is done with the derived class of `VariableAttributeLoader`,
      // `CustomAttributeLoader`.
      prisms::CustomAttributeLoader attribute_loader;
      attribute_loader.init_variable_attributes();
      std::map<unsigned int, prisms::VariableAttributes> var_attributes =
        attribute_loader.get_var_attributes();

      // Load in parameters
      prisms::InputFileReader input_file_reader(parameters_filename, var_attributes);

      // Run problem based on the number of dimensions and element degree
      switch (input_file_reader.get_dim())
        {
          case 1:
            {
              prisms::UserInputParameters<1> user_inputs(
                input_file_reader,
                input_file_reader.get_parameter_handler());
              switch (user_inputs.get_spatial_discretization().get_degree())
                {
                  case 1:
                    {
                      std::shared_ptr<prisms::PDEOperator<1, 1, double>> pde_operator =
                        std::make_shared<prisms::CustomPDE<1, 1, double>>(user_inputs);
                      std::shared_ptr<prisms::PDEOperator<1, 1, float>>
                        pde_operator_float =
                          std::make_shared<prisms::CustomPDE<1, 1, float>>(user_inputs);

                      prisms::PDEProblem<1, 1, double> problem(user_inputs,
                                                               pde_operator,
                                                               pde_operator_float);
                      problem.run();
                      break;
                    }
                  case 2:
                    {
                      std::shared_ptr<prisms::PDEOperator<1, 2, double>> pde_operator =
                        std::make_shared<prisms::CustomPDE<1, 2, double>>(user_inputs);
                      std::shared_ptr<prisms::PDEOperator<1, 2, float>>
                        pde_operator_float =
                          std::make_shared<prisms::CustomPDE<1, 2, float>>(user_inputs);

                      prisms::PDEProblem<1, 2, double> problem(user_inputs,
                                                               pde_operator,
                                                               pde_operator_float);
                      problem.run();
                      break;
                    }
                  case 3:
                    {
                      std::shared_ptr<prisms::PDEOperator<1, 3, double>> pde_operator =
                        std::make_shared<prisms::CustomPDE<1, 3, double>>(user_inputs);
                      std::shared_ptr<prisms::PDEOperator<1, 3, float>>
                        pde_operator_float =
                          std::make_shared<prisms::CustomPDE<1, 3, float>>(user_inputs);

                      prisms::PDEProblem<1, 3, double> problem(user_inputs,
                                                               pde_operator,
                                                               pde_operator_float);
                      problem.run();
                      break;
                    }
                  default:
                    throw std::runtime_error("Invalid element degree");
                }
              break;
            }
          case 2:
            {
              prisms::UserInputParameters<2> user_inputs(
                input_file_reader,
                input_file_reader.get_parameter_handler());
              switch (user_inputs.get_spatial_discretization().get_degree())
                {
                  case 1:
                    {
                      std::shared_ptr<prisms::PDEOperator<2, 1, double>> pde_operator =
                        std::make_shared<prisms::CustomPDE<2, 1, double>>(user_inputs);
                      std::shared_ptr<prisms::PDEOperator<2, 1, float>>
                        pde_operator_float =
                          std::make_shared<prisms::CustomPDE<2, 1, float>>(user_inputs);

                      prisms::PDEProblem<2, 1, double> problem(user_inputs,
                                                               pde_operator,
                                                               pde_operator_float);
                      problem.run();
                      break;
                    }
                  case 2:
                    {
                      std::shared_ptr<prisms::PDEOperator<2, 2, double>> pde_operator =
                        std::make_shared<prisms::CustomPDE<2, 2, double>>(user_inputs);
                      std::shared_ptr<prisms::PDEOperator<2, 2, float>>
                        pde_operator_float =
                          std::make_shared<prisms::CustomPDE<2, 2, float>>(user_inputs);

                      prisms::PDEProblem<2, 2, double> problem(user_inputs,
                                                               pde_operator,
                                                               pde_operator_float);
                      problem.run();
                      break;
                    }
                  case 3:
                    {
                      std::shared_ptr<prisms::PDEOperator<2, 3, double>> pde_operator =
                        std::make_shared<prisms::CustomPDE<2, 3, double>>(user_inputs);
                      std::shared_ptr<prisms::PDEOperator<2, 3, float>>
                        pde_operator_float =
                          std::make_shared<prisms::CustomPDE<2, 3, float>>(user_inputs);

                      prisms::PDEProblem<2, 3, double> problem(user_inputs,
                                                               pde_operator,
                                                               pde_operator_float);
                      problem.run();
                      break;
                    }
                  default:
                    throw std::runtime_error("Invalid element degree");
                }
              break;
            }
          case 3:
            {
              prisms::UserInputParameters<3> user_inputs(
                input_file_reader,
                input_file_reader.get_parameter_handler());
              switch (user_inputs.get_spatial_discretization().get_degree())
                {
                  case 1:
                    {
                      std::shared_ptr<prisms::PDEOperator<3, 1, double>> pde_operator =
                        std::make_shared<prisms::CustomPDE<3, 1, double>>(user_inputs);
                      std::shared_ptr<prisms::PDEOperator<3, 1, float>>
                        pde_operator_float =
                          std::make_shared<prisms::CustomPDE<3, 1, float>>(user_inputs);

                      prisms::PDEProblem<3, 1, double> problem(user_inputs,
                                                               pde_operator,
                                                               pde_operator_float);
                      problem.run();
                      break;
                    }
                  case 2:
                    {
                      std::shared_ptr<prisms::PDEOperator<3, 2, double>> pde_operator =
                        std::make_shared<prisms::CustomPDE<3, 2, double>>(user_inputs);
                      std::shared_ptr<prisms::PDEOperator<3, 2, float>>
                        pde_operator_float =
                          std::make_shared<prisms::CustomPDE<3, 2, float>>(user_inputs);

                      prisms::PDEProblem<3, 2, double> problem(user_inputs,
                                                               pde_operator,
                                                               pde_operator_float);
                      problem.run();
                      break;
                    }
                  case 3:
                    {
                      std::shared_ptr<prisms::PDEOperator<3, 3, double>> pde_operator =
                        std::make_shared<prisms::CustomPDE<3, 3, double>>(user_inputs);
                      std::shared_ptr<prisms::PDEOperator<3, 3, float>>
                        pde_operator_float =
                          std::make_shared<prisms::CustomPDE<3, 3, float>>(user_inputs);

                      prisms::PDEProblem<3, 3, double> problem(user_inputs,
                                                               pde_operator,
                                                               pde_operator_float);
                      problem.run();
                      break;
                    }
                  default:
                    throw std::runtime_error("Invalid element degree");
                }
              break;
            }
          default:
            throw std::runtime_error("Invalid number of dimensions");
        }

          // Caliper config manager closure
#ifdef PRISMS_PF_WITH_CALIPER
      // Flush output before finalizing MPI
      mgr.flush();
#endif
    }

  catch (std::exception &exc)
    {
      std::cerr << '\n'
                << '\n'
                << "----------------------------------------------------" << '\n';
      std::cerr << "Exception on processing: " << '\n'
                << exc.what() << '\n'
                << "Aborting!" << '\n'
                << "----------------------------------------------------" << '\n';
      return 1;
    }

  catch (...)
    {
      std::cerr << '\n'
                << '\n'
                << "----------------------------------------------------" << '\n';
      std::cerr << "Unknown exception!" << '\n'
                << "Aborting!" << '\n'
                << "----------------------------------------------------" << '\n';
      return 1;
    }

  return 0;
}