# Module to run tests on galaxy modules
from __future__ import print_function, absolute_import, division, unicode_literals

# TEST_UNICODE_LITERALS

import pytest
import os
import numpy as np
import importlib_resources

from astropy.table import Table
from astropy import units 
from astropy.io import fits 
from astropy.coordinates import SkyCoord
from astropy.nddata import Cutout2D
from astropy.wcs import WCS

from frb import frb
from frb.galaxies import photom
from frb.surveys.catalog_utils import convert_mags_to_flux


def test_dust_correct():

    correct = photom.extinction_correction('GMOS_S_r', 0.138)
    assert np.isclose(correct, 1.3869201954307397)

def test_flux_conversion():

    # Create a dummy table that should get converted in a known way
    tab = Table()
    tab['DES_r'] = [20.]
    tab['DES_r_err'] = 0.5

    fluxunits = 'mJy'

    fluxtab = convert_mags_to_flux(tab, fluxunits, exact_mag_err=False)

    # Check fluxes
    assert np.isclose(fluxtab['DES_r'], 0.036307805), "Check AB flux conversion."

    # Check errors
    assert np.isclose(fluxtab['DES_r_err'], 0.016720362110466937), "Check AB flux error."


def test_fractional_flux():
    isize = 5
    # FRB and HG
    frbname = 'FRB20180924B'
    frbdat = frb.FRB.by_name(frbname)
    # frbcoord = frbdat.coord
    hg = frbdat.grab_host()
    # Read cutout
    cutout_file = importlib_resources.files('frb.tests.files')/'FRB180924_cutout.fits'
    hdul = fits.open(cutout_file)

    hgcoord = hg.coord
    size = units.Quantity((isize, isize), units.arcsec)
    cutout = Cutout2D(hdul[0].data, hgcoord, size, wcs=WCS(hdul[0].header))

    # Run
    med_ff, sig_ff, f_weight = photom.fractional_flux(cutout, frbdat, hg)    

    assert np.isclose(sig_ff, 0.2906803236219953)


