-- Section 10

-- Exceptional expressions:

data Expr = Val Value | Add Expr Expr | If Expr Expr Expr | Catch Expr Expr
            deriving Show

data Value = I Int | B Bool | Throw | Error
             deriving Show

-- Evaluation

folde :: (Value -> a) -> (a -> a -> a) ->
            (a -> a -> a -> a) -> (a -> a -> a) -> Expr -> a
folde val add cond catch = f
   where
      f (Val v)     = val v
      f (Add x y)   = add (f x) (f y)
      f (If x y z)  = cond (f x) (f y) (f z)
      f (Catch x y) = catch (f x) (f y)

eval :: Expr -> Value
eval = folde id add cond catch

add :: Value -> Value -> Value
add (I n) (I m) = I (n+m)
add Throw _     = Throw
add (I _) Throw = Throw
add _     _     = Error

cond :: Value -> Value -> Value -> Value
cond (B b) v w = if b then v else w
cond Throw _ _ = Throw
cond _     _ _ = Error

catch :: Value -> Value -> Value
catch Throw v = v
catch v     _ = v

-- Types:

data Type = INT | INT' | BOOL | BOOL' | THROW | ERROR | TOP
            deriving (Show, Eq)

tval :: Value -> Type
tval (I _) = INT
tval (B _) = BOOL
tval Throw = THROW
tval Error = ERROR

instance Ord Type where
   -- (<=) :: Type -> Type -> Bool
   ERROR <= _     = True
   _     <= TOP   = True
   BOOL' <= BOOL  = True
   BOOL' <= THROW = True
   INT'  <= INT   = True
   INT'  <= THROW = True
   t     <= t'    = t == t'

(/\) :: Type -> Type -> Type
BOOL  /\ INT   = ERROR
INT   /\ BOOL  = ERROR
BOOL  /\ THROW = BOOL'
THROW /\ BOOL  = BOOL'
INT   /\ THROW = INT'
THROW /\ INT   = INT'
INT'  /\ BOOL' = ERROR
BOOL' /\ INT'  = ERROR
t     /\ t'    | t  <= t'  = t
               | t' <= t   = t'
               | t  == t'  = t
               | otherwise = ERROR

(~>) :: Bool -> Type -> Type
b ~> t = if b then t else TOP

-- Type checking:

texp :: Expr -> Type
texp = folde tval add' cond' catch'

add' :: Type -> Type -> Type
add' t t'  =  ((t <= INT && t' <= INT)                   ~> INT)
           /\ ((t <= THROW || (t <= INT && t' <= THROW)) ~> THROW)
           /\ ((t <= BOOL  || (t <= INT && t' <= BOOL))  ~> ERROR)

cond' :: Type -> Type -> Type -> Type
cond' s t t'  =  ((s <= BOOL)  ~> (t /\ t'))
              /\ ((s <= INT)   ~> ERROR)
              /\ ((s <= THROW) ~> THROW)

catch' :: Type -> Type -> Type
catch' t t'  =  ((t <= THROW) ~> t')
             /\ ((t <= ERROR) ~> ERROR)
             /\ ((t <= BOOL)  ~> BOOL)
             /\ ((t <= INT)   ~> INT)
