module Reasoning.Implication where

open import Relation.Binary hiding (_⇔_) renaming (_⇒_ to implies)
open import Relation.Binary.Structures
open import Relation.Binary.Definitions
open import Relation.Binary.PropositionalEquality
  using (subst; sym; cong)
open import Relation.Binary.PropositionalEquality.Properties
  using (isEquivalence)
open import Relation.Binary.Bundles using (Poset; Preorder)

open import Agda.Builtin.Equality
open import Level
open import Function hiding (_⇔_)

open import Reasoning.Definitions
  using (_⇔_; to; from) public

variable ℓ : Level


{- This file provides a proof that backwards implication ⇐ forms a
   preorder. This allows us to use it within the preorder "calculation"
   syntax, and so present the calculations in the same style as in
   the paper. -}


_⇐_ : Rel (Set ℓ) ℓ
x ⇐ y = y → x
infixl 1 _⇐_

⇐-reflexive : implies {ℓ₁ = ℓ} _⇔_ _⇐_
⇐-reflexive = from


⇐-trans : Transitive {ℓ = ℓ} _⇐_
⇐-trans f g x = f (g x)

⇔-trans : Transitive {ℓ = ℓ} _⇔_
to (⇔-trans f g) x = g .to (f .to x)
from (⇔-trans f g) x = f .from (g .from x)


⇔-refl : Reflexive {ℓ = ℓ} _⇔_
⇔-refl .to x = x
⇔-refl .from x = x

⇔-sym : Symmetric {ℓ = ℓ} _⇔_
⇔-sym x .to = x .from
⇔-sym x .from = x .to

⇔-isEquivalence : IsEquivalence {ℓ = ℓ} _⇔_
⇔-isEquivalence = record
  { refl = ⇔-refl
  ; sym = ⇔-sym
  ; trans = ⇔-trans }

⇔-cong : {A : Set ℓ} (f : A → Set) → Congruent _≡_ _⇔_ f
⇔-cong f {x} eq = subst (λ t → f x ⇔ f t) eq ⇔-refl


⇐-isPreorder : IsPreorder {ℓ = ℓ} _⇔_ _⇐_
⇐-isPreorder = record
  { isEquivalence = ⇔-isEquivalence
  ; reflexive = ⇐-reflexive
  ; trans = ⇐-trans }


⇐-preorder : Preorder _ _ _
⇐-preorder = record
  { Carrier = Set
  ; _≈_ = _⇔_
  ; _≲_ = _⇐_
  ; isPreorder = ⇐-isPreorder }


variable P Q R : Set

with₂ : {A : Set} → {f g : A → A → Set}
      → ((t t' : A) → f t t' ⇔ g t t')
      → ({t t' : A} → f t t') ⇔ ({t t' : A} → g t t')
with₂ x .to y {t} {t'} = x t t' .to y
with₂ x .from y {t} {t'} = x t t' .from y

with₃ : {A : Set} → {f g : A → A → A → Set}
      → ((s t t' : A) → f s t t' ⇔ g s t t')
      → ({s t t' : A} → f s t t') ⇔ ({s t t' : A} → g s t t')
with₃ x .to y {s} {t} {t'} = x s t t' .to y
with₃ x .from y {s} {t} {t'} = x s t t' .from y
