module Positivity.Calculations where

open import Data.Integer
open import Data.Nat using (suc; s≤s; z≤n)
open import Data.Product renaming (_×_ to _∧_)

open import Reasoning.Syntax
open import Reasoning.Implication

open import Positivity.Definitions

open use-⇐ (⇐-preorder)


{- In this file we formalise the calculations in Section 2 of the
   paper, on Positivity Checking. -}

{- We begin by showing a lemma which we will make use of later on. In
   the paper, this is written as "addition preserves positivity". It
   states that adding two positive integers yields another positive
   integer. -}
addition-preserves-positivity : ∀ (n m : ℤ)
  → +0 < n ∧ +0 < m
  → +0 < n + m
addition-preserves-positivity +0 (+ m) (pn , pm) = pm
addition-preserves-positivity (+ suc n) (+ m) (pn , pm) = +<+ (s≤s z≤n)


{- The calculations are presented in Agda in a calculational style, as
   in the paper. As much as possible, we try to stick to the same
   syntax, defined in Reasoning.Syntax.

   The only difference is where in the paper we add defining clauses
   to isPos as a calculation step, here the steps follow by
   definition. This is because Agda does not permit partial
   definitions to be built up in this way, and so we are obliged to
   provide the full definition in Positivity.Definitions, and then
   "pretend" here that we are performing a calculation. In actual
   fact, the below is a *proof*, not a calculation, but it can still
   be viewed as a calculation in this way. -}

correct : ∀ (e : Expr) → isPos e → eval e > +0

correct (val n) = begin
  eval (val n) > +0
    ⇔⟨⟩
  n > +0
    ⇔⟨⟩
  isPos (val n)
    ∎

correct (add x y) = begin
  eval (add x y) > +0
    ⇔⟨⟩
  eval x + eval y > +0
    ⇐⟨ addition-preserves-positivity (eval x) (eval y) ⟩
  (eval x > +0 ∧ eval y > +0)
    ⇐⟨ (λ (px , py) → (correct _ px , correct _ py)) ⟩
  (isPos x ∧ isPos y)
    ⇔⟨⟩
  isPos (add x y)
    ∎
