open import Data.Empty
open import Data.Product
open import Data.Sum renaming (_⊎_ to _∨_)

open import Relation.Binary.Definitions using (Decidable)
open import Relation.Binary.Core using (Rel)
open import Relation.Binary.Bundles using (Poset)
open import Relation.Binary.Lattice using (IsBoundedMeetSemilattice)
open import Relation.Binary.PropositionalEquality using (sym)

open import Algebra using (Op₂)

open import Agda.Builtin.Equality using (_≡_)
open import Level

open import Reasoning.Implication

{- This module provides a number of useful properties for reasoning
   about the monotonicity of functions within a particular ordering.

   For unary, binary, and ternary functions, we prove properties
   relating to: constant functions, functions which select a particular
   argument, and the interaction with conjunction, disjunction,
   greatest-lower-bounds, and the composition operator ⇛.

   The properties of monotonicity correspond to the idea in the paper
   that we get monotonicity "for free", for our constraint-based
   definitions. This is demonstrated the Calculation modules: these
   monotonicity lemmas are used in an "automatic" way, as directed
   purely by the structure of e.g. the add' function.

   We show additional properties relating to ⇛ and ⊓, in particular
   the three properties used in the paper (⇛-collect, ⇛-subsume, and
   ⇛-disjunct), as well as a handful of other useful lemmas. -}
module Monotonicity.Properties
  (Type : Set)
  (_≲_ : Rel Type zero)
  (_≲?_ : Decidable _≲_)
  (_∧_ : Op₂ Type)
  (TOP : Type)
  (⊓-isSemilattice : IsBoundedMeetSemilattice _≡_ _≲_ _∧_ TOP)
  where

open import Monotonicity.Definitions Type _≲_ _∧_ TOP ⊓-isSemilattice
  public

open Poset (≤-poset)
  public
  using (_≥_)

open import Relation.Binary.Properties.Poset (≤-poset)
  public
  using (≥-poset)

open import Relation.Binary.Lattice.Properties.BoundedMeetSemilattice
  ⊓-meetSemilattice
  public
  
open import Relation.Binary.Lattice.Properties.MeetSemilattice
  meetSemilattice
  public

open Composing


{- Lifting/unlifting implication into True decidable props -}

wit : ∀ {A B : Set} → {a? : Dec A} → {b? : Dec B}
      → (True a? → True b?) → A → B
wit f p = toWitness (f (fromWitness p))

unwit : ∀ {A B : Set} → {a? : Dec A} → {b? : Dec B}
      → (A → B) → (True a? → True b?)
unwit f a = fromWitness (f (toWitness a))



{- Properties about monotonicity of unary functions -}

mono₁-⊓ : ∀ {f g}
        → MonoT f → MonoT g
        → MonoT (λ t → f t ∧ g t)
mono₁-⊓ mf mg t t' t≤t' = ∧-monotonic (mf t t' t≤t') (mg t t' t≤t')

mono₁-⇛ : ∀ {P f g}
        → MonoD {P} f → MonoT g
        → MonoT (λ t → f t ⇛ g t)
mono₁-⇛ {f = f} {g = g} mf mg t t' t≤t' with f t'
... | no p = maximum (f t ⇛ g t)
... | yes p with f t
...   | yes q = mg t t' t≤t'
...   | no q = ⊥-elim (q (wit (mf t t' t≤t') p))

mono₁-const : (k : Type) → MonoT (λ t → k)
mono₁-const k t t' t≤t' = reflexive _≡_.refl

mono₁-∧? : ∀ {P Q f g}
         → MonoD {P} f → MonoD {Q} g
         → MonoD (λ t → f t ∧? g t)
mono₁-∧? mf mg t t' t≤t' a with toWitness a
... | p , q = fromWitness
  ( wit (mf t t' t≤t') p , wit (mg t t' t≤t') q )

mono₁-∨? : ∀ {P Q f g}
         → MonoD {P} f → MonoD {Q} g
         → MonoD (λ t → f t ∨? g t)
mono₁-∨? mf mg t t' t≤t' a with toWitness a
... | inj₁ p = fromWitness (inj₁ (wit (mf t t' t≤t') p))
... | inj₂ q = fromWitness (inj₂ (wit (mg t t' t≤t') q))



{- Properties about ⇛ and ⊓ -}
⇛-collect : ∀ {A : Set} (b : Dec A) t t'
          → (b ⇛ t) ∧ (b ⇛ t') ≡ b ⇛ (t ∧ t')          
⇛-collect (yes b) t t' = _≡_.refl
⇛-collect (no b) t t' rewrite ∧-idempotent TOP = _≡_.refl

⇛-subsume : ∀ {B B' : Set} (b : Dec B) (b' : Dec B') t t'
          → (B' → B) → t ≤ t' → (b ⇛ t) ∧ (b' ⇛ t') ≡ b ⇛ t
⇛-subsume (yes a) (yes b) t t' h lt rewrite ∧-comm t t' = y≤x⇒x∧y≈y lt
⇛-subsume (yes a) (no b) t t' h lt rewrite identityʳ t = _≡_.refl
⇛-subsume (no a) (yes b) t t' h lt = ⊥-elim (a (h b))
⇛-subsume (no a) (no b) t t' h lt rewrite identityˡ TOP = _≡_.refl
  
⇛-disjunct : ∀ {B B' : Set} (b : Dec B) (b' : Dec B') t
           → (b ⇛ t) ∧ (b' ⇛ t) ≡ (b ∨? b') ⇛ t
⇛-disjunct (yes b) (yes b') t rewrite ∧-idempotent t = _≡_.refl
⇛-disjunct (yes b) (no b') t rewrite identityʳ t = _≡_.refl
⇛-disjunct (no b) (yes b') t rewrite identityˡ t = _≡_.refl
⇛-disjunct (no b) (no b') t rewrite identityˡ TOP = _≡_.refl



{- Properties relating ⊓ and ⇛ to inequalities on functions defined
   using them. -}
⇛-use : ∀ {P : Set} {t : Type} (b : Dec P)
      → (a : P)
      → (b ⇛ t) ≡ t
⇛-use (yes b) a = _≡_.refl
⇛-use (no b) a = ⊥-elim (b a)

⇛-def : {P : Set} {D : Dec P} {f u : Type}
      → (P → f ≤ u) ⇔ (f ≤ (D ⇛ u))
to (⇛-def {D = D} {f = f}) h with D
... | yes p = h p
... | no p = maximum f
from (⇛-def {D = D}) r lt with D
... | yes p = r
... | no p = ⊥-elim (p lt)

⇛-glbₗ : ∀ {P : Set} {t t' : Type} (b : Dec P)
       → (a : P)
       → ((b ⇛ t) ∧ t') ≤ t
⇛-glbₗ {t = t} {t' = t'} (yes b) a = x∧y≤x t t'
⇛-glbₗ (no b) a = ⊥-elim (b a)

⇛-glbᵣ : ∀ {Q : Set} {t t' : Type} (b' : Dec Q)
       → (a : Q)
       → (t ∧ (b' ⇛ t')) ≤ t'
⇛-glbᵣ {Q} {t = t} {t' = t'} b' a
  rewrite ∧-comm t (b' ⇛ t') = ⇛-glbₗ b' a

⇛-left : ∀ {P Q : Set} {t : Type} (b : Dec P) (b' : Dec Q)
       → ((b ∨? b') ⇛ t) ≤ (b ⇛ t)
⇛-left (yes b) b' = reflexive _≡_.refl
⇛-left {t = t} (no b) b' = maximum _

⇛-right : ∀ {P Q : Set} {t : Type} (b : Dec P) (b' : Dec Q)
        → ((b ∨? b') ⇛ t) ≤ (b' ⇛ t)
⇛-right (yes b) (yes b') = reflexive _≡_.refl
⇛-right (no b) (yes b') = reflexive _≡_.refl
⇛-right (yes b) (no b') = maximum _
⇛-right (no b) (no b') = maximum _

⇛-weaken : {P Q : Set} {t t' : Type} (b : Dec P) (b' : Dec Q)
      → (P → Q) → t' ≤ t → (b' ⇛ t') ≤ (b ⇛ t)
⇛-weaken {t = t} {t' = t'} b b' f lt
  rewrite sym (⇛-subsume b' b t' t f lt)
  = x∧y≤y (b' ⇛ t') (b ⇛ t)


{- Properties about monotonicity of binary functions -}

mono₂-≤?-constₗ : (k : Type) → MonoD₂ (λ t _ → t ≲? k)
mono₂-≤?-constₗ k t t' u u' t≤t' u≤u' x = unwit (trans t≤t') x

mono₂-≤?-constᵣ : (k : Type) → MonoD₂ (λ _ u → u ≲? k)
mono₂-≤?-constᵣ k t t' u u' t≤t' u≤u' x = unwit (trans u≤u') x

mono₂-const : (k : Type) → MonoT₂ (λ t u → k)
mono₂-const k t t' u u' t≤t' u≤u' = reflexive _≡_.refl

mono₂-fst : MonoT₂ (λ t u → t)
mono₂-fst t t' u u' t≤t' u≤u' = t≤t'

mono₂-snd : MonoT₂ (λ t u → u)
mono₂-snd t t' u u' t≤t' u≤u' = u≤u'

mono₂-∧? : ∀ {P Q : Type → Type → Set} {f g}
         → MonoD₂ {P} f → MonoD₂ {Q} g
         → MonoD₂ (λ t u → f t u ∧? g t u)
mono₂-∧? mf mg = D.mono-1-2
  (λ k → mono₁-∧? (D.mono-fixₗ k mf) (D.mono-fixₗ k mg))
  (λ k → mono₁-∧? (D.mono-fixᵣ k mf) (D.mono-fixᵣ k mg))

mono₂-∨? : ∀ {P Q : Type → Type → Set} {f g}
         → MonoD₂ {P} f → MonoD₂ {Q} g
         → MonoD₂ (λ t u → f t u ∨? g t u)
mono₂-∨? mf mg = D.mono-1-2
  (λ k → mono₁-∨? (D.mono-fixₗ k mf) (D.mono-fixₗ k mg))
  (λ k → mono₁-∨? (D.mono-fixᵣ k mf) (D.mono-fixᵣ k mg))

mono₂-⊓ : ∀ {f g}
         → MonoT₂ f → MonoT₂ g
         → MonoT₂ (λ t u → f t u ∧ g t u)
mono₂-⊓ mf mg = T.mono-1-2 {λ _ _ → Type}
  (λ k → mono₁-⊓ (T.mono-fixₗ {K₂} k mf) (T.mono-fixₗ {K₂} k mg))
  (λ k → mono₁-⊓ (T.mono-fixᵣ {K₂} k mf) (T.mono-fixᵣ {K₂} k mg))

mono₂-⇛ : ∀ {P : Type → Type → Set} {f g}
        → MonoD₂ {P} f → MonoT₂ g
        → MonoT₂ (λ t u → f t u ⇛ g t u)
mono₂-⇛ mf mg = T.mono-1-2 {λ _ _ → Type}
  (λ k → mono₁-⇛ (D.mono-fixₗ k mf) (T.mono-fixₗ {K₂} k mg))
  (λ k → mono₁-⇛ (D.mono-fixᵣ k mf) (T.mono-fixᵣ {K₂} k mg))       



{- Properties about monotonicity of three-place functions -}

mono₃-const : (k : Type) → MonoT₃ (λ t u v → k)
mono₃-const k t t' u u' v v' t≤t' u≤u' v≤v' = reflexive _≡_.refl

mono₃-u : MonoT₃ (λ t u v → u)
mono₃-u t t' u u' v v' t≤t' u≤u' v≤v' = u≤u'

mono₃-v : MonoT₃ (λ t u v → v)
mono₃-v t t' u u' v v' t≤t' u≤u' v≤v' = v≤v'

mono₃-≤?-const₁ : (k : Type) → MonoD₃ (λ t _ _ → t ≲? k)
mono₃-≤?-const₁ k t t' u u' v v' t≤t' u≤u' v≤v' x = unwit (trans t≤t') x

mono₃-∧? : ∀ {P Q : Type → Type → Type → Set} {f g}
         → MonoD₃ {P} f → MonoD₃ {Q} g
         → MonoD₃ (λ t u v → f t u v ∧? g t u v)
mono₃-∧? mf mg = D.mono-1-3
  (λ j k → mono₁-∧? (D.mono-fix₁ j k mf) (D.mono-fix₁ j k mg))
  (λ j k → mono₁-∧? (D.mono-fix₂ j k mf) (D.mono-fix₂ j k mg))
  (λ j k → mono₁-∧? (D.mono-fix₃ j k mf) (D.mono-fix₃ j k mg))

mono₃-∨? : ∀ {P Q : Type → Type → Type → Set} {f g}
         → MonoD₃ {P} f → MonoD₃ {Q} g
         → MonoD₃ (λ t u v → f t u v ∨? g t u v)
mono₃-∨? mf mg = D.mono-1-3
  (λ j k → mono₁-∨? (D.mono-fix₁ j k mf) (D.mono-fix₁ j k mg))
  (λ j k → mono₁-∨? (D.mono-fix₂ j k mf) (D.mono-fix₂ j k mg))
  (λ j k → mono₁-∨? (D.mono-fix₃ j k mf) (D.mono-fix₃ j k mg))

mono₃-⊓ : ∀ {f g}
         → MonoT₃ f → MonoT₃ g
         → MonoT₃ (λ t u v → f t u v ∧ g t u v)
mono₃-⊓ mf mg = T.mono-1-3 {K₃}
  (λ j k → mono₁-⊓ (T.mono-fix₁ {K₃} j k mf) (T.mono-fix₁ {K₃} j k mg))
  (λ j k → mono₁-⊓ (T.mono-fix₂ {K₃} j k mf) (T.mono-fix₂ {K₃} j k mg))
  (λ j k → mono₁-⊓ (T.mono-fix₃ {K₃} j k mf) (T.mono-fix₃ {K₃} j k mg))

mono₃-⇛ : ∀ {P : Type → Type → Type → Set} {f g}
        → MonoD₃ {P} f → MonoT₃ g
        → MonoT₃ (λ t u v → f t u v ⇛ g t u v)
mono₃-⇛ mf mg = T.mono-1-3 {K₃}
  (λ j k → mono₁-⇛ (D.mono-fix₁ j k mf) (T.mono-fix₁ {K₃} j k mg))
  (λ j k → mono₁-⇛ (D.mono-fix₂ j k mf) (T.mono-fix₂ {K₃} j k mg))
  (λ j k → mono₁-⇛ (D.mono-fix₃ j k mf) (T.mono-fix₃ {K₃} j k mg))    
