open import Data.Product renaming (_×_ to _∧_)

open import Relation.Binary.Core using (REL; Rel)
open import Relation.Binary.Indexed.Heterogeneous using (IRel; Transitive)
open import Relation.Binary using (IsPartialOrder)
open import Relation.Binary.Lattice
  using (IsBoundedMeetSemilattice; BoundedMeetSemilattice)

open import Relation.Nullary.Decidable
  using (Dec; _because_; yes; no; _→-dec_; True; toWitness; fromWitness; map′)
  renaming (_⊎-dec_ to _∨?_; _×-dec_ to _∧?_)

open import Algebra using (Op₂)

open import Agda.Builtin.Equality using (_≡_)
open import Level


{- This file does two primary things:

     * Defines the notion of monotonicity for functions.
     * Defines the ⇛ constraint composition operator.
-}


{- In this module, we define what it means for a function to be
   monotone w.r.t. a given semilattice. As such, the module is
   parameterised over a carrier set, an ordering, and a greatest-
   lower-bound and TOP forming a semi-lattice.

   We provide definitions of monotonicity for functions of one, two,
   and three arguments. We also define monotonicity for decidable
   propositions (where the ordering is given by implication). -}
module Monotonicity.Definitions
  (Type : Set)
  (_≲_ : Rel Type zero)
  (_∧_ : Op₂ Type)
  (TOP : Type)
  (⊓-isSemilattice : IsBoundedMeetSemilattice _≡_ _≲_ _∧_ TOP)
  where

⊓-meetSemilattice : BoundedMeetSemilattice _ _ _
⊓-meetSemilattice = record
  { Carrier = Type
  ; _≈_ = _≡_
  ; _∧_ = _∧_
  ; ⊤ = TOP
  ; isBoundedMeetSemilattice = ⊓-isSemilattice }


open BoundedMeetSemilattice (⊓-meetSemilattice)
  public
  using (maximum; reflexive; _≤_; x∧y≤x; x∧y≤y; trans; refl; meetSemilattice)
  renaming (poset to ≤-poset; preorder to ≤-preorder)


{- Here, we define monotonicity generally in terms of an indexed
   relation ≲. This can be instantiated later on to an ordering on
   types (given by the module-wide semilattice above), and also to the
   ordering on decidable propositions (given by implication).

   Here we also provide some simple propositions to reason about
   monotone functions. Many more properties are defined in the other
   module, Monotonicity.Properties. -}
module Monotonicity
  (D : Set → Set)
  (_≲_ : IRel D zero)
  (≲-trans : Transitive D _≲_)
  where

  {- What does monotonicity mean, for functions of different arities? -}
  Mono : {P : Type → Set}
       → (f : (t : Type) → D (P t)) → Set
  Mono f = ∀ t t'
         → (t≤t' : t ≤ t')
         → f t ≲ f t'

  Mono₂ : {P : Type → Type → Set}
        → (f : (t u : Type) → D (P t u)) → Set
  Mono₂ f = ∀ t t' u u'
          → (t≤t' : t ≤ t') → (u≤u' : u ≤ u')
          → f t u ≲ f t' u'

  Mono₃ : {P : Type → Type → Type → Set}
        → (f : (t u v : Type) → D (P t u v)) → Set
  Mono₃ f = ∀ t t' u u' v v'
          → (t≤t' : t ≤ t') → (u≤u' : u ≤ u') → (v≤v' : v ≤ v')
          → f t u v ≲ f t' u' v'

  {- Fixing all but one arguments to an n-ary function is mono, if the
     n-ary function itself is n-monotone -}
  mono-fixₗ : ∀ {P f} → (k : Type) → Mono₂ {P} f → Mono (λ t → f t k)
  mono-fixₗ k mf t t' t≤t' = mf t t' k k t≤t' refl

  mono-fixᵣ : ∀ {P f} → (k : Type) → Mono₂ {P} f → Mono (λ t → f k t)
  mono-fixᵣ k mf t t' t≤t' = mf k k t t' refl t≤t'

  mono-fix₁ : ∀ {P f} → (j k : Type) → Mono₃ {P} f → Mono (λ t → f t j k)
  mono-fix₁ j k mf t t' t≤t' = mf t t' j j k k t≤t' refl refl

  mono-fix₂ : ∀ {P f} → (j k : Type) → Mono₃ {P} f → Mono (λ t → f j t k)
  mono-fix₂ j k mf t t' t≤t' = mf j j t t' k k refl t≤t' refl

  mono-fix₃ : ∀ {P f} → (j k : Type) → Mono₃ {P} f → Mono (λ t → f j k t)
  mono-fix₃ j k mf t t' t≤t' = mf j j k k t t' refl refl t≤t'

  {- Conversion of monotone unary functions to higher arity monotone
     functions, argument-wise -}
  mono-1-2 : ∀ {P f}
         → (∀ (k : Type) → Mono (λ t → f t k))
         → (∀ (k : Type) → Mono (λ t → f k t))
         → Mono₂ {P} f
  mono-1-2 l r t t' u u' t≤t' u≤u' =
    ≲-trans(r t u u' u≤u') (l u' t t' t≤t')

  mono-1-3 : ∀ {P f}
           → (∀ j k → Mono (λ t → f t j k))
           → (∀ j k → Mono (λ t → f j t k))
           → (∀ j k → Mono (λ t → f j k t))
           → Mono₃ {P} f
  mono-1-3 a b c t t' u u' v v' t≤t' u≤u' v≤v' =
    ≲-trans (a u v t t' t≤t')
      (≲-trans (b t' v u u' u≤u') (c t' u' v v' v≤v'))



{- Instantiating monotonicity to constant Type, and decidable functions -}

open module T = Monotonicity (λ _ → Type) _≤_ trans
  public
  using ()

open module D = Monotonicity (λ t → Dec t) (λ a b → True b → True a) (λ f g x → f (g x))
  public
  using ()
  renaming (Mono to MonoD; Mono₂ to MonoD₂; Mono₃ to MonoD₃)

{- We give special names to T.Monoₙ, instantiated to the constant Type
   functors. These represent monotonicity of functions of:

     * Type -> Type
     * Type -> Type -> Type
     * Type -> Type -> Type -> Type -}

K₁ : Type → Set
K₁ = λ _ → Type

K₂ : Type → Type → Set
K₂ = λ _ _ → Type

K₃ : Type → Type → Type → Set
K₃ = λ _ _ _ → Type

MonoT = T.Mono {K₁}
MonoT₂ = T.Mono₂ {K₂}
MonoT₃ = T.Mono₃ {K₃}



{- Here we define the ⇛ operator for composing constraints into
   function definitions -}
module Composing where
  open import Relation.Nullary.Decidable
    renaming (_×-dec_ to _∧?_; _⊎-dec_ to _∨?_)
    public

  _⇛_ : {l : Level} → {P : Set l} → Dec P → Type → Type
  yes _ ⇛ t = t
  no _ ⇛ t = TOP
  infixr 10 _⇛_

