open import Data.Product renaming (_×_ to _∧_)
open import Data.Empty

open import Relation.Binary.Core using (Rel)
open import Relation.Binary.Definitions using (Decidable)
open import Relation.Nullary.Decidable using (Dec; yes; no)
open import Relation.Nullary.Negation using (¬_)

open import Agda.Builtin.Equality
open import Level

{- In this module we define the notion of "comparison". This roughly
   corresponds to Haskell's 'Ordering' type. A comparison between two
   objects is either "less than", "equal", or "greater than". Thus, a
   type in which two terms *always* have a comparison is a total order.

   'Comparison' can also be seen as a proposition: whether two terms
   can be ordered in any way. Therefore, ¬ Comparison t t' is the
   proposition that two terms are incomparable.

   Here, we are only interested in defining the notion of comparison
   for decidable ordering/equality relations. Therefore, whether or
   not any two elements are comparable is itself decidable, as
   witnessed by _compare_. -}
module Monotonicity.Comparison
  (Type : Set)
  (_≲_ : Rel Type zero)
  (_≲?_ : Decidable _≲_)
  (_==_ : Decidable (_≡_ {A = Type}))
  where

private variable t t' : Type

data Comparison : Type → Type → Set where
  LT : t ≲ t' → Comparison t t'
  EQ : t ≡ t' → Comparison t t'
  GT : t' ≲ t → Comparison t t'

open Comparison public

incomparable : ¬ t ≡ t' → ¬ t ≲ t' → ¬ t' ≲ t → ¬ Comparison t t'
incomparable p q r (LT x) = q x
incomparable p q r (EQ x) = p x
incomparable p q r (GT x) = r x

_compare_ : ∀ (t t' : Type) → Dec (Comparison t t')
t compare t' with t ≲? t' | t' ≲? t | t == t'
... | yes p | _     | _     = yes (LT p)
... | no p  | yes q | _     = yes (GT q)
... | no p  | no q  | yes r = yes (EQ r)
... | no p  | no q  | no r  = no (incomparable r p q)

incomparable′ : ¬ Comparison t t' → (¬ t ≡ t' ∧ ¬ t ≲ t' ∧ ¬ t' ≲ t)
incomparable′ {t} {t'} nc with t compare t'
... | yes p = ⊥-elim (nc p)
... | no np = (λ eq → ⊥-elim (nc (EQ eq)))
            , (λ lt → ⊥-elim (nc (LT lt)))
            , (λ gt → ⊥-elim (nc (GT gt)))
