module Exceptions.Fusion where

open import Agda.Builtin.Equality

open import Exceptions.Definitions
open import Exceptions.Properties
open import Reasoning.Definitions

open Ordering

open import Reasoning.Syntax
open use-≥ (≥-poset)


{- Similar to the analogous Conditionals.Fusion module, here we define
   the fold operator 'folde' for the exceptions language, according to
   Section 9 in the paper. We also state and prove the fold-fusion
   property. -}


folde : {A : Set}
      → (Value → A) → (A → A → A) → (A → A → A → A) → (A → A → A)
      → Expr → A
folde {A} val add cond catch = go
  where
    go : Expr → A
    go (Val v) = val v
    go (Add x y) = add (go x) (go y)
    go (If x y z) = cond (go x) (go y) (go z)
    go (Catch x y) = catch (go x) (go y)

fusion : {A : Set}
     → (h : A → Type)

     -- cata operators
     → (val : Value → A)
     → (add : A → A → A)
     → (cond : A → A → A → A)
     → (catch : A → A → A)
     → (val' : Value → Type)
     → (add' : Type → Type → Type)
     → (cond' : Type → Type → Type → Type)
     → (catch' : Type → Type → Type)

     -- monotonicity of add' and cond'
     → MonoT₂ add'
     → MonoT₃ cond'
     → MonoT₂ catch'

     -- homomorphism properties
     → (∀ {x}     → h (val x) ≥ val' x)
     → (∀ {x y}   → h (add x y) ≥ add' (h x) (h y))
     → (∀ {x y z} → h (cond x y z) ≥ cond' (h x) (h y) (h z))
     → (∀ {x y}   → h (catch x y) ≥ catch' (h x) (h y))
     -------------------------------------------------------------
     → ∀ e
     → (h (folde val add cond catch e)) ≥ folde val' add' cond' catch' e


fusion h val add cond catch
         val' add' cond' catch'
         ma mc mca hv ha hc hca e
  = go e
  where
    go : (e : Expr)
       → h (folde val add cond catch e) ≥ folde val' add' cond' catch' e

    go (Val x) = begin
      h (folde val add cond catch (Val x))
        ≡⟨⟩
      h (val x)
        ≥⟨ hv ⟩
      val' x
        ≡⟨⟩
      folde val' add' cond' catch' (Val x)
        ∎

    go (Add x y) = begin
      h (folde val add cond catch (Add x y))
        ≡⟨⟩
      h (add (folde val add cond catch x) (folde val add cond catch y))
        ≥⟨ ha ⟩
      add' (h (folde val add cond catch x)) (h (folde val add cond catch y))
        ≥⟨ ma _ _ _ _ (go x) (go y) ⟩
      add' (folde val' add' cond' catch' x) (folde val' add' cond' catch' y)
        ≡⟨⟩
      folde val' add' cond' catch' (Add x y)
        ∎

    go (If x y z) = begin
      h (folde val add cond catch (If x y z))
        ≡⟨⟩
      h (cond (folde val add cond catch x)
              (folde val add cond catch y)
              (folde val add cond catch z))
        ≥⟨ hc ⟩
      cond' (h (folde val add cond catch x))
            (h (folde val add cond catch y))
            (h (folde val add cond catch z))
        ≥⟨ mc _ _ _ _ _ _ (go x) (go y) (go z) ⟩
      cond' (folde val' add' cond' catch' x)
            (folde val' add' cond' catch' y)
            (folde val' add' cond' catch' z)
        ≡⟨⟩
      folde val' add' cond' catch' (If x y z)
        ∎

    go (Catch x y) = begin
      h (folde val add cond catch (Catch x y))
        ≡⟨⟩
      h (catch (folde val add cond catch x) (folde val add cond catch y))
        ≥⟨ hca ⟩
      catch' (h (folde val add cond catch x)) (h (folde val add cond catch y))
        ≥⟨ mca _ _ _ _ (go x) (go y) ⟩
      catch' (folde val' add' cond' catch' x) (folde val' add' cond' catch' y)
        ≡⟨⟩
      folde val' add' cond' catch' (Catch x y)
        ∎
