module Exceptions.Definitions where

open import Data.Integer using (ℤ; _+_)
open import Data.Nat using (ℕ; _≟_)
open import Data.Bool using (Bool; if_then_else_; true; false)
open import Data.Sum renaming (_⊎_ to _∨_)
open import Data.Product renaming (_×_ to _∧_)

open import Relation.Binary using (Rel)
open import Relation.Binary.PropositionalEquality
open import Relation.Nullary.Decidable
  using (Dec; yes; no; via-injection)
  renaming (_⊎-dec_ to _∨?_; _×-dec_ to _∧?_)
open import Relation.Nullary.Negation using (¬_)
open import Algebra using (Op₂)
open import Agda.Builtin.Equality
open import Level

open import Reasoning.Definitions


{- This file contains the definitions for a language consisting of
   integers and booleans, addition, and conditionals, as with
   Conditionals.Definitions. This language also has exceptions, and
   a try-catch construct to deal with them. We also define the set of
   types which we end up with in the paper in Section 10.

   This file therefore corresponds to Sections 9 and 10. -}


data Value : Set where
  I : ℤ → Value
  B : Bool → Value
  Throw : Value
  Error : Value

data Expr : Set where
  Val : Value → Expr
  Add : Expr → Expr → Expr
  If : Expr → Expr → Expr → Expr
  Catch : Expr → Expr → Expr



add : Value → Value → Value
add (I n) (I m) = I (n + m)
add Throw _     = Throw
add (I _) Throw = Throw
add _     _     = Error

cond : Value → Value → Value → Value
cond (B b) v w = if b then v else w
cond Throw _ _ = Throw
cond _     _ _ = Error

catch : Value → Value → Value
catch Throw v = v
catch v     _ = v



data Type : Set where
  INT BOOL INT? BOOL? THROW ERROR TOP : Type

tval : Value → Type
tval (I x) = INT
tval (B x) = BOOL
tval Throw = THROW
tval Error = ERROR

variable t t' u u' : Type



{- For convenience and brevity, we define equality in terms of an
   injective map to the natural numbers. -}
module Equality where
  inj : Type → ℕ
  inj INT = 0
  inj BOOL = 1
  inj INT? = 2
  inj BOOL? = 3
  inj THROW = 4
  inj ERROR = 5
  inj TOP = 6

  is-inj : ∀ {x y : Type} → inj x ≡ inj y → x ≡ y
  is-inj {INT} {INT} eq = refl
  is-inj {BOOL} {BOOL} eq = refl
  is-inj {INT?} {INT?} eq = refl
  is-inj {BOOL?} {BOOL?} eq = refl
  is-inj {THROW} {THROW} eq = refl
  is-inj {ERROR} {ERROR} eq = refl
  is-inj {TOP} {TOP} eq = refl
  
  _==_ : ∀ (t t' : Type) → Dec (t ≡ t')
  _==_ = via-injection {S = setoid Type} {T = setoid ℕ}
    (record { to = inj; cong = cong inj; injective = is-inj }) _≟_


module Ordering where
  open Equality
  
  data _≤_ : Rel Type zero where
    ≤-bot : ERROR ≤ t
    ≤-top : t ≤ TOP
    ≤-refl : t ≤ t
    INT?≤INT : INT? ≤ INT
    INT?≤THROW : INT? ≤ THROW
    BOOL?≤BOOL : BOOL? ≤ BOOL
    BOOL?≤THROW : BOOL? ≤ THROW

  _≤?_ : ∀ (t t' : Type) → Dec (t ≤ t')
  t ≤? TOP = yes ≤-top
  ERROR ≤? t' = yes ≤-bot
  INT ≤? INT = yes ≤-refl
  INT ≤? BOOL = no λ ()
  INT ≤? INT? = no λ ()
  INT ≤? BOOL? = no λ ()
  INT ≤? THROW = no λ ()
  INT ≤? ERROR = no λ ()
  BOOL ≤? INT = no λ ()
  BOOL ≤? BOOL = yes ≤-refl
  BOOL ≤? INT? = no λ ()
  BOOL ≤? BOOL? = no λ ()
  BOOL ≤? THROW = no λ ()
  BOOL ≤? ERROR = no λ ()
  INT? ≤? INT = yes INT?≤INT
  INT? ≤? BOOL = no λ ()
  INT? ≤? INT? = yes ≤-refl
  INT? ≤? BOOL? = no λ ()
  INT? ≤? THROW = yes INT?≤THROW
  INT? ≤? ERROR = no λ ()
  BOOL? ≤? INT = no λ ()
  BOOL? ≤? BOOL = yes BOOL?≤BOOL
  BOOL? ≤? INT? = no λ ()
  BOOL? ≤? BOOL? = yes ≤-refl
  BOOL? ≤? THROW = yes BOOL?≤THROW
  BOOL? ≤? ERROR = no λ ()
  THROW ≤? INT = no λ ()
  THROW ≤? BOOL = no λ ()
  THROW ≤? INT? = no λ ()
  THROW ≤? BOOL? = no λ ()
  THROW ≤? THROW = yes ≤-refl
  THROW ≤? ERROR = no λ ()
  TOP ≤? INT = no λ ()
  TOP ≤? BOOL = no λ ()
  TOP ≤? INT? = no λ ()
  TOP ≤? BOOL? = no λ ()
  TOP ≤? THROW = no λ ()
  TOP ≤? ERROR = no λ ()


module GLB where
  open Equality
  open Ordering
  open import Monotonicity.Comparison Type _≤_ _≤?_ _==_

  _⊓_ : Op₂ Type
  t ⊓ t' with t compare t'
  ... | yes (LT _) = t
  ... | yes (EQ _) = t
  ... | yes (GT _) = t'
  ... | no p with INT? ≤? t ∧? INT? ≤? t'
                | BOOL? ≤? t ∧? BOOL? ≤? t'
  ...   | yes q | _     = INT?
  ...   | no q  | yes r = BOOL?
  ...   | no q  | no r  = ERROR
  infixr 15 _⊓_

