module Exceptions.Calculations.Fusion where

open import Data.Product renaming (_×_ to _∧_; proj₁ to l; proj₂ to r)
open import Data.Sum using (inj₁; inj₂; [_,_]) renaming (_⊎_ to _∨_)
open import Data.Bool using (Bool; true; false)
open import Data.Integer using (_+_)
open import Data.Empty
open import Relation.Binary.PropositionalEquality
  using (_≡_; cong; subst; sym)

open import Function using (id; _∘_)
open import Exceptions.Definitions
open import Exceptions.Fusion
open import Exceptions.Properties hiding (_≤_)

open Equality
open Ordering
open GLB
open Composing

{- In this file, similar to <Conditionals/Calculations/Fusion.agda>,
   we formalise the fold-fusion calculations given in the paper. The
   calculations presented here correspond to those in Section 9.

   We begin by defining type-checking algebras add', cond', and catch'
   which together constitute the folding operators defining texp. Each
   of these functions is defined by composing a number of constraints
   together, taking their greatest lower bound. This corresponds
   precisely to the definitions given in the paper.

   Writing the definitions in this way gives us monotonicity for free,
   as explained in the paper and as demonstrated in Monotone, below. -}

eval : Expr → Value
eval = folde id add cond catch

add' : Type → Type → Type
add' t t' = c₁ ⊓ c₂ ⊓ c₃
  module Add where
    c₁ = (t ≤? INT ∧? t' ≤? INT)                   ⇛ INT
    c₂ = (t ≤? THROW ∨? (t ≤? INT ∧? t' ≤? THROW)) ⇛ THROW
    c₃ = (t ≤? BOOL ∨? (t ≤? INT ∧? t' ≤? BOOL))   ⇛ ERROR

cond' : Type → Type → Type → Type
cond' s t t' = c₁ ⊓ c₂ ⊓ c₃
  module Cond where
    c₁ = (s ≤? BOOL)  ⇛ t ⊓ t'
    c₂ = (s ≤? INT)   ⇛ ERROR
    c₃ = (s ≤? THROW) ⇛ THROW

catch' : Type → Type → Type
catch' t t' = c₁ ⊓ c₂ ⊓ c₃ ⊓ c₄
  module Catch where
    c₁ = (t ≤? THROW) ⇛ t'
    c₂ = (t ≤? ERROR) ⇛ ERROR
    c₃ = (t ≤? BOOL)  ⇛ BOOL
    c₄ = (t ≤? INT)   ⇛ INT

texp : Expr → Type
texp = folde tval add' cond' catch'


{- Here, we write down the specific sets of constraints which we
   derived from the calculations in the paper. The calculations are
   shown and formalised later on, in the Homo module below.

   In this module we show that the calculated constraints do in fact
   yield the definitions we gave above, by making entirely syntax-
   directed use of various lemmas relating to the greatest lower bound
   and its interaction with ⇛. In doing so, we demonstrate that the
   definitions trivially follow from these constraints.

   AllAdd, AllCond, and AllCatch are propositions representing all
   of the constraints for each function respectively being true. We
   prove these three propositions here. -}
module Constraints where
  open import Reasoning.Syntax
  open import Reasoning.Implication
  open use-⇐ (⇐-preorder)

  AllAdd AllCond AllCatch : Set
  add₁ add₂ add₃ add₄ : Set
  cond₁ cond₂ cond₃ cond₄ cond₅ : Set
  catch₁ catch₂ catch₃ catch₄ : Set

  add₁ = ∀ {t t'}
    → add' t t' ≤ ((t ≤? INT ∧? t' ≤? INT) ⇛ INT)
  add₂ = ∀ {t t'}
    → add' t t' ≤ ((t ≤? THROW) ⇛ THROW)
  add₃ = ∀ {t t'}
    → add' t t' ≤ ((t ≤? INT ∧? t' ≤? THROW) ⇛ THROW)
  add₄ = ∀ {t t'}
    → add' t t' ≤ ((t ≤? BOOL ∨? (t ≤? INT ∧? t' ≤? BOOL)) ⇛ ERROR)

  cond₁ = ∀ {s t t'} → cond' s t t' ≤ ((s ≤? BOOL) ⇛ t)
  cond₂ = ∀ {s t t'} → cond' s t t' ≤ ((s ≤? BOOL) ⇛ t')
  cond₃ = ∀ {s t t'} → cond' s t t' ≤ ((s ≤? INT) ⇛ ERROR)
  cond₄ = ∀ {s t t'} → cond' s t t' ≤ ((s ≤? ERROR) ⇛ ERROR)
  cond₅ = ∀ {s t t'} → cond' s t t' ≤ ((s ≤? THROW) ⇛ THROW)

  catch₁ = ∀ {t t'} → catch' t t' ≤ ((t ≤? THROW) ⇛ t')
  catch₂ = ∀ {t t'} → catch' t t' ≤ ((t ≤? ERROR) ⇛ ERROR)
  catch₃ = ∀ {t t'} → catch' t t' ≤ ((t ≤? BOOL) ⇛ BOOL)
  catch₄ = ∀ {t t'} → catch' t t' ≤ ((t ≤? INT) ⇛ INT)

  AllAdd = add₁ ∧ add₂ ∧ add₃ ∧ add₄
  AllCond = cond₁ ∧ cond₂ ∧ cond₃ ∧ cond₄ ∧ cond₅
  AllCatch = catch₁ ∧ catch₂ ∧ catch₃ ∧ catch₄

  all-add : AllAdd
  l all-add {t} {t'} =
    x∧y≤x _ (Add.c₂ t t' ⊓ Add.c₃ t t')
  l (r all-add) {t} {t'} =
    trans (trans (x∧y≤y _ _) (x∧y≤x _ (Add.c₃ t t')))
          (⇛-left (t ≤? THROW) (t ≤? INT ∧? t' ≤? THROW))
  l (r (r all-add)) {t} {t'} =
    trans (trans (x∧y≤y _ _) (x∧y≤x _ (Add.c₃ t t')))
          (⇛-right (t ≤? THROW) (t ≤? INT ∧? t' ≤? THROW))
  r (r (r all-add)) {t} {t'} =
    trans (x∧y≤y _ _) (x∧y≤y _ (Add.c₃ t t'))

  all-cond : AllCond
  l all-cond {s} {t} {t'} =
    trans (x∧y≤x _ (Cond.c₂ s t t' ⊓ Cond.c₃ s t t'))
          (⇛-weaken (s ≤? BOOL) (s ≤? BOOL) id (x∧y≤x t t'))
  l (r all-cond) {s} {t} {t'} =
    trans (x∧y≤x _ (Cond.c₂ s t t' ⊓ Cond.c₃ s t t'))
          (⇛-weaken (s ≤? BOOL) (s ≤? BOOL) id (x∧y≤y t t'))
  l (r (r all-cond)) {s} {t} {t'} =
    trans (x∧y≤y _ _)
          (x∧y≤x _ (Cond.c₃ s t t'))
  l (r (r (r all-cond))) {s} {t} {t'} =
    trans (x∧y≤y _ _)
      (trans (x∧y≤x _ (Cond.c₃ s t t'))
        (⇛-weaken (s ≤? ERROR) (s ≤? INT)
          (λ b → subst (λ u → u ≤ INT) (sym (≤-has-bot b)) ≤-bot)
          ≤-bot))
  r (r (r (r all-cond))) {s} {t} {t'} =
    trans (x∧y≤y _ _)
      (trans (x∧y≤y _ (Cond.c₃ s t t'))
        ≤-refl)

  all-catch : AllCatch
  l all-catch {t} {t'} =
    x∧y≤x _ (Catch.c₂ t t' ⊓ Catch.c₃ t t' ⊓ Catch.c₄ t t')
  l (r all-catch) {t} {t'} =
    trans (x∧y≤y _ _)
          (x∧y≤x _ (Catch.c₃ t t' ⊓ Catch.c₄ t t'))
  l (r (r all-catch)) {t} {t'} =
    trans (x∧y≤y _ _)
      (trans (x∧y≤y _ _)
        (x∧y≤x _ (Catch.c₄ t t')))
  r (r (r all-catch)) {t} {t'} =
    trans (x∧y≤y _ _)
      (trans (x∧y≤y _ _)
        (x∧y≤y _ (Catch.c₄ t t')))
  

{- Here we trivially show monotonicity of add', cond', and catch',
   using the fact that any operators defined with certain combinations
   of the ⇛ and ⊓ operations are monotone.

   As in the analogous proofs for the Conditional language, the proof
   structure here is entirely trivial and directed by the syntax of
   add', cond', and catch'. -}
module Monotone where
  add'-mono : MonoT₂ add'
  add'-mono = mono₂-⊓
    (mono₂-⇛
      (mono₂-∧? (mono₂-≤?-constₗ INT) (mono₂-≤?-constᵣ INT))
      (mono₂-const INT))
    (mono₂-⊓
      (mono₂-⇛
        (mono₂-∨?
          (mono₂-≤?-constₗ THROW)
          (mono₂-∧? (mono₂-≤?-constₗ INT) (mono₂-≤?-constᵣ THROW)))
        (mono₂-const THROW))
      (mono₂-⇛
        (mono₂-∨?
          (mono₂-≤?-constₗ BOOL)
          (mono₂-∧? (mono₂-≤?-constₗ INT) (mono₂-≤?-constᵣ BOOL)))
        (mono₂-const ERROR)))

  cond'-mono : MonoT₃ cond'
  cond'-mono = mono₃-⊓
    (mono₃-⇛ (mono₃-≤?-const₁ BOOL)
             (mono₃-⊓ mono₃-u mono₃-v))
    (mono₃-⊓
      (mono₃-⇛ (mono₃-≤?-const₁ INT) (mono₃-const ERROR))
      (mono₃-⇛ (mono₃-≤?-const₁ THROW) (mono₃-const THROW)))

  catch'-mono : MonoT₂ catch'
  catch'-mono = mono₂-⊓
    (mono₂-⇛ (mono₂-≤?-constₗ THROW) mono₂-snd)
    (mono₂-⊓
      (mono₂-⇛ (mono₂-≤?-constₗ ERROR) (mono₂-const ERROR))
      (mono₂-⊓
        (mono₂-⇛ (mono₂-≤?-constₗ BOOL) (mono₂-const BOOL))
        (mono₂-⇛ (mono₂-≤?-constₗ INT) (mono₂-const INT))))


{- Now we prove the homomorphism properties of add', cond', and catch'
   in the same way as shown in the calculations in the paper.

   Specifically, we show that the constraints listed above in the
   Constraints module imply the homomorphism properties we need. This,
   combined then with the proofs that those constraints do in fact
   hold for our ⊓-defined definitions, lets us establish homomorphism
   for these three algebras. -}
module Homo where
  open Monotone
  open import Reasoning.Syntax
  open import Reasoning.Implication
  open use-⇐ (⇐-preorder)

  homo-add : ∀ x y
           → Constraints.AllAdd
           → tval (add x y) ≥ add' (tval x) (tval y)

  homo-add (I n) (I m) = begin
    (add' (tval (I n)) (tval (I m)) ≤ tval (add (I n) (I m)))
      ⇔⟨⟩
    (add' INT INT ≤ tval (I (n + m)))
      ⇔⟨⟩
    (add' INT INT ≤ INT)
      ⇔⟨ record
        { to = λ h {t} {t'} (p , q) → add'-mono t INT t' INT p q
        ; from = λ h → h {INT} {INT} (≤-refl , ≤-refl) } ⟩
    (∀ {t t'} → t ≤ INT ∧ t' ≤ INT → add' t t' ≤ INT)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? INT ∧? t' ≤? INT) ⇛ INT))
      ⇔⟨⟩
    Constraints.add₁ ⇐⟨ l ⟩ Constraints.AllAdd ∎

  homo-add Throw y = begin
    (add' (tval Throw) (tval y) ≤ tval (add Throw y))
      ⇔⟨⟩
    (add' THROW (tval y) ≤ THROW)
      ⇐⟨ (λ x → x {THROW} {tval y}) ⟩
    (∀ {t t'} → add' THROW t' ≤ THROW)
      ⇔⟨ record { to = λ h {t} {t'} lt → add'-mono t THROW t' t' lt ≤-refl
                ; from = λ h {t'} → h {THROW} {t'} ≤-refl } ⟩
    (∀ {t t'} → t ≤ THROW → add' t t' ≤ THROW)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? THROW) ⇛ THROW))
      ⇔⟨⟩
    Constraints.add₂ ⇐⟨ l ∘ r ⟩ Constraints.AllAdd ∎

  homo-add (I n) Throw = begin
    (add' (tval (I n)) (tval Throw) ≤ tval (add (I n) Throw))
      ⇔⟨⟩
    (add' INT THROW ≤ tval Throw)
      ⇔⟨⟩
    (add' INT THROW ≤ THROW)
      ⇔⟨ record
        { to = λ lt {t} {t'} (p , q) → add'-mono t INT t' THROW p q
        ; from = λ h → h {INT} {THROW} (≤-refl , ≤-refl) } ⟩
    (∀ {t t'} → t ≤ INT ∧ t' ≤ THROW → add' t t' ≤ THROW)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? INT ∧? t' ≤? THROW) ⇛ THROW))
      ⇔⟨⟩
    Constraints.add₃ ⇐⟨ l ∘ r ∘ r ⟩ Constraints.AllAdd ∎

  {- The following cases correspond to the case listed in the paper
     as "x = Error or x = B b or (x = I n and (y = Error or y = B b))".

     Agda does not permit us to write patterns like this, so we have
     to enumerate them separately. The logic within each calculation
     is basically the same though, and agrees with the calculations
     given in the paper. -}
   
  homo-add Error y = begin
    (add' (tval Error) (tval y) ≤ tval (add Error y))
      ⇔⟨⟩
    (add' ERROR (tval y) ≤ ERROR)
      ⇐⟨ (λ x → x {ERROR} {tval y}) ⟩
    (∀ {t t'} → add' ERROR t' ≤ ERROR)
      ⇔⟨ record
        { to = λ lt {t} {t'} p → trans
          (add'-mono t ERROR t' t' p ≤-refl)
          (lt {ERROR} {t'})
        ; from = λ h {t} {t'} → h {ERROR} {t'} ≤-refl } ⟩
    (∀ {t t'} → t ≤ ERROR → add' t t' ≤ ERROR)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? ERROR) ⇛ ERROR))
      ⇐⟨ (λ h {t} {t'} →
        trans (h {t} {t'})
          (trans (⇛-left (t ≤? BOOL) _)
            (⇛-weaken (t ≤? ERROR) (t ≤? BOOL) ≤-use-bot ≤-refl))) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? BOOL ∨? (t ≤? INT ∧? t' ≤? BOOL)) ⇛ ERROR))
      ⇔⟨⟩
    Constraints.add₄ ⇐⟨ r ∘ r ∘ r ⟩ Constraints.AllAdd ∎

  homo-add (B b) y = begin
    (add' (tval (B b)) (tval y) ≤ tval (add (B b) y))
      ⇔⟨⟩
    (add' BOOL (tval y) ≤ ERROR)
      ⇐⟨ (λ x → x {BOOL} {tval y}) ⟩
    (∀ {t t'} → add' BOOL t' ≤ ERROR)
      ⇔⟨ record
        { to = λ k {t} {t'} p → add'-mono t BOOL t' t' p ≤-refl
        ; from = λ h → h {BOOL} {tval y} ≤-refl } ⟩
    (∀ {t t'} → t ≤ BOOL → add' t t' ≤ ERROR)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? BOOL) ⇛ ERROR))
      ⇐⟨ (λ h {t} {t'} →
        trans (h {t} {t'})
          (⇛-left (t ≤? BOOL) _)) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? BOOL ∨? (t ≤? INT ∧? t' ≤? BOOL)) ⇛ ERROR))
      ⇔⟨⟩
    Constraints.add₄ ⇐⟨ r ∘ r ∘ r ⟩ Constraints.AllAdd ∎

  homo-add (I n) Error = begin
    (add' (tval (I n)) (tval Error) ≤ tval (add (I n) Error))
      ⇔⟨⟩
    (add' INT ERROR ≤ ERROR)
      ⇔⟨ record
        { to = λ k {t} {t'} (p , q) → add'-mono t INT t' ERROR p q
        ; from = λ h → h {INT} {ERROR} (≤-refl , ≤-refl) } ⟩
    (∀ {t t'} → t ≤ INT ∧ t' ≤ ERROR → add' t t' ≤ ERROR)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? INT ∧? t' ≤? ERROR) ⇛ ERROR))
      ⇐⟨ (λ h {t} {t'} →
        trans (h {t} {t'})
          (trans (⇛-right (t ≤? BOOL) _)
            (⇛-weaken ((t ≤? INT) ∧? (t' ≤? ERROR)) _
              (λ (t≤INT , t'≤ERROR) → t≤INT , ≤-use-bot t'≤ERROR)
              ≤-bot))) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? BOOL ∨? (t ≤? INT ∧? t' ≤? BOOL)) ⇛ ERROR))
      ⇔⟨⟩
    Constraints.add₄ ⇐⟨ r ∘ r ∘ r ⟩ Constraints.AllAdd ∎

  homo-add (I n) (B b) = begin
    (add' (tval (I n)) (tval (B b)) ≤ tval (add (I n) (B b)))
      ⇔⟨⟩
    (add' INT BOOL ≤ ERROR)
      ⇔⟨ record
        { to = λ lt {t} {t'} (p , q) → add'-mono t INT t' BOOL p q
        ; from = λ h → h {INT} {BOOL} (≤-refl , ≤-refl) }  ⟩
    (∀ {t t'} → t ≤ INT ∧ t' ≤ BOOL → add' t t' ≤ ERROR)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? INT ∧? t' ≤? BOOL) ⇛ ERROR))
      ⇐⟨ (λ h {t} {t'} →
        trans (h {t} {t'})
          (⇛-right (t ≤? BOOL) _)) ⟩
    (∀ {t t'} → add' t t' ≤ ((t ≤? BOOL ∨? (t ≤? INT ∧? t' ≤? BOOL)) ⇛ ERROR))
      ⇔⟨⟩
    Constraints.add₄ ⇐⟨ r ∘ r ∘ r ⟩ Constraints.AllAdd ∎
  
  homo-cond : ∀ x y z
    → Constraints.AllCond
    → tval (cond x y z) ≥ cond' (tval x) (tval y) (tval z)

  homo-cond (B true) y z = begin
    (cond' (tval (B true)) (tval y) (tval z) ≤ tval (cond (B true) y z))
      ⇔⟨⟩
    (cond' BOOL (tval y) (tval z) ≤ tval y)
      ⇐⟨ (λ x → x {BOOL} {tval y} {tval z}) ⟩
    (∀ {s t t'} → cond' BOOL t t' ≤ t)
      ⇔⟨ record
        { to = λ k {s} {t} {t'} lt →
          trans
            (cond'-mono s BOOL t t t' t' lt ≤-refl ≤-refl)
            (k {s} {t} {t'})
        ; from = λ h {s} {t} {t'} → h {BOOL} {t} {t'} ≤-refl } ⟩
    (∀ {s t t'} → s ≤ BOOL → cond' s t t' ≤ t)
      ⇔⟨ with₃ (λ s t t' → ⇛-def) ⟩
    (∀ {s t t'} → cond' s t t' ≤ ((s ≤? BOOL) ⇛ t))
      ⇔⟨⟩
    Constraints.cond₁ ⇐⟨ l ⟩ Constraints.AllCond ∎

  homo-cond (B false) y z = begin
    (cond' (tval (B false)) (tval y) (tval z) ≤ tval (cond (B false) y z))
      ⇔⟨⟩
    (cond' BOOL (tval y) (tval z) ≤ tval z)
      ⇐⟨ (λ x → x {BOOL} {tval y} {tval z}) ⟩
    (∀ {s t t'} → cond' BOOL t t' ≤ t')
      ⇔⟨ record
        { to = λ k {s} {t} {t'} lt →
          trans
            (cond'-mono s BOOL t t t' t' lt ≤-refl ≤-refl)
            (k {s} {t} {t'})
        ; from = λ h {s} {t} {t'} → h {BOOL} {t} {t'} ≤-refl } ⟩
    (∀ {s t t'} → s ≤ BOOL → cond' s t t' ≤ t')
      ⇔⟨ with₃ (λ s t t' → ⇛-def) ⟩
    (∀ {s t t'} → cond' s t t' ≤ ((s ≤? BOOL) ⇛ t'))
      ⇔⟨⟩
    Constraints.cond₂ ⇐⟨ l ∘ r ⟩ Constraints.AllCond ∎

  homo-cond (I n) y z = begin
    (cond' (tval (I n)) (tval y) (tval z) ≤ tval (cond (I n) y z))
      ⇔⟨⟩
    (cond' INT (tval y) (tval z) ≤ tval Error)
      ⇐⟨ (λ x → x {INT} {tval y} {tval z}) ⟩
    (∀ {s t t'} → cond' INT t t' ≤ ERROR)
      ⇔⟨ record
        { to = λ k {s} {t} {t'} lt →
          trans
            (cond'-mono s INT t t t' t' lt ≤-refl ≤-refl)
            (k {s} {t} {t'})
        ; from = λ h {s} {t} {t'} → h {INT} {t} {t'} ≤-refl } ⟩
    (∀ {s t t'} → s ≤ INT → cond' s t t' ≤ ERROR)
      ⇔⟨ with₃ (λ s t t' → ⇛-def) ⟩
    (∀ {s t t'} → cond' s t t' ≤ ((s ≤? INT) ⇛ ERROR))
      ⇔⟨⟩
    Constraints.cond₃ ⇐⟨ l ∘ r ∘ r ⟩ Constraints.AllCond ∎
    
  homo-cond Error y z = begin
    (cond' (tval Error) (tval y) (tval z) ≤ tval (cond Error y z))
      ⇔⟨⟩
    (cond' ERROR (tval y) (tval z) ≤ tval Error)
      ⇔⟨⟩
    (cond' ERROR (tval y) (tval z) ≤ ERROR)
      ⇐⟨ (λ x → x {ERROR} {tval y} {tval z}) ⟩
    (∀ {s t t'} → cond' ERROR t t' ≤ ERROR)
      ⇔⟨ record
        { to = λ k {s} {t} {t'} lt →
          trans
            (cond'-mono s ERROR t t t' t' lt ≤-refl ≤-refl)
            (k {s} {t} {t'})
        ; from = λ h {s} {t} {t'} → h {ERROR} {t} {t'} ≤-refl } ⟩
    (∀ {s t t'} → s ≤ ERROR → cond' s t t' ≤ ERROR)
      ⇔⟨ with₃ (λ s t t' → ⇛-def) ⟩
    (∀ {s t t'} → cond' s t t' ≤ ((s ≤? ERROR) ⇛ ERROR))
      ⇔⟨⟩
    Constraints.cond₄ ⇐⟨ l ∘ r ∘ r ∘ r ⟩ Constraints.AllCond ∎

  homo-cond Throw y z = begin
    (cond' (tval Throw) (tval y) (tval z) ≤ tval (cond Throw y z))
      ⇔⟨⟩
    (cond' THROW (tval y) (tval z) ≤ THROW)
      ⇐⟨ (λ k → k {TOP} {tval y} {tval z}) ⟩
    (∀ {s t t'} → cond' THROW t t' ≤ THROW)
      ⇔⟨ with₃ (λ s t t' → record
        { to = λ h lt → cond'-mono s THROW t t t' t' lt ≤-refl ≤-refl
        ; from = λ h → ≤-refl }) ⟩
    (∀ {s t t'} → s ≤ THROW → cond' s t t' ≤ THROW)
      ⇔⟨ with₃ (λ s t t' → ⇛-def) ⟩
    (∀ {s t t'} → cond' s t t' ≤ ((s ≤? THROW) ⇛ THROW))
      ⇔⟨⟩
    Constraints.cond₅ ⇐⟨ r ∘ r ∘ r ∘ r ⟩ Constraints.AllCond ∎

  homo-catch : ∀ x y
    → Constraints.AllCatch
    → tval (catch x y) ≥ catch' (tval x) (tval y)

  homo-catch Throw y = begin
    (catch' (tval Throw) (tval y) ≤ tval (catch Throw y))
      ⇔⟨⟩
    (catch' THROW (tval y) ≤ tval y)
      ⇐⟨ (λ k → k {TOP} {tval y}) ⟩
    (∀ {t t'} → catch' THROW t' ≤ t')
      ⇔⟨ with₂ (λ t t' → record
        { to = λ h lt → catch'-mono t THROW t' t' lt ≤-refl
        ; from = λ h → ≤-refl }) ⟩
    (∀ {t t'} → t ≤ THROW → catch' t t' ≤ t')
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → catch' t t' ≤ ((t ≤? THROW) ⇛ t'))
      ⇔⟨⟩
    Constraints.catch₁ ⇐⟨ l ⟩ Constraints.AllCatch ∎

  homo-catch Error y = begin
    (catch' (tval Error) (tval y) ≤ tval (catch Error y))
      ⇔⟨⟩
    (catch' ERROR (tval y) ≤ ERROR)
      ⇐⟨ (λ k → k {TOP} {tval y}) ⟩
    (∀ {t t'} → catch' ERROR t' ≤ ERROR)
      ⇔⟨ record
        { to = λ h {t} {t'} lt → catch'-mono t ERROR t' TOP lt ≤-top
        ; from = λ h {t} {t'} → h {t = ERROR} {t' = t'} ≤-refl } ⟩
    (∀ {t t'} → t ≤ ERROR → catch' t t' ≤ ERROR)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → catch' t t' ≤ ((t ≤? ERROR) ⇛ ERROR))
      ⇔⟨⟩
    Constraints.catch₂ ⇐⟨ l ∘ r ⟩ Constraints.AllCatch ∎

  homo-catch (B b) y = begin
    (catch' (tval (B b)) (tval y) ≤ tval (catch (B b) y))
      ⇔⟨⟩
    (catch' BOOL (tval y) ≤ BOOL)
      ⇐⟨ (λ k → k {TOP} {tval y}) ⟩
    (∀ {t t'} → catch' BOOL t' ≤ BOOL)
      ⇔⟨ with₂ (λ t t' → record
        { to = λ h lt → catch'-mono t BOOL t' t' lt ≤-refl
        ; from = λ h → ≤-refl }) ⟩
    (∀ {t t'} → t ≤ BOOL → catch' t t' ≤ BOOL)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → catch' t t' ≤ ((t ≤? BOOL) ⇛ BOOL))
      ⇔⟨⟩
    Constraints.catch₃ ⇐⟨ l ∘ r ∘ r ⟩ Constraints.AllCatch ∎
      
  homo-catch (I n) y = begin
    (catch' (tval (I n)) (tval y) ≤ tval (catch (I n) y))
      ⇔⟨⟩
    (catch' INT (tval y) ≤ INT)
      ⇐⟨ (λ k → k {TOP} {tval y}) ⟩
    (∀ {t t'} → catch' INT t' ≤ INT)
      ⇔⟨ with₂ (λ t t' → record
        { to = λ h lt → catch'-mono t INT t' t' lt ≤-refl
        ; from = λ h → ≤-refl }) ⟩
    (∀ {t t'} → t ≤ INT → catch' t t' ≤ INT)
      ⇔⟨ with₂ (λ t t' → ⇛-def) ⟩
    (∀ {t t'} → catch' t t' ≤ ((t ≤? INT) ⇛ INT))
      ⇔⟨⟩
    Constraints.catch₄ ⇐⟨ r ∘ r ∘ r ⟩ Constraints.AllCatch ∎
 

{- Finally, we can bring together the proofs of homomorphism conditions
   and monotonicity, as in Conditionals.Calculations.Fusion. This
   establishes texp as a correct type-checker. -}
correctness : Correct eval texp
correctness x = fusion tval
  id add cond catch
  tval add' cond' catch'
  Monotone.add'-mono Monotone.cond'-mono Monotone.catch'-mono
  --------------------------------------
  (λ {x} → ≤-refl)
  (λ {x y} → Homo.homo-add x y Constraints.all-add)
  (λ {x y z} → Homo.homo-cond x y z Constraints.all-cond)
  (λ {x y} → Homo.homo-catch x y Constraints.all-catch)
  --------------------------------------
  x
