module Conditionals.Properties where

open import Data.Product renaming (_×_ to _∧_)
open import Data.Sum renaming (_⊎_ to _∨_)
open import Data.Empty

open import Relation.Binary
  using (Transitive; Antisymmetric; IsPreorder; IsPartialOrder)
  renaming (_⇒_ to _R⇒_)
  
open import Relation.Binary.Lattice
  using ( IsMeetSemilattice; IsBoundedMeetSemilattice
        ; Infimum; MeetSemilattice )
        
open import Relation.Binary.PropositionalEquality
open import Relation.Binary.Bundles using (Poset; Preorder)

open import Relation.Nullary.Decidable
  using (yes; no)
  renaming (_⊎-dec_ to _∨?_)

open import Algebra using (Idempotent)

open import Agda.Builtin.Equality
open import Level

open import Reasoning.Implication
open import Conditionals.Definitions

open Ordering
open GLB
open Equality
open import Monotonicity.Comparison Type _≤_ _≤?_ _==_


{- This file does two main things:

     * Establishes that ≤ and ⊓ provide a bounded meet-semilattice
       structure for types.

     * States what "correctness" means. -}


{- Properties of ≤ ordering. It is a partial order, and it has both
   a top and bottom element. -}
≤-reflexive : _≡_ R⇒ _≤_
≤-reflexive refl = ≤-refl

≤-trans : Transitive _≤_
≤-trans ≤-bot q = ≤-bot
≤-trans ≤-refl q = q
≤-trans ≤-top ≤-top = ≤-top
≤-trans ≤-top ≤-refl = ≤-top

≤-antisym : Antisymmetric _≡_ _≤_
≤-antisym ≤-bot ≤-bot = refl
≤-antisym ≤-refl ≤-bot = refl
≤-antisym ≤-bot ≤-refl = refl
≤-antisym ≤-refl ≤-refl = refl
≤-antisym ≤-top ≤-top = refl
≤-antisym ≤-top ≤-refl = refl
≤-antisym ≤-refl ≤-top = refl

≤-isPreorder : IsPreorder _≡_ _≤_
≤-isPreorder = record { isEquivalence = isEquivalence
                      ; reflexive = ≤-reflexive
                      ; trans = ≤-trans }

≤-isPartialOrder : IsPartialOrder _≡_ _≤_
≤-isPartialOrder = record { isPreorder = ≤-isPreorder
                          ; antisym = ≤-antisym }

≤-has-top : TOP ≤ t' → TOP ≡ t'
≤-has-top ≤-top = refl
≤-has-top ≤-refl = refl

≤-has-bot : t ≤ ERROR → t ≡ ERROR
≤-has-bot ≤-bot = refl
≤-has-bot ≤-refl = refl



{- Properties of ⊓. It forms a bounded meet-semilattice with ≤ -}
⊓-idem : Idempotent _≡_ _⊓_
⊓-idem INT = refl
⊓-idem BOOL = refl
⊓-idem ERROR = refl
⊓-idem TOP = refl

⊓-topₗ : TOP ⊓ t' ≡ t'
⊓-topₗ {t'} with TOP compare t'
... | yes (LT x) = ≤-has-top x
... | yes (EQ x) = x
... | yes (GT x) = refl
... | no p = ⊥-elim (p (GT ≤-top))

⊓-topᵣ : t ⊓ TOP ≡ t
⊓-topᵣ = refl

⊓-is-lowerₗ : (t ⊓ t') ≤ t
⊓-is-lowerₗ {t} {t'} with t compare t'
... | yes (LT x) = ≤-refl
... | yes (EQ x) = ≤-refl
... | yes (GT x) = x
... | no p = ≤-bot

⊓-is-lowerᵣ : (t ⊓ t') ≤ t'
⊓-is-lowerᵣ {t} {t'} with t compare t'
... | yes (LT x) = x
... | yes (EQ x) rewrite x = ≤-refl
... | yes (GT x) = ≤-refl
... | no p = ≤-bot

⊓-is-greatest : u ≤ t → u ≤ t' → u ≤ (t ⊓ t')
⊓-is-greatest ≤-bot g = ≤-bot
⊓-is-greatest {t' = t'} ≤-top g rewrite ⊓-topₗ {t'} = g
⊓-is-greatest ≤-refl ≤-bot = ≤-bot
⊓-is-greatest {u = u} ≤-refl ≤-top rewrite ⊓-topᵣ {u} = ≤-refl
⊓-is-greatest {u = u} ≤-refl ≤-refl rewrite ⊓-idem u = ≤-refl

⊓-infimum : Infimum _≤_ _⊓_
⊓-infimum x y = ⊓-is-lowerₗ {x} {y}
              , ⊓-is-lowerᵣ
              , λ u → ⊓-is-greatest

⊓-isMeetSemilattice : IsMeetSemilattice _≡_ _≤_ _⊓_
⊓-isMeetSemilattice = record
  { isPartialOrder = ≤-isPartialOrder
  ; infimum = ⊓-infimum }

⊓-isBoundedMeetSemilattice : IsBoundedMeetSemilattice _≡_ _≤_ _⊓_ TOP
⊓-isBoundedMeetSemilattice = record
  { isMeetSemilattice = ⊓-isMeetSemilattice
  ; maximum = λ x → ≤-top }




{- Miscellaneous properties -}
add-bool : ∀ x y
  → (tval x ≤ BOOL) ∨ (tval y ≤ BOOL)
  → add x y ≡ Error
add-bool (B b) y (inj₁ p) = refl
add-bool Error y (inj₁ p) = refl
add-bool (I n) (B b') (inj₂ q) = refl
add-bool (B b) (B b') (inj₂ q) = refl
add-bool Error (B b') (inj₂ q) = refl
add-bool (I n) Error (inj₂ q) = refl
add-bool (B b) Error (inj₂ q) = refl
add-bool Error Error (inj₂ q) = refl




{- Expose the monotonicity definitions and properties -}
open import Monotonicity.Properties
  Type _≤_ _≤?_ _⊓_ TOP ⊓-isBoundedMeetSemilattice
  public



{- What is correctness? -}
Correct : (eval : Expr → Value)
        → (texp : Expr → Type)
        → Set
Correct eval texp = ∀ (x : Expr) → tval (eval x) ≥ texp x
