module Conditionals.Fusion where

open import Agda.Builtin.Equality

open import Conditionals.Definitions
open import Conditionals.Properties
open import Reasoning.Definitions

open Ordering

open import Reasoning.Syntax
open use-≥ (≥-poset)


{- In this file, we take the definitions of the condition language in
   Conditionals.Definitions, and provide a folding operator 'folde'.
   This is according to the definition given in the paper Section 6.
   We also state and prove the fold-fusion property. -}


folde : {A : Set}
      → (Value → A) → (A → A → A) → (A → A → A → A)
      → Expr → A
folde {A} val add cond = f
  where
    f : Expr → A
    f (Val v) = val v
    f (Add x y) = add (f x) (f y)
    f (If x y z) = cond (f x) (f y) (f z)

fusion : {A : Set}
     → (h : A → Type)

     -- cata operators
     → (val : Value → A)
     → (add : A → A → A)
     → (cond : A → A → A → A)
     → (val' : Value → Type)
     → (add' : Type → Type → Type)
     → (cond' : Type → Type → Type → Type)

     -- monotonicity of add' and cond'
     → MonoT₂ add'
     → MonoT₃ cond'

     -- homomorphism properties
     → (∀ {x}     → h (val x) ≥ val' x)
     → (∀ {x y}   → h (add x y) ≥ add' (h x) (h y))
     → (∀ {x y z} → h (cond x y z) ≥ cond' (h x) (h y) (h z))
     -------------------------------------------------------------
     → ∀ e
     → (h (folde val add cond e)) ≥ folde val' add' cond' e


fusion h val add cond val' add' cond' ma mc hv ha hc e = go e
  where
    go : (e : Expr)
       → h (folde val add cond e) ≥ folde val' add' cond' e

    go (Val x) = begin
      h (folde val add cond (Val x))
        ≡⟨⟩
      h (val x)
        ≥⟨ hv ⟩
      val' x
        ≡⟨⟩
      folde val' add' cond' (Val x)
        ∎

    go (Add x y) = begin
      h (folde val add cond (Add x y))
        ≡⟨⟩
      h (add (folde val add cond x) (folde val add cond y))
        ≥⟨ ha ⟩
      add' (h (folde val add cond x)) (h (folde val add cond y))
        ≥⟨ ma _ _ _ _ (go x) (go y) ⟩
      add' (folde val' add' cond' x) (folde val' add' cond' y)
        ≡⟨⟩
      folde val' add' cond' (Add x y)
        ∎

    go (If x y z) = begin
      h (folde val add cond (If x y z))
        ≡⟨⟩
      h (cond (folde val add cond x)
              (folde val add cond y)
              (folde val add cond z))
        ≥⟨ hc ⟩
      cond' (h (folde val add cond x))
            (h (folde val add cond y))
            (h (folde val add cond z))
        ≥⟨ mc _ _ _ _ _ _ (go x) (go y) (go z) ⟩
      cond' (folde val' add' cond' x)
            (folde val' add' cond' y)
            (folde val' add' cond' z)
        ≡⟨⟩
      folde val' add' cond' (If x y z)
        ∎
