module Conditionals.Definitions where

open import Data.Integer using (ℤ; _+_)
open import Data.Bool using (Bool; if_then_else_; true; false)
open import Data.Sum renaming (_⊎_ to _∨_)

open import Relation.Binary using (Rel)
open import Relation.Binary.PropositionalEquality
open import Relation.Nullary.Decidable using (Dec; yes; no)
open import Relation.Nullary.Negation using (¬_)
open import Algebra using (Op₂)
open import Agda.Builtin.Equality
open import Level

open import Reasoning.Definitions


{- This file contains the definitions for a language consisting of
   integers and booleans, addition, and conditional expressions, as is
   presented in Section 3 in the paper. The algebras add and cond are
   defined here, which constitute the semantics of the language.

   In addition, we define the set of types as in Section 4 (augmented
   with TOP, which comes later in the paper), and also decidable
   equality and ordering over these types. -}


data Value : Set where
  I : ℤ → Value
  B : Bool → Value
  Error : Value

data Expr : Set where
  Val : Value → Expr
  Add : Expr → Expr → Expr
  If : Expr → Expr → Expr → Expr



add : Value → Value → Value
add (I n) (I m) = I (n + m)
add _ _ = Error

cond : Value → Value → Value → Value
cond (B b) v w = if b then v else w
cond _ _ _ = Error



data Type : Set where
  INT BOOL ERROR TOP : Type

tval : Value → Type
tval (I x) = INT
tval (B x) = BOOL
tval Error = ERROR

variable t t' u u' : Type



{- Note that the definitions for decidable equality and ordering are
   not particularly pretty. This is a limitation of Agda, but note that
   ≤? directly reflects the lattice structure shown in the paper. -}


module Equality where
  _==_ : ∀ (t t' : Type) → Dec (t ≡ t')
  INT == INT = yes refl
  BOOL == BOOL = yes refl
  ERROR == ERROR = yes refl
  TOP == TOP = yes refl
  INT == BOOL = no (λ ())
  INT == ERROR = no (λ ())
  INT == TOP = no (λ ())
  BOOL == INT = no (λ ())
  BOOL == ERROR = no (λ ())
  BOOL == TOP = no (λ ())
  ERROR == INT = no (λ ())
  ERROR == BOOL = no (λ ())
  ERROR == TOP = no (λ ())
  TOP == INT = no (λ ())
  TOP == BOOL = no (λ ())
  TOP == ERROR = no (λ ())



module Ordering where
  open Equality
  
  data _≤_ : Rel Type 0ℓ where
    ≤-bot : ERROR ≤ t
    ≤-top : t ≤ TOP
    ≤-refl : t ≤ t

  _≤?_ : ∀ (t t' : Type) → Dec (t ≤ t')
  t ≤? TOP = yes ≤-top
  ERROR ≤? INT = yes ≤-bot
  ERROR ≤? BOOL = yes ≤-bot
  ERROR ≤? ERROR = yes ≤-bot
  INT ≤? INT = yes ≤-refl
  INT ≤? BOOL = no λ ()
  INT ≤? ERROR = no λ ()
  BOOL ≤? INT = no λ ()
  BOOL ≤? BOOL = yes ≤-refl
  BOOL ≤? ERROR = no λ ()
  TOP ≤? INT = no λ ()
  TOP ≤? BOOL = no λ ()
  TOP ≤? ERROR = no λ ()
  infixl 7 _≤?_
  


module GLB where
  open Equality
  open Ordering
  open import Monotonicity.Comparison Type _≤_ _≤?_ _==_

  _⊓_ : Op₂ Type
  t ⊓ t' with t compare t'
  ... | yes (LT _) = t
  ... | yes (EQ _) = t
  ... | yes (GT _) = t'
  ... | no p = ERROR
  infixl 15 _⊓_

