module Conditionals.Calculations.Fusion where

open import Data.Product renaming (_×_ to _∧_; proj₁ to l; proj₂ to r)
open import Data.Sum using (inj₁; inj₂) renaming (_⊎_ to _∨_)
open import Data.Bool using (Bool; true; false)
open import Data.Integer using (_+_)
open import Data.Empty
open import Relation.Binary.PropositionalEquality using (_≡_; cong)

open import Function using (id; _∘_)

open import Conditionals.Definitions
open import Conditionals.Fusion
open import Conditionals.Properties hiding (_≤_)

open Equality
open Ordering
open GLB
open Composing

{- In this file, we formalise the calculations given in the paper in
   Sections 6, 7, and 8. That is, we take an algebraic approach to
   evaluation and type-checking (using a fold operator defined in
   Conditionals.Fusion).

   We calculate a set of sufficient constraints, (see in Constraints),
   which together imply the homomorphism conditions for applying
   fold fusion (see in Homo).

   In calculating the type checking algebras in this fashion, we get
   monotonicity "for free", here shown by the trivial and entirely
   syntax-directed composition of some combinators. See Monotonicity
   for these proofs, and Conditionals.Properties for the combinators
   which we make use of. -}


{- To begin with, eval is given as a fold. So too is texp, defined by
   add' and cond'. These definitions are constructed in a calculational
   manner, as described in the paper, but Agda does not allow us to
   "define" them via calculation, so for now we just write them down. -}

eval : Expr → Value
eval = folde id add cond

add' : Type → Type → Type
add' t t' = c₁ ⊓ c₂
  module Add where
    c₁ = (t ≤? INT ∧? t' ≤? INT)   ⇛ INT
    c₂ = (t ≤? BOOL ∨? t' ≤? BOOL) ⇛ ERROR

cond' : Type → Type → Type → Type
cond' s t t' = c₁ ⊓ c₂
  module Cond where
    c₁ = (s ≤? BOOL) ⇛ t ⊓ t'
    c₂ = (s ≤? INT)  ⇛ ERROR

texp : Expr → Type
texp = folde tval add' cond'


{- Here we define the constraints defining add' and cond', which are
   shown calculationally below in the Homo module. We also prove,
   again by calculation, that the constraints follow by definition of
   add' and cond' (in all-add and all-cond).

   AllAdd and AllCond are propositions representing all of the add'
   constraints being true and all of the cond' being true, resp.

   Proofs of AllAdd and AllCond are named all-add and all-cond, and
   follow easily from the calculated definitions. -}
module Constraints where
  open import Reasoning.Syntax
  open import Reasoning.Implication
  open use-⇐ (⇐-preorder)

  add₁ add₂ cond₁ cond₂ cond₃ cond₄ AllAdd AllCond : Set
  
  add₁ = add' INT INT ≤ INT
  add₂ = ∀ {t t'} → t ≤ BOOL → t' ≤ BOOL → add' t t' ≤ ERROR

  cond₁ = ∀ {t t'} → cond' BOOL t t'  ≤ t
  cond₂ = ∀ {t t'} → cond' BOOL t t'  ≤ t'
  cond₃ = ∀ {t t'} → cond' INT t t'   ≤ ERROR
  cond₄ = ∀ {t t'} → cond' ERROR t t' ≤ ERROR
  
  AllAdd = add₁ ∧ add₂
  AllCond = cond₁ ∧ cond₂ ∧ cond₃ ∧ cond₄

  all-add : AllAdd
  l all-add = ≤-refl
  r all-add ≤-bot ≤-bot = ≤-bot
  r all-add ≤-bot ≤-refl = ≤-bot
  r all-add ≤-refl q = ≤-bot

  all-cond : AllCond
  l all-cond {t} {t'} = trans
    (⇛-glbₗ {t' = Cond.c₂ BOOL t t'}
      (BOOL ≤? BOOL) (reflexive _≡_.refl))
      (x∧y≤x t t')
  l (r all-cond) {t} {t'} = trans
    (⇛-glbₗ {t' = Cond.c₂ BOOL t t'}
      (BOOL ≤? BOOL) (reflexive _≡_.refl))
      (x∧y≤y t t')
  l (r (r all-cond)) {t} {t'} =
    ⇛-glbᵣ {t = Cond.c₁ INT t t'}
      (INT ≤? INT) (reflexive _≡_.refl)
  r (r (r all-cond)) {t} {t'} =
    ⇛-glbᵣ {t = Cond.c₁ ERROR t t'}
      (ERROR ≤? INT) ≤-bot


{- Here we trivially show monotonicity of add' and cond', using the
   fact that any operators defined with certain combinations of the
   ⇛ and ⊓ operations are monotone. -}
module Monotone where
  add'-mono : MonoT₂ add'
  add'-mono = mono₂-⊓
    (mono₂-⇛
      (mono₂-∧? (mono₂-≤?-constₗ INT) (mono₂-≤?-constᵣ INT))
      (mono₂-const INT))
    (mono₂-⇛
      (mono₂-∨? (mono₂-≤?-constₗ BOOL) (mono₂-≤?-constᵣ BOOL))
      (mono₂-const ERROR))

  cond'-mono : MonoT₃ cond'
  cond'-mono = mono₃-⊓
    (mono₃-⇛
      (mono₃-≤?-const₁ BOOL)
      (mono₃-⊓ mono₃-u mono₃-v))
    (mono₃-⇛
      (mono₃-≤?-const₁ INT) (mono₃-const ERROR))


{- Here we prove the homomorphism properties of add' and cond', in the
   same calculational style as is shown in the paper. We show that the
   constraints derived above are sufficient to prove these properties,
   and then later apply the fact that these constraints are satisfied
   by the actual (calculated) definitions.

   As much as Agda permits, these calculations take the exact same
   route as the corresponding calculations in Section 7. -}
module Homo where
  open import Reasoning.Syntax
  open import Reasoning.Implication
  open use-⇐ (⇐-preorder)

  homo-add-non-INT : ∀ x y
    → (tval x ≤ BOOL ∨ tval y ≤ BOOL)
    → Constraints.AllAdd
    → tval (add x y) ≥ add' (tval x) (tval y)

  homo-add : ∀ x y
    → Constraints.AllAdd
    → tval (add x y) ≥ add' (tval x) (tval y)

  homo-add (I n) (I m) = begin
    (tval (add (I n) (I m)) ≥ add' (tval (I n)) (tval (I m)))
      ⇔⟨⟩
    (tval (I (n + m)) ≥ add' INT INT)
      ⇔⟨⟩
    Constraints.add₁ ⇐⟨ l ⟩ Constraints.AllAdd ∎
  
  {- It's not super easy in Agda to write a pattern like we do in the
     paper: "x ≢ I n or y ≢ I m", so instead, we define a lemma which
     proves this case when "tval x ≢ INT or tval y ≢ INT" and then
     apply this in each case manually. The lemma is proved below. -}
  homo-add (B b) y = homo-add-non-INT (B b) y (inj₁ ≤-refl)
  homo-add Error y = homo-add-non-INT Error y (inj₁ ≤-bot)
  homo-add (I n) (B x) = homo-add-non-INT (I n) (B x) (inj₂ ≤-refl)
  homo-add (I n) Error = homo-add-non-INT (I n) Error (inj₂ ≤-bot)

  {- homo-add x y
     where x = B b or x = Error or (x = I n and (y = B x or y = Error)) -}
  homo-add-non-INT x y p = begin
    (tval (add x y) ≥ add' (tval x) (tval y))
      ⇔⟨ ⇔-cong (λ t → t ≥ add' (tval x) (tval y))
                (cong tval (add-bool x y p)) ⟩
    (tval Error ≥ add' (tval x) (tval y))
      ⇔⟨⟩
    (ERROR ≥ add' (tval x) (tval y))
      ⇐⟨ (λ k → ⇛-glbᵣ (tval x ≤? BOOL ∨? tval y ≤? BOOL) p) ⟩
    Constraints.add₂ ⇐⟨ r ⟩ Constraints.AllAdd ∎


  homo-cond : ∀ x y z
    → Constraints.AllCond
    → tval (cond x y z) ≥ cond' (tval x) (tval y) (tval z)
  homo-cond (B true) y z = begin
    (tval (cond (B true) y z) ≥ cond' (tval (B true)) (tval y) (tval z))
      ⇔⟨⟩
    (tval y ≥ cond' BOOL (tval y) (tval z))
      ⇐⟨ (λ k → k {tval y} {tval z}) ⟩
    Constraints.cond₁ ⇐⟨ l ⟩ Constraints.AllCond ∎
    
  homo-cond (B false) y z = begin
    (tval (cond (B false) y z) ≥ cond' (tval (B false)) (tval y) (tval z))
      ⇔⟨⟩
    (tval z ≥ cond' BOOL (tval y) (tval z))
      ⇐⟨ (λ k → k {tval y} {tval z}) ⟩
    Constraints.cond₂ ⇐⟨ l ∘ r ⟩ Constraints.AllCond ∎
    
  homo-cond (I n) y z = begin
    (tval (cond (I n) y z) ≥ cond' (tval (I n)) (tval y) (tval z))
      ⇔⟨⟩
    (tval Error ≥ cond' INT (tval y) (tval z))
      ⇔⟨⟩
    (ERROR ≥ cond' INT (tval y) (tval z))
      ⇐⟨ (λ k → k {tval y} {tval z}) ⟩
    Constraints.cond₃ ⇐⟨ l ∘ r ∘ r ⟩ Constraints.AllCond ∎
  
  homo-cond Error y z = begin
    (tval (cond Error y z) ≥ cond' (tval Error) (tval y) (tval z))
      ⇔⟨⟩
    (tval Error ≥ cond' ERROR (tval y) (tval z))
      ⇔⟨⟩
    (ERROR ≥ cond' ERROR (tval y) (tval z))
      ⇐⟨ (λ k → k {tval y} {tval z}) ⟩
    Constraints.cond₄ ⇐⟨ r ∘ r ∘ r ⟩ Constraints.AllCond ∎


{- Finally, we can bring together the proofs of the homomorphism
   conditions (Homo.homo-add and Homo.homo-cond) with proofs that
   these conditions are in fact implied by our calculated definitions
   (Constraints.all-add and Constraints.all-cond) in order to apply
   the fold-fusion law and show that texp is a correct type-checker. -}
correctness : Correct eval texp
correctness x = fusion tval
  id add cond
  tval add' cond'
  Monotone.add'-mono Monotone.cond'-mono
  --------------------------------------
  (λ {x} → ≤-refl)
  (λ {x y} → Homo.homo-add x y Constraints.all-add)
  (λ {x y z} → Homo.homo-cond x y z Constraints.all-cond)
  --------------------------------------
  x
