################################################################################
# This script provides the function to calculate activities of gene signatures.
#
# Usage:
#        It should be imported by other scripts.
################################################################################
pacman::p_load("gplots")

cal_HWactivity <- function(data, weight, model_size, sd_cutoff = 2.5,
                           activity_plot = FALSE, activity_plot_fd = NULL) {
  # This function calculates the activity of each gene signature in the model
  # specified by the weight matrix for each sample in the data compendium.

  # Input:
  # data: a data frame of the input gene expression compendium
  # weight: a data frame of a model's weight matrix
  # model_size: int, number of nodes in the model
  # sd_cutoff: number of standard deviations from mean to be considered as
  #            high-weight, default to 2.5
  # activity_plot: logical, plot the distribution of a signature's activity
  #                across samples in the input data if set to TRUE
  # activity_plot_fd: the folder to save activity plots

  # Return:
  # HWactivity_perGene: a data frame of activities per gene signature per
  #                     sample


  HWactivity_perGene_pos <- matrix(, nrow = ncol(data), ncol = model_size)
  HWactivity_perGene_neg <- matrix(, nrow = ncol(data), ncol = model_size)

  for (node in 1:model_size) {

    # positive side
    pos_cutoff <- mean(weight[, node]) + sd_cutoff * sd(weight[, node])
    this_node_HWG_index <- weight[, node] >= pos_cutoff
    HWactivity_perGene_pos[, node] <- t(t(weight[this_node_HWG_index, node]) %*%
                                          as.matrix(data[this_node_HWG_index, ]))
    if (sum(this_node_HWG_index) > 0) {
      HWactivity_perGene_pos[, node] <- HWactivity_perGene_pos[, node] /
        sum(this_node_HWG_index)
    }

    # negative side
    neg_cutoff <- mean(weight[, node]) - sd_cutoff * sd(weight[, node])
    this_node_HWG_index <- weight[, node] <= neg_cutoff
    HWactivity_perGene_neg[, node] <- t(t(weight[this_node_HWG_index, node]) %*%
                                          as.matrix(data[this_node_HWG_index, ]))
    if (sum(this_node_HWG_index) > 0) {
      HWactivity_perGene_neg[, node] <- HWactivity_perGene_neg[, node] /
        sum(this_node_HWG_index)
    }

    # plot the distribution of activities
    if (activity_plot) {

      pdf(file.path(activity_plot_fd, paste0("Node", node, "pos.pdf")),
          height = 5, width = 5)
      plot(density(HWactivity_perGene_pos[, node]),
           main = paste("Node", node, "pos", sep = ""))
      dev.off()

      pdf(file.path(activity_plot_fd, paste0("Node", node, "neg.pdf")),
          height = 5, width = 5)
      plot(density(HWactivity_perGene_neg[, node]),
           main = paste("Node", node, "neg", sep = ""))
      dev.off()
    }

  }

  # combine positive and negative sides
  HWactivity_perGene <- cbind(HWactivity_perGene_pos, HWactivity_perGene_neg)
  rownames(HWactivity_perGene) <- colnames(data)
  colnames(HWactivity_perGene) <- c(paste0("Node", seq(1, model_size), "pos"),
                                    paste0("Node", seq(1, model_size), "neg"))

  return(HWactivity_perGene)

}

plot.HWactivity.per.exp <- function(HWactivity.exp, dataset, outputFile){
  # This function makes a heatmap of signature activities for one experiment
  # across all signatures.
  #
  # Inputs:
  # HWactivity.exp: a data matrix that stores signature activities for each
  #                 sample in the experiment
  # dataset: character, the name of the experiment
  # outputFile: file path to save the output heatmap

  # get the minimun value of each column
  min_val <- apply(HWactivity.exp, 2, min)
  # transform the values by substracting minimun of each column
  HWactivity.exp_scaled <- scale(HWactivity.exp, center = min_val, scale = F)

  width <- 3 + nrow(HWactivity.exp)/6
  height <- ncol(HWactivity.exp)/10 + 4
  # activity heatmap color panel
  activity.color <- colorpanel(255, rgb(0/255, 176/255, 240/255),
                               rgb(230/255, 230/255, 230/255),
                               rgb(255/255, 255/255, 0/255))
  # plot activity heatmap
  pdf(outputFile, width = width, height = height)
  heatmap.2(t(HWactivity.exp_scaled), Rowv = T, Colv = F, trace = "none",
            margins = c(10, 4), main = dataset, cexRow = 0.6, cexCol = 0.4,
            lhei = c(0.1, 1), col = activity.color)
  dev.off()
}

plot.HWactivity.per.signature <- function(HWactivity.sig, signature_name,
                                          key_range, outputFile){
  # This function makes a heatmap of signature activities for one signature
  # across samples.
  #
  # Inputs:
  # HWactivity.sig: a named vector that stores each sample's activity for one
  #                 signature
  # signature_name: character, the name of the signature
  # key_range: a vector that specify color breaks of the heatmap
  # outputFile: file path to save the output heatmap

  # skip if all values are 0 or if all values are equal
  if (!all(HWactivity.sig == 0) & !(length(unique(HWactivity.sig)) == 1)) {
    # to draw a heatmap, it needs at least two column.  so two same columns are
    # combined.
    combined <- cbind(HWactivity.sig, HWactivity.sig)
    # heatmap color key is determined by activity range

    activity.color <- colorpanel(length(key_range) - 1,
                                 rgb(0/255, 176/255, 240/255),
                                 rgb(230/255, 230/255, 230/255),
                                 rgb(255/255, 255/255, 0/255))
    height <- 2 + length(HWactivity.sig)/10
    pdf(outputFile, width = 5, height = height)
    par(cex.main = 0.75)
    heatmap.2(as.matrix(combined), trace = "none", symbreaks = F, symkey = F,
              Colv = F, Rowv = F, col = activity.color, labCol = c("", ""),
              cexCol = 0.1, main = signature_name, cexRow = 0.3,
              keysize = 2, lhei = c(1, 1), margin = c(1, 10),
              density.info = "none", breaks = key_range)
    dev.off()
  }
}
