clc; clear;

% --- System Parameters ---
SNR_dB = -20:0.2:20;                % SNR range in dB
SNR = 10.^(SNR_dB/10);              % Linear scale

BW = 125e3;                         % Bandwidth in Hz (standard LoRa)
codingRate = 4/5;                   % CR = 4/5 (standard LoRa coding rate)
nPreamble = 8;                      % Number of preamble symbols
explicitHeader = true;              % Using explicit header
CRC = true;                         % Using CRC
payload_bytes = 20;                 % Payload size in bytes
packet_len_bytes = payload_bytes;   % Without overhead for simplicity

% --- LoRa SF-specific Parameters ---
SF_values = [7, 8, 9, 10, 11, 12];  % Include SF12
SF_labels = {'SF7','SF8','SF9','SF10','SF11', 'SF12'};
num_SFs = length(SF_values);
colors = lines(num_SFs);            % Color palette

% --- LoRa Performance Parameters (based on literature) ---
% Realistic Rayleigh fading performance requires ~6-8dB more SNR vs AWGN
% SNR values where PER = 0.5 (midpoint of waterfall curve)
% These values are based on empirical data and Semtech documentation
sf_thresholds = [7.5, 4.5, 1.5, -1.5, -4, -7];  % Midpoints for PER=0.5 under Rayleigh
sf_slopes = [1.2, 1.1, 1.0, 0.9, 0.8, 0.7];     % Different slopes for each SF

% --- Preallocate matrices ---
N = length(SNR_dB);
ToA = zeros(1, num_SFs);                % Time on Air
Rb = zeros(1, num_SFs);                 % Raw bit rate
throughput = zeros(num_SFs, N);         % Effective throughput
PER = zeros(num_SFs, N);                % Packet Error Rate
spectral_efficiency = zeros(num_SFs, N); % Spectral efficiency

% --- More realistic Rayleigh fading PER model ---
% Using logistic function with SF-specific parameters
for i = 1:num_SFs
    SF = SF_values(i);
    a = sf_slopes(i);  % Slope parameter - higher SFs have gentler slopes
    
    % Calculate Time on Air - Using realistic LoRa airtime formula
    % Number of symbols in payload
    payloadSymNb = 8 + max(ceil((8*packet_len_bytes - 4*SF + 28 + 16*CRC - 20*~explicitHeader)/(4*SF)) * codingRate, 0);
    
    % Total packet duration
    Tsym = 2^SF / BW;  % Symbol period
    Tpreamble = (nPreamble + 4.25) * Tsym;
    Tpayload = payloadSymNb * Tsym;
    ToA(i) = Tpreamble + Tpayload;  % seconds
    
    % Raw bit rate (bps)
    Rb(i) = SF * (BW / 2^SF) * codingRate;
    
    % PER calculation using modified logistic function for Rayleigh
    % Each SF has different reliability characteristics
    PER(i, :) = 1 ./ (1 + exp(a * (SNR_dB - sf_thresholds(i))));
    
    % Calculate effective throughput - bits per second considering PER
    throughput(i, :) = (1 - PER(i, :)) * (payload_bytes * 8) ./ ToA(i);
    
    % Spectral efficiency (bits/s/Hz)
    spectral_efficiency(i, :) = throughput(i, :) / BW;
end

% --- ACM Throughput Envelope ---
% At each SNR point, ACM chooses the SF with highest throughput
acm_throughput = max(throughput, [], 1);

% --- Throughput Gain vs Fixed SF7 ---
% Make sure to handle regions where SF7 doesn't work
sf7_throughput = throughput(1, :);
min_valid_throughput = 1e-6;  % Define a minimum value to avoid division by zero
sf7_throughput_safe = max(sf7_throughput, min_valid_throughput);
gain_acm = acm_throughput ./ sf7_throughput_safe;

% Cap the maximum gain to a realistic value to avoid infinity spikes
max_realistic_gain = 1000;  % Maximum 1000x gain 
gain_acm = min(gain_acm, max_realistic_gain);

% --- FIXED Energy Efficiency Analysis ---
% Typical LoRa power consumption (mW)
tx_power_mW = 25;  % Transmit power consumption (25mW at 14dBm)

% Energy consumption per bit (µJ/bit) - CORRECTED CALCULATION
energy_per_bit = zeros(num_SFs, N);
for i = 1:num_SFs
    % Prevent divide-by-zero and unrealistic values
    PER_safe = max(PER(i, :), 1e-8);  % Minimum PER to prevent division by zero
    success_rate = 1 - PER_safe;
    
    % Energy per successful bit transmission including retransmissions
    % Expected number of transmissions = 1 / success_rate
    expected_transmissions = 1 ./ success_rate;
    
    % Energy per bit (µJ/bit) = Power (mW) × Time (s) × Expected transmissions / Bits
    energy_per_bit(i, :) = (tx_power_mW * 1000) * ToA(i) .* expected_transmissions ./ (payload_bytes * 8);
    
    % Cap unrealistic values (above 10 J/bit)
    energy_per_bit(i, :) = min(energy_per_bit(i, :), 1e7);  % Max 10 J/bit = 10^7 µJ/bit
end

% ACM energy efficiency - choose SF with minimum energy per bit at each SNR
acm_energy = zeros(1, N);
acm_energy_sf = zeros(1, N);  % Track which SF is selected for energy efficiency
for j = 1:N
    % Find SF with minimum energy consumption that has reasonable PER
    PER_threshold = 0.9;  % Allow higher PER for energy optimization
    working_SFs = find(PER(:, j) < PER_threshold);
    
    if ~isempty(working_SFs)
        [acm_energy(j), best_idx] = min(energy_per_bit(working_SFs, j));
        acm_energy_sf(j) = SF_values(working_SFs(best_idx));
    else
        % If all SFs have very high PER, choose the most energy-efficient
        [acm_energy(j), best_idx] = min(energy_per_bit(:, j));
        acm_energy_sf(j) = SF_values(best_idx);
    end
end

% --- SF Selection by ACM (for throughput) ---
acm_sf_selection = zeros(1, N);
for j = 1:N
    [~, idx] = max(throughput(:, j));
    acm_sf_selection(j) = SF_values(idx);
end

% --- Optimal SNR Operating Ranges ---
sf_optimal_ranges = cell(num_SFs, 1);
range_start = 1;

for j = 2:N
    if acm_sf_selection(j) ~= acm_sf_selection(j-1)
        % SF transition point
        sf_idx = find(SF_values == acm_sf_selection(j-1));
        if ~isempty(sf_idx)
            range_end = j-1;
            sf_optimal_ranges{sf_idx} = [sf_optimal_ranges{sf_idx}; SNR_dB(range_start), SNR_dB(range_end)];
            range_start = j;
        end
    end
end

% Add the last range
sf_idx = find(SF_values == acm_sf_selection(end));
sf_optimal_ranges{sf_idx} = [sf_optimal_ranges{sf_idx}; SNR_dB(range_start), SNR_dB(end)];

% =======================
% === Begin Plotting ===
% =======================

% --- Plot 1: PER vs. SNR ---
figure;
hold on;
for i = 1:num_SFs
    plot(SNR_dB, PER(i, :), 'LineWidth', 2, 'Color', colors(i,:), 'DisplayName', SF_labels{i});
end
title('PER vs. SNR under Rayleigh Fading');
xlabel('SNR (dB)'); ylabel('Packet Error Rate');
ylim([0 1]); xlim([min(SNR_dB) max(SNR_dB)]);
grid on; legend('show', 'Location', 'northeast');
set(gca, 'FontSize', 12);

% --- Plot 2: Throughput vs. SNR (Fixed SFs) ---
figure;
hold on;
for i = 1:num_SFs
    plot(SNR_dB, throughput(i, :)/1000, 'LineWidth', 2, 'Color', colors(i,:), ...
        'DisplayName', sprintf('%s (ToA = %.0f ms)', SF_labels{i}, ToA(i)*1000));
end
title('Throughput vs. SNR for Fixed LoRa Configurations');
xlabel('SNR (dB)'); ylabel('Throughput (kbps)');
grid on; legend('show', 'Location', 'northwest');
ylim([0 max(acm_throughput/1000)*1.1]);
set(gca, 'FontSize', 12);

% --- Plot 3: ACM Throughput Envelope vs Fixed SFs ---
figure;
hold on;
for i = 1:num_SFs
    plot(SNR_dB, throughput(i, :)/1000, '--', 'LineWidth', 1, 'Color', colors(i,:), ...
        'DisplayName', sprintf('%s', SF_labels{i}));
end
plot(SNR_dB, acm_throughput/1000, 'k-', 'LineWidth', 2.5, 'DisplayName', 'ACM Envelope');

% Add annotations for optimal SF regions
y_pos = max(acm_throughput/1000)*0.85;
for i = 1:num_SFs
    ranges = sf_optimal_ranges{i};
    if ~isempty(ranges)
        for r = 1:size(ranges, 1)
            x_start = ranges(r, 1);
            x_end = ranges(r, 2);
            x_mid = (x_start + x_end) / 2;
            text(x_mid, y_pos, SF_labels{i}, 'HorizontalAlignment', 'center', ...
                'BackgroundColor', [0.9 0.9 0.9], 'FontWeight', 'bold');
        end
    end
end

title('ACM Throughput Envelope with Optimal SF Regions');
xlabel('SNR (dB)'); ylabel('Throughput (kbps)');
grid on; legend('show', 'Location', 'northwest');
ylim([0 max(acm_throughput/1000)*1.1]);
set(gca, 'FontSize', 12);

% --- Plot 4: Realistic Throughput Gain of ACM vs Fixed SF7 ---
figure;
plot(SNR_dB, gain_acm, 'g', 'LineWidth', 2);
title('Throughput Gain of ACM vs. Fixed SF7');
xlabel('SNR (dB)'); ylabel('Throughput Gain (×)');
grid on;
set(gca, 'YScale', 'log');  % Log scale for better visualization
ylim([0.1 max_realistic_gain]);
set(gca, 'FontSize', 12);

% --- Plot 5: FIXED Energy Efficiency Comparison ---
figure;
hold on;
for i = 1:num_SFs
    plot(SNR_dB, energy_per_bit(i, :), '--', 'LineWidth', 1, 'Color', colors(i,:), ...
        'DisplayName', SF_labels{i});
end
plot(SNR_dB, acm_energy, 'k-', 'LineWidth', 2.5, 'DisplayName', 'ACM Energy Optimal');
title('Energy Efficiency Comparison');
xlabel('SNR (dB)'); ylabel('Energy per Bit (µJ/bit)');
grid on; legend('show', 'Location', 'northeast');
set(gca, 'YScale', 'log');  % Log scale for better visualization
ylim([1 max(max(energy_per_bit))*2]);
set(gca, 'FontSize', 12);

% --- Plot 6: Spectral Efficiency Comparison ---
figure;
hold on;
for i = 1:num_SFs
    plot(SNR_dB, spectral_efficiency(i, :), '--', 'LineWidth', 1, 'Color', colors(i,:), ...
        'DisplayName', SF_labels{i});
end
plot(SNR_dB, acm_throughput/BW, 'k-', 'LineWidth', 2.5, 'DisplayName', 'ACM');
title('Spectral Efficiency Comparison');
xlabel('SNR (dB)'); ylabel('Spectral Efficiency (bits/s/Hz)');
grid on; legend('show', 'Location', 'northwest');
set(gca, 'FontSize', 12);

% --- Table with Key Metrics ---
fprintf('=== LoRa Performance Metrics ===\n');
fprintf('SF\tToA (ms)\tRaw Rate (bps)\tMin SNR (dB)*\n');
fprintf('--------------------------------------------------\n');
for i = 1:num_SFs
    % Find minimum SNR for PER < 10%
    min_SNR_idx = find(PER(i, :) < 0.1, 1, 'first');
    if isempty(min_SNR_idx)
        min_SNR = 'N/A';
    else
        min_SNR = sprintf('%.1f', SNR_dB(min_SNR_idx));
    end
    
    fprintf('%s\t%.1f\t\t%.1f\t\t%s\n', ...
        SF_labels{i}, ToA(i)*1000, Rb(i), min_SNR);
end
fprintf('--------------------------------------------------\n');
fprintf('* Minimum SNR for PER < 10%% under Rayleigh fading\n\n');

% --- Print ACM Gain Statistics ---
fprintf('=== ACM Performance Gain Statistics ===\n');
% Calculate gain statistics at different SNR ranges
low_SNR = SNR_dB < -5;
med_SNR = SNR_dB >= -5 & SNR_dB < 5;
high_SNR = SNR_dB >= 5;

fprintf('SNR Range\tAvg Gain\tMax Gain\n');
fprintf('--------------------------------------\n');
fprintf('Low (<-5 dB)\t%.1fx\t\t%.1fx\n', ...
    mean(gain_acm(low_SNR)), max(gain_acm(low_SNR)));
fprintf('Med (-5 to 5 dB)\t%.1fx\t\t%.1fx\n', ...
    mean(gain_acm(med_SNR)), max(gain_acm(med_SNR)));
fprintf('High (>5 dB)\t%.1fx\t\t%.1fx\n', ...
    mean(gain_acm(high_SNR)), max(gain_acm(high_SNR)));
fprintf('Overall\t\t%.1fx\t\t%.1fx\n', ...
    mean(gain_acm), max(gain_acm));

% ========================
% === Export to CSV ======
% ========================
T_data = array2table(SNR_dB.', 'VariableNames', {'SNR_dB'});

% Throughput columns
for i = 1:num_SFs
    T_data.([SF_labels{i} '_Throughput_bps']) = throughput(i, :).';
    T_data.([SF_labels{i} '_PER']) = PER(i, :).';
    T_data.([SF_labels{i} '_Energy_uJ_per_bit']) = energy_per_bit(i, :).';
end

% ACM results
T_data.ACM_Throughput_bps = acm_throughput.';
T_data.ACM_Energy_uJ_per_bit = acm_energy.';
T_data.Gain_vs_Fixed_SF7 = gain_acm.';
T_data.ACM_Selected_SF = acm_sf_selection.';
T_data.ACM_Energy_Selected_SF = acm_energy_sf.';

% Write CSV
writetable(T_data, 'LoRa_ACM_Realistic_Performance.csv');