import argparse
import os
import pathlib

from common.generator import HotpatchGenerator
from common.binaryfile import BinaryFile

def directory_path(path : str) -> str:
    if os.path.isdir(path):
        return path
    else:
        raise argparse.ArgumentTypeError(f"argument value '{path}' is not a valid path")

def get_files(path : str, filename : str, suffixes : list) -> str:
    object_paths = []
    for suffix in suffixes:
        object_paths += [ file for file in pathlib.Path(path).rglob(suffix) ]

    for object_path in object_paths:
        if filename in object_path.stem:
            return str(object_path)

    return None

def find_cve_project_dir(project_path, cve):
    cve_dirs = []
    for root, dirs, files in os.walk(project_path):
        for dir_name in dirs:
            if dir_name == cve:
                full_path = os.path.join(root, dir_name)
                cve_dirs.append(full_path)

    if len(cve_dirs) > 1:
        print("[error]: multiple directories for the same CVE, please adjust your path.")
        for cve_dir in cve_dirs:
            print("\t-", cve_dir)
        return None
    

    if len(cve_dirs) == 0:
        print(f"[error]: no directory for cve '{cve}' found.")
        return None
    
    return cve_dirs[0]

def main():
    parser = argparse.ArgumentParser("Kintsugi Hotpatch Generator")
    parser.add_argument("--project_path", required=True, type=directory_path, help="Path to the directory containing the CVE projects")
    parser.add_argument("--binary", required=True, type=str, help="Name of the ELF binary")
    parser.add_argument("--cve", required=True, type=str, help="CVE of the project")
    parser.add_argument("--out-dir", default="include", type=str, help="Output path for the header file containing the hotpatch data")
    args = parser.parse_args()


    cve_path = find_cve_project_dir(args.project_path, args.cve)
    if cve_path is None:
        print("[error]: could not get CVE path.")
        return
    
    print(f"[info]: found cve path: {cve_path}")

    hotpatch_object_file_path = get_files(cve_path, args.cve, [ "*.o", "*.obj" ])
    elf_file_path = get_files(cve_path, args.binary, [ "*.elf", "*.out" ])

    print(f"[info]: loading hotpatch object file: {hotpatch_object_file_path}")
    hotpatch_object_file = BinaryFile()
    if not hotpatch_object_file.load(hotpatch_object_file_path):
        print(f"[error]: could not load hotpatch object file: '{hotpatch_object_file_path}'")
        return
    

    print(f"[info]: loading elf binary file: {elf_file_path}")
    binary_file = BinaryFile()
    if not binary_file.load(elf_file_path):
        print(f"[error]: could not load elf binary file: '{elf_file_path}'")
        return

    print("[info]: generating hotpatch...")
    generator = HotpatchGenerator(hotpatch_object_file, binary_file)
    hotpatch_data = generator.generate_hotpatch()

    if hotpatch_data is None:
        print(f"[error]: could not generate hotpatch for '{args.cve}'")
        return
    
    print(f"[info]: successfully generated hotpatch of {len(hotpatch_data)} bytes for {args.cve}!")

    # hotpatch array string
    hotpatch_array_lines = []
    curr_data = []
    for i, data in enumerate(hotpatch_data):
        if i > 0 and (i % 8) == 0:
            hotpatch_array_lines.append("\t" + ", ".join([ f"0x{b:02X}" for b in curr_data]))
            curr_data = []
        curr_data.append(data)
    
    if len(hotpatch_array_lines) > 0:
        hotpatch_array_lines.append("\t" + ", ".join([ f"0x{b:02X}" for b in curr_data]))

    hotpatch_str = ", \n".join(hotpatch_array_lines)

    # store the hotpatch as an include that can be used
    hotpatch_include_str =  f"/* AUTOGENERATED FILE BY KINTSUGI */\n" \
                            f"#ifndef HP_HOTPATCH_CODE_H_\n" \
                            f"#define HP_HOTPATCH_CODE_H_\n" \
                            f"\n" \
                            f"#include <stdint.h>\n" \
                            f"\n" \
                            f"static const uint8_t HOTPATCH_CODE[{len(hotpatch_data)}] = " + "{\n" \
                            f"{hotpatch_str}\n"\
                            f"" + "};\n" \
                            f"\n" \
                            f"#endif /* HP_HOTPATCH_CODE_H_ */\n"
    
    header_path = f"{cve_path}/{args.out_dir}/hp_code_data.h"
    with open(header_path, "w") as f:
        f.write(hotpatch_include_str)
    print(f"[info]: stored hotpatch in header file '{header_path}'")

if __name__ == "__main__":
    main()