# Quaternion

```@meta
CurrentModule = ReferenceFrameRotations
```

```@setup quaternions
using ReferenceFrameRotations
```

Quaternions are hypercomplex number with 4 dimensions that can be used to represent 3D
rotations. In this package, a quaternion ``\mathbf{q}`` is represented by

```math
\mathbf{q} =
  q_0 + q_1 \cdot \mathbf{i} + q_2 \cdot \mathbf{j} + q_3 \cdot \mathbf{k} =
  r + \mathbf{v}
```

using the following immutable structure:

```julia
struct Quaternion{T}
    q0::T
    q1::T
    q2::T
    q3::T
end
```

## Initialization

There are several ways to create a quaternion.

- Provide all the elements:

```@repl quaternions
q = Quaternion(1.0, 0.0, 0.0, 0.0)
```

- Provide the real and imaginary parts as separated numbers:

```@repl quaternions
r = sqrt(2) / 2

v = [sqrt(2) / 2, 0, 0]

q = Quaternion(r,v)
```

- Provide the real and imaginary parts as one single vector:

```@repl quaternions
v = [1., 2., 3., 4.]

q = Quaternion(v)
```

- Provide just the imaginary part, in this case the real part will be 0:

```@repl quaternions
v = [1., 0., 0.]

q = Quaternion(v)
```

- Create an identity quaternion:

```@repl quaternions
q = Quaternion{Float64}(I)  # Creates an identity quaternion of type `Float64`.

q = Quaternion(1.0I)  # Creates an identity quaternion of type `Float64`.

q = Quaternion{Float32}(I)  # Creates an identity quaternion of type `Float32`.

q = Quaternion(1.0f0I)  # Creates an identity quaternion of type `Float32`.

a = Quaternion(I,q)  # Creates an identity quaternion with the same type of `q`.

q = Quaternion(I)
```

- Create an additive identity quaternion using the `zero` function:

```@repl quaternions
q = zero(Quaternion)

q = zero(Quaternion{Float32})

a = zero(q)
```

- Create an multiplicative identity quaternion using the `one` function:

```@repl quaternions
q = one(Quaternion)

q = one(Quaternion{Float32})

a = one(q)
```

!!! note

    Individual elements of the quaternion can be accessed by:

    ```julia
    q.q0
    q.q1
    q.q2
    q.q3
    ```

    or using linear indexing:

    ```julia
    q[1]
    q[2]
    q[3]
    q[4]
    ```

    Notice that, in this case, the index 1 refers to the real part of the quaternion.

!!! warning

    Since the type `Quaternion` is **immutable**, its components cannot be changed
    individually after the creation. Hence, the following operation will lead to an error:

    ```julia
    q.q0 = 1.0  # This is not defined and will not work.
    ```

    If you want to modify a single value for the quaternion, you need to create another
    one:

    ```julia
    q = Quaternion(1.0, q.q1, q.q2, q.q3)
    ```

    This can be annoying sometimes, but using an immutable type provided a huge performance
    boost for the algorithm.

## Operations

### Sum, subtraction, and scalar multiplication

The sum between quaternions, the subtraction between quaternions, and the multiplication
between a quaternion and a scalar are defined as usual:

```math
\begin{aligned}
  \mathbf{q}_a + \mathbf{q}_b &= (q_{a,0} + q_{b,0}) +
                                 (q_{a,1} + q_{b,1}) \cdot \mathbf{i} +
                                 (q_{a,2} + q_{b,2}) \cdot \mathbf{j} +
                                 (q_{a,3} + q_{b,3}) \cdot \mathbf{k} \\
  \mathbf{q}_a - \mathbf{q}_b &= (q_{a,0} - q_{b,0}) +
                                 (q_{a,1} - q_{b,1}) \cdot \mathbf{i} +
                                 (q_{a,2} - q_{b,2}) \cdot \mathbf{j} +
                                 (q_{a,3} - q_{b,3}) \cdot \mathbf{k} \\
  \lambda \cdot \mathbf{q} &= (\lambda \cdot q_0) +
                              (\lambda \cdot q_1) \cdot \mathbf{i} +
                              (\lambda \cdot q_2) \cdot \mathbf{j} +
                              (\lambda \cdot q_3) \cdot \mathbf{k}
\end{aligned}
```

```@repl quaternions
q1 = Quaternion(1.0, 1.0, 0.0, 0.0)

q2 = Quaternion(1.0, 2.0, 3.0, 4.0)

q1 + q2

q1 - q2

q1 = Quaternion(1.0, 2.0, 3.0, 4.0)

q1 * 3

4 * q1

5q1
```

### Multiplication between quaternions

The multiplication between quaternions is defined using the Hamilton product:

```math
\begin{aligned}
  \mathbf{q}_1 &= r_1 + \mathbf{v}_1\ , \\
  \mathbf{q}_2 &= r_2 + \mathbf{v}_2\ , \\
  \mathbf{q}_1 \cdot \mathbf{q}_2 &= r_1 \cdot r_2 -
                                     \mathbf{v}_1 \cdot \mathbf{v}_2 +
                                     r_1 \cdot \mathbf{v}_2 +
                                     r_2 \cdot \mathbf{v}_1 +
                                     \mathbf{v}_1 \times \mathbf{v}_2\ .
\end{aligned}
```

```@repl quaternions
q1 = Quaternion(cosd(15), sind(15), 0.0, 0.0)

q2 = Quaternion(cosd(30), sind(30), 0.0, 0.0)

q1 * q2
```

If a quaternion ``\mathbf{q}`` is multiplied by a vector ``\mathbf{v}``, then the vector is
converted to a quaternion with real part 0, ``\mathbf{q}_v = 0 + \mathbf{v}``, and the
quaternion multiplication is performed as usual:

```math
\begin{aligned}
\mathbf{q}   &= r + \mathbf{w}\ , \\
\mathbf{q}_v &= 0 + \mathbf{v}\ , \\
\mathbf{q} \cdot \mathbf{v} \triangleq \mathbf{q} \cdot \mathbf{q}_v &=
  - \mathbf{w} \cdot \mathbf{v} + r \cdot \mathbf{v} + \mathbf{w} \times \mathbf{v}\ , \\
\mathbf{v} \cdot \mathbf{q} \triangleq \mathbf{q}_v \cdot \mathbf{q} &=
  - \mathbf{v} \cdot \mathbf{w} + r \cdot \mathbf{v} + \mathbf{v} \times \mathbf{w}\ .
\end{aligned}
```

```@repl quaternions
q1 = Quaternion(cosd(22.5), sind(22.5), 0.0, 0.0)

v = [0; 1; 0]

v * q1

q1 * v
```

### Division between quaternions

Given this definition of the product between two quaternions, we can define the
multiplicative inverse of a quaternion by:

```math
\mathbf{q}^{-1} \triangleq \frac{\bar{\mathbf{q}}}{|\mathbf{q}|^2} =
  \frac{q_0 - q_1 \cdot \mathbf{i} - q_2 \cdot \mathbf{j} - q_3 \cdot \mathbf{k}}
       {q_0^2 + q_1^2 + q_2^2 + q_3^2}
```

Notice that, in this case, one gets:

```math
\mathbf{q} \cdot \mathbf{q}^{-1} = 1
```

!!! note

    ``\bar{\mathbf{q}}``, which is the quaternion conjugate, can be computed
    using `conj(q)`.

    ``|\mathbf{q}|``, which is the quaternion norm, can be computed using
    `norm(q)`.

    The quaternion inverse can be computed using `inv(q)`.

!!! warning

    The exponentiation operator is not defined for quaternions. Hence, `q^(-1)`
    or `q^2` will throw an error.

The right division (`/`) between two quaternions ``\mathbf{q}_1`` and ``\mathbf{q}_2`` is
defined as the following Hamilton product:

```math
\mathbf{q}_1\ /\ \mathbf{q}_2 = \mathbf{q}_1 \cdot \mathbf{q}_2^{-1}\ .
```

The left division (`\`) between two quaternions ``\mathbf{q}_1`` and ``\mathbf{q}_2`` is
defined as the following Hamilton product:

```math
\mathbf{q}_1\ \backslash\ \mathbf{q}_2 = \mathbf{q}_1^{-1} \cdot \mathbf{q}_2\ .
```

```@repl quaternions
q1 = Quaternion(cosd(45 + 22.5), sind(45 + 22.5), 0.0, 0.0)

q2 = Quaternion(cosd(22.5), sind(22.5), 0.0, 0.0)

q1 / q2

q1 \ q2

q1 \ q2 * q1 / q2
```

If a division operation (right-division or left-division) is performed between a vector
``\mathbf{v}`` and a quaternion, then the vector ``\mathbf{v}`` is converted to a quaternion
real part 0, ``\mathbf{q}_v = 0 + \mathbf{v}``, and the division operation is performed as
defined earlier.

```math
\begin{aligned}
  \mathbf{v}\ /\ \mathbf{q}          &\triangleq \mathbf{q}_v      \cdot \mathbf{q}^{-1}   \\
  \mathbf{v}\ \backslash\ \mathbf{q} &\triangleq \mathbf{q}_v^{-1} \cdot \mathbf{q}        \\
  \mathbf{q}\ /\ \mathbf{v}          &\triangleq \mathbf{q}        \cdot \mathbf{q}_v^{-1} \\
  \mathbf{q}\ \backslash\ \mathbf{v} &\triangleq \mathbf{q}^{-1}   \cdot \mathbf{q}_v
\end{aligned}
```

```@repl quaternions
q1 = Quaternion(cosd(22.5), sind(22.5), 0.0, 0.0)

v  = [0; 1; 0]

q1 \ v

v \ q1
```

### Other operations

There are also the following functions available:

```@repl quaternions
q = Quaternion(1.0, 2.0, 3.0, 4.0)

conj(q)  # Returns the complex conjugate of the quaternion.

copy(q)  # Creates a copy of the quaternion.

inv(q)   # Computes the multiplicative inverse of the quaternion.

inv(q) * q

imag(q)  # Returns the vectorial / imaginary part of the quaternion.

norm(q)  # Computes the norm of the quaternion.

real(q)  # Returns the real part of the quaternion.

vect(q)  # Returns the vectorial / imaginary part of the quaternion.
```

!!! note

    The operation `a / q` is equal to `a * inv(q)` if `a` is a scalar.

### Converting reference frames using quaternions

Given the reference frames **A** and **B**, let ``\mathbf{w}`` be a unitary vector in which
a rotation about it of an angle ``\theta`` aligns the reference frame **A** with the
reference frame **B** (in this case, ``\mathbf{w}`` is aligned with the Euler Axis and
``\theta`` is the Euler angle). Construct the following quaternion:

```math
\mathbf{q}_{ba} =
  \cos\left(\frac{\theta}{2}\right) +
  \sin\left(\frac{\theta}{2}\right) \cdot \mathbf{w}\ .
```

Then, a vector ``\mathbf{v}`` represented in reference frame **A** (``\mathbf{v}_a``) can be
represented in reference frame **B** using:

```math
\mathbf{v}_b =
  \texttt{vect}\left(\mathbf{q}_{ba}^{-1} \cdot \mathbf{v}_a \cdot \mathbf{q}_{ba}\right)\ .
```

Hence:

```julia
qBA = Quaternion(cosd(22.5), sind(22.5), 0.0, 0.0)

vA  = [0, 1, 0]

vB  = vect(qBA \ vA * qBA) # Equivalent to: vect(inv(qBA)*vA*qBA)

vB
```

!!! note

    A `SArray` is returned instead of the usual `Array`. This is a static vector created by
    the package [**StaticArrays**](https://github.com/JuliaArrays/StaticArrays.jl).
    Generally, you can treat this vector as any other one. The only downside is that you
    cannot modify individual components because it is immutable.
