#!/usr/bin/env python3
"""
WMAP Fine Structure Constant (α = 1/137) Detection Analysis
Professional analysis code for MNRAS publication

Author: Michael Kevin Baines
Date: August 2025
"""

import numpy as np
import matplotlib.pyplot as plt
import healpy as hp
from scipy import stats
import json
from datetime import datetime
import warnings
warnings.filterwarnings('ignore')

# Set publication-quality plot parameters
plt.rcParams['font.size'] = 12
plt.rcParams['font.family'] = 'serif'
plt.rcParams['axes.labelsize'] = 14
plt.rcParams['axes.titlesize'] = 16
plt.rcParams['xtick.labelsize'] = 12
plt.rcParams['ytick.labelsize'] = 12
plt.rcParams['legend.fontsize'] = 11
plt.rcParams['figure.figsize'] = (10, 6)
plt.rcParams['figure.dpi'] = 100

# =============================================================================
# CONFIGURATION
# =============================================================================

# Mathematical constants
ALPHA = 1/137.035999  # Fine structure constant
PI = np.pi

# Analysis parameters
ALPHA_MULTIPOLE = 7  # Target: α × 1000 ≈ 7.30
PI_MULTIPOLE = 565   # Target: π × 180 ≈ 565.5
ANALYSIS_WINDOW = 10 # ±10 multipoles around target
LMAX = 800          # Maximum multipole for analysis

# WMAP configuration
WMAP_BANDS = {
    'K':  {'freq': 23, 'file': 'wmap_band_imap_r9_9yr_K_v5.fits'},
    'Ka': {'freq': 33, 'file': 'wmap_band_imap_r9_9yr_Ka_v5.fits'},
    'Q':  {'freq': 41, 'file': 'wmap_band_imap_r9_9yr_Q_v5.fits'},
    'V':  {'freq': 61, 'file': 'wmap_band_imap_r9_9yr_V_v5.fits'},
    'W':  {'freq': 94, 'file': 'wmap_band_imap_r9_9yr_W_v5.fits'}
}

# =============================================================================
# ANALYSIS FUNCTIONS
# =============================================================================

def load_wmap_data():
    """Load WMAP data from FITS files"""
    print("=" * 70)
    print("LOADING WMAP 9-YEAR DATA")
    print("=" * 70)
    
    loaded_data = {}
    
    for band, info in WMAP_BANDS.items():
        try:
            print(f"Loading {band}-band ({info['freq']} GHz)...", end=" ")
            data = hp.read_map(info['file'], verbose=False)
            loaded_data[band] = {
                'map': data,
                'freq': info['freq'],
                'nside': hp.get_nside(data),
                'npix': len(data)
            }
            print(f"✓ Success (NSIDE={loaded_data[band]['nside']})")
        except Exception as e:
            print(f"✗ Failed: {str(e)}")
    
    print(f"\nSuccessfully loaded {len(loaded_data)}/5 WMAP bands")
    return loaded_data

def compute_power_spectrum(map_data, lmax=LMAX):
    """Compute angular power spectrum from map"""
    cl = hp.anafast(map_data, lmax=lmax)
    ell = np.arange(len(cl))
    
    # Convert to D_ℓ = ℓ(ℓ+1)C_ℓ/(2π) in μK²
    dl = ell * (ell + 1) * cl / (2 * np.pi) * 1e12  # Convert to μK²
    
    return ell, cl, dl

def analyze_multipole_signal(dl, ell, target_multipole, window=ANALYSIS_WINDOW):
    """Analyze signal at specific multipole with background estimation"""
    
    # Extract signal region
    signal_range = np.where((ell >= target_multipole - window) & 
                           (ell <= target_multipole + window))[0]
    signal_values = dl[signal_range]
    signal_mean = np.mean(signal_values)
    
    # Extract background regions (avoiding signal)
    bg_left = np.where((ell >= target_multipole - 50) & 
                      (ell < target_multipole - window - 5))[0]
    bg_right = np.where((ell > target_multipole + window + 5) & 
                       (ell <= target_multipole + 50))[0]
    
    # Combine background regions
    bg_indices = np.concatenate([bg_left, bg_right])
    if len(bg_indices) > 0:
        bg_values = dl[bg_indices]
        bg_mean = np.mean(bg_values)
        bg_std = np.std(bg_values)
    else:
        # Fallback if standard regions unavailable
        bg_values = dl[target_multipole + window + 5:target_multipole + 100]
        bg_mean = np.mean(bg_values)
        bg_std = np.std(bg_values)
    
    # Calculate statistics
    enhancement = (signal_mean - bg_mean) / bg_mean * 100 if bg_mean != 0 else 0
    significance = (signal_mean - bg_mean) / bg_std if bg_std != 0 else 0
    
    # Peak value at exact multipole
    peak_value = dl[target_multipole] if target_multipole < len(dl) else 0
    
    return {
        'signal_mean': signal_mean,
        'background_mean': bg_mean,
        'background_std': bg_std,
        'enhancement_percent': enhancement,
        'significance_sigma': significance,
        'peak_value': peak_value,
        'signal_values': signal_values,
        'background_values': bg_values
    }

def analyze_alpha_detection(wmap_data):
    """Main analysis: detect α = 1/137 at multipole ℓ = 7"""
    print("\n" + "=" * 70)
    print("ANALYZING α = 1/137 DETECTION AT ℓ = 7")
    print("=" * 70)
    print(f"Target value: α × 1000 = {ALPHA * 1000:.3f}")
    print(f"Analysis window: ℓ = {ALPHA_MULTIPOLE} ± {ANALYSIS_WINDOW}")
    
    results = {}
    
    for band, data in wmap_data.items():
        print(f"\n{band}-band ({data['freq']} GHz):")
        print("-" * 40)
        
        # Compute power spectrum
        ell, cl, dl = compute_power_spectrum(data['map'])
        
        # Analyze α signal
        analysis = analyze_multipole_signal(dl, ell, ALPHA_MULTIPOLE)
        
        # Store results
        results[band] = {
            'frequency': data['freq'],
            'analysis': analysis,
            'power_spectrum': {'ell': ell, 'cl': cl, 'dl': dl}
        }
        
        # Print results
        print(f"Signal (ℓ={ALPHA_MULTIPOLE}): {analysis['signal_mean']:.1f} μK²")
        print(f"Background: {analysis['background_mean']:.1f} ± {analysis['background_std']:.1f} μK²")
        print(f"Enhancement: {analysis['enhancement_percent']:+.1f}%")
        print(f"Significance: {analysis['significance_sigma']:.2f}σ")
        
        # Detection assessment
        if abs(analysis['significance_sigma']) > 5.0:
            print(">>> DISCOVERY-LEVEL DETECTION! <<<")
        elif abs(analysis['significance_sigma']) > 3.0:
            print(">>> EVIDENCE-LEVEL DETECTION <<<")
    
    return results

def analyze_pi_crosscheck(wmap_data):
    """Cross-validation: analyze π at multipole ℓ = 565"""
    print("\n" + "=" * 70)
    print("CROSS-VALIDATION: π DETECTION AT ℓ = 565")
    print("=" * 70)
    print(f"Target value: π × 180 = {PI * 180:.3f}")
    
    results = {}
    
    for band, data in wmap_data.items():
        # Compute power spectrum
        ell, cl, dl = compute_power_spectrum(data['map'])
        
        # Analyze π signal
        analysis = analyze_multipole_signal(dl, ell, PI_MULTIPOLE)
        
        # Store results
        results[band] = {
            'frequency': data['freq'],
            'analysis': analysis
        }
        
        print(f"{band}-band: {analysis['significance_sigma']:+.2f}σ", end="  ")
    
    print("\n")
    return results

def monte_carlo_validation(wmap_data, n_simulations=100):
    """Monte Carlo validation for false positive rate"""
    print("\n" + "=" * 70)
    print("MONTE CARLO VALIDATION")
    print("=" * 70)
    print(f"Running {n_simulations} simulations...")
    
    # Get typical map properties from first band
    first_band = list(wmap_data.values())[0]
    nside = first_band['nside']
    
    # Run simulations
    max_significances = []
    
    for i in range(n_simulations):
        if (i + 1) % 20 == 0:
            print(f"Progress: {i+1}/{n_simulations}")
        
        # Generate random Gaussian map
        np.random.seed(42 + i)  # Reproducible randomness
        random_map = hp.synfast(np.ones(LMAX+1), nside, verbose=False)
        
        # Test random multipole (avoiding very low ℓ)
        test_multipole = np.random.randint(10, 100)
        
        # Compute power spectrum
        ell, cl, dl = compute_power_spectrum(random_map)
        
        # Analyze signal
        analysis = analyze_multipole_signal(dl, ell, test_multipole)
        max_significances.append(abs(analysis['significance_sigma']))
    
    # Calculate statistics
    max_sig = np.max(max_significances)
    false_positives_3sig = np.sum(np.array(max_significances) > 3.0)
    false_positives_5sig = np.sum(np.array(max_significances) > 5.0)
    
    print(f"\nMonte Carlo Results:")
    print(f"Maximum significance in random data: {max_sig:.2f}σ")
    print(f"False positives >3σ: {false_positives_3sig}/{n_simulations} ({100*false_positives_3sig/n_simulations:.1f}%)")
    print(f"False positives >5σ: {false_positives_5sig}/{n_simulations} ({100*false_positives_5sig/n_simulations:.1f}%)")
    
    return {
        'n_simulations': n_simulations,
        'max_significance': max_sig,
        'significances': max_significances,
        'fp_rate_3sig': false_positives_3sig / n_simulations,
        'fp_rate_5sig': false_positives_5sig / n_simulations
    }

def create_summary_plots(alpha_results, pi_results, mc_results):
    """Create publication-quality plots"""
    
    # Set up figure with subplots
    fig = plt.figure(figsize=(15, 10))
    
    # ========== Plot 1: α Detection Across Bands ==========
    ax1 = plt.subplot(2, 2, 1)
    
    bands = ['K', 'Ka', 'Q', 'V', 'W']
    frequencies = [23, 33, 41, 61, 94]
    alpha_sigs = [alpha_results[b]['analysis']['significance_sigma'] for b in bands]
    
    # Plot data
    ax1.plot(frequencies, alpha_sigs, 'o-', color='#4fc3f7', markersize=10, linewidth=2.5)
    
    # Add significance thresholds
    ax1.axhline(y=5.0, color='red', linestyle='--', alpha=0.7, label='5σ Discovery')
    ax1.axhline(y=3.0, color='orange', linestyle='--', alpha=0.7, label='3σ Evidence')
    ax1.axhline(y=-5.0, color='red', linestyle='--', alpha=0.7)
    ax1.axhline(y=-3.0, color='orange', linestyle='--', alpha=0.7)
    
    # Formatting
    ax1.set_xlabel('Frequency (GHz)')
    ax1.set_ylabel('Significance (σ)')
    ax1.set_title('α = 1/137 Detection at ℓ = 7', fontsize=16, fontweight='bold')
    ax1.grid(True, alpha=0.3)
    ax1.legend()
    
    # Add value labels
    for i, (f, s) in enumerate(zip(frequencies, alpha_sigs)):
        ax1.text(f, s+0.3, f'{s:.2f}σ', ha='center', fontsize=10)
    
    # ========== Plot 2: Monte Carlo Validation ==========
    ax2 = plt.subplot(2, 2, 2)
    
    # Histogram of MC significances
    ax2.hist(mc_results['significances'], bins=30, alpha=0.7, color='#81c784', edgecolor='black')
    
    # Mark observed maximum
    max_observed = max([abs(s) for s in alpha_sigs])
    ax2.axvline(x=max_observed, color='red', linewidth=3, label=f'Observed: {max_observed:.2f}σ')
    
    # Formatting
    ax2.set_xlabel('Maximum Significance (σ)')
    ax2.set_ylabel('Frequency')
    ax2.set_title('Monte Carlo Validation (100 simulations)', fontsize=16, fontweight='bold')
    ax2.legend()
    ax2.grid(True, alpha=0.3)
    
    # ========== Plot 3: α vs π Comparison ==========
    ax3 = plt.subplot(2, 2, 3)
    
    pi_sigs = [pi_results[b]['analysis']['significance_sigma'] for b in bands]
    
    # Plot both
    ax3.plot(frequencies, alpha_sigs, 'o-', color='#4fc3f7', markersize=10, 
             linewidth=2.5, label='α (ℓ=7)')
    ax3.plot(frequencies, pi_sigs, 's-', color='#ffeb3b', markersize=8, 
             linewidth=2, label='π (ℓ=565)')
    
    # Formatting
    ax3.set_xlabel('Frequency (GHz)')
    ax3.set_ylabel('Significance (σ)')
    ax3.set_title('Cross-Validation: α vs π', fontsize=16, fontweight='bold')
    ax3.legend()
    ax3.grid(True, alpha=0.3)
    ax3.axhline(y=0, color='gray', linestyle='-', alpha=0.5)
    
    # ========== Plot 4: Power Spectrum Detail ==========
    ax4 = plt.subplot(2, 2, 4)
    
    # Show Ka-band spectrum around ℓ=7
    ka_spectrum = alpha_results['Ka']['power_spectrum']
    ell_range = np.where((ka_spectrum['ell'] >= 2) & (ka_spectrum['ell'] <= 20))[0]
    
    ax4.semilogy(ka_spectrum['ell'][ell_range], ka_spectrum['dl'][ell_range], 
                 'k-', linewidth=2)
    ax4.axvline(x=7, color='red', linewidth=2, linestyle='--', label='α target (ℓ=7)')
    ax4.scatter([7], [ka_spectrum['dl'][7]], color='red', s=100, zorder=5)
    
    # Formatting
    ax4.set_xlabel('Multipole ℓ')
    ax4.set_ylabel('D_ℓ (μK²)')
    ax4.set_title('Ka-band Power Spectrum (ℓ = 2-20)', fontsize=16, fontweight='bold')
    ax4.legend()
    ax4.grid(True, alpha=0.3)
    
    plt.tight_layout()
    plt.savefig('wmap_alpha_detection_analysis.png', dpi=300, bbox_inches='tight')
    plt.show()

def save_results(alpha_results, pi_results, mc_results):
    """Save analysis results to JSON"""
    
    # Prepare results dictionary
    output = {
        'metadata': {
            'analysis_date': datetime.now().isoformat(),
            'target_constant': 'Fine Structure Constant α = 1/137.036',
            'target_multipole': ALPHA_MULTIPOLE,
            'analysis_window': ANALYSIS_WINDOW
        },
        'alpha_detection': {},
        'pi_crosscheck': {},
        'monte_carlo': {
            'n_simulations': mc_results['n_simulations'],
            'max_significance': mc_results['max_significance'],
            'fp_rate_3sig': mc_results['fp_rate_3sig'],
            'fp_rate_5sig': mc_results['fp_rate_5sig']
        }
    }
    
    # Extract key results
    for band in ['K', 'Ka', 'Q', 'V', 'W']:
        output['alpha_detection'][band] = {
            'frequency_ghz': alpha_results[band]['frequency'],
            'significance_sigma': alpha_results[band]['analysis']['significance_sigma'],
            'enhancement_percent': alpha_results[band]['analysis']['enhancement_percent'],
            'signal_mean': alpha_results[band]['analysis']['signal_mean'],
            'background_mean': alpha_results[band]['analysis']['background_mean'],
            'background_std': alpha_results[band]['analysis']['background_std']
        }
        
        output['pi_crosscheck'][band] = {
            'frequency_ghz': pi_results[band]['frequency'],
            'significance_sigma': pi_results[band]['analysis']['significance_sigma']
        }
    
    # Save to file
    with open('wmap_alpha_analysis_results.json', 'w') as f:
        json.dump(output, f, indent=2)
    
    print("\nResults saved to: wmap_alpha_analysis_results.json")

# =============================================================================
# MAIN ANALYSIS EXECUTION
# =============================================================================

def main():
    """Execute complete analysis pipeline"""
    
    print("╔" + "═" * 68 + "╗")
    print("║       WMAP α = 1/137 DETECTION ANALYSIS - PROFESSIONAL CODE       ║")
    print("╚" + "═" * 68 + "╝")
    print(f"Analysis started: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    
    # 1. Load WMAP data
    wmap_data = load_wmap_data()
    
    if len(wmap_data) == 0:
        print("ERROR: No WMAP data loaded. Please check file paths.")
        return None
    
    # 2. Analyze α detection
    alpha_results = analyze_alpha_detection(wmap_data)
    
    # 3. Cross-validation with π
    pi_results = analyze_pi_crosscheck(wmap_data)
    
    # 4. Monte Carlo validation
    mc_results = monte_carlo_validation(wmap_data, n_simulations=100)
    
    # 5. Create summary plots
    create_summary_plots(alpha_results, pi_results, mc_results)
    
    # 6. Save results
    save_results(alpha_results, pi_results, mc_results)
    
    # 7. Print summary
    print("\n" + "=" * 70)
    print("ANALYSIS SUMMARY")
    print("=" * 70)
    
    # Find maximum significance
    max_sig_band = None
    max_sig_value = 0
    
    for band in ['K', 'Ka', 'Q', 'V', 'W']:
        sig = abs(alpha_results[band]['analysis']['significance_sigma'])
        if sig > max_sig_value:
            max_sig_value = sig
            max_sig_band = band
    
    print(f"Maximum α detection: {max_sig_value:.2f}σ in {max_sig_band}-band")
    print(f"Monte Carlo false positive rate: {mc_results['fp_rate_5sig']*100:.3f}%")
    
    if max_sig_value > 5.0:
        print("\n>>> DISCOVERY-LEVEL DETECTION OF α = 1/137 IN CMB! <<<")
    elif max_sig_value > 3.0:
        print("\n>>> EVIDENCE-LEVEL DETECTION OF α = 1/137 IN CMB <<<")
    
    print(f"\nAnalysis completed: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    
    return {
        'alpha_results': alpha_results,
        'pi_results': pi_results,
        'mc_results': mc_results
    }

# Run the analysis
if __name__ == "__main__":
    results = main()
