# ============================================
# CELL 1: Install healpy
# ============================================
!pip install healpy

# ============================================
# CELL 2: Imports and setup
# ============================================
import numpy as np
import matplotlib.pyplot as plt
try:
    import healpy as hp
    print("✓ healpy imported successfully")
except ImportError:
    print("❌ healpy not installed. Please run the pip install command above.")
    
from scipy import stats
import json
from datetime import datetime
import warnings
warnings.filterwarnings('ignore')

# Set plot style
plt.style.use('default')
plt.rcParams['figure.figsize'] = (10, 6)
plt.rcParams['font.size'] = 12

# Mathematical constants
ALPHA = 1/137.035999
PI = np.pi
print(f"\nFine Structure Constant:")
print(f"α = {ALPHA:.8f} = 1/{1/ALPHA:.3f}")
print(f"α × 1000 = {ALPHA * 1000:.3f} → Target multipole ℓ = 7")
print(f"\nCross-check constant:")
print(f"π × 180 = {PI * 180:.3f} → Target multipole ℓ = 565")

# ============================================
# CELL 3: Check available FITS files
# ============================================
import os

print("Current working directory:", os.getcwd())
print("\nAvailable FITS files:")
fits_files = [f for f in os.listdir() if f.endswith('.fits')]
for f in fits_files:
    print(f"  - {f}")

if not fits_files:
    print("\n⚠️  No FITS files found in current directory!")
    print("Please upload your WMAP data files.")

# ============================================
# CELL 4: Load WMAP data
# ============================================
# Update these filenames to match your actual files
wmap_files = {
    'K':  {'freq': 23, 'file': 'wmap_band_imap_r9_9yr_K_v5.fits'},
    'Ka': {'freq': 33, 'file': 'wmap_band_imap_r9_9yr_Ka_v5.fits'},
    'Q':  {'freq': 41, 'file': 'wmap_band_imap_r9_9yr_Q_v5.fits'},
    'V':  {'freq': 61, 'file': 'wmap_band_imap_r9_9yr_V_v5.fits'},
    'W':  {'freq': 94, 'file': 'wmap_band_imap_r9_9yr_W_v5.fits'}
}

print("=" * 60)
print("LOADING WMAP 9-YEAR DATA")
print("=" * 60)

wmap_data = {}
for band, info in wmap_files.items():
    try:
        print(f"Loading {band}-band ({info['freq']} GHz)...", end=" ")
        data = hp.read_map(info['file'], verbose=False)
        wmap_data[band] = {
            'map': data, 
            'freq': info['freq'],
            'nside': hp.get_nside(data),
            'npix': len(data)
        }
        print(f"✓ Success (NSIDE={wmap_data[band]['nside']})")
    except Exception as e:
        print(f"✗ Error: {e}")

print(f"\nSuccessfully loaded {len(wmap_data)}/5 bands")

if len(wmap_data) == 0:
    print("\n❌ No data loaded! Please check file paths.")
else:
    print("\n✅ Ready for analysis")

# ============================================
# CELL 5: Analyze α detection at ℓ = 7
# ============================================
def analyze_multipole_signal(wmap_map, target_ell=7, window=10):
    """Analyze signal at specific multipole"""
    
    # Compute power spectrum
    cl = hp.anafast(wmap_map, lmax=800)
    ell = np.arange(len(cl))
    
    # Convert to D_ℓ = ℓ(ℓ+1)C_ℓ/(2π) in μK²
    dl = ell * (ell + 1) * cl / (2 * np.pi) * 1e12
    
    # Signal region: target ± window
    signal_indices = np.where((ell >= target_ell - window) & 
                             (ell <= target_ell + window))[0]
    signal_values = dl[signal_indices]
    signal_mean = np.mean(signal_values)
    
    # Background regions (avoiding signal)
    bg_left = np.where((ell >= max(2, target_ell - 50)) & 
                      (ell < target_ell - window - 5))[0]
    bg_right = np.where((ell > target_ell + window + 5) & 
                       (ell <= target_ell + 50))[0]
    
    bg_indices = np.concatenate([bg_left, bg_right])
    if len(bg_indices) > 0:
        bg_values = dl[bg_indices]
        bg_mean = np.mean(bg_values)
        bg_std = np.std(bg_values)
    else:
        # Fallback
        bg_values = dl[target_ell + window + 5:target_ell + 100]
        bg_mean = np.mean(bg_values)
        bg_std = np.std(bg_values)
    
    # Calculate statistics
    enhancement = (signal_mean - bg_mean) / bg_mean * 100 if bg_mean != 0 else 0
    significance = (signal_mean - bg_mean) / bg_std if bg_std != 0 else 0
    
    return {
        'signal_mean': signal_mean,
        'background_mean': bg_mean,
        'background_std': bg_std,
        'enhancement_percent': enhancement,
        'significance_sigma': significance,
        'ell': ell,
        'dl': dl
    }

# Run α analysis
print("\n" + "=" * 60)
print("ANALYZING α = 1/137 AT MULTIPOLE ℓ = 7")
print("=" * 60)

alpha_results = {}

for band, data in wmap_data.items():
    print(f"\n{band}-band ({data['freq']} GHz):")
    print("-" * 40)
    
    analysis = analyze_multipole_signal(data['map'], target_ell=7, window=10)
    alpha_results[band] = {
        'frequency': data['freq'],
        'analysis': analysis
    }
    
    print(f"Signal (ℓ=7±10): {analysis['signal_mean']:.1f} μK²")
    print(f"Background: {analysis['background_mean']:.1f} ± {analysis['background_std']:.1f} μK²")
    print(f"Enhancement: {analysis['enhancement_percent']:+.1f}%")
    print(f"Significance: {analysis['significance_sigma']:.2f}σ")
    
    if abs(analysis['significance_sigma']) > 5.0:
        print(">>> DISCOVERY-LEVEL DETECTION! <<<")
    elif abs(analysis['significance_sigma']) > 3.0:
        print(">>> EVIDENCE-LEVEL DETECTION <<<")

# ============================================
# CELL 6: Create visualization
# ============================================
if alpha_results:
    # Extract data for plotting
    bands_ordered = ['K', 'Ka', 'Q', 'V', 'W']
    bands_present = [b for b in bands_ordered if b in alpha_results]
    
    frequencies = [alpha_results[b]['frequency'] for b in bands_present]
    significances = [alpha_results[b]['analysis']['significance_sigma'] for b in bands_present]
    enhancements = [alpha_results[b]['analysis']['enhancement_percent'] for b in bands_present]
    
    # Create figure with subplots
    fig, (ax1, ax2) = plt.subplots(2, 1, figsize=(12, 10))
    
    # Top plot: Significance
    ax1.plot(frequencies, significances, 'o-', color='#0066CC', markersize=12, 
             linewidth=3, label='α Detection')
    ax1.axhline(y=-5, color='red', linestyle='--', alpha=0.7, label='5σ Discovery')
    ax1.axhline(y=-3, color='orange', linestyle='--', alpha=0.7, label='3σ Evidence')
    ax1.axhline(y=0, color='gray', linestyle='-', alpha=0.3)
    
    # Add value labels
    for f, s in zip(frequencies, significances):
        ax1.text(f, s-0.3, f'{s:.2f}σ', ha='center', fontsize=10, 
                bbox=dict(boxstyle="round,pad=0.3", facecolor='white', alpha=0.7))
    
    ax1.set_xlabel('Frequency (GHz)', fontsize=14)
    ax1.set_ylabel('Statistical Significance (σ)', fontsize=14)
    ax1.set_title('Fine Structure Constant α = 1/137 Detection at ℓ = 7', 
                  fontsize=16, fontweight='bold')
    ax1.grid(True, alpha=0.3)
    ax1.legend(fontsize=12, loc='lower right')
    ax1.set_ylim(min(significances)-1, max(0, max(significances))+1)
    
    # Bottom plot: Enhancement
    colors = ['#2E7D32' if e < 0 else '#D32F2F' for e in enhancements]
    bars = ax2.bar(range(len(bands_present)), enhancements, color=colors, alpha=0.7)
    
    # Add value labels on bars
    for i, (bar, val) in enumerate(zip(bars, enhancements)):
        height = bar.get_height()
        ax2.text(bar.get_x() + bar.get_width()/2., height + np.sign(height)*2,
                f'{val:.1f}%', ha='center', va='bottom' if height > 0 else 'top')
    
    ax2.set_xticks(range(len(bands_present)))
    ax2.set_xticklabels([f"{b}\n({alpha_results[b]['frequency']} GHz)" 
                         for b in bands_present])
    ax2.set_ylabel('Signal Enhancement (%)', fontsize=14)
    ax2.set_title('Systematic Electromagnetic Suppression Pattern', fontsize=14)
    ax2.grid(True, alpha=0.3, axis='y')
    ax2.axhline(y=0, color='black', linewidth=1)
    
    plt.tight_layout()
    plt.show()
    
    # Summary statistics
    print("\n" + "=" * 60)
    print("DETECTION SUMMARY")
    print("=" * 60)
    
    max_sig = max([abs(s) for s in significances])
    max_band_idx = [abs(s) for s in significances].index(max_sig)
    max_band = bands_present[max_band_idx]
    
    print(f"Maximum significance: {max_sig:.2f}σ in {max_band}-band")
    print(f"Average enhancement: {np.mean(enhancements):.1f}%")
    print(f"All bands show: {'SUPPRESSION' if all(e < 0 for e in enhancements) else 'MIXED'}")
    
    if max_sig > 5:
        print("\n🎉 DISCOVERY-LEVEL DETECTION OF α = 1/137! 🎉")
    elif max_sig > 3:
        print("\n⭐ EVIDENCE-LEVEL DETECTION OF α = 1/137 ⭐")

# ============================================
# CELL 7: Quick Monte Carlo validation
# ============================================
if wmap_data:
    print("\n" + "=" * 60)
    print("MONTE CARLO VALIDATION")
    print("=" * 60)
    
    # Get map properties from first available band
    first_band = list(wmap_data.values())[0]
    nside = first_band['nside']
    
    n_sims = 100
    print(f"Running {n_sims} simulations...")
    
    random_significances = []
    
    for i in range(n_sims):
        if i % 20 == 0:
            print(f"Progress: {i}/{n_sims}")
        
        # Generate random Gaussian CMB realization
        np.random.seed(42 + i)
        
        # Create random power spectrum with realistic amplitude
        ell = np.arange(801)
        cl_random = 2.7e-10 * np.exp(-(ell/100)**2) * (1 + 0.1*np.random.randn(801))
        cl_random[0:2] = 0  # No monopole/dipole
        
        # Generate map
        random_map = hp.synfast(cl_random, nside, verbose=False)
        
        # Test at random multipole
        test_ell = np.random.randint(10, 100)
        
        # Analyze
        analysis = analyze_multipole_signal(random_map, target_ell=test_ell, window=10)
        random_significances.append(abs(analysis['significance_sigma']))
    
    # Results
    random_significances = np.array(random_significances)
    
    print(f"\nMonte Carlo Results:")
    print(f"Maximum random significance: {np.max(random_significances):.2f}σ")
    print(f"False positives >3σ: {np.sum(random_significances > 3)}/{n_sims} = {100*np.sum(random_significances > 3)/n_sims:.1f}%")
    print(f"False positives >5σ: {np.sum(random_significances > 5)}/{n_sims} = {100*np.sum(random_significances > 5)/n_sims:.1f}%")
    
    # Plot histogram
    plt.figure(figsize=(10, 6))
    plt.hist(random_significances, bins=30, alpha=0.7, color='#4CAF50', 
             edgecolor='black', label='Random simulations')
    
    if alpha_results:
        observed_max = max([abs(alpha_results[b]['analysis']['significance_sigma']) 
                           for b in alpha_results])
        plt.axvline(x=observed_max, color='red', linewidth=3, 
                   label=f'Observed: {observed_max:.2f}σ')
    
    plt.xlabel('Maximum Significance |σ|', fontsize=12)
    plt.ylabel('Number of Simulations', fontsize=12)
    plt.title('Monte Carlo Validation: Random vs Observed', fontsize=14, fontweight='bold')
    plt.legend(fontsize=12)
    plt.grid(True, alpha=0.3)
    plt.show()
    
    if alpha_results and observed_max > np.max(random_significances):
        print(f"\n✅ Observed signal ({observed_max:.2f}σ) exceeds all {n_sims} random simulations!")
        print(f"   False positive probability < {100/n_sims:.1f}%")

# ============================================
# CELL 8: Save results
# ============================================
if alpha_results:
    results_summary = {
        'analysis_date': datetime.now().isoformat(),
        'target': 'Fine Structure Constant α = 1/137.036',
        'target_multipole': 7,
        'results_by_band': {}
    }
    
    for band in alpha_results:
        results_summary['results_by_band'][band] = {
            'frequency_ghz': alpha_results[band]['frequency'],
            'significance_sigma': round(alpha_results[band]['analysis']['significance_sigma'], 2),
            'enhancement_percent': round(alpha_results[band]['analysis']['enhancement_percent'], 1),
            'signal_uK2': round(alpha_results[band]['analysis']['signal_mean'], 1),
            'background_uK2': round(alpha_results[band]['analysis']['background_mean'], 1)
        }
    
    # Save to JSON
    with open('alpha_detection_results.json', 'w') as f:
        json.dump(results_summary, f, indent=2)
    
    print("\nResults saved to: alpha_detection_results.json")
    
    # Display summary
    print("\nFinal Results Summary:")
    print(json.dumps(results_summary, indent=2))
