/*
 * SPDX-FileCopyrightText: 2022-2023 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Unlicense OR CC0-1.0
 */
/* itwt Example

   This example code is in the Public Domain (or CC0 licensed, at your option.)

   Unless required by applicable law or agreed to in writing, this
   software is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
   CONDITIONS OF ANY KIND, either express or implied.
*/

/*
   this example shows how to use itwt
   set a router or a AP using the same SSID&PASSWORD as configuration of this example.
   start esp32c6 and when it connected to AP it will setup itwt.
*/
#include <netdb.h>
#include "freertos/FreeRTOS.h"
#include "freertos/event_groups.h"
#include "esp_wifi.h"
#include "esp_log.h"
#include "esp_event.h"
#include "nvs_flash.h"
#include "esp_console.h"
#include "cmd_system.h"
#include "wifi_cmd.h"
#include "esp_wifi_he.h"
#include "esp_pm.h"
#include "esp_timer.h"
#include "esp_private/wifi.h"

#include "lwip/err.h"
#include "lwip/etharp.h"
#include "lwip/sockets.h"
#include "lwip/sys.h"
#include <lwip/netdb.h>

#include <rom/ets_sys.h>

/*******************************************************
 *                Constants
 *******************************************************/
static const char *TAG = "itwt";

/*******************************************************
 *                Structures
 *******************************************************/

/*******************************************************
 *                Variable Definitions
 *******************************************************/
/*set the ssid and password via "idf.py menuconfig"*/
#define DEFAULT_SSID CONFIG_EXAMPLE_WIFI_SSID
#define DEFAULT_PWD CONFIG_EXAMPLE_WIFI_PASSWORD
#define ITWT_SETUP_SUCCESS 1

#define MAX_PAYLOAD_SIZE 1450
#define TWT_OFFSET CONFIG_EXAMPLE_TWT_OFFSET
#define TDMA_OFFSET CONFIG_EXAMPLE_TDMA_OFFSET
#define TWT_FREQUENCY CONFIG_EXAMPLE_TWT_FREQUENCY
static int PAYLOAD_SIZE = CONFIG_EXAMPLE_UDP_PACKET_SIZE;
static int NO_PACKETS = CONFIG_EXAMPLE_UDP_NO_PACKETS;
static int MIN_WAKE_DURA = CONFIG_EXAMPLE_ITWT_MIN_WAKE_DURA;
#define BEACON_PERIOD 102400

#if CONFIG_EXAMPLE_TDMA_ENABLE
uint8_t tdma_enabled = 1;
#else
uint8_t tdma_enabled = 0;
#endif

#if CONFIG_EXAMPLE_ITWT_ENABLE
uint8_t twt_enabled = 1;
#else
uint8_t twt_enabled = 0;
#endif

#if CONFIG_EXAMPLE_ENABLE_STATIC_ARP
uint8_t static_arp_enabled = 1;
#else
uint8_t static_arp_enabled = 0;
#endif

#if CONFIG_EXAMPLE_ITWT_TRIGGER_ENABLE
uint8_t trigger_enabled = 1;
#else
uint8_t trigger_enabled = 0;
#endif

#if CONFIG_EXAMPLE_ITWT_ANNOUNCED
uint8_t flow_type_announced = 1;
#else
uint8_t flow_type_announced = 0;
#endif

esp_netif_t *netif_sta = NULL;
const int CONNECTED_BIT = BIT0;
const int DISCONNECTED_BIT = BIT1;
EventGroupHandle_t wifi_event_group;

static const char *UDPTAG = "udp";
static char payload[MAX_PAYLOAD_SIZE];

/* // TASPER
const int twt_offsets[] = { 0, 5120, 25600, 35840, 40960, 61440, 66560, 71680, 81920, 87040}; //units of 2.56ms
const int twt_packets[] = { 2, 8,  4,  2,  8,  2,  2,  4,  2,  2};
const int twt_min_wake[]= {22, 88, 44, 22, 88, 22, 22, 44, 22, 22};
const int twt_payloads[]= {1200, 1200, 1200, 1200, 1200, 1200, 1200, 1200, 1200, 1200};
*/

// Shortest first
const int twt_offsets[] = { 0, 5120, 20480, 25600, 40960, 61440, 66560, 71680, 81920, 87040}; //units of 2.56ms
const int twt_packets[] = { 2,  4,  2,  2,  8,  2,  2,  4,  2, 0};
const int twt_min_wake[]= {22, 44, 22, 22, 88, 22, 22, 44, 22, 22};
const int twt_payloads[]= {1200, 1200, 1200, 1200, 1200, 1200, 1200, 1200, 1200, 1200};



#if CONFIG_EXAMPLE_ITWT_CUSTOM_SCHEDULE
uint8_t custom_schedule = 1;
#else
uint8_t custom_schedule = 0;
#endif


uint8_t sta_id;

static int sock;
TaskHandle_t udp_task;

/*******************************************************
 *                Function Declarations
 *******************************************************/

/*******************************************************
 *                Function Definitions
 *******************************************************/

static void oneshot_udp_timer_callback(void* arg){
    struct sockaddr_in dest_addr = *(struct sockaddr_in *)arg;

    ESP_LOGI(UDPTAG, "Sending message of %d udp packet(s) at %lld", NO_PACKETS, esp_timer_get_time());

    int cnt;
    for (cnt = 0; cnt < NO_PACKETS; cnt++)
    {
        payload[0] = (char) (cnt + '0');
        int err = sendto(sock, payload, PAYLOAD_SIZE, 0, (struct sockaddr *)&dest_addr, sizeof(dest_addr));
           
        if (err < 0) {
            int errsv = errno;
            ESP_LOGE(UDPTAG, "Error occurred during sending: errno %d", errsv);
            ESP_LOGE(UDPTAG, "%s", strerror(errsv));
        }
    }
    ESP_LOGD(UDPTAG, "Message sent");  
}

static void udp_client_task(void *pvParameters){
    //char rx_buffer[128];
    //char host_ip[] = CONFIG_EXAMPLE_UDP_HOST_IP_ADDR;
    int addr_family = 0;
    int ip_protocol = 0;
    
    while(true) {
        struct sockaddr_in dest_addr;
        dest_addr.sin_addr.s_addr = inet_addr(CONFIG_EXAMPLE_UDP_HOST_IP_ADDR);
        dest_addr.sin_family = AF_INET;
        dest_addr.sin_port = htons(CONFIG_EXAMPLE_UDP_HOST_PORT);
        addr_family = AF_INET;
        ip_protocol = IPPROTO_IP;

        sock = socket(addr_family, SOCK_DGRAM, ip_protocol);
        if (sock < 0) {
            ESP_LOGE(UDPTAG, "Unable to create socket: errno %d", errno);
            break;
        }

        // Set timeout
        struct timeval timeout;
        timeout.tv_sec = 10;
        timeout.tv_usec = 0;
        setsockopt (sock, SOL_SOCKET, SO_RCVTIMEO, &timeout, sizeof timeout);

        ESP_LOGI(UDPTAG, "Socket created, sending to %s:%d", CONFIG_EXAMPLE_UDP_HOST_IP_ADDR, CONFIG_EXAMPLE_UDP_HOST_PORT);


        const esp_timer_create_args_t oneshot_timer_args = {
            .callback = &oneshot_udp_timer_callback,
            .arg = (void*) &dest_addr,
            .name = "udp-tx"
        };

        esp_timer_handle_t oneshot_timer;
        ESP_ERROR_CHECK(esp_timer_create(&oneshot_timer_args, &oneshot_timer));
        
        int packet_generation_offset = TWT_OFFSET;
        if (twt_enabled) {
            // let TWT figure out the packet schedule
            packet_generation_offset = 0;
        }
        if (tdma_enabled && !twt_enabled) {
            packet_generation_offset += TDMA_OFFSET * sta_id;
        }
        if (custom_schedule) {
            packet_generation_offset = TWT_OFFSET + twt_offsets[sta_id];
        }

        while (true) {
            int64_t timing_advance_us = 1000;
            // send data TWT_OFFSET after beacon
            int64_t tsf_time = esp_wifi_get_tsf_time(WIFI_IF_STA);
            int64_t current_time = esp_timer_get_time();
            int64_t tsf_target_time = tsf_time - (tsf_time % (TWT_FREQUENCY * BEACON_PERIOD)) + (TWT_FREQUENCY * BEACON_PERIOD);
            int64_t wait_time = tsf_target_time - tsf_time + packet_generation_offset;
            sprintf(payload, "X,%lld,%d\n", tsf_target_time, sta_id);

            ESP_ERROR_CHECK(esp_timer_start_once(oneshot_timer, wait_time - 35 - timing_advance_us)); // remove 35us of overhead

            ESP_LOGI(UDPTAG, "TSF current time: %lld", tsf_time);
            ESP_LOGI(UDPTAG, "TSF target time:  %lld", tsf_time + wait_time);
     
            ESP_LOGI(UDPTAG, "Current time:   %lld", current_time);
            ESP_LOGI(UDPTAG, "Target TX time: %lld", current_time + wait_time);

            ESP_LOGD(UDPTAG, "Suspending for %d ms" , (int)((TWT_FREQUENCY * BEACON_PERIOD)/1000)-2);
            vTaskDelay((int)((TWT_FREQUENCY * BEACON_PERIOD)/1000)-2 / portTICK_PERIOD_MS);

        }

        if (sock != -1) {
            ESP_LOGE(UDPTAG, "Shutting down socket and restarting...");
            shutdown(sock, 0);
            close(sock);
        }
    }
    vTaskDelete(NULL);
}

static void example_set_static_ip(esp_netif_t *netif)
{
#if CONFIG_EXAMPLE_ENABLE_STATIC_IP
    if (esp_netif_dhcpc_stop(netif) != ESP_OK) {
        ESP_LOGE(TAG, "Failed to stop dhcp client");
        return;
    }
    esp_netif_ip_info_t ip;
    memset(&ip, 0 , sizeof(esp_netif_ip_info_t));
    ip.ip.addr = ipaddr_addr(CONFIG_EXAMPLE_STATIC_IP_ADDR);
    ip.netmask.addr = ipaddr_addr(CONFIG_EXAMPLE_STATIC_NETMASK_ADDR);
    ip.gw.addr = ipaddr_addr(CONFIG_EXAMPLE_STATIC_GW_ADDR);
    if (esp_netif_set_ip_info(netif, &ip) != ESP_OK) {
        ESP_LOGE(TAG, "Failed to set ip info");
        return;
    }
    ESP_LOGI(TAG, "Success to set static ip: %s, netmask: %s, gw: %s",
             CONFIG_EXAMPLE_STATIC_IP_ADDR, CONFIG_EXAMPLE_STATIC_NETMASK_ADDR, CONFIG_EXAMPLE_STATIC_GW_ADDR);
#endif
}

static const char *itwt_probe_status_to_str(wifi_itwt_probe_status_t status)
{
    switch (status) {
    case ITWT_PROBE_FAIL:                 return "itwt probe fail";
    case ITWT_PROBE_SUCCESS:              return "itwt probe success";
    case ITWT_PROBE_TIMEOUT:              return "itwt probe timeout";
    case ITWT_PROBE_STA_DISCONNECTED:     return "Sta disconnected";
    default:                              return "Unknown status";
    }
}


static void oneshot_twt_timer_callback(void* arg){
    esp_err_t err = ESP_OK;
    wifi_twt_setup_config_t* setup_config = (wifi_twt_setup_config_t*) arg;

    ESP_LOGI(TAG, "Demanding TWT at %lld", esp_timer_get_time());
    ESP_LOGI(TAG, "Demanding TWT at %lld", esp_wifi_get_tsf_time(WIFI_IF_STA));

    err = esp_wifi_sta_itwt_setup(setup_config);
    if (err != ESP_OK) {
        ESP_LOGE(TAG, "itwt setup failed, err:0x%x", err);
    }    
}

static void setup_twt()
{
    wifi_twt_setup_config_t setup_config = {
        .setup_cmd = TWT_DEMAND,
        .flow_id = 0,
        .twt_id = CONFIG_EXAMPLE_ITWT_ID,
        .flow_type = flow_type_announced ? 0 : 1,
        .min_wake_dura = MIN_WAKE_DURA,
        .wake_duration_unit = CONFIG_EXAMPLE_ITWT_WAKE_DURATION_UNIT,
        .wake_invl_expn = CONFIG_EXAMPLE_ITWT_WAKE_INVL_EXPN,
        .wake_invl_mant = CONFIG_EXAMPLE_ITWT_WAKE_INVL_MANT,
        .trigger = trigger_enabled,
        .timeout_time_ms = CONFIG_EXAMPLE_ITWT_SETUP_TIMEOUT_TIME_MS,
    };

    int packet_generation_offset = TWT_OFFSET;
    if (tdma_enabled)
        packet_generation_offset += TDMA_OFFSET * sta_id;
    if (custom_schedule)
        packet_generation_offset = TWT_OFFSET + twt_offsets[sta_id];
    
    ESP_ERROR_CHECK(esp_wifi_sta_itwt_set_target_wake_time_offset(packet_generation_offset));

    // timer help starting the TWT service period considering TWT_FREQUENCY
    const esp_timer_create_args_t oneshot_timer_args = {
            .callback = &oneshot_twt_timer_callback,
            .arg = (void*) &setup_config,
            .name = "twt-setup"
        };
    
    esp_timer_handle_t oneshot_timer;
    ESP_ERROR_CHECK(esp_timer_create(&oneshot_timer_args, &oneshot_timer));

    int64_t tsf_time = esp_wifi_get_tsf_time(WIFI_IF_STA);
    int64_t current_time = esp_timer_get_time();

    int64_t tsf_request_target_time = tsf_time - (tsf_time % (TWT_FREQUENCY * BEACON_PERIOD)) + (TWT_FREQUENCY * BEACON_PERIOD);
    int64_t tsf_target_time = tsf_request_target_time + (2* TWT_FREQUENCY * BEACON_PERIOD) + packet_generation_offset;
    int64_t wait_time = tsf_request_target_time - tsf_time;

    ESP_LOGI(TAG, "TSF current time: %lld", tsf_time);
    ESP_LOGI(TAG, "TSF target TWT:  %lld", tsf_target_time);

    ESP_LOGI(TAG, "Current time:      %lld", current_time);
    ESP_LOGI(TAG, "Target TWT demand: %lld", current_time + wait_time);

    ESP_ERROR_CHECK(esp_timer_start_once(oneshot_timer, wait_time));

    vTaskDelay((TickType_t)((2000) / portTICK_PERIOD_MS));

}

static void got_ip_handler(void *arg, esp_event_base_t event_base,
                           int32_t event_id, void *event_data)
{
    xEventGroupClearBits(wifi_event_group, DISCONNECTED_BIT);
    xEventGroupSetBits(wifi_event_group, CONNECTED_BIT);
    ip_event_got_ip_t* event = (ip_event_got_ip_t*) event_data;
    sta_id = esp_ip4_addr4_16(&event->ip_info.ip)-10;
    //ESP_LOGI(TAG, "got ip:" IPSTR, IP2STR(&event->ip_info.ip));
    ESP_LOGI(TAG, "sta ID: %d", sta_id);

    if (custom_schedule) {
        PAYLOAD_SIZE = twt_payloads[sta_id];
        NO_PACKETS = twt_packets[sta_id];
        MIN_WAKE_DURA = twt_min_wake[sta_id];
    }

#if CONFIG_EXAMPLE_ENABLE_STATIC_ARP
    ip4_addr_t ipaddr;
    ipaddr.addr = ipaddr_addr(CONFIG_EXAMPLE_UDP_HOST_IP_ADDR);
    
    struct eth_addr ethaddr;
    char macStr[] = CONFIG_EXAMPLE_UDP_HOST_MAC_ADDR;
    for (int i = 0; i < 6; ++i) {
        char byteStr[3] = { macStr[i*2], macStr[i*2 + 1], '\0' };
        ethaddr.addr[i] = (uint8_t) strtoul(byteStr, NULL, 16);
    }
    ESP_LOGI(TAG, "IP %s - MAC %02x:%02x:%02x:%02x:%02x:%02x",
        ip4addr_ntoa(&ipaddr),
        ethaddr.addr[0], ethaddr.addr[1], ethaddr.addr[2],
        ethaddr.addr[3], ethaddr.addr[4], ethaddr.addr[5]);  
    ESP_ERROR_CHECK(etharp_add_static_entry(&ipaddr, &ethaddr));
#endif


    /* setup a trigger-based announce individual TWT agreement. */
    wifi_phy_mode_t phymode;
    wifi_config_t sta_cfg = { 0, };
    esp_wifi_get_config(WIFI_IF_STA, &sta_cfg);
    esp_wifi_sta_get_negotiated_phymode(&phymode);
    
    if (phymode != WIFI_PHY_MODE_HE20)
    {
        ESP_LOGE(TAG, "Must be in 11ax mode to support itwt");
        return;
    }

    if (twt_enabled)
        setup_twt();

    xTaskCreate(udp_client_task, "udp_client", 4096, NULL, 5, &udp_task);

}

static void start_handler(void *arg, esp_event_base_t event_base,
                            int32_t event_id, void *event_data)
{
    ESP_LOGI(TAG, "sta connect to %s", DEFAULT_SSID);
    ESP_ERROR_CHECK(esp_wifi_connect());
}

static void disconnect_handler(void *arg, esp_event_base_t event_base,
                               int32_t event_id, void *event_data)
{
    ESP_LOGI(TAG, "sta disconnect, reconnect...");
    xEventGroupClearBits(wifi_event_group, CONNECTED_BIT);
    vTaskDelete(udp_task);
    ESP_ERROR_CHECK(esp_wifi_connect());
}

static void itwt_setup_handler(void *arg, esp_event_base_t event_base,
                               int32_t event_id, void *event_data)
{
    wifi_event_sta_itwt_setup_t *setup = (wifi_event_sta_itwt_setup_t *) event_data;
    if (setup->status == ITWT_SETUP_SUCCESS) {
        /* TWT Wake Interval = TWT Wake Interval Mantissa * (2 ^ TWT Wake Interval Exponent) */
        ESP_LOGI(TAG, "<WIFI_EVENT_ITWT_SETUP>twt_id:%d, flow_id:%d, %s, %s, wake_dura:%d, wake_dura_unit:%d, wake_invl_e:%d, wake_invl_m:%d", setup->config.twt_id,
                setup->config.flow_id, setup->config.trigger ? "trigger-enabled" : "non-trigger-enabled", setup->config.flow_type ? "unannounced" : "announced",
                setup->config.min_wake_dura, setup->config.wake_duration_unit, setup->config.wake_invl_expn, setup->config.wake_invl_mant);
        ESP_LOGI(TAG, "<WIFI_EVENT_ITWT_SETUP>target wake time:%lld, wake duration:%d us, service period:%d us", setup->target_wake_time, setup->config.min_wake_dura << (setup->config.wake_duration_unit == 1 ? 10 : 8),
                setup->config.wake_invl_mant << setup->config.wake_invl_expn);
    } else {
        if (setup->status == ESP_ERR_WIFI_TWT_SETUP_TIMEOUT) {
            ESP_LOGE(TAG, "<WIFI_EVENT_ITWT_SETUP>twt_id:%d, timeout of receiving twt setup response frame", setup->config.twt_id);
        } else if (setup->status == ESP_ERR_WIFI_TWT_SETUP_TXFAIL) {
            ESP_LOGE(TAG, "<WIFI_EVENT_ITWT_SETUP>twt_id:%d, twt setup frame tx failed, reason: %d", setup->config.twt_id, setup->reason);
        } else if (setup->status == ESP_ERR_WIFI_TWT_SETUP_REJECT) {
            ESP_LOGE(TAG, "<WIFI_EVENT_ITWT_SETUP>twt_id:%d, twt setup request was rejected, setup cmd: %d", setup->config.twt_id, setup->config.setup_cmd);
        } else {
            ESP_LOGE(TAG, "<WIFI_EVENT_ITWT_SETUP>twt_id:%d, twt setup failed, status: %d", setup->config.twt_id, setup->status);
        }
    }
}

static void itwt_teardown_handler(void *arg, esp_event_base_t event_base,
                                  int32_t event_id, void *event_data)
{
    wifi_event_sta_itwt_teardown_t *teardown = (wifi_event_sta_itwt_teardown_t *) event_data;
    ESP_LOGI(TAG, "<WIFI_EVENT_ITWT_TEARDOWN>flow_id %d%s", teardown->flow_id, (teardown->flow_id == 8) ? "(all twt)" : "");
}

static void itwt_suspend_handler(void *arg, esp_event_base_t event_base,
                                 int32_t event_id, void *event_data)
{
    wifi_event_sta_itwt_suspend_t *suspend = (wifi_event_sta_itwt_suspend_t *) event_data;
    ESP_LOGI(TAG, "<WIFI_EVENT_ITWT_SUSPEND>status:%d, flow_id_bitmap:0x%x, actual_suspend_time_ms:[%lu %lu %lu %lu %lu %lu %lu %lu]",
             suspend->status, suspend->flow_id_bitmap,
             suspend->actual_suspend_time_ms[0], suspend->actual_suspend_time_ms[1], suspend->actual_suspend_time_ms[2], suspend->actual_suspend_time_ms[3],
             suspend->actual_suspend_time_ms[4], suspend->actual_suspend_time_ms[5], suspend->actual_suspend_time_ms[6], suspend->actual_suspend_time_ms[7]);
}

static void itwt_probe_handler(void *arg, esp_event_base_t event_base,
                               int32_t event_id, void *event_data)
{
    wifi_event_sta_itwt_probe_t *probe = (wifi_event_sta_itwt_probe_t *) event_data;
    ESP_LOGI(TAG, "<WIFI_EVENT_ITWT_PROBE>status:%s, reason:0x%x", itwt_probe_status_to_str(probe->status), probe->reason);
}

static void wifi_itwt(void)
{
    ESP_ERROR_CHECK(esp_netif_init());
    wifi_event_group = xEventGroupCreate();
    ESP_ERROR_CHECK(esp_event_loop_create_default());
    netif_sta = esp_netif_create_default_wifi_sta();
    assert(netif_sta);

    wifi_init_config_t cfg = WIFI_INIT_CONFIG_DEFAULT();
    ESP_ERROR_CHECK(esp_wifi_init(&cfg));
    esp_event_handler_instance_t instance_got_ip;

    ESP_ERROR_CHECK(esp_event_handler_instance_register(WIFI_EVENT,
                    WIFI_EVENT_STA_START,
                    &start_handler,
                    NULL,
                    NULL));
    ESP_ERROR_CHECK(esp_event_handler_instance_register(WIFI_EVENT,
                    WIFI_EVENT_STA_DISCONNECTED,
                    &disconnect_handler,
                    NULL,
                    NULL));
    ESP_ERROR_CHECK(esp_event_handler_instance_register(IP_EVENT,
                    IP_EVENT_STA_GOT_IP,
                    &got_ip_handler,
                    NULL,
                    &instance_got_ip));
    /* itwt */
    ESP_ERROR_CHECK(esp_event_handler_instance_register(WIFI_EVENT,
                    WIFI_EVENT_ITWT_SETUP,
                    &itwt_setup_handler,
                    NULL,
                    NULL));
    ESP_ERROR_CHECK(esp_event_handler_instance_register(WIFI_EVENT,
                    WIFI_EVENT_ITWT_TEARDOWN,
                    &itwt_teardown_handler,
                    NULL,
                    NULL));
    ESP_ERROR_CHECK(esp_event_handler_instance_register(WIFI_EVENT,
                    WIFI_EVENT_ITWT_SUSPEND,
                    &itwt_suspend_handler,
                    NULL,
                    NULL));
    ESP_ERROR_CHECK(esp_event_handler_instance_register(WIFI_EVENT,
                    WIFI_EVENT_ITWT_PROBE,
                    &itwt_probe_handler,
                    NULL,
                    NULL));

    wifi_config_t wifi_config = {
        .sta = {
            .ssid = DEFAULT_SSID,
            .password = DEFAULT_PWD,
            .failure_retry_cnt = 5,
        },
    };
    ESP_ERROR_CHECK(esp_wifi_set_mode(WIFI_MODE_STA));
    ESP_ERROR_CHECK(esp_wifi_set_config(WIFI_IF_STA, &wifi_config));

    esp_wifi_set_bandwidth(WIFI_IF_STA, WIFI_BW_HT20);
    esp_wifi_set_protocol(WIFI_IF_STA, WIFI_PROTOCOL_11AX);
    // The following is fundamental, pay attention!
    esp_wifi_set_ps(WIFI_PS_NONE);

    //ESP_ERROR_CHECK(esp_wifi_config_80211_tx_rate(WIFI_IF_STA, WIFI_PHY_RATE_MCS1_SGI));

#if CONFIG_EXAMPLE_ENABLE_STATIC_IP
    example_set_static_ip(netif_sta);
#endif

    ESP_ERROR_CHECK(esp_wifi_start());
    ESP_ERROR_CHECK(esp_wifi_internal_set_fix_rate(WIFI_IF_STA, true, WIFI_PHY_RATE_MCS0_SGI));

#if CONFIG_ESP_WIFI_ENABLE_WIFI_RX_STATS
#if CONFIG_ESP_WIFI_ENABLE_WIFI_RX_MU_STATS
    esp_wifi_enable_rx_statistics(true, true);
#else
    esp_wifi_enable_rx_statistics(true, false);
#endif
#endif
#if CONFIG_ESP_WIFI_ENABLE_WIFI_TX_STATS
    esp_wifi_enable_tx_statistics(ESP_WIFI_ACI_VO, true); //VO, mgmt
    esp_wifi_enable_tx_statistics(ESP_WIFI_ACI_BE, true); //BE, data
#endif



}

static int wifi_cmd_tsf_get(int argc, char **argv)
{
    
    int64_t tsf_time = esp_wifi_get_tsf_time(WIFI_IF_STA);
    if (tsf_time == 0)
        return 1;
    else {
        ESP_LOGI(TAG, "(tsf) time: %lld", tsf_time);
        return 0;
    }
}





void app_main(void)
{
    // Initialize NVS
    esp_err_t ret = nvs_flash_init();
    if (ret == ESP_ERR_NVS_NO_FREE_PAGES || ret == ESP_ERR_NVS_NEW_VERSION_FOUND) {
        ESP_ERROR_CHECK(nvs_flash_erase());
        ret = nvs_flash_init();
    }

#if CONFIG_PM_ENABLE
    // Configure dynamic frequency scaling:
    // maximum and minimum frequencies are set in sdkconfig,
    // automatic light sleep is enabled if tickless idle support is enabled.
    esp_pm_config_t pm_config = {
        .max_freq_mhz = CONFIG_EXAMPLE_MAX_CPU_FREQ_MHZ,
        .min_freq_mhz = CONFIG_EXAMPLE_MIN_CPU_FREQ_MHZ,
#if CONFIG_FREERTOS_USE_TICKLESS_IDLE
        .light_sleep_enable = true
#endif
    };
    ESP_ERROR_CHECK( esp_pm_configure(&pm_config) );
    ESP_ERROR_CHECK( ret );

#else
    printf("\n =================================================\n");
    printf(" |                   Test WiFi 6 itwt             |\n");
    printf(" =================================================\n\n");

    esp_console_repl_t *repl = NULL;
    esp_console_repl_config_t repl_config = ESP_CONSOLE_REPL_CONFIG_DEFAULT();
    repl_config.prompt = "itwt>";

    // init console REPL environment
#if CONFIG_ESP_CONSOLE_UART
    esp_console_dev_uart_config_t uart_config = ESP_CONSOLE_DEV_UART_CONFIG_DEFAULT();
    ESP_ERROR_CHECK(esp_console_new_repl_uart(&uart_config, &repl_config, &repl));
#elif CONFIG_ESP_CONSOLE_USB_CDC
    esp_console_dev_usb_cdc_config_t cdc_config = ESP_CONSOLE_DEV_CDC_CONFIG_DEFAULT();
    ESP_ERROR_CHECK(esp_console_new_repl_usb_cdc(&cdc_config, &repl_config, &repl));
#elif CONFIG_ESP_CONSOLE_USB_SERIAL_JTAG
    esp_console_dev_usb_serial_jtag_config_t usbjtag_config = ESP_CONSOLE_DEV_USB_SERIAL_JTAG_CONFIG_DEFAULT();
    ESP_ERROR_CHECK(esp_console_new_repl_usb_serial_jtag(&usbjtag_config, &repl_config, &repl));
#endif

    // start console REPL
    ESP_ERROR_CHECK(esp_console_start_repl(repl));
#endif

    //start wifi
    wifi_itwt();
    // register commands
    register_system();
    register_wifi_itwt();
    register_wifi_stats();

    const esp_console_cmd_t tsf_cmd = {
        .command = "tsf",
        .help = "tsf time report",
        .hint = NULL,
        .func = &wifi_cmd_tsf_get,
        .argtable = NULL
    };
    ESP_ERROR_CHECK(esp_console_cmd_register(&tsf_cmd));
    ESP_LOGI(TAG, "TDMA %d, TWT %d", tdma_enabled, twt_enabled);
}
