
      module icepack_shortwave_data

      use icepack_kinds
      use icepack_warnings, only: warnstr, icepack_warnings_add
      use icepack_warnings, only: icepack_warnings_setabort, icepack_warnings_aborted
      use icepack_parameters, only: c0
      use icepack_tracers, only: nmodal1, nmodal2, max_aero

      implicit none

      private

      integer (kind=int_kind), parameter, public :: &
         nspint_3bd = 3, & ! number of solar spectral bands
         nspint_5bd = 5    ! number of solar spectral bands (snicar snow)

      character (len=char_len), public :: &
         rsnw_datatype = 'unknown'

      ! dEdd 3-band data
      real (kind=dbl_kind), dimension (nspint_3bd), public :: &
         ! inherent optical properties (iop)
         !   mn = specified mean (or base) value
         !    k = extinction coefficient (/m)
         !    w = single scattering albedo
         !    g = asymmetry parameter
         ki_ssl_mn_3bd, wi_ssl_mn_3bd, gi_ssl_mn_3bd, & ! ice surface scattering layer (ssl) iops
         ki_dl_mn_3bd,  wi_dl_mn_3bd,  gi_dl_mn_3bd , & ! ice drained layer (dl) iops
         ki_int_mn_3bd, wi_int_mn_3bd, gi_int_mn_3bd, & ! ice interior layer (int) iops
         ki_p_ssl_mn,   wi_p_ssl_mn,   gi_p_ssl_mn  , & ! ponded ice surface scattering layer (ssl) iops
         ki_p_int_mn,   wi_p_int_mn,   gi_p_int_mn  , & ! ponded ice interior layer (int) iops
         kw,            ww,            gw               ! iops for pond water and underlying ocean

      real (kind=dbl_kind), allocatable, public :: &
         gaer_bc_3bd (:,:),  & ! gaer_bc_3bd, modalAsymmetryParameter5band
         kaer_bc_3bd (:,:),  & ! kaer_bc_3bd, modalMassExtinctionCrossSection5band
         waer_bc_3bd (:,:),  & ! waer_bc_3bd, modalSingleScatterAlbedo5band
         gaer_3bd    (:,:),  & ! gaer_3bd, aerosolAsymmetryParameter5band
         kaer_3bd    (:,:),  & ! kaer_3bd, aerosolMassExtinctionCrossSection5band
         waer_3bd    (:,:),  & ! waer_3bd, aerosolSingleScatterAlbedo5band
         bcenh_3bd   (:,:,:)   ! bcenh_3bd, modalBCabsorptionParameter5band

      integer (kind=int_kind), public :: &
         nmbrad_snw         ! number of snow grain radii in tables
      real (kind=dbl_kind), allocatable, public :: &
         rsnw_tab(:)  , &   ! snow grain radii (micro-meters) for table
         Qs_tab  (:,:), &   ! snow extinction efficiency (unitless)
         ws_tab  (:,:), &   ! snow single scattering albedo (unitless)
         gs_tab  (:,:)      ! snow asymmetry parameter (unitless)

      ! dEdd 5-band data
      real (kind=dbl_kind), dimension (nspint_5bd), public :: &
         ki_ssl_mn_5bd, wi_ssl_mn_5bd, gi_ssl_mn_5bd, & ! ice surface scattering layer (ssl) iops
         ki_dl_mn_5bd,  wi_dl_mn_5bd,  gi_dl_mn_5bd , & ! ice drained layer (dl) iops
         ki_int_mn_5bd, wi_int_mn_5bd, gi_int_mn_5bd    ! ice interior layer (int) iops

      real (kind=dbl_kind), allocatable, public :: &
         gaer_bc_5bd (:,:),  & ! gaer_bc_5bd, modalAsymmetryParameter5band
         kaer_bc_5bd (:,:),  & ! kaer_bc_5bd, modalMassExtinctionCrossSection5band
         waer_bc_5bd (:,:),  & ! waer_bc_5bd, modalSingleScatterAlbedo5band
         gaer_5bd    (:,:),  & ! gaer_5bd, aerosolAsymmetryParameter5band
         kaer_5bd    (:,:),  & ! kaer_5bd, aerosolMassExtinctionCrossSection5band
         waer_5bd    (:,:),  & ! waer_5bd, aerosolSingleScatterAlbedo5band
         bcenh_5bd   (:,:,:)   ! bcenh_5bd, modalBCabsorptionParameter5band

      ! dEdd 5-band data SSP SNICAR
      integer (kind=int_kind), public :: &
         nmbrad_snicar      ! number of snow grain radii in SNICAR SSP tables

      real (kind=dbl_kind), public :: &
         rsnw_snicar_min, & ! minimum snow radius - integer value used for indexing
         rsnw_snicar_max    ! maximum snow radius - integer value used for indexing

      real (kind=dbl_kind), pointer, public :: &
         ! these are pointers instead of allocatables because icepack_shortwave has local names
         ! TODO: should make these allocatable and update icepack_shortwave names OR
         ! maybe make all the data here pointers.
         rsnw_snicar_tab(:), & ! snow grain radii (10^-6 m)
         ssp_snwextdr(:,:),  & ! snow mass extinction cross section (m2/kg), direct
         ssp_snwextdf(:,:),  & ! snow mass extinction cross section (m2/kg), diffuse
         ssp_snwalbdr(:,:),  & ! snow single scatter albedo (fraction), direct
         ssp_snwalbdf(:,:),  & ! snow single scatter albedo (fraction), diffuse
         ssp_sasymmdr(:,:),  & ! snow asymmetry factor (cos(theta)), direct
         ssp_sasymmdf(:,:)     ! snow asymmetry factor (cos(theta)), diffuse

      ! dEdd SNICAR SSP data 5 band

      public :: icepack_shortwave_init_dEdd3band
      public :: icepack_shortwave_init_dEdd5band
      public :: icepack_shortwave_init_snicartest
      public :: icepack_shortwave_init_snicar

!=======================================================================
      contains
!=======================================================================
      ! tabular data needed for the original dEdd scheme

      subroutine icepack_shortwave_init_dEdd3band

      ! local variables

      integer (kind=int_kind) :: n

      character (len=*),parameter :: subname='(icepack_shortwave_init_dEdd3band)'

      nmbrad_snw = 32 ! number of snow grain radii in tables

      allocate(rsnw_tab(           nmbrad_snw))
      allocate(Qs_tab  (nspint_3bd,nmbrad_snw))
      allocate(ws_tab  (nspint_3bd,nmbrad_snw))
      allocate(gs_tab  (nspint_3bd,nmbrad_snw))

      ! snow grain radii (micro-meters) for table
      rsnw_datatype = 'sorted'
      rsnw_tab = (/ &   ! snow grain radius for each table entry (micro-meters)
          5._dbl_kind,    7._dbl_kind,   10._dbl_kind,   15._dbl_kind, &
         20._dbl_kind,   30._dbl_kind,   40._dbl_kind,   50._dbl_kind, &
         65._dbl_kind,   80._dbl_kind,  100._dbl_kind,  120._dbl_kind, &
        140._dbl_kind,  170._dbl_kind,  200._dbl_kind,  240._dbl_kind, &
        290._dbl_kind,  350._dbl_kind,  420._dbl_kind,  500._dbl_kind, &
        570._dbl_kind,  660._dbl_kind,  760._dbl_kind,  870._dbl_kind, &
       1000._dbl_kind, 1100._dbl_kind, 1250._dbl_kind, 1400._dbl_kind, &
       1600._dbl_kind, 1800._dbl_kind, 2000._dbl_kind, 2500._dbl_kind/)

      ! snow extinction efficiency (unitless)
      Qs_tab = reshape((/ &
          2.131798_dbl_kind,  2.187756_dbl_kind,  2.267358_dbl_kind, &
          2.104499_dbl_kind,  2.148345_dbl_kind,  2.236078_dbl_kind, &
          2.081580_dbl_kind,  2.116885_dbl_kind,  2.175067_dbl_kind, &
          2.062595_dbl_kind,  2.088937_dbl_kind,  2.130242_dbl_kind, &
          2.051403_dbl_kind,  2.072422_dbl_kind,  2.106610_dbl_kind, &
          2.039223_dbl_kind,  2.055389_dbl_kind,  2.080586_dbl_kind, &
          2.032383_dbl_kind,  2.045751_dbl_kind,  2.066394_dbl_kind, &
          2.027920_dbl_kind,  2.039388_dbl_kind,  2.057224_dbl_kind, &
          2.023444_dbl_kind,  2.033137_dbl_kind,  2.048055_dbl_kind, &
          2.020412_dbl_kind,  2.028840_dbl_kind,  2.041874_dbl_kind, &
          2.017608_dbl_kind,  2.024863_dbl_kind,  2.036046_dbl_kind, &
          2.015592_dbl_kind,  2.022021_dbl_kind,  2.031954_dbl_kind, &
          2.014083_dbl_kind,  2.019887_dbl_kind,  2.028853_dbl_kind, &
          2.012368_dbl_kind,  2.017471_dbl_kind,  2.025353_dbl_kind, &
          2.011092_dbl_kind,  2.015675_dbl_kind,  2.022759_dbl_kind, &
          2.009837_dbl_kind,  2.013897_dbl_kind,  2.020168_dbl_kind, &
          2.008668_dbl_kind,  2.012252_dbl_kind,  2.017781_dbl_kind, &
          2.007627_dbl_kind,  2.010813_dbl_kind,  2.015678_dbl_kind, &
          2.006764_dbl_kind,  2.009577_dbl_kind,  2.013880_dbl_kind, &
          2.006037_dbl_kind,  2.008520_dbl_kind,  2.012382_dbl_kind, &
          2.005528_dbl_kind,  2.007807_dbl_kind,  2.011307_dbl_kind, &
          2.005025_dbl_kind,  2.007079_dbl_kind,  2.010280_dbl_kind, &
          2.004562_dbl_kind,  2.006440_dbl_kind,  2.009333_dbl_kind, &
          2.004155_dbl_kind,  2.005898_dbl_kind,  2.008523_dbl_kind, &
          2.003794_dbl_kind,  2.005379_dbl_kind,  2.007795_dbl_kind, &
          2.003555_dbl_kind,  2.005041_dbl_kind,  2.007329_dbl_kind, &
          2.003264_dbl_kind,  2.004624_dbl_kind,  2.006729_dbl_kind, &
          2.003037_dbl_kind,  2.004291_dbl_kind,  2.006230_dbl_kind, &
          2.002776_dbl_kind,  2.003929_dbl_kind,  2.005700_dbl_kind, &
          2.002590_dbl_kind,  2.003627_dbl_kind,  2.005276_dbl_kind, &
          2.002395_dbl_kind,  2.003391_dbl_kind,  2.004904_dbl_kind, &
          2.002071_dbl_kind,  2.002922_dbl_kind,  2.004241_dbl_kind/), &
          (/nspint_3bd,nmbrad_snw/))

      ! snow single scattering albedo (unitless)
      ws_tab = reshape((/ &
         0.9999994_dbl_kind,  0.9999673_dbl_kind,  0.9954589_dbl_kind, &
         0.9999992_dbl_kind,  0.9999547_dbl_kind,  0.9938576_dbl_kind, &
         0.9999990_dbl_kind,  0.9999382_dbl_kind,  0.9917989_dbl_kind, &
         0.9999985_dbl_kind,  0.9999123_dbl_kind,  0.9889724_dbl_kind, &
         0.9999979_dbl_kind,  0.9998844_dbl_kind,  0.9866190_dbl_kind, &
         0.9999970_dbl_kind,  0.9998317_dbl_kind,  0.9823021_dbl_kind, &
         0.9999960_dbl_kind,  0.9997800_dbl_kind,  0.9785269_dbl_kind, &
         0.9999951_dbl_kind,  0.9997288_dbl_kind,  0.9751601_dbl_kind, &
         0.9999936_dbl_kind,  0.9996531_dbl_kind,  0.9706974_dbl_kind, &
         0.9999922_dbl_kind,  0.9995783_dbl_kind,  0.9667577_dbl_kind, &
         0.9999903_dbl_kind,  0.9994798_dbl_kind,  0.9621007_dbl_kind, &
         0.9999885_dbl_kind,  0.9993825_dbl_kind,  0.9579541_dbl_kind, &
         0.9999866_dbl_kind,  0.9992862_dbl_kind,  0.9541924_dbl_kind, &
         0.9999838_dbl_kind,  0.9991434_dbl_kind,  0.9490959_dbl_kind, &
         0.9999810_dbl_kind,  0.9990025_dbl_kind,  0.9444940_dbl_kind, &
         0.9999772_dbl_kind,  0.9988171_dbl_kind,  0.9389141_dbl_kind, &
         0.9999726_dbl_kind,  0.9985890_dbl_kind,  0.9325819_dbl_kind, &
         0.9999670_dbl_kind,  0.9983199_dbl_kind,  0.9256405_dbl_kind, &
         0.9999605_dbl_kind,  0.9980117_dbl_kind,  0.9181533_dbl_kind, &
         0.9999530_dbl_kind,  0.9976663_dbl_kind,  0.9101540_dbl_kind, &
         0.9999465_dbl_kind,  0.9973693_dbl_kind,  0.9035031_dbl_kind, &
         0.9999382_dbl_kind,  0.9969939_dbl_kind,  0.8953134_dbl_kind, &
         0.9999289_dbl_kind,  0.9965848_dbl_kind,  0.8865789_dbl_kind, &
         0.9999188_dbl_kind,  0.9961434_dbl_kind,  0.8773350_dbl_kind, &
         0.9999068_dbl_kind,  0.9956323_dbl_kind,  0.8668233_dbl_kind, &
         0.9998975_dbl_kind,  0.9952464_dbl_kind,  0.8589990_dbl_kind, &
         0.9998837_dbl_kind,  0.9946782_dbl_kind,  0.8476493_dbl_kind, &
         0.9998699_dbl_kind,  0.9941218_dbl_kind,  0.8367318_dbl_kind, &
         0.9998515_dbl_kind,  0.9933966_dbl_kind,  0.8227881_dbl_kind, &
         0.9998332_dbl_kind,  0.9926888_dbl_kind,  0.8095131_dbl_kind, &
         0.9998148_dbl_kind,  0.9919968_dbl_kind,  0.7968620_dbl_kind, &
         0.9997691_dbl_kind,  0.9903277_dbl_kind,  0.7677887_dbl_kind/), &
         (/nspint_3bd,nmbrad_snw/))

      ! snow asymmetry parameter (unitless)
      gs_tab = reshape((/ &
          0.859913_dbl_kind,  0.848003_dbl_kind,  0.824415_dbl_kind, &
          0.867130_dbl_kind,  0.858150_dbl_kind,  0.848445_dbl_kind, &
          0.873381_dbl_kind,  0.867221_dbl_kind,  0.861714_dbl_kind, &
          0.878368_dbl_kind,  0.874879_dbl_kind,  0.874036_dbl_kind, &
          0.881462_dbl_kind,  0.879661_dbl_kind,  0.881299_dbl_kind, &
          0.884361_dbl_kind,  0.883903_dbl_kind,  0.890184_dbl_kind, &
          0.885937_dbl_kind,  0.886256_dbl_kind,  0.895393_dbl_kind, &
          0.886931_dbl_kind,  0.887769_dbl_kind,  0.899072_dbl_kind, &
          0.887894_dbl_kind,  0.889255_dbl_kind,  0.903285_dbl_kind, &
          0.888515_dbl_kind,  0.890236_dbl_kind,  0.906588_dbl_kind, &
          0.889073_dbl_kind,  0.891127_dbl_kind,  0.910152_dbl_kind, &
          0.889452_dbl_kind,  0.891750_dbl_kind,  0.913100_dbl_kind, &
          0.889730_dbl_kind,  0.892213_dbl_kind,  0.915621_dbl_kind, &
          0.890026_dbl_kind,  0.892723_dbl_kind,  0.918831_dbl_kind, &
          0.890238_dbl_kind,  0.893099_dbl_kind,  0.921540_dbl_kind, &
          0.890441_dbl_kind,  0.893474_dbl_kind,  0.924581_dbl_kind, &
          0.890618_dbl_kind,  0.893816_dbl_kind,  0.927701_dbl_kind, &
          0.890762_dbl_kind,  0.894123_dbl_kind,  0.930737_dbl_kind, &
          0.890881_dbl_kind,  0.894397_dbl_kind,  0.933568_dbl_kind, &
          0.890975_dbl_kind,  0.894645_dbl_kind,  0.936148_dbl_kind, &
          0.891035_dbl_kind,  0.894822_dbl_kind,  0.937989_dbl_kind, &
          0.891097_dbl_kind,  0.895020_dbl_kind,  0.939949_dbl_kind, &
          0.891147_dbl_kind,  0.895212_dbl_kind,  0.941727_dbl_kind, &
          0.891189_dbl_kind,  0.895399_dbl_kind,  0.943339_dbl_kind, &
          0.891225_dbl_kind,  0.895601_dbl_kind,  0.944915_dbl_kind, &
          0.891248_dbl_kind,  0.895745_dbl_kind,  0.945950_dbl_kind, &
          0.891277_dbl_kind,  0.895951_dbl_kind,  0.947288_dbl_kind, &
          0.891299_dbl_kind,  0.896142_dbl_kind,  0.948438_dbl_kind, &
          0.891323_dbl_kind,  0.896388_dbl_kind,  0.949762_dbl_kind, &
          0.891340_dbl_kind,  0.896623_dbl_kind,  0.950916_dbl_kind, &
          0.891356_dbl_kind,  0.896851_dbl_kind,  0.951945_dbl_kind, &
          0.891386_dbl_kind,  0.897399_dbl_kind,  0.954156_dbl_kind/), &
          (/nspint_3bd,nmbrad_snw/))

      ! ice surface scattering layer (ssl) iops
      ki_ssl_mn_3bd = (/ 1000.1_dbl_kind, 1003.7_dbl_kind, 7042._dbl_kind/)
      wi_ssl_mn_3bd = (/ .9999_dbl_kind,  .9963_dbl_kind,  .9088_dbl_kind/)
      gi_ssl_mn_3bd = (/  .94_dbl_kind,     .94_dbl_kind,    .94_dbl_kind/)

      ! ice drained layer (dl) iops
      ki_dl_mn_3bd = (/ 100.2_dbl_kind, 107.7_dbl_kind,  1309._dbl_kind /)
      wi_dl_mn_3bd = (/ .9980_dbl_kind,  .9287_dbl_kind, .0305_dbl_kind /)
      gi_dl_mn_3bd = (/ .94_dbl_kind,     .94_dbl_kind,    .94_dbl_kind /)

      ! ice interior layer (int) iops
      ki_int_mn_3bd = (/  20.2_dbl_kind,  27.7_dbl_kind,  1445._dbl_kind /)
      wi_int_mn_3bd = (/ .9901_dbl_kind, .7223_dbl_kind,  .0277_dbl_kind /)
      gi_int_mn_3bd = (/ .94_dbl_kind,    .94_dbl_kind,     .94_dbl_kind /)

      ! ponded ice surface scattering layer (ssl) iops
      ki_p_ssl_mn = (/ 70.2_dbl_kind,  77.7_dbl_kind,  1309._dbl_kind/)
      wi_p_ssl_mn = (/ .9972_dbl_kind, .9009_dbl_kind, .0305_dbl_kind/)
      gi_p_ssl_mn = (/ .94_dbl_kind,   .94_dbl_kind,   .94_dbl_kind  /)

      ! ponded ice interior layer (int) iops
      ki_p_int_mn = (/  20.2_dbl_kind,  27.7_dbl_kind, 1445._dbl_kind/)
      wi_p_int_mn = (/ .9901_dbl_kind, .7223_dbl_kind, .0277_dbl_kind/)
      gi_p_int_mn = (/ .94_dbl_kind,   .94_dbl_kind,   .94_dbl_kind  /)

      ! inherent optical property (iop) arrays for pond water and underlying ocean
      ! kw = Pond water extinction coefficient (/m)
      ! ww = Pond water single scattering albedo
      ! gw = Pond water asymmetry parameter
      kw = (/ 0.20_dbl_kind,   12.0_dbl_kind,   729._dbl_kind /)
      ww = (/ 0.00_dbl_kind,   0.00_dbl_kind,   0.00_dbl_kind /)
      gw = (/ 0.00_dbl_kind,   0.00_dbl_kind,   0.00_dbl_kind /)

      !--- extracted from netcdf file
      !--- snicar_optics_5bnd_mam_c140303_mpas_cice (this is actually 3 band)
      !    gaer_bc_3bd (:,:)   = modalAsymmetryParameter(:,:)
      !    kaer_bc_3bd (:,:)   = modalMassExtinctionCrossSection(:,:)
      !    waer_bc_3bd (:,:)   = modalSingleScatterAlbedo(:,:)
      !    bcenh_3bd   (:,:,:) = modalBCabsorptionParameter(:,:,:)
      !
      !--- spectralIntervals = 4.5e-13, 9.45e-13, 3.095e-12 ;
      !--- bcEffectiveRadius = 0.05, 0.1, 0.15, 0.2, 0.25, 0.3, 0.35, 0.4, 0.45, 0.5 ;
      !--- iceGrainEffectiveRadius = 50, 100, 250, 500, 750, 1000, 1250, 1500 ;
      !
      !--- extracted from CICE model, was single precision data statements
      !    gaer_bc (:,:)
      !    kaer_bc (:,:)
      !    waer_bc (:,:)

      allocate(gaer_bc_3bd(nspint_3bd,nmodal1) )
      allocate(kaer_bc_3bd(nspint_3bd,nmodal1) )
      allocate(waer_bc_3bd(nspint_3bd,nmodal1) )
      allocate(gaer_3bd   (nspint_3bd,max_aero))
      allocate(kaer_3bd   (nspint_3bd,max_aero))
      allocate(waer_3bd   (nspint_3bd,max_aero))
      allocate(bcenh_3bd  (nspint_3bd,nmodal1,nmodal2))

      gaer_bc_3bd (:,:)   = reshape((/ &
       0.283282988564031_dbl_kind, 0.196444209821980_dbl_kind, 0.104976473902976_dbl_kind, &
       0.444877326083453_dbl_kind, 0.326147707342261_dbl_kind, 0.196121968923488_dbl_kind, &
       0.547243414035631_dbl_kind, 0.416106187964493_dbl_kind, 0.263903486903711_dbl_kind, &
       0.617111563012282_dbl_kind, 0.484745531707601_dbl_kind, 0.319218974395050_dbl_kind, &
       0.666728525631754_dbl_kind, 0.539228555802301_dbl_kind, 0.366323180358996_dbl_kind, &
       0.702956870835387_dbl_kind, 0.583372441336763_dbl_kind, 0.407316408184865_dbl_kind, &
       0.730016668453191_dbl_kind, 0.619595539349710_dbl_kind, 0.443436944107423_dbl_kind, &
       0.750635997128011_dbl_kind, 0.649589805870541_dbl_kind, 0.475512089138887_dbl_kind, &
       0.766634959089444_dbl_kind, 0.674609076223658_dbl_kind, 0.504145461809103_dbl_kind, &
       0.779256641759228_dbl_kind, 0.695614224933709_dbl_kind, 0.529805346632687_dbl_kind/), &
       (/nspint_3bd,nmodal1/))

      kaer_bc_3bd (:,:)   = reshape((/ &
       12955.4473151973_dbl_kind, 5946.89461205564_dbl_kind, 2772.33366387720_dbl_kind, &
       12085.3066388712_dbl_kind, 7438.83131367992_dbl_kind, 3657.13084442081_dbl_kind, &
       9753.99697536893_dbl_kind, 7342.87139082553_dbl_kind, 4187.79303607928_dbl_kind, &
       7815.74879345131_dbl_kind, 6659.65096365965_dbl_kind, 4337.98863414228_dbl_kind, &
       6381.28194381772_dbl_kind, 5876.78408231865_dbl_kind, 4254.65053724305_dbl_kind, &
       5326.93163497508_dbl_kind, 5156.74531505734_dbl_kind, 4053.66581550147_dbl_kind, &
       4538.09762614960_dbl_kind, 4538.60874501597_dbl_kind, 3804.10884202567_dbl_kind, &
       3934.17604000777_dbl_kind, 4020.20798667897_dbl_kind, 3543.27199302277_dbl_kind, &
       3461.20655708248_dbl_kind, 3587.80961820605_dbl_kind, 3289.98060303894_dbl_kind, &
       3083.03396032095_dbl_kind, 3226.27231329114_dbl_kind, 3052.91440681137_dbl_kind/), &
       (/nspint_3bd,nmodal1/))

      waer_bc_3bd (:,:)   = reshape((/ &
       0.261071919959011_dbl_kind, 0.158608047940651_dbl_kind, 0.0653546447770291_dbl_kind, &
       0.375593873543050_dbl_kind, 0.303181671502553_dbl_kind, 0.194832290545495_dbl_kind, &
       0.422240383488477_dbl_kind, 0.369134186611324_dbl_kind, 0.278752556671685_dbl_kind, &
       0.447772153910671_dbl_kind, 0.405033725319593_dbl_kind, 0.330260831965086_dbl_kind, &
       0.464443094570456_dbl_kind, 0.427439117980081_dbl_kind, 0.364256689383418_dbl_kind, &
       0.476668995985241_dbl_kind, 0.442854173154887_dbl_kind, 0.388270470928338_dbl_kind, &
       0.486347881475941_dbl_kind, 0.454284736567521_dbl_kind, 0.406167596922937_dbl_kind, &
       0.494397834153785_dbl_kind, 0.463279526357470_dbl_kind, 0.420084410794128_dbl_kind, &
       0.501307856563459_dbl_kind, 0.470696914968199_dbl_kind, 0.431284889617716_dbl_kind, &
       0.507362336297419_dbl_kind, 0.477038272961243_dbl_kind, 0.440559363958571_dbl_kind/), &
       (/nspint_3bd,nmodal1/))

      ! tcraig, copied from CICE, promote to dbl_kind
      kaer_3bd = reshape((/ &      ! aerosol mass extinction cross section (m2/kg)
          11580.61872_dbl_kind,   5535.41835_dbl_kind,   2793.79690_dbl_kind, &
          25798.96479_dbl_kind,  11536.03871_dbl_kind,   4688.24207_dbl_kind, &
            196.49772_dbl_kind,    204.14078_dbl_kind,    214.42287_dbl_kind, &
           2665.85867_dbl_kind,   2256.71027_dbl_kind,    820.36024_dbl_kind, &
            840.78295_dbl_kind,   1028.24656_dbl_kind,   1163.03298_dbl_kind, &
            387.51211_dbl_kind,    414.68808_dbl_kind,    450.29814_dbl_kind/), &
            (/nspint_3bd,max_aero/))
      waer_3bd = reshape((/ &      ! aerosol single scatter albedo (fraction)
              0.29003_dbl_kind,      0.17349_dbl_kind,      0.06613_dbl_kind, &
              0.51731_dbl_kind,      0.41609_dbl_kind,      0.21324_dbl_kind, &
              0.84467_dbl_kind,      0.94216_dbl_kind,      0.95666_dbl_kind, &
              0.97764_dbl_kind,      0.99402_dbl_kind,      0.98552_dbl_kind, &
              0.94146_dbl_kind,      0.98527_dbl_kind,      0.99093_dbl_kind, &
              0.90034_dbl_kind,      0.96543_dbl_kind,      0.97678_dbl_kind/), &
              (/nspint_3bd,max_aero/))
      gaer_3bd = reshape((/ &      ! aerosol asymmetry parameter (cos(theta))
              0.35445_dbl_kind,      0.19838_dbl_kind,      0.08857_dbl_kind, &
              0.52581_dbl_kind,      0.32384_dbl_kind,      0.14970_dbl_kind, &
              0.83162_dbl_kind,      0.78306_dbl_kind,      0.74375_dbl_kind, &
              0.68861_dbl_kind,      0.70836_dbl_kind,      0.54171_dbl_kind, &
              0.70239_dbl_kind,      0.66115_dbl_kind,      0.71983_dbl_kind, &
              0.78734_dbl_kind,      0.73580_dbl_kind,      0.64411_dbl_kind/), &
              (/nspint_3bd,max_aero/))

      bcenh_3bd   (:,:,:) = reshape((/ &
       1.95264863821642_dbl_kind, 2.05550877768593_dbl_kind, 1.70240332828865_dbl_kind, &
       1.99163464431877_dbl_kind, 2.02176016911438_dbl_kind, 1.68864818773335_dbl_kind, &
       2.08039249602289_dbl_kind, 2.04220861750766_dbl_kind, 1.66912312377214_dbl_kind, &
       2.16942689206486_dbl_kind, 2.08596540854892_dbl_kind, 1.66160699987308_dbl_kind, &
       2.24943093516846_dbl_kind, 2.13660234877153_dbl_kind, 1.66384052314060_dbl_kind, &
       2.31967028980740_dbl_kind, 2.18749561126022_dbl_kind, 1.67219028250399_dbl_kind, &
       2.38127471771344_dbl_kind, 2.23610708358262_dbl_kind, 1.68413091787831_dbl_kind, &
       2.43560783392705_dbl_kind, 2.28159277966039_dbl_kind, 1.69809375282139_dbl_kind, &
       2.48387711610947_dbl_kind, 2.32381536608936_dbl_kind, 1.71312401386438_dbl_kind, &
       2.52707191213977_dbl_kind, 2.36292350082796_dbl_kind, 1.72863990449135_dbl_kind, &
       1.92998070985211_dbl_kind, 2.00057297786255_dbl_kind, 1.46439607292737_dbl_kind, &
       1.96828588712172_dbl_kind, 1.96789325006899_dbl_kind, 1.45080052583184_dbl_kind, &
       2.05617049294061_dbl_kind, 1.98793085930286_dbl_kind, 1.42948868363647_dbl_kind, &
       2.14447875560037_dbl_kind, 2.03061959947511_dbl_kind, 1.41873460329624_dbl_kind, &
       2.22391980172086_dbl_kind, 2.07998145729518_dbl_kind, 1.41704273834779_dbl_kind, &
       2.29372360046412_dbl_kind, 2.12957637725258_dbl_kind, 1.42123091826813_dbl_kind, &
       2.35498486656548_dbl_kind, 2.17693833665883_dbl_kind, 1.42900797904431_dbl_kind, &
       2.40904101847001_dbl_kind, 2.22124880688538_dbl_kind, 1.43892088962935_dbl_kind, &
       2.45708129191118_dbl_kind, 2.26237623779669_dbl_kind, 1.45006788911609_dbl_kind, &
       2.50008250129713_dbl_kind, 2.30046686882928_dbl_kind, 1.46188722662250_dbl_kind, &
       1.91329346734012_dbl_kind, 1.96543390843360_dbl_kind, 1.14194244916007_dbl_kind, &
       1.95121371985675_dbl_kind, 1.93263565295399_dbl_kind, 1.12725592122175_dbl_kind, &
       2.03809762910398_dbl_kind, 1.95185585527707_dbl_kind, 1.10247502663803_dbl_kind, &
       2.12530005514579_dbl_kind, 1.99345975452191_dbl_kind, 1.08665719062047_dbl_kind, &
       2.20367089050177_dbl_kind, 2.04169792795632_dbl_kind, 1.07921619336383_dbl_kind, &
       2.27248123187072_dbl_kind, 2.09021776382565_dbl_kind, 1.07748810634133_dbl_kind, &
       2.33283448215489_dbl_kind, 2.13658191370477_dbl_kind, 1.07944256077146_dbl_kind, &
       2.38606519288941_dbl_kind, 2.17997684658703_dbl_kind, 1.08374903741219_dbl_kind, &
       2.43335573206486_dbl_kind, 2.22026683524201_dbl_kind, 1.08955635350460_dbl_kind, &
       2.47567508771849_dbl_kind, 2.25759085163530_dbl_kind, 1.09631629142051_dbl_kind, &
       1.90934210233419_dbl_kind, 1.95036222309469_dbl_kind, 0.909292665461334_dbl_kind, &
       1.94718753993654_dbl_kind, 1.91707244220432_dbl_kind, 0.894718400010399_dbl_kind, &
       2.03413257206460_dbl_kind, 1.93555229772563_dbl_kind, 0.868918153028977_dbl_kind, &
       2.12144970426002_dbl_kind, 1.97639279591636_dbl_kind, 0.850765821245711_dbl_kind, &
       2.19996950008693_dbl_kind, 2.02391750274938_dbl_kind, 0.840259615891209_dbl_kind, &
       2.26895256024478_dbl_kind, 2.07179265196092_dbl_kind, 0.835148764099313_dbl_kind, &
       2.32949533824322_dbl_kind, 2.11758088518527_dbl_kind, 0.833634959766446_dbl_kind, &
       2.38292734368215_dbl_kind, 2.16046235899349_dbl_kind, 0.834510361060840_dbl_kind, &
       2.43042710618676_dbl_kind, 2.20029348070725_dbl_kind, 0.836985635757155_dbl_kind, &
       2.47296056325651_dbl_kind, 2.23720560395594_dbl_kind, 0.840541121927357_dbl_kind, &
       1.89949243983108_dbl_kind, 1.93683846999834_dbl_kind, 0.787365738156841_dbl_kind, &
       1.93727235487729_dbl_kind, 1.90329555375166_dbl_kind, 0.773444742139757_dbl_kind, &
       2.02369838860644_dbl_kind, 1.92122803438124_dbl_kind, 0.748247354378510_dbl_kind, &
       2.11044213773949_dbl_kind, 1.96146568420286_dbl_kind, 0.729898511422122_dbl_kind, &
       2.18840765391269_dbl_kind, 2.00841288916682_dbl_kind, 0.718628437773005_dbl_kind, &
       2.25686720991682_dbl_kind, 2.05575959668218_dbl_kind, 0.712430381055335_dbl_kind, &
       2.31691587317745_dbl_kind, 2.10107207746932_dbl_kind, 0.709662234578170_dbl_kind, &
       2.36987969534290_dbl_kind, 2.14352706751582_dbl_kind, 0.709208099450235_dbl_kind, &
       2.41693409934405_dbl_kind, 2.18297543800191_dbl_kind, 0.710331652717846_dbl_kind, &
       2.45904265244829_dbl_kind, 2.21954279828598_dbl_kind, 0.712543468490739_dbl_kind, &
       1.89513078257241_dbl_kind, 1.92263839595509_dbl_kind, 0.706461800324346_dbl_kind, &
       1.93309313777527_dbl_kind, 1.88899078061693_dbl_kind, 0.693226914812576_dbl_kind, &
       2.01957651148981_dbl_kind, 1.90645089504406_dbl_kind, 0.669064908581365_dbl_kind, &
       2.10637427535360_dbl_kind, 1.94612617504146_dbl_kind, 0.651180172594725_dbl_kind, &
       2.18442157999672_dbl_kind, 1.99251964154984_dbl_kind, 0.639897070268354_dbl_kind, &
       2.25299528565588_dbl_kind, 2.03935253218284_dbl_kind, 0.633381538773347_dbl_kind, &
       2.31318616048548_dbl_kind, 2.08419838569356_dbl_kind, 0.630113597076398_dbl_kind, &
       2.36631356022073_dbl_kind, 2.12623247442447_dbl_kind, 0.629053974248532_dbl_kind, &
       2.41354632072806_dbl_kind, 2.16530128719337_dbl_kind, 0.629513431452439_dbl_kind, &
       2.45584239569185_dbl_kind, 2.20152541593779_dbl_kind, 0.631031550232408_dbl_kind, &
       1.88966200391793_dbl_kind, 1.91138224467403_dbl_kind, 0.645037746856209_dbl_kind, &
       1.92722145035491_dbl_kind, 1.87746007578606_dbl_kind, 0.632544812562624_dbl_kind, &
       2.01312358620330_dbl_kind, 1.89440019770827_dbl_kind, 0.609599412746477_dbl_kind, &
       2.09932210400679_dbl_kind, 1.93352303806653_dbl_kind, 0.592457728466318_dbl_kind, &
       2.17678426739086_dbl_kind, 1.97939606834949_dbl_kind, 0.581485988000951_dbl_kind, &
       2.24479350026512_dbl_kind, 2.02575746617653_dbl_kind, 0.574991121329954_dbl_kind, &
       2.30444200946822_dbl_kind, 2.07018181146210_dbl_kind, 0.571558829839628_dbl_kind, &
       2.35704963947562_dbl_kind, 2.11184012144586_dbl_kind, 0.570218484962620_dbl_kind, &
       2.40378548089834_dbl_kind, 2.15057324446996_dbl_kind, 0.570324425190625_dbl_kind, &
       2.44560753617498_dbl_kind, 2.18649621439102_dbl_kind, 0.571444075492751_dbl_kind, &
       1.88643160779142_dbl_kind, 1.89913501159848_dbl_kind, 0.595263574413264_dbl_kind, &
       1.92407257560116_dbl_kind, 1.86500658393517_dbl_kind, 0.583463057793767_dbl_kind, &
       2.00998096470402_dbl_kind, 1.88145449003472_dbl_kind, 0.561733888485346_dbl_kind, &
       2.09618177016992_dbl_kind, 1.92003164909457_dbl_kind, 0.545410325196173_dbl_kind, &
       2.17365367623383_dbl_kind, 1.96538149140099_dbl_kind, 0.534870485223569_dbl_kind, &
       2.24167959830890_dbl_kind, 2.01126422674813_dbl_kind, 0.528539343400941_dbl_kind, &
       2.30134946305542_dbl_kind, 2.05525778466667_dbl_kind, 0.525094898482761_dbl_kind, &
       2.35398116406859_dbl_kind, 2.09653013835900_dbl_kind, 0.523627801843702_dbl_kind, &
       2.40074243898899_dbl_kind, 2.13491705468600_dbl_kind, 0.523532027540127_dbl_kind, &
       2.44259042500067_dbl_kind, 2.17052832643712_dbl_kind, 0.524400814046362_dbl_kind/), &
       (/nspint_3bd,nmodal1,nmodal2/))

      end subroutine icepack_shortwave_init_dEdd3band

!=======================================================================

      subroutine icepack_shortwave_init_dEdd5band

      ! local variables

      integer (kind=int_kind) :: n

      character (len=*),parameter :: subname='(icepack_shortwave_init_dEdd5band)'

      ! The 5band data given in this section are based on CICE and SNICAR band choice:
      ! SNICAR band 1 = CICE band 1
      ! SNICAR band 2 + SNICAR band 3 = CICE band 2
      ! SNICAR band 4 + SNICAR band 5 = CICE band 3

      ! ice surface scattering layer (ssl) iops
      ki_ssl_mn_5bd = (/ 1000.1_dbl_kind, 1003.7_dbl_kind, 1003.7_dbl_kind, &
                         7042._dbl_kind, 7042._dbl_kind /)
      wi_ssl_mn_5bd = (/ .9999_dbl_kind, .9963_dbl_kind, .9963_dbl_kind, &
                         .9088_dbl_kind, .9088_dbl_kind /)
      gi_ssl_mn_5bd = (/ .94_dbl_kind, .94_dbl_kind, .94_dbl_kind, &
                         .94_dbl_kind, .94_dbl_kind /)

      ! ice drained layer (dl) iops
      ki_dl_mn_5bd = (/ 100.2_dbl_kind, 107.7_dbl_kind, 107.7_dbl_kind, &
                        1309._dbl_kind, 1309._dbl_kind /)
      wi_dl_mn_5bd = (/ .9980_dbl_kind, .9287_dbl_kind, .9287_dbl_kind, &
                        .0305_dbl_kind, .0305_dbl_kind /)
      gi_dl_mn_5bd = (/ .94_dbl_kind, .94_dbl_kind, .94_dbl_kind, &
                        .94_dbl_kind, .94_dbl_kind /)

      ! ice interior layer (int) iops
      ki_int_mn_5bd = (/ 20.2_dbl_kind, 27.7_dbl_kind, 27.7_dbl_kind, &
                         1445._dbl_kind, 1445._dbl_kind/)
      wi_int_mn_5bd = (/ .9901_dbl_kind, .7223_dbl_kind, .7223_dbl_kind, &
                         .0277_dbl_kind, .0277_dbl_kind /)
      gi_int_mn_5bd = (/ .94_dbl_kind, .94_dbl_kind, .94_dbl_kind, &
                         .94_dbl_kind, .94_dbl_kind /)

      !--- extracted from netcdf file
      !--- snicar_optics_5bnd_snow_and_aerosols.nc
      !   gaer_bc_5bd (:,:)   = modalAsymmetryParameter5band(:,:)
      !   kaer_bc_5bd (:,:)   = modalMassExtinctionCrossSection5band(:,:)
      !   waer_bc_5bd (:,:)   = modalSingleScatterAlbedo5band(:,:)
      !   gaer_5bd    (:,:)   = aerosolAsymmetryParameter5band(:,:)
      !   kaer_5bd    (:,:)   = aerosolMassExtinctionCrossSection5band(:,:)
      !   waer_5bd    (:,:)   = aerosolSingleScatterAlbedo5band(:,:)
      !   bcenh_5bd   (:,:,:) = modalBCabsorptionParameter5band(:,:,:)

      allocate(gaer_bc_5bd(nspint_5bd,nmodal1) )
      allocate(kaer_bc_5bd(nspint_5bd,nmodal1) )
      allocate(waer_bc_5bd(nspint_5bd,nmodal1) )
      allocate(gaer_5bd   (nspint_5bd,max_aero))
      allocate(kaer_5bd   (nspint_5bd,max_aero))
      allocate(waer_5bd   (nspint_5bd,max_aero))
      allocate(bcenh_5bd  (nspint_5bd,nmodal1,nmodal2))

      gaer_bc_5bd(1:5 ,1) = (/ 0.283180657713357_dbl_kind, 0.209482088735706_dbl_kind, 0.166441061642889_dbl_kind, 0.138879205492097_dbl_kind, 0.0862017497093632_dbl_kind /)
      gaer_bc_5bd(1:5 ,2) = (/ 0.444732109704644_dbl_kind, 0.342651789871816_dbl_kind, 0.285303488980226_dbl_kind, 0.247698976619574_dbl_kind, 0.167948761949466_dbl_kind /)
      gaer_bc_5bd(1:5 ,3) = (/ 0.547091281292997_dbl_kind, 0.433969936661441_dbl_kind, 0.368367957620211_dbl_kind, 0.324886781461864_dbl_kind, 0.229171714711908_dbl_kind /)
      gaer_bc_5bd(1:5 ,4) = (/ 0.616967685845435_dbl_kind, 0.503089780780303_dbl_kind, 0.433284180500838_dbl_kind, 0.386200764739288_dbl_kind, 0.279644654892079_dbl_kind /)
      gaer_bc_5bd(1:5 ,5) = (/ 0.666598670419791_dbl_kind, 0.557497118318872_dbl_kind, 0.486364162004934_dbl_kind, 0.437315965706898_dbl_kind, 0.32323402731446_dbl_kind /)
      gaer_bc_5bd(1:5 ,6) = (/ 0.702842367361724_dbl_kind, 0.601186554981124_dbl_kind, 0.530715174055473_dbl_kind, 0.480926756372334_dbl_kind, 0.361764440902239_dbl_kind /)
      gaer_bc_5bd(1:5 ,7) = (/ 0.729916826617441_dbl_kind, 0.636706604014777_dbl_kind, 0.568216759394959_dbl_kind, 0.5185983947025_dbl_kind, 0.396259725869959_dbl_kind /)
      gaer_bc_5bd(1:5 ,8) = (/ 0.750549324881687_dbl_kind, 0.665849031271645_dbl_kind, 0.60016461579108_dbl_kind, 0.551378204757893_dbl_kind, 0.427376921191516_dbl_kind /)
      gaer_bc_5bd(1:5 ,9) = (/ 0.766559757465102_dbl_kind, 0.689942199993075_dbl_kind, 0.627527163812553_dbl_kind, 0.580037712812519_dbl_kind, 0.455582792418271_dbl_kind /)
      gaer_bc_5bd(1:5 ,10) = (/ 0.779191279240598_dbl_kind, 0.71000026362677_dbl_kind, 0.651064109101405_dbl_kind, 0.605182083087546_dbl_kind, 0.48123462598527_dbl_kind /)

      kaer_bc_5bd(1:5 ,1) = (/ 12944.1191669303_dbl_kind, 6433.8546129199_dbl_kind, 4379.35815079177_dbl_kind, 3492.51850776132_dbl_kind, 2271.71785345005_dbl_kind /)
      kaer_bc_5bd(1:5 ,2) = (/ 12080.1397436577_dbl_kind, 7942.86489089621_dbl_kind, 5812.47284115357_dbl_kind, 4685.4656136674_dbl_kind, 2943.75841342826_dbl_kind /)
      kaer_bc_5bd(1:5 ,3) = (/ 9752.02484228556_dbl_kind, 7698.94532514882_dbl_kind, 6190.90781492957_dbl_kind, 5232.51964498223_dbl_kind, 3463.69216361937_dbl_kind /)
      kaer_bc_5bd(1:5 ,4) = (/ 7815.10345723545_dbl_kind, 6881.58135306449_dbl_kind, 5939.77102314034_dbl_kind, 5240.81549971547_dbl_kind, 3712.12933475335_dbl_kind /)
      kaer_bc_5bd(1:5 ,5) = (/ 6381.18061078149_dbl_kind, 6007.04138576162_dbl_kind, 5453.04529077459_dbl_kind, 4977.11342455862_dbl_kind, 3753.42937518735_dbl_kind /)
      kaer_bc_5bd(1:5 ,6) = (/ 5327.05076509938_dbl_kind, 5229.1647601636_dbl_kind, 4920.29504756471_dbl_kind, 4609.03071475928_dbl_kind, 3667.92739329671_dbl_kind /)
      kaer_bc_5bd(1:5 ,7) = (/ 4538.29991082855_dbl_kind, 4575.50607606948_dbl_kind, 4417.44045147505_dbl_kind, 4220.99458200724_dbl_kind, 3514.14534131843_dbl_kind /)
      kaer_bc_5bd(1:5 ,8) = (/ 3934.40205337053_dbl_kind, 4035.52102830084_dbl_kind, 3969.24444889817_dbl_kind, 3850.95782137369_dbl_kind, 3328.91108844436_dbl_kind /)
      kaer_bc_5bd(1:5 ,9) = (/ 3461.43069443711_dbl_kind, 3590.1138938721_dbl_kind, 3579.26043074527_dbl_kind, 3513.60273858667_dbl_kind, 3133.888122911_dbl_kind /)
      kaer_bc_5bd(1:5 ,10) = (/ 3083.24574223473_dbl_kind, 3220.83851695162_dbl_kind, 3243.02488284297_dbl_kind, 3212.5874258974_dbl_kind, 2941.20365863075_dbl_kind /)

      waer_bc_5bd(1:5 ,1) = (/ 0.260958704071654_dbl_kind, 0.16805845111727_dbl_kind, 0.11411468355425_dbl_kind, 0.0845902711417414_dbl_kind, 0.0447050517464244_dbl_kind /)
      waer_bc_5bd(1:5 ,2) = (/ 0.375522113298175_dbl_kind, 0.31235052399053_dbl_kind, 0.26288498209907_dbl_kind, 0.22723691750397_dbl_kind, 0.159075641626696_dbl_kind /)
      waer_bc_5bd(1:5 ,3) = (/ 0.422185378147481_dbl_kind, 0.376555779043625_dbl_kind, 0.33936407097995_dbl_kind, 0.310215627605754_dbl_kind, 0.245905521445928_dbl_kind /)
      waer_bc_5bd(1:5 ,4) = (/ 0.447724362399737_dbl_kind, 0.411125469751176_dbl_kind, 0.382216922622453_dbl_kind, 0.358710570753989_dbl_kind, 0.302535187827444_dbl_kind /)
      waer_bc_5bd(1:5 ,5) = (/ 0.464398385288602_dbl_kind, 0.432573637322427_dbl_kind, 0.40912325097991_dbl_kind, 0.389777582190663_dbl_kind, 0.340897831337878_dbl_kind /)
      waer_bc_5bd(1:5 ,6) = (/ 0.476625494212487_dbl_kind, 0.447308005754231_dbl_kind, 0.427510535024552_dbl_kind, 0.411226212689519_dbl_kind, 0.368348762641282_dbl_kind /)
      waer_bc_5bd(1:5 ,7) = (/ 0.486304785993923_dbl_kind, 0.458261662359298_dbl_kind, 0.440925436651717_dbl_kind, 0.426916054656286_dbl_kind, 0.38894085673284_dbl_kind /)
      waer_bc_5bd(1:5 ,8) = (/ 0.494354868140575_dbl_kind, 0.4669281500285_dbl_kind, 0.451247969458627_dbl_kind, 0.43894010429319_dbl_kind, 0.40499178204751_dbl_kind /)
      waer_bc_5bd(1:5 ,9) = (/ 0.501264988054309_dbl_kind, 0.47412360289998_dbl_kind, 0.459551339156444_dbl_kind, 0.448519783558473_dbl_kind, 0.417901240195357_dbl_kind /)
      waer_bc_5bd(1:5 ,10) = (/ 0.507319636821098_dbl_kind, 0.480317417860846_dbl_kind, 0.466481312779405_dbl_kind, 0.456408044720941_dbl_kind, 0.428557864653526_dbl_kind /)

      bcenh_5bd(1:5 ,1,1) = (/ 2.01085868366548_dbl_kind, 2.04133313210786_dbl_kind, 2.07508000300547_dbl_kind, 1.96633438024084_dbl_kind, 1.41689382160734_dbl_kind /)
      bcenh_5bd(1:5 ,2,1) = (/ 2.06240193763163_dbl_kind, 2.00779624447446_dbl_kind, 2.03535206238544_dbl_kind, 1.93696335633222_dbl_kind, 1.42085892146856_dbl_kind /)
      bcenh_5bd(1:5 ,3,1) = (/ 2.16245417515345_dbl_kind, 2.03367512749659_dbl_kind, 2.03217185662912_dbl_kind, 1.92216538758257_dbl_kind, 1.41057880313765_dbl_kind /)
      bcenh_5bd(1:5 ,4,1) = (/ 2.26075013581323_dbl_kind, 2.08270396815669_dbl_kind, 2.05598320187322_dbl_kind, 1.93005028646658_dbl_kind, 1.40362921606229_dbl_kind /)
      bcenh_5bd(1:5 ,5,1) = (/ 2.34824623202129_dbl_kind, 2.13751008192628_dbl_kind, 2.09204530638863_dbl_kind, 1.95135851042228_dbl_kind, 1.4023767146298_dbl_kind /)
      bcenh_5bd(1:5 ,6,1) = (/ 2.4245821026032_dbl_kind, 2.19151236157797_dbl_kind, 2.13262602986381_dbl_kind, 1.97935233613122_dbl_kind, 1.40560606928656_dbl_kind /)
      bcenh_5bd(1:5 ,7,1) = (/ 2.49120967497054_dbl_kind, 2.24239779808293_dbl_kind, 2.17377807033259_dbl_kind, 2.01017733622214_dbl_kind, 1.41189694037042_dbl_kind /)
      bcenh_5bd(1:5 ,8,1) = (/ 2.54974120400747_dbl_kind, 2.2895387842995_dbl_kind, 2.21382769032984_dbl_kind, 2.04172990500999_dbl_kind, 1.42017827052267_dbl_kind /)
      bcenh_5bd(1:5 ,9,1) = (/ 2.60156969843265_dbl_kind, 2.33296656241086_dbl_kind, 2.25201772913892_dbl_kind, 2.07288930404508_dbl_kind, 1.42971387107226_dbl_kind /)
      bcenh_5bd(1:5 ,10,1) = (/ 2.64782205756202_dbl_kind, 2.37295227907137_dbl_kind, 2.28808322951764_dbl_kind, 2.10308108574029_dbl_kind, 1.44000988888373_dbl_kind /)
      bcenh_5bd(1:5 ,1,2) = (/ 1.93095248830255_dbl_kind, 2.00671847586146_dbl_kind, 2.00719477963142_dbl_kind, 1.84522141815519_dbl_kind, 1.0759084306113_dbl_kind /)
      bcenh_5bd(1:5 ,2,2) = (/ 1.9692253934491_dbl_kind, 1.97261097531431_dbl_kind, 1.97018044809597_dbl_kind, 1.81698805576196_dbl_kind, 1.07910761278501_dbl_kind /)
      bcenh_5bd(1:5 ,3,2) = (/ 2.0569305238012_dbl_kind, 1.99733517977522_dbl_kind, 1.96791189681145_dbl_kind, 1.80196522204901_dbl_kind, 1.07130151751929_dbl_kind /)
      bcenh_5bd(1:5 ,4,2) = (/ 2.14496096212491_dbl_kind, 2.04502142936022_dbl_kind, 1.99149039012804_dbl_kind, 1.80835190349595_dbl_kind, 1.06588237810724_dbl_kind /)
      bcenh_5bd(1:5 ,5,2) = (/ 2.22407771486646_dbl_kind, 2.09850387281167_dbl_kind, 2.02679018049782_dbl_kind, 1.82750693049481_dbl_kind, 1.06472566849715_dbl_kind /)
      bcenh_5bd(1:5 ,6,2) = (/ 2.29354361051981_dbl_kind, 2.15127410474977_dbl_kind, 2.06629477010126_dbl_kind, 1.85307952087923_dbl_kind, 1.06695130591868_dbl_kind /)
      bcenh_5bd(1:5 ,7,2) = (/ 2.35447198629787_dbl_kind, 2.20103710784166_dbl_kind, 2.10637650326798_dbl_kind, 1.88142263401342_dbl_kind, 1.07150133206641_dbl_kind /)
      bcenh_5bd(1:5 ,8,2) = (/ 2.40820994383032_dbl_kind, 2.24716243339933_dbl_kind, 2.14535203369221_dbl_kind, 1.91053883285925_dbl_kind, 1.07757202807584_dbl_kind /)
      bcenh_5bd(1:5 ,9,2) = (/ 2.45595109315931_dbl_kind, 2.28967140910979_dbl_kind, 2.18249764512321_dbl_kind, 1.93935882511077_dbl_kind, 1.08460855322213_dbl_kind /)
      bcenh_5bd(1:5 ,10,2) = (/ 2.49867376345289_dbl_kind, 2.32882370197717_dbl_kind, 2.21756305829645_dbl_kind, 1.96733017018785_dbl_kind, 1.09223739315287_dbl_kind /)
      bcenh_5bd(1:5 ,1,3) = (/ 1.91585309392631_dbl_kind, 1.96967352324366_dbl_kind, 1.96009968122517_dbl_kind, 1.68080965604958_dbl_kind, 0.59179696632248_dbl_kind /)
      bcenh_5bd(1:5 ,2,3) = (/ 1.95350884507749_dbl_kind, 1.93742852972506_dbl_kind, 1.92365694040103_dbl_kind, 1.65420258116929_dbl_kind, 0.593740739460229_dbl_kind /)
      bcenh_5bd(1:5 ,3,3) = (/ 2.04035073356608_dbl_kind, 1.96236202642718_dbl_kind, 1.92122593826873_dbl_kind, 1.6389418244479_dbl_kind, 0.589846868165608_dbl_kind /)
      bcenh_5bd(1:5 ,4,3) = (/ 2.12759603009265_dbl_kind, 2.00961752368397_dbl_kind, 1.94409212769693_dbl_kind, 1.6433504376276_dbl_kind, 0.587192323341285_dbl_kind /)
      bcenh_5bd(1:5 ,5,3) = (/ 2.20604474780993_dbl_kind, 2.06245283901475_dbl_kind, 1.97843797212714_dbl_kind, 1.65962990443543_dbl_kind, 0.586790173437158_dbl_kind /)
      bcenh_5bd(1:5 ,6,3) = (/ 2.27494671729217_dbl_kind, 2.11452075480442_dbl_kind, 2.01700243919087_dbl_kind, 1.68195669719917_dbl_kind, 0.588180543026505_dbl_kind /)
      bcenh_5bd(1:5 ,7,3) = (/ 2.33539514630735_dbl_kind, 2.16358888576982_dbl_kind, 2.05606743772564_dbl_kind, 1.70696657730997_dbl_kind, 0.590803092976719_dbl_kind /)
      bcenh_5bd(1:5 ,8,3) = (/ 2.38871953142276_dbl_kind, 2.20905043144983_dbl_kind, 2.09406451723796_dbl_kind, 1.7328071260566_dbl_kind, 0.594230523360484_dbl_kind /)
      bcenh_5bd(1:5 ,9,3) = (/ 2.43609975131583_dbl_kind, 2.25093431540464_dbl_kind, 2.1302842143625_dbl_kind, 1.75847955247898_dbl_kind, 0.598167328207445_dbl_kind /)
      bcenh_5bd(1:5 ,10,3) = (/ 2.47850378585028_dbl_kind, 2.28950111063737_dbl_kind, 2.16448018258101_dbl_kind, 1.78346181192778_dbl_kind, 0.602414429295388_dbl_kind /)
      bcenh_5bd(1:5 ,1,4) = (/ 1.90672644767849_dbl_kind, 1.95728818232617_dbl_kind, 1.91897760144748_dbl_kind, 1.52351339105988_dbl_kind, 0.2856750536761_dbl_kind /)
      bcenh_5bd(1:5 ,2,4) = (/ 1.94433061417967_dbl_kind, 1.92520391124637_dbl_kind, 1.88324629837065_dbl_kind, 1.49885892176041_dbl_kind, 0.286675433333491_dbl_kind /)
      bcenh_5bd(1:5 ,3,4) = (/ 2.03088473930652_dbl_kind, 1.94993972535292_dbl_kind, 1.88079705335263_dbl_kind, 1.4841126290055_dbl_kind, 0.285045919810642_dbl_kind /)
      bcenh_5bd(1:5 ,4,4) = (/ 2.11781824183707_dbl_kind, 1.99686494434747_dbl_kind, 1.90313037410511_dbl_kind, 1.48729958815793_dbl_kind, 0.283994128672286_dbl_kind /)
      bcenh_5bd(1:5 ,5,4) = (/ 2.19597517944483_dbl_kind, 2.04934129186202_dbl_kind, 1.93671708289151_dbl_kind, 1.50138740089174_dbl_kind, 0.283979887790801_dbl_kind /)
      bcenh_5bd(1:5 ,6,4) = (/ 2.26461303017531_dbl_kind, 2.10106009736148_dbl_kind, 1.97435597077173_dbl_kind, 1.521073249671_dbl_kind, 0.284789256577519_dbl_kind /)
      bcenh_5bd(1:5 ,7,4) = (/ 2.3248240816686_dbl_kind, 2.14980196580654_dbl_kind, 2.01256991861794_dbl_kind, 1.54328272090356_dbl_kind, 0.286162594084878_dbl_kind /)
      bcenh_5bd(1:5 ,8,4) = (/ 2.3779349732592_dbl_kind, 2.1949631106279_dbl_kind, 2.04974391892188_dbl_kind, 1.56631751882822_dbl_kind, 0.287901980282528_dbl_kind /)
      bcenh_5bd(1:5 ,9,4) = (/ 2.42512255483062_dbl_kind, 2.23657164478165_dbl_kind, 2.08518221689877_dbl_kind, 1.58925799258076_dbl_kind, 0.2898706320179_dbl_kind /)
      bcenh_5bd(1:5 ,10,4) = (/ 2.46735205354131_dbl_kind, 2.274886005391_dbl_kind, 2.1186427508733_dbl_kind, 1.61162011811624_dbl_kind, 0.291976528925237_dbl_kind /)
      bcenh_5bd(1:5 ,1,5) = (/ 1.89867647191471_dbl_kind, 1.94807456713651_dbl_kind, 1.881998255131_dbl_kind, 1.4042673172787_dbl_kind, 0.160591841632584_dbl_kind /)
      bcenh_5bd(1:5 ,2,5) = (/ 1.93619524467095_dbl_kind, 1.91604124756277_dbl_kind, 1.84690908713309_dbl_kind, 1.38128161669944_dbl_kind, 0.161178869198082_dbl_kind /)
      bcenh_5bd(1:5 ,3,5) = (/ 2.0223409055861_dbl_kind, 1.94057626469304_dbl_kind, 1.84443760681477_dbl_kind, 1.3673028890037_dbl_kind, 0.160360280890161_dbl_kind /)
      bcenh_5bd(1:5 ,4,5) = (/ 2.10882091872199_dbl_kind, 1.98721817857084_dbl_kind, 1.86628271260137_dbl_kind, 1.36990699665007_dbl_kind, 0.159858752620932_dbl_kind /)
      bcenh_5bd(1:5 ,5,5) = (/ 2.18654488079119_dbl_kind, 2.03940039330662_dbl_kind, 1.89917614160759_dbl_kind, 1.38261979356805_dbl_kind, 0.159921181308761_dbl_kind /)
      bcenh_5bd(1:5 ,6,5) = (/ 2.25478556857455_dbl_kind, 2.09083933491043_dbl_kind, 1.93605537053396_dbl_kind, 1.40054119827002_dbl_kind, 0.160430275376893_dbl_kind /)
      bcenh_5bd(1:5 ,7,5) = (/ 2.31463677856253_dbl_kind, 2.13932311562854_dbl_kind, 1.97350315915034_dbl_kind, 1.42082635090228_dbl_kind, 0.161244236354913_dbl_kind /)
      bcenh_5bd(1:5 ,8,5) = (/ 2.36742266598385_dbl_kind, 2.18424880883966_dbl_kind, 2.00993636804643_dbl_kind, 1.44190170672523_dbl_kind, 0.162255036646285_dbl_kind /)
      bcenh_5bd(1:5 ,9,5) = (/ 2.41431655157797_dbl_kind, 2.22564299805269_dbl_kind, 2.04467151274813_dbl_kind, 1.46291363103416_dbl_kind, 0.163388069865313_dbl_kind /)
      bcenh_5bd(1:5 ,10,5) = (/ 2.45628005869902_dbl_kind, 2.26376189572524_dbl_kind, 2.07747034796467_dbl_kind, 1.48341153176869_dbl_kind, 0.164593206197625_dbl_kind /)
      bcenh_5bd(1:5 ,1,6) = (/ 1.89418663016821_dbl_kind, 1.94091710412984_dbl_kind, 1.84622848287114_dbl_kind, 1.30255089681936_dbl_kind, 0.098483858462836_dbl_kind /)
      bcenh_5bd(1:5 ,2,6) = (/ 1.93159232417903_dbl_kind, 1.90886115724064_dbl_kind, 1.81172529149757_dbl_kind, 1.28111548264923_dbl_kind, 0.0988562059622643_dbl_kind /)
      bcenh_5bd(1:5 ,3,6) = (/ 2.01753914444024_dbl_kind, 1.93319311880197_dbl_kind, 1.80921451143445_dbl_kind, 1.26796068554138_dbl_kind, 0.0983978116731492_dbl_kind /)
      bcenh_5bd(1:5 ,4,6) = (/ 2.10383351130493_dbl_kind, 1.97958216029892_dbl_kind, 1.83057653581059_dbl_kind, 1.27021211695359_dbl_kind, 0.0981299700083134_dbl_kind /)
      bcenh_5bd(1:5 ,5,6) = (/ 2.18140322971526_dbl_kind, 2.03149297511205_dbl_kind, 1.86279162835321_dbl_kind, 1.28187019646779_dbl_kind, 0.098199185372061_dbl_kind /)
      bcenh_5bd(1:5 ,6,6) = (/ 2.24951550749452_dbl_kind, 2.08269363694761_dbl_kind, 1.89892485058734_dbl_kind, 1.29838388707867_dbl_kind, 0.0985349399895448_dbl_kind /)
      bcenh_5bd(1:5 ,7,6) = (/ 2.30926076563282_dbl_kind, 2.13096027537119_dbl_kind, 1.93562599960626_dbl_kind, 1.3171088717836_dbl_kind, 0.0990521610544031_dbl_kind /)
      bcenh_5bd(1:5 ,8,6) = (/ 2.36195834728809_dbl_kind, 2.17568957538176_dbl_kind, 1.97133789840367_dbl_kind, 1.33658147936655_dbl_kind, 0.0996860550013943_dbl_kind /)
      bcenh_5bd(1:5 ,9,6) = (/ 2.40877775516896_dbl_kind, 2.21690619911267_dbl_kind, 2.0053887500366_dbl_kind, 1.35600690986615_dbl_kind, 0.100391895795296_dbl_kind /)
      bcenh_5bd(1:5 ,10,6) = (/ 2.45067776198751_dbl_kind, 2.25486408585975_dbl_kind, 2.03754387915688_dbl_kind, 1.37496497414564_dbl_kind, 0.101139658510172_dbl_kind /)
      bcenh_5bd(1:5 ,1,7) = (/ 1.8898407988674_dbl_kind, 1.93349866190539_dbl_kind, 1.81165805631067_dbl_kind, 1.21086211627226_dbl_kind, 0.0639927299752619_dbl_kind /)
      bcenh_5bd(1:5 ,2,7) = (/ 1.92726743592625_dbl_kind, 1.90143347118491_dbl_kind, 1.77772917537379_dbl_kind, 1.19106001747711_dbl_kind, 0.064241780706825_dbl_kind /)
      bcenh_5bd(1:5 ,3,7) = (/ 2.01310466603341_dbl_kind, 1.92556961967173_dbl_kind, 1.77518652640106_dbl_kind, 1.17869634139483_dbl_kind, 0.0639657849860226_dbl_kind /)
      bcenh_5bd(1:5 ,4,7) = (/ 2.09927399606475_dbl_kind, 1.97170510850255_dbl_kind, 1.79608613617107_dbl_kind, 1.18057464600288_dbl_kind, 0.0638113078299023_dbl_kind /)
      bcenh_5bd(1:5 ,5,7) = (/ 2.17672489152732_dbl_kind, 2.02337778137591_dbl_kind, 1.82764917978241_dbl_kind, 1.19116266115224_dbl_kind, 0.0638713068477278_dbl_kind /)
      bcenh_5bd(1:5 ,6,7) = (/ 2.2447323037953_dbl_kind, 2.07433923799976_dbl_kind, 1.86305867559311_dbl_kind, 1.20841096309504_dbl_kind, 0.0641007406105561_dbl_kind /)
      bcenh_5bd(1:5 ,7,7) = (/ 2.30438425846541_dbl_kind, 2.12238686220719_dbl_kind, 1.89904000093039_dbl_kind, 1.22579100032108_dbl_kind, 0.0644452998987116_dbl_kind /)
      bcenh_5bd(1:5 ,8,7) = (/ 2.3569984474224_dbl_kind, 2.16691740841079_dbl_kind, 1.93405621565909_dbl_kind, 1.24387586495275_dbl_kind, 0.0648636211415022_dbl_kind /)
      bcenh_5bd(1:5 ,9,7) = (/ 2.4037427324266_dbl_kind, 2.20795386848329_dbl_kind, 1.96744690263512_dbl_kind, 1.2619237632143_dbl_kind, 0.0653271600625114_dbl_kind /)
      bcenh_5bd(1:5 ,10,7) = (/ 2.44557448046568_dbl_kind, 2.24574805834547_dbl_kind, 1.99898089500271_dbl_kind, 1.27954213788229_dbl_kind, 0.0658167678826765_dbl_kind /)
      bcenh_5bd(1:5 ,1,8) = (/ 1.88623754217592_dbl_kind, 1.9263648659218_dbl_kind, 1.77795918152236_dbl_kind, 1.13227177143423_dbl_kind, 0.0434147123324349_dbl_kind /)
      bcenh_5bd(1:5 ,2,8) = (/ 1.92369424467697_dbl_kind, 1.8942558864284_dbl_kind, 1.74459158351539_dbl_kind, 1.1135113265668_dbl_kind, 0.0435893033924684_dbl_kind /)
      bcenh_5bd(1:5 ,3,8) = (/ 2.00945846267708_dbl_kind, 1.91818299493273_dbl_kind, 1.74202123236172_dbl_kind, 1.10187775406683_dbl_kind, 0.043414854260165_dbl_kind /)
      bcenh_5bd(1:5 ,4,8) = (/ 2.09554589746963_dbl_kind, 1.96406200367393_dbl_kind, 1.76247271743548_dbl_kind, 1.10366507349996_dbl_kind, 0.0433211088148566_dbl_kind /)
      bcenh_5bd(1:5 ,5,8) = (/ 2.17292182481109_dbl_kind, 2.01547921320266_dbl_kind, 1.79340178912967_dbl_kind, 1.11366185926902_dbl_kind, 0.0433701544141689_dbl_kind /)
      bcenh_5bd(1:5 ,6,8) = (/ 2.24086221282091_dbl_kind, 2.06620238481792_dbl_kind, 1.82812292792437_dbl_kind, 1.12790506098243_dbl_kind, 0.0435319637410063_dbl_kind /)
      bcenh_5bd(1:5 ,7,8) = (/ 2.30045372513647_dbl_kind, 2.11403292203795_dbl_kind, 1.86340498643423_dbl_kind, 1.14408997199748_dbl_kind, 0.0437702684169416_dbl_kind /)
      bcenh_5bd(1:5 ,8,8) = (/ 2.35301272208646_dbl_kind, 2.15836713295615_dbl_kind, 1.89774509946483_dbl_kind, 1.16093986738741_dbl_kind, 0.0440574325725808_dbl_kind /)
      bcenh_5bd(1:5 ,9,8) = (/ 2.39970609592023_dbl_kind, 2.19922606974399_dbl_kind, 1.9304940073545_dbl_kind, 1.17776068893983_dbl_kind, 0.044374386922134_dbl_kind /)
      bcenh_5bd(1:5 ,10,8) = (/ 2.44149053760016_dbl_kind, 2.23685927813827_dbl_kind, 1.96142401595058_dbl_kind, 1.19418484664932_dbl_kind, 0.0447083456286085_dbl_kind /)

      gaer_5bd(1:5 ,1) = (/ 0.350231881885906_dbl_kind, 0.211924244128064_dbl_kind, 0.146188682542913_dbl_kind, 0.112009439045293_dbl_kind, 0.060565694843084_dbl_kind /)
      gaer_5bd(1:5 ,2) = (/ 0.521517715996158_dbl_kind, 0.34457189840306_dbl_kind, 0.244048159248401_dbl_kind, 0.188518513380877_dbl_kind, 0.103316928297739_dbl_kind /)
      gaer_5bd(1:5 ,3) = (/ 0.690908112844937_dbl_kind, 0.717759065247993_dbl_kind, 0.671511248292627_dbl_kind, 0.614225462567888_dbl_kind, 0.436682950958558_dbl_kind /)
      gaer_5bd(1:5 ,4) = (/ 0.699478684452806_dbl_kind, 0.651992387581091_dbl_kind, 0.695738438913831_dbl_kind, 0.724417176862696_dbl_kind, 0.701481090364134_dbl_kind /)
      gaer_5bd(1:5 ,5) = (/ 0.785636278417498_dbl_kind, 0.749796744517699_dbl_kind, 0.683301177698451_dbl_kind, 0.629720518882672_dbl_kind, 0.665531587501598_dbl_kind /)
      gaer_5bd(1:5 ,6) = (/ 0.829747716843613_dbl_kind, 0.785685490791679_dbl_kind, 0.772583689570701_dbl_kind, 0.757648539242581_dbl_kind, 0.728843851070574_dbl_kind /)

      kaer_5bd(1:5 ,1) = (/ 11398.4540724821_dbl_kind, 5922.76076637376_dbl_kind, 4039.88947595266_dbl_kind, 3261.62137894056_dbl_kind, 2223.60028513459_dbl_kind /)
      kaer_5bd(1:5 ,2) = (/ 25368.6111954733_dbl_kind, 12520.3846877849_dbl_kind, 7738.643174918_dbl_kind, 5744.35461327268_dbl_kind, 3526.76546641382_dbl_kind /)
      kaer_5bd(1:5 ,3) = (/ 2686.90326329624_dbl_kind, 2419.98140297723_dbl_kind, 1627.51690973548_dbl_kind, 1138.23252303209_dbl_kind, 466.104227277046_dbl_kind /)
      kaer_5bd(1:5 ,4) = (/ 841.089434044834_dbl_kind, 987.406197502421_dbl_kind, 1183.52284776972_dbl_kind, 1267.30625580205_dbl_kind, 993.497508579304_dbl_kind /)
      kaer_5bd(1:5 ,5) = (/ 387.85423560755_dbl_kind, 419.109723948302_dbl_kind, 399.559447343404_dbl_kind, 397.191283865122_dbl_kind, 503.14317519429_dbl_kind /)
      kaer_5bd(1:5 ,6) = (/ 196.638063554016_dbl_kind, 202.877379461792_dbl_kind, 208.304425287341_dbl_kind, 204.723737634461_dbl_kind, 228.755667038372_dbl_kind /)

      waer_5bd(1:5 ,1) = (/ 0.287685315976181_dbl_kind, 0.186577277125224_dbl_kind, 0.123152237089201_dbl_kind, 0.0883462885905543_dbl_kind, 0.0403421562269378_dbl_kind /)
      waer_5bd(1:5 ,2) = (/ 0.515945305512804_dbl_kind, 0.434313626536424_dbl_kind, 0.346103765992635_dbl_kind, 0.275522926330555_dbl_kind, 0.138576096442815_dbl_kind /)
      waer_5bd(1:5 ,3) = (/ 0.97891105715305_dbl_kind, 0.994175916042451_dbl_kind, 0.993357580762207_dbl_kind, 0.992545751316266_dbl_kind, 0.953291550046772_dbl_kind /)
      waer_5bd(1:5 ,4) = (/ 0.943752248802793_dbl_kind, 0.984191668599419_dbl_kind, 0.989309063917025_dbl_kind, 0.991793946836264_dbl_kind, 0.982999590668913_dbl_kind /)
      waer_5bd(1:5 ,5) = (/ 0.904044530646049_dbl_kind, 0.964651629694555_dbl_kind, 0.968275809551522_dbl_kind, 0.972598419874107_dbl_kind, 0.977612418329876_dbl_kind /)
      waer_5bd(1:5 ,6) = (/ 0.849818195355416_dbl_kind, 0.940460325044343_dbl_kind, 0.948316305534169_dbl_kind, 0.952841175117807_dbl_kind, 0.955379528193802_dbl_kind /)

      end subroutine icepack_shortwave_init_dEdd5band

!=======================================================================
      subroutine icepack_shortwave_init_snicartest()

      character (len=*),parameter :: subname='(icepack_shortwave_init_snicartest)'

         nmbrad_snicar  = 5   ! snow grain radius number SNICAR SSP tables

         allocate(rsnw_snicar_tab(nmbrad_snicar))         ! snow grain radii
         rsnw_snicar_tab = (/ &   ! snow grain radius for each table entry (micro-meters)
            6._dbl_kind, 37._dbl_kind, 221._dbl_kind, 600._dbl_kind, 1340._dbl_kind/)
         rsnw_snicar_min = rsnw_snicar_tab(1)             ! minimum snow radius - integer value used for indexing
         rsnw_snicar_max = rsnw_snicar_tab(nmbrad_snicar) ! maximum snow radius - integer value used for indexing
         rsnw_datatype = 'sorted'

         allocate(ssp_snwextdr(nspint_5bd,nmbrad_snicar)) ! extinction coefficient, direct
         allocate(ssp_snwextdf(nspint_5bd,nmbrad_snicar)) ! extinction coefficient, diffuse
         allocate(ssp_snwalbdr(nspint_5bd,nmbrad_snicar)) ! single-scattering albedo, direct
         allocate(ssp_snwalbdf(nspint_5bd,nmbrad_snicar)) ! single-scattering albedo, diffuse
         allocate(ssp_sasymmdr(nspint_5bd,nmbrad_snicar)) ! snow asymmetry factor, direct
         allocate(ssp_sasymmdf(nspint_5bd,nmbrad_snicar)) ! snow asymmetry factor, diffuse

         ! tcraig, these data statements are not consistent with the array index order
         ! reshape from nmbrad,nspint to nspint,nmbrad with order = 2,1

         ssp_snwextdr = reshape((/ &
               46.27374983_dbl_kind, 24.70286257_dbl_kind, 6.54918455_dbl_kind, 2.60356240_dbl_kind, 1.19616800_dbl_kind,   &
               46.56827715_dbl_kind, 24.81790668_dbl_kind, 6.56181227_dbl_kind, 2.60604155_dbl_kind, 1.19682614_dbl_kind, &
               46.76114033_dbl_kind, 24.90468677_dbl_kind, 6.56950323_dbl_kind, 2.60780948_dbl_kind, 1.19737512_dbl_kind, &
               46.97539920_dbl_kind, 24.94951550_dbl_kind, 6.57687695_dbl_kind, 2.60937383_dbl_kind, 1.19774008_dbl_kind, &
               47.48598349_dbl_kind, 25.14100194_dbl_kind, 6.59708024_dbl_kind, 2.61372576_dbl_kind, 1.19897351_dbl_kind /), &
               (/nmbrad_snicar,nspint_5bd/), order=(/ 2, 1 /))

         ssp_snwextdf = reshape((/ &
               46.26936158_dbl_kind, 24.70165487_dbl_kind, 6.54903637_dbl_kind, 2.60353051_dbl_kind, 1.19615825_dbl_kind, &
               46.56628244_dbl_kind, 24.81707286_dbl_kind, 6.56164279_dbl_kind, 2.60601584_dbl_kind, 1.19681690_dbl_kind,  &
               46.75501968_dbl_kind, 24.90175807_dbl_kind, 6.56932140_dbl_kind, 2.60775795_dbl_kind, 1.19736237_dbl_kind, &
               46.95368476_dbl_kind, 24.94497414_dbl_kind, 6.57612007_dbl_kind, 2.60924059_dbl_kind, 1.19770981_dbl_kind, &
               47.29620774_dbl_kind, 25.07135850_dbl_kind, 6.58916980_dbl_kind, 2.61198929_dbl_kind, 1.19850197_dbl_kind /), &
               (/nmbrad_snicar,nspint_5bd/), order=(/ 2, 1 /))

         ssp_snwalbdr = reshape((/ &
               0.99999581_dbl_kind, 0.99999231_dbl_kind, 0.99997215_dbl_kind, 0.99993093_dbl_kind, 0.99985157_dbl_kind, &
               0.99987892_dbl_kind, 0.99978212_dbl_kind, 0.99923115_dbl_kind, 0.99817946_dbl_kind, 0.99628374_dbl_kind, &
               0.99909682_dbl_kind, 0.99835523_dbl_kind, 0.99418008_dbl_kind, 0.98592085_dbl_kind, 0.97062632_dbl_kind, &
               0.99308867_dbl_kind, 0.98972448_dbl_kind, 0.97261528_dbl_kind, 0.93947995_dbl_kind, 0.88207117_dbl_kind, &
               0.93322249_dbl_kind, 0.89645776_dbl_kind, 0.75765643_dbl_kind, 0.62728830_dbl_kind, 0.55435033_dbl_kind /), &
               (/nmbrad_snicar,nspint_5bd/), order=(/ 2, 1 /))

         ssp_snwalbdf = reshape((/ &
               0.99999586_dbl_kind, 0.99999240_dbl_kind, 0.99997249_dbl_kind, 0.99993178_dbl_kind, 0.99985340_dbl_kind,  &
               0.99988441_dbl_kind, 0.99979202_dbl_kind, 0.99926745_dbl_kind, 0.99826865_dbl_kind, 0.99647135_dbl_kind, &
               0.99910997_dbl_kind, 0.99837683_dbl_kind, 0.99426171_dbl_kind, 0.98610431_dbl_kind, 0.97097643_dbl_kind, &
               0.99476731_dbl_kind, 0.99161140_dbl_kind, 0.97441209_dbl_kind, 0.94127464_dbl_kind, 0.88440735_dbl_kind, &
               0.94198370_dbl_kind, 0.90613207_dbl_kind, 0.77042376_dbl_kind, 0.63887161_dbl_kind, 0.55660980_dbl_kind /), &
               (/nmbrad_snicar,nspint_5bd/), order=(/ 2, 1 /))

         ssp_sasymmdr = reshape((/ &
               0.88503036_dbl_kind, 0.88778473_dbl_kind, 0.89049023_dbl_kind, 0.89112501_dbl_kind, 0.89136157_dbl_kind, &
               0.88495225_dbl_kind, 0.88905367_dbl_kind, 0.89315385_dbl_kind, 0.89433673_dbl_kind, 0.89500270_dbl_kind,  &
               0.88440201_dbl_kind, 0.88928256_dbl_kind, 0.89503166_dbl_kind, 0.89762648_dbl_kind, 0.90045378_dbl_kind, &
               0.88590371_dbl_kind, 0.89350221_dbl_kind, 0.90525156_dbl_kind, 0.91314567_dbl_kind, 0.92157748_dbl_kind, &
               0.90335370_dbl_kind, 0.91778261_dbl_kind, 0.94615574_dbl_kind, 0.96323447_dbl_kind, 0.97167644_dbl_kind /), &
               (/nmbrad_snicar,nspint_5bd/), order=(/ 2, 1 /))

         ssp_sasymmdf = reshape((/ &
               0.88500571_dbl_kind, 0.88773868_dbl_kind, 0.89042496_dbl_kind, 0.89105530_dbl_kind, 0.89128949_dbl_kind, &
               0.88495225_dbl_kind, 0.88905367_dbl_kind, 0.89315385_dbl_kind, 0.89433673_dbl_kind, 0.89500270_dbl_kind, &
               0.88441433_dbl_kind, 0.88929133_dbl_kind, 0.89500611_dbl_kind, 0.89756091_dbl_kind, 0.90035504_dbl_kind, &
               0.88495554_dbl_kind, 0.89201556_dbl_kind, 0.90204619_dbl_kind, 0.90914885_dbl_kind, 0.91769988_dbl_kind, &
               0.89620237_dbl_kind, 0.90998944_dbl_kind, 0.94126152_dbl_kind, 0.96209938_dbl_kind, 0.97266310_dbl_kind  /), &
               (/nmbrad_snicar,nspint_5bd/), order=(/ 2, 1 /))

      end subroutine icepack_shortwave_init_snicartest

!=======================================================================

      subroutine icepack_shortwave_init_snicar()

! NO_SNICARHC turns off hardcoded tables to reduce compile time
#ifdef NO_SNICARHC
      character(len=*),parameter :: subname='(icepack_shortwave_init_snicar)'
      call icepack_warnings_add(subname//' ERROR: large shortwave snicar tables not compiled')
      call icepack_warnings_add(subname//' ERROR: NO_SNICARHC CPP should be turned off')
      call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
      return
#else
      integer (kind=int_kind) :: &
         n        ! loop counter

      ! data in SNICAR SSP file
   real (kind=dbl_kind), allocatable :: iceAsymmetryParameterDirect(:,:)
   real (kind=dbl_kind), allocatable :: iceAsymmetryParameterDiffuse(:,:)
   real (kind=dbl_kind), allocatable :: iceSingleScatterAlbedoDirect(:,:)
   real (kind=dbl_kind), allocatable :: iceSingleScatterAlbedoDiffuse(:,:)
   real (kind=dbl_kind), allocatable :: iceMassExtinctionCrossSectionDirect(:,:)
   real (kind=dbl_kind), allocatable :: iceMassExtinctionCrossSectionDiffuse(:,:)
   real (kind=dbl_kind), allocatable :: bcEffectiveRadius(:)
   real (kind=dbl_kind), allocatable :: iceGrainEffectiveRadius(:)

      character(len=*),parameter :: subname='(icepack_shortwave_init_snicar)'

   !--- extracted from netcdf file
   !--- snicar_optics_5bnd_snow_and_aerosols.nc

   nmbrad_snicar   = 1471

   allocate(iceAsymmetryParameterDirect(nspint_5bd, nmbrad_snicar))
   allocate(iceAsymmetryParameterDiffuse(nspint_5bd, nmbrad_snicar))
   allocate(iceSingleScatterAlbedoDirect(nspint_5bd, nmbrad_snicar))
   allocate(iceSingleScatterAlbedoDiffuse(nspint_5bd, nmbrad_snicar))
   allocate(iceMassExtinctionCrossSectionDirect(nspint_5bd, nmbrad_snicar))
   allocate(iceMassExtinctionCrossSectionDiffuse(nspint_5bd, nmbrad_snicar))
   allocate(bcEffectiveRadius(nmodal1))
   allocate(iceGrainEffectiveRadius(nmodal2))

   iceAsymmetryParameterDirect(1:5 ,1) = (/ 0.883812826619966_dbl_kind, 0.883341999504413_dbl_kind, 0.882286247177836_dbl_kind, 0.883104831607199_dbl_kind, 0.898615034787299_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,2) = (/ 0.884045703655972_dbl_kind, 0.883651330111304_dbl_kind, 0.882694799122514_dbl_kind, 0.88363100888832_dbl_kind, 0.89949035376612_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,3) = (/ 0.884265594993862_dbl_kind, 0.883942205670049_dbl_kind, 0.88307926463881_dbl_kind, 0.884130950189708_dbl_kind, 0.900327839087815_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,4) = (/ 0.88447330083083_dbl_kind, 0.884216304103572_dbl_kind, 0.88344091213276_dbl_kind, 0.884606491166985_dbl_kind, 0.901130420272937_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,5) = (/ 0.884669552343808_dbl_kind, 0.884475131690979_dbl_kind, 0.883781038633209_dbl_kind, 0.885059383458455_dbl_kind, 0.901900722489371_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,6) = (/ 0.884855026960007_dbl_kind, 0.884720041157208_dbl_kind, 0.884100962450369_dbl_kind, 0.885491277499437_dbl_kind, 0.90264110549347_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,7) = (/ 0.885030359518704_dbl_kind, 0.884952247979408_dbl_kind, 0.884402009458808_dbl_kind, 0.885903711207814_dbl_kind, 0.903353696417132_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,8) = (/ 0.885196150263205_dbl_kind, 0.885172844911776_dbl_kind, 0.884685496767126_dbl_kind, 0.886298104374309_dbl_kind, 0.904040417746719_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,9) = (/ 0.885352970401633_dbl_kind, 0.885382814828977_dbl_kind, 0.884952716319368_dbl_kind, 0.886675757994174_dbl_kind, 0.904703011428127_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,10) = (/ 0.885501365822281_dbl_kind, 0.885583042041594_dbl_kind, 0.885204920022277_dbl_kind, 0.887037857517288_dbl_kind, 0.905343059764095_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,11) = (/ 0.885641859430929_dbl_kind, 0.885774322259615_dbl_kind, 0.88544330728219_dbl_kind, 0.887385478949465_dbl_kind, 0.905962003597254_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,12) = (/ 0.885774952484183_dbl_kind, 0.885957371382396_dbl_kind, 0.885669015328004_dbl_kind, 0.88771959682008_dbl_kind, 0.906561158161349_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,13) = (/ 0.885901125217878_dbl_kind, 0.886132833283754_dbl_kind, 0.885883112352245_dbl_kind, 0.888041093178004_dbl_kind, 0.907141726910893_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,14) = (/ 0.886020837008229_dbl_kind, 0.886301286744091_dbl_kind, 0.886086593283499_dbl_kind, 0.888350766948108_dbl_kind, 0.907704813590579_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,15) = (/ 0.886134526252419_dbl_kind, 0.886463251661636_dbl_kind, 0.886280377878174_dbl_kind, 0.888649343148456_dbl_kind, 0.908251432770758_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,16) = (/ 0.886242610112709_dbl_kind, 0.886619194654339_dbl_kind, 0.886465310761095_dbl_kind, 0.888937481619419_dbl_kind, 0.908782519047961_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,17) = (/ 0.886345484232351_dbl_kind, 0.886769534144212_dbl_kind, 0.886642163031875_dbl_kind, 0.889215785043267_dbl_kind, 0.909298935086562_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,18) = (/ 0.8864435225017_dbl_kind, 0.886914644998123_dbl_kind, 0.886811635071063_dbl_kind, 0.889484806134571_dbl_kind, 0.909801478657418_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,19) = (/ 0.886537076928103_dbl_kind, 0.887054862783442_dbl_kind, 0.886974360214836_dbl_kind, 0.889745053959067_dbl_kind, 0.910290888810795_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,20) = (/ 0.886626477642934_dbl_kind, 0.887190487683911_dbl_kind, 0.887130909010696_dbl_kind, 0.889996999394228_dbl_kind, 0.910767851303848_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,21) = (/ 0.886712033062894_dbl_kind, 0.887321788110551_dbl_kind, 0.887281793813395_dbl_kind, 0.8902410797823_dbl_kind, 0.911233003387176_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,22) = (/ 0.886794030210059_dbl_kind, 0.887449004034129_dbl_kind, 0.887427473525989_dbl_kind, 0.890477702849466_dbl_kind, 0.911686938040782_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,23) = (/ 0.886872735185552_dbl_kind, 0.887572350059476_dbl_kind, 0.887568358333269_dbl_kind, 0.890707249976559_dbl_kind, 0.912130207736829_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,24) = (/ 0.886948393784847_dbl_kind, 0.887692018257494_dbl_kind, 0.88770481431232_dbl_kind, 0.890930078910336_dbl_kind, 0.912563327795312_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,25) = (/ 0.887021232237997_dbl_kind, 0.887808180767587_dbl_kind, 0.887837167837215_dbl_kind, 0.891146526002195_dbl_kind, 0.912986779388719_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,26) = (/ 0.887091458055303_dbl_kind, 0.887920992181373_dbl_kind, 0.887965709721658_dbl_kind, 0.89135690805533_dbl_kind, 0.913401012243196_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,27) = (/ 0.887159260957581_dbl_kind, 0.888030591717439_dbl_kind, 0.888090699065235_dbl_kind, 0.891561523853312_dbl_kind, 0.913806447076307_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,28) = (/ 0.88722481387005_dbl_kind, 0.888137105196449_dbl_kind, 0.888212366786088_dbl_kind, 0.891760655433949_dbl_kind, 0.914203477805249_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,29) = (/ 0.887288273959547_dbl_kind, 0.888240646825891_dbl_kind, 0.888330918836063_dbl_kind, 0.891954569162996_dbl_kind, 0.914592473554032_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,30) = (/ 0.887349783696164_dbl_kind, 0.888341320803912_dbl_kind, 0.88844653910415_dbl_kind, 0.892143516653255_dbl_kind, 0.914973780483771_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,31) = (/ 0.887409471922131_dbl_kind, 0.888439222752064_dbl_kind, 0.888559392021019_dbl_kind, 0.8923277355663_dbl_kind, 0.915347723466474_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,32) = (/ 0.887467454912848_dbl_kind, 0.888534440987063_dbl_kind, 0.888669624882114_dbl_kind, 0.892507450326607_dbl_kind, 0.915714607619647_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,33) = (/ 0.88752383741706_dbl_kind, 0.888627057642028_dbl_kind, 0.888777369909652_dbl_kind, 0.892682872771361_dbl_kind, 0.916074719716489_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,34) = (/ 0.887578713665373_dbl_kind, 0.888717149647815_dbl_kind, 0.888882746075307_dbl_kind, 0.8928542027537_dbl_kind, 0.91642832948429_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,35) = (/ 0.887632168338369_dbl_kind, 0.888804789585207_dbl_kind, 0.888985860705803_dbl_kind, 0.893021628712539_dbl_kind, 0.916775690801912_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,36) = (/ 0.887684277487558_dbl_kind, 0.888890046418659_dbl_kind, 0.889086810893211_dbl_kind, 0.893185328218354_dbl_kind, 0.917117042805751_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,37) = (/ 0.887735109404193_dbl_kind, 0.888972986122176_dbl_kind, 0.889185684730915_dbl_kind, 0.893345468501316_dbl_kind, 0.91745261091235_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,38) = (/ 0.887784725432596_dbl_kind, 0.889053672207634_dbl_kind, 0.889282562394872_dbl_kind, 0.893502206965805_dbl_kind, 0.917782607764873_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,39) = (/ 0.887833180726068_dbl_kind, 0.889132166165486_dbl_kind, 0.889377517088377_dbl_kind, 0.893655691693575_dbl_kind, 0.918107234109712_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,40) = (/ 0.887880524944661_dbl_kind, 0.889208527827369_dbl_kind, 0.889470615866909_dbl_kind, 0.893806061936493_dbl_kind, 0.918426679608872_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,41) = (/ 0.887926802895125_dbl_kind, 0.889282815659608_dbl_kind, 0.889561920358066_dbl_kind, 0.893953448598883_dbl_kind, 0.918741123593164_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,42) = (/ 0.887972055114191_dbl_kind, 0.889355086996071_dbl_kind, 0.889651487389992_dbl_kind, 0.894097974708859_dbl_kind, 0.919050735760675_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,43) = (/ 0.888016318397028_dbl_kind, 0.889425398218217_dbl_kind, 0.889739369540202_dbl_kind, 0.894239755877666_dbl_kind, 0.919355676824658_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,44) = (/ 0.888059626273239_dbl_kind, 0.8894938048896_dbl_kind, 0.889825615615308_dbl_kind, 0.894378900745858_dbl_kind, 0.919656099114526_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,45) = (/ 0.888102009433131_dbl_kind, 0.889560361851463_dbl_kind, 0.889910271070832_dbl_kind, 0.894515511415094_dbl_kind, 0.919952147133357_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,46) = (/ 0.888143496107316_dbl_kind, 0.889625123285456_dbl_kind, 0.889993378379132_dbl_kind, 0.894649683864386_dbl_kind, 0.920243958075076_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,47) = (/ 0.888184112402814_dbl_kind, 0.889688142748925_dbl_kind, 0.890074977352411_dbl_kind, 0.894781508349754_dbl_kind, 0.920531662304148_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,48) = (/ 0.888223882598948_dbl_kind, 0.889749473187648_dbl_kind, 0.890155105426786_dbl_kind, 0.894911069786426_dbl_kind, 0.920815383800515_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,49) = (/ 0.888262829406337_dbl_kind, 0.88980916693036_dbl_kind, 0.890233797912649_dbl_kind, 0.895038448112892_dbl_kind, 0.921095240572238_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,50) = (/ 0.888300974192223_dbl_kind, 0.889867275668895_dbl_kind, 0.890311088215733_dbl_kind, 0.895163718636358_dbl_kind, 0.921371345038144_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,51) = (/ 0.888338337175304_dbl_kind, 0.889923850427301_dbl_kind, 0.890387008032717_dbl_kind, 0.895286952359318_dbl_kind, 0.921643804382689_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,52) = (/ 0.888374937593115_dbl_kind, 0.889978941522852_dbl_kind, 0.890461587524629_dbl_kind, 0.89540821628718_dbl_kind, 0.921912720884974_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,53) = (/ 0.888410793844827_dbl_kind, 0.890032598521469_dbl_kind, 0.890534855470855_dbl_kind, 0.895527573717057_dbl_kind, 0.922178192223876_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,54) = (/ 0.888445923612195_dbl_kind, 0.890084870189707_dbl_kind, 0.890606839406125_dbl_kind, 0.89564508450799_dbl_kind, 0.922440311760995_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,55) = (/ 0.88848034396117_dbl_kind, 0.890135804445115_dbl_kind, 0.890677565742526_dbl_kind, 0.895760805333015_dbl_kind, 0.922699168803066_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,56) = (/ 0.888514071426546_dbl_kind, 0.890185448306509_dbl_kind, 0.890747059878304_dbl_kind, 0.895874789913616_dbl_kind, 0.922954848845391_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,57) = (/ 0.888547122081761_dbl_kind, 0.890233847845397_dbl_kind, 0.890815346294935_dbl_kind, 0.895987089237189_dbl_kind, 0.923207433797687_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,58) = (/ 0.888579511595852_dbl_kind, 0.890281048139569_dbl_kind, 0.890882448643774_dbl_kind, 0.896097751758237_dbl_kind, 0.923457002193696_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,59) = (/ 0.888611255279319_dbl_kind, 0.890327093229695_dbl_kind, 0.890948389823394_dbl_kind, 0.89620682358406_dbl_kind, 0.923703629385799_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,60) = (/ 0.888642368120525_dbl_kind, 0.890372026079506_dbl_kind, 0.891013192048583_dbl_kind, 0.89631434864577_dbl_kind, 0.923947387725758_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,61) = (/ 0.888672864814057_dbl_kind, 0.89041588854007_dbl_kind, 0.891076876911846_dbl_kind, 0.896420368855462_dbl_kind, 0.92418834673268_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,62) = (/ 0.888702759782336_dbl_kind, 0.890458721318461_dbl_kind, 0.891139465438151_dbl_kind, 0.896524924250401_dbl_kind, 0.924426573249183_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,63) = (/ 0.888732067191594_dbl_kind, 0.890500563951055_dbl_kind, 0.891200978133579_dbl_kind, 0.896628053125094_dbl_kind, 0.924662131586672_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,64) = (/ 0.888760800963244_dbl_kind, 0.890541454781536_dbl_kind, 0.891261435028453_dbl_kind, 0.896729792152077_dbl_kind, 0.924895083660592_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,65) = (/ 0.888788974781478_dbl_kind, 0.890581430943656_dbl_kind, 0.891320855715475_dbl_kind, 0.896830176492261_dbl_kind, 0.925125489116406_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,66) = (/ 0.888816602097876_dbl_kind, 0.890620528348689_dbl_kind, 0.891379259383328_dbl_kind, 0.896929239895646_dbl_kind, 0.925353405447064_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,67) = (/ 0.888843696133657_dbl_kind, 0.890658781677486_dbl_kind, 0.891436664846189_dbl_kind, 0.897027014793168_dbl_kind, 0.925578888102588_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,68) = (/ 0.888870269880154_dbl_kind, 0.890696224376954_dbl_kind, 0.89149309056951_dbl_kind, 0.897123532380426_dbl_kind, 0.925801990592405_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,69) = (/ 0.888896336097955_dbl_kind, 0.890732888660789_dbl_kind, 0.891548554692451_dbl_kind, 0.897218822693995_dbl_kind, 0.926022764580998_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,70) = (/ 0.888921907315145_dbl_kind, 0.890768805514231_dbl_kind, 0.891603075047273_dbl_kind, 0.897312914680984_dbl_kind, 0.926241259977366_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,71) = (/ 0.888946995824941_dbl_kind, 0.890804004702586_dbl_kind, 0.891656669176004_dbl_kind, 0.897405836262464_dbl_kind, 0.926457525018792_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,72) = (/ 0.888971613683035_dbl_kind, 0.890838514783288_dbl_kind, 0.891709354344638_dbl_kind, 0.897497614391339_dbl_kind, 0.926671606349338_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,73) = (/ 0.888995772704832_dbl_kind, 0.890872363121181_dbl_kind, 0.891761147555155_dbl_kind, 0.89758827510521_dbl_kind, 0.926883549093492_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,74) = (/ 0.88901948446278_dbl_kind, 0.890905575906795_dbl_kind, 0.891812065555573_dbl_kind, 0.897677843574715_dbl_kind, 0.927093396925292_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,75) = (/ 0.88904276028393_dbl_kind, 0.8909381781773_dbl_kind, 0.891862124848268_dbl_kind, 0.897766344147812_dbl_kind, 0.9273011921333_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,76) = (/ 0.889065611247823_dbl_kind, 0.890970193839887_dbl_kind, 0.891911341696774_dbl_kind, 0.897853800390416_dbl_kind, 0.927506975681712_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,77) = (/ 0.889088048184792_dbl_kind, 0.891001645697287_dbl_kind, 0.891959732131233_dbl_kind, 0.897940235123779_dbl_kind, 0.927710787267906_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,78) = (/ 0.889110081674722_dbl_kind, 0.891032555475178_dbl_kind, 0.892007311952706_dbl_kind, 0.898025670458946_dbl_kind, 0.927912665376661_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,79) = (/ 0.889131722046316_dbl_kind, 0.891062943851212_dbl_kind, 0.892054096736468_dbl_kind, 0.898110127828608_dbl_kind, 0.928112647331301_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,80) = (/ 0.88915297937687_dbl_kind, 0.891092830485425_dbl_kind, 0.892100101834468_dbl_kind, 0.898193628016638_dbl_kind, 0.928310769341983_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,81) = (/ 0.889173863492567_dbl_kind, 0.891122234051793_dbl_kind, 0.892145342377078_dbl_kind, 0.898276191185543_dbl_kind, 0.928507066551312_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,82) = (/ 0.889194383969292_dbl_kind, 0.891151172270705_dbl_kind, 0.89218983327426_dbl_kind, 0.898357836902074_dbl_kind, 0.928701573077486_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,83) = (/ 0.889214550133932_dbl_kind, 0.89117966194216_dbl_kind, 0.892233589216265_dbl_kind, 0.898438584161196_dbl_kind, 0.928894322055124_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,84) = (/ 0.889234371066155_dbl_kind, 0.891207718979478_dbl_kind, 0.892276624673959_dbl_kind, 0.898518451408575_dbl_kind, 0.929085345673951_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,85) = (/ 0.889253855600639_dbl_kind, 0.891235358443346_dbl_kind, 0.892318953898884_dbl_kind, 0.898597456561759_dbl_kind, 0.929274675215449_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,86) = (/ 0.889273012329711_dbl_kind, 0.891262594576028_dbl_kind, 0.892360590923125_dbl_kind, 0.898675617030181_dbl_kind, 0.929462341087646_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,87) = (/ 0.889291849606374_dbl_kind, 0.891289440835594_dbl_kind, 0.892401549559063_dbl_kind, 0.898752949734104_dbl_kind, 0.929648372858128_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,88) = (/ 0.88931037554767_dbl_kind, 0.891315909930016_dbl_kind, 0.892441843399064_dbl_kind, 0.898829471122627_dbl_kind, 0.929832799285416_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,89) = (/ 0.889328598038372_dbl_kind, 0.891342013850997_dbl_kind, 0.892481485815188_dbl_kind, 0.898905197190808_dbl_kind, 0.930015648348784_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,90) = (/ 0.88934652473494_dbl_kind, 0.891367763907424_dbl_kind, 0.892520489958936_dbl_kind, 0.898980143496028_dbl_kind, 0.930196947276639_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,91) = (/ 0.889364163069721_dbl_kind, 0.891393170758348_dbl_kind, 0.892558868761093_dbl_kind, 0.899054325173636_dbl_kind, 0.930376722573528_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,92) = (/ 0.889381520255364_dbl_kind, 0.89141824444537_dbl_kind, 0.892596634931703_dbl_kind, 0.899127756951936_dbl_kind, 0.930555000045876_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,93) = (/ 0.889398603289403_dbl_kind, 0.891442994424384_dbl_kind, 0.892633800960189_dbl_kind, 0.899200453166574_dbl_kind, 0.93073180482651_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,94) = (/ 0.889415418958989_dbl_kind, 0.891467429596578_dbl_kind, 0.892670379115655_dbl_kind, 0.899272427774362_dbl_kind, 0.930907161398059_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,95) = (/ 0.889431973845748_dbl_kind, 0.891491558338644_dbl_kind, 0.892706381447383_dbl_kind, 0.899343694366579_dbl_kind, 0.931081093615284_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,96) = (/ 0.889448274330712_dbl_kind, 0.891515388532144_dbl_kind, 0.892741819785529_dbl_kind, 0.899414266181781_dbl_kind, 0.9312536247264_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,97) = (/ 0.889464326599331_dbl_kind, 0.891538927591983_dbl_kind, 0.892776705742045_dbl_kind, 0.899484156118131_dbl_kind, 0.931424777393451_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,98) = (/ 0.889480136646522_dbl_kind, 0.891562182493956_dbl_kind, 0.892811050711799_dbl_kind, 0.899553376745303_dbl_kind, 0.9315945737118_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,99) = (/ 0.889495710281738_dbl_kind, 0.891585159801335_dbl_kind, 0.892844865873934_dbl_kind, 0.899621940315948_dbl_kind, 0.931763035228747_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,100) = (/ 0.889511053134045_dbl_kind, 0.891607865690487_dbl_kind, 0.892878162193432_dbl_kind, 0.899689858776747_dbl_kind, 0.931930182961392_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,101) = (/ 0.889526170657182_dbl_kind, 0.891630305975495_dbl_kind, 0.892910950422888_dbl_kind, 0.899757143779079_dbl_kind, 0.932096037413687_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,102) = (/ 0.889541068134592_dbl_kind, 0.891652486131773_dbl_kind, 0.892943241104495_dbl_kind, 0.899823806689298_dbl_kind, 0.932260618592831_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,103) = (/ 0.889555750684416_dbl_kind, 0.891674411318688_dbl_kind, 0.89297504457222_dbl_kind, 0.899889858598637_dbl_kind, 0.93242394602494_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,104) = (/ 0.88957022326442_dbl_kind, 0.891696086401164_dbl_kind, 0.893006370954164_dbl_kind, 0.899955310332755_dbl_kind, 0.932586038770114_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,105) = (/ 0.889584490676875_dbl_kind, 0.891717515970294_dbl_kind, 0.893037230175099_dbl_kind, 0.900020172460922_dbl_kind, 0.932746915436868_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,106) = (/ 0.889598557573349_dbl_kind, 0.891738704362963_dbl_kind, 0.893067631959158_dbl_kind, 0.900084455304856_dbl_kind, 0.932906594196008_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,107) = (/ 0.889612428459424_dbl_kind, 0.891759655680491_dbl_kind, 0.893097585832683_dbl_kind, 0.900148168947233_dbl_kind, 0.933065092793949_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,108) = (/ 0.889626107699325_dbl_kind, 0.891780373806316_dbl_kind, 0.893127101127191_dbl_kind, 0.90021132323985_dbl_kind, 0.933222428565522_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,109) = (/ 0.889639599520457_dbl_kind, 0.891800862422733_dbl_kind, 0.893156186982477_dbl_kind, 0.900273927811472_dbl_kind, 0.933378618446283_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,110) = (/ 0.889652908017841_dbl_kind, 0.891821125026716_dbl_kind, 0.893184852349796_dbl_kind, 0.900335992075362_dbl_kind, 0.933533678984354_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,111) = (/ 0.889666037158456_dbl_kind, 0.891841164944837_dbl_kind, 0.893213105995161_dbl_kind, 0.900397525236498_dbl_kind, 0.93368762635183_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,112) = (/ 0.889678990785469_dbl_kind, 0.891860985347313_dbl_kind, 0.893240956502698_dbl_kind, 0.900458536298484_dbl_kind, 0.933840476355749_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,113) = (/ 0.889691772622367_dbl_kind, 0.891880589261207_dbl_kind, 0.89326841227807_dbl_kind, 0.900519034070173_dbl_kind, 0.933992244448662_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,114) = (/ 0.889704386276983_dbl_kind, 0.891899979582808_dbl_kind, 0.893295481551944_dbl_kind, 0.900579027171998_dbl_kind, 0.934142945738833_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,115) = (/ 0.889716835245406_dbl_kind, 0.891919159089218_dbl_kind, 0.893322172383498_dbl_kind, 0.900638524042021_dbl_kind, 0.934292595000051_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,116) = (/ 0.889729122915796_dbl_kind, 0.891938130449177_dbl_kind, 0.893348492663946_dbl_kind, 0.900697532941712_dbl_kind, 0.934441206681117_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,117) = (/ 0.889741252572081_dbl_kind, 0.891956896233159_dbl_kind, 0.893374450120073_dbl_kind, 0.900756061961465_dbl_kind, 0.934588794914984_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,118) = (/ 0.889753227397555_dbl_kind, 0.891975458922756_dbl_kind, 0.893400052317773_dbl_kind, 0.900814119025858_dbl_kind, 0.934735373527575_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,119) = (/ 0.889765050478368_dbl_kind, 0.891993820919397_dbl_kind, 0.893425306665571_dbl_kind, 0.900871711898663_dbl_kind, 0.934880956046323_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,120) = (/ 0.889776724806911_dbl_kind, 0.89201198455242_dbl_kind, 0.893450220418122_dbl_kind, 0.900928848187619_dbl_kind, 0.935025555708392_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,121) = (/ 0.889788253285097_dbl_kind, 0.892029952086527_dbl_kind, 0.893474800679693_dbl_kind, 0.900985535348982_dbl_kind, 0.935169185468642_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,122) = (/ 0.88979963872755_dbl_kind, 0.89204772572866_dbl_kind, 0.893499054407582_dbl_kind, 0.901041780691841_dbl_kind, 0.935311858007322_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,123) = (/ 0.889810883864676_dbl_kind, 0.892065307634319_dbl_kind, 0.893522988415516_dbl_kind, 0.901097591382232_dbl_kind, 0.935453585737501_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,124) = (/ 0.889821991345661_dbl_kind, 0.892082699913359_dbl_kind, 0.893546609376982_dbl_kind, 0.901152974447049_dbl_kind, 0.935594380812278_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,125) = (/ 0.889832963741352_dbl_kind, 0.892099904635271_dbl_kind, 0.893569923828501_dbl_kind, 0.901207936777753_dbl_kind, 0.935734255131741_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,126) = (/ 0.889843803547056_dbl_kind, 0.892116923834021_dbl_kind, 0.893592938172838_dbl_kind, 0.901262485133906_dbl_kind, 0.935873220349706_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,127) = (/ 0.889854513185248_dbl_kind, 0.892133759512413_dbl_kind, 0.893615658682148_dbl_kind, 0.90131662614652_dbl_kind, 0.936011287880251_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,128) = (/ 0.889865095008185_dbl_kind, 0.892150413646063_dbl_kind, 0.893638091501042_dbl_kind, 0.901370366321242_dbl_kind, 0.936148468904038_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,129) = (/ 0.88987555130044_dbl_kind, 0.892166888186953_dbl_kind, 0.893660242649585_dbl_kind, 0.901423712041384_dbl_kind, 0.936284774374435_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,130) = (/ 0.889885884281343_dbl_kind, 0.89218318506664_dbl_kind, 0.893682118026204_dbl_kind, 0.901476669570793_dbl_kind, 0.93642021502346_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,131) = (/ 0.889896096107353_dbl_kind, 0.892199306199101_dbl_kind, 0.89370372341053_dbl_kind, 0.901529245056586_dbl_kind, 0.936554801367527_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,132) = (/ 0.889906188874331_dbl_kind, 0.892215253483271_dbl_kind, 0.893725064466145_dbl_kind, 0.90158144453175_dbl_kind, 0.936688543713033_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,133) = (/ 0.88991616461976_dbl_kind, 0.892231028805268_dbl_kind, 0.893746146743251_dbl_kind, 0.901633273917611_dbl_kind, 0.936821452161768_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,134) = (/ 0.889926025324862_dbl_kind, 0.892246634040347_dbl_kind, 0.893766975681261_dbl_kind, 0.901684739026194_dbl_kind, 0.936953536616166_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,135) = (/ 0.889935772916667_dbl_kind, 0.892262071054592_dbl_kind, 0.893787556611294_dbl_kind, 0.901735845562456_dbl_kind, 0.93708480678439_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,136) = (/ 0.889945409269996_dbl_kind, 0.892277341706361_dbl_kind, 0.893807894758593_dbl_kind, 0.901786599126435_dbl_kind, 0.937215272185281_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,137) = (/ 0.889954936209373_dbl_kind, 0.892292447847522_dbl_kind, 0.893827995244867_dbl_kind, 0.901837005215285_dbl_kind, 0.937344942153155_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,138) = (/ 0.889964355510884_dbl_kind, 0.892307391324473_dbl_kind, 0.893847863090537_dbl_kind, 0.901887069225227_dbl_kind, 0.937473825842461_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,139) = (/ 0.889973668903956_dbl_kind, 0.892322173978979_dbl_kind, 0.893867503216914_dbl_kind, 0.901936796453422_dbl_kind, 0.937601932232299_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,140) = (/ 0.889982878073081_dbl_kind, 0.892336797648837_dbl_kind, 0.893886920448292_dbl_kind, 0.901986192099754_dbl_kind, 0.937729270130816_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,141) = (/ 0.889991984659472_dbl_kind, 0.892351264168377_dbl_kind, 0.893906119513959_dbl_kind, 0.902035261268552_dbl_kind, 0.937855848179466_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,142) = (/ 0.890000990262672_dbl_kind, 0.892365575368832_dbl_kind, 0.893925105050154_dbl_kind, 0.902084008970236_dbl_kind, 0.937981674857158_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,143) = (/ 0.890009896442088_dbl_kind, 0.892379733078562_dbl_kind, 0.893943881601921_dbl_kind, 0.90213244012291_dbl_kind, 0.938106758484284_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,144) = (/ 0.890018704718483_dbl_kind, 0.892393739123175_dbl_kind, 0.893962453624917_dbl_kind, 0.902180559553897_dbl_kind, 0.938231107226625_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,145) = (/ 0.890027416575406_dbl_kind, 0.892407595325534_dbl_kind, 0.893980825487141_dbl_kind, 0.902228372001216_dbl_kind, 0.938354729099155_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,146) = (/ 0.890036033460573_dbl_kind, 0.892421303505667_dbl_kind, 0.893999001470602_dbl_kind, 0.902275882115018_dbl_kind, 0.93847763196974_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,147) = (/ 0.890044556787193_dbl_kind, 0.892434865480602_dbl_kind, 0.894016985772921_dbl_kind, 0.902323094458982_dbl_kind, 0.938599823562726_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,148) = (/ 0.89005298793525_dbl_kind, 0.892448283064117_dbl_kind, 0.894034782508872_dbl_kind, 0.902370013511659_dbl_kind, 0.938721311462439_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,149) = (/ 0.890061328252731_dbl_kind, 0.892461558066431_dbl_kind, 0.894052395711877_dbl_kind, 0.902416643667796_dbl_kind, 0.938842103116563_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,150) = (/ 0.890069579056809_dbl_kind, 0.892474692293834_dbl_kind, 0.894069829335429_dbl_kind, 0.90246298923962_dbl_kind, 0.938962205839464_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,151) = (/ 0.890077741634987_dbl_kind, 0.89248768754827_dbl_kind, 0.894087087254481_dbl_kind, 0.902509054458088_dbl_kind, 0.93908162681539_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,152) = (/ 0.890085817246188_dbl_kind, 0.892500545626875_dbl_kind, 0.894104173266771_dbl_kind, 0.902554843474122_dbl_kind, 0.939200373101591_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,153) = (/ 0.890093807121806_dbl_kind, 0.892513268321485_dbl_kind, 0.894121091094114_dbl_kind, 0.902600360359815_dbl_kind, 0.939318451631365_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,154) = (/ 0.890101712466724_dbl_kind, 0.892525857418106_dbl_kind, 0.894137844383644_dbl_kind, 0.90264560910961_dbl_kind, 0.939435869217003_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,155) = (/ 0.890109534460279_dbl_kind, 0.892538314696369_dbl_kind, 0.894154436709015_dbl_kind, 0.902690593641473_dbl_kind, 0.939552632552679_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,156) = (/ 0.890117274257199_dbl_kind, 0.892550641928962_dbl_kind, 0.894170871571565_dbl_kind, 0.902735317798036_dbl_kind, 0.939668748217231_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,157) = (/ 0.890124932988498_dbl_kind, 0.892562840881054_dbl_kind, 0.894187152401448_dbl_kind, 0.902779785347736_dbl_kind, 0.939784222676889_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,158) = (/ 0.890132511762329_dbl_kind, 0.892574913309701_dbl_kind, 0.894203282558726_dbl_kind, 0.902823999985931_dbl_kind, 0.939899062287924_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,159) = (/ 0.890140011664821_dbl_kind, 0.892586860963255_dbl_kind, 0.894219265334438_dbl_kind, 0.902867965336011_dbl_kind, 0.940013273299224_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,160) = (/ 0.890147433760857_dbl_kind, 0.89259868558077_dbl_kind, 0.894235103951636_dbl_kind, 0.902911684950495_dbl_kind, 0.9401268618548_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,161) = (/ 0.890154779094839_dbl_kind, 0.892610388891411_dbl_kind, 0.894250801566393_dbl_kind, 0.90295516231212_dbl_kind, 0.940239833996224_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,162) = (/ 0.890162048691411_dbl_kind, 0.892621972613864_dbl_kind, 0.894266361268792_dbl_kind, 0.902998400834917_dbl_kind, 0.940352195665015_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,163) = (/ 0.890169243556154_dbl_kind, 0.89263343845576_dbl_kind, 0.894281786083894_dbl_kind, 0.903041403865289_dbl_kind, 0.940463952704933_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,164) = (/ 0.890176364676258_dbl_kind, 0.892644788113103_dbl_kind, 0.894297078972673_dbl_kind, 0.903084174683071_dbl_kind, 0.940575110864248_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,165) = (/ 0.890183413021149_dbl_kind, 0.892656023269718_dbl_kind, 0.89431224283295_dbl_kind, 0.90312671650259_dbl_kind, 0.940685675797919_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,166) = (/ 0.890190389543108_dbl_kind, 0.892667145596705_dbl_kind, 0.894327280500298_dbl_kind, 0.903169032473719_dbl_kind, 0.940795653069726_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,167) = (/ 0.890197295177848_dbl_kind, 0.892678156751916_dbl_kind, 0.894342194748938_dbl_kind, 0.903211125682924_dbl_kind, 0.940905048154354_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,168) = (/ 0.890204130845076_dbl_kind, 0.892689058379446_dbl_kind, 0.894356988292616_dbl_kind, 0.903252999154304_dbl_kind, 0.941013866439407_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,169) = (/ 0.890210897449021_dbl_kind, 0.892699852109146_dbl_kind, 0.894371663785475_dbl_kind, 0.903294655850631_dbl_kind, 0.941122113227381_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,170) = (/ 0.890217595878947_dbl_kind, 0.892710539556146_dbl_kind, 0.894386223822905_dbl_kind, 0.903336098674379_dbl_kind, 0.941229793737574_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,171) = (/ 0.890224227009638_dbl_kind, 0.892721122320416_dbl_kind, 0.894400670942399_dbl_kind, 0.903377330468759_dbl_kind, 0.941336913107958_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,172) = (/ 0.890230791701858_dbl_kind, 0.892731601986329_dbl_kind, 0.894415007624378_dbl_kind, 0.903418354018734_dbl_kind, 0.94144347639699_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,173) = (/ 0.890237290802799_dbl_kind, 0.892741980122261_dbl_kind, 0.894429236293035_dbl_kind, 0.903459172052042_dbl_kind, 0.941549488585386_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,174) = (/ 0.890243725146499_dbl_kind, 0.892752258280207_dbl_kind, 0.894443359317146_dbl_kind, 0.90349978724021_dbl_kind, 0.941654954577845_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,175) = (/ 0.890250095554245_dbl_kind, 0.892762437995419_dbl_kind, 0.894457379010895_dbl_kind, 0.903540202199558_dbl_kind, 0.941759879204716_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,176) = (/ 0.890256402834956_dbl_kind, 0.892772520786066_dbl_kind, 0.89447129763468_dbl_kind, 0.903580419492211_dbl_kind, 0.941864267223648_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,177) = (/ 0.890262647785552_dbl_kind, 0.892782508152925_dbl_kind, 0.894485117395924_dbl_kind, 0.90362044162709_dbl_kind, 0.941968123321168_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,178) = (/ 0.890268831191296_dbl_kind, 0.892792401579084_dbl_kind, 0.894498840449872_dbl_kind, 0.903660271060909_dbl_kind, 0.942071452114239_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,179) = (/ 0.890274953826125_dbl_kind, 0.892802202529674_dbl_kind, 0.894512468900394_dbl_kind, 0.903699910199155_dbl_kind, 0.942174258151766_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,180) = (/ 0.890281016452973_dbl_kind, 0.892811912451624_dbl_kind, 0.89452600480078_dbl_kind, 0.903739361397083_dbl_kind, 0.942276545916068_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,181) = (/ 0.89028701982406_dbl_kind, 0.892821532773436_dbl_kind, 0.894539450154529_dbl_kind, 0.903778626960671_dbl_kind, 0.942378319824318_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,182) = (/ 0.890292964681184_dbl_kind, 0.892831064904983_dbl_kind, 0.894552806916142_dbl_kind, 0.903817709147606_dbl_kind, 0.942479584229921_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,183) = (/ 0.890298851755988_dbl_kind, 0.892840510237329_dbl_kind, 0.894566076991908_dbl_kind, 0.903856610168232_dbl_kind, 0.942580343423893_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,184) = (/ 0.890304681770221_dbl_kind, 0.892849870142572_dbl_kind, 0.894579262240688_dbl_kind, 0.903895332186508_dbl_kind, 0.942680601636176_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,185) = (/ 0.890310455435979_dbl_kind, 0.892859145973705_dbl_kind, 0.894592364474701_dbl_kind, 0.90393387732095_dbl_kind, 0.942780363036926_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,186) = (/ 0.890316173455936_dbl_kind, 0.892868339064495_dbl_kind, 0.894605385460302_dbl_kind, 0.903972247645572_dbl_kind, 0.942879631737776_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,187) = (/ 0.890321836523572_dbl_kind, 0.892877450729394_dbl_kind, 0.89461832691876_dbl_kind, 0.904010445190813_dbl_kind, 0.942978411793066_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,188) = (/ 0.890327445323374_dbl_kind, 0.892886482263455_dbl_kind, 0.894631190527038_dbl_kind, 0.904048471944454_dbl_kind, 0.943076707201024_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,189) = (/ 0.890333000531039_dbl_kind, 0.892895434942271_dbl_kind, 0.894643977918568_dbl_kind, 0.904086329852532_dbl_kind, 0.94317452190494_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,190) = (/ 0.890338502813661_dbl_kind, 0.892904310021941_dbl_kind, 0.894656690684018_dbl_kind, 0.904124020820239_dbl_kind, 0.943271859794294_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,191) = (/ 0.890343952829913_dbl_kind, 0.892913108739043_dbl_kind, 0.89466933037207_dbl_kind, 0.904161546712814_dbl_kind, 0.943368724705867_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,192) = (/ 0.890349351230213_dbl_kind, 0.892921832310624_dbl_kind, 0.894681898490184_dbl_kind, 0.904198909356424_dbl_kind, 0.943465120424809_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,193) = (/ 0.890354698656887_dbl_kind, 0.892930481934216_dbl_kind, 0.894694396505362_dbl_kind, 0.904236110539033_dbl_kind, 0.943561050685701_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,194) = (/ 0.890359995744324_dbl_kind, 0.892939058787859_dbl_kind, 0.894706825844915_dbl_kind, 0.904273152011265_dbl_kind, 0.943656519173563_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,195) = (/ 0.890365243119116_dbl_kind, 0.892947564030143_dbl_kind, 0.894719187897221_dbl_kind, 0.904310035487251_dbl_kind, 0.943751529524865_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,196) = (/ 0.890370441400203_dbl_kind, 0.892955998800262_dbl_kind, 0.894731484012482_dbl_kind, 0.904346762645468_dbl_kind, 0.94384608532849_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,197) = (/ 0.890375591198999_dbl_kind, 0.892964364218084_dbl_kind, 0.894743715503474_dbl_kind, 0.904383335129562_dbl_kind, 0.94394019012669_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,198) = (/ 0.890380693119516_dbl_kind, 0.892972661384232_dbl_kind, 0.894755883646305_dbl_kind, 0.904419754549168_dbl_kind, 0.944033847415996_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,199) = (/ 0.890385747758485_dbl_kind, 0.892980891380182_dbl_kind, 0.894767989681153_dbl_kind, 0.904456022480708_dbl_kind, 0.944127060648135_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,200) = (/ 0.890390755705467_dbl_kind, 0.892989055268367_dbl_kind, 0.89478003481301_dbl_kind, 0.904492140468185_dbl_kind, 0.9442198332309_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,201) = (/ 0.89039571754296_dbl_kind, 0.892997154092297_dbl_kind, 0.894792020212421_dbl_kind, 0.90452811002396_dbl_kind, 0.944312168529002_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,202) = (/ 0.890400633846501_dbl_kind, 0.893005188876689_dbl_kind, 0.894803947016219_dbl_kind, 0.904563932629516_dbl_kind, 0.944404069864913_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,203) = (/ 0.89040550518476_dbl_kind, 0.8930131606276_dbl_kind, 0.894815816328244_dbl_kind, 0.904599609736216_dbl_kind, 0.94449554051968_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,204) = (/ 0.890410332119637_dbl_kind, 0.893021070332585_dbl_kind, 0.894827629220075_dbl_kind, 0.90463514276604_dbl_kind, 0.944586583733713_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,205) = (/ 0.890415115206346_dbl_kind, 0.893028918960848_dbl_kind, 0.894839386731741_dbl_kind, 0.904670533112314_dbl_kind, 0.944677202707563_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,206) = (/ 0.890419854993499_dbl_kind, 0.89303670746341_dbl_kind, 0.894851089872438_dbl_kind, 0.904705782140428_dbl_kind, 0.944767400602674_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,207) = (/ 0.890424552023191_dbl_kind, 0.893044436773286_dbl_kind, 0.894862739621225_dbl_kind, 0.90474089118853_dbl_kind, 0.94485718054213_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,208) = (/ 0.890429206831069_dbl_kind, 0.89305210780566_dbl_kind, 0.894874336927732_dbl_kind, 0.904775861568227_dbl_kind, 0.944946545611362_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,209) = (/ 0.890433819946412_dbl_kind, 0.89305972145808_dbl_kind, 0.894885882712848_dbl_kind, 0.904810694565251_dbl_kind, 0.945035498858857_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,210) = (/ 0.890438391892199_dbl_kind, 0.893067278610647_dbl_kind, 0.894897377869406_dbl_kind, 0.904845391440127_dbl_kind, 0.945124043296831_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,211) = (/ 0.890442923185176_dbl_kind, 0.893074780126221_dbl_kind, 0.894908823262861_dbl_kind, 0.90487995342882_dbl_kind, 0.945212181901913_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,212) = (/ 0.890447414335921_dbl_kind, 0.893082226850622_dbl_kind, 0.894920219731962_dbl_kind, 0.904914381743377_dbl_kind, 0.945299917615786_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,213) = (/ 0.890451865848909_dbl_kind, 0.893089619612844_dbl_kind, 0.894931568089419_dbl_kind, 0.904948677572543_dbl_kind, 0.945387253345819_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,214) = (/ 0.890456278222571_dbl_kind, 0.893096959225268_dbl_kind, 0.89494286912255_dbl_kind, 0.904982842082378_dbl_kind, 0.945474191965697_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,215) = (/ 0.89046065194935_dbl_kind, 0.893104246483884_dbl_kind, 0.894954123593934_dbl_kind, 0.905016876416853_dbl_kind, 0.945560736316016_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,216) = (/ 0.890464987515763_dbl_kind, 0.893111482168512_dbl_kind, 0.894965332242052_dbl_kind, 0.905050781698433_dbl_kind, 0.945646889204888_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,217) = (/ 0.89046928540245_dbl_kind, 0.893118667043029_dbl_kind, 0.894976495781913_dbl_kind, 0.90508455902865_dbl_kind, 0.945732653408498_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,218) = (/ 0.890473546084232_dbl_kind, 0.893125801855597_dbl_kind, 0.894987614905679_dbl_kind, 0.905118209488663_dbl_kind, 0.945818031671683_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,219) = (/ 0.89047777003016_dbl_kind, 0.893132887338899_dbl_kind, 0.894998690283278_dbl_kind, 0.905151734139801_dbl_kind, 0.945903026708477_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,220) = (/ 0.890481957703569_dbl_kind, 0.893139924210366_dbl_kind, 0.895009722563011_dbl_kind, 0.9051851340241_dbl_kind, 0.945987641202643_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,221) = (/ 0.890486109562121_dbl_kind, 0.893146913172419_dbl_kind, 0.895020712372143_dbl_kind, 0.905218410164824_dbl_kind, 0.946071877808206_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,222) = (/ 0.890490226057859_dbl_kind, 0.893153854912704_dbl_kind, 0.895031660317492_dbl_kind, 0.905251563566971_dbl_kind, 0.946155739149948_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,223) = (/ 0.890494307637257_dbl_kind, 0.893160750104329_dbl_kind, 0.895042566986007_dbl_kind, 0.905284595217769_dbl_kind, 0.946239227823929_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,224) = (/ 0.890498354741257_dbl_kind, 0.893167599406106_dbl_kind, 0.895053432945334_dbl_kind, 0.905317506087168_dbl_kind, 0.946322346397961_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,225) = (/ 0.890502367805325_dbl_kind, 0.893174403462789_dbl_kind, 0.895064258744375_dbl_kind, 0.905350297128298_dbl_kind, 0.946405097412085_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,226) = (/ 0.890506347259488_dbl_kind, 0.893181162905316_dbl_kind, 0.895075044913836_dbl_kind, 0.905382969277944_dbl_kind, 0.946487483379042_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,227) = (/ 0.890510293528387_dbl_kind, 0.89318787835105_dbl_kind, 0.895085791966765_dbl_kind, 0.905415523456985_dbl_kind, 0.946569506784724_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,228) = (/ 0.890514207031312_dbl_kind, 0.893194550404015_dbl_kind, 0.895096500399085_dbl_kind, 0.905447960570836_dbl_kind, 0.946651170088617_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,229) = (/ 0.890518088182255_dbl_kind, 0.893201179655143_dbl_kind, 0.895107170690107_dbl_kind, 0.90548028150987_dbl_kind, 0.946732475724245_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,230) = (/ 0.890521937389948_dbl_kind, 0.89320776668251_dbl_kind, 0.895117803303045_dbl_kind, 0.905512487149837_dbl_kind, 0.946813426099576_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,231) = (/ 0.890525755057904_dbl_kind, 0.893214312051572_dbl_kind, 0.895128398685513_dbl_kind, 0.905544578352263_dbl_kind, 0.946894023597459_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,232) = (/ 0.890529541584468_dbl_kind, 0.89322081631541_dbl_kind, 0.895138957270016_dbl_kind, 0.905576555964845_dbl_kind, 0.946974270576003_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,233) = (/ 0.890533297362853_dbl_kind, 0.893227280014958_dbl_kind, 0.895149479474426_dbl_kind, 0.905608420821832_dbl_kind, 0.947054169369002_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,234) = (/ 0.890537022781183_dbl_kind, 0.893233703679248_dbl_kind, 0.895159965702458_dbl_kind, 0.905640173744398_dbl_kind, 0.947133722286298_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,235) = (/ 0.890540718222539_dbl_kind, 0.893240087825636_dbl_kind, 0.895170416344125_dbl_kind, 0.905671815541_dbl_kind, 0.947212931614173_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,236) = (/ 0.890544384064999_dbl_kind, 0.893246432960038_dbl_kind, 0.89518083177619_dbl_kind, 0.905703347007733_dbl_kind, 0.947291799615714_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,237) = (/ 0.890548020681681_dbl_kind, 0.89325273957716_dbl_kind, 0.895191212362607_dbl_kind, 0.905734768928667_dbl_kind, 0.947370328531177_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,238) = (/ 0.890551628440784_dbl_kind, 0.893259008160728_dbl_kind, 0.895201558454949_dbl_kind, 0.905766082076183_dbl_kind, 0.947448520578338_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,239) = (/ 0.890555207705634_dbl_kind, 0.893265239183714_dbl_kind, 0.89521187039283_dbl_kind, 0.905797287211289_dbl_kind, 0.947526377952841_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,240) = (/ 0.890558758834722_dbl_kind, 0.893271433108558_dbl_kind, 0.895222148504316_dbl_kind, 0.905828385083938_dbl_kind, 0.947603902828547_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,241) = (/ 0.890562282181751_dbl_kind, 0.893277590387393_dbl_kind, 0.895232393106328_dbl_kind, 0.905859376433329_dbl_kind, 0.947681097357838_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,242) = (/ 0.890565778095674_dbl_kind, 0.893283711462267_dbl_kind, 0.89524260450503_dbl_kind, 0.9058902619882_dbl_kind, 0.94775796367197_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,243) = (/ 0.89056924692074_dbl_kind, 0.893289796765358_dbl_kind, 0.895252782996214_dbl_kind, 0.905921042467116_dbl_kind, 0.947834503881375_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,244) = (/ 0.890572688996536_dbl_kind, 0.893295846719186_dbl_kind, 0.895262928865668_dbl_kind, 0.905951718578747_dbl_kind, 0.947910720075973_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,245) = (/ 0.890576104658031_dbl_kind, 0.893301861736834_dbl_kind, 0.895273042389549_dbl_kind, 0.905982291022134_dbl_kind, 0.947986614325482_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,246) = (/ 0.890579494235617_dbl_kind, 0.893307842222149_dbl_kind, 0.895283123834725_dbl_kind, 0.906012760486949_dbl_kind, 0.948062188679704_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,247) = (/ 0.890582858055151_dbl_kind, 0.893313788569957_dbl_kind, 0.89529317345913_dbl_kind, 0.906043127653751_dbl_kind, 0.948137445168825_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,248) = (/ 0.890586196438006_dbl_kind, 0.893319701166259_dbl_kind, 0.895303191512094_dbl_kind, 0.906073393194232_dbl_kind, 0.9482123858037_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,249) = (/ 0.890589509701102_dbl_kind, 0.893325580388442_dbl_kind, 0.89531317823467_dbl_kind, 0.906103557771447_dbl_kind, 0.948287012576124_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,250) = (/ 0.890592798156964_dbl_kind, 0.893331426605472_dbl_kind, 0.895323133859957_dbl_kind, 0.906133622040054_dbl_kind, 0.948361327459115_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,251) = (/ 0.890596062113753_dbl_kind, 0.893337240178088_dbl_kind, 0.895333058613402_dbl_kind, 0.906163586646529_dbl_kind, 0.948435332407175_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,252) = (/ 0.890599301875317_dbl_kind, 0.893343021459001_dbl_kind, 0.895342952713108_dbl_kind, 0.906193452229384_dbl_kind, 0.94850902935655_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,253) = (/ 0.890602517741234_dbl_kind, 0.893348770793082_dbl_kind, 0.895352816370121_dbl_kind, 0.906223219419379_dbl_kind, 0.948582420225493_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,254) = (/ 0.890605710006854_dbl_kind, 0.893354488517543_dbl_kind, 0.895362649788713_dbl_kind, 0.906252888839719_dbl_kind, 0.948655506914516_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,255) = (/ 0.890608878963347_dbl_kind, 0.893360174962132_dbl_kind, 0.895372453166665_dbl_kind, 0.906282461106255_dbl_kind, 0.948728291306627_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,256) = (/ 0.890612024897743_dbl_kind, 0.893365830449303_dbl_kind, 0.895382226695527_dbl_kind, 0.906311936827671_dbl_kind, 0.94880077526758_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,257) = (/ 0.89061514809298_dbl_kind, 0.8933714552944_dbl_kind, 0.895391970560879_dbl_kind, 0.906341316605669_dbl_kind, 0.948872960646107_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,258) = (/ 0.890618248827944_dbl_kind, 0.893377049805833_dbl_kind, 0.895401684942586_dbl_kind, 0.906370601035144_dbl_kind, 0.948944849274153_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,259) = (/ 0.89062132737752_dbl_kind, 0.893382614285244_dbl_kind, 0.89541137001504_dbl_kind, 0.906399790704358_dbl_kind, 0.949016442967102_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,260) = (/ 0.890624384012631_dbl_kind, 0.893388149027681_dbl_kind, 0.895421025947395_dbl_kind, 0.906428886195106_dbl_kind, 0.949087743523997_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,261) = (/ 0.890627419000288_dbl_kind, 0.89339365432176_dbl_kind, 0.895430652903798_dbl_kind, 0.906457888082877_dbl_kind, 0.949158752727763_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,262) = (/ 0.890630432603627_dbl_kind, 0.893399130449832_dbl_kind, 0.89544025104361_dbl_kind, 0.906486796937009_dbl_kind, 0.949229472345416_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,263) = (/ 0.890633425081963_dbl_kind, 0.893404577688137_dbl_kind, 0.895449820521624_dbl_kind, 0.906515613320838_dbl_kind, 0.949299904128278_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,264) = (/ 0.890636396690826_dbl_kind, 0.893409996306965_dbl_kind, 0.895459361488266_dbl_kind, 0.906544337791848_dbl_kind, 0.94937004981218_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,265) = (/ 0.890639347682012_dbl_kind, 0.893415386570807_dbl_kind, 0.895468874089801_dbl_kind, 0.906572970901807_dbl_kind, 0.949439911117666_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,266) = (/ 0.890642278303628_dbl_kind, 0.893420748738507_dbl_kind, 0.895478358468526_dbl_kind, 0.906601513196904_dbl_kind, 0.949509489750189_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,267) = (/ 0.89064518880013_dbl_kind, 0.893426083063409_dbl_kind, 0.89548781476296_dbl_kind, 0.906629965217883_dbl_kind, 0.949578787400303_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,268) = (/ 0.890648079412375_dbl_kind, 0.893431389793501_dbl_kind, 0.895497243108019_dbl_kind, 0.906658327500169_dbl_kind, 0.949647805743865_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,269) = (/ 0.890650950377661_dbl_kind, 0.893436669171556_dbl_kind, 0.895506643635198_dbl_kind, 0.906686600573986_dbl_kind, 0.949716546442204_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,270) = (/ 0.890653801929776_dbl_kind, 0.893441921435271_dbl_kind, 0.895516016472736_dbl_kind, 0.906714784964487_dbl_kind, 0.949785011142327_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,271) = (/ 0.890656634299036_dbl_kind, 0.893447146817406_dbl_kind, 0.895525361745779_dbl_kind, 0.906742881191855_dbl_kind, 0.949853201477077_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,272) = (/ 0.890659447712336_dbl_kind, 0.893452345545912_dbl_kind, 0.89553467957654_dbl_kind, 0.906770889771425_dbl_kind, 0.949921119065326_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,273) = (/ 0.890662242393189_dbl_kind, 0.893457517844062_dbl_kind, 0.895543970084447_dbl_kind, 0.906798811213786_dbl_kind, 0.949988765512144_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,274) = (/ 0.890665018561774_dbl_kind, 0.893462663930581_dbl_kind, 0.895553233386291_dbl_kind, 0.906826646024887_dbl_kind, 0.950056142408963_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,275) = (/ 0.890667776434978_dbl_kind, 0.893467784019767_dbl_kind, 0.895562469596366_dbl_kind, 0.906854394706134_dbl_kind, 0.95012325133376_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,276) = (/ 0.89067051622644_dbl_kind, 0.893472878321614_dbl_kind, 0.895571678826604_dbl_kind, 0.906882057754493_dbl_kind, 0.950190093851205_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,277) = (/ 0.890673238146594_dbl_kind, 0.893477947041932_dbl_kind, 0.895580861186707_dbl_kind, 0.906909635662573_dbl_kind, 0.950256671512831_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,278) = (/ 0.890675942402715_dbl_kind, 0.89348299038246_dbl_kind, 0.895590016784269_dbl_kind, 0.906937128918729_dbl_kind, 0.950322985857199_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,279) = (/ 0.890678629198958_dbl_kind, 0.893488008540983_dbl_kind, 0.895599145724898_dbl_kind, 0.906964538007141_dbl_kind, 0.950389038410037_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,280) = (/ 0.890681298736405_dbl_kind, 0.89349300171144_dbl_kind, 0.895608248112332_dbl_kind, 0.906991863407902_dbl_kind, 0.950454830684409_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,281) = (/ 0.890683951213107_dbl_kind, 0.893497970084034_dbl_kind, 0.89561732404855_dbl_kind, 0.907019105597098_dbl_kind, 0.950520364180862_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,282) = (/ 0.890686586824124_dbl_kind, 0.893502913845336_dbl_kind, 0.895626373633877_dbl_kind, 0.907046265046888_dbl_kind, 0.950585640387567_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,283) = (/ 0.890689205761571_dbl_kind, 0.89350783317839_dbl_kind, 0.895635396967087_dbl_kind, 0.90707334222558_dbl_kind, 0.950650660780472_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,284) = (/ 0.890691808214656_dbl_kind, 0.893512728262813_dbl_kind, 0.895644394145503_dbl_kind, 0.907100337597706_dbl_kind, 0.950715426823444_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,285) = (/ 0.890694394369728_dbl_kind, 0.893517599274892_dbl_kind, 0.895653365265087_dbl_kind, 0.907127251624091_dbl_kind, 0.95077993996841_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,286) = (/ 0.89069696441031_dbl_kind, 0.893522446387682_dbl_kind, 0.895662310420533_dbl_kind, 0.907154084761925_dbl_kind, 0.950844201655489_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,287) = (/ 0.890699518517147_dbl_kind, 0.893527269771097_dbl_kind, 0.895671229705351_dbl_kind, 0.907180837464827_dbl_kind, 0.950908213313136_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,288) = (/ 0.890702056868247_dbl_kind, 0.893532069592002_dbl_kind, 0.895680123211953_dbl_kind, 0.907207510182909_dbl_kind, 0.950971976358275_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,289) = (/ 0.890704579638915_dbl_kind, 0.893536846014303_dbl_kind, 0.895688991031725_dbl_kind, 0.907234103362845_dbl_kind, 0.951035492196426_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,290) = (/ 0.890707087001801_dbl_kind, 0.893541599199031_dbl_kind, 0.89569783325511_dbl_kind, 0.907260617447922_dbl_kind, 0.951098762221836_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,291) = (/ 0.890709579126936_dbl_kind, 0.89354632930443_dbl_kind, 0.895706649971673_dbl_kind, 0.907287052878104_dbl_kind, 0.951161787817607_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,292) = (/ 0.890712056181773_dbl_kind, 0.893551036486034_dbl_kind, 0.895715441270175_dbl_kind, 0.90731341009009_dbl_kind, 0.951224570355825_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,293) = (/ 0.890714518331224_dbl_kind, 0.893555720896752_dbl_kind, 0.895724207238634_dbl_kind, 0.907339689517365_dbl_kind, 0.951287111197675_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,294) = (/ 0.890716965737701_dbl_kind, 0.893560382686943_dbl_kind, 0.895732947964389_dbl_kind, 0.907365891590251_dbl_kind, 0.95134941169357_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,295) = (/ 0.890719398561154_dbl_kind, 0.893565022004492_dbl_kind, 0.895741663534162_dbl_kind, 0.907392016735967_dbl_kind, 0.95141147318327_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,296) = (/ 0.890721816959114_dbl_kind, 0.893569638994883_dbl_kind, 0.895750354034107_dbl_kind, 0.907418065378669_dbl_kind, 0.951473296995985_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,297) = (/ 0.890724221086719_dbl_kind, 0.893574233801275_dbl_kind, 0.895759019549874_dbl_kind, 0.907444037939506_dbl_kind, 0.951534884450519_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,298) = (/ 0.890726611096764_dbl_kind, 0.893578806564567_dbl_kind, 0.895767660166648_dbl_kind, 0.907469934836658_dbl_kind, 0.951596236855355_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,299) = (/ 0.890728987139732_dbl_kind, 0.893583357423466_dbl_kind, 0.895776275969209_dbl_kind, 0.90749575648539_dbl_kind, 0.951657355508784_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,300) = (/ 0.890731349363831_dbl_kind, 0.893587886514558_dbl_kind, 0.895784867041969_dbl_kind, 0.907521503298091_dbl_kind, 0.951718241699009_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,301) = (/ 0.89073369791503_dbl_kind, 0.893592393972367_dbl_kind, 0.89579343346902_dbl_kind, 0.907547175684316_dbl_kind, 0.951778896704261_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,302) = (/ 0.890736032937098_dbl_kind, 0.893596879929419_dbl_kind, 0.895801975334173_dbl_kind, 0.907572774050828_dbl_kind, 0.951839321792894_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,303) = (/ 0.890738354571636_dbl_kind, 0.893601344516303_dbl_kind, 0.895810492720997_dbl_kind, 0.907598298801638_dbl_kind, 0.951899518223498_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,304) = (/ 0.890740662958114_dbl_kind, 0.893605787861732_dbl_kind, 0.895818985712859_dbl_kind, 0.907623750338047_dbl_kind, 0.951959487245005_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,305) = (/ 0.890742958233905_dbl_kind, 0.893610210092596_dbl_kind, 0.895827454392951_dbl_kind, 0.907649129058673_dbl_kind, 0.952019230096788_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,306) = (/ 0.890745240534321_dbl_kind, 0.893614611334018_dbl_kind, 0.895835898844331_dbl_kind, 0.907674435359502_dbl_kind, 0.952078748008758_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,307) = (/ 0.890747509992643_dbl_kind, 0.893618991709415_dbl_kind, 0.895844319149949_dbl_kind, 0.907699669633909_dbl_kind, 0.952138042201471_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,308) = (/ 0.890749766740158_dbl_kind, 0.89362335134054_dbl_kind, 0.895852715392676_dbl_kind, 0.907724832272704_dbl_kind, 0.952197113886219_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,309) = (/ 0.890752010906192_dbl_kind, 0.893627690347541_dbl_kind, 0.895861087655332_dbl_kind, 0.907749923664156_dbl_kind, 0.952255964265132_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,310) = (/ 0.890754242618139_dbl_kind, 0.893632008849004_dbl_kind, 0.895869436020708_dbl_kind, 0.907774944194034_dbl_kind, 0.952314594531271_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,311) = (/ 0.890756462001498_dbl_kind, 0.893636306962009_dbl_kind, 0.895877760571597_dbl_kind, 0.907799894245628_dbl_kind, 0.95237300586872_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,312) = (/ 0.890758669179901_dbl_kind, 0.893640584802167_dbl_kind, 0.895886061390807_dbl_kind, 0.907824774199791_dbl_kind, 0.952431199452679_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,313) = (/ 0.890760864275148_dbl_kind, 0.893644842483671_dbl_kind, 0.895894338561189_dbl_kind, 0.907849584434959_dbl_kind, 0.952489176449556_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,314) = (/ 0.890763047407233_dbl_kind, 0.89364908011934_dbl_kind, 0.895902592165651_dbl_kind, 0.907874325327185_dbl_kind, 0.952546938017058_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,315) = (/ 0.890765218694378_dbl_kind, 0.893653297820657_dbl_kind, 0.895910822287176_dbl_kind, 0.907898997250166_dbl_kind, 0.952604485304275_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,316) = (/ 0.890767378253065_dbl_kind, 0.893657495697814_dbl_kind, 0.895919029008843_dbl_kind, 0.907923600575269_dbl_kind, 0.952661819451771_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,317) = (/ 0.890769526198059_dbl_kind, 0.893661673859751_dbl_kind, 0.895927212413834_dbl_kind, 0.907948135671559_dbl_kind, 0.952718941591673_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,318) = (/ 0.890771662642443_dbl_kind, 0.893665832414191_dbl_kind, 0.895935372585457_dbl_kind, 0.907972602905825_dbl_kind, 0.95277585284775_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,319) = (/ 0.890773787697645_dbl_kind, 0.893669971467683_dbl_kind, 0.895943509607149_dbl_kind, 0.907997002642603_dbl_kind, 0.95283255433549_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,320) = (/ 0.890775901473467_dbl_kind, 0.893674091125633_dbl_kind, 0.895951623562495_dbl_kind, 0.908021335244203_dbl_kind, 0.952889047162205_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,321) = (/ 0.890778004078111_dbl_kind, 0.893678191492343_dbl_kind, 0.895959714535234_dbl_kind, 0.908045601070731_dbl_kind, 0.952945332427089_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,322) = (/ 0.890780095618208_dbl_kind, 0.893682272671041_dbl_kind, 0.895967782609272_dbl_kind, 0.908069800480115_dbl_kind, 0.953001411221312_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,323) = (/ 0.890782176198846_dbl_kind, 0.893686334763916_dbl_kind, 0.895975827868684_dbl_kind, 0.908093933828122_dbl_kind, 0.953057284628095_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,324) = (/ 0.890784245923595_dbl_kind, 0.893690377872151_dbl_kind, 0.895983850397729_dbl_kind, 0.908118001468386_dbl_kind, 0.953112953722786_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,325) = (/ 0.890786304894532_dbl_kind, 0.89369440209595_dbl_kind, 0.89599185028085_dbl_kind, 0.908142003752429_dbl_kind, 0.953168419572941_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,326) = (/ 0.890788353212271_dbl_kind, 0.893698407534569_dbl_kind, 0.895999827602683_dbl_kind, 0.908165941029678_dbl_kind, 0.953223683238401_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,327) = (/ 0.890790390975987_dbl_kind, 0.893702394286346_dbl_kind, 0.896007782448063_dbl_kind, 0.908189813647487_dbl_kind, 0.95327874577136_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,328) = (/ 0.890792418283437_dbl_kind, 0.893706362448728_dbl_kind, 0.896015714902023_dbl_kind, 0.908213621951161_dbl_kind, 0.953333608216449_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,329) = (/ 0.89079443523099_dbl_kind, 0.893710312118298_dbl_kind, 0.896023625049802_dbl_kind, 0.90823736628397_dbl_kind, 0.953388271610799_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,330) = (/ 0.890796441913649_dbl_kind, 0.893714243390797_dbl_kind, 0.896031512976845_dbl_kind, 0.908261046987171_dbl_kind, 0.953442736984121_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,331) = (/ 0.890798438425073_dbl_kind, 0.893718156361155_dbl_kind, 0.896039378768807_dbl_kind, 0.908284664400028_dbl_kind, 0.953497005358774_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,332) = (/ 0.890800424857606_dbl_kind, 0.893722051123511_dbl_kind, 0.896047222511551_dbl_kind, 0.908308218859826_dbl_kind, 0.953551077749837_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,333) = (/ 0.890802401302293_dbl_kind, 0.893725927771238_dbl_kind, 0.896055044291151_dbl_kind, 0.908331710701893_dbl_kind, 0.953604955165176_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,334) = (/ 0.890804367848906_dbl_kind, 0.893729786396966_dbl_kind, 0.896062844193893_dbl_kind, 0.908355140259615_dbl_kind, 0.953658638605515_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,335) = (/ 0.890806324585969_dbl_kind, 0.893733627092599_dbl_kind, 0.89607062230627_dbl_kind, 0.908378507864455_dbl_kind, 0.953712129064504_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,336) = (/ 0.890808271600774_dbl_kind, 0.893737449949343_dbl_kind, 0.896078378714986_dbl_kind, 0.908401813845968_dbl_kind, 0.953765427528779_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,337) = (/ 0.890810208979408_dbl_kind, 0.893741255057719_dbl_kind, 0.896086113506953_dbl_kind, 0.908425058531816_dbl_kind, 0.953818534978042_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,338) = (/ 0.890812136806771_dbl_kind, 0.893745042507589_dbl_kind, 0.896093826769286_dbl_kind, 0.90844824224779_dbl_kind, 0.953871452385111_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,339) = (/ 0.890814055166596_dbl_kind, 0.893748812388171_dbl_kind, 0.896101518589306_dbl_kind, 0.908471365317818_dbl_kind, 0.953924180715994_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,340) = (/ 0.890815964141473_dbl_kind, 0.893752564788056_dbl_kind, 0.896109189054532_dbl_kind, 0.908494428063985_dbl_kind, 0.953976720929953_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,341) = (/ 0.890817863812866_dbl_kind, 0.893756299795227_dbl_kind, 0.89611683825268_dbl_kind, 0.908517430806549_dbl_kind, 0.954029073979559_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,342) = (/ 0.890819754261132_dbl_kind, 0.893760017497078_dbl_kind, 0.896124466271661_dbl_kind, 0.908540373863951_dbl_kind, 0.954081240810759_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,343) = (/ 0.890821635565545_dbl_kind, 0.893763717980426_dbl_kind, 0.896132073199575_dbl_kind, 0.908563257552833_dbl_kind, 0.954133222362943_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,344) = (/ 0.890823507804309_dbl_kind, 0.893767401331528_dbl_kind, 0.896139659124706_dbl_kind, 0.908586082188052_dbl_kind, 0.954185019568989_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,345) = (/ 0.89082537105458_dbl_kind, 0.893771067636096_dbl_kind, 0.896147224135521_dbl_kind, 0.908608848082693_dbl_kind, 0.954236633355345_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,346) = (/ 0.890827225392486_dbl_kind, 0.893774716979314_dbl_kind, 0.89615476832066_dbl_kind, 0.908631555548083_dbl_kind, 0.954288064642065_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,347) = (/ 0.890829070893139_dbl_kind, 0.893778349445847_dbl_kind, 0.896162291768937_dbl_kind, 0.908654204893804_dbl_kind, 0.954339314342882_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,348) = (/ 0.890830907630657_dbl_kind, 0.893781965119856_dbl_kind, 0.89616979456933_dbl_kind, 0.908676796427707_dbl_kind, 0.954390383365264_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,349) = (/ 0.890832735678183_dbl_kind, 0.893785564085015_dbl_kind, 0.896177276810976_dbl_kind, 0.908699330455922_dbl_kind, 0.954441272610464_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,350) = (/ 0.890834555107896_dbl_kind, 0.893789146424516_dbl_kind, 0.89618473858317_dbl_kind, 0.908721807282878_dbl_kind, 0.954491982973588_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,351) = (/ 0.890836365991034_dbl_kind, 0.893792712221085_dbl_kind, 0.896192179975351_dbl_kind, 0.908744227211306_dbl_kind, 0.954542515343638_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,352) = (/ 0.890838168397903_dbl_kind, 0.893796261556996_dbl_kind, 0.896199601077102_dbl_kind, 0.908766590542255_dbl_kind, 0.954592870603572_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,353) = (/ 0.890839962397902_dbl_kind, 0.893799794514074_dbl_kind, 0.896207001978142_dbl_kind, 0.908788897575108_dbl_kind, 0.954643049630368_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,354) = (/ 0.890841748059531_dbl_kind, 0.893803311173712_dbl_kind, 0.896214382768321_dbl_kind, 0.908811148607589_dbl_kind, 0.95469305329506_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,355) = (/ 0.89084352545041_dbl_kind, 0.893806811616881_dbl_kind, 0.896221743537608_dbl_kind, 0.908833343935775_dbl_kind, 0.954742882462802_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,356) = (/ 0.890845294637294_dbl_kind, 0.893810295924135_dbl_kind, 0.896229084376092_dbl_kind, 0.908855483854109_dbl_kind, 0.954792537992923_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,357) = (/ 0.890847055686085_dbl_kind, 0.893813764175623_dbl_kind, 0.896236405373971_dbl_kind, 0.908877568655412_dbl_kind, 0.954842020738972_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,358) = (/ 0.890848808661852_dbl_kind, 0.8938172164511_dbl_kind, 0.896243706621541_dbl_kind, 0.908899598630891_dbl_kind, 0.954891331548773_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,359) = (/ 0.890850553628839_dbl_kind, 0.893820652829931_dbl_kind, 0.896250988209201_dbl_kind, 0.908921574070151_dbl_kind, 0.954940471264469_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,360) = (/ 0.890852290650483_dbl_kind, 0.893824073391102_dbl_kind, 0.896258250227431_dbl_kind, 0.908943495261208_dbl_kind, 0.95498944072259_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,361) = (/ 0.890854019789427_dbl_kind, 0.893827478213228_dbl_kind, 0.896265492766797_dbl_kind, 0.908965362490495_dbl_kind, 0.955038240754077_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,362) = (/ 0.890855741107532_dbl_kind, 0.893830867374557_dbl_kind, 0.896272715917938_dbl_kind, 0.908987176042878_dbl_kind, 0.955086872184355_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,363) = (/ 0.89085745466589_dbl_kind, 0.893834240952982_dbl_kind, 0.896279919771557_dbl_kind, 0.909008936201661_dbl_kind, 0.955135335833363_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,364) = (/ 0.890859160524842_dbl_kind, 0.893837599026044_dbl_kind, 0.89628710441842_dbl_kind, 0.909030643248597_dbl_kind, 0.955183632515618_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,365) = (/ 0.890860858743981_dbl_kind, 0.893840941670941_dbl_kind, 0.896294269949344_dbl_kind, 0.909052297463903_dbl_kind, 0.955231763040245_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,366) = (/ 0.890862549382174_dbl_kind, 0.893844268964533_dbl_kind, 0.896301416455189_dbl_kind, 0.909073899126263_dbl_kind, 0.95527972821104_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,367) = (/ 0.890864232497568_dbl_kind, 0.893847580983346_dbl_kind, 0.896308544026855_dbl_kind, 0.90909544851284_dbl_kind, 0.955327528826511_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,368) = (/ 0.890865908147604_dbl_kind, 0.893850877803583_dbl_kind, 0.896315652755269_dbl_kind, 0.909116945899286_dbl_kind, 0.955375165679916_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,369) = (/ 0.89086757638903_dbl_kind, 0.893854159501124_dbl_kind, 0.896322742731383_dbl_kind, 0.909138391559753_dbl_kind, 0.955422639559315_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,370) = (/ 0.890869237277909_dbl_kind, 0.893857426151535_dbl_kind, 0.896329814046162_dbl_kind, 0.909159785766897_dbl_kind, 0.955469951247623_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,371) = (/ 0.890870890869633_dbl_kind, 0.89386067783007_dbl_kind, 0.896336866790581_dbl_kind, 0.909181128791892_dbl_kind, 0.955517101522638_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,372) = (/ 0.890872537218933_dbl_kind, 0.893863914611678_dbl_kind, 0.896343901055613_dbl_kind, 0.909202420904438_dbl_kind, 0.955564091157094_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,373) = (/ 0.890874176379889_dbl_kind, 0.893867136571005_dbl_kind, 0.896350916932224_dbl_kind, 0.909223662372766_dbl_kind, 0.955610920918704_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,374) = (/ 0.890875808405942_dbl_kind, 0.893870343782402_dbl_kind, 0.896357914511367_dbl_kind, 0.909244853463653_dbl_kind, 0.955657591570203_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,375) = (/ 0.890877433349905_dbl_kind, 0.893873536319926_dbl_kind, 0.896364893883973_dbl_kind, 0.909265994442424_dbl_kind, 0.955704103869385_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,376) = (/ 0.890879051263969_dbl_kind, 0.893876714257345_dbl_kind, 0.896371855140943_dbl_kind, 0.909287085572966_dbl_kind, 0.955750458569156_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,377) = (/ 0.890880662199716_dbl_kind, 0.893879877668141_dbl_kind, 0.896378798373141_dbl_kind, 0.909308127117733_dbl_kind, 0.955796656417567_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,378) = (/ 0.89088226620813_dbl_kind, 0.893883026625515_dbl_kind, 0.896385723671391_dbl_kind, 0.909329119337754_dbl_kind, 0.955842698157851_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,379) = (/ 0.890883863339603_dbl_kind, 0.893886161202389_dbl_kind, 0.89639263112646_dbl_kind, 0.909350062492643_dbl_kind, 0.955888584528478_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,380) = (/ 0.890885453643947_dbl_kind, 0.89388928147141_dbl_kind, 0.896399520829063_dbl_kind, 0.909370956840607_dbl_kind, 0.955934316263182_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,381) = (/ 0.890887037170401_dbl_kind, 0.893892387504952_dbl_kind, 0.896406392869849_dbl_kind, 0.909391802638453_dbl_kind, 0.955979894091005_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,382) = (/ 0.890888613967643_dbl_kind, 0.89389547937512_dbl_kind, 0.896413247339391_dbl_kind, 0.909412600141593_dbl_kind, 0.956025318736338_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,383) = (/ 0.890890184083795_dbl_kind, 0.893898557153753_dbl_kind, 0.896420084328189_dbl_kind, 0.909433349604058_dbl_kind, 0.956070590918965_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,384) = (/ 0.890891747566435_dbl_kind, 0.893901620912426_dbl_kind, 0.896426903926652_dbl_kind, 0.909454051278501_dbl_kind, 0.956115711354084_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,385) = (/ 0.890893304462601_dbl_kind, 0.89390467072245_dbl_kind, 0.8964337062251_dbl_kind, 0.909474705416207_dbl_kind, 0.956160680752364_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,386) = (/ 0.890894854818805_dbl_kind, 0.893907706654879_dbl_kind, 0.896440491313751_dbl_kind, 0.909495312267095_dbl_kind, 0.956205499819975_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,387) = (/ 0.890896398681037_dbl_kind, 0.893910728780511_dbl_kind, 0.896447259282719_dbl_kind, 0.909515872079735_dbl_kind, 0.956250169258624_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,388) = (/ 0.890897936094773_dbl_kind, 0.893913737169885_dbl_kind, 0.896454010222005_dbl_kind, 0.909536385101346_dbl_kind, 0.956294689765591_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,389) = (/ 0.890899467104985_dbl_kind, 0.893916731893292_dbl_kind, 0.896460744221489_dbl_kind, 0.909556851577808_dbl_kind, 0.956339062033775_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,390) = (/ 0.890900991756147_dbl_kind, 0.893919713020768_dbl_kind, 0.896467461370926_dbl_kind, 0.909577271753669_dbl_kind, 0.956383286751714_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,391) = (/ 0.890902510092243_dbl_kind, 0.893922680622103_dbl_kind, 0.89647416175994_dbl_kind, 0.909597645872149_dbl_kind, 0.956427364603642_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,392) = (/ 0.890904022156773_dbl_kind, 0.893925634766835_dbl_kind, 0.896480845478015_dbl_kind, 0.909617974175151_dbl_kind, 0.956471296269497_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,393) = (/ 0.890905527992762_dbl_kind, 0.893928575524261_dbl_kind, 0.896487512614492_dbl_kind, 0.909638256903265_dbl_kind, 0.956515082424984_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,394) = (/ 0.890907027642765_dbl_kind, 0.89393150296343_dbl_kind, 0.896494163258557_dbl_kind, 0.909658494295776_dbl_kind, 0.956558723741594_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,395) = (/ 0.890908521148877_dbl_kind, 0.89393441715315_dbl_kind, 0.896500797499245_dbl_kind, 0.909678686590669_dbl_kind, 0.956602220886638_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,396) = (/ 0.890910008552735_dbl_kind, 0.893937318161986_dbl_kind, 0.896507415425424_dbl_kind, 0.90969883402464_dbl_kind, 0.956645574523288_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,397) = (/ 0.890911489895531_dbl_kind, 0.893940206058264_dbl_kind, 0.896514017125792_dbl_kind, 0.909718936833096_dbl_kind, 0.956688785310607_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,398) = (/ 0.890912965218012_dbl_kind, 0.893943080910068_dbl_kind, 0.896520602688876_dbl_kind, 0.909738995250167_dbl_kind, 0.956731853903579_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,399) = (/ 0.890914434560489_dbl_kind, 0.893945942785247_dbl_kind, 0.89652717220302_dbl_kind, 0.909759009508714_dbl_kind, 0.956774780953149_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,400) = (/ 0.890915897962846_dbl_kind, 0.89394879175141_dbl_kind, 0.896533725756382_dbl_kind, 0.909778979840324_dbl_kind, 0.95681756710625_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,401) = (/ 0.890917355464538_dbl_kind, 0.893951627875934_dbl_kind, 0.896540263436928_dbl_kind, 0.909798906475331_dbl_kind, 0.956860213005839_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,402) = (/ 0.89091880710461_dbl_kind, 0.893954451225956_dbl_kind, 0.896546785332427_dbl_kind, 0.909818789642814_dbl_kind, 0.956902719290926_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,403) = (/ 0.890920252921689_dbl_kind, 0.893957261868383_dbl_kind, 0.896553291530446_dbl_kind, 0.909838629570604_dbl_kind, 0.956945086596606_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,404) = (/ 0.890921692953997_dbl_kind, 0.893960059869885_dbl_kind, 0.896559782118342_dbl_kind, 0.90985842648529_dbl_kind, 0.956987315554091_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,405) = (/ 0.890923127239359_dbl_kind, 0.893962845296903_dbl_kind, 0.896566257183262_dbl_kind, 0.909878180612229_dbl_kind, 0.957029406790746_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,406) = (/ 0.890924555815203_dbl_kind, 0.893965618215642_dbl_kind, 0.89657271681213_dbl_kind, 0.909897892175547_dbl_kind, 0.957071360930106_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,407) = (/ 0.890925978718565_dbl_kind, 0.89396837869208_dbl_kind, 0.896579161091652_dbl_kind, 0.909917561398149_dbl_kind, 0.957113178591925_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,408) = (/ 0.890927395986101_dbl_kind, 0.893971126791962_dbl_kind, 0.8965855901083_dbl_kind, 0.909937188501719_dbl_kind, 0.957154860392189_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,409) = (/ 0.890928807654086_dbl_kind, 0.893973862580803_dbl_kind, 0.896592003948316_dbl_kind, 0.909956773706734_dbl_kind, 0.957196406943157_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,410) = (/ 0.890930213758422_dbl_kind, 0.89397658612389_dbl_kind, 0.896598402697703_dbl_kind, 0.909976317232464_dbl_kind, 0.957237818853385_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,411) = (/ 0.89093161433464_dbl_kind, 0.89397929748628_dbl_kind, 0.896604786442223_dbl_kind, 0.909995819296982_dbl_kind, 0.957279096727756_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,412) = (/ 0.890933009417912_dbl_kind, 0.893981996732802_dbl_kind, 0.896611155267386_dbl_kind, 0.910015280117163_dbl_kind, 0.957320241167512_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,413) = (/ 0.890934399043045_dbl_kind, 0.893984683928057_dbl_kind, 0.896617509258454_dbl_kind, 0.910034699908697_dbl_kind, 0.957361252770279_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,414) = (/ 0.890935783244496_dbl_kind, 0.893987359136419_dbl_kind, 0.896623848500433_dbl_kind, 0.910054078886091_dbl_kind, 0.957402132130096_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,415) = (/ 0.890937162056371_dbl_kind, 0.893990022422036_dbl_kind, 0.896630173078065_dbl_kind, 0.910073417262677_dbl_kind, 0.957442879837445_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,416) = (/ 0.890938535512431_dbl_kind, 0.893992673848826_dbl_kind, 0.896636483075829_dbl_kind, 0.910092715250613_dbl_kind, 0.957483496479276_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,417) = (/ 0.890939903646096_dbl_kind, 0.893995313480486_dbl_kind, 0.896642778577935_dbl_kind, 0.910111973060892_dbl_kind, 0.957523982639039_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,418) = (/ 0.890941266490451_dbl_kind, 0.893997941380481_dbl_kind, 0.896649059668318_dbl_kind, 0.91013119090335_dbl_kind, 0.957564338896703_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,419) = (/ 0.890942624078248_dbl_kind, 0.894000557612057_dbl_kind, 0.896655326430639_dbl_kind, 0.910150368986665_dbl_kind, 0.957604565828794_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,420) = (/ 0.890943976441913_dbl_kind, 0.894003162238229_dbl_kind, 0.896661578948275_dbl_kind, 0.910169507518367_dbl_kind, 0.957644664008414_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,421) = (/ 0.890945323613548_dbl_kind, 0.894005755321791_dbl_kind, 0.896667817304318_dbl_kind, 0.910188606704842_dbl_kind, 0.957684634005265_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,422) = (/ 0.890946665624935_dbl_kind, 0.894008336925311_dbl_kind, 0.896674041581575_dbl_kind, 0.910207666751338_dbl_kind, 0.957724476385684_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,423) = (/ 0.890948002507541_dbl_kind, 0.894010907111131_dbl_kind, 0.896680251862554_dbl_kind, 0.91022668786197_dbl_kind, 0.957764191712667_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,424) = (/ 0.890949334292523_dbl_kind, 0.894013465941372_dbl_kind, 0.896686448229474_dbl_kind, 0.910245670239725_dbl_kind, 0.957803780545883_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,425) = (/ 0.89095066101073_dbl_kind, 0.894016013477929_dbl_kind, 0.896692630764251_dbl_kind, 0.910264614086467_dbl_kind, 0.95784324344172_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,426) = (/ 0.890951982692707_dbl_kind, 0.894018549782474_dbl_kind, 0.896698799548497_dbl_kind, 0.910283519602942_dbl_kind, 0.957882580953291_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,427) = (/ 0.8909532993687_dbl_kind, 0.894021074916454_dbl_kind, 0.896704954663522_dbl_kind, 0.910302386988786_dbl_kind, 0.957921793630468_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,428) = (/ 0.890954611068659_dbl_kind, 0.894023588941097_dbl_kind, 0.896711096190322_dbl_kind, 0.910321216442527_dbl_kind, 0.957960882019909_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,429) = (/ 0.890955917822242_dbl_kind, 0.894026091917402_dbl_kind, 0.896717224209585_dbl_kind, 0.910340008161589_dbl_kind, 0.957999846665076_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,430) = (/ 0.890957219658817_dbl_kind, 0.89402858390615_dbl_kind, 0.896723338801681_dbl_kind, 0.910358762342302_dbl_kind, 0.958038688106262_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,431) = (/ 0.890958516607469_dbl_kind, 0.894031064967897_dbl_kind, 0.89672944004666_dbl_kind, 0.910377479179905_dbl_kind, 0.958077406880616_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,432) = (/ 0.890959808696999_dbl_kind, 0.894033535162977_dbl_kind, 0.896735528024254_dbl_kind, 0.910396158868547_dbl_kind, 0.958116003522167_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,433) = (/ 0.89096109595593_dbl_kind, 0.894035994551503_dbl_kind, 0.89674160281387_dbl_kind, 0.910414801601297_dbl_kind, 0.958154478561845_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,434) = (/ 0.890962378412512_dbl_kind, 0.894038443193362_dbl_kind, 0.896747664494586_dbl_kind, 0.910433407570148_dbl_kind, 0.958192832527505_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,435) = (/ 0.890963656094722_dbl_kind, 0.894040881148224_dbl_kind, 0.896753713145151_dbl_kind, 0.91045197696602_dbl_kind, 0.958231065943955_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,436) = (/ 0.890964929030268_dbl_kind, 0.894043308475533_dbl_kind, 0.896759748843983_dbl_kind, 0.910470509978768_dbl_kind, 0.95826917933297_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,437) = (/ 0.890966197246594_dbl_kind, 0.894045725234513_dbl_kind, 0.896765771669166_dbl_kind, 0.910489006797183_dbl_kind, 0.958307173213327_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,438) = (/ 0.890967460770881_dbl_kind, 0.894048131484167_dbl_kind, 0.896771781698444_dbl_kind, 0.910507467609002_dbl_kind, 0.958345048100811_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,439) = (/ 0.890968719630053_dbl_kind, 0.894050527283276_dbl_kind, 0.896777779009223_dbl_kind, 0.910525892600908_dbl_kind, 0.958382804508251_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,440) = (/ 0.890969973850776_dbl_kind, 0.8940529126904_dbl_kind, 0.896783763678568_dbl_kind, 0.910544281958536_dbl_kind, 0.958420442945538_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,441) = (/ 0.890971223459464_dbl_kind, 0.894055287763877_dbl_kind, 0.8967897357832_dbl_kind, 0.910562635866481_dbl_kind, 0.958457963919645_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,442) = (/ 0.890972468482282_dbl_kind, 0.894057652561827_dbl_kind, 0.896795695399492_dbl_kind, 0.910580954508299_dbl_kind, 0.95849536793465_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,443) = (/ 0.890973708945147_dbl_kind, 0.894060007142145_dbl_kind, 0.896801642603471_dbl_kind, 0.910599238066513_dbl_kind, 0.958532655491757_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,444) = (/ 0.890974944873733_dbl_kind, 0.894062351562508_dbl_kind, 0.896807577470813_dbl_kind, 0.91061748672262_dbl_kind, 0.958569827089319_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,445) = (/ 0.890976176293472_dbl_kind, 0.894064685880372_dbl_kind, 0.896813500076841_dbl_kind, 0.910635700657092_dbl_kind, 0.958606883222854_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,446) = (/ 0.890977403229557_dbl_kind, 0.894067010152973_dbl_kind, 0.896819410496527_dbl_kind, 0.910653880049384_dbl_kind, 0.958643824385075_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,447) = (/ 0.890978625706948_dbl_kind, 0.894069324437328_dbl_kind, 0.896825308804484_dbl_kind, 0.910672025077935_dbl_kind, 0.958680651065901_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,448) = (/ 0.890979843750369_dbl_kind, 0.894071628790231_dbl_kind, 0.896831195074971_dbl_kind, 0.910690135920177_dbl_kind, 0.95871736375248_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,449) = (/ 0.890981057384317_dbl_kind, 0.894073923268258_dbl_kind, 0.896837069381885_dbl_kind, 0.910708212752539_dbl_kind, 0.958753962929216_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,450) = (/ 0.890982266633059_dbl_kind, 0.894076207927766_dbl_kind, 0.896842931798763_dbl_kind, 0.910726255750447_dbl_kind, 0.958790449077779_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,451) = (/ 0.890983471520638_dbl_kind, 0.894078482824892_dbl_kind, 0.896848782398784_dbl_kind, 0.910744265088334_dbl_kind, 0.958826822677133_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,452) = (/ 0.890984672070875_dbl_kind, 0.894080748015553_dbl_kind, 0.896854621254758_dbl_kind, 0.910762240939642_dbl_kind, 0.958863084203552_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,453) = (/ 0.89098586830737_dbl_kind, 0.894083003555448_dbl_kind, 0.896860448439134_dbl_kind, 0.910780183476828_dbl_kind, 0.958899234130638_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,454) = (/ 0.890987060253509_dbl_kind, 0.894085249500056_dbl_kind, 0.896866264023993_dbl_kind, 0.910798092871368_dbl_kind, 0.958935272929344_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,455) = (/ 0.890988247932458_dbl_kind, 0.894087485904637_dbl_kind, 0.896872068081049_dbl_kind, 0.910815969293761_dbl_kind, 0.958971201067993_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,456) = (/ 0.890989431367177_dbl_kind, 0.894089712824235_dbl_kind, 0.89687786068165_dbl_kind, 0.910833812913534_dbl_kind, 0.95900701901229_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,457) = (/ 0.89099061058041_dbl_kind, 0.894091930313673_dbl_kind, 0.896883641896771_dbl_kind, 0.910851623899248_dbl_kind, 0.959042727225353_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,458) = (/ 0.890991785594698_dbl_kind, 0.894094138427556_dbl_kind, 0.896889411797018_dbl_kind, 0.910869402418499_dbl_kind, 0.959078326167723_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,459) = (/ 0.890992956432374_dbl_kind, 0.894096337220272_dbl_kind, 0.896895170452626_dbl_kind, 0.910887148637927_dbl_kind, 0.95911381629738_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,460) = (/ 0.89099412311557_dbl_kind, 0.894098526745991_dbl_kind, 0.896900917933457_dbl_kind, 0.910904862723217_dbl_kind, 0.959149198069774_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,461) = (/ 0.890995285666217_dbl_kind, 0.894100707058666_dbl_kind, 0.896906654308999_dbl_kind, 0.910922544839106_dbl_kind, 0.959184471937828_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,462) = (/ 0.890996444106047_dbl_kind, 0.89410287821203_dbl_kind, 0.896912379648368_dbl_kind, 0.910940195149385_dbl_kind, 0.959219638351965_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,463) = (/ 0.890997598456597_dbl_kind, 0.894105040259602_dbl_kind, 0.896918094020302_dbl_kind, 0.910957813816905_dbl_kind, 0.959254697760127_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,464) = (/ 0.890998748739209_dbl_kind, 0.894107193254683_dbl_kind, 0.896923797493166_dbl_kind, 0.910975401003582_dbl_kind, 0.959289650607785_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,465) = (/ 0.890999894975036_dbl_kind, 0.894109337250356_dbl_kind, 0.896929490134949_dbl_kind, 0.910992956870402_dbl_kind, 0.959324497337963_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,466) = (/ 0.891001037185038_dbl_kind, 0.894111472299489_dbl_kind, 0.896935172013262_dbl_kind, 0.911010481577422_dbl_kind, 0.959359238391253_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,467) = (/ 0.891002175389989_dbl_kind, 0.894113598454734_dbl_kind, 0.896940843195338_dbl_kind, 0.911027975283777_dbl_kind, 0.959393874205835_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,468) = (/ 0.89100330961048_dbl_kind, 0.894115715768525_dbl_kind, 0.896946503748035_dbl_kind, 0.911045438147686_dbl_kind, 0.959428405217489_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,469) = (/ 0.891004439866916_dbl_kind, 0.894117824293081_dbl_kind, 0.89695215373783_dbl_kind, 0.911062870326453_dbl_kind, 0.959462831859614_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,470) = (/ 0.891005566179522_dbl_kind, 0.894119924080408_dbl_kind, 0.896957793230826_dbl_kind, 0.911080271976474_dbl_kind, 0.959497154563246_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,471) = (/ 0.891006688568344_dbl_kind, 0.894122015182293_dbl_kind, 0.896963422292741_dbl_kind, 0.911097643253239_dbl_kind, 0.959531373757078_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,472) = (/ 0.891007807053251_dbl_kind, 0.89412409765031_dbl_kind, 0.896969040988919_dbl_kind, 0.91111498431134_dbl_kind, 0.959565489867462_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,473) = (/ 0.891008921653938_dbl_kind, 0.894126171535817_dbl_kind, 0.896974649384324_dbl_kind, 0.91113229530447_dbl_kind, 0.959599503318449_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,474) = (/ 0.891010032389924_dbl_kind, 0.894128236889959_dbl_kind, 0.896980247543539_dbl_kind, 0.911149576385435_dbl_kind, 0.95963341453178_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,475) = (/ 0.891011139280561_dbl_kind, 0.894130293763665_dbl_kind, 0.896985835530768_dbl_kind, 0.911166827706149_dbl_kind, 0.95966722392692_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,476) = (/ 0.891012242345028_dbl_kind, 0.894132342207651_dbl_kind, 0.896991413409838_dbl_kind, 0.911184049417649_dbl_kind, 0.959700931921068_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,477) = (/ 0.891013341602339_dbl_kind, 0.894134382272419_dbl_kind, 0.896996981244193_dbl_kind, 0.911201241670087_dbl_kind, 0.959734538929171_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,478) = (/ 0.891014437071342_dbl_kind, 0.894136414008258_dbl_kind, 0.897002539096901_dbl_kind, 0.911218404612748_dbl_kind, 0.95976804536394_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,479) = (/ 0.891015528770722_dbl_kind, 0.894138437465243_dbl_kind, 0.897008087030648_dbl_kind, 0.911235538394041_dbl_kind, 0.959801451635865_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,480) = (/ 0.891016616719002_dbl_kind, 0.894140452693236_dbl_kind, 0.897013625107743_dbl_kind, 0.911252643161514_dbl_kind, 0.959834758153237_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,481) = (/ 0.891017700934544_dbl_kind, 0.894142459741888_dbl_kind, 0.897019153390114_dbl_kind, 0.911269719061852_dbl_kind, 0.959867965322156_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,482) = (/ 0.891018781435554_dbl_kind, 0.894144458660634_dbl_kind, 0.897024671939311_dbl_kind, 0.911286766240884_dbl_kind, 0.959901073546544_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,483) = (/ 0.891019858240079_dbl_kind, 0.894146449498702_dbl_kind, 0.897030180816508_dbl_kind, 0.911303784843586_dbl_kind, 0.95993408322817_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,484) = (/ 0.891020931366015_dbl_kind, 0.894148432305104_dbl_kind, 0.897035680082496_dbl_kind, 0.911320775014087_dbl_kind, 0.959966994766654_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,485) = (/ 0.891022000831104_dbl_kind, 0.894150407128642_dbl_kind, 0.897041169797693_dbl_kind, 0.91133773689567_dbl_kind, 0.959999808559492_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,486) = (/ 0.891023066652933_dbl_kind, 0.894152374017907_dbl_kind, 0.897046650022137_dbl_kind, 0.911354670630781_dbl_kind, 0.960032525002059_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,487) = (/ 0.891024128848947_dbl_kind, 0.89415433302128_dbl_kind, 0.897052120815489_dbl_kind, 0.911371576361029_dbl_kind, 0.960065144487631_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,488) = (/ 0.891025187436437_dbl_kind, 0.894156284186929_dbl_kind, 0.897057582237035_dbl_kind, 0.911388454227191_dbl_kind, 0.960097667407401_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,489) = (/ 0.891026242432551_dbl_kind, 0.894158227562815_dbl_kind, 0.897063034345685_dbl_kind, 0.911405304369219_dbl_kind, 0.960130094150485_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,490) = (/ 0.891027293854293_dbl_kind, 0.894160163196686_dbl_kind, 0.89706847719997_dbl_kind, 0.911422126926242_dbl_kind, 0.960162425103946_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,491) = (/ 0.891028341718523_dbl_kind, 0.894162091136083_dbl_kind, 0.897073910858053_dbl_kind, 0.91143892203657_dbl_kind, 0.960194660652801_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,492) = (/ 0.891029386041961_dbl_kind, 0.894164011428334_dbl_kind, 0.897079335377716_dbl_kind, 0.911455689837699_dbl_kind, 0.960226801180034_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,493) = (/ 0.891030426841186_dbl_kind, 0.894165924120563_dbl_kind, 0.897084750816372_dbl_kind, 0.911472430466314_dbl_kind, 0.960258847066616_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,494) = (/ 0.891031464132642_dbl_kind, 0.894167829259681_dbl_kind, 0.89709015723106_dbl_kind, 0.911489144058296_dbl_kind, 0.960290798691514_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,495) = (/ 0.891032497932633_dbl_kind, 0.894169726892394_dbl_kind, 0.897095554678445_dbl_kind, 0.911505830748723_dbl_kind, 0.960322656431706_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,496) = (/ 0.891033528257331_dbl_kind, 0.894171617065198_dbl_kind, 0.897100943214824_dbl_kind, 0.911522490671876_dbl_kind, 0.960354420662189_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,497) = (/ 0.891034555122776_dbl_kind, 0.894173499824381_dbl_kind, 0.897106322896122_dbl_kind, 0.911539123961243_dbl_kind, 0.960386091756007_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,498) = (/ 0.891035578544871_dbl_kind, 0.894175375216025_dbl_kind, 0.897111693777893_dbl_kind, 0.911555730749521_dbl_kind, 0.960417670084243_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,499) = (/ 0.891036598539395_dbl_kind, 0.894177243286005_dbl_kind, 0.897117055915326_dbl_kind, 0.911572311168624_dbl_kind, 0.960449156016051_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,500) = (/ 0.891037615121995_dbl_kind, 0.89417910407999_dbl_kind, 0.897122409363237_dbl_kind, 0.911588865349685_dbl_kind, 0.960480549918658_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,501) = (/ 0.891038628308191_dbl_kind, 0.894180957643439_dbl_kind, 0.89712775417608_dbl_kind, 0.911605393423059_dbl_kind, 0.960511852157379_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,502) = (/ 0.891039638113379_dbl_kind, 0.89418280402161_dbl_kind, 0.89713309040794_dbl_kind, 0.911621895518329_dbl_kind, 0.960543063095633_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,503) = (/ 0.891040644552828_dbl_kind, 0.894184643259553_dbl_kind, 0.897138418112539_dbl_kind, 0.911638371764309_dbl_kind, 0.960574183094953_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,504) = (/ 0.891041647641688_dbl_kind, 0.894186475402112_dbl_kind, 0.897143737343232_dbl_kind, 0.911654822289049_dbl_kind, 0.960605212514998_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,505) = (/ 0.891042647394983_dbl_kind, 0.894188300493928_dbl_kind, 0.897149048153016_dbl_kind, 0.911671247219839_dbl_kind, 0.960636151713565_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,506) = (/ 0.891043643827621_dbl_kind, 0.894190118579436_dbl_kind, 0.897154350594521_dbl_kind, 0.911687646683211_dbl_kind, 0.960667001046601_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,507) = (/ 0.891044636954388_dbl_kind, 0.894191929702867_dbl_kind, 0.897159644720021_dbl_kind, 0.911704020804948_dbl_kind, 0.960697760868222_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,508) = (/ 0.891045626789957_dbl_kind, 0.894193733908251_dbl_kind, 0.897164930581427_dbl_kind, 0.91172036971008_dbl_kind, 0.960728431530714_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,509) = (/ 0.891046613348882_dbl_kind, 0.894195531239409_dbl_kind, 0.897170208230292_dbl_kind, 0.911736693522897_dbl_kind, 0.960759013384554_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,510) = (/ 0.891047596645602_dbl_kind, 0.894197321739965_dbl_kind, 0.897175477717815_dbl_kind, 0.911752992366947_dbl_kind, 0.960789506778415_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,511) = (/ 0.891048576694445_dbl_kind, 0.894199105453335_dbl_kind, 0.897180739094835_dbl_kind, 0.911769266365042_dbl_kind, 0.960819912059184_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,512) = (/ 0.891049553509626_dbl_kind, 0.894200882422737_dbl_kind, 0.897185992411838_dbl_kind, 0.911785515639263_dbl_kind, 0.96085022957197_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,513) = (/ 0.89105052710525_dbl_kind, 0.894202652691185_dbl_kind, 0.897191237718956_dbl_kind, 0.911801740310961_dbl_kind, 0.960880459660115_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,514) = (/ 0.891051497495313_dbl_kind, 0.894204416301491_dbl_kind, 0.89719647506597_dbl_kind, 0.911817940500765_dbl_kind, 0.96091060266521_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,515) = (/ 0.8910524646937_dbl_kind, 0.894206173296267_dbl_kind, 0.89720170450231_dbl_kind, 0.911834116328581_dbl_kind, 0.960940658927101_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,516) = (/ 0.891053428714195_dbl_kind, 0.894207923717922_dbl_kind, 0.897206926077053_dbl_kind, 0.911850267913603_dbl_kind, 0.960970628783902_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,517) = (/ 0.89105438957047_dbl_kind, 0.894209667608668_dbl_kind, 0.897212139838933_dbl_kind, 0.911866395374309_dbl_kind, 0.961000512572007_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,518) = (/ 0.8910553472761_dbl_kind, 0.894211405010515_dbl_kind, 0.897217345836334_dbl_kind, 0.911882498828471_dbl_kind, 0.961030310626103_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,519) = (/ 0.891056301844549_dbl_kind, 0.894213135965272_dbl_kind, 0.897222544117294_dbl_kind, 0.911898578393154_dbl_kind, 0.961060023279177_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,520) = (/ 0.891057253289186_dbl_kind, 0.894214860514551_dbl_kind, 0.897227734729508_dbl_kind, 0.911914634184727_dbl_kind, 0.961089650862527_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,521) = (/ 0.891058201623277_dbl_kind, 0.894216578699763_dbl_kind, 0.897232917720329_dbl_kind, 0.911930666318859_dbl_kind, 0.96111919370578_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,522) = (/ 0.891059146859987_dbl_kind, 0.894218290562124_dbl_kind, 0.897238093136768_dbl_kind, 0.911946674910526_dbl_kind, 0.961148652136893_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,523) = (/ 0.891060089012386_dbl_kind, 0.894219996142647_dbl_kind, 0.897243261025495_dbl_kind, 0.911962660074019_dbl_kind, 0.961178026482167_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,524) = (/ 0.891061028093444_dbl_kind, 0.894221695482151_dbl_kind, 0.897248421432844_dbl_kind, 0.91197862192294_dbl_kind, 0.961207317066265_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,525) = (/ 0.891061964116037_dbl_kind, 0.894223388621256_dbl_kind, 0.897253574404811_dbl_kind, 0.911994560570214_dbl_kind, 0.96123652421221_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,526) = (/ 0.891062897092945_dbl_kind, 0.894225075600386_dbl_kind, 0.897258719987057_dbl_kind, 0.912010476128084_dbl_kind, 0.961265648241405_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,527) = (/ 0.891063827036856_dbl_kind, 0.894226756459767_dbl_kind, 0.897263858224908_dbl_kind, 0.912026368708125_dbl_kind, 0.961294689473636_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,528) = (/ 0.891064753960366_dbl_kind, 0.89422843123943_dbl_kind, 0.897268989163359_dbl_kind, 0.91204223842124_dbl_kind, 0.961323648227092_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,529) = (/ 0.891065677875976_dbl_kind, 0.894230099979209_dbl_kind, 0.897274112847075_dbl_kind, 0.912058085377666_dbl_kind, 0.961352524818363_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,530) = (/ 0.8910665987961_dbl_kind, 0.894231762718745_dbl_kind, 0.89727922932039_dbl_kind, 0.912073909686978_dbl_kind, 0.961381319562461_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,531) = (/ 0.891067516733063_dbl_kind, 0.894233419497482_dbl_kind, 0.897284338627311_dbl_kind, 0.912089711458096_dbl_kind, 0.961410032772826_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,532) = (/ 0.8910684316991_dbl_kind, 0.894235070354669_dbl_kind, 0.897289440811522_dbl_kind, 0.912105490799281_dbl_kind, 0.961438664761331_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,533) = (/ 0.891069343706359_dbl_kind, 0.894236715329362_dbl_kind, 0.897294535916377_dbl_kind, 0.912121247818149_dbl_kind, 0.961467215838297_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,534) = (/ 0.891070252766904_dbl_kind, 0.894238354460424_dbl_kind, 0.897299623984913_dbl_kind, 0.912136982621664_dbl_kind, 0.961495686312506_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,535) = (/ 0.891071158892713_dbl_kind, 0.894239987786524_dbl_kind, 0.897304705059842_dbl_kind, 0.912152695316149_dbl_kind, 0.961524076491203_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,536) = (/ 0.89107206209568_dbl_kind, 0.894241615346135_dbl_kind, 0.897309779183559_dbl_kind, 0.912168386007292_dbl_kind, 0.96155238668011_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,537) = (/ 0.891072962387614_dbl_kind, 0.894243237177543_dbl_kind, 0.897314846398139_dbl_kind, 0.912184054800139_dbl_kind, 0.961580617183435_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,538) = (/ 0.891073859780247_dbl_kind, 0.894244853318837_dbl_kind, 0.897319906745343_dbl_kind, 0.912199701799107_dbl_kind, 0.961608768303883_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,539) = (/ 0.891074754285226_dbl_kind, 0.894246463807918_dbl_kind, 0.897324960266616_dbl_kind, 0.912215327107985_dbl_kind, 0.96163684034266_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,540) = (/ 0.891075645914119_dbl_kind, 0.894248068682493_dbl_kind, 0.897330007003091_dbl_kind, 0.912230930829938_dbl_kind, 0.961664833599489_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,541) = (/ 0.891076534678416_dbl_kind, 0.894249667980078_dbl_kind, 0.89733504699559_dbl_kind, 0.912246513067509_dbl_kind, 0.961692748372615_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,542) = (/ 0.891077420589527_dbl_kind, 0.894251261738002_dbl_kind, 0.897340080284626_dbl_kind, 0.912262073922626_dbl_kind, 0.961720584958818_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,543) = (/ 0.891078303658788_dbl_kind, 0.894252849993399_dbl_kind, 0.897345106910403_dbl_kind, 0.912277613496601_dbl_kind, 0.96174834365341_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,544) = (/ 0.891079183897456_dbl_kind, 0.894254432783218_dbl_kind, 0.897350126912819_dbl_kind, 0.912293131890137_dbl_kind, 0.961776024750266_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,545) = (/ 0.891080061316715_dbl_kind, 0.894256010144216_dbl_kind, 0.897355140331471_dbl_kind, 0.912308629203331_dbl_kind, 0.961803628541817_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,546) = (/ 0.891080935927673_dbl_kind, 0.894257582112962_dbl_kind, 0.89736014720565_dbl_kind, 0.912324105535678_dbl_kind, 0.961831155319055_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,547) = (/ 0.891081807741365_dbl_kind, 0.894259148725836_dbl_kind, 0.897365147574347_dbl_kind, 0.912339560986074_dbl_kind, 0.961858605371558_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,548) = (/ 0.891082676768754_dbl_kind, 0.89426071001903_dbl_kind, 0.897370141476254_dbl_kind, 0.912354995652817_dbl_kind, 0.961885978987486_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,549) = (/ 0.891083543020733_dbl_kind, 0.894262266028551_dbl_kind, 0.897375128949768_dbl_kind, 0.912370409633615_dbl_kind, 0.961913276453591_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,550) = (/ 0.89108440650812_dbl_kind, 0.894263816790215_dbl_kind, 0.897380110032987_dbl_kind, 0.91238580302559_dbl_kind, 0.961940498055237_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,551) = (/ 0.891085267241668_dbl_kind, 0.894265362339655_dbl_kind, 0.897385084763718_dbl_kind, 0.912401175925273_dbl_kind, 0.96196764407639_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,552) = (/ 0.891086125232058_dbl_kind, 0.894266902712314_dbl_kind, 0.897390053179474_dbl_kind, 0.912416528428621_dbl_kind, 0.961994714799642_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,553) = (/ 0.891086980489903_dbl_kind, 0.894268437943454_dbl_kind, 0.897395015317479_dbl_kind, 0.912431860631006_dbl_kind, 0.962021710506212_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,554) = (/ 0.891087833025749_dbl_kind, 0.894269968068146_dbl_kind, 0.897399971214669_dbl_kind, 0.912447172627231_dbl_kind, 0.962048631475957_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,555) = (/ 0.891088682850078_dbl_kind, 0.894271493121281_dbl_kind, 0.897404920907692_dbl_kind, 0.912462464511526_dbl_kind, 0.962075477987378_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,556) = (/ 0.891089529973302_dbl_kind, 0.894273013137562_dbl_kind, 0.897409864432912_dbl_kind, 0.912477736377553_dbl_kind, 0.96210225031763_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,557) = (/ 0.891090374405772_dbl_kind, 0.89427452815151_dbl_kind, 0.89741480182641_dbl_kind, 0.912492988318411_dbl_kind, 0.962128948742529_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,558) = (/ 0.891091216157772_dbl_kind, 0.894276038197462_dbl_kind, 0.897419733123985_dbl_kind, 0.912508220426638_dbl_kind, 0.962155573536564_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,559) = (/ 0.891092055239525_dbl_kind, 0.894277543309572_dbl_kind, 0.897424658361156_dbl_kind, 0.912523432794214_dbl_kind, 0.962182124972899_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,560) = (/ 0.89109289166119_dbl_kind, 0.89427904352181_dbl_kind, 0.897429577573167_dbl_kind, 0.912538625512568_dbl_kind, 0.962208603323385_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,561) = (/ 0.891093725432865_dbl_kind, 0.894280538867965_dbl_kind, 0.897434490794982_dbl_kind, 0.912553798672575_dbl_kind, 0.962235008858566_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,562) = (/ 0.891094556564587_dbl_kind, 0.894282029381643_dbl_kind, 0.897439398061294_dbl_kind, 0.912568952364564_dbl_kind, 0.962261341847687_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,563) = (/ 0.891095385066332_dbl_kind, 0.89428351509627_dbl_kind, 0.897444299406523_dbl_kind, 0.912584086678324_dbl_kind, 0.962287602558706_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,564) = (/ 0.891096210948018_dbl_kind, 0.89428499604509_dbl_kind, 0.897449194864818_dbl_kind, 0.912599201703097_dbl_kind, 0.962313791258297_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,565) = (/ 0.891097034219502_dbl_kind, 0.894286472261167_dbl_kind, 0.897454084470059_dbl_kind, 0.912614297527595_dbl_kind, 0.962339908211855_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,566) = (/ 0.891097854890587_dbl_kind, 0.894287943777384_dbl_kind, 0.897458968255859_dbl_kind, 0.91262937423999_dbl_kind, 0.962365953683513_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,567) = (/ 0.891098672971012_dbl_kind, 0.894289410626447_dbl_kind, 0.897463846255566_dbl_kind, 0.912644431927927_dbl_kind, 0.96239192793614_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,568) = (/ 0.891099488470467_dbl_kind, 0.894290872840879_dbl_kind, 0.897468718502265_dbl_kind, 0.912659470678524_dbl_kind, 0.962417831231356_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,569) = (/ 0.89110030139858_dbl_kind, 0.894292330453027_dbl_kind, 0.897473585028779_dbl_kind, 0.912674490578374_dbl_kind, 0.962443663829536_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,570) = (/ 0.891101111764928_dbl_kind, 0.89429378349506_dbl_kind, 0.897478445867671_dbl_kind, 0.912689491713549_dbl_kind, 0.962469425989809_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,571) = (/ 0.891101919579031_dbl_kind, 0.894295231998966_dbl_kind, 0.897483301051245_dbl_kind, 0.912704474169604_dbl_kind, 0.962495117970089_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,572) = (/ 0.891102724850355_dbl_kind, 0.894296675996561_dbl_kind, 0.897488150611552_dbl_kind, 0.912719438031582_dbl_kind, 0.962520740027057_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,573) = (/ 0.891103527588314_dbl_kind, 0.894298115519479_dbl_kind, 0.897492994580386_dbl_kind, 0.91273438338401_dbl_kind, 0.96254629241618_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,574) = (/ 0.89110432780227_dbl_kind, 0.894299550599181_dbl_kind, 0.897497832989287_dbl_kind, 0.912749310310913_dbl_kind, 0.962571775391717_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,575) = (/ 0.89110512550153_dbl_kind, 0.894300981266949_dbl_kind, 0.897502665869547_dbl_kind, 0.912764218895806_dbl_kind, 0.962597189206727_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,576) = (/ 0.891105920695354_dbl_kind, 0.894302407553893_dbl_kind, 0.897507493252208_dbl_kind, 0.912779109221705_dbl_kind, 0.962622534113075_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,577) = (/ 0.891106713392946_dbl_kind, 0.894303829490946_dbl_kind, 0.897512315168063_dbl_kind, 0.912793981371129_dbl_kind, 0.962647810361437_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,578) = (/ 0.891107503603466_dbl_kind, 0.894305247108866_dbl_kind, 0.897517131647662_dbl_kind, 0.9128088354261_dbl_kind, 0.962673018201309_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,579) = (/ 0.891108291336019_dbl_kind, 0.894306660438239_dbl_kind, 0.897521942721309_dbl_kind, 0.912823671468146_dbl_kind, 0.962698157881019_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,580) = (/ 0.891109076599664_dbl_kind, 0.894308069509475_dbl_kind, 0.897526748419067_dbl_kind, 0.91283848957831_dbl_kind, 0.962723229647719_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,581) = (/ 0.891109859403412_dbl_kind, 0.894309474352813_dbl_kind, 0.897531548770759_dbl_kind, 0.912853289837145_dbl_kind, 0.96274823374741_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,582) = (/ 0.891110639756226_dbl_kind, 0.894310874998318_dbl_kind, 0.897536343805968_dbl_kind, 0.912868072324724_dbl_kind, 0.96277317042494_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,583) = (/ 0.891111417667019_dbl_kind, 0.894312271475883_dbl_kind, 0.897541133554041_dbl_kind, 0.91288283712064_dbl_kind, 0.962798039924004_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,584) = (/ 0.891112193144663_dbl_kind, 0.89431366381523_dbl_kind, 0.897545918044091_dbl_kind, 0.912897584304006_dbl_kind, 0.962822842487164_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,585) = (/ 0.891112966197978_dbl_kind, 0.89431505204591_dbl_kind, 0.897550697304996_dbl_kind, 0.912912313953465_dbl_kind, 0.962847578355847_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,586) = (/ 0.891113736835742_dbl_kind, 0.894316436197302_dbl_kind, 0.897555471365404_dbl_kind, 0.912927026147185_dbl_kind, 0.962872247770356_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,587) = (/ 0.891114505066689_dbl_kind, 0.894317816298615_dbl_kind, 0.897560240253731_dbl_kind, 0.91294172096287_dbl_kind, 0.962896850969874_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,588) = (/ 0.891115270899507_dbl_kind, 0.894319192378891_dbl_kind, 0.897565003998167_dbl_kind, 0.912956398477758_dbl_kind, 0.962921388192469_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,589) = (/ 0.891116034342838_dbl_kind, 0.894320564466998_dbl_kind, 0.897569762626675_dbl_kind, 0.912971058768623_dbl_kind, 0.962945859675101_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,590) = (/ 0.891116795405285_dbl_kind, 0.89432193259164_dbl_kind, 0.897574516166993_dbl_kind, 0.912985701911782_dbl_kind, 0.962970265653639_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,591) = (/ 0.891117554095406_dbl_kind, 0.894323296781348_dbl_kind, 0.897579264646636_dbl_kind, 0.913000327983094_dbl_kind, 0.962994606362848_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,592) = (/ 0.891118310421716_dbl_kind, 0.894324657064489_dbl_kind, 0.897584008092898_dbl_kind, 0.913014937057967_dbl_kind, 0.963018882036411_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,593) = (/ 0.89111906439269_dbl_kind, 0.894326013469263_dbl_kind, 0.897588746532853_dbl_kind, 0.913029529211356_dbl_kind, 0.963043092906925_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,594) = (/ 0.891119816016759_dbl_kind, 0.894327366023699_dbl_kind, 0.897593479993357_dbl_kind, 0.913044104517772_dbl_kind, 0.963067239205919_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,595) = (/ 0.891120565302317_dbl_kind, 0.894328714755665_dbl_kind, 0.897598208501052_dbl_kind, 0.913058663051277_dbl_kind, 0.963091321163851_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,596) = (/ 0.891121312257714_dbl_kind, 0.894330059692858_dbl_kind, 0.897602932082362_dbl_kind, 0.913073204885495_dbl_kind, 0.96311533901011_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,597) = (/ 0.891122056891263_dbl_kind, 0.894331400862814_dbl_kind, 0.8976076507635_dbl_kind, 0.913087730093608_dbl_kind, 0.963139292973037_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,598) = (/ 0.891122799211234_dbl_kind, 0.894332738292903_dbl_kind, 0.897612364570469_dbl_kind, 0.913102238748365_dbl_kind, 0.96316318327992_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,599) = (/ 0.891123539225863_dbl_kind, 0.894334072010328_dbl_kind, 0.897617073529061_dbl_kind, 0.913116730922077_dbl_kind, 0.963187010156996_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,600) = (/ 0.891124276943343_dbl_kind, 0.894335402042134_dbl_kind, 0.897621777664859_dbl_kind, 0.913131206686628_dbl_kind, 0.963210773829475_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,601) = (/ 0.891125012371831_dbl_kind, 0.894336728415196_dbl_kind, 0.897626477003242_dbl_kind, 0.913145666113473_dbl_kind, 0.963234474521525_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,602) = (/ 0.891125745519447_dbl_kind, 0.894338051156232_dbl_kind, 0.897631171569384_dbl_kind, 0.913160109273643_dbl_kind, 0.963258112456292_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,603) = (/ 0.891126476394271_dbl_kind, 0.894339370291793_dbl_kind, 0.897635861388255_dbl_kind, 0.913174536237743_dbl_kind, 0.963281687855901_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,604) = (/ 0.891127205004348_dbl_kind, 0.894340685848273_dbl_kind, 0.897640546484625_dbl_kind, 0.913188947075963_dbl_kind, 0.963305200941456_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,605) = (/ 0.891127931357689_dbl_kind, 0.8943419978519_dbl_kind, 0.897645226883063_dbl_kind, 0.913203341858071_dbl_kind, 0.963328651933062_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,606) = (/ 0.891128655462265_dbl_kind, 0.894343306328746_dbl_kind, 0.89764990260794_dbl_kind, 0.913217720653426_dbl_kind, 0.96335204104981_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,607) = (/ 0.891129377326014_dbl_kind, 0.894344611304719_dbl_kind, 0.897654573683432_dbl_kind, 0.91323208353097_dbl_kind, 0.963375368509803_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,608) = (/ 0.891130096956837_dbl_kind, 0.894345912805568_dbl_kind, 0.897659240133518_dbl_kind, 0.913246430559239_dbl_kind, 0.963398634530144_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,609) = (/ 0.891130814362601_dbl_kind, 0.894347210856886_dbl_kind, 0.897663901981986_dbl_kind, 0.913260761806361_dbl_kind, 0.963421839326951_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,610) = (/ 0.89113152955114_dbl_kind, 0.894348505484104_dbl_kind, 0.897668559252431_dbl_kind, 0.913275077340063_dbl_kind, 0.963444983115361_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,611) = (/ 0.891132242530251_dbl_kind, 0.894349796712497_dbl_kind, 0.897673211968256_dbl_kind, 0.913289377227668_dbl_kind, 0.963468066109541_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,612) = (/ 0.891132953307701_dbl_kind, 0.894351084567179_dbl_kind, 0.897677860152679_dbl_kind, 0.9133036615361_dbl_kind, 0.963491088522678_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,613) = (/ 0.891133661891221_dbl_kind, 0.894352369073112_dbl_kind, 0.897682503828728_dbl_kind, 0.913317930331889_dbl_kind, 0.963514050567002_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,614) = (/ 0.89113436828851_dbl_kind, 0.894353650255098_dbl_kind, 0.897687143019247_dbl_kind, 0.91333218368117_dbl_kind, 0.963536952453781_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,615) = (/ 0.891135072507235_dbl_kind, 0.894354928137784_dbl_kind, 0.897691777746896_dbl_kind, 0.913346421649687_dbl_kind, 0.96355979439333_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,616) = (/ 0.891135774555029_dbl_kind, 0.894356202745662_dbl_kind, 0.897696408034152_dbl_kind, 0.913360644302797_dbl_kind, 0.963582576595011_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,617) = (/ 0.891136474439494_dbl_kind, 0.894357474103068_dbl_kind, 0.89770103390331_dbl_kind, 0.913374851705469_dbl_kind, 0.963605299267251_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,618) = (/ 0.891137172168203_dbl_kind, 0.894358742234185_dbl_kind, 0.897705655376487_dbl_kind, 0.913389043922289_dbl_kind, 0.963627962617533_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,619) = (/ 0.891137867748694_dbl_kind, 0.894360007163041_dbl_kind, 0.897710272475624_dbl_kind, 0.913403221017463_dbl_kind, 0.96365056685241_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,620) = (/ 0.891138561188475_dbl_kind, 0.894361268913511_dbl_kind, 0.89771488522248_dbl_kind, 0.913417383054818_dbl_kind, 0.963673112177504_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,621) = (/ 0.891139252495025_dbl_kind, 0.894362527509318_dbl_kind, 0.897719493638645_dbl_kind, 0.913431530097805_dbl_kind, 0.963695598797518_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,622) = (/ 0.891139941675792_dbl_kind, 0.894363782974032_dbl_kind, 0.897724097745531_dbl_kind, 0.913445662209499_dbl_kind, 0.963718026916236_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,623) = (/ 0.891140628738194_dbl_kind, 0.894365035331072_dbl_kind, 0.89772869756438_dbl_kind, 0.913459779452607_dbl_kind, 0.963740396736531_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,624) = (/ 0.891141313689619_dbl_kind, 0.894366284603705_dbl_kind, 0.897733293116264_dbl_kind, 0.913473881889466_dbl_kind, 0.963762708460369_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,625) = (/ 0.891141996537426_dbl_kind, 0.894367530815048_dbl_kind, 0.897737884422083_dbl_kind, 0.913487969582046_dbl_kind, 0.963784962288811_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,626) = (/ 0.891142677288945_dbl_kind, 0.894368773988068_dbl_kind, 0.897742471502574_dbl_kind, 0.913502042591953_dbl_kind, 0.963807158422025_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,627) = (/ 0.891143355951478_dbl_kind, 0.894370014145582_dbl_kind, 0.897747054378304_dbl_kind, 0.91351610098043_dbl_kind, 0.963829297059285_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,628) = (/ 0.891144032532295_dbl_kind, 0.894371251310261_dbl_kind, 0.897751633069676_dbl_kind, 0.913530144808364_dbl_kind, 0.963851378398975_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,629) = (/ 0.891144707038644_dbl_kind, 0.894372485504621_dbl_kind, 0.897756207596929_dbl_kind, 0.913544174136281_dbl_kind, 0.963873402638601_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,630) = (/ 0.89114537947774_dbl_kind, 0.894373716751038_dbl_kind, 0.897760777980142_dbl_kind, 0.913558189024355_dbl_kind, 0.96389536997479_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,631) = (/ 0.891146049856773_dbl_kind, 0.894374945071735_dbl_kind, 0.89776534423923_dbl_kind, 0.913572189532406_dbl_kind, 0.963917280603295_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,632) = (/ 0.891146718182903_dbl_kind, 0.894376170488792_dbl_kind, 0.897769906393952_dbl_kind, 0.913586175719903_dbl_kind, 0.963939134719003_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,633) = (/ 0.891147384463266_dbl_kind, 0.894377393024139_dbl_kind, 0.897774464463908_dbl_kind, 0.913600147645969_dbl_kind, 0.963960932515935_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,634) = (/ 0.891148048704968_dbl_kind, 0.894378612699565_dbl_kind, 0.897779018468541_dbl_kind, 0.91361410536938_dbl_kind, 0.963982674187258_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,635) = (/ 0.891148710915092_dbl_kind, 0.89437982953671_dbl_kind, 0.897783568427139_dbl_kind, 0.91362804894857_dbl_kind, 0.964004359925281_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,636) = (/ 0.891149371100692_dbl_kind, 0.894381043557072_dbl_kind, 0.897788114358836_dbl_kind, 0.913641978441629_dbl_kind, 0.964025989921464_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,637) = (/ 0.891150029268797_dbl_kind, 0.894382254782005_dbl_kind, 0.897792656282613_dbl_kind, 0.913655893906309_dbl_kind, 0.964047564366429_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,638) = (/ 0.891150685426409_dbl_kind, 0.894383463232717_dbl_kind, 0.897797194217301_dbl_kind, 0.913669795400027_dbl_kind, 0.964069083449949_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,639) = (/ 0.891151339580505_dbl_kind, 0.894384668930277_dbl_kind, 0.897801728181581_dbl_kind, 0.913683682979864_dbl_kind, 0.964090547360968_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,640) = (/ 0.891151991738037_dbl_kind, 0.89438587189561_dbl_kind, 0.897806258193984_dbl_kind, 0.913697556702567_dbl_kind, 0.964111956287597_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,641) = (/ 0.891152641905932_dbl_kind, 0.894387072149498_dbl_kind, 0.897810784272896_dbl_kind, 0.913711416624554_dbl_kind, 0.96413331041712_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,642) = (/ 0.89115329009109_dbl_kind, 0.894388269712584_dbl_kind, 0.897815306436554_dbl_kind, 0.913725262801916_dbl_kind, 0.964154609935999_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,643) = (/ 0.89115393630039_dbl_kind, 0.89438946460537_dbl_kind, 0.897819824703053_dbl_kind, 0.913739095290414_dbl_kind, 0.964175855029884_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,644) = (/ 0.891154580540682_dbl_kind, 0.894390656848218_dbl_kind, 0.897824339090343_dbl_kind, 0.91375291414549_dbl_kind, 0.964197045883605_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,645) = (/ 0.891155222818795_dbl_kind, 0.89439184646135_dbl_kind, 0.897828849616233_dbl_kind, 0.91376671942226_dbl_kind, 0.964218182681189_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,646) = (/ 0.891155863141534_dbl_kind, 0.894393033464851_dbl_kind, 0.89783335629839_dbl_kind, 0.913780511175521_dbl_kind, 0.964239265605854_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,647) = (/ 0.891156501515675_dbl_kind, 0.894394217878665_dbl_kind, 0.897837859154342_dbl_kind, 0.913794289459754_dbl_kind, 0.964260294840022_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,648) = (/ 0.891157137947978_dbl_kind, 0.8943953997226_dbl_kind, 0.897842358201477_dbl_kind, 0.913808054329123_dbl_kind, 0.964281270565317_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,649) = (/ 0.891157772445174_dbl_kind, 0.894396579016329_dbl_kind, 0.897846853457048_dbl_kind, 0.913821805837477_dbl_kind, 0.964302192962576_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,650) = (/ 0.891158405013972_dbl_kind, 0.894397755779383_dbl_kind, 0.897851344938171_dbl_kind, 0.913835544038356_dbl_kind, 0.964323062211843_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,651) = (/ 0.891159035661059_dbl_kind, 0.894398930031164_dbl_kind, 0.897855832661826_dbl_kind, 0.913849268984988_dbl_kind, 0.964343878492382_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,652) = (/ 0.891159664393097_dbl_kind, 0.894400101790933_dbl_kind, 0.897860316644861_dbl_kind, 0.913862980730294_dbl_kind, 0.964364641982681_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,653) = (/ 0.891160291216726_dbl_kind, 0.89440127107782_dbl_kind, 0.897864796903992_dbl_kind, 0.913876679326891_dbl_kind, 0.964385352860451_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,654) = (/ 0.891160916138564_dbl_kind, 0.894402437910817_dbl_kind, 0.897869273455801_dbl_kind, 0.913890364827089_dbl_kind, 0.964406011302632_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,655) = (/ 0.891161539165208_dbl_kind, 0.894403602308786_dbl_kind, 0.897873746316743_dbl_kind, 0.913904037282898_dbl_kind, 0.964426617485402_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,656) = (/ 0.891162160303228_dbl_kind, 0.894404764290456_dbl_kind, 0.897878215503141_dbl_kind, 0.91391769674603_dbl_kind, 0.96444717158417_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,657) = (/ 0.891162779559177_dbl_kind, 0.894405923874421_dbl_kind, 0.897882681031192_dbl_kind, 0.913931343267895_dbl_kind, 0.964467673773595_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,658) = (/ 0.891163396939583_dbl_kind, 0.894407081079144_dbl_kind, 0.897887142916966_dbl_kind, 0.913944976899611_dbl_kind, 0.964488124227574_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,659) = (/ 0.891164012450953_dbl_kind, 0.894408235922959_dbl_kind, 0.897891601176409_dbl_kind, 0.913958597691999_dbl_kind, 0.964508523119263_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,660) = (/ 0.891164626099773_dbl_kind, 0.894409388424066_dbl_kind, 0.89789605582534_dbl_kind, 0.913972205695589_dbl_kind, 0.964528870621065_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,661) = (/ 0.891165237892506_dbl_kind, 0.894410538600537_dbl_kind, 0.897900506879455_dbl_kind, 0.913985800960621_dbl_kind, 0.964549166904642_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,662) = (/ 0.891165847835595_dbl_kind, 0.894411686470314_dbl_kind, 0.897904954354329_dbl_kind, 0.913999383537047_dbl_kind, 0.964569412140923_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,663) = (/ 0.891166455935462_dbl_kind, 0.894412832051211_dbl_kind, 0.897909398265415_dbl_kind, 0.91401295347453_dbl_kind, 0.964589606500098_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,664) = (/ 0.891167062198506_dbl_kind, 0.894413975360911_dbl_kind, 0.897913838628048_dbl_kind, 0.914026510822451_dbl_kind, 0.964609750151625_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,665) = (/ 0.891167666631108_dbl_kind, 0.894415116416973_dbl_kind, 0.89791827545744_dbl_kind, 0.914040055629906_dbl_kind, 0.964629843264242_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,666) = (/ 0.891168269239625_dbl_kind, 0.894416255236825_dbl_kind, 0.897922708768689_dbl_kind, 0.914053587945711_dbl_kind, 0.964649886005964_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,667) = (/ 0.891168870030397_dbl_kind, 0.894417391837773_dbl_kind, 0.897927138576773_dbl_kind, 0.914067107818404_dbl_kind, 0.964669878544077_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,668) = (/ 0.89116946900974_dbl_kind, 0.894418526236993_dbl_kind, 0.897931564896556_dbl_kind, 0.914080615296242_dbl_kind, 0.964689821045165_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,669) = (/ 0.891170066183953_dbl_kind, 0.894419658451537_dbl_kind, 0.897935987742788_dbl_kind, 0.914094110427209_dbl_kind, 0.964709713675095_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,670) = (/ 0.89117066155931_dbl_kind, 0.894420788498332_dbl_kind, 0.897940407130103_dbl_kind, 0.914107593259014_dbl_kind, 0.964729556599023_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,671) = (/ 0.89117125514207_dbl_kind, 0.894421916394182_dbl_kind, 0.897944823073024_dbl_kind, 0.914121063839095_dbl_kind, 0.964749349981411_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,672) = (/ 0.89117184693847_dbl_kind, 0.894423042155767_dbl_kind, 0.897949235585961_dbl_kind, 0.914134522214616_dbl_kind, 0.96476909398601_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,673) = (/ 0.891172436954726_dbl_kind, 0.894424165799642_dbl_kind, 0.897953644683215_dbl_kind, 0.914147968432478_dbl_kind, 0.964788788775883_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,674) = (/ 0.891173025197036_dbl_kind, 0.894425287342242_dbl_kind, 0.897958050378976_dbl_kind, 0.91416140253931_dbl_kind, 0.964808434513396_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,675) = (/ 0.891173611671576_dbl_kind, 0.89442640679988_dbl_kind, 0.897962452687325_dbl_kind, 0.914174824581479_dbl_kind, 0.964828031360228_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,676) = (/ 0.891174196384506_dbl_kind, 0.894427524188747_dbl_kind, 0.897966851622235_dbl_kind, 0.914188234605084_dbl_kind, 0.964847579477369_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,677) = (/ 0.891174779341964_dbl_kind, 0.894428639524912_dbl_kind, 0.897971247197574_dbl_kind, 0.914201632655968_dbl_kind, 0.964867079025134_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,678) = (/ 0.891175360550069_dbl_kind, 0.894429752824327_dbl_kind, 0.897975639427103_dbl_kind, 0.914215018779709_dbl_kind, 0.964886530163148_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,679) = (/ 0.891175940014921_dbl_kind, 0.894430864102822_dbl_kind, 0.897980028324478_dbl_kind, 0.914228393021628_dbl_kind, 0.964905933050377_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,680) = (/ 0.891176517742601_dbl_kind, 0.894431973376109_dbl_kind, 0.897984413903251_dbl_kind, 0.914241755426789_dbl_kind, 0.964925287845101_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,681) = (/ 0.891177093739171_dbl_kind, 0.894433080659784_dbl_kind, 0.897988796176871_dbl_kind, 0.914255106040001_dbl_kind, 0.964944594704939_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,682) = (/ 0.891177668010675_dbl_kind, 0.894434185969321_dbl_kind, 0.897993175158686_dbl_kind, 0.91426844490582_dbl_kind, 0.964963853786848_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,683) = (/ 0.891178240563136_dbl_kind, 0.89443528932008_dbl_kind, 0.897997550861941_dbl_kind, 0.914281772068547_dbl_kind, 0.964983065247118_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,684) = (/ 0.891178811402559_dbl_kind, 0.894436390727302_dbl_kind, 0.898001923299783_dbl_kind, 0.914295087572235_dbl_kind, 0.965002229241386_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,685) = (/ 0.891179380534934_dbl_kind, 0.894437490206117_dbl_kind, 0.898006292485257_dbl_kind, 0.914308391460689_dbl_kind, 0.965021345924632_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,686) = (/ 0.891179947966226_dbl_kind, 0.894438587771534_dbl_kind, 0.898010658431312_dbl_kind, 0.914321683777463_dbl_kind, 0.965040415451188_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,687) = (/ 0.891180513702387_dbl_kind, 0.89443968343845_dbl_kind, 0.898015021150797_dbl_kind, 0.914334964565869_dbl_kind, 0.965059437974737_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,688) = (/ 0.891181077749348_dbl_kind, 0.894440777221648_dbl_kind, 0.898019380656467_dbl_kind, 0.914348233868973_dbl_kind, 0.965078413648318_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,689) = (/ 0.891181640113023_dbl_kind, 0.894441869135797_dbl_kind, 0.898023736960978_dbl_kind, 0.914361491729599_dbl_kind, 0.965097342624332_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,690) = (/ 0.891182200799305_dbl_kind, 0.894442959195454_dbl_kind, 0.898028090076895_dbl_kind, 0.914374738190329_dbl_kind, 0.965116225054536_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,691) = (/ 0.891182759814074_dbl_kind, 0.894444047415062_dbl_kind, 0.898032440016686_dbl_kind, 0.914387973293507_dbl_kind, 0.965135061090058_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,692) = (/ 0.891183317163187_dbl_kind, 0.894445133808953_dbl_kind, 0.898036786792725_dbl_kind, 0.914401197081237_dbl_kind, 0.965153850881397_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,693) = (/ 0.891183872852486_dbl_kind, 0.894446218391348_dbl_kind, 0.898041130417296_dbl_kind, 0.914414409595387_dbl_kind, 0.96517259457842_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,694) = (/ 0.891184426887794_dbl_kind, 0.894447301176358_dbl_kind, 0.89804547090259_dbl_kind, 0.914427610877591_dbl_kind, 0.965191292330372_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,695) = (/ 0.891184979274916_dbl_kind, 0.894448382177983_dbl_kind, 0.898049808260709_dbl_kind, 0.91444080096925_dbl_kind, 0.965209944285876_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,696) = (/ 0.891185530019641_dbl_kind, 0.894449461410114_dbl_kind, 0.898054142503661_dbl_kind, 0.91445397991153_dbl_kind, 0.965228550592936_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,697) = (/ 0.891186079127738_dbl_kind, 0.894450538886533_dbl_kind, 0.89805847364337_dbl_kind, 0.914467147745369_dbl_kind, 0.965247111398944_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,698) = (/ 0.891186626604959_dbl_kind, 0.894451614620914_dbl_kind, 0.898062801691668_dbl_kind, 0.914480304511475_dbl_kind, 0.965265626850677_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,699) = (/ 0.89118717245704_dbl_kind, 0.894452688626824_dbl_kind, 0.8980671266603_dbl_kind, 0.914493450250327_dbl_kind, 0.965284097094305_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,700) = (/ 0.891187716689699_dbl_kind, 0.894453760917723_dbl_kind, 0.898071448560926_dbl_kind, 0.914506585002181_dbl_kind, 0.965302522275393_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,701) = (/ 0.891188259308635_dbl_kind, 0.894454831506961_dbl_kind, 0.898075767405118_dbl_kind, 0.914519708807064_dbl_kind, 0.965320902538902_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,702) = (/ 0.891188800319531_dbl_kind, 0.894455900407788_dbl_kind, 0.898080083204362_dbl_kind, 0.914532821704784_dbl_kind, 0.965339238029198_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,703) = (/ 0.891189339728052_dbl_kind, 0.894456967633345_dbl_kind, 0.898084395970064_dbl_kind, 0.914545923734922_dbl_kind, 0.965357528890044_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,704) = (/ 0.891189877539848_dbl_kind, 0.89445803319667_dbl_kind, 0.898088705713541_dbl_kind, 0.914559014936843_dbl_kind, 0.965375775264613_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,705) = (/ 0.89119041376055_dbl_kind, 0.894459097110693_dbl_kind, 0.898093012446031_dbl_kind, 0.914572095349689_dbl_kind, 0.96539397729549_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,706) = (/ 0.891190948395771_dbl_kind, 0.894460159388247_dbl_kind, 0.898097316178686_dbl_kind, 0.914585165012388_dbl_kind, 0.965412135124669_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,707) = (/ 0.891191481451109_dbl_kind, 0.894461220042056_dbl_kind, 0.898101616922581_dbl_kind, 0.914598223963647_dbl_kind, 0.965430248893564_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,708) = (/ 0.891192012932144_dbl_kind, 0.894462279084746_dbl_kind, 0.898105914688706_dbl_kind, 0.914611272241961_dbl_kind, 0.965448318743001_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,709) = (/ 0.891192542844439_dbl_kind, 0.89446333652884_dbl_kind, 0.898110209487974_dbl_kind, 0.91462430988561_dbl_kind, 0.965466344813234_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,710) = (/ 0.891193071193542_dbl_kind, 0.894464392386757_dbl_kind, 0.898114501331217_dbl_kind, 0.914637336932663_dbl_kind, 0.965484327243939_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,711) = (/ 0.891193597984981_dbl_kind, 0.894465446670819_dbl_kind, 0.898118790229188_dbl_kind, 0.914650353420976_dbl_kind, 0.965502266174219_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,712) = (/ 0.891194123224269_dbl_kind, 0.894466499393247_dbl_kind, 0.898123076192565_dbl_kind, 0.914663359388195_dbl_kind, 0.965520161742608_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,713) = (/ 0.891194646916904_dbl_kind, 0.894467550566162_dbl_kind, 0.898127359231945_dbl_kind, 0.91467635487176_dbl_kind, 0.965538014087074_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,714) = (/ 0.891195169068364_dbl_kind, 0.894468600201586_dbl_kind, 0.898131639357851_dbl_kind, 0.9146893399089_dbl_kind, 0.965555823345019_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,715) = (/ 0.891195689684113_dbl_kind, 0.894469648311444_dbl_kind, 0.898135916580729_dbl_kind, 0.914702314536642_dbl_kind, 0.965573589653285_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,716) = (/ 0.891196208769597_dbl_kind, 0.894470694907562_dbl_kind, 0.89814019091095_dbl_kind, 0.914715278791807_dbl_kind, 0.965591313148157_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,717) = (/ 0.891196726330246_dbl_kind, 0.894471740001669_dbl_kind, 0.898144462358811_dbl_kind, 0.91472823271101_dbl_kind, 0.965608993965362_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,718) = (/ 0.891197242371473_dbl_kind, 0.894472783605398_dbl_kind, 0.898148730934535_dbl_kind, 0.914741176330668_dbl_kind, 0.965626632240074_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,719) = (/ 0.891197756898677_dbl_kind, 0.894473825730285_dbl_kind, 0.898152996648271_dbl_kind, 0.914754109686994_dbl_kind, 0.965644228106921_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,720) = (/ 0.891198269917236_dbl_kind, 0.894474866387773_dbl_kind, 0.898157259510097_dbl_kind, 0.914767032816004_dbl_kind, 0.965661781699979_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,721) = (/ 0.891198781432518_dbl_kind, 0.894475905589207_dbl_kind, 0.898161519530017_dbl_kind, 0.914779945753513_dbl_kind, 0.965679293152783_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,722) = (/ 0.891199291449868_dbl_kind, 0.89447694334584_dbl_kind, 0.898165776717966_dbl_kind, 0.914792848535141_dbl_kind, 0.965696762598325_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,723) = (/ 0.891199799974619_dbl_kind, 0.89447797966883_dbl_kind, 0.898170031083806_dbl_kind, 0.914805741196312_dbl_kind, 0.965714190169058_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,724) = (/ 0.891200307012088_dbl_kind, 0.894479014569244_dbl_kind, 0.898174282637332_dbl_kind, 0.914818623772254_dbl_kind, 0.965731575996896_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,725) = (/ 0.891200812567572_dbl_kind, 0.894480048058052_dbl_kind, 0.898178531388266_dbl_kind, 0.914831496298001_dbl_kind, 0.965748920213225_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,726) = (/ 0.891201316646358_dbl_kind, 0.894481080146137_dbl_kind, 0.898182777346264_dbl_kind, 0.914844358808398_dbl_kind, 0.965766222948894_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,727) = (/ 0.89120181925371_dbl_kind, 0.894482110844288_dbl_kind, 0.898187020520913_dbl_kind, 0.914857211338096_dbl_kind, 0.965783484334229_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,728) = (/ 0.891202320394882_dbl_kind, 0.894483140163204_dbl_kind, 0.898191260921733_dbl_kind, 0.914870053921557_dbl_kind, 0.965800704499026_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,729) = (/ 0.891202820075108_dbl_kind, 0.894484168113492_dbl_kind, 0.898195498558175_dbl_kind, 0.914882886593054_dbl_kind, 0.965817883572559_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,730) = (/ 0.891203318299608_dbl_kind, 0.894485194705671_dbl_kind, 0.898199733439626_dbl_kind, 0.914895709386673_dbl_kind, 0.965835021683585_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,731) = (/ 0.891203815073586_dbl_kind, 0.894486219950171_dbl_kind, 0.898203965575406_dbl_kind, 0.914908522336313_dbl_kind, 0.965852118960334_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,732) = (/ 0.891204310402227_dbl_kind, 0.894487243857332_dbl_kind, 0.89820819497477_dbl_kind, 0.914921325475689_dbl_kind, 0.96586917553053_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,733) = (/ 0.891204804290706_dbl_kind, 0.894488266437406_dbl_kind, 0.898212421646908_dbl_kind, 0.914934118838329_dbl_kind, 0.96588619152138_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,734) = (/ 0.891205296744178_dbl_kind, 0.89448928770056_dbl_kind, 0.898216645600947_dbl_kind, 0.914946902457583_dbl_kind, 0.96590316705958_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,735) = (/ 0.891205787767782_dbl_kind, 0.89449030765687_dbl_kind, 0.898220866845949_dbl_kind, 0.914959676366613_dbl_kind, 0.965920102271319_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,736) = (/ 0.891206277366643_dbl_kind, 0.894491326316329_dbl_kind, 0.898225085390915_dbl_kind, 0.914972440598405_dbl_kind, 0.965936997282282_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,737) = (/ 0.89120676554587_dbl_kind, 0.894492343688843_dbl_kind, 0.898229301244781_dbl_kind, 0.914985195185763_dbl_kind, 0.965953852217649_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,738) = (/ 0.891207252310557_dbl_kind, 0.894493359784231_dbl_kind, 0.898233514416423_dbl_kind, 0.914997940161314_dbl_kind, 0.9659706672021_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,739) = (/ 0.891207737665779_dbl_kind, 0.89449437461223_dbl_kind, 0.898237724914656_dbl_kind, 0.915010675557507_dbl_kind, 0.96598744235982_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,740) = (/ 0.891208221616599_dbl_kind, 0.894495388182492_dbl_kind, 0.898241932748232_dbl_kind, 0.915023401406615_dbl_kind, 0.966004177814493_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,741) = (/ 0.891208704168063_dbl_kind, 0.894496400504585_dbl_kind, 0.898246137925846_dbl_kind, 0.915036117740734_dbl_kind, 0.966020873689313_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,742) = (/ 0.891209185325201_dbl_kind, 0.894497411587992_dbl_kind, 0.898250340456131_dbl_kind, 0.915048824591788_dbl_kind, 0.966037530106985_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,743) = (/ 0.891209665093029_dbl_kind, 0.894498421442116_dbl_kind, 0.89825454034766_dbl_kind, 0.915061521991526_dbl_kind, 0.966054147189722_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,744) = (/ 0.891210143476546_dbl_kind, 0.894499430076279_dbl_kind, 0.89825873760895_dbl_kind, 0.915074209971528_dbl_kind, 0.966070725059255_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,745) = (/ 0.891210620480735_dbl_kind, 0.894500437499717_dbl_kind, 0.898262932248458_dbl_kind, 0.9150868885632_dbl_kind, 0.966087263836831_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,746) = (/ 0.891211096110566_dbl_kind, 0.894501443721589_dbl_kind, 0.898267124274583_dbl_kind, 0.915099557797778_dbl_kind, 0.966103763643211_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,747) = (/ 0.891211570370992_dbl_kind, 0.894502448750971_dbl_kind, 0.89827131369567_dbl_kind, 0.915112217706333_dbl_kind, 0.966120224598681_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,748) = (/ 0.89121204326695_dbl_kind, 0.894503452596863_dbl_kind, 0.898275500520002_dbl_kind, 0.915124868319763_dbl_kind, 0.966136646823055_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,749) = (/ 0.891212514803362_dbl_kind, 0.89450445526818_dbl_kind, 0.89827968475581_dbl_kind, 0.915137509668802_dbl_kind, 0.966153030435663_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,750) = (/ 0.891212984985136_dbl_kind, 0.894505456773763_dbl_kind, 0.898283866411269_dbl_kind, 0.915150141784019_dbl_kind, 0.966169375555372_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,751) = (/ 0.891213453817163_dbl_kind, 0.894506457122373_dbl_kind, 0.898288045494496_dbl_kind, 0.915162764695815_dbl_kind, 0.966185682300571_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,752) = (/ 0.89121392130432_dbl_kind, 0.894507456322692_dbl_kind, 0.898292222013556_dbl_kind, 0.915175378434431_dbl_kind, 0.966201950789192_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,753) = (/ 0.891214387451468_dbl_kind, 0.894508454383325_dbl_kind, 0.89829639597646_dbl_kind, 0.915187983029941_dbl_kind, 0.966218181138693_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,754) = (/ 0.891214852263452_dbl_kind, 0.894509451312804_dbl_kind, 0.898300567391162_dbl_kind, 0.915200578512262_dbl_kind, 0.966234373466077_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,755) = (/ 0.891215315745103_dbl_kind, 0.89451044711958_dbl_kind, 0.898304736265566_dbl_kind, 0.915213164911147_dbl_kind, 0.966250527887875_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,756) = (/ 0.891215777901238_dbl_kind, 0.89451144181203_dbl_kind, 0.898308902607522_dbl_kind, 0.915225742256188_dbl_kind, 0.966266644520175_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,757) = (/ 0.891216238736655_dbl_kind, 0.894512435398456_dbl_kind, 0.898313066424827_dbl_kind, 0.915238310576822_dbl_kind, 0.966282723478596_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,758) = (/ 0.891216698256141_dbl_kind, 0.894513427887086_dbl_kind, 0.898317227725226_dbl_kind, 0.915250869902326_dbl_kind, 0.966298764878308_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,759) = (/ 0.891217156464465_dbl_kind, 0.894514419286074_dbl_kind, 0.898321386516415_dbl_kind, 0.915263420261819_dbl_kind, 0.966314768834034_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,760) = (/ 0.891217613366383_dbl_kind, 0.894515409603498_dbl_kind, 0.898325542806035_dbl_kind, 0.915275961684266_dbl_kind, 0.96633073546004_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,761) = (/ 0.891218068966634_dbl_kind, 0.894516398847365_dbl_kind, 0.898329696601678_dbl_kind, 0.915288494198475_dbl_kind, 0.96634666487015_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,762) = (/ 0.891218523269943_dbl_kind, 0.89451738702561_dbl_kind, 0.898333847910889_dbl_kind, 0.915301017833102_dbl_kind, 0.966362557177736_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,763) = (/ 0.891218976281021_dbl_kind, 0.894518374146093_dbl_kind, 0.898337996741157_dbl_kind, 0.915313532616648_dbl_kind, 0.966378412495734_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,764) = (/ 0.891219428004561_dbl_kind, 0.894519360216606_dbl_kind, 0.898342143099927_dbl_kind, 0.915326038577464_dbl_kind, 0.966394230936641_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,765) = (/ 0.891219878445244_dbl_kind, 0.894520345244866_dbl_kind, 0.898346286994593_dbl_kind, 0.915338535743746_dbl_kind, 0.966410012612504_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,766) = (/ 0.891220327607735_dbl_kind, 0.894521329238524_dbl_kind, 0.898350428432501_dbl_kind, 0.915351024143544_dbl_kind, 0.966425757634944_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,767) = (/ 0.891220775496683_dbl_kind, 0.894522312205157_dbl_kind, 0.898354567420947_dbl_kind, 0.915363503804755_dbl_kind, 0.966441466115144_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,768) = (/ 0.891221222116725_dbl_kind, 0.894523294152274_dbl_kind, 0.898358703967182_dbl_kind, 0.915375974755128_dbl_kind, 0.966457138163851_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,769) = (/ 0.891221667472481_dbl_kind, 0.894524275087316_dbl_kind, 0.898362838078409_dbl_kind, 0.915388437022267_dbl_kind, 0.966472773891388_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,770) = (/ 0.891222111568555_dbl_kind, 0.894525255017652_dbl_kind, 0.898366969761783_dbl_kind, 0.915400890633625_dbl_kind, 0.966488373407646_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,771) = (/ 0.891222554409538_dbl_kind, 0.894526233950587_dbl_kind, 0.898371099024413_dbl_kind, 0.915413335616513_dbl_kind, 0.966503936822088_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,772) = (/ 0.891222996000006_dbl_kind, 0.894527211893355_dbl_kind, 0.898375225873365_dbl_kind, 0.915425771998094_dbl_kind, 0.966519464243757_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,773) = (/ 0.891223436344521_dbl_kind, 0.894528188853125_dbl_kind, 0.898379350315653_dbl_kind, 0.915438199805389_dbl_kind, 0.966534955781269_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,774) = (/ 0.891223875447627_dbl_kind, 0.894529164837_dbl_kind, 0.898383472358252_dbl_kind, 0.915450619065275_dbl_kind, 0.966550411542827_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,775) = (/ 0.891224313313858_dbl_kind, 0.894530139852014_dbl_kind, 0.898387592008088_dbl_kind, 0.915463029804485_dbl_kind, 0.966565831636205_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,776) = (/ 0.891224749947729_dbl_kind, 0.894531113905136_dbl_kind, 0.898391709272045_dbl_kind, 0.915475432049612_dbl_kind, 0.96658121616877_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,777) = (/ 0.891225185353743_dbl_kind, 0.894532087003274_dbl_kind, 0.898395824156962_dbl_kind, 0.915487825827109_dbl_kind, 0.966596565247473_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,778) = (/ 0.891225619536388_dbl_kind, 0.894533059153264_dbl_kind, 0.898399936669633_dbl_kind, 0.915500211163289_dbl_kind, 0.966611878978846_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,779) = (/ 0.891226052500135_dbl_kind, 0.894534030361885_dbl_kind, 0.89840404681681_dbl_kind, 0.915512588084323_dbl_kind, 0.96662715746902_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,780) = (/ 0.891226484249443_dbl_kind, 0.894535000635846_dbl_kind, 0.898408154605202_dbl_kind, 0.915524956616246_dbl_kind, 0.966642400823712_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,781) = (/ 0.891226914788757_dbl_kind, 0.894535969981798_dbl_kind, 0.898412260041475_dbl_kind, 0.915537316784957_dbl_kind, 0.966657609148232_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,782) = (/ 0.891227344122503_dbl_kind, 0.894536938406326_dbl_kind, 0.898416363132253_dbl_kind, 0.915549668616217_dbl_kind, 0.96667278254749_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,783) = (/ 0.891227772255099_dbl_kind, 0.894537905915953_dbl_kind, 0.898420463884118_dbl_kind, 0.915562012135651_dbl_kind, 0.966687921125989_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,784) = (/ 0.891228199190942_dbl_kind, 0.894538872517141_dbl_kind, 0.89842456230361_dbl_kind, 0.915574347368748_dbl_kind, 0.966703024987833_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,785) = (/ 0.891228624934419_dbl_kind, 0.89453983821629_dbl_kind, 0.898428658397228_dbl_kind, 0.915586674340866_dbl_kind, 0.966718094236727_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,786) = (/ 0.8912290494899_dbl_kind, 0.89454080301974_dbl_kind, 0.89843275217143_dbl_kind, 0.915598993077228_dbl_kind, 0.966733128975979_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,787) = (/ 0.891229472861741_dbl_kind, 0.894541766933768_dbl_kind, 0.898436843632635_dbl_kind, 0.915611303602924_dbl_kind, 0.9667481293085_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,788) = (/ 0.891229895054286_dbl_kind, 0.894542729964594_dbl_kind, 0.898440932787218_dbl_kind, 0.915623605942913_dbl_kind, 0.966763095336813_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,789) = (/ 0.891230316071861_dbl_kind, 0.894543692118376_dbl_kind, 0.89844501964152_dbl_kind, 0.915635900122023_dbl_kind, 0.966778027163043_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,790) = (/ 0.89123073591878_dbl_kind, 0.894544653401214_dbl_kind, 0.898449104201836_dbl_kind, 0.915648186164951_dbl_kind, 0.966792924888929_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,791) = (/ 0.89123115459934_dbl_kind, 0.89454561381915_dbl_kind, 0.898453186474427_dbl_kind, 0.915660464096267_dbl_kind, 0.966807788615824_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,792) = (/ 0.891231572117828_dbl_kind, 0.894546573378166_dbl_kind, 0.898457266465513_dbl_kind, 0.91567273394041_dbl_kind, 0.966822618444691_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,793) = (/ 0.891231988478513_dbl_kind, 0.894547532084186_dbl_kind, 0.898461344181274_dbl_kind, 0.915684995721693_dbl_kind, 0.96683741447611_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,794) = (/ 0.89123240368565_dbl_kind, 0.89454848994308_dbl_kind, 0.898465419627856_dbl_kind, 0.915697249464299_dbl_kind, 0.96685217681028_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,795) = (/ 0.891232817743483_dbl_kind, 0.894549446960657_dbl_kind, 0.898469492811363_dbl_kind, 0.915709495192288_dbl_kind, 0.966866905547019_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,796) = (/ 0.891233230656236_dbl_kind, 0.894550403142673_dbl_kind, 0.898473563737862_dbl_kind, 0.915721732929594_dbl_kind, 0.966881600785766_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,797) = (/ 0.891233642428124_dbl_kind, 0.894551358494824_dbl_kind, 0.898477632413387_dbl_kind, 0.915733962700025_dbl_kind, 0.966896262625584_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,798) = (/ 0.891234053063345_dbl_kind, 0.894552313022754_dbl_kind, 0.89848169884393_dbl_kind, 0.915746184527265_dbl_kind, 0.966910891165158_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,799) = (/ 0.891234462566085_dbl_kind, 0.894553266732051_dbl_kind, 0.898485763035447_dbl_kind, 0.915758398434875_dbl_kind, 0.966925486502804_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,800) = (/ 0.891234870940513_dbl_kind, 0.894554219628247_dbl_kind, 0.898489824993862_dbl_kind, 0.915770604446294_dbl_kind, 0.966940048736462_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,801) = (/ 0.891235278190786_dbl_kind, 0.894555171716819_dbl_kind, 0.898493884725058_dbl_kind, 0.915782802584839_dbl_kind, 0.966954577963702_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,802) = (/ 0.891235684321047_dbl_kind, 0.894556123003193_dbl_kind, 0.898497942234886_dbl_kind, 0.915794992873705_dbl_kind, 0.966969074281729_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,803) = (/ 0.891236089335423_dbl_kind, 0.894557073492741_dbl_kind, 0.898501997529158_dbl_kind, 0.915807175335969_dbl_kind, 0.96698353778738_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,804) = (/ 0.891236493238028_dbl_kind, 0.894558023190779_dbl_kind, 0.898506050613654_dbl_kind, 0.915819349994585_dbl_kind, 0.966997968577127_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,805) = (/ 0.891236896032962_dbl_kind, 0.894558972102573_dbl_kind, 0.898510101494118_dbl_kind, 0.91583151687239_dbl_kind, 0.967012366747077_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,806) = (/ 0.891237297724312_dbl_kind, 0.894559920233336_dbl_kind, 0.898514150176259_dbl_kind, 0.915843675992103_dbl_kind, 0.967026732392979_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,807) = (/ 0.891237698316149_dbl_kind, 0.89456086758823_dbl_kind, 0.898518196665754_dbl_kind, 0.915855827376324_dbl_kind, 0.967041065610219_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,808) = (/ 0.891238097812531_dbl_kind, 0.894561814172364_dbl_kind, 0.898522240968244_dbl_kind, 0.915867971047539_dbl_kind, 0.967055366493826_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,809) = (/ 0.891238496217503_dbl_kind, 0.894562759990798_dbl_kind, 0.898526283089336_dbl_kind, 0.915880107028114_dbl_kind, 0.967069635138476_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,810) = (/ 0.891238893535095_dbl_kind, 0.894563705048539_dbl_kind, 0.898530323034604_dbl_kind, 0.915892235340303_dbl_kind, 0.967083871638481_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,811) = (/ 0.891239289769322_dbl_kind, 0.894564649350546_dbl_kind, 0.898534360809592_dbl_kind, 0.915904356006242_dbl_kind, 0.96709807608781_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,812) = (/ 0.891239684924187_dbl_kind, 0.894565592901728_dbl_kind, 0.898538396419807_dbl_kind, 0.915916469047957_dbl_kind, 0.967112248580073_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,813) = (/ 0.891240079003679_dbl_kind, 0.894566535706944_dbl_kind, 0.898542429870724_dbl_kind, 0.915928574487357_dbl_kind, 0.967126389208535_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,814) = (/ 0.891240472011772_dbl_kind, 0.894567477771004_dbl_kind, 0.898546461167789_dbl_kind, 0.91594067234624_dbl_kind, 0.96714049806611_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,815) = (/ 0.891240863952428_dbl_kind, 0.894568419098669_dbl_kind, 0.898550490316414_dbl_kind, 0.915952762646291_dbl_kind, 0.967154575245366_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,816) = (/ 0.891241254829593_dbl_kind, 0.894569359694653_dbl_kind, 0.898554517321978_dbl_kind, 0.915964845409085_dbl_kind, 0.967168620838524_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,817) = (/ 0.8912416446472_dbl_kind, 0.894570299563621_dbl_kind, 0.89855854218983_dbl_kind, 0.915976920656085_dbl_kind, 0.967182634937465_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,818) = (/ 0.89124203340917_dbl_kind, 0.894571238710193_dbl_kind, 0.898562564925289_dbl_kind, 0.915988988408644_dbl_kind, 0.967196617633724_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,819) = (/ 0.891242421119408_dbl_kind, 0.894572177138938_dbl_kind, 0.89856658553364_dbl_kind, 0.916001048688007_dbl_kind, 0.967210569018498_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,820) = (/ 0.891242807781806_dbl_kind, 0.894573114854383_dbl_kind, 0.89857060402014_dbl_kind, 0.916013101515308_dbl_kind, 0.967224489182648_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,821) = (/ 0.891243193400243_dbl_kind, 0.894574051861005_dbl_kind, 0.898574620390015_dbl_kind, 0.916025146911575_dbl_kind, 0.96723837821669_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,822) = (/ 0.891243577978585_dbl_kind, 0.894574988163237_dbl_kind, 0.89857863464846_dbl_kind, 0.916037184897728_dbl_kind, 0.967252236210809_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,823) = (/ 0.891243961520681_dbl_kind, 0.894575923765466_dbl_kind, 0.898582646800641_dbl_kind, 0.916049215494578_dbl_kind, 0.967266063254858_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,824) = (/ 0.891244344030371_dbl_kind, 0.894576858672035_dbl_kind, 0.898586656851695_dbl_kind, 0.916061238722834_dbl_kind, 0.967279859438352_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,825) = (/ 0.891244725511478_dbl_kind, 0.89457779288724_dbl_kind, 0.898590664806728_dbl_kind, 0.916073254603095_dbl_kind, 0.967293624850477_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,826) = (/ 0.891245105967814_dbl_kind, 0.894578726415336_dbl_kind, 0.898594670670818_dbl_kind, 0.916085263155857_dbl_kind, 0.967307359580091_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,827) = (/ 0.891245485403175_dbl_kind, 0.89457965926053_dbl_kind, 0.898598674449014_dbl_kind, 0.916097264401513_dbl_kind, 0.967321063715723_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,828) = (/ 0.891245863821345_dbl_kind, 0.894580591426988_dbl_kind, 0.898602676146334_dbl_kind, 0.91610925836035_dbl_kind, 0.967334737345572_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,829) = (/ 0.891246241226094_dbl_kind, 0.894581522918833_dbl_kind, 0.898606675767773_dbl_kind, 0.916121245052554_dbl_kind, 0.967348380557515_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,830) = (/ 0.89124661762118_dbl_kind, 0.894582453740145_dbl_kind, 0.898610673318293_dbl_kind, 0.916133224498206_dbl_kind, 0.967361993439107_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,831) = (/ 0.891246993010346_dbl_kind, 0.894583383894959_dbl_kind, 0.898614668802829_dbl_kind, 0.916145196717288_dbl_kind, 0.967375576077576_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,832) = (/ 0.891247367397323_dbl_kind, 0.894584313387271_dbl_kind, 0.898618662226291_dbl_kind, 0.916157161729678_dbl_kind, 0.967389128559832_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,833) = (/ 0.891247740785825_dbl_kind, 0.894585242221033_dbl_kind, 0.898622653593558_dbl_kind, 0.916169119555155_dbl_kind, 0.967402650972465_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,834) = (/ 0.891248113179559_dbl_kind, 0.894586170400158_dbl_kind, 0.898626642909484_dbl_kind, 0.916181070213398_dbl_kind, 0.967416143401749_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,835) = (/ 0.891248484582214_dbl_kind, 0.894587097928515_dbl_kind, 0.898630630178895_dbl_kind, 0.916193013723986_dbl_kind, 0.967429605933636_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,836) = (/ 0.891248854997466_dbl_kind, 0.894588024809934_dbl_kind, 0.898634615406592_dbl_kind, 0.916204950106398_dbl_kind, 0.96744303865377_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,837) = (/ 0.891249224428981_dbl_kind, 0.894588951048205_dbl_kind, 0.898638598597347_dbl_kind, 0.916216879380017_dbl_kind, 0.967456441647477_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,838) = (/ 0.891249592880407_dbl_kind, 0.894589876647078_dbl_kind, 0.898642579755907_dbl_kind, 0.916228801564127_dbl_kind, 0.967469814999771_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,839) = (/ 0.891249960355383_dbl_kind, 0.89459080161026_dbl_kind, 0.898646558886994_dbl_kind, 0.916240716677914_dbl_kind, 0.967483158795355_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,840) = (/ 0.891250326857534_dbl_kind, 0.894591725941423_dbl_kind, 0.898650535995301_dbl_kind, 0.91625262474047_dbl_kind, 0.967496473118626_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,841) = (/ 0.891250692390469_dbl_kind, 0.894592649644199_dbl_kind, 0.898654511085498_dbl_kind, 0.916264525770788_dbl_kind, 0.96750975805367_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,842) = (/ 0.891251056957787_dbl_kind, 0.894593572722178_dbl_kind, 0.89865848416223_dbl_kind, 0.916276419787767_dbl_kind, 0.967523013684265_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,843) = (/ 0.891251420563075_dbl_kind, 0.894594495178917_dbl_kind, 0.898662455230116_dbl_kind, 0.916288306810211_dbl_kind, 0.967536240093887_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,844) = (/ 0.891251783209902_dbl_kind, 0.89459541701793_dbl_kind, 0.898666424293748_dbl_kind, 0.91630018685683_dbl_kind, 0.967549437365706_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,845) = (/ 0.89125214490183_dbl_kind, 0.894596338242697_dbl_kind, 0.898670391357697_dbl_kind, 0.91631205994624_dbl_kind, 0.967562605582593_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,846) = (/ 0.891252505642403_dbl_kind, 0.894597258856658_dbl_kind, 0.898674356426505_dbl_kind, 0.916323926096964_dbl_kind, 0.96757574482711_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,847) = (/ 0.891252865435154_dbl_kind, 0.894598178863219_dbl_kind, 0.898678319504695_dbl_kind, 0.916335785327432_dbl_kind, 0.96758885518153_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,848) = (/ 0.891253224283605_dbl_kind, 0.894599098265745_dbl_kind, 0.898682280596761_dbl_kind, 0.916347637655981_dbl_kind, 0.967601936727816_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,849) = (/ 0.891253582191262_dbl_kind, 0.894600017067569_dbl_kind, 0.898686239707176_dbl_kind, 0.916359483100859_dbl_kind, 0.967614989547641_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,850) = (/ 0.89125393916162_dbl_kind, 0.894600935271985_dbl_kind, 0.898690196840389_dbl_kind, 0.91637132168022_dbl_kind, 0.967628013722385_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,851) = (/ 0.891254295198161_dbl_kind, 0.894601852882253_dbl_kind, 0.898694152000823_dbl_kind, 0.91638315341213_dbl_kind, 0.967641009333123_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,852) = (/ 0.891254650304353_dbl_kind, 0.894602769901597_dbl_kind, 0.898698105192882_dbl_kind, 0.916394978314564_dbl_kind, 0.967653976460646_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,853) = (/ 0.891255004483653_dbl_kind, 0.894603686333205_dbl_kind, 0.898702056420942_dbl_kind, 0.916406796405406_dbl_kind, 0.967666915185451_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,854) = (/ 0.891255357739504_dbl_kind, 0.894604602180231_dbl_kind, 0.89870600568936_dbl_kind, 0.916418607702455_dbl_kind, 0.967679825587744_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,855) = (/ 0.891255710075338_dbl_kind, 0.894605517445796_dbl_kind, 0.898709953002469_dbl_kind, 0.916430412223417_dbl_kind, 0.967692707747436_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,856) = (/ 0.891256061494571_dbl_kind, 0.894606432132983_dbl_kind, 0.89871389836458_dbl_kind, 0.916442209985914_dbl_kind, 0.967705561744161_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,857) = (/ 0.891256412000609_dbl_kind, 0.894607346244845_dbl_kind, 0.898717841779979_dbl_kind, 0.916454001007479_dbl_kind, 0.96771838765726_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,858) = (/ 0.891256761596846_dbl_kind, 0.8946082597844_dbl_kind, 0.898721783252933_dbl_kind, 0.916465785305558_dbl_kind, 0.967731185565785_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,859) = (/ 0.891257110286661_dbl_kind, 0.894609172754632_dbl_kind, 0.898725722787686_dbl_kind, 0.916477562897511_dbl_kind, 0.967743955548511_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,860) = (/ 0.891257458073422_dbl_kind, 0.894610085158492_dbl_kind, 0.898729660388459_dbl_kind, 0.916489333800611_dbl_kind, 0.967756697683927_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,861) = (/ 0.891257804960485_dbl_kind, 0.894610996998901_dbl_kind, 0.898733596059455_dbl_kind, 0.916501098032049_dbl_kind, 0.96776941205024_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,862) = (/ 0.891258150951192_dbl_kind, 0.894611908278744_dbl_kind, 0.898737529804852_dbl_kind, 0.916512855608929_dbl_kind, 0.967782098725377_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,863) = (/ 0.891258496048873_dbl_kind, 0.894612819000876_dbl_kind, 0.898741461628809_dbl_kind, 0.91652460654827_dbl_kind, 0.967794757786982_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,864) = (/ 0.891258840256847_dbl_kind, 0.89461372916812_dbl_kind, 0.898745391535462_dbl_kind, 0.916536350867008_dbl_kind, 0.967807389312431_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,865) = (/ 0.891259183578417_dbl_kind, 0.894614638783266_dbl_kind, 0.898749319528927_dbl_kind, 0.916548088581996_dbl_kind, 0.967819993378813_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,866) = (/ 0.891259526016879_dbl_kind, 0.894615547849076_dbl_kind, 0.898753245613301_dbl_kind, 0.916559819710005_dbl_kind, 0.967832570062947_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,867) = (/ 0.891259867575512_dbl_kind, 0.894616456368278_dbl_kind, 0.898757169792659_dbl_kind, 0.916571544267721_dbl_kind, 0.967845119441373_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,868) = (/ 0.891260208257585_dbl_kind, 0.89461736434357_dbl_kind, 0.898761092071056_dbl_kind, 0.916583262271751_dbl_kind, 0.967857641590365_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,869) = (/ 0.891260548066354_dbl_kind, 0.894618271777621_dbl_kind, 0.898765012452526_dbl_kind, 0.91659497373862_dbl_kind, 0.967870136585919_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,870) = (/ 0.891260887005063_dbl_kind, 0.894619178673068_dbl_kind, 0.898768930941085_dbl_kind, 0.91660667868477_dbl_kind, 0.967882604503762_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,871) = (/ 0.891261225076943_dbl_kind, 0.894620085032519_dbl_kind, 0.898772847540727_dbl_kind, 0.916618377126566_dbl_kind, 0.967895045419355_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,872) = (/ 0.891261562285215_dbl_kind, 0.894620990858555_dbl_kind, 0.89877676225543_dbl_kind, 0.916630069080291_dbl_kind, 0.967907459407882_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,873) = (/ 0.891261898633086_dbl_kind, 0.894621896153723_dbl_kind, 0.898780675089148_dbl_kind, 0.916641754562147_dbl_kind, 0.967919846544272_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,874) = (/ 0.891262234123751_dbl_kind, 0.894622800920544_dbl_kind, 0.898784586045819_dbl_kind, 0.916653433588262_dbl_kind, 0.967932206903179_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,875) = (/ 0.891262568760393_dbl_kind, 0.894623705161509_dbl_kind, 0.898788495129361_dbl_kind, 0.91666510617468_dbl_kind, 0.967944540558992_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,876) = (/ 0.891262902546184_dbl_kind, 0.894624608879083_dbl_kind, 0.898792402343673_dbl_kind, 0.916676772337371_dbl_kind, 0.967956847585842_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,877) = (/ 0.891263235484283_dbl_kind, 0.8946255120757_dbl_kind, 0.898796307692636_dbl_kind, 0.916688432092225_dbl_kind, 0.967969128057593_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,878) = (/ 0.891263567577836_dbl_kind, 0.894626414753766_dbl_kind, 0.898800211180111_dbl_kind, 0.916700085455056_dbl_kind, 0.967981382047852_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,879) = (/ 0.89126389882998_dbl_kind, 0.894627316915662_dbl_kind, 0.898804112809942_dbl_kind, 0.916711732441601_dbl_kind, 0.967993609629958_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,880) = (/ 0.891264229243838_dbl_kind, 0.89462821856374_dbl_kind, 0.898808012585954_dbl_kind, 0.91672337306752_dbl_kind, 0.968005810877_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,881) = (/ 0.891264558822521_dbl_kind, 0.894629119700323_dbl_kind, 0.898811910511955_dbl_kind, 0.916735007348399_dbl_kind, 0.968017985861802_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,882) = (/ 0.891264887569129_dbl_kind, 0.894630020327711_dbl_kind, 0.898815806591734_dbl_kind, 0.916746635299747_dbl_kind, 0.968030134656937_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,883) = (/ 0.891265215486749_dbl_kind, 0.894630920448174_dbl_kind, 0.898819700829063_dbl_kind, 0.916758256936999_dbl_kind, 0.968042257334718_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,884) = (/ 0.891265542578458_dbl_kind, 0.894631820063957_dbl_kind, 0.898823593227695_dbl_kind, 0.916769872275516_dbl_kind, 0.968054353967203_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,885) = (/ 0.891265868847321_dbl_kind, 0.89463271917728_dbl_kind, 0.898827483791367_dbl_kind, 0.916781481330583_dbl_kind, 0.968066424626204_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,886) = (/ 0.891266194296389_dbl_kind, 0.894633617790334_dbl_kind, 0.8988313725238_dbl_kind, 0.916793084117414_dbl_kind, 0.968078469383265_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,887) = (/ 0.891266518928704_dbl_kind, 0.894634515905288_dbl_kind, 0.898835259428694_dbl_kind, 0.916804680651148_dbl_kind, 0.968090488309698_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,888) = (/ 0.891266842747294_dbl_kind, 0.894635413524283_dbl_kind, 0.898839144509735_dbl_kind, 0.916816270946852_dbl_kind, 0.968102481476552_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,889) = (/ 0.891267165755178_dbl_kind, 0.894636310649435_dbl_kind, 0.898843027770592_dbl_kind, 0.916827855019521_dbl_kind, 0.968114448954628_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,890) = (/ 0.891267487955362_dbl_kind, 0.894637207282836_dbl_kind, 0.898846909214918_dbl_kind, 0.916839432884077_dbl_kind, 0.968126390814484_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,891) = (/ 0.891267809350839_dbl_kind, 0.894638103426554_dbl_kind, 0.898850788846346_dbl_kind, 0.916851004555372_dbl_kind, 0.968138307126426_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,892) = (/ 0.891268129944594_dbl_kind, 0.894638999082631_dbl_kind, 0.898854666668497_dbl_kind, 0.916862570048185_dbl_kind, 0.968150197960516_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,893) = (/ 0.891268449739598_dbl_kind, 0.894639894253085_dbl_kind, 0.898858542684973_dbl_kind, 0.916874129377228_dbl_kind, 0.968162063386576_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,894) = (/ 0.89126876873881_dbl_kind, 0.894640788939911_dbl_kind, 0.898862416899362_dbl_kind, 0.916885682557139_dbl_kind, 0.968173903474175_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,895) = (/ 0.891269086945179_dbl_kind, 0.894641683145079_dbl_kind, 0.898866289315234_dbl_kind, 0.916897229602488_dbl_kind, 0.968185718292644_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,896) = (/ 0.891269404361644_dbl_kind, 0.894642576870537_dbl_kind, 0.898870159936144_dbl_kind, 0.916908770527776_dbl_kind, 0.968197507911073_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,897) = (/ 0.891269720991129_dbl_kind, 0.894643470118208_dbl_kind, 0.898874028765633_dbl_kind, 0.916920305347432_dbl_kind, 0.968209272398311_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,898) = (/ 0.891270036836549_dbl_kind, 0.894644362889992_dbl_kind, 0.898877895807224_dbl_kind, 0.916931834075822_dbl_kind, 0.968221011822966_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,899) = (/ 0.891270351900808_dbl_kind, 0.894645255187769_dbl_kind, 0.898881761064426_dbl_kind, 0.91694335672724_dbl_kind, 0.968232726253407_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,900) = (/ 0.891270666186797_dbl_kind, 0.894646147013393_dbl_kind, 0.898885624540733_dbl_kind, 0.916954873315913_dbl_kind, 0.968244415757771_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,901) = (/ 0.891270979697398_dbl_kind, 0.894647038368697_dbl_kind, 0.898889486239623_dbl_kind, 0.916966383855999_dbl_kind, 0.968256080403949_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,902) = (/ 0.89127129243548_dbl_kind, 0.894647929255493_dbl_kind, 0.898893346164561_dbl_kind, 0.916977888361593_dbl_kind, 0.968267720259603_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,903) = (/ 0.891271604403903_dbl_kind, 0.894648819675567_dbl_kind, 0.898897204318996_dbl_kind, 0.91698938684672_dbl_kind, 0.968279335392158_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,904) = (/ 0.891271915605512_dbl_kind, 0.894649709630689_dbl_kind, 0.898901060706362_dbl_kind, 0.91700087932534_dbl_kind, 0.968290925868811_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,905) = (/ 0.891272226043145_dbl_kind, 0.894650599122602_dbl_kind, 0.898904915330078_dbl_kind, 0.917012365811347_dbl_kind, 0.968302491756516_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,906) = (/ 0.891272535719628_dbl_kind, 0.894651488153032_dbl_kind, 0.898908768193551_dbl_kind, 0.917023846318572_dbl_kind, 0.968314033122006_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,907) = (/ 0.891272844637773_dbl_kind, 0.89465237672368_dbl_kind, 0.898912619300171_dbl_kind, 0.917035320860775_dbl_kind, 0.968325550031774_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,908) = (/ 0.891273152800385_dbl_kind, 0.894653264836231_dbl_kind, 0.898916468653317_dbl_kind, 0.917046789451659_dbl_kind, 0.968337042552091_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,909) = (/ 0.891273460210257_dbl_kind, 0.894654152492344_dbl_kind, 0.898920316256351_dbl_kind, 0.917058252104856_dbl_kind, 0.968348510748997_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,910) = (/ 0.89127376687017_dbl_kind, 0.89465503969366_dbl_kind, 0.898924162112624_dbl_kind, 0.917069708833938_dbl_kind, 0.968359954688303_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,911) = (/ 0.891274072782893_dbl_kind, 0.894655926441802_dbl_kind, 0.89892800622547_dbl_kind, 0.917081159652414_dbl_kind, 0.968371374435594_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,912) = (/ 0.891274377951188_dbl_kind, 0.894656812738368_dbl_kind, 0.898931848598212_dbl_kind, 0.917092604573727_dbl_kind, 0.96838277005623_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,913) = (/ 0.891274682377804_dbl_kind, 0.89465769858494_dbl_kind, 0.89893568923416_dbl_kind, 0.917104043611259_dbl_kind, 0.968394141615345_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,914) = (/ 0.891274986065477_dbl_kind, 0.894658583983078_dbl_kind, 0.898939528136608_dbl_kind, 0.91711547677833_dbl_kind, 0.96840548917785_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,915) = (/ 0.891275289016936_dbl_kind, 0.894659468934325_dbl_kind, 0.89894336530884_dbl_kind, 0.917126904088196_dbl_kind, 0.968416812808436_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,916) = (/ 0.891275591234898_dbl_kind, 0.894660353440202_dbl_kind, 0.898947200754124_dbl_kind, 0.917138325554053_dbl_kind, 0.968428112571565_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,917) = (/ 0.891275892722069_dbl_kind, 0.894661237502212_dbl_kind, 0.898951034475718_dbl_kind, 0.917149741189036_dbl_kind, 0.968439388531483_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,918) = (/ 0.891276193481145_dbl_kind, 0.89466212112184_dbl_kind, 0.898954866476864_dbl_kind, 0.917161151006219_dbl_kind, 0.968450640752217_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,919) = (/ 0.89127649351481_dbl_kind, 0.89466300430055_dbl_kind, 0.898958696760794_dbl_kind, 0.917172555018613_dbl_kind, 0.968461869297575_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,920) = (/ 0.891276792825738_dbl_kind, 0.894663887039791_dbl_kind, 0.898962525330726_dbl_kind, 0.917183953239172_dbl_kind, 0.96847307423114_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,921) = (/ 0.891277091416595_dbl_kind, 0.894664769340991_dbl_kind, 0.898966352189867_dbl_kind, 0.917195345680788_dbl_kind, 0.968484255616288_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,922) = (/ 0.891277389290031_dbl_kind, 0.89466565120556_dbl_kind, 0.89897017734141_dbl_kind, 0.917206732356294_dbl_kind, 0.968495413516169_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,923) = (/ 0.891277686448692_dbl_kind, 0.89466653263489_dbl_kind, 0.898974000788536_dbl_kind, 0.917218113278465_dbl_kind, 0.968506547993722_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,924) = (/ 0.89127798289521_dbl_kind, 0.894667413630356_dbl_kind, 0.898977822534415_dbl_kind, 0.917229488460016_dbl_kind, 0.968517659111676_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,925) = (/ 0.891278278632205_dbl_kind, 0.894668294193316_dbl_kind, 0.898981642582205_dbl_kind, 0.917240857913604_dbl_kind, 0.968528746932538_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,926) = (/ 0.89127857366229_dbl_kind, 0.894669174325108_dbl_kind, 0.898985460935052_dbl_kind, 0.917252221651826_dbl_kind, 0.968539811518605_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,927) = (/ 0.891278867988067_dbl_kind, 0.894670054027056_dbl_kind, 0.898989277596088_dbl_kind, 0.917263579687224_dbl_kind, 0.968550852931969_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,928) = (/ 0.891279161612127_dbl_kind, 0.894670933300464_dbl_kind, 0.898993092568437_dbl_kind, 0.917274932032281_dbl_kind, 0.968561871234499_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,929) = (/ 0.891279454537049_dbl_kind, 0.894671812146622_dbl_kind, 0.898996905855209_dbl_kind, 0.917286278699424_dbl_kind, 0.968572866487866_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,930) = (/ 0.891279746765407_dbl_kind, 0.894672690566801_dbl_kind, 0.899000717459504_dbl_kind, 0.917297619701022_dbl_kind, 0.968583838753518_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,931) = (/ 0.891280038299758_dbl_kind, 0.894673568562255_dbl_kind, 0.899004527384411_dbl_kind, 0.917308955049387_dbl_kind, 0.968594788092713_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,932) = (/ 0.891280329142655_dbl_kind, 0.894674446134224_dbl_kind, 0.899008335633005_dbl_kind, 0.917320284756777_dbl_kind, 0.968605714566484_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,933) = (/ 0.891280619296636_dbl_kind, 0.894675323283931_dbl_kind, 0.899012142208353_dbl_kind, 0.917331608835393_dbl_kind, 0.968616618235669_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,934) = (/ 0.891280908764233_dbl_kind, 0.894676200012581_dbl_kind, 0.899015947113512_dbl_kind, 0.91734292729738_dbl_kind, 0.968627499160892_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,935) = (/ 0.891281197547965_dbl_kind, 0.894677076321366_dbl_kind, 0.899019750351523_dbl_kind, 0.917354240154828_dbl_kind, 0.968638357402581_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,936) = (/ 0.891281485650343_dbl_kind, 0.89467795221146_dbl_kind, 0.899023551925421_dbl_kind, 0.917365547419773_dbl_kind, 0.968649193020952_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,937) = (/ 0.891281773073866_dbl_kind, 0.894678827684023_dbl_kind, 0.899027351838229_dbl_kind, 0.917376849104196_dbl_kind, 0.96866000607602_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,938) = (/ 0.891282059821025_dbl_kind, 0.894679702740199_dbl_kind, 0.899031150092959_dbl_kind, 0.917388145220023_dbl_kind, 0.968670796627604_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,939) = (/ 0.8912823458943_dbl_kind, 0.894680577381117_dbl_kind, 0.899034946692614_dbl_kind, 0.917399435779129_dbl_kind, 0.968681564735313_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,940) = (/ 0.891282631296161_dbl_kind, 0.89468145160789_dbl_kind, 0.899038741640184_dbl_kind, 0.917410720793331_dbl_kind, 0.968692310458559_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,941) = (/ 0.891282916029069_dbl_kind, 0.894682325421616_dbl_kind, 0.899042534938652_dbl_kind, 0.917422000274397_dbl_kind, 0.968703033856555_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,942) = (/ 0.891283200095475_dbl_kind, 0.894683198823382_dbl_kind, 0.899046326590989_dbl_kind, 0.917433274234039_dbl_kind, 0.968713734988311_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,943) = (/ 0.891283483497819_dbl_kind, 0.894684071814255_dbl_kind, 0.899050116600156_dbl_kind, 0.917444542683918_dbl_kind, 0.968724413912648_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,944) = (/ 0.891283766238533_dbl_kind, 0.894684944395291_dbl_kind, 0.899053904969105_dbl_kind, 0.917455805635643_dbl_kind, 0.968735070688177_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,945) = (/ 0.891284048320039_dbl_kind, 0.89468581656753_dbl_kind, 0.899057691700776_dbl_kind, 0.917467063100768_dbl_kind, 0.968745705373322_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,946) = (/ 0.891284329744748_dbl_kind, 0.894686688332_dbl_kind, 0.899061476798104_dbl_kind, 0.9174783150908_dbl_kind, 0.968756318026311_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,947) = (/ 0.891284610515062_dbl_kind, 0.894687559689712_dbl_kind, 0.89906526026401_dbl_kind, 0.91748956161719_dbl_kind, 0.968766908705172_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,948) = (/ 0.891284890633375_dbl_kind, 0.894688430641666_dbl_kind, 0.899069042101406_dbl_kind, 0.917500802691341_dbl_kind, 0.96877747746774_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,949) = (/ 0.891285170102068_dbl_kind, 0.894689301188846_dbl_kind, 0.899072822313197_dbl_kind, 0.917512038324603_dbl_kind, 0.968788024371661_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,950) = (/ 0.891285448923517_dbl_kind, 0.894690171332225_dbl_kind, 0.899076600902276_dbl_kind, 0.917523268528278_dbl_kind, 0.968798549474384_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,951) = (/ 0.891285727100085_dbl_kind, 0.89469104107276_dbl_kind, 0.89908037787153_dbl_kind, 0.917534493313616_dbl_kind, 0.96880905283317_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,952) = (/ 0.891286004634126_dbl_kind, 0.894691910411397_dbl_kind, 0.899084153223833_dbl_kind, 0.917545712691816_dbl_kind, 0.968819534505084_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,953) = (/ 0.891286281527985_dbl_kind, 0.894692779349067_dbl_kind, 0.899087926962052_dbl_kind, 0.91755692667403_dbl_kind, 0.968829994547008_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,954) = (/ 0.891286557783999_dbl_kind, 0.894693647886688_dbl_kind, 0.899091699089046_dbl_kind, 0.917568135271359_dbl_kind, 0.968840433015628_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,955) = (/ 0.891286833404495_dbl_kind, 0.894694516025168_dbl_kind, 0.899095469607663_dbl_kind, 0.917579338494855_dbl_kind, 0.968850849967445_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,956) = (/ 0.891287108391787_dbl_kind, 0.894695383765399_dbl_kind, 0.899099238520745_dbl_kind, 0.917590536355522_dbl_kind, 0.968861245458773_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,957) = (/ 0.891287382748187_dbl_kind, 0.894696251108263_dbl_kind, 0.899103005831123_dbl_kind, 0.917601728864314_dbl_kind, 0.968871619545735_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,958) = (/ 0.891287656475989_dbl_kind, 0.894697118054627_dbl_kind, 0.899106771541619_dbl_kind, 0.917612916032139_dbl_kind, 0.968881972284273_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,959) = (/ 0.891287929577486_dbl_kind, 0.894697984605347_dbl_kind, 0.89911053565505_dbl_kind, 0.917624097869854_dbl_kind, 0.968892303730138_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,960) = (/ 0.891288202054956_dbl_kind, 0.894698850761268_dbl_kind, 0.89911429817422_dbl_kind, 0.917635274388271_dbl_kind, 0.968902613938899_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,961) = (/ 0.891288473910671_dbl_kind, 0.894699716523222_dbl_kind, 0.899118059101929_dbl_kind, 0.917646445598154_dbl_kind, 0.968912902965944_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,962) = (/ 0.891288745146893_dbl_kind, 0.894700581892028_dbl_kind, 0.899121818440965_dbl_kind, 0.917657611510219_dbl_kind, 0.968923170866473_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,963) = (/ 0.891289015765874_dbl_kind, 0.894701446868495_dbl_kind, 0.899125576194111_dbl_kind, 0.917668772135135_dbl_kind, 0.968933417695506_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,964) = (/ 0.891289285769858_dbl_kind, 0.894702311453418_dbl_kind, 0.89912933236414_dbl_kind, 0.917679927483527_dbl_kind, 0.968943643507879_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,965) = (/ 0.89128955516108_dbl_kind, 0.894703175647585_dbl_kind, 0.899133086953818_dbl_kind, 0.917691077565969_dbl_kind, 0.96895384835825_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,966) = (/ 0.891289823941765_dbl_kind, 0.894704039451768_dbl_kind, 0.899136839965904_dbl_kind, 0.917702222392994_dbl_kind, 0.968964032301097_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,967) = (/ 0.891290092114131_dbl_kind, 0.89470490286673_dbl_kind, 0.899140591403146_dbl_kind, 0.917713361975085_dbl_kind, 0.968974195390714_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,968) = (/ 0.891290359680385_dbl_kind, 0.894705765893223_dbl_kind, 0.899144341268288_dbl_kind, 0.917724496322684_dbl_kind, 0.96898433768122_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,969) = (/ 0.891290626642725_dbl_kind, 0.894706628531985_dbl_kind, 0.899148089564063_dbl_kind, 0.917735625446182_dbl_kind, 0.968994459226556_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,970) = (/ 0.891290893003342_dbl_kind, 0.894707490783748_dbl_kind, 0.899151836293199_dbl_kind, 0.917746749355929_dbl_kind, 0.969004560080486_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,971) = (/ 0.891291158764418_dbl_kind, 0.894708352649231_dbl_kind, 0.899155581458416_dbl_kind, 0.91775786806223_dbl_kind, 0.969014640296593_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,972) = (/ 0.891291423928124_dbl_kind, 0.894709214129141_dbl_kind, 0.899159325062427_dbl_kind, 0.917768981575345_dbl_kind, 0.969024699928291_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,973) = (/ 0.891291688496625_dbl_kind, 0.894710075224175_dbl_kind, 0.899163067107935_dbl_kind, 0.917780089905488_dbl_kind, 0.969034739028811_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,974) = (/ 0.891291952472075_dbl_kind, 0.894710935935022_dbl_kind, 0.89916680759764_dbl_kind, 0.917791193062833_dbl_kind, 0.969044757651216_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,975) = (/ 0.891292215856621_dbl_kind, 0.894711796262358_dbl_kind, 0.899170546534231_dbl_kind, 0.917802291057506_dbl_kind, 0.969054755848393_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,976) = (/ 0.8912924786524_dbl_kind, 0.894712656206851_dbl_kind, 0.899174283920392_dbl_kind, 0.917813383899592_dbl_kind, 0.969064733673054_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,977) = (/ 0.891292740861542_dbl_kind, 0.894713515769155_dbl_kind, 0.8991780197588_dbl_kind, 0.917824471599132_dbl_kind, 0.969074691177741_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,978) = (/ 0.891293002486166_dbl_kind, 0.89471437494992_dbl_kind, 0.899181754052124_dbl_kind, 0.917835554166124_dbl_kind, 0.969084628414828_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,979) = (/ 0.891293263528385_dbl_kind, 0.894715233749781_dbl_kind, 0.899185486803027_dbl_kind, 0.917846631610524_dbl_kind, 0.969094545436507_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,980) = (/ 0.891293523990301_dbl_kind, 0.894716092169366_dbl_kind, 0.899189218014166_dbl_kind, 0.917857703942245_dbl_kind, 0.969104442294809_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,981) = (/ 0.891293783874011_dbl_kind, 0.894716950209293_dbl_kind, 0.899192947688188_dbl_kind, 0.917868771171157_dbl_kind, 0.969114319041596_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,982) = (/ 0.8912940431816_dbl_kind, 0.89471780787017_dbl_kind, 0.899196675827738_dbl_kind, 0.917879833307089_dbl_kind, 0.969124175728554_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,983) = (/ 0.891294301915145_dbl_kind, 0.894718665152595_dbl_kind, 0.899200402435451_dbl_kind, 0.917890890359828_dbl_kind, 0.969134012407205_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,984) = (/ 0.891294560076718_dbl_kind, 0.894719522057159_dbl_kind, 0.899204127513957_dbl_kind, 0.917901942339119_dbl_kind, 0.969143829128904_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,985) = (/ 0.891294817668378_dbl_kind, 0.894720378584442_dbl_kind, 0.899207851065879_dbl_kind, 0.917912989254667_dbl_kind, 0.969153625944837_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,986) = (/ 0.891295074692179_dbl_kind, 0.894721234735014_dbl_kind, 0.899211573093834_dbl_kind, 0.917924031116135_dbl_kind, 0.969163402906029_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,987) = (/ 0.891295331150165_dbl_kind, 0.894722090509439_dbl_kind, 0.899215293600432_dbl_kind, 0.917935067933143_dbl_kind, 0.96917316006333_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,988) = (/ 0.891295587044373_dbl_kind, 0.89472294590827_dbl_kind, 0.899219012588278_dbl_kind, 0.917946099715275_dbl_kind, 0.969182897467433_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,989) = (/ 0.89129584237683_dbl_kind, 0.894723800932052_dbl_kind, 0.899222730059971_dbl_kind, 0.917957126472072_dbl_kind, 0.969192615168866_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,990) = (/ 0.891296097149557_dbl_kind, 0.89472465558132_dbl_kind, 0.899226446018101_dbl_kind, 0.917968148213034_dbl_kind, 0.969202313217987_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,991) = (/ 0.891296351364564_dbl_kind, 0.894725509856602_dbl_kind, 0.899230160465255_dbl_kind, 0.917979164947623_dbl_kind, 0.969211991665_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,992) = (/ 0.891296605023856_dbl_kind, 0.894726363758419_dbl_kind, 0.899233873404014_dbl_kind, 0.91799017668526_dbl_kind, 0.969221650559939_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,993) = (/ 0.891296858129429_dbl_kind, 0.894727217287279_dbl_kind, 0.899237584836952_dbl_kind, 0.918001183435327_dbl_kind, 0.969231289952681_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,994) = (/ 0.891297110683268_dbl_kind, 0.894728070443687_dbl_kind, 0.899241294766637_dbl_kind, 0.918012185207167_dbl_kind, 0.969240909892942_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,995) = (/ 0.891297362687354_dbl_kind, 0.894728923228135_dbl_kind, 0.899245003195632_dbl_kind, 0.918023182010084_dbl_kind, 0.969250510430276_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,996) = (/ 0.891297614143658_dbl_kind, 0.89472977564111_dbl_kind, 0.899248710126493_dbl_kind, 0.918034173853343_dbl_kind, 0.969260091614076_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,997) = (/ 0.891297865054144_dbl_kind, 0.894730627683092_dbl_kind, 0.899252415561773_dbl_kind, 0.91804516074617_dbl_kind, 0.969269653493577_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,998) = (/ 0.891298115420766_dbl_kind, 0.89473147935455_dbl_kind, 0.899256119504017_dbl_kind, 0.918056142697753_dbl_kind, 0.969279196117858_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,999) = (/ 0.891298365245472_dbl_kind, 0.894732330655946_dbl_kind, 0.899259821955766_dbl_kind, 0.918067119717242_dbl_kind, 0.96928871953584_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1000) = (/ 0.891298614530201_dbl_kind, 0.894733181587737_dbl_kind, 0.899263522919554_dbl_kind, 0.918078091813749_dbl_kind, 0.96929822379628_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1001) = (/ 0.891298863276887_dbl_kind, 0.894734032150368_dbl_kind, 0.89926722239791_dbl_kind, 0.918089058996348_dbl_kind, 0.969307708947787_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1002) = (/ 0.89129911148745_dbl_kind, 0.894734882344282_dbl_kind, 0.89927092039336_dbl_kind, 0.918100021274076_dbl_kind, 0.969317175038811_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1003) = (/ 0.891299359163809_dbl_kind, 0.894735732169908_dbl_kind, 0.899274616908421_dbl_kind, 0.918110978655933_dbl_kind, 0.969326622117643_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1004) = (/ 0.891299606307872_dbl_kind, 0.894736581627674_dbl_kind, 0.899278311945608_dbl_kind, 0.91812193115088_dbl_kind, 0.969336050232424_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1005) = (/ 0.891299852921538_dbl_kind, 0.894737430717996_dbl_kind, 0.899282005507428_dbl_kind, 0.918132878767844_dbl_kind, 0.969345459431143_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1006) = (/ 0.891300099006701_dbl_kind, 0.894738279441287_dbl_kind, 0.899285697596386_dbl_kind, 0.918143821515713_dbl_kind, 0.969354849761626_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1007) = (/ 0.891300344565247_dbl_kind, 0.894739127797948_dbl_kind, 0.899289388214979_dbl_kind, 0.918154759403339_dbl_kind, 0.969364221271555_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1008) = (/ 0.891300589599052_dbl_kind, 0.894739975788377_dbl_kind, 0.899293077365701_dbl_kind, 0.918165692439538_dbl_kind, 0.969373574008455_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1009) = (/ 0.891300834109986_dbl_kind, 0.894740823412965_dbl_kind, 0.899296765051041_dbl_kind, 0.918176620633091_dbl_kind, 0.969382908019702_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1010) = (/ 0.891301078099912_dbl_kind, 0.894741670672093_dbl_kind, 0.899300451273482_dbl_kind, 0.918187543992741_dbl_kind, 0.96939222335252_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1011) = (/ 0.891301321570685_dbl_kind, 0.894742517566139_dbl_kind, 0.899304136035501_dbl_kind, 0.918198462527198_dbl_kind, 0.96940152005398_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1012) = (/ 0.891301564524152_dbl_kind, 0.894743364095472_dbl_kind, 0.899307819339575_dbl_kind, 0.918209376245133_dbl_kind, 0.969410798171008_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1013) = (/ 0.891301806962152_dbl_kind, 0.894744210260454_dbl_kind, 0.899311501188171_dbl_kind, 0.918220285155184_dbl_kind, 0.969420057750377_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1014) = (/ 0.891302048886519_dbl_kind, 0.894745056061444_dbl_kind, 0.899315181583754_dbl_kind, 0.918231189265955_dbl_kind, 0.969429298838713_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1015) = (/ 0.891302290299076_dbl_kind, 0.89474590149879_dbl_kind, 0.899318860528784_dbl_kind, 0.918242088586013_dbl_kind, 0.969438521482491_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1016) = (/ 0.891302531201641_dbl_kind, 0.894746746572838_dbl_kind, 0.899322538025717_dbl_kind, 0.918252983123891_dbl_kind, 0.969447725728041_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1017) = (/ 0.891302771596026_dbl_kind, 0.894747591283924_dbl_kind, 0.899326214077002_dbl_kind, 0.918263872888087_dbl_kind, 0.969456911621549_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1018) = (/ 0.891303011484031_dbl_kind, 0.894748435632381_dbl_kind, 0.899329888685088_dbl_kind, 0.918274757887067_dbl_kind, 0.969466079209046_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1019) = (/ 0.891303250867453_dbl_kind, 0.894749279618533_dbl_kind, 0.899333561852415_dbl_kind, 0.918285638129259_dbl_kind, 0.969475228536426_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1020) = (/ 0.89130348974808_dbl_kind, 0.894750123242699_dbl_kind, 0.89933723358142_dbl_kind, 0.918296513623061_dbl_kind, 0.96948435964943_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1021) = (/ 0.891303728127693_dbl_kind, 0.894750966505195_dbl_kind, 0.899340903874537_dbl_kind, 0.918307384376834_dbl_kind, 0.969493472593661_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1022) = (/ 0.891303966008066_dbl_kind, 0.894751809406327_dbl_kind, 0.899344572734195_dbl_kind, 0.918318250398909_dbl_kind, 0.969502567414575_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1023) = (/ 0.891304203390964_dbl_kind, 0.894752651946397_dbl_kind, 0.89934824016282_dbl_kind, 0.918329111697578_dbl_kind, 0.969511644157481_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1024) = (/ 0.891304440278148_dbl_kind, 0.8947534941257_dbl_kind, 0.899351906162829_dbl_kind, 0.918339968281107_dbl_kind, 0.969520702867554_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1025) = (/ 0.89130467667137_dbl_kind, 0.894754335944529_dbl_kind, 0.899355570736641_dbl_kind, 0.918350820157723_dbl_kind, 0.969529743589817_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1026) = (/ 0.891304912572373_dbl_kind, 0.894755177403167_dbl_kind, 0.899359233886668_dbl_kind, 0.918361667335624_dbl_kind, 0.969538766369155_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1027) = (/ 0.891305147982898_dbl_kind, 0.894756018501895_dbl_kind, 0.899362895615317_dbl_kind, 0.918372509822972_dbl_kind, 0.969547771250312_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1028) = (/ 0.891305382904674_dbl_kind, 0.894756859240985_dbl_kind, 0.899366555924992_dbl_kind, 0.9183833476279_dbl_kind, 0.969556758277889_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1029) = (/ 0.891305617339426_dbl_kind, 0.894757699620707_dbl_kind, 0.899370214818095_dbl_kind, 0.918394180758506_dbl_kind, 0.969565727496353_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1030) = (/ 0.89130585128887_dbl_kind, 0.894758539641324_dbl_kind, 0.899373872297021_dbl_kind, 0.918405009222858_dbl_kind, 0.969574678950022_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1031) = (/ 0.891306084754718_dbl_kind, 0.894759379303095_dbl_kind, 0.899377528364162_dbl_kind, 0.91841583302899_dbl_kind, 0.969583612683082_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1032) = (/ 0.89130631773867_dbl_kind, 0.894760218606272_dbl_kind, 0.899381183021906_dbl_kind, 0.918426652184905_dbl_kind, 0.969592528739576_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1033) = (/ 0.891306550242423_dbl_kind, 0.894761057551103_dbl_kind, 0.899384836272639_dbl_kind, 0.918437466698575_dbl_kind, 0.969601427163411_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1034) = (/ 0.891306782267669_dbl_kind, 0.894761896137831_dbl_kind, 0.899388488118742_dbl_kind, 0.91844827657794_dbl_kind, 0.969610307998356_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1035) = (/ 0.891307013816087_dbl_kind, 0.894762734366694_dbl_kind, 0.899392138562591_dbl_kind, 0.91845908183091_dbl_kind, 0.969619171288044_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1036) = (/ 0.891307244889354_dbl_kind, 0.894763572237926_dbl_kind, 0.899395787606559_dbl_kind, 0.918469882465361_dbl_kind, 0.969628017075969_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1037) = (/ 0.891307475489139_dbl_kind, 0.894764409751753_dbl_kind, 0.899399435253017_dbl_kind, 0.918480678489141_dbl_kind, 0.969636845405492_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1038) = (/ 0.891307705617104_dbl_kind, 0.894765246908401_dbl_kind, 0.899403081504331_dbl_kind, 0.918491469910065_dbl_kind, 0.969645656319835_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1039) = (/ 0.891307935274903_dbl_kind, 0.894766083708088_dbl_kind, 0.899406726362864_dbl_kind, 0.91850225673592_dbl_kind, 0.969654449862087_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1040) = (/ 0.891308164464187_dbl_kind, 0.894766920151027_dbl_kind, 0.899410369830973_dbl_kind, 0.91851303897446_dbl_kind, 0.969663226075204_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1041) = (/ 0.891308393186595_dbl_kind, 0.894767756237429_dbl_kind, 0.899414011911015_dbl_kind, 0.91852381663341_dbl_kind, 0.969671985002004_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1042) = (/ 0.891308621443765_dbl_kind, 0.894768591967499_dbl_kind, 0.899417652605343_dbl_kind, 0.918534589720465_dbl_kind, 0.969680726685176_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1043) = (/ 0.891308849237324_dbl_kind, 0.894769427341437_dbl_kind, 0.899421291916304_dbl_kind, 0.918545358243288_dbl_kind, 0.969689451167275_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1044) = (/ 0.891309076568893_dbl_kind, 0.89477026235944_dbl_kind, 0.899424929846243_dbl_kind, 0.918556122209516_dbl_kind, 0.96969815849072_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1045) = (/ 0.89130930344009_dbl_kind, 0.894771097021699_dbl_kind, 0.899428566397503_dbl_kind, 0.918566881626754_dbl_kind, 0.969706848697803_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1046) = (/ 0.891309529852522_dbl_kind, 0.894771931328403_dbl_kind, 0.899432201572423_dbl_kind, 0.918577636502576_dbl_kind, 0.969715521830681_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1047) = (/ 0.891309755807793_dbl_kind, 0.894772765279734_dbl_kind, 0.899435835373336_dbl_kind, 0.91858838684453_dbl_kind, 0.969724177931383_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1048) = (/ 0.891309981307496_dbl_kind, 0.894773598875873_dbl_kind, 0.899439467802576_dbl_kind, 0.918599132660133_dbl_kind, 0.969732817041804_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1049) = (/ 0.891310206353223_dbl_kind, 0.894774432116993_dbl_kind, 0.89944309886247_dbl_kind, 0.918609873956873_dbl_kind, 0.969741439203714_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1050) = (/ 0.891310430946556_dbl_kind, 0.894775265003268_dbl_kind, 0.899446728555345_dbl_kind, 0.918620610742209_dbl_kind, 0.969750044458751_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1051) = (/ 0.891310655089072_dbl_kind, 0.894776097534863_dbl_kind, 0.899450356883521_dbl_kind, 0.918631343023572_dbl_kind, 0.969758632848421_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1052) = (/ 0.89131087878234_dbl_kind, 0.894776929711942_dbl_kind, 0.89945398384932_dbl_kind, 0.918642070808365_dbl_kind, 0.969767204414107_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1053) = (/ 0.891311102027925_dbl_kind, 0.894777761534664_dbl_kind, 0.899457609455057_dbl_kind, 0.91865279410396_dbl_kind, 0.969775759197058_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1054) = (/ 0.891311324827384_dbl_kind, 0.894778593003186_dbl_kind, 0.899461233703043_dbl_kind, 0.918663512917704_dbl_kind, 0.969784297238405_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1055) = (/ 0.891311547182268_dbl_kind, 0.894779424117659_dbl_kind, 0.899464856595591_dbl_kind, 0.918674227256913_dbl_kind, 0.969792818579143_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1056) = (/ 0.891311769094121_dbl_kind, 0.894780254878231_dbl_kind, 0.899468478135005_dbl_kind, 0.918684937128877_dbl_kind, 0.969801323260142_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1057) = (/ 0.891311990564484_dbl_kind, 0.894781085285046_dbl_kind, 0.899472098323591_dbl_kind, 0.918695642540857_dbl_kind, 0.96980981132215_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1058) = (/ 0.891312211594887_dbl_kind, 0.894781915338246_dbl_kind, 0.899475717163648_dbl_kind, 0.918706343500088_dbl_kind, 0.969818282805787_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1059) = (/ 0.891312432186858_dbl_kind, 0.894782745037968_dbl_kind, 0.899479334657475_dbl_kind, 0.918717040013775_dbl_kind, 0.969826737751549_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1060) = (/ 0.891312652341915_dbl_kind, 0.894783574384346_dbl_kind, 0.899482950807367_dbl_kind, 0.918727732089097_dbl_kind, 0.969835176199804_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1061) = (/ 0.891312872061573_dbl_kind, 0.89478440337751_dbl_kind, 0.899486565615616_dbl_kind, 0.918738419733206_dbl_kind, 0.969843598190802_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1062) = (/ 0.89131309134734_dbl_kind, 0.894785232017587_dbl_kind, 0.89949017908451_dbl_kind, 0.918749102953226_dbl_kind, 0.969852003764662_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1063) = (/ 0.891313310200717_dbl_kind, 0.894786060304701_dbl_kind, 0.899493791216338_dbl_kind, 0.918759781756256_dbl_kind, 0.969860392961387_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1064) = (/ 0.8913135286232_dbl_kind, 0.894786888238972_dbl_kind, 0.89949740201338_dbl_kind, 0.918770456149365_dbl_kind, 0.969868765820853_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1065) = (/ 0.891313746616278_dbl_kind, 0.894787715820518_dbl_kind, 0.899501011477919_dbl_kind, 0.918781126139597_dbl_kind, 0.969877122382816_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1066) = (/ 0.891313964181434_dbl_kind, 0.894788543049452_dbl_kind, 0.899504619612233_dbl_kind, 0.91879179173397_dbl_kind, 0.969885462686905_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1067) = (/ 0.891314181320146_dbl_kind, 0.894789369925885_dbl_kind, 0.899508226418597_dbl_kind, 0.918802452939474_dbl_kind, 0.969893786772635_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1068) = (/ 0.891314398033886_dbl_kind, 0.894790196449925_dbl_kind, 0.899511831899282_dbl_kind, 0.918813109763076_dbl_kind, 0.969902094679396_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1069) = (/ 0.891314614324119_dbl_kind, 0.894791022621676_dbl_kind, 0.89951543605656_dbl_kind, 0.918823762211712_dbl_kind, 0.96991038644646_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1070) = (/ 0.891314830192305_dbl_kind, 0.89479184844124_dbl_kind, 0.899519038892696_dbl_kind, 0.918834410292296_dbl_kind, 0.969918662112975_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1071) = (/ 0.891315045639896_dbl_kind, 0.894792673908716_dbl_kind, 0.899522640409955_dbl_kind, 0.918845054011713_dbl_kind, 0.969926921717975_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1072) = (/ 0.891315260668342_dbl_kind, 0.894793499024199_dbl_kind, 0.8995262406106_dbl_kind, 0.918855693376824_dbl_kind, 0.969935165300367_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1073) = (/ 0.891315475279083_dbl_kind, 0.894794323787782_dbl_kind, 0.899529839496888_dbl_kind, 0.918866328394466_dbl_kind, 0.96994339289895_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1074) = (/ 0.891315689473557_dbl_kind, 0.894795148199555_dbl_kind, 0.899533437071078_dbl_kind, 0.918876959071447_dbl_kind, 0.969951604552394_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1075) = (/ 0.891315903253193_dbl_kind, 0.894795972259606_dbl_kind, 0.899537033335422_dbl_kind, 0.918887585414551_dbl_kind, 0.969959800299262_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1076) = (/ 0.891316116619416_dbl_kind, 0.894796795968019_dbl_kind, 0.899540628292172_dbl_kind, 0.918898207430539_dbl_kind, 0.969967980177988_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1077) = (/ 0.891316329573645_dbl_kind, 0.894797619324875_dbl_kind, 0.899544221943576_dbl_kind, 0.918908825126142_dbl_kind, 0.969976144226898_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1078) = (/ 0.891316542117293_dbl_kind, 0.894798442330254_dbl_kind, 0.899547814291881_dbl_kind, 0.91891943850807_dbl_kind, 0.969984292484202_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1079) = (/ 0.891316754251767_dbl_kind, 0.894799264984233_dbl_kind, 0.899551405339332_dbl_kind, 0.918930047583008_dbl_kind, 0.969992424987985_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1080) = (/ 0.891316965978468_dbl_kind, 0.894800087286885_dbl_kind, 0.899554995088168_dbl_kind, 0.918940652357613_dbl_kind, 0.970000541776229_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1081) = (/ 0.891317177298794_dbl_kind, 0.894800909238281_dbl_kind, 0.89955858354063_dbl_kind, 0.918951252838521_dbl_kind, 0.970008642886788_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1082) = (/ 0.891317388214133_dbl_kind, 0.894801730838491_dbl_kind, 0.899562170698952_dbl_kind, 0.918961849032342_dbl_kind, 0.970016728357409_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1083) = (/ 0.891317598725872_dbl_kind, 0.894802552087581_dbl_kind, 0.89956575656537_dbl_kind, 0.918972440945661_dbl_kind, 0.970024798225726_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1084) = (/ 0.891317808835388_dbl_kind, 0.894803372985616_dbl_kind, 0.899569341142114_dbl_kind, 0.91898302858504_dbl_kind, 0.970032852529255_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1085) = (/ 0.891318018544056_dbl_kind, 0.894804193532656_dbl_kind, 0.899572924431413_dbl_kind, 0.918993611957014_dbl_kind, 0.9700408913054_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1086) = (/ 0.891318227853244_dbl_kind, 0.894805013728762_dbl_kind, 0.899576506435495_dbl_kind, 0.919004191068099_dbl_kind, 0.970048914591452_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1087) = (/ 0.891318436764314_dbl_kind, 0.89480583357399_dbl_kind, 0.899580087156583_dbl_kind, 0.919014765924783_dbl_kind, 0.970056922424589_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1088) = (/ 0.891318645278623_dbl_kind, 0.894806653068395_dbl_kind, 0.8995836665969_dbl_kind, 0.91902533653353_dbl_kind, 0.970064914841877_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1089) = (/ 0.891318853397523_dbl_kind, 0.89480747221203_dbl_kind, 0.899587244758664_dbl_kind, 0.919035902900783_dbl_kind, 0.970072891880273_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1090) = (/ 0.891319061122359_dbl_kind, 0.894808291004946_dbl_kind, 0.899590821644094_dbl_kind, 0.91904646503296_dbl_kind, 0.970080853576616_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1091) = (/ 0.891319268454473_dbl_kind, 0.894809109447189_dbl_kind, 0.899594397255404_dbl_kind, 0.919057022936456_dbl_kind, 0.970088799967643_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1092) = (/ 0.891319475395201_dbl_kind, 0.894809927538808_dbl_kind, 0.899597971594806_dbl_kind, 0.91906757661764_dbl_kind, 0.970096731089974_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1093) = (/ 0.891319681945871_dbl_kind, 0.894810745279846_dbl_kind, 0.899601544664511_dbl_kind, 0.919078126082862_dbl_kind, 0.970104646980119_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1094) = (/ 0.891319888107807_dbl_kind, 0.894811562670344_dbl_kind, 0.899605116466726_dbl_kind, 0.919088671338447_dbl_kind, 0.970112547674482_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1095) = (/ 0.891320093882331_dbl_kind, 0.894812379710344_dbl_kind, 0.899608687003658_dbl_kind, 0.919099212390695_dbl_kind, 0.970120433209357_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1096) = (/ 0.891320299270755_dbl_kind, 0.894813196399883_dbl_kind, 0.899612256277509_dbl_kind, 0.919109749245887_dbl_kind, 0.970128303620923_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1097) = (/ 0.891320504274388_dbl_kind, 0.894814012738999_dbl_kind, 0.899615824290481_dbl_kind, 0.919120281910277_dbl_kind, 0.970136158945261_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1098) = (/ 0.891320708894533_dbl_kind, 0.894814828727723_dbl_kind, 0.899619391044774_dbl_kind, 0.9191308103901_dbl_kind, 0.970143999218333_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1099) = (/ 0.891320913132488_dbl_kind, 0.894815644366091_dbl_kind, 0.899622956542582_dbl_kind, 0.919141334691565_dbl_kind, 0.970151824476005_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1100) = (/ 0.891321116989546_dbl_kind, 0.894816459654132_dbl_kind, 0.8996265207861_dbl_kind, 0.919151854820862_dbl_kind, 0.970159634754022_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1101) = (/ 0.891321320466996_dbl_kind, 0.894817274591875_dbl_kind, 0.899630083777522_dbl_kind, 0.919162370784156_dbl_kind, 0.970167430088035_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1102) = (/ 0.891321523566118_dbl_kind, 0.894818089179347_dbl_kind, 0.899633645519037_dbl_kind, 0.91917288258759_dbl_kind, 0.97017521051358_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1103) = (/ 0.89132172628819_dbl_kind, 0.894818903416574_dbl_kind, 0.899637206012832_dbl_kind, 0.919183390237286_dbl_kind, 0.970182976066092_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1104) = (/ 0.891321928634486_dbl_kind, 0.894819717303579_dbl_kind, 0.899640765261094_dbl_kind, 0.919193893739344_dbl_kind, 0.970190726780897_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1105) = (/ 0.891322130606271_dbl_kind, 0.894820530840385_dbl_kind, 0.899644323266005_dbl_kind, 0.91920439309984_dbl_kind, 0.970198462693213_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1106) = (/ 0.891322332204807_dbl_kind, 0.894821344027013_dbl_kind, 0.899647880029746_dbl_kind, 0.919214888324831_dbl_kind, 0.970206183838161_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1107) = (/ 0.891322533431352_dbl_kind, 0.89482215686348_dbl_kind, 0.899651435554498_dbl_kind, 0.919225379420349_dbl_kind, 0.970213890250754_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1108) = (/ 0.891322734287157_dbl_kind, 0.894822969349805_dbl_kind, 0.899654989842435_dbl_kind, 0.919235866392408_dbl_kind, 0.970221581965898_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1109) = (/ 0.89132293477347_dbl_kind, 0.894823781486003_dbl_kind, 0.899658542895734_dbl_kind, 0.919246349246996_dbl_kind, 0.970229259018399_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1110) = (/ 0.891323134891531_dbl_kind, 0.894824593272089_dbl_kind, 0.899662094716567_dbl_kind, 0.919256827990083_dbl_kind, 0.970236921442953_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1111) = (/ 0.891323334642577_dbl_kind, 0.894825404708075_dbl_kind, 0.899665645307103_dbl_kind, 0.919267302627618_dbl_kind, 0.970244569274162_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1112) = (/ 0.891323534027841_dbl_kind, 0.894826215793973_dbl_kind, 0.899669194669512_dbl_kind, 0.919277773165525_dbl_kind, 0.97025220254652_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1113) = (/ 0.891323733048549_dbl_kind, 0.894827026529793_dbl_kind, 0.89967274280596_dbl_kind, 0.919288239609711_dbl_kind, 0.970259821294419_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1114) = (/ 0.891323931705924_dbl_kind, 0.894827836915544_dbl_kind, 0.899676289718609_dbl_kind, 0.919298701966058_dbl_kind, 0.97026742555215_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1115) = (/ 0.891324130001181_dbl_kind, 0.894828646951232_dbl_kind, 0.899679835409623_dbl_kind, 0.919309160240431_dbl_kind, 0.970275015353903_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1116) = (/ 0.891324327935533_dbl_kind, 0.894829456636864_dbl_kind, 0.899683379881161_dbl_kind, 0.919319614438671_dbl_kind, 0.970282590733764_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1117) = (/ 0.891324525510189_dbl_kind, 0.894830265972445_dbl_kind, 0.899686923135379_dbl_kind, 0.9193300645666_dbl_kind, 0.970290151725722_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1118) = (/ 0.891324722726349_dbl_kind, 0.894831074957978_dbl_kind, 0.899690465174435_dbl_kind, 0.919340510630018_dbl_kind, 0.970297698363663_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1119) = (/ 0.891324919585211_dbl_kind, 0.894831883593465_dbl_kind, 0.899694006000481_dbl_kind, 0.919350952634704_dbl_kind, 0.970305230681373_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1120) = (/ 0.891325116087969_dbl_kind, 0.894832691878908_dbl_kind, 0.899697545615668_dbl_kind, 0.919361390586419_dbl_kind, 0.97031274871254_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1121) = (/ 0.89132531223581_dbl_kind, 0.894833499814306_dbl_kind, 0.899701084022145_dbl_kind, 0.919371824490902_dbl_kind, 0.97032025249075_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1122) = (/ 0.891325508029918_dbl_kind, 0.894834307399657_dbl_kind, 0.89970462122206_dbl_kind, 0.91938225435387_dbl_kind, 0.970327742049491_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1123) = (/ 0.89132570347147_dbl_kind, 0.89483511463496_dbl_kind, 0.899708157217557_dbl_kind, 0.919392680181023_dbl_kind, 0.970335217422156_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1124) = (/ 0.891325898561642_dbl_kind, 0.89483592152021_dbl_kind, 0.899711692010778_dbl_kind, 0.919403101978038_dbl_kind, 0.970342678642034_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1125) = (/ 0.891326093301601_dbl_kind, 0.894836728055404_dbl_kind, 0.899715225603866_dbl_kind, 0.919413519750573_dbl_kind, 0.970350125742316_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1126) = (/ 0.891326287692512_dbl_kind, 0.894837534240536_dbl_kind, 0.899718757998958_dbl_kind, 0.919423933504266_dbl_kind, 0.970357558756103_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1127) = (/ 0.891326481735536_dbl_kind, 0.894838340075598_dbl_kind, 0.89972228919819_dbl_kind, 0.919434343244736_dbl_kind, 0.970364977716392_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1128) = (/ 0.891326675431826_dbl_kind, 0.894839145560583_dbl_kind, 0.899725819203697_dbl_kind, 0.919444748977581_dbl_kind, 0.970372382656082_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1129) = (/ 0.891326868782534_dbl_kind, 0.894839950695483_dbl_kind, 0.899729348017612_dbl_kind, 0.919455150708379_dbl_kind, 0.970379773607978_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1130) = (/ 0.891327061788805_dbl_kind, 0.894840755480289_dbl_kind, 0.899732875642064_dbl_kind, 0.919465548442689_dbl_kind, 0.970387150604794_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1131) = (/ 0.89132725445178_dbl_kind, 0.894841559914988_dbl_kind, 0.89973640207918_dbl_kind, 0.91947594218605_dbl_kind, 0.970394513679139_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1132) = (/ 0.891327446772596_dbl_kind, 0.894842363999571_dbl_kind, 0.899739927331089_dbl_kind, 0.919486331943982_dbl_kind, 0.970401862863531_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1133) = (/ 0.891327638752386_dbl_kind, 0.894843167734025_dbl_kind, 0.899743451399912_dbl_kind, 0.919496717721986_dbl_kind, 0.970409198190392_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1134) = (/ 0.891327830392276_dbl_kind, 0.894843971118337_dbl_kind, 0.899746974287773_dbl_kind, 0.919507099525542_dbl_kind, 0.970416519692052_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1135) = (/ 0.89132802169339_dbl_kind, 0.894844774152492_dbl_kind, 0.899750495996789_dbl_kind, 0.919517477360111_dbl_kind, 0.970423827400742_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1136) = (/ 0.891328212656847_dbl_kind, 0.894845576836477_dbl_kind, 0.899754016529079_dbl_kind, 0.919527851231138_dbl_kind, 0.9704311213486_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1137) = (/ 0.89132840328376_dbl_kind, 0.894846379170275_dbl_kind, 0.89975753588676_dbl_kind, 0.919538221144044_dbl_kind, 0.970438401567673_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1138) = (/ 0.891328593575239_dbl_kind, 0.894847181153871_dbl_kind, 0.899761054071942_dbl_kind, 0.919548587104235_dbl_kind, 0.970445668089911_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1139) = (/ 0.891328783532389_dbl_kind, 0.894847982787247_dbl_kind, 0.899764571086738_dbl_kind, 0.919558949117095_dbl_kind, 0.970452920947174_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1140) = (/ 0.891328973156312_dbl_kind, 0.894848784070386_dbl_kind, 0.899768086933256_dbl_kind, 0.919569307187992_dbl_kind, 0.970460160171226_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1141) = (/ 0.891329162448103_dbl_kind, 0.894849585003269_dbl_kind, 0.899771601613604_dbl_kind, 0.919579661322274_dbl_kind, 0.970467385793743_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1142) = (/ 0.891329351408855_dbl_kind, 0.894850385585877_dbl_kind, 0.899775115129887_dbl_kind, 0.919590011525268_dbl_kind, 0.970474597846302_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1143) = (/ 0.891329540039655_dbl_kind, 0.894851185818192_dbl_kind, 0.899778627484206_dbl_kind, 0.919600357802286_dbl_kind, 0.970481796360395_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1144) = (/ 0.891329728341585_dbl_kind, 0.894851985700191_dbl_kind, 0.899782138678663_dbl_kind, 0.919610700158619_dbl_kind, 0.970488981367416_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1145) = (/ 0.891329916315727_dbl_kind, 0.894852785231854_dbl_kind, 0.899785648715355_dbl_kind, 0.919621038599542_dbl_kind, 0.970496152898676_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1146) = (/ 0.891330103963153_dbl_kind, 0.894853584413161_dbl_kind, 0.89978915759638_dbl_kind, 0.91963137313031_dbl_kind, 0.970503310985387_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1147) = (/ 0.891330291284936_dbl_kind, 0.894854383244089_dbl_kind, 0.89979266532383_dbl_kind, 0.919641703756157_dbl_kind, 0.970510455658677_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1148) = (/ 0.891330478282139_dbl_kind, 0.894855181724614_dbl_kind, 0.899796171899799_dbl_kind, 0.919652030482303_dbl_kind, 0.970517586949578_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1149) = (/ 0.891330664955826_dbl_kind, 0.894855979854715_dbl_kind, 0.899799677326375_dbl_kind, 0.919662353313949_dbl_kind, 0.970524704889036_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1150) = (/ 0.891330851307053_dbl_kind, 0.894856777634367_dbl_kind, 0.899803181605646_dbl_kind, 0.919672672256275_dbl_kind, 0.970531809507907_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1151) = (/ 0.891331037336875_dbl_kind, 0.894857575063547_dbl_kind, 0.899806684739698_dbl_kind, 0.919682987314447_dbl_kind, 0.970538900836955_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1152) = (/ 0.89133122304634_dbl_kind, 0.894858372142229_dbl_kind, 0.899810186730614_dbl_kind, 0.919693298493611_dbl_kind, 0.97054597890686_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1153) = (/ 0.891331408436494_dbl_kind, 0.894859168870389_dbl_kind, 0.899813687580475_dbl_kind, 0.919703605798893_dbl_kind, 0.970553043748209_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1154) = (/ 0.891331593508377_dbl_kind, 0.894859965248002_dbl_kind, 0.89981718729136_dbl_kind, 0.919713909235405_dbl_kind, 0.9705600953915_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1155) = (/ 0.891331778263026_dbl_kind, 0.894860761275041_dbl_kind, 0.899820685865345_dbl_kind, 0.919724208808239_dbl_kind, 0.970567133867149_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1156) = (/ 0.891331962701473_dbl_kind, 0.894861556951481_dbl_kind, 0.899824183304505_dbl_kind, 0.91973450452247_dbl_kind, 0.970574159205478_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1157) = (/ 0.891332146824747_dbl_kind, 0.894862352277295_dbl_kind, 0.899827679610913_dbl_kind, 0.919744796383154_dbl_kind, 0.970581171436723_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1158) = (/ 0.891332330633872_dbl_kind, 0.894863147252456_dbl_kind, 0.899831174786638_dbl_kind, 0.919755084395333_dbl_kind, 0.970588170591038_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1159) = (/ 0.891332514129869_dbl_kind, 0.894863941876936_dbl_kind, 0.899834668833748_dbl_kind, 0.919765368564026_dbl_kind, 0.970595156698483_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1160) = (/ 0.891332697313753_dbl_kind, 0.894864736150709_dbl_kind, 0.899838161754308_dbl_kind, 0.919775648894239_dbl_kind, 0.970602129789035_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1161) = (/ 0.891332880186537_dbl_kind, 0.894865530073745_dbl_kind, 0.899841653550383_dbl_kind, 0.91978592539096_dbl_kind, 0.970609089892584_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1162) = (/ 0.891333062749228_dbl_kind, 0.894866323646018_dbl_kind, 0.899845144224032_dbl_kind, 0.919796198059157_dbl_kind, 0.970616037038936_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1163) = (/ 0.891333245002831_dbl_kind, 0.8948671168675_dbl_kind, 0.899848633777316_dbl_kind, 0.919806466903785_dbl_kind, 0.970622971257811_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1164) = (/ 0.891333426948347_dbl_kind, 0.89486790973816_dbl_kind, 0.89985212221229_dbl_kind, 0.919816731929778_dbl_kind, 0.970629892578838_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1165) = (/ 0.89133360858677_dbl_kind, 0.89486870225797_dbl_kind, 0.89985560953101_dbl_kind, 0.919826993142055_dbl_kind, 0.970636801031573_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1166) = (/ 0.891333789919093_dbl_kind, 0.894869494426902_dbl_kind, 0.899859095735526_dbl_kind, 0.919837250545516_dbl_kind, 0.970643696645475_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1167) = (/ 0.891333970946304_dbl_kind, 0.894870286244927_dbl_kind, 0.899862580827889_dbl_kind, 0.919847504145046_dbl_kind, 0.970650579449925_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1168) = (/ 0.891334151669388_dbl_kind, 0.894871077712013_dbl_kind, 0.899866064810146_dbl_kind, 0.919857753945513_dbl_kind, 0.970657449474219_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1169) = (/ 0.891334332089325_dbl_kind, 0.894871868828133_dbl_kind, 0.899869547684342_dbl_kind, 0.919867999951767_dbl_kind, 0.970664306747569_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1170) = (/ 0.891334512207091_dbl_kind, 0.894872659593256_dbl_kind, 0.899873029452521_dbl_kind, 0.919878242168642_dbl_kind, 0.970671151299104_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1171) = (/ 0.891334692023659_dbl_kind, 0.894873450007353_dbl_kind, 0.899876510116722_dbl_kind, 0.919888480600954_dbl_kind, 0.970677983157867_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1172) = (/ 0.891334871539997_dbl_kind, 0.894874240070393_dbl_kind, 0.899879989678984_dbl_kind, 0.919898715253504_dbl_kind, 0.970684802352823_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1173) = (/ 0.891335050757071_dbl_kind, 0.894875029782346_dbl_kind, 0.899883468141343_dbl_kind, 0.919908946131076_dbl_kind, 0.970691608912852_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1174) = (/ 0.891335229675841_dbl_kind, 0.894875819143184_dbl_kind, 0.899886945505832_dbl_kind, 0.919919173238436_dbl_kind, 0.970698402866747_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1175) = (/ 0.891335408297264_dbl_kind, 0.894876608152873_dbl_kind, 0.899890421774482_dbl_kind, 0.919929396580335_dbl_kind, 0.970705184243228_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1176) = (/ 0.891335586622295_dbl_kind, 0.894877396811387_dbl_kind, 0.899893896949323_dbl_kind, 0.919939616161507_dbl_kind, 0.970711953070926_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1177) = (/ 0.891335764651881_dbl_kind, 0.894878185118693_dbl_kind, 0.899897371032379_dbl_kind, 0.919949831986669_dbl_kind, 0.970718709378394_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1178) = (/ 0.89133594238697_dbl_kind, 0.89487897307476_dbl_kind, 0.899900844025675_dbl_kind, 0.919960044060524_dbl_kind, 0.970725453194104_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1179) = (/ 0.891336119828503_dbl_kind, 0.894879760679561_dbl_kind, 0.899904315931234_dbl_kind, 0.919970252387756_dbl_kind, 0.970732184546444_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1180) = (/ 0.891336296977419_dbl_kind, 0.894880547933062_dbl_kind, 0.899907786751073_dbl_kind, 0.919980456973033_dbl_kind, 0.970738903463724_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1181) = (/ 0.891336473834652_dbl_kind, 0.894881334835235_dbl_kind, 0.89991125648721_dbl_kind, 0.919990657821009_dbl_kind, 0.970745609974172_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1182) = (/ 0.891336650401133_dbl_kind, 0.89488212138605_dbl_kind, 0.899914725141659_dbl_kind, 0.920000854936319_dbl_kind, 0.97075230410594_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1183) = (/ 0.891336826677789_dbl_kind, 0.894882907585475_dbl_kind, 0.899918192716431_dbl_kind, 0.920011048323584_dbl_kind, 0.970758985887093_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1184) = (/ 0.891337002665544_dbl_kind, 0.894883693433481_dbl_kind, 0.899921659213536_dbl_kind, 0.92002123798741_dbl_kind, 0.970765655345623_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1185) = (/ 0.891337178365317_dbl_kind, 0.894884478930037_dbl_kind, 0.899925124634981_dbl_kind, 0.920031423932383_dbl_kind, 0.97077231250944_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1186) = (/ 0.891337353778025_dbl_kind, 0.894885264075114_dbl_kind, 0.899928588982769_dbl_kind, 0.920041606163077_dbl_kind, 0.970778957406372_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1187) = (/ 0.89133752890458_dbl_kind, 0.894886048868682_dbl_kind, 0.899932052258904_dbl_kind, 0.920051784684048_dbl_kind, 0.970785590064178_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1188) = (/ 0.891337703745891_dbl_kind, 0.894886833310712_dbl_kind, 0.899935514465384_dbl_kind, 0.920061959499838_dbl_kind, 0.970792210510529_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1189) = (/ 0.891337878302864_dbl_kind, 0.894887617401172_dbl_kind, 0.899938975604207_dbl_kind, 0.92007213061497_dbl_kind, 0.970798818773018_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1190) = (/ 0.8913380525764_dbl_kind, 0.894888401140035_dbl_kind, 0.899942435677365_dbl_kind, 0.920082298033955_dbl_kind, 0.970805414879168_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1191) = (/ 0.891338226567397_dbl_kind, 0.89488918452727_dbl_kind, 0.899945894686851_dbl_kind, 0.920092461761287_dbl_kind, 0.970811998856415_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1192) = (/ 0.89133840027675_dbl_kind, 0.894889967562848_dbl_kind, 0.899949352634655_dbl_kind, 0.920102621801444_dbl_kind, 0.970818570732125_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1193) = (/ 0.89133857370535_dbl_kind, 0.894890750246741_dbl_kind, 0.899952809522762_dbl_kind, 0.920112778158887_dbl_kind, 0.970825130533583_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1194) = (/ 0.891338746854085_dbl_kind, 0.894891532578919_dbl_kind, 0.899956265353157_dbl_kind, 0.920122930838066_dbl_kind, 0.970831678287999_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1195) = (/ 0.891338919723838_dbl_kind, 0.894892314559354_dbl_kind, 0.899959720127821_dbl_kind, 0.92013307984341_dbl_kind, 0.970838214022505_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1196) = (/ 0.89133909231549_dbl_kind, 0.894893096188017_dbl_kind, 0.899963173848733_dbl_kind, 0.920143225179337_dbl_kind, 0.970844737764155_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1197) = (/ 0.891339264629918_dbl_kind, 0.894893877464881_dbl_kind, 0.899966626517868_dbl_kind, 0.920153366850247_dbl_kind, 0.970851249539934_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1198) = (/ 0.891339436667996_dbl_kind, 0.894894658389917_dbl_kind, 0.899970078137201_dbl_kind, 0.920163504860526_dbl_kind, 0.970857749376741_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1199) = (/ 0.891339608430593_dbl_kind, 0.894895438963098_dbl_kind, 0.899973528708703_dbl_kind, 0.920173639214544_dbl_kind, 0.970864237301409_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1200) = (/ 0.891339779918577_dbl_kind, 0.894896219184396_dbl_kind, 0.89997697823434_dbl_kind, 0.920183769916657_dbl_kind, 0.970870713340692_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1201) = (/ 0.891339951132809_dbl_kind, 0.894896999053784_dbl_kind, 0.89998042671608_dbl_kind, 0.920193896971204_dbl_kind, 0.970877177521265_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1202) = (/ 0.891340122074152_dbl_kind, 0.894897778571235_dbl_kind, 0.899983874155884_dbl_kind, 0.920204020382511_dbl_kind, 0.970883629869733_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1203) = (/ 0.891340292743459_dbl_kind, 0.894898557736722_dbl_kind, 0.899987320555712_dbl_kind, 0.920214140154887_dbl_kind, 0.970890070412626_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1204) = (/ 0.891340463141585_dbl_kind, 0.894899336550219_dbl_kind, 0.899990765917522_dbl_kind, 0.920224256292627_dbl_kind, 0.9708964991764_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1205) = (/ 0.89134063326938_dbl_kind, 0.894900115011701_dbl_kind, 0.899994210243269_dbl_kind, 0.920234368800011_dbl_kind, 0.970902916187435_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1206) = (/ 0.891340803127688_dbl_kind, 0.894900893121141_dbl_kind, 0.899997653534904_dbl_kind, 0.920244477681303_dbl_kind, 0.97090932147204_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1207) = (/ 0.891340972717354_dbl_kind, 0.894901670878514_dbl_kind, 0.900001095794376_dbl_kind, 0.920254582940754_dbl_kind, 0.970915715056445_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1208) = (/ 0.891341142039217_dbl_kind, 0.894902448283795_dbl_kind, 0.900004537023631_dbl_kind, 0.920264684582599_dbl_kind, 0.970922096966813_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1209) = (/ 0.891341311094113_dbl_kind, 0.894903225336959_dbl_kind, 0.900007977224614_dbl_kind, 0.920274782611058_dbl_kind, 0.970928467229232_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1210) = (/ 0.891341479882875_dbl_kind, 0.894904002037982_dbl_kind, 0.900011416399264_dbl_kind, 0.920284877030338_dbl_kind, 0.970934825869718_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1211) = (/ 0.891341648406332_dbl_kind, 0.894904778386841_dbl_kind, 0.90001485454952_dbl_kind, 0.920294967844627_dbl_kind, 0.970941172914209_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1212) = (/ 0.891341816665312_dbl_kind, 0.894905554383511_dbl_kind, 0.900018291677315_dbl_kind, 0.920305055058103_dbl_kind, 0.97094750838858_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1213) = (/ 0.891341984660637_dbl_kind, 0.894906330027969_dbl_kind, 0.900021727784584_dbl_kind, 0.920315138674928_dbl_kind, 0.970953832318627_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1214) = (/ 0.891342152393127_dbl_kind, 0.894907105320193_dbl_kind, 0.900025162873253_dbl_kind, 0.920325218699249_dbl_kind, 0.970960144730075_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1215) = (/ 0.891342319863598_dbl_kind, 0.894907880260161_dbl_kind, 0.90002859694525_dbl_kind, 0.920335295135196_dbl_kind, 0.97096644564858_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1216) = (/ 0.891342487072864_dbl_kind, 0.894908654847849_dbl_kind, 0.900032030002499_dbl_kind, 0.92034536798689_dbl_kind, 0.970972735099726_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1217) = (/ 0.891342654021736_dbl_kind, 0.894909429083238_dbl_kind, 0.900035462046919_dbl_kind, 0.920355437258433_dbl_kind, 0.970979013109024_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1218) = (/ 0.891342820711019_dbl_kind, 0.894910202966305_dbl_kind, 0.900038893080429_dbl_kind, 0.920365502953915_dbl_kind, 0.970985279701918_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1219) = (/ 0.891342987141518_dbl_kind, 0.89491097649703_dbl_kind, 0.900042323104941_dbl_kind, 0.920375565077408_dbl_kind, 0.970991534903778_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1220) = (/ 0.891343153314034_dbl_kind, 0.894911749675393_dbl_kind, 0.90004575212237_dbl_kind, 0.920385623632976_dbl_kind, 0.970997778739902_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1221) = (/ 0.891343319229363_dbl_kind, 0.894912522501374_dbl_kind, 0.900049180134623_dbl_kind, 0.920395678624662_dbl_kind, 0.971004011235523_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1222) = (/ 0.891343484888301_dbl_kind, 0.894913294974954_dbl_kind, 0.900052607143605_dbl_kind, 0.9204057300565_dbl_kind, 0.971010232415805_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1223) = (/ 0.891343650291638_dbl_kind, 0.894914067096114_dbl_kind, 0.900056033151219_dbl_kind, 0.920415777932505_dbl_kind, 0.971016442305834_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1224) = (/ 0.891343815440162_dbl_kind, 0.894914838864836_dbl_kind, 0.900059458159365_dbl_kind, 0.920425822256683_dbl_kind, 0.971022640930633_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1225) = (/ 0.891343980334659_dbl_kind, 0.894915610281101_dbl_kind, 0.90006288216994_dbl_kind, 0.920435863033021_dbl_kind, 0.971028828315157_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1226) = (/ 0.891344144975909_dbl_kind, 0.894916381344893_dbl_kind, 0.900066305184836_dbl_kind, 0.920445900265495_dbl_kind, 0.971035004484288_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1227) = (/ 0.891344309364692_dbl_kind, 0.894917152056195_dbl_kind, 0.900069727205944_dbl_kind, 0.920455933958066_dbl_kind, 0.971041169462839_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1228) = (/ 0.891344473501784_dbl_kind, 0.89491792241499_dbl_kind, 0.900073148235151_dbl_kind, 0.920465964114681_dbl_kind, 0.971047323275561_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1229) = (/ 0.891344637387957_dbl_kind, 0.894918692421264_dbl_kind, 0.900076568274342_dbl_kind, 0.920475990739272_dbl_kind, 0.97105346594713_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1230) = (/ 0.89134480102398_dbl_kind, 0.894919462074999_dbl_kind, 0.900079987325398_dbl_kind, 0.92048601383576_dbl_kind, 0.971059597502154_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1231) = (/ 0.89134496441062_dbl_kind, 0.894920231376183_dbl_kind, 0.900083405390195_dbl_kind, 0.920496033408048_dbl_kind, 0.97106571796518_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1232) = (/ 0.891345127548641_dbl_kind, 0.894921000324799_dbl_kind, 0.90008682247061_dbl_kind, 0.920506049460028_dbl_kind, 0.97107182736068_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1233) = (/ 0.891345290438803_dbl_kind, 0.894921768920836_dbl_kind, 0.900090238568513_dbl_kind, 0.920516061995577_dbl_kind, 0.971077925713063_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1234) = (/ 0.891345453081863_dbl_kind, 0.89492253716428_dbl_kind, 0.900093653685773_dbl_kind, 0.920526071018558_dbl_kind, 0.971084013046669_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1235) = (/ 0.891345615478576_dbl_kind, 0.894923305055118_dbl_kind, 0.900097067824255_dbl_kind, 0.920536076532822_dbl_kind, 0.971090089385772_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1236) = (/ 0.891345777629693_dbl_kind, 0.894924072593339_dbl_kind, 0.900100480985821_dbl_kind, 0.920546078542204_dbl_kind, 0.971096154754577_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1237) = (/ 0.891345939535965_dbl_kind, 0.89492483977893_dbl_kind, 0.900103893172329_dbl_kind, 0.920556077050527_dbl_kind, 0.97110220917723_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1238) = (/ 0.891346101198134_dbl_kind, 0.894925606611883_dbl_kind, 0.900107304385635_dbl_kind, 0.920566072061598_dbl_kind, 0.971108252677798_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1239) = (/ 0.891346262616944_dbl_kind, 0.894926373092185_dbl_kind, 0.900110714627591_dbl_kind, 0.920576063579213_dbl_kind, 0.971114285280296_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1240) = (/ 0.891346423793136_dbl_kind, 0.894927139219828_dbl_kind, 0.900114123900046_dbl_kind, 0.920586051607153_dbl_kind, 0.971120307008661_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1241) = (/ 0.891346584727445_dbl_kind, 0.894927904994803_dbl_kind, 0.900117532204844_dbl_kind, 0.920596036149185_dbl_kind, 0.971126317886776_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1242) = (/ 0.891346745420607_dbl_kind, 0.894928670417101_dbl_kind, 0.900120939543829_dbl_kind, 0.920606017209065_dbl_kind, 0.971132317938449_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1243) = (/ 0.891346905873351_dbl_kind, 0.894929435486714_dbl_kind, 0.900124345918838_dbl_kind, 0.920615994790531_dbl_kind, 0.971138307187425_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1244) = (/ 0.891347066086405_dbl_kind, 0.894930200203637_dbl_kind, 0.900127751331708_dbl_kind, 0.920625968897312_dbl_kind, 0.97114428565739_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1245) = (/ 0.891347226060496_dbl_kind, 0.894930964567861_dbl_kind, 0.900131155784271_dbl_kind, 0.920635939533121_dbl_kind, 0.971150253371959_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1246) = (/ 0.891347385796346_dbl_kind, 0.894931728579381_dbl_kind, 0.900134559278354_dbl_kind, 0.920645906701658_dbl_kind, 0.971156210354684_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1247) = (/ 0.891347545294673_dbl_kind, 0.894932492238193_dbl_kind, 0.900137961815784_dbl_kind, 0.92065587040661_dbl_kind, 0.971162156629055_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1248) = (/ 0.891347704556196_dbl_kind, 0.89493325554429_dbl_kind, 0.900141363398381_dbl_kind, 0.92066583065165_dbl_kind, 0.971168092218495_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1249) = (/ 0.891347863581627_dbl_kind, 0.89493401849767_dbl_kind, 0.900144764027965_dbl_kind, 0.920675787440439_dbl_kind, 0.971174017146364_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1250) = (/ 0.891348022371678_dbl_kind, 0.89493478109833_dbl_kind, 0.90014816370635_dbl_kind, 0.920685740776624_dbl_kind, 0.971179931435959_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1251) = (/ 0.891348180927057_dbl_kind, 0.894935543346266_dbl_kind, 0.900151562435348_dbl_kind, 0.920695690663838_dbl_kind, 0.971185835110515_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1252) = (/ 0.89134833924847_dbl_kind, 0.894936305241476_dbl_kind, 0.900154960216766_dbl_kind, 0.9207056371057_dbl_kind, 0.9711917281932_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1253) = (/ 0.891348497336619_dbl_kind, 0.89493706678396_dbl_kind, 0.900158357052408_dbl_kind, 0.920715580105819_dbl_kind, 0.97119761070712_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1254) = (/ 0.891348655192204_dbl_kind, 0.894937827973717_dbl_kind, 0.900161752944077_dbl_kind, 0.920725519667789_dbl_kind, 0.971203482675326_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1255) = (/ 0.891348812815923_dbl_kind, 0.894938588810747_dbl_kind, 0.900165147893568_dbl_kind, 0.920735455795189_dbl_kind, 0.971209344120794_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1256) = (/ 0.89134897020847_dbl_kind, 0.894939349295051_dbl_kind, 0.900168541902675_dbl_kind, 0.920745388491589_dbl_kind, 0.971215195066444_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1257) = (/ 0.891349127370537_dbl_kind, 0.894940109426629_dbl_kind, 0.90017193497319_dbl_kind, 0.920755317760542_dbl_kind, 0.971221035535134_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1258) = (/ 0.891349284302812_dbl_kind, 0.894940869205485_dbl_kind, 0.900175327106897_dbl_kind, 0.920765243605589_dbl_kind, 0.971226865549659_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1259) = (/ 0.891349441005982_dbl_kind, 0.894941628631621_dbl_kind, 0.900178718305579_dbl_kind, 0.92077516603026_dbl_kind, 0.971232685132755_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1260) = (/ 0.891349597480731_dbl_kind, 0.894942387705041_dbl_kind, 0.900182108571017_dbl_kind, 0.92078508503807_dbl_kind, 0.971238494307089_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1261) = (/ 0.89134975372774_dbl_kind, 0.894943146425749_dbl_kind, 0.900185497904986_dbl_kind, 0.920795000632521_dbl_kind, 0.971244293095275_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1262) = (/ 0.891349909747685_dbl_kind, 0.89494390479375_dbl_kind, 0.900188886309257_dbl_kind, 0.920804912817103_dbl_kind, 0.971250081519864_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1263) = (/ 0.891350065541245_dbl_kind, 0.894944662809049_dbl_kind, 0.900192273785599_dbl_kind, 0.920814821595292_dbl_kind, 0.97125585960334_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1264) = (/ 0.89135022110909_dbl_kind, 0.894945420471654_dbl_kind, 0.900195660335776_dbl_kind, 0.920824726970553_dbl_kind, 0.971261627368133_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1265) = (/ 0.891350376451892_dbl_kind, 0.894946177781571_dbl_kind, 0.900199045961549_dbl_kind, 0.920834628946334_dbl_kind, 0.971267384836611_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1266) = (/ 0.891350531570317_dbl_kind, 0.894946934738809_dbl_kind, 0.900202430664675_dbl_kind, 0.920844527526077_dbl_kind, 0.97127313203108_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1267) = (/ 0.891350686465032_dbl_kind, 0.894947691343375_dbl_kind, 0.900205814446907_dbl_kind, 0.920854422713204_dbl_kind, 0.971278868973787_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1268) = (/ 0.891350841136698_dbl_kind, 0.89494844759528_dbl_kind, 0.900209197309994_dbl_kind, 0.920864314511129_dbl_kind, 0.971284595686918_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1269) = (/ 0.891350995585975_dbl_kind, 0.894949203494533_dbl_kind, 0.900212579255683_dbl_kind, 0.920874202923251_dbl_kind, 0.971290312192601_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1270) = (/ 0.89135114981352_dbl_kind, 0.894949959041145_dbl_kind, 0.900215960285715_dbl_kind, 0.920884087952957_dbl_kind, 0.9712960185129_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1271) = (/ 0.891351303819988_dbl_kind, 0.894950714235128_dbl_kind, 0.900219340401828_dbl_kind, 0.920893969603621_dbl_kind, 0.971301714669828_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1272) = (/ 0.89135145760603_dbl_kind, 0.894951469076494_dbl_kind, 0.900222719605757_dbl_kind, 0.920903847878605_dbl_kind, 0.971307400685329_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1273) = (/ 0.891351611172298_dbl_kind, 0.894952223565257_dbl_kind, 0.900226097899231_dbl_kind, 0.920913722781257_dbl_kind, 0.971313076581296_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1274) = (/ 0.891351764519436_dbl_kind, 0.894952977701429_dbl_kind, 0.900229475283977_dbl_kind, 0.920923594314914_dbl_kind, 0.971318742379558_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1275) = (/ 0.891351917648089_dbl_kind, 0.894953731485026_dbl_kind, 0.900232851761718_dbl_kind, 0.9209334624829_dbl_kind, 0.971324398101889_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1276) = (/ 0.8913520705589_dbl_kind, 0.894954484916063_dbl_kind, 0.900236227334172_dbl_kind, 0.920943327288524_dbl_kind, 0.971330043770001_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1277) = (/ 0.891352223252508_dbl_kind, 0.894955237994557_dbl_kind, 0.900239602003053_dbl_kind, 0.920953188735087_dbl_kind, 0.971335679405553_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1278) = (/ 0.891352375729548_dbl_kind, 0.894955990720524_dbl_kind, 0.900242975770073_dbl_kind, 0.920963046825873_dbl_kind, 0.971341305030137_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1279) = (/ 0.891352527990657_dbl_kind, 0.894956743093982_dbl_kind, 0.900246348636939_dbl_kind, 0.920972901564156_dbl_kind, 0.971346920665297_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1280) = (/ 0.891352680036464_dbl_kind, 0.894957495114949_dbl_kind, 0.900249720605352_dbl_kind, 0.920982752953196_dbl_kind, 0.971352526332514_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1281) = (/ 0.891352831867601_dbl_kind, 0.894958246783445_dbl_kind, 0.900253091677012_dbl_kind, 0.920992600996244_dbl_kind, 0.971358122053212_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1282) = (/ 0.891352983484692_dbl_kind, 0.89495899809949_dbl_kind, 0.900256461853614_dbl_kind, 0.921002445696532_dbl_kind, 0.971363707848758_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1283) = (/ 0.891353134888364_dbl_kind, 0.894959749063104_dbl_kind, 0.900259831136847_dbl_kind, 0.921012287057287_dbl_kind, 0.971369283740466_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1284) = (/ 0.891353286079237_dbl_kind, 0.89496049967431_dbl_kind, 0.9002631995284_dbl_kind, 0.92102212508172_dbl_kind, 0.971374849749585_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1285) = (/ 0.891353437057932_dbl_kind, 0.89496124993313_dbl_kind, 0.900266567029954_dbl_kind, 0.921031959773027_dbl_kind, 0.971380405897312_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1286) = (/ 0.891353587825064_dbl_kind, 0.894961999839585_dbl_kind, 0.900269933643188_dbl_kind, 0.921041791134396_dbl_kind, 0.971385952204789_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1287) = (/ 0.89135373838125_dbl_kind, 0.894962749393702_dbl_kind, 0.900273299369776_dbl_kind, 0.921051619169003_dbl_kind, 0.971391488693096_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1288) = (/ 0.8913538887271_dbl_kind, 0.894963498595505_dbl_kind, 0.900276664211388_dbl_kind, 0.921061443880006_dbl_kind, 0.971397015383263_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1289) = (/ 0.891354038863225_dbl_kind, 0.894964247445019_dbl_kind, 0.900280028169692_dbl_kind, 0.921071265270557_dbl_kind, 0.971402532296261_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1290) = (/ 0.891354188790232_dbl_kind, 0.894964995942271_dbl_kind, 0.900283391246347_dbl_kind, 0.921081083343793_dbl_kind, 0.971408039453005_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1291) = (/ 0.891354338508726_dbl_kind, 0.894965744087288_dbl_kind, 0.900286753443012_dbl_kind, 0.921090898102839_dbl_kind, 0.971413536874356_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1292) = (/ 0.89135448801931_dbl_kind, 0.894966491880098_dbl_kind, 0.900290114761343_dbl_kind, 0.921100709550808_dbl_kind, 0.971419024581117_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1293) = (/ 0.891354637322583_dbl_kind, 0.894967239320729_dbl_kind, 0.900293475202985_dbl_kind, 0.921110517690799_dbl_kind, 0.971424502594038_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1294) = (/ 0.891354786419145_dbl_kind, 0.894967986409212_dbl_kind, 0.900296834769587_dbl_kind, 0.921120322525901_dbl_kind, 0.971429970933811_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1295) = (/ 0.89135493530959_dbl_kind, 0.894968733145577_dbl_kind, 0.900300193462788_dbl_kind, 0.921130124059191_dbl_kind, 0.971435429621079_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1296) = (/ 0.891355083994511_dbl_kind, 0.894969479529856_dbl_kind, 0.900303551284224_dbl_kind, 0.921139922293733_dbl_kind, 0.971440878676421_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1297) = (/ 0.891355232474499_dbl_kind, 0.894970225562079_dbl_kind, 0.900306908235529_dbl_kind, 0.921149717232578_dbl_kind, 0.971446318120371_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1298) = (/ 0.891355380750144_dbl_kind, 0.894970971242281_dbl_kind, 0.90031026431833_dbl_kind, 0.921159508878768_dbl_kind, 0.971451747973401_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1299) = (/ 0.89135552882203_dbl_kind, 0.894971716570495_dbl_kind, 0.900313619534252_dbl_kind, 0.921169297235328_dbl_kind, 0.971457168255933_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1300) = (/ 0.891355676690743_dbl_kind, 0.894972461546755_dbl_kind, 0.900316973884913_dbl_kind, 0.921179082305276_dbl_kind, 0.971462578988336_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1301) = (/ 0.891355824356863_dbl_kind, 0.894973206171097_dbl_kind, 0.90032032737193_dbl_kind, 0.921188864091614_dbl_kind, 0.97146798019092_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1302) = (/ 0.891355971820971_dbl_kind, 0.894973950443558_dbl_kind, 0.900323679996911_dbl_kind, 0.921198642597335_dbl_kind, 0.971473371883946_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1303) = (/ 0.891356119083642_dbl_kind, 0.894974694364174_dbl_kind, 0.900327031761465_dbl_kind, 0.921208417825418_dbl_kind, 0.971478754087618_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1304) = (/ 0.891356266145452_dbl_kind, 0.894975437932983_dbl_kind, 0.900330382667192_dbl_kind, 0.921218189778832_dbl_kind, 0.971484126822091_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1305) = (/ 0.891356413006974_dbl_kind, 0.894976181150023_dbl_kind, 0.900333732715691_dbl_kind, 0.921227958460531_dbl_kind, 0.971489490107461_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1306) = (/ 0.891356559668777_dbl_kind, 0.894976924015335_dbl_kind, 0.900337081908555_dbl_kind, 0.921237723873459_dbl_kind, 0.971494843963777_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1307) = (/ 0.89135670613143_dbl_kind, 0.894977666528958_dbl_kind, 0.900340430247371_dbl_kind, 0.921247486020549_dbl_kind, 0.971500188411031_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1308) = (/ 0.891356852395499_dbl_kind, 0.894978408690935_dbl_kind, 0.900343777733726_dbl_kind, 0.92125724490472_dbl_kind, 0.97150552346916_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1309) = (/ 0.891356998461547_dbl_kind, 0.894979150501306_dbl_kind, 0.900347124369198_dbl_kind, 0.921267000528882_dbl_kind, 0.971510849158058_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1310) = (/ 0.891357144330135_dbl_kind, 0.894979891960115_dbl_kind, 0.900350470155362_dbl_kind, 0.921276752895929_dbl_kind, 0.971516165497559_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1311) = (/ 0.891357290001823_dbl_kind, 0.894980633067405_dbl_kind, 0.900353815093791_dbl_kind, 0.921286502008746_dbl_kind, 0.971521472507443_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1312) = (/ 0.891357435477168_dbl_kind, 0.894981373823221_dbl_kind, 0.900357159186049_dbl_kind, 0.921296247870206_dbl_kind, 0.971526770207445_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1313) = (/ 0.891357580756724_dbl_kind, 0.894982114227609_dbl_kind, 0.900360502433698_dbl_kind, 0.92130599048317_dbl_kind, 0.97153205861724_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1314) = (/ 0.891357725841045_dbl_kind, 0.894982854280615_dbl_kind, 0.900363844838296_dbl_kind, 0.921315729850487_dbl_kind, 0.97153733775646_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1315) = (/ 0.89135787073068_dbl_kind, 0.894983593982284_dbl_kind, 0.900367186401396_dbl_kind, 0.921325465974994_dbl_kind, 0.971542607644678_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1316) = (/ 0.891358015426177_dbl_kind, 0.894984333332667_dbl_kind, 0.900370527124546_dbl_kind, 0.921335198859516_dbl_kind, 0.971547868301423_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1317) = (/ 0.891358159928084_dbl_kind, 0.894985072331811_dbl_kind, 0.900373867009288_dbl_kind, 0.921344928506867_dbl_kind, 0.971553119746164_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1318) = (/ 0.891358304236944_dbl_kind, 0.894985810979765_dbl_kind, 0.900377206057162_dbl_kind, 0.92135465491985_dbl_kind, 0.971558361998327_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1319) = (/ 0.891358448353298_dbl_kind, 0.894986549276581_dbl_kind, 0.900380544269702_dbl_kind, 0.921364378101254_dbl_kind, 0.971563595077282_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1320) = (/ 0.891358592277686_dbl_kind, 0.894987287222309_dbl_kind, 0.900383881648438_dbl_kind, 0.921374098053859_dbl_kind, 0.97156881900235_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1321) = (/ 0.891358736010647_dbl_kind, 0.894988024817002_dbl_kind, 0.900387218194894_dbl_kind, 0.921383814780431_dbl_kind, 0.971574033792799_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1322) = (/ 0.891358879552715_dbl_kind, 0.894988762060713_dbl_kind, 0.90039055391059_dbl_kind, 0.921393528283725_dbl_kind, 0.971579239467854_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1323) = (/ 0.891359022904425_dbl_kind, 0.894989498953494_dbl_kind, 0.900393888797043_dbl_kind, 0.921403238566485_dbl_kind, 0.97158443604668_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1324) = (/ 0.891359166066306_dbl_kind, 0.894990235495403_dbl_kind, 0.900397222855763_dbl_kind, 0.921412945631444_dbl_kind, 0.971589623548399_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1325) = (/ 0.891359309038888_dbl_kind, 0.894990971686492_dbl_kind, 0.900400556088256_dbl_kind, 0.921422649481322_dbl_kind, 0.97159480199208_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1326) = (/ 0.8913594518227_dbl_kind, 0.894991707526819_dbl_kind, 0.900403888496023_dbl_kind, 0.921432350118827_dbl_kind, 0.971599971396743_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1327) = (/ 0.891359594418264_dbl_kind, 0.894992443016442_dbl_kind, 0.900407220080562_dbl_kind, 0.921442047546657_dbl_kind, 0.971605131781357_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1328) = (/ 0.891359736826106_dbl_kind, 0.894993178155417_dbl_kind, 0.900410550843363_dbl_kind, 0.921451741767498_dbl_kind, 0.971610283164844_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1329) = (/ 0.891359879046744_dbl_kind, 0.894993912943804_dbl_kind, 0.900413880785915_dbl_kind, 0.921461432784025_dbl_kind, 0.971615425566076_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1330) = (/ 0.891360021080699_dbl_kind, 0.894994647381663_dbl_kind, 0.900417209909699_dbl_kind, 0.921471120598899_dbl_kind, 0.971620559003872_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1331) = (/ 0.891360162928487_dbl_kind, 0.894995381469053_dbl_kind, 0.900420538216193_dbl_kind, 0.921480805214772_dbl_kind, 0.971625683497007_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1332) = (/ 0.891360304590623_dbl_kind, 0.894996115206038_dbl_kind, 0.900423865706869_dbl_kind, 0.921490486634284_dbl_kind, 0.971630799064208_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1333) = (/ 0.89136044606762_dbl_kind, 0.894996848592678_dbl_kind, 0.900427192383195_dbl_kind, 0.921500164860063_dbl_kind, 0.971635905724147_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1334) = (/ 0.891360587359987_dbl_kind, 0.894997581629037_dbl_kind, 0.900430518246635_dbl_kind, 0.921509839894727_dbl_kind, 0.971641003495451_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1335) = (/ 0.891360728468236_dbl_kind, 0.894998314315179_dbl_kind, 0.900433843298646_dbl_kind, 0.921519511740879_dbl_kind, 0.971646092396703_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1336) = (/ 0.891360869392872_dbl_kind, 0.894999046651168_dbl_kind, 0.900437167540681_dbl_kind, 0.921529180401115_dbl_kind, 0.97165117244643_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1337) = (/ 0.891361010134398_dbl_kind, 0.894999778637071_dbl_kind, 0.900440490974189_dbl_kind, 0.921538845878017_dbl_kind, 0.971656243663115_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1338) = (/ 0.89136115069332_dbl_kind, 0.895000510272953_dbl_kind, 0.900443813600613_dbl_kind, 0.921548508174156_dbl_kind, 0.971661306065192_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1339) = (/ 0.891361291070137_dbl_kind, 0.895001241558882_dbl_kind, 0.900447135421391_dbl_kind, 0.921558167292092_dbl_kind, 0.971666359671049_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1340) = (/ 0.891361431265349_dbl_kind, 0.895001972494926_dbl_kind, 0.900450456437958_dbl_kind, 0.921567823234373_dbl_kind, 0.971671404499024_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1341) = (/ 0.891361571279451_dbl_kind, 0.895002703081154_dbl_kind, 0.900453776651741_dbl_kind, 0.921577476003537_dbl_kind, 0.971676440567408_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1342) = (/ 0.89136171111294_dbl_kind, 0.895003433317636_dbl_kind, 0.900457096064164_dbl_kind, 0.921587125602108_dbl_kind, 0.971681467894447_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1343) = (/ 0.891361850766307_dbl_kind, 0.895004163204442_dbl_kind, 0.900460414676645_dbl_kind, 0.921596772032602_dbl_kind, 0.971686486498335_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1344) = (/ 0.891361990240045_dbl_kind, 0.895004892741645_dbl_kind, 0.9004637324906_dbl_kind, 0.921606415297522_dbl_kind, 0.971691496397226_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1345) = (/ 0.891362129534642_dbl_kind, 0.895005621929316_dbl_kind, 0.900467049507435_dbl_kind, 0.92161605539936_dbl_kind, 0.971696497609219_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1346) = (/ 0.891362268650585_dbl_kind, 0.895006350767528_dbl_kind, 0.900470365728554_dbl_kind, 0.921625692340596_dbl_kind, 0.971701490152372_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1347) = (/ 0.89136240758836_dbl_kind, 0.895007079256356_dbl_kind, 0.900473681155356_dbl_kind, 0.9216353261237_dbl_kind, 0.971706474044696_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1348) = (/ 0.891362546348449_dbl_kind, 0.895007807395874_dbl_kind, 0.900476995789235_dbl_kind, 0.921644956751129_dbl_kind, 0.971711449304153_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1349) = (/ 0.891362684931336_dbl_kind, 0.895008535186159_dbl_kind, 0.900480309631578_dbl_kind, 0.92165458422533_dbl_kind, 0.971716415948659_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1350) = (/ 0.891362823337498_dbl_kind, 0.895009262627286_dbl_kind, 0.90048362268377_dbl_kind, 0.92166420854874_dbl_kind, 0.971721373996084_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1351) = (/ 0.891362961567415_dbl_kind, 0.895009989719333_dbl_kind, 0.900486934947187_dbl_kind, 0.921673829723782_dbl_kind, 0.971726323464257_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1352) = (/ 0.891363099621561_dbl_kind, 0.895010716462379_dbl_kind, 0.900490246423205_dbl_kind, 0.921683447752869_dbl_kind, 0.971731264370952_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1353) = (/ 0.89136323750041_dbl_kind, 0.895011442856501_dbl_kind, 0.90049355711319_dbl_kind, 0.921693062638406_dbl_kind, 0.971736196733904_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1354) = (/ 0.891363375204435_dbl_kind, 0.895012168901781_dbl_kind, 0.900496867018505_dbl_kind, 0.92170267438278_dbl_kind, 0.9717411205708_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1355) = (/ 0.891363512734106_dbl_kind, 0.895012894598298_dbl_kind, 0.900500176140508_dbl_kind, 0.921712282988373_dbl_kind, 0.97174603589928_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1356) = (/ 0.891363650089891_dbl_kind, 0.895013619946135_dbl_kind, 0.900503484480552_dbl_kind, 0.921721888457553_dbl_kind, 0.971750942736947_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1357) = (/ 0.891363787272257_dbl_kind, 0.895014344945373_dbl_kind, 0.900506792039984_dbl_kind, 0.921731490792677_dbl_kind, 0.971755841101346_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1358) = (/ 0.891363924281669_dbl_kind, 0.895015069596096_dbl_kind, 0.900510098820146_dbl_kind, 0.921741089996093_dbl_kind, 0.971760731009985_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1359) = (/ 0.891364061118588_dbl_kind, 0.895015793898388_dbl_kind, 0.900513404822375_dbl_kind, 0.921750686070135_dbl_kind, 0.971765612480325_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1360) = (/ 0.891364197783477_dbl_kind, 0.895016517852333_dbl_kind, 0.900516710048003_dbl_kind, 0.921760279017128_dbl_kind, 0.971770485529788_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1361) = (/ 0.891364334276795_dbl_kind, 0.895017241458017_dbl_kind, 0.900520014498357_dbl_kind, 0.921769868839384_dbl_kind, 0.971775350175737_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1362) = (/ 0.891364470598999_dbl_kind, 0.895017964715528_dbl_kind, 0.900523318174758_dbl_kind, 0.921779455539207_dbl_kind, 0.971780206435506_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1363) = (/ 0.891364606750545_dbl_kind, 0.895018687624951_dbl_kind, 0.900526621078522_dbl_kind, 0.921789039118886_dbl_kind, 0.971785054326376_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1364) = (/ 0.891364742731885_dbl_kind, 0.895019410186375_dbl_kind, 0.90052992321096_dbl_kind, 0.921798619580702_dbl_kind, 0.971789893865588_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1365) = (/ 0.891364878543474_dbl_kind, 0.895020132399889_dbl_kind, 0.900533224573377_dbl_kind, 0.921808196926925_dbl_kind, 0.971794725070332_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1366) = (/ 0.891365014185761_dbl_kind, 0.895020854265582_dbl_kind, 0.900536525167074_dbl_kind, 0.921817771159811_dbl_kind, 0.971799547957764_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1367) = (/ 0.891365149659194_dbl_kind, 0.895021575783547_dbl_kind, 0.900539824993346_dbl_kind, 0.921827342281609_dbl_kind, 0.971804362544989_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1368) = (/ 0.891365284964221_dbl_kind, 0.895022296953872_dbl_kind, 0.900543124053482_dbl_kind, 0.921836910294554_dbl_kind, 0.971809168849072_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1369) = (/ 0.891365420101285_dbl_kind, 0.895023017776651_dbl_kind, 0.900546422348768_dbl_kind, 0.921846475200871_dbl_kind, 0.971813966887031_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1370) = (/ 0.891365555070831_dbl_kind, 0.895023738251977_dbl_kind, 0.900549719880481_dbl_kind, 0.921856037002774_dbl_kind, 0.971818756675843_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1371) = (/ 0.8913656898733_dbl_kind, 0.895024458379944_dbl_kind, 0.900553016649896_dbl_kind, 0.921865595702468_dbl_kind, 0.971823538232442_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1372) = (/ 0.891365824509131_dbl_kind, 0.895025178160645_dbl_kind, 0.90055631265828_dbl_kind, 0.921875151302143_dbl_kind, 0.971828311573717_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1373) = (/ 0.891365958978763_dbl_kind, 0.895025897594177_dbl_kind, 0.900559607906898_dbl_kind, 0.921884703803982_dbl_kind, 0.971833076716515_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1374) = (/ 0.891366093282632_dbl_kind, 0.895026616680636_dbl_kind, 0.900562902397006_dbl_kind, 0.921894253210156_dbl_kind, 0.971837833677643_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1375) = (/ 0.891366227421173_dbl_kind, 0.895027335420117_dbl_kind, 0.900566196129857_dbl_kind, 0.921903799522822_dbl_kind, 0.971842582473859_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1376) = (/ 0.891366361394818_dbl_kind, 0.895028053812721_dbl_kind, 0.900569489106697_dbl_kind, 0.921913342744131_dbl_kind, 0.971847323121882_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1377) = (/ 0.891366495203998_dbl_kind, 0.895028771858544_dbl_kind, 0.900572781328769_dbl_kind, 0.92192288287622_dbl_kind, 0.971852055638391_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1378) = (/ 0.891366628849144_dbl_kind, 0.895029489557687_dbl_kind, 0.900576072797307_dbl_kind, 0.921932419921217_dbl_kind, 0.971856780040018_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1379) = (/ 0.891366762330682_dbl_kind, 0.89503020691025_dbl_kind, 0.900579363513544_dbl_kind, 0.921941953881237_dbl_kind, 0.971861496343354_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1380) = (/ 0.89136689564904_dbl_kind, 0.895030923916333_dbl_kind, 0.900582653478702_dbl_kind, 0.921951484758386_dbl_kind, 0.971866204564953_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1381) = (/ 0.891367028804641_dbl_kind, 0.895031640576037_dbl_kind, 0.900585942694003_dbl_kind, 0.921961012554757_dbl_kind, 0.971870904721318_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1382) = (/ 0.891367161797908_dbl_kind, 0.895032356889468_dbl_kind, 0.900589231160661_dbl_kind, 0.921970537272437_dbl_kind, 0.971875596828916_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1383) = (/ 0.891367294629263_dbl_kind, 0.895033072856725_dbl_kind, 0.900592518879884_dbl_kind, 0.921980058913497_dbl_kind, 0.971880280904172_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1384) = (/ 0.891367427299124_dbl_kind, 0.895033788477916_dbl_kind, 0.900595805852876_dbl_kind, 0.921989577479998_dbl_kind, 0.971884956963467_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1385) = (/ 0.891367559807909_dbl_kind, 0.895034503753143_dbl_kind, 0.900599092080834_dbl_kind, 0.921999092973994_dbl_kind, 0.971889625023145_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1386) = (/ 0.891367692156035_dbl_kind, 0.895035218682513_dbl_kind, 0.90060237756495_dbl_kind, 0.922008605397524_dbl_kind, 0.971894285099503_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1387) = (/ 0.891367824343917_dbl_kind, 0.895035933266132_dbl_kind, 0.900605662306412_dbl_kind, 0.922018114752618_dbl_kind, 0.9718989372088_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1388) = (/ 0.891367956371965_dbl_kind, 0.895036647504107_dbl_kind, 0.900608946306401_dbl_kind, 0.922027621041296_dbl_kind, 0.971903581367255_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1389) = (/ 0.891368088240594_dbl_kind, 0.895037361396547_dbl_kind, 0.900612229566091_dbl_kind, 0.922037124265567_dbl_kind, 0.971908217591045_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1390) = (/ 0.891368219950211_dbl_kind, 0.89503807494356_dbl_kind, 0.900615512086654_dbl_kind, 0.922046624427427_dbl_kind, 0.971912845896302_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1391) = (/ 0.891368351501225_dbl_kind, 0.895038788145255_dbl_kind, 0.900618793869255_dbl_kind, 0.922056121528864_dbl_kind, 0.971917466299125_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1392) = (/ 0.891368482894042_dbl_kind, 0.895039501001744_dbl_kind, 0.90062207491505_dbl_kind, 0.922065615571854_dbl_kind, 0.971922078815565_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1393) = (/ 0.891368614129068_dbl_kind, 0.895040213513136_dbl_kind, 0.900625355225196_dbl_kind, 0.922075106558364_dbl_kind, 0.971926683461638_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1394) = (/ 0.891368745206704_dbl_kind, 0.895040925679544_dbl_kind, 0.900628634800839_dbl_kind, 0.922084594490348_dbl_kind, 0.971931280253318_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1395) = (/ 0.891368876127353_dbl_kind, 0.89504163750108_dbl_kind, 0.900631913643121_dbl_kind, 0.922094079369751_dbl_kind, 0.971935869206538_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1396) = (/ 0.891369006891416_dbl_kind, 0.895042348977858_dbl_kind, 0.900635191753179_dbl_kind, 0.922103561198506_dbl_kind, 0.971940450337188_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1397) = (/ 0.891369137499289_dbl_kind, 0.895043060109992_dbl_kind, 0.900638469132145_dbl_kind, 0.922113039978537_dbl_kind, 0.971945023661125_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1398) = (/ 0.891369267951371_dbl_kind, 0.895043770897596_dbl_kind, 0.900641745781144_dbl_kind, 0.922122515711757_dbl_kind, 0.971949589194163_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1399) = (/ 0.891369398248056_dbl_kind, 0.895044481340786_dbl_kind, 0.900645021701296_dbl_kind, 0.922131988400066_dbl_kind, 0.971954146952073_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1400) = (/ 0.891369528389739_dbl_kind, 0.895045191439679_dbl_kind, 0.900648296893714_dbl_kind, 0.922141458045357_dbl_kind, 0.971958696950591_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1401) = (/ 0.891369658376811_dbl_kind, 0.895045901194391_dbl_kind, 0.900651571359507_dbl_kind, 0.922150924649511_dbl_kind, 0.971963239205411_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1402) = (/ 0.891369788209664_dbl_kind, 0.89504661060504_dbl_kind, 0.900654845099779_dbl_kind, 0.922160388214397_dbl_kind, 0.971967773732188_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1403) = (/ 0.891369917888685_dbl_kind, 0.895047319671745_dbl_kind, 0.900658118115626_dbl_kind, 0.922169848741876_dbl_kind, 0.971972300546538_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1404) = (/ 0.891370047414264_dbl_kind, 0.895048028394624_dbl_kind, 0.90066139040814_dbl_kind, 0.922179306233797_dbl_kind, 0.971976819664039_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1405) = (/ 0.891370176786785_dbl_kind, 0.895048736773799_dbl_kind, 0.900664661978407_dbl_kind, 0.922188760691998_dbl_kind, 0.971981331100227_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1406) = (/ 0.891370306006635_dbl_kind, 0.89504944480939_dbl_kind, 0.900667932827507_dbl_kind, 0.922198212118307_dbl_kind, 0.971985834870602_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1407) = (/ 0.891370435074195_dbl_kind, 0.895050152501517_dbl_kind, 0.900671202956515_dbl_kind, 0.922207660514541_dbl_kind, 0.971990330990623_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1408) = (/ 0.891370563989846_dbl_kind, 0.895050859850304_dbl_kind, 0.900674472366498_dbl_kind, 0.922217105882509_dbl_kind, 0.971994819475714_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1409) = (/ 0.89137069275397_dbl_kind, 0.895051566855873_dbl_kind, 0.900677741058522_dbl_kind, 0.922226548224006_dbl_kind, 0.971999300341255_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1410) = (/ 0.891370821366945_dbl_kind, 0.895052273518348_dbl_kind, 0.900681009033642_dbl_kind, 0.922235987540819_dbl_kind, 0.972003773602593_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1411) = (/ 0.891370949829147_dbl_kind, 0.895052979837853_dbl_kind, 0.90068427629291_dbl_kind, 0.922245423834723_dbl_kind, 0.972008239275032_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1412) = (/ 0.891371078140953_dbl_kind, 0.895053685814514_dbl_kind, 0.900687542837373_dbl_kind, 0.922254857107483_dbl_kind, 0.972012697373843_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1413) = (/ 0.891371206302735_dbl_kind, 0.895054391448455_dbl_kind, 0.90069080866807_dbl_kind, 0.922264287360855_dbl_kind, 0.972017147914253_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1414) = (/ 0.891371334314868_dbl_kind, 0.895055096739804_dbl_kind, 0.900694073786036_dbl_kind, 0.922273714596582_dbl_kind, 0.972021590911455_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1415) = (/ 0.891371462177721_dbl_kind, 0.895055801688687_dbl_kind, 0.900697338192299_dbl_kind, 0.922283138816399_dbl_kind, 0.972026026380603_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1416) = (/ 0.891371589891666_dbl_kind, 0.895056506295233_dbl_kind, 0.900700601887883_dbl_kind, 0.922292560022028_dbl_kind, 0.972030454336813_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1417) = (/ 0.891371717457068_dbl_kind, 0.89505721055957_dbl_kind, 0.900703864873804_dbl_kind, 0.922301978215185_dbl_kind, 0.972034874795164_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1418) = (/ 0.891371844874297_dbl_kind, 0.895057914481827_dbl_kind, 0.900707127151075_dbl_kind, 0.922311393397569_dbl_kind, 0.972039287770697_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1419) = (/ 0.891371972143716_dbl_kind, 0.895058618062134_dbl_kind, 0.900710388720699_dbl_kind, 0.922320805570875_dbl_kind, 0.972043693278416_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1420) = (/ 0.89137209926569_dbl_kind, 0.895059321300623_dbl_kind, 0.900713649583678_dbl_kind, 0.922330214736784_dbl_kind, 0.972048091333287_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1421) = (/ 0.891372226240581_dbl_kind, 0.895060024197423_dbl_kind, 0.900716909741004_dbl_kind, 0.922339620896968_dbl_kind, 0.972052481950241_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1422) = (/ 0.89137235306875_dbl_kind, 0.895060726752667_dbl_kind, 0.900720169193667_dbl_kind, 0.922349024053087_dbl_kind, 0.972056865144167_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1423) = (/ 0.891372479750556_dbl_kind, 0.895061428966489_dbl_kind, 0.900723427942649_dbl_kind, 0.922358424206793_dbl_kind, 0.972061240929924_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1424) = (/ 0.891372606286359_dbl_kind, 0.89506213083902_dbl_kind, 0.900726685988925_dbl_kind, 0.922367821359727_dbl_kind, 0.972065609322327_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1425) = (/ 0.891372732676514_dbl_kind, 0.895062832370396_dbl_kind, 0.900729943333468_dbl_kind, 0.922377215513518_dbl_kind, 0.97206997033616_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1426) = (/ 0.891372858921376_dbl_kind, 0.895063533560751_dbl_kind, 0.900733199977241_dbl_kind, 0.922386606669788_dbl_kind, 0.972074323986165_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1427) = (/ 0.8913729850213_dbl_kind, 0.89506423441022_dbl_kind, 0.900736455921203_dbl_kind, 0.922395994830145_dbl_kind, 0.972078670287055_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1428) = (/ 0.891373110976639_dbl_kind, 0.89506493491894_dbl_kind, 0.900739711166309_dbl_kind, 0.922405379996189_dbl_kind, 0.9720830092535_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1429) = (/ 0.891373236787743_dbl_kind, 0.895065635087047_dbl_kind, 0.900742965713505_dbl_kind, 0.922414762169509_dbl_kind, 0.972087340900133_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1430) = (/ 0.891373362454962_dbl_kind, 0.895066334914678_dbl_kind, 0.900746219563732_dbl_kind, 0.922424141351684_dbl_kind, 0.972091665241558_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1431) = (/ 0.891373487978645_dbl_kind, 0.895067034401971_dbl_kind, 0.900749472717927_dbl_kind, 0.922433517544284_dbl_kind, 0.972095982292336_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1432) = (/ 0.891373613359138_dbl_kind, 0.895067733549066_dbl_kind, 0.900752725177019_dbl_kind, 0.922442890748865_dbl_kind, 0.972100292066994_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1433) = (/ 0.891373738596787_dbl_kind, 0.895068432356102_dbl_kind, 0.900755976941932_dbl_kind, 0.922452260966976_dbl_kind, 0.972104594580028_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1434) = (/ 0.891373863691936_dbl_kind, 0.895069130823218_dbl_kind, 0.900759228013584_dbl_kind, 0.922461628200156_dbl_kind, 0.972108889845887_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1435) = (/ 0.891373988644929_dbl_kind, 0.895069828950555_dbl_kind, 0.900762478392887_dbl_kind, 0.922470992449932_dbl_kind, 0.972113177878996_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1436) = (/ 0.891374113456106_dbl_kind, 0.895070526738254_dbl_kind, 0.900765728080748_dbl_kind, 0.922480353717823_dbl_kind, 0.972117458693738_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1437) = (/ 0.891374238125809_dbl_kind, 0.895071224186458_dbl_kind, 0.900768977078066_dbl_kind, 0.922489712005334_dbl_kind, 0.972121732304463_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1438) = (/ 0.891374362654374_dbl_kind, 0.895071921295308_dbl_kind, 0.900772225385736_dbl_kind, 0.922499067313964_dbl_kind, 0.972125998725485_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1439) = (/ 0.891374487042142_dbl_kind, 0.895072618064948_dbl_kind, 0.900775473004647_dbl_kind, 0.922508419645199_dbl_kind, 0.97213025797108_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1440) = (/ 0.891374611289446_dbl_kind, 0.895073314495522_dbl_kind, 0.900778719935682_dbl_kind, 0.922517769000517_dbl_kind, 0.972134510055496_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1441) = (/ 0.891374735396623_dbl_kind, 0.895074010587174_dbl_kind, 0.900781966179717_dbl_kind, 0.922527115381384_dbl_kind, 0.972138754992937_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1442) = (/ 0.891374859364006_dbl_kind, 0.895074706340049_dbl_kind, 0.900785211737622_dbl_kind, 0.922536458789258_dbl_kind, 0.972142992797579_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1443) = (/ 0.891374983191927_dbl_kind, 0.895075401754293_dbl_kind, 0.900788456610264_dbl_kind, 0.922545799225584_dbl_kind, 0.972147223483561_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1444) = (/ 0.891375106880717_dbl_kind, 0.895076096830051_dbl_kind, 0.9007917007985_dbl_kind, 0.9225551366918_dbl_kind, 0.972151447064986_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1445) = (/ 0.891375230430705_dbl_kind, 0.895076791567471_dbl_kind, 0.900794944303184_dbl_kind, 0.922564471189331_dbl_kind, 0.972155663555923_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1446) = (/ 0.89137535384222_dbl_kind, 0.895077485966701_dbl_kind, 0.900798187125163_dbl_kind, 0.922573802719594_dbl_kind, 0.972159872970406_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1447) = (/ 0.891375477115588_dbl_kind, 0.895078180027888_dbl_kind, 0.900801429265278_dbl_kind, 0.922583131283995_dbl_kind, 0.972164075322439_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1448) = (/ 0.891375600251136_dbl_kind, 0.895078873751182_dbl_kind, 0.900804670724365_dbl_kind, 0.92259245688393_dbl_kind, 0.972168270625984_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1449) = (/ 0.891375723249188_dbl_kind, 0.89507956713673_dbl_kind, 0.900807911503253_dbl_kind, 0.922601779520786_dbl_kind, 0.972172458894975_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1450) = (/ 0.891375846110066_dbl_kind, 0.895080260184683_dbl_kind, 0.900811151602764_dbl_kind, 0.922611099195938_dbl_kind, 0.972176640143311_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1451) = (/ 0.891375968834094_dbl_kind, 0.895080952895192_dbl_kind, 0.900814391023717_dbl_kind, 0.922620415910753_dbl_kind, 0.972180814384854_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1452) = (/ 0.891376091421591_dbl_kind, 0.895081645268407_dbl_kind, 0.900817629766923_dbl_kind, 0.922629729666587_dbl_kind, 0.972184981633431_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1453) = (/ 0.891376213872876_dbl_kind, 0.895082337304481_dbl_kind, 0.900820867833187_dbl_kind, 0.922639040464785_dbl_kind, 0.972189141902845_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1454) = (/ 0.891376336188268_dbl_kind, 0.895083029003565_dbl_kind, 0.900824105223309_dbl_kind, 0.922648348306685_dbl_kind, 0.972193295206854_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1455) = (/ 0.891376458368084_dbl_kind, 0.895083720365812_dbl_kind, 0.900827341938082_dbl_kind, 0.922657653193611_dbl_kind, 0.972197441559186_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1456) = (/ 0.891376580412638_dbl_kind, 0.895084411391376_dbl_kind, 0.900830577978295_dbl_kind, 0.92266695512688_dbl_kind, 0.972201580973539_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1457) = (/ 0.891376702322246_dbl_kind, 0.895085102080409_dbl_kind, 0.900833813344728_dbl_kind, 0.922676254107798_dbl_kind, 0.972205713463571_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1458) = (/ 0.89137682409722_dbl_kind, 0.895085792433068_dbl_kind, 0.900837048038157_dbl_kind, 0.922685550137662_dbl_kind, 0.972209839042916_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1459) = (/ 0.891376945737871_dbl_kind, 0.895086482449506_dbl_kind, 0.900840282059352_dbl_kind, 0.922694843217757_dbl_kind, 0.972213957725165_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1460) = (/ 0.891377067244512_dbl_kind, 0.89508717212988_dbl_kind, 0.900843515409077_dbl_kind, 0.92270413334936_dbl_kind, 0.972218069523879_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1461) = (/ 0.891377188617451_dbl_kind, 0.895087861474345_dbl_kind, 0.900846748088089_dbl_kind, 0.922713420533738_dbl_kind, 0.97222217445259_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1462) = (/ 0.891377309856996_dbl_kind, 0.895088550483058_dbl_kind, 0.90084998009714_dbl_kind, 0.922722704772146_dbl_kind, 0.972226272524794_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1463) = (/ 0.891377430963453_dbl_kind, 0.895089239156178_dbl_kind, 0.900853211436976_dbl_kind, 0.922731986065832_dbl_kind, 0.972230363753956_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1464) = (/ 0.891377551937129_dbl_kind, 0.895089927493859_dbl_kind, 0.900856442108336_dbl_kind, 0.922741264416032_dbl_kind, 0.972234448153503_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1465) = (/ 0.891377672778328_dbl_kind, 0.895090615496263_dbl_kind, 0.900859672111955_dbl_kind, 0.922750539823973_dbl_kind, 0.972238525736836_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1466) = (/ 0.891377793487354_dbl_kind, 0.895091303163547_dbl_kind, 0.900862901448559_dbl_kind, 0.922759812290872_dbl_kind, 0.972242596517318_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1467) = (/ 0.891377914064508_dbl_kind, 0.89509199049587_dbl_kind, 0.900866130118871_dbl_kind, 0.922769081817937_dbl_kind, 0.972246660508282_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1468) = (/ 0.891378034510091_dbl_kind, 0.895092677493393_dbl_kind, 0.900869358123606_dbl_kind, 0.922778348406364_dbl_kind, 0.972250717723031_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1469) = (/ 0.891378154824403_dbl_kind, 0.895093364156276_dbl_kind, 0.900872585463475_dbl_kind, 0.922787612057341_dbl_kind, 0.972254768174833_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1470) = (/ 0.891378275007743_dbl_kind, 0.89509405048468_dbl_kind, 0.900875812139181_dbl_kind, 0.922796872772046_dbl_kind, 0.972258811876922_dbl_kind /)
   iceAsymmetryParameterDirect(1:5 ,1471) = (/ 0.891378395060407_dbl_kind, 0.895094736478766_dbl_kind, 0.900879038151421_dbl_kind, 0.922806130551648_dbl_kind, 0.972262848842506_dbl_kind /)

   iceAsymmetryParameterDiffuse(1:5 ,1) = (/ 0.883796988056542_dbl_kind, 0.883342090644533_dbl_kind, 0.882303225047584_dbl_kind, 0.882254720448352_dbl_kind, 0.891906690982411_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,2) = (/ 0.884028281638843_dbl_kind, 0.883650540642183_dbl_kind, 0.882710900566814_dbl_kind, 0.882764613086411_dbl_kind, 0.892694276417957_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,3) = (/ 0.88424663334592_dbl_kind, 0.883940569462334_dbl_kind, 0.883094503579836_dbl_kind, 0.883248326043141_dbl_kind, 0.893450073209044_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,4) = (/ 0.884452844861344_dbl_kind, 0.884213851534574_dbl_kind, 0.883455324541469_dbl_kind, 0.883707612999668_dbl_kind, 0.894176631663937_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,5) = (/ 0.884647649374281_dbl_kind, 0.884471892549507_dbl_kind, 0.883794679592248_dbl_kind, 0.884144176622858_dbl_kind, 0.894876220141942_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,6) = (/ 0.884831726410077_dbl_kind, 0.884716046213931_dbl_kind, 0.884113902083967_dbl_kind, 0.884559641943785_dbl_kind, 0.895550861780034_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,7) = (/ 0.885005712686694_dbl_kind, 0.884947529649023_dbl_kind, 0.884414328395376_dbl_kind, 0.884955537372027_dbl_kind, 0.89620236592712_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,8) = (/ 0.885170209913341_dbl_kind, 0.885167437327416_dbl_kind, 0.88469728162147_dbl_kind, 0.885333283129094_dbl_kind, 0.896832355262637_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,9) = (/ 0.885325790246527_dbl_kind, 0.885376753562385_dbl_kind, 0.884964055525829_dbl_kind, 0.885694186171259_dbl_kind, 0.897442289306865_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,10) = (/ 0.885472999967418_dbl_kind, 0.885576363634085_dbl_kind, 0.885215900222602_dbl_kind, 0.886039440351782_dbl_kind, 0.898033484848693_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,11) = (/ 0.885612361828784_dbl_kind, 0.88576706367635_dbl_kind, 0.885454010372427_dbl_kind, 0.88637013050785_dbl_kind, 0.898607133697787_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,12) = (/ 0.885744376429824_dbl_kind, 0.885949569463488_dbl_kind, 0.885679516194706_dbl_kind, 0.886687239246645_dbl_kind, 0.899164318091874_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,13) = (/ 0.885869522905364_dbl_kind, 0.886124524237356_dbl_kind, 0.885893477276041_dbl_kind, 0.886991655375074_dbl_kind, 0.899706024040709_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,14) = (/ 0.885988259157575_dbl_kind, 0.886292505706632_dbl_kind, 0.886096878953615_dbl_kind, 0.887284183119532_dbl_kind, 0.900233152855044_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,15) = (/ 0.886101021809947_dbl_kind, 0.886454032336609_dbl_kind, 0.886290630941009_dbl_kind, 0.887565551484452_dbl_kind, 0.9007465310844_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,16) = (/ 0.886208226022785_dbl_kind, 0.886609569031906_dbl_kind, 0.886475567816117_dbl_kind, 0.88783642328275_dbl_kind, 0.901246919067011_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,17) = (/ 0.886310265275357_dbl_kind, 0.886759532298146_dbl_kind, 0.886652450985984_dbl_kind, 0.888097403528819_dbl_kind, 0.901735018276774_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,18) = (/ 0.886407511191253_dbl_kind, 0.886904294953166_dbl_kind, 0.886821971765953_dbl_kind, 0.888349047012215_dbl_kind, 0.902211477633972_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,19) = (/ 0.886500313459644_dbl_kind, 0.887044190444398_dbl_kind, 0.886984755248813_dbl_kind, 0.888591864968568_dbl_kind, 0.902676898928897_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,20) = (/ 0.886588999885604_dbl_kind, 0.887179516817139_dbl_kind, 0.887141364685484_dbl_kind, 0.888826330836459_dbl_kind, 0.903131841490282_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,21) = (/ 0.886673876586886_dbl_kind, 0.887310540368572_dbl_kind, 0.887292306146555_dbl_kind, 0.889052885138958_dbl_kind, 0.903576826213942_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,22) = (/ 0.886755228342161_dbl_kind, 0.88743749901457_dbl_kind, 0.887438033279919_dbl_kind, 0.889271939560447_dbl_kind, 0.904012339051525_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,23) = (/ 0.886833319086356_dbl_kind, 0.887560605390382_dbl_kind, 0.887578952021718_dbl_kind, 0.889483880307311_dbl_kind, 0.90443883404501_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,24) = (/ 0.886908392541867_dbl_kind, 0.887680049701907_dbl_kind, 0.887715425154613_dbl_kind, 0.889689070848517_dbl_kind, 0.904856735979786_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,25) = (/ 0.886980672969772_dbl_kind, 0.887796002341302_dbl_kind, 0.887847776638672_dbl_kind, 0.889887854132129_dbl_kind, 0.905266442717668_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,26) = (/ 0.887050366022347_dbl_kind, 0.887908616278673_dbl_kind, 0.887976295665916_dbl_kind, 0.890080554368865_dbl_kind, 0.905668327261321_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,27) = (/ 0.887117659676757_dbl_kind, 0.888018029240549_dbl_kind, 0.888101240410255_dbl_kind, 0.890267478465699_dbl_kind, 0.906062739592946_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,28) = (/ 0.887182725229628_dbl_kind, 0.888124365685244_dbl_kind, 0.888222841460672_dbl_kind, 0.890448917182832_dbl_kind, 0.906450008322743_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,29) = (/ 0.887245718332769_dbl_kind, 0.888227738585142_dbl_kind, 0.888341304937679_dbl_kind, 0.890625146077036_dbl_kind, 0.906830442176463_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,30) = (/ 0.887306780051657_dbl_kind, 0.88832825102602_dbl_kind, 0.888456815301928_dbl_kind, 0.890796426284211_dbl_kind, 0.907204331346206_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,31) = (/ 0.887366037929941_dbl_kind, 0.888425997633761_dbl_kind, 0.888569537870031_dbl_kind, 0.890963005184391_dbl_kind, 0.907571948724267_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,32) = (/ 0.887423607045217_dbl_kind, 0.888521065839075_dbl_kind, 0.888679621056604_dbl_kind, 0.891125116983819_dbl_kind, 0.907933551036209_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,33) = (/ 0.887479591043387_dbl_kind, 0.888613536991061_dbl_kind, 0.888787198363803_dbl_kind, 0.891282983241_dbl_kind, 0.908289379886484_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,34) = (/ 0.88753408314102_dbl_kind, 0.888703487330578_dbl_kind, 0.888892390140641_dbl_kind, 0.891436813357139_dbl_kind, 0.90863966272739_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,35) = (/ 0.887587167087193_dbl_kind, 0.888790988834415_dbl_kind, 0.88899530513433_dbl_kind, 0.891586805045823_dbl_kind, 0.908984613760256_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,36) = (/ 0.887638918078174_dbl_kind, 0.888876109941195_dbl_kind, 0.889096041855276_dbl_kind, 0.891733144792322_dbl_kind, 0.909324434776124_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,37) = (/ 0.887689403620121_dbl_kind, 0.888958916169717_dbl_kind, 0.889194689776193_dbl_kind, 0.891876008309212_dbl_kind, 0.909659315941875_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,38) = (/ 0.887738684336504_dbl_kind, 0.889039470640159_dbl_kind, 0.889291330384397_dbl_kind, 0.892015560992235_dbl_kind, 0.909989436536712_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,39) = (/ 0.887786814718402_dbl_kind, 0.889117834508164_dbl_kind, 0.889386038104745_dbl_kind, 0.892151958378099_dbl_kind, 0.910314965643127_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,40) = (/ 0.887833843816991_dbl_kind, 0.889194067321354_dbl_kind, 0.88947888110906_dbl_kind, 0.892285346604339_dbl_kind, 0.910636062795733_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,41) = (/ 0.887879815878559_dbl_kind, 0.8892682273073_dbl_kind, 0.889569922026239_dbl_kind, 0.892415862870263_dbl_kind, 0.910952878590902_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,42) = (/ 0.887924770923229_dbl_kind, 0.889340371601377_dbl_kind, 0.889659218565671_dbl_kind, 0.892543635897201_dbl_kind, 0.911265555259685_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,43) = (/ 0.887968745269213_dbl_kind, 0.889410556422354_dbl_kind, 0.889746824065113_dbl_kind, 0.892668786385915_dbl_kind, 0.911574227206191_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,44) = (/ 0.888011772004978_dbl_kind, 0.88947883720293_dbl_kind, 0.889832787972822_dbl_kind, 0.892791427468755_dbl_kind, 0.911879021513383_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,45) = (/ 0.888053881412039_dbl_kind, 0.889545268681829_dbl_kind, 0.889917156272457_dbl_kind, 0.892911665154172_dbl_kind, 0.912180058418001_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,46) = (/ 0.888095101341409_dbl_kind, 0.889609904963444_dbl_kind, 0.889999971858194_dbl_kind, 0.893029598761265_dbl_kind, 0.912477451756221_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,47) = (/ 0.888135457546868_dbl_kind, 0.889672799550429_dbl_kind, 0.890081274866456_dbl_kind, 0.893145321342271_dbl_kind, 0.912771309381574_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,48) = (/ 0.888174973978335_dbl_kind, 0.889734005354067_dbl_kind, 0.890161102969814_dbl_kind, 0.893258920091117_dbl_kind, 0.913061733556469_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,49) = (/ 0.888213673038585_dbl_kind, 0.889793574686733_dbl_kind, 0.890239491637826_dbl_kind, 0.893370476736499_dbl_kind, 0.913348821318716_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,50) = (/ 0.888251575806578_dbl_kind, 0.889851559240207_dbl_kind, 0.890316474368906_dbl_kind, 0.893480067918188_dbl_kind, 0.913632664824293_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,51) = (/ 0.888288702230505_dbl_kind, 0.88990801005319_dbl_kind, 0.890392082896773_dbl_kind, 0.893587765545606_dbl_kind, 0.913913351667631_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,52) = (/ 0.888325071293592_dbl_kind, 0.889962977470874_dbl_kind, 0.890466347374466_dbl_kind, 0.893693637138017_dbl_kind, 0.914190965180586_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,53) = (/ 0.888360701155516_dbl_kind, 0.890016511099083_dbl_kind, 0.890539296538565_dbl_kind, 0.89379774614589_dbl_kind, 0.914465584711298_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,54) = (/ 0.88839560927213_dbl_kind, 0.890068659755085_dbl_kind, 0.890610957855826_dbl_kind, 0.893900152253291_dbl_kind, 0.914737285884037_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,55) = (/ 0.888429812496003_dbl_kind, 0.890119471416878_dbl_kind, 0.890681357654177_dbl_kind, 0.894000911661363_dbl_kind, 0.915006140841175_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,56) = (/ 0.888463327160124_dbl_kind, 0.890168993172454_dbl_kind, 0.890750521239729_dbl_kind, 0.894100077353134_dbl_kind, 0.915272218468297_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,57) = (/ 0.888496169146878_dbl_kind, 0.890217271170279_dbl_kind, 0.890818473001267_dbl_kind, 0.894197699340066_dbl_kind, 0.915535584603522_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,58) = (/ 0.888528353944274_dbl_kind, 0.890264350571979_dbl_kind, 0.890885236503478_dbl_kind, 0.894293824890895_dbl_kind, 0.915796302232_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,59) = (/ 0.888559896691174_dbl_kind, 0.890310275508056_dbl_kind, 0.890950834570019_dbl_kind, 0.894388498743404_dbl_kind, 0.916054431666545_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,60) = (/ 0.88859081221313_dbl_kind, 0.89035508903723_dbl_kind, 0.891015289357418_dbl_kind, 0.894481763299879_dbl_kind, 0.9163100307153_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,61) = (/ 0.888621115050245_dbl_kind, 0.890398833109884_dbl_kind, 0.891078622420656_dbl_kind, 0.894573658807056_dbl_kind, 0.916563154837322_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,62) = (/ 0.888650819478345_dbl_kind, 0.89044154853592_dbl_kind, 0.89114085477123_dbl_kind, 0.894664223521376_dbl_kind, 0.916813857286906_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,63) = (/ 0.888679939524565_dbl_kind, 0.890483274957257_dbl_kind, 0.891202006928368_dbl_kind, 0.894753493860463_dbl_kind, 0.917062189247434_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,64) = (/ 0.888708488978359_dbl_kind, 0.890524050825058_dbl_kind, 0.891262098964049_dbl_kind, 0.894841504541665_dbl_kind, 0.917308199955472_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,65) = (/ 0.888736481398781_dbl_kind, 0.890563913381718_dbl_kind, 0.891321150542374_dbl_kind, 0.894928288708573_dbl_kind, 0.91755193681586_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,66) = (/ 0.888763930118802_dbl_kind, 0.89060289864758_dbl_kind, 0.89137918095385_dbl_kind, 0.895013878046367_dbl_kind, 0.917793445508402_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,67) = (/ 0.888790848247292_dbl_kind, 0.890641041412248_dbl_kind, 0.891436209145022_dbl_kind, 0.895098302886845_dbl_kind, 0.91803277008682_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,68) = (/ 0.888817248669241_dbl_kind, 0.890678375230371_dbl_kind, 0.891492253743939_dbl_kind, 0.895181592303958_dbl_kind, 0.918269953070515_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,69) = (/ 0.888843144044675_dbl_kind, 0.890714932421694_dbl_kind, 0.891547333081832_dbl_kind, 0.895263774200623_dbl_kind, 0.9185050355297_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,70) = (/ 0.888868546806661_dbl_kind, 0.890750744075159_dbl_kind, 0.891601465211407_dbl_kind, 0.895344875387595_dbl_kind, 0.9187380571644_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,71) = (/ 0.888893469158747_dbl_kind, 0.890785840056822_dbl_kind, 0.891654667922093_dbl_kind, 0.895424921655057_dbl_kind, 0.918969056377788_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,72) = (/ 0.888917923072088_dbl_kind, 0.890820249021316_dbl_kind, 0.891706958752572_dbl_kind, 0.895503937837647_dbl_kind, 0.919198070344288_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,73) = (/ 0.888941920282494_dbl_kind, 0.890853998426608_dbl_kind, 0.891758355000908_dbl_kind, 0.895581947873499_dbl_kind, 0.91942513507287_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,74) = (/ 0.88896547228757_dbl_kind, 0.890887114551756_dbl_kind, 0.89180887373255_dbl_kind, 0.895658974857909_dbl_kind, 0.91965028546588_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,75) = (/ 0.888988590344082_dbl_kind, 0.890919622517414_dbl_kind, 0.891858531786464_dbl_kind, 0.895735041092142_dbl_kind, 0.919873555373783_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,76) = (/ 0.889011285465668_dbl_kind, 0.890951546308795_dbl_kind, 0.891907345779658_dbl_kind, 0.895810168127874_dbl_kind, 0.920094977646107_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,77) = (/ 0.889033568420953_dbl_kind, 0.890982908800816_dbl_kind, 0.891955332110304_dbl_kind, 0.895884376807738_dbl_kind, 0.920314584178915_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,78) = (/ 0.889055449732138_dbl_kind, 0.891013731785184_dbl_kind, 0.892002506959671_dbl_kind, 0.895957687302357_dbl_kind, 0.920532405959048_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,79) = (/ 0.889076939674086_dbl_kind, 0.891044035999144_dbl_kind, 0.892048886293071_dbl_kind, 0.896030119144265_dbl_kind, 0.920748473105413_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,80) = (/ 0.889098048273922_dbl_kind, 0.89107384115566_dbl_kind, 0.89209448585997_dbl_kind, 0.896101691259039_dbl_kind, 0.920962814907536_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,81) = (/ 0.889118785311168_dbl_kind, 0.891103165974784_dbl_kind, 0.892139321193441_dbl_kind, 0.896172421993956_dbl_kind, 0.921175459861586_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,82) = (/ 0.889139160318374_dbl_kind, 0.89113202821599_dbl_kind, 0.892183407609094_dbl_kind, 0.896242329144436_dbl_kind, 0.921386435704079_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,83) = (/ 0.889159182582262_dbl_kind, 0.891160444711281_dbl_kind, 0.892226760203618_dbl_kind, 0.896311429978535_dbl_kind, 0.921595769443438_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,84) = (/ 0.889178861145343_dbl_kind, 0.891188431398837_dbl_kind, 0.89226939385304_dbl_kind, 0.896379741259696_dbl_kind, 0.921803487389562_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,85) = (/ 0.889198204807981_dbl_kind, 0.891216003357059_dbl_kind, 0.892311323210821_dbl_kind, 0.896447279267942_dbl_kind, 0.922009615181575_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,86) = (/ 0.889217222130878_dbl_kind, 0.891243174838803_dbl_kind, 0.892352562705864_dbl_kind, 0.896514059819706_dbl_kind, 0.922214177813886_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,87) = (/ 0.88923592143794_dbl_kind, 0.891269959305676_dbl_kind, 0.89239312654053_dbl_kind, 0.896580098286422_dbl_kind, 0.92241719966067_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,88) = (/ 0.889254310819507_dbl_kind, 0.891296369462232_dbl_kind, 0.892433028688715_dbl_kind, 0.896645409612034_dbl_kind, 0.922618704498939_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,89) = (/ 0.889272398135877_dbl_kind, 0.891322417289928_dbl_kind, 0.892472282894062_dbl_kind, 0.896710008329519_dbl_kind, 0.922818715530237_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,90) = (/ 0.889290191021132_dbl_kind, 0.891348114080751_dbl_kind, 0.892510902668348_dbl_kind, 0.896773908576533_dbl_kind, 0.923017255401135_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,91) = (/ 0.889307696887197_dbl_kind, 0.891373470470375_dbl_kind, 0.892548901290091_dbl_kind, 0.896837124110269_dbl_kind, 0.923214346222563_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,92) = (/ 0.889324922928117_dbl_kind, 0.891398496470771_dbl_kind, 0.892586291803415_dbl_kind, 0.896899668321597_dbl_kind, 0.923410009588088_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,93) = (/ 0.889341876124513_dbl_kind, 0.89142320150218_dbl_kind, 0.892623087017193_dbl_kind, 0.896961554248552_dbl_kind, 0.923604266591208_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,94) = (/ 0.889358563248189_dbl_kind, 0.891447594424376_dbl_kind, 0.89265929950449_dbl_kind, 0.897022794589233_dbl_kind, 0.923797137841743_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,95) = (/ 0.889374990866859_dbl_kind, 0.891471683567148_dbl_kind, 0.892694941602318_dbl_kind, 0.897083401714142_dbl_kind, 0.923988643481367_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,96) = (/ 0.889391165348976_dbl_kind, 0.891495476759951_dbl_kind, 0.892730025411724_dbl_kind, 0.897143387678026_dbl_kind, 0.924178803198377_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,97) = (/ 0.88940709286862_dbl_kind, 0.89151898136068_dbl_kind, 0.892764562798188_dbl_kind, 0.897202764231238_dbl_kind, 0.924367636241708_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,98) = (/ 0.889422779410445_dbl_kind, 0.89154220428353_dbl_kind, 0.89279856539236_dbl_kind, 0.897261542830655_dbl_kind, 0.924555161434289_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,99) = (/ 0.889438230774645_dbl_kind, 0.891565152025896_dbl_kind, 0.892832044591116_dbl_kind, 0.897319734650179_dbl_kind, 0.92474139718577_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,100) = (/ 0.889453452581931_dbl_kind, 0.891587830694315_dbl_kind, 0.892865011558921_dbl_kind, 0.897377350590835_dbl_kind, 0.924926361504649_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,101) = (/ 0.8894684502785_dbl_kind, 0.891610246029412_dbl_kind, 0.892897477229501_dbl_kind, 0.897434401290483_dbl_kind, 0.925110072009855_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,102) = (/ 0.889483229140977_dbl_kind, 0.891632403429845_dbl_kind, 0.89292945230781_dbl_kind, 0.897490897133172_dbl_kind, 0.92529254594184_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,103) = (/ 0.889497794281325_dbl_kind, 0.891654307975246_dbl_kind, 0.892960947272257_dbl_kind, 0.897546848258136_dbl_kind, 0.925473800173167_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,104) = (/ 0.889512150651693_dbl_kind, 0.891675964448158_dbl_kind, 0.892991972377214_dbl_kind, 0.897602264568443_dbl_kind, 0.925653851218676_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,105) = (/ 0.889526303049222_dbl_kind, 0.891697377354962_dbl_kind, 0.893022537655749_dbl_kind, 0.897657155739317_dbl_kind, 0.925832715245224_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,106) = (/ 0.889540256120764_dbl_kind, 0.891718550945817_dbl_kind, 0.893052652922592_dbl_kind, 0.897711531226132_dbl_kind, 0.926010408081041_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,107) = (/ 0.889554014367539_dbl_kind, 0.891739489233612_dbl_kind, 0.893082327777296_dbl_kind, 0.897765400272093_dbl_kind, 0.926186945224716_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,108) = (/ 0.889567582149695_dbl_kind, 0.891760196011949_dbl_kind, 0.893111571607599_dbl_kind, 0.897818771915606_dbl_kind, 0.926362341853852_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,109) = (/ 0.889580963690795_dbl_kind, 0.891780674872182_dbl_kind, 0.893140393592933_dbl_kind, 0.897871654997342_dbl_kind, 0.926536612833389_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,110) = (/ 0.889594163082194_dbl_kind, 0.891800929219512_dbl_kind, 0.893168802708094_dbl_kind, 0.897924058167015_dbl_kind, 0.926709772723631_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,111) = (/ 0.889607184287334_dbl_kind, 0.891820962288188_dbl_kind, 0.893196807727023_dbl_kind, 0.897975989889868_dbl_kind, 0.926881835787992_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,112) = (/ 0.889620031145927_dbl_kind, 0.891840777155813_dbl_kind, 0.893224417226713_dbl_kind, 0.898027458452871_dbl_kind, 0.927052816000474_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,113) = (/ 0.889632707378049_dbl_kind, 0.891860376756798_dbl_kind, 0.893251639591186_dbl_kind, 0.89807847197065_dbl_kind, 0.927222727052889_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,114) = (/ 0.889645216588119_dbl_kind, 0.891879763894978_dbl_kind, 0.893278483015553_dbl_kind, 0.898129038391152_dbl_kind, 0.927391582361845_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,115) = (/ 0.889657562268782_dbl_kind, 0.891898941255431_dbl_kind, 0.893304955510124_dbl_kind, 0.898179165501033_dbl_kind, 0.927559395075515_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,116) = (/ 0.889669747804685_dbl_kind, 0.891917911415517_dbl_kind, 0.89333106490455_dbl_kind, 0.898228860930815_dbl_kind, 0.927726178080177_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,117) = (/ 0.889681776476153_dbl_kind, 0.891936676855171_dbl_kind, 0.893356818851999_dbl_kind, 0.89827813215977_dbl_kind, 0.927891944006573_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,118) = (/ 0.889693651462752_dbl_kind, 0.891955239966484_dbl_kind, 0.893382224833324_dbl_kind, 0.898326986520589_dbl_kind, 0.928056705236054_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,119) = (/ 0.889705375846762_dbl_kind, 0.891973603062593_dbl_kind, 0.893407290161237_dbl_kind, 0.898375431203798_dbl_kind, 0.928220473906561_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,120) = (/ 0.889716952616536_dbl_kind, 0.891991768385917_dbl_kind, 0.893432021984454_dbl_kind, 0.898423473261967_dbl_kind, 0.928383261918425_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,121) = (/ 0.889728384669772_dbl_kind, 0.892009738115769_dbl_kind, 0.893456427291817_dbl_kind, 0.89847111961369_dbl_kind, 0.928545080939997_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,122) = (/ 0.889739674816671_dbl_kind, 0.892027514375368_dbl_kind, 0.893480512916363_dbl_kind, 0.898518377047369_dbl_kind, 0.928705942413132_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,123) = (/ 0.889750825783011_dbl_kind, 0.89204509923828_dbl_kind, 0.893504285539359_dbl_kind, 0.898565252224786_dbl_kind, 0.928865857558508_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,124) = (/ 0.889761840213119_dbl_kind, 0.892062494734333_dbl_kind, 0.893527751694255_dbl_kind, 0.898611751684493_dbl_kind, 0.929024837380814_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,125) = (/ 0.889772720672754_dbl_kind, 0.892079702855005_dbl_kind, 0.893550917770585_dbl_kind, 0.898657881845016_dbl_kind, 0.929182892673788_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,126) = (/ 0.889783469651891_dbl_kind, 0.892096725558345_dbl_kind, 0.893573790017786_dbl_kind, 0.898703649007882_dbl_kind, 0.929340034025132_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,127) = (/ 0.889794089567424_dbl_kind, 0.89211356477343_dbl_kind, 0.893596374548932_dbl_kind, 0.898749059360483_dbl_kind, 0.929496271821287_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,128) = (/ 0.88980458276578_dbl_kind, 0.892130222404395_dbl_kind, 0.893618677344391_dbl_kind, 0.898794118978777_dbl_kind, 0.929651616252099_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,129) = (/ 0.889814951525444_dbl_kind, 0.89214670033406_dbl_kind, 0.893640704255381_dbl_kind, 0.898838833829849_dbl_kind, 0.929806077315359_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,130) = (/ 0.889825198059403_dbl_kind, 0.892163000427178_dbl_kind, 0.893662461007437_dbl_kind, 0.898883209774316_dbl_kind, 0.929959664821222_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,131) = (/ 0.889835324517515_dbl_kind, 0.892179124533331_dbl_kind, 0.893683953203773_dbl_kind, 0.898927252568607_dbl_kind, 0.930112388396531_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,132) = (/ 0.889845332988786_dbl_kind, 0.892195074489491_dbl_kind, 0.893705186328558_dbl_kind, 0.898970967867124_dbl_kind, 0.930264257489025_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,133) = (/ 0.889855225503586_dbl_kind, 0.892210852122277_dbl_kind, 0.893726165750073_dbl_kind, 0.899014361224263_dbl_kind, 0.930415281371442_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,134) = (/ 0.889865004035781_dbl_kind, 0.892226459249918_dbl_kind, 0.893746896723777_dbl_kind, 0.89905743809635_dbl_kind, 0.930565469145523_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,135) = (/ 0.889874670504792_dbl_kind, 0.892241897683955_dbl_kind, 0.893767384395265_dbl_kind, 0.899100203843452_dbl_kind, 0.930714829745929_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,136) = (/ 0.889884226777591_dbl_kind, 0.892257169230691_dbl_kind, 0.893787633803127_dbl_kind, 0.899142663731108_dbl_kind, 0.930863371944046_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,137) = (/ 0.88989367467062_dbl_kind, 0.892272275692416_dbl_kind, 0.893807649881699_dbl_kind, 0.899184822931961_dbl_kind, 0.931011104351718_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,138) = (/ 0.889903015951649_dbl_kind, 0.892287218868419_dbl_kind, 0.893827437463713_dbl_kind, 0.899226686527304_dbl_kind, 0.931158035424871_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,139) = (/ 0.889912252341568_dbl_kind, 0.892302000555806_dbl_kind, 0.893847001282854_dbl_kind, 0.899268259508566_dbl_kind, 0.93130417346707_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,140) = (/ 0.889921385516117_dbl_kind, 0.892316622550138_dbl_kind, 0.89386634597621_dbl_kind, 0.899309546778707_dbl_kind, 0.93144952663298_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,141) = (/ 0.88993041710755_dbl_kind, 0.892331086645918_dbl_kind, 0.893885476086627_dbl_kind, 0.899350553153564_dbl_kind, 0.931594102931748_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,142) = (/ 0.88993934870625_dbl_kind, 0.892345394636909_dbl_kind, 0.893904396064976_dbl_kind, 0.89939128336313_dbl_kind, 0.931737910230305_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,143) = (/ 0.889948181862275_dbl_kind, 0.892359548316342_dbl_kind, 0.893923110272318_dbl_kind, 0.899431742052778_dbl_kind, 0.931880956256597_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,144) = (/ 0.889956918086858_dbl_kind, 0.892373549476983_dbl_kind, 0.893941622981989_dbl_kind, 0.899471933784446_dbl_kind, 0.932023248602722_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,145) = (/ 0.889965558853845_dbl_kind, 0.8923873999111_dbl_kind, 0.893959938381592_dbl_kind, 0.899511863037769_dbl_kind, 0.932164794728016_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,146) = (/ 0.889974105601087_dbl_kind, 0.892401101410333_dbl_kind, 0.893978060574913_dbl_kind, 0.899551534211174_dbl_kind, 0.932305601962053_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,147) = (/ 0.889982559731775_dbl_kind, 0.892414655765474_dbl_kind, 0.893995993583749_dbl_kind, 0.899590951622938_dbl_kind, 0.932445677507579_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,148) = (/ 0.889990922615729_dbl_kind, 0.892428064766164_dbl_kind, 0.89401374134966_dbl_kind, 0.899630119512222_dbl_kind, 0.932585028443373_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,149) = (/ 0.889999195590644_dbl_kind, 0.892441330200542_dbl_kind, 0.894031307735658_dbl_kind, 0.899669042040067_dbl_kind, 0.932723661727045_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,150) = (/ 0.890007379963272_dbl_kind, 0.892454453854806_dbl_kind, 0.894048696527809_dbl_kind, 0.899707723290373_dbl_kind, 0.93286158419777_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,151) = (/ 0.890015477010583_dbl_kind, 0.89246743751275_dbl_kind, 0.894065911436783_dbl_kind, 0.899746167270855_dbl_kind, 0.93299880257895_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,152) = (/ 0.890023487980864_dbl_kind, 0.892480282955243_dbl_kind, 0.894082956099329_dbl_kind, 0.89978437791398_dbl_kind, 0.933135323480825_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,153) = (/ 0.89003141409478_dbl_kind, 0.892492991959673_dbl_kind, 0.894099834079695_dbl_kind, 0.89982235907789_dbl_kind, 0.933271153403011_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,154) = (/ 0.890039256546397_dbl_kind, 0.892505566299365_dbl_kind, 0.894116548870995_dbl_kind, 0.89986011454731_dbl_kind, 0.933406298736986_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,155) = (/ 0.890047016504164_dbl_kind, 0.892518007742974_dbl_kind, 0.894133103896507_dbl_kind, 0.899897648034447_dbl_kind, 0.933540765768519_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,156) = (/ 0.890054695111852_dbl_kind, 0.892530318053858_dbl_kind, 0.894149502510941_dbl_kind, 0.89993496317988_dbl_kind, 0.933674560680028_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,157) = (/ 0.890062293489465_dbl_kind, 0.892542498989439_dbl_kind, 0.894165748001639_dbl_kind, 0.899972063553443_dbl_kind, 0.933807689552898_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,158) = (/ 0.890069812734101_dbl_kind, 0.892554552300554_dbl_kind, 0.894181843589748_dbl_kind, 0.900008952655102_dbl_kind, 0.933940158369741_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,159) = (/ 0.89007725392079_dbl_kind, 0.892566479730804_dbl_kind, 0.894197792431336_dbl_kind, 0.90004563391583_dbl_kind, 0.934071973016593_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,160) = (/ 0.890084618103295_dbl_kind, 0.892578283015897_dbl_kind, 0.894213597618481_dbl_kind, 0.900082110698477_dbl_kind, 0.934203139285071_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,161) = (/ 0.890091906314871_dbl_kind, 0.892589963883005_dbl_kind, 0.894229262180314_dbl_kind, 0.900118386298642_dbl_kind, 0.93433366287447_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,162) = (/ 0.890099119569008_dbl_kind, 0.892601524050109_dbl_kind, 0.894244789084037_dbl_kind, 0.90015446394554_dbl_kind, 0.934463549393815_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,163) = (/ 0.890106258860131_dbl_kind, 0.892612965225373_dbl_kind, 0.894260181235903_dbl_kind, 0.900190346802877_dbl_kind, 0.934592804363861_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,164) = (/ 0.890113325164272_dbl_kind, 0.892624289106512_dbl_kind, 0.894275441482172_dbl_kind, 0.900226037969713_dbl_kind, 0.934721433219048_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,165) = (/ 0.890120319439716_dbl_kind, 0.892635497380186_dbl_kind, 0.894290572610038_dbl_kind, 0.900261540481343_dbl_kind, 0.934849441309407_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,166) = (/ 0.890127242627623_dbl_kind, 0.8926465917214_dbl_kind, 0.894305577348532_dbl_kind, 0.90029685731017_dbl_kind, 0.934976833902422_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,167) = (/ 0.89013409565261_dbl_kind, 0.89265757379293_dbl_kind, 0.894320458369401_dbl_kind, 0.900331991366577_dbl_kind, 0.935103616184845_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,168) = (/ 0.890140879423323_dbl_kind, 0.892668445244759_dbl_kind, 0.894335218287973_dbl_kind, 0.900366945499816_dbl_kind, 0.935229793264468_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,169) = (/ 0.890147594832972_dbl_kind, 0.892679207713537_dbl_kind, 0.894349859663997_dbl_kind, 0.900401722498884_dbl_kind, 0.935355370171856_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,170) = (/ 0.890154242759848_dbl_kind, 0.892689862822069_dbl_kind, 0.894364385002467_dbl_kind, 0.900436325093411_dbl_kind, 0.935480351862039_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,171) = (/ 0.890160824067813_dbl_kind, 0.892700412178809_dbl_kind, 0.894378796754433_dbl_kind, 0.900470755954549_dbl_kind, 0.935604743216145_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,172) = (/ 0.890167339606775_dbl_kind, 0.892710857377391_dbl_kind, 0.8943930973178_dbl_kind, 0.900505017695863_dbl_kind, 0.935728549043026_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,173) = (/ 0.890173790213129_dbl_kind, 0.892721199996178_dbl_kind, 0.894407289038108_dbl_kind, 0.900539112874225_dbl_kind, 0.935851774080815_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,174) = (/ 0.890180176710189_dbl_kind, 0.892731441597832_dbl_kind, 0.894421374209307_dbl_kind, 0.90057304399071_dbl_kind, 0.93597442299846_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,175) = (/ 0.890186499908594_dbl_kind, 0.892741583728918_dbl_kind, 0.894435355074518_dbl_kind, 0.900606813491491_dbl_kind, 0.936096500397221_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,176) = (/ 0.890192760606695_dbl_kind, 0.892751627919514_dbl_kind, 0.894449233826792_dbl_kind, 0.900640423768745_dbl_kind, 0.936218010812117_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,177) = (/ 0.890198959590929_dbl_kind, 0.892761575682869_dbl_kind, 0.894463012609848_dbl_kind, 0.90067387716155_dbl_kind, 0.936338958713361_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,178) = (/ 0.890205097636167_dbl_kind, 0.892771428515064_dbl_kind, 0.894476693518821_dbl_kind, 0.900707175956787_dbl_kind, 0.936459348507738_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,179) = (/ 0.890211175506047_dbl_kind, 0.892781187894709_dbl_kind, 0.894490278600985_dbl_kind, 0.900740322390046_dbl_kind, 0.936579184539964_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,180) = (/ 0.890217193953302_dbl_kind, 0.892790855282665_dbl_kind, 0.894503769856489_dbl_kind, 0.900773318646522_dbl_kind, 0.936698471093999_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,181) = (/ 0.890223153720052_dbl_kind, 0.892800432121781_dbl_kind, 0.894517169239074_dbl_kind, 0.900806166861925_dbl_kind, 0.936817212394342_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,182) = (/ 0.890229055538101_dbl_kind, 0.892809919836661_dbl_kind, 0.894530478656791_dbl_kind, 0.900838869123377_dbl_kind, 0.936935412607287_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,183) = (/ 0.890234900129204_dbl_kind, 0.892819319833457_dbl_kind, 0.894543699972717_dbl_kind, 0.900871427470311_dbl_kind, 0.937053075842141_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,184) = (/ 0.890240688205337_dbl_kind, 0.892828633499672_dbl_kind, 0.894556835005664_dbl_kind, 0.900903843895372_dbl_kind, 0.937170206152427_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,185) = (/ 0.890246420468931_dbl_kind, 0.892837862204005_dbl_kind, 0.894569885530885_dbl_kind, 0.900936120345312_dbl_kind, 0.937286807537053_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,186) = (/ 0.890252097613118_dbl_kind, 0.892847007296195_dbl_kind, 0.894582853280778_dbl_kind, 0.900968258721879_dbl_kind, 0.937402883941437_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,187) = (/ 0.890257720321949_dbl_kind, 0.892856070106906_dbl_kind, 0.89459573994559_dbl_kind, 0.901000260882711_dbl_kind, 0.937518439258629_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,188) = (/ 0.890263289270606_dbl_kind, 0.89286505194762_dbl_kind, 0.894608547174113_dbl_kind, 0.901032128642219_dbl_kind, 0.93763347733039_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,189) = (/ 0.890268805125602_dbl_kind, 0.892873954110557_dbl_kind, 0.894621276574383_dbl_kind, 0.901063863772464_dbl_kind, 0.937748001948244_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,190) = (/ 0.890274268544973_dbl_kind, 0.892882777868608_dbl_kind, 0.89463392971437_dbl_kind, 0.901095468004041_dbl_kind, 0.937862016854509_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,191) = (/ 0.890279680178459_dbl_kind, 0.892891524475292_dbl_kind, 0.894646508122675_dbl_kind, 0.90112694302694_dbl_kind, 0.937975525743309_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,192) = (/ 0.890285040667673_dbl_kind, 0.892900195164733_dbl_kind, 0.894659013289215_dbl_kind, 0.901158290491414_dbl_kind, 0.938088532261543_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,193) = (/ 0.890290350646263_dbl_kind, 0.892908791151641_dbl_kind, 0.894671446665914_dbl_kind, 0.90118951200884_dbl_kind, 0.938201040009846_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,194) = (/ 0.890295610740071_dbl_kind, 0.892917313631333_dbl_kind, 0.894683809667384_dbl_kind, 0.901220609152562_dbl_kind, 0.938313052543525_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,195) = (/ 0.890300821567272_dbl_kind, 0.892925763779745_dbl_kind, 0.894696103671609_dbl_kind, 0.901251583458742_dbl_kind, 0.938424573373459_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,196) = (/ 0.890305983738519_dbl_kind, 0.892934142753477_dbl_kind, 0.894708330020626_dbl_kind, 0.901282436427189_dbl_kind, 0.938535605966993_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,197) = (/ 0.890311097857072_dbl_kind, 0.892942451689843_dbl_kind, 0.894720490021202_dbl_kind, 0.90131316952219_dbl_kind, 0.938646153748803_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,198) = (/ 0.89031616451892_dbl_kind, 0.892950691706944_dbl_kind, 0.894732584945503_dbl_kind, 0.901343784173329_dbl_kind, 0.938756220101732_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,199) = (/ 0.890321184312904_dbl_kind, 0.892958863903745_dbl_kind, 0.894744616031772_dbl_kind, 0.901374281776295_dbl_kind, 0.938865808367626_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,200) = (/ 0.890326157820826_dbl_kind, 0.892966969360168_dbl_kind, 0.894756584484991_dbl_kind, 0.901404663693682_dbl_kind, 0.938974921848123_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,201) = (/ 0.890331085617559_dbl_kind, 0.892975009137204_dbl_kind, 0.894768491477549_dbl_kind, 0.901434931255784_dbl_kind, 0.939083563805445_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,202) = (/ 0.890335968271143_dbl_kind, 0.892982984277024_dbl_kind, 0.894780338149899_dbl_kind, 0.901465085761369_dbl_kind, 0.939191737463164_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,203) = (/ 0.890340806342889_dbl_kind, 0.892990895803112_dbl_kind, 0.894792125611219_dbl_kind, 0.901495128478456_dbl_kind, 0.93929944600694_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,204) = (/ 0.890345600387466_dbl_kind, 0.892998744720399_dbl_kind, 0.89480385494006_dbl_kind, 0.901525060645072_dbl_kind, 0.939406692585242_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,205) = (/ 0.890350350952986_dbl_kind, 0.893006532015415_dbl_kind, 0.894815527184995_dbl_kind, 0.901554883470003_dbl_kind, 0.93951348031008_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,206) = (/ 0.890355058581094_dbl_kind, 0.893014258656449_dbl_kind, 0.894827143365265_dbl_kind, 0.901584598133529_dbl_kind, 0.939619812257674_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,207) = (/ 0.890359723807043_dbl_kind, 0.893021925593706_dbl_kind, 0.894838704471412_dbl_kind, 0.901614205788157_dbl_kind, 0.939725691469137_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,208) = (/ 0.890364347159771_dbl_kind, 0.893029533759487_dbl_kind, 0.894850211465919_dbl_kind, 0.901643707559329_dbl_kind, 0.939831120951139_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,209) = (/ 0.890368929161974_dbl_kind, 0.893037084068371_dbl_kind, 0.894861665283834_dbl_kind, 0.901673104546136_dbl_kind, 0.939936103676537_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,210) = (/ 0.890373470330174_dbl_kind, 0.893044577417399_dbl_kind, 0.89487306683339_dbl_kind, 0.901702397822_dbl_kind, 0.940040642585012_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,211) = (/ 0.890377971174789_dbl_kind, 0.893052014686269_dbl_kind, 0.894884416996629_dbl_kind, 0.901731588435364_dbl_kind, 0.940144740583677_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,212) = (/ 0.890382432200192_dbl_kind, 0.893059396737539_dbl_kind, 0.894895716630005_dbl_kind, 0.901760677410354_dbl_kind, 0.940248400547673_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,213) = (/ 0.890386853904778_dbl_kind, 0.893066724416829_dbl_kind, 0.894906966564992_dbl_kind, 0.901789665747438_dbl_kind, 0.940351625320754_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,214) = (/ 0.890391236781017_dbl_kind, 0.893073998553031_dbl_kind, 0.894918167608682_dbl_kind, 0.901818554424072_dbl_kind, 0.940454417715846_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,215) = (/ 0.89039558131552_dbl_kind, 0.89308121995853_dbl_kind, 0.894929320544375_dbl_kind, 0.901847344395331_dbl_kind, 0.940556780515612_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,216) = (/ 0.890399887989083_dbl_kind, 0.893088389429413_dbl_kind, 0.894940426132164_dbl_kind, 0.901876036594527_dbl_kind, 0.940658716472989_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,217) = (/ 0.890404157276751_dbl_kind, 0.893095507745699_dbl_kind, 0.894951485109509_dbl_kind, 0.901904631933822_dbl_kind, 0.940760228311716_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,218) = (/ 0.890408389647863_dbl_kind, 0.893102575671562_dbl_kind, 0.894962498191815_dbl_kind, 0.901933131304815_dbl_kind, 0.940861318726845_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,219) = (/ 0.890412585566105_dbl_kind, 0.893109593955558_dbl_kind, 0.894973466072985_dbl_kind, 0.901961535579134_dbl_kind, 0.94096199038525_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,220) = (/ 0.89041674548956_dbl_kind, 0.893116563330856_dbl_kind, 0.894984389425981_dbl_kind, 0.901989845609002_dbl_kind, 0.941062245926115_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,221) = (/ 0.890420869870753_dbl_kind, 0.893123484515476_dbl_kind, 0.894995268903368_dbl_kind, 0.902018062227796_dbl_kind, 0.941162087961412_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,222) = (/ 0.890424959156701_dbl_kind, 0.893130358212517_dbl_kind, 0.895006105137859_dbl_kind, 0.902046186250595_dbl_kind, 0.941261519076372_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,223) = (/ 0.890429013788957_dbl_kind, 0.893137185110393_dbl_kind, 0.895016898742837_dbl_kind, 0.90207421847471_dbl_kind, 0.941360541829935_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,224) = (/ 0.890433034203655_dbl_kind, 0.893143965883078_dbl_kind, 0.895027650312887_dbl_kind, 0.902102159680212_dbl_kind, 0.941459158755206_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,225) = (/ 0.890437020831557_dbl_kind, 0.893150701190335_dbl_kind, 0.895038360424307_dbl_kind, 0.902130010630436_dbl_kind, 0.941557372359877_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,226) = (/ 0.890440974098092_dbl_kind, 0.893157391677959_dbl_kind, 0.895049029635615_dbl_kind, 0.902157772072483_dbl_kind, 0.941655185126665_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,227) = (/ 0.890444894423402_dbl_kind, 0.893164037978018_dbl_kind, 0.895059658488047_dbl_kind, 0.902185444737706_dbl_kind, 0.941752599513719_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,228) = (/ 0.890448782222384_dbl_kind, 0.893170640709085_dbl_kind, 0.895070247506048_dbl_kind, 0.90221302934218_dbl_kind, 0.941849617955025_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,229) = (/ 0.890452637904733_dbl_kind, 0.893177200476486_dbl_kind, 0.895080797197752_dbl_kind, 0.90224052658717_dbl_kind, 0.941946242860804_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,230) = (/ 0.890456461874981_dbl_kind, 0.893183717872529_dbl_kind, 0.895091308055457_dbl_kind, 0.902267937159579_dbl_kind, 0.9420424766179_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,231) = (/ 0.890460254532541_dbl_kind, 0.893190193476748_dbl_kind, 0.895101780556084_dbl_kind, 0.90229526173239_dbl_kind, 0.942138321590155_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,232) = (/ 0.890464016271748_dbl_kind, 0.893196627856137_dbl_kind, 0.895112215161639_dbl_kind, 0.90232250096509_dbl_kind, 0.942233780118777_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,233) = (/ 0.890467747481897_dbl_kind, 0.893203021565388_dbl_kind, 0.895122612319652_dbl_kind, 0.902349655504094_dbl_kind, 0.9423288545227_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,234) = (/ 0.890471448547291_dbl_kind, 0.893209375147124_dbl_kind, 0.895132972463622_dbl_kind, 0.902376725983144_dbl_kind, 0.942423547098941_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,235) = (/ 0.89047511984727_dbl_kind, 0.89321568913213_dbl_kind, 0.895143296013439_dbl_kind, 0.902403713023711_dbl_kind, 0.942517860122935_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,236) = (/ 0.890478761756264_dbl_kind, 0.893221964039592_dbl_kind, 0.895153583375812_dbl_kind, 0.902430617235375_dbl_kind, 0.942611795848882_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,237) = (/ 0.890482374643827_dbl_kind, 0.893228200377318_dbl_kind, 0.895163834944673_dbl_kind, 0.902457439216201_dbl_kind, 0.942705356510065_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,238) = (/ 0.890485958874678_dbl_kind, 0.893234398641973_dbl_kind, 0.895174051101583_dbl_kind, 0.902484179553104_dbl_kind, 0.942798544319174_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,239) = (/ 0.890489514808742_dbl_kind, 0.893240559319301_dbl_kind, 0.895184232216127_dbl_kind, 0.902510838822203_dbl_kind, 0.942891361468627_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,240) = (/ 0.890493042801193_dbl_kind, 0.893246682884354_dbl_kind, 0.895194378646298_dbl_kind, 0.90253741758916_dbl_kind, 0.942983810130863_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,241) = (/ 0.890496543202492_dbl_kind, 0.893252769801706_dbl_kind, 0.895204490738872_dbl_kind, 0.902563916409519_dbl_kind, 0.943075892458654_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,242) = (/ 0.89050001635843_dbl_kind, 0.893258820525683_dbl_kind, 0.89521456882978_dbl_kind, 0.90259033582903_dbl_kind, 0.943167610585391_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,243) = (/ 0.890503462610168_dbl_kind, 0.893264835500571_dbl_kind, 0.895224613244465_dbl_kind, 0.902616676383961_dbl_kind, 0.94325896662537_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,244) = (/ 0.890506882294278_dbl_kind, 0.893270815160839_dbl_kind, 0.895234624298231_dbl_kind, 0.902642938601404_dbl_kind, 0.943349962674082_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,245) = (/ 0.890510275742783_dbl_kind, 0.893276759931343_dbl_kind, 0.895244602296592_dbl_kind, 0.902669122999574_dbl_kind, 0.943440600808471_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,246) = (/ 0.890513643283203_dbl_kind, 0.893282670227544_dbl_kind, 0.895254547535598_dbl_kind, 0.902695230088093_dbl_kind, 0.943530883087214_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,247) = (/ 0.890516985238593_dbl_kind, 0.89328854645571_dbl_kind, 0.895264460302168_dbl_kind, 0.902721260368271_dbl_kind, 0.943620811550982_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,248) = (/ 0.890520301927583_dbl_kind, 0.89329438901312_dbl_kind, 0.895274340874405_dbl_kind, 0.902747214333375_dbl_kind, 0.943710388222692_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,249) = (/ 0.890523593664424_dbl_kind, 0.893300198288269_dbl_kind, 0.895284189521903_dbl_kind, 0.902773092468891_dbl_kind, 0.943799615107758_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,250) = (/ 0.89052686075903_dbl_kind, 0.893305974661065_dbl_kind, 0.895294006506055_dbl_kind, 0.902798895252778_dbl_kind, 0.943888494194338_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,251) = (/ 0.890530103517015_dbl_kind, 0.893311718503024_dbl_kind, 0.89530379208034_dbl_kind, 0.902824623155714_dbl_kind, 0.943977027453582_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,252) = (/ 0.890533322239742_dbl_kind, 0.893317430177464_dbl_kind, 0.895313546490614_dbl_kind, 0.90285027664133_dbl_kind, 0.944065216839851_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,253) = (/ 0.890536517224361_dbl_kind, 0.893323110039694_dbl_kind, 0.895323269975387_dbl_kind, 0.902875856166446_dbl_kind, 0.94415306429096_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,254) = (/ 0.890539688763852_dbl_kind, 0.893328758437203_dbl_kind, 0.895332962766091_dbl_kind, 0.902901362181291_dbl_kind, 0.944240571728397_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,255) = (/ 0.89054283714707_dbl_kind, 0.893334375709842_dbl_kind, 0.895342625087346_dbl_kind, 0.902926795129716_dbl_kind, 0.944327741057547_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,256) = (/ 0.890545962658789_dbl_kind, 0.893339962190006_dbl_kind, 0.895352257157217_dbl_kind, 0.902952155449404_dbl_kind, 0.944414574167906_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,257) = (/ 0.890549065579738_dbl_kind, 0.893345518202811_dbl_kind, 0.895361859187458_dbl_kind, 0.902977443572076_dbl_kind, 0.944501072933293_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,258) = (/ 0.890552146186652_dbl_kind, 0.89335104406627_dbl_kind, 0.895371431383759_dbl_kind, 0.903002659923676_dbl_kind, 0.944587239212056_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,259) = (/ 0.890555204752312_dbl_kind, 0.893356540091464_dbl_kind, 0.895380973945975_dbl_kind, 0.903027804924568_dbl_kind, 0.94467307484728_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,260) = (/ 0.890558241545586_dbl_kind, 0.893362006582714_dbl_kind, 0.895390487068356_dbl_kind, 0.903052878989715_dbl_kind, 0.944758581666974_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,261) = (/ 0.890561256831479_dbl_kind, 0.89336744383774_dbl_kind, 0.895399970939771_dbl_kind, 0.90307788252885_dbl_kind, 0.94484376148428_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,262) = (/ 0.890564250871169_dbl_kind, 0.893372852147832_dbl_kind, 0.895409425743915_dbl_kind, 0.903102815946653_dbl_kind, 0.944928616097656_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,263) = (/ 0.890567223922053_dbl_kind, 0.893378231798002_dbl_kind, 0.895418851659524_dbl_kind, 0.90312767964291_dbl_kind, 0.94501314729106_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,264) = (/ 0.890570176237793_dbl_kind, 0.893383583067148_dbl_kind, 0.895428248860569_dbl_kind, 0.903152474012669_dbl_kind, 0.945097356834142_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,265) = (/ 0.890573108068358_dbl_kind, 0.893388906228202_dbl_kind, 0.895437617516459_dbl_kind, 0.9031771994464_dbl_kind, 0.945181246482415_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,266) = (/ 0.890576019660067_dbl_kind, 0.893394201548282_dbl_kind, 0.895446957792221_dbl_kind, 0.903201856330132_dbl_kind, 0.945264817977436_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,267) = (/ 0.890578911255631_dbl_kind, 0.893399469288844_dbl_kind, 0.89545626984869_dbl_kind, 0.903226445045602_dbl_kind, 0.945348073046979_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,268) = (/ 0.8905817830942_dbl_kind, 0.893404709705821_dbl_kind, 0.895465553842682_dbl_kind, 0.903250965970389_dbl_kind, 0.945431013405204_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,269) = (/ 0.890584635411406_dbl_kind, 0.893409923049766_dbl_kind, 0.895474809927167_dbl_kind, 0.903275419478046_dbl_kind, 0.945513640752813_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,270) = (/ 0.890587468439402_dbl_kind, 0.893415109565995_dbl_kind, 0.895484038251432_dbl_kind, 0.903299805938226_dbl_kind, 0.945595956777235_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,271) = (/ 0.890590282406912_dbl_kind, 0.893420269494718_dbl_kind, 0.895493238961243_dbl_kind, 0.903324125716805_dbl_kind, 0.94567796315276_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,272) = (/ 0.890593077539268_dbl_kind, 0.893425403071177_dbl_kind, 0.895502412199003_dbl_kind, 0.903348379176001_dbl_kind, 0.945759661540713_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,273) = (/ 0.890595854058458_dbl_kind, 0.893430510525769_dbl_kind, 0.895511558103893_dbl_kind, 0.903372566674482_dbl_kind, 0.945841053589605_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,274) = (/ 0.890598612183167_dbl_kind, 0.893435592084184_dbl_kind, 0.895520676812023_dbl_kind, 0.903396688567483_dbl_kind, 0.945922140935279_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,275) = (/ 0.89060135212882_dbl_kind, 0.893440647967518_dbl_kind, 0.895529768456569_dbl_kind, 0.903420745206905_dbl_kind, 0.946002925201057_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,276) = (/ 0.890604074107625_dbl_kind, 0.893445678392407_dbl_kind, 0.895538833167909_dbl_kind, 0.903444736941421_dbl_kind, 0.946083407997899_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,277) = (/ 0.890606778328615_dbl_kind, 0.893450683571135_dbl_kind, 0.895547871073749_dbl_kind, 0.903468664116564_dbl_kind, 0.946163590924526_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,278) = (/ 0.890609464997693_dbl_kind, 0.893455663711758_dbl_kind, 0.895556882299253_dbl_kind, 0.903492527074833_dbl_kind, 0.946243475567573_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,279) = (/ 0.89061213431767_dbl_kind, 0.893460619018216_dbl_kind, 0.895565866967161_dbl_kind, 0.903516326155773_dbl_kind, 0.946323063501727_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,280) = (/ 0.890614786488313_dbl_kind, 0.893465549690444_dbl_kind, 0.895574825197902_dbl_kind, 0.903540061696068_dbl_kind, 0.946402356289853_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,281) = (/ 0.890617421706381_dbl_kind, 0.89347045592448_dbl_kind, 0.895583757109712_dbl_kind, 0.903563734029617_dbl_kind, 0.946481355483134_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,282) = (/ 0.890620040165671_dbl_kind, 0.893475337912574_dbl_kind, 0.895592662818739_dbl_kind, 0.903587343487623_dbl_kind, 0.946560062621202_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,283) = (/ 0.890622642057059_dbl_kind, 0.893480195843288_dbl_kind, 0.895601542439144_dbl_kind, 0.903610890398659_dbl_kind, 0.946638479232261_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,284) = (/ 0.890625227568537_dbl_kind, 0.893485029901599_dbl_kind, 0.895610396083206_dbl_kind, 0.903634375088752_dbl_kind, 0.946716606833219_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,285) = (/ 0.89062779688526_dbl_kind, 0.893489840268999_dbl_kind, 0.895619223861414_dbl_kind, 0.903657797881446_dbl_kind, 0.946794446929803_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,286) = (/ 0.890630350189585_dbl_kind, 0.89349462712359_dbl_kind, 0.895628025882561_dbl_kind, 0.903681159097874_dbl_kind, 0.946872001016691_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,287) = (/ 0.890632887661107_dbl_kind, 0.893499390640179_dbl_kind, 0.89563680225383_dbl_kind, 0.903704459056823_dbl_kind, 0.946949270577626_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,288) = (/ 0.890635409476708_dbl_kind, 0.893504130990368_dbl_kind, 0.895645553080884_dbl_kind, 0.903727698074793_dbl_kind, 0.94702625708553_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,289) = (/ 0.890637915810587_dbl_kind, 0.893508848342646_dbl_kind, 0.895654278467945_dbl_kind, 0.903750876466061_dbl_kind, 0.947102962002626_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,290) = (/ 0.890640406834309_dbl_kind, 0.893513542862474_dbl_kind, 0.895662978517871_dbl_kind, 0.903773994542738_dbl_kind, 0.94717938678055_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,291) = (/ 0.890642882716836_dbl_kind, 0.893518214712372_dbl_kind, 0.895671653332234_dbl_kind, 0.903797052614822_dbl_kind, 0.947255532860463_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,292) = (/ 0.890645343624572_dbl_kind, 0.893522864052_dbl_kind, 0.895680303011391_dbl_kind, 0.903820050990251_dbl_kind, 0.947331401673153_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,293) = (/ 0.890647789721401_dbl_kind, 0.893527491038242_dbl_kind, 0.895688927654552_dbl_kind, 0.903842989974958_dbl_kind, 0.947406994639158_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,294) = (/ 0.890650221168721_dbl_kind, 0.893532095825279_dbl_kind, 0.895697527359848_dbl_kind, 0.903865869872914_dbl_kind, 0.947482313168859_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,295) = (/ 0.890652638125486_dbl_kind, 0.893536678564673_dbl_kind, 0.895706102224396_dbl_kind, 0.903888690986177_dbl_kind, 0.947557358662593_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,296) = (/ 0.890655040748246_dbl_kind, 0.893541239405435_dbl_kind, 0.895714652344355_dbl_kind, 0.903911453614939_dbl_kind, 0.94763213251075_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,297) = (/ 0.890657429191175_dbl_kind, 0.893545778494104_dbl_kind, 0.89572317781499_dbl_kind, 0.903934158057563_dbl_kind, 0.947706636093878_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,298) = (/ 0.89065980360612_dbl_kind, 0.89355029597481_dbl_kind, 0.895731678730721_dbl_kind, 0.90395680461063_dbl_kind, 0.947780870782784_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,299) = (/ 0.890662164142628_dbl_kind, 0.89355479198935_dbl_kind, 0.895740155185182_dbl_kind, 0.903979393568975_dbl_kind, 0.94785483793863_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,300) = (/ 0.890664510947988_dbl_kind, 0.893559266677254_dbl_kind, 0.895748607271269_dbl_kind, 0.904001925225723_dbl_kind, 0.947928538913026_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,301) = (/ 0.890666844167263_dbl_kind, 0.893563720175846_dbl_kind, 0.895757035081189_dbl_kind, 0.904024399872328_dbl_kind, 0.948001975048135_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,302) = (/ 0.890669163943329_dbl_kind, 0.89356815262031_dbl_kind, 0.895765438706507_dbl_kind, 0.904046817798608_dbl_kind, 0.948075147676756_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,303) = (/ 0.890671470416907_dbl_kind, 0.893572564143755_dbl_kind, 0.895773818238184_dbl_kind, 0.904069179292773_dbl_kind, 0.948148058122426_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,304) = (/ 0.890673763726599_dbl_kind, 0.893576954877268_dbl_kind, 0.895782173766632_dbl_kind, 0.904091484641461_dbl_kind, 0.948220707699503_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,305) = (/ 0.890676044008924_dbl_kind, 0.893581324949981_dbl_kind, 0.89579050538174_dbl_kind, 0.904113734129765_dbl_kind, 0.94829309771326_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,306) = (/ 0.890678311398349_dbl_kind, 0.893585674489117_dbl_kind, 0.895798813172922_dbl_kind, 0.904135928041265_dbl_kind, 0.948365229459975_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,307) = (/ 0.890680566027323_dbl_kind, 0.893590003620057_dbl_kind, 0.895807097229146_dbl_kind, 0.904158066658051_dbl_kind, 0.94843710422701_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,308) = (/ 0.890682808026313_dbl_kind, 0.893594312466386_dbl_kind, 0.895815357638973_dbl_kind, 0.904180150260754_dbl_kind, 0.948508723292908_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,309) = (/ 0.890685037523831_dbl_kind, 0.893598601149946_dbl_kind, 0.895823594490591_dbl_kind, 0.904202179128567_dbl_kind, 0.948580087927469_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,310) = (/ 0.890687254646475_dbl_kind, 0.893602869790888_dbl_kind, 0.89583180787184_dbl_kind, 0.904224153539272_dbl_kind, 0.948651199391836_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,311) = (/ 0.89068945951895_dbl_kind, 0.893607118507721_dbl_kind, 0.895839997870247_dbl_kind, 0.904246073769259_dbl_kind, 0.948722058938578_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,312) = (/ 0.89069165226411_dbl_kind, 0.893611347417361_dbl_kind, 0.895848164573048_dbl_kind, 0.904267940093552_dbl_kind, 0.948792667811767_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,313) = (/ 0.890693833002981_dbl_kind, 0.893615556635173_dbl_kind, 0.895856308067224_dbl_kind, 0.904289752785828_dbl_kind, 0.94886302724707_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,314) = (/ 0.890696001854796_dbl_kind, 0.893619746275021_dbl_kind, 0.895864428439515_dbl_kind, 0.904311512118437_dbl_kind, 0.94893313847181_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,315) = (/ 0.890698158937023_dbl_kind, 0.893623916449308_dbl_kind, 0.89587252577645_dbl_kind, 0.904333218362419_dbl_kind, 0.949003002705056_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,316) = (/ 0.890700304365399_dbl_kind, 0.893628067269021_dbl_kind, 0.895880600164368_dbl_kind, 0.904354871787526_dbl_kind, 0.949072621157696_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,317) = (/ 0.890702438253952_dbl_kind, 0.893632198843771_dbl_kind, 0.895888651689437_dbl_kind, 0.904376472662236_dbl_kind, 0.949141995032512_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,318) = (/ 0.890704560715037_dbl_kind, 0.893636311281832_dbl_kind, 0.895896680437677_dbl_kind, 0.904398021253769_dbl_kind, 0.949211125524257_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,319) = (/ 0.89070667185936_dbl_kind, 0.893640404690182_dbl_kind, 0.895904686494977_dbl_kind, 0.904419517828107_dbl_kind, 0.949280013819725_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,320) = (/ 0.890708771796011_dbl_kind, 0.893644479174539_dbl_kind, 0.89591266994711_dbl_kind, 0.904440962650005_dbl_kind, 0.949348661097827_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,321) = (/ 0.890710860632484_dbl_kind, 0.893648534839396_dbl_kind, 0.895920630879752_dbl_kind, 0.904462355983005_dbl_kind, 0.94941706852966_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,322) = (/ 0.890712938474712_dbl_kind, 0.893652571788059_dbl_kind, 0.895928569378498_dbl_kind, 0.904483698089453_dbl_kind, 0.949485237278577_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,323) = (/ 0.890715005427091_dbl_kind, 0.89365659012268_dbl_kind, 0.895936485528873_dbl_kind, 0.904504989230507_dbl_kind, 0.949553168500265_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,324) = (/ 0.890717061592505_dbl_kind, 0.893660589944291_dbl_kind, 0.895944379416349_dbl_kind, 0.904526229666154_dbl_kind, 0.949620863342799_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,325) = (/ 0.890719107072354_dbl_kind, 0.893664571352833_dbl_kind, 0.895952251126352_dbl_kind, 0.904547419655221_dbl_kind, 0.949688322946724_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,326) = (/ 0.890721141966578_dbl_kind, 0.893668534447192_dbl_kind, 0.895960100744283_dbl_kind, 0.904568559455382_dbl_kind, 0.949755548445114_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,327) = (/ 0.890723166373687_dbl_kind, 0.893672479325226_dbl_kind, 0.895967928355515_dbl_kind, 0.904589649323173_dbl_kind, 0.949822540963639_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,328) = (/ 0.890725180390778_dbl_kind, 0.893676406083793_dbl_kind, 0.895975734045416_dbl_kind, 0.904610689514001_dbl_kind, 0.949889301620638_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,329) = (/ 0.890727184113566_dbl_kind, 0.893680314818784_dbl_kind, 0.89598351789935_dbl_kind, 0.904631680282152_dbl_kind, 0.94995583152717_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,330) = (/ 0.890729177636404_dbl_kind, 0.893684205625145_dbl_kind, 0.895991280002688_dbl_kind, 0.904652621880803_dbl_kind, 0.950022131787091_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,331) = (/ 0.890731161052312_dbl_kind, 0.893688078596907_dbl_kind, 0.895999020440814_dbl_kind, 0.904673514562028_dbl_kind, 0.950088203497112_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,332) = (/ 0.890733134452994_dbl_kind, 0.893691933827211_dbl_kind, 0.896006739299135_dbl_kind, 0.904694358576808_dbl_kind, 0.950154047746855_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,333) = (/ 0.890735097928863_dbl_kind, 0.893695771408333_dbl_kind, 0.896014436663085_dbl_kind, 0.904715154175037_dbl_kind, 0.950219665618927_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,334) = (/ 0.890737051569067_dbl_kind, 0.893699591431706_dbl_kind, 0.896022112618128_dbl_kind, 0.904735901605532_dbl_kind, 0.950285058188969_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,335) = (/ 0.890738995461507_dbl_kind, 0.893703393987945_dbl_kind, 0.896029767249769_dbl_kind, 0.904756601116038_dbl_kind, 0.950350226525722_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,336) = (/ 0.89074092969286_dbl_kind, 0.89370717916687_dbl_kind, 0.896037400643554_dbl_kind, 0.904777252953238_dbl_kind, 0.950415171691088_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,337) = (/ 0.890742854348601_dbl_kind, 0.893710947057527_dbl_kind, 0.896045012885075_dbl_kind, 0.904797857362753_dbl_kind, 0.950479894740183_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,338) = (/ 0.890744769513023_dbl_kind, 0.893714697748209_dbl_kind, 0.896052604059973_dbl_kind, 0.904818414589156_dbl_kind, 0.950544396721397_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,339) = (/ 0.890746675269259_dbl_kind, 0.893718431326476_dbl_kind, 0.896060174253943_dbl_kind, 0.904838924875972_dbl_kind, 0.950608678676456_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,340) = (/ 0.890748571699301_dbl_kind, 0.893722147879176_dbl_kind, 0.896067723552733_dbl_kind, 0.904859388465686_dbl_kind, 0.950672741640472_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,341) = (/ 0.890750458884021_dbl_kind, 0.893725847492461_dbl_kind, 0.896075252042149_dbl_kind, 0.904879805599747_dbl_kind, 0.950736586642002_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,342) = (/ 0.89075233690319_dbl_kind, 0.893729530251809_dbl_kind, 0.896082759808054_dbl_kind, 0.904900176518576_dbl_kind, 0.950800214703099_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,343) = (/ 0.890754205835497_dbl_kind, 0.893733196242042_dbl_kind, 0.896090246936371_dbl_kind, 0.904920501461565_dbl_kind, 0.950863626839376_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,344) = (/ 0.890756065758568_dbl_kind, 0.893736845547337_dbl_kind, 0.896097713513084_dbl_kind, 0.904940780667089_dbl_kind, 0.950926824060051_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,345) = (/ 0.890757916748985_dbl_kind, 0.89374047825125_dbl_kind, 0.896105159624234_dbl_kind, 0.904961014372503_dbl_kind, 0.950989807368005_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,346) = (/ 0.890759758882305_dbl_kind, 0.893744094436729_dbl_kind, 0.896112585355927_dbl_kind, 0.904981202814152_dbl_kind, 0.951052577759832_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,347) = (/ 0.890761592233074_dbl_kind, 0.893747694186127_dbl_kind, 0.896119990794325_dbl_kind, 0.905001346227371_dbl_kind, 0.951115136225895_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,348) = (/ 0.890763416874849_dbl_kind, 0.893751277581222_dbl_kind, 0.896127376025653_dbl_kind, 0.90502144484649_dbl_kind, 0.951177483750372_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,349) = (/ 0.890765232880216_dbl_kind, 0.893754844703226_dbl_kind, 0.89613474113619_dbl_kind, 0.905041498904837_dbl_kind, 0.951239621311313_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,350) = (/ 0.890767040320799_dbl_kind, 0.893758395632805_dbl_kind, 0.896142086212277_dbl_kind, 0.905061508634744_dbl_kind, 0.951301549880687_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,351) = (/ 0.890768839267288_dbl_kind, 0.893761930450085_dbl_kind, 0.896149411340308_dbl_kind, 0.905081474267545_dbl_kind, 0.951363270424429_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,352) = (/ 0.890770629789444_dbl_kind, 0.893765449234672_dbl_kind, 0.896156716606732_dbl_kind, 0.905101396033585_dbl_kind, 0.951424783902502_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,353) = (/ 0.890772411956123_dbl_kind, 0.893768952065659_dbl_kind, 0.896164002098047_dbl_kind, 0.905121274162218_dbl_kind, 0.951486091268927_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,354) = (/ 0.89077418583529_dbl_kind, 0.893772439021644_dbl_kind, 0.896171267900807_dbl_kind, 0.905141108881813_dbl_kind, 0.951547193471846_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,355) = (/ 0.890775951494029_dbl_kind, 0.893775910180735_dbl_kind, 0.896178514101606_dbl_kind, 0.905160900419754_dbl_kind, 0.951608091453564_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,356) = (/ 0.890777708998566_dbl_kind, 0.893779365620565_dbl_kind, 0.896185740787088_dbl_kind, 0.905180649002446_dbl_kind, 0.951668786150597_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,357) = (/ 0.89077945841428_dbl_kind, 0.893782805418303_dbl_kind, 0.896192948043937_dbl_kind, 0.905200354855312_dbl_kind, 0.951729278493722_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,358) = (/ 0.890781199805715_dbl_kind, 0.893786229650663_dbl_kind, 0.896200135958875_dbl_kind, 0.905220018202802_dbl_kind, 0.951789569408015_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,359) = (/ 0.890782933236598_dbl_kind, 0.893789638393914_dbl_kind, 0.896207304618661_dbl_kind, 0.905239639268389_dbl_kind, 0.951849659812906_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,360) = (/ 0.890784658769853_dbl_kind, 0.893793031723891_dbl_kind, 0.896214454110085_dbl_kind, 0.905259218274577_dbl_kind, 0.951909550622219_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,361) = (/ 0.890786376467611_dbl_kind, 0.893796409716005_dbl_kind, 0.896221584519967_dbl_kind, 0.905278755442895_dbl_kind, 0.951969242744217_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,362) = (/ 0.890788086391227_dbl_kind, 0.893799772445248_dbl_kind, 0.896228695935151_dbl_kind, 0.905298250993909_dbl_kind, 0.952028737081647_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,363) = (/ 0.890789788601292_dbl_kind, 0.893803119986207_dbl_kind, 0.896235788442503_dbl_kind, 0.905317705147216_dbl_kind, 0.952088034531788_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,364) = (/ 0.890791483157644_dbl_kind, 0.893806452413067_dbl_kind, 0.896242862128907_dbl_kind, 0.905337118121449_dbl_kind, 0.952147135986491_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,365) = (/ 0.890793170119385_dbl_kind, 0.893809769799623_dbl_kind, 0.896249917081262_dbl_kind, 0.905356490134278_dbl_kind, 0.952206042332215_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,366) = (/ 0.890794849544888_dbl_kind, 0.893813072219288_dbl_kind, 0.896256953386471_dbl_kind, 0.905375821402412_dbl_kind, 0.952264754450083_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,367) = (/ 0.890796521491812_dbl_kind, 0.893816359745097_dbl_kind, 0.896263971131448_dbl_kind, 0.905395112141599_dbl_kind, 0.952323273215913_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,368) = (/ 0.890798186017116_dbl_kind, 0.893819632449717_dbl_kind, 0.896270970403106_dbl_kind, 0.905414362566632_dbl_kind, 0.95238159950027_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,369) = (/ 0.890799843177065_dbl_kind, 0.893822890405452_dbl_kind, 0.896277951288353_dbl_kind, 0.905433572891346_dbl_kind, 0.952439734168496_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,370) = (/ 0.890801493027247_dbl_kind, 0.893826133684253_dbl_kind, 0.896284913874091_dbl_kind, 0.905452743328619_dbl_kind, 0.952497678080756_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,371) = (/ 0.890803135622581_dbl_kind, 0.89382936235772_dbl_kind, 0.89629185824721_dbl_kind, 0.905471874090379_dbl_kind, 0.952555432092086_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,372) = (/ 0.890804771017329_dbl_kind, 0.893832576497111_dbl_kind, 0.896298784494582_dbl_kind, 0.905490965387599_dbl_kind, 0.952612997052416_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,373) = (/ 0.890806399265105_dbl_kind, 0.893835776173348_dbl_kind, 0.896305692703057_dbl_kind, 0.905510017430303_dbl_kind, 0.952670373806625_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,374) = (/ 0.890808020418889_dbl_kind, 0.893838961457022_dbl_kind, 0.896312582959462_dbl_kind, 0.905529030427564_dbl_kind, 0.952727563194574_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,375) = (/ 0.890809634531035_dbl_kind, 0.893842132418397_dbl_kind, 0.896319455350588_dbl_kind, 0.905548004587507_dbl_kind, 0.952784566051144_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,376) = (/ 0.890811241653279_dbl_kind, 0.893845289127419_dbl_kind, 0.896326309963195_dbl_kind, 0.905566940117311_dbl_kind, 0.952841383206277_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,377) = (/ 0.890812841836755_dbl_kind, 0.893848431653718_dbl_kind, 0.896333146883999_dbl_kind, 0.905585837223208_dbl_kind, 0.952898015485015_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,378) = (/ 0.890814435131998_dbl_kind, 0.893851560066613_dbl_kind, 0.896339966199675_dbl_kind, 0.905604696110486_dbl_kind, 0.952954463707528_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,379) = (/ 0.890816021588957_dbl_kind, 0.89385467443512_dbl_kind, 0.896346767996843_dbl_kind, 0.905623516983492_dbl_kind, 0.95301072868917_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,380) = (/ 0.890817601257006_dbl_kind, 0.893857774827951_dbl_kind, 0.89635355236207_dbl_kind, 0.905642300045627_dbl_kind, 0.953066811240496_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,381) = (/ 0.890819174184947_dbl_kind, 0.893860861313524_dbl_kind, 0.896360319381864_dbl_kind, 0.905661045499353_dbl_kind, 0.953122712167312_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,382) = (/ 0.890820740421026_dbl_kind, 0.893863933959963_dbl_kind, 0.896367069142668_dbl_kind, 0.905679753546193_dbl_kind, 0.953178432270706_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,383) = (/ 0.890822300012937_dbl_kind, 0.893866992835106_dbl_kind, 0.896373801730854_dbl_kind, 0.905698424386732_dbl_kind, 0.953233972347082_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,384) = (/ 0.890823853007833_dbl_kind, 0.893870038006504_dbl_kind, 0.896380517232721_dbl_kind, 0.905717058220615_dbl_kind, 0.953289333188205_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,385) = (/ 0.890825399452333_dbl_kind, 0.89387306954143_dbl_kind, 0.896387215734487_dbl_kind, 0.905735655246554_dbl_kind, 0.953344515581222_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,386) = (/ 0.89082693939253_dbl_kind, 0.893876087506877_dbl_kind, 0.896393897322285_dbl_kind, 0.905754215662325_dbl_kind, 0.953399520308711_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,387) = (/ 0.890828472874001_dbl_kind, 0.893879091969567_dbl_kind, 0.896400562082161_dbl_kind, 0.905772739664768_dbl_kind, 0.953454348148701_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,388) = (/ 0.890829999941814_dbl_kind, 0.893882082995952_dbl_kind, 0.896407210100066_dbl_kind, 0.905791227449795_dbl_kind, 0.953508999874722_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,389) = (/ 0.890831520640535_dbl_kind, 0.893885060652214_dbl_kind, 0.89641384146185_dbl_kind, 0.905809679212383_dbl_kind, 0.953563476255827_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,390) = (/ 0.890833035014234_dbl_kind, 0.893888025004274_dbl_kind, 0.89642045625326_dbl_kind, 0.90582809514658_dbl_kind, 0.953617778056632_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,391) = (/ 0.890834543106498_dbl_kind, 0.893890976117792_dbl_kind, 0.896427054559935_dbl_kind, 0.905846475445506_dbl_kind, 0.953671906037343_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,392) = (/ 0.890836044960433_dbl_kind, 0.893893914058167_dbl_kind, 0.896433636467399_dbl_kind, 0.905864820301353_dbl_kind, 0.953725860953796_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,393) = (/ 0.890837540618672_dbl_kind, 0.893896838890546_dbl_kind, 0.896440202061059_dbl_kind, 0.905883129905385_dbl_kind, 0.953779643557485_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,394) = (/ 0.890839030123385_dbl_kind, 0.893899750679822_dbl_kind, 0.896446751426195_dbl_kind, 0.905901404447942_dbl_kind, 0.953833254595596_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,395) = (/ 0.890840513516282_dbl_kind, 0.893902649490636_dbl_kind, 0.896453284647963_dbl_kind, 0.90591964411844_dbl_kind, 0.953886694811042_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,396) = (/ 0.890841990838623_dbl_kind, 0.893905535387385_dbl_kind, 0.896459801811383_dbl_kind, 0.905937849105374_dbl_kind, 0.953939964942486_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,397) = (/ 0.890843462131222_dbl_kind, 0.893908408434217_dbl_kind, 0.89646630300134_dbl_kind, 0.905956019596314_dbl_kind, 0.953993065724381_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,398) = (/ 0.890844927434455_dbl_kind, 0.893911268695039_dbl_kind, 0.896472788302574_dbl_kind, 0.905974155777912_dbl_kind, 0.954045997886996_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,399) = (/ 0.890846386788265_dbl_kind, 0.893914116233516_dbl_kind, 0.896479257799679_dbl_kind, 0.905992257835902_dbl_kind, 0.954098762156453_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,400) = (/ 0.890847840232171_dbl_kind, 0.893916951113075_dbl_kind, 0.896485711577098_dbl_kind, 0.9060103259551_dbl_kind, 0.954151359254748_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,401) = (/ 0.89084928780527_dbl_kind, 0.893919773396905_dbl_kind, 0.896492149719116_dbl_kind, 0.906028360319405_dbl_kind, 0.954203789899791_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,402) = (/ 0.890850729546248_dbl_kind, 0.89392258314796_dbl_kind, 0.89649857230986_dbl_kind, 0.906046361111802_dbl_kind, 0.95425605480543_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,403) = (/ 0.89085216549338_dbl_kind, 0.893925380428964_dbl_kind, 0.896504979433289_dbl_kind, 0.906064328514364_dbl_kind, 0.954308154681482_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,404) = (/ 0.890853595684541_dbl_kind, 0.893928165302404_dbl_kind, 0.896511371173193_dbl_kind, 0.906082262708249_dbl_kind, 0.954360090233764_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,405) = (/ 0.890855020157207_dbl_kind, 0.893930937830543_dbl_kind, 0.896517747613189_dbl_kind, 0.906100163873708_dbl_kind, 0.954411862164119_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,406) = (/ 0.890856438948465_dbl_kind, 0.893933698075412_dbl_kind, 0.896524108836714_dbl_kind, 0.906118032190082_dbl_kind, 0.954463471170449_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,407) = (/ 0.890857852095016_dbl_kind, 0.893936446098819_dbl_kind, 0.896530454927021_dbl_kind, 0.906135867835804_dbl_kind, 0.954514917946743_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,408) = (/ 0.890859259633181_dbl_kind, 0.893939181962346_dbl_kind, 0.896536785967179_dbl_kind, 0.906153670988401_dbl_kind, 0.954566203183099_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,409) = (/ 0.890860661598905_dbl_kind, 0.89394190572735_dbl_kind, 0.896543102040063_dbl_kind, 0.906171441824496_dbl_kind, 0.954617327565762_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,410) = (/ 0.890862058027762_dbl_kind, 0.893944617454968_dbl_kind, 0.896549403228353_dbl_kind, 0.90618918051981_dbl_kind, 0.95466829177714_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,411) = (/ 0.890863448954963_dbl_kind, 0.893947317206117_dbl_kind, 0.896555689614529_dbl_kind, 0.906206887249162_dbl_kind, 0.954719096495851_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,412) = (/ 0.890864834415359_dbl_kind, 0.893950005041494_dbl_kind, 0.896561961280867_dbl_kind, 0.906224562186471_dbl_kind, 0.954769742396724_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,413) = (/ 0.890866214443444_dbl_kind, 0.893952681021577_dbl_kind, 0.896568218309434_dbl_kind, 0.906242205504759_dbl_kind, 0.95482023015085_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,414) = (/ 0.890867589073362_dbl_kind, 0.89395534520663_dbl_kind, 0.896574460782089_dbl_kind, 0.906259817376152_dbl_kind, 0.954870560425593_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,415) = (/ 0.890868958338913_dbl_kind, 0.8939579976567_dbl_kind, 0.896580688780469_dbl_kind, 0.90627739797188_dbl_kind, 0.954920733884629_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,416) = (/ 0.890870322273553_dbl_kind, 0.89396063843162_dbl_kind, 0.896586902385996_dbl_kind, 0.906294947462282_dbl_kind, 0.954970751187955_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,417) = (/ 0.890871680910404_dbl_kind, 0.893963267591009_dbl_kind, 0.896593101679866_dbl_kind, 0.906312466016805_dbl_kind, 0.95502061299194_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,418) = (/ 0.890873034282253_dbl_kind, 0.893965885194276_dbl_kind, 0.896599286743048_dbl_kind, 0.906329953804006_dbl_kind, 0.955070319949327_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,419) = (/ 0.890874382421561_dbl_kind, 0.893968491300618_dbl_kind, 0.896605457656279_dbl_kind, 0.906347410991556_dbl_kind, 0.955119872709271_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,420) = (/ 0.890875725360464_dbl_kind, 0.89397108596902_dbl_kind, 0.896611614500063_dbl_kind, 0.90636483774624_dbl_kind, 0.955169271917366_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,421) = (/ 0.890877063130779_dbl_kind, 0.893973669258263_dbl_kind, 0.896617757354664_dbl_kind, 0.906382234233957_dbl_kind, 0.955218518215662_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,422) = (/ 0.890878395764007_dbl_kind, 0.893976241226914_dbl_kind, 0.896623886300103_dbl_kind, 0.906399600619728_dbl_kind, 0.955267612242694_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,423) = (/ 0.890879723291338_dbl_kind, 0.893978801933337_dbl_kind, 0.896630001416157_dbl_kind, 0.906416937067692_dbl_kind, 0.955316554633513_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,424) = (/ 0.890881045743655_dbl_kind, 0.893981351435687_dbl_kind, 0.896636102782354_dbl_kind, 0.906434243741109_dbl_kind, 0.9553653460197_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,425) = (/ 0.890882363151536_dbl_kind, 0.893983889791916_dbl_kind, 0.896642190477968_dbl_kind, 0.906451520802366_dbl_kind, 0.955413987029395_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,426) = (/ 0.890883675545261_dbl_kind, 0.89398641705977_dbl_kind, 0.896648264582019_dbl_kind, 0.906468768412973_dbl_kind, 0.955462478287324_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,427) = (/ 0.890884982954811_dbl_kind, 0.893988933296792_dbl_kind, 0.896654325173265_dbl_kind, 0.906485986733568_dbl_kind, 0.955510820414819_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,428) = (/ 0.890886285409879_dbl_kind, 0.893991438560319_dbl_kind, 0.896660372330204_dbl_kind, 0.906503175923923_dbl_kind, 0.955559014029845_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,429) = (/ 0.890887582939866_dbl_kind, 0.893993932907492_dbl_kind, 0.89666640613107_dbl_kind, 0.906520336142936_dbl_kind, 0.955607059747016_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,430) = (/ 0.890888875573888_dbl_kind, 0.893996416395243_dbl_kind, 0.896672426653824_dbl_kind, 0.906537467548646_dbl_kind, 0.955654958177632_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,431) = (/ 0.890890163340783_dbl_kind, 0.893998889080308_dbl_kind, 0.896678433976159_dbl_kind, 0.906554570298225_dbl_kind, 0.955702709929685_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,432) = (/ 0.890891446269105_dbl_kind, 0.894001351019222_dbl_kind, 0.896684428175491_dbl_kind, 0.906571644547981_dbl_kind, 0.955750315607899_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,433) = (/ 0.890892724387139_dbl_kind, 0.89400380226832_dbl_kind, 0.89669040932896_dbl_kind, 0.906588690453368_dbl_kind, 0.95579777581374_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,434) = (/ 0.890893997722894_dbl_kind, 0.894006242883737_dbl_kind, 0.896696377513425_dbl_kind, 0.906605708168981_dbl_kind, 0.955845091145443_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,435) = (/ 0.890895266304116_dbl_kind, 0.894008672921413_dbl_kind, 0.89670233280546_dbl_kind, 0.906622697848559_dbl_kind, 0.955892262198036_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,436) = (/ 0.89089653015828_dbl_kind, 0.894011092437086_dbl_kind, 0.896708275281356_dbl_kind, 0.906639659644991_dbl_kind, 0.955939289563361_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,437) = (/ 0.890897789312605_dbl_kind, 0.894013501486301_dbl_kind, 0.896714205017113_dbl_kind, 0.906656593710313_dbl_kind, 0.955986173830094_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,438) = (/ 0.890899043794048_dbl_kind, 0.894015900124405_dbl_kind, 0.89672012208844_dbl_kind, 0.906673500195717_dbl_kind, 0.95603291558377_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,439) = (/ 0.890900293629313_dbl_kind, 0.894018288406549_dbl_kind, 0.896726026570754_dbl_kind, 0.906690379251546_dbl_kind, 0.956079515406797_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,440) = (/ 0.890901538844849_dbl_kind, 0.894020666387689_dbl_kind, 0.896731918539172_dbl_kind, 0.906707231027303_dbl_kind, 0.956125973878493_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,441) = (/ 0.890902779466858_dbl_kind, 0.894023034122585_dbl_kind, 0.896737798068515_dbl_kind, 0.906724055671648_dbl_kind, 0.956172291575088_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,442) = (/ 0.890904015521294_dbl_kind, 0.894025391665806_dbl_kind, 0.8967436652333_dbl_kind, 0.906740853332405_dbl_kind, 0.95621846906976_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,443) = (/ 0.890905247033869_dbl_kind, 0.894027739071723_dbl_kind, 0.896749520107744_dbl_kind, 0.906757624156561_dbl_kind, 0.956264506932649_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,444) = (/ 0.890906474030054_dbl_kind, 0.894030076394517_dbl_kind, 0.896755362765754_dbl_kind, 0.906774368290271_dbl_kind, 0.956310405730875_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,445) = (/ 0.890907696535082_dbl_kind, 0.894032403688176_dbl_kind, 0.896761193280931_dbl_kind, 0.906791085878858_dbl_kind, 0.956356166028567_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,446) = (/ 0.890908914573948_dbl_kind, 0.894034721006493_dbl_kind, 0.896767011726566_dbl_kind, 0.906807777066819_dbl_kind, 0.956401788386877_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,447) = (/ 0.890910128171422_dbl_kind, 0.894037028403073_dbl_kind, 0.896772818175637_dbl_kind, 0.906824441997823_dbl_kind, 0.956447273363999_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,448) = (/ 0.890911337352038_dbl_kind, 0.894039325931327_dbl_kind, 0.896778612700806_dbl_kind, 0.906841080814719_dbl_kind, 0.956492621515199_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,449) = (/ 0.890912542140106_dbl_kind, 0.894041613644477_dbl_kind, 0.89678439537442_dbl_kind, 0.906857693659533_dbl_kind, 0.956537833392817_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,450) = (/ 0.89091374255971_dbl_kind, 0.894043891595553_dbl_kind, 0.896790166268508_dbl_kind, 0.906874280673477_dbl_kind, 0.956582909546304_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,451) = (/ 0.890914938634714_dbl_kind, 0.894046159837395_dbl_kind, 0.896795925454777_dbl_kind, 0.906890841996943_dbl_kind, 0.956627850522235_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,452) = (/ 0.890916130388763_dbl_kind, 0.894048418422655_dbl_kind, 0.896801673004614_dbl_kind, 0.906907377769517_dbl_kind, 0.956672656864322_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,453) = (/ 0.890917317845284_dbl_kind, 0.894050667403795_dbl_kind, 0.896807408989081_dbl_kind, 0.906923888129969_dbl_kind, 0.956717329113443_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,454) = (/ 0.890918501027492_dbl_kind, 0.894052906833087_dbl_kind, 0.896813133478912_dbl_kind, 0.906940373216269_dbl_kind, 0.956761867807656_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,455) = (/ 0.890919679958389_dbl_kind, 0.894055136762617_dbl_kind, 0.896818846544519_dbl_kind, 0.906956833165578_dbl_kind, 0.95680627348222_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,456) = (/ 0.890920854660768_dbl_kind, 0.894057357244281_dbl_kind, 0.89682454825598_dbl_kind, 0.906973268114257_dbl_kind, 0.95685054666961_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,457) = (/ 0.890922025157215_dbl_kind, 0.894059568329788_dbl_kind, 0.896830238683046_dbl_kind, 0.90698967819787_dbl_kind, 0.956894687899537_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,458) = (/ 0.890923191470111_dbl_kind, 0.89406177007066_dbl_kind, 0.896835917895135_dbl_kind, 0.907006063551185_dbl_kind, 0.956938697698968_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,459) = (/ 0.890924353621638_dbl_kind, 0.894063962518232_dbl_kind, 0.896841585961331_dbl_kind, 0.907022424308177_dbl_kind, 0.956982576592148_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,460) = (/ 0.890925511633775_dbl_kind, 0.894066145723653_dbl_kind, 0.896847242950385_dbl_kind, 0.90703876060203_dbl_kind, 0.957026325100604_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,461) = (/ 0.890926665528304_dbl_kind, 0.894068319737884_dbl_kind, 0.89685288893071_dbl_kind, 0.907055072565145_dbl_kind, 0.95706994374318_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,462) = (/ 0.890927815326813_dbl_kind, 0.894070484611704_dbl_kind, 0.896858523970384_dbl_kind, 0.907071360329134_dbl_kind, 0.957113433036041_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,463) = (/ 0.890928961050696_dbl_kind, 0.894072640395702_dbl_kind, 0.896864148137145_dbl_kind, 0.907087624024832_dbl_kind, 0.957156793492699_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,464) = (/ 0.890930102721157_dbl_kind, 0.894074787140285_dbl_kind, 0.896869761498391_dbl_kind, 0.907103863782294_dbl_kind, 0.957200025624032_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,465) = (/ 0.890931240359211_dbl_kind, 0.894076924895674_dbl_kind, 0.896875364121182_dbl_kind, 0.9071200797308_dbl_kind, 0.957243129938289_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,466) = (/ 0.890932373985684_dbl_kind, 0.894079053711906_dbl_kind, 0.896880956072232_dbl_kind, 0.907136271998859_dbl_kind, 0.957286106941121_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,467) = (/ 0.890933503621222_dbl_kind, 0.894081173638833_dbl_kind, 0.896886537417918_dbl_kind, 0.907152440714209_dbl_kind, 0.957328957135593_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,468) = (/ 0.890934629286286_dbl_kind, 0.894083284726125_dbl_kind, 0.896892108224267_dbl_kind, 0.907168586003824_dbl_kind, 0.957371681022191_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,469) = (/ 0.890935751001157_dbl_kind, 0.894085387023265_dbl_kind, 0.896897668556968_dbl_kind, 0.907184707993914_dbl_kind, 0.95741427909886_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,470) = (/ 0.890936868785937_dbl_kind, 0.894087480579557_dbl_kind, 0.896903218481358_dbl_kind, 0.907200806809929_dbl_kind, 0.957456751861001_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,471) = (/ 0.890937982660553_dbl_kind, 0.89408956544412_dbl_kind, 0.896908758062432_dbl_kind, 0.907216882576563_dbl_kind, 0.957499099801499_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,472) = (/ 0.890939092644758_dbl_kind, 0.894091641665891_dbl_kind, 0.896914287364839_dbl_kind, 0.907232935417753_dbl_kind, 0.957541323410728_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,473) = (/ 0.89094019875813_dbl_kind, 0.894093709293622_dbl_kind, 0.896919806452876_dbl_kind, 0.90724896545669_dbl_kind, 0.957583423176586_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,474) = (/ 0.890941301020081_dbl_kind, 0.894095768375887_dbl_kind, 0.896925315390496_dbl_kind, 0.907264972815816_dbl_kind, 0.957625399584487_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,475) = (/ 0.890942399449852_dbl_kind, 0.894097818961077_dbl_kind, 0.896930814241301_dbl_kind, 0.907280957616825_dbl_kind, 0.957667253117397_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,476) = (/ 0.890943494066516_dbl_kind, 0.8940998610974_dbl_kind, 0.896936303068544_dbl_kind, 0.907296919980676_dbl_kind, 0.957708984255844_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,477) = (/ 0.890944584888984_dbl_kind, 0.894101894832884_dbl_kind, 0.896941781935128_dbl_kind, 0.907312860027583_dbl_kind, 0.957750593477922_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,478) = (/ 0.890945671936003_dbl_kind, 0.894103920215375_dbl_kind, 0.896947250903607_dbl_kind, 0.907328777877031_dbl_kind, 0.957792081259326_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,479) = (/ 0.890946755226159_dbl_kind, 0.894105937292541_dbl_kind, 0.896952710036183_dbl_kind, 0.907344673647771_dbl_kind, 0.957833448073355_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,480) = (/ 0.890947834777879_dbl_kind, 0.894107946111867_dbl_kind, 0.896958159394708_dbl_kind, 0.907360547457824_dbl_kind, 0.957874694390925_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,481) = (/ 0.890948910609433_dbl_kind, 0.894109946720659_dbl_kind, 0.89696359904068_dbl_kind, 0.907376399424488_dbl_kind, 0.957915820680597_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,482) = (/ 0.890949982738935_dbl_kind, 0.894111939166043_dbl_kind, 0.896969029035249_dbl_kind, 0.907392229664338_dbl_kind, 0.957956827408581_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,483) = (/ 0.890951051184345_dbl_kind, 0.894113923494966_dbl_kind, 0.896974449439211_dbl_kind, 0.90740803829323_dbl_kind, 0.957997715038749_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,484) = (/ 0.890952115963472_dbl_kind, 0.894115899754194_dbl_kind, 0.896979860313011_dbl_kind, 0.907423825426305_dbl_kind, 0.958038484032665_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,485) = (/ 0.890953177093972_dbl_kind, 0.894117867990314_dbl_kind, 0.896985261716741_dbl_kind, 0.907439591177993_dbl_kind, 0.958079134849583_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,486) = (/ 0.890954234593356_dbl_kind, 0.894119828249738_dbl_kind, 0.89699065371014_dbl_kind, 0.90745533566201_dbl_kind, 0.958119667946467_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,487) = (/ 0.890955288478985_dbl_kind, 0.894121780578694_dbl_kind, 0.896996036352597_dbl_kind, 0.907471058991372_dbl_kind, 0.95816008377801_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,488) = (/ 0.890956338768075_dbl_kind, 0.894123725023236_dbl_kind, 0.897001409703149_dbl_kind, 0.907486761278389_dbl_kind, 0.958200382796645_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,489) = (/ 0.890957385477701_dbl_kind, 0.894125661629236_dbl_kind, 0.897006773820477_dbl_kind, 0.907502442634674_dbl_kind, 0.958240565452556_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,490) = (/ 0.890958428624792_dbl_kind, 0.894127590442392_dbl_kind, 0.897012128762914_dbl_kind, 0.907518103171144_dbl_kind, 0.958280632193702_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,491) = (/ 0.89095946822614_dbl_kind, 0.894129511508222_dbl_kind, 0.897017474588439_dbl_kind, 0.907533742998021_dbl_kind, 0.958320583465815_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,492) = (/ 0.890960504298396_dbl_kind, 0.894131424872068_dbl_kind, 0.897022811354678_dbl_kind, 0.90754936222484_dbl_kind, 0.958360419712428_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,493) = (/ 0.890961536858074_dbl_kind, 0.894133330579092_dbl_kind, 0.897028139118909_dbl_kind, 0.907564960960451_dbl_kind, 0.958400141374887_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,494) = (/ 0.890962565921554_dbl_kind, 0.894135228674281_dbl_kind, 0.897033457938055_dbl_kind, 0.907580539313019_dbl_kind, 0.958439748892355_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,495) = (/ 0.890963591505078_dbl_kind, 0.894137119202448_dbl_kind, 0.89703876786869_dbl_kind, 0.907596097390031_dbl_kind, 0.958479242701837_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,496) = (/ 0.89096461362476_dbl_kind, 0.894139002208224_dbl_kind, 0.897044068967035_dbl_kind, 0.907611635298298_dbl_kind, 0.958518623238186_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,497) = (/ 0.890965632296581_dbl_kind, 0.894140877736067_dbl_kind, 0.897049361288964_dbl_kind, 0.907627153143958_dbl_kind, 0.958557890934119_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,498) = (/ 0.89096664753639_dbl_kind, 0.89414274583026_dbl_kind, 0.897054644889997_dbl_kind, 0.907642651032483_dbl_kind, 0.958597046220233_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,499) = (/ 0.890967659359911_dbl_kind, 0.894144606534907_dbl_kind, 0.897059919825308_dbl_kind, 0.907658129068676_dbl_kind, 0.958636089525012_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,500) = (/ 0.89096866778274_dbl_kind, 0.894146459893938_dbl_kind, 0.89706518614972_dbl_kind, 0.907673587356676_dbl_kind, 0.958675021274843_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,501) = (/ 0.890969672820348_dbl_kind, 0.894148305951107_dbl_kind, 0.897070443917707_dbl_kind, 0.907689025999969_dbl_kind, 0.958713841894034_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,502) = (/ 0.890970674488083_dbl_kind, 0.894150144749995_dbl_kind, 0.897075693183395_dbl_kind, 0.907704445101379_dbl_kind, 0.958752551804815_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,503) = (/ 0.890971672801168_dbl_kind, 0.894151976334005_dbl_kind, 0.897080934000563_dbl_kind, 0.907719844763082_dbl_kind, 0.958791151427364_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,504) = (/ 0.890972667774709_dbl_kind, 0.894153800746367_dbl_kind, 0.897086166422642_dbl_kind, 0.907735225086603_dbl_kind, 0.958829641179811_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,505) = (/ 0.890973659423687_dbl_kind, 0.894155618030135_dbl_kind, 0.897091390502719_dbl_kind, 0.907750586172823_dbl_kind, 0.958868021478251_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,506) = (/ 0.890974647762971_dbl_kind, 0.89415742822819_dbl_kind, 0.897096606293533_dbl_kind, 0.907765928121981_dbl_kind, 0.95890629273676_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,507) = (/ 0.890975632807309_dbl_kind, 0.894159231383238_dbl_kind, 0.897101813847479_dbl_kind, 0.907781251033674_dbl_kind, 0.958944455367407_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,508) = (/ 0.890976614571335_dbl_kind, 0.894161027537811_dbl_kind, 0.897107013216606_dbl_kind, 0.907796555006867_dbl_kind, 0.958982509780264_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,509) = (/ 0.890977593069567_dbl_kind, 0.894162816734268_dbl_kind, 0.897112204452623_dbl_kind, 0.907811840139894_dbl_kind, 0.959020456383418_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,510) = (/ 0.890978568316414_dbl_kind, 0.894164599014794_dbl_kind, 0.897117387606893_dbl_kind, 0.907827106530456_dbl_kind, 0.959058295582985_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,511) = (/ 0.890979540326169_dbl_kind, 0.8941663744214_dbl_kind, 0.897122562730439_dbl_kind, 0.907842354275634_dbl_kind, 0.959096027783121_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,512) = (/ 0.890980509113019_dbl_kind, 0.894168142995924_dbl_kind, 0.897127729873941_dbl_kind, 0.907857583471882_dbl_kind, 0.959133653386036_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,513) = (/ 0.890981474691037_dbl_kind, 0.894169904780032_dbl_kind, 0.897132889087742_dbl_kind, 0.907872794215041_dbl_kind, 0.959171172791996_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,514) = (/ 0.890982437074194_dbl_kind, 0.894171659815215_dbl_kind, 0.897138040421843_dbl_kind, 0.907887986600334_dbl_kind, 0.959208586399355_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,515) = (/ 0.89098339627635_dbl_kind, 0.894173408142794_dbl_kind, 0.897143183925907_dbl_kind, 0.907903160722371_dbl_kind, 0.959245894604544_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,516) = (/ 0.890984352311263_dbl_kind, 0.894175149803916_dbl_kind, 0.897148319649259_dbl_kind, 0.907918316675158_dbl_kind, 0.959283097802094_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,517) = (/ 0.890985305192585_dbl_kind, 0.894176884839556_dbl_kind, 0.897153447640891_dbl_kind, 0.907933454552093_dbl_kind, 0.959320196384651_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,518) = (/ 0.890986254933866_dbl_kind, 0.894178613290517_dbl_kind, 0.897158567949455_dbl_kind, 0.907948574445973_dbl_kind, 0.959357190742977_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,519) = (/ 0.890987201548554_dbl_kind, 0.894180335197431_dbl_kind, 0.897163680623272_dbl_kind, 0.907963676448998_dbl_kind, 0.959394081265968_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,520) = (/ 0.890988145049997_dbl_kind, 0.894182050600756_dbl_kind, 0.897168785710329_dbl_kind, 0.907978760652772_dbl_kind, 0.959430868340668_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,521) = (/ 0.890989085451446_dbl_kind, 0.89418375954078_dbl_kind, 0.897173883258278_dbl_kind, 0.907993827148308_dbl_kind, 0.959467552352267_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,522) = (/ 0.890990022766048_dbl_kind, 0.894185462057621_dbl_kind, 0.897178973314443_dbl_kind, 0.908008876026034_dbl_kind, 0.95950413368413_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,523) = (/ 0.89099095700686_dbl_kind, 0.894187158191224_dbl_kind, 0.897184055925817_dbl_kind, 0.908023907375788_dbl_kind, 0.95954061271779_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,524) = (/ 0.89099188818684_dbl_kind, 0.894188847981363_dbl_kind, 0.897189131139064_dbl_kind, 0.908038921286831_dbl_kind, 0.959576989832977_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,525) = (/ 0.890992816318851_dbl_kind, 0.894190531467642_dbl_kind, 0.89719419900052_dbl_kind, 0.908053917847844_dbl_kind, 0.959613265407609_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,526) = (/ 0.890993741415663_dbl_kind, 0.894192208689495_dbl_kind, 0.897199259556195_dbl_kind, 0.908068897146935_dbl_kind, 0.959649439817826_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,527) = (/ 0.890994663489955_dbl_kind, 0.894193879686185_dbl_kind, 0.897204312851772_dbl_kind, 0.908083859271639_dbl_kind, 0.959685513437973_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,528) = (/ 0.890995582554313_dbl_kind, 0.894195544496803_dbl_kind, 0.897209358932614_dbl_kind, 0.908098804308926_dbl_kind, 0.959721486640634_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,529) = (/ 0.890996498621233_dbl_kind, 0.894197203160273_dbl_kind, 0.897214397843756_dbl_kind, 0.908113732345199_dbl_kind, 0.959757359796631_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,530) = (/ 0.890997411703124_dbl_kind, 0.894198855715348_dbl_kind, 0.897219429629914_dbl_kind, 0.908128643466301_dbl_kind, 0.959793133275041_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,531) = (/ 0.890998321812304_dbl_kind, 0.89420050220061_dbl_kind, 0.897224454335485_dbl_kind, 0.908143537757515_dbl_kind, 0.959828807443192_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,532) = (/ 0.890999228961006_dbl_kind, 0.894202142654475_dbl_kind, 0.897229472004543_dbl_kind, 0.908158415303573_dbl_kind, 0.959864382666696_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,533) = (/ 0.891000133161375_dbl_kind, 0.894203777115185_dbl_kind, 0.897234482680849_dbl_kind, 0.908173276188654_dbl_kind, 0.959899859309437_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,534) = (/ 0.891001034425475_dbl_kind, 0.894205405620817_dbl_kind, 0.897239486407844_dbl_kind, 0.908188120496388_dbl_kind, 0.959935237733597_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,535) = (/ 0.891001932765283_dbl_kind, 0.894207028209277_dbl_kind, 0.897244483228656_dbl_kind, 0.908202948309862_dbl_kind, 0.959970518299652_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,536) = (/ 0.891002828192691_dbl_kind, 0.894208644918302_dbl_kind, 0.897249473186099_dbl_kind, 0.908217759711622_dbl_kind, 0.960005701366396_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,537) = (/ 0.891003720719516_dbl_kind, 0.894210255785463_dbl_kind, 0.897254456322674_dbl_kind, 0.908232554783674_dbl_kind, 0.960040787290943_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,538) = (/ 0.891004610357486_dbl_kind, 0.894211860848159_dbl_kind, 0.89725943268057_dbl_kind, 0.90824733360749_dbl_kind, 0.960075776428736_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,539) = (/ 0.891005497118254_dbl_kind, 0.894213460143625_dbl_kind, 0.897264402301672_dbl_kind, 0.908262096264012_dbl_kind, 0.960110669133562_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,540) = (/ 0.891006381013394_dbl_kind, 0.894215053708923_dbl_kind, 0.89726936522755_dbl_kind, 0.908276842833652_dbl_kind, 0.960145465757554_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,541) = (/ 0.8910072620544_dbl_kind, 0.894216641580952_dbl_kind, 0.897274321499471_dbl_kind, 0.908291573396297_dbl_kind, 0.960180166651207_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,542) = (/ 0.891008140252689_dbl_kind, 0.894218223796441_dbl_kind, 0.897279271158398_dbl_kind, 0.908306288031313_dbl_kind, 0.960214772163386_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,543) = (/ 0.891009015619603_dbl_kind, 0.894219800391952_dbl_kind, 0.897284214244987_dbl_kind, 0.908320986817547_dbl_kind, 0.960249282641336_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,544) = (/ 0.891009888166408_dbl_kind, 0.894221371403879_dbl_kind, 0.897289150799596_dbl_kind, 0.90833566983333_dbl_kind, 0.960283698430688_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,545) = (/ 0.891010757904296_dbl_kind, 0.89422293686845_dbl_kind, 0.897294080862278_dbl_kind, 0.908350337156483_dbl_kind, 0.960318019875468_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,546) = (/ 0.891011624844386_dbl_kind, 0.894224496821728_dbl_kind, 0.897299004472789_dbl_kind, 0.908364988864315_dbl_kind, 0.960352247318115_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,547) = (/ 0.891012488997722_dbl_kind, 0.894226051299604_dbl_kind, 0.897303921670589_dbl_kind, 0.908379625033631_dbl_kind, 0.960386381099478_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,548) = (/ 0.891013350375279_dbl_kind, 0.894227600337808_dbl_kind, 0.897308832494838_dbl_kind, 0.908394245740735_dbl_kind, 0.960420421558836_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,549) = (/ 0.89101420898796_dbl_kind, 0.894229143971902_dbl_kind, 0.897313736984404_dbl_kind, 0.908408851061427_dbl_kind, 0.960454369033896_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,550) = (/ 0.891015064846599_dbl_kind, 0.89423068223728_dbl_kind, 0.897318635177863_dbl_kind, 0.908423441071016_dbl_kind, 0.960488223860814_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,551) = (/ 0.891015917961959_dbl_kind, 0.894232215169173_dbl_kind, 0.897323527113495_dbl_kind, 0.908438015844313_dbl_kind, 0.960521986374191_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,552) = (/ 0.891016768344736_dbl_kind, 0.894233742802645_dbl_kind, 0.897328412829296_dbl_kind, 0.908452575455643_dbl_kind, 0.960555656907095_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,553) = (/ 0.891017616005558_dbl_kind, 0.894235265172594_dbl_kind, 0.897333292362968_dbl_kind, 0.908467119978841_dbl_kind, 0.96058923579106_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,554) = (/ 0.891018460954986_dbl_kind, 0.894236782313754_dbl_kind, 0.89733816575193_dbl_kind, 0.908481649487262_dbl_kind, 0.960622723356097_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,555) = (/ 0.891019303203518_dbl_kind, 0.894238294260693_dbl_kind, 0.897343033033315_dbl_kind, 0.908496164053776_dbl_kind, 0.960656119930707_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,556) = (/ 0.891020142761582_dbl_kind, 0.894239801047815_dbl_kind, 0.897347894243972_dbl_kind, 0.908510663750776_dbl_kind, 0.960689425841883_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,557) = (/ 0.891020979639546_dbl_kind, 0.894241302709359_dbl_kind, 0.897352749420467_dbl_kind, 0.908525148650186_dbl_kind, 0.960722641415123_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,558) = (/ 0.891021813847712_dbl_kind, 0.8942427992794_dbl_kind, 0.897357598599087_dbl_kind, 0.908539618823451_dbl_kind, 0.960755766974438_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,559) = (/ 0.891022645396321_dbl_kind, 0.894244290791847_dbl_kind, 0.89736244181584_dbl_kind, 0.908554074341552_dbl_kind, 0.96078880284236_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,560) = (/ 0.89102347429555_dbl_kind, 0.894245777280447_dbl_kind, 0.897367279106457_dbl_kind, 0.908568515275004_dbl_kind, 0.960821749339944_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,561) = (/ 0.891024300555517_dbl_kind, 0.894247258778784_dbl_kind, 0.897372110506391_dbl_kind, 0.908582941693858_dbl_kind, 0.960854606786791_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,562) = (/ 0.891025124186278_dbl_kind, 0.894248735320275_dbl_kind, 0.897376936050826_dbl_kind, 0.908597353667708_dbl_kind, 0.960887375501044_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,563) = (/ 0.891025945197829_dbl_kind, 0.894250206938178_dbl_kind, 0.897381755774668_dbl_kind, 0.90861175126569_dbl_kind, 0.960920055799397_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,564) = (/ 0.891026763600109_dbl_kind, 0.894251673665583_dbl_kind, 0.897386569712557_dbl_kind, 0.908626134556487_dbl_kind, 0.960952647997107_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,565) = (/ 0.891027579402996_dbl_kind, 0.894253135535421_dbl_kind, 0.897391377898861_dbl_kind, 0.908640503608333_dbl_kind, 0.960985152408002_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,566) = (/ 0.891028392616313_dbl_kind, 0.89425459258046_dbl_kind, 0.897396180367683_dbl_kind, 0.908654858489012_dbl_kind, 0.961017569344488_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,567) = (/ 0.891029203249823_dbl_kind, 0.894256044833303_dbl_kind, 0.897400977152857_dbl_kind, 0.908669199265866_dbl_kind, 0.961049899117554_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,568) = (/ 0.891030011313237_dbl_kind, 0.894257492326394_dbl_kind, 0.897405768287956_dbl_kind, 0.908683526005793_dbl_kind, 0.961082142036785_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,569) = (/ 0.891030816816206_dbl_kind, 0.894258935092013_dbl_kind, 0.897410553806291_dbl_kind, 0.908697838775255_dbl_kind, 0.961114298410366_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,570) = (/ 0.891031619768329_dbl_kind, 0.894260373162278_dbl_kind, 0.897415333740907_dbl_kind, 0.908712137640277_dbl_kind, 0.961146368545092_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,571) = (/ 0.89103242017915_dbl_kind, 0.894261806569146_dbl_kind, 0.897420108124596_dbl_kind, 0.90872642266645_dbl_kind, 0.961178352746376_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,572) = (/ 0.891033218058158_dbl_kind, 0.894263235344414_dbl_kind, 0.89742487698989_dbl_kind, 0.908740693918937_dbl_kind, 0.961210251318252_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,573) = (/ 0.891034013414792_dbl_kind, 0.894264659519717_dbl_kind, 0.897429640369065_dbl_kind, 0.908754951462473_dbl_kind, 0.96124206456339_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,574) = (/ 0.891034806258437_dbl_kind, 0.894266079126527_dbl_kind, 0.897434398294142_dbl_kind, 0.908769195361369_dbl_kind, 0.961273792783097_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,575) = (/ 0.891035596598426_dbl_kind, 0.894267494196161_dbl_kind, 0.897439150796892_dbl_kind, 0.908783425679515_dbl_kind, 0.961305436277327_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,576) = (/ 0.891036384444041_dbl_kind, 0.89426890475977_dbl_kind, 0.897443897908833_dbl_kind, 0.908797642480382_dbl_kind, 0.961336995344692_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,577) = (/ 0.891037169804515_dbl_kind, 0.894270310848348_dbl_kind, 0.897448639661234_dbl_kind, 0.908811845827022_dbl_kind, 0.961368470282463_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,578) = (/ 0.891037952689028_dbl_kind, 0.894271712492727_dbl_kind, 0.897453376085119_dbl_kind, 0.908826035782082_dbl_kind, 0.961399861386583_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,579) = (/ 0.891038733106715_dbl_kind, 0.894273109723583_dbl_kind, 0.897458107211262_dbl_kind, 0.908840212407791_dbl_kind, 0.961431168951668_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,580) = (/ 0.89103951106666_dbl_kind, 0.89427450257143_dbl_kind, 0.897462833070196_dbl_kind, 0.908854375765974_dbl_kind, 0.961462393271021_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,581) = (/ 0.891040286577898_dbl_kind, 0.894275891066624_dbl_kind, 0.897467553692211_dbl_kind, 0.908868525918052_dbl_kind, 0.961493534636637_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,582) = (/ 0.891041059649417_dbl_kind, 0.894277275239361_dbl_kind, 0.897472269107354_dbl_kind, 0.908882662925041_dbl_kind, 0.961524593339205_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,583) = (/ 0.891041830290161_dbl_kind, 0.89427865511968_dbl_kind, 0.897476979345435_dbl_kind, 0.908896786847561_dbl_kind, 0.961555569668124_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,584) = (/ 0.891042598509023_dbl_kind, 0.89428003073746_dbl_kind, 0.897481684436026_dbl_kind, 0.908910897745834_dbl_kind, 0.961586463911505_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,585) = (/ 0.891043364314854_dbl_kind, 0.894281402122425_dbl_kind, 0.897486384408464_dbl_kind, 0.908924995679688_dbl_kind, 0.961617276356178_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,586) = (/ 0.891044127716457_dbl_kind, 0.89428276930414_dbl_kind, 0.897491079291848_dbl_kind, 0.908939080708561_dbl_kind, 0.961648007287696_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,587) = (/ 0.891044888722592_dbl_kind, 0.89428413231201_dbl_kind, 0.89749576911505_dbl_kind, 0.9089531528915_dbl_kind, 0.961678656990351_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,588) = (/ 0.891045647341973_dbl_kind, 0.894285491175287_dbl_kind, 0.897500453906705_dbl_kind, 0.90896721228717_dbl_kind, 0.961709225747172_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,589) = (/ 0.891046403583272_dbl_kind, 0.894286845923063_dbl_kind, 0.897505133695223_dbl_kind, 0.908981258953849_dbl_kind, 0.961739713839938_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,590) = (/ 0.891047157455117_dbl_kind, 0.894288196584276_dbl_kind, 0.897509808508784_dbl_kind, 0.908995292949437_dbl_kind, 0.961770121549176_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,591) = (/ 0.891047908966093_dbl_kind, 0.894289543187705_dbl_kind, 0.897514478375344_dbl_kind, 0.909009314331454_dbl_kind, 0.961800449154183_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,592) = (/ 0.891048658124744_dbl_kind, 0.894290885761976_dbl_kind, 0.897519143322633_dbl_kind, 0.909023323157047_dbl_kind, 0.961830696933015_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,593) = (/ 0.891049404939569_dbl_kind, 0.894292224335557_dbl_kind, 0.897523803378158_dbl_kind, 0.909037319482988_dbl_kind, 0.961860865162508_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,594) = (/ 0.891050149419031_dbl_kind, 0.894293558936763_dbl_kind, 0.897528458569206_dbl_kind, 0.90905130336568_dbl_kind, 0.961890954118275_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,595) = (/ 0.891050891571548_dbl_kind, 0.894294889593749_dbl_kind, 0.897533108922843_dbl_kind, 0.909065274861157_dbl_kind, 0.961920964074718_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,596) = (/ 0.891051631405499_dbl_kind, 0.894296216334523_dbl_kind, 0.897537754465919_dbl_kind, 0.90907923402509_dbl_kind, 0.961950895305031_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,597) = (/ 0.891052368929221_dbl_kind, 0.894297539186932_dbl_kind, 0.897542395225065_dbl_kind, 0.909093180912784_dbl_kind, 0.96198074808121_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,598) = (/ 0.891053104151018_dbl_kind, 0.894298858178671_dbl_kind, 0.8975470312267_dbl_kind, 0.909107115579188_dbl_kind, 0.962010522674058_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,599) = (/ 0.891053837079147_dbl_kind, 0.894300173337282_dbl_kind, 0.897551662497029_dbl_kind, 0.909121038078889_dbl_kind, 0.96204021935319_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,600) = (/ 0.891054567721831_dbl_kind, 0.894301484690152_dbl_kind, 0.897556289062043_dbl_kind, 0.909134948466122_dbl_kind, 0.962069838387041_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,601) = (/ 0.891055296087255_dbl_kind, 0.894302792264516_dbl_kind, 0.897560910947527_dbl_kind, 0.909148846794767_dbl_kind, 0.962099380042871_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,602) = (/ 0.891056022183564_dbl_kind, 0.894304096087456_dbl_kind, 0.897565528179055_dbl_kind, 0.909162733118356_dbl_kind, 0.962128844586773_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,603) = (/ 0.891056746018868_dbl_kind, 0.894305396185901_dbl_kind, 0.897570140781994_dbl_kind, 0.90917660749007_dbl_kind, 0.962158232283678_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,604) = (/ 0.891057467601239_dbl_kind, 0.894306692586628_dbl_kind, 0.897574748781508_dbl_kind, 0.909190469962748_dbl_kind, 0.962187543397359_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,605) = (/ 0.891058186938713_dbl_kind, 0.894307985316261_dbl_kind, 0.897579352202556_dbl_kind, 0.909204320588883_dbl_kind, 0.962216778190446_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,606) = (/ 0.89105890403929_dbl_kind, 0.894309274401276_dbl_kind, 0.897583951069894_dbl_kind, 0.909218159420628_dbl_kind, 0.96224593692442_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,607) = (/ 0.891059618910934_dbl_kind, 0.894310559867992_dbl_kind, 0.89758854540808_dbl_kind, 0.909231986509799_dbl_kind, 0.962275019859625_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,608) = (/ 0.891060331561573_dbl_kind, 0.894311841742583_dbl_kind, 0.897593135241469_dbl_kind, 0.909245801907874_dbl_kind, 0.962304027255278_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,609) = (/ 0.891061041999102_dbl_kind, 0.894313120051069_dbl_kind, 0.897597720594223_dbl_kind, 0.909259605665998_dbl_kind, 0.962332959369464_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,610) = (/ 0.89106175023138_dbl_kind, 0.894314394819321_dbl_kind, 0.897602301490306_dbl_kind, 0.909273397834985_dbl_kind, 0.962361816459158_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,611) = (/ 0.891062456266234_dbl_kind, 0.89431566607306_dbl_kind, 0.897606877953488_dbl_kind, 0.90928717846532_dbl_kind, 0.962390598780211_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,612) = (/ 0.891063160111453_dbl_kind, 0.894316933837856_dbl_kind, 0.897611450007344_dbl_kind, 0.909300947607161_dbl_kind, 0.962419306587375_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,613) = (/ 0.891063861774798_dbl_kind, 0.894318198139133_dbl_kind, 0.897616017675261_dbl_kind, 0.90931470531034_dbl_kind, 0.962447940134297_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,614) = (/ 0.891064561263992_dbl_kind, 0.894319459002164_dbl_kind, 0.897620580980435_dbl_kind, 0.909328451624369_dbl_kind, 0.962476499673531_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,615) = (/ 0.891065258586729_dbl_kind, 0.894320716452073_dbl_kind, 0.897625139945873_dbl_kind, 0.90934218659844_dbl_kind, 0.962504985456536_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,616) = (/ 0.891065953750669_dbl_kind, 0.894321970513839_dbl_kind, 0.897629694594396_dbl_kind, 0.909355910281423_dbl_kind, 0.962533397733689_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,617) = (/ 0.891066646763439_dbl_kind, 0.894323221212291_dbl_kind, 0.897634244948638_dbl_kind, 0.909369622721879_dbl_kind, 0.96256173675429_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,618) = (/ 0.891067337632636_dbl_kind, 0.89432446857211_dbl_kind, 0.897638791031051_dbl_kind, 0.90938332396805_dbl_kind, 0.962590002766564_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,619) = (/ 0.891068026365826_dbl_kind, 0.894325712617832_dbl_kind, 0.897643332863905_dbl_kind, 0.909397014067869_dbl_kind, 0.962618196017669_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,620) = (/ 0.891068712970543_dbl_kind, 0.894326953373846_dbl_kind, 0.897647870469287_dbl_kind, 0.909410693068962_dbl_kind, 0.962646316753705_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,621) = (/ 0.891069397454289_dbl_kind, 0.894328190864393_dbl_kind, 0.897652403869107_dbl_kind, 0.909424361018644_dbl_kind, 0.962674365219707_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,622) = (/ 0.891070079824538_dbl_kind, 0.894329425113572_dbl_kind, 0.897656933085094_dbl_kind, 0.909438017963926_dbl_kind, 0.962702341659669_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,623) = (/ 0.891070760088732_dbl_kind, 0.894330656145332_dbl_kind, 0.897661458138803_dbl_kind, 0.909451663951522_dbl_kind, 0.962730246316533_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,624) = (/ 0.891071438254286_dbl_kind, 0.894331883983481_dbl_kind, 0.897665979051613_dbl_kind, 0.909465299027837_dbl_kind, 0.962758079432205_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,625) = (/ 0.89107211432858_dbl_kind, 0.89433310865168_dbl_kind, 0.89767049584473_dbl_kind, 0.909478923238984_dbl_kind, 0.962785841247556_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,626) = (/ 0.891072788318971_dbl_kind, 0.894334330173447_dbl_kind, 0.897675008539185_dbl_kind, 0.909492536630776_dbl_kind, 0.962813532002426_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,627) = (/ 0.891073460232784_dbl_kind, 0.894335548572156_dbl_kind, 0.897679517155842_dbl_kind, 0.909506139248733_dbl_kind, 0.962841151935636_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,628) = (/ 0.891074130077314_dbl_kind, 0.894336763871036_dbl_kind, 0.897684021715394_dbl_kind, 0.909519731138084_dbl_kind, 0.962868701284981_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,629) = (/ 0.891074797859831_dbl_kind, 0.894337976093175_dbl_kind, 0.897688522238364_dbl_kind, 0.909533312343767_dbl_kind, 0.96289618028725_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,630) = (/ 0.891075463587574_dbl_kind, 0.894339185261518_dbl_kind, 0.897693018745111_dbl_kind, 0.90954688291043_dbl_kind, 0.962923589178221_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,631) = (/ 0.891076127267755_dbl_kind, 0.894340391398867_dbl_kind, 0.897697511255827_dbl_kind, 0.909560442882436_dbl_kind, 0.96295092819267_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,632) = (/ 0.89107678890756_dbl_kind, 0.894341594527883_dbl_kind, 0.897701999790541_dbl_kind, 0.909573992303866_dbl_kind, 0.962978197564377_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,633) = (/ 0.891077448514145_dbl_kind, 0.894342794671085_dbl_kind, 0.897706484369119_dbl_kind, 0.909587531218516_dbl_kind, 0.963005397526127_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,634) = (/ 0.891078106094641_dbl_kind, 0.894343991850853_dbl_kind, 0.897710965011267_dbl_kind, 0.909601059669903_dbl_kind, 0.963032528309719_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,635) = (/ 0.89107876165615_dbl_kind, 0.894345186089422_dbl_kind, 0.897715441736528_dbl_kind, 0.909614577701265_dbl_kind, 0.963059590145972_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,636) = (/ 0.89107941520575_dbl_kind, 0.894346377408893_dbl_kind, 0.897719914564291_dbl_kind, 0.909628085355566_dbl_kind, 0.963086583264723_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,637) = (/ 0.891080066750492_dbl_kind, 0.894347565831221_dbl_kind, 0.897724383513784_dbl_kind, 0.909641582675492_dbl_kind, 0.963113507894843_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,638) = (/ 0.891080716297398_dbl_kind, 0.894348751378227_dbl_kind, 0.897728848604081_dbl_kind, 0.909655069703459_dbl_kind, 0.963140364264233_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,639) = (/ 0.891081363853468_dbl_kind, 0.894349934071591_dbl_kind, 0.897733309854099_dbl_kind, 0.909668546481613_dbl_kind, 0.963167152599832_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,640) = (/ 0.891082009425675_dbl_kind, 0.894351113932854_dbl_kind, 0.897737767282606_dbl_kind, 0.909682013051829_dbl_kind, 0.963193873127622_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,641) = (/ 0.891082653020965_dbl_kind, 0.89435229098342_dbl_kind, 0.897742220908215_dbl_kind, 0.909695469455715_dbl_kind, 0.96322052607263_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,642) = (/ 0.89108329464626_dbl_kind, 0.894353465244556_dbl_kind, 0.897746670749389_dbl_kind, 0.909708915734619_dbl_kind, 0.963247111658944_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,643) = (/ 0.891083934308458_dbl_kind, 0.89435463673739_dbl_kind, 0.89775111682444_dbl_kind, 0.909722351929619_dbl_kind, 0.963273630109698_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,644) = (/ 0.89108457201443_dbl_kind, 0.894355805482915_dbl_kind, 0.897755559151536_dbl_kind, 0.909735778081537_dbl_kind, 0.963300081647099_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,645) = (/ 0.891085207771024_dbl_kind, 0.894356971501988_dbl_kind, 0.897759997748693_dbl_kind, 0.909749194230932_dbl_kind, 0.963326466492414_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,646) = (/ 0.891085841585064_dbl_kind, 0.894358134815329_dbl_kind, 0.897764432633785_dbl_kind, 0.90976260041811_dbl_kind, 0.963352784865984_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,647) = (/ 0.891086473463347_dbl_kind, 0.894359295443525_dbl_kind, 0.897768863824541_dbl_kind, 0.909775996683114_dbl_kind, 0.963379036987229_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,648) = (/ 0.89108710341265_dbl_kind, 0.894360453407025_dbl_kind, 0.897773291338545_dbl_kind, 0.909789383065739_dbl_kind, 0.963405223074646_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,649) = (/ 0.891087731439723_dbl_kind, 0.894361608726147_dbl_kind, 0.897777715193241_dbl_kind, 0.909802759605526_dbl_kind, 0.963431343345821_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,650) = (/ 0.891088357551293_dbl_kind, 0.894362761421072_dbl_kind, 0.897782135405932_dbl_kind, 0.909816126341765_dbl_kind, 0.963457398017427_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,651) = (/ 0.891088981754065_dbl_kind, 0.89436391151185_dbl_kind, 0.89778655199378_dbl_kind, 0.909829483313496_dbl_kind, 0.963483387305241_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,652) = (/ 0.891089604054719_dbl_kind, 0.894365059018397_dbl_kind, 0.897790964973811_dbl_kind, 0.909842830559515_dbl_kind, 0.963509311424126_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,653) = (/ 0.891090224459913_dbl_kind, 0.894366203960497_dbl_kind, 0.897795374362912_dbl_kind, 0.90985616811837_dbl_kind, 0.963535170588062_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,654) = (/ 0.891090842976282_dbl_kind, 0.894367346357801_dbl_kind, 0.897799780177835_dbl_kind, 0.909869496028366_dbl_kind, 0.963560965010128_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,655) = (/ 0.891091459610437_dbl_kind, 0.89436848622983_dbl_kind, 0.897804182435197_dbl_kind, 0.909882814327566_dbl_kind, 0.963586694902525_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,656) = (/ 0.891092074368969_dbl_kind, 0.894369623595972_dbl_kind, 0.89780858115148_dbl_kind, 0.909896123053793_dbl_kind, 0.963612360476564_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,657) = (/ 0.891092687258444_dbl_kind, 0.894370758475487_dbl_kind, 0.897812976343037_dbl_kind, 0.909909422244632_dbl_kind, 0.963637961942686_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,658) = (/ 0.891093298285406_dbl_kind, 0.894371890887501_dbl_kind, 0.897817368026086_dbl_kind, 0.909922711937429_dbl_kind, 0.963663499510451_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,659) = (/ 0.891093907456379_dbl_kind, 0.894373020851015_dbl_kind, 0.897821756216717_dbl_kind, 0.909935992169296_dbl_kind, 0.963688973388557_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,660) = (/ 0.891094514777863_dbl_kind, 0.894374148384898_dbl_kind, 0.897826140930889_dbl_kind, 0.909949262977113_dbl_kind, 0.963714383784832_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,661) = (/ 0.891095120256337_dbl_kind, 0.89437527350789_dbl_kind, 0.897830522184436_dbl_kind, 0.909962524397523_dbl_kind, 0.963739730906244_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,662) = (/ 0.891095723898259_dbl_kind, 0.894376396238604_dbl_kind, 0.897834899993063_dbl_kind, 0.909975776466943_dbl_kind, 0.963765014958912_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,663) = (/ 0.891096325710063_dbl_kind, 0.894377516595527_dbl_kind, 0.897839274372349_dbl_kind, 0.909989019221561_dbl_kind, 0.963790236148093_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,664) = (/ 0.891096925698165_dbl_kind, 0.894378634597014_dbl_kind, 0.89784364533775_dbl_kind, 0.910002252697334_dbl_kind, 0.963815394678205_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,665) = (/ 0.891097523868958_dbl_kind, 0.894379750261298_dbl_kind, 0.897848012904598_dbl_kind, 0.910015476929997_dbl_kind, 0.963840490752819_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,666) = (/ 0.891098120228814_dbl_kind, 0.894380863606484_dbl_kind, 0.897852377088101_dbl_kind, 0.910028691955058_dbl_kind, 0.963865524574666_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,667) = (/ 0.891098714784083_dbl_kind, 0.894381974650551_dbl_kind, 0.897856737903347_dbl_kind, 0.910041897807805_dbl_kind, 0.963890496345646_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,668) = (/ 0.891099307541097_dbl_kind, 0.894383083411352_dbl_kind, 0.897861095365304_dbl_kind, 0.910055094523303_dbl_kind, 0.963915406266827_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,669) = (/ 0.891099898506165_dbl_kind, 0.894384189906618_dbl_kind, 0.89786544948882_dbl_kind, 0.910068282136398_dbl_kind, 0.96394025453845_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,670) = (/ 0.891100487685576_dbl_kind, 0.894385294153953_dbl_kind, 0.897869800288624_dbl_kind, 0.910081460681717_dbl_kind, 0.963965041359932_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,671) = (/ 0.8911010750856_dbl_kind, 0.894386396170839_dbl_kind, 0.897874147779329_dbl_kind, 0.910094630193672_dbl_kind, 0.963989766929875_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,672) = (/ 0.891101660712484_dbl_kind, 0.894387495974633_dbl_kind, 0.897878491975432_dbl_kind, 0.910107790706459_dbl_kind, 0.964014431446066_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,673) = (/ 0.891102244572456_dbl_kind, 0.894388593582572_dbl_kind, 0.897882832891314_dbl_kind, 0.910120942254061_dbl_kind, 0.964039035105478_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,674) = (/ 0.891102826671725_dbl_kind, 0.894389689011769_dbl_kind, 0.897887170541242_dbl_kind, 0.910134084870247_dbl_kind, 0.964063578104285_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,675) = (/ 0.891103407016479_dbl_kind, 0.894390782279215_dbl_kind, 0.89789150493937_dbl_kind, 0.910147218588578_dbl_kind, 0.964088060637854_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,676) = (/ 0.891103985612887_dbl_kind, 0.894391873401781_dbl_kind, 0.89789583609974_dbl_kind, 0.910160343442403_dbl_kind, 0.964112482900752_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,677) = (/ 0.891104562467096_dbl_kind, 0.894392962396217_dbl_kind, 0.897900164036281_dbl_kind, 0.910173459464866_dbl_kind, 0.96413684508676_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,678) = (/ 0.891105137585236_dbl_kind, 0.894394049279152_dbl_kind, 0.897904488762816_dbl_kind, 0.910186566688901_dbl_kind, 0.964161147388859_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,679) = (/ 0.891105710973418_dbl_kind, 0.894395134067096_dbl_kind, 0.897908810293053_dbl_kind, 0.910199665147241_dbl_kind, 0.96418538999925_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,680) = (/ 0.891106282637729_dbl_kind, 0.894396216776441_dbl_kind, 0.897913128640597_dbl_kind, 0.910212754872413_dbl_kind, 0.964209573109349_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,681) = (/ 0.891106852584241_dbl_kind, 0.894397297423458_dbl_kind, 0.897917443818943_dbl_kind, 0.910225835896741_dbl_kind, 0.964233696909792_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,682) = (/ 0.891107420819006_dbl_kind, 0.894398376024302_dbl_kind, 0.897921755841479_dbl_kind, 0.910238908252351_dbl_kind, 0.964257761590448_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,683) = (/ 0.891107987348057_dbl_kind, 0.89439945259501_dbl_kind, 0.89792606472149_dbl_kind, 0.910251971971167_dbl_kind, 0.964281767340403_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,684) = (/ 0.891108552177405_dbl_kind, 0.894400527151501_dbl_kind, 0.897930370472154_dbl_kind, 0.910265027084917_dbl_kind, 0.964305714347987_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,685) = (/ 0.891109115313047_dbl_kind, 0.894401599709578_dbl_kind, 0.897934673106547_dbl_kind, 0.91027807362513_dbl_kind, 0.964329602800757_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,686) = (/ 0.891109676760957_dbl_kind, 0.894402670284929_dbl_kind, 0.897938972637641_dbl_kind, 0.91029111162314_dbl_kind, 0.964353432885517_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,687) = (/ 0.891110236527093_dbl_kind, 0.894403738893125_dbl_kind, 0.897943269078307_dbl_kind, 0.91030414111009_dbl_kind, 0.964377204788316_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,688) = (/ 0.891110794617392_dbl_kind, 0.894404805549624_dbl_kind, 0.897947562441315_dbl_kind, 0.910317162116925_dbl_kind, 0.964400918694442_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,689) = (/ 0.891111351037775_dbl_kind, 0.894405870269767_dbl_kind, 0.897951852739336_dbl_kind, 0.910330174674403_dbl_kind, 0.964424574788445_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,690) = (/ 0.891111905794141_dbl_kind, 0.894406933068782_dbl_kind, 0.897956139984939_dbl_kind, 0.91034317881309_dbl_kind, 0.964448173254122_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,691) = (/ 0.891112458892375_dbl_kind, 0.894407993961785_dbl_kind, 0.897960424190598_dbl_kind, 0.910356174563363_dbl_kind, 0.964471714274534_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,692) = (/ 0.89111301033834_dbl_kind, 0.894409052963779_dbl_kind, 0.897964705368686_dbl_kind, 0.910369161955413_dbl_kind, 0.964495198032001_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,693) = (/ 0.891113560137882_dbl_kind, 0.894410110089653_dbl_kind, 0.897968983531484_dbl_kind, 0.910382141019241_dbl_kind, 0.964518624708112_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,694) = (/ 0.89111410829683_dbl_kind, 0.894411165354186_dbl_kind, 0.897973258691174_dbl_kind, 0.910395111784667_dbl_kind, 0.964541994483723_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,695) = (/ 0.891114654820994_dbl_kind, 0.894412218772044_dbl_kind, 0.897977530859842_dbl_kind, 0.910408074281325_dbl_kind, 0.964565307538965_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,696) = (/ 0.891115199716164_dbl_kind, 0.894413270357785_dbl_kind, 0.897981800049484_dbl_kind, 0.910421028538668_dbl_kind, 0.964588564053242_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,697) = (/ 0.891115742988116_dbl_kind, 0.894414320125852_dbl_kind, 0.897986066272001_dbl_kind, 0.910433974585965_dbl_kind, 0.964611764205243_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,698) = (/ 0.891116284642604_dbl_kind, 0.894415368090585_dbl_kind, 0.8979903295392_dbl_kind, 0.910446912452307_dbl_kind, 0.964634908172938_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,699) = (/ 0.891116824685367_dbl_kind, 0.894416414266209_dbl_kind, 0.897994589862798_dbl_kind, 0.910459842166606_dbl_kind, 0.964657996133584_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,700) = (/ 0.891117363122126_dbl_kind, 0.894417458666844_dbl_kind, 0.897998847254422_dbl_kind, 0.910472763757596_dbl_kind, 0.964681028263729_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,701) = (/ 0.891117899958583_dbl_kind, 0.8944185013065_dbl_kind, 0.898003101725606_dbl_kind, 0.910485677253834_dbl_kind, 0.964704004739216_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,702) = (/ 0.891118435200424_dbl_kind, 0.894419542199081_dbl_kind, 0.898007353287798_dbl_kind, 0.910498582683701_dbl_kind, 0.964726925735183_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,703) = (/ 0.891118968853316_dbl_kind, 0.894420581358384_dbl_kind, 0.898011601952357_dbl_kind, 0.910511480075408_dbl_kind, 0.964749791426071_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,704) = (/ 0.891119500922909_dbl_kind, 0.894421618798097_dbl_kind, 0.898015847730552_dbl_kind, 0.910524369456986_dbl_kind, 0.964772601985627_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,705) = (/ 0.891120031414836_dbl_kind, 0.894422654531805_dbl_kind, 0.898020090633568_dbl_kind, 0.910537250856302_dbl_kind, 0.9647953575869_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,706) = (/ 0.891120560334712_dbl_kind, 0.894423688572988_dbl_kind, 0.898024330672503_dbl_kind, 0.910550124301046_dbl_kind, 0.964818058402254_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,707) = (/ 0.891121087688135_dbl_kind, 0.894424720935018_dbl_kind, 0.89802856785837_dbl_kind, 0.910562989818741_dbl_kind, 0.964840704603366_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,708) = (/ 0.891121613480686_dbl_kind, 0.894425751631165_dbl_kind, 0.898032802202095_dbl_kind, 0.910575847436742_dbl_kind, 0.964863296361233_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,709) = (/ 0.891122137717928_dbl_kind, 0.894426780674595_dbl_kind, 0.898037033714523_dbl_kind, 0.910588697182237_dbl_kind, 0.964885833846167_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,710) = (/ 0.891122660405409_dbl_kind, 0.894427808078371_dbl_kind, 0.898041262406416_dbl_kind, 0.910601539082247_dbl_kind, 0.964908317227811_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,711) = (/ 0.891123181548657_dbl_kind, 0.894428833855454_dbl_kind, 0.898045488288451_dbl_kind, 0.910614373163628_dbl_kind, 0.964930746675129_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,712) = (/ 0.891123701153184_dbl_kind, 0.8944298580187_dbl_kind, 0.898049711371225_dbl_kind, 0.910627199453072_dbl_kind, 0.964953122356422_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,713) = (/ 0.891124219224487_dbl_kind, 0.894430880580866_dbl_kind, 0.898053931665255_dbl_kind, 0.910640017977108_dbl_kind, 0.964975444439317_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,714) = (/ 0.891124735768043_dbl_kind, 0.894431901554608_dbl_kind, 0.898058149180975_dbl_kind, 0.910652828762104_dbl_kind, 0.96499771309079_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,715) = (/ 0.891125250789314_dbl_kind, 0.894432920952479_dbl_kind, 0.898062363928742_dbl_kind, 0.910665631834268_dbl_kind, 0.965019928477142_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,716) = (/ 0.891125764293745_dbl_kind, 0.894433938786936_dbl_kind, 0.898066575918834_dbl_kind, 0.910678427219646_dbl_kind, 0.965042090764026_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,717) = (/ 0.891126276286764_dbl_kind, 0.894434955070334_dbl_kind, 0.898070785161449_dbl_kind, 0.910691214944127_dbl_kind, 0.965064200116444_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,718) = (/ 0.891126786773782_dbl_kind, 0.894435969814927_dbl_kind, 0.89807499166671_dbl_kind, 0.910703995033444_dbl_kind, 0.965086256698743_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,719) = (/ 0.891127295760193_dbl_kind, 0.894436983032876_dbl_kind, 0.89807919544466_dbl_kind, 0.910716767513169_dbl_kind, 0.965108260674619_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,720) = (/ 0.891127803251377_dbl_kind, 0.894437994736241_dbl_kind, 0.898083396505269_dbl_kind, 0.910729532408725_dbl_kind, 0.965130212207133_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,721) = (/ 0.891128309252693_dbl_kind, 0.894439004936983_dbl_kind, 0.89808759485843_dbl_kind, 0.910742289745374_dbl_kind, 0.965152111458699_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,722) = (/ 0.891128813769487_dbl_kind, 0.894440013646971_dbl_kind, 0.898091790513962_dbl_kind, 0.910755039548229_dbl_kind, 0.965173958591093_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,723) = (/ 0.891129316807087_dbl_kind, 0.894441020877973_dbl_kind, 0.898095983481608_dbl_kind, 0.91076778184225_dbl_kind, 0.965195753765457_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,724) = (/ 0.891129818370804_dbl_kind, 0.894442026641665_dbl_kind, 0.89810017377104_dbl_kind, 0.910780516652245_dbl_kind, 0.965217497142302_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,725) = (/ 0.891130318465935_dbl_kind, 0.894443030949625_dbl_kind, 0.898104361391855_dbl_kind, 0.910793244002871_dbl_kind, 0.965239188881509_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,726) = (/ 0.891130817097759_dbl_kind, 0.894444033813338_dbl_kind, 0.89810854635358_dbl_kind, 0.910805963918636_dbl_kind, 0.965260829142331_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,727) = (/ 0.891131314271538_dbl_kind, 0.894445035244196_dbl_kind, 0.898112728665668_dbl_kind, 0.910818676423902_dbl_kind, 0.965282418083403_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,728) = (/ 0.891131809992518_dbl_kind, 0.894446035253494_dbl_kind, 0.898116908337503_dbl_kind, 0.91083138154288_dbl_kind, 0.965303955862732_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,729) = (/ 0.89113230426593_dbl_kind, 0.894447033852438_dbl_kind, 0.898121085378397_dbl_kind, 0.910844079299637_dbl_kind, 0.965325442637715_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,730) = (/ 0.891132797096987_dbl_kind, 0.894448031052139_dbl_kind, 0.898125259797594_dbl_kind, 0.910856769718094_dbl_kind, 0.965346878565132_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,731) = (/ 0.891133288490888_dbl_kind, 0.894449026863617_dbl_kind, 0.898129431604267_dbl_kind, 0.910869452822028_dbl_kind, 0.96536826380115_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,732) = (/ 0.891133778452814_dbl_kind, 0.8944500212978_dbl_kind, 0.898133600807521_dbl_kind, 0.910882128635071_dbl_kind, 0.965389598501331_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,733) = (/ 0.89113426698793_dbl_kind, 0.894451014365525_dbl_kind, 0.898137767416395_dbl_kind, 0.910894797180713_dbl_kind, 0.96541088282063_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,734) = (/ 0.891134754101387_dbl_kind, 0.894452006077541_dbl_kind, 0.898141931439857_dbl_kind, 0.910907458482304_dbl_kind, 0.965432116913397_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,735) = (/ 0.891135239798316_dbl_kind, 0.894452996444504_dbl_kind, 0.898146092886812_dbl_kind, 0.91092011256305_dbl_kind, 0.965453300933384_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,736) = (/ 0.891135724083837_dbl_kind, 0.894453985476982_dbl_kind, 0.898150251766097_dbl_kind, 0.910932759446021_dbl_kind, 0.965474435033747_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,737) = (/ 0.89113620696305_dbl_kind, 0.894454973185456_dbl_kind, 0.898154408086482_dbl_kind, 0.910945399154145_dbl_kind, 0.965495519367044_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,738) = (/ 0.89113668844104_dbl_kind, 0.894455959580316_dbl_kind, 0.898158561856675_dbl_kind, 0.910958031710215_dbl_kind, 0.965516554085247_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,739) = (/ 0.891137168522878_dbl_kind, 0.894456944671865_dbl_kind, 0.898162713085317_dbl_kind, 0.910970657136884_dbl_kind, 0.965537539339735_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,740) = (/ 0.891137647213617_dbl_kind, 0.894457928470321_dbl_kind, 0.898166861780986_dbl_kind, 0.91098327545667_dbl_kind, 0.965558475281304_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,741) = (/ 0.891138124518294_dbl_kind, 0.894458910985813_dbl_kind, 0.898171007952197_dbl_kind, 0.910995886691957_dbl_kind, 0.96557936206016_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,742) = (/ 0.891138600441933_dbl_kind, 0.894459892228384_dbl_kind, 0.898175151607402_dbl_kind, 0.911008490864992_dbl_kind, 0.965600199825939_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,743) = (/ 0.891139074989539_dbl_kind, 0.894460872207993_dbl_kind, 0.898179292754989_dbl_kind, 0.91102108799789_dbl_kind, 0.965620988727692_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,744) = (/ 0.891139548166102_dbl_kind, 0.894461850934513_dbl_kind, 0.898183431403288_dbl_kind, 0.911033678112635_dbl_kind, 0.965641728913896_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,745) = (/ 0.891140019976599_dbl_kind, 0.894462828417732_dbl_kind, 0.898187567560563_dbl_kind, 0.911046261231076_dbl_kind, 0.965662420532459_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,746) = (/ 0.891140490425987_dbl_kind, 0.894463804667356_dbl_kind, 0.898191701235022_dbl_kind, 0.911058837374934_dbl_kind, 0.965683063730715_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,747) = (/ 0.891140959519211_dbl_kind, 0.894464779693005_dbl_kind, 0.898195832434809_dbl_kind, 0.911071406565797_dbl_kind, 0.965703658655434_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,748) = (/ 0.891141427261198_dbl_kind, 0.894465753504216_dbl_kind, 0.898199961168011_dbl_kind, 0.911083968825127_dbl_kind, 0.965724205452819_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,749) = (/ 0.891141893656861_dbl_kind, 0.894466726110447_dbl_kind, 0.898204087442654_dbl_kind, 0.911096524174254_dbl_kind, 0.965744704268512_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,750) = (/ 0.891142358711096_dbl_kind, 0.894467697521069_dbl_kind, 0.898208211266707_dbl_kind, 0.911109072634384_dbl_kind, 0.9657651552476_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,751) = (/ 0.891142822428785_dbl_kind, 0.894468667745377_dbl_kind, 0.89821233264808_dbl_kind, 0.911121614226596_dbl_kind, 0.965785558534608_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,752) = (/ 0.891143284814793_dbl_kind, 0.894469636792581_dbl_kind, 0.898216451594625_dbl_kind, 0.911134148971839_dbl_kind, 0.965805914273509_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,753) = (/ 0.891143745873971_dbl_kind, 0.894470604671813_dbl_kind, 0.898220568114137_dbl_kind, 0.911146676890942_dbl_kind, 0.965826222607726_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,754) = (/ 0.891144205611153_dbl_kind, 0.894471571392123_dbl_kind, 0.898224682214355_dbl_kind, 0.911159198004607_dbl_kind, 0.96584648368013_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,755) = (/ 0.891144664031159_dbl_kind, 0.894472536962483_dbl_kind, 0.898228793902962_dbl_kind, 0.911171712333413_dbl_kind, 0.965866697633051_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,756) = (/ 0.891145121138793_dbl_kind, 0.894473501391786_dbl_kind, 0.898232903187585_dbl_kind, 0.911184219897818_dbl_kind, 0.965886864608268_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,757) = (/ 0.891145576938843_dbl_kind, 0.894474464688848_dbl_kind, 0.898237010075794_dbl_kind, 0.911196720718155_dbl_kind, 0.965906984747029_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,758) = (/ 0.891146031436082_dbl_kind, 0.894475426862404_dbl_kind, 0.898241114575109_dbl_kind, 0.911209214814638_dbl_kind, 0.96592705819003_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,759) = (/ 0.891146484635269_dbl_kind, 0.894476387921114_dbl_kind, 0.898245216692989_dbl_kind, 0.911221702207362_dbl_kind, 0.965947085077443_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,760) = (/ 0.891146936541146_dbl_kind, 0.894477347873562_dbl_kind, 0.898249316436845_dbl_kind, 0.911234182916299_dbl_kind, 0.965967065548901_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,761) = (/ 0.89114738715844_dbl_kind, 0.894478306728252_dbl_kind, 0.898253413814033_dbl_kind, 0.911246656961306_dbl_kind, 0.965986999743504_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,762) = (/ 0.891147836491864_dbl_kind, 0.894479264493616_dbl_kind, 0.898257508831854_dbl_kind, 0.911259124362119_dbl_kind, 0.966006887799827_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,763) = (/ 0.891148284546115_dbl_kind, 0.89448022117801_dbl_kind, 0.898261601497559_dbl_kind, 0.911271585138359_dbl_kind, 0.966026729855917_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,764) = (/ 0.891148731325874_dbl_kind, 0.894481176789712_dbl_kind, 0.898265691818347_dbl_kind, 0.91128403930953_dbl_kind, 0.966046526049296_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,765) = (/ 0.891149176835808_dbl_kind, 0.894482131336931_dbl_kind, 0.898269779801364_dbl_kind, 0.911296486895019_dbl_kind, 0.966066276516965_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,766) = (/ 0.891149621080568_dbl_kind, 0.894483084827796_dbl_kind, 0.898273865453707_dbl_kind, 0.9113089279141_dbl_kind, 0.966085981395409_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,767) = (/ 0.891150064064791_dbl_kind, 0.894484037270369_dbl_kind, 0.898277948782419_dbl_kind, 0.911321362385933_dbl_kind, 0.966105640820592_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,768) = (/ 0.891150505793098_dbl_kind, 0.894484988672635_dbl_kind, 0.898282029794496_dbl_kind, 0.911333790329562_dbl_kind, 0.966125254927967_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,769) = (/ 0.891150946270095_dbl_kind, 0.894485939042508_dbl_kind, 0.898286108496882_dbl_kind, 0.911346211763921_dbl_kind, 0.966144823852471_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,770) = (/ 0.891151385500374_dbl_kind, 0.894486888387829_dbl_kind, 0.898290184896474_dbl_kind, 0.911358626707831_dbl_kind, 0.966164347728535_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,771) = (/ 0.89115182348851_dbl_kind, 0.89448783671637_dbl_kind, 0.898294259000118_dbl_kind, 0.911371035180001_dbl_kind, 0.96618382669008_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,772) = (/ 0.891152260239065_dbl_kind, 0.894488784035829_dbl_kind, 0.898298330814613_dbl_kind, 0.911383437199031_dbl_kind, 0.966203260870527_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,773) = (/ 0.891152695756585_dbl_kind, 0.894489730353837_dbl_kind, 0.898302400346708_dbl_kind, 0.91139583278341_dbl_kind, 0.966222650402787_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,774) = (/ 0.891153130045601_dbl_kind, 0.894490675677954_dbl_kind, 0.898306467603106_dbl_kind, 0.911408221951519_dbl_kind, 0.966241995419277_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,775) = (/ 0.89115356311063_dbl_kind, 0.894491620015668_dbl_kind, 0.898310532590461_dbl_kind, 0.911420604721629_dbl_kind, 0.966261296051913_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,776) = (/ 0.891153994956172_dbl_kind, 0.894492563374401_dbl_kind, 0.898314595315383_dbl_kind, 0.911432981111905_dbl_kind, 0.966280552432115_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,777) = (/ 0.891154425586716_dbl_kind, 0.894493505761506_dbl_kind, 0.898318655784433_dbl_kind, 0.911445351140404_dbl_kind, 0.96629976469081_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,778) = (/ 0.891154855006732_dbl_kind, 0.894494447184268_dbl_kind, 0.898322714004127_dbl_kind, 0.911457714825078_dbl_kind, 0.966318932958434_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,779) = (/ 0.891155283220677_dbl_kind, 0.894495387649903_dbl_kind, 0.898326769980934_dbl_kind, 0.91147007218377_dbl_kind, 0.966338057364935_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,780) = (/ 0.891155710232995_dbl_kind, 0.894496327165562_dbl_kind, 0.898330823721279_dbl_kind, 0.911482423234224_dbl_kind, 0.96635713803977_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,781) = (/ 0.891156136048111_dbl_kind, 0.894497265738329_dbl_kind, 0.898334875231543_dbl_kind, 0.911494767994072_dbl_kind, 0.966376175111917_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,782) = (/ 0.891156560670439_dbl_kind, 0.89449820337522_dbl_kind, 0.898338924518058_dbl_kind, 0.91150710648085_dbl_kind, 0.96639516870987_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,783) = (/ 0.891156984104378_dbl_kind, 0.894499140083189_dbl_kind, 0.898342971587119_dbl_kind, 0.911519438711986_dbl_kind, 0.966414118961639_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,784) = (/ 0.891157406354309_dbl_kind, 0.894500075869121_dbl_kind, 0.89834701644497_dbl_kind, 0.911531764704808_dbl_kind, 0.966433025994759_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,785) = (/ 0.891157827424602_dbl_kind, 0.89450101073984_dbl_kind, 0.898351059097817_dbl_kind, 0.911544084476541_dbl_kind, 0.96645188993629_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,786) = (/ 0.891158247319611_dbl_kind, 0.894501944702102_dbl_kind, 0.898355099551819_dbl_kind, 0.911556398044311_dbl_kind, 0.966470710912818_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,787) = (/ 0.891158666043675_dbl_kind, 0.894502877762602_dbl_kind, 0.898359137813095_dbl_kind, 0.911568705425142_dbl_kind, 0.966489489050455_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,788) = (/ 0.891159083601118_dbl_kind, 0.894503809927971_dbl_kind, 0.898363173887721_dbl_kind, 0.911581006635959_dbl_kind, 0.966508224474845_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,789) = (/ 0.891159499996252_dbl_kind, 0.894504741204777_dbl_kind, 0.89836720778173_dbl_kind, 0.911593301693588_dbl_kind, 0.966526917311169_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,790) = (/ 0.891159915233372_dbl_kind, 0.894505671599525_dbl_kind, 0.898371239501115_dbl_kind, 0.911605590614756_dbl_kind, 0.966545567684133_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,791) = (/ 0.891160329316758_dbl_kind, 0.89450660111866_dbl_kind, 0.898375269051827_dbl_kind, 0.911617873416094_dbl_kind, 0.966564175717988_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,792) = (/ 0.891160742250678_dbl_kind, 0.894507529768563_dbl_kind, 0.898379296439776_dbl_kind, 0.911630150114134_dbl_kind, 0.966582741536526_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,793) = (/ 0.891161154039384_dbl_kind, 0.894508457555556_dbl_kind, 0.898383321670832_dbl_kind, 0.911642420725312_dbl_kind, 0.96660126526307_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,794) = (/ 0.891161564687113_dbl_kind, 0.894509384485899_dbl_kind, 0.898387344750823_dbl_kind, 0.91165468526597_dbl_kind, 0.966619747020492_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,795) = (/ 0.891161974198089_dbl_kind, 0.894510310565793_dbl_kind, 0.89839136568554_dbl_kind, 0.911666943752352_dbl_kind, 0.966638186931215_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,796) = (/ 0.891162382576521_dbl_kind, 0.89451123580138_dbl_kind, 0.898395384480732_dbl_kind, 0.911679196200608_dbl_kind, 0.966656585117197_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,797) = (/ 0.891162789826602_dbl_kind, 0.894512160198739_dbl_kind, 0.898399401142111_dbl_kind, 0.911691442626795_dbl_kind, 0.966674941699957_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,798) = (/ 0.891163195952515_dbl_kind, 0.894513083763895_dbl_kind, 0.898403415675349_dbl_kind, 0.911703683046877_dbl_kind, 0.966693256800558_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,799) = (/ 0.891163600958424_dbl_kind, 0.894514006502812_dbl_kind, 0.898407428086078_dbl_kind, 0.911715917476724_dbl_kind, 0.966711530539617_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,800) = (/ 0.891164004848481_dbl_kind, 0.894514928421397_dbl_kind, 0.898411438379894_dbl_kind, 0.911728145932113_dbl_kind, 0.966729763037312_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,801) = (/ 0.891164407626823_dbl_kind, 0.894515849525498_dbl_kind, 0.898415446562356_dbl_kind, 0.911740368428733_dbl_kind, 0.96674795441337_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,802) = (/ 0.891164809297573_dbl_kind, 0.894516769820907_dbl_kind, 0.898419452638982_dbl_kind, 0.911752584982177_dbl_kind, 0.966766104787085_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,803) = (/ 0.891165209864841_dbl_kind, 0.894517689313362_dbl_kind, 0.898423456615256_dbl_kind, 0.911764795607952_dbl_kind, 0.966784214277306_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,804) = (/ 0.891165609332721_dbl_kind, 0.894518608008539_dbl_kind, 0.898427458496624_dbl_kind, 0.911777000321472_dbl_kind, 0.96680228300245_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,805) = (/ 0.891166007705292_dbl_kind, 0.894519525912064_dbl_kind, 0.898431458288495_dbl_kind, 0.911789199138063_dbl_kind, 0.966820311080498_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,806) = (/ 0.891166404986623_dbl_kind, 0.894520443029503_dbl_kind, 0.898435455996241_dbl_kind, 0.911801392072964_dbl_kind, 0.966838298628995_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,807) = (/ 0.891166801180764_dbl_kind, 0.894521359366371_dbl_kind, 0.898439451625201_dbl_kind, 0.911813579141322_dbl_kind, 0.96685624576506_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,808) = (/ 0.891167196291756_dbl_kind, 0.894522274928127_dbl_kind, 0.898443445180676_dbl_kind, 0.9118257603582_dbl_kind, 0.966874152605381_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,809) = (/ 0.891167590323619_dbl_kind, 0.894523189720175_dbl_kind, 0.898447436667932_dbl_kind, 0.911837935738572_dbl_kind, 0.966892019266216_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,810) = (/ 0.891167983280367_dbl_kind, 0.894524103747865_dbl_kind, 0.8984514260922_dbl_kind, 0.911850105297327_dbl_kind, 0.966909845863402_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,811) = (/ 0.891168375165993_dbl_kind, 0.894525017016497_dbl_kind, 0.898455413458676_dbl_kind, 0.911862269049267_dbl_kind, 0.966927632512352_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,812) = (/ 0.891168765984481_dbl_kind, 0.894525929531315_dbl_kind, 0.898459398772523_dbl_kind, 0.911874427009107_dbl_kind, 0.966945379328057_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,813) = (/ 0.891169155739798_dbl_kind, 0.894526841297512_dbl_kind, 0.898463382038868_dbl_kind, 0.911886579191483_dbl_kind, 0.966963086425086_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,814) = (/ 0.891169544435898_dbl_kind, 0.894527752320229_dbl_kind, 0.898467363262806_dbl_kind, 0.91189872561094_dbl_kind, 0.966980753917596_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,815) = (/ 0.891169932076723_dbl_kind, 0.894528662604553_dbl_kind, 0.898471342449397_dbl_kind, 0.911910866281945_dbl_kind, 0.966998381919323_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,816) = (/ 0.891170318666197_dbl_kind, 0.894529572155525_dbl_kind, 0.898475319603669_dbl_kind, 0.911923001218877_dbl_kind, 0.967015970543589_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,817) = (/ 0.891170704208234_dbl_kind, 0.894530480978129_dbl_kind, 0.898479294730615_dbl_kind, 0.911935130436035_dbl_kind, 0.967033519903311_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,818) = (/ 0.891171088706732_dbl_kind, 0.894531389077305_dbl_kind, 0.898483267835197_dbl_kind, 0.911947253947637_dbl_kind, 0.967051030110986_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,819) = (/ 0.891171472165577_dbl_kind, 0.894532296457937_dbl_kind, 0.898487238922345_dbl_kind, 0.911959371767816_dbl_kind, 0.967068501278709_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,820) = (/ 0.891171854588638_dbl_kind, 0.894533203124865_dbl_kind, 0.898491207996955_dbl_kind, 0.911971483910628_dbl_kind, 0.967085933518165_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,821) = (/ 0.891172235979774_dbl_kind, 0.894534109082873_dbl_kind, 0.898495175063894_dbl_kind, 0.911983590390047_dbl_kind, 0.967103326940637_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,822) = (/ 0.891172616342829_dbl_kind, 0.894535014336704_dbl_kind, 0.898499140127994_dbl_kind, 0.911995691219963_dbl_kind, 0.967120681657004_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,823) = (/ 0.891172995681631_dbl_kind, 0.894535918891047_dbl_kind, 0.898503103194058_dbl_kind, 0.912007786414195_dbl_kind, 0.967137997777741_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,824) = (/ 0.891173373999997_dbl_kind, 0.894536822750545_dbl_kind, 0.898507064266858_dbl_kind, 0.912019875986475_dbl_kind, 0.967155275412928_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,825) = (/ 0.891173751301731_dbl_kind, 0.894537725919794_dbl_kind, 0.898511023351134_dbl_kind, 0.91203195995046_dbl_kind, 0.967172514672245_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,826) = (/ 0.891174127590621_dbl_kind, 0.894538628403341_dbl_kind, 0.898514980451596_dbl_kind, 0.912044038319731_dbl_kind, 0.967189715664973_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,827) = (/ 0.891174502870442_dbl_kind, 0.894539530205687_dbl_kind, 0.898518935572924_dbl_kind, 0.912056111107788_dbl_kind, 0.967206878500002_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,828) = (/ 0.891174877144957_dbl_kind, 0.894540431331288_dbl_kind, 0.898522888719769_dbl_kind, 0.912068178328055_dbl_kind, 0.967224003285833_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,829) = (/ 0.891175250417912_dbl_kind, 0.894541331784551_dbl_kind, 0.898526839896751_dbl_kind, 0.912080239993883_dbl_kind, 0.967241090130568_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,830) = (/ 0.891175622693045_dbl_kind, 0.894542231569839_dbl_kind, 0.898530789108461_dbl_kind, 0.912092296118541_dbl_kind, 0.967258139141928_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,831) = (/ 0.891175993974076_dbl_kind, 0.894543130691471_dbl_kind, 0.898534736359459_dbl_kind, 0.912104346715228_dbl_kind, 0.967275150427239_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,832) = (/ 0.891176364264712_dbl_kind, 0.894544029153717_dbl_kind, 0.898538681654281_dbl_kind, 0.912116391797066_dbl_kind, 0.967292124093446_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,833) = (/ 0.891176733568649_dbl_kind, 0.894544926960807_dbl_kind, 0.898542624997428_dbl_kind, 0.912128431377102_dbl_kind, 0.967309060247111_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,834) = (/ 0.891177101889567_dbl_kind, 0.894545824116924_dbl_kind, 0.898546566393378_dbl_kind, 0.91214046546831_dbl_kind, 0.967325958994408_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,835) = (/ 0.891177469231135_dbl_kind, 0.894546720626207_dbl_kind, 0.898550505846578_dbl_kind, 0.912152494083589_dbl_kind, 0.967342820441139_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,836) = (/ 0.891177835597006_dbl_kind, 0.894547616492753_dbl_kind, 0.898554443361448_dbl_kind, 0.912164517235767_dbl_kind, 0.96735964469272_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,837) = (/ 0.891178200990823_dbl_kind, 0.894548511720615_dbl_kind, 0.89855837894238_dbl_kind, 0.912176534937599_dbl_kind, 0.967376431854188_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,838) = (/ 0.891178565416212_dbl_kind, 0.894549406313804_dbl_kind, 0.898562312593737_dbl_kind, 0.912188547201767_dbl_kind, 0.967393182030215_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,839) = (/ 0.891178928876789_dbl_kind, 0.894550300276287_dbl_kind, 0.898566244319859_dbl_kind, 0.912200554040883_dbl_kind, 0.967409895325087_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,840) = (/ 0.891179291376155_dbl_kind, 0.89455119361199_dbl_kind, 0.898570174125055_dbl_kind, 0.912212555467485_dbl_kind, 0.967426571842724_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,841) = (/ 0.891179652917898_dbl_kind, 0.894552086324797_dbl_kind, 0.898574102013609_dbl_kind, 0.912224551494044_dbl_kind, 0.967443211686676_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,842) = (/ 0.891180013505594_dbl_kind, 0.894552978418552_dbl_kind, 0.898578027989779_dbl_kind, 0.912236542132958_dbl_kind, 0.967459814960118_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,843) = (/ 0.891180373142804_dbl_kind, 0.894553869897055_dbl_kind, 0.898581952057796_dbl_kind, 0.912248527396557_dbl_kind, 0.967476381765862_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,844) = (/ 0.891180731833077_dbl_kind, 0.894554760764068_dbl_kind, 0.898585874221865_dbl_kind, 0.912260507297101_dbl_kind, 0.967492912206357_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,845) = (/ 0.89118108957995_dbl_kind, 0.894555651023311_dbl_kind, 0.898589794486166_dbl_kind, 0.91227248184678_dbl_kind, 0.967509406383678_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,846) = (/ 0.891181446386945_dbl_kind, 0.894556540678466_dbl_kind, 0.898593712854852_dbl_kind, 0.912284451057719_dbl_kind, 0.967525864399547_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,847) = (/ 0.891181802257571_dbl_kind, 0.894557429733174_dbl_kind, 0.898597629332052_dbl_kind, 0.912296414941971_dbl_kind, 0.96754228635532_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,848) = (/ 0.891182157195326_dbl_kind, 0.894558318191036_dbl_kind, 0.898601543921871_dbl_kind, 0.912308373511524_dbl_kind, 0.967558672351992_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,849) = (/ 0.891182511203695_dbl_kind, 0.894559206055615_dbl_kind, 0.898605456628385_dbl_kind, 0.912320326778299_dbl_kind, 0.967575022490204_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,850) = (/ 0.891182864286146_dbl_kind, 0.894560093330435_dbl_kind, 0.89860936745565_dbl_kind, 0.912332274754149_dbl_kind, 0.967591336870239_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,851) = (/ 0.89118321644614_dbl_kind, 0.894560980018983_dbl_kind, 0.898613276407697_dbl_kind, 0.912344217450862_dbl_kind, 0.967607615592023_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,852) = (/ 0.891183567687121_dbl_kind, 0.894561866124706_dbl_kind, 0.898617183488528_dbl_kind, 0.912356154880159_dbl_kind, 0.967623858755129_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,853) = (/ 0.891183918012521_dbl_kind, 0.894562751651015_dbl_kind, 0.898621088702128_dbl_kind, 0.912368087053698_dbl_kind, 0.967640066458781_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,854) = (/ 0.891184267425761_dbl_kind, 0.894563636601283_dbl_kind, 0.898624992052453_dbl_kind, 0.912380013983069_dbl_kind, 0.96765623880185_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,855) = (/ 0.891184615930246_dbl_kind, 0.894564520978845_dbl_kind, 0.898628893543438_dbl_kind, 0.9123919356798_dbl_kind, 0.967672375882858_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,856) = (/ 0.891184963529373_dbl_kind, 0.894565404787001_dbl_kind, 0.898632793178994_dbl_kind, 0.912403852155355_dbl_kind, 0.967688477799981_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,857) = (/ 0.891185310226521_dbl_kind, 0.894566288029015_dbl_kind, 0.898636690963009_dbl_kind, 0.912415763421133_dbl_kind, 0.967704544651047_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,858) = (/ 0.89118565602506_dbl_kind, 0.894567170708112_dbl_kind, 0.898640586899347_dbl_kind, 0.91242766948847_dbl_kind, 0.967720576533542_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,859) = (/ 0.891186000928346_dbl_kind, 0.894568052827485_dbl_kind, 0.898644480991853_dbl_kind, 0.912439570368639_dbl_kind, 0.967736573544607_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,860) = (/ 0.891186344939723_dbl_kind, 0.894568934390289_dbl_kind, 0.898648373244345_dbl_kind, 0.912451466072853_dbl_kind, 0.967752535781044_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,861) = (/ 0.891186688062522_dbl_kind, 0.894569815399645_dbl_kind, 0.898652263660622_dbl_kind, 0.91246335661226_dbl_kind, 0.967768463339312_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,862) = (/ 0.891187030300061_dbl_kind, 0.894570695858639_dbl_kind, 0.898656152244458_dbl_kind, 0.912475241997948_dbl_kind, 0.967784356315533_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,863) = (/ 0.891187371655646_dbl_kind, 0.894571575770323_dbl_kind, 0.898660038999608_dbl_kind, 0.912487122240944_dbl_kind, 0.967800214805491_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,864) = (/ 0.891187712132572_dbl_kind, 0.894572455137714_dbl_kind, 0.898663923929804_dbl_kind, 0.912498997352213_dbl_kind, 0.967816038904637_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,865) = (/ 0.891188051734118_dbl_kind, 0.894573333963795_dbl_kind, 0.898667807038757_dbl_kind, 0.912510867342661_dbl_kind, 0.967831828708085_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,866) = (/ 0.891188390463554_dbl_kind, 0.894574212251516_dbl_kind, 0.898671688330156_dbl_kind, 0.912522732223134_dbl_kind, 0.967847584310616_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,867) = (/ 0.891188728324137_dbl_kind, 0.894575090003795_dbl_kind, 0.89867556780767_dbl_kind, 0.912534592004416_dbl_kind, 0.967863305806682_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,868) = (/ 0.89118906531911_dbl_kind, 0.894575967223515_dbl_kind, 0.898679445474946_dbl_kind, 0.912546446697237_dbl_kind, 0.967878993290403_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,869) = (/ 0.891189401451705_dbl_kind, 0.894576843913527_dbl_kind, 0.898683321335612_dbl_kind, 0.912558296312262_dbl_kind, 0.967894646855571_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,870) = (/ 0.891189736725141_dbl_kind, 0.894577720076649_dbl_kind, 0.898687195393274_dbl_kind, 0.912570140860103_dbl_kind, 0.967910266595652_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,871) = (/ 0.891190071142627_dbl_kind, 0.89457859571567_dbl_kind, 0.898691067651518_dbl_kind, 0.912581980351311_dbl_kind, 0.967925852603785_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,872) = (/ 0.891190404707355_dbl_kind, 0.894579470833342_dbl_kind, 0.898694938113911_dbl_kind, 0.91259381479638_dbl_kind, 0.967941404972783_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,873) = (/ 0.89119073742251_dbl_kind, 0.89458034543239_dbl_kind, 0.898698806783998_dbl_kind, 0.912605644205746_dbl_kind, 0.96795692379514_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,874) = (/ 0.891191069291263_dbl_kind, 0.894581219515506_dbl_kind, 0.898702673665308_dbl_kind, 0.912617468589791_dbl_kind, 0.967972409163027_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,875) = (/ 0.891191400316771_dbl_kind, 0.89458209308535_dbl_kind, 0.898706538761345_dbl_kind, 0.912629287958837_dbl_kind, 0.967987861168293_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,876) = (/ 0.891191730502181_dbl_kind, 0.894582966144555_dbl_kind, 0.898710402075599_dbl_kind, 0.912641102323152_dbl_kind, 0.968003279902468_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,877) = (/ 0.891192059850628_dbl_kind, 0.894583838695719_dbl_kind, 0.898714263611539_dbl_kind, 0.912652911692947_dbl_kind, 0.968018665456768_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,878) = (/ 0.891192388365234_dbl_kind, 0.894584710741412_dbl_kind, 0.898718123372614_dbl_kind, 0.91266471607838_dbl_kind, 0.968034017922087_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,879) = (/ 0.89119271604911_dbl_kind, 0.894585582284176_dbl_kind, 0.898721981362254_dbl_kind, 0.912676515489551_dbl_kind, 0.968049337389012_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,880) = (/ 0.891193042905354_dbl_kind, 0.894586453326521_dbl_kind, 0.898725837583873_dbl_kind, 0.912688309936508_dbl_kind, 0.96806462394781_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,881) = (/ 0.891193368937052_dbl_kind, 0.894587323870929_dbl_kind, 0.898729692040865_dbl_kind, 0.912700099429242_dbl_kind, 0.968079877688437_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,882) = (/ 0.891193694147281_dbl_kind, 0.894588193919851_dbl_kind, 0.898733544736604_dbl_kind, 0.912711883977693_dbl_kind, 0.968095098700539_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,883) = (/ 0.891194018539101_dbl_kind, 0.894589063475712_dbl_kind, 0.89873739567445_dbl_kind, 0.912723663591746_dbl_kind, 0.968110287073455_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,884) = (/ 0.891194342115565_dbl_kind, 0.894589932540907_dbl_kind, 0.898741244857741_dbl_kind, 0.912735438281231_dbl_kind, 0.968125442896211_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,885) = (/ 0.891194664879712_dbl_kind, 0.894590801117803_dbl_kind, 0.898745092289801_dbl_kind, 0.91274720805593_dbl_kind, 0.96814056625753_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,886) = (/ 0.891194986834569_dbl_kind, 0.894591669208739_dbl_kind, 0.898748937973933_dbl_kind, 0.912758972925568_dbl_kind, 0.968155657245824_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,887) = (/ 0.891195307983152_dbl_kind, 0.894592536816027_dbl_kind, 0.898752781913424_dbl_kind, 0.912770732899819_dbl_kind, 0.96817071594921_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,888) = (/ 0.891195628328466_dbl_kind, 0.894593403941951_dbl_kind, 0.898756624111546_dbl_kind, 0.912782487988307_dbl_kind, 0.968185742455493_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,889) = (/ 0.891195947873503_dbl_kind, 0.894594270588768_dbl_kind, 0.898760464571551_dbl_kind, 0.912794238200602_dbl_kind, 0.968200736852177_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,890) = (/ 0.891196266621244_dbl_kind, 0.894595136758707_dbl_kind, 0.898764303296675_dbl_kind, 0.912805983546224_dbl_kind, 0.968215699226473_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,891) = (/ 0.891196584574658_dbl_kind, 0.894596002453973_dbl_kind, 0.898768140290138_dbl_kind, 0.912817724034644_dbl_kind, 0.968230629665284_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,892) = (/ 0.891196901736703_dbl_kind, 0.894596867676741_dbl_kind, 0.898771975555143_dbl_kind, 0.912829459675281_dbl_kind, 0.96824552825522_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,893) = (/ 0.891197218110326_dbl_kind, 0.894597732429164_dbl_kind, 0.898775809094877_dbl_kind, 0.912841190477502_dbl_kind, 0.968260395082593_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,894) = (/ 0.891197533698461_dbl_kind, 0.894598596713365_dbl_kind, 0.898779640912509_dbl_kind, 0.912852916450629_dbl_kind, 0.968275230233417_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,895) = (/ 0.891197848504032_dbl_kind, 0.894599460531445_dbl_kind, 0.898783471011196_dbl_kind, 0.912864637603929_dbl_kind, 0.968290033793418_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,896) = (/ 0.891198162529951_dbl_kind, 0.894600323885476_dbl_kind, 0.898787299394075_dbl_kind, 0.912876353946624_dbl_kind, 0.968304805848021_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,897) = (/ 0.891198475779118_dbl_kind, 0.894601186777509_dbl_kind, 0.898791126064269_dbl_kind, 0.912888065487886_dbl_kind, 0.968319546482366_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,898) = (/ 0.891198788254424_dbl_kind, 0.894602049209565_dbl_kind, 0.898794951024886_dbl_kind, 0.912899772236839_dbl_kind, 0.968334255781299_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,899) = (/ 0.891199099958746_dbl_kind, 0.894602911183645_dbl_kind, 0.898798774279018_dbl_kind, 0.912911474202557_dbl_kind, 0.968348933829376_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,900) = (/ 0.89119941089495_dbl_kind, 0.894603772701723_dbl_kind, 0.898802595829741_dbl_kind, 0.91292317139407_dbl_kind, 0.968363580710864_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,901) = (/ 0.891199721065892_dbl_kind, 0.89460463376575_dbl_kind, 0.898806415680118_dbl_kind, 0.912934863820356_dbl_kind, 0.968378196509749_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,902) = (/ 0.891200030474418_dbl_kind, 0.894605494377652_dbl_kind, 0.898810233833195_dbl_kind, 0.912946551490349_dbl_kind, 0.968392781309726_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,903) = (/ 0.891200339123359_dbl_kind, 0.894606354539331_dbl_kind, 0.898814050292005_dbl_kind, 0.912958234412937_dbl_kind, 0.968407335194205_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,904) = (/ 0.891200647015538_dbl_kind, 0.894607214252667_dbl_kind, 0.898817865059564_dbl_kind, 0.912969912596957_dbl_kind, 0.968421858246315_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,905) = (/ 0.891200954153766_dbl_kind, 0.894608073519515_dbl_kind, 0.898821678138876_dbl_kind, 0.912981586051204_dbl_kind, 0.968436350548902_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,906) = (/ 0.891201260540844_dbl_kind, 0.894608932341709_dbl_kind, 0.89882548953293_dbl_kind, 0.912993254784427_dbl_kind, 0.968450812184529_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,907) = (/ 0.891201566179559_dbl_kind, 0.894609790721058_dbl_kind, 0.8988292992447_dbl_kind, 0.913004918805326_dbl_kind, 0.968465243235486_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,908) = (/ 0.89120187107269_dbl_kind, 0.89461064865935_dbl_kind, 0.898833107277147_dbl_kind, 0.913016578122559_dbl_kind, 0.968479643783772_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,909) = (/ 0.891202175223004_dbl_kind, 0.894611506158348_dbl_kind, 0.898836913633217_dbl_kind, 0.913028232744738_dbl_kind, 0.968494013911123_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,910) = (/ 0.891202478633258_dbl_kind, 0.894612363219797_dbl_kind, 0.898840718315843_dbl_kind, 0.913039882680431_dbl_kind, 0.968508353698984_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,911) = (/ 0.891202781306195_dbl_kind, 0.894613219845417_dbl_kind, 0.898844521327946_dbl_kind, 0.913051527938159_dbl_kind, 0.968522663228538_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,912) = (/ 0.89120308324455_dbl_kind, 0.894614076036907_dbl_kind, 0.89884832267243_dbl_kind, 0.913063168526403_dbl_kind, 0.968536942580682_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,913) = (/ 0.891203384451048_dbl_kind, 0.894614931795945_dbl_kind, 0.898852122352189_dbl_kind, 0.913074804453597_dbl_kind, 0.968551191836051_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,914) = (/ 0.891203684928401_dbl_kind, 0.894615787124186_dbl_kind, 0.898855920370102_dbl_kind, 0.913086435728135_dbl_kind, 0.968565411074998_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,915) = (/ 0.891203984679311_dbl_kind, 0.894616642023265_dbl_kind, 0.898859716729035_dbl_kind, 0.913098062358363_dbl_kind, 0.968579600377614_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,916) = (/ 0.89120428370647_dbl_kind, 0.894617496494797_dbl_kind, 0.898863511431844_dbl_kind, 0.913109684352587_dbl_kind, 0.968593759823714_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,917) = (/ 0.891204582012557_dbl_kind, 0.894618350540375_dbl_kind, 0.898867304481367_dbl_kind, 0.913121301719072_dbl_kind, 0.968607889492849_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,918) = (/ 0.891204879600243_dbl_kind, 0.894619204161573_dbl_kind, 0.898871095880435_dbl_kind, 0.913132914466037_dbl_kind, 0.968621989464298_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,919) = (/ 0.891205176472188_dbl_kind, 0.894620057359943_dbl_kind, 0.898874885631863_dbl_kind, 0.913144522601662_dbl_kind, 0.968636059817075_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,920) = (/ 0.891205472631041_dbl_kind, 0.894620910137017_dbl_kind, 0.898878673738454_dbl_kind, 0.913156126134083_dbl_kind, 0.968650100629934_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,921) = (/ 0.891205768079439_dbl_kind, 0.894621762494308_dbl_kind, 0.898882460203_dbl_kind, 0.913167725071398_dbl_kind, 0.968664111981358_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,922) = (/ 0.891206062820011_dbl_kind, 0.894622614433309_dbl_kind, 0.89888624502828_dbl_kind, 0.91317931942166_dbl_kind, 0.968678093949569_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,923) = (/ 0.891206356855373_dbl_kind, 0.894623465955494_dbl_kind, 0.898890028217062_dbl_kind, 0.913190909192882_dbl_kind, 0.968692046612529_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,924) = (/ 0.891206650188134_dbl_kind, 0.894624317062317_dbl_kind, 0.898893809772103_dbl_kind, 0.913202494393038_dbl_kind, 0.968705970047937_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,925) = (/ 0.89120694282089_dbl_kind, 0.894625167755214_dbl_kind, 0.898897589696145_dbl_kind, 0.913214075030061_dbl_kind, 0.968719864333233_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,926) = (/ 0.891207234756226_dbl_kind, 0.8946260180356_dbl_kind, 0.898901367991922_dbl_kind, 0.913225651111844_dbl_kind, 0.9687337295456_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,927) = (/ 0.891207525996718_dbl_kind, 0.894626867904873_dbl_kind, 0.898905144662154_dbl_kind, 0.913237222646239_dbl_kind, 0.968747565761958_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,928) = (/ 0.891207816544934_dbl_kind, 0.894627717364412_dbl_kind, 0.898908919709552_dbl_kind, 0.91324878964106_dbl_kind, 0.968761373058976_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,929) = (/ 0.891208106403427_dbl_kind, 0.894628566415578_dbl_kind, 0.898912693136814_dbl_kind, 0.913260352104082_dbl_kind, 0.968775151513062_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,930) = (/ 0.891208395574743_dbl_kind, 0.894629415059714_dbl_kind, 0.898916464946628_dbl_kind, 0.913271910043039_dbl_kind, 0.968788901200377_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,931) = (/ 0.891208684061417_dbl_kind, 0.894630263298144_dbl_kind, 0.898920235141672_dbl_kind, 0.913283463465628_dbl_kind, 0.968802622196822_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,932) = (/ 0.891208971865973_dbl_kind, 0.894631111132175_dbl_kind, 0.898924003724611_dbl_kind, 0.913295012379508_dbl_kind, 0.968816314578047_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,933) = (/ 0.891209258990928_dbl_kind, 0.894631958563095_dbl_kind, 0.898927770698101_dbl_kind, 0.913306556792298_dbl_kind, 0.968829978419451_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,934) = (/ 0.891209545438784_dbl_kind, 0.894632805592179_dbl_kind, 0.898931536064786_dbl_kind, 0.91331809671158_dbl_kind, 0.968843613796182_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,935) = (/ 0.891209831212037_dbl_kind, 0.894633652220679_dbl_kind, 0.898935299827302_dbl_kind, 0.9133296321449_dbl_kind, 0.968857220783141_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,936) = (/ 0.891210116313172_dbl_kind, 0.894634498449834_dbl_kind, 0.898939061988272_dbl_kind, 0.913341163099763_dbl_kind, 0.968870799454978_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,937) = (/ 0.891210400744663_dbl_kind, 0.894635344280863_dbl_kind, 0.898942822550312_dbl_kind, 0.913352689583639_dbl_kind, 0.968884349886093_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,938) = (/ 0.891210684508974_dbl_kind, 0.894636189714972_dbl_kind, 0.898946581516024_dbl_kind, 0.913364211603962_dbl_kind, 0.968897872150648_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,939) = (/ 0.891210967608562_dbl_kind, 0.894637034753347_dbl_kind, 0.898950338888004_dbl_kind, 0.913375729168128_dbl_kind, 0.968911366322552_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,940) = (/ 0.89121125004587_dbl_kind, 0.89463787939716_dbl_kind, 0.898954094668835_dbl_kind, 0.913387242283497_dbl_kind, 0.968924832475472_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,941) = (/ 0.891211531823334_dbl_kind, 0.894638723647566_dbl_kind, 0.898957848861093_dbl_kind, 0.913398750957393_dbl_kind, 0.968938270682833_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,942) = (/ 0.891211812943378_dbl_kind, 0.894639567505704_dbl_kind, 0.898961601467342_dbl_kind, 0.913410255197104_dbl_kind, 0.968951681017813_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,943) = (/ 0.89121209340842_dbl_kind, 0.894640410972696_dbl_kind, 0.898965352490138_dbl_kind, 0.913421755009881_dbl_kind, 0.968965063553355_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,944) = (/ 0.891212373220864_dbl_kind, 0.89464125404965_dbl_kind, 0.898969101932028_dbl_kind, 0.913433250402942_dbl_kind, 0.968978418362157_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,945) = (/ 0.891212652383106_dbl_kind, 0.894642096737658_dbl_kind, 0.898972849795549_dbl_kind, 0.913444741383468_dbl_kind, 0.968991745516678_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,946) = (/ 0.891212930897534_dbl_kind, 0.894642939037797_dbl_kind, 0.898976596083227_dbl_kind, 0.913456227958608_dbl_kind, 0.969005045089141_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,947) = (/ 0.891213208766524_dbl_kind, 0.894643780951128_dbl_kind, 0.898980340797585_dbl_kind, 0.913467710135471_dbl_kind, 0.969018317151528_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,948) = (/ 0.891213485992444_dbl_kind, 0.894644622478697_dbl_kind, 0.898984083941129_dbl_kind, 0.913479187921137_dbl_kind, 0.969031561775588_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,949) = (/ 0.89121376257765_dbl_kind, 0.894645463621535_dbl_kind, 0.898987825516362_dbl_kind, 0.913490661322649_dbl_kind, 0.969044779032829_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,950) = (/ 0.891214038524492_dbl_kind, 0.89464630438066_dbl_kind, 0.898991565525777_dbl_kind, 0.913502130347015_dbl_kind, 0.969057968994534_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,951) = (/ 0.891214313835307_dbl_kind, 0.894647144757074_dbl_kind, 0.898995303971857_dbl_kind, 0.913513595001212_dbl_kind, 0.969071131731737_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,952) = (/ 0.891214588512427_dbl_kind, 0.894647984751764_dbl_kind, 0.898999040857077_dbl_kind, 0.913525055292182_dbl_kind, 0.969084267315254_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,953) = (/ 0.891214862558169_dbl_kind, 0.894648824365704_dbl_kind, 0.899002776183906_dbl_kind, 0.913536511226835_dbl_kind, 0.969097375815662_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,954) = (/ 0.891215135974846_dbl_kind, 0.894649663599853_dbl_kind, 0.899006509954802_dbl_kind, 0.913547962812045_dbl_kind, 0.969110457303307_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,955) = (/ 0.891215408764757_dbl_kind, 0.894650502455157_dbl_kind, 0.899010242172215_dbl_kind, 0.913559410054656_dbl_kind, 0.969123511848308_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,956) = (/ 0.891215680930195_dbl_kind, 0.894651340932548_dbl_kind, 0.899013972838589_dbl_kind, 0.913570852961479_dbl_kind, 0.969136539520551_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,957) = (/ 0.891215952473442_dbl_kind, 0.894652179032943_dbl_kind, 0.899017701956357_dbl_kind, 0.913582291539291_dbl_kind, 0.969149540389697_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,958) = (/ 0.891216223396773_dbl_kind, 0.894653016757248_dbl_kind, 0.899021429527947_dbl_kind, 0.91359372579484_dbl_kind, 0.969162514525179_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,959) = (/ 0.891216493702449_dbl_kind, 0.894653854106352_dbl_kind, 0.899025155555777_dbl_kind, 0.913605155734837_dbl_kind, 0.9691754619962_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,960) = (/ 0.891216763392728_dbl_kind, 0.894654691081136_dbl_kind, 0.899028880042259_dbl_kind, 0.913616581365967_dbl_kind, 0.969188382871744_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,961) = (/ 0.891217032469854_dbl_kind, 0.894655527682462_dbl_kind, 0.899032602989796_dbl_kind, 0.91362800269488_dbl_kind, 0.969201277220564_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,962) = (/ 0.891217300936065_dbl_kind, 0.894656363911184_dbl_kind, 0.899036324400786_dbl_kind, 0.913639419728194_dbl_kind, 0.969214145111193_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,963) = (/ 0.891217568793587_dbl_kind, 0.89465719976814_dbl_kind, 0.899040044277617_dbl_kind, 0.913650832472499_dbl_kind, 0.969226986611943_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,964) = (/ 0.89121783604464_dbl_kind, 0.894658035254158_dbl_kind, 0.89904376262267_dbl_kind, 0.913662240934353_dbl_kind, 0.9692398017909_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,965) = (/ 0.891218102691432_dbl_kind, 0.89465887037005_dbl_kind, 0.89904747943832_dbl_kind, 0.913673645120281_dbl_kind, 0.96925259071593_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,966) = (/ 0.891218368736166_dbl_kind, 0.89465970511662_dbl_kind, 0.899051194726935_dbl_kind, 0.913685045036781_dbl_kind, 0.969265353454678_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,967) = (/ 0.891218634181031_dbl_kind, 0.894660539494657_dbl_kind, 0.899054908490874_dbl_kind, 0.91369644069032_dbl_kind, 0.969278090074577_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,968) = (/ 0.891218899028212_dbl_kind, 0.894661373504937_dbl_kind, 0.899058620732491_dbl_kind, 0.913707832087333_dbl_kind, 0.969290800642828_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,969) = (/ 0.891219163279882_dbl_kind, 0.894662207148227_dbl_kind, 0.899062331454133_dbl_kind, 0.913719219234227_dbl_kind, 0.969303485226426_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,970) = (/ 0.891219426938206_dbl_kind, 0.89466304042528_dbl_kind, 0.89906604065814_dbl_kind, 0.913730602137378_dbl_kind, 0.969316143892148_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,971) = (/ 0.891219690005341_dbl_kind, 0.894663873336837_dbl_kind, 0.899069748346844_dbl_kind, 0.913741980803135_dbl_kind, 0.969328776706547_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,972) = (/ 0.891219952483433_dbl_kind, 0.894664705883631_dbl_kind, 0.899073454522573_dbl_kind, 0.913753355237814_dbl_kind, 0.96934138373597_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,973) = (/ 0.891220214374621_dbl_kind, 0.894665538066378_dbl_kind, 0.899077159187646_dbl_kind, 0.913764725447707_dbl_kind, 0.969353965046548_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,974) = (/ 0.891220475681036_dbl_kind, 0.894666369885788_dbl_kind, 0.899080862344379_dbl_kind, 0.913776091439072_dbl_kind, 0.969366520704194_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,975) = (/ 0.8912207364048_dbl_kind, 0.894667201342556_dbl_kind, 0.899084563995077_dbl_kind, 0.913787453218143_dbl_kind, 0.969379050774614_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,976) = (/ 0.891220996548024_dbl_kind, 0.894668032437368_dbl_kind, 0.899088264142043_dbl_kind, 0.913798810791121_dbl_kind, 0.969391555323299_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,977) = (/ 0.891221256112813_dbl_kind, 0.894668863170898_dbl_kind, 0.899091962787571_dbl_kind, 0.913810164164183_dbl_kind, 0.969404034415531_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,978) = (/ 0.891221515101263_dbl_kind, 0.89466969354381_dbl_kind, 0.899095659933952_dbl_kind, 0.913821513343476_dbl_kind, 0.969416488116384_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,979) = (/ 0.891221773515459_dbl_kind, 0.894670523556757_dbl_kind, 0.899099355583467_dbl_kind, 0.913832858335119_dbl_kind, 0.969428916490717_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,980) = (/ 0.891222031357482_dbl_kind, 0.894671353210381_dbl_kind, 0.899103049738396_dbl_kind, 0.913844199145204_dbl_kind, 0.969441319603186_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,981) = (/ 0.891222288629401_dbl_kind, 0.894672182505315_dbl_kind, 0.899106742401008_dbl_kind, 0.913855535779795_dbl_kind, 0.969453697518237_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,982) = (/ 0.891222545333278_dbl_kind, 0.89467301144218_dbl_kind, 0.89911043357357_dbl_kind, 0.913866868244929_dbl_kind, 0.969466050300112_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,983) = (/ 0.891222801471166_dbl_kind, 0.894673840021587_dbl_kind, 0.899114123258343_dbl_kind, 0.913878196546615_dbl_kind, 0.969478378012843_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,984) = (/ 0.89122305704511_dbl_kind, 0.894674668244138_dbl_kind, 0.89911781145758_dbl_kind, 0.913889520690837_dbl_kind, 0.969490680720261_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,985) = (/ 0.891223312057146_dbl_kind, 0.894675496110425_dbl_kind, 0.899121498173531_dbl_kind, 0.913900840683551_dbl_kind, 0.969502958485991_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,986) = (/ 0.891223566509302_dbl_kind, 0.894676323621028_dbl_kind, 0.89912518340844_dbl_kind, 0.913912156530688_dbl_kind, 0.969515211373454_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,987) = (/ 0.891223820403599_dbl_kind, 0.894677150776519_dbl_kind, 0.899128867164546_dbl_kind, 0.91392346823815_dbl_kind, 0.96952743944587_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,988) = (/ 0.891224073742048_dbl_kind, 0.894677977577461_dbl_kind, 0.899132549444081_dbl_kind, 0.913934775811814_dbl_kind, 0.969539642766254_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,989) = (/ 0.891224326526653_dbl_kind, 0.894678804024407_dbl_kind, 0.899136230249273_dbl_kind, 0.913946079257533_dbl_kind, 0.969551821397425_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,990) = (/ 0.891224578759409_dbl_kind, 0.894679630117898_dbl_kind, 0.899139909582346_dbl_kind, 0.913957378581131_dbl_kind, 0.969563975401998_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,991) = (/ 0.891224830442302_dbl_kind, 0.894680455858471_dbl_kind, 0.899143587445517_dbl_kind, 0.913968673788408_dbl_kind, 0.96957610484239_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,992) = (/ 0.891225081577313_dbl_kind, 0.894681281246648_dbl_kind, 0.899147263841_dbl_kind, 0.913979964885139_dbl_kind, 0.969588209780819_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,993) = (/ 0.891225332166412_dbl_kind, 0.894682106282946_dbl_kind, 0.899150938771002_dbl_kind, 0.913991251877072_dbl_kind, 0.969600290279303_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,994) = (/ 0.891225582211561_dbl_kind, 0.894682930967871_dbl_kind, 0.899154612237727_dbl_kind, 0.914002534769932_dbl_kind, 0.969612346399665_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,995) = (/ 0.891225831714716_dbl_kind, 0.894683755301921_dbl_kind, 0.899158284243373_dbl_kind, 0.914013813569417_dbl_kind, 0.969624378203534_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,996) = (/ 0.891226080677824_dbl_kind, 0.894684579285586_dbl_kind, 0.899161954790135_dbl_kind, 0.914025088281203_dbl_kind, 0.969636385752338_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,997) = (/ 0.891226329102824_dbl_kind, 0.894685402919345_dbl_kind, 0.899165623880201_dbl_kind, 0.914036358910937_dbl_kind, 0.969648369107312_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,998) = (/ 0.891226576991646_dbl_kind, 0.89468622620367_dbl_kind, 0.899169291515756_dbl_kind, 0.914047625464247_dbl_kind, 0.969660328329503_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,999) = (/ 0.891226824346213_dbl_kind, 0.894687049139025_dbl_kind, 0.899172957698981_dbl_kind, 0.914058887946731_dbl_kind, 0.969672263479754_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1000) = (/ 0.891227071168442_dbl_kind, 0.894687871725864_dbl_kind, 0.899176622432052_dbl_kind, 0.914070146363969_dbl_kind, 0.969684174618725_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1001) = (/ 0.891227317460239_dbl_kind, 0.894688693964635_dbl_kind, 0.899180285717139_dbl_kind, 0.91408140072151_dbl_kind, 0.969696061806877_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1002) = (/ 0.891227563223504_dbl_kind, 0.894689515855775_dbl_kind, 0.899183947556411_dbl_kind, 0.914092651024887_dbl_kind, 0.969707925104485_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1003) = (/ 0.89122780846013_dbl_kind, 0.894690337399715_dbl_kind, 0.899187607952031_dbl_kind, 0.914103897279603_dbl_kind, 0.96971976457163_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1004) = (/ 0.891228053171999_dbl_kind, 0.894691158596878_dbl_kind, 0.899191266906156_dbl_kind, 0.914115139491141_dbl_kind, 0.969731580268205_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1005) = (/ 0.891228297360988_dbl_kind, 0.894691979447679_dbl_kind, 0.899194924420944_dbl_kind, 0.914126377664959_dbl_kind, 0.969743372253915_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1006) = (/ 0.891228541028967_dbl_kind, 0.894692799952524_dbl_kind, 0.899198580498543_dbl_kind, 0.914137611806492_dbl_kind, 0.969755140588275_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1007) = (/ 0.891228784177796_dbl_kind, 0.894693620111813_dbl_kind, 0.899202235141102_dbl_kind, 0.914148841921155_dbl_kind, 0.969766885330615_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1008) = (/ 0.891229026809329_dbl_kind, 0.894694439925936_dbl_kind, 0.899205888350762_dbl_kind, 0.914160068014334_dbl_kind, 0.969778606540073_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1009) = (/ 0.891229268925412_dbl_kind, 0.894695259395278_dbl_kind, 0.899209540129664_dbl_kind, 0.914171290091399_dbl_kind, 0.969790304275612_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1010) = (/ 0.891229510527882_dbl_kind, 0.894696078520216_dbl_kind, 0.899213190479944_dbl_kind, 0.914182508157693_dbl_kind, 0.969801978595992_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1011) = (/ 0.891229751618572_dbl_kind, 0.894696897301119_dbl_kind, 0.899216839403731_dbl_kind, 0.914193722218539_dbl_kind, 0.969813629559806_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1012) = (/ 0.891229992199304_dbl_kind, 0.894697715738349_dbl_kind, 0.899220486903155_dbl_kind, 0.914204932279236_dbl_kind, 0.969825257225456_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1013) = (/ 0.891230232271893_dbl_kind, 0.894698533832262_dbl_kind, 0.899224132980341_dbl_kind, 0.914216138345061_dbl_kind, 0.969836861651157_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1014) = (/ 0.89123047183815_dbl_kind, 0.894699351583205_dbl_kind, 0.899227777637409_dbl_kind, 0.914227340421272_dbl_kind, 0.969848442894949_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1015) = (/ 0.891230710899874_dbl_kind, 0.89470016899152_dbl_kind, 0.899231420876477_dbl_kind, 0.914238538513101_dbl_kind, 0.969860001014686_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1016) = (/ 0.89123094945886_dbl_kind, 0.89470098605754_dbl_kind, 0.899235062699658_dbl_kind, 0.914249732625762_dbl_kind, 0.96987153606804_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1017) = (/ 0.891231187516893_dbl_kind, 0.894701802781592_dbl_kind, 0.899238703109063_dbl_kind, 0.914260922764444_dbl_kind, 0.969883048112505_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1018) = (/ 0.891231425075753_dbl_kind, 0.894702619163998_dbl_kind, 0.8992423421068_dbl_kind, 0.914272108934319_dbl_kind, 0.969894537205393_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1019) = (/ 0.891231662137211_dbl_kind, 0.894703435205072_dbl_kind, 0.899245979694974_dbl_kind, 0.914283291140534_dbl_kind, 0.969906003403842_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1020) = (/ 0.891231898703032_dbl_kind, 0.894704250905121_dbl_kind, 0.899249615875684_dbl_kind, 0.914294469388215_dbl_kind, 0.969917446764807_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1021) = (/ 0.891232134774974_dbl_kind, 0.894705066264445_dbl_kind, 0.899253250651029_dbl_kind, 0.91430564368247_dbl_kind, 0.969928867345064_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1022) = (/ 0.891232370354786_dbl_kind, 0.89470588128334_dbl_kind, 0.899256884023102_dbl_kind, 0.914316814028383_dbl_kind, 0.969940265201216_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1023) = (/ 0.891232605444211_dbl_kind, 0.894706695962094_dbl_kind, 0.899260515993997_dbl_kind, 0.91432798043102_dbl_kind, 0.969951640389691_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1024) = (/ 0.891232840044986_dbl_kind, 0.89470751030099_dbl_kind, 0.899264146565801_dbl_kind, 0.914339142895427_dbl_kind, 0.969962992966734_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1025) = (/ 0.891233074158838_dbl_kind, 0.894708324300303_dbl_kind, 0.8992677757406_dbl_kind, 0.914350301426625_dbl_kind, 0.969974322988425_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1026) = (/ 0.89123330778749_dbl_kind, 0.894709137960303_dbl_kind, 0.899271403520476_dbl_kind, 0.914361456029618_dbl_kind, 0.969985630510661_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1027) = (/ 0.891233540932655_dbl_kind, 0.894709951281255_dbl_kind, 0.899275029907511_dbl_kind, 0.914372606709391_dbl_kind, 0.969996915589172_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1028) = (/ 0.891233773596043_dbl_kind, 0.894710764263416_dbl_kind, 0.899278654903779_dbl_kind, 0.914383753470907_dbl_kind, 0.970008178279508_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1029) = (/ 0.891234005779352_dbl_kind, 0.89471157690704_dbl_kind, 0.899282278511356_dbl_kind, 0.914394896319109_dbl_kind, 0.970019418637055_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1030) = (/ 0.891234237484277_dbl_kind, 0.894712389212371_dbl_kind, 0.899285900732314_dbl_kind, 0.914406035258923_dbl_kind, 0.97003063671702_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1031) = (/ 0.891234468712505_dbl_kind, 0.894713201179652_dbl_kind, 0.899289521568721_dbl_kind, 0.914417170295251_dbl_kind, 0.970041832574444_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1032) = (/ 0.891234699465715_dbl_kind, 0.894714012809117_dbl_kind, 0.899293141022643_dbl_kind, 0.914428301432979_dbl_kind, 0.970053006264197_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1033) = (/ 0.89123492974558_dbl_kind, 0.894714824100997_dbl_kind, 0.899296759096146_dbl_kind, 0.914439428676974_dbl_kind, 0.970064157840974_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1034) = (/ 0.891235159553766_dbl_kind, 0.894715635055516_dbl_kind, 0.899300375791288_dbl_kind, 0.914450552032079_dbl_kind, 0.97007528735931_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1035) = (/ 0.891235388891933_dbl_kind, 0.894716445672891_dbl_kind, 0.899303991110129_dbl_kind, 0.914461671503125_dbl_kind, 0.970086394873561_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1036) = (/ 0.891235617761733_dbl_kind, 0.894717255953339_dbl_kind, 0.899307605054725_dbl_kind, 0.914472787094917_dbl_kind, 0.970097480437925_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1037) = (/ 0.891235846164811_dbl_kind, 0.894718065897066_dbl_kind, 0.89931121762713_dbl_kind, 0.914483898812248_dbl_kind, 0.970108544106431_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1038) = (/ 0.891236074102807_dbl_kind, 0.894718875504277_dbl_kind, 0.899314828829396_dbl_kind, 0.914495006659886_dbl_kind, 0.970119585932935_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1039) = (/ 0.891236301577353_dbl_kind, 0.894719684775168_dbl_kind, 0.89931843866357_dbl_kind, 0.914506110642584_dbl_kind, 0.970130605971131_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1040) = (/ 0.891236528590075_dbl_kind, 0.894720493709934_dbl_kind, 0.899322047131701_dbl_kind, 0.914517210765076_dbl_kind, 0.970141604274551_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1041) = (/ 0.891236755142591_dbl_kind, 0.894721302308762_dbl_kind, 0.899325654235833_dbl_kind, 0.914528307032078_dbl_kind, 0.97015258089656_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1042) = (/ 0.891236981236514_dbl_kind, 0.894722110571836_dbl_kind, 0.899329259978007_dbl_kind, 0.914539399448285_dbl_kind, 0.970163535890356_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1043) = (/ 0.891237206873449_dbl_kind, 0.894722918499334_dbl_kind, 0.899332864360265_dbl_kind, 0.914550488018378_dbl_kind, 0.970174469308977_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1044) = (/ 0.891237432054996_dbl_kind, 0.89472372609143_dbl_kind, 0.899336467384644_dbl_kind, 0.914561572747017_dbl_kind, 0.970185381205298_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1045) = (/ 0.891237656782747_dbl_kind, 0.894724533348293_dbl_kind, 0.89934006905318_dbl_kind, 0.914572653638845_dbl_kind, 0.970196271632029_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1046) = (/ 0.891237881058289_dbl_kind, 0.894725340270087_dbl_kind, 0.899343669367906_dbl_kind, 0.914583730698488_dbl_kind, 0.970207140641725_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1047) = (/ 0.891238104883201_dbl_kind, 0.894726146856972_dbl_kind, 0.899347268330855_dbl_kind, 0.914594803930552_dbl_kind, 0.970217988286772_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1048) = (/ 0.891238328259055_dbl_kind, 0.894726953109104_dbl_kind, 0.899350865944057_dbl_kind, 0.914605873339629_dbl_kind, 0.970228814619399_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1049) = (/ 0.891238551187421_dbl_kind, 0.894727759026632_dbl_kind, 0.899354462209539_dbl_kind, 0.914616938930291_dbl_kind, 0.970239619691676_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1050) = (/ 0.891238773669856_dbl_kind, 0.894728564609705_dbl_kind, 0.899358057129327_dbl_kind, 0.914628000707093_dbl_kind, 0.970250403555518_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1051) = (/ 0.891238995707915_dbl_kind, 0.894729369858463_dbl_kind, 0.899361650705446_dbl_kind, 0.914639058674572_dbl_kind, 0.97026116626267_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1052) = (/ 0.891239217303147_dbl_kind, 0.894730174773044_dbl_kind, 0.899365242939917_dbl_kind, 0.914650112837251_dbl_kind, 0.97027190786473_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1053) = (/ 0.891239438457092_dbl_kind, 0.894730979353584_dbl_kind, 0.89936883383476_dbl_kind, 0.914661163199633_dbl_kind, 0.970282628413133_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1054) = (/ 0.891239659171285_dbl_kind, 0.89473178360021_dbl_kind, 0.899372423391995_dbl_kind, 0.914672209766205_dbl_kind, 0.970293327959159_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1055) = (/ 0.891239879447254_dbl_kind, 0.894732587513049_dbl_kind, 0.899376011613638_dbl_kind, 0.914683252541437_dbl_kind, 0.970304006553931_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1056) = (/ 0.891240099286524_dbl_kind, 0.894733391092221_dbl_kind, 0.899379598501704_dbl_kind, 0.914694291529783_dbl_kind, 0.970314664248418_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1057) = (/ 0.891240318690608_dbl_kind, 0.894734194337846_dbl_kind, 0.899383184058207_dbl_kind, 0.914705326735679_dbl_kind, 0.970325301093431_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1058) = (/ 0.891240537661019_dbl_kind, 0.894734997250035_dbl_kind, 0.899386768285158_dbl_kind, 0.914716358163547_dbl_kind, 0.970335917139631_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1059) = (/ 0.89124075619926_dbl_kind, 0.8947357998289_dbl_kind, 0.899390351184567_dbl_kind, 0.914727385817789_dbl_kind, 0.970346512437518_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1060) = (/ 0.891240974306828_dbl_kind, 0.894736602074545_dbl_kind, 0.899393932758443_dbl_kind, 0.914738409702794_dbl_kind, 0.970357087037447_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1061) = (/ 0.891241191985214_dbl_kind, 0.894737403987073_dbl_kind, 0.899397513008792_dbl_kind, 0.914749429822934_dbl_kind, 0.970367640989616_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1062) = (/ 0.891241409235906_dbl_kind, 0.894738205566583_dbl_kind, 0.89940109193762_dbl_kind, 0.914760446182563_dbl_kind, 0.970378174344068_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1063) = (/ 0.891241626060381_dbl_kind, 0.894739006813169_dbl_kind, 0.899404669546931_dbl_kind, 0.914771458786023_dbl_kind, 0.970388687150701_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1064) = (/ 0.891241842460115_dbl_kind, 0.894739807726924_dbl_kind, 0.899408245838727_dbl_kind, 0.914782467637635_dbl_kind, 0.970399179459256_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1065) = (/ 0.891242058436573_dbl_kind, 0.894740608307934_dbl_kind, 0.899411820815008_dbl_kind, 0.914793472741708_dbl_kind, 0.970409651319328_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1066) = (/ 0.891242273991218_dbl_kind, 0.894741408556284_dbl_kind, 0.899415394477773_dbl_kind, 0.914804474102535_dbl_kind, 0.970420102780359_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1067) = (/ 0.891242489125505_dbl_kind, 0.894742208472056_dbl_kind, 0.899418966829022_dbl_kind, 0.914815471724391_dbl_kind, 0.970430533891643_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1068) = (/ 0.891242703840884_dbl_kind, 0.894743008055326_dbl_kind, 0.899422537870749_dbl_kind, 0.914826465611537_dbl_kind, 0.970440944702322_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1069) = (/ 0.891242918138798_dbl_kind, 0.89474380730617_dbl_kind, 0.899426107604951_dbl_kind, 0.91483745576822_dbl_kind, 0.970451335261396_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1070) = (/ 0.891243132020684_dbl_kind, 0.894744606224657_dbl_kind, 0.899429676033621_dbl_kind, 0.914848442198669_dbl_kind, 0.970461705617713_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1071) = (/ 0.891243345487977_dbl_kind, 0.894745404810857_dbl_kind, 0.899433243158751_dbl_kind, 0.9148594249071_dbl_kind, 0.970472055819971_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1072) = (/ 0.891243558542099_dbl_kind, 0.894746203064832_dbl_kind, 0.899436808982332_dbl_kind, 0.914870403897712_dbl_kind, 0.970482385916727_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1073) = (/ 0.891243771184473_dbl_kind, 0.894747000986647_dbl_kind, 0.899440373506354_dbl_kind, 0.91488137917469_dbl_kind, 0.970492695956392_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1074) = (/ 0.891243983416513_dbl_kind, 0.894747798576357_dbl_kind, 0.899443936732806_dbl_kind, 0.914892350742204_dbl_kind, 0.970502985987225_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1075) = (/ 0.891244195239628_dbl_kind, 0.89474859583402_dbl_kind, 0.899447498663675_dbl_kind, 0.914903318604409_dbl_kind, 0.970513256057347_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1076) = (/ 0.89124440665522_dbl_kind, 0.894749392759688_dbl_kind, 0.899451059300947_dbl_kind, 0.914914282765446_dbl_kind, 0.970523506214729_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1077) = (/ 0.891244617664686_dbl_kind, 0.894750189353409_dbl_kind, 0.899454618646606_dbl_kind, 0.91492524322944_dbl_kind, 0.970533736507203_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1078) = (/ 0.89124482826942_dbl_kind, 0.894750985615232_dbl_kind, 0.899458176702636_dbl_kind, 0.914936200000501_dbl_kind, 0.970543946982453_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1079) = (/ 0.891245038470804_dbl_kind, 0.8947517815452_dbl_kind, 0.89946173347102_dbl_kind, 0.914947153082726_dbl_kind, 0.970554137688023_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1080) = (/ 0.891245248270222_dbl_kind, 0.894752577143353_dbl_kind, 0.899465288953739_dbl_kind, 0.914958102480198_dbl_kind, 0.970564308671313_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1081) = (/ 0.891245457669047_dbl_kind, 0.894753372409731_dbl_kind, 0.899468843152773_dbl_kind, 0.914969048196983_dbl_kind, 0.970574459979582_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1082) = (/ 0.891245666668647_dbl_kind, 0.894754167344369_dbl_kind, 0.899472396070101_dbl_kind, 0.914979990237134_dbl_kind, 0.970584591659946_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1083) = (/ 0.891245875270388_dbl_kind, 0.894754961947301_dbl_kind, 0.8994759477077_dbl_kind, 0.914990928604693_dbl_kind, 0.970594703759385_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1084) = (/ 0.891246083475625_dbl_kind, 0.894755756218556_dbl_kind, 0.899479498067549_dbl_kind, 0.915001863303681_dbl_kind, 0.970604796324732_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1085) = (/ 0.891246291285714_dbl_kind, 0.894756550158162_dbl_kind, 0.899483047151621_dbl_kind, 0.915012794338111_dbl_kind, 0.970614869402684_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1086) = (/ 0.891246498701998_dbl_kind, 0.894757343766145_dbl_kind, 0.899486594961893_dbl_kind, 0.91502372171198_dbl_kind, 0.970624923039798_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1087) = (/ 0.891246705725821_dbl_kind, 0.894758137042528_dbl_kind, 0.899490141500337_dbl_kind, 0.915034645429271_dbl_kind, 0.970634957282494_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1088) = (/ 0.891246912358518_dbl_kind, 0.894758929987331_dbl_kind, 0.899493686768927_dbl_kind, 0.915045565493954_dbl_kind, 0.970644972177045_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1089) = (/ 0.89124711860142_dbl_kind, 0.894759722600573_dbl_kind, 0.899497230769634_dbl_kind, 0.915056481909982_dbl_kind, 0.970654967769603_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1090) = (/ 0.891247324455852_dbl_kind, 0.894760514882269_dbl_kind, 0.899500773504427_dbl_kind, 0.9150673946813_dbl_kind, 0.970664944106167_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1091) = (/ 0.891247529923133_dbl_kind, 0.894761306832432_dbl_kind, 0.899504314975278_dbl_kind, 0.915078303811834_dbl_kind, 0.970674901232603_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1092) = (/ 0.891247735004579_dbl_kind, 0.894762098451075_dbl_kind, 0.899507855184154_dbl_kind, 0.9150892093055_dbl_kind, 0.970684839194647_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1093) = (/ 0.891247939701497_dbl_kind, 0.894762889738205_dbl_kind, 0.899511394133024_dbl_kind, 0.915100111166198_dbl_kind, 0.970694758037891_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1094) = (/ 0.891248144015193_dbl_kind, 0.894763680693829_dbl_kind, 0.899514931823854_dbl_kind, 0.915111009397818_dbl_kind, 0.970704657807799_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1095) = (/ 0.891248347946963_dbl_kind, 0.894764471317953_dbl_kind, 0.899518468258609_dbl_kind, 0.915121904004234_dbl_kind, 0.970714538549695_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1096) = (/ 0.891248551498102_dbl_kind, 0.894765261610578_dbl_kind, 0.899522003439254_dbl_kind, 0.915132794989308_dbl_kind, 0.970724400308768_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1097) = (/ 0.891248754669897_dbl_kind, 0.894766051571705_dbl_kind, 0.899525537367754_dbl_kind, 0.915143682356887_dbl_kind, 0.970734243130078_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1098) = (/ 0.891248957463631_dbl_kind, 0.894766841201333_dbl_kind, 0.899529070046071_dbl_kind, 0.915154566110808_dbl_kind, 0.970744067058547_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1099) = (/ 0.891249159880582_dbl_kind, 0.894767630499458_dbl_kind, 0.899532601476168_dbl_kind, 0.915165446254892_dbl_kind, 0.970753872138966_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1100) = (/ 0.891249361922022_dbl_kind, 0.894768419466074_dbl_kind, 0.899536131660006_dbl_kind, 0.91517632279295_dbl_kind, 0.970763658415996_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1101) = (/ 0.891249563589217_dbl_kind, 0.894769208101174_dbl_kind, 0.899539660599543_dbl_kind, 0.915187195728777_dbl_kind, 0.970773425934162_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1102) = (/ 0.89124976488343_dbl_kind, 0.894769996404749_dbl_kind, 0.899543188296742_dbl_kind, 0.915198065066157_dbl_kind, 0.970783174737859_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1103) = (/ 0.891249965805918_dbl_kind, 0.894770784376787_dbl_kind, 0.899546714753559_dbl_kind, 0.915208930808863_dbl_kind, 0.970792904871352_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1104) = (/ 0.891250166357932_dbl_kind, 0.894771572017276_dbl_kind, 0.899550239971953_dbl_kind, 0.91521979296065_dbl_kind, 0.970802616378774_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1105) = (/ 0.89125036654072_dbl_kind, 0.8947723593262_dbl_kind, 0.899553763953881_dbl_kind, 0.915230651525268_dbl_kind, 0.970812309304128_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1106) = (/ 0.891250566355522_dbl_kind, 0.894773146303543_dbl_kind, 0.899557286701297_dbl_kind, 0.915241506506446_dbl_kind, 0.970821983691289_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1107) = (/ 0.891250765803577_dbl_kind, 0.894773932949288_dbl_kind, 0.899560808216159_dbl_kind, 0.915252357907908_dbl_kind, 0.970831639584_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1108) = (/ 0.891250964886114_dbl_kind, 0.894774719263414_dbl_kind, 0.89956432850042_dbl_kind, 0.915263205733362_dbl_kind, 0.97084127702588_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1109) = (/ 0.891251163604361_dbl_kind, 0.894775505245899_dbl_kind, 0.899567847556032_dbl_kind, 0.915274049986503_dbl_kind, 0.970850896060415_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1110) = (/ 0.89125136195954_dbl_kind, 0.89477629089672_dbl_kind, 0.89957136538495_dbl_kind, 0.915284890671015_dbl_kind, 0.970860496730966_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1111) = (/ 0.891251559952866_dbl_kind, 0.894777076215854_dbl_kind, 0.899574881989125_dbl_kind, 0.91529572779057_dbl_kind, 0.970870079080765_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1112) = (/ 0.891251757585553_dbl_kind, 0.894777861203273_dbl_kind, 0.899578397370508_dbl_kind, 0.915306561348829_dbl_kind, 0.970879643152919_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1113) = (/ 0.891251954858806_dbl_kind, 0.894778645858949_dbl_kind, 0.899581911531049_dbl_kind, 0.915317391349437_dbl_kind, 0.970889188990406_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1114) = (/ 0.891252151773829_dbl_kind, 0.894779430182854_dbl_kind, 0.899585424472698_dbl_kind, 0.91532821779603_dbl_kind, 0.970898716636084_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1115) = (/ 0.891252348331817_dbl_kind, 0.894780214174957_dbl_kind, 0.899588936197403_dbl_kind, 0.915339040692231_dbl_kind, 0.970908226132676_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1116) = (/ 0.891252544533963_dbl_kind, 0.894780997835227_dbl_kind, 0.899592446707112_dbl_kind, 0.915349860041654_dbl_kind, 0.970917717522789_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1117) = (/ 0.891252740381455_dbl_kind, 0.894781781163627_dbl_kind, 0.899595956003773_dbl_kind, 0.915360675847896_dbl_kind, 0.970927190848898_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1118) = (/ 0.891252935875476_dbl_kind, 0.894782564160126_dbl_kind, 0.899599464089331_dbl_kind, 0.915371488114545_dbl_kind, 0.970936646153364_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1119) = (/ 0.891253131017202_dbl_kind, 0.894783346824687_dbl_kind, 0.899602970965731_dbl_kind, 0.915382296845179_dbl_kind, 0.970946083478409_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1120) = (/ 0.891253325807808_dbl_kind, 0.894784129157271_dbl_kind, 0.899606476634919_dbl_kind, 0.91539310204336_dbl_kind, 0.970955502866149_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1121) = (/ 0.891253520248462_dbl_kind, 0.894784911157841_dbl_kind, 0.899609981098839_dbl_kind, 0.915403903712643_dbl_kind, 0.970964904358562_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1122) = (/ 0.891253714340327_dbl_kind, 0.894785692826357_dbl_kind, 0.899613484359432_dbl_kind, 0.915414701856568_dbl_kind, 0.970974287997516_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1123) = (/ 0.891253908084563_dbl_kind, 0.894786474162776_dbl_kind, 0.899616986418643_dbl_kind, 0.915425496478666_dbl_kind, 0.970983653824748_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1124) = (/ 0.891254101482323_dbl_kind, 0.894787255167058_dbl_kind, 0.899620487278411_dbl_kind, 0.915436287582453_dbl_kind, 0.970993001881877_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1125) = (/ 0.891254294534758_dbl_kind, 0.894788035839159_dbl_kind, 0.899623986940678_dbl_kind, 0.915447075171438_dbl_kind, 0.9710023322104_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1126) = (/ 0.891254487243012_dbl_kind, 0.894788816179034_dbl_kind, 0.899627485407385_dbl_kind, 0.915457859249115_dbl_kind, 0.971011644851698_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1127) = (/ 0.891254679608226_dbl_kind, 0.894789596186638_dbl_kind, 0.899630982680468_dbl_kind, 0.915468639818969_dbl_kind, 0.971020939847024_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1128) = (/ 0.891254871631534_dbl_kind, 0.894790375861923_dbl_kind, 0.899634478761868_dbl_kind, 0.915479416884472_dbl_kind, 0.971030217237517_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1129) = (/ 0.89125506331407_dbl_kind, 0.894791155204843_dbl_kind, 0.899637973653522_dbl_kind, 0.915490190449088_dbl_kind, 0.971039477064194_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1130) = (/ 0.891255254656959_dbl_kind, 0.894791934215349_dbl_kind, 0.899641467357366_dbl_kind, 0.915500960516265_dbl_kind, 0.971048719367955_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1131) = (/ 0.891255445661323_dbl_kind, 0.89479271289339_dbl_kind, 0.899644959875336_dbl_kind, 0.915511727089444_dbl_kind, 0.971057944189577_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1132) = (/ 0.89125563632828_dbl_kind, 0.894793491238917_dbl_kind, 0.899648451209367_dbl_kind, 0.915522490172052_dbl_kind, 0.971067151569725_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1133) = (/ 0.891255826658942_dbl_kind, 0.894794269251878_dbl_kind, 0.899651941361394_dbl_kind, 0.915533249767508_dbl_kind, 0.971076341548942_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1134) = (/ 0.89125601665442_dbl_kind, 0.89479504693222_dbl_kind, 0.899655430333351_dbl_kind, 0.915544005879219_dbl_kind, 0.971085514167656_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1135) = (/ 0.891256206315815_dbl_kind, 0.894795824279889_dbl_kind, 0.899658918127169_dbl_kind, 0.915554758510578_dbl_kind, 0.971094669466176_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1136) = (/ 0.89125639564423_dbl_kind, 0.894796601294833_dbl_kind, 0.899662404744781_dbl_kind, 0.915565507664971_dbl_kind, 0.971103807484697_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1137) = (/ 0.891256584640757_dbl_kind, 0.894797377976995_dbl_kind, 0.899665890188118_dbl_kind, 0.915576253345772_dbl_kind, 0.971112928263295_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1138) = (/ 0.891256773306488_dbl_kind, 0.89479815432632_dbl_kind, 0.899669374459109_dbl_kind, 0.915586995556345_dbl_kind, 0.971122031841934_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1139) = (/ 0.89125696164251_dbl_kind, 0.894798930342752_dbl_kind, 0.899672857559686_dbl_kind, 0.91559773430004_dbl_kind, 0.971131118260458_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1140) = (/ 0.891257149649905_dbl_kind, 0.894799706026232_dbl_kind, 0.899676339491775_dbl_kind, 0.915608469580201_dbl_kind, 0.971140187558601_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1141) = (/ 0.89125733732975_dbl_kind, 0.894800481376704_dbl_kind, 0.899679820257304_dbl_kind, 0.915619201400158_dbl_kind, 0.971149239775978_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1142) = (/ 0.891257524683118_dbl_kind, 0.894801256394108_dbl_kind, 0.899683299858202_dbl_kind, 0.915629929763232_dbl_kind, 0.971158274952098_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1143) = (/ 0.891257711711079_dbl_kind, 0.894802031078385_dbl_kind, 0.899686778296392_dbl_kind, 0.915640654672732_dbl_kind, 0.971167293126344_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1144) = (/ 0.891257898414698_dbl_kind, 0.894802805429475_dbl_kind, 0.899690255573802_dbl_kind, 0.915651376131959_dbl_kind, 0.971176294337996_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1145) = (/ 0.891258084795034_dbl_kind, 0.894803579447316_dbl_kind, 0.899693731692355_dbl_kind, 0.9156620941442_dbl_kind, 0.971185278626217_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1146) = (/ 0.891258270853145_dbl_kind, 0.894804353131849_dbl_kind, 0.899697206653974_dbl_kind, 0.915672808712734_dbl_kind, 0.971194246030057_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1147) = (/ 0.891258456590081_dbl_kind, 0.89480512648301_dbl_kind, 0.899700680460582_dbl_kind, 0.91568351984083_dbl_kind, 0.971203196588457_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1148) = (/ 0.891258642006891_dbl_kind, 0.894805899500738_dbl_kind, 0.899704153114101_dbl_kind, 0.915694227531746_dbl_kind, 0.971212130340246_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1149) = (/ 0.891258827104618_dbl_kind, 0.894806672184969_dbl_kind, 0.899707624616451_dbl_kind, 0.915704931788728_dbl_kind, 0.971221047324134_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1150) = (/ 0.891259011884302_dbl_kind, 0.89480744453564_dbl_kind, 0.899711094969553_dbl_kind, 0.915715632615015_dbl_kind, 0.97122994757873_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1151) = (/ 0.891259196346977_dbl_kind, 0.894808216552686_dbl_kind, 0.899714564175325_dbl_kind, 0.915726330013833_dbl_kind, 0.971238831142531_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1152) = (/ 0.891259380493674_dbl_kind, 0.894808988236044_dbl_kind, 0.899718032235686_dbl_kind, 0.915737023988399_dbl_kind, 0.971247698053918_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1153) = (/ 0.891259564325421_dbl_kind, 0.894809759585648_dbl_kind, 0.899721499152551_dbl_kind, 0.91574771454192_dbl_kind, 0.971256548351167_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1154) = (/ 0.89125974784324_dbl_kind, 0.894810530601433_dbl_kind, 0.899724964927839_dbl_kind, 0.915758401677591_dbl_kind, 0.971265382072444_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1155) = (/ 0.89125993104815_dbl_kind, 0.894811301283333_dbl_kind, 0.899728429563463_dbl_kind, 0.915769085398601_dbl_kind, 0.971274199255804_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1156) = (/ 0.891260113941165_dbl_kind, 0.894812071631281_dbl_kind, 0.899731893061339_dbl_kind, 0.915779765708126_dbl_kind, 0.971282999939197_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1157) = (/ 0.891260296523296_dbl_kind, 0.894812841645212_dbl_kind, 0.899735355423379_dbl_kind, 0.915790442609331_dbl_kind, 0.97129178416046_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1158) = (/ 0.891260478795549_dbl_kind, 0.894813611325058_dbl_kind, 0.899738816651496_dbl_kind, 0.915801116105373_dbl_kind, 0.971300551957328_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1159) = (/ 0.891260660758926_dbl_kind, 0.894814380670752_dbl_kind, 0.899742276747602_dbl_kind, 0.9158117861994_dbl_kind, 0.971309303367423_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1160) = (/ 0.891260842414426_dbl_kind, 0.894815149682226_dbl_kind, 0.899745735713606_dbl_kind, 0.915822452894547_dbl_kind, 0.971318038428263_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1161) = (/ 0.891261023763043_dbl_kind, 0.894815918359412_dbl_kind, 0.899749193551417_dbl_kind, 0.915833116193942_dbl_kind, 0.971326757177258_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1162) = (/ 0.891261204805768_dbl_kind, 0.894816686702242_dbl_kind, 0.899752650262945_dbl_kind, 0.915843776100702_dbl_kind, 0.971335459651711_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1163) = (/ 0.891261385543586_dbl_kind, 0.894817454710648_dbl_kind, 0.899756105850098_dbl_kind, 0.915854432617933_dbl_kind, 0.971344145888822_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1164) = (/ 0.89126156597748_dbl_kind, 0.894818222384561_dbl_kind, 0.89975956031478_dbl_kind, 0.915865085748734_dbl_kind, 0.971352815925684_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1165) = (/ 0.89126174610843_dbl_kind, 0.894818989723912_dbl_kind, 0.899763013658897_dbl_kind, 0.915875735496193_dbl_kind, 0.971361469799281_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1166) = (/ 0.891261925937408_dbl_kind, 0.894819756728632_dbl_kind, 0.899766465884353_dbl_kind, 0.915886381863387_dbl_kind, 0.971370107546498_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1167) = (/ 0.891262105465387_dbl_kind, 0.894820523398651_dbl_kind, 0.899769916993053_dbl_kind, 0.915897024853385_dbl_kind, 0.971378729204112_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1168) = (/ 0.891262284693333_dbl_kind, 0.894821289733901_dbl_kind, 0.899773366986897_dbl_kind, 0.915907664469245_dbl_kind, 0.971387334808797_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1169) = (/ 0.891262463622207_dbl_kind, 0.89482205573431_dbl_kind, 0.899776815867786_dbl_kind, 0.915918300714018_dbl_kind, 0.971395924397121_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1170) = (/ 0.891262642252972_dbl_kind, 0.894822821399811_dbl_kind, 0.899780263637621_dbl_kind, 0.915928933590741_dbl_kind, 0.971404498005552_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1171) = (/ 0.89126282058658_dbl_kind, 0.894823586730332_dbl_kind, 0.8997837102983_dbl_kind, 0.915939563102448_dbl_kind, 0.971413055670451_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1172) = (/ 0.891262998623983_dbl_kind, 0.894824351725803_dbl_kind, 0.899787155851721_dbl_kind, 0.915950189252156_dbl_kind, 0.971421597428081_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1173) = (/ 0.89126317636613_dbl_kind, 0.894825116386156_dbl_kind, 0.899790600299779_dbl_kind, 0.915960812042878_dbl_kind, 0.971430123314597_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1174) = (/ 0.891263353813963_dbl_kind, 0.894825880711319_dbl_kind, 0.899794043644371_dbl_kind, 0.915971431477615_dbl_kind, 0.971438633366056_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1175) = (/ 0.891263530968423_dbl_kind, 0.894826644701222_dbl_kind, 0.899797485887391_dbl_kind, 0.91598204755936_dbl_kind, 0.971447127618413_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1176) = (/ 0.891263707830446_dbl_kind, 0.894827408355794_dbl_kind, 0.89980092703073_dbl_kind, 0.915992660291096_dbl_kind, 0.97145560610752_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1177) = (/ 0.891263884400965_dbl_kind, 0.894828171674967_dbl_kind, 0.899804367076282_dbl_kind, 0.916003269675795_dbl_kind, 0.971464068869128_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1178) = (/ 0.891264060680909_dbl_kind, 0.894828934658669_dbl_kind, 0.899807806025936_dbl_kind, 0.916013875716424_dbl_kind, 0.971472515938888_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1179) = (/ 0.891264236671202_dbl_kind, 0.89482969730683_dbl_kind, 0.899811243881583_dbl_kind, 0.916024478415935_dbl_kind, 0.971480947352351_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1180) = (/ 0.891264412372767_dbl_kind, 0.89483045961938_dbl_kind, 0.899814680645109_dbl_kind, 0.916035077777276_dbl_kind, 0.971489363144967_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1181) = (/ 0.89126458778652_dbl_kind, 0.894831221596248_dbl_kind, 0.899818116318401_dbl_kind, 0.916045673803382_dbl_kind, 0.971497763352089_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1182) = (/ 0.891264762913376_dbl_kind, 0.894831983237365_dbl_kind, 0.899821550903346_dbl_kind, 0.916056266497181_dbl_kind, 0.971506148008969_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1183) = (/ 0.891264937754245_dbl_kind, 0.894832744542661_dbl_kind, 0.899824984401827_dbl_kind, 0.91606685586159_dbl_kind, 0.971514517150756_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1184) = (/ 0.891265112310034_dbl_kind, 0.894833505512065_dbl_kind, 0.899828416815726_dbl_kind, 0.916077441899519_dbl_kind, 0.971522870812507_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1185) = (/ 0.891265286581646_dbl_kind, 0.894834266145507_dbl_kind, 0.899831848146927_dbl_kind, 0.916088024613867_dbl_kind, 0.971531209029177_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1186) = (/ 0.891265460569981_dbl_kind, 0.894835026442918_dbl_kind, 0.899835278397308_dbl_kind, 0.916098604007524_dbl_kind, 0.971539531835623_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1187) = (/ 0.891265634275935_dbl_kind, 0.894835786404228_dbl_kind, 0.899838707568749_dbl_kind, 0.916109180083373_dbl_kind, 0.971547839266607_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1188) = (/ 0.8912658077004_dbl_kind, 0.894836546029368_dbl_kind, 0.899842135663128_dbl_kind, 0.916119752844286_dbl_kind, 0.971556131356791_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1189) = (/ 0.891265980844265_dbl_kind, 0.894837305318268_dbl_kind, 0.89984556268232_dbl_kind, 0.916130322293125_dbl_kind, 0.971564408140741_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1190) = (/ 0.891266153708415_dbl_kind, 0.894838064270858_dbl_kind, 0.8998489886282_dbl_kind, 0.916140888432745_dbl_kind, 0.971572669652925_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1191) = (/ 0.891266326293732_dbl_kind, 0.894838822887072_dbl_kind, 0.899852413502641_dbl_kind, 0.916151451265992_dbl_kind, 0.971580915927717_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1192) = (/ 0.891266498601095_dbl_kind, 0.894839581166837_dbl_kind, 0.899855837307516_dbl_kind, 0.916162010795701_dbl_kind, 0.971589146999394_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1193) = (/ 0.891266670631377_dbl_kind, 0.894840339110088_dbl_kind, 0.899859260044695_dbl_kind, 0.916172567024701_dbl_kind, 0.971597362902137_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1194) = (/ 0.891266842385451_dbl_kind, 0.894841096716755_dbl_kind, 0.899862681716046_dbl_kind, 0.916183119955809_dbl_kind, 0.971605563670033_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1195) = (/ 0.891267013864184_dbl_kind, 0.894841853986769_dbl_kind, 0.899866102323438_dbl_kind, 0.916193669591835_dbl_kind, 0.971613749337069_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1196) = (/ 0.89126718506844_dbl_kind, 0.894842610920064_dbl_kind, 0.899869521868735_dbl_kind, 0.91620421593558_dbl_kind, 0.971621919937147_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1197) = (/ 0.891267355999081_dbl_kind, 0.894843367516571_dbl_kind, 0.899872940353804_dbl_kind, 0.916214758989835_dbl_kind, 0.971630075504065_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1198) = (/ 0.891267526656964_dbl_kind, 0.894844123776224_dbl_kind, 0.899876357780506_dbl_kind, 0.916225298757383_dbl_kind, 0.97163821607153_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1199) = (/ 0.891267697042942_dbl_kind, 0.894844879698954_dbl_kind, 0.899879774150703_dbl_kind, 0.916235835240998_dbl_kind, 0.971646341673159_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1200) = (/ 0.891267867157866_dbl_kind, 0.894845635284695_dbl_kind, 0.899883189466256_dbl_kind, 0.916246368443447_dbl_kind, 0.971654452342471_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1201) = (/ 0.891268037002585_dbl_kind, 0.894846390533381_dbl_kind, 0.89988660372902_dbl_kind, 0.916256898367483_dbl_kind, 0.971662548112893_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1202) = (/ 0.891268206577941_dbl_kind, 0.894847145444946_dbl_kind, 0.899890016940855_dbl_kind, 0.916267425015857_dbl_kind, 0.971670629017761_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1203) = (/ 0.891268375884776_dbl_kind, 0.894847900019323_dbl_kind, 0.899893429103615_dbl_kind, 0.916277948391305_dbl_kind, 0.971678695090314_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1204) = (/ 0.891268544923926_dbl_kind, 0.894848654256448_dbl_kind, 0.899896840219153_dbl_kind, 0.916288468496559_dbl_kind, 0.971686746363707_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1205) = (/ 0.891268713696225_dbl_kind, 0.894849408156254_dbl_kind, 0.89990025028932_dbl_kind, 0.91629898533434_dbl_kind, 0.971694782870993_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1206) = (/ 0.891268882202504_dbl_kind, 0.894850161718678_dbl_kind, 0.899903659315968_dbl_kind, 0.91630949890736_dbl_kind, 0.97170280464514_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1207) = (/ 0.891269050443591_dbl_kind, 0.894850914943655_dbl_kind, 0.899907067300943_dbl_kind, 0.916320009218325_dbl_kind, 0.971710811719024_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1208) = (/ 0.891269218420308_dbl_kind, 0.89485166783112_dbl_kind, 0.899910474246094_dbl_kind, 0.916330516269927_dbl_kind, 0.971718804125431_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1209) = (/ 0.891269386133477_dbl_kind, 0.89485242038101_dbl_kind, 0.899913880153264_dbl_kind, 0.916341020064857_dbl_kind, 0.971726781897053_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1210) = (/ 0.891269553583915_dbl_kind, 0.894853172593262_dbl_kind, 0.899917285024297_dbl_kind, 0.916351520605789_dbl_kind, 0.971734745066491_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1211) = (/ 0.891269720772437_dbl_kind, 0.894853924467814_dbl_kind, 0.899920688861035_dbl_kind, 0.916362017895395_dbl_kind, 0.971742693666261_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1212) = (/ 0.891269887699853_dbl_kind, 0.894854676004602_dbl_kind, 0.899924091665316_dbl_kind, 0.916372511936335_dbl_kind, 0.971750627728785_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1213) = (/ 0.891270054366971_dbl_kind, 0.894855427203565_dbl_kind, 0.899927493438979_dbl_kind, 0.916383002731261_dbl_kind, 0.971758547286397_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1214) = (/ 0.891270220774596_dbl_kind, 0.894856178064641_dbl_kind, 0.899930894183859_dbl_kind, 0.916393490282817_dbl_kind, 0.971766452371344_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1215) = (/ 0.891270386923528_dbl_kind, 0.894856928587769_dbl_kind, 0.899934293901791_dbl_kind, 0.916403974593638_dbl_kind, 0.971774343015779_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1216) = (/ 0.891270552814566_dbl_kind, 0.894857678772889_dbl_kind, 0.899937692594608_dbl_kind, 0.916414455666352_dbl_kind, 0.971782219251772_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1217) = (/ 0.891270718448506_dbl_kind, 0.894858428619941_dbl_kind, 0.899941090264138_dbl_kind, 0.916424933503574_dbl_kind, 0.971790081111301_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1218) = (/ 0.891270883826138_dbl_kind, 0.894859178128865_dbl_kind, 0.899944486912212_dbl_kind, 0.916435408107917_dbl_kind, 0.971797928626257_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1219) = (/ 0.891271048948252_dbl_kind, 0.894859927299602_dbl_kind, 0.899947882540655_dbl_kind, 0.916445879481981_dbl_kind, 0.971805761828445_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1220) = (/ 0.891271213815633_dbl_kind, 0.894860676132092_dbl_kind, 0.899951277151292_dbl_kind, 0.916456347628357_dbl_kind, 0.971813580749579_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1221) = (/ 0.891271378429064_dbl_kind, 0.894861424626279_dbl_kind, 0.899954670745945_dbl_kind, 0.91646681254963_dbl_kind, 0.97182138542129_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1222) = (/ 0.891271542789323_dbl_kind, 0.894862172782105_dbl_kind, 0.899958063326435_dbl_kind, 0.916477274248376_dbl_kind, 0.97182917587512_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1223) = (/ 0.891271706897187_dbl_kind, 0.894862920599512_dbl_kind, 0.899961454894582_dbl_kind, 0.916487732727162_dbl_kind, 0.971836952142523_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1224) = (/ 0.89127187075343_dbl_kind, 0.894863668078444_dbl_kind, 0.899964845452201_dbl_kind, 0.916498187988548_dbl_kind, 0.97184471425487_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1225) = (/ 0.891272034358821_dbl_kind, 0.894864415218845_dbl_kind, 0.899968235001107_dbl_kind, 0.916508640035083_dbl_kind, 0.971852462243445_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1226) = (/ 0.891272197714127_dbl_kind, 0.89486516202066_dbl_kind, 0.899971623543113_dbl_kind, 0.916519088869309_dbl_kind, 0.971860196139444_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1227) = (/ 0.891272360820112_dbl_kind, 0.894865908483833_dbl_kind, 0.899975011080028_dbl_kind, 0.91652953449376_dbl_kind, 0.971867915973982_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1228) = (/ 0.891272523677538_dbl_kind, 0.894866654608311_dbl_kind, 0.899978397613662_dbl_kind, 0.91653997691096_dbl_kind, 0.971875621778085_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1229) = (/ 0.891272686287162_dbl_kind, 0.89486740039404_dbl_kind, 0.899981783145821_dbl_kind, 0.916550416123428_dbl_kind, 0.971883313582695_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1230) = (/ 0.891272848649739_dbl_kind, 0.894868145840967_dbl_kind, 0.899985167678309_dbl_kind, 0.916560852133671_dbl_kind, 0.971890991418672_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1231) = (/ 0.891273010766022_dbl_kind, 0.894868890949039_dbl_kind, 0.899988551212927_dbl_kind, 0.91657128494419_dbl_kind, 0.971898655316788_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1232) = (/ 0.891273172636758_dbl_kind, 0.894869635718205_dbl_kind, 0.899991933751475_dbl_kind, 0.916581714557476_dbl_kind, 0.971906305307734_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1233) = (/ 0.891273334262695_dbl_kind, 0.894870380148412_dbl_kind, 0.899995315295752_dbl_kind, 0.916592140976013_dbl_kind, 0.971913941422116_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1234) = (/ 0.891273495644576_dbl_kind, 0.894871124239611_dbl_kind, 0.899998695847551_dbl_kind, 0.916602564202275_dbl_kind, 0.971921563690456_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1235) = (/ 0.891273656783139_dbl_kind, 0.894871867991752_dbl_kind, 0.900002075408667_dbl_kind, 0.916612984238731_dbl_kind, 0.971929172143193_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1236) = (/ 0.891273817679123_dbl_kind, 0.894872611404784_dbl_kind, 0.90000545398089_dbl_kind, 0.916623401087838_dbl_kind, 0.971936766810687_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1237) = (/ 0.891273978333262_dbl_kind, 0.894873354478659_dbl_kind, 0.900008831566008_dbl_kind, 0.916633814752048_dbl_kind, 0.971944347723209_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1238) = (/ 0.891274138746287_dbl_kind, 0.89487409721333_dbl_kind, 0.900012208165807_dbl_kind, 0.916644225233801_dbl_kind, 0.971951914910951_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1239) = (/ 0.891274298918927_dbl_kind, 0.894874839608749_dbl_kind, 0.900015583782072_dbl_kind, 0.916654632535531_dbl_kind, 0.971959468404024_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1240) = (/ 0.891274458851907_dbl_kind, 0.894875581664868_dbl_kind, 0.900018958416585_dbl_kind, 0.916665036659665_dbl_kind, 0.971967008232455_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1241) = (/ 0.89127461854595_dbl_kind, 0.894876323381642_dbl_kind, 0.900022332071123_dbl_kind, 0.916675437608619_dbl_kind, 0.971974534426188_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1242) = (/ 0.891274778001776_dbl_kind, 0.894877064759025_dbl_kind, 0.900025704747464_dbl_kind, 0.916685835384802_dbl_kind, 0.971982047015093_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1243) = (/ 0.891274937220102_dbl_kind, 0.894877805796973_dbl_kind, 0.900029076447382_dbl_kind, 0.916696229990616_dbl_kind, 0.971989546028953_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1244) = (/ 0.891275096201642_dbl_kind, 0.894878546495441_dbl_kind, 0.90003244717265_dbl_kind, 0.916706621428453_dbl_kind, 0.971997031497468_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1245) = (/ 0.891275254947107_dbl_kind, 0.894879286854387_dbl_kind, 0.900035816925036_dbl_kind, 0.916717009700696_dbl_kind, 0.972004503450263_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1246) = (/ 0.891275413457206_dbl_kind, 0.894880026873767_dbl_kind, 0.900039185706307_dbl_kind, 0.916727394809723_dbl_kind, 0.972011961916881_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1247) = (/ 0.891275571732646_dbl_kind, 0.894880766553539_dbl_kind, 0.900042553518228_dbl_kind, 0.916737776757902_dbl_kind, 0.972019406926785_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1248) = (/ 0.891275729774127_dbl_kind, 0.894881505893662_dbl_kind, 0.900045920362561_dbl_kind, 0.916748155547592_dbl_kind, 0.972026838509356_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1249) = (/ 0.891275887582351_dbl_kind, 0.894882244894096_dbl_kind, 0.900049286241065_dbl_kind, 0.916758531181145_dbl_kind, 0.972034256693901_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1250) = (/ 0.891276045158016_dbl_kind, 0.894882983554801_dbl_kind, 0.900052651155497_dbl_kind, 0.916768903660904_dbl_kind, 0.972041661509639_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1251) = (/ 0.891276202501815_dbl_kind, 0.894883721875737_dbl_kind, 0.900056015107612_dbl_kind, 0.916779272989204_dbl_kind, 0.972049052985724_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1252) = (/ 0.891276359614442_dbl_kind, 0.894884459856867_dbl_kind, 0.900059378099161_dbl_kind, 0.916789639168373_dbl_kind, 0.972056431151216_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1253) = (/ 0.891276516496584_dbl_kind, 0.894885197498153_dbl_kind, 0.900062740131892_dbl_kind, 0.916800002200732_dbl_kind, 0.972063796035105_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1254) = (/ 0.891276673148928_dbl_kind, 0.894885934799557_dbl_kind, 0.900066101207553_dbl_kind, 0.916810362088587_dbl_kind, 0.972071147666302_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1255) = (/ 0.891276829572159_dbl_kind, 0.894886671761044_dbl_kind, 0.900069461327887_dbl_kind, 0.916820718834245_dbl_kind, 0.97207848607364_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1256) = (/ 0.891276985766957_dbl_kind, 0.894887408382578_dbl_kind, 0.900072820494635_dbl_kind, 0.916831072439998_dbl_kind, 0.972085811285873_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1257) = (/ 0.891277141734001_dbl_kind, 0.894888144664126_dbl_kind, 0.900076178709535_dbl_kind, 0.916841422908134_dbl_kind, 0.972093123331681_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1258) = (/ 0.891277297473965_dbl_kind, 0.894888880605652_dbl_kind, 0.900079535974323_dbl_kind, 0.916851770240931_dbl_kind, 0.97210042223966_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1259) = (/ 0.891277452987523_dbl_kind, 0.894889616207124_dbl_kind, 0.900082892290733_dbl_kind, 0.916862114440658_dbl_kind, 0.972107708038336_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1260) = (/ 0.891277608275347_dbl_kind, 0.894890351468509_dbl_kind, 0.900086247660492_dbl_kind, 0.916872455509579_dbl_kind, 0.972114980756156_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1261) = (/ 0.891277763338102_dbl_kind, 0.894891086389777_dbl_kind, 0.90008960208533_dbl_kind, 0.916882793449947_dbl_kind, 0.97212224042149_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1262) = (/ 0.891277918176455_dbl_kind, 0.894891820970896_dbl_kind, 0.90009295556697_dbl_kind, 0.916893128264008_dbl_kind, 0.972129487062633_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1263) = (/ 0.891278072791067_dbl_kind, 0.894892555211837_dbl_kind, 0.900096308107135_dbl_kind, 0.916903459954_dbl_kind, 0.9721367207078_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1264) = (/ 0.891278227182599_dbl_kind, 0.89489328911257_dbl_kind, 0.900099659707541_dbl_kind, 0.916913788522154_dbl_kind, 0.97214394138514_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1265) = (/ 0.891278381351708_dbl_kind, 0.894894022673067_dbl_kind, 0.900103010369907_dbl_kind, 0.916924113970689_dbl_kind, 0.972151149122717_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1266) = (/ 0.891278535299048_dbl_kind, 0.894894755893301_dbl_kind, 0.900106360095944_dbl_kind, 0.916934436301821_dbl_kind, 0.972158343948523_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1267) = (/ 0.891278689025271_dbl_kind, 0.894895488773245_dbl_kind, 0.900109708887362_dbl_kind, 0.916944755517755_dbl_kind, 0.972165525890475_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1268) = (/ 0.891278842531027_dbl_kind, 0.894896221312872_dbl_kind, 0.90011305674587_dbl_kind, 0.916955071620686_dbl_kind, 0.972172694976415_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1269) = (/ 0.891278995816963_dbl_kind, 0.894896953512158_dbl_kind, 0.90011640367317_dbl_kind, 0.916965384612807_dbl_kind, 0.972179851234115_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1270) = (/ 0.891279148883724_dbl_kind, 0.894897685371079_dbl_kind, 0.900119749670963_dbl_kind, 0.916975694496298_dbl_kind, 0.972186994691266_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1271) = (/ 0.89127930173195_dbl_kind, 0.894898416889611_dbl_kind, 0.900123094740948_dbl_kind, 0.916986001273332_dbl_kind, 0.972194125375488_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1272) = (/ 0.891279454362281_dbl_kind, 0.894899148067732_dbl_kind, 0.90012643888482_dbl_kind, 0.916996304946074_dbl_kind, 0.972201243314328_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1273) = (/ 0.891279606775354_dbl_kind, 0.894899878905419_dbl_kind, 0.900129782104271_dbl_kind, 0.917006605516682_dbl_kind, 0.972208348535258_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1274) = (/ 0.891279758971804_dbl_kind, 0.894900609402653_dbl_kind, 0.90013312440099_dbl_kind, 0.917016902987304_dbl_kind, 0.972215441065681_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1275) = (/ 0.891279910952261_dbl_kind, 0.894901339559412_dbl_kind, 0.900136465776663_dbl_kind, 0.917027197360083_dbl_kind, 0.972222520932919_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1276) = (/ 0.891280062717356_dbl_kind, 0.894902069375677_dbl_kind, 0.900139806232971_dbl_kind, 0.91703748863715_dbl_kind, 0.97222958816423_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1277) = (/ 0.891280214267715_dbl_kind, 0.894902798851431_dbl_kind, 0.900143145771596_dbl_kind, 0.917047776820632_dbl_kind, 0.972236642786791_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1278) = (/ 0.891280365603962_dbl_kind, 0.894903527986655_dbl_kind, 0.900146484394213_dbl_kind, 0.917058061912645_dbl_kind, 0.972243684827714_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1279) = (/ 0.89128051672672_dbl_kind, 0.894904256781333_dbl_kind, 0.900149822102495_dbl_kind, 0.917068343915297_dbl_kind, 0.972250714314036_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1280) = (/ 0.891280667636607_dbl_kind, 0.894904985235449_dbl_kind, 0.900153158898113_dbl_kind, 0.91707862283069_dbl_kind, 0.972257731272723_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1281) = (/ 0.89128081833424_dbl_kind, 0.894905713348988_dbl_kind, 0.900156494782733_dbl_kind, 0.917088898660918_dbl_kind, 0.972264735730664_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1282) = (/ 0.891280968820235_dbl_kind, 0.894906441121936_dbl_kind, 0.900159829758019_dbl_kind, 0.917099171408064_dbl_kind, 0.972271727714683_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1283) = (/ 0.891281119095203_dbl_kind, 0.89490716855428_dbl_kind, 0.900163163825632_dbl_kind, 0.917109441074206_dbl_kind, 0.972278707251534_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1284) = (/ 0.891281269159753_dbl_kind, 0.894907895646006_dbl_kind, 0.900166496987227_dbl_kind, 0.917119707661413_dbl_kind, 0.972285674367893_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1285) = (/ 0.891281419014494_dbl_kind, 0.894908622397105_dbl_kind, 0.900169829244459_dbl_kind, 0.917129971171745_dbl_kind, 0.972292629090372_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1286) = (/ 0.89128156866003_dbl_kind, 0.894909348807565_dbl_kind, 0.900173160598978_dbl_kind, 0.917140231607255_dbl_kind, 0.972299571445506_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1287) = (/ 0.891281718096963_dbl_kind, 0.894910074877376_dbl_kind, 0.900176491052431_dbl_kind, 0.917150488969989_dbl_kind, 0.97230650145977_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1288) = (/ 0.891281867325895_dbl_kind, 0.894910800606529_dbl_kind, 0.900179820606462_dbl_kind, 0.917160743261983_dbl_kind, 0.972313419159556_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1289) = (/ 0.891282016347422_dbl_kind, 0.894911525995017_dbl_kind, 0.900183149262711_dbl_kind, 0.917170994485266_dbl_kind, 0.972320324571197_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1290) = (/ 0.891282165162139_dbl_kind, 0.894912251042832_dbl_kind, 0.900186477022815_dbl_kind, 0.917181242641859_dbl_kind, 0.972327217720949_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1291) = (/ 0.891282313770641_dbl_kind, 0.894912975749969_dbl_kind, 0.900189803888406_dbl_kind, 0.917191487733774_dbl_kind, 0.972334098635003_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1292) = (/ 0.891282462173517_dbl_kind, 0.894913700116421_dbl_kind, 0.900193129861116_dbl_kind, 0.917201729763017_dbl_kind, 0.97234096733948_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1293) = (/ 0.891282610371356_dbl_kind, 0.894914424142184_dbl_kind, 0.90019645494257_dbl_kind, 0.917211968731584_dbl_kind, 0.97234782386043_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1294) = (/ 0.891282758364745_dbl_kind, 0.894915147827255_dbl_kind, 0.900199779134391_dbl_kind, 0.917222204641463_dbl_kind, 0.972354668223839_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1295) = (/ 0.891282906154267_dbl_kind, 0.894915871171631_dbl_kind, 0.900203102438199_dbl_kind, 0.917232437494637_dbl_kind, 0.972361500455619_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1296) = (/ 0.891283053740503_dbl_kind, 0.89491659417531_dbl_kind, 0.900206424855611_dbl_kind, 0.917242667293078_dbl_kind, 0.972368320581616_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1297) = (/ 0.891283201124034_dbl_kind, 0.894917316838292_dbl_kind, 0.900209746388237_dbl_kind, 0.917252894038751_dbl_kind, 0.972375128627609_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1298) = (/ 0.891283348305434_dbl_kind, 0.894918039160576_dbl_kind, 0.900213067037686_dbl_kind, 0.917263117733612_dbl_kind, 0.972381924619308_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1299) = (/ 0.89128349528528_dbl_kind, 0.894918761142164_dbl_kind, 0.900216386805564_dbl_kind, 0.91727333837961_dbl_kind, 0.972388708582355_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1300) = (/ 0.891283642064145_dbl_kind, 0.894919482783057_dbl_kind, 0.900219705693473_dbl_kind, 0.917283555978687_dbl_kind, 0.972395480542324_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1301) = (/ 0.891283788642597_dbl_kind, 0.894920204083258_dbl_kind, 0.90022302370301_dbl_kind, 0.917293770532775_dbl_kind, 0.972402240524728_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1302) = (/ 0.891283935021204_dbl_kind, 0.89492092504277_dbl_kind, 0.900226340835769_dbl_kind, 0.917303982043798_dbl_kind, 0.972408988555003_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1303) = (/ 0.891284081200533_dbl_kind, 0.894921645661599_dbl_kind, 0.900229657093341_dbl_kind, 0.917314190513675_dbl_kind, 0.972415724658525_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1304) = (/ 0.891284227181147_dbl_kind, 0.89492236593975_dbl_kind, 0.900232972477313_dbl_kind, 0.917324395944313_dbl_kind, 0.9724224488606_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1305) = (/ 0.891284372963607_dbl_kind, 0.894923085877228_dbl_kind, 0.900236286989267_dbl_kind, 0.917334598337615_dbl_kind, 0.972429161186473_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1306) = (/ 0.891284518548472_dbl_kind, 0.894923805474042_dbl_kind, 0.900239600630784_dbl_kind, 0.917344797695471_dbl_kind, 0.972435861661314_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1307) = (/ 0.891284663936299_dbl_kind, 0.8949245247302_dbl_kind, 0.900242913403437_dbl_kind, 0.917354994019768_dbl_kind, 0.972442550310238_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1308) = (/ 0.891284809127643_dbl_kind, 0.894925243645711_dbl_kind, 0.900246225308801_dbl_kind, 0.917365187312383_dbl_kind, 0.972449227158283_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1309) = (/ 0.891284954123055_dbl_kind, 0.894925962220584_dbl_kind, 0.900249536348442_dbl_kind, 0.917375377575184_dbl_kind, 0.972455892230431_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1310) = (/ 0.891285098923087_dbl_kind, 0.894926680454832_dbl_kind, 0.900252846523924_dbl_kind, 0.917385564810032_dbl_kind, 0.972462545551591_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1311) = (/ 0.891285243528286_dbl_kind, 0.894927398348465_dbl_kind, 0.900256155836807_dbl_kind, 0.917395749018781_dbl_kind, 0.972469187146614_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1312) = (/ 0.891285387939199_dbl_kind, 0.894928115901496_dbl_kind, 0.90025946428865_dbl_kind, 0.917405930203274_dbl_kind, 0.97247581704028_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1313) = (/ 0.891285532156368_dbl_kind, 0.89492883311394_dbl_kind, 0.900262771881002_dbl_kind, 0.917416108365351_dbl_kind, 0.972482435257307_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1314) = (/ 0.891285676180336_dbl_kind, 0.894929549985811_dbl_kind, 0.900266078615415_dbl_kind, 0.917426283506838_dbl_kind, 0.972489041822349_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1315) = (/ 0.891285820011643_dbl_kind, 0.894930266517125_dbl_kind, 0.900269384493431_dbl_kind, 0.917436455629557_dbl_kind, 0.972495636759998_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1316) = (/ 0.891285963650825_dbl_kind, 0.894930982707898_dbl_kind, 0.900272689516592_dbl_kind, 0.917446624735322_dbl_kind, 0.972502220094773_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1317) = (/ 0.891286107098418_dbl_kind, 0.894931698558147_dbl_kind, 0.900275993686434_dbl_kind, 0.917456790825936_dbl_kind, 0.972508791851139_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1318) = (/ 0.891286250354955_dbl_kind, 0.894932414067892_dbl_kind, 0.900279297004491_dbl_kind, 0.917466953903198_dbl_kind, 0.972515352053492_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1319) = (/ 0.891286393420968_dbl_kind, 0.89493312923715_dbl_kind, 0.900282599472291_dbl_kind, 0.917477113968896_dbl_kind, 0.972521900726165_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1320) = (/ 0.891286536296985_dbl_kind, 0.894933844065943_dbl_kind, 0.900285901091359_dbl_kind, 0.917487271024812_dbl_kind, 0.97252843789343_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1321) = (/ 0.891286678983533_dbl_kind, 0.894934558554292_dbl_kind, 0.900289201863216_dbl_kind, 0.917497425072717_dbl_kind, 0.972534963579496_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1322) = (/ 0.891286821481137_dbl_kind, 0.894935272702219_dbl_kind, 0.900292501789378_dbl_kind, 0.917507576114377_dbl_kind, 0.972541477808504_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1323) = (/ 0.89128696379032_dbl_kind, 0.894935986509746_dbl_kind, 0.900295800871358_dbl_kind, 0.91751772415155_dbl_kind, 0.972547980604536_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1324) = (/ 0.891287105911602_dbl_kind, 0.894936699976898_dbl_kind, 0.900299099110665_dbl_kind, 0.917527869185984_dbl_kind, 0.972554471991613_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1325) = (/ 0.891287247845502_dbl_kind, 0.8949374131037_dbl_kind, 0.900302396508803_dbl_kind, 0.917538011219421_dbl_kind, 0.972560951993693_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1326) = (/ 0.891287389592538_dbl_kind, 0.894938125890177_dbl_kind, 0.900305693067271_dbl_kind, 0.917548150253591_dbl_kind, 0.972567420634668_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1327) = (/ 0.891287531153222_dbl_kind, 0.894938838336357_dbl_kind, 0.900308988787566_dbl_kind, 0.917558286290222_dbl_kind, 0.972573877938372_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1328) = (/ 0.89128767252807_dbl_kind, 0.894939550442266_dbl_kind, 0.90031228367118_dbl_kind, 0.917568419331031_dbl_kind, 0.972580323928576_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1329) = (/ 0.891287813717589_dbl_kind, 0.894940262207933_dbl_kind, 0.900315577719602_dbl_kind, 0.917578549377726_dbl_kind, 0.972586758628991_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1330) = (/ 0.89128795472229_dbl_kind, 0.894940973633389_dbl_kind, 0.900318870934313_dbl_kind, 0.917588676432008_dbl_kind, 0.972593182063261_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1331) = (/ 0.891288095542679_dbl_kind, 0.894941684718663_dbl_kind, 0.900322163316794_dbl_kind, 0.91759880049557_dbl_kind, 0.972599594254977_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1332) = (/ 0.891288236179259_dbl_kind, 0.894942395463788_dbl_kind, 0.900325454868519_dbl_kind, 0.917608921570097_dbl_kind, 0.972605995227664_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1333) = (/ 0.891288376632535_dbl_kind, 0.894943105868794_dbl_kind, 0.900328745590959_dbl_kind, 0.917619039657266_dbl_kind, 0.972612385004787_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1334) = (/ 0.891288516903007_dbl_kind, 0.894943815933716_dbl_kind, 0.900332035485581_dbl_kind, 0.917629154758746_dbl_kind, 0.972618763609752_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1335) = (/ 0.891288656991172_dbl_kind, 0.894944525658588_dbl_kind, 0.900335324553846_dbl_kind, 0.917639266876197_dbl_kind, 0.9726251310659_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1336) = (/ 0.891288796897528_dbl_kind, 0.894945235043445_dbl_kind, 0.900338612797213_dbl_kind, 0.917649376011273_dbl_kind, 0.972631487396517_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1337) = (/ 0.89128893662257_dbl_kind, 0.894945944088322_dbl_kind, 0.900341900217135_dbl_kind, 0.917659482165619_dbl_kind, 0.972637832624828_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1338) = (/ 0.89128907616679_dbl_kind, 0.894946652793257_dbl_kind, 0.900345186815061_dbl_kind, 0.917669585340872_dbl_kind, 0.972644166773995_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1339) = (/ 0.89128921553068_dbl_kind, 0.894947361158289_dbl_kind, 0.900348472592436_dbl_kind, 0.91767968553866_dbl_kind, 0.972650489867124_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1340) = (/ 0.891289354714727_dbl_kind, 0.894948069183455_dbl_kind, 0.900351757550699_dbl_kind, 0.917689782760603_dbl_kind, 0.972656801927258_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1341) = (/ 0.891289493719421_dbl_kind, 0.894948776868795_dbl_kind, 0.900355041691288_dbl_kind, 0.917699877008316_dbl_kind, 0.972663102977385_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1342) = (/ 0.891289632545243_dbl_kind, 0.894949484214351_dbl_kind, 0.900358325015633_dbl_kind, 0.917709968283402_dbl_kind, 0.972669393040429_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1343) = (/ 0.89128977119268_dbl_kind, 0.894950191220163_dbl_kind, 0.900361607525162_dbl_kind, 0.917720056587458_dbl_kind, 0.972675672139259_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1344) = (/ 0.891289909662212_dbl_kind, 0.894950897886275_dbl_kind, 0.900364889221296_dbl_kind, 0.917730141922073_dbl_kind, 0.972681940296684_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1345) = (/ 0.891290047954317_dbl_kind, 0.89495160421273_dbl_kind, 0.900368170105454_dbl_kind, 0.917740224288829_dbl_kind, 0.972688197535457_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1346) = (/ 0.891290186069474_dbl_kind, 0.894952310199572_dbl_kind, 0.900371450179051_dbl_kind, 0.917750303689296_dbl_kind, 0.972694443878262_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1347) = (/ 0.891290324008158_dbl_kind, 0.894953015846847_dbl_kind, 0.900374729443494_dbl_kind, 0.91776038012504_dbl_kind, 0.972700679347739_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1348) = (/ 0.891290461770843_dbl_kind, 0.8949537211546_dbl_kind, 0.900378007900188_dbl_kind, 0.917770453597618_dbl_kind, 0.972706903966462_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1349) = (/ 0.891290599358001_dbl_kind, 0.89495442612288_dbl_kind, 0.900381285550533_dbl_kind, 0.917780524108578_dbl_kind, 0.972713117756949_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1350) = (/ 0.891290736770103_dbl_kind, 0.894955130751734_dbl_kind, 0.900384562395924_dbl_kind, 0.91779059165946_dbl_kind, 0.972719320741659_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1351) = (/ 0.891290874007615_dbl_kind, 0.894955835041211_dbl_kind, 0.900387838437753_dbl_kind, 0.917800656251797_dbl_kind, 0.972725512942994_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1352) = (/ 0.891291011071005_dbl_kind, 0.894956538991362_dbl_kind, 0.900391113677405_dbl_kind, 0.917810717887113_dbl_kind, 0.972731694383301_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1353) = (/ 0.891291147960736_dbl_kind, 0.894957242602237_dbl_kind, 0.900394388116262_dbl_kind, 0.917820776566924_dbl_kind, 0.972737865084869_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1354) = (/ 0.891291284677272_dbl_kind, 0.894957945873887_dbl_kind, 0.900397661755702_dbl_kind, 0.91783083229274_dbl_kind, 0.972744025069926_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1355) = (/ 0.891291421221074_dbl_kind, 0.894958648806366_dbl_kind, 0.900400934597095_dbl_kind, 0.917840885066059_dbl_kind, 0.972750174360648_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1356) = (/ 0.891291557592601_dbl_kind, 0.894959351399728_dbl_kind, 0.90040420664181_dbl_kind, 0.917850934888374_dbl_kind, 0.972756312979151_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1357) = (/ 0.891291693792309_dbl_kind, 0.894960053654026_dbl_kind, 0.90040747789121_dbl_kind, 0.91786098176117_dbl_kind, 0.972762440947498_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1358) = (/ 0.891291829820655_dbl_kind, 0.894960755569317_dbl_kind, 0.900410748346652_dbl_kind, 0.917871025685922_dbl_kind, 0.972768558287695_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1359) = (/ 0.891291965678091_dbl_kind, 0.894961457145657_dbl_kind, 0.900414018009491_dbl_kind, 0.917881066664099_dbl_kind, 0.972774665021688_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1360) = (/ 0.891292101365072_dbl_kind, 0.894962158383103_dbl_kind, 0.900417286881075_dbl_kind, 0.917891104697159_dbl_kind, 0.97278076117137_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1361) = (/ 0.891292236882044_dbl_kind, 0.894962859281713_dbl_kind, 0.900420554962749_dbl_kind, 0.917901139786556_dbl_kind, 0.97278684675858_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1362) = (/ 0.891292372229458_dbl_kind, 0.894963559841546_dbl_kind, 0.90042382225585_dbl_kind, 0.917911171933733_dbl_kind, 0.972792921805099_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1363) = (/ 0.89129250740776_dbl_kind, 0.894964260062663_dbl_kind, 0.900427088761714_dbl_kind, 0.917921201140125_dbl_kind, 0.972798986332651_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1364) = (/ 0.891292642417395_dbl_kind, 0.894964959945124_dbl_kind, 0.900430354481671_dbl_kind, 0.917931227407161_dbl_kind, 0.972805040362909_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1365) = (/ 0.891292777258806_dbl_kind, 0.894965659488991_dbl_kind, 0.900433619417044_dbl_kind, 0.917941250736259_dbl_kind, 0.972811083917489_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1366) = (/ 0.891292911932433_dbl_kind, 0.894966358694328_dbl_kind, 0.900436883569155_dbl_kind, 0.917951271128832_dbl_kind, 0.97281711701795_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1367) = (/ 0.891293046438718_dbl_kind, 0.894967057561197_dbl_kind, 0.900440146939318_dbl_kind, 0.917961288586283_dbl_kind, 0.9728231396858_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1368) = (/ 0.891293180778098_dbl_kind, 0.894967756089663_dbl_kind, 0.900443409528843_dbl_kind, 0.917971303110007_dbl_kind, 0.972829151942487_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1369) = (/ 0.891293314951008_dbl_kind, 0.894968454279791_dbl_kind, 0.900446671339037_dbl_kind, 0.917981314701391_dbl_kind, 0.972835153809411_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1370) = (/ 0.891293448957883_dbl_kind, 0.894969152131649_dbl_kind, 0.900449932371199_dbl_kind, 0.917991323361815_dbl_kind, 0.972841145307915_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1371) = (/ 0.891293582799157_dbl_kind, 0.894969849645303_dbl_kind, 0.900453192626624_dbl_kind, 0.918001329092649_dbl_kind, 0.972847126459285_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1372) = (/ 0.891293716475259_dbl_kind, 0.894970546820822_dbl_kind, 0.900456452106605_dbl_kind, 0.918011331895257_dbl_kind, 0.972853097284758_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1373) = (/ 0.89129384998662_dbl_kind, 0.894971243658273_dbl_kind, 0.900459710812426_dbl_kind, 0.918021331770995_dbl_kind, 0.972859057805515_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1374) = (/ 0.891293983333667_dbl_kind, 0.894971940157728_dbl_kind, 0.900462968745368_dbl_kind, 0.918031328721206_dbl_kind, 0.972865008042683_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1375) = (/ 0.891294116516825_dbl_kind, 0.894972636319257_dbl_kind, 0.900466225906708_dbl_kind, 0.918041322747233_dbl_kind, 0.972870948017333_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1376) = (/ 0.89129424953652_dbl_kind, 0.894973332142933_dbl_kind, 0.900469482297715_dbl_kind, 0.918051313850405_dbl_kind, 0.97287687775049_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1377) = (/ 0.891294382393174_dbl_kind, 0.894974027628826_dbl_kind, 0.900472737919656_dbl_kind, 0.918061302032043_dbl_kind, 0.97288279726312_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1378) = (/ 0.891294515087208_dbl_kind, 0.894974722777012_dbl_kind, 0.900475992773792_dbl_kind, 0.918071287293463_dbl_kind, 0.972888706576137_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1379) = (/ 0.891294647619041_dbl_kind, 0.894975417587564_dbl_kind, 0.900479246861377_dbl_kind, 0.918081269635971_dbl_kind, 0.972894605710403_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1380) = (/ 0.89129477998909_dbl_kind, 0.894976112060557_dbl_kind, 0.900482500183664_dbl_kind, 0.918091249060866_dbl_kind, 0.972900494686727_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1381) = (/ 0.891294912197773_dbl_kind, 0.894976806196068_dbl_kind, 0.900485752741898_dbl_kind, 0.918101225569437_dbl_kind, 0.972906373525866_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1382) = (/ 0.891295044245503_dbl_kind, 0.894977499994174_dbl_kind, 0.900489004537319_dbl_kind, 0.918111199162966_dbl_kind, 0.972912242248525_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1383) = (/ 0.891295176132692_dbl_kind, 0.894978193454953_dbl_kind, 0.900492255571163_dbl_kind, 0.918121169842728_dbl_kind, 0.972918100875355_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1384) = (/ 0.891295307859753_dbl_kind, 0.894978886578482_dbl_kind, 0.900495505844661_dbl_kind, 0.918131137609988_dbl_kind, 0.972923949426956_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1385) = (/ 0.891295439427094_dbl_kind, 0.894979579364843_dbl_kind, 0.900498755359037_dbl_kind, 0.918141102466003_dbl_kind, 0.972929787923877_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1386) = (/ 0.891295570835124_dbl_kind, 0.894980271814115_dbl_kind, 0.900502004115513_dbl_kind, 0.918151064412025_dbl_kind, 0.972935616386615_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1387) = (/ 0.891295702084249_dbl_kind, 0.89498096392638_dbl_kind, 0.900505252115303_dbl_kind, 0.918161023449292_dbl_kind, 0.972941434835614_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1388) = (/ 0.891295833174873_dbl_kind, 0.89498165570172_dbl_kind, 0.900508499359617_dbl_kind, 0.91817097957904_dbl_kind, 0.972947243291269_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1389) = (/ 0.891295964107399_dbl_kind, 0.894982347140217_dbl_kind, 0.900511745849661_dbl_kind, 0.918180932802493_dbl_kind, 0.972953041773922_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1390) = (/ 0.89129609488223_dbl_kind, 0.894983038241957_dbl_kind, 0.900514991586633_dbl_kind, 0.918190883120868_dbl_kind, 0.972958830303865_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1391) = (/ 0.891296225499765_dbl_kind, 0.894983729007024_dbl_kind, 0.900518236571728_dbl_kind, 0.918200830535375_dbl_kind, 0.972964608901336_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1392) = (/ 0.891296355960403_dbl_kind, 0.894984419435503_dbl_kind, 0.900521480806136_dbl_kind, 0.918210775047213_dbl_kind, 0.972970377586528_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1393) = (/ 0.891296486264539_dbl_kind, 0.894985109527482_dbl_kind, 0.90052472429104_dbl_kind, 0.918220716657577_dbl_kind, 0.972976136379579_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1394) = (/ 0.89129661641257_dbl_kind, 0.894985799283046_dbl_kind, 0.900527967027618_dbl_kind, 0.91823065536765_dbl_kind, 0.972981885300576_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1395) = (/ 0.89129674640489_dbl_kind, 0.894986488702285_dbl_kind, 0.900531209017047_dbl_kind, 0.918240591178608_dbl_kind, 0.97298762436956_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1396) = (/ 0.891296876241889_dbl_kind, 0.894987177785287_dbl_kind, 0.900534450260492_dbl_kind, 0.91825052409162_dbl_kind, 0.972993353606518_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1397) = (/ 0.89129700592396_dbl_kind, 0.894987866532143_dbl_kind, 0.900537690759117_dbl_kind, 0.918260454107847_dbl_kind, 0.972999073031387_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1398) = (/ 0.89129713545149_dbl_kind, 0.894988554942943_dbl_kind, 0.90054093051408_dbl_kind, 0.918270381228439_dbl_kind, 0.973004782664058_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1399) = (/ 0.891297264824868_dbl_kind, 0.894989243017779_dbl_kind, 0.900544169526534_dbl_kind, 0.918280305454542_dbl_kind, 0.973010482524368_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1400) = (/ 0.89129739404448_dbl_kind, 0.894989930756743_dbl_kind, 0.900547407797626_dbl_kind, 0.91829022678729_dbl_kind, 0.973016172632106_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1401) = (/ 0.89129752311071_dbl_kind, 0.894990618159928_dbl_kind, 0.900550645328498_dbl_kind, 0.918300145227812_dbl_kind, 0.973021853007012_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1402) = (/ 0.891297652023941_dbl_kind, 0.894991305227429_dbl_kind, 0.900553882120285_dbl_kind, 0.918310060777226_dbl_kind, 0.973027523668778_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1403) = (/ 0.891297780784554_dbl_kind, 0.89499199195934_dbl_kind, 0.90055711817412_dbl_kind, 0.918319973436644_dbl_kind, 0.973033184637043_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1404) = (/ 0.89129790939293_dbl_kind, 0.894992678355757_dbl_kind, 0.900560353491129_dbl_kind, 0.918329883207169_dbl_kind, 0.973038835931401_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1405) = (/ 0.891298037849446_dbl_kind, 0.894993364416776_dbl_kind, 0.900563588072432_dbl_kind, 0.918339790089897_dbl_kind, 0.973044477571393_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1406) = (/ 0.891298166154481_dbl_kind, 0.894994050142496_dbl_kind, 0.900566821919144_dbl_kind, 0.918349694085913_dbl_kind, 0.973050109576516_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1407) = (/ 0.891298294308409_dbl_kind, 0.894994735533014_dbl_kind, 0.900570055032375_dbl_kind, 0.918359595196298_dbl_kind, 0.973055731966217_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1408) = (/ 0.891298422311604_dbl_kind, 0.894995420588429_dbl_kind, 0.900573287413229_dbl_kind, 0.91836949342212_dbl_kind, 0.973061344759893_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1409) = (/ 0.89129855016444_dbl_kind, 0.89499610530884_dbl_kind, 0.900576519062806_dbl_kind, 0.918379388764442_dbl_kind, 0.973066947976893_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1410) = (/ 0.891298677867286_dbl_kind, 0.89499678969435_dbl_kind, 0.900579749982199_dbl_kind, 0.91838928122432_dbl_kind, 0.973072541636518_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1411) = (/ 0.891298805420514_dbl_kind, 0.894997473745059_dbl_kind, 0.900582980172496_dbl_kind, 0.918399170802798_dbl_kind, 0.973078125758024_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1412) = (/ 0.89129893282449_dbl_kind, 0.894998157461068_dbl_kind, 0.900586209634781_dbl_kind, 0.918409057500915_dbl_kind, 0.973083700360616_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1413) = (/ 0.891299060079582_dbl_kind, 0.894998840842481_dbl_kind, 0.90058943837013_dbl_kind, 0.9184189413197_dbl_kind, 0.97308926546345_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1414) = (/ 0.891299187186155_dbl_kind, 0.894999523889403_dbl_kind, 0.900592666379615_dbl_kind, 0.918428822260175_dbl_kind, 0.97309482108564_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1415) = (/ 0.891299314144572_dbl_kind, 0.895000206601938_dbl_kind, 0.900595893664303_dbl_kind, 0.918438700323354_dbl_kind, 0.973100367246247_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1416) = (/ 0.891299440955197_dbl_kind, 0.89500088898019_dbl_kind, 0.900599120225255_dbl_kind, 0.918448575510242_dbl_kind, 0.973105903964288_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1417) = (/ 0.89129956761839_dbl_kind, 0.895001571024267_dbl_kind, 0.900602346063525_dbl_kind, 0.918458447821835_dbl_kind, 0.973111431258732_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1418) = (/ 0.891299694134511_dbl_kind, 0.895002252734275_dbl_kind, 0.900605571180165_dbl_kind, 0.918468317259123_dbl_kind, 0.973116949148499_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1419) = (/ 0.891299820503918_dbl_kind, 0.895002934110322_dbl_kind, 0.900608795576219_dbl_kind, 0.918478183823086_dbl_kind, 0.973122457652467_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1420) = (/ 0.891299946726968_dbl_kind, 0.895003615152516_dbl_kind, 0.900612019252725_dbl_kind, 0.918488047514697_dbl_kind, 0.973127956789461_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1421) = (/ 0.891300072804015_dbl_kind, 0.895004295860968_dbl_kind, 0.900615242210718_dbl_kind, 0.918497908334921_dbl_kind, 0.973133446578268_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1422) = (/ 0.891300198735414_dbl_kind, 0.895004976235786_dbl_kind, 0.900618464451224_dbl_kind, 0.918507766284713_dbl_kind, 0.973138927037616_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1423) = (/ 0.891300324521518_dbl_kind, 0.895005656277082_dbl_kind, 0.900621685975267_dbl_kind, 0.918517621365022_dbl_kind, 0.973144398186202_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1424) = (/ 0.891300450162678_dbl_kind, 0.895006335984968_dbl_kind, 0.900624906783862_dbl_kind, 0.918527473576789_dbl_kind, 0.973149860042664_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1425) = (/ 0.891300575659242_dbl_kind, 0.895007015359555_dbl_kind, 0.900628126878023_dbl_kind, 0.918537322920943_dbl_kind, 0.973155312625601_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1426) = (/ 0.891300701011561_dbl_kind, 0.895007694400958_dbl_kind, 0.900631346258753_dbl_kind, 0.918547169398409_dbl_kind, 0.973160755953563_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1427) = (/ 0.891300826219981_dbl_kind, 0.895008373109289_dbl_kind, 0.900634564927054_dbl_kind, 0.918557013010102_dbl_kind, 0.973166190045058_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1428) = (/ 0.891300951284847_dbl_kind, 0.895009051484665_dbl_kind, 0.900637782883919_dbl_kind, 0.91856685375693_dbl_kind, 0.973171614918544_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1429) = (/ 0.891301076206504_dbl_kind, 0.895009729527199_dbl_kind, 0.900641000130338_dbl_kind, 0.918576691639792_dbl_kind, 0.973177030592431_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1430) = (/ 0.891301200985295_dbl_kind, 0.895010407237009_dbl_kind, 0.900644216667295_dbl_kind, 0.918586526659578_dbl_kind, 0.973182437085098_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1431) = (/ 0.891301325621561_dbl_kind, 0.89501108461421_dbl_kind, 0.900647432495767_dbl_kind, 0.918596358817171_dbl_kind, 0.973187834414862_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1432) = (/ 0.891301450115644_dbl_kind, 0.895011761658922_dbl_kind, 0.900650647616725_dbl_kind, 0.918606188113446_dbl_kind, 0.973193222600002_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1433) = (/ 0.89130157446788_dbl_kind, 0.895012438371262_dbl_kind, 0.900653862031137_dbl_kind, 0.918616014549269_dbl_kind, 0.973198601658754_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1434) = (/ 0.891301698678609_dbl_kind, 0.895013114751349_dbl_kind, 0.900657075739964_dbl_kind, 0.918625838125497_dbl_kind, 0.973203971609305_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1435) = (/ 0.891301822748166_dbl_kind, 0.895013790799304_dbl_kind, 0.90066028874416_dbl_kind, 0.918635658842983_dbl_kind, 0.973209332469803_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1436) = (/ 0.891301946676886_dbl_kind, 0.895014466515246_dbl_kind, 0.900663501044675_dbl_kind, 0.918645476702567_dbl_kind, 0.973214684258343_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1437) = (/ 0.891302070465103_dbl_kind, 0.895015141899298_dbl_kind, 0.900666712642454_dbl_kind, 0.918655291705081_dbl_kind, 0.973220026992985_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1438) = (/ 0.891302194113149_dbl_kind, 0.895015816951581_dbl_kind, 0.900669923538434_dbl_kind, 0.918665103851354_dbl_kind, 0.973225360691737_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1439) = (/ 0.891302317621356_dbl_kind, 0.895016491672217_dbl_kind, 0.900673133733548_dbl_kind, 0.9186749131422_dbl_kind, 0.973230685372569_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1440) = (/ 0.891302440990053_dbl_kind, 0.895017166061332_dbl_kind, 0.900676343228723_dbl_kind, 0.918684719578429_dbl_kind, 0.973236001053401_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1441) = (/ 0.891302564219568_dbl_kind, 0.895017840119048_dbl_kind, 0.900679552024881_dbl_kind, 0.918694523160844_dbl_kind, 0.973241307752113_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1442) = (/ 0.891302687310228_dbl_kind, 0.895018513845491_dbl_kind, 0.900682760122936_dbl_kind, 0.918704323890235_dbl_kind, 0.973246605486544_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1443) = (/ 0.891302810262359_dbl_kind, 0.895019187240786_dbl_kind, 0.900685967523799_dbl_kind, 0.918714121767388_dbl_kind, 0.973251894274482_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1444) = (/ 0.891302933076286_dbl_kind, 0.89501986030506_dbl_kind, 0.900689174228373_dbl_kind, 0.91872391679308_dbl_kind, 0.973257174133677_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1445) = (/ 0.891303055752333_dbl_kind, 0.895020533038439_dbl_kind, 0.900692380237558_dbl_kind, 0.918733708968078_dbl_kind, 0.973262445081832_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1446) = (/ 0.89130317829082_dbl_kind, 0.895021205441051_dbl_kind, 0.900695585552245_dbl_kind, 0.918743498293142_dbl_kind, 0.973267707136612_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1447) = (/ 0.891303300692068_dbl_kind, 0.895021877513026_dbl_kind, 0.900698790173322_dbl_kind, 0.918753284769025_dbl_kind, 0.973272960315633_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1448) = (/ 0.891303422956397_dbl_kind, 0.895022549254491_dbl_kind, 0.90070199410167_dbl_kind, 0.918763068396469_dbl_kind, 0.973278204636473_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1449) = (/ 0.891303545084126_dbl_kind, 0.895023220665577_dbl_kind, 0.900705197338166_dbl_kind, 0.918772849176211_dbl_kind, 0.973283440116661_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1450) = (/ 0.891303667075571_dbl_kind, 0.895023891746414_dbl_kind, 0.900708399883677_dbl_kind, 0.918782627108979_dbl_kind, 0.97328866677369_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1451) = (/ 0.891303788931048_dbl_kind, 0.895024562497133_dbl_kind, 0.900711601739068_dbl_kind, 0.91879240219549_dbl_kind, 0.973293884625008_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1452) = (/ 0.89130391065087_dbl_kind, 0.895025232917867_dbl_kind, 0.900714802905199_dbl_kind, 0.918802174436457_dbl_kind, 0.973299093688017_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1453) = (/ 0.891304032235353_dbl_kind, 0.895025903008746_dbl_kind, 0.90071800338292_dbl_kind, 0.918811943832581_dbl_kind, 0.973304293980082_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1454) = (/ 0.891304153684807_dbl_kind, 0.895026572769906_dbl_kind, 0.900721203173078_dbl_kind, 0.918821710384559_dbl_kind, 0.973309485518524_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1455) = (/ 0.891304274999542_dbl_kind, 0.895027242201479_dbl_kind, 0.900724402276516_dbl_kind, 0.918831474093075_dbl_kind, 0.973314668320618_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1456) = (/ 0.89130439617987_dbl_kind, 0.8950279113036_dbl_kind, 0.900727600694067_dbl_kind, 0.918841234958809_dbl_kind, 0.973319842403605_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1457) = (/ 0.891304517226098_dbl_kind, 0.895028580076404_dbl_kind, 0.900730798426562_dbl_kind, 0.918850992982431_dbl_kind, 0.973325007784675_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1458) = (/ 0.891304638138532_dbl_kind, 0.895029248520027_dbl_kind, 0.900733995474824_dbl_kind, 0.918860748164603_dbl_kind, 0.973330164480983_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1459) = (/ 0.891304758917479_dbl_kind, 0.895029916634606_dbl_kind, 0.90073719183967_dbl_kind, 0.918870500505978_dbl_kind, 0.973335312509639_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1460) = (/ 0.891304879563244_dbl_kind, 0.895030584420277_dbl_kind, 0.900740387521913_dbl_kind, 0.918880250007203_dbl_kind, 0.973340451887712_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1461) = (/ 0.89130500007613_dbl_kind, 0.895031251877178_dbl_kind, 0.900743582522359_dbl_kind, 0.918889996668914_dbl_kind, 0.973345582632231_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1462) = (/ 0.891305120456439_dbl_kind, 0.895031919005448_dbl_kind, 0.900746776841807_dbl_kind, 0.918899740491742_dbl_kind, 0.973350704760184_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1463) = (/ 0.891305240704472_dbl_kind, 0.895032585805226_dbl_kind, 0.900749970481054_dbl_kind, 0.918909481476307_dbl_kind, 0.973355818288515_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1464) = (/ 0.891305360820529_dbl_kind, 0.895033252276651_dbl_kind, 0.900753163440887_dbl_kind, 0.918919219623223_dbl_kind, 0.973360923234129_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1465) = (/ 0.89130548080491_dbl_kind, 0.895033918419863_dbl_kind, 0.900756355722089_dbl_kind, 0.918928954933093_dbl_kind, 0.973366019613892_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1466) = (/ 0.89130560065791_dbl_kind, 0.895034584235004_dbl_kind, 0.900759547325438_dbl_kind, 0.918938687406517_dbl_kind, 0.973371107444624_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1467) = (/ 0.891305720379828_dbl_kind, 0.895035249722214_dbl_kind, 0.900762738251705_dbl_kind, 0.91894841704408_dbl_kind, 0.973376186743106_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1468) = (/ 0.891305839970957_dbl_kind, 0.895035914881637_dbl_kind, 0.900765928501655_dbl_kind, 0.918958143846365_dbl_kind, 0.973381257526084_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1469) = (/ 0.891305959431592_dbl_kind, 0.895036579713415_dbl_kind, 0.900769118076048_dbl_kind, 0.918967867813943_dbl_kind, 0.973386319810258_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1470) = (/ 0.891306078762026_dbl_kind, 0.89503724421769_dbl_kind, 0.900772306975637_dbl_kind, 0.918977588947379_dbl_kind, 0.973391373612288_dbl_kind /)
   iceAsymmetryParameterDiffuse(1:5 ,1471) = (/ 0.891306197962551_dbl_kind, 0.895037908394608_dbl_kind, 0.900775495201171_dbl_kind, 0.918987307247229_dbl_kind, 0.973396418948797_dbl_kind /)

   iceSingleScatterAlbedoDirect(1:5 ,1) = (/ 0.999996473539642_dbl_kind, 0.999897920739992_dbl_kind, 0.999240565661103_dbl_kind, 0.993823361190655_dbl_kind, 0.941294661423717_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,2) = (/ 0.999996361173434_dbl_kind, 0.999894735419621_dbl_kind, 0.999216637256531_dbl_kind, 0.993697069727296_dbl_kind, 0.939921805058392_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,3) = (/ 0.999996249406051_dbl_kind, 0.999891558786757_dbl_kind, 0.999192716658491_dbl_kind, 0.993572481152941_dbl_kind, 0.938560323564721_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,4) = (/ 0.999996138181685_dbl_kind, 0.999888389969221_dbl_kind, 0.999168786870456_dbl_kind, 0.993449465601411_dbl_kind, 0.937209923402721_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,5) = (/ 0.999996027434647_dbl_kind, 0.999885228003346_dbl_kind, 0.999144833681027_dbl_kind, 0.993327900863047_dbl_kind, 0.935870325891656_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,6) = (/ 0.999995917092138_dbl_kind, 0.999882071888276_dbl_kind, 0.99912084599924_dbl_kind, 0.993207672400846_dbl_kind, 0.934541266588599_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,7) = (/ 0.999995807076844_dbl_kind, 0.999878920635777_dbl_kind, 0.999096815972237_dbl_kind, 0.993088673448881_dbl_kind, 0.93322249430228_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,8) = (/ 0.999995697309289_dbl_kind, 0.999875773312412_dbl_kind, 0.999072738926153_dbl_kind, 0.992970805123498_dbl_kind, 0.931913769984942_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,9) = (/ 0.99999558770994_dbl_kind, 0.999872629072474_dbl_kind, 0.999048613171924_dbl_kind, 0.992853976494181_dbl_kind, 0.930614865653367_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,10) = (/ 0.999995478201023_dbl_kind, 0.999869487181161_dbl_kind, 0.999024439714989_dbl_kind, 0.992738104579_dbl_kind, 0.929325563419527_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,11) = (/ 0.99999536870806_dbl_kind, 0.999866347028399_dbl_kind, 0.999000221903186_dbl_kind, 0.992623114246065_dbl_kind, 0.928045654661507_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,12) = (/ 0.999995259161125_dbl_kind, 0.999863208134334_dbl_kind, 0.998975965041614_dbl_kind, 0.992508938015847_dbl_kind, 0.926774939334079_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,13) = (/ 0.999995149495825_dbl_kind, 0.999860070147774_dbl_kind, 0.998951675997609_dbl_kind, 0.992395515769161_dbl_kind, 0.925513225400971_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,14) = (/ 0.999995039654026_dbl_kind, 0.999856932839081_dbl_kind, 0.998927362813756_dbl_kind, 0.992282794372209_dbl_kind, 0.924260328363991_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,15) = (/ 0.999994929584356_dbl_kind, 0.999853796088973_dbl_kind, 0.998903034342147_dbl_kind, 0.992170727233891_dbl_kind, 0.9230160708634_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,16) = (/ 0.999994819242493_dbl_kind, 0.999850659874595_dbl_kind, 0.998878699909057_dbl_kind, 0.992059273812212_dbl_kind, 0.921780282327223_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,17) = (/ 0.999994708591275_dbl_kind, 0.99984752425406_dbl_kind, 0.998854369015923_dbl_kind, 0.991948399086661_dbl_kind, 0.920552798651865_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,18) = (/ 0.999994597600659_dbl_kind, 0.999844389350458_dbl_kind, 0.998830051079715_dbl_kind, 0.991838073012411_dbl_kind, 0.919333461901503_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,19) = (/ 0.999994486247544_dbl_kind, 0.99984125533615_dbl_kind, 0.998805755213725_dbl_kind, 0.991728269970545_dbl_kind, 0.918122120018432_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,20) = (/ 0.999994374515509_dbl_kind, 0.999838122417929_dbl_kind, 0.998781490048113_dbl_kind, 0.991618968226489_dbl_kind, 0.916918626540208_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,21) = (/ 0.999994262394443_dbl_kind, 0.999834990823491_dbl_kind, 0.998757263588354_dbl_kind, 0.991510149406748_dbl_kind, 0.915722840322303_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,22) = (/ 0.999994149880139_dbl_kind, 0.999831860789493_dbl_kind, 0.998733083108911_dbl_kind, 0.991401798001929_dbl_kind, 0.914534625266845_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,23) = (/ 0.999994036973832_dbl_kind, 0.999828732551317_dbl_kind, 0.998708955078855_dbl_kind, 0.991293900902114_dbl_kind, 0.913353850058993_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,24) = (/ 0.99999392368171_dbl_kind, 0.999825606334613_dbl_kind, 0.998684885115909_dbl_kind, 0.991186446968852_dbl_kind, 0.912180387913036_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,25) = (/ 0.999993810014412_dbl_kind, 0.999822482348553_dbl_kind, 0.998660877965212_dbl_kind, 0.991079426646468_dbl_kind, 0.91101411633025_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,26) = (/ 0.999993695986521_dbl_kind, 0.999819360780719_dbl_kind, 0.998636937499187_dbl_kind, 0.990972831614054_dbl_kind, 0.909854916870194_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,27) = (/ 0.999993581616066_dbl_kind, 0.999816241793466_dbl_kind, 0.998613066734978_dbl_kind, 0.990866654478355_dbl_kind, 0.908702674936746_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,28) = (/ 0.99999346692402_dbl_kind, 0.999813125521626_dbl_kind, 0.998589267866166_dbl_kind, 0.990760888506839_dbl_kind, 0.907557279579561_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,29) = (/ 0.99999335193384_dbl_kind, 0.999810012071344_dbl_kind, 0.998565542305738_dbl_kind, 0.990655527399463_dbl_kind, 0.906418623311165_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,30) = (/ 0.999993236671004_dbl_kind, 0.999806901519907_dbl_kind, 0.998541890737549_dbl_kind, 0.990550565097123_dbl_kind, 0.905286601939379_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,31) = (/ 0.999993121162588_dbl_kind, 0.999803793916362_dbl_kind, 0.998518313173872_dbl_kind, 0.990445995624275_dbl_kind, 0.90416111441443_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,32) = (/ 0.999993005436863_dbl_kind, 0.999800689282775_dbl_kind, 0.998494809016887_dbl_kind, 0.990341812963_dbl_kind, 0.903042062689752_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,33) = (/ 0.99999288952292_dbl_kind, 0.999797587615991_dbl_kind, 0.998471377122302_dbl_kind, 0.990238010955531_dbl_kind, 0.901929351595277_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,34) = (/ 0.999992773450328_dbl_kind, 0.999794488889745_dbl_kind, 0.99844801586356_dbl_kind, 0.99013458323225_dbl_kind, 0.900822888721985_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,35) = (/ 0.99999265724882_dbl_kind, 0.999791393057014_dbl_kind, 0.998424723195369_dbl_kind, 0.990031523162107_dbl_kind, 0.899722584316255_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,36) = (/ 0.999992540948004_dbl_kind, 0.999788300052515_dbl_kind, 0.998401496715505_dbl_kind, 0.989928823822514_dbl_kind, 0.898628351182785_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,37) = (/ 0.999992424577107_dbl_kind, 0.999785209795256_dbl_kind, 0.998378333724095_dbl_kind, 0.989826477985864_dbl_kind, 0.897540104594721_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,38) = (/ 0.99999230816475_dbl_kind, 0.99978212219108_dbl_kind, 0.998355231279746_dbl_kind, 0.989724478120003_dbl_kind, 0.896457762209863_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,39) = (/ 0.99999219173874_dbl_kind, 0.999779037135125_dbl_kind, 0.99833218625207_dbl_kind, 0.989622816400156_dbl_kind, 0.895381243991876_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,40) = (/ 0.999992075325901_dbl_kind, 0.999775954514193_dbl_kind, 0.998309195370299_dbl_kind, 0.989521484730012_dbl_kind, 0.894310472135542_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,41) = (/ 0.999991958951922_dbl_kind, 0.99977287420895_dbl_kind, 0.998286255267812_dbl_kind, 0.989420474769906_dbl_kind, 0.893245370995306_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,42) = (/ 0.999991842641231_dbl_kind, 0.999769796095979_dbl_kind, 0.998263362522481_dbl_kind, 0.989319777970234_dbl_kind, 0.892185867016399_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,43) = (/ 0.999991726416893_dbl_kind, 0.999766720049631_dbl_kind, 0.998240513692866_dbl_kind, 0.989219385608479_dbl_kind, 0.891131888668029_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,44) = (/ 0.999991610300526_dbl_kind, 0.999763645943698_dbl_kind, 0.998217705350313_dbl_kind, 0.98911928882842_dbl_kind, 0.890083366378218_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,45) = (/ 0.999991494312241_dbl_kind, 0.999760573652885_dbl_kind, 0.998194934107099_dbl_kind, 0.989019478680316_dbl_kind, 0.889040232469953_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,46) = (/ 0.999991378470595_dbl_kind, 0.999757503054088_dbl_kind, 0.998172196640793_dbl_kind, 0.988919946161042_dbl_kind, 0.888002421098439_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,47) = (/ 0.999991262792566_dbl_kind, 0.999754434027491_dbl_kind, 0.998149489715045_dbl_kind, 0.988820682253317_dbl_kind, 0.886969868189292_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,48) = (/ 0.999991147293544_dbl_kind, 0.999751366457485_dbl_kind, 0.998126810197022_dbl_kind, 0.988721677963367_dbl_kind, 0.885942511377617_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,49) = (/ 0.999991031987326_dbl_kind, 0.999748300233412_dbl_kind, 0.998104155071743_dbl_kind, 0.988622924356465_dbl_kind, 0.88492028994791_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,50) = (/ 0.999990916886139_dbl_kind, 0.999745235250165_dbl_kind, 0.998081521453549_dbl_kind, 0.988524412589953_dbl_kind, 0.883903144774828_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,51) = (/ 0.999990802000661_dbl_kind, 0.999742171408633_dbl_kind, 0.998058906594972_dbl_kind, 0.988426133943462_dbl_kind, 0.88289101826485_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,52) = (/ 0.999990687340057_dbl_kind, 0.999739108616031_dbl_kind, 0.998036307893254_dbl_kind, 0.988328079846139_dbl_kind, 0.88188385429892_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,53) = (/ 0.999990572912027_dbl_kind, 0.999736046786104_dbl_kind, 0.998013722894732_dbl_kind, 0.988230241900782_dbl_kind, 0.880881598176128_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,54) = (/ 0.999990458722857_dbl_kind, 0.99973298583923_dbl_kind, 0.997991149297371_dbl_kind, 0.988132611904862_dbl_kind, 0.879884196558546_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,55) = (/ 0.999990344777472_dbl_kind, 0.999729925702442_dbl_kind, 0.997968584951615_dbl_kind, 0.988035181868449_dbl_kind, 0.878891597417318_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,56) = (/ 0.999990231079508_dbl_kind, 0.999726866309362_dbl_kind, 0.997946027859809_dbl_kind, 0.987937944029165_dbl_kind, 0.877903749980053_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,57) = (/ 0.999990117631371_dbl_kind, 0.999723807600081_dbl_kind, 0.99792347617437_dbl_kind, 0.987840890864242_dbl_kind, 0.876920604679668_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,58) = (/ 0.999990004434316_dbl_kind, 0.999720749520974_dbl_kind, 0.997900928194887_dbl_kind, 0.987744015099875_dbl_kind, 0.875942113104744_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,59) = (/ 0.999989891488511_dbl_kind, 0.999717692024476_dbl_kind, 0.997878382364332_dbl_kind, 0.987647309718035_dbl_kind, 0.874968227951409_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,60) = (/ 0.999989778793117_dbl_kind, 0.999714635068829_dbl_kind, 0.997855837264532_dbl_kind, 0.987550767960936_dbl_kind, 0.873998902976937_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,61) = (/ 0.999989666346362_dbl_kind, 0.999711578617787_dbl_kind, 0.997833291611034_dbl_kind, 0.98745438333336_dbl_kind, 0.873034092954959_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,62) = (/ 0.999989554145612_dbl_kind, 0.999708522640321_dbl_kind, 0.997810744247507_dbl_kind, 0.987358149603049_dbl_kind, 0.872073753632452_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,63) = (/ 0.999989442187455_dbl_kind, 0.999705467110289_dbl_kind, 0.997788194139775_dbl_kind, 0.987262060799374_dbl_kind, 0.871117841688458_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,64) = (/ 0.999989330467762_dbl_kind, 0.99970241200612_dbl_kind, 0.997765640369605_dbl_kind, 0.987166111210474_dbl_kind, 0.870166314694558_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,65) = (/ 0.999989218981767_dbl_kind, 0.999699357310482_dbl_kind, 0.997743082128317_dbl_kind, 0.987070295379083_dbl_kind, 0.869219131077165_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,66) = (/ 0.999989107724132_dbl_kind, 0.999696303009957_dbl_kind, 0.997720518710311_dbl_kind, 0.986974608097212_dbl_kind, 0.868276250081496_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,67) = (/ 0.99998899668902_dbl_kind, 0.999693249094721_dbl_kind, 0.997697949506573_dbl_kind, 0.98687904439989_dbl_kind, 0.867337631737369_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,68) = (/ 0.999988885870154_dbl_kind, 0.999690195558232_dbl_kind, 0.997675373998225_dbl_kind, 0.986783599558102_dbl_kind, 0.86640323682667_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,69) = (/ 0.99998877526088_dbl_kind, 0.999687142396932_dbl_kind, 0.99765279175015_dbl_kind, 0.986688269071112_dbl_kind, 0.865473026852525_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,70) = (/ 0.999988664854231_dbl_kind, 0.999684089609956_dbl_kind, 0.99763020240477_dbl_kind, 0.986593048658292_dbl_kind, 0.864546964010107_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,71) = (/ 0.999988554642975_dbl_kind, 0.999681037198863_dbl_kind, 0.997607605675971_dbl_kind, 0.98649793425059_dbl_kind, 0.863625011159058_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,72) = (/ 0.999988444619675_dbl_kind, 0.999677985167374_dbl_kind, 0.997585001343239_dbl_kind, 0.98640292198176_dbl_kind, 0.862707131797436_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,73) = (/ 0.999988334776732_dbl_kind, 0.999674933521132_dbl_kind, 0.997562389246002_dbl_kind, 0.986308008179464_dbl_kind, 0.861793290037168_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,74) = (/ 0.999988225106432_dbl_kind, 0.999671882267478_dbl_kind, 0.997539769278229_dbl_kind, 0.986213189356309_dbl_kind, 0.860883450580945_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,75) = (/ 0.999988115600994_dbl_kind, 0.999668831415243_dbl_kind, 0.997517141383254_dbl_kind, 0.986118462200938_dbl_kind, 0.859977578700462_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,76) = (/ 0.999988006252603_dbl_kind, 0.999665780974552_dbl_kind, 0.997494505548878_dbl_kind, 0.986023823569227_dbl_kind, 0.859075640216005_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,77) = (/ 0.999987897053447_dbl_kind, 0.99966273095665_dbl_kind, 0.997471861802732_dbl_kind, 0.985929270475626_dbl_kind, 0.858177601477257_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,78) = (/ 0.999987787995752_dbl_kind, 0.999659681373747_dbl_kind, 0.997449210207908_dbl_kind, 0.985834800084737_dbl_kind, 0.857283429345278_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,79) = (/ 0.999987679071813_dbl_kind, 0.999656632238866_dbl_kind, 0.997426550858855_dbl_kind, 0.98574040970313_dbl_kind, 0.85639309117566_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,80) = (/ 0.999987570274012_dbl_kind, 0.999653583565711_dbl_kind, 0.997403883877544_dbl_kind, 0.985646096771452_dbl_kind, 0.855506554802647_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,81) = (/ 0.999987461594855_dbl_kind, 0.999650535368563_dbl_kind, 0.997381209409902_dbl_kind, 0.985551858856839_dbl_kind, 0.85462378852434_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,82) = (/ 0.99998735302698_dbl_kind, 0.999647487662163_dbl_kind, 0.997358527622488_dbl_kind, 0.985457693645672_dbl_kind, 0.853744761088761_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,83) = (/ 0.999987244563187_dbl_kind, 0.999644440461628_dbl_kind, 0.997335838699428_dbl_kind, 0.985363598936656_dbl_kind, 0.85286944168084_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,84) = (/ 0.999987136196446_dbl_kind, 0.999641393782366_dbl_kind, 0.997313142839596_dbl_kind, 0.98526957263426_dbl_kind, 0.851997799910171_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,85) = (/ 0.999987027919913_dbl_kind, 0.999638347640009_dbl_kind, 0.99729044025402_dbl_kind, 0.985175612742506_dbl_kind, 0.851129805799573_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,86) = (/ 0.999986919726943_dbl_kind, 0.999635302050357_dbl_kind, 0.997267731163508_dbl_kind, 0.985081717359106_dbl_kind, 0.850265429774311_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,87) = (/ 0.999986811611099_dbl_kind, 0.999632257029318_dbl_kind, 0.997245015796497_dbl_kind, 0.984987884669948_dbl_kind, 0.849404642652022_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,88) = (/ 0.999986703566159_dbl_kind, 0.999629212592869_dbl_kind, 0.997222294387102_dbl_kind, 0.98489411294392_dbl_kind, 0.848547415633198_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,89) = (/ 0.999986595586122_dbl_kind, 0.999626168757019_dbl_kind, 0.997199567173354_dbl_kind, 0.984800400528062_dbl_kind, 0.847693720292266_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,90) = (/ 0.999986487665212_dbl_kind, 0.999623125537779_dbl_kind, 0.997176834395619_dbl_kind, 0.98470674584304_dbl_kind, 0.846843528569165_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,91) = (/ 0.99998637979788_dbl_kind, 0.999620082951137_dbl_kind, 0.997154096295194_dbl_kind, 0.984613147378922_dbl_kind, 0.845996812761414_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,92) = (/ 0.99998627197881_dbl_kind, 0.999617041013038_dbl_kind, 0.997131353113063_dbl_kind, 0.984519603691242_dbl_kind, 0.845153545516615_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,93) = (/ 0.999986164202915_dbl_kind, 0.999613999739369_dbl_kind, 0.99710860508879_dbl_kind, 0.984426113397355_dbl_kind, 0.84431369982535_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,94) = (/ 0.999986056465334_dbl_kind, 0.999610959145946_dbl_kind, 0.997085852459573_dbl_kind, 0.984332675173032_dbl_kind, 0.843477249014463_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,95) = (/ 0.999985948761437_dbl_kind, 0.999607919248509_dbl_kind, 0.997063095459413_dbl_kind, 0.984239287749321_dbl_kind, 0.842644166740675_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,96) = (/ 0.999985841086818_dbl_kind, 0.99960488006271_dbl_kind, 0.997040334318406_dbl_kind, 0.984145949909627_dbl_kind, 0.8418144269845_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,97) = (/ 0.999985733437291_dbl_kind, 0.999601841604114_dbl_kind, 0.99701756926215_dbl_kind, 0.984052660487007_dbl_kind, 0.840988004044465_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,98) = (/ 0.999985625808889_dbl_kind, 0.999598803888194_dbl_kind, 0.996994800511252_dbl_kind, 0.983959418361672_dbl_kind, 0.840164872531574_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,99) = (/ 0.999985518197856_dbl_kind, 0.999595766930333_dbl_kind, 0.996972028280929_dbl_kind, 0.983866222458669_dbl_kind, 0.839345007364012_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,100) = (/ 0.999985410600644_dbl_kind, 0.99959273074582_dbl_kind, 0.996949252780691_dbl_kind, 0.983773071745727_dbl_kind, 0.838528383762069_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,101) = (/ 0.999985303013908_dbl_kind, 0.999589695349857_dbl_kind, 0.996926474214107_dbl_kind, 0.983679965231275_dbl_kind, 0.837714977243259_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,102) = (/ 0.999985195434496_dbl_kind, 0.999586660757551_dbl_kind, 0.99690369277864_dbl_kind, 0.983586901962591_dbl_kind, 0.836904763617621_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,103) = (/ 0.999985087859447_dbl_kind, 0.999583626983927_dbl_kind, 0.99688090866554_dbl_kind, 0.983493881024083_dbl_kind, 0.836097718983174_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,104) = (/ 0.999984980285986_dbl_kind, 0.99958059404392_dbl_kind, 0.996858122059804_dbl_kind, 0.983400901535699_dbl_kind, 0.835293819721546_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,105) = (/ 0.999984872711509_dbl_kind, 0.99957756195238_dbl_kind, 0.996835333140173_dbl_kind, 0.983307962651435_dbl_kind, 0.834493042493715_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,106) = (/ 0.999984765133589_dbl_kind, 0.999574530724072_dbl_kind, 0.996812542079185_dbl_kind, 0.983215063557941_dbl_kind, 0.833695364235881_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,107) = (/ 0.999984657549955_dbl_kind, 0.999571500373681_dbl_kind, 0.996789749043262_dbl_kind, 0.983122203473226_dbl_kind, 0.832900762155469_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,108) = (/ 0.999984549958498_dbl_kind, 0.999568470915806_dbl_kind, 0.996766954192832_dbl_kind, 0.983029381645431_dbl_kind, 0.832109213727222_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,109) = (/ 0.999984442357257_dbl_kind, 0.99956544236496_dbl_kind, 0.99674415768248_dbl_kind, 0.98293659735168_dbl_kind, 0.831320696689381_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,110) = (/ 0.999984334744413_dbl_kind, 0.999562414735575_dbl_kind, 0.996721359661123_dbl_kind, 0.982843849896993_dbl_kind, 0.83053518903999_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,111) = (/ 0.999984227118285_dbl_kind, 0.999559388041995_dbl_kind, 0.996698560272209_dbl_kind, 0.982751138613253_dbl_kind, 0.829752669033236_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,112) = (/ 0.99998411947732_dbl_kind, 0.999556362298471_dbl_kind, 0.996675759653931_dbl_kind, 0.982658462858233_dbl_kind, 0.828973115175904_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,113) = (/ 0.999984011820093_dbl_kind, 0.999553337519166_dbl_kind, 0.996652957939451_dbl_kind, 0.982565822014663_dbl_kind, 0.828196506223886_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,114) = (/ 0.99998390414529_dbl_kind, 0.999550313718141_dbl_kind, 0.996630155257145_dbl_kind, 0.982473215489331_dbl_kind, 0.827422821178748_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,115) = (/ 0.999983796451713_dbl_kind, 0.99954729090936_dbl_kind, 0.996607351730843_dbl_kind, 0.982380642712235_dbl_kind, 0.826652039284379_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,116) = (/ 0.999983688738269_dbl_kind, 0.999544269106678_dbl_kind, 0.996584547480085_dbl_kind, 0.982288103135756_dbl_kind, 0.825884140023673_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,117) = (/ 0.999983581003962_dbl_kind, 0.999541248323839_dbl_kind, 0.99656174262037_dbl_kind, 0.982195596233865_dbl_kind, 0.825119103115293_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,118) = (/ 0.999983473247892_dbl_kind, 0.999538228574463_dbl_kind, 0.996538937263417_dbl_kind, 0.982103121501354_dbl_kind, 0.824356908510459_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,119) = (/ 0.999983365469249_dbl_kind, 0.999535209872047_dbl_kind, 0.996516131517411_dbl_kind, 0.98201067845309_dbl_kind, 0.823597536389802_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,120) = (/ 0.999983257667303_dbl_kind, 0.999532192229956_dbl_kind, 0.996493325487261_dbl_kind, 0.981918266623293_dbl_kind, 0.82284096716026_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,121) = (/ 0.999983149841407_dbl_kind, 0.999529175661408_dbl_kind, 0.996470519274839_dbl_kind, 0.981825885564829_dbl_kind, 0.822087181452019_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,122) = (/ 0.999983041990985_dbl_kind, 0.999526160179473_dbl_kind, 0.996447712979227_dbl_kind, 0.981733534848527_dbl_kind, 0.821336160115486_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,123) = (/ 0.999982934115533_dbl_kind, 0.999523145797061_dbl_kind, 0.996424906696945_dbl_kind, 0.981641214062499_dbl_kind, 0.820587884218327_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,124) = (/ 0.999982826214608_dbl_kind, 0.999520132526913_dbl_kind, 0.996402100522179_dbl_kind, 0.981548922811498_dbl_kind, 0.819842335042523_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,125) = (/ 0.999982718287834_dbl_kind, 0.999517120381593_dbl_kind, 0.996379294546999_dbl_kind, 0.981456660716262_dbl_kind, 0.819099494081472_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,126) = (/ 0.999982610334887_dbl_kind, 0.999514109373478_dbl_kind, 0.996356488861564_dbl_kind, 0.981364427412895_dbl_kind, 0.818359343037132_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,127) = (/ 0.9999825023555_dbl_kind, 0.999511099514746_dbl_kind, 0.996333683554323_dbl_kind, 0.981272222552251_dbl_kind, 0.817621863817198_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,128) = (/ 0.999982394349454_dbl_kind, 0.999508090817371_dbl_kind, 0.9963108787122_dbl_kind, 0.981180045799326_dbl_kind, 0.816887038532311_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,129) = (/ 0.999982286316578_dbl_kind, 0.999505083293109_dbl_kind, 0.996288074420773_dbl_kind, 0.981087896832675_dbl_kind, 0.816154849493315_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,130) = (/ 0.999982178256744_dbl_kind, 0.999502076953489_dbl_kind, 0.996265270764436_dbl_kind, 0.98099577534383_dbl_kind, 0.815425279208539_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,131) = (/ 0.999982070169865_dbl_kind, 0.999499071809809_dbl_kind, 0.996242467826558_dbl_kind, 0.980903681036733_dbl_kind, 0.814698310381101_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,132) = (/ 0.999981962055892_dbl_kind, 0.999496067873118_dbl_kind, 0.996219665689625_dbl_kind, 0.980811613627188_dbl_kind, 0.813973925906295_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,133) = (/ 0.999981853914809_dbl_kind, 0.999493065154213_dbl_kind, 0.996196864435375_dbl_kind, 0.980719572842316_dbl_kind, 0.813252108868931_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,134) = (/ 0.999981745746636_dbl_kind, 0.999490063663629_dbl_kind, 0.996174064144912_dbl_kind, 0.980627558420019_dbl_kind, 0.812532842540798_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,135) = (/ 0.999981637551422_dbl_kind, 0.999487063411626_dbl_kind, 0.996151264898828_dbl_kind, 0.980535570108479_dbl_kind, 0.811816110378088_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,136) = (/ 0.999981529329242_dbl_kind, 0.99948406440819_dbl_kind, 0.996128466777293_dbl_kind, 0.980443607665639_dbl_kind, 0.811101896018897_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,137) = (/ 0.999981421080201_dbl_kind, 0.999481066663014_dbl_kind, 0.996105669860151_dbl_kind, 0.980351670858719_dbl_kind, 0.810390183280735_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,138) = (/ 0.999981312804427_dbl_kind, 0.999478070185498_dbl_kind, 0.996082874226994_dbl_kind, 0.980259759463736_dbl_kind, 0.809680956158078_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,139) = (/ 0.99998120450207_dbl_kind, 0.999475074984739_dbl_kind, 0.996060079957238_dbl_kind, 0.980167873265038_dbl_kind, 0.808974198819959_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,140) = (/ 0.999981096173299_dbl_kind, 0.999472081069523_dbl_kind, 0.996037287130174_dbl_kind, 0.980076012054852_dbl_kind, 0.808269895607564_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,141) = (/ 0.999980987818305_dbl_kind, 0.999469088448323_dbl_kind, 0.996014495825026_dbl_kind, 0.979984175632846_dbl_kind, 0.807568031031901_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,142) = (/ 0.999980879437296_dbl_kind, 0.999466097129288_dbl_kind, 0.99599170612099_dbl_kind, 0.979892363805704_dbl_kind, 0.806868589771456_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,143) = (/ 0.999980771030497_dbl_kind, 0.999463107120239_dbl_kind, 0.995968918097265_dbl_kind, 0.979800576386713_dbl_kind, 0.806171556669916_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,144) = (/ 0.999980662598146_dbl_kind, 0.999460118428667_dbl_kind, 0.995946131833081_dbl_kind, 0.979708813195372_dbl_kind, 0.805476916733903_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,145) = (/ 0.999980554140495_dbl_kind, 0.999457131061723_dbl_kind, 0.995923347407716_dbl_kind, 0.979617074056996_dbl_kind, 0.804784655130745_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,146) = (/ 0.999980445657813_dbl_kind, 0.999454145026219_dbl_kind, 0.995900564900505_dbl_kind, 0.979525358802362_dbl_kind, 0.804094757186281_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,147) = (/ 0.999980337150376_dbl_kind, 0.999451160328625_dbl_kind, 0.995877784390842_dbl_kind, 0.979433667267341_dbl_kind, 0.803407208382678_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,148) = (/ 0.999980228618474_dbl_kind, 0.999448176975059_dbl_kind, 0.995855005958184_dbl_kind, 0.979341999292564_dbl_kind, 0.802721994356317_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,149) = (/ 0.999980120062404_dbl_kind, 0.999445194971292_dbl_kind, 0.995832229682032_dbl_kind, 0.979250354723093_dbl_kind, 0.802039100895661_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,150) = (/ 0.999980011482475_dbl_kind, 0.999442214322745_dbl_kind, 0.995809455641926_dbl_kind, 0.979158733408104_dbl_kind, 0.80135851393919_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,151) = (/ 0.999979902879002_dbl_kind, 0.99943923503448_dbl_kind, 0.995786683917419_dbl_kind, 0.979067135200596_dbl_kind, 0.800680219573345_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,152) = (/ 0.99997979425231_dbl_kind, 0.999436257111212_dbl_kind, 0.995763914588057_dbl_kind, 0.9789755599571_dbl_kind, 0.800004204030512_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,153) = (/ 0.999979685602729_dbl_kind, 0.999433280557297_dbl_kind, 0.99574114773335_dbl_kind, 0.978884007537404_dbl_kind, 0.799330453687026_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,154) = (/ 0.999979576930596_dbl_kind, 0.999430305376737_dbl_kind, 0.995718383432736_dbl_kind, 0.978792477804303_dbl_kind, 0.798658955061216_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,155) = (/ 0.999979468236252_dbl_kind, 0.999427331573181_dbl_kind, 0.995695621765552_dbl_kind, 0.97870097062335_dbl_kind, 0.797989694811458_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,156) = (/ 0.999979359520047_dbl_kind, 0.999424359149926_dbl_kind, 0.995672862810991_dbl_kind, 0.978609485862623_dbl_kind, 0.797322659734281_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,157) = (/ 0.999979250782333_dbl_kind, 0.999421388109913_dbl_kind, 0.99565010664806_dbl_kind, 0.978518023392506_dbl_kind, 0.796657836762484_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,158) = (/ 0.999979142023465_dbl_kind, 0.99941841845574_dbl_kind, 0.995627353355539_dbl_kind, 0.978426583085493_dbl_kind, 0.795995212963282_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,159) = (/ 0.999979033243804_dbl_kind, 0.99941545018965_dbl_kind, 0.995604603011935_dbl_kind, 0.978335164815978_dbl_kind, 0.795334775536489_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,160) = (/ 0.999978924443714_dbl_kind, 0.999412483313548_dbl_kind, 0.995581855695429_dbl_kind, 0.978243768460088_dbl_kind, 0.794676511812718_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,161) = (/ 0.999978815623561_dbl_kind, 0.999409517828994_dbl_kind, 0.995559111483838_dbl_kind, 0.978152393895505_dbl_kind, 0.794020409251605_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,162) = (/ 0.999978706783713_dbl_kind, 0.999406553737209_dbl_kind, 0.995536370454552_dbl_kind, 0.978061041001314_dbl_kind, 0.793366455440072_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,163) = (/ 0.999978597924541_dbl_kind, 0.999403591039081_dbl_kind, 0.995513632684492_dbl_kind, 0.977969709657853_dbl_kind, 0.792714638090616_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,164) = (/ 0.999978489046419_dbl_kind, 0.999400629735167_dbl_kind, 0.995490898250059_dbl_kind, 0.977878399746578_dbl_kind, 0.792064945039606_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,165) = (/ 0.999978380149719_dbl_kind, 0.9993976698257_dbl_kind, 0.995468167227075_dbl_kind, 0.977787111149943_dbl_kind, 0.791417364245611_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,166) = (/ 0.999978271234818_dbl_kind, 0.99939471131059_dbl_kind, 0.995445439690737_dbl_kind, 0.977695843751276_dbl_kind, 0.790771883787782_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,167) = (/ 0.999978162302091_dbl_kind, 0.999391754189433_dbl_kind, 0.995422715715565_dbl_kind, 0.977604597434684_dbl_kind, 0.790128491864212_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,168) = (/ 0.999978053351915_dbl_kind, 0.999388798461513_dbl_kind, 0.995399995375349_dbl_kind, 0.97751337208495_dbl_kind, 0.789487176790356_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,169) = (/ 0.999977944384667_dbl_kind, 0.99938584412581_dbl_kind, 0.995377278743097_dbl_kind, 0.977422167587457_dbl_kind, 0.788847926997462_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,170) = (/ 0.999977835400724_dbl_kind, 0.999382891181004_dbl_kind, 0.99535456589099_dbl_kind, 0.977330983828098_dbl_kind, 0.78821073103102_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,171) = (/ 0.999977726400462_dbl_kind, 0.999379939625484_dbl_kind, 0.995331856890327_dbl_kind, 0.97723982069322_dbl_kind, 0.787575577549248_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,172) = (/ 0.999977617384259_dbl_kind, 0.99937698945735_dbl_kind, 0.99530915181148_dbl_kind, 0.977148678069556_dbl_kind, 0.786942455321594_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,173) = (/ 0.999977508352489_dbl_kind, 0.999374040674424_dbl_kind, 0.995286450723847_dbl_kind, 0.977057555844175_dbl_kind, 0.786311353227264_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,174) = (/ 0.999977399305528_dbl_kind, 0.999371093274253_dbl_kind, 0.995263753695807_dbl_kind, 0.976966453904437_dbl_kind, 0.785682260253755_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,175) = (/ 0.999977290243749_dbl_kind, 0.999368147254119_dbl_kind, 0.995241060794672_dbl_kind, 0.976875372137952_dbl_kind, 0.785055165495449_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,176) = (/ 0.999977181167526_dbl_kind, 0.999365202611041_dbl_kind, 0.995218372086649_dbl_kind, 0.976784310432556_dbl_kind, 0.784430058152181_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,177) = (/ 0.999977072077228_dbl_kind, 0.999362259341788_dbl_kind, 0.995195687636795_dbl_kind, 0.976693268676275_dbl_kind, 0.783806927527871_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,178) = (/ 0.999976962973224_dbl_kind, 0.999359317442882_dbl_kind, 0.995173007508982_dbl_kind, 0.976602246757313_dbl_kind, 0.783185763029147_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,179) = (/ 0.999976853855885_dbl_kind, 0.999356376910607_dbl_kind, 0.995150331765853_dbl_kind, 0.97651124456404_dbl_kind, 0.782566554164004_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,180) = (/ 0.999976744725574_dbl_kind, 0.999353437741016_dbl_kind, 0.995127660468796_dbl_kind, 0.976420261984975_dbl_kind, 0.781949290540488_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,181) = (/ 0.999976635582654_dbl_kind, 0.999350499929937_dbl_kind, 0.995104993677897_dbl_kind, 0.976329298908794_dbl_kind, 0.781333961865375_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,182) = (/ 0.999976526427487_dbl_kind, 0.999347563472983_dbl_kind, 0.99508233145192_dbl_kind, 0.976238355224321_dbl_kind, 0.780720557942906_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,183) = (/ 0.999976417260431_dbl_kind, 0.999344628365557_dbl_kind, 0.995059673848271_dbl_kind, 0.97614743082054_dbl_kind, 0.78010906867351_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,184) = (/ 0.999976308081842_dbl_kind, 0.999341694602861_dbl_kind, 0.995037020922968_dbl_kind, 0.976056525586603_dbl_kind, 0.779499484052566_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,185) = (/ 0.999976198892073_dbl_kind, 0.9993387621799_dbl_kind, 0.99501437273062_dbl_kind, 0.975965639411841_dbl_kind, 0.77889179416918_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,186) = (/ 0.999976089691474_dbl_kind, 0.999335831091497_dbl_kind, 0.994991729324402_dbl_kind, 0.975874772185787_dbl_kind, 0.778285989204964_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,187) = (/ 0.999975980480392_dbl_kind, 0.999332901332293_dbl_kind, 0.99496909075603_dbl_kind, 0.975783923798188_dbl_kind, 0.777682059432865_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,188) = (/ 0.999975871259171_dbl_kind, 0.999329972896755_dbl_kind, 0.994946457075739_dbl_kind, 0.975693094139035_dbl_kind, 0.777079995215985_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,189) = (/ 0.99997576202815_dbl_kind, 0.999327045779189_dbl_kind, 0.994923828332274_dbl_kind, 0.975602283098581_dbl_kind, 0.776479787006433_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,190) = (/ 0.999975652787668_dbl_kind, 0.999324119973742_dbl_kind, 0.994901204572863_dbl_kind, 0.975511490567376_dbl_kind, 0.775881425344182_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,191) = (/ 0.999975543538057_dbl_kind, 0.99932119547441_dbl_kind, 0.994878585843212_dbl_kind, 0.975420716436295_dbl_kind, 0.775284900855964_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,192) = (/ 0.999975434279646_dbl_kind, 0.999318272275047_dbl_kind, 0.994855972187487_dbl_kind, 0.975329960596564_dbl_kind, 0.774690204254155_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,193) = (/ 0.999975325012762_dbl_kind, 0.99931535036937_dbl_kind, 0.994833363648303_dbl_kind, 0.975239222939803_dbl_kind, 0.774097326335703_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,194) = (/ 0.999975215737726_dbl_kind, 0.999312429750969_dbl_kind, 0.994810760266721_dbl_kind, 0.975148503358054_dbl_kind, 0.773506257981057_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,195) = (/ 0.999975106454855_dbl_kind, 0.99930951041331_dbl_kind, 0.994788162082239_dbl_kind, 0.97505780174382_dbl_kind, 0.772916990153109_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,196) = (/ 0.999974997164465_dbl_kind, 0.999306592349744_dbl_kind, 0.994765569132782_dbl_kind, 0.974967117990098_dbl_kind, 0.772329513896167_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,197) = (/ 0.999974887866863_dbl_kind, 0.999303675553515_dbl_kind, 0.994742981454706_dbl_kind, 0.974876451990427_dbl_kind, 0.77174382033493_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,198) = (/ 0.999974778562356_dbl_kind, 0.999300760017763_dbl_kind, 0.994720399082797_dbl_kind, 0.974785803638914_dbl_kind, 0.77115990067348_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,199) = (/ 0.999974669251242_dbl_kind, 0.999297845735533_dbl_kind, 0.994697822050264_dbl_kind, 0.974695172830284_dbl_kind, 0.770577746194309_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,200) = (/ 0.99997455993382_dbl_kind, 0.999294932699782_dbl_kind, 0.994675250388748_dbl_kind, 0.974604559459912_dbl_kind, 0.769997348257317_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,201) = (/ 0.99997445061038_dbl_kind, 0.999292020903382_dbl_kind, 0.994652684128321_dbl_kind, 0.974513963423868_dbl_kind, 0.769418698298877_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,202) = (/ 0.99997434128121_dbl_kind, 0.999289110339131_dbl_kind, 0.994630123297497_dbl_kind, 0.974423384618953_dbl_kind, 0.768841787830874_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,203) = (/ 0.999974231946592_dbl_kind, 0.999286200999753_dbl_kind, 0.99460756792323_dbl_kind, 0.974332822942743_dbl_kind, 0.768266608439772_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,204) = (/ 0.999974122606803_dbl_kind, 0.99928329287791_dbl_kind, 0.99458501803093_dbl_kind, 0.974242278293624_dbl_kind, 0.767693151785713_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,205) = (/ 0.999974013262117_dbl_kind, 0.999280385966203_dbl_kind, 0.994562473644468_dbl_kind, 0.974151750570835_dbl_kind, 0.767121409601591_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,206) = (/ 0.999973903912801_dbl_kind, 0.999277480257179_dbl_kind, 0.994539934786189_dbl_kind, 0.974061239674505_dbl_kind, 0.766551373692173_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,207) = (/ 0.999973794559118_dbl_kind, 0.999274575743337_dbl_kind, 0.99451740147692_dbl_kind, 0.97397074550569_dbl_kind, 0.765983035933223_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,208) = (/ 0.999973685201327_dbl_kind, 0.999271672417135_dbl_kind, 0.994494873735989_dbl_kind, 0.973880267966415_dbl_kind, 0.76541638827062_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,209) = (/ 0.999973575839682_dbl_kind, 0.99926877027099_dbl_kind, 0.994472351581235_dbl_kind, 0.973789806959708_dbl_kind, 0.764851422719525_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,210) = (/ 0.999973466474429_dbl_kind, 0.999265869297286_dbl_kind, 0.994449835029023_dbl_kind, 0.973699362389637_dbl_kind, 0.76428813136353_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,211) = (/ 0.999973357105814_dbl_kind, 0.999262969488384_dbl_kind, 0.994427324094264_dbl_kind, 0.973608934161341_dbl_kind, 0.763726506353836_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,212) = (/ 0.999973247734073_dbl_kind, 0.999260070836615_dbl_kind, 0.994404818790429_dbl_kind, 0.973518522181074_dbl_kind, 0.76316653990843_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,213) = (/ 0.999973138359441_dbl_kind, 0.999257173334295_dbl_kind, 0.99438231912957_dbl_kind, 0.973428126356231_dbl_kind, 0.762608224311284_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,214) = (/ 0.999973028982144_dbl_kind, 0.999254276973726_dbl_kind, 0.994359825122334_dbl_kind, 0.973337746595381_dbl_kind, 0.762051551911567_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,215) = (/ 0.999972919602407_dbl_kind, 0.999251381747198_dbl_kind, 0.994337336777992_dbl_kind, 0.973247382808302_dbl_kind, 0.761496515122857_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,216) = (/ 0.999972810220448_dbl_kind, 0.999248487646995_dbl_kind, 0.994314854104451_dbl_kind, 0.973157034906006_dbl_kind, 0.760943106422373_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,217) = (/ 0.99997270083648_dbl_kind, 0.9992455946654_dbl_kind, 0.994292377108279_dbl_kind, 0.973066702800772_dbl_kind, 0.760391318350221_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,218) = (/ 0.999972591450709_dbl_kind, 0.999242702794697_dbl_kind, 0.994269905794728_dbl_kind, 0.972976386406175_dbl_kind, 0.759841143508639_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,219) = (/ 0.99997248206334_dbl_kind, 0.999239812027178_dbl_kind, 0.994247440167754_dbl_kind, 0.972886085637106_dbl_kind, 0.75929257456126_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,220) = (/ 0.999972372674571_dbl_kind, 0.999236922355139_dbl_kind, 0.994224980230044_dbl_kind, 0.972795800409804_dbl_kind, 0.758745604232396_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,221) = (/ 0.999972263284593_dbl_kind, 0.999234033770892_dbl_kind, 0.994202525983037_dbl_kind, 0.972705530641877_dbl_kind, 0.758200225306301_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,222) = (/ 0.999972153893596_dbl_kind, 0.999231146266766_dbl_kind, 0.994180077426948_dbl_kind, 0.972615276252323_dbl_kind, 0.757656430626487_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,223) = (/ 0.999972044501762_dbl_kind, 0.999228259835104_dbl_kind, 0.994157634560794_dbl_kind, 0.972525037161557_dbl_kind, 0.757114213095009_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,224) = (/ 0.999971935109269_dbl_kind, 0.999225374468276_dbl_kind, 0.994135197382418_dbl_kind, 0.972434813291424_dbl_kind, 0.756573565671794_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,225) = (/ 0.999971825716289_dbl_kind, 0.999222490158672_dbl_kind, 0.994112765888517_dbl_kind, 0.972344604565223_dbl_kind, 0.756034481373946_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,226) = (/ 0.999971716322992_dbl_kind, 0.999219606898713_dbl_kind, 0.994090340074664_dbl_kind, 0.972254410907722_dbl_kind, 0.755496953275093_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,227) = (/ 0.999971606929541_dbl_kind, 0.999216724680849_dbl_kind, 0.994067919935335_dbl_kind, 0.972164232245174_dbl_kind, 0.754960974504724_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,228) = (/ 0.999971497536094_dbl_kind, 0.999213843497562_dbl_kind, 0.994045505463937_dbl_kind, 0.972074068505333_dbl_kind, 0.754426538247536_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,229) = (/ 0.999971388142806_dbl_kind, 0.999210963341367_dbl_kind, 0.994023096652831_dbl_kind, 0.971983919617467_dbl_kind, 0.753893637742801_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,230) = (/ 0.999971278749825_dbl_kind, 0.99920808420482_dbl_kind, 0.994000693493361_dbl_kind, 0.971893785512366_dbl_kind, 0.753362266283723_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,231) = (/ 0.999971169357295_dbl_kind, 0.999205206080514_dbl_kind, 0.993978295975882_dbl_kind, 0.971803666122363_dbl_kind, 0.752832417216826_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,232) = (/ 0.999971059965358_dbl_kind, 0.999202328961084_dbl_kind, 0.993955904089781_dbl_kind, 0.971713561381331_dbl_kind, 0.752304083941336_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,233) = (/ 0.999970950574148_dbl_kind, 0.999199452839207_dbl_kind, 0.993933517823508_dbl_kind, 0.9716234712247_dbl_kind, 0.751777259908575_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,234) = (/ 0.999970841183796_dbl_kind, 0.999196577707605_dbl_kind, 0.993911137164602_dbl_kind, 0.971533395589462_dbl_kind, 0.751251938621359_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,235) = (/ 0.99997073179443_dbl_kind, 0.999193703559048_dbl_kind, 0.993888762099717_dbl_kind, 0.971443334414173_dbl_kind, 0.750728113633418_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,236) = (/ 0.99997062240617_dbl_kind, 0.999190830386352_dbl_kind, 0.993866392614647_dbl_kind, 0.971353287638964_dbl_kind, 0.750205778548806_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,237) = (/ 0.999970513019135_dbl_kind, 0.999187958182386_dbl_kind, 0.993844028694358_dbl_kind, 0.971263255205536_dbl_kind, 0.749684927021327_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,238) = (/ 0.999970403633438_dbl_kind, 0.999185086940066_dbl_kind, 0.993821670323006_dbl_kind, 0.971173237057172_dbl_kind, 0.749165552753981_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,239) = (/ 0.99997029424919_dbl_kind, 0.999182216652363_dbl_kind, 0.993799317483969_dbl_kind, 0.97108323313873_dbl_kind, 0.748647649498387_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,240) = (/ 0.999970184866494_dbl_kind, 0.999179347312299_dbl_kind, 0.993776970159873_dbl_kind, 0.970993243396647_dbl_kind, 0.748131211054249_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,241) = (/ 0.999970075485454_dbl_kind, 0.999176478912954_dbl_kind, 0.993754628332615_dbl_kind, 0.970903267778934_dbl_kind, 0.747616231268801_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,242) = (/ 0.999969966106167_dbl_kind, 0.99917361144746_dbl_kind, 0.993732291983391_dbl_kind, 0.970813306235183_dbl_kind, 0.747102704036279_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,243) = (/ 0.999969856728725_dbl_kind, 0.999170744909008_dbl_kind, 0.993709961092718_dbl_kind, 0.970723358716549_dbl_kind, 0.746590623297386_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,244) = (/ 0.999969747353219_dbl_kind, 0.999167879290844_dbl_kind, 0.993687635640467_dbl_kind, 0.97063342517576_dbl_kind, 0.746079983038774_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,245) = (/ 0.999969637979734_dbl_kind, 0.999165014586275_dbl_kind, 0.993665315605877_dbl_kind, 0.970543505567103_dbl_kind, 0.745570777292526_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,246) = (/ 0.999969528608355_dbl_kind, 0.999162150788665_dbl_kind, 0.993643000967589_dbl_kind, 0.970453599846421_dbl_kind, 0.745063000135648_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,247) = (/ 0.999969419239159_dbl_kind, 0.999159287891435_dbl_kind, 0.993620691703666_dbl_kind, 0.970363707971106_dbl_kind, 0.744556645689571_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,248) = (/ 0.999969309872221_dbl_kind, 0.99915642588807_dbl_kind, 0.993598387791615_dbl_kind, 0.970273829900089_dbl_kind, 0.744051708119653_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,249) = (/ 0.999969200507614_dbl_kind, 0.999153564772115_dbl_kind, 0.99357608920842_dbl_kind, 0.970183965593835_dbl_kind, 0.743548181634687_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,250) = (/ 0.999969091145406_dbl_kind, 0.999150704537171_dbl_kind, 0.993553795930553_dbl_kind, 0.970094115014328_dbl_kind, 0.743046060486428_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,251) = (/ 0.999968981785664_dbl_kind, 0.999147845176906_dbl_kind, 0.993531507934006_dbl_kind, 0.970004278125065_dbl_kind, 0.742545338969108_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,252) = (/ 0.999968872428448_dbl_kind, 0.999144986685047_dbl_kind, 0.99350922519431_dbl_kind, 0.969914454891044_dbl_kind, 0.742046011418976_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,253) = (/ 0.999968763073819_dbl_kind, 0.99914212905538_dbl_kind, 0.99348694768656_dbl_kind, 0.96982464527875_dbl_kind, 0.74154807221382_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,254) = (/ 0.999968653721832_dbl_kind, 0.999139272281758_dbl_kind, 0.993464675385433_dbl_kind, 0.969734849256143_dbl_kind, 0.741051515772528_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,255) = (/ 0.999968544372542_dbl_kind, 0.99913641635809_dbl_kind, 0.993442408265215_dbl_kind, 0.969645066792647_dbl_kind, 0.740556336554622_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,256) = (/ 0.999968435025998_dbl_kind, 0.999133561278352_dbl_kind, 0.993420146299816_dbl_kind, 0.969555297859136_dbl_kind, 0.74006252905982_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,257) = (/ 0.999968325682249_dbl_kind, 0.99913070703658_dbl_kind, 0.993397889462797_dbl_kind, 0.969465542427914_dbl_kind, 0.739570087827593_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,258) = (/ 0.999968216341338_dbl_kind, 0.99912785362687_dbl_kind, 0.993375637727388_dbl_kind, 0.969375800472706_dbl_kind, 0.739079007436728_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,259) = (/ 0.999968107003309_dbl_kind, 0.999125001043383_dbl_kind, 0.993353391066506_dbl_kind, 0.969286071968641_dbl_kind, 0.738589282504911_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,260) = (/ 0.999967997668201_dbl_kind, 0.99912214928034_dbl_kind, 0.993331149452778_dbl_kind, 0.969196356892233_dbl_kind, 0.73810090768829_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,261) = (/ 0.999967888336053_dbl_kind, 0.999119298332024_dbl_kind, 0.993308912858558_dbl_kind, 0.969106655221368_dbl_kind, 0.737613877681063_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,262) = (/ 0.999967779006899_dbl_kind, 0.999116448192779_dbl_kind, 0.993286681255949_dbl_kind, 0.969016966935284_dbl_kind, 0.737128187215068_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,263) = (/ 0.999967669680772_dbl_kind, 0.99911359885701_dbl_kind, 0.993264454616817_dbl_kind, 0.968927292014558_dbl_kind, 0.736643831059374_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,264) = (/ 0.999967560357703_dbl_kind, 0.999110750319183_dbl_kind, 0.993242232912814_dbl_kind, 0.968837630441079_dbl_kind, 0.736160804019883_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,265) = (/ 0.99996745103772_dbl_kind, 0.999107902573826_dbl_kind, 0.99322001611539_dbl_kind, 0.968747982198039_dbl_kind, 0.735679100938921_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,266) = (/ 0.99996734172085_dbl_kind, 0.999105055615524_dbl_kind, 0.993197804195817_dbl_kind, 0.968658347269912_dbl_kind, 0.735198716694859_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,267) = (/ 0.999967232407117_dbl_kind, 0.999102209438926_dbl_kind, 0.993175597125201_dbl_kind, 0.968568725642431_dbl_kind, 0.734719646201725_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,268) = (/ 0.999967123096545_dbl_kind, 0.999099364038736_dbl_kind, 0.993153394874497_dbl_kind, 0.968479117302576_dbl_kind, 0.734241884408816_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,269) = (/ 0.999967013789153_dbl_kind, 0.999096519409721_dbl_kind, 0.993131197414535_dbl_kind, 0.968389522238543_dbl_kind, 0.733765426300326_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,270) = (/ 0.999966904484962_dbl_kind, 0.999093675546706_dbl_kind, 0.993109004716021_dbl_kind, 0.968299940439736_dbl_kind, 0.73329026689497_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,271) = (/ 0.999966795183987_dbl_kind, 0.99909083244457_dbl_kind, 0.993086816749564_dbl_kind, 0.968210371896743_dbl_kind, 0.732816401245625_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,272) = (/ 0.999966685886246_dbl_kind, 0.999087990098257_dbl_kind, 0.993064633485686_dbl_kind, 0.968120816601312_dbl_kind, 0.732343824438956_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,273) = (/ 0.999966576591752_dbl_kind, 0.999085148502762_dbl_kind, 0.993042454894836_dbl_kind, 0.968031274546336_dbl_kind, 0.731872531595063_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,274) = (/ 0.999966467300519_dbl_kind, 0.999082307653139_dbl_kind, 0.993020280947407_dbl_kind, 0.967941745725826_dbl_kind, 0.731402517867131_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,275) = (/ 0.999966358012557_dbl_kind, 0.999079467544499_dbl_kind, 0.992998111613745_dbl_kind, 0.9678522301349_dbl_kind, 0.730933778441072_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,276) = (/ 0.999966248727877_dbl_kind, 0.999076628172008_dbl_kind, 0.992975946864168_dbl_kind, 0.967762727769752_dbl_kind, 0.730466308535194_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,277) = (/ 0.999966139446488_dbl_kind, 0.999073789530887_dbl_kind, 0.992953786668975_dbl_kind, 0.967673238627639_dbl_kind, 0.730000103399841_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,278) = (/ 0.999966030168397_dbl_kind, 0.999070951616412_dbl_kind, 0.992931630998457_dbl_kind, 0.967583762706854_dbl_kind, 0.729535158317075_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,279) = (/ 0.999965920893611_dbl_kind, 0.999068114423912_dbl_kind, 0.992909479822915_dbl_kind, 0.967494300006708_dbl_kind, 0.729071468600336_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,280) = (/ 0.999965811622135_dbl_kind, 0.999065277948771_dbl_kind, 0.992887333112667_dbl_kind, 0.967404850527509_dbl_kind, 0.728609029594107_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,281) = (/ 0.999965702353973_dbl_kind, 0.999062442186425_dbl_kind, 0.992865190838059_dbl_kind, 0.967315414270543_dbl_kind, 0.72814783667361_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,282) = (/ 0.99996559308913_dbl_kind, 0.999059607132363_dbl_kind, 0.99284305296948_dbl_kind, 0.967225991238045_dbl_kind, 0.727687885244465_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,283) = (/ 0.999965483827607_dbl_kind, 0.999056772782123_dbl_kind, 0.992820919477367_dbl_kind, 0.967136581433188_dbl_kind, 0.727229170742386_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,284) = (/ 0.999965374569406_dbl_kind, 0.999053939131298_dbl_kind, 0.992798790332222_dbl_kind, 0.967047184860056_dbl_kind, 0.726771688632869_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,285) = (/ 0.999965265314529_dbl_kind, 0.99905110617553_dbl_kind, 0.992776665504616_dbl_kind, 0.966957801523621_dbl_kind, 0.726315434410877_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,286) = (/ 0.999965156062975_dbl_kind, 0.99904827391051_dbl_kind, 0.992754544965203_dbl_kind, 0.966868431429732_dbl_kind, 0.725860403600549_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,287) = (/ 0.999965046814744_dbl_kind, 0.999045442331979_dbl_kind, 0.992732428684724_dbl_kind, 0.966779074585082_dbl_kind, 0.725406591754877_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,288) = (/ 0.999964937569835_dbl_kind, 0.999042611435725_dbl_kind, 0.992710316634023_dbl_kind, 0.966689730997194_dbl_kind, 0.724953994455439_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,289) = (/ 0.999964828328246_dbl_kind, 0.999039781217588_dbl_kind, 0.992688208784049_dbl_kind, 0.9666004006744_dbl_kind, 0.72450260731208_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,290) = (/ 0.999964719089975_dbl_kind, 0.999036951673451_dbl_kind, 0.992666105105867_dbl_kind, 0.966511083625819_dbl_kind, 0.724052425962642_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,291) = (/ 0.999964609855019_dbl_kind, 0.999034122799246_dbl_kind, 0.992644005570669_dbl_kind, 0.966421779861337_dbl_kind, 0.723603446072664_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,292) = (/ 0.999964500623374_dbl_kind, 0.999031294590952_dbl_kind, 0.992621910149773_dbl_kind, 0.966332489391586_dbl_kind, 0.72315566333511_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,293) = (/ 0.99996439139504_dbl_kind, 0.999028467044591_dbl_kind, 0.992599818814641_dbl_kind, 0.966243212227926_dbl_kind, 0.72270907347008_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,294) = (/ 0.999964282170009_dbl_kind, 0.999025640156233_dbl_kind, 0.992577731536878_dbl_kind, 0.96615394838242_dbl_kind, 0.722263672224545_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,295) = (/ 0.999964172948279_dbl_kind, 0.999022813921991_dbl_kind, 0.992555648288242_dbl_kind, 0.966064697867823_dbl_kind, 0.721819455372074_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,296) = (/ 0.999964063729844_dbl_kind, 0.999019988338021_dbl_kind, 0.992533569040652_dbl_kind, 0.965975460697554_dbl_kind, 0.721376418712556_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,297) = (/ 0.999963954514701_dbl_kind, 0.999017163400523_dbl_kind, 0.992511493766189_dbl_kind, 0.965886236885677_dbl_kind, 0.720934558071951_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,298) = (/ 0.999963845302845_dbl_kind, 0.99901433910574_dbl_kind, 0.992489422437108_dbl_kind, 0.96579702644689_dbl_kind, 0.720493869302008_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,299) = (/ 0.999963736094269_dbl_kind, 0.999011515449956_dbl_kind, 0.992467355025838_dbl_kind, 0.965707829396495_dbl_kind, 0.720054348280025_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,300) = (/ 0.99996362688897_dbl_kind, 0.999008692429498_dbl_kind, 0.992445291504992_dbl_kind, 0.965618645750386_dbl_kind, 0.719615990908586_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,301) = (/ 0.999963517686942_dbl_kind, 0.999005870040732_dbl_kind, 0.992423231847371_dbl_kind, 0.965529475525029_dbl_kind, 0.719178793115302_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,302) = (/ 0.999963408488179_dbl_kind, 0.999003048280066_dbl_kind, 0.992401176025965_dbl_kind, 0.965440318737442_dbl_kind, 0.718742750852572_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,303) = (/ 0.999963299292678_dbl_kind, 0.999000227143946_dbl_kind, 0.992379124013962_dbl_kind, 0.965351175405177_dbl_kind, 0.718307860097333_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,304) = (/ 0.99996319010043_dbl_kind, 0.99899740662886_dbl_kind, 0.992357075784752_dbl_kind, 0.965262045546307_dbl_kind, 0.717874116850821_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,305) = (/ 0.999963080911432_dbl_kind, 0.998994586731331_dbl_kind, 0.992335031311929_dbl_kind, 0.965172929179398_dbl_kind, 0.717441517138319_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,306) = (/ 0.999962971725678_dbl_kind, 0.998991767447924_dbl_kind, 0.992312990569295_dbl_kind, 0.965083826323501_dbl_kind, 0.717010057008937_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,307) = (/ 0.999962862543164_dbl_kind, 0.998988948775238_dbl_kind, 0.992290953530868_dbl_kind, 0.964994736998132_dbl_kind, 0.716579732535358_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,308) = (/ 0.999962753363883_dbl_kind, 0.998986130709912_dbl_kind, 0.992268920170878_dbl_kind, 0.964905661223248_dbl_kind, 0.716150539813624_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,309) = (/ 0.999962644187832_dbl_kind, 0.998983313248619_dbl_kind, 0.992246890463776_dbl_kind, 0.964816599019245_dbl_kind, 0.715722474962901_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,310) = (/ 0.999962535015005_dbl_kind, 0.998980496388071_dbl_kind, 0.992224864384238_dbl_kind, 0.964727550406923_dbl_kind, 0.715295534125251_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,311) = (/ 0.999962425845398_dbl_kind, 0.998977680125012_dbl_kind, 0.992202841907161_dbl_kind, 0.96463851540749_dbl_kind, 0.714869713465411_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,312) = (/ 0.999962316679006_dbl_kind, 0.998974864456225_dbl_kind, 0.992180823007672_dbl_kind, 0.964549494042524_dbl_kind, 0.71444500917057_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,313) = (/ 0.999962207515825_dbl_kind, 0.998972049378525_dbl_kind, 0.992158807661129_dbl_kind, 0.964460486333979_dbl_kind, 0.714021417450157_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,314) = (/ 0.999962098355852_dbl_kind, 0.998969234888762_dbl_kind, 0.992136795843121_dbl_kind, 0.964371492304151_dbl_kind, 0.713598934535621_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,315) = (/ 0.999961989199082_dbl_kind, 0.998966420983819_dbl_kind, 0.992114787529472_dbl_kind, 0.964282511975675_dbl_kind, 0.713177556680224_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,316) = (/ 0.999961880045513_dbl_kind, 0.998963607660612_dbl_kind, 0.992092782696244_dbl_kind, 0.964193545371505_dbl_kind, 0.712757280158825_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,317) = (/ 0.999961770895139_dbl_kind, 0.998960794916093_dbl_kind, 0.992070781319736_dbl_kind, 0.964104592514899_dbl_kind, 0.712338101267675_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,318) = (/ 0.99996166174796_dbl_kind, 0.998957982747241_dbl_kind, 0.992048783376488_dbl_kind, 0.964015653429406_dbl_kind, 0.711920016324215_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,319) = (/ 0.999961552603973_dbl_kind, 0.998955171151071_dbl_kind, 0.99202678884328_dbl_kind, 0.963926728138853_dbl_kind, 0.711503021666876_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,320) = (/ 0.999961443463174_dbl_kind, 0.998952360124628_dbl_kind, 0.992004797697137_dbl_kind, 0.963837816667326_dbl_kind, 0.711087113654864_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,321) = (/ 0.999961334325563_dbl_kind, 0.998949549664989_dbl_kind, 0.991982809915325_dbl_kind, 0.963748919039165_dbl_kind, 0.710672288667985_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,322) = (/ 0.999961225191137_dbl_kind, 0.99894673976926_dbl_kind, 0.991960825475356_dbl_kind, 0.963660035278939_dbl_kind, 0.710258543106431_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,323) = (/ 0.999961116059896_dbl_kind, 0.99894393043458_dbl_kind, 0.991938844354989_dbl_kind, 0.963571165411445_dbl_kind, 0.709845873390601_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,324) = (/ 0.999961006931838_dbl_kind, 0.998941121658116_dbl_kind, 0.991916866532228_dbl_kind, 0.963482309461687_dbl_kind, 0.709434275960904_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,325) = (/ 0.999960897806964_dbl_kind, 0.998938313437063_dbl_kind, 0.991894891985324_dbl_kind, 0.963393467454867_dbl_kind, 0.709023747277573_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,326) = (/ 0.999960788685272_dbl_kind, 0.998935505768648_dbl_kind, 0.991872920692775_dbl_kind, 0.96330463941637_dbl_kind, 0.708614283820477_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,327) = (/ 0.999960679566763_dbl_kind, 0.998932698650126_dbl_kind, 0.991850952633329_dbl_kind, 0.963215825371754_dbl_kind, 0.70820588208894_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,328) = (/ 0.999960570451438_dbl_kind, 0.998929892078781_dbl_kind, 0.991828987785978_dbl_kind, 0.963127025346741_dbl_kind, 0.707798538601555_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,329) = (/ 0.999960461339298_dbl_kind, 0.998927086051923_dbl_kind, 0.991807026129966_dbl_kind, 0.963038239367198_dbl_kind, 0.707392249896021_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,330) = (/ 0.999960352230345_dbl_kind, 0.998924280566893_dbl_kind, 0.991785067644783_dbl_kind, 0.962949467459133_dbl_kind, 0.706987012528944_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,331) = (/ 0.999960243124578_dbl_kind, 0.998921475621058_dbl_kind, 0.991763112310166_dbl_kind, 0.96286070964868_dbl_kind, 0.706582823075677_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,332) = (/ 0.999960134022002_dbl_kind, 0.998918671211811_dbl_kind, 0.991741160106103_dbl_kind, 0.962771965962089_dbl_kind, 0.70617967813014_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,333) = (/ 0.999960024922619_dbl_kind, 0.998915867336572_dbl_kind, 0.991719211012825_dbl_kind, 0.962683236425717_dbl_kind, 0.705777574304657_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,334) = (/ 0.999959915826432_dbl_kind, 0.998913063992791_dbl_kind, 0.991697265010814_dbl_kind, 0.962594521066015_dbl_kind, 0.70537650822978_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,335) = (/ 0.999959806733443_dbl_kind, 0.99891026117794_dbl_kind, 0.991675322080796_dbl_kind, 0.962505819909524_dbl_kind, 0.704976476554127_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,336) = (/ 0.999959697643657_dbl_kind, 0.998907458889519_dbl_kind, 0.991653382203744_dbl_kind, 0.962417132982855_dbl_kind, 0.704577475944218_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,337) = (/ 0.999959588557078_dbl_kind, 0.998904657125055_dbl_kind, 0.991631445360875_dbl_kind, 0.962328460312691_dbl_kind, 0.704179503084307_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,338) = (/ 0.99995947947371_dbl_kind, 0.998901855882098_dbl_kind, 0.991609511533653_dbl_kind, 0.962239801925769_dbl_kind, 0.703782554676226_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,339) = (/ 0.999959370393559_dbl_kind, 0.998899055158225_dbl_kind, 0.991587580703782_dbl_kind, 0.962151157848876_dbl_kind, 0.703386627439228_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,340) = (/ 0.99995926131663_dbl_kind, 0.998896254951036_dbl_kind, 0.991565652853211_dbl_kind, 0.962062528108837_dbl_kind, 0.702991718109829_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,341) = (/ 0.999959152242929_dbl_kind, 0.998893455258158_dbl_kind, 0.991543727964131_dbl_kind, 0.96197391273251_dbl_kind, 0.702597823441648_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,342) = (/ 0.99995904317246_dbl_kind, 0.998890656077241_dbl_kind, 0.991521806018971_dbl_kind, 0.961885311746774_dbl_kind, 0.70220494020526_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,343) = (/ 0.999958934105233_dbl_kind, 0.99888785740596_dbl_kind, 0.991499887000404_dbl_kind, 0.961796725178528_dbl_kind, 0.701813065188042_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,344) = (/ 0.999958825041252_dbl_kind, 0.998885059242014_dbl_kind, 0.991477970891338_dbl_kind, 0.961708153054671_dbl_kind, 0.701422195194027_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,345) = (/ 0.999958715980525_dbl_kind, 0.998882261583125_dbl_kind, 0.991456057674919_dbl_kind, 0.961619595402104_dbl_kind, 0.701032327043751_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,346) = (/ 0.99995860692306_dbl_kind, 0.99887946442704_dbl_kind, 0.991434147334527_dbl_kind, 0.961531052247723_dbl_kind, 0.700643457574112_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,347) = (/ 0.999958497868866_dbl_kind, 0.998876667771528_dbl_kind, 0.991412239853781_dbl_kind, 0.961442523618405_dbl_kind, 0.700255583638222_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,348) = (/ 0.999958388817949_dbl_kind, 0.998873871614382_dbl_kind, 0.991390335216529_dbl_kind, 0.961354009541009_dbl_kind, 0.699868702105266_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,349) = (/ 0.99995827977032_dbl_kind, 0.998871075953418_dbl_kind, 0.991368433406853_dbl_kind, 0.96126551004236_dbl_kind, 0.699482809860364_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,350) = (/ 0.999958170725986_dbl_kind, 0.998868280786474_dbl_kind, 0.991346534409063_dbl_kind, 0.961177025149255_dbl_kind, 0.699097903804431_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,351) = (/ 0.999958061684958_dbl_kind, 0.998865486111411_dbl_kind, 0.991324638207699_dbl_kind, 0.961088554888443_dbl_kind, 0.698713980854031_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,352) = (/ 0.999957952647244_dbl_kind, 0.998862691926114_dbl_kind, 0.99130274478753_dbl_kind, 0.961000099286631_dbl_kind, 0.698331037941254_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,353) = (/ 0.999957843612857_dbl_kind, 0.998859898228489_dbl_kind, 0.991280854133545_dbl_kind, 0.960911658370467_dbl_kind, 0.697949072013576_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,354) = (/ 0.999957734581804_dbl_kind, 0.998857105016461_dbl_kind, 0.991258966230963_dbl_kind, 0.960823232166546_dbl_kind, 0.697568080033724_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,355) = (/ 0.999957625554097_dbl_kind, 0.998854312287984_dbl_kind, 0.991237081065221_dbl_kind, 0.960734820701393_dbl_kind, 0.697188058979547_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,356) = (/ 0.999957516529748_dbl_kind, 0.998851520041028_dbl_kind, 0.991215198621976_dbl_kind, 0.960646424001466_dbl_kind, 0.696809005843885_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,357) = (/ 0.999957407508767_dbl_kind, 0.998848728273586_dbl_kind, 0.991193318887107_dbl_kind, 0.96055804209315_dbl_kind, 0.696430917634451_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,358) = (/ 0.999957298491166_dbl_kind, 0.998845936983672_dbl_kind, 0.991171441846707_dbl_kind, 0.960469675002745_dbl_kind, 0.696053791373685_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,359) = (/ 0.999957189476957_dbl_kind, 0.998843146169325_dbl_kind, 0.991149567487084_dbl_kind, 0.960381322756471_dbl_kind, 0.695677624098646_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,360) = (/ 0.999957080466151_dbl_kind, 0.998840355828601_dbl_kind, 0.991127695794761_dbl_kind, 0.960292985380457_dbl_kind, 0.695302412860879_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,361) = (/ 0.999956971458761_dbl_kind, 0.998837565959578_dbl_kind, 0.99110582675647_dbl_kind, 0.960204662900739_dbl_kind, 0.694928154726301_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,362) = (/ 0.9999568624548_dbl_kind, 0.998834776560356_dbl_kind, 0.991083960359154_dbl_kind, 0.960116355343253_dbl_kind, 0.694554846775071_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,363) = (/ 0.999956753454281_dbl_kind, 0.998831987629054_dbl_kind, 0.991062096589962_dbl_kind, 0.960028062733836_dbl_kind, 0.69418248610147_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,364) = (/ 0.999956644457217_dbl_kind, 0.998829199163814_dbl_kind, 0.99104023543625_dbl_kind, 0.959939785098218_dbl_kind, 0.6938110698138_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,365) = (/ 0.99995653546362_dbl_kind, 0.998826411162798_dbl_kind, 0.991018376885577_dbl_kind, 0.959851522462021_dbl_kind, 0.693440595034244_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,366) = (/ 0.999956426473506_dbl_kind, 0.998823623624187_dbl_kind, 0.990996520925703_dbl_kind, 0.959763274850751_dbl_kind, 0.693071058898769_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,367) = (/ 0.999956317486886_dbl_kind, 0.998820836546182_dbl_kind, 0.990974667544587_dbl_kind, 0.959675042289801_dbl_kind, 0.692702458556997_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,368) = (/ 0.999956208503776_dbl_kind, 0.998818049927008_dbl_kind, 0.990952816730389_dbl_kind, 0.959586824804442_dbl_kind, 0.692334791172108_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,369) = (/ 0.999956099524189_dbl_kind, 0.998815263764905_dbl_kind, 0.99093096847146_dbl_kind, 0.959498622419823_dbl_kind, 0.691968053920715_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,370) = (/ 0.999955990548139_dbl_kind, 0.998812478058136_dbl_kind, 0.990909122756348_dbl_kind, 0.959410435160967_dbl_kind, 0.691602243992762_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,371) = (/ 0.999955881575642_dbl_kind, 0.998809692804984_dbl_kind, 0.99088727957379_dbl_kind, 0.959322263052767_dbl_kind, 0.691237358591414_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,372) = (/ 0.999955772606711_dbl_kind, 0.99880690800375_dbl_kind, 0.990865438912715_dbl_kind, 0.959234106119989_dbl_kind, 0.690873394932944_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,373) = (/ 0.999955663641362_dbl_kind, 0.998804123652756_dbl_kind, 0.990843600762238_dbl_kind, 0.95914596438726_dbl_kind, 0.690510350246634_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,374) = (/ 0.999955554679609_dbl_kind, 0.998801339750342_dbl_kind, 0.99082176511166_dbl_kind, 0.959057837879074_dbl_kind, 0.690148221774662_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,375) = (/ 0.999955445721468_dbl_kind, 0.998798556294871_dbl_kind, 0.990799931950463_dbl_kind, 0.958969726619784_dbl_kind, 0.689787006772011_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,376) = (/ 0.999955336766953_dbl_kind, 0.99879577328472_dbl_kind, 0.990778101268317_dbl_kind, 0.958881630633606_dbl_kind, 0.689426702506348_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,377) = (/ 0.99995522781608_dbl_kind, 0.998792990718291_dbl_kind, 0.990756273055062_dbl_kind, 0.958793549944609_dbl_kind, 0.689067306257936_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,378) = (/ 0.999955118868865_dbl_kind, 0.998790208594001_dbl_kind, 0.990734447300722_dbl_kind, 0.958705484576719_dbl_kind, 0.688708815319532_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,379) = (/ 0.999955009925322_dbl_kind, 0.998787426910288_dbl_kind, 0.990712623995495_dbl_kind, 0.958617434553719_dbl_kind, 0.688351226996278_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,380) = (/ 0.999954900985468_dbl_kind, 0.998784645665607_dbl_kind, 0.990690803129748_dbl_kind, 0.958529399899237_dbl_kind, 0.68799453860562_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,381) = (/ 0.999954792049317_dbl_kind, 0.998781864858435_dbl_kind, 0.990668984694025_dbl_kind, 0.958441380636755_dbl_kind, 0.687638747477189_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,382) = (/ 0.999954683116886_dbl_kind, 0.998779084487266_dbl_kind, 0.990647168679035_dbl_kind, 0.958353376789604_dbl_kind, 0.687283850952725_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,383) = (/ 0.999954574188191_dbl_kind, 0.998776304550612_dbl_kind, 0.990625355075655_dbl_kind, 0.958265388380963_dbl_kind, 0.686929846385967_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,384) = (/ 0.999954465263248_dbl_kind, 0.998773525047006_dbl_kind, 0.990603543874926_dbl_kind, 0.958177415433853_dbl_kind, 0.686576731142572_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,385) = (/ 0.99995435634207_dbl_kind, 0.998770745974997_dbl_kind, 0.990581735068054_dbl_kind, 0.958089457971145_dbl_kind, 0.68622450260001_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,386) = (/ 0.999954247424676_dbl_kind, 0.998767967333153_dbl_kind, 0.990559928646403_dbl_kind, 0.958001516015551_dbl_kind, 0.685873158147483_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,387) = (/ 0.999954138511081_dbl_kind, 0.998765189120062_dbl_kind, 0.990538124601499_dbl_kind, 0.957913589589627_dbl_kind, 0.685522695185821_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,388) = (/ 0.999954029601301_dbl_kind, 0.998762411334329_dbl_kind, 0.99051632292502_dbl_kind, 0.957825678715771_dbl_kind, 0.685173111127409_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,389) = (/ 0.999953920695353_dbl_kind, 0.998759633974578_dbl_kind, 0.990494523608803_dbl_kind, 0.957737783416221_dbl_kind, 0.684824403396078_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,390) = (/ 0.99995381179325_dbl_kind, 0.99875685703945_dbl_kind, 0.990472726644836_dbl_kind, 0.957649903713056_dbl_kind, 0.68447656942704_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,391) = (/ 0.99995370289501_dbl_kind, 0.998754080527605_dbl_kind, 0.990450932025258_dbl_kind, 0.957562039628197_dbl_kind, 0.684129606666778_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,392) = (/ 0.999953594000649_dbl_kind, 0.99875130443772_dbl_kind, 0.990429139742355_dbl_kind, 0.957474191183401_dbl_kind, 0.683783512572983_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,393) = (/ 0.999953485110182_dbl_kind, 0.998748528768491_dbl_kind, 0.990407349788562_dbl_kind, 0.957386358400268_dbl_kind, 0.683438284614445_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,394) = (/ 0.999953376223625_dbl_kind, 0.998745753518633_dbl_kind, 0.990385562156456_dbl_kind, 0.957298541300232_dbl_kind, 0.683093920270988_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,395) = (/ 0.999953267340995_dbl_kind, 0.998742978686875_dbl_kind, 0.990363776838761_dbl_kind, 0.957210739904569_dbl_kind, 0.682750417033378_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,396) = (/ 0.999953158462306_dbl_kind, 0.998740204271967_dbl_kind, 0.990341993828336_dbl_kind, 0.957122954234391_dbl_kind, 0.68240777240324_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,397) = (/ 0.999953049587575_dbl_kind, 0.998737430272674_dbl_kind, 0.990320213118184_dbl_kind, 0.957035184310648_dbl_kind, 0.682065983892986_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,398) = (/ 0.999952940716817_dbl_kind, 0.998734656687783_dbl_kind, 0.990298434701441_dbl_kind, 0.95694743015413_dbl_kind, 0.681725049025718_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,399) = (/ 0.999952831850047_dbl_kind, 0.998731883516093_dbl_kind, 0.99027665857138_dbl_kind, 0.95685969178546_dbl_kind, 0.68138496533516_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,400) = (/ 0.999952722987282_dbl_kind, 0.998729110756423_dbl_kind, 0.990254884721407_dbl_kind, 0.956771969225106_dbl_kind, 0.681045730365585_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,401) = (/ 0.999952614128535_dbl_kind, 0.998726338407608_dbl_kind, 0.990233113145058_dbl_kind, 0.956684262493368_dbl_kind, 0.680707341671718_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,402) = (/ 0.999952505273824_dbl_kind, 0.998723566468505_dbl_kind, 0.990211343835999_dbl_kind, 0.956596571610384_dbl_kind, 0.680369796818674_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,403) = (/ 0.999952396423162_dbl_kind, 0.99872079493798_dbl_kind, 0.990189576788022_dbl_kind, 0.956508896596135_dbl_kind, 0.680033093381879_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,404) = (/ 0.999952287576566_dbl_kind, 0.998718023814924_dbl_kind, 0.990167811995047_dbl_kind, 0.956421237470436_dbl_kind, 0.679697228946989_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,405) = (/ 0.999952178734049_dbl_kind, 0.998715253098238_dbl_kind, 0.990146049451115_dbl_kind, 0.956333594252944_dbl_kind, 0.679362201109824_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,406) = (/ 0.999952069895627_dbl_kind, 0.998712482786846_dbl_kind, 0.99012428915039_dbl_kind, 0.956245966963152_dbl_kind, 0.679028007476284_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,407) = (/ 0.999951961061315_dbl_kind, 0.998709712879685_dbl_kind, 0.990102531087156_dbl_kind, 0.956158355620397_dbl_kind, 0.67869464566228_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,408) = (/ 0.999951852231127_dbl_kind, 0.99870694337571_dbl_kind, 0.990080775255815_dbl_kind, 0.956070760243853_dbl_kind, 0.67836211329367_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,409) = (/ 0.999951743405077_dbl_kind, 0.998704174273892_dbl_kind, 0.990059021650885_dbl_kind, 0.955983180852535_dbl_kind, 0.678030408006171_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,410) = (/ 0.99995163458318_dbl_kind, 0.998701405573218_dbl_kind, 0.990037270266998_dbl_kind, 0.9558956174653_dbl_kind, 0.677699527445304_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,411) = (/ 0.99995152576545_dbl_kind, 0.998698637272694_dbl_kind, 0.990015521098901_dbl_kind, 0.955808070100847_dbl_kind, 0.677369469266312_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,412) = (/ 0.9999514169519_dbl_kind, 0.99869586937134_dbl_kind, 0.989993774141451_dbl_kind, 0.955720538777718_dbl_kind, 0.677040231134098_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,413) = (/ 0.999951308142545_dbl_kind, 0.998693101868193_dbl_kind, 0.989972029389613_dbl_kind, 0.955633023514297_dbl_kind, 0.676711810723154_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,414) = (/ 0.999951199337399_dbl_kind, 0.998690334762307_dbl_kind, 0.989950286838461_dbl_kind, 0.955545524328813_dbl_kind, 0.676384205717488_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,415) = (/ 0.999951090536475_dbl_kind, 0.998687568052751_dbl_kind, 0.989928546483174_dbl_kind, 0.955458041239339_dbl_kind, 0.676057413810567_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,416) = (/ 0.999950981739785_dbl_kind, 0.99868480173861_dbl_kind, 0.989906808319037_dbl_kind, 0.955370574263794_dbl_kind, 0.675731432705244_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,417) = (/ 0.999950872947344_dbl_kind, 0.998682035818987_dbl_kind, 0.989885072341434_dbl_kind, 0.955283123419942_dbl_kind, 0.675406260113689_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,418) = (/ 0.999950764159164_dbl_kind, 0.998679270292996_dbl_kind, 0.989863338545855_dbl_kind, 0.955195688725399_dbl_kind, 0.67508189375733_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,419) = (/ 0.999950655375259_dbl_kind, 0.998676505159773_dbl_kind, 0.989841606927885_dbl_kind, 0.955108270197621_dbl_kind, 0.674758331366788_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,420) = (/ 0.999950546595639_dbl_kind, 0.998673740418466_dbl_kind, 0.989819877483208_dbl_kind, 0.955020867853922_dbl_kind, 0.674435570681807_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,421) = (/ 0.999950437820318_dbl_kind, 0.998670976068239_dbl_kind, 0.989798150207605_dbl_kind, 0.954933481711459_dbl_kind, 0.674113609451199_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,422) = (/ 0.999950329049308_dbl_kind, 0.998668212108272_dbl_kind, 0.98977642509695_dbl_kind, 0.954846111787244_dbl_kind, 0.673792445432773_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,423) = (/ 0.999950220282622_dbl_kind, 0.99866544853776_dbl_kind, 0.989754702147211_dbl_kind, 0.954758758098138_dbl_kind, 0.673472076393282_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,424) = (/ 0.99995011152027_dbl_kind, 0.998662685355914_dbl_kind, 0.989732981354446_dbl_kind, 0.95467142066086_dbl_kind, 0.673152500108355_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,425) = (/ 0.999950002762265_dbl_kind, 0.998659922561959_dbl_kind, 0.989711262714805_dbl_kind, 0.954584099491977_dbl_kind, 0.672833714362434_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,426) = (/ 0.999949894008617_dbl_kind, 0.998657160155138_dbl_kind, 0.989689546224523_dbl_kind, 0.954496794607914_dbl_kind, 0.672515716948725_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,427) = (/ 0.999949785259337_dbl_kind, 0.998654398134705_dbl_kind, 0.989667831879926_dbl_kind, 0.954409506024955_dbl_kind, 0.672198505669124_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,428) = (/ 0.999949676514438_dbl_kind, 0.998651636499931_dbl_kind, 0.989646119677421_dbl_kind, 0.954322233759235_dbl_kind, 0.671882078334173_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,429) = (/ 0.999949567773928_dbl_kind, 0.998648875250103_dbl_kind, 0.989624409613502_dbl_kind, 0.954234977826753_dbl_kind, 0.671566432762991_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,430) = (/ 0.999949459037819_dbl_kind, 0.99864611438452_dbl_kind, 0.989602701684744_dbl_kind, 0.954147738243365_dbl_kind, 0.671251566783216_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,431) = (/ 0.999949350306121_dbl_kind, 0.998643353902499_dbl_kind, 0.989580995887802_dbl_kind, 0.954060515024789_dbl_kind, 0.670937478230957_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,432) = (/ 0.999949241578843_dbl_kind, 0.998640593803368_dbl_kind, 0.989559292219413_dbl_kind, 0.953973308186604_dbl_kind, 0.670624164950727_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,433) = (/ 0.999949132855996_dbl_kind, 0.998637834086472_dbl_kind, 0.989537590676389_dbl_kind, 0.953886117744251_dbl_kind, 0.670311624795394_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,434) = (/ 0.999949024137589_dbl_kind, 0.99863507475117_dbl_kind, 0.989515891255622_dbl_kind, 0.953798943713038_dbl_kind, 0.669999855626119_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,435) = (/ 0.999948915423631_dbl_kind, 0.998632315796835_dbl_kind, 0.989494193954076_dbl_kind, 0.953711786108137_dbl_kind, 0.669688855312311_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,436) = (/ 0.999948806714131_dbl_kind, 0.998629557222853_dbl_kind, 0.989472498768791_dbl_kind, 0.953624644944587_dbl_kind, 0.669378621731561_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,437) = (/ 0.999948698009097_dbl_kind, 0.998626799028625_dbl_kind, 0.989450805696881_dbl_kind, 0.953537520237294_dbl_kind, 0.669069152769597_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,438) = (/ 0.99994858930854_dbl_kind, 0.998624041213567_dbl_kind, 0.989429114735527_dbl_kind, 0.953450412001034_dbl_kind, 0.668760446320226_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,439) = (/ 0.999948480612466_dbl_kind, 0.998621283777107_dbl_kind, 0.989407425881983_dbl_kind, 0.953363320250454_dbl_kind, 0.66845250028528_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,440) = (/ 0.999948371920883_dbl_kind, 0.998618526718688_dbl_kind, 0.989385739133574_dbl_kind, 0.953276245000072_dbl_kind, 0.668145312574569_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,441) = (/ 0.999948263233799_dbl_kind, 0.998615770037767_dbl_kind, 0.989364054487685_dbl_kind, 0.953189186264278_dbl_kind, 0.667838881105824_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,442) = (/ 0.999948154551223_dbl_kind, 0.998613013733811_dbl_kind, 0.989342371941774_dbl_kind, 0.953102144057338_dbl_kind, 0.667533203804648_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,443) = (/ 0.999948045873161_dbl_kind, 0.998610257806306_dbl_kind, 0.989320691493364_dbl_kind, 0.953015118393393_dbl_kind, 0.667228278604466_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,444) = (/ 0.99994793719962_dbl_kind, 0.998607502254747_dbl_kind, 0.989299013140035_dbl_kind, 0.952928109286459_dbl_kind, 0.666924103446468_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,445) = (/ 0.999947828530607_dbl_kind, 0.998604747078645_dbl_kind, 0.989277336879437_dbl_kind, 0.952841116750431_dbl_kind, 0.666620676279574_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,446) = (/ 0.999947719866127_dbl_kind, 0.99860199227752_dbl_kind, 0.989255662709277_dbl_kind, 0.952754140799084_dbl_kind, 0.666317995060363_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,447) = (/ 0.999947611206189_dbl_kind, 0.998599237850912_dbl_kind, 0.989233990627324_dbl_kind, 0.952667181446074_dbl_kind, 0.666016057753048_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,448) = (/ 0.999947502550798_dbl_kind, 0.998596483798366_dbl_kind, 0.989212320631405_dbl_kind, 0.952580238704936_dbl_kind, 0.665714862329406_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,449) = (/ 0.999947393899957_dbl_kind, 0.998593730119444_dbl_kind, 0.989190652719404_dbl_kind, 0.952493312589091_dbl_kind, 0.665414406768741_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,450) = (/ 0.999947285253675_dbl_kind, 0.998590976813721_dbl_kind, 0.989168986889264_dbl_kind, 0.952406403111842_dbl_kind, 0.665114689057843_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,451) = (/ 0.999947176611956_dbl_kind, 0.998588223880784_dbl_kind, 0.989147323138981_dbl_kind, 0.95231951028638_dbl_kind, 0.664815707190919_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,452) = (/ 0.999947067974804_dbl_kind, 0.998585471320231_dbl_kind, 0.989125661466608_dbl_kind, 0.95223263412578_dbl_kind, 0.664517459169569_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,453) = (/ 0.999946959342225_dbl_kind, 0.998582719131673_dbl_kind, 0.989104001870249_dbl_kind, 0.952145774643007_dbl_kind, 0.664219943002731_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,454) = (/ 0.999946850714223_dbl_kind, 0.998579967314734_dbl_kind, 0.989082344348061_dbl_kind, 0.952058931850917_dbl_kind, 0.663923156706629_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,455) = (/ 0.999946742090801_dbl_kind, 0.998577215869049_dbl_kind, 0.989060688898253_dbl_kind, 0.951972105762252_dbl_kind, 0.66362709830474_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,456) = (/ 0.999946633471965_dbl_kind, 0.998574464794268_dbl_kind, 0.989039035519083_dbl_kind, 0.95188529638965_dbl_kind, 0.663331765827736_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,457) = (/ 0.999946524857717_dbl_kind, 0.998571714090048_dbl_kind, 0.989017384208861_dbl_kind, 0.951798503745642_dbl_kind, 0.66303715731345_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,458) = (/ 0.99994641624806_dbl_kind, 0.99856896375606_dbl_kind, 0.98899573496594_dbl_kind, 0.95171172784265_dbl_kind, 0.662743270806823_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,459) = (/ 0.999946307642999_dbl_kind, 0.998566213791989_dbl_kind, 0.988974087788724_dbl_kind, 0.951624968692995_dbl_kind, 0.66245010435987_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,460) = (/ 0.999946199042535_dbl_kind, 0.998563464197526_dbl_kind, 0.988952442675663_dbl_kind, 0.951538226308895_dbl_kind, 0.662157656031628_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,461) = (/ 0.999946090446672_dbl_kind, 0.99856071497238_dbl_kind, 0.98893079962525_dbl_kind, 0.951451500702463_dbl_kind, 0.661865923888117_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,462) = (/ 0.999945981855412_dbl_kind, 0.998557966116266_dbl_kind, 0.988909158636025_dbl_kind, 0.951364791885714_dbl_kind, 0.661574906002299_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,463) = (/ 0.999945873268756_dbl_kind, 0.998555217628913_dbl_kind, 0.988887519706568_dbl_kind, 0.951278099870565_dbl_kind, 0.66128460045403_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,464) = (/ 0.999945764686708_dbl_kind, 0.998552469510059_dbl_kind, 0.988865882835505_dbl_kind, 0.95119142466883_dbl_kind, 0.660995005330022_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,465) = (/ 0.999945656109267_dbl_kind, 0.998549721759456_dbl_kind, 0.9888442480215_dbl_kind, 0.951104766292231_dbl_kind, 0.660706118723807_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,466) = (/ 0.999945547536436_dbl_kind, 0.998546974376862_dbl_kind, 0.988822615263259_dbl_kind, 0.951018124752391_dbl_kind, 0.660417938735684_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,467) = (/ 0.999945438968215_dbl_kind, 0.998544227362051_dbl_kind, 0.988800984559529_dbl_kind, 0.950931500060839_dbl_kind, 0.660130463472682_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,468) = (/ 0.999945330404607_dbl_kind, 0.998541480714804_dbl_kind, 0.988779355909094_dbl_kind, 0.950844892229013_dbl_kind, 0.659843691048533_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,469) = (/ 0.99994522184561_dbl_kind, 0.998538734434914_dbl_kind, 0.988757729310775_dbl_kind, 0.950758301268255_dbl_kind, 0.659557619583611_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,470) = (/ 0.999945113291225_dbl_kind, 0.998535988522183_dbl_kind, 0.988736104763433_dbl_kind, 0.950671727189818_dbl_kind, 0.659272247204908_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,471) = (/ 0.999945004741453_dbl_kind, 0.998533242976425_dbl_kind, 0.988714482265963_dbl_kind, 0.950585170004866_dbl_kind, 0.658987572045988_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,472) = (/ 0.999944896196293_dbl_kind, 0.998530497797462_dbl_kind, 0.988692861817295_dbl_kind, 0.950498629724473_dbl_kind, 0.658703592246947_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,473) = (/ 0.999944787655745_dbl_kind, 0.998527752985128_dbl_kind, 0.988671243416394_dbl_kind, 0.950412106359624_dbl_kind, 0.658420305954382_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,474) = (/ 0.999944679119808_dbl_kind, 0.998525008539265_dbl_kind, 0.988649627062261_dbl_kind, 0.950325599921221_dbl_kind, 0.658137711321343_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,475) = (/ 0.999944570588481_dbl_kind, 0.998522264459727_dbl_kind, 0.988628012753927_dbl_kind, 0.950239110420078_dbl_kind, 0.657855806507305_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,476) = (/ 0.999944462061764_dbl_kind, 0.998519520746374_dbl_kind, 0.988606400490456_dbl_kind, 0.950152637866924_dbl_kind, 0.657574589678116_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,477) = (/ 0.999944353539654_dbl_kind, 0.99851677739908_dbl_kind, 0.988584790270945_dbl_kind, 0.950066182272408_dbl_kind, 0.657294059005978_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,478) = (/ 0.99994424502215_dbl_kind, 0.998514034417725_dbl_kind, 0.988563182094519_dbl_kind, 0.949979743647096_dbl_kind, 0.657014212669399_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,479) = (/ 0.999944136509251_dbl_kind, 0.998511291802198_dbl_kind, 0.988541575960335_dbl_kind, 0.94989332200147_dbl_kind, 0.656735048853157_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,480) = (/ 0.999944028000954_dbl_kind, 0.9985085495524_dbl_kind, 0.988519971867579_dbl_kind, 0.949806917345935_dbl_kind, 0.656456565748261_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,481) = (/ 0.999943919497256_dbl_kind, 0.998505807668238_dbl_kind, 0.988498369815464_dbl_kind, 0.949720529690817_dbl_kind, 0.656178761551931_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,482) = (/ 0.999943810998156_dbl_kind, 0.99850306614963_dbl_kind, 0.988476769803233_dbl_kind, 0.949634159046363_dbl_kind, 0.655901634467538_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,483) = (/ 0.999943702503651_dbl_kind, 0.998500324996503_dbl_kind, 0.988455171830153_dbl_kind, 0.949547805422744_dbl_kind, 0.655625182704587_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,484) = (/ 0.999943594013737_dbl_kind, 0.998497584208789_dbl_kind, 0.98843357589552_dbl_kind, 0.949461468830054_dbl_kind, 0.655349404478681_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,485) = (/ 0.999943485528411_dbl_kind, 0.998494843786433_dbl_kind, 0.988411981998657_dbl_kind, 0.949375149278316_dbl_kind, 0.655074298011472_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,486) = (/ 0.999943377047671_dbl_kind, 0.998492103729386_dbl_kind, 0.988390390138909_dbl_kind, 0.949288846777473_dbl_kind, 0.654799861530643_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,487) = (/ 0.999943268571512_dbl_kind, 0.998489364037607_dbl_kind, 0.988368800315645_dbl_kind, 0.949202561337402_dbl_kind, 0.654526093269862_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,488) = (/ 0.99994316009993_dbl_kind, 0.998486624711063_dbl_kind, 0.988347212528261_dbl_kind, 0.949116292967903_dbl_kind, 0.65425299146876_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,489) = (/ 0.999943051632921_dbl_kind, 0.998483885749733_dbl_kind, 0.988325626776174_dbl_kind, 0.949030041678708_dbl_kind, 0.653980554372884_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,490) = (/ 0.999942943170481_dbl_kind, 0.998481147153598_dbl_kind, 0.988304043058824_dbl_kind, 0.948943807479478_dbl_kind, 0.653708780233675_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,491) = (/ 0.999942834712606_dbl_kind, 0.99847840892265_dbl_kind, 0.988282461375673_dbl_kind, 0.948857590379805_dbl_kind, 0.653437667308431_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,492) = (/ 0.99994272625929_dbl_kind, 0.998475671056888_dbl_kind, 0.988260881726205_dbl_kind, 0.948771390389212_dbl_kind, 0.653167213860265_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,493) = (/ 0.999942617810529_dbl_kind, 0.998472933556319_dbl_kind, 0.988239304109923_dbl_kind, 0.948685207517158_dbl_kind, 0.652897418158094_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,494) = (/ 0.999942509366319_dbl_kind, 0.998470196420957_dbl_kind, 0.988217728526352_dbl_kind, 0.948599041773035_dbl_kind, 0.652628278476583_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,495) = (/ 0.999942400926652_dbl_kind, 0.998467459650824_dbl_kind, 0.988196154975036_dbl_kind, 0.948512893166166_dbl_kind, 0.652359793096134_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,496) = (/ 0.999942292491524_dbl_kind, 0.998464723245948_dbl_kind, 0.988174583455537_dbl_kind, 0.948426761705814_dbl_kind, 0.652091960302838_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,497) = (/ 0.999942184060929_dbl_kind, 0.998461987206364_dbl_kind, 0.988153013967437_dbl_kind, 0.948340647401175_dbl_kind, 0.651824778388452_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,498) = (/ 0.999942075634861_dbl_kind, 0.998459251532116_dbl_kind, 0.988131446510336_dbl_kind, 0.948254550261387_dbl_kind, 0.651558245650369_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,499) = (/ 0.999941967213314_dbl_kind, 0.998456516223252_dbl_kind, 0.988109881083849_dbl_kind, 0.948168470295521_dbl_kind, 0.651292360391578_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,500) = (/ 0.999941858796282_dbl_kind, 0.998453781279828_dbl_kind, 0.988088317687611_dbl_kind, 0.948082407512589_dbl_kind, 0.651027120920648_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,501) = (/ 0.999941750383758_dbl_kind, 0.998451046701908_dbl_kind, 0.988066756321272_dbl_kind, 0.947996361921544_dbl_kind, 0.650762525551685_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,502) = (/ 0.999941641975736_dbl_kind, 0.99844831248956_dbl_kind, 0.988045196984499_dbl_kind, 0.947910333531278_dbl_kind, 0.650498572604305_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,503) = (/ 0.999941533572208_dbl_kind, 0.99844557864286_dbl_kind, 0.988023639676971_dbl_kind, 0.947824322350625_dbl_kind, 0.650235260403612_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,504) = (/ 0.999941425173168_dbl_kind, 0.99844284516189_dbl_kind, 0.988002084398386_dbl_kind, 0.947738328388363_dbl_kind, 0.649972587280158_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,505) = (/ 0.999941316778608_dbl_kind, 0.998440112046736_dbl_kind, 0.987980531148455_dbl_kind, 0.947652351653208_dbl_kind, 0.649710551569915_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,506) = (/ 0.999941208388521_dbl_kind, 0.998437379297495_dbl_kind, 0.987958979926902_dbl_kind, 0.947566392153824_dbl_kind, 0.649449151614257_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,507) = (/ 0.999941100002899_dbl_kind, 0.998434646914264_dbl_kind, 0.987937430733465_dbl_kind, 0.947480449898819_dbl_kind, 0.649188385759914_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,508) = (/ 0.999940991621736_dbl_kind, 0.99843191489715_dbl_kind, 0.987915883567896_dbl_kind, 0.947394524896745_dbl_kind, 0.648928252358956_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,509) = (/ 0.999940883245022_dbl_kind, 0.998429183246263_dbl_kind, 0.987894338429959_dbl_kind, 0.9473086171561_dbl_kind, 0.648668749768765_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,510) = (/ 0.999940774872749_dbl_kind, 0.998426451961721_dbl_kind, 0.98787279531943_dbl_kind, 0.94722272668533_dbl_kind, 0.648409876352_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,511) = (/ 0.999940666504911_dbl_kind, 0.998423721043646_dbl_kind, 0.987851254236097_dbl_kind, 0.947136853492828_dbl_kind, 0.648151630476569_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,512) = (/ 0.999940558141496_dbl_kind, 0.998420990492165_dbl_kind, 0.987829715179759_dbl_kind, 0.947050997586932_dbl_kind, 0.647894010515609_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,513) = (/ 0.999940449782499_dbl_kind, 0.99841826030741_dbl_kind, 0.987808178150225_dbl_kind, 0.946965158975932_dbl_kind, 0.647637014847449_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,514) = (/ 0.999940341427909_dbl_kind, 0.99841553048952_dbl_kind, 0.987786643147319_dbl_kind, 0.946879337668068_dbl_kind, 0.647380641855594_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,515) = (/ 0.999940233077719_dbl_kind, 0.998412801038637_dbl_kind, 0.987765110170869_dbl_kind, 0.946793533671527_dbl_kind, 0.647124889928687_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,516) = (/ 0.999940124731918_dbl_kind, 0.998410071954909_dbl_kind, 0.987743579220716_dbl_kind, 0.946707746994448_dbl_kind, 0.646869757460488_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,517) = (/ 0.999940016390498_dbl_kind, 0.998407343238489_dbl_kind, 0.987722050296711_dbl_kind, 0.946621977644922_dbl_kind, 0.646615242849846_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,518) = (/ 0.999939908053449_dbl_kind, 0.998404614889534_dbl_kind, 0.987700523398713_dbl_kind, 0.946536225630991_dbl_kind, 0.646361344500677_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,519) = (/ 0.999939799720762_dbl_kind, 0.998401886908204_dbl_kind, 0.987678998526587_dbl_kind, 0.94645049096065_dbl_kind, 0.646108060821926_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,520) = (/ 0.999939691392428_dbl_kind, 0.998399159294667_dbl_kind, 0.987657475680212_dbl_kind, 0.946364773641848_dbl_kind, 0.645855390227555_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,521) = (/ 0.999939583068437_dbl_kind, 0.998396432049093_dbl_kind, 0.98763595485947_dbl_kind, 0.946279073682483_dbl_kind, 0.645603331136509_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,522) = (/ 0.999939474748777_dbl_kind, 0.998393705171656_dbl_kind, 0.987614436064253_dbl_kind, 0.946193391090415_dbl_kind, 0.645351881972695_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,523) = (/ 0.999939366433442_dbl_kind, 0.998390978662536_dbl_kind, 0.987592919294459_dbl_kind, 0.946107725873452_dbl_kind, 0.64510104116495_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,524) = (/ 0.999939258122419_dbl_kind, 0.998388252521915_dbl_kind, 0.987571404549994_dbl_kind, 0.946022078039361_dbl_kind, 0.644850807147026_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,525) = (/ 0.999939149815698_dbl_kind, 0.99838552674998_dbl_kind, 0.987549891830769_dbl_kind, 0.945936447595864_dbl_kind, 0.644601178357552_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,526) = (/ 0.999939041513269_dbl_kind, 0.998382801346921_dbl_kind, 0.987528381136703_dbl_kind, 0.945850834550639_dbl_kind, 0.644352153240026_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,527) = (/ 0.999938933215123_dbl_kind, 0.998380076312933_dbl_kind, 0.987506872467719_dbl_kind, 0.945765238911321_dbl_kind, 0.644103730242772_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,528) = (/ 0.999938824921246_dbl_kind, 0.998377351648214_dbl_kind, 0.987485365823748_dbl_kind, 0.945679660685505_dbl_kind, 0.643855907818929_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,529) = (/ 0.99993871663163_dbl_kind, 0.998374627352965_dbl_kind, 0.987463861204725_dbl_kind, 0.945594099880739_dbl_kind, 0.643608684426422_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,530) = (/ 0.999938608346264_dbl_kind, 0.998371903427391_dbl_kind, 0.987442358610589_dbl_kind, 0.945508556504534_dbl_kind, 0.643362058527943_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,531) = (/ 0.999938500065137_dbl_kind, 0.9983691798717_dbl_kind, 0.987420858041285_dbl_kind, 0.945423030564356_dbl_kind, 0.643116028590914_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,532) = (/ 0.999938391788236_dbl_kind, 0.998366456686104_dbl_kind, 0.987399359496763_dbl_kind, 0.945337522067635_dbl_kind, 0.642870593087482_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,533) = (/ 0.999938283515552_dbl_kind, 0.998363733870816_dbl_kind, 0.987377862976976_dbl_kind, 0.945252031021758_dbl_kind, 0.642625750494476_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,534) = (/ 0.999938175247073_dbl_kind, 0.998361011426055_dbl_kind, 0.987356368481882_dbl_kind, 0.945166557434071_dbl_kind, 0.642381499293403_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,535) = (/ 0.999938066982788_dbl_kind, 0.998358289352041_dbl_kind, 0.987334876011439_dbl_kind, 0.945081101311884_dbl_kind, 0.642137837970413_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,536) = (/ 0.999937958722684_dbl_kind, 0.998355567648996_dbl_kind, 0.987313385565616_dbl_kind, 0.944995662662467_dbl_kind, 0.641894765016276_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,537) = (/ 0.999937850466751_dbl_kind, 0.998352846317149_dbl_kind, 0.987291897144375_dbl_kind, 0.94491024149305_dbl_kind, 0.641652278926361_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,538) = (/ 0.999937742214977_dbl_kind, 0.998350125356725_dbl_kind, 0.987270410747691_dbl_kind, 0.944824837810827_dbl_kind, 0.641410378200626_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,539) = (/ 0.99993763396735_dbl_kind, 0.998347404767957_dbl_kind, 0.987248926375534_dbl_kind, 0.944739451622953_dbl_kind, 0.64116906134357_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,540) = (/ 0.999937525723859_dbl_kind, 0.998344684551078_dbl_kind, 0.987227444027879_dbl_kind, 0.944654082936546_dbl_kind, 0.640928326864237_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,541) = (/ 0.999937417484491_dbl_kind, 0.998341964706326_dbl_kind, 0.987205963704706_dbl_kind, 0.944568731758689_dbl_kind, 0.640688173276174_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,542) = (/ 0.999937309249234_dbl_kind, 0.998339245233936_dbl_kind, 0.987184485405992_dbl_kind, 0.944483398096427_dbl_kind, 0.64044859909742_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,543) = (/ 0.999937201018075_dbl_kind, 0.99833652613415_dbl_kind, 0.987163009131718_dbl_kind, 0.944398081956767_dbl_kind, 0.640209602850488_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,544) = (/ 0.999937092791005_dbl_kind, 0.998333807407211_dbl_kind, 0.987141534881867_dbl_kind, 0.944312783346683_dbl_kind, 0.639971183062329_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,545) = (/ 0.99993698456801_dbl_kind, 0.998331089053362_dbl_kind, 0.987120062656422_dbl_kind, 0.944227502273115_dbl_kind, 0.639733338264324_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,546) = (/ 0.999936876349076_dbl_kind, 0.99832837107285_dbl_kind, 0.987098592455369_dbl_kind, 0.944142238742963_dbl_kind, 0.639496066992259_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,547) = (/ 0.999936768134193_dbl_kind, 0.998325653465924_dbl_kind, 0.987077124278691_dbl_kind, 0.944056992763096_dbl_kind, 0.639259367786298_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,548) = (/ 0.999936659923346_dbl_kind, 0.998322936232834_dbl_kind, 0.987055658126377_dbl_kind, 0.943971764340348_dbl_kind, 0.639023239190971_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,549) = (/ 0.999936551716525_dbl_kind, 0.99832021937383_dbl_kind, 0.987034193998412_dbl_kind, 0.943886553481518_dbl_kind, 0.638787679755151_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,550) = (/ 0.999936443513717_dbl_kind, 0.998317502889166_dbl_kind, 0.987012731894783_dbl_kind, 0.943801360193371_dbl_kind, 0.63855268803203_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,551) = (/ 0.999936335314908_dbl_kind, 0.998314786779096_dbl_kind, 0.986991271815477_dbl_kind, 0.943716184482641_dbl_kind, 0.638318262579102_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,552) = (/ 0.999936227120085_dbl_kind, 0.998312071043877_dbl_kind, 0.986969813760479_dbl_kind, 0.943631026356026_dbl_kind, 0.638084401958141_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,553) = (/ 0.999936118929238_dbl_kind, 0.998309355683767_dbl_kind, 0.986948357729778_dbl_kind, 0.943545885820191_dbl_kind, 0.637851104735181_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,554) = (/ 0.999936010742351_dbl_kind, 0.998306640699023_dbl_kind, 0.986926903723357_dbl_kind, 0.94346076288177_dbl_kind, 0.637618369480497_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,555) = (/ 0.999935902559413_dbl_kind, 0.998303926089905_dbl_kind, 0.986905451741203_dbl_kind, 0.943375657547365_dbl_kind, 0.637386194768589_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,556) = (/ 0.999935794380409_dbl_kind, 0.998301211856675_dbl_kind, 0.986884001783301_dbl_kind, 0.943290569823544_dbl_kind, 0.637154579178153_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,557) = (/ 0.999935686205329_dbl_kind, 0.998298497999593_dbl_kind, 0.986862553849632_dbl_kind, 0.943205499716842_dbl_kind, 0.63692352129207_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,558) = (/ 0.999935578034157_dbl_kind, 0.998295784518923_dbl_kind, 0.98684110794018_dbl_kind, 0.943120447233767_dbl_kind, 0.636693019697382_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,559) = (/ 0.999935469866882_dbl_kind, 0.998293071414929_dbl_kind, 0.986819664054924_dbl_kind, 0.943035412380791_dbl_kind, 0.63646307298528_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,560) = (/ 0.99993536170349_dbl_kind, 0.998290358687874_dbl_kind, 0.986798222193845_dbl_kind, 0.942950395164356_dbl_kind, 0.636233679751073_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,561) = (/ 0.999935253543966_dbl_kind, 0.998287646338024_dbl_kind, 0.98677678235692_dbl_kind, 0.942865395590876_dbl_kind, 0.636004838594176_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,562) = (/ 0.9999351453883_dbl_kind, 0.998284934365645_dbl_kind, 0.986755344544125_dbl_kind, 0.94278041366673_dbl_kind, 0.635776548118101_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,563) = (/ 0.999935037236477_dbl_kind, 0.998282222771002_dbl_kind, 0.986733908755434_dbl_kind, 0.942695449398268_dbl_kind, 0.635548806930413_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,564) = (/ 0.999934929088484_dbl_kind, 0.998279511554362_dbl_kind, 0.986712474990819_dbl_kind, 0.94261050279181_dbl_kind, 0.635321613642741_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,565) = (/ 0.999934820944308_dbl_kind, 0.998276800715993_dbl_kind, 0.98669104325025_dbl_kind, 0.942525573853646_dbl_kind, 0.635094966870737_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,566) = (/ 0.999934712803934_dbl_kind, 0.998274090256162_dbl_kind, 0.986669613533694_dbl_kind, 0.942440662590036_dbl_kind, 0.634868865234074_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,567) = (/ 0.99993460466735_dbl_kind, 0.998271380175137_dbl_kind, 0.986648185841116_dbl_kind, 0.942355769007209_dbl_kind, 0.634643307356414_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,568) = (/ 0.999934496534542_dbl_kind, 0.998268670473185_dbl_kind, 0.986626760172479_dbl_kind, 0.942270893111365_dbl_kind, 0.634418291865401_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,569) = (/ 0.999934388405497_dbl_kind, 0.998265961150574_dbl_kind, 0.986605336527742_dbl_kind, 0.942186034908676_dbl_kind, 0.634193817392636_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,570) = (/ 0.999934280280201_dbl_kind, 0.998263252207573_dbl_kind, 0.986583914906862_dbl_kind, 0.942101194405282_dbl_kind, 0.633969882573665_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,571) = (/ 0.999934172158641_dbl_kind, 0.99826054364445_dbl_kind, 0.986562495309794_dbl_kind, 0.942016371607296_dbl_kind, 0.633746486047961_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,572) = (/ 0.999934064040803_dbl_kind, 0.998257835461473_dbl_kind, 0.986541077736488_dbl_kind, 0.941931566520802_dbl_kind, 0.633523626458898_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,573) = (/ 0.999933955926673_dbl_kind, 0.998255127658909_dbl_kind, 0.986519662186892_dbl_kind, 0.941846779151853_dbl_kind, 0.633301302453745_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,574) = (/ 0.999933847816238_dbl_kind, 0.998252420237026_dbl_kind, 0.986498248660951_dbl_kind, 0.941762009506476_dbl_kind, 0.633079512683645_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,575) = (/ 0.999933739709484_dbl_kind, 0.998249713196092_dbl_kind, 0.986476837158605_dbl_kind, 0.941677257590668_dbl_kind, 0.632858255803595_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,576) = (/ 0.999933631606398_dbl_kind, 0.998247006536374_dbl_kind, 0.986455427679792_dbl_kind, 0.941592523410397_dbl_kind, 0.63263753047243_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,577) = (/ 0.999933523506966_dbl_kind, 0.998244300258138_dbl_kind, 0.986434020224448_dbl_kind, 0.941507806971602_dbl_kind, 0.632417335352815_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,578) = (/ 0.999933415411175_dbl_kind, 0.998241594361651_dbl_kind, 0.986412614792501_dbl_kind, 0.941423108280199_dbl_kind, 0.632197669111211_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,579) = (/ 0.99993330731901_dbl_kind, 0.998238888847178_dbl_kind, 0.986391211383879_dbl_kind, 0.94133842734207_dbl_kind, 0.631978530417876_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,580) = (/ 0.999933199230458_dbl_kind, 0.998236183714986_dbl_kind, 0.986369809998504_dbl_kind, 0.94125376416307_dbl_kind, 0.631759917946839_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,581) = (/ 0.999933091145506_dbl_kind, 0.998233478965337_dbl_kind, 0.986348410636296_dbl_kind, 0.941169118749029_dbl_kind, 0.631541830375885_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,582) = (/ 0.999932983064138_dbl_kind, 0.998230774598497_dbl_kind, 0.986327013297169_dbl_kind, 0.941084491105747_dbl_kind, 0.631324266386541_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,583) = (/ 0.999932874986343_dbl_kind, 0.998228070614728_dbl_kind, 0.986305617981036_dbl_kind, 0.940999881238995_dbl_kind, 0.631107224664056_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,584) = (/ 0.999932766912107_dbl_kind, 0.998225367014293_dbl_kind, 0.986284224687801_dbl_kind, 0.940915289154521_dbl_kind, 0.630890703897395_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,585) = (/ 0.999932658841414_dbl_kind, 0.998222663797452_dbl_kind, 0.986262833417367_dbl_kind, 0.940830714858041_dbl_kind, 0.63067470277921_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,586) = (/ 0.999932550774253_dbl_kind, 0.998219960964467_dbl_kind, 0.986241444169632_dbl_kind, 0.940746158355246_dbl_kind, 0.630459220005833_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,587) = (/ 0.999932442710609_dbl_kind, 0.998217258515599_dbl_kind, 0.986220056944491_dbl_kind, 0.940661619651797_dbl_kind, 0.630244254277255_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,588) = (/ 0.999932334650467_dbl_kind, 0.998214556451104_dbl_kind, 0.986198671741832_dbl_kind, 0.940577098753332_dbl_kind, 0.630029804297116_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,589) = (/ 0.999932226593817_dbl_kind, 0.998211854771241_dbl_kind, 0.98617728856154_dbl_kind, 0.940492595665458_dbl_kind, 0.629815868772688_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,590) = (/ 0.999932118540641_dbl_kind, 0.998209153476266_dbl_kind, 0.986155907403495_dbl_kind, 0.940408110393757_dbl_kind, 0.629602446414857_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,591) = (/ 0.999932010490929_dbl_kind, 0.998206452566436_dbl_kind, 0.986134528267573_dbl_kind, 0.940323642943783_dbl_kind, 0.629389535938112_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,592) = (/ 0.999931902444665_dbl_kind, 0.998203752042003_dbl_kind, 0.986113151153643_dbl_kind, 0.940239193321061_dbl_kind, 0.629177136060524_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,593) = (/ 0.999931794401837_dbl_kind, 0.998201051903222_dbl_kind, 0.986091776061572_dbl_kind, 0.940154761531094_dbl_kind, 0.62896524550374_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,594) = (/ 0.99993168636243_dbl_kind, 0.998198352150343_dbl_kind, 0.986070402991221_dbl_kind, 0.940070347579354_dbl_kind, 0.62875386299296_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,595) = (/ 0.99993157832643_dbl_kind, 0.998195652783616_dbl_kind, 0.986049031942446_dbl_kind, 0.939985951471288_dbl_kind, 0.628542987256922_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,596) = (/ 0.999931470293825_dbl_kind, 0.998192953803292_dbl_kind, 0.986027662915099_dbl_kind, 0.939901573212314_dbl_kind, 0.6283326170279_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,597) = (/ 0.9999313622646_dbl_kind, 0.998190255209616_dbl_kind, 0.986006295909023_dbl_kind, 0.939817212807826_dbl_kind, 0.628122751041673_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,598) = (/ 0.999931254238743_dbl_kind, 0.998187557002837_dbl_kind, 0.985984930924063_dbl_kind, 0.93973287026319_dbl_kind, 0.627913388037522_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,599) = (/ 0.999931146216239_dbl_kind, 0.998184859183197_dbl_kind, 0.985963567960052_dbl_kind, 0.939648545583744_dbl_kind, 0.627704526758207_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,600) = (/ 0.999931038197076_dbl_kind, 0.998182161750939_dbl_kind, 0.985942207016822_dbl_kind, 0.939564238774801_dbl_kind, 0.627496165949962_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,601) = (/ 0.999930930181238_dbl_kind, 0.998179464706306_dbl_kind, 0.985920848094198_dbl_kind, 0.939479949841648_dbl_kind, 0.627288304362478_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,602) = (/ 0.999930822168715_dbl_kind, 0.998176768049537_dbl_kind, 0.985899491192001_dbl_kind, 0.939395678789542_dbl_kind, 0.627080940748879_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,603) = (/ 0.99993071415949_dbl_kind, 0.99817407178087_dbl_kind, 0.985878136310046_dbl_kind, 0.939311425623718_dbl_kind, 0.626874073865725_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,604) = (/ 0.999930606153551_dbl_kind, 0.998171375900542_dbl_kind, 0.985856783448142_dbl_kind, 0.93922719034938_dbl_kind, 0.626667702472989_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,605) = (/ 0.999930498150886_dbl_kind, 0.998168680408785_dbl_kind, 0.985835432606093_dbl_kind, 0.939142972971707_dbl_kind, 0.626461825334038_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,606) = (/ 0.999930390151479_dbl_kind, 0.998165985305834_dbl_kind, 0.9858140837837_dbl_kind, 0.939058773495854_dbl_kind, 0.626256441215633_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,607) = (/ 0.999930282155319_dbl_kind, 0.998163290591921_dbl_kind, 0.985792736980755_dbl_kind, 0.938974591926945_dbl_kind, 0.626051548887901_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,608) = (/ 0.999930174162391_dbl_kind, 0.998160596267273_dbl_kind, 0.985771392197046_dbl_kind, 0.938890428270081_dbl_kind, 0.62584714712434_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,609) = (/ 0.999930066172682_dbl_kind, 0.998157902332117_dbl_kind, 0.985750049432356_dbl_kind, 0.938806282530335_dbl_kind, 0.625643234701782_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,610) = (/ 0.99992995818618_dbl_kind, 0.99815520878668_dbl_kind, 0.985728708686462_dbl_kind, 0.938722154712753_dbl_kind, 0.625439810400395_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,611) = (/ 0.99992985020287_dbl_kind, 0.998152515631183_dbl_kind, 0.985707369959136_dbl_kind, 0.938638044822355_dbl_kind, 0.625236873003674_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,612) = (/ 0.999929742222741_dbl_kind, 0.99814982286585_dbl_kind, 0.985686033250142_dbl_kind, 0.938553952864135_dbl_kind, 0.625034421298414_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,613) = (/ 0.999929634245777_dbl_kind, 0.998147130490899_dbl_kind, 0.985664698559243_dbl_kind, 0.938469878843058_dbl_kind, 0.624832454074708_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,614) = (/ 0.999929526271967_dbl_kind, 0.998144438506546_dbl_kind, 0.985643365886191_dbl_kind, 0.938385822764067_dbl_kind, 0.624630970125924_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,615) = (/ 0.999929418301298_dbl_kind, 0.998141746913008_dbl_kind, 0.985622035230738_dbl_kind, 0.938301784632074_dbl_kind, 0.624429968248706_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,616) = (/ 0.999929310333755_dbl_kind, 0.998139055710497_dbl_kind, 0.985600706592625_dbl_kind, 0.938217764451968_dbl_kind, 0.624229447242954_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,617) = (/ 0.999929202369327_dbl_kind, 0.998136364899223_dbl_kind, 0.985579379971589_dbl_kind, 0.938133762228607_dbl_kind, 0.624029405911803_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,618) = (/ 0.999929094408_dbl_kind, 0.998133674479395_dbl_kind, 0.985558055367366_dbl_kind, 0.938049777966826_dbl_kind, 0.623829843061625_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,619) = (/ 0.999928986449762_dbl_kind, 0.998130984451221_dbl_kind, 0.985536732779678_dbl_kind, 0.937965811671434_dbl_kind, 0.623630757502011_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,620) = (/ 0.999928878494598_dbl_kind, 0.998128294814904_dbl_kind, 0.985515412208248_dbl_kind, 0.937881863347209_dbl_kind, 0.623432148045754_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,621) = (/ 0.999928770542498_dbl_kind, 0.998125605570646_dbl_kind, 0.98549409365279_dbl_kind, 0.937797932998908_dbl_kind, 0.623234013508842_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,622) = (/ 0.999928662593446_dbl_kind, 0.998122916718646_dbl_kind, 0.985472777113013_dbl_kind, 0.937714020631256_dbl_kind, 0.623036352710447_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,623) = (/ 0.999928554647433_dbl_kind, 0.998120228259101_dbl_kind, 0.98545146258862_dbl_kind, 0.937630126248955_dbl_kind, 0.622839164472908_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,624) = (/ 0.999928446704443_dbl_kind, 0.998117540192207_dbl_kind, 0.98543015007931_dbl_kind, 0.937546249856678_dbl_kind, 0.62264244762172_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,625) = (/ 0.999928338764465_dbl_kind, 0.998114852518157_dbl_kind, 0.985408839584772_dbl_kind, 0.937462391459074_dbl_kind, 0.622446200985526_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,626) = (/ 0.999928230827486_dbl_kind, 0.998112165237141_dbl_kind, 0.985387531104694_dbl_kind, 0.937378551060763_dbl_kind, 0.622250423396105_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,627) = (/ 0.999928122893494_dbl_kind, 0.998109478349345_dbl_kind, 0.985366224638755_dbl_kind, 0.937294728666339_dbl_kind, 0.622055113688352_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,628) = (/ 0.999928014962475_dbl_kind, 0.998106791854955_dbl_kind, 0.985344920186631_dbl_kind, 0.937210924280368_dbl_kind, 0.621860270700276_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,629) = (/ 0.999927907034418_dbl_kind, 0.998104105754155_dbl_kind, 0.985323617747988_dbl_kind, 0.93712713790739_dbl_kind, 0.621665893272984_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,630) = (/ 0.99992779910931_dbl_kind, 0.998101420047124_dbl_kind, 0.985302317322489_dbl_kind, 0.93704336955192_dbl_kind, 0.62147198025067_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,631) = (/ 0.999927691187138_dbl_kind, 0.998098734734042_dbl_kind, 0.985281018909793_dbl_kind, 0.936959619218444_dbl_kind, 0.621278530480604_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,632) = (/ 0.999927583267891_dbl_kind, 0.998096049815081_dbl_kind, 0.985259722509549_dbl_kind, 0.936875886911419_dbl_kind, 0.621085542813119_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,633) = (/ 0.999927475351555_dbl_kind, 0.998093365290417_dbl_kind, 0.985238428121403_dbl_kind, 0.936792172635282_dbl_kind, 0.620893016101601_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,634) = (/ 0.999927367438119_dbl_kind, 0.998090681160218_dbl_kind, 0.985217135744992_dbl_kind, 0.936708476394435_dbl_kind, 0.62070094920248_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,635) = (/ 0.999927259527572_dbl_kind, 0.998087997424653_dbl_kind, 0.985195845379954_dbl_kind, 0.936624798193259_dbl_kind, 0.620509340975215_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,636) = (/ 0.999927151619898_dbl_kind, 0.998085314083887_dbl_kind, 0.985174557025913_dbl_kind, 0.936541138036103_dbl_kind, 0.620318190282285_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,637) = (/ 0.999927043715089_dbl_kind, 0.998082631138081_dbl_kind, 0.985153270682492_dbl_kind, 0.936457495927294_dbl_kind, 0.620127495989175_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,638) = (/ 0.999926935813131_dbl_kind, 0.998079948587397_dbl_kind, 0.985131986349306_dbl_kind, 0.936373871871128_dbl_kind, 0.619937256964372_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,639) = (/ 0.999926827914013_dbl_kind, 0.998077266431991_dbl_kind, 0.985110704025967_dbl_kind, 0.936290265871875_dbl_kind, 0.619747472079343_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,640) = (/ 0.999926720017722_dbl_kind, 0.998074584672017_dbl_kind, 0.985089423712079_dbl_kind, 0.936206677933779_dbl_kind, 0.619558140208541_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,641) = (/ 0.999926612124246_dbl_kind, 0.998071903307629_dbl_kind, 0.985068145407239_dbl_kind, 0.936123108061054_dbl_kind, 0.619369260229375_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,642) = (/ 0.999926504233574_dbl_kind, 0.998069222338974_dbl_kind, 0.985046869111041_dbl_kind, 0.936039556257891_dbl_kind, 0.619180831022213_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,643) = (/ 0.999926396345695_dbl_kind, 0.998066541766199_dbl_kind, 0.985025594823073_dbl_kind, 0.935956022528449_dbl_kind, 0.618992851470364_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,644) = (/ 0.999926288460596_dbl_kind, 0.998063861589447_dbl_kind, 0.985004322542914_dbl_kind, 0.935872506876864_dbl_kind, 0.618805320460075_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,645) = (/ 0.999926180578265_dbl_kind, 0.998061181808862_dbl_kind, 0.984983052270141_dbl_kind, 0.93578900930724_dbl_kind, 0.618618236880514_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,646) = (/ 0.999926072698693_dbl_kind, 0.998058502424579_dbl_kind, 0.984961784004324_dbl_kind, 0.935705529823658_dbl_kind, 0.618431599623759_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,647) = (/ 0.999925964821865_dbl_kind, 0.998055823436735_dbl_kind, 0.984940517745027_dbl_kind, 0.935622068430169_dbl_kind, 0.618245407584792_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,648) = (/ 0.999925856947772_dbl_kind, 0.998053144845463_dbl_kind, 0.984919253491806_dbl_kind, 0.935538625130798_dbl_kind, 0.618059659661491_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,649) = (/ 0.999925749076402_dbl_kind, 0.998050466650892_dbl_kind, 0.984897991244217_dbl_kind, 0.935455199929541_dbl_kind, 0.617874354754607_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,650) = (/ 0.999925641207744_dbl_kind, 0.998047788853151_dbl_kind, 0.984876731001804_dbl_kind, 0.935371792830366_dbl_kind, 0.617689491767771_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,651) = (/ 0.999925533341786_dbl_kind, 0.998045111452362_dbl_kind, 0.984855472764109_dbl_kind, 0.935288403837216_dbl_kind, 0.617505069607474_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,652) = (/ 0.999925425478517_dbl_kind, 0.998042434448649_dbl_kind, 0.984834216530669_dbl_kind, 0.935205032954005_dbl_kind, 0.617321087183056_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,653) = (/ 0.999925317617925_dbl_kind, 0.998039757842129_dbl_kind, 0.984812962301012_dbl_kind, 0.935121680184618_dbl_kind, 0.617137543406701_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,654) = (/ 0.999925209760002_dbl_kind, 0.998037081632919_dbl_kind, 0.984791710074663_dbl_kind, 0.935038345532915_dbl_kind, 0.616954437193424_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,655) = (/ 0.999925101904734_dbl_kind, 0.998034405821133_dbl_kind, 0.984770459851141_dbl_kind, 0.934955029002725_dbl_kind, 0.616771767461064_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,656) = (/ 0.999924994052111_dbl_kind, 0.998031730406879_dbl_kind, 0.984749211629957_dbl_kind, 0.934871730597852_dbl_kind, 0.61658953313027_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,657) = (/ 0.999924886202121_dbl_kind, 0.998029055390267_dbl_kind, 0.98472796541062_dbl_kind, 0.934788450322072_dbl_kind, 0.616407733124495_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,658) = (/ 0.999924778354755_dbl_kind, 0.9980263807714_dbl_kind, 0.984706721192632_dbl_kind, 0.934705188179129_dbl_kind, 0.616226366369984_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,659) = (/ 0.999924670510002_dbl_kind, 0.998023706550381_dbl_kind, 0.984685478975488_dbl_kind, 0.934621944172745_dbl_kind, 0.616045431795769_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,660) = (/ 0.99992456266785_dbl_kind, 0.998021032727308_dbl_kind, 0.98466423875868_dbl_kind, 0.934538718306611_dbl_kind, 0.615864928333653_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,661) = (/ 0.999924454828289_dbl_kind, 0.998018359302277_dbl_kind, 0.984643000541692_dbl_kind, 0.93445551058439_dbl_kind, 0.615684854918203_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,662) = (/ 0.999924346991309_dbl_kind, 0.998015686275383_dbl_kind, 0.984621764324005_dbl_kind, 0.934372321009717_dbl_kind, 0.615505210486745_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,663) = (/ 0.999924239156899_dbl_kind, 0.998013013646715_dbl_kind, 0.984600530105092_dbl_kind, 0.9342891495862_dbl_kind, 0.615325993979347_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,664) = (/ 0.999924131325048_dbl_kind, 0.99801034141636_dbl_kind, 0.984579297884423_dbl_kind, 0.934205996317418_dbl_kind, 0.615147204338817_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,665) = (/ 0.999924023495747_dbl_kind, 0.998007669584403_dbl_kind, 0.98455806766146_dbl_kind, 0.934122861206921_dbl_kind, 0.614968840510687_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,666) = (/ 0.999923915668984_dbl_kind, 0.998004998150927_dbl_kind, 0.984536839435662_dbl_kind, 0.934039744258234_dbl_kind, 0.614790901443211_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,667) = (/ 0.999923807844751_dbl_kind, 0.99800232711601_dbl_kind, 0.984515613206481_dbl_kind, 0.93395664547485_dbl_kind, 0.614613386087352_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,668) = (/ 0.999923700023035_dbl_kind, 0.997999656479728_dbl_kind, 0.984494388973365_dbl_kind, 0.933873564860236_dbl_kind, 0.614436293396768_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,669) = (/ 0.999923592203828_dbl_kind, 0.997996986242154_dbl_kind, 0.984473166735755_dbl_kind, 0.93379050241783_dbl_kind, 0.614259622327816_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,670) = (/ 0.999923484387119_dbl_kind, 0.997994316403359_dbl_kind, 0.984451946493088_dbl_kind, 0.933707458151042_dbl_kind, 0.614083371839531_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,671) = (/ 0.999923376572899_dbl_kind, 0.997991646963408_dbl_kind, 0.984430728244795_dbl_kind, 0.933624432063254_dbl_kind, 0.613907540893621_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,672) = (/ 0.999923268761157_dbl_kind, 0.997988977922368_dbl_kind, 0.984409511990302_dbl_kind, 0.933541424157817_dbl_kind, 0.613732128454463_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,673) = (/ 0.999923160951883_dbl_kind, 0.9979863092803_dbl_kind, 0.984388297729032_dbl_kind, 0.93345843443806_dbl_kind, 0.613557133489081_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,674) = (/ 0.999923053145069_dbl_kind, 0.997983641037263_dbl_kind, 0.984367085460397_dbl_kind, 0.933375462907275_dbl_kind, 0.613382554967157_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,675) = (/ 0.999922945340703_dbl_kind, 0.997980973193311_dbl_kind, 0.98434587518381_dbl_kind, 0.933292509568733_dbl_kind, 0.613208391861004_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,676) = (/ 0.999922837538776_dbl_kind, 0.997978305748499_dbl_kind, 0.984324666898677_dbl_kind, 0.933209574425671_dbl_kind, 0.613034643145569_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,677) = (/ 0.99992272973928_dbl_kind, 0.997975638702876_dbl_kind, 0.984303460604395_dbl_kind, 0.933126657481301_dbl_kind, 0.612861307798418_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,678) = (/ 0.999922621942203_dbl_kind, 0.997972972056489_dbl_kind, 0.984282256300362_dbl_kind, 0.933043758738807_dbl_kind, 0.612688384799731_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,679) = (/ 0.999922514147538_dbl_kind, 0.997970305809382_dbl_kind, 0.984261053985966_dbl_kind, 0.932960878201339_dbl_kind, 0.612515873132293_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,680) = (/ 0.999922406355273_dbl_kind, 0.997967639961599_dbl_kind, 0.984239853660594_dbl_kind, 0.932878015872025_dbl_kind, 0.612343771781483_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,681) = (/ 0.999922298565402_dbl_kind, 0.997964974513175_dbl_kind, 0.984218655323625_dbl_kind, 0.93279517175396_dbl_kind, 0.612172079735269_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,682) = (/ 0.999922190777912_dbl_kind, 0.997962309464148_dbl_kind, 0.984197458974434_dbl_kind, 0.932712345850213_dbl_kind, 0.612000795984199_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,683) = (/ 0.999922082992797_dbl_kind, 0.997959644814549_dbl_kind, 0.98417626461239_dbl_kind, 0.932629538163821_dbl_kind, 0.611829919521391_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,684) = (/ 0.999921975210046_dbl_kind, 0.99795698056441_dbl_kind, 0.984155072236861_dbl_kind, 0.932546748697798_dbl_kind, 0.611659449342526_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,685) = (/ 0.99992186742965_dbl_kind, 0.997954316713756_dbl_kind, 0.984133881847205_dbl_kind, 0.932463977455123_dbl_kind, 0.611489384445841_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,686) = (/ 0.999921759651601_dbl_kind, 0.997951653262613_dbl_kind, 0.984112693442779_dbl_kind, 0.932381224438749_dbl_kind, 0.611319723832118_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,687) = (/ 0.99992165187589_dbl_kind, 0.997948990211001_dbl_kind, 0.984091507022933_dbl_kind, 0.9322984896516_dbl_kind, 0.611150466504678_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,688) = (/ 0.999921544102508_dbl_kind, 0.997946327558939_dbl_kind, 0.984070322587011_dbl_kind, 0.932215773096573_dbl_kind, 0.610981611469372_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,689) = (/ 0.999921436331445_dbl_kind, 0.997943665306442_dbl_kind, 0.984049140134358_dbl_kind, 0.932133074776533_dbl_kind, 0.610813157734578_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,690) = (/ 0.999921328562694_dbl_kind, 0.997941003453524_dbl_kind, 0.984027959664306_dbl_kind, 0.932050394694318_dbl_kind, 0.61064510431118_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,691) = (/ 0.999921220796245_dbl_kind, 0.997938342000195_dbl_kind, 0.98400678117619_dbl_kind, 0.931967732852737_dbl_kind, 0.610477450212579_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,692) = (/ 0.999921113032091_dbl_kind, 0.997935680946461_dbl_kind, 0.983985604669336_dbl_kind, 0.93188508925457_dbl_kind, 0.610310194454666_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,693) = (/ 0.999921005270221_dbl_kind, 0.997933020292328_dbl_kind, 0.983964430143066_dbl_kind, 0.931802463902568_dbl_kind, 0.610143336055829_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,694) = (/ 0.99992089751063_dbl_kind, 0.997930360037795_dbl_kind, 0.983943257596697_dbl_kind, 0.931719856799453_dbl_kind, 0.609976874036938_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,695) = (/ 0.999920789753306_dbl_kind, 0.997927700182863_dbl_kind, 0.983922087029545_dbl_kind, 0.931637267947918_dbl_kind, 0.609810807421335_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,696) = (/ 0.999920681998243_dbl_kind, 0.997925040727527_dbl_kind, 0.983900918440916_dbl_kind, 0.931554697350628_dbl_kind, 0.609645135234833_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,697) = (/ 0.999920574245433_dbl_kind, 0.99792238167178_dbl_kind, 0.983879751830116_dbl_kind, 0.931472145010217_dbl_kind, 0.609479856505708_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,698) = (/ 0.999920466494866_dbl_kind, 0.997919723015613_dbl_kind, 0.983858587196443_dbl_kind, 0.931389610929293_dbl_kind, 0.609314970264689_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,699) = (/ 0.999920358746536_dbl_kind, 0.997917064759013_dbl_kind, 0.983837424539195_dbl_kind, 0.931307095110432_dbl_kind, 0.609150475544943_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,700) = (/ 0.999920251000433_dbl_kind, 0.997914406901965_dbl_kind, 0.98381626385766_dbl_kind, 0.931224597556183_dbl_kind, 0.608986371382079_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,701) = (/ 0.999920143256549_dbl_kind, 0.997911749444451_dbl_kind, 0.983795105151126_dbl_kind, 0.931142118269066_dbl_kind, 0.608822656814142_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,702) = (/ 0.999920035514879_dbl_kind, 0.99790909238645_dbl_kind, 0.983773948418876_dbl_kind, 0.93105965725157_dbl_kind, 0.608659330881589_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,703) = (/ 0.999919927775411_dbl_kind, 0.997906435727939_dbl_kind, 0.983752793660187_dbl_kind, 0.930977214506157_dbl_kind, 0.608496392627302_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,704) = (/ 0.999919820038142_dbl_kind, 0.997903779468892_dbl_kind, 0.983731640874332_dbl_kind, 0.930894790035261_dbl_kind, 0.608333841096569_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,705) = (/ 0.99991971230306_dbl_kind, 0.997901123609279_dbl_kind, 0.983710490060583_dbl_kind, 0.930812383841282_dbl_kind, 0.608171675337073_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,706) = (/ 0.999919604570159_dbl_kind, 0.997898468149069_dbl_kind, 0.983689341218203_dbl_kind, 0.930729995926597_dbl_kind, 0.608009894398901_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,707) = (/ 0.999919496839432_dbl_kind, 0.997895813088228_dbl_kind, 0.983668194346454_dbl_kind, 0.93064762629355_dbl_kind, 0.60784849733452_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,708) = (/ 0.99991938911087_dbl_kind, 0.997893158426718_dbl_kind, 0.983647049444593_dbl_kind, 0.930565274944459_dbl_kind, 0.607687483198776_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,709) = (/ 0.999919281384468_dbl_kind, 0.997890504164499_dbl_kind, 0.983625906511874_dbl_kind, 0.930482941881609_dbl_kind, 0.607526851048887_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,710) = (/ 0.999919173660216_dbl_kind, 0.997887850301531_dbl_kind, 0.983604765547545_dbl_kind, 0.930400627107259_dbl_kind, 0.607366599944441_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,711) = (/ 0.999919065938108_dbl_kind, 0.997885196837767_dbl_kind, 0.983583626550851_dbl_kind, 0.93031833062364_dbl_kind, 0.607206728947382_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,712) = (/ 0.999918958218137_dbl_kind, 0.997882543773159_dbl_kind, 0.983562489521034_dbl_kind, 0.930236052432949_dbl_kind, 0.607047237122002_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,713) = (/ 0.999918850500295_dbl_kind, 0.997879891107657_dbl_kind, 0.98354135445733_dbl_kind, 0.930153792537359_dbl_kind, 0.60688812353494_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,714) = (/ 0.999918742784576_dbl_kind, 0.997877238841209_dbl_kind, 0.983520221358972_dbl_kind, 0.930071550939013_dbl_kind, 0.606729387255171_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,715) = (/ 0.999918635070972_dbl_kind, 0.997874586973757_dbl_kind, 0.983499090225191_dbl_kind, 0.929989327640021_dbl_kind, 0.606571027354003_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,716) = (/ 0.999918527359476_dbl_kind, 0.997871935505245_dbl_kind, 0.983477961055212_dbl_kind, 0.929907122642471_dbl_kind, 0.606413042905064_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,717) = (/ 0.999918419650081_dbl_kind, 0.997869284435611_dbl_kind, 0.983456833848256_dbl_kind, 0.929824935948415_dbl_kind, 0.606255432984298_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,718) = (/ 0.999918311942781_dbl_kind, 0.997866633764791_dbl_kind, 0.983435708603541_dbl_kind, 0.92974276755988_dbl_kind, 0.606098196669966_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,719) = (/ 0.999918204237568_dbl_kind, 0.99786398349272_dbl_kind, 0.983414585320283_dbl_kind, 0.929660617478865_dbl_kind, 0.605941333042624_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,720) = (/ 0.999918096534437_dbl_kind, 0.997861333619329_dbl_kind, 0.98339346399769_dbl_kind, 0.929578485707335_dbl_kind, 0.605784841185124_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,721) = (/ 0.99991798883338_dbl_kind, 0.997858684144547_dbl_kind, 0.983372344634972_dbl_kind, 0.929496372247232_dbl_kind, 0.605628720182616_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,722) = (/ 0.999917881134391_dbl_kind, 0.997856035068299_dbl_kind, 0.983351227231331_dbl_kind, 0.929414277100465_dbl_kind, 0.605472969122526_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,723) = (/ 0.999917773437463_dbl_kind, 0.99785338639051_dbl_kind, 0.983330111785967_dbl_kind, 0.929332200268915_dbl_kind, 0.605317587094556_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,724) = (/ 0.99991766574259_dbl_kind, 0.997850738111102_dbl_kind, 0.983308998298076_dbl_kind, 0.929250141754435_dbl_kind, 0.605162573190683_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,725) = (/ 0.999917558049765_dbl_kind, 0.997848090229992_dbl_kind, 0.983287886766852_dbl_kind, 0.929168101558849_dbl_kind, 0.605007926505144_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,726) = (/ 0.999917450358981_dbl_kind, 0.997845442747096_dbl_kind, 0.983266777191484_dbl_kind, 0.92908607968395_dbl_kind, 0.604853646134431_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,727) = (/ 0.999917342670234_dbl_kind, 0.997842795662329_dbl_kind, 0.983245669571158_dbl_kind, 0.929004076131505_dbl_kind, 0.604699731177291_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,728) = (/ 0.999917234983516_dbl_kind, 0.9978401489756_dbl_kind, 0.983224563905056_dbl_kind, 0.928922090903251_dbl_kind, 0.60454618073471_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,729) = (/ 0.999917127298822_dbl_kind, 0.997837502686821_dbl_kind, 0.983203460192359_dbl_kind, 0.928840124000896_dbl_kind, 0.604392993909917_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,730) = (/ 0.999917019616144_dbl_kind, 0.997834856795897_dbl_kind, 0.983182358432243_dbl_kind, 0.928758175426119_dbl_kind, 0.604240169808363_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,731) = (/ 0.999916911935478_dbl_kind, 0.997832211302729_dbl_kind, 0.983161258623879_dbl_kind, 0.928676245180571_dbl_kind, 0.604087707537735_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,732) = (/ 0.999916804256818_dbl_kind, 0.997829566207223_dbl_kind, 0.983140160766439_dbl_kind, 0.928594333265873_dbl_kind, 0.603935606207928_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,733) = (/ 0.999916696580157_dbl_kind, 0.997826921509276_dbl_kind, 0.983119064859088_dbl_kind, 0.928512439683619_dbl_kind, 0.603783864931057_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,734) = (/ 0.999916588905488_dbl_kind, 0.997824277208783_dbl_kind, 0.983097970900991_dbl_kind, 0.928430564435373_dbl_kind, 0.603632482821437_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,735) = (/ 0.999916481232808_dbl_kind, 0.997821633305641_dbl_kind, 0.983076878891306_dbl_kind, 0.928348707522671_dbl_kind, 0.603481458995584_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,736) = (/ 0.99991637356211_dbl_kind, 0.99781898979974_dbl_kind, 0.983055788829191_dbl_kind, 0.928266868947019_dbl_kind, 0.60333079257221_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,737) = (/ 0.999916265893388_dbl_kind, 0.997816346690971_dbl_kind, 0.983034700713802_dbl_kind, 0.928185048709897_dbl_kind, 0.603180482672214_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,738) = (/ 0.999916158226637_dbl_kind, 0.99781370397922_dbl_kind, 0.983013614544288_dbl_kind, 0.928103246812752_dbl_kind, 0.603030528418674_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,739) = (/ 0.999916050561851_dbl_kind, 0.997811061664373_dbl_kind, 0.982992530319799_dbl_kind, 0.928021463257009_dbl_kind, 0.60288092893684_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,740) = (/ 0.999915942899024_dbl_kind, 0.997808419746311_dbl_kind, 0.982971448039479_dbl_kind, 0.927939698044059_dbl_kind, 0.602731683354139_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,741) = (/ 0.999915835238152_dbl_kind, 0.997805778224915_dbl_kind, 0.982950367702473_dbl_kind, 0.927857951175265_dbl_kind, 0.602582790800153_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,742) = (/ 0.999915727579229_dbl_kind, 0.997803137100063_dbl_kind, 0.982929289307917_dbl_kind, 0.927776222651965_dbl_kind, 0.602434250406624_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,743) = (/ 0.999915619922249_dbl_kind, 0.99780049637163_dbl_kind, 0.982908212854951_dbl_kind, 0.927694512475465_dbl_kind, 0.602286061307446_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,744) = (/ 0.999915512267207_dbl_kind, 0.997797856039492_dbl_kind, 0.982887138342707_dbl_kind, 0.927612820647045_dbl_kind, 0.602138222638657_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,745) = (/ 0.999915404614099_dbl_kind, 0.997795216103517_dbl_kind, 0.982866065770318_dbl_kind, 0.927531147167954_dbl_kind, 0.601990733538433_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,746) = (/ 0.999915296962919_dbl_kind, 0.997792576563575_dbl_kind, 0.982844995136913_dbl_kind, 0.927449492039416_dbl_kind, 0.601843593147083_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,747) = (/ 0.999915189313662_dbl_kind, 0.997789937419533_dbl_kind, 0.982823926441618_dbl_kind, 0.927367855262626_dbl_kind, 0.601696800607047_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,748) = (/ 0.999915081666322_dbl_kind, 0.997787298671257_dbl_kind, 0.982802859683556_dbl_kind, 0.927286236838746_dbl_kind, 0.601550355062878_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,749) = (/ 0.999914974020896_dbl_kind, 0.997784660318607_dbl_kind, 0.982781794861848_dbl_kind, 0.927204636768917_dbl_kind, 0.601404255661254_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,750) = (/ 0.999914866377378_dbl_kind, 0.997782022361444_dbl_kind, 0.982760731975613_dbl_kind, 0.927123055054248_dbl_kind, 0.601258501550956_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,751) = (/ 0.999914758735762_dbl_kind, 0.997779384799626_dbl_kind, 0.982739671023967_dbl_kind, 0.92704149169582_dbl_kind, 0.601113091882871_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,752) = (/ 0.999914651096045_dbl_kind, 0.997776747633009_dbl_kind, 0.982718612006023_dbl_kind, 0.926959946694686_dbl_kind, 0.600968025809986_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,753) = (/ 0.999914543458222_dbl_kind, 0.997774110861447_dbl_kind, 0.982697554920894_dbl_kind, 0.926878420051873_dbl_kind, 0.600823302487378_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,754) = (/ 0.999914435822288_dbl_kind, 0.997771474484792_dbl_kind, 0.982676499767687_dbl_kind, 0.926796911768375_dbl_kind, 0.600678921072212_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,755) = (/ 0.999914328188238_dbl_kind, 0.997768838502892_dbl_kind, 0.98265544654551_dbl_kind, 0.926715421845165_dbl_kind, 0.600534880723732_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,756) = (/ 0.999914220556068_dbl_kind, 0.997766202915596_dbl_kind, 0.982634395253466_dbl_kind, 0.926633950283184_dbl_kind, 0.600391180603262_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,757) = (/ 0.999914112925773_dbl_kind, 0.997763567722749_dbl_kind, 0.982613345890657_dbl_kind, 0.926552497083344_dbl_kind, 0.60024781987419_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,758) = (/ 0.999914005297349_dbl_kind, 0.997760932924194_dbl_kind, 0.982592298456185_dbl_kind, 0.926471062246533_dbl_kind, 0.600104797701975_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,759) = (/ 0.999913897670792_dbl_kind, 0.997758298519772_dbl_kind, 0.982571252949145_dbl_kind, 0.926389645773608_dbl_kind, 0.599962113254129_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,760) = (/ 0.999913790046096_dbl_kind, 0.997755664509324_dbl_kind, 0.982550209368635_dbl_kind, 0.926308247665401_dbl_kind, 0.59981976570022_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,761) = (/ 0.999913682423259_dbl_kind, 0.997753030892685_dbl_kind, 0.982529167713746_dbl_kind, 0.926226867922715_dbl_kind, 0.599677754211862_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,762) = (/ 0.999913574802274_dbl_kind, 0.997750397669692_dbl_kind, 0.982508127983571_dbl_kind, 0.926145506546326_dbl_kind, 0.599536077962714_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,763) = (/ 0.999913467183139_dbl_kind, 0.997747764840177_dbl_kind, 0.982487090177198_dbl_kind, 0.926064163536978_dbl_kind, 0.599394736128467_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,764) = (/ 0.999913359565849_dbl_kind, 0.997745132403973_dbl_kind, 0.982466054293717_dbl_kind, 0.925982838895397_dbl_kind, 0.59925372788685_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,765) = (/ 0.999913251950401_dbl_kind, 0.997742500360909_dbl_kind, 0.98244502033221_dbl_kind, 0.925901532622275_dbl_kind, 0.599113052417612_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,766) = (/ 0.999913144336789_dbl_kind, 0.997739868710811_dbl_kind, 0.982423988291764_dbl_kind, 0.925820244718275_dbl_kind, 0.598972708902526_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,767) = (/ 0.99991303672501_dbl_kind, 0.997737237453505_dbl_kind, 0.982402958171458_dbl_kind, 0.925738975184038_dbl_kind, 0.598832696525377_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,768) = (/ 0.999912929115061_dbl_kind, 0.997734606588815_dbl_kind, 0.982381929970374_dbl_kind, 0.925657724020176_dbl_kind, 0.598693014471964_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,769) = (/ 0.999912821506937_dbl_kind, 0.997731976116564_dbl_kind, 0.982360903687588_dbl_kind, 0.925576491227273_dbl_kind, 0.598553661930089_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,770) = (/ 0.999912713900634_dbl_kind, 0.99772934603657_dbl_kind, 0.982339879322179_dbl_kind, 0.925495276805886_dbl_kind, 0.598414638089547_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,771) = (/ 0.999912606296149_dbl_kind, 0.997726716348653_dbl_kind, 0.98231885687322_dbl_kind, 0.925414080756544_dbl_kind, 0.598275942142137_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,772) = (/ 0.999912498693478_dbl_kind, 0.997724087052627_dbl_kind, 0.982297836339783_dbl_kind, 0.925332903079752_dbl_kind, 0.59813757328164_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,773) = (/ 0.999912391092617_dbl_kind, 0.997721458148308_dbl_kind, 0.982276817720941_dbl_kind, 0.925251743775987_dbl_kind, 0.597999530703825_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,774) = (/ 0.999912283493562_dbl_kind, 0.997718829635508_dbl_kind, 0.982255801015763_dbl_kind, 0.925170602845697_dbl_kind, 0.597861813606433_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,775) = (/ 0.999912175896311_dbl_kind, 0.997716201514038_dbl_kind, 0.982234786223318_dbl_kind, 0.925089480289305_dbl_kind, 0.597724421189185_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,776) = (/ 0.999912068300859_dbl_kind, 0.997713573783707_dbl_kind, 0.982213773342672_dbl_kind, 0.925008376107208_dbl_kind, 0.597587352653763_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,777) = (/ 0.999911960707203_dbl_kind, 0.997710946444322_dbl_kind, 0.98219276237289_dbl_kind, 0.924927290299775_dbl_kind, 0.597450607203819_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,778) = (/ 0.999911853115339_dbl_kind, 0.997708319495687_dbl_kind, 0.982171753313036_dbl_kind, 0.924846222867351_dbl_kind, 0.597314184044957_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,779) = (/ 0.999911745525265_dbl_kind, 0.997705692937609_dbl_kind, 0.982150746162173_dbl_kind, 0.924765173810249_dbl_kind, 0.597178082384736_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,780) = (/ 0.999911637936977_dbl_kind, 0.997703066769889_dbl_kind, 0.982129740919361_dbl_kind, 0.924684143128763_dbl_kind, 0.597042301432662_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,781) = (/ 0.999911530350471_dbl_kind, 0.997700440992325_dbl_kind, 0.98210873758366_dbl_kind, 0.924603130823154_dbl_kind, 0.596906840400182_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,782) = (/ 0.999911422765745_dbl_kind, 0.997697815604718_dbl_kind, 0.982087736154128_dbl_kind, 0.924522136893661_dbl_kind, 0.596771698500686_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,783) = (/ 0.999911315182794_dbl_kind, 0.997695190606865_dbl_kind, 0.982066736629823_dbl_kind, 0.924441161340495_dbl_kind, 0.596636874949485_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,784) = (/ 0.999911207601616_dbl_kind, 0.997692565998559_dbl_kind, 0.982045739009801_dbl_kind, 0.924360204163841_dbl_kind, 0.596502368963832_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,785) = (/ 0.999911100022208_dbl_kind, 0.997689941779597_dbl_kind, 0.982024743293114_dbl_kind, 0.924279265363859_dbl_kind, 0.59636817976289_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,786) = (/ 0.999910992444567_dbl_kind, 0.997687317949769_dbl_kind, 0.982003749478819_dbl_kind, 0.924198344940682_dbl_kind, 0.596234306567746_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,787) = (/ 0.99991088486869_dbl_kind, 0.997684694508866_dbl_kind, 0.981982757565966_dbl_kind, 0.924117442894417_dbl_kind, 0.596100748601397_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,788) = (/ 0.999910777294573_dbl_kind, 0.997682071456676_dbl_kind, 0.981961767553608_dbl_kind, 0.924036559225147_dbl_kind, 0.59596750508875_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,789) = (/ 0.999910669722215_dbl_kind, 0.997679448792988_dbl_kind, 0.981940779440794_dbl_kind, 0.923955693932927_dbl_kind, 0.595834575256612_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,790) = (/ 0.999910562151611_dbl_kind, 0.997676826517587_dbl_kind, 0.981919793226574_dbl_kind, 0.923874847017787_dbl_kind, 0.595701958333689_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,791) = (/ 0.999910454582759_dbl_kind, 0.997674204630256_dbl_kind, 0.981898808909996_dbl_kind, 0.923794018479732_dbl_kind, 0.59556965355058_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,792) = (/ 0.999910347015657_dbl_kind, 0.997671583130778_dbl_kind, 0.981877826490107_dbl_kind, 0.923713208318742_dbl_kind, 0.595437660139773_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,793) = (/ 0.999910239450301_dbl_kind, 0.997668962018937_dbl_kind, 0.981856845965955_dbl_kind, 0.92363241653477_dbl_kind, 0.595305977335636_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,794) = (/ 0.999910131886688_dbl_kind, 0.997666341294508_dbl_kind, 0.981835867336583_dbl_kind, 0.923551643127744_dbl_kind, 0.595174604374421_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,795) = (/ 0.999910024324818_dbl_kind, 0.997663720957272_dbl_kind, 0.981814890601038_dbl_kind, 0.923470888097569_dbl_kind, 0.595043540494248_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,796) = (/ 0.999909916764686_dbl_kind, 0.997661101007005_dbl_kind, 0.981793915758364_dbl_kind, 0.923390151444121_dbl_kind, 0.59491278493511_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,797) = (/ 0.999909809206289_dbl_kind, 0.997658481443483_dbl_kind, 0.981772942807603_dbl_kind, 0.923309433167256_dbl_kind, 0.594782336938862_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,798) = (/ 0.999909701649627_dbl_kind, 0.997655862266478_dbl_kind, 0.981751971747798_dbl_kind, 0.923228733266798_dbl_kind, 0.594652195749221_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,799) = (/ 0.999909594094695_dbl_kind, 0.997653243475765_dbl_kind, 0.981731002577991_dbl_kind, 0.923148051742554_dbl_kind, 0.594522360611758_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,800) = (/ 0.999909486541491_dbl_kind, 0.997650625071111_dbl_kind, 0.981710035297224_dbl_kind, 0.923067388594301_dbl_kind, 0.594392830773887_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,801) = (/ 0.999909378990014_dbl_kind, 0.997648007052289_dbl_kind, 0.981689069904536_dbl_kind, 0.922986743821791_dbl_kind, 0.594263605484881_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,802) = (/ 0.99990927144026_dbl_kind, 0.997645389419066_dbl_kind, 0.981668106398968_dbl_kind, 0.922906117424755_dbl_kind, 0.594134683995843_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,803) = (/ 0.999909163892227_dbl_kind, 0.997642772171208_dbl_kind, 0.981647144779559_dbl_kind, 0.922825509402897_dbl_kind, 0.594006065559718_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,804) = (/ 0.999909056345913_dbl_kind, 0.997640155308482_dbl_kind, 0.981626185045349_dbl_kind, 0.922744919755898_dbl_kind, 0.59387774943128_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,805) = (/ 0.999908948801317_dbl_kind, 0.99763753883065_dbl_kind, 0.981605227195375_dbl_kind, 0.922664348483412_dbl_kind, 0.593749734867127_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,806) = (/ 0.999908841258435_dbl_kind, 0.997634922737477_dbl_kind, 0.981584271228676_dbl_kind, 0.92258379558507_dbl_kind, 0.593622021125687_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,807) = (/ 0.999908733717265_dbl_kind, 0.997632307028722_dbl_kind, 0.98156331714429_dbl_kind, 0.922503261060482_dbl_kind, 0.593494607467197_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,808) = (/ 0.999908626177806_dbl_kind, 0.997629691704148_dbl_kind, 0.981542364941254_dbl_kind, 0.922422744909228_dbl_kind, 0.593367493153714_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,809) = (/ 0.999908518640054_dbl_kind, 0.997627076763511_dbl_kind, 0.981521414618604_dbl_kind, 0.922342247130869_dbl_kind, 0.593240677449101_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,810) = (/ 0.999908411104009_dbl_kind, 0.997624462206571_dbl_kind, 0.981500466175377_dbl_kind, 0.92226176772494_dbl_kind, 0.593114159619027_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,811) = (/ 0.999908303569667_dbl_kind, 0.997621848033083_dbl_kind, 0.98147951961061_dbl_kind, 0.922181306690951_dbl_kind, 0.592987938930958_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,812) = (/ 0.999908196037028_dbl_kind, 0.997619234242802_dbl_kind, 0.98145857492334_dbl_kind, 0.922100864028391_dbl_kind, 0.592862014654156_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,813) = (/ 0.999908088506088_dbl_kind, 0.997616620835483_dbl_kind, 0.981437632112601_dbl_kind, 0.922020439736723_dbl_kind, 0.592736386059675_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,814) = (/ 0.999907980976848_dbl_kind, 0.997614007810877_dbl_kind, 0.98141669117743_dbl_kind, 0.921940033815389_dbl_kind, 0.592611052420355_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,815) = (/ 0.999907873449303_dbl_kind, 0.997611395168738_dbl_kind, 0.981395752116863_dbl_kind, 0.921859646263804_dbl_kind, 0.592486013010816_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,816) = (/ 0.999907765923452_dbl_kind, 0.997608782908814_dbl_kind, 0.981374814929934_dbl_kind, 0.921779277081363_dbl_kind, 0.592361267107458_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,817) = (/ 0.999907658399295_dbl_kind, 0.997606171030855_dbl_kind, 0.981353879615679_dbl_kind, 0.921698926267437_dbl_kind, 0.592236813988453_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,818) = (/ 0.999907550876828_dbl_kind, 0.997603559534609_dbl_kind, 0.981332946173135_dbl_kind, 0.921618593821372_dbl_kind, 0.59211265293374_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,819) = (/ 0.99990744335605_dbl_kind, 0.997600948419824_dbl_kind, 0.981312014601337_dbl_kind, 0.921538279742493_dbl_kind, 0.591988783225027_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,820) = (/ 0.99990733583696_dbl_kind, 0.997598337686244_dbl_kind, 0.981291084899318_dbl_kind, 0.921457984030102_dbl_kind, 0.591865204145774_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,821) = (/ 0.999907228319555_dbl_kind, 0.997595727333614_dbl_kind, 0.981270157066118_dbl_kind, 0.921377706683476_dbl_kind, 0.591741914981205_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,822) = (/ 0.999907120803835_dbl_kind, 0.997593117361679_dbl_kind, 0.981249231100769_dbl_kind, 0.921297447701871_dbl_kind, 0.591618915018292_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,823) = (/ 0.999907013289797_dbl_kind, 0.997590507770179_dbl_kind, 0.981228307002308_dbl_kind, 0.921217207084521_dbl_kind, 0.59149620354575_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,824) = (/ 0.999906905777439_dbl_kind, 0.997587898558858_dbl_kind, 0.981207384769771_dbl_kind, 0.921136984830636_dbl_kind, 0.591373779854044_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,825) = (/ 0.999906798266761_dbl_kind, 0.997585289727455_dbl_kind, 0.981186464402194_dbl_kind, 0.921056780939405_dbl_kind, 0.591251643235371_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,826) = (/ 0.999906690757762_dbl_kind, 0.99758268127571_dbl_kind, 0.981165545898614_dbl_kind, 0.920976595409992_dbl_kind, 0.591129792983665_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,827) = (/ 0.999906583250438_dbl_kind, 0.997580073203361_dbl_kind, 0.981144629258068_dbl_kind, 0.920896428241541_dbl_kind, 0.59100822839459_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,828) = (/ 0.99990647574479_dbl_kind, 0.997577465510144_dbl_kind, 0.981123714479591_dbl_kind, 0.920816279433175_dbl_kind, 0.590886948765537_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,829) = (/ 0.999906368240815_dbl_kind, 0.997574858195799_dbl_kind, 0.981102801562223_dbl_kind, 0.920736148983992_dbl_kind, 0.590765953395612_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,830) = (/ 0.999906260738513_dbl_kind, 0.997572251260057_dbl_kind, 0.981081890504999_dbl_kind, 0.92065603689307_dbl_kind, 0.590645241585645_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,831) = (/ 0.999906153237881_dbl_kind, 0.997569644702655_dbl_kind, 0.98106098130696_dbl_kind, 0.920575943159463_dbl_kind, 0.590524812638177_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,832) = (/ 0.999906045738919_dbl_kind, 0.997567038523326_dbl_kind, 0.981040073967141_dbl_kind, 0.920495867782207_dbl_kind, 0.590404665857457_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,833) = (/ 0.999905938241625_dbl_kind, 0.997564432721802_dbl_kind, 0.981019168484584_dbl_kind, 0.920415810760313_dbl_kind, 0.59028480054944_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,834) = (/ 0.999905830745999_dbl_kind, 0.997561827297815_dbl_kind, 0.980998264858327_dbl_kind, 0.920335772092772_dbl_kind, 0.59016521602178_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,835) = (/ 0.999905723252038_dbl_kind, 0.997559222251096_dbl_kind, 0.98097736308741_dbl_kind, 0.920255751778554_dbl_kind, 0.590045911583824_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,836) = (/ 0.999905615759741_dbl_kind, 0.997556617581374_dbl_kind, 0.980956463170874_dbl_kind, 0.920175749816605_dbl_kind, 0.589926886546621_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,837) = (/ 0.999905508269108_dbl_kind, 0.997554013288378_dbl_kind, 0.98093556510776_dbl_kind, 0.920095766205855_dbl_kind, 0.589808140222899_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,838) = (/ 0.999905400780138_dbl_kind, 0.997551409371837_dbl_kind, 0.980914668897109_dbl_kind, 0.920015800945207_dbl_kind, 0.58968967192707_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,839) = (/ 0.999905293292829_dbl_kind, 0.997548805831477_dbl_kind, 0.980893774537964_dbl_kind, 0.919935854033547_dbl_kind, 0.589571480975237_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,840) = (/ 0.99990518580718_dbl_kind, 0.997546202667024_dbl_kind, 0.980872882029368_dbl_kind, 0.919855925469738_dbl_kind, 0.589453566685163_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,841) = (/ 0.99990507832319_dbl_kind, 0.997543599878205_dbl_kind, 0.980851991370363_dbl_kind, 0.919776015252626_dbl_kind, 0.589335928376295_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,842) = (/ 0.999904970840858_dbl_kind, 0.997540997464744_dbl_kind, 0.980831102559997_dbl_kind, 0.91969612338103_dbl_kind, 0.589218565369741_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,843) = (/ 0.999904863360184_dbl_kind, 0.997538395426364_dbl_kind, 0.980810215597311_dbl_kind, 0.919616249853753_dbl_kind, 0.589101476988275_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,844) = (/ 0.999904755881165_dbl_kind, 0.997535793762789_dbl_kind, 0.980789330481355_dbl_kind, 0.919536394669577_dbl_kind, 0.588984662556332_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,845) = (/ 0.999904648403802_dbl_kind, 0.997533192473742_dbl_kind, 0.980768447211173_dbl_kind, 0.919456557827263_dbl_kind, 0.5888681214_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,846) = (/ 0.999904540928094_dbl_kind, 0.997530591558943_dbl_kind, 0.980747565785814_dbl_kind, 0.919376739325553_dbl_kind, 0.588751852847023_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,847) = (/ 0.999904433454039_dbl_kind, 0.997527991018113_dbl_kind, 0.980726686204325_dbl_kind, 0.919296939163166_dbl_kind, 0.588635856226783_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,848) = (/ 0.999904325981637_dbl_kind, 0.997525390850974_dbl_kind, 0.980705808465757_dbl_kind, 0.919217157338804_dbl_kind, 0.588520130870319_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,849) = (/ 0.999904218510886_dbl_kind, 0.997522791057243_dbl_kind, 0.98068493256916_dbl_kind, 0.919137393851148_dbl_kind, 0.588404676110299_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,850) = (/ 0.999904111041786_dbl_kind, 0.997520191636639_dbl_kind, 0.980664058513585_dbl_kind, 0.919057648698859_dbl_kind, 0.588289491281033_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,851) = (/ 0.999904003574337_dbl_kind, 0.997517592588881_dbl_kind, 0.980643186298085_dbl_kind, 0.918977921880578_dbl_kind, 0.58817457571846_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,852) = (/ 0.999903896108537_dbl_kind, 0.997514993913685_dbl_kind, 0.980622315921713_dbl_kind, 0.918898213394929_dbl_kind, 0.588059928760146_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,853) = (/ 0.999903788644386_dbl_kind, 0.997512395610768_dbl_kind, 0.980601447383524_dbl_kind, 0.918818523240513_dbl_kind, 0.587945549745285_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,854) = (/ 0.999903681181883_dbl_kind, 0.997509797679846_dbl_kind, 0.980580580682573_dbl_kind, 0.918738851415914_dbl_kind, 0.587831438014687_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,855) = (/ 0.999903573721028_dbl_kind, 0.997507200120634_dbl_kind, 0.980559715817916_dbl_kind, 0.918659197919694_dbl_kind, 0.587717592910781_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,856) = (/ 0.999903466261819_dbl_kind, 0.997504602932848_dbl_kind, 0.980538852788612_dbl_kind, 0.918579562750401_dbl_kind, 0.587604013777607_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,857) = (/ 0.999903358804258_dbl_kind, 0.9975020061162_dbl_kind, 0.980517991593719_dbl_kind, 0.918499945906559_dbl_kind, 0.587490699960814_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,858) = (/ 0.999903251348341_dbl_kind, 0.997499409670403_dbl_kind, 0.980497132232297_dbl_kind, 0.918420347386677_dbl_kind, 0.587377650807653_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,859) = (/ 0.99990314389407_dbl_kind, 0.997496813595173_dbl_kind, 0.980476274703409_dbl_kind, 0.918340767189241_dbl_kind, 0.587264865666982_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,860) = (/ 0.999903036441443_dbl_kind, 0.997494217890218_dbl_kind, 0.980455419006116_dbl_kind, 0.918261205312723_dbl_kind, 0.587152343889251_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,861) = (/ 0.99990292899046_dbl_kind, 0.997491622555253_dbl_kind, 0.980434565139483_dbl_kind, 0.918181661755573_dbl_kind, 0.587040084826506_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,862) = (/ 0.999902821541121_dbl_kind, 0.997489027589987_dbl_kind, 0.980413713102575_dbl_kind, 0.918102136516224_dbl_kind, 0.586928087832377_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,863) = (/ 0.999902714093425_dbl_kind, 0.997486432994131_dbl_kind, 0.980392862894459_dbl_kind, 0.918022629593091_dbl_kind, 0.586816352262087_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,864) = (/ 0.999902606647371_dbl_kind, 0.997483838767396_dbl_kind, 0.9803720145142_dbl_kind, 0.91794314098457_dbl_kind, 0.586704877472435_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,865) = (/ 0.99990249920296_dbl_kind, 0.99748124490949_dbl_kind, 0.980351167960872_dbl_kind, 0.91786367068904_dbl_kind, 0.5865936628218_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,866) = (/ 0.99990239176019_dbl_kind, 0.997478651420121_dbl_kind, 0.980330323233543_dbl_kind, 0.917784218704859_dbl_kind, 0.586482707670137_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,867) = (/ 0.999902284319062_dbl_kind, 0.997476058299_dbl_kind, 0.980309480331285_dbl_kind, 0.917704785030373_dbl_kind, 0.58637201137897_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,868) = (/ 0.999902176879575_dbl_kind, 0.997473465545833_dbl_kind, 0.980288639253173_dbl_kind, 0.917625369663906_dbl_kind, 0.586261573311388_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,869) = (/ 0.999902069441728_dbl_kind, 0.997470873160328_dbl_kind, 0.980267799998282_dbl_kind, 0.917545972603762_dbl_kind, 0.586151392832047_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,870) = (/ 0.999901962005523_dbl_kind, 0.997468281142192_dbl_kind, 0.980246962565687_dbl_kind, 0.917466593848235_dbl_kind, 0.586041469307157_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,871) = (/ 0.999901854570957_dbl_kind, 0.997465689491132_dbl_kind, 0.980226126954469_dbl_kind, 0.917387233395595_dbl_kind, 0.58593180210449_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,872) = (/ 0.99990174713803_dbl_kind, 0.997463098206853_dbl_kind, 0.980205293163707_dbl_kind, 0.917307891244098_dbl_kind, 0.585822390593363_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,873) = (/ 0.999901639706745_dbl_kind, 0.99746050728906_dbl_kind, 0.980184461192481_dbl_kind, 0.917228567391983_dbl_kind, 0.585713234144647_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,874) = (/ 0.999901532277098_dbl_kind, 0.99745791673746_dbl_kind, 0.980163631039875_dbl_kind, 0.917149261837469_dbl_kind, 0.585604332130757_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,875) = (/ 0.99990142484909_dbl_kind, 0.997455326551757_dbl_kind, 0.980142802704974_dbl_kind, 0.917069974578761_dbl_kind, 0.585495683925645_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,876) = (/ 0.999901317422721_dbl_kind, 0.997452736731655_dbl_kind, 0.980121976186866_dbl_kind, 0.916990705614047_dbl_kind, 0.585387288904802_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,877) = (/ 0.999901209997991_dbl_kind, 0.997450147276859_dbl_kind, 0.980101151484636_dbl_kind, 0.916911454941497_dbl_kind, 0.585279146445253_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,878) = (/ 0.9999011025749_dbl_kind, 0.997447558187071_dbl_kind, 0.980080328597375_dbl_kind, 0.916832222559266_dbl_kind, 0.585171255925555_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,879) = (/ 0.999900995153447_dbl_kind, 0.997444969461994_dbl_kind, 0.980059507524176_dbl_kind, 0.916753008465492_dbl_kind, 0.585063616725786_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,880) = (/ 0.999900887733633_dbl_kind, 0.997442381101333_dbl_kind, 0.980038688264132_dbl_kind, 0.916673812658295_dbl_kind, 0.584956228227555_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,881) = (/ 0.999900780315457_dbl_kind, 0.99743979310479_dbl_kind, 0.980017870816338_dbl_kind, 0.91659463513578_dbl_kind, 0.584849089813981_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,882) = (/ 0.99990067289892_dbl_kind, 0.997437205472066_dbl_kind, 0.979997055179891_dbl_kind, 0.916515475896039_dbl_kind, 0.584742200869703_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,883) = (/ 0.99990056548402_dbl_kind, 0.997434618202863_dbl_kind, 0.97997624135389_dbl_kind, 0.916436334937143_dbl_kind, 0.584635560780873_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,884) = (/ 0.999900458070759_dbl_kind, 0.997432031296883_dbl_kind, 0.979955429337437_dbl_kind, 0.916357212257152_dbl_kind, 0.584529168935151_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,885) = (/ 0.999900350659136_dbl_kind, 0.997429444753827_dbl_kind, 0.979934619129633_dbl_kind, 0.916278107854105_dbl_kind, 0.584423024721699_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,886) = (/ 0.999900243249152_dbl_kind, 0.997426858573397_dbl_kind, 0.979913810729584_dbl_kind, 0.91619902172603_dbl_kind, 0.584317127531181_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,887) = (/ 0.999900135840805_dbl_kind, 0.997424272755292_dbl_kind, 0.979893004136397_dbl_kind, 0.916119953870936_dbl_kind, 0.584211476755762_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,888) = (/ 0.999900028434096_dbl_kind, 0.997421687299213_dbl_kind, 0.97987219934918_dbl_kind, 0.916040904286822_dbl_kind, 0.584106071789096_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,889) = (/ 0.999899921029027_dbl_kind, 0.997419102204861_dbl_kind, 0.979851396367044_dbl_kind, 0.915961872971664_dbl_kind, 0.584000912026334_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,890) = (/ 0.999899813625596_dbl_kind, 0.997416517471934_dbl_kind, 0.979830595189102_dbl_kind, 0.91588285992343_dbl_kind, 0.583895996864109_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,891) = (/ 0.999899706223803_dbl_kind, 0.997413933100134_dbl_kind, 0.979809795814468_dbl_kind, 0.91580386514007_dbl_kind, 0.583791325700539_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,892) = (/ 0.999899598823649_dbl_kind, 0.997411349089159_dbl_kind, 0.97978899824226_dbl_kind, 0.915724888619517_dbl_kind, 0.583686897935221_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,893) = (/ 0.999899491425133_dbl_kind, 0.997408765438708_dbl_kind, 0.979768202471597_dbl_kind, 0.915645930359695_dbl_kind, 0.583582712969234_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,894) = (/ 0.999899384028257_dbl_kind, 0.997406182148479_dbl_kind, 0.979747408501599_dbl_kind, 0.915566990358507_dbl_kind, 0.583478770205121_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,895) = (/ 0.99989927663302_dbl_kind, 0.997403599218174_dbl_kind, 0.979726616331391_dbl_kind, 0.915488068613846_dbl_kind, 0.583375069046905_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,896) = (/ 0.999899169239422_dbl_kind, 0.997401016647489_dbl_kind, 0.979705825960098_dbl_kind, 0.915409165123589_dbl_kind, 0.583271608900066_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,897) = (/ 0.999899061847464_dbl_kind, 0.997398434436123_dbl_kind, 0.979685037386848_dbl_kind, 0.915330279885598_dbl_kind, 0.583168389171551_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,898) = (/ 0.999898954457145_dbl_kind, 0.997395852583774_dbl_kind, 0.979664250610771_dbl_kind, 0.915251412897723_dbl_kind, 0.583065409269769_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,899) = (/ 0.999898847068466_dbl_kind, 0.997393271090141_dbl_kind, 0.979643465630999_dbl_kind, 0.915172564157799_dbl_kind, 0.582962668604578_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,900) = (/ 0.999898739681427_dbl_kind, 0.997390689954921_dbl_kind, 0.979622682446665_dbl_kind, 0.915093733663645_dbl_kind, 0.582860166587296_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,901) = (/ 0.999898632296029_dbl_kind, 0.997388109177813_dbl_kind, 0.979601901056908_dbl_kind, 0.915014921413069_dbl_kind, 0.582757902630684_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,902) = (/ 0.999898524912272_dbl_kind, 0.997385528758513_dbl_kind, 0.979581121460868_dbl_kind, 0.914936127403864_dbl_kind, 0.582655876148953_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,903) = (/ 0.999898417530156_dbl_kind, 0.99738294869672_dbl_kind, 0.979560343657684_dbl_kind, 0.914857351633812_dbl_kind, 0.582554086557754_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,904) = (/ 0.999898310149682_dbl_kind, 0.99738036899213_dbl_kind, 0.979539567646501_dbl_kind, 0.914778594100676_dbl_kind, 0.582452533274176_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,905) = (/ 0.999898202770848_dbl_kind, 0.997377789644441_dbl_kind, 0.979518793426466_dbl_kind, 0.914699854802212_dbl_kind, 0.582351215716746_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,906) = (/ 0.999898095393657_dbl_kind, 0.99737521065335_dbl_kind, 0.979498020996727_dbl_kind, 0.91462113373616_dbl_kind, 0.582250133305422_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,907) = (/ 0.999897988018108_dbl_kind, 0.997372632018556_dbl_kind, 0.979477250356436_dbl_kind, 0.914542430900244_dbl_kind, 0.582149285461589_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,908) = (/ 0.999897880644202_dbl_kind, 0.997370053739753_dbl_kind, 0.979456481504745_dbl_kind, 0.914463746292181_dbl_kind, 0.582048671608062_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,909) = (/ 0.999897773271939_dbl_kind, 0.99736747581664_dbl_kind, 0.979435714440812_dbl_kind, 0.914385079909672_dbl_kind, 0.581948291169069_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,910) = (/ 0.999897665901321_dbl_kind, 0.997364898248914_dbl_kind, 0.979414949163794_dbl_kind, 0.914306431750405_dbl_kind, 0.581848143570269_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,911) = (/ 0.999897558532346_dbl_kind, 0.997362321036271_dbl_kind, 0.979394185672854_dbl_kind, 0.914227801812057_dbl_kind, 0.581748228238727_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,912) = (/ 0.999897451165015_dbl_kind, 0.997359744178409_dbl_kind, 0.979373423967154_dbl_kind, 0.91414919009229_dbl_kind, 0.581648544602922_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,913) = (/ 0.999897343799329_dbl_kind, 0.997357167675024_dbl_kind, 0.979352664045862_dbl_kind, 0.914070596588757_dbl_kind, 0.581549092092748_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,914) = (/ 0.999897236435289_dbl_kind, 0.997354591525813_dbl_kind, 0.979331905908145_dbl_kind, 0.913992021299096_dbl_kind, 0.581449870139492_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,915) = (/ 0.999897129072895_dbl_kind, 0.997352015730473_dbl_kind, 0.979311149553176_dbl_kind, 0.913913464220936_dbl_kind, 0.581350878175857_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,916) = (/ 0.999897021712147_dbl_kind, 0.997349440288701_dbl_kind, 0.979290394980129_dbl_kind, 0.91383492535189_dbl_kind, 0.581252115635934_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,917) = (/ 0.999896914353045_dbl_kind, 0.997346865200194_dbl_kind, 0.97926964218818_dbl_kind, 0.913756404689562_dbl_kind, 0.581153581955217_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,918) = (/ 0.999896806995592_dbl_kind, 0.997344290464648_dbl_kind, 0.97924889117651_dbl_kind, 0.913677902231544_dbl_kind, 0.581055276570586_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,919) = (/ 0.999896699639786_dbl_kind, 0.997341716081759_dbl_kind, 0.9792281419443_dbl_kind, 0.913599417975415_dbl_kind, 0.580957198920314_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,920) = (/ 0.999896592285628_dbl_kind, 0.997339142051227_dbl_kind, 0.979207394490736_dbl_kind, 0.913520951918746_dbl_kind, 0.580859348444061_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,921) = (/ 0.99989648493312_dbl_kind, 0.997336568372747_dbl_kind, 0.979186648815005_dbl_kind, 0.913442504059094_dbl_kind, 0.580761724582866_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,922) = (/ 0.999896377582261_dbl_kind, 0.997333995046016_dbl_kind, 0.979165904916298_dbl_kind, 0.913364074394002_dbl_kind, 0.580664326779149_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,923) = (/ 0.999896270233052_dbl_kind, 0.997331422070732_dbl_kind, 0.979145162793808_dbl_kind, 0.913285662921009_dbl_kind, 0.580567154476706_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,924) = (/ 0.999896162885494_dbl_kind, 0.997328849446591_dbl_kind, 0.979124422446731_dbl_kind, 0.913207269637637_dbl_kind, 0.580470207120704_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,925) = (/ 0.999896055539588_dbl_kind, 0.997326277173292_dbl_kind, 0.979103683874266_dbl_kind, 0.913128894541398_dbl_kind, 0.580373484157686_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,926) = (/ 0.999895948195333_dbl_kind, 0.99732370525053_dbl_kind, 0.979082947075615_dbl_kind, 0.913050537629798_dbl_kind, 0.580276985035552_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,927) = (/ 0.999895840852731_dbl_kind, 0.997321133678005_dbl_kind, 0.979062212049983_dbl_kind, 0.912972198900325_dbl_kind, 0.580180709203572_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,928) = (/ 0.999895733511783_dbl_kind, 0.997318562455413_dbl_kind, 0.979041478796578_dbl_kind, 0.912893878350464_dbl_kind, 0.580084656112373_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,929) = (/ 0.999895626172489_dbl_kind, 0.99731599158245_dbl_kind, 0.97902074731461_dbl_kind, 0.912815575977685_dbl_kind, 0.579988825213941_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,930) = (/ 0.999895518834849_dbl_kind, 0.997313421058817_dbl_kind, 0.979000017603292_dbl_kind, 0.912737291779449_dbl_kind, 0.579893215961615_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,931) = (/ 0.999895411498865_dbl_kind, 0.997310850884211_dbl_kind, 0.97897928966184_dbl_kind, 0.912659025753204_dbl_kind, 0.579797827810083_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,932) = (/ 0.999895304164537_dbl_kind, 0.997308281058329_dbl_kind, 0.978958563489476_dbl_kind, 0.912580777896396_dbl_kind, 0.579702660215379_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,933) = (/ 0.999895196831865_dbl_kind, 0.997305711580871_dbl_kind, 0.97893783908542_dbl_kind, 0.912502548206451_dbl_kind, 0.579607712634888_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,934) = (/ 0.999895089500851_dbl_kind, 0.997303142451532_dbl_kind, 0.978917116448897_dbl_kind, 0.912424336680793_dbl_kind, 0.57951298452733_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,935) = (/ 0.999894982171496_dbl_kind, 0.997300573670014_dbl_kind, 0.978896395579138_dbl_kind, 0.912346143316833_dbl_kind, 0.579418475352763_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,936) = (/ 0.999894874843799_dbl_kind, 0.997298005236015_dbl_kind, 0.978875676475374_dbl_kind, 0.912267968111973_dbl_kind, 0.579324184572582_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,937) = (/ 0.999894767517762_dbl_kind, 0.997295437149232_dbl_kind, 0.978854959136838_dbl_kind, 0.912189811063606_dbl_kind, 0.579230111649514_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,938) = (/ 0.999894660193386_dbl_kind, 0.997292869409366_dbl_kind, 0.978834243562769_dbl_kind, 0.912111672169115_dbl_kind, 0.579136256047613_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,939) = (/ 0.999894552870671_dbl_kind, 0.997290302016115_dbl_kind, 0.978813529752407_dbl_kind, 0.912033551425875_dbl_kind, 0.579042617232259_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,940) = (/ 0.999894445549618_dbl_kind, 0.997287734969179_dbl_kind, 0.978792817704998_dbl_kind, 0.911955448831251_dbl_kind, 0.578949194670152_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,941) = (/ 0.999894338230228_dbl_kind, 0.997285168268258_dbl_kind, 0.978772107419787_dbl_kind, 0.911877364382599_dbl_kind, 0.578855987829316_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,942) = (/ 0.999894230912503_dbl_kind, 0.997282601913051_dbl_kind, 0.978751398896025_dbl_kind, 0.911799298077269_dbl_kind, 0.578762996179085_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,943) = (/ 0.999894123596441_dbl_kind, 0.99728003590326_dbl_kind, 0.978730692132967_dbl_kind, 0.911721249912597_dbl_kind, 0.578670219190114_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,944) = (/ 0.999894016282045_dbl_kind, 0.997277470238583_dbl_kind, 0.978709987129868_dbl_kind, 0.911643219885915_dbl_kind, 0.57857765633436_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,945) = (/ 0.999893908969316_dbl_kind, 0.997274904918721_dbl_kind, 0.97868928388599_dbl_kind, 0.911565207994545_dbl_kind, 0.578485307085092_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,946) = (/ 0.999893801658254_dbl_kind, 0.997272339943377_dbl_kind, 0.978668582400594_dbl_kind, 0.911487214235802_dbl_kind, 0.578393170916881_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,947) = (/ 0.99989369434886_dbl_kind, 0.997269775312249_dbl_kind, 0.978647882672949_dbl_kind, 0.91140923860699_dbl_kind, 0.578301247305598_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,948) = (/ 0.999893587041135_dbl_kind, 0.99726721102504_dbl_kind, 0.978627184702323_dbl_kind, 0.911331281105408_dbl_kind, 0.578209535728416_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,949) = (/ 0.99989347973508_dbl_kind, 0.997264647081452_dbl_kind, 0.97860648848799_dbl_kind, 0.911253341728345_dbl_kind, 0.578118035663794_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,950) = (/ 0.999893372430696_dbl_kind, 0.997262083481186_dbl_kind, 0.978585794029227_dbl_kind, 0.911175420473083_dbl_kind, 0.578026746591495_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,951) = (/ 0.999893265127984_dbl_kind, 0.997259520223945_dbl_kind, 0.978565101325314_dbl_kind, 0.911097517336899_dbl_kind, 0.577935667992558_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,952) = (/ 0.999893157826944_dbl_kind, 0.997256957309431_dbl_kind, 0.978544410375533_dbl_kind, 0.911019632317056_dbl_kind, 0.577844799349316_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,953) = (/ 0.999893050527578_dbl_kind, 0.997254394737346_dbl_kind, 0.978523721179173_dbl_kind, 0.910941765410815_dbl_kind, 0.577754140145383_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,954) = (/ 0.999892943229886_dbl_kind, 0.997251832507395_dbl_kind, 0.978503033735522_dbl_kind, 0.910863916615429_dbl_kind, 0.577663689865649_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,955) = (/ 0.99989283593387_dbl_kind, 0.99724927061928_dbl_kind, 0.978482348043876_dbl_kind, 0.910786085928143_dbl_kind, 0.577573447996285_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,956) = (/ 0.999892728639531_dbl_kind, 0.997246709072705_dbl_kind, 0.97846166410353_dbl_kind, 0.910708273346193_dbl_kind, 0.577483414024735_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,957) = (/ 0.99989262134687_dbl_kind, 0.997244147867374_dbl_kind, 0.978440981913785_dbl_kind, 0.910630478866811_dbl_kind, 0.577393587439713_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,958) = (/ 0.999892514055887_dbl_kind, 0.997241587002991_dbl_kind, 0.978420301473945_dbl_kind, 0.910552702487221_dbl_kind, 0.577303967731197_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,959) = (/ 0.999892406766583_dbl_kind, 0.997239026479262_dbl_kind, 0.978399622783318_dbl_kind, 0.910474944204642_dbl_kind, 0.577214554390438_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,960) = (/ 0.99989229947896_dbl_kind, 0.997236466295891_dbl_kind, 0.978378945841216_dbl_kind, 0.910397204016283_dbl_kind, 0.577125346909938_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,961) = (/ 0.999892192193019_dbl_kind, 0.997233906452584_dbl_kind, 0.978358270646952_dbl_kind, 0.91031948191935_dbl_kind, 0.577036344783469_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,962) = (/ 0.999892084908761_dbl_kind, 0.997231346949046_dbl_kind, 0.978337597199845_dbl_kind, 0.910241777911039_dbl_kind, 0.576947547506049_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,963) = (/ 0.999891977626187_dbl_kind, 0.997228787784984_dbl_kind, 0.978316925499217_dbl_kind, 0.910164091988544_dbl_kind, 0.576858954573958_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,964) = (/ 0.999891870345297_dbl_kind, 0.997226228960104_dbl_kind, 0.978296255544393_dbl_kind, 0.91008642414905_dbl_kind, 0.576770565484717_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,965) = (/ 0.999891763066094_dbl_kind, 0.997223670474114_dbl_kind, 0.978275587334702_dbl_kind, 0.910008774389736_dbl_kind, 0.576682379737103_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,966) = (/ 0.999891655788578_dbl_kind, 0.997221112326721_dbl_kind, 0.978254920869477_dbl_kind, 0.909931142707779_dbl_kind, 0.576594396831128_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,967) = (/ 0.999891548512749_dbl_kind, 0.997218554517631_dbl_kind, 0.978234256148055_dbl_kind, 0.909853529100344_dbl_kind, 0.576506616268053_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,968) = (/ 0.99989144123861_dbl_kind, 0.997215997046554_dbl_kind, 0.978213593169776_dbl_kind, 0.909775933564595_dbl_kind, 0.576419037550371_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,969) = (/ 0.999891333966161_dbl_kind, 0.997213439913199_dbl_kind, 0.978192931933983_dbl_kind, 0.90969835609769_dbl_kind, 0.576331660181818_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,970) = (/ 0.999891226695405_dbl_kind, 0.997210883117272_dbl_kind, 0.978172272440023_dbl_kind, 0.909620796696779_dbl_kind, 0.576244483667353_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,971) = (/ 0.99989111942634_dbl_kind, 0.997208326658486_dbl_kind, 0.978151614687249_dbl_kind, 0.909543255359011_dbl_kind, 0.576157507513173_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,972) = (/ 0.999891012158969_dbl_kind, 0.997205770536549_dbl_kind, 0.978130958675014_dbl_kind, 0.909465732081526_dbl_kind, 0.576070731226698_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,973) = (/ 0.999890904893294_dbl_kind, 0.99720321475117_dbl_kind, 0.978110304402679_dbl_kind, 0.909388226861461_dbl_kind, 0.57598415431657_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,974) = (/ 0.999890797629315_dbl_kind, 0.997200659302063_dbl_kind, 0.978089651869605_dbl_kind, 0.909310739695947_dbl_kind, 0.575897776292653_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,975) = (/ 0.999890690367033_dbl_kind, 0.997198104188937_dbl_kind, 0.978069001075159_dbl_kind, 0.909233270582113_dbl_kind, 0.575811596666035_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,976) = (/ 0.99989058310645_dbl_kind, 0.997195549411504_dbl_kind, 0.97804835201871_dbl_kind, 0.909155819517078_dbl_kind, 0.575725614949012_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,977) = (/ 0.999890475847567_dbl_kind, 0.997192994969476_dbl_kind, 0.978027704699635_dbl_kind, 0.909078386497962_dbl_kind, 0.575639830655094_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,978) = (/ 0.999890368590385_dbl_kind, 0.997190440862564_dbl_kind, 0.978007059117309_dbl_kind, 0.909000971521879_dbl_kind, 0.575554243299004_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,979) = (/ 0.999890261334905_dbl_kind, 0.997187887090484_dbl_kind, 0.977986415271116_dbl_kind, 0.908923574585936_dbl_kind, 0.575468852396663_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,980) = (/ 0.999890154081128_dbl_kind, 0.997185333652948_dbl_kind, 0.977965773160439_dbl_kind, 0.908846195687238_dbl_kind, 0.575383657465208_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,981) = (/ 0.999890046829057_dbl_kind, 0.99718278054967_dbl_kind, 0.977945132784671_dbl_kind, 0.908768834822889_dbl_kind, 0.575298658022969_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,982) = (/ 0.999889939578692_dbl_kind, 0.997180227780365_dbl_kind, 0.977924494143203_dbl_kind, 0.908691491989983_dbl_kind, 0.575213853589478_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,983) = (/ 0.999889832330033_dbl_kind, 0.997177675344747_dbl_kind, 0.977903857235434_dbl_kind, 0.908614167185614_dbl_kind, 0.575129243685458_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,984) = (/ 0.999889725083083_dbl_kind, 0.997175123242532_dbl_kind, 0.977883222060766_dbl_kind, 0.90853686040687_dbl_kind, 0.575044827832828_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,985) = (/ 0.999889617837844_dbl_kind, 0.997172571473437_dbl_kind, 0.977862588618602_dbl_kind, 0.90845957165084_dbl_kind, 0.574960605554698_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,986) = (/ 0.999889510594316_dbl_kind, 0.997170020037177_dbl_kind, 0.977841956908354_dbl_kind, 0.908382300914604_dbl_kind, 0.574876576375359_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,987) = (/ 0.9998894033525_dbl_kind, 0.99716746893347_dbl_kind, 0.977821326929436_dbl_kind, 0.908305048195244_dbl_kind, 0.574792739820296_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,988) = (/ 0.999889296112399_dbl_kind, 0.997164918162032_dbl_kind, 0.977800698681263_dbl_kind, 0.908227813489833_dbl_kind, 0.574709095416165_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,989) = (/ 0.999889188874012_dbl_kind, 0.997162367722584_dbl_kind, 0.977780072163259_dbl_kind, 0.908150596795445_dbl_kind, 0.574625642690809_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,990) = (/ 0.999889081637342_dbl_kind, 0.997159817614843_dbl_kind, 0.977759447374849_dbl_kind, 0.908073398109151_dbl_kind, 0.574542381173239_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,991) = (/ 0.99988897440239_dbl_kind, 0.997157267838527_dbl_kind, 0.977738824315463_dbl_kind, 0.907996217428016_dbl_kind, 0.574459310393647_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,992) = (/ 0.999888867169158_dbl_kind, 0.997154718393358_dbl_kind, 0.977718202984535_dbl_kind, 0.907919054749106_dbl_kind, 0.574376429883392_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,993) = (/ 0.999888759937646_dbl_kind, 0.997152169279055_dbl_kind, 0.977697583381503_dbl_kind, 0.907841910069485_dbl_kind, 0.574293739174995_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,994) = (/ 0.999888652707856_dbl_kind, 0.99714962049534_dbl_kind, 0.97767696550581_dbl_kind, 0.907764783386206_dbl_kind, 0.574211237802156_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,995) = (/ 0.99988854547979_dbl_kind, 0.997147072041933_dbl_kind, 0.977656349356902_dbl_kind, 0.907687674696332_dbl_kind, 0.574128925299722_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,996) = (/ 0.999888438253449_dbl_kind, 0.997144523918557_dbl_kind, 0.97763573493423_dbl_kind, 0.907610583996916_dbl_kind, 0.574046801203707_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,997) = (/ 0.999888331028835_dbl_kind, 0.997141976124934_dbl_kind, 0.977615122237247_dbl_kind, 0.907533511285009_dbl_kind, 0.573964865051281_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,998) = (/ 0.999888223805948_dbl_kind, 0.997139428660788_dbl_kind, 0.977594511265416_dbl_kind, 0.907456456557663_dbl_kind, 0.573883116380769_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,999) = (/ 0.999888116584791_dbl_kind, 0.997136881525842_dbl_kind, 0.977573902018197_dbl_kind, 0.907379419811926_dbl_kind, 0.573801554731643_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1000) = (/ 0.999888009365364_dbl_kind, 0.997134334719821_dbl_kind, 0.977553294495059_dbl_kind, 0.907302401044846_dbl_kind, 0.573720179644525_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1001) = (/ 0.99988790214767_dbl_kind, 0.997131788242451_dbl_kind, 0.977532688695474_dbl_kind, 0.907225400253467_dbl_kind, 0.57363899066119_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1002) = (/ 0.999887794931709_dbl_kind, 0.997129242093455_dbl_kind, 0.977512084618919_dbl_kind, 0.907148417434835_dbl_kind, 0.573557987324543_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1003) = (/ 0.999887687717484_dbl_kind, 0.997126696272562_dbl_kind, 0.977491482264873_dbl_kind, 0.90707145258599_dbl_kind, 0.573477169178638_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1004) = (/ 0.999887580504995_dbl_kind, 0.997124150779496_dbl_kind, 0.977470881632821_dbl_kind, 0.906994505703973_dbl_kind, 0.573396535768666_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1005) = (/ 0.999887473294245_dbl_kind, 0.997121605613988_dbl_kind, 0.977450282722253_dbl_kind, 0.906917576785827_dbl_kind, 0.573316086640953_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1006) = (/ 0.999887366085234_dbl_kind, 0.997119060775763_dbl_kind, 0.977429685532663_dbl_kind, 0.906840665828587_dbl_kind, 0.573235821342956_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1007) = (/ 0.999887258877965_dbl_kind, 0.997116516264552_dbl_kind, 0.977409090063548_dbl_kind, 0.906763772829291_dbl_kind, 0.57315573942326_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1008) = (/ 0.999887151672439_dbl_kind, 0.997113972080082_dbl_kind, 0.977388496314411_dbl_kind, 0.90668689778498_dbl_kind, 0.573075840431582_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1009) = (/ 0.999887044468657_dbl_kind, 0.997111428222085_dbl_kind, 0.97736790428476_dbl_kind, 0.906610040692687_dbl_kind, 0.572996123918759_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1010) = (/ 0.999886937266622_dbl_kind, 0.997108884690292_dbl_kind, 0.977347313974103_dbl_kind, 0.906533201549447_dbl_kind, 0.572916589436752_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1011) = (/ 0.999886830066334_dbl_kind, 0.997106341484433_dbl_kind, 0.977326725381959_dbl_kind, 0.906456380352298_dbl_kind, 0.572837236538638_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1012) = (/ 0.999886722867796_dbl_kind, 0.99710379860424_dbl_kind, 0.977306138507847_dbl_kind, 0.906379577098275_dbl_kind, 0.572758064778614_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1013) = (/ 0.999886615671009_dbl_kind, 0.997101256049448_dbl_kind, 0.977285553351292_dbl_kind, 0.906302791784411_dbl_kind, 0.572679073711987_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1014) = (/ 0.999886508475974_dbl_kind, 0.997098713819788_dbl_kind, 0.977264969911823_dbl_kind, 0.906226024407739_dbl_kind, 0.572600262895183_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1015) = (/ 0.999886401282693_dbl_kind, 0.997096171914995_dbl_kind, 0.977244388188974_dbl_kind, 0.906149274965298_dbl_kind, 0.572521631885725_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1016) = (/ 0.999886294091168_dbl_kind, 0.997093630334804_dbl_kind, 0.977223808182283_dbl_kind, 0.906072543454119_dbl_kind, 0.572443180242253_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1017) = (/ 0.999886186901401_dbl_kind, 0.997091089078949_dbl_kind, 0.977203229891294_dbl_kind, 0.905995829871239_dbl_kind, 0.572364907524498_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1018) = (/ 0.999886079713392_dbl_kind, 0.997088548147167_dbl_kind, 0.977182653315553_dbl_kind, 0.905919134213691_dbl_kind, 0.572286813293306_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1019) = (/ 0.999885972527145_dbl_kind, 0.997086007539196_dbl_kind, 0.977162078454613_dbl_kind, 0.905842456478513_dbl_kind, 0.572208897110609_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1020) = (/ 0.999885865342661_dbl_kind, 0.997083467254773_dbl_kind, 0.977141505308031_dbl_kind, 0.905765796662741_dbl_kind, 0.572131158539441_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1021) = (/ 0.999885758159941_dbl_kind, 0.997080927293635_dbl_kind, 0.977120933875367_dbl_kind, 0.905689154763412_dbl_kind, 0.572053597143927_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1022) = (/ 0.999885650978986_dbl_kind, 0.997078387655522_dbl_kind, 0.977100364156188_dbl_kind, 0.905612530777563_dbl_kind, 0.571976212489284_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1023) = (/ 0.999885543799799_dbl_kind, 0.997075848340173_dbl_kind, 0.977079796150065_dbl_kind, 0.905535924702233_dbl_kind, 0.571899004141812_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1024) = (/ 0.999885436622383_dbl_kind, 0.997073309347331_dbl_kind, 0.977059229856573_dbl_kind, 0.905459336534463_dbl_kind, 0.571821971668899_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1025) = (/ 0.999885329446737_dbl_kind, 0.997070770676734_dbl_kind, 0.977038665275291_dbl_kind, 0.905382766271292_dbl_kind, 0.571745114639017_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1026) = (/ 0.999885222272865_dbl_kind, 0.997068232328125_dbl_kind, 0.977018102405807_dbl_kind, 0.905306213909765_dbl_kind, 0.571668432621717_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1027) = (/ 0.999885115100766_dbl_kind, 0.997065694301246_dbl_kind, 0.976997541247707_dbl_kind, 0.905229679446922_dbl_kind, 0.571591925187625_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1028) = (/ 0.999885007930446_dbl_kind, 0.997063156595843_dbl_kind, 0.976976981800588_dbl_kind, 0.905153162879811_dbl_kind, 0.571515591908449_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1029) = (/ 0.999884900761904_dbl_kind, 0.997060619211657_dbl_kind, 0.976956424064047_dbl_kind, 0.90507666420548_dbl_kind, 0.571439432356959_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1030) = (/ 0.999884793595142_dbl_kind, 0.997058082148435_dbl_kind, 0.97693586803769_dbl_kind, 0.905000183420974_dbl_kind, 0.571363446107001_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1031) = (/ 0.999884686430161_dbl_kind, 0.997055545405921_dbl_kind, 0.976915313721124_dbl_kind, 0.904923720523345_dbl_kind, 0.571287632733486_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1032) = (/ 0.999884579266966_dbl_kind, 0.997053008983863_dbl_kind, 0.976894761113963_dbl_kind, 0.904847275509646_dbl_kind, 0.571211991812395_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1033) = (/ 0.999884472105556_dbl_kind, 0.997050472882008_dbl_kind, 0.976874210215827_dbl_kind, 0.904770848376932_dbl_kind, 0.571136522920763_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1034) = (/ 0.999884364945935_dbl_kind, 0.997047937100102_dbl_kind, 0.976853661026338_dbl_kind, 0.904694439122259_dbl_kind, 0.571061225636689_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1035) = (/ 0.999884257788102_dbl_kind, 0.997045401637897_dbl_kind, 0.976833113545124_dbl_kind, 0.904618047742686_dbl_kind, 0.570986099539332_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1036) = (/ 0.999884150632061_dbl_kind, 0.99704286649514_dbl_kind, 0.976812567771819_dbl_kind, 0.904541674235275_dbl_kind, 0.570911144208898_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1037) = (/ 0.999884043477814_dbl_kind, 0.997040331671582_dbl_kind, 0.97679202370606_dbl_kind, 0.904465318597088_dbl_kind, 0.570836359226652_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1038) = (/ 0.999883936325363_dbl_kind, 0.997037797166975_dbl_kind, 0.976771481347491_dbl_kind, 0.904388980825196_dbl_kind, 0.570761744174904_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1039) = (/ 0.999883829174708_dbl_kind, 0.99703526298107_dbl_kind, 0.976750940695759_dbl_kind, 0.904312660916665_dbl_kind, 0.570687298637017_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1040) = (/ 0.999883722025854_dbl_kind, 0.99703272911362_dbl_kind, 0.976730401750518_dbl_kind, 0.904236358868569_dbl_kind, 0.570613022197391_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1041) = (/ 0.999883614878801_dbl_kind, 0.997030195564379_dbl_kind, 0.976709864511425_dbl_kind, 0.904160074677982_dbl_kind, 0.570538914441473_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1042) = (/ 0.999883507733552_dbl_kind, 0.9970276623331_dbl_kind, 0.976689328978145_dbl_kind, 0.904083808341986_dbl_kind, 0.570464974955748_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1043) = (/ 0.999883400590107_dbl_kind, 0.99702512941954_dbl_kind, 0.976668795150343_dbl_kind, 0.904007559857658_dbl_kind, 0.570391203327737_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1044) = (/ 0.999883293448471_dbl_kind, 0.997022596823453_dbl_kind, 0.976648263027694_dbl_kind, 0.903931329222088_dbl_kind, 0.570317599146003_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1045) = (/ 0.999883186308643_dbl_kind, 0.997020064544598_dbl_kind, 0.976627732609876_dbl_kind, 0.903855116432362_dbl_kind, 0.57024416200013_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1046) = (/ 0.999883079170628_dbl_kind, 0.997017532582731_dbl_kind, 0.976607203896571_dbl_kind, 0.903778921485572_dbl_kind, 0.570170891480739_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1047) = (/ 0.999882972034426_dbl_kind, 0.997015000937612_dbl_kind, 0.97658667688747_dbl_kind, 0.903702744378817_dbl_kind, 0.570097787179478_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1048) = (/ 0.99988286490004_dbl_kind, 0.997012469608999_dbl_kind, 0.976566151582264_dbl_kind, 0.903626585109195_dbl_kind, 0.570024848689016_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1049) = (/ 0.999882757767472_dbl_kind, 0.997009938596652_dbl_kind, 0.976545627980652_dbl_kind, 0.903550443673809_dbl_kind, 0.569952075603048_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1050) = (/ 0.999882650636723_dbl_kind, 0.997007407900333_dbl_kind, 0.976525106082339_dbl_kind, 0.903474320069768_dbl_kind, 0.569879467516288_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1051) = (/ 0.999882543507797_dbl_kind, 0.997004877519805_dbl_kind, 0.976504585887033_dbl_kind, 0.903398214294184_dbl_kind, 0.569807024024466_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1052) = (/ 0.999882436380695_dbl_kind, 0.997002347454828_dbl_kind, 0.976484067394449_dbl_kind, 0.903322126344173_dbl_kind, 0.56973474472433_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1053) = (/ 0.99988232925542_dbl_kind, 0.996999817705168_dbl_kind, 0.976463550604305_dbl_kind, 0.903246056216856_dbl_kind, 0.56966262921364_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1054) = (/ 0.999882222131973_dbl_kind, 0.996997288270588_dbl_kind, 0.976443035516327_dbl_kind, 0.90317000390936_dbl_kind, 0.569590677091164_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1055) = (/ 0.999882115010357_dbl_kind, 0.996994759150853_dbl_kind, 0.976422522130245_dbl_kind, 0.903093969418811_dbl_kind, 0.569518887956681_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1056) = (/ 0.999882007890574_dbl_kind, 0.996992230345731_dbl_kind, 0.976402010445793_dbl_kind, 0.903017952742348_dbl_kind, 0.569447261410972_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1057) = (/ 0.999881900772626_dbl_kind, 0.996989701854988_dbl_kind, 0.976381500462712_dbl_kind, 0.902941953877108_dbl_kind, 0.569375797055821_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1058) = (/ 0.999881793656515_dbl_kind, 0.996987173678392_dbl_kind, 0.976360992180749_dbl_kind, 0.902865972820235_dbl_kind, 0.569304494494018_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1059) = (/ 0.999881686542245_dbl_kind, 0.996984645815712_dbl_kind, 0.976340485599654_dbl_kind, 0.90279000956888_dbl_kind, 0.569233353329346_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1060) = (/ 0.999881579429817_dbl_kind, 0.996982118266719_dbl_kind, 0.976319980719183_dbl_kind, 0.902714064120197_dbl_kind, 0.569162373166586_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1061) = (/ 0.999881472319233_dbl_kind, 0.996979591031182_dbl_kind, 0.976299477539098_dbl_kind, 0.902638136471348_dbl_kind, 0.569091553611511_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1062) = (/ 0.999881365210495_dbl_kind, 0.996977064108873_dbl_kind, 0.976278976059168_dbl_kind, 0.902562226619497_dbl_kind, 0.569020894270887_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1063) = (/ 0.999881258103606_dbl_kind, 0.996974537499565_dbl_kind, 0.976258476279164_dbl_kind, 0.902486334561813_dbl_kind, 0.568950394752468_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1064) = (/ 0.999881150998569_dbl_kind, 0.996972011203031_dbl_kind, 0.976237978198866_dbl_kind, 0.902410460295476_dbl_kind, 0.568880054664995_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1065) = (/ 0.999881043895386_dbl_kind, 0.996969485219045_dbl_kind, 0.976217481818055_dbl_kind, 0.902334603817668_dbl_kind, 0.568809873618189_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1066) = (/ 0.999880936794058_dbl_kind, 0.996966959547383_dbl_kind, 0.976196987136521_dbl_kind, 0.902258765125576_dbl_kind, 0.568739851222757_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1067) = (/ 0.99988082969459_dbl_kind, 0.996964434187821_dbl_kind, 0.976176494154059_dbl_kind, 0.902182944216396_dbl_kind, 0.568669987090386_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1068) = (/ 0.999880722596981_dbl_kind, 0.996961909140137_dbl_kind, 0.976156002870469_dbl_kind, 0.902107141087328_dbl_kind, 0.568600280833738_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1069) = (/ 0.999880615501237_dbl_kind, 0.996959384404107_dbl_kind, 0.976135513285556_dbl_kind, 0.902031355735577_dbl_kind, 0.568530732066445_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1070) = (/ 0.999880508407358_dbl_kind, 0.996956859979511_dbl_kind, 0.976115025399132_dbl_kind, 0.901955588158358_dbl_kind, 0.568461340403122_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1071) = (/ 0.999880401315347_dbl_kind, 0.996954335866128_dbl_kind, 0.976094539211013_dbl_kind, 0.90187983835289_dbl_kind, 0.568392105459344_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1072) = (/ 0.999880294225207_dbl_kind, 0.996951812063742_dbl_kind, 0.976074054721021_dbl_kind, 0.901804106316399_dbl_kind, 0.568323026851658_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1073) = (/ 0.999880187136941_dbl_kind, 0.996949288572131_dbl_kind, 0.976053571928984_dbl_kind, 0.901728392046119_dbl_kind, 0.568254104197576_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1074) = (/ 0.99988008005055_dbl_kind, 0.99694676539108_dbl_kind, 0.976033090834736_dbl_kind, 0.901652695539288_dbl_kind, 0.56818533711557_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1075) = (/ 0.999879972966037_dbl_kind, 0.996944242520372_dbl_kind, 0.976012611438116_dbl_kind, 0.901577016793152_dbl_kind, 0.56811672522508_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1076) = (/ 0.999879865883405_dbl_kind, 0.996941719959792_dbl_kind, 0.975992133738969_dbl_kind, 0.901501355804967_dbl_kind, 0.568048268146496_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1077) = (/ 0.999879758802657_dbl_kind, 0.996939197709125_dbl_kind, 0.975971657737145_dbl_kind, 0.901425712571991_dbl_kind, 0.567979965501166_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1078) = (/ 0.999879651723794_dbl_kind, 0.996936675768158_dbl_kind, 0.9759511834325_dbl_kind, 0.901350087091493_dbl_kind, 0.567911816911396_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1079) = (/ 0.99987954464682_dbl_kind, 0.99693415413668_dbl_kind, 0.975930710824896_dbl_kind, 0.901274479360748_dbl_kind, 0.567843822000439_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1080) = (/ 0.999879437571738_dbl_kind, 0.996931632814477_dbl_kind, 0.9759102399142_dbl_kind, 0.901198889377039_dbl_kind, 0.567775980392501_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1081) = (/ 0.99987933049855_dbl_kind, 0.996929111801341_dbl_kind, 0.975889770700288_dbl_kind, 0.901123317137656_dbl_kind, 0.567708291712729_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1082) = (/ 0.999879223427258_dbl_kind, 0.99692659109706_dbl_kind, 0.975869303183035_dbl_kind, 0.901047762639898_dbl_kind, 0.56764075558722_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1083) = (/ 0.999879116357865_dbl_kind, 0.996924070701427_dbl_kind, 0.97584883736233_dbl_kind, 0.900972225881069_dbl_kind, 0.567573371643011_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1084) = (/ 0.999879009290374_dbl_kind, 0.996921550614235_dbl_kind, 0.975828373238062_dbl_kind, 0.900896706858483_dbl_kind, 0.56750613950808_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1085) = (/ 0.999878902224788_dbl_kind, 0.996919030835277_dbl_kind, 0.975807910810127_dbl_kind, 0.900821205569462_dbl_kind, 0.56743905881134_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1086) = (/ 0.99987879516111_dbl_kind, 0.996916511364347_dbl_kind, 0.975787450078428_dbl_kind, 0.900745722011337_dbl_kind, 0.567372129182643_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1087) = (/ 0.999878688099342_dbl_kind, 0.996913992201242_dbl_kind, 0.975766991042875_dbl_kind, 0.900670256181444_dbl_kind, 0.567305350252772_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1088) = (/ 0.999878581039487_dbl_kind, 0.996911473345757_dbl_kind, 0.975746533703379_dbl_kind, 0.900594808077132_dbl_kind, 0.567238721653442_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1089) = (/ 0.999878473981548_dbl_kind, 0.996908954797691_dbl_kind, 0.975726078059863_dbl_kind, 0.900519377695753_dbl_kind, 0.567172243017298_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1090) = (/ 0.999878366925527_dbl_kind, 0.996906436556841_dbl_kind, 0.975705624112253_dbl_kind, 0.900443965034673_dbl_kind, 0.567105913977907_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1091) = (/ 0.999878259871429_dbl_kind, 0.996903918623008_dbl_kind, 0.975685171860479_dbl_kind, 0.900368570091263_dbl_kind, 0.567039734169764_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1092) = (/ 0.999878152819254_dbl_kind, 0.996901400995991_dbl_kind, 0.975664721304481_dbl_kind, 0.900293192862904_dbl_kind, 0.566973703228288_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1093) = (/ 0.999878045769007_dbl_kind, 0.996898883675594_dbl_kind, 0.975644272444202_dbl_kind, 0.900217833346987_dbl_kind, 0.56690782078981_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1094) = (/ 0.99987793872069_dbl_kind, 0.996896366661618_dbl_kind, 0.975623825279593_dbl_kind, 0.900142491540911_dbl_kind, 0.566842086491586_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1095) = (/ 0.999877831674307_dbl_kind, 0.996893849953867_dbl_kind, 0.975603379810609_dbl_kind, 0.900067167442082_dbl_kind, 0.566776499971785_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1096) = (/ 0.999877724629859_dbl_kind, 0.996891333552146_dbl_kind, 0.975582936037213_dbl_kind, 0.899991861047921_dbl_kind, 0.566711060869488_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1097) = (/ 0.999877617587351_dbl_kind, 0.996888817456261_dbl_kind, 0.975562493959373_dbl_kind, 0.899916572355851_dbl_kind, 0.566645768824689_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1098) = (/ 0.999877510546785_dbl_kind, 0.996886301666018_dbl_kind, 0.975542053577064_dbl_kind, 0.89984130136331_dbl_kind, 0.566580623478289_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1099) = (/ 0.999877403508164_dbl_kind, 0.996883786181227_dbl_kind, 0.975521614890266_dbl_kind, 0.899766048067745_dbl_kind, 0.566515624472092_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1100) = (/ 0.999877296471491_dbl_kind, 0.996881271001695_dbl_kind, 0.975501177898966_dbl_kind, 0.899690812466609_dbl_kind, 0.566450771448817_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1101) = (/ 0.99987718943677_dbl_kind, 0.996878756127232_dbl_kind, 0.975480742603156_dbl_kind, 0.89961559455737_dbl_kind, 0.566386064052075_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1102) = (/ 0.999877082404002_dbl_kind, 0.996876241557651_dbl_kind, 0.975460309002836_dbl_kind, 0.899540394337502_dbl_kind, 0.566321501926381_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1103) = (/ 0.999876975373193_dbl_kind, 0.996873727292762_dbl_kind, 0.97543987709801_dbl_kind, 0.899465211804491_dbl_kind, 0.566257084717145_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1104) = (/ 0.999876868344344_dbl_kind, 0.99687121333238_dbl_kind, 0.975419446888691_dbl_kind, 0.899390046955833_dbl_kind, 0.566192812070676_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1105) = (/ 0.999876761317459_dbl_kind, 0.996868699676319_dbl_kind, 0.975399018374896_dbl_kind, 0.899314899789031_dbl_kind, 0.566128683634177_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1106) = (/ 0.999876654292541_dbl_kind, 0.996866186324394_dbl_kind, 0.975378591556649_dbl_kind, 0.899239770301604_dbl_kind, 0.566064699055739_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1107) = (/ 0.999876547269593_dbl_kind, 0.996863673276421_dbl_kind, 0.97535816643398_dbl_kind, 0.899164658491078_dbl_kind, 0.566000857984339_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1108) = (/ 0.999876440248619_dbl_kind, 0.996861160532217_dbl_kind, 0.975337743006926_dbl_kind, 0.89908956435499_dbl_kind, 0.565937160069853_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1109) = (/ 0.999876333229621_dbl_kind, 0.996858648091604_dbl_kind, 0.97531732127553_dbl_kind, 0.899014487890887_dbl_kind, 0.565873604963027_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1110) = (/ 0.999876226212603_dbl_kind, 0.996856135954399_dbl_kind, 0.97529690123984_dbl_kind, 0.89893942909633_dbl_kind, 0.5658101923155_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1111) = (/ 0.999876119197568_dbl_kind, 0.996853624120424_dbl_kind, 0.975276482899913_dbl_kind, 0.898864387968886_dbl_kind, 0.565746921779786_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1112) = (/ 0.99987601218452_dbl_kind, 0.9968511125895_dbl_kind, 0.975256066255811_dbl_kind, 0.898789364506138_dbl_kind, 0.565683793009284_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1113) = (/ 0.999875905173463_dbl_kind, 0.996848601361451_dbl_kind, 0.975235651307603_dbl_kind, 0.898714358705676_dbl_kind, 0.565620805658259_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1114) = (/ 0.999875798164398_dbl_kind, 0.996846090436101_dbl_kind, 0.975215238055361_dbl_kind, 0.898639370565105_dbl_kind, 0.565557959381857_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1115) = (/ 0.99987569115733_dbl_kind, 0.996843579813274_dbl_kind, 0.97519482649917_dbl_kind, 0.898564400082038_dbl_kind, 0.565495253836093_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1116) = (/ 0.999875584152262_dbl_kind, 0.996841069492799_dbl_kind, 0.975174416639116_dbl_kind, 0.898489447254102_dbl_kind, 0.565432688677857_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1117) = (/ 0.999875477149198_dbl_kind, 0.996838559474502_dbl_kind, 0.975154008475294_dbl_kind, 0.898414512078933_dbl_kind, 0.565370263564901_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1118) = (/ 0.999875370148141_dbl_kind, 0.996836049758213_dbl_kind, 0.975133602007805_dbl_kind, 0.898339594554182_dbl_kind, 0.565307978155844_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1119) = (/ 0.999875263149094_dbl_kind, 0.996833540343759_dbl_kind, 0.975113197236755_dbl_kind, 0.89826469467751_dbl_kind, 0.565245832110168_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1120) = (/ 0.999875156152062_dbl_kind, 0.996831031230974_dbl_kind, 0.97509279416226_dbl_kind, 0.898189812446588_dbl_kind, 0.56518382508822_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1121) = (/ 0.999875049157047_dbl_kind, 0.996828522419689_dbl_kind, 0.975072392784439_dbl_kind, 0.898114947859103_dbl_kind, 0.565121956751201_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1122) = (/ 0.999874942164053_dbl_kind, 0.996826013909736_dbl_kind, 0.97505199310342_dbl_kind, 0.898040100912752_dbl_kind, 0.565060226761172_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1123) = (/ 0.999874835173085_dbl_kind, 0.996823505700952_dbl_kind, 0.975031595119337_dbl_kind, 0.897965271605243_dbl_kind, 0.564998634781049_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1124) = (/ 0.999874728184145_dbl_kind, 0.99682099779317_dbl_kind, 0.975011198832331_dbl_kind, 0.897890459934298_dbl_kind, 0.564937180474601_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1125) = (/ 0.999874621197236_dbl_kind, 0.996818490186229_dbl_kind, 0.974990804242548_dbl_kind, 0.897815665897653_dbl_kind, 0.564875863506446_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1126) = (/ 0.999874514212364_dbl_kind, 0.996815982879965_dbl_kind, 0.974970411350142_dbl_kind, 0.897740889493053_dbl_kind, 0.564814683542052_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1127) = (/ 0.999874407229532_dbl_kind, 0.996813475874216_dbl_kind, 0.974950020155275_dbl_kind, 0.897666130718258_dbl_kind, 0.564753640247738_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1128) = (/ 0.999874300248742_dbl_kind, 0.996810969168826_dbl_kind, 0.974929630658112_dbl_kind, 0.897591389571041_dbl_kind, 0.564692733290662_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1129) = (/ 0.99987419327_dbl_kind, 0.996808462763634_dbl_kind, 0.97490924285883_dbl_kind, 0.897516666049186_dbl_kind, 0.564631962338822_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1130) = (/ 0.999874086293308_dbl_kind, 0.996805956658482_dbl_kind, 0.974888856757608_dbl_kind, 0.897441960150491_dbl_kind, 0.564571327061066_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1131) = (/ 0.999873979318672_dbl_kind, 0.996803450853215_dbl_kind, 0.974868472354635_dbl_kind, 0.897367271872769_dbl_kind, 0.564510827127072_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1132) = (/ 0.999873872346093_dbl_kind, 0.996800945347677_dbl_kind, 0.974848089650105_dbl_kind, 0.897292601213842_dbl_kind, 0.56445046220736_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1133) = (/ 0.999873765375577_dbl_kind, 0.996798440141714_dbl_kind, 0.974827708644219_dbl_kind, 0.897217948171551_dbl_kind, 0.564390231973278_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1134) = (/ 0.999873658407126_dbl_kind, 0.996795935235174_dbl_kind, 0.974807329337186_dbl_kind, 0.897143312743745_dbl_kind, 0.564330136097018_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1135) = (/ 0.999873551440747_dbl_kind, 0.996793430627904_dbl_kind, 0.974786951729222_dbl_kind, 0.897068694928289_dbl_kind, 0.564270174251588_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1136) = (/ 0.999873444476441_dbl_kind, 0.996790926319756_dbl_kind, 0.974766575820548_dbl_kind, 0.896994094723062_dbl_kind, 0.564210346110835_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1137) = (/ 0.999873337514214_dbl_kind, 0.996788422310579_dbl_kind, 0.974746201611392_dbl_kind, 0.896919512125957_dbl_kind, 0.564150651349425_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1138) = (/ 0.999873230554069_dbl_kind, 0.996785918600225_dbl_kind, 0.974725829101993_dbl_kind, 0.896844947134879_dbl_kind, 0.564091089642853_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1139) = (/ 0.99987312359601_dbl_kind, 0.996783415188548_dbl_kind, 0.974705458292592_dbl_kind, 0.896770399747748_dbl_kind, 0.564031660667434_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1140) = (/ 0.999873016640041_dbl_kind, 0.996780912075402_dbl_kind, 0.97468508918344_dbl_kind, 0.896695869962501_dbl_kind, 0.563972364100306_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1141) = (/ 0.999872909686166_dbl_kind, 0.996778409260643_dbl_kind, 0.974664721774794_dbl_kind, 0.896621357777083_dbl_kind, 0.563913199619421_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1142) = (/ 0.99987280273439_dbl_kind, 0.996775906744127_dbl_kind, 0.974644356066918_dbl_kind, 0.896546863189459_dbl_kind, 0.563854166903549_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1143) = (/ 0.999872695784717_dbl_kind, 0.996773404525712_dbl_kind, 0.974623992060084_dbl_kind, 0.896472386197604_dbl_kind, 0.563795265632274_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1144) = (/ 0.999872588837151_dbl_kind, 0.996770902605259_dbl_kind, 0.974603629754571_dbl_kind, 0.896397926799513_dbl_kind, 0.563736495485996_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1145) = (/ 0.999872481891696_dbl_kind, 0.996768400982628_dbl_kind, 0.974583269150662_dbl_kind, 0.89632348499319_dbl_kind, 0.563677856145917_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1146) = (/ 0.999872374948356_dbl_kind, 0.99676589965768_dbl_kind, 0.974562910248652_dbl_kind, 0.896249060776656_dbl_kind, 0.563619347294051_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1147) = (/ 0.999872268007136_dbl_kind, 0.996763398630277_dbl_kind, 0.974542553048841_dbl_kind, 0.896174654147949_dbl_kind, 0.563560968613225_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1148) = (/ 0.999872161068039_dbl_kind, 0.996760897900285_dbl_kind, 0.974522197551534_dbl_kind, 0.896100265105119_dbl_kind, 0.563502719787058_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1149) = (/ 0.999872054131071_dbl_kind, 0.996758397467569_dbl_kind, 0.974501843757048_dbl_kind, 0.896025893646231_dbl_kind, 0.563444600499981_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1150) = (/ 0.999871947196236_dbl_kind, 0.996755897331996_dbl_kind, 0.974481491665703_dbl_kind, 0.895951539769367_dbl_kind, 0.563386610437219_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1151) = (/ 0.999871840263537_dbl_kind, 0.996753397493433_dbl_kind, 0.974461141277829_dbl_kind, 0.895877203472624_dbl_kind, 0.5633287492848_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1152) = (/ 0.99987173333298_dbl_kind, 0.996750897951749_dbl_kind, 0.974440792593761_dbl_kind, 0.895802884754113_dbl_kind, 0.563271016729545_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1153) = (/ 0.99987162640457_dbl_kind, 0.996748398706816_dbl_kind, 0.974420445613843_dbl_kind, 0.895728583611962_dbl_kind, 0.56321341245907_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1154) = (/ 0.999871519478309_dbl_kind, 0.996745899758504_dbl_kind, 0.974400100338426_dbl_kind, 0.895654300044315_dbl_kind, 0.563155936161783_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1155) = (/ 0.999871412554204_dbl_kind, 0.996743401106688_dbl_kind, 0.974379756767869_dbl_kind, 0.895580034049329_dbl_kind, 0.563098587526885_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1156) = (/ 0.999871305632258_dbl_kind, 0.996740902751241_dbl_kind, 0.974359414902536_dbl_kind, 0.895505785625178_dbl_kind, 0.563041366244365_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1157) = (/ 0.999871198712477_dbl_kind, 0.996738404692038_dbl_kind, 0.9743390747428_dbl_kind, 0.895431554770055_dbl_kind, 0.562984272004995_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1158) = (/ 0.999871091794864_dbl_kind, 0.996735906928956_dbl_kind, 0.974318736289044_dbl_kind, 0.895357341482164_dbl_kind, 0.562927304500333_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1159) = (/ 0.999870984879424_dbl_kind, 0.996733409461874_dbl_kind, 0.974298399541654_dbl_kind, 0.895283145759729_dbl_kind, 0.562870463422723_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1160) = (/ 0.999870877966163_dbl_kind, 0.996730912290671_dbl_kind, 0.974278064501025_dbl_kind, 0.895208967600989_dbl_kind, 0.562813748465286_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1161) = (/ 0.999870771055085_dbl_kind, 0.996728415415226_dbl_kind, 0.974257731167562_dbl_kind, 0.895134807004199_dbl_kind, 0.562757159321924_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1162) = (/ 0.999870664146194_dbl_kind, 0.996725918835422_dbl_kind, 0.974237399541673_dbl_kind, 0.895060663967632_dbl_kind, 0.562700695687314_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1163) = (/ 0.999870557239496_dbl_kind, 0.996723422551144_dbl_kind, 0.974217069623778_dbl_kind, 0.894986538489573_dbl_kind, 0.562644357256907_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1164) = (/ 0.999870450334995_dbl_kind, 0.996720926562274_dbl_kind, 0.974196741414302_dbl_kind, 0.89491243056833_dbl_kind, 0.562588143726933_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1165) = (/ 0.999870343432697_dbl_kind, 0.9967184308687_dbl_kind, 0.974176414913678_dbl_kind, 0.894838340202223_dbl_kind, 0.562532054794387_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1166) = (/ 0.999870236532605_dbl_kind, 0.996715935470306_dbl_kind, 0.974156090122348_dbl_kind, 0.894764267389592_dbl_kind, 0.562476090157036_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1167) = (/ 0.999870129634725_dbl_kind, 0.996713440366982_dbl_kind, 0.974135767040759_dbl_kind, 0.894690212128791_dbl_kind, 0.562420249513411_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1168) = (/ 0.999870022739062_dbl_kind, 0.996710945558618_dbl_kind, 0.974115445669368_dbl_kind, 0.894616174418195_dbl_kind, 0.562364532562815_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1169) = (/ 0.999869915845622_dbl_kind, 0.996708451045106_dbl_kind, 0.97409512600864_dbl_kind, 0.894542154256192_dbl_kind, 0.562308939005307_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1170) = (/ 0.999869808954407_dbl_kind, 0.996705956826336_dbl_kind, 0.974074808059044_dbl_kind, 0.89446815164119_dbl_kind, 0.562253468541714_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1171) = (/ 0.999869702065426_dbl_kind, 0.996703462902202_dbl_kind, 0.974054491821063_dbl_kind, 0.894394166571613_dbl_kind, 0.562198120873619_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1172) = (/ 0.999869595178681_dbl_kind, 0.9967009692726_dbl_kind, 0.974034177295181_dbl_kind, 0.894320199045904_dbl_kind, 0.562142895703362_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1173) = (/ 0.999869488294179_dbl_kind, 0.996698475937426_dbl_kind, 0.974013864481896_dbl_kind, 0.894246249062523_dbl_kind, 0.562087792734044_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1174) = (/ 0.999869381411924_dbl_kind, 0.996695982896578_dbl_kind, 0.973993553381708_dbl_kind, 0.894172316619946_dbl_kind, 0.562032811669514_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1175) = (/ 0.999869274531922_dbl_kind, 0.996693490149953_dbl_kind, 0.97397324399513_dbl_kind, 0.89409840171667_dbl_kind, 0.561977952214379_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1176) = (/ 0.999869167654179_dbl_kind, 0.996690997697453_dbl_kind, 0.973952936322679_dbl_kind, 0.894024504351208_dbl_kind, 0.561923214073993_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1177) = (/ 0.999869060778697_dbl_kind, 0.99668850553898_dbl_kind, 0.973932630364883_dbl_kind, 0.893950624522092_dbl_kind, 0.561868596954457_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1178) = (/ 0.999868953905485_dbl_kind, 0.996686013674435_dbl_kind, 0.973912326122275_dbl_kind, 0.89387676222787_dbl_kind, 0.561814100562624_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1179) = (/ 0.999868847034546_dbl_kind, 0.996683522103724_dbl_kind, 0.973892023595398_dbl_kind, 0.893802917467109_dbl_kind, 0.561759724606084_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1180) = (/ 0.999868740165887_dbl_kind, 0.996681030826752_dbl_kind, 0.973871722784803_dbl_kind, 0.893729090238398_dbl_kind, 0.561705468793179_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1181) = (/ 0.999868633299513_dbl_kind, 0.996678539843425_dbl_kind, 0.973851423691048_dbl_kind, 0.89365528054034_dbl_kind, 0.561651332832987_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1182) = (/ 0.999868526435429_dbl_kind, 0.996676049153653_dbl_kind, 0.973831126314699_dbl_kind, 0.893581488371558_dbl_kind, 0.561597316435323_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1183) = (/ 0.99986841957364_dbl_kind, 0.996673558757345_dbl_kind, 0.973810830656331_dbl_kind, 0.893507713730695_dbl_kind, 0.561543419310746_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1184) = (/ 0.999868312714153_dbl_kind, 0.996671068654412_dbl_kind, 0.973790536716526_dbl_kind, 0.89343395661641_dbl_kind, 0.561489641170543_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1185) = (/ 0.999868205856973_dbl_kind, 0.996668578844766_dbl_kind, 0.973770244495876_dbl_kind, 0.893360217027384_dbl_kind, 0.561435981726742_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1186) = (/ 0.999868099002105_dbl_kind, 0.996666089328322_dbl_kind, 0.973749953994979_dbl_kind, 0.893286494962315_dbl_kind, 0.5613824406921_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1187) = (/ 0.999867992149555_dbl_kind, 0.996663600104993_dbl_kind, 0.973729665214441_dbl_kind, 0.89321279041992_dbl_kind, 0.561329017780101_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1188) = (/ 0.99986788529933_dbl_kind, 0.996661111174697_dbl_kind, 0.973709378154879_dbl_kind, 0.893139103398937_dbl_kind, 0.561275712704964_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1189) = (/ 0.999867778451434_dbl_kind, 0.996658622537353_dbl_kind, 0.973689092816915_dbl_kind, 0.893065433898121_dbl_kind, 0.561222525181627_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1190) = (/ 0.999867671605873_dbl_kind, 0.996656134192878_dbl_kind, 0.973668809201181_dbl_kind, 0.892991781916247_dbl_kind, 0.561169454925758_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1191) = (/ 0.999867564762653_dbl_kind, 0.996653646141193_dbl_kind, 0.973648527308318_dbl_kind, 0.892918147452112_dbl_kind, 0.561116501653746_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1192) = (/ 0.99986745792178_dbl_kind, 0.996651158382222_dbl_kind, 0.973628247138971_dbl_kind, 0.892844530504529_dbl_kind, 0.561063665082702_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1193) = (/ 0.999867351083261_dbl_kind, 0.996648670915886_dbl_kind, 0.9736079686938_dbl_kind, 0.892770931072333_dbl_kind, 0.561010944930455_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1194) = (/ 0.9998672442471_dbl_kind, 0.996646183742111_dbl_kind, 0.973587691973469_dbl_kind, 0.892697349154378_dbl_kind, 0.560958340915551_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1195) = (/ 0.999867137413304_dbl_kind, 0.996643696860825_dbl_kind, 0.973567416978649_dbl_kind, 0.892623784749537_dbl_kind, 0.560905852757254_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1196) = (/ 0.999867030581879_dbl_kind, 0.996641210271952_dbl_kind, 0.973547143710023_dbl_kind, 0.892550237856707_dbl_kind, 0.560853480175542_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1197) = (/ 0.99986692375283_dbl_kind, 0.996638723975423_dbl_kind, 0.973526872168281_dbl_kind, 0.892476708474798_dbl_kind, 0.560801222891101_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1198) = (/ 0.999866816926165_dbl_kind, 0.996636237971169_dbl_kind, 0.973506602354122_dbl_kind, 0.892403196602748_dbl_kind, 0.560749080625332_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1199) = (/ 0.999866710101889_dbl_kind, 0.996633752259121_dbl_kind, 0.973486334268252_dbl_kind, 0.89232970223951_dbl_kind, 0.560697053100342_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1200) = (/ 0.999866603280008_dbl_kind, 0.996631266839212_dbl_kind, 0.973466067911386_dbl_kind, 0.892256225384059_dbl_kind, 0.560645140038947_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1201) = (/ 0.999866496460528_dbl_kind, 0.996628781711377_dbl_kind, 0.973445803284249_dbl_kind, 0.892182766035392_dbl_kind, 0.560593341164665_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1202) = (/ 0.999866389643457_dbl_kind, 0.996626296875553_dbl_kind, 0.973425540387572_dbl_kind, 0.892109324192524_dbl_kind, 0.56054165620172_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1203) = (/ 0.999866282828799_dbl_kind, 0.996623812331675_dbl_kind, 0.973405279222099_dbl_kind, 0.892035899854492_dbl_kind, 0.560490084875035_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1204) = (/ 0.999866176016562_dbl_kind, 0.996621328079686_dbl_kind, 0.973385019788575_dbl_kind, 0.891962493020356_dbl_kind, 0.560438626910236_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1205) = (/ 0.999866069206752_dbl_kind, 0.996618844119523_dbl_kind, 0.973364762087761_dbl_kind, 0.891889103689192_dbl_kind, 0.560387282033647_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1206) = (/ 0.999865962399375_dbl_kind, 0.996616360451128_dbl_kind, 0.973344506120423_dbl_kind, 0.891815731860102_dbl_kind, 0.56033604997228_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1207) = (/ 0.999865855594438_dbl_kind, 0.996613877074447_dbl_kind, 0.973324251887338_dbl_kind, 0.891742377532206_dbl_kind, 0.560284930453854_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1208) = (/ 0.999865748791947_dbl_kind, 0.996611393989422_dbl_kind, 0.973303999389288_dbl_kind, 0.891669040704646_dbl_kind, 0.560233923206773_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1209) = (/ 0.999865641991909_dbl_kind, 0.996608911196_dbl_kind, 0.973283748627068_dbl_kind, 0.891595721376587_dbl_kind, 0.560183027960134_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1210) = (/ 0.99986553519433_dbl_kind, 0.996606428694128_dbl_kind, 0.973263499601478_dbl_kind, 0.891522419547212_dbl_kind, 0.560132244443722_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1211) = (/ 0.999865428399217_dbl_kind, 0.996603946483756_dbl_kind, 0.973243252313329_dbl_kind, 0.891449135215729_dbl_kind, 0.560081572388015_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1212) = (/ 0.999865321606576_dbl_kind, 0.996601464564835_dbl_kind, 0.97322300676344_dbl_kind, 0.891375868381366_dbl_kind, 0.560031011524167_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1213) = (/ 0.999865214816416_dbl_kind, 0.996598982937315_dbl_kind, 0.973202762952639_dbl_kind, 0.891302619043373_dbl_kind, 0.559980561584026_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1214) = (/ 0.999865108028741_dbl_kind, 0.996596501601151_dbl_kind, 0.973182520881764_dbl_kind, 0.891229387201022_dbl_kind, 0.559930222300117_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1215) = (/ 0.99986500124356_dbl_kind, 0.996594020556298_dbl_kind, 0.97316228055166_dbl_kind, 0.891156172853606_dbl_kind, 0.559879993405646_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1216) = (/ 0.999864894460878_dbl_kind, 0.996591539802711_dbl_kind, 0.973142041963182_dbl_kind, 0.891082976000442_dbl_kind, 0.559829874634503_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1217) = (/ 0.999864787680703_dbl_kind, 0.996589059340349_dbl_kind, 0.973121805117192_dbl_kind, 0.891009796640866_dbl_kind, 0.559779865721248_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1218) = (/ 0.999864680903042_dbl_kind, 0.996586579169171_dbl_kind, 0.973101570014566_dbl_kind, 0.89093663477424_dbl_kind, 0.55972996640112_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1219) = (/ 0.999864574127902_dbl_kind, 0.996584099289137_dbl_kind, 0.973081336656182_dbl_kind, 0.890863490399946_dbl_kind, 0.559680176410033_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1220) = (/ 0.999864467355289_dbl_kind, 0.99658161970021_dbl_kind, 0.973061105042933_dbl_kind, 0.890790363517388_dbl_kind, 0.559630495484572_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1221) = (/ 0.999864360585211_dbl_kind, 0.996579140402353_dbl_kind, 0.973040875175717_dbl_kind, 0.890717254125994_dbl_kind, 0.559580923361992_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1222) = (/ 0.999864253817675_dbl_kind, 0.996576661395532_dbl_kind, 0.973020647055444_dbl_kind, 0.890644162225216_dbl_kind, 0.55953145978022_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1223) = (/ 0.999864147052688_dbl_kind, 0.996574182679714_dbl_kind, 0.973000420683031_dbl_kind, 0.890571087814525_dbl_kind, 0.559482104477845_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1224) = (/ 0.999864040290258_dbl_kind, 0.996571704254867_dbl_kind, 0.972980196059406_dbl_kind, 0.890498030893416_dbl_kind, 0.559432857194126_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1225) = (/ 0.999863933530391_dbl_kind, 0.996569226120959_dbl_kind, 0.972959973185503_dbl_kind, 0.89042499146141_dbl_kind, 0.559383717668985_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1226) = (/ 0.999863826773096_dbl_kind, 0.996566748277964_dbl_kind, 0.972939752062269_dbl_kind, 0.890351969518047_dbl_kind, 0.559334685643003_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1227) = (/ 0.999863720018378_dbl_kind, 0.996564270725852_dbl_kind, 0.972919532690658_dbl_kind, 0.890278965062892_dbl_kind, 0.559285760857425_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1228) = (/ 0.999863613266246_dbl_kind, 0.996561793464599_dbl_kind, 0.972899315071633_dbl_kind, 0.890205978095533_dbl_kind, 0.559236943054155_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1229) = (/ 0.999863506516708_dbl_kind, 0.996559316494181_dbl_kind, 0.972879099206167_dbl_kind, 0.890133008615582_dbl_kind, 0.559188231975748_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1230) = (/ 0.99986339976977_dbl_kind, 0.996556839814573_dbl_kind, 0.972858885095243_dbl_kind, 0.890060056622673_dbl_kind, 0.559139627365424_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1231) = (/ 0.99986329302544_dbl_kind, 0.996554363425755_dbl_kind, 0.972838672739852_dbl_kind, 0.889987122116464_dbl_kind, 0.55909112896705_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1232) = (/ 0.999863186283726_dbl_kind, 0.996551887327708_dbl_kind, 0.972818462140994_dbl_kind, 0.88991420509664_dbl_kind, 0.559042736525149_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1233) = (/ 0.999863079544636_dbl_kind, 0.996549411520412_dbl_kind, 0.97279825329968_dbl_kind, 0.889841305562902_dbl_kind, 0.558994449784892_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1234) = (/ 0.999862972808177_dbl_kind, 0.996546936003852_dbl_kind, 0.97277804621693_dbl_kind, 0.889768423514982_dbl_kind, 0.558946268492099_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1235) = (/ 0.999862866074357_dbl_kind, 0.996544460778013_dbl_kind, 0.972757840893772_dbl_kind, 0.889695558952634_dbl_kind, 0.55889819239324_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1236) = (/ 0.999862759343183_dbl_kind, 0.996541985842879_dbl_kind, 0.972737637331244_dbl_kind, 0.889622711875633_dbl_kind, 0.558850221235427_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1237) = (/ 0.999862652614664_dbl_kind, 0.996539511198441_dbl_kind, 0.972717435530394_dbl_kind, 0.889549882283782_dbl_kind, 0.558802354766419_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1238) = (/ 0.999862545888807_dbl_kind, 0.996537036844685_dbl_kind, 0.97269723549228_dbl_kind, 0.889477070176908_dbl_kind, 0.558754592734616_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1239) = (/ 0.999862439165621_dbl_kind, 0.996534562781605_dbl_kind, 0.97267703721797_dbl_kind, 0.889404275554858_dbl_kind, 0.55870693488906_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1240) = (/ 0.999862332445113_dbl_kind, 0.996532089009191_dbl_kind, 0.972656840708537_dbl_kind, 0.889331498417508_dbl_kind, 0.558659380979428_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1241) = (/ 0.999862225727291_dbl_kind, 0.99652961552744_dbl_kind, 0.97263664596507_dbl_kind, 0.889258738764756_dbl_kind, 0.558611930756042_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1242) = (/ 0.999862119012164_dbl_kind, 0.996527142336345_dbl_kind, 0.972616452988663_dbl_kind, 0.889185996596527_dbl_kind, 0.558564583969855_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1243) = (/ 0.999862012299739_dbl_kind, 0.996524669435903_dbl_kind, 0.972596261780421_dbl_kind, 0.889113271912767_dbl_kind, 0.558517340372452_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1244) = (/ 0.999861905590025_dbl_kind, 0.996522196826115_dbl_kind, 0.972576072341459_dbl_kind, 0.889040564713449_dbl_kind, 0.558470199716055_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1245) = (/ 0.999861798883031_dbl_kind, 0.99651972450698_dbl_kind, 0.972555884672902_dbl_kind, 0.888967874998573_dbl_kind, 0.558423161753518_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1246) = (/ 0.999861692178762_dbl_kind, 0.9965172524785_dbl_kind, 0.972535698775884_dbl_kind, 0.888895202768159_dbl_kind, 0.55837622623832_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1247) = (/ 0.999861585477231_dbl_kind, 0.996514780740676_dbl_kind, 0.972515514651549_dbl_kind, 0.888822548022256_dbl_kind, 0.558329392924573_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1248) = (/ 0.999861478778442_dbl_kind, 0.996512309293516_dbl_kind, 0.972495332301049_dbl_kind, 0.888749910760936_dbl_kind, 0.55828266156701_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1249) = (/ 0.999861372082407_dbl_kind, 0.996509838137026_dbl_kind, 0.97247515172555_dbl_kind, 0.888677290984298_dbl_kind, 0.558236031920993_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1250) = (/ 0.999861265389132_dbl_kind, 0.996507367271214_dbl_kind, 0.972454972926223_dbl_kind, 0.888604688692465_dbl_kind, 0.558189503742507_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1251) = (/ 0.999861158698627_dbl_kind, 0.996504896696089_dbl_kind, 0.972434795904253_dbl_kind, 0.888532103885585_dbl_kind, 0.558143076788154_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1252) = (/ 0.999861052010899_dbl_kind, 0.996502426411663_dbl_kind, 0.972414620660831_dbl_kind, 0.888459536563834_dbl_kind, 0.558096750815163_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1253) = (/ 0.999860945325959_dbl_kind, 0.996499956417948_dbl_kind, 0.972394447197161_dbl_kind, 0.88838698672741_dbl_kind, 0.558050525581375_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1254) = (/ 0.999860838643813_dbl_kind, 0.996497486714958_dbl_kind, 0.972374275514456_dbl_kind, 0.88831445437654_dbl_kind, 0.558004400845251_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1255) = (/ 0.999860731964472_dbl_kind, 0.996495017302711_dbl_kind, 0.972354105613936_dbl_kind, 0.888241939511475_dbl_kind, 0.55795837636587_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1256) = (/ 0.999860625287944_dbl_kind, 0.996492548181222_dbl_kind, 0.972333937496838_dbl_kind, 0.888169442132493_dbl_kind, 0.557912451902918_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1257) = (/ 0.999860518614238_dbl_kind, 0.996490079350513_dbl_kind, 0.972313771164401_dbl_kind, 0.888096962239896_dbl_kind, 0.557866627216703_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1258) = (/ 0.999860411943362_dbl_kind, 0.996487610810603_dbl_kind, 0.972293606617879_dbl_kind, 0.888024499834013_dbl_kind, 0.557820902068129_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1259) = (/ 0.999860305275326_dbl_kind, 0.996485142561515_dbl_kind, 0.972273443858535_dbl_kind, 0.887952054915202_dbl_kind, 0.557775276218724_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1260) = (/ 0.999860198610138_dbl_kind, 0.996482674603271_dbl_kind, 0.97225328288764_dbl_kind, 0.887879627483842_dbl_kind, 0.557729749430617_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1261) = (/ 0.999860091947808_dbl_kind, 0.9964802069359_dbl_kind, 0.972233123706479_dbl_kind, 0.887807217540343_dbl_kind, 0.557684321466541_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1262) = (/ 0.999859985288345_dbl_kind, 0.996477739559427_dbl_kind, 0.972212966316344_dbl_kind, 0.887734825085138_dbl_kind, 0.557638992089838_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1263) = (/ 0.999859878631758_dbl_kind, 0.99647527247388_dbl_kind, 0.972192810718538_dbl_kind, 0.887662450118689_dbl_kind, 0.557593761064451_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1264) = (/ 0.999859771978056_dbl_kind, 0.996472805679292_dbl_kind, 0.972172656914375_dbl_kind, 0.887590092641483_dbl_kind, 0.557548628154921_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1265) = (/ 0.999859665327248_dbl_kind, 0.996470339175694_dbl_kind, 0.972152504905177_dbl_kind, 0.887517752654036_dbl_kind, 0.557503593126397_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1266) = (/ 0.999859558679344_dbl_kind, 0.996467872963119_dbl_kind, 0.972132354692279_dbl_kind, 0.887445430156888_dbl_kind, 0.557458655744617_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1267) = (/ 0.999859452034354_dbl_kind, 0.996465407041602_dbl_kind, 0.972112206277025_dbl_kind, 0.887373125150607_dbl_kind, 0.557413815775926_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1268) = (/ 0.999859345392286_dbl_kind, 0.996462941411182_dbl_kind, 0.972092059660769_dbl_kind, 0.887300837635788_dbl_kind, 0.557369072987253_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1269) = (/ 0.999859238753151_dbl_kind, 0.996460476071894_dbl_kind, 0.972071914844874_dbl_kind, 0.887228567613052_dbl_kind, 0.557324427146131_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1270) = (/ 0.999859132116957_dbl_kind, 0.996458011023782_dbl_kind, 0.972051771830718_dbl_kind, 0.88715631508305_dbl_kind, 0.557279878020677_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1271) = (/ 0.999859025483715_dbl_kind, 0.996455546266886_dbl_kind, 0.972031630619684_dbl_kind, 0.887084080046459_dbl_kind, 0.557235425379605_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1272) = (/ 0.999858918853432_dbl_kind, 0.996453081801248_dbl_kind, 0.972011491213168_dbl_kind, 0.88701186250398_dbl_kind, 0.557191068992216_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1273) = (/ 0.999858812226122_dbl_kind, 0.996450617626916_dbl_kind, 0.971991353612577_dbl_kind, 0.886939662456347_dbl_kind, 0.557146808628397_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1274) = (/ 0.999858705601792_dbl_kind, 0.996448153743935_dbl_kind, 0.971971217819326_dbl_kind, 0.886867479904317_dbl_kind, 0.557102644058625_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1275) = (/ 0.999858598980451_dbl_kind, 0.996445690152354_dbl_kind, 0.971951083834844_dbl_kind, 0.886795314848677_dbl_kind, 0.557058575053961_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1276) = (/ 0.999858492362111_dbl_kind, 0.996443226852222_dbl_kind, 0.971930951660566_dbl_kind, 0.886723167290241_dbl_kind, 0.557014601386045_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1277) = (/ 0.999858385746781_dbl_kind, 0.996440763843591_dbl_kind, 0.971910821297942_dbl_kind, 0.88665103722985_dbl_kind, 0.556970722827105_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1278) = (/ 0.999858279134472_dbl_kind, 0.996438301126516_dbl_kind, 0.971890692748429_dbl_kind, 0.886578924668373_dbl_kind, 0.556926939149942_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1279) = (/ 0.999858172525192_dbl_kind, 0.99643583870105_dbl_kind, 0.971870566013498_dbl_kind, 0.88650682960671_dbl_kind, 0.556883250127943_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1280) = (/ 0.999858065918952_dbl_kind, 0.996433376567249_dbl_kind, 0.971850441094627_dbl_kind, 0.886434752045785_dbl_kind, 0.556839655535072_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1281) = (/ 0.999857959315763_dbl_kind, 0.996430914725174_dbl_kind, 0.971830317993307_dbl_kind, 0.886362691986551_dbl_kind, 0.556796155145864_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1282) = (/ 0.999857852715636_dbl_kind, 0.996428453174884_dbl_kind, 0.97181019671104_dbl_kind, 0.886290649429991_dbl_kind, 0.556752748735428_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1283) = (/ 0.999857746118578_dbl_kind, 0.996425991916439_dbl_kind, 0.971790077249336_dbl_kind, 0.886218624377115_dbl_kind, 0.556709436079454_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1284) = (/ 0.999857639524603_dbl_kind, 0.996423530949904_dbl_kind, 0.971769959609718_dbl_kind, 0.886146616828961_dbl_kind, 0.556666216954197_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1285) = (/ 0.999857532933719_dbl_kind, 0.996421070275344_dbl_kind, 0.971749843793719_dbl_kind, 0.886074626786596_dbl_kind, 0.556623091136482_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1286) = (/ 0.999857426345937_dbl_kind, 0.996418609892824_dbl_kind, 0.971729729802883_dbl_kind, 0.886002654251115_dbl_kind, 0.556580058403704_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1287) = (/ 0.999857319761268_dbl_kind, 0.996416149802415_dbl_kind, 0.971709617638764_dbl_kind, 0.885930699223643_dbl_kind, 0.556537118533826_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1288) = (/ 0.999857213179723_dbl_kind, 0.996413690004184_dbl_kind, 0.971689507302929_dbl_kind, 0.885858761705332_dbl_kind, 0.55649427130538_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1289) = (/ 0.999857106601311_dbl_kind, 0.996411230498205_dbl_kind, 0.971669398796953_dbl_kind, 0.885786841697364_dbl_kind, 0.556451516497449_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1290) = (/ 0.999857000026044_dbl_kind, 0.996408771284551_dbl_kind, 0.971649292122423_dbl_kind, 0.885714939200949_dbl_kind, 0.556408853889696_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1291) = (/ 0.999856893453932_dbl_kind, 0.996406312363295_dbl_kind, 0.971629187280938_dbl_kind, 0.885643054217327_dbl_kind, 0.556366283262337_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1292) = (/ 0.999856786884987_dbl_kind, 0.996403853734518_dbl_kind, 0.971609084274105_dbl_kind, 0.885571186747766_dbl_kind, 0.556323804396142_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1293) = (/ 0.999856680319218_dbl_kind, 0.996401395398294_dbl_kind, 0.971588983103547_dbl_kind, 0.885499336793563_dbl_kind, 0.556281417072453_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1294) = (/ 0.999856573756638_dbl_kind, 0.996398937354705_dbl_kind, 0.971568883770892_dbl_kind, 0.885427504356046_dbl_kind, 0.556239121073158_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1295) = (/ 0.999856467197257_dbl_kind, 0.996396479603834_dbl_kind, 0.971548786277783_dbl_kind, 0.88535568943657_dbl_kind, 0.556196916180705_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1296) = (/ 0.999856360641086_dbl_kind, 0.996394022145763_dbl_kind, 0.971528690625872_dbl_kind, 0.88528389203652_dbl_kind, 0.556154802178096_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1297) = (/ 0.999856254088136_dbl_kind, 0.996391564980578_dbl_kind, 0.971508596816825_dbl_kind, 0.885212112157312_dbl_kind, 0.556112778848886_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1298) = (/ 0.999856147538418_dbl_kind, 0.996389108108366_dbl_kind, 0.971488504852314_dbl_kind, 0.885140349800389_dbl_kind, 0.556070845977183_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1299) = (/ 0.999856040991945_dbl_kind, 0.996386651529215_dbl_kind, 0.971468414734027_dbl_kind, 0.885068604967226_dbl_kind, 0.556029003347638_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1300) = (/ 0.999855934448726_dbl_kind, 0.996384195243216_dbl_kind, 0.97144832646366_dbl_kind, 0.884996877659325_dbl_kind, 0.55598725074546_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1301) = (/ 0.999855827908773_dbl_kind, 0.996381739250461_dbl_kind, 0.971428240042922_dbl_kind, 0.884925167878221_dbl_kind, 0.555945587956397_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1302) = (/ 0.999855721372098_dbl_kind, 0.996379283551044_dbl_kind, 0.971408155473532_dbl_kind, 0.884853475625477_dbl_kind, 0.555904014766753_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1303) = (/ 0.999855614838713_dbl_kind, 0.996376828145061_dbl_kind, 0.971388072757222_dbl_kind, 0.884781800902686_dbl_kind, 0.555862530963362_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1304) = (/ 0.999855508308628_dbl_kind, 0.996374373032608_dbl_kind, 0.971367991895731_dbl_kind, 0.88471014371147_dbl_kind, 0.555821136333612_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1305) = (/ 0.999855401781855_dbl_kind, 0.996371918213785_dbl_kind, 0.971347912890814_dbl_kind, 0.884638504053483_dbl_kind, 0.555779830665433_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1306) = (/ 0.999855295258406_dbl_kind, 0.996369463688693_dbl_kind, 0.971327835744236_dbl_kind, 0.884566881930409_dbl_kind, 0.555738613747287_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1307) = (/ 0.999855188738293_dbl_kind, 0.996367009457433_dbl_kind, 0.971307760457771_dbl_kind, 0.88449527734396_dbl_kind, 0.555697485368179_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1308) = (/ 0.999855082221529_dbl_kind, 0.99636455552011_dbl_kind, 0.971287687033206_dbl_kind, 0.884423690295882_dbl_kind, 0.555656445317654_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1309) = (/ 0.999854975708122_dbl_kind, 0.996362101876831_dbl_kind, 0.971267615472341_dbl_kind, 0.884352120787947_dbl_kind, 0.555615493385792_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1310) = (/ 0.999854869198089_dbl_kind, 0.996359648527702_dbl_kind, 0.971247545776984_dbl_kind, 0.884280568821961_dbl_kind, 0.555574629363203_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1311) = (/ 0.999854762691437_dbl_kind, 0.996357195472834_dbl_kind, 0.971227477948956_dbl_kind, 0.88420903439976_dbl_kind, 0.555533853041036_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1312) = (/ 0.999854656188181_dbl_kind, 0.996354742712337_dbl_kind, 0.971207411990091_dbl_kind, 0.884137517523207_dbl_kind, 0.55549316421097_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1313) = (/ 0.999854549688333_dbl_kind, 0.996352290246325_dbl_kind, 0.971187347902231_dbl_kind, 0.884066018194202_dbl_kind, 0.555452562665213_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1314) = (/ 0.999854443191905_dbl_kind, 0.996349838074912_dbl_kind, 0.971167285687233_dbl_kind, 0.88399453641467_dbl_kind, 0.555412048196505_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1315) = (/ 0.999854336698908_dbl_kind, 0.996347386198214_dbl_kind, 0.971147225346962_dbl_kind, 0.883923072186571_dbl_kind, 0.555371620598112_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1316) = (/ 0.999854230209355_dbl_kind, 0.996344934616349_dbl_kind, 0.971127166883298_dbl_kind, 0.883851625511892_dbl_kind, 0.555331279663826_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1317) = (/ 0.99985412372326_dbl_kind, 0.996342483329439_dbl_kind, 0.97110711029813_dbl_kind, 0.883780196392656_dbl_kind, 0.555291025187963_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1318) = (/ 0.999854017240632_dbl_kind, 0.996340032337602_dbl_kind, 0.971087055593361_dbl_kind, 0.883708784830911_dbl_kind, 0.555250856965369_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1319) = (/ 0.999853910761487_dbl_kind, 0.996337581640966_dbl_kind, 0.971067002770901_dbl_kind, 0.883637390828742_dbl_kind, 0.555210774791405_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1320) = (/ 0.999853804285836_dbl_kind, 0.996335131239652_dbl_kind, 0.971046951832676_dbl_kind, 0.88356601438826_dbl_kind, 0.555170778461957_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1321) = (/ 0.999853697813691_dbl_kind, 0.99633268113379_dbl_kind, 0.971026902780625_dbl_kind, 0.883494655511612_dbl_kind, 0.555130867773432_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1322) = (/ 0.999853591345066_dbl_kind, 0.996330231323508_dbl_kind, 0.971006855616692_dbl_kind, 0.883423314200973_dbl_kind, 0.555091042522749_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1323) = (/ 0.999853484879973_dbl_kind, 0.996327781808935_dbl_kind, 0.970986810342839_dbl_kind, 0.883351990458551_dbl_kind, 0.555051302507352_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1324) = (/ 0.999853378418424_dbl_kind, 0.996325332590206_dbl_kind, 0.970966766961036_dbl_kind, 0.883280684286586_dbl_kind, 0.555011647525194_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1325) = (/ 0.999853271960435_dbl_kind, 0.996322883667452_dbl_kind, 0.970946725473267_dbl_kind, 0.883209395687347_dbl_kind, 0.554972077374749_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1326) = (/ 0.999853165506015_dbl_kind, 0.99632043504081_dbl_kind, 0.970926685881527_dbl_kind, 0.883138124663137_dbl_kind, 0.554932591854997_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1327) = (/ 0.999853059055179_dbl_kind, 0.996317986710419_dbl_kind, 0.97090664818782_dbl_kind, 0.883066871216291_dbl_kind, 0.554893190765437_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1328) = (/ 0.99985295260794_dbl_kind, 0.996315538676417_dbl_kind, 0.970886612394168_dbl_kind, 0.882995635349173_dbl_kind, 0.554853873906073_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1329) = (/ 0.999852846164312_dbl_kind, 0.996313090938945_dbl_kind, 0.9708665785026_dbl_kind, 0.882924417064183_dbl_kind, 0.55481464107742_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1330) = (/ 0.999852739724306_dbl_kind, 0.996310643498148_dbl_kind, 0.970846546515156_dbl_kind, 0.882853216363748_dbl_kind, 0.554775492080501_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1331) = (/ 0.999852633287938_dbl_kind, 0.996308196354168_dbl_kind, 0.970826516433893_dbl_kind, 0.882782033250332_dbl_kind, 0.554736426716843_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1332) = (/ 0.999852526855218_dbl_kind, 0.996305749507152_dbl_kind, 0.970806488260875_dbl_kind, 0.882710867726428_dbl_kind, 0.554697444788483_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1333) = (/ 0.999852420426162_dbl_kind, 0.996303302957251_dbl_kind, 0.970786461998181_dbl_kind, 0.882639719794562_dbl_kind, 0.554658546097958_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1334) = (/ 0.999852314000783_dbl_kind, 0.996300856704612_dbl_kind, 0.970766437647899_dbl_kind, 0.882568589457291_dbl_kind, 0.554619730448312_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1335) = (/ 0.999852207579095_dbl_kind, 0.996298410749389_dbl_kind, 0.970746415212132_dbl_kind, 0.882497476717206_dbl_kind, 0.554580997643082_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1336) = (/ 0.99985210116111_dbl_kind, 0.996295965091734_dbl_kind, 0.970726394692993_dbl_kind, 0.882426381576931_dbl_kind, 0.554542347486315_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1337) = (/ 0.999851994746843_dbl_kind, 0.996293519731804_dbl_kind, 0.970706376092608_dbl_kind, 0.882355304039119_dbl_kind, 0.554503779782549_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1338) = (/ 0.999851888336307_dbl_kind, 0.996291074669756_dbl_kind, 0.970686359413114_dbl_kind, 0.882284244106458_dbl_kind, 0.554465294336823_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1339) = (/ 0.999851781929517_dbl_kind, 0.996288629905749_dbl_kind, 0.970666344656662_dbl_kind, 0.882213201781669_dbl_kind, 0.554426890954671_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1340) = (/ 0.999851675526486_dbl_kind, 0.996286185439943_dbl_kind, 0.970646331825414_dbl_kind, 0.882142177067503_dbl_kind, 0.554388569442124_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1341) = (/ 0.999851569127228_dbl_kind, 0.996283741272503_dbl_kind, 0.970626320921542_dbl_kind, 0.882071169966748_dbl_kind, 0.554350329605704_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1342) = (/ 0.999851462731757_dbl_kind, 0.996281297403591_dbl_kind, 0.970606311947233_dbl_kind, 0.88200018048222_dbl_kind, 0.554312171252425_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1343) = (/ 0.999851356340087_dbl_kind, 0.996278853833375_dbl_kind, 0.970586304904685_dbl_kind, 0.88192920861677_dbl_kind, 0.554274094189795_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1344) = (/ 0.999851249952233_dbl_kind, 0.996276410562024_dbl_kind, 0.97056629979611_dbl_kind, 0.881858254373283_dbl_kind, 0.554236098225809_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1345) = (/ 0.999851143568207_dbl_kind, 0.996273967589705_dbl_kind, 0.970546296623728_dbl_kind, 0.881787317754674_dbl_kind, 0.554198183168952_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1346) = (/ 0.999851037188027_dbl_kind, 0.996271524916593_dbl_kind, 0.970526295389776_dbl_kind, 0.881716398763895_dbl_kind, 0.554160348828196_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1347) = (/ 0.999850930811704_dbl_kind, 0.99626908254286_dbl_kind, 0.9705062960965_dbl_kind, 0.881645497403927_dbl_kind, 0.554122595012998_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1348) = (/ 0.999850824439254_dbl_kind, 0.996266640468681_dbl_kind, 0.970486298746158_dbl_kind, 0.881574613677786_dbl_kind, 0.5540849215333_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1349) = (/ 0.999850718070691_dbl_kind, 0.996264198694235_dbl_kind, 0.970466303341025_dbl_kind, 0.881503747588522_dbl_kind, 0.554047328199528_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1350) = (/ 0.999850611706031_dbl_kind, 0.9962617572197_dbl_kind, 0.970446309883382_dbl_kind, 0.881432899139216_dbl_kind, 0.554009814822592_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1351) = (/ 0.999850505345287_dbl_kind, 0.996259316045257_dbl_kind, 0.970426318375525_dbl_kind, 0.881362068332984_dbl_kind, 0.553972381213878_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1352) = (/ 0.999850398988473_dbl_kind, 0.99625687517109_dbl_kind, 0.970406328819764_dbl_kind, 0.881291255172976_dbl_kind, 0.553935027185257_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1353) = (/ 0.999850292635606_dbl_kind, 0.996254434597382_dbl_kind, 0.97038634121842_dbl_kind, 0.881220459662373_dbl_kind, 0.553897752549075_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1354) = (/ 0.9998501862867_dbl_kind, 0.99625199432432_dbl_kind, 0.970366355573825_dbl_kind, 0.881149681804391_dbl_kind, 0.553860557118158_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1355) = (/ 0.999850079941769_dbl_kind, 0.996249554352093_dbl_kind, 0.970346371888326_dbl_kind, 0.881078921602281_dbl_kind, 0.553823440705806_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1356) = (/ 0.999849973600829_dbl_kind, 0.99624711468089_dbl_kind, 0.97032639016428_dbl_kind, 0.881008179059323_dbl_kind, 0.553786403125791_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1357) = (/ 0.999849867263895_dbl_kind, 0.996244675310904_dbl_kind, 0.970306410404058_dbl_kind, 0.880937454178838_dbl_kind, 0.553749444192365_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1358) = (/ 0.999849760930982_dbl_kind, 0.996242236242329_dbl_kind, 0.970286432610043_dbl_kind, 0.880866746964172_dbl_kind, 0.553712563720252_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1359) = (/ 0.999849654602106_dbl_kind, 0.996239797475359_dbl_kind, 0.97026645678463_dbl_kind, 0.880796057418712_dbl_kind, 0.553675761524635_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1360) = (/ 0.999849548277281_dbl_kind, 0.996237359010193_dbl_kind, 0.970246482930229_dbl_kind, 0.880725385545876_dbl_kind, 0.553639037421185_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1361) = (/ 0.999849441956522_dbl_kind, 0.996234920847029_dbl_kind, 0.970226511049258_dbl_kind, 0.880654731349114_dbl_kind, 0.553602391226028_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1362) = (/ 0.999849335639846_dbl_kind, 0.996232482986071_dbl_kind, 0.970206541144152_dbl_kind, 0.880584094831912_dbl_kind, 0.553565822755763_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1363) = (/ 0.999849229327268_dbl_kind, 0.99623004542752_dbl_kind, 0.970186573217357_dbl_kind, 0.880513475997792_dbl_kind, 0.553529331827455_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1364) = (/ 0.999849123018803_dbl_kind, 0.996227608171581_dbl_kind, 0.970166607271328_dbl_kind, 0.880442874850307_dbl_kind, 0.553492918258632_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1365) = (/ 0.999849016714467_dbl_kind, 0.99622517121846_dbl_kind, 0.970146643308541_dbl_kind, 0.880372291393044_dbl_kind, 0.553456581867287_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1366) = (/ 0.999848910414276_dbl_kind, 0.996222734568368_dbl_kind, 0.970126681331476_dbl_kind, 0.880301725629626_dbl_kind, 0.553420322471877_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1367) = (/ 0.999848804118246_dbl_kind, 0.996220298221512_dbl_kind, 0.970106721342629_dbl_kind, 0.880231177563711_dbl_kind, 0.553384139891318_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1368) = (/ 0.999848697826391_dbl_kind, 0.996217862178109_dbl_kind, 0.970086763344512_dbl_kind, 0.880160647198986_dbl_kind, 0.553348033944988_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1369) = (/ 0.99984859153873_dbl_kind, 0.996215426438369_dbl_kind, 0.970066807339643_dbl_kind, 0.88009013453918_dbl_kind, 0.553312004452723_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1370) = (/ 0.999848485255276_dbl_kind, 0.99621299100251_dbl_kind, 0.970046853330559_dbl_kind, 0.88001963958805_dbl_kind, 0.553276051234816_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1371) = (/ 0.999848378976047_dbl_kind, 0.99621055587075_dbl_kind, 0.970026901319806_dbl_kind, 0.879949162349391_dbl_kind, 0.553240174112019_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1372) = (/ 0.999848272701058_dbl_kind, 0.996208121043309_dbl_kind, 0.970006951309943_dbl_kind, 0.87987870282703_dbl_kind, 0.553204372905537_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1373) = (/ 0.999848166430327_dbl_kind, 0.996205686520408_dbl_kind, 0.969987003303544_dbl_kind, 0.879808261024831_dbl_kind, 0.553168647437032_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1374) = (/ 0.999848060163868_dbl_kind, 0.99620325230227_dbl_kind, 0.969967057303194_dbl_kind, 0.879737836946692_dbl_kind, 0.553132997528615_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1375) = (/ 0.999847953901699_dbl_kind, 0.996200818389122_dbl_kind, 0.969947113311491_dbl_kind, 0.879667430596545_dbl_kind, 0.553097423002853_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1376) = (/ 0.999847847643836_dbl_kind, 0.99619838478119_dbl_kind, 0.969927171331046_dbl_kind, 0.879597041978355_dbl_kind, 0.553061923682762_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1377) = (/ 0.999847741390295_dbl_kind, 0.996195951478704_dbl_kind, 0.969907231364482_dbl_kind, 0.879526671096126_dbl_kind, 0.553026499391806_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1378) = (/ 0.999847635141093_dbl_kind, 0.996193518481895_dbl_kind, 0.969887293414438_dbl_kind, 0.879456317953894_dbl_kind, 0.552991149953901_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1379) = (/ 0.999847528896247_dbl_kind, 0.996191085790995_dbl_kind, 0.969867357483561_dbl_kind, 0.879385982555729_dbl_kind, 0.552955875193406_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1380) = (/ 0.999847422655774_dbl_kind, 0.99618865340624_dbl_kind, 0.969847423574516_dbl_kind, 0.879315664905738_dbl_kind, 0.55292067493513_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1381) = (/ 0.99984731641969_dbl_kind, 0.996186221327866_dbl_kind, 0.969827491689977_dbl_kind, 0.879245365008062_dbl_kind, 0.552885549004322_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1382) = (/ 0.999847210188011_dbl_kind, 0.996183789556111_dbl_kind, 0.969807561832632_dbl_kind, 0.879175082866878_dbl_kind, 0.552850497226683_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1383) = (/ 0.999847103960756_dbl_kind, 0.996181358091216_dbl_kind, 0.969787634005183_dbl_kind, 0.879104818486395_dbl_kind, 0.552815519428346_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1384) = (/ 0.999846997737942_dbl_kind, 0.996178926933424_dbl_kind, 0.969767708210345_dbl_kind, 0.879034571870861_dbl_kind, 0.552780615435895_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1385) = (/ 0.999846891519585_dbl_kind, 0.996176496082979_dbl_kind, 0.969747784450845_dbl_kind, 0.878964343024556_dbl_kind, 0.552745785076347_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1386) = (/ 0.999846785305702_dbl_kind, 0.996174065540127_dbl_kind, 0.969727862729423_dbl_kind, 0.878894131951798_dbl_kind, 0.552711028177161_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1387) = (/ 0.999846679096311_dbl_kind, 0.996171635305116_dbl_kind, 0.969707943048832_dbl_kind, 0.878823938656937_dbl_kind, 0.552676344566236_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1388) = (/ 0.999846572891429_dbl_kind, 0.996169205378194_dbl_kind, 0.969688025411839_dbl_kind, 0.878753763144362_dbl_kind, 0.552641734071906_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1389) = (/ 0.999846466691073_dbl_kind, 0.996166775759615_dbl_kind, 0.969668109821224_dbl_kind, 0.878683605418493_dbl_kind, 0.552607196522938_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1390) = (/ 0.999846360495261_dbl_kind, 0.996164346449633_dbl_kind, 0.969648196279779_dbl_kind, 0.878613465483789_dbl_kind, 0.552572731748538_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1391) = (/ 0.999846254304011_dbl_kind, 0.996161917448502_dbl_kind, 0.969628284790309_dbl_kind, 0.878543343344744_dbl_kind, 0.552538339578344_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1392) = (/ 0.99984614811734_dbl_kind, 0.99615948875648_dbl_kind, 0.969608375355635_dbl_kind, 0.878473239005883_dbl_kind, 0.552504019842426_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1393) = (/ 0.999846041935266_dbl_kind, 0.996157060373827_dbl_kind, 0.969588467978587_dbl_kind, 0.878403152471774_dbl_kind, 0.552469772371284_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1394) = (/ 0.999845935757808_dbl_kind, 0.996154632300803_dbl_kind, 0.96956856266201_dbl_kind, 0.878333083747014_dbl_kind, 0.55243559699585_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1395) = (/ 0.999845829584981_dbl_kind, 0.996152204537673_dbl_kind, 0.969548659408763_dbl_kind, 0.878263032836238_dbl_kind, 0.552401493547483_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1396) = (/ 0.999845723416806_dbl_kind, 0.9961497770847_dbl_kind, 0.969528758221718_dbl_kind, 0.878192999744118_dbl_kind, 0.552367461857972_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1397) = (/ 0.999845617253299_dbl_kind, 0.996147349942152_dbl_kind, 0.96950885910376_dbl_kind, 0.87812298447536_dbl_kind, 0.552333501759529_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1398) = (/ 0.999845511094478_dbl_kind, 0.996144923110298_dbl_kind, 0.969488962057784_dbl_kind, 0.878052987034705_dbl_kind, 0.552299613084799_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1399) = (/ 0.999845404940363_dbl_kind, 0.996142496589409_dbl_kind, 0.969469067086704_dbl_kind, 0.87798300742693_dbl_kind, 0.552265795666842_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1400) = (/ 0.99984529879097_dbl_kind, 0.996140070379757_dbl_kind, 0.969449174193444_dbl_kind, 0.87791304565685_dbl_kind, 0.552232049339148_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1401) = (/ 0.999845192646319_dbl_kind, 0.996137644481616_dbl_kind, 0.969429283380941_dbl_kind, 0.877843101729315_dbl_kind, 0.552198373935626_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1402) = (/ 0.999845086506428_dbl_kind, 0.996135218895264_dbl_kind, 0.969409394652146_dbl_kind, 0.877773175649207_dbl_kind, 0.55216476929061_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1403) = (/ 0.999844980371315_dbl_kind, 0.996132793620978_dbl_kind, 0.969389508010022_dbl_kind, 0.877703267421449_dbl_kind, 0.552131235238848_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1404) = (/ 0.999844874241_dbl_kind, 0.996130368659039_dbl_kind, 0.969369623457549_dbl_kind, 0.877633377050998_dbl_kind, 0.552097771615513_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1405) = (/ 0.999844768115499_dbl_kind, 0.996127944009729_dbl_kind, 0.969349740997716_dbl_kind, 0.877563504542846_dbl_kind, 0.55206437825619_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1406) = (/ 0.999844661994833_dbl_kind, 0.996125519673333_dbl_kind, 0.969329860633529_dbl_kind, 0.877493649902022_dbl_kind, 0.552031054996887_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1407) = (/ 0.99984455587902_dbl_kind, 0.996123095650135_dbl_kind, 0.969309982368003_dbl_kind, 0.877423813133591_dbl_kind, 0.551997801674024_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1408) = (/ 0.999844449768079_dbl_kind, 0.996120671940425_dbl_kind, 0.969290106204171_dbl_kind, 0.877353994242652_dbl_kind, 0.551964618124436_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1409) = (/ 0.99984434366203_dbl_kind, 0.996118248544492_dbl_kind, 0.969270232145076_dbl_kind, 0.877284193234345_dbl_kind, 0.551931504185371_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1410) = (/ 0.999844237560889_dbl_kind, 0.996115825462627_dbl_kind, 0.969250360193777_dbl_kind, 0.877214410113841_dbl_kind, 0.551898459694491_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1411) = (/ 0.999844131464678_dbl_kind, 0.996113402695126_dbl_kind, 0.969230490353344_dbl_kind, 0.877144644886349_dbl_kind, 0.551865484489868_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1412) = (/ 0.999844025373416_dbl_kind, 0.996110980242281_dbl_kind, 0.969210622626862_dbl_kind, 0.877074897557115_dbl_kind, 0.551832578409985_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1413) = (/ 0.99984391928712_dbl_kind, 0.996108558104393_dbl_kind, 0.969190757017429_dbl_kind, 0.87700516813142_dbl_kind, 0.551799741293735_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1414) = (/ 0.999843813205812_dbl_kind, 0.99610613628176_dbl_kind, 0.969170893528157_dbl_kind, 0.876935456614583_dbl_kind, 0.551766972980416_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1415) = (/ 0.99984370712951_dbl_kind, 0.996103714774682_dbl_kind, 0.969151032162169_dbl_kind, 0.876865763011956_dbl_kind, 0.551734273309737_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1416) = (/ 0.999843601058233_dbl_kind, 0.996101293583465_dbl_kind, 0.969131172922606_dbl_kind, 0.876796087328931_dbl_kind, 0.55170164212181_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1417) = (/ 0.999843494992002_dbl_kind, 0.996098872708412_dbl_kind, 0.969111315812618_dbl_kind, 0.876726429570934_dbl_kind, 0.551669079257155_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1418) = (/ 0.999843388930836_dbl_kind, 0.996096452149831_dbl_kind, 0.969091460835372_dbl_kind, 0.876656789743428_dbl_kind, 0.551636584556691_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1419) = (/ 0.999843282874755_dbl_kind, 0.996094031908031_dbl_kind, 0.969071607994045_dbl_kind, 0.876587167851913_dbl_kind, 0.551604157861744_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1420) = (/ 0.999843176823778_dbl_kind, 0.996091611983323_dbl_kind, 0.96905175729183_dbl_kind, 0.876517563901924_dbl_kind, 0.551571799014042_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1421) = (/ 0.999843070777925_dbl_kind, 0.99608919237602_dbl_kind, 0.969031908731935_dbl_kind, 0.876447977899035_dbl_kind, 0.551539507855711_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1422) = (/ 0.999842964737217_dbl_kind, 0.996086773086436_dbl_kind, 0.969012062317578_dbl_kind, 0.876378409848852_dbl_kind, 0.551507284229278_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1423) = (/ 0.999842858701674_dbl_kind, 0.996084354114888_dbl_kind, 0.968992218051991_dbl_kind, 0.876308859757022_dbl_kind, 0.551475127977671_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1424) = (/ 0.999842752671315_dbl_kind, 0.996081935461696_dbl_kind, 0.968972375938423_dbl_kind, 0.876239327629227_dbl_kind, 0.551443038944207_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1425) = (/ 0.99984264664616_dbl_kind, 0.996079517127178_dbl_kind, 0.968952535980133_dbl_kind, 0.876169813471185_dbl_kind, 0.551411016972612_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1426) = (/ 0.999842540626231_dbl_kind, 0.996077099111659_dbl_kind, 0.968932698180395_dbl_kind, 0.87610031728865_dbl_kind, 0.551379061906997_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1427) = (/ 0.999842434611547_dbl_kind, 0.996074681415462_dbl_kind, 0.968912862542498_dbl_kind, 0.876030839087416_dbl_kind, 0.551347173591875_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1428) = (/ 0.99984232860213_dbl_kind, 0.996072264038914_dbl_kind, 0.968893029069741_dbl_kind, 0.875961378873308_dbl_kind, 0.551315351872147_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1429) = (/ 0.999842222597998_dbl_kind, 0.996069846982343_dbl_kind, 0.96887319776544_dbl_kind, 0.875891936652193_dbl_kind, 0.551283596593106_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1430) = (/ 0.999842116599174_dbl_kind, 0.996067430246079_dbl_kind, 0.968853368632923_dbl_kind, 0.875822512429971_dbl_kind, 0.551251907600444_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1431) = (/ 0.999842010605677_dbl_kind, 0.996065013830455_dbl_kind, 0.968833541675533_dbl_kind, 0.875753106212581_dbl_kind, 0.551220284740234_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1432) = (/ 0.999841904617529_dbl_kind, 0.996062597735804_dbl_kind, 0.968813716896626_dbl_kind, 0.875683718005999_dbl_kind, 0.551188727858944_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1433) = (/ 0.99984179863475_dbl_kind, 0.996060181962462_dbl_kind, 0.968793894299571_dbl_kind, 0.875614347816234_dbl_kind, 0.55115723680343_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1434) = (/ 0.999841692657362_dbl_kind, 0.996057766510767_dbl_kind, 0.968774073887751_dbl_kind, 0.875544995649336_dbl_kind, 0.551125811420931_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1435) = (/ 0.999841586685384_dbl_kind, 0.99605535138106_dbl_kind, 0.968754255664563_dbl_kind, 0.875475661511389_dbl_kind, 0.551094451559077_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1436) = (/ 0.999841480718839_dbl_kind, 0.996052936573683_dbl_kind, 0.968734439633419_dbl_kind, 0.875406345408516_dbl_kind, 0.551063157065882_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1437) = (/ 0.999841374757748_dbl_kind, 0.996050522088977_dbl_kind, 0.968714625797741_dbl_kind, 0.875337047346876_dbl_kind, 0.551031927789742_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1438) = (/ 0.999841268802131_dbl_kind, 0.99604810792729_dbl_kind, 0.968694814160968_dbl_kind, 0.875267767332662_dbl_kind, 0.55100076357944_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1439) = (/ 0.99984116285201_dbl_kind, 0.996045694088969_dbl_kind, 0.968675004726553_dbl_kind, 0.87519850537211_dbl_kind, 0.550969664284135_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1440) = (/ 0.999841056907407_dbl_kind, 0.996043280574363_dbl_kind, 0.968655197497962_dbl_kind, 0.875129261471487_dbl_kind, 0.550938629753378_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1441) = (/ 0.999840950968343_dbl_kind, 0.996040867383824_dbl_kind, 0.968635392478672_dbl_kind, 0.8750600356371_dbl_kind, 0.550907659837087_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1442) = (/ 0.99984084503484_dbl_kind, 0.996038454517706_dbl_kind, 0.968615589672178_dbl_kind, 0.87499082787529_dbl_kind, 0.550876754385571_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1443) = (/ 0.999840739106918_dbl_kind, 0.996036041976363_dbl_kind, 0.968595789081987_dbl_kind, 0.87492163819244_dbl_kind, 0.550845913249509_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1444) = (/ 0.999840633184601_dbl_kind, 0.996033629760153_dbl_kind, 0.968575990711619_dbl_kind, 0.874852466594966_dbl_kind, 0.55081513627996_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1445) = (/ 0.999840527267909_dbl_kind, 0.996031217869434_dbl_kind, 0.968556194564609_dbl_kind, 0.874783313089319_dbl_kind, 0.550784423328362_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1446) = (/ 0.999840421356866_dbl_kind, 0.99602880630457_dbl_kind, 0.968536400644505_dbl_kind, 0.874714177681993_dbl_kind, 0.550753774246522_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1447) = (/ 0.999840315451491_dbl_kind, 0.996026395065921_dbl_kind, 0.96851660895487_dbl_kind, 0.874645060379515_dbl_kind, 0.550723188886629_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1448) = (/ 0.999840209551808_dbl_kind, 0.996023984153853_dbl_kind, 0.96849681949928_dbl_kind, 0.874575961188449_dbl_kind, 0.550692667101239_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1449) = (/ 0.999840103657839_dbl_kind, 0.996021573568733_dbl_kind, 0.968477032281324_dbl_kind, 0.874506880115397_dbl_kind, 0.550662208743281_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1450) = (/ 0.999839997769606_dbl_kind, 0.996019163310929_dbl_kind, 0.968457247304606_dbl_kind, 0.874437817166997_dbl_kind, 0.55063181366606_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1451) = (/ 0.999839891887133_dbl_kind, 0.996016753380814_dbl_kind, 0.968437464572744_dbl_kind, 0.874368772349926_dbl_kind, 0.550601481723247_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1452) = (/ 0.999839786010439_dbl_kind, 0.996014343778759_dbl_kind, 0.96841768408937_dbl_kind, 0.874299745670896_dbl_kind, 0.550571212768882_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1453) = (/ 0.999839680139548_dbl_kind, 0.996011934505139_dbl_kind, 0.968397905858128_dbl_kind, 0.874230737136656_dbl_kind, 0.550541006657379_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1454) = (/ 0.999839574274484_dbl_kind, 0.99600952556033_dbl_kind, 0.968378129882678_dbl_kind, 0.874161746753993_dbl_kind, 0.550510863243511_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1455) = (/ 0.999839468415267_dbl_kind, 0.996007116944711_dbl_kind, 0.968358356166693_dbl_kind, 0.874092774529732_dbl_kind, 0.550480782382425_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1456) = (/ 0.999839362561922_dbl_kind, 0.996004708658663_dbl_kind, 0.968338584713859_dbl_kind, 0.874023820470734_dbl_kind, 0.550450763929629_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1457) = (/ 0.999839256714471_dbl_kind, 0.996002300702567_dbl_kind, 0.968318815527878_dbl_kind, 0.873954884583895_dbl_kind, 0.550420807740996_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1458) = (/ 0.999839150872936_dbl_kind, 0.995999893076809_dbl_kind, 0.968299048612465_dbl_kind, 0.873885966876151_dbl_kind, 0.550390913672766_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1459) = (/ 0.999839045037341_dbl_kind, 0.995997485781774_dbl_kind, 0.968279283971349_dbl_kind, 0.873817067354474_dbl_kind, 0.550361081581537_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1460) = (/ 0.999838939207709_dbl_kind, 0.99599507881785_dbl_kind, 0.968259521608271_dbl_kind, 0.873748186025874_dbl_kind, 0.55033131132427_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1461) = (/ 0.999838833384063_dbl_kind, 0.995992672185428_dbl_kind, 0.968239761526987_dbl_kind, 0.873679322897396_dbl_kind, 0.550301602758291_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1462) = (/ 0.999838727566425_dbl_kind, 0.9959902658849_dbl_kind, 0.96822000373127_dbl_kind, 0.873610477976124_dbl_kind, 0.550271955741279_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1463) = (/ 0.999838621754821_dbl_kind, 0.995987859916659_dbl_kind, 0.968200248224902_dbl_kind, 0.873541651269178_dbl_kind, 0.550242370131277_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1464) = (/ 0.999838515949272_dbl_kind, 0.995985454281102_dbl_kind, 0.968180495011684_dbl_kind, 0.873472842783716_dbl_kind, 0.550212845786682_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1465) = (/ 0.999838410149801_dbl_kind, 0.995983048978626_dbl_kind, 0.968160744095425_dbl_kind, 0.873404052526932_dbl_kind, 0.550183382566253_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1466) = (/ 0.999838304356434_dbl_kind, 0.995980644009632_dbl_kind, 0.968140995479954_dbl_kind, 0.873335280506058_dbl_kind, 0.5501539803291_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1467) = (/ 0.999838198569193_dbl_kind, 0.99597823937452_dbl_kind, 0.968121249169109_dbl_kind, 0.873266526728362_dbl_kind, 0.550124638934689_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1468) = (/ 0.999838092788102_dbl_kind, 0.995975835073695_dbl_kind, 0.968101505166746_dbl_kind, 0.873197791201152_dbl_kind, 0.550095358242842_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1469) = (/ 0.999837987013184_dbl_kind, 0.995973431107561_dbl_kind, 0.968081763476731_dbl_kind, 0.873129073931769_dbl_kind, 0.550066138113734_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1470) = (/ 0.999837881244465_dbl_kind, 0.995971027476527_dbl_kind, 0.968062024102947_dbl_kind, 0.873060374927594_dbl_kind, 0.55003697840789_dbl_kind /)
   iceSingleScatterAlbedoDirect(1:5 ,1471) = (/ 0.999837775481967_dbl_kind, 0.995968624181001_dbl_kind, 0.968042287049291_dbl_kind, 0.872991694196043_dbl_kind, 0.550007878986191_dbl_kind /)

   iceSingleScatterAlbedoDiffuse(1:5 ,1) = (/ 0.99999652090225_dbl_kind, 0.999902541003866_dbl_kind, 0.999252020157292_dbl_kind, 0.995412713638889_dbl_kind, 0.949846089277264_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,2) = (/ 0.999996410054143_dbl_kind, 0.999899503226746_dbl_kind, 0.999228399618862_dbl_kind, 0.995303298087803_dbl_kind, 0.948509250075099_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,3) = (/ 0.999996299750575_dbl_kind, 0.999896473321404_dbl_kind, 0.99920477600596_dbl_kind, 0.995194732374268_dbl_kind, 0.947183340709259_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,4) = (/ 0.99999618993626_dbl_kind, 0.999893450431926_dbl_kind, 0.99918113312949_dbl_kind, 0.995086927144935_dbl_kind, 0.945868086144487_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,5) = (/ 0.999996080547224_dbl_kind, 0.999890433632407_dbl_kind, 0.999157457639793_dbl_kind, 0.994979802131183_dbl_kind, 0.944563224486756_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,6) = (/ 0.999995971513353_dbl_kind, 0.999887421976042_dbl_kind, 0.999133739332467_dbl_kind, 0.994873285280009_dbl_kind, 0.943268506620646_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,7) = (/ 0.999995862760764_dbl_kind, 0.999884414539864_dbl_kind, 0.999109971247562_dbl_kind, 0.994767312029555_dbl_kind, 0.941983695382215_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,8) = (/ 0.999995754213958_dbl_kind, 0.999881410462392_dbl_kind, 0.999086149600374_dbl_kind, 0.994661824708817_dbl_kind, 0.940708564569374_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,9) = (/ 0.999995645797729_dbl_kind, 0.999878408972699_dbl_kind, 0.999062273582888_dbl_kind, 0.994556772033718_dbl_kind, 0.939442897974212_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,10) = (/ 0.999995537438816_dbl_kind, 0.999875409410544_dbl_kind, 0.999038345072532_dbl_kind, 0.994452108671675_dbl_kind, 0.938186488532096_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,11) = (/ 0.999995429067298_dbl_kind, 0.999872411237903_dbl_kind, 0.999014368280627_dbl_kind, 0.99434779485037_dbl_kind, 0.936939137620628_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,12) = (/ 0.999995320617729_dbl_kind, 0.999869414042851_dbl_kind, 0.998990349367928_dbl_kind, 0.994243795991595_dbl_kind, 0.935700654503753_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,13) = (/ 0.999995212030031_dbl_kind, 0.999866417536997_dbl_kind, 0.998966296049477_dbl_kind, 0.994140082356331_dbl_kind, 0.93447085589643_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,14) = (/ 0.99999510325014_dbl_kind, 0.999863421547835_dbl_kind, 0.99894221720615_dbl_kind, 0.994036628692172_dbl_kind, 0.933249565618159_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,15) = (/ 0.999994994230469_dbl_kind, 0.999860426007369_dbl_kind, 0.998918122515939_dbl_kind, 0.993933413878208_dbl_kind, 0.932036614304369_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,16) = (/ 0.999994884930152_dbl_kind, 0.999857430938265_dbl_kind, 0.998894022114204_dbl_kind, 0.993830420565761_dbl_kind, 0.930831839149533_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,17) = (/ 0.999994775315146_dbl_kind, 0.999854436438646_dbl_kind, 0.998869926289007_dbl_kind, 0.99372763481573_dbl_kind, 0.929635083662521_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,18) = (/ 0.999994665358178_dbl_kind, 0.99985144266647_dbl_kind, 0.998845845214982_dbl_kind, 0.993625045734948_dbl_kind, 0.928446197421538_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,19) = (/ 0.999994555038579_dbl_kind, 0.999848449824228_dbl_kind, 0.998821788727136_dbl_kind, 0.993522645115012_dbl_kind, 0.927265035821632_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,20) = (/ 0.99999444434203_dbl_kind, 0.999845458144543_dbl_kind, 0.998797766134329_dbl_kind, 0.993420427077533_dbl_kind, 0.926091459812696_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,21) = (/ 0.999994333260216_dbl_kind, 0.999842467877046_dbl_kind, 0.998773786070956_dbl_kind, 0.993318387729923_dbl_kind, 0.9249253356289_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,22) = (/ 0.999994221790443_dbl_kind, 0.999839479276826_dbl_kind, 0.998749856384439_dbl_kind, 0.993216524835637_dbl_kind, 0.923766534512734_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,23) = (/ 0.999994109935196_dbl_kind, 0.999836492594539_dbl_kind, 0.998725984055546_dbl_kind, 0.993114837502444_dbl_kind, 0.922614932437868_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,24) = (/ 0.999993997701684_dbl_kind, 0.999833508068264_dbl_kind, 0.998702175148166_dbl_kind, 0.993013325891694_dbl_kind, 0.921470409835264_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,25) = (/ 0.99999388510136_dbl_kind, 0.999830525917023_dbl_kind, 0.998678434784983_dbl_kind, 0.99291199095103_dbl_kind, 0.920332851326813_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,26) = (/ 0.999993772149446_dbl_kind, 0.999827546335907_dbl_kind, 0.998654767145464_dbl_kind, 0.992810834172239_dbl_kind, 0.919202145470054_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,27) = (/ 0.99999365886445_dbl_kind, 0.999824569492653_dbl_kind, 0.998631175482641_dbl_kind, 0.992709857375342_dbl_kind, 0.91807818451689_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,28) = (/ 0.999993545267703_dbl_kind, 0.999821595525516_dbl_kind, 0.998607662155357_dbl_kind, 0.992609062519367_dbl_kind, 0.916960864188264_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,29) = (/ 0.999993431382901_dbl_kind, 0.999818624542291_dbl_kind, 0.998584228672848_dbl_kind, 0.992508451539675_dbl_kind, 0.915850083466017_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,30) = (/ 0.99999331723568_dbl_kind, 0.999815656620279_dbl_kind, 0.998560875748808_dbl_kind, 0.992408026211212_dbl_kind, 0.914745744402413_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,31) = (/ 0.999993202853193_dbl_kind, 0.999812691807057_dbl_kind, 0.998537603362396_dbl_kind, 0.992307788036584_dbl_kind, 0.913647751947186_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,32) = (/ 0.999993088263737_dbl_kind, 0.999809730121874_dbl_kind, 0.998514410823891_dbl_kind, 0.992207738157586_dbl_kind, 0.912556013791496_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,33) = (/ 0.999992973496378_dbl_kind, 0.999806771557545_dbl_kind, 0.998491296843058_dbl_kind, 0.992107877288416_dbl_kind, 0.911470440227775_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,34) = (/ 0.999992858580627_dbl_kind, 0.999803816082697_dbl_kind, 0.998468259598521_dbl_kind, 0.992008205668736_dbl_kind, 0.910390944024266_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,35) = (/ 0.999992743546132_dbl_kind, 0.999800863644254_dbl_kind, 0.998445296806749_dbl_kind, 0.991908723034527_dbl_kind, 0.909317440312792_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,36) = (/ 0.999992628422393_dbl_kind, 0.999797914170069_dbl_kind, 0.998422405789482_dbl_kind, 0.991809428604657_dbl_kind, 0.908249846488327_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,37) = (/ 0.999992513238518_dbl_kind, 0.999794967571603_dbl_kind, 0.998399583538682_dbl_kind, 0.991710321081086_dbl_kind, 0.907188082118862_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,38) = (/ 0.999992398022995_dbl_kind, 0.999792023746599_dbl_kind, 0.998376826778272_dbl_kind, 0.991611398660643_dbl_kind, 0.906132068864145_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,39) = (/ 0.999992282803493_dbl_kind, 0.999789082581672_dbl_kind, 0.998354132022134_dbl_kind, 0.991512659056419_dbl_kind, 0.905081730401947_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,40) = (/ 0.999992167606692_dbl_kind, 0.999786143954798_dbl_kind, 0.998331495627974_dbl_kind, 0.991414099526906_dbl_kind, 0.904036992360624_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,41) = (/ 0.999992052458132_dbl_kind, 0.999783207737638_dbl_kind, 0.998308913846818_dbl_kind, 0.991315716911179_dbl_kind, 0.902997782256833_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,42) = (/ 0.999991937382088_dbl_kind, 0.999780273797694_dbl_kind, 0.998286382868009_dbl_kind, 0.991217507668518_dbl_kind, 0.90196402943746_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,43) = (/ 0.999991822401468_dbl_kind, 0.999777342000278_dbl_kind, 0.99826389885969_dbl_kind, 0.991119467921065_dbl_kind, 0.900935665024904_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,44) = (/ 0.999991707537731_dbl_kind, 0.999774412210272_dbl_kind, 0.998241458004797_dbl_kind, 0.991021593498265_dbl_kind, 0.899912621864986_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,45) = (/ 0.999991592810822_dbl_kind, 0.999771484293697_dbl_kind, 0.998219056532714_dbl_kind, 0.990923879981993_dbl_kind, 0.898894834476914_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,46) = (/ 0.999991478239131_dbl_kind, 0.999768558119063_dbl_kind, 0.99819669074674_dbl_kind, 0.99082632275141_dbl_kind, 0.897882239004825_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,47) = (/ 0.999991363839461_dbl_kind, 0.999765633558546_dbl_kind, 0.998174357047574_dbl_kind, 0.990728917026805_dbl_kind, 0.896874773170513_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,48) = (/ 0.99999124962702_dbl_kind, 0.99976271048896_dbl_kind, 0.998152051953067_dbl_kind, 0.990631657911731_dbl_kind, 0.895872376227055_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,49) = (/ 0.999991135615418_dbl_kind, 0.999759788792555_dbl_kind, 0.998129772114491_dbl_kind, 0.990534540432962_dbl_kind, 0.89487498891319_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,50) = (/ 0.999991021816687_dbl_kind, 0.999756868357656_dbl_kind, 0.998107514329592_dbl_kind, 0.990437559577848_dbl_kind, 0.893882553408211_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,51) = (/ 0.9999909082413_dbl_kind, 0.99975394907914_dbl_kind, 0.998085275552696_dbl_kind, 0.990340710328807_dbl_kind, 0.892895013287379_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,52) = (/ 0.999990794898213_dbl_kind, 0.999751030858791_dbl_kind, 0.998063052902156_dbl_kind, 0.990243987694739_dbl_kind, 0.891912313477792_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,53) = (/ 0.999990681794904_dbl_kind, 0.999748113605511_dbl_kind, 0.998040843665384_dbl_kind, 0.990147386739289_dbl_kind, 0.890934400214682_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,54) = (/ 0.999990568937429_dbl_kind, 0.999745197235436_dbl_kind, 0.998018645301754_dbl_kind, 0.990050902605887_dbl_kind, 0.889961220998227_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,55) = (/ 0.999990456330476_dbl_kind, 0.999742281671948_dbl_kind, 0.997996455443608_dbl_kind, 0.989954530539612_dbl_kind, 0.888992724550883_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,56) = (/ 0.999990343977435_dbl_kind, 0.999739366845606_dbl_kind, 0.9979742718956_dbl_kind, 0.989858265905964_dbl_kind, 0.888028860775323_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,57) = (/ 0.999990231880456_dbl_kind, 0.999736452694005_dbl_kind, 0.997952092632626_dbl_kind, 0.989762104206637_dbl_kind, 0.88706958071307_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,58) = (/ 0.999990120040531_dbl_kind, 0.999733539161577_dbl_kind, 0.99792991579651_dbl_kind, 0.989666041092457_dbl_kind, 0.886114836503855_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,59) = (/ 0.999990008457558_dbl_kind, 0.999730626199336_dbl_kind, 0.997907739691673_dbl_kind, 0.989570072373657_dbl_kind, 0.885164581345854_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,60) = (/ 0.999989897130422_dbl_kind, 0.999727713764598_dbl_kind, 0.997885562779938_dbl_kind, 0.989474194027683_dbl_kind, 0.884218769456816_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,61) = (/ 0.999989786057066_dbl_kind, 0.999724801820653_dbl_kind, 0.997863383674645_dbl_kind, 0.989378402204711_dbl_kind, 0.883277356036192_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,62) = (/ 0.99998967523457_dbl_kind, 0.999721890336434_dbl_kind, 0.99784120113422_dbl_kind, 0.989282693231118_dbl_kind, 0.882340297228336_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,63) = (/ 0.999989564659228_dbl_kind, 0.999718979286153_dbl_kind, 0.997819014055325_dbl_kind, 0.989187063611078_dbl_kind, 0.881407550086788_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,64) = (/ 0.999989454326614_dbl_kind, 0.999716068648942_dbl_kind, 0.997796821465704_dbl_kind, 0.989091510026522_dbl_kind, 0.880479072539767_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,65) = (/ 0.999989344231667_dbl_kind, 0.999713158408486_dbl_kind, 0.997774622516845_dbl_kind, 0.988996029335641_dbl_kind, 0.879554823356831_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,66) = (/ 0.999989234368745_dbl_kind, 0.999710248552654_dbl_kind, 0.997752416476528_dbl_kind, 0.988900618570138_dbl_kind, 0.878634762116829_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,67) = (/ 0.999989124731711_dbl_kind, 0.999707339073146_dbl_kind, 0.997730202721353_dbl_kind, 0.988805274931413_dbl_kind, 0.877718849177038_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,68) = (/ 0.999989015313987_dbl_kind, 0.999704429965137_dbl_kind, 0.997707980729306_dbl_kind, 0.988709995785843_dbl_kind, 0.876807045643657_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,69) = (/ 0.999988906108622_dbl_kind, 0.999701521226943_dbl_kind, 0.997685750072437_dbl_kind, 0.988614778659336_dbl_kind, 0.875899313343513_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,70) = (/ 0.999988797108349_dbl_kind, 0.999698612859691_dbl_kind, 0.99766351040968_dbl_kind, 0.988519621231299_dbl_kind, 0.874995614797072_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,71) = (/ 0.999988688305644_dbl_kind, 0.999695704867021_dbl_kind, 0.997641261479875_dbl_kind, 0.988424521328148_dbl_kind, 0.874095913192701_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,72) = (/ 0.999988579692776_dbl_kind, 0.999692797254788_dbl_kind, 0.997619003095011_dbl_kind, 0.9883294769165_dbl_kind, 0.873200172362171_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,73) = (/ 0.999988471261862_dbl_kind, 0.999689890030787_dbl_kind, 0.997596735133723_dbl_kind, 0.988234486096147_dbl_kind, 0.872308356757381_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,74) = (/ 0.99998836300491_dbl_kind, 0.999686983204507_dbl_kind, 0.997574457535074_dbl_kind, 0.988139547092911_dbl_kind, 0.871420431428259_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,75) = (/ 0.999988254913861_dbl_kind, 0.999684076786888_dbl_kind, 0.99755217029261_dbl_kind, 0.988044658251466_dbl_kind, 0.870536362001835_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,76) = (/ 0.999988146980635_dbl_kind, 0.99968117079011_dbl_kind, 0.997529873448732_dbl_kind, 0.987949818028208_dbl_kind, 0.869656114662401_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,77) = (/ 0.999988039197161_dbl_kind, 0.99967826522739_dbl_kind, 0.997507567089372_dbl_kind, 0.987855024984224_dbl_kind, 0.868779656132768_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,78) = (/ 0.999987931555411_dbl_kind, 0.999675360112806_dbl_kind, 0.997485251338975_dbl_kind, 0.987760277778427_dbl_kind, 0.867906953656532_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,79) = (/ 0.999987824047436_dbl_kind, 0.999672455461132_dbl_kind, 0.997462926355802_dbl_kind, 0.987665575160893_dbl_kind, 0.867037974981354_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,80) = (/ 0.999987716665386_dbl_kind, 0.999669551287692_dbl_kind, 0.997440592327533_dbl_kind, 0.987570915966445_dbl_kind, 0.866172688343139_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,81) = (/ 0.999987609401539_dbl_kind, 0.999666647608233_dbl_kind, 0.997418249467183_dbl_kind, 0.987476299108502_dbl_kind, 0.865311062451126_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,82) = (/ 0.999987502248321_dbl_kind, 0.999663744438809_dbl_kind, 0.997395898009312_dbl_kind, 0.98738172357321_dbl_kind, 0.86445306647384_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,83) = (/ 0.999987395198322_dbl_kind, 0.99966084179568_dbl_kind, 0.997373538206521_dbl_kind, 0.987287188413908_dbl_kind, 0.863598670025796_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,84) = (/ 0.999987288244319_dbl_kind, 0.999657939695223_dbl_kind, 0.997351170326238_dbl_kind, 0.987192692745875_dbl_kind, 0.862747843154996_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,85) = (/ 0.999987181379282_dbl_kind, 0.999655038153864_dbl_kind, 0.997328794647762_dbl_kind, 0.987098235741428_dbl_kind, 0.86190055633111_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,86) = (/ 0.999987074596391_dbl_kind, 0.999652137188007_dbl_kind, 0.997306411459578_dbl_kind, 0.98700381662532_dbl_kind, 0.861056780434318_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,87) = (/ 0.999986967889043_dbl_kind, 0.999649236813984_dbl_kind, 0.997284021056908_dbl_kind, 0.986909434670475_dbl_kind, 0.860216486744775_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,88) = (/ 0.999986861250863_dbl_kind, 0.999646337048014_dbl_kind, 0.997261623739507_dbl_kind, 0.986815089194025_dbl_kind, 0.859379646932654_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,89) = (/ 0.999986754675707_dbl_kind, 0.999643437906162_dbl_kind, 0.997239219809672_dbl_kind, 0.986720779553657_dbl_kind, 0.858546233048721_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,90) = (/ 0.999986648157663_dbl_kind, 0.999640539404315_dbl_kind, 0.997216809570468_dbl_kind, 0.986626505144253_dbl_kind, 0.85771621751542_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,91) = (/ 0.999986541691063_dbl_kind, 0.99963764155816_dbl_kind, 0.99719439332415_dbl_kind, 0.986532265394814_dbl_kind, 0.856889573118407_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,92) = (/ 0.999986435270476_dbl_kind, 0.999634744383173_dbl_kind, 0.997171971370771_dbl_kind, 0.986438059765654_dbl_kind, 0.856066272998523_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,93) = (/ 0.99998632889071_dbl_kind, 0.999631847894602_dbl_kind, 0.997149544006959_dbl_kind, 0.986343887745842_dbl_kind, 0.855246290644169_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,94) = (/ 0.999986222546814_dbl_kind, 0.999628952107467_dbl_kind, 0.997127111524859_dbl_kind, 0.986249748850899_dbl_kind, 0.85442959988405_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,95) = (/ 0.999986116234074_dbl_kind, 0.999626057036559_dbl_kind, 0.997104674211228_dbl_kind, 0.986155642620701_dbl_kind, 0.853616174880241_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,96) = (/ 0.999986009948009_dbl_kind, 0.999623162696438_dbl_kind, 0.997082232346662_dbl_kind, 0.986061568617604_dbl_kind, 0.852805990121591_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,97) = (/ 0.999985903684369_dbl_kind, 0.99962026910144_dbl_kind, 0.997059786204952_dbl_kind, 0.985967526424762_dbl_kind, 0.851999020417401_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,98) = (/ 0.999985797439131_dbl_kind, 0.999617376265686_dbl_kind, 0.997037336052564_dbl_kind, 0.985873515644613_dbl_kind, 0.851195240891383_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,99) = (/ 0.999985691208491_dbl_kind, 0.999614484203083_dbl_kind, 0.99701488214822_dbl_kind, 0.98577953589754_dbl_kind, 0.850394626975844_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,100) = (/ 0.999985584988862_dbl_kind, 0.999611592927342_dbl_kind, 0.996992424742573_dbl_kind, 0.985685586820682_dbl_kind, 0.849597154406122_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,101) = (/ 0.999985478776867_dbl_kind, 0.999608702451985_dbl_kind, 0.996969964077986_dbl_kind, 0.985591668066861_dbl_kind, 0.848802799215213_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,102) = (/ 0.999985372569333_dbl_kind, 0.999605812790352_dbl_kind, 0.99694750038837_dbl_kind, 0.985497779303661_dbl_kind, 0.848011537728598_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,103) = (/ 0.999985266363281_dbl_kind, 0.999602923955619_dbl_kind, 0.996925033899118_dbl_kind, 0.985403920212584_dbl_kind, 0.847223346559245_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,104) = (/ 0.999985160155927_dbl_kind, 0.999600035960804_dbl_kind, 0.996902564827085_dbl_kind, 0.985310090488336_dbl_kind, 0.846438202602784_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,105) = (/ 0.999985053944667_dbl_kind, 0.999597148818781_dbl_kind, 0.996880093380627_dbl_kind, 0.985216289838166_dbl_kind, 0.845656083032821_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,106) = (/ 0.999984947727074_dbl_kind, 0.999594262542289_dbl_kind, 0.99685761975971_dbl_kind, 0.98512251798131_dbl_kind, 0.844876965296402_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,107) = (/ 0.999984841500891_dbl_kind, 0.999591377143941_dbl_kind, 0.996835144156033_dbl_kind, 0.98502877464848_dbl_kind, 0.844100827109607_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,108) = (/ 0.999984735264022_dbl_kind, 0.999588492636235_dbl_kind, 0.996812666753216_dbl_kind, 0.984935059581425_dbl_kind, 0.843327646453255_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,109) = (/ 0.999984629014528_dbl_kind, 0.999585609031559_dbl_kind, 0.996790187727007_dbl_kind, 0.984841372532532_dbl_kind, 0.842557401568737_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,110) = (/ 0.999984522750617_dbl_kind, 0.999582726342205_dbl_kind, 0.996767707245523_dbl_kind, 0.984747713264471_dbl_kind, 0.841790070953947_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,111) = (/ 0.999984416470638_dbl_kind, 0.999579844580366_dbl_kind, 0.996745225469509_dbl_kind, 0.984654081549879_dbl_kind, 0.841025633359302_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,112) = (/ 0.999984310173075_dbl_kind, 0.999576963758147_dbl_kind, 0.996722742552621_dbl_kind, 0.984560477171064_dbl_kind, 0.840264067783874_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,113) = (/ 0.99998420385654_dbl_kind, 0.999574083887569_dbl_kind, 0.996700258641724_dbl_kind, 0.984466899919747_dbl_kind, 0.839505353471591_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,114) = (/ 0.999984097519765_dbl_kind, 0.99957120498057_dbl_kind, 0.996677773877201_dbl_kind, 0.984373349596805_dbl_kind, 0.838749469907526_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,115) = (/ 0.999983991161597_dbl_kind, 0.999568327049009_dbl_kind, 0.996655288393265_dbl_kind, 0.984279826012045_dbl_kind, 0.837996396814269_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,116) = (/ 0.999983884780993_dbl_kind, 0.999565450104663_dbl_kind, 0.996632802318287_dbl_kind, 0.98418632898398_dbl_kind, 0.83724611414836_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,117) = (/ 0.999983778377012_dbl_kind, 0.999562574159233_dbl_kind, 0.996610315775117_dbl_kind, 0.984092858339611_dbl_kind, 0.836498602096801_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,118) = (/ 0.999983671948808_dbl_kind, 0.999559699224336_dbl_kind, 0.996587828881413_dbl_kind, 0.983999413914221_dbl_kind, 0.835753841073636_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,119) = (/ 0.999983565495629_dbl_kind, 0.999556825311511_dbl_kind, 0.996565341749958_dbl_kind, 0.983905995551167_dbl_kind, 0.83501181171658_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,120) = (/ 0.999983459016807_dbl_kind, 0.999553952432205_dbl_kind, 0.996542854488988_dbl_kind, 0.983812603101673_dbl_kind, 0.834272494883725_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,121) = (/ 0.999983352511757_dbl_kind, 0.99955108059778_dbl_kind, 0.996520367202502_dbl_kind, 0.983719236424625_dbl_kind, 0.833535871650307_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,122) = (/ 0.999983245979964_dbl_kind, 0.999548209819501_dbl_kind, 0.996497879990567_dbl_kind, 0.98362589538636_dbl_kind, 0.832801923305503_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,123) = (/ 0.999983139420992_dbl_kind, 0.999545340108536_dbl_kind, 0.996475392949624_dbl_kind, 0.983532579860451_dbl_kind, 0.83207063134932_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,124) = (/ 0.999983032834466_dbl_kind, 0.999542471475946_dbl_kind, 0.996452906172769_dbl_kind, 0.983439289727497_dbl_kind, 0.831341977489505_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,125) = (/ 0.999982926220074_dbl_kind, 0.999539603932678_dbl_kind, 0.996430419750044_dbl_kind, 0.983346024874902_dbl_kind, 0.830615943638526_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,126) = (/ 0.999982819577563_dbl_kind, 0.999536737489563_dbl_kind, 0.996407933768693_dbl_kind, 0.983252785196645_dbl_kind, 0.829892511910596_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,127) = (/ 0.999982712906737_dbl_kind, 0.9995338721573_dbl_kind, 0.996385448313431_dbl_kind, 0.983159570593056_dbl_kind, 0.829171664618747_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,128) = (/ 0.999982606207443_dbl_kind, 0.999531007946459_dbl_kind, 0.996362963466681_dbl_kind, 0.98306638097058_dbl_kind, 0.828453384271957_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,129) = (/ 0.999982499479584_dbl_kind, 0.999528144867461_dbl_kind, 0.996340479308808_dbl_kind, 0.982973216241541_dbl_kind, 0.827737653572308_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,130) = (/ 0.999982392723098_dbl_kind, 0.999525282930578_dbl_kind, 0.996317995918343_dbl_kind, 0.982880076323899_dbl_kind, 0.82702445541221_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,131) = (/ 0.99998228593797_dbl_kind, 0.999522422145921_dbl_kind, 0.996295513372184_dbl_kind, 0.982786961141006_dbl_kind, 0.826313772871662_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,132) = (/ 0.999982179124222_dbl_kind, 0.999519562523431_dbl_kind, 0.996273031745793_dbl_kind, 0.982693870621355_dbl_kind, 0.825605589215544_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,133) = (/ 0.999982072281908_dbl_kind, 0.999516704072871_dbl_kind, 0.996250551113374_dbl_kind, 0.982600804698336_dbl_kind, 0.824899887890984_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,134) = (/ 0.999981965411117_dbl_kind, 0.999513846803814_dbl_kind, 0.996228071548045_dbl_kind, 0.982507763309975_dbl_kind, 0.824196652524729_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,135) = (/ 0.999981858511968_dbl_kind, 0.99951099072564_dbl_kind, 0.99620559312199_dbl_kind, 0.982414746398689_dbl_kind, 0.823495866920593_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,136) = (/ 0.999981751584604_dbl_kind, 0.999508135847521_dbl_kind, 0.996183115906598_dbl_kind, 0.982321753911022_dbl_kind, 0.822797515056917_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,137) = (/ 0.999981644629197_dbl_kind, 0.999505282178418_dbl_kind, 0.996160639972599_dbl_kind, 0.982228785797401_dbl_kind, 0.822101581084095_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,138) = (/ 0.999981537645943_dbl_kind, 0.999502429727066_dbl_kind, 0.996138165390179_dbl_kind, 0.982135842011876_dbl_kind, 0.821408049322124_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,139) = (/ 0.999981430635057_dbl_kind, 0.999499578501971_dbl_kind, 0.996115692229085_dbl_kind, 0.982042922511872_dbl_kind, 0.820716904258198_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,140) = (/ 0.999981323596774_dbl_kind, 0.9994967285114_dbl_kind, 0.996093220558721_dbl_kind, 0.981950027257937_dbl_kind, 0.820028130544342_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,141) = (/ 0.999981216531351_dbl_kind, 0.999493879763374_dbl_kind, 0.996070750448232_dbl_kind, 0.981857156213496_dbl_kind, 0.819341712995087_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,142) = (/ 0.999981109439057_dbl_kind, 0.999491032265659_dbl_kind, 0.996048281966574_dbl_kind, 0.981764309344609_dbl_kind, 0.818657636585168_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,143) = (/ 0.999981002320179_dbl_kind, 0.999488186025761_dbl_kind, 0.996025815182581_dbl_kind, 0.981671486619726_dbl_kind, 0.817975886447293_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,144) = (/ 0.999980895175015_dbl_kind, 0.999485341050918_dbl_kind, 0.996003350165018_dbl_kind, 0.98157868800946_dbl_kind, 0.817296447869903_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,145) = (/ 0.999980788003882_dbl_kind, 0.999482497348093_dbl_kind, 0.99598088698262_dbl_kind, 0.98148591348635_dbl_kind, 0.816619306295007_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,146) = (/ 0.9999806808071_dbl_kind, 0.99947965492397_dbl_kind, 0.995958425704136_dbl_kind, 0.981393163024642_dbl_kind, 0.815944447316037_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,147) = (/ 0.999980573585007_dbl_kind, 0.999476813784949_dbl_kind, 0.995935966398346_dbl_kind, 0.981300436600063_dbl_kind, 0.815271856675732_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,148) = (/ 0.999980466337946_dbl_kind, 0.999473973937138_dbl_kind, 0.99591350913409_dbl_kind, 0.981207734189618_dbl_kind, 0.814601520264071_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,149) = (/ 0.999980359066271_dbl_kind, 0.999471135386351_dbl_kind, 0.995891053980275_dbl_kind, 0.981115055771377_dbl_kind, 0.813933424116233_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,150) = (/ 0.999980251770341_dbl_kind, 0.999468298138102_dbl_kind, 0.99586860100588_dbl_kind, 0.981022401324282_dbl_kind, 0.813267554410581_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,151) = (/ 0.999980144450525_dbl_kind, 0.999465462197606_dbl_kind, 0.995846150279957_dbl_kind, 0.980929770827952_dbl_kind, 0.812603897466711_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,152) = (/ 0.999980037107196_dbl_kind, 0.999462627569769_dbl_kind, 0.995823701871624_dbl_kind, 0.980837164262501_dbl_kind, 0.811942439743485_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,153) = (/ 0.999979929740733_dbl_kind, 0.999459794259191_dbl_kind, 0.995801255850049_dbl_kind, 0.980744581608363_dbl_kind, 0.811283167837145_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,154) = (/ 0.999979822351522_dbl_kind, 0.999456962270161_dbl_kind, 0.99577881228444_dbl_kind, 0.980652022846124_dbl_kind, 0.810626068479424_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,155) = (/ 0.999979714939948_dbl_kind, 0.999454131606656_dbl_kind, 0.995756371244014_dbl_kind, 0.980559487956366_dbl_kind, 0.809971128535706_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,156) = (/ 0.999979607506407_dbl_kind, 0.999451302272341_dbl_kind, 0.995733932797981_dbl_kind, 0.980466976919508_dbl_kind, 0.809318335003218_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,157) = (/ 0.99997950005129_dbl_kind, 0.999448474270567_dbl_kind, 0.995711497015504_dbl_kind, 0.980374489715671_dbl_kind, 0.808667675009229_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,158) = (/ 0.999979392574998_dbl_kind, 0.999445647604369_dbl_kind, 0.995689063965675_dbl_kind, 0.98028202632454_dbl_kind, 0.808019135809311_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,159) = (/ 0.99997928507793_dbl_kind, 0.999442822276471_dbl_kind, 0.995666633717475_dbl_kind, 0.980189586725241_dbl_kind, 0.807372704785609_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,160) = (/ 0.999979177560487_dbl_kind, 0.999439998289281_dbl_kind, 0.995644206339731_dbl_kind, 0.980097170896216_dbl_kind, 0.806728369445143_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,161) = (/ 0.999979070023074_dbl_kind, 0.999437175644896_dbl_kind, 0.995621781901088_dbl_kind, 0.980004778815125_dbl_kind, 0.806086117418144_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,162) = (/ 0.999978962466096_dbl_kind, 0.999434354345101_dbl_kind, 0.995599360469951_dbl_kind, 0.979912410458737_dbl_kind, 0.80544593645641_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,163) = (/ 0.999978854889955_dbl_kind, 0.999431534391374_dbl_kind, 0.995576942114453_dbl_kind, 0.979820065802838_dbl_kind, 0.804807814431694_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,164) = (/ 0.999978747295059_dbl_kind, 0.999428715784884_dbl_kind, 0.995554526902404_dbl_kind, 0.97972774482215_dbl_kind, 0.804171739334112_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,165) = (/ 0.999978639681813_dbl_kind, 0.999425898526498_dbl_kind, 0.995532114901242_dbl_kind, 0.97963544749025_dbl_kind, 0.803537699270602_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,166) = (/ 0.999978532050622_dbl_kind, 0.999423082616778_dbl_kind, 0.995509706177993_dbl_kind, 0.979543173779508_dbl_kind, 0.802905682463364_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,167) = (/ 0.999978424401891_dbl_kind, 0.999420268055991_dbl_kind, 0.995487300799218_dbl_kind, 0.979450923661013_dbl_kind, 0.802275677248369_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,168) = (/ 0.999978316736026_dbl_kind, 0.999417454844109_dbl_kind, 0.995464898830967_dbl_kind, 0.979358697104534_dbl_kind, 0.801647672073872_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,169) = (/ 0.999978209053428_dbl_kind, 0.999414642980811_dbl_kind, 0.995442500338732_dbl_kind, 0.979266494078463_dbl_kind, 0.80102165549895_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,170) = (/ 0.999978101354499_dbl_kind, 0.999411832465493_dbl_kind, 0.995420105387395_dbl_kind, 0.979174314549777_dbl_kind, 0.800397616192066_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,171) = (/ 0.999977993639641_dbl_kind, 0.999409023297268_dbl_kind, 0.995397714041188_dbl_kind, 0.979082158484009_dbl_kind, 0.799775542929663_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,172) = (/ 0.999977885909253_dbl_kind, 0.999406215474968_dbl_kind, 0.995375326363639_dbl_kind, 0.978990025845219_dbl_kind, 0.799155424594772_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,173) = (/ 0.999977778163731_dbl_kind, 0.999403408997159_dbl_kind, 0.995352942417534_dbl_kind, 0.978897916595973_dbl_kind, 0.798537250175645_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,174) = (/ 0.99997767040347_dbl_kind, 0.999400603862135_dbl_kind, 0.995330562264861_dbl_kind, 0.978805830697329_dbl_kind, 0.797921008764414_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,175) = (/ 0.999977562628864_dbl_kind, 0.99939780006793_dbl_kind, 0.995308185966779_dbl_kind, 0.978713768108834_dbl_kind, 0.79730668955577_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,176) = (/ 0.999977454840303_dbl_kind, 0.999394997612322_dbl_kind, 0.995285813583564_dbl_kind, 0.978621728788511_dbl_kind, 0.796694281845663_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,177) = (/ 0.999977347038174_dbl_kind, 0.999392196492838_dbl_kind, 0.995263445174575_dbl_kind, 0.978529712692876_dbl_kind, 0.796083775030023_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,178) = (/ 0.999977239222864_dbl_kind, 0.99938939670676_dbl_kind, 0.995241080798208_dbl_kind, 0.978437719776932_dbl_kind, 0.795475158603502_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,179) = (/ 0.999977131394755_dbl_kind, 0.999386598251134_dbl_kind, 0.995218720511865_dbl_kind, 0.97834574999419_dbl_kind, 0.794868422158232_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,180) = (/ 0.999977023554226_dbl_kind, 0.999383801122771_dbl_kind, 0.995196364371906_dbl_kind, 0.978253803296685_dbl_kind, 0.794263555382618_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,181) = (/ 0.999976915701653_dbl_kind, 0.999381005318256_dbl_kind, 0.995174012433626_dbl_kind, 0.978161879634993_dbl_kind, 0.793660548060127_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,182) = (/ 0.99997680783741_dbl_kind, 0.999378210833957_dbl_kind, 0.995151664751209_dbl_kind, 0.978069978958262_dbl_kind, 0.79305939006812_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,183) = (/ 0.999976699961866_dbl_kind, 0.999375417666027_dbl_kind, 0.995129321377706_dbl_kind, 0.977978101214239_dbl_kind, 0.792460071376681_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,184) = (/ 0.999976592075387_dbl_kind, 0.999372625810411_dbl_kind, 0.995106982364996_dbl_kind, 0.977886246349303_dbl_kind, 0.791862582047488_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,185) = (/ 0.999976484178335_dbl_kind, 0.999369835262856_dbl_kind, 0.995084647763765_dbl_kind, 0.977794414308498_dbl_kind, 0.791266912232674_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,186) = (/ 0.999976376271069_dbl_kind, 0.999367046018918_dbl_kind, 0.995062317623474_dbl_kind, 0.97770260503558_dbl_kind, 0.790673052173732_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,187) = (/ 0.999976268353944_dbl_kind, 0.999364258073959_dbl_kind, 0.995039991992336_dbl_kind, 0.977610818473051_dbl_kind, 0.790080992200427_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,188) = (/ 0.99997616042731_dbl_kind, 0.999361471423168_dbl_kind, 0.995017670917293_dbl_kind, 0.977519054562204_dbl_kind, 0.789490722729714_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,189) = (/ 0.999976052491515_dbl_kind, 0.999358686061558_dbl_kind, 0.994995354443995_dbl_kind, 0.977427313243179_dbl_kind, 0.78890223426469_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,190) = (/ 0.999975944546899_dbl_kind, 0.999355901983976_dbl_kind, 0.994973042616778_dbl_kind, 0.977335594454998_dbl_kind, 0.788315517393555_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,191) = (/ 0.999975836593802_dbl_kind, 0.999353119185106_dbl_kind, 0.994950735478652_dbl_kind, 0.977243898135624_dbl_kind, 0.787730562788591_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,192) = (/ 0.999975728632557_dbl_kind, 0.999350337659484_dbl_kind, 0.994928433071278_dbl_kind, 0.977152224222018_dbl_kind, 0.787147361205152_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,193) = (/ 0.999975620663494_dbl_kind, 0.999347557401495_dbl_kind, 0.994906135434962_dbl_kind, 0.977060572650183_dbl_kind, 0.786565903480672_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,194) = (/ 0.999975512686936_dbl_kind, 0.999344778405385_dbl_kind, 0.994883842608635_dbl_kind, 0.976968943355223_dbl_kind, 0.785986180533689_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,195) = (/ 0.999975404703205_dbl_kind, 0.999342000665268_dbl_kind, 0.99486155462985_dbl_kind, 0.976877336271401_dbl_kind, 0.785408183362884_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,196) = (/ 0.999975296712615_dbl_kind, 0.999339224175128_dbl_kind, 0.994839271534769_dbl_kind, 0.976785751332197_dbl_kind, 0.784831903046139_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,197) = (/ 0.999975188715477_dbl_kind, 0.999336448928831_dbl_kind, 0.994816993358154_dbl_kind, 0.976694188470364_dbl_kind, 0.784257330739599_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,198) = (/ 0.999975080712098_dbl_kind, 0.999333674920127_dbl_kind, 0.99479472013337_dbl_kind, 0.976602647617985_dbl_kind, 0.783684457676751_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,199) = (/ 0.999974972702777_dbl_kind, 0.999330902142659_dbl_kind, 0.994772451892375_dbl_kind, 0.976511128706538_dbl_kind, 0.783113275167518_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,200) = (/ 0.99997486468781_dbl_kind, 0.999328130589967_dbl_kind, 0.994750188665717_dbl_kind, 0.976419631666949_dbl_kind, 0.782543774597386_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,201) = (/ 0.999974756667488_dbl_kind, 0.999325360255493_dbl_kind, 0.994727930482538_dbl_kind, 0.976328156429653_dbl_kind, 0.781975947426493_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,202) = (/ 0.999974648642099_dbl_kind, 0.999322591132594_dbl_kind, 0.994705677370573_dbl_kind, 0.976236702924659_dbl_kind, 0.781409785188795_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,203) = (/ 0.999974540611922_dbl_kind, 0.999319823214539_dbl_kind, 0.994683429356153_dbl_kind, 0.976145271081601_dbl_kind, 0.780845279491197_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,204) = (/ 0.999974432577232_dbl_kind, 0.99931705649452_dbl_kind, 0.994661186464207_dbl_kind, 0.976053860829802_dbl_kind, 0.780282422012708_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,205) = (/ 0.999974324538301_dbl_kind, 0.999314290965656_dbl_kind, 0.994638948718271_dbl_kind, 0.975962472098331_dbl_kind, 0.77972120450363_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,206) = (/ 0.999974216495393_dbl_kind, 0.999311526620998_dbl_kind, 0.994616716140492_dbl_kind, 0.975871104816065_dbl_kind, 0.779161618784724_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,207) = (/ 0.99997410844877_dbl_kind, 0.999308763453536_dbl_kind, 0.994594488751636_dbl_kind, 0.975779758911737_dbl_kind, 0.778603656746421_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,208) = (/ 0.999974000398685_dbl_kind, 0.999306001456205_dbl_kind, 0.994572266571097_dbl_kind, 0.975688434314007_dbl_kind, 0.778047310348016_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,209) = (/ 0.999973892345388_dbl_kind, 0.999303240621886_dbl_kind, 0.99455004961691_dbl_kind, 0.975597130951504_dbl_kind, 0.777492571616892_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,210) = (/ 0.999973784289124_dbl_kind, 0.999300480943415_dbl_kind, 0.994527837905758_dbl_kind, 0.975505848752892_dbl_kind, 0.776939432647752_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,211) = (/ 0.999973676230132_dbl_kind, 0.999297722413585_dbl_kind, 0.994505631452988_dbl_kind, 0.975414587646915_dbl_kind, 0.776387885601855_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,212) = (/ 0.999973568168646_dbl_kind, 0.999294965025156_dbl_kind, 0.994483430272623_dbl_kind, 0.975323347562458_dbl_kind, 0.775837922706266_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,213) = (/ 0.999973460104894_dbl_kind, 0.999292208770853_dbl_kind, 0.994461234377375_dbl_kind, 0.975232128428598_dbl_kind, 0.775289536253126_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,214) = (/ 0.999973352039099_dbl_kind, 0.999289453643375_dbl_kind, 0.994439043778664_dbl_kind, 0.975140930174648_dbl_kind, 0.774742718598906_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,215) = (/ 0.99997324397148_dbl_kind, 0.999286699635397_dbl_kind, 0.99441685848663_dbl_kind, 0.975049752730217_dbl_kind, 0.774197462163714_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,216) = (/ 0.999973135902251_dbl_kind, 0.999283946739577_dbl_kind, 0.994394678510153_dbl_kind, 0.974958596025251_dbl_kind, 0.773653759430559_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,217) = (/ 0.999973027831616_dbl_kind, 0.999281194948555_dbl_kind, 0.994372503856869_dbl_kind, 0.974867459990081_dbl_kind, 0.773111602944678_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,218) = (/ 0.999972919759781_dbl_kind, 0.999278444254963_dbl_kind, 0.994350334533188_dbl_kind, 0.974776344555475_dbl_kind, 0.772570985312822_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,219) = (/ 0.999972811686941_dbl_kind, 0.999275694651427_dbl_kind, 0.994328170544317_dbl_kind, 0.974685249652674_dbl_kind, 0.772031899202598_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,220) = (/ 0.999972703613288_dbl_kind, 0.999272946130567_dbl_kind, 0.994306011894272_dbl_kind, 0.974594175213442_dbl_kind, 0.771494337341787_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,221) = (/ 0.999972595539011_dbl_kind, 0.999270198685008_dbl_kind, 0.994283858585907_dbl_kind, 0.974503121170103_dbl_kind, 0.770958292517678_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,222) = (/ 0.999972487464289_dbl_kind, 0.999267452307373_dbl_kind, 0.994261710620926_dbl_kind, 0.974412087455583_dbl_kind, 0.770423757576431_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,223) = (/ 0.999972379389301_dbl_kind, 0.999264706990299_dbl_kind, 0.994239567999914_dbl_kind, 0.974321074003454_dbl_kind, 0.769890725422417_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,224) = (/ 0.999972271314217_dbl_kind, 0.999261962726428_dbl_kind, 0.99421743072235_dbl_kind, 0.974230080747957_dbl_kind, 0.769359189017588_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,225) = (/ 0.999972163239206_dbl_kind, 0.999259219508419_dbl_kind, 0.994195298786632_dbl_kind, 0.974139107624053_dbl_kind, 0.768829141380858_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,226) = (/ 0.999972055164426_dbl_kind, 0.999256477328946_dbl_kind, 0.994173172190099_dbl_kind, 0.974048154567451_dbl_kind, 0.768300575587466_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,227) = (/ 0.999971947090038_dbl_kind, 0.999253736180704_dbl_kind, 0.994151050929058_dbl_kind, 0.973957221514641_dbl_kind, 0.767773484768384_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,228) = (/ 0.999971839016193_dbl_kind, 0.99925099605641_dbl_kind, 0.994128934998801_dbl_kind, 0.973866308402921_dbl_kind, 0.767247862109704_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,229) = (/ 0.999971730943037_dbl_kind, 0.999248256948805_dbl_kind, 0.994106824393628_dbl_kind, 0.973775415170436_dbl_kind, 0.766723700852043_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,230) = (/ 0.999971622870712_dbl_kind, 0.999245518850656_dbl_kind, 0.994084719106879_dbl_kind, 0.973684541756199_dbl_kind, 0.766200994289957_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,231) = (/ 0.999971514799358_dbl_kind, 0.999242781754763_dbl_kind, 0.994062619130943_dbl_kind, 0.973593688100118_dbl_kind, 0.765679735771363_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,232) = (/ 0.999971406729107_dbl_kind, 0.999240045653955_dbl_kind, 0.994040524457297_dbl_kind, 0.973502854143028_dbl_kind, 0.765159918696967_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,233) = (/ 0.999971298660089_dbl_kind, 0.999237310541098_dbl_kind, 0.994018435076519_dbl_kind, 0.973412039826702_dbl_kind, 0.764641536519696_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,234) = (/ 0.999971190592426_dbl_kind, 0.99923457640909_dbl_kind, 0.993996350978317_dbl_kind, 0.973321245093888_dbl_kind, 0.764124582744138_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,235) = (/ 0.999971082526241_dbl_kind, 0.999231843250871_dbl_kind, 0.99397427215155_dbl_kind, 0.973230469888321_dbl_kind, 0.763609050926_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,236) = (/ 0.999970974461647_dbl_kind, 0.999229111059419_dbl_kind, 0.993952198584255_dbl_kind, 0.973139714154742_dbl_kind, 0.763094934671553_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,237) = (/ 0.999970866398756_dbl_kind, 0.999226379827753_dbl_kind, 0.993930130263666_dbl_kind, 0.973048977838919_dbl_kind, 0.762582227637104_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,238) = (/ 0.999970758337675_dbl_kind, 0.999223649548937_dbl_kind, 0.993908067176246_dbl_kind, 0.972958260887665_dbl_kind, 0.762070923528462_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,239) = (/ 0.999970650278508_dbl_kind, 0.999220920216078_dbl_kind, 0.9938860093077_dbl_kind, 0.972867563248849_dbl_kind, 0.761561016100414_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,240) = (/ 0.999970542221352_dbl_kind, 0.999218191822329_dbl_kind, 0.993863956643007_dbl_kind, 0.972776884871412_dbl_kind, 0.761052499156207_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,241) = (/ 0.999970434166303_dbl_kind, 0.999215464360892_dbl_kind, 0.993841909166441_dbl_kind, 0.97268622570538_dbl_kind, 0.760545366547042_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,242) = (/ 0.999970326113453_dbl_kind, 0.999212737825016_dbl_kind, 0.993819866861593_dbl_kind, 0.972595585701875_dbl_kind, 0.760039612171563_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,243) = (/ 0.999970218062887_dbl_kind, 0.999210012208002_dbl_kind, 0.993797829711396_dbl_kind, 0.972504964813123_dbl_kind, 0.759535229975361_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,244) = (/ 0.999970110014689_dbl_kind, 0.9992072875032_dbl_kind, 0.993775797698147_dbl_kind, 0.972414362992467_dbl_kind, 0.759032213950485_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,245) = (/ 0.99997000196894_dbl_kind, 0.999204563704011_dbl_kind, 0.993753770803532_dbl_kind, 0.972323780194369_dbl_kind, 0.758530558134953_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,246) = (/ 0.999969893925715_dbl_kind, 0.999201840803891_dbl_kind, 0.993731749008644_dbl_kind, 0.972233216374423_dbl_kind, 0.758030256612273_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,247) = (/ 0.999969785885088_dbl_kind, 0.999199118796349_dbl_kind, 0.99370973229401_dbl_kind, 0.972142671489353_dbl_kind, 0.757531303510963_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,248) = (/ 0.999969677847126_dbl_kind, 0.99919639767495_dbl_kind, 0.993687720639612_dbl_kind, 0.97205214549702_dbl_kind, 0.757033693004094_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,249) = (/ 0.999969569811898_dbl_kind, 0.999193677433308_dbl_kind, 0.99366571402491_dbl_kind, 0.971961638356432_dbl_kind, 0.756537419308819_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,250) = (/ 0.999969461779464_dbl_kind, 0.999190958065101_dbl_kind, 0.993643712428859_dbl_kind, 0.971871150027731_dbl_kind, 0.756042476685912_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,251) = (/ 0.999969353749885_dbl_kind, 0.999188239564056_dbl_kind, 0.993621715829934_dbl_kind, 0.97178068047221_dbl_kind, 0.755548859439325_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,252) = (/ 0.999969245723216_dbl_kind, 0.999185521923963_dbl_kind, 0.993599724206154_dbl_kind, 0.9716902296523_dbl_kind, 0.755056561915738_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,253) = (/ 0.999969137699513_dbl_kind, 0.999182805138663_dbl_kind, 0.993577737535097_dbl_kind, 0.971599797531576_dbl_kind, 0.754565578504116_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,254) = (/ 0.999969029678825_dbl_kind, 0.999180089202059_dbl_kind, 0.993555755793924_dbl_kind, 0.971509384074754_dbl_kind, 0.754075903635274_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,255) = (/ 0.999968921661199_dbl_kind, 0.999177374108109_dbl_kind, 0.993533778959398_dbl_kind, 0.971418989247686_dbl_kind, 0.753587531781447_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,256) = (/ 0.999968813646681_dbl_kind, 0.999174659850831_dbl_kind, 0.993511807007905_dbl_kind, 0.971328613017362_dbl_kind, 0.753100457455869_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,257) = (/ 0.999968705635312_dbl_kind, 0.9991719464243_dbl_kind, 0.993489839915472_dbl_kind, 0.971238255351893_dbl_kind, 0.752614675212339_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,258) = (/ 0.999968597627133_dbl_kind, 0.999169233822651_dbl_kind, 0.993467877657788_dbl_kind, 0.971147916220522_dbl_kind, 0.752130179644818_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,259) = (/ 0.999968489622181_dbl_kind, 0.999166522040074_dbl_kind, 0.99344592021022_dbl_kind, 0.971057595593603_dbl_kind, 0.75164696538702_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,260) = (/ 0.999968381620489_dbl_kind, 0.999163811070821_dbl_kind, 0.993423967547838_dbl_kind, 0.970967293442605_dbl_kind, 0.751165027111993_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,261) = (/ 0.999968273622092_dbl_kind, 0.9991611009092_dbl_kind, 0.993402019645422_dbl_kind, 0.970877009740099_dbl_kind, 0.750684359531727_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,262) = (/ 0.999968165627018_dbl_kind, 0.999158391549579_dbl_kind, 0.993380076477493_dbl_kind, 0.970786744459749_dbl_kind, 0.75020495739676_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,263) = (/ 0.999968057635296_dbl_kind, 0.999155682986385_dbl_kind, 0.99335813801832_dbl_kind, 0.970696497576305_dbl_kind, 0.74972681549578_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,264) = (/ 0.999967949646952_dbl_kind, 0.9991529752141_dbl_kind, 0.993336204241942_dbl_kind, 0.970606269065596_dbl_kind, 0.749249928655242_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,265) = (/ 0.999967841662009_dbl_kind, 0.99915026822727_dbl_kind, 0.993314275122185_dbl_kind, 0.970516058904514_dbl_kind, 0.748774291738979_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,266) = (/ 0.999967733680489_dbl_kind, 0.99914756202049_dbl_kind, 0.993292350632676_dbl_kind, 0.970425867071007_dbl_kind, 0.748299899647833_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,267) = (/ 0.999967625702412_dbl_kind, 0.999144856588422_dbl_kind, 0.99327043074686_dbl_kind, 0.970335693544067_dbl_kind, 0.747826747319277_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,268) = (/ 0.999967517727797_dbl_kind, 0.99914215192578_dbl_kind, 0.993248515438019_dbl_kind, 0.97024553830372_dbl_kind, 0.747354829727042_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,269) = (/ 0.999967409756659_dbl_kind, 0.999139448027335_dbl_kind, 0.993226604679279_dbl_kind, 0.97015540133101_dbl_kind, 0.746884141880754_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,270) = (/ 0.999967301789013_dbl_kind, 0.999136744887919_dbl_kind, 0.993204698443636_dbl_kind, 0.970065282607989_dbl_kind, 0.746414678825582_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,271) = (/ 0.999967193824873_dbl_kind, 0.999134042502415_dbl_kind, 0.99318279670396_dbl_kind, 0.969975182117703_dbl_kind, 0.745946435641867_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,272) = (/ 0.99996708586425_dbl_kind, 0.999131340865766_dbl_kind, 0.993160899433019_dbl_kind, 0.969885099844181_dbl_kind, 0.745479407444774_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,273) = (/ 0.999966977907154_dbl_kind, 0.99912863997297_dbl_kind, 0.993139006603482_dbl_kind, 0.969795035772415_dbl_kind, 0.745013589383953_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,274) = (/ 0.999966869953594_dbl_kind, 0.999125939819079_dbl_kind, 0.993117118187943_dbl_kind, 0.969704989888355_dbl_kind, 0.744548976643178_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,275) = (/ 0.999966762003578_dbl_kind, 0.999123240399202_dbl_kind, 0.993095234158929_dbl_kind, 0.969614962178884_dbl_kind, 0.744085564440019_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,276) = (/ 0.999966654057112_dbl_kind, 0.999120541708501_dbl_kind, 0.993073354488911_dbl_kind, 0.969524952631811_dbl_kind, 0.743623348025503_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,277) = (/ 0.999966546114201_dbl_kind, 0.999117843742192_dbl_kind, 0.99305147915032_dbl_kind, 0.969434961235855_dbl_kind, 0.743162322683773_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,278) = (/ 0.999966438174849_dbl_kind, 0.999115146495547_dbl_kind, 0.993029608115559_dbl_kind, 0.969344987980625_dbl_kind, 0.742702483731777_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,279) = (/ 0.999966330239059_dbl_kind, 0.999112449963886_dbl_kind, 0.993007741357011_dbl_kind, 0.969255032856609_dbl_kind, 0.742243826518917_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,280) = (/ 0.999966222306832_dbl_kind, 0.999109754142588_dbl_kind, 0.992985878847057_dbl_kind, 0.969165095855157_dbl_kind, 0.74178634642675_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,281) = (/ 0.999966114378171_dbl_kind, 0.999107059027078_dbl_kind, 0.992964020558079_dbl_kind, 0.969075176968464_dbl_kind, 0.741330038868662_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,282) = (/ 0.999966006453073_dbl_kind, 0.999104364612837_dbl_kind, 0.992942166462478_dbl_kind, 0.968985276189557_dbl_kind, 0.740874899289553_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,283) = (/ 0.999965898531541_dbl_kind, 0.999101670895395_dbl_kind, 0.992920316532679_dbl_kind, 0.968895393512272_dbl_kind, 0.740420923165522_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,284) = (/ 0.999965790613571_dbl_kind, 0.999098977870333_dbl_kind, 0.992898470741146_dbl_kind, 0.968805528931246_dbl_kind, 0.739968106003572_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,285) = (/ 0.999965682699162_dbl_kind, 0.999096285533282_dbl_kind, 0.992876629060384_dbl_kind, 0.968715682441895_dbl_kind, 0.739516443341295_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,286) = (/ 0.999965574788309_dbl_kind, 0.999093593879923_dbl_kind, 0.99285479146296_dbl_kind, 0.9686258540404_dbl_kind, 0.739065930746577_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,287) = (/ 0.999965466881012_dbl_kind, 0.999090902905984_dbl_kind, 0.9928329579215_dbl_kind, 0.968536043723689_dbl_kind, 0.738616563817301_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,288) = (/ 0.999965358977264_dbl_kind, 0.999088212607243_dbl_kind, 0.992811128408709_dbl_kind, 0.968446251489419_dbl_kind, 0.738168338181055_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,289) = (/ 0.999965251077062_dbl_kind, 0.999085522979524_dbl_kind, 0.992789302897367_dbl_kind, 0.968356477335964_dbl_kind, 0.737721249494836_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,290) = (/ 0.999965143180403_dbl_kind, 0.999082834018701_dbl_kind, 0.992767481360351_dbl_kind, 0.968266721262392_dbl_kind, 0.737275293444771_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,291) = (/ 0.999965035287276_dbl_kind, 0.999080145720691_dbl_kind, 0.992745663770634_dbl_kind, 0.968176983268454_dbl_kind, 0.736830465745827_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,292) = (/ 0.999964927397681_dbl_kind, 0.999077458081463_dbl_kind, 0.992723850101291_dbl_kind, 0.968087263354565_dbl_kind, 0.736386762141539_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,293) = (/ 0.999964819511608_dbl_kind, 0.999074771097022_dbl_kind, 0.992702040325515_dbl_kind, 0.967997561521785_dbl_kind, 0.735944178403721_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,294) = (/ 0.999964711629052_dbl_kind, 0.999072084763427_dbl_kind, 0.992680234416617_dbl_kind, 0.967907877771804_dbl_kind, 0.735502710332207_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,295) = (/ 0.999964603750008_dbl_kind, 0.999069399076776_dbl_kind, 0.992658432348036_dbl_kind, 0.967818212106928_dbl_kind, 0.735062353754568_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,296) = (/ 0.999964495874467_dbl_kind, 0.999066714033214_dbl_kind, 0.992636634093341_dbl_kind, 0.967728564530057_dbl_kind, 0.734623104525846_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,297) = (/ 0.999964388002423_dbl_kind, 0.999064029628926_dbl_kind, 0.992614839626247_dbl_kind, 0.967638935044674_dbl_kind, 0.734184958528296_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,298) = (/ 0.999964280133867_dbl_kind, 0.999061345860142_dbl_kind, 0.992593048920609_dbl_kind, 0.967549323654827_dbl_kind, 0.73374791167112_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,299) = (/ 0.999964172268795_dbl_kind, 0.999058662723133_dbl_kind, 0.992571261950436_dbl_kind, 0.967459730365108_dbl_kind, 0.733311959890207_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,300) = (/ 0.999964064407198_dbl_kind, 0.999055980214211_dbl_kind, 0.992549478689896_dbl_kind, 0.967370155180642_dbl_kind, 0.73287709914788_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,301) = (/ 0.999963956549068_dbl_kind, 0.999053298329732_dbl_kind, 0.992527699113315_dbl_kind, 0.967280598107071_dbl_kind, 0.732443325432642_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,302) = (/ 0.999963848694399_dbl_kind, 0.999050617066087_dbl_kind, 0.992505923195188_dbl_kind, 0.967191059150534_dbl_kind, 0.732010634758921_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,303) = (/ 0.999963740843182_dbl_kind, 0.999047936419713_dbl_kind, 0.992484150910185_dbl_kind, 0.967101538317655_dbl_kind, 0.731579023166839_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,304) = (/ 0.999963632995411_dbl_kind, 0.999045256387082_dbl_kind, 0.992462382233149_dbl_kind, 0.967012035615524_dbl_kind, 0.731148486721948_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,305) = (/ 0.999963525151079_dbl_kind, 0.999042576964706_dbl_kind, 0.992440617139105_dbl_kind, 0.966922551051685_dbl_kind, 0.730719021514996_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,306) = (/ 0.999963417310179_dbl_kind, 0.999039898149137_dbl_kind, 0.992418855603264_dbl_kind, 0.966833084634119_dbl_kind, 0.730290623661695_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,307) = (/ 0.999963309472703_dbl_kind, 0.999037219936961_dbl_kind, 0.992397097601025_dbl_kind, 0.966743636371225_dbl_kind, 0.729863289302474_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,308) = (/ 0.999963201638644_dbl_kind, 0.999034542324804_dbl_kind, 0.992375343107981_dbl_kind, 0.96665420627181_dbl_kind, 0.729437014602253_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,309) = (/ 0.999963093807998_dbl_kind, 0.999031865309331_dbl_kind, 0.99235359209992_dbl_kind, 0.966564794345074_dbl_kind, 0.729011795750206_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,310) = (/ 0.999962985980755_dbl_kind, 0.999029188887237_dbl_kind, 0.992331844552831_dbl_kind, 0.966475400600591_dbl_kind, 0.728587628959543_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,311) = (/ 0.999962878156912_dbl_kind, 0.999026513055258_dbl_kind, 0.992310100442905_dbl_kind, 0.9663860250483_dbl_kind, 0.72816451046727_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,312) = (/ 0.99996277033646_dbl_kind, 0.999023837810163_dbl_kind, 0.992288359746538_dbl_kind, 0.966296667698482_dbl_kind, 0.72774243653398_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,313) = (/ 0.999962662519396_dbl_kind, 0.999021163148757_dbl_kind, 0.992266622440334_dbl_kind, 0.966207328561758_dbl_kind, 0.727321403443617_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,314) = (/ 0.999962554705713_dbl_kind, 0.999018489067878_dbl_kind, 0.992244888501113_dbl_kind, 0.966118007649064_dbl_kind, 0.726901407503279_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,315) = (/ 0.999962446895407_dbl_kind, 0.999015815564399_dbl_kind, 0.9922231579059_dbl_kind, 0.966028704971642_dbl_kind, 0.726482445042978_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,316) = (/ 0.999962339088471_dbl_kind, 0.999013142635225_dbl_kind, 0.992201430631942_dbl_kind, 0.965939420541027_dbl_kind, 0.726064512415446_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,317) = (/ 0.999962231284902_dbl_kind, 0.999010470277296_dbl_kind, 0.992179706656701_dbl_kind, 0.965850154369033_dbl_kind, 0.725647605995917_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,318) = (/ 0.999962123484696_dbl_kind, 0.999007798487583_dbl_kind, 0.992157985957861_dbl_kind, 0.965760906467737_dbl_kind, 0.725231722181917_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,319) = (/ 0.999962015687849_dbl_kind, 0.999005127263089_dbl_kind, 0.992136268513324_dbl_kind, 0.965671676849472_dbl_kind, 0.724816857393056_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,320) = (/ 0.999961907894357_dbl_kind, 0.999002456600851_dbl_kind, 0.992114554301217_dbl_kind, 0.965582465526807_dbl_kind, 0.724403008070837_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,321) = (/ 0.999961800104216_dbl_kind, 0.998999786497934_dbl_kind, 0.992092843299891_dbl_kind, 0.96549327251254_dbl_kind, 0.723990170678434_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,322) = (/ 0.999961692317426_dbl_kind, 0.998997116951435_dbl_kind, 0.992071135487923_dbl_kind, 0.965404097819683_dbl_kind, 0.723578341700514_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,323) = (/ 0.999961584533981_dbl_kind, 0.998994447958484_dbl_kind, 0.992049430844116_dbl_kind, 0.965314941461452_dbl_kind, 0.723167517643023_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,324) = (/ 0.999961476753881_dbl_kind, 0.998991779516236_dbl_kind, 0.992027729347502_dbl_kind, 0.965225803451254_dbl_kind, 0.722757695033003_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,325) = (/ 0.999961368977124_dbl_kind, 0.998989111621882_dbl_kind, 0.992006030977341_dbl_kind, 0.965136683802672_dbl_kind, 0.722348870418393_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,326) = (/ 0.999961261203709_dbl_kind, 0.998986444272635_dbl_kind, 0.991984335713123_dbl_kind, 0.965047582529463_dbl_kind, 0.721941040367841_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,327) = (/ 0.999961153433634_dbl_kind, 0.998983777465745_dbl_kind, 0.991962643534569_dbl_kind, 0.964958499645535_dbl_kind, 0.721534201470513_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,328) = (/ 0.9999610456669_dbl_kind, 0.998981111198484_dbl_kind, 0.99194095442163_dbl_kind, 0.964869435164945_dbl_kind, 0.721128350335915_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,329) = (/ 0.999960937903506_dbl_kind, 0.998978445468155_dbl_kind, 0.991919268354488_dbl_kind, 0.964780389101881_dbl_kind, 0.720723483593696_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,330) = (/ 0.999960830143453_dbl_kind, 0.998975780272091_dbl_kind, 0.991897585313558_dbl_kind, 0.964691361470661_dbl_kind, 0.720319597893475_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,331) = (/ 0.999960722386741_dbl_kind, 0.998973115607647_dbl_kind, 0.991875905279484_dbl_kind, 0.964602352285714_dbl_kind, 0.719916689904664_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,332) = (/ 0.999960614633371_dbl_kind, 0.998970451472211_dbl_kind, 0.991854228233146_dbl_kind, 0.964513361561574_dbl_kind, 0.71951475631628_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,333) = (/ 0.999960506883345_dbl_kind, 0.998967787863196_dbl_kind, 0.991832554155653_dbl_kind, 0.96442438931287_dbl_kind, 0.719113793836775_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,334) = (/ 0.999960399136665_dbl_kind, 0.998965124778039_dbl_kind, 0.991810883028347_dbl_kind, 0.964335435554313_dbl_kind, 0.718713799193864_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,335) = (/ 0.999960291393335_dbl_kind, 0.998962462214207_dbl_kind, 0.991789214832802_dbl_kind, 0.964246500300693_dbl_kind, 0.718314769134348_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,336) = (/ 0.999960183653353_dbl_kind, 0.998959800169192_dbl_kind, 0.991767549550821_dbl_kind, 0.964157583566863_dbl_kind, 0.717916700423947_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,337) = (/ 0.999960075916728_dbl_kind, 0.998957138640512_dbl_kind, 0.991745887164442_dbl_kind, 0.964068685367738_dbl_kind, 0.717519589847129_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,338) = (/ 0.999959968183459_dbl_kind, 0.998954477625707_dbl_kind, 0.991724227655931_dbl_kind, 0.963979805718276_dbl_kind, 0.717123434206951_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,339) = (/ 0.999959860453553_dbl_kind, 0.998951817122349_dbl_kind, 0.991702571007785_dbl_kind, 0.963890944633479_dbl_kind, 0.716728230324885_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,340) = (/ 0.999959752727013_dbl_kind, 0.99894915712803_dbl_kind, 0.99168091720273_dbl_kind, 0.96380210212838_dbl_kind, 0.716333975040659_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,341) = (/ 0.999959645003845_dbl_kind, 0.998946497640366_dbl_kind, 0.991659266223722_dbl_kind, 0.963713278218035_dbl_kind, 0.715940665212101_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,342) = (/ 0.999959537284051_dbl_kind, 0.998943838657002_dbl_kind, 0.991637618053944_dbl_kind, 0.963624472917518_dbl_kind, 0.715548297714972_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,343) = (/ 0.999959429567642_dbl_kind, 0.998941180175604_dbl_kind, 0.991615972676807_dbl_kind, 0.963535686241909_dbl_kind, 0.715156869442815_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,344) = (/ 0.999959321854618_dbl_kind, 0.998938522193861_dbl_kind, 0.991594330075947_dbl_kind, 0.963446918206291_dbl_kind, 0.714766377306795_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,345) = (/ 0.99995921414499_dbl_kind, 0.998935864709489_dbl_kind, 0.991572690235228_dbl_kind, 0.96335816882574_dbl_kind, 0.714376818235549_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,346) = (/ 0.999959106438762_dbl_kind, 0.998933207720225_dbl_kind, 0.991551053138739_dbl_kind, 0.963269438115317_dbl_kind, 0.713988189175036_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,347) = (/ 0.999958998735943_dbl_kind, 0.998930551223832_dbl_kind, 0.991529418770787_dbl_kind, 0.963180726090067_dbl_kind, 0.713600487088373_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,348) = (/ 0.999958891036538_dbl_kind, 0.998927895218092_dbl_kind, 0.991507787115909_dbl_kind, 0.963092032765007_dbl_kind, 0.713213708955704_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,349) = (/ 0.999958783340558_dbl_kind, 0.998925239700814_dbl_kind, 0.991486158158857_dbl_kind, 0.963003358155116_dbl_kind, 0.712827851774046_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,350) = (/ 0.999958675648008_dbl_kind, 0.998922584669828_dbl_kind, 0.991464531884606_dbl_kind, 0.962914702275344_dbl_kind, 0.712442912557137_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,351) = (/ 0.999958567958898_dbl_kind, 0.998919930122985_dbl_kind, 0.991442908278348_dbl_kind, 0.962826065140586_dbl_kind, 0.7120588883353_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,352) = (/ 0.999958460273238_dbl_kind, 0.998917276058161_dbl_kind, 0.991421287325496_dbl_kind, 0.962737446765693_dbl_kind, 0.711675776155301_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,353) = (/ 0.999958352591035_dbl_kind, 0.998914622473254_dbl_kind, 0.991399669011674_dbl_kind, 0.962648847165456_dbl_kind, 0.711293573080206_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,354) = (/ 0.999958244912299_dbl_kind, 0.99891196936618_dbl_kind, 0.991378053322726_dbl_kind, 0.962560266354603_dbl_kind, 0.710912276189237_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,355) = (/ 0.99995813723704_dbl_kind, 0.998909316734883_dbl_kind, 0.991356440244704_dbl_kind, 0.9624717043478_dbl_kind, 0.710531882577642_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,356) = (/ 0.999958029565269_dbl_kind, 0.998906664577323_dbl_kind, 0.991334829763875_dbl_kind, 0.962383161159634_dbl_kind, 0.710152389356557_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,357) = (/ 0.999957921896995_dbl_kind, 0.998904012891485_dbl_kind, 0.991313221866715_dbl_kind, 0.962294636804622_dbl_kind, 0.70977379365287_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,358) = (/ 0.999957814232228_dbl_kind, 0.998901361675374_dbl_kind, 0.99129161653991_dbl_kind, 0.962206131297193_dbl_kind, 0.709396092609089_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,359) = (/ 0.999957706570981_dbl_kind, 0.998898710927016_dbl_kind, 0.991270013770352_dbl_kind, 0.962117644651695_dbl_kind, 0.709019283383208_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,360) = (/ 0.999957598913265_dbl_kind, 0.998896060644459_dbl_kind, 0.991248413545137_dbl_kind, 0.962029176882381_dbl_kind, 0.708643363148576_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,361) = (/ 0.99995749125909_dbl_kind, 0.998893410825771_dbl_kind, 0.991226815851568_dbl_kind, 0.961940728003411_dbl_kind, 0.708268329093777_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,362) = (/ 0.999957383608467_dbl_kind, 0.998890761469041_dbl_kind, 0.991205220677148_dbl_kind, 0.961852298028847_dbl_kind, 0.707894178422491_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,363) = (/ 0.99995727596141_dbl_kind, 0.998888112572378_dbl_kind, 0.991183628009581_dbl_kind, 0.961763886972647_dbl_kind, 0.707520908353374_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,364) = (/ 0.999957168317931_dbl_kind, 0.998885464133912_dbl_kind, 0.991162037836771_dbl_kind, 0.961675494848664_dbl_kind, 0.707148516119934_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,365) = (/ 0.999957060678042_dbl_kind, 0.998882816151794_dbl_kind, 0.991140450146816_dbl_kind, 0.961587121670639_dbl_kind, 0.706776998970409_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,366) = (/ 0.999956953041754_dbl_kind, 0.998880168624194_dbl_kind, 0.991118864928011_dbl_kind, 0.961498767452203_dbl_kind, 0.70640635416763_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,367) = (/ 0.999956845409082_dbl_kind, 0.998877521549302_dbl_kind, 0.991097282168846_dbl_kind, 0.961410432206867_dbl_kind, 0.706036578988929_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,368) = (/ 0.999956737780037_dbl_kind, 0.998874874925329_dbl_kind, 0.991075701858001_dbl_kind, 0.961322115948024_dbl_kind, 0.705667670725991_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,369) = (/ 0.999956630154634_dbl_kind, 0.998872228750506_dbl_kind, 0.991054123984344_dbl_kind, 0.961233818688945_dbl_kind, 0.705299626684752_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,370) = (/ 0.999956522532885_dbl_kind, 0.998869583023082_dbl_kind, 0.991032548536935_dbl_kind, 0.961145540442774_dbl_kind, 0.704932444185278_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,371) = (/ 0.999956414914804_dbl_kind, 0.998866937741327_dbl_kind, 0.991010975505017_dbl_kind, 0.96105728122253_dbl_kind, 0.704566120561651_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,372) = (/ 0.999956307300405_dbl_kind, 0.99886429290353_dbl_kind, 0.990989404878019_dbl_kind, 0.9609690410411_dbl_kind, 0.704200653161853_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,373) = (/ 0.999956199689701_dbl_kind, 0.998861648508_dbl_kind, 0.990967836645552_dbl_kind, 0.960880819911236_dbl_kind, 0.703836039347651_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,374) = (/ 0.999956092082707_dbl_kind, 0.998859004553065_dbl_kind, 0.990946270797406_dbl_kind, 0.960792617845557_dbl_kind, 0.703472276494487_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,375) = (/ 0.999955984479435_dbl_kind, 0.99885636103707_dbl_kind, 0.990924707323553_dbl_kind, 0.960704434856545_dbl_kind, 0.703109361991374_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,376) = (/ 0.999955876879901_dbl_kind, 0.998853717958384_dbl_kind, 0.990903146214137_dbl_kind, 0.960616270956543_dbl_kind, 0.702747293240771_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,377) = (/ 0.999955769284118_dbl_kind, 0.998851075315391_dbl_kind, 0.990881587459482_dbl_kind, 0.960528126157753_dbl_kind, 0.702386067658494_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,378) = (/ 0.999955661692101_dbl_kind, 0.998848433106493_dbl_kind, 0.990860031050081_dbl_kind, 0.960440000472231_dbl_kind, 0.702025682673589_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,379) = (/ 0.999955554103865_dbl_kind, 0.998845791330115_dbl_kind, 0.990838476976601_dbl_kind, 0.960351893911894_dbl_kind, 0.701666135728239_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,380) = (/ 0.999955446519423_dbl_kind, 0.998843149984698_dbl_kind, 0.990816925229875_dbl_kind, 0.960263806488507_dbl_kind, 0.70130742427766_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,381) = (/ 0.999955338938791_dbl_kind, 0.998840509068703_dbl_kind, 0.990795375800904_dbl_kind, 0.960175738213694_dbl_kind, 0.700949545789984_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,382) = (/ 0.999955231361983_dbl_kind, 0.998837868580605_dbl_kind, 0.990773828680856_dbl_kind, 0.960087689098924_dbl_kind, 0.700592497746175_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,383) = (/ 0.999955123789014_dbl_kind, 0.998835228518906_dbl_kind, 0.990752283861062_dbl_kind, 0.959999659155521_dbl_kind, 0.700236277639905_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,384) = (/ 0.999955016219898_dbl_kind, 0.998832588882118_dbl_kind, 0.990730741333011_dbl_kind, 0.959911648394655_dbl_kind, 0.699880882977482_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,385) = (/ 0.99995490865465_dbl_kind, 0.998829949668776_dbl_kind, 0.990709201088356_dbl_kind, 0.959823656827347_dbl_kind, 0.699526311277718_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,386) = (/ 0.999954801093284_dbl_kind, 0.99882731087743_dbl_kind, 0.990687663118904_dbl_kind, 0.959735684464461_dbl_kind, 0.699172560071856_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,387) = (/ 0.999954693535816_dbl_kind, 0.998824672506653_dbl_kind, 0.990666127416619_dbl_kind, 0.959647731316712_dbl_kind, 0.698819626903462_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,388) = (/ 0.99995458598226_dbl_kind, 0.998822034555031_dbl_kind, 0.990644593973619_dbl_kind, 0.959559797394656_dbl_kind, 0.698467509328324_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,389) = (/ 0.999954478432632_dbl_kind, 0.99881939702117_dbl_kind, 0.990623062782172_dbl_kind, 0.959471882708699_dbl_kind, 0.698116204914369_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,390) = (/ 0.999954370886945_dbl_kind, 0.998816759903694_dbl_kind, 0.990601533834698_dbl_kind, 0.959383987269087_dbl_kind, 0.697765711241559_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,391) = (/ 0.999954263345215_dbl_kind, 0.998814123201244_dbl_kind, 0.990580007123762_dbl_kind, 0.959296111085913_dbl_kind, 0.6974160259018_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,392) = (/ 0.999954155807456_dbl_kind, 0.998811486912481_dbl_kind, 0.990558482642079_dbl_kind, 0.959208254169113_dbl_kind, 0.69706714649885_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,393) = (/ 0.999954048273683_dbl_kind, 0.99880885103608_dbl_kind, 0.990536960382504_dbl_kind, 0.959120416528467_dbl_kind, 0.696719070648225_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,394) = (/ 0.99995394074391_dbl_kind, 0.998806215570736_dbl_kind, 0.990515440338036_dbl_kind, 0.959032598173596_dbl_kind, 0.696371795977111_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,395) = (/ 0.999953833218153_dbl_kind, 0.998803580515161_dbl_kind, 0.990493922501814_dbl_kind, 0.958944799113967_dbl_kind, 0.696025320124275_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,396) = (/ 0.999953725696424_dbl_kind, 0.998800945868083_dbl_kind, 0.990472406867115_dbl_kind, 0.958857019358889_dbl_kind, 0.695679640739975_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,397) = (/ 0.99995361817874_dbl_kind, 0.998798311628251_dbl_kind, 0.990450893427352_dbl_kind, 0.958769258917514_dbl_kind, 0.695334755485868_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,398) = (/ 0.999953510665114_dbl_kind, 0.998795677794427_dbl_kind, 0.990429382176076_dbl_kind, 0.958681517798837_dbl_kind, 0.694990662034932_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,399) = (/ 0.99995340315556_dbl_kind, 0.998793044365392_dbl_kind, 0.990407873106965_dbl_kind, 0.958593796011698_dbl_kind, 0.694647358071366_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,400) = (/ 0.999953295650093_dbl_kind, 0.998790411339944_dbl_kind, 0.990386366213831_dbl_kind, 0.958506093564779_dbl_kind, 0.694304841290526_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,401) = (/ 0.999953188148727_dbl_kind, 0.998787778716899_dbl_kind, 0.990364861490617_dbl_kind, 0.958418410466606_dbl_kind, 0.693963109398813_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,402) = (/ 0.999953080651475_dbl_kind, 0.998785146495088_dbl_kind, 0.990343358931389_dbl_kind, 0.958330746725551_dbl_kind, 0.693622160113616_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,403) = (/ 0.999952973158352_dbl_kind, 0.99878251467336_dbl_kind, 0.990321858530341_dbl_kind, 0.95824310234983_dbl_kind, 0.693281991163213_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,404) = (/ 0.999952865669371_dbl_kind, 0.99877988325058_dbl_kind, 0.99030036028179_dbl_kind, 0.958155477347502_dbl_kind, 0.692942600286692_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,405) = (/ 0.999952758184546_dbl_kind, 0.998777252225631_dbl_kind, 0.990278864180174_dbl_kind, 0.958067871726477_dbl_kind, 0.692603985233871_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,406) = (/ 0.999952650703889_dbl_kind, 0.998774621597412_dbl_kind, 0.990257370220052_dbl_kind, 0.957980285494506_dbl_kind, 0.692266143765224_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,407) = (/ 0.999952543227416_dbl_kind, 0.998771991364839_dbl_kind, 0.990235878396101_dbl_kind, 0.957892718659189_dbl_kind, 0.691929073651788_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,408) = (/ 0.999952435755139_dbl_kind, 0.998769361526841_dbl_kind, 0.990214388703113_dbl_kind, 0.957805171227974_dbl_kind, 0.691592772675096_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,409) = (/ 0.99995232828707_dbl_kind, 0.998766732082369_dbl_kind, 0.990192901135996_dbl_kind, 0.957717643208158_dbl_kind, 0.691257238627094_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,410) = (/ 0.999952220823223_dbl_kind, 0.998764103030387_dbl_kind, 0.990171415689771_dbl_kind, 0.957630134606886_dbl_kind, 0.690922469310066_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,411) = (/ 0.99995211336361_dbl_kind, 0.998761474369874_dbl_kind, 0.99014993235957_dbl_kind, 0.957542645431154_dbl_kind, 0.690588462536552_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,412) = (/ 0.999952005908245_dbl_kind, 0.998758846099829_dbl_kind, 0.990128451140632_dbl_kind, 0.957455175687807_dbl_kind, 0.690255216129282_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,413) = (/ 0.999951898457138_dbl_kind, 0.998756218219264_dbl_kind, 0.990106972028307_dbl_kind, 0.957367725383545_dbl_kind, 0.689922727921095_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,414) = (/ 0.999951791010304_dbl_kind, 0.998753590727209_dbl_kind, 0.990085495018051_dbl_kind, 0.957280294524918_dbl_kind, 0.68959099575486_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,415) = (/ 0.999951683567752_dbl_kind, 0.998750963622708_dbl_kind, 0.990064020105422_dbl_kind, 0.95719288311833_dbl_kind, 0.689260017483413_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,416) = (/ 0.999951576129496_dbl_kind, 0.998748336904822_dbl_kind, 0.990042547286081_dbl_kind, 0.957105491170043_dbl_kind, 0.688929790969478_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,417) = (/ 0.999951468695548_dbl_kind, 0.998745710572626_dbl_kind, 0.990021076555793_dbl_kind, 0.957018118686168_dbl_kind, 0.688600314085596_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,418) = (/ 0.999951361265918_dbl_kind, 0.998743084625214_dbl_kind, 0.989999607910418_dbl_kind, 0.95693076567268_dbl_kind, 0.688271584714053_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,419) = (/ 0.999951253840618_dbl_kind, 0.998740459061692_dbl_kind, 0.989978141345918_dbl_kind, 0.956843432135409_dbl_kind, 0.687943600746813_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,420) = (/ 0.999951146419659_dbl_kind, 0.998737833881185_dbl_kind, 0.989956676858348_dbl_kind, 0.956756118080043_dbl_kind, 0.687616360085439_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,421) = (/ 0.999951039003052_dbl_kind, 0.99873520908283_dbl_kind, 0.98993521444386_dbl_kind, 0.956668823512132_dbl_kind, 0.687289860641037_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,422) = (/ 0.999950931590806_dbl_kind, 0.998732584665781_dbl_kind, 0.989913754098696_dbl_kind, 0.956581548437088_dbl_kind, 0.686964100334175_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,423) = (/ 0.999950824182934_dbl_kind, 0.998729960629208_dbl_kind, 0.989892295819192_dbl_kind, 0.956494292860183_dbl_kind, 0.686639077094818_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,424) = (/ 0.999950716779445_dbl_kind, 0.998727336972293_dbl_kind, 0.989870839601773_dbl_kind, 0.956407056786556_dbl_kind, 0.68631478886227_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,425) = (/ 0.999950609380349_dbl_kind, 0.998724713694239_dbl_kind, 0.989849385442952_dbl_kind, 0.95631984022121_dbl_kind, 0.685991233585093_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,426) = (/ 0.999950501985655_dbl_kind, 0.998722090794256_dbl_kind, 0.989827933339329_dbl_kind, 0.956232643169014_dbl_kind, 0.685668409221052_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,427) = (/ 0.999950394595374_dbl_kind, 0.998719468271576_dbl_kind, 0.989806483287592_dbl_kind, 0.956145465634706_dbl_kind, 0.685346313737042_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,428) = (/ 0.999950287209515_dbl_kind, 0.998716846125443_dbl_kind, 0.989785035284508_dbl_kind, 0.956058307622892_dbl_kind, 0.685024945109027_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,429) = (/ 0.999950179828087_dbl_kind, 0.998714224355113_dbl_kind, 0.98976358932693_dbl_kind, 0.95597116913805_dbl_kind, 0.684704301321977_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,430) = (/ 0.999950072451098_dbl_kind, 0.998711602959862_dbl_kind, 0.989742145411789_dbl_kind, 0.955884050184527_dbl_kind, 0.684384380369798_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,431) = (/ 0.999949965078558_dbl_kind, 0.998708981938976_dbl_kind, 0.989720703536097_dbl_kind, 0.955796950766549_dbl_kind, 0.684065180255278_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,432) = (/ 0.999949857710473_dbl_kind, 0.998706361291757_dbl_kind, 0.989699263696947_dbl_kind, 0.95570987088821_dbl_kind, 0.683746698990015_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,433) = (/ 0.999949750346854_dbl_kind, 0.998703741017523_dbl_kind, 0.989677825891503_dbl_kind, 0.955622810553486_dbl_kind, 0.683428934594363_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,434) = (/ 0.999949642987707_dbl_kind, 0.998701121115604_dbl_kind, 0.989656390117008_dbl_kind, 0.955535769766225_dbl_kind, 0.683111885097366_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,435) = (/ 0.999949535633041_dbl_kind, 0.998698501585343_dbl_kind, 0.989634956370778_dbl_kind, 0.955448748530158_dbl_kind, 0.682795548536699_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,436) = (/ 0.999949428282862_dbl_kind, 0.998695882426102_dbl_kind, 0.989613524650201_dbl_kind, 0.955361746848895_dbl_kind, 0.682479922958606_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,437) = (/ 0.999949320937178_dbl_kind, 0.998693263637251_dbl_kind, 0.989592094952736_dbl_kind, 0.955274764725927_dbl_kind, 0.682165006417844_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,438) = (/ 0.999949213595996_dbl_kind, 0.998690645218178_dbl_kind, 0.989570667275913_dbl_kind, 0.955187802164632_dbl_kind, 0.681850796977617_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,439) = (/ 0.999949106259322_dbl_kind, 0.998688027168283_dbl_kind, 0.98954924161733_dbl_kind, 0.955100859168267_dbl_kind, 0.681537292709525_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,440) = (/ 0.999948998927162_dbl_kind, 0.998685409486981_dbl_kind, 0.989527817974651_dbl_kind, 0.95501393573998_dbl_kind, 0.681224491693502_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,441) = (/ 0.999948891599524_dbl_kind, 0.998682792173698_dbl_kind, 0.989506396345607_dbl_kind, 0.954927031882804_dbl_kind, 0.680912392017759_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,442) = (/ 0.999948784276412_dbl_kind, 0.998680175227876_dbl_kind, 0.989484976727993_dbl_kind, 0.954840147599664_dbl_kind, 0.680600991778727_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,443) = (/ 0.999948676957833_dbl_kind, 0.998677558648969_dbl_kind, 0.989463559119668_dbl_kind, 0.954753282893373_dbl_kind, 0.680290289081003_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,444) = (/ 0.999948569643791_dbl_kind, 0.998674942436445_dbl_kind, 0.989442143518554_dbl_kind, 0.954666437766636_dbl_kind, 0.679980282037283_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,445) = (/ 0.999948462334292_dbl_kind, 0.998672326589784_dbl_kind, 0.989420729922631_dbl_kind, 0.954579612222056_dbl_kind, 0.679670968768326_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,446) = (/ 0.999948355029342_dbl_kind, 0.998669711108481_dbl_kind, 0.989399318329942_dbl_kind, 0.954492806262123_dbl_kind, 0.67936234740288_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,447) = (/ 0.999948247728942_dbl_kind, 0.99866709599204_dbl_kind, 0.989377908738586_dbl_kind, 0.954406019889235_dbl_kind, 0.679054416077639_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,448) = (/ 0.9999481404331_dbl_kind, 0.998664481239984_dbl_kind, 0.989356501146721_dbl_kind, 0.954319253105679_dbl_kind, 0.678747172937182_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,449) = (/ 0.999948033141818_dbl_kind, 0.998661866851844_dbl_kind, 0.98933509555256_dbl_kind, 0.954232505913643_dbl_kind, 0.678440616133927_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,450) = (/ 0.999947925855101_dbl_kind, 0.998659252827163_dbl_kind, 0.989313691954371_dbl_kind, 0.95414577831522_dbl_kind, 0.678134743828069_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,451) = (/ 0.999947818572952_dbl_kind, 0.998656639165501_dbl_kind, 0.989292290350477_dbl_kind, 0.954059070312403_dbl_kind, 0.677829554187536_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,452) = (/ 0.999947711295374_dbl_kind, 0.998654025866426_dbl_kind, 0.989270890739253_dbl_kind, 0.95397238190709_dbl_kind, 0.67752504538793_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,453) = (/ 0.99994760402237_dbl_kind, 0.998651412929521_dbl_kind, 0.989249493119125_dbl_kind, 0.953885713101084_dbl_kind, 0.677221215612479_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,454) = (/ 0.999947496753943_dbl_kind, 0.99864880035438_dbl_kind, 0.98922809748857_dbl_kind, 0.953799063896095_dbl_kind, 0.676918063051985_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,455) = (/ 0.999947389490097_dbl_kind, 0.998646188140609_dbl_kind, 0.989206703846116_dbl_kind, 0.953712434293743_dbl_kind, 0.67661558590477_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,456) = (/ 0.999947282230831_dbl_kind, 0.998643576287827_dbl_kind, 0.989185312190337_dbl_kind, 0.953625824295555_dbl_kind, 0.676313782376632_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,457) = (/ 0.99994717497615_dbl_kind, 0.998640964795663_dbl_kind, 0.989163922519856_dbl_kind, 0.953539233902974_dbl_kind, 0.67601265068079_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,458) = (/ 0.999947067726054_dbl_kind, 0.998638353663759_dbl_kind, 0.98914253483334_dbl_kind, 0.953452663117354_dbl_kind, 0.675712189037837_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,459) = (/ 0.999946960480545_dbl_kind, 0.99863574289177_dbl_kind, 0.989121149129507_dbl_kind, 0.95336611193996_dbl_kind, 0.675412395675687_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,460) = (/ 0.999946853239624_dbl_kind, 0.99863313247936_dbl_kind, 0.989099765407112_dbl_kind, 0.953279580371981_dbl_kind, 0.675113268829531_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,461) = (/ 0.999946746003293_dbl_kind, 0.998630522426205_dbl_kind, 0.989078383664957_dbl_kind, 0.953193068414514_dbl_kind, 0.67481480674179_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,462) = (/ 0.999946638771552_dbl_kind, 0.998627912731993_dbl_kind, 0.989057003901887_dbl_kind, 0.953106576068581_dbl_kind, 0.674517007662057_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,463) = (/ 0.9999465315444_dbl_kind, 0.998625303396423_dbl_kind, 0.989035626116787_dbl_kind, 0.953020103335124_dbl_kind, 0.674219869847063_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,464) = (/ 0.99994642432184_dbl_kind, 0.998622694419205_dbl_kind, 0.989014250308583_dbl_kind, 0.952933650215003_dbl_kind, 0.67392339156062_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,465) = (/ 0.99994631710387_dbl_kind, 0.998620085800061_dbl_kind, 0.98899287647624_dbl_kind, 0.952847216709003_dbl_kind, 0.673627571073579_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,466) = (/ 0.99994620989049_dbl_kind, 0.998617477538719_dbl_kind, 0.988971504618764_dbl_kind, 0.952760802817836_dbl_kind, 0.673332406663782_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,467) = (/ 0.9999461026817_dbl_kind, 0.998614869634927_dbl_kind, 0.988950134735193_dbl_kind, 0.952674408542136_dbl_kind, 0.673037896616018_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,468) = (/ 0.999945995477499_dbl_kind, 0.998612262088434_dbl_kind, 0.988928766824609_dbl_kind, 0.952588033882466_dbl_kind, 0.672744039221976_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,469) = (/ 0.999945888277886_dbl_kind, 0.998609654899005_dbl_kind, 0.988907400886124_dbl_kind, 0.952501678839317_dbl_kind, 0.672450832780201_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,470) = (/ 0.999945781082858_dbl_kind, 0.998607048066414_dbl_kind, 0.988886036918888_dbl_kind, 0.952415343413111_dbl_kind, 0.672158275596044_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,471) = (/ 0.999945673892417_dbl_kind, 0.998604441590445_dbl_kind, 0.988864674922085_dbl_kind, 0.952329027604202_dbl_kind, 0.671866365981628_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,472) = (/ 0.999945566706558_dbl_kind, 0.998601835470892_dbl_kind, 0.988843314894931_dbl_kind, 0.952242731412873_dbl_kind, 0.671575102255794_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,473) = (/ 0.999945459525281_dbl_kind, 0.99859922970756_dbl_kind, 0.988821956836676_dbl_kind, 0.952156454839346_dbl_kind, 0.671284482744067_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,474) = (/ 0.999945352348583_dbl_kind, 0.998596624300263_dbl_kind, 0.988800600746601_dbl_kind, 0.952070197883774_dbl_kind, 0.670994505778599_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,475) = (/ 0.99994524517646_dbl_kind, 0.998594019248825_dbl_kind, 0.988779246624018_dbl_kind, 0.951983960546248_dbl_kind, 0.670705169698142_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,476) = (/ 0.999945138008912_dbl_kind, 0.99859141455308_dbl_kind, 0.988757894468269_dbl_kind, 0.951897742826798_dbl_kind, 0.670416472847994_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,477) = (/ 0.999945030845935_dbl_kind, 0.998588810212872_dbl_kind, 0.988736544278726_dbl_kind, 0.951811544725392_dbl_kind, 0.670128413579963_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,478) = (/ 0.999944923687525_dbl_kind, 0.998586206228052_dbl_kind, 0.988715196054788_dbl_kind, 0.95172536624194_dbl_kind, 0.669840990252318_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,479) = (/ 0.999944816533679_dbl_kind, 0.998583602598484_dbl_kind, 0.988693849795883_dbl_kind, 0.951639207376292_dbl_kind, 0.669554201229763_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,480) = (/ 0.999944709384394_dbl_kind, 0.998580999324039_dbl_kind, 0.988672505501466_dbl_kind, 0.951553068128241_dbl_kind, 0.669268044883371_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,481) = (/ 0.999944602239666_dbl_kind, 0.998578396404596_dbl_kind, 0.988651163171019_dbl_kind, 0.951466948497526_dbl_kind, 0.668982519590568_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,482) = (/ 0.99994449509949_dbl_kind, 0.998575793840046_dbl_kind, 0.988629822804048_dbl_kind, 0.95138084848383_dbl_kind, 0.66869762373508_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,483) = (/ 0.999944387963862_dbl_kind, 0.998573191630288_dbl_kind, 0.988608484400084_dbl_kind, 0.951294768086782_dbl_kind, 0.668413355706891_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,484) = (/ 0.999944280832779_dbl_kind, 0.998570589775226_dbl_kind, 0.988587147958685_dbl_kind, 0.951208707305961_dbl_kind, 0.668129713902213_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,485) = (/ 0.999944173706234_dbl_kind, 0.998567988274777_dbl_kind, 0.988565813479428_dbl_kind, 0.951122666140892_dbl_kind, 0.667846696723433_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,486) = (/ 0.999944066584223_dbl_kind, 0.998565387128866_dbl_kind, 0.988544480961917_dbl_kind, 0.951036644591053_dbl_kind, 0.667564302579089_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,487) = (/ 0.99994395946674_dbl_kind, 0.998562786337425_dbl_kind, 0.988523150405777_dbl_kind, 0.950950642655872_dbl_kind, 0.667282529883816_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,488) = (/ 0.999943852353782_dbl_kind, 0.998560185900395_dbl_kind, 0.988501821810653_dbl_kind, 0.950864660334729_dbl_kind, 0.667001377058323_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,489) = (/ 0.999943745245341_dbl_kind, 0.998557585817724_dbl_kind, 0.988480495176213_dbl_kind, 0.95077869762696_dbl_kind, 0.66672084252934_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,490) = (/ 0.999943638141412_dbl_kind, 0.99855498608937_dbl_kind, 0.988459170502143_dbl_kind, 0.95069275453185_dbl_kind, 0.666440924729589_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,491) = (/ 0.999943531041989_dbl_kind, 0.998552386715297_dbl_kind, 0.988437847788152_dbl_kind, 0.950606831048647_dbl_kind, 0.666161622097745_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,492) = (/ 0.999943423947066_dbl_kind, 0.998549787695478_dbl_kind, 0.988416527033964_dbl_kind, 0.950520927176552_dbl_kind, 0.665882933078396_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,493) = (/ 0.999943316856637_dbl_kind, 0.998547189029894_dbl_kind, 0.988395208239325_dbl_kind, 0.950435042914722_dbl_kind, 0.665604856122011_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,494) = (/ 0.999943209770694_dbl_kind, 0.998544590718532_dbl_kind, 0.988373891403997_dbl_kind, 0.950349178262278_dbl_kind, 0.665327389684893_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,495) = (/ 0.999943102689231_dbl_kind, 0.998541992761388_dbl_kind, 0.988352576527759_dbl_kind, 0.950263333218296_dbl_kind, 0.665050532229159_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,496) = (/ 0.999942995612242_dbl_kind, 0.998539395158464_dbl_kind, 0.988331263610409_dbl_kind, 0.950177507781817_dbl_kind, 0.664774282222687_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,497) = (/ 0.999942888539718_dbl_kind, 0.998536797909772_dbl_kind, 0.988309952651759_dbl_kind, 0.950091701951841_dbl_kind, 0.664498638139091_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,498) = (/ 0.999942781471652_dbl_kind, 0.998534201015326_dbl_kind, 0.988288643651637_dbl_kind, 0.950005915727334_dbl_kind, 0.664223598457681_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,499) = (/ 0.999942674408038_dbl_kind, 0.998531604475151_dbl_kind, 0.988267336609887_dbl_kind, 0.949920149107222_dbl_kind, 0.663949161663427_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,500) = (/ 0.999942567348866_dbl_kind, 0.998529008289279_dbl_kind, 0.988246031526367_dbl_kind, 0.949834402090399_dbl_kind, 0.663675326246929_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,501) = (/ 0.999942460294131_dbl_kind, 0.998526412457747_dbl_kind, 0.988224728400949_dbl_kind, 0.949748674675725_dbl_kind, 0.663402090704375_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,502) = (/ 0.999942353243823_dbl_kind, 0.998523816980599_dbl_kind, 0.988203427233518_dbl_kind, 0.949662966862027_dbl_kind, 0.663129453537516_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,503) = (/ 0.999942246197933_dbl_kind, 0.998521221857886_dbl_kind, 0.988182128023974_dbl_kind, 0.949577278648096_dbl_kind, 0.662857413253619_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,504) = (/ 0.999942139156455_dbl_kind, 0.998518627089666_dbl_kind, 0.988160830772228_dbl_kind, 0.949491610032698_dbl_kind, 0.662585968365445_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,505) = (/ 0.999942032119378_dbl_kind, 0.998516032676001_dbl_kind, 0.988139535478203_dbl_kind, 0.949405961014562_dbl_kind, 0.662315117391211_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,506) = (/ 0.999941925086694_dbl_kind, 0.998513438616962_dbl_kind, 0.988118242141834_dbl_kind, 0.949320331592391_dbl_kind, 0.662044858854551_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,507) = (/ 0.999941818058395_dbl_kind, 0.998510844912623_dbl_kind, 0.988096950763067_dbl_kind, 0.949234721764861_dbl_kind, 0.661775191284497_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,508) = (/ 0.999941711034471_dbl_kind, 0.998508251563068_dbl_kind, 0.988075661341861_dbl_kind, 0.949149131530616_dbl_kind, 0.661506113215427_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,509) = (/ 0.999941604014913_dbl_kind, 0.998505658568383_dbl_kind, 0.98805437387818_dbl_kind, 0.949063560888275_dbl_kind, 0.661237623187047_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,510) = (/ 0.999941496999712_dbl_kind, 0.998503065928662_dbl_kind, 0.988033088372003_dbl_kind, 0.948978009836433_dbl_kind, 0.660969719744358_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,511) = (/ 0.999941389988858_dbl_kind, 0.998500473644004_dbl_kind, 0.988011804823316_dbl_kind, 0.948892478373654_dbl_kind, 0.660702401437612_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,512) = (/ 0.999941282982341_dbl_kind, 0.998497881714513_dbl_kind, 0.987990523232114_dbl_kind, 0.948806966498485_dbl_kind, 0.660435666822298_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,513) = (/ 0.999941175980153_dbl_kind, 0.998495290140299_dbl_kind, 0.987969243598401_dbl_kind, 0.948721474209442_dbl_kind, 0.660169514459088_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,514) = (/ 0.999941068982281_dbl_kind, 0.998492698921478_dbl_kind, 0.987947965922189_dbl_kind, 0.948636001505021_dbl_kind, 0.659903942913831_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,515) = (/ 0.999940961988718_dbl_kind, 0.998490108058169_dbl_kind, 0.987926690203499_dbl_kind, 0.948550548383698_dbl_kind, 0.6596389507575_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,516) = (/ 0.999940854999452_dbl_kind, 0.998487517550498_dbl_kind, 0.987905416442357_dbl_kind, 0.948465114843923_dbl_kind, 0.659374536566173_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,517) = (/ 0.999940748014472_dbl_kind, 0.998484927398596_dbl_kind, 0.987884144638797_dbl_kind, 0.94837970088413_dbl_kind, 0.659110698921002_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,518) = (/ 0.999940641033769_dbl_kind, 0.998482337602596_dbl_kind, 0.987862874792862_dbl_kind, 0.948294306502728_dbl_kind, 0.658847436408173_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,519) = (/ 0.999940534057332_dbl_kind, 0.998479748162641_dbl_kind, 0.987841606904596_dbl_kind, 0.94820893169811_dbl_kind, 0.658584747618891_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,520) = (/ 0.999940427085149_dbl_kind, 0.998477159078874_dbl_kind, 0.987820340974054_dbl_kind, 0.948123576468649_dbl_kind, 0.658322631149335_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,521) = (/ 0.99994032011721_dbl_kind, 0.998474570351445_dbl_kind, 0.987799077001295_dbl_kind, 0.9480382408127_dbl_kind, 0.658061085600641_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,522) = (/ 0.999940213153505_dbl_kind, 0.998471981980506_dbl_kind, 0.98777781498638_dbl_kind, 0.9479529247286_dbl_kind, 0.657800109578862_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,523) = (/ 0.999940106194022_dbl_kind, 0.998469393966217_dbl_kind, 0.987756554929381_dbl_kind, 0.947867628214672_dbl_kind, 0.657539701694945_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,524) = (/ 0.999939999238749_dbl_kind, 0.998466806308739_dbl_kind, 0.98773529683037_dbl_kind, 0.947782351269218_dbl_kind, 0.657279860564699_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,525) = (/ 0.999939892287675_dbl_kind, 0.998464219008239_dbl_kind, 0.987714040689423_dbl_kind, 0.947697093890529_dbl_kind, 0.65702058480877_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,526) = (/ 0.999939785340789_dbl_kind, 0.998461632064886_dbl_kind, 0.987692786506623_dbl_kind, 0.947611856076879_dbl_kind, 0.65676187305261_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,527) = (/ 0.999939678398079_dbl_kind, 0.998459045478855_dbl_kind, 0.987671534282054_dbl_kind, 0.947526637826529_dbl_kind, 0.656503723926443_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,528) = (/ 0.999939571459534_dbl_kind, 0.998456459250326_dbl_kind, 0.987650284015804_dbl_kind, 0.947441439137725_dbl_kind, 0.656246136065247_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,529) = (/ 0.999939464525141_dbl_kind, 0.998453873379478_dbl_kind, 0.987629035707964_dbl_kind, 0.947356260008699_dbl_kind, 0.655989108108722_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,530) = (/ 0.999939357594889_dbl_kind, 0.998451287866497_dbl_kind, 0.987607789358629_dbl_kind, 0.947271100437671_dbl_kind, 0.655732638701258_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,531) = (/ 0.999939250668765_dbl_kind, 0.998448702711573_dbl_kind, 0.987586544967894_dbl_kind, 0.947185960422851_dbl_kind, 0.655476726491914_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,532) = (/ 0.999939143746759_dbl_kind, 0.998446117914897_dbl_kind, 0.987565302535858_dbl_kind, 0.947100839962435_dbl_kind, 0.655221370134388_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,533) = (/ 0.999939036828856_dbl_kind, 0.998443533476667_dbl_kind, 0.987544062062621_dbl_kind, 0.94701573905461_dbl_kind, 0.654966568286983_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,534) = (/ 0.999938929915045_dbl_kind, 0.998440949397079_dbl_kind, 0.987522823548285_dbl_kind, 0.946930657697549_dbl_kind, 0.654712319612593_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,535) = (/ 0.999938823005314_dbl_kind, 0.998438365676337_dbl_kind, 0.987501586992952_dbl_kind, 0.946845595889419_dbl_kind, 0.654458622778671_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,536) = (/ 0.99993871609965_dbl_kind, 0.998435782314646_dbl_kind, 0.987480352396728_dbl_kind, 0.946760553628375_dbl_kind, 0.654205476457197_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,537) = (/ 0.999938609198041_dbl_kind, 0.998433199312212_dbl_kind, 0.987459119759716_dbl_kind, 0.946675530912564_dbl_kind, 0.653952879324656_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,538) = (/ 0.999938502300474_dbl_kind, 0.998430616669248_dbl_kind, 0.987437889082024_dbl_kind, 0.946590527740122_dbl_kind, 0.653700830062018_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,539) = (/ 0.999938395406937_dbl_kind, 0.998428034385968_dbl_kind, 0.987416660363756_dbl_kind, 0.946505544109182_dbl_kind, 0.653449327354697_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,540) = (/ 0.999938288517415_dbl_kind, 0.998425452462586_dbl_kind, 0.987395433605019_dbl_kind, 0.946420580017861_dbl_kind, 0.653198369892542_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,541) = (/ 0.999938181631898_dbl_kind, 0.998422870899322_dbl_kind, 0.987374208805919_dbl_kind, 0.946335635464278_dbl_kind, 0.6529479563698_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,542) = (/ 0.999938074750371_dbl_kind, 0.998420289696397_dbl_kind, 0.987352985966561_dbl_kind, 0.946250710446539_dbl_kind, 0.652698085485095_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,543) = (/ 0.999937967872822_dbl_kind, 0.998417708854036_dbl_kind, 0.987331765087051_dbl_kind, 0.946165804962744_dbl_kind, 0.652448755941403_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,544) = (/ 0.999937860999237_dbl_kind, 0.998415128372463_dbl_kind, 0.987310546167491_dbl_kind, 0.946080919010989_dbl_kind, 0.652199966446025_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,545) = (/ 0.999937754129604_dbl_kind, 0.998412548251908_dbl_kind, 0.987289329207989_dbl_kind, 0.945996052589362_dbl_kind, 0.651951715710566_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,546) = (/ 0.999937647263909_dbl_kind, 0.998409968492601_dbl_kind, 0.987268114208642_dbl_kind, 0.945911205695948_dbl_kind, 0.651704002450906_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,547) = (/ 0.999937540402139_dbl_kind, 0.998407389094772_dbl_kind, 0.987246901169554_dbl_kind, 0.945826378328825_dbl_kind, 0.651456825387178_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,548) = (/ 0.999937433544281_dbl_kind, 0.998404810058658_dbl_kind, 0.987225690090824_dbl_kind, 0.945741570486066_dbl_kind, 0.651210183243743_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,549) = (/ 0.999937326690322_dbl_kind, 0.998402231384494_dbl_kind, 0.987204480972548_dbl_kind, 0.945656782165742_dbl_kind, 0.650964074749167_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,550) = (/ 0.999937219840246_dbl_kind, 0.998399653072518_dbl_kind, 0.987183273814822_dbl_kind, 0.945572013365917_dbl_kind, 0.650718498636197_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,551) = (/ 0.999937112994042_dbl_kind, 0.99839707512297_dbl_kind, 0.98716206861774_dbl_kind, 0.945487264084654_dbl_kind, 0.650473453641734_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,552) = (/ 0.999937006151697_dbl_kind, 0.998394497536091_dbl_kind, 0.987140865381394_dbl_kind, 0.94540253432001_dbl_kind, 0.650228938506818_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,553) = (/ 0.999936899313195_dbl_kind, 0.998391920312123_dbl_kind, 0.98711966410587_dbl_kind, 0.945317824070042_dbl_kind, 0.649984951976594_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,554) = (/ 0.999936792478525_dbl_kind, 0.998389343451311_dbl_kind, 0.987098464791257_dbl_kind, 0.9452331333328_dbl_kind, 0.649741492800295_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,555) = (/ 0.99993668564767_dbl_kind, 0.9983867669539_dbl_kind, 0.987077267437637_dbl_kind, 0.945148462106335_dbl_kind, 0.649498559731219_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,556) = (/ 0.999936578820619_dbl_kind, 0.998384190820138_dbl_kind, 0.98705607204509_dbl_kind, 0.945063810388694_dbl_kind, 0.649256151526704_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,557) = (/ 0.999936471997358_dbl_kind, 0.998381615050272_dbl_kind, 0.987034878613693_dbl_kind, 0.944979178177923_dbl_kind, 0.649014266948107_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,558) = (/ 0.999936365177872_dbl_kind, 0.998379039644551_dbl_kind, 0.987013687143521_dbl_kind, 0.944894565472065_dbl_kind, 0.648772904760781_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,559) = (/ 0.999936258362149_dbl_kind, 0.998376464603225_dbl_kind, 0.986992497634645_dbl_kind, 0.944809972269163_dbl_kind, 0.648532063734053_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,560) = (/ 0.999936151550174_dbl_kind, 0.998373889926546_dbl_kind, 0.986971310087131_dbl_kind, 0.944725398567255_dbl_kind, 0.648291742641201_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,561) = (/ 0.999936044741933_dbl_kind, 0.998371315614765_dbl_kind, 0.986950124501043_dbl_kind, 0.944640844364383_dbl_kind, 0.648051940259434_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,562) = (/ 0.999935937937412_dbl_kind, 0.998368741668136_dbl_kind, 0.986928940876441_dbl_kind, 0.944556309658586_dbl_kind, 0.647812655369865_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,563) = (/ 0.999935831136598_dbl_kind, 0.998366168086912_dbl_kind, 0.98690775921338_dbl_kind, 0.944471794447901_dbl_kind, 0.6475738867575_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,564) = (/ 0.999935724339475_dbl_kind, 0.998363594871348_dbl_kind, 0.986886579511912_dbl_kind, 0.944387298730364_dbl_kind, 0.647335633211202_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,565) = (/ 0.999935617546032_dbl_kind, 0.998361022021698_dbl_kind, 0.986865401772084_dbl_kind, 0.944302822504014_dbl_kind, 0.647097893523683_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,566) = (/ 0.999935510756253_dbl_kind, 0.998358449538218_dbl_kind, 0.98684422599394_dbl_kind, 0.94421836576689_dbl_kind, 0.646860666491474_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,567) = (/ 0.999935403970124_dbl_kind, 0.998355877421163_dbl_kind, 0.986823052177518_dbl_kind, 0.944133928517024_dbl_kind, 0.646623950914911_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,568) = (/ 0.999935297187632_dbl_kind, 0.99835330567079_dbl_kind, 0.986801880322854_dbl_kind, 0.944049510752458_dbl_kind, 0.646387745598105_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,569) = (/ 0.999935190408762_dbl_kind, 0.998350734287355_dbl_kind, 0.986780710429975_dbl_kind, 0.943965112471227_dbl_kind, 0.646152049348931_dbl_kind /)
   iceSingleScatterAlbedoDiffuse(1:5 ,570) = (/ 0.999935083633501_dbl_kind, 0.998348163271116_dbl_kind, 0.986759542498908_dbl_kind, 0.94388073367137_dbl_kind, 0.645916860979002_dbl_kind /)
   iceSingle