!=======================================================================
!
! The albedo and absorbed/transmitted flux parameterizations for
! snow over ice, bare ice and ponded ice.
!
! Presently, two methods are included:
!   (1) CCSM3
!   (2) Delta-Eddington
! as two distinct routines.
! Either can be called from the ice driver.
!
! The Delta-Eddington method is described here:
!
! Briegleb, B. P., and B. Light (2007): A Delta-Eddington Multiple
!    Scattering Parameterization for Solar Radiation in the Sea Ice
!    Component of the Community Climate System Model, NCAR Technical
!    Note  NCAR/TN-472+STR  February 2007
!
! name: originally ice_albedo
!
! authors:  Bruce P. Briegleb, NCAR
!           Elizabeth C. Hunke and William H. Lipscomb, LANL
! 2005, WHL: Moved absorbed_solar from icepack_therm_vertical to this
!            module and changed name from ice_albedo
! 2006, WHL: Added Delta Eddington routines from Bruce Briegleb
! 2006, ECH: Changed data statements in Delta Eddington routines (no
!            longer hardwired)
!            Converted to free source form (F90)
! 2007, BPB: Completely updated Delta-Eddington code, so that:
!            (1) multiple snow layers enabled (i.e. nslyr > 1)
!            (2) included SSL for snow surface absorption
!            (3) added Sswabs for internal snow layer absorption
!            (4) variable sea ice layers allowed (i.e. not hardwired)
!            (5) updated all inherent optical properties
!            (6) included algae absorption for sea ice lowest layer
!            (7) very complete internal documentation included
! 2007, ECH: Improved efficiency
! 2008, BPB: Added aerosols to Delta Eddington code
! 2013, ECH: merged with NCAR version, cleaned up

      module icepack_shortwave

      use icepack_kinds
      use icepack_parameters, only: c0, c1, c1p5, c2, c3, c4, c10
      use icepack_parameters, only: p01, p1, p15, p25, p5, p75, puny
      use icepack_parameters, only: argcheck
      use icepack_parameters, only: albocn, Timelt, snowpatch, awtvdr, awtidr, awtvdf, awtidf
      use icepack_parameters, only: kappav, hs_min, rhofresh, rhos, rhoi
      use icepack_parameters, only: rsnw_fall, snwredist, rsnw_tmax
      use icepack_parameters, only: hi_ssl, hs_ssl, min_bgc, sk_l, snwlvlfac, snwgrain
      use icepack_parameters, only: z_tracers, skl_bgc, calc_tsfc, shortwave, kalg
      use icepack_parameters, only: R_ice, R_pnd, R_snw, dT_mlt, rsnw_mlt, hs0, hs1, hp1
      use icepack_parameters, only: pndaspect, albedo_type, albicev, albicei, albsnowv, albsnowi, ahmax
      use icepack_parameters, only: snw_ssp_table, modal_aero, semi_implicit_Tsfc
      use icepack_parameters, only: dEdd_algae
      use icepack_parameters, only: hpmin, hp0

      use icepack_tracers,    only: ncat, nilyr, nslyr, nblyr
      use icepack_tracers,    only: ntrcr, nbtrcr_sw
      use icepack_tracers,    only: tr_pond_lvl, tr_pond_topo, tr_pond_sealvl
      use icepack_tracers,    only: tr_lvl
      use icepack_tracers,    only: tr_bgc_N, tr_aero
      use icepack_tracers,    only: nt_bgc_N, nt_zaero
      use icepack_tracers,    only: tr_zaero, nlt_chl_sw, nlt_zaero_sw
      use icepack_tracers,    only: n_algae, n_aero, n_zaero
      use icepack_tracers,    only: nmodal1, nmodal2, max_aero
      use icepack_shortwave_data, only: nspint_3bd, nspint_5bd, rsnw_datatype
      use icepack_zbgc_shared,only: R_chl2N, F_abs_chl
      use icepack_zbgc_shared,only: remap_zbgc, igrid, swgrid
      use icepack_orbital,    only: compute_coszen
      use icepack_warnings,   only: warnstr, icepack_warnings_add
      use icepack_warnings,   only: icepack_warnings_setabort, icepack_warnings_aborted

      ! dEdd 3-band data
      use icepack_shortwave_data, only: &
         ! inherent optical properties (iop)
         !    k = extinction coefficient (/m)
         !    w = single scattering albedo
         !    g = asymmetry parameter
         ki_ssl_mn_3bd, wi_ssl_mn_3bd, gi_ssl_mn_3bd, & ! ice surface scattering layer (ssl) iops
         ki_dl_mn_3bd,  wi_dl_mn_3bd,  gi_dl_mn_3bd , & ! ice drained layer (dl) iops
         ki_int_mn_3bd, wi_int_mn_3bd, gi_int_mn_3bd, & ! ice interior layer (int) iops
         ki_p_ssl_mn,   wi_p_ssl_mn,   gi_p_ssl_mn  , & ! ponded ice surface scattering layer (ssl) iops
         ki_p_int_mn,   wi_p_int_mn,   gi_p_int_mn  , & ! ponded ice interior layer (int) iops
         kw,            ww,            gw               ! iops for pond water and underlying ocean
      use icepack_shortwave_data, only: &
         gaer_bc_3bd, kaer_bc_3bd, waer_bc_3bd, bcenh_3bd, &
         gaer_3bd, kaer_3bd, waer_3bd
      use icepack_shortwave_data, only: &
         nmbrad_snw, & ! number of snow grain radii in tables
         rsnw_tab,   & ! snow grain radii (micro-meters) for table
         Qs_tab,     & ! snow extinction efficiency (unitless)
         ws_tab,     & ! snow single scattering albedo (unitless)
         gs_tab        ! snow asymmetry parameter (unitless)

      ! dEdd 5-band data
      use icepack_shortwave_data, only: &
         ki_ssl_mn_5bd, wi_ssl_mn_5bd, gi_ssl_mn_5bd, & ! ice surface scattering layer (ssl) iops
         ki_dl_mn_5bd,  wi_dl_mn_5bd,  gi_dl_mn_5bd , & ! ice drained layer (dl) iops
         ki_int_mn_5bd, wi_int_mn_5bd, gi_int_mn_5bd    ! ice interior layer (int) iops
      use icepack_shortwave_data, only: &
         gaer_bc_5bd, kaer_bc_5bd, waer_bc_5bd, bcenh_5bd, &
         gaer_5bd, kaer_5bd, waer_5bd
      use icepack_shortwave_data, only: &
         nmbrad_snicar  , & ! number of snow grain radii in SNICAR SSP tables
         rsnw_snicar_min, & ! minimum snow radius
         rsnw_snicar_max, & ! maximum snow radius
         ssp_snwextdr, ssp_snwalbdr, ssp_sasymmdr, &
         ssp_snwextdf, ssp_snwalbdf, ssp_sasymmdf, &
         rsnw_snicar_tab

      implicit none

      private
      public :: icepack_prep_radiation, &
                icepack_init_radiation, &
                icepack_step_radiation

      real (kind=dbl_kind), parameter :: &
         exp_argmax = c10    ! maximum argument of exponential

      ! dEdd tuning parameters, set in namelist
      !   R_ice     ! sea ice tuning parameter; +1 > 1sig increase in albedo
      !   R_pnd     ! ponded ice tuning parameter; +1 > 1sig increase in albedo
      !   R_snw     ! snow tuning parameter; +1 > ~.01 change in broadband albedo
      !   dT_mlt    ! change in temp for non-melt to melt snow grain radius change (C)
      !   rsnw_mlt  ! maximum melting snow grain radius (10^-6 m)
      !   pndaspect ! ratio of pond depth to pond fraction
      !   hs0       ! snow depth for transition to bare sea ice (m)
      !   hs1       ! tapering parameter for snow on pond ice
      !   hp1       ! critical parameter for pond ice thickness
      !   kalg      ! algae absorption coefficient

!=======================================================================

      contains

!=======================================================================
!autodocument_start icepack_init_radiation
! Initialize data needed for shortwave radiation calculations
! This should be called after values are set via icepack_init_parameters

      subroutine icepack_init_radiation()

!autodocument_end
      use icepack_shortwave_data, only: icepack_shortwave_init_dEdd3band
      use icepack_shortwave_data, only: icepack_shortwave_init_dEdd5band
      use icepack_shortwave_data, only: icepack_shortwave_init_snicar
      use icepack_shortwave_data, only: icepack_shortwave_init_snicartest

      ! local variables

      integer (kind=int_kind) :: n

      character (len=*),parameter :: subname='(icepack_init_radiation)'

      !-----------------------------------------------------------------
      ! Set dEdd parameter tables
      !-----------------------------------------------------------------

      if (shortwave(1:4) == 'dEdd') then
         call icepack_shortwave_init_dEdd3band()
         if (icepack_warnings_aborted(subname)) return
      endif

      if (trim(shortwave) == 'dEdd_snicar_ad') then
         call icepack_shortwave_init_dEdd5band()
         if (icepack_warnings_aborted(subname)) return

         if (trim(snw_ssp_table) == 'test') then ! 5x5 test table
            call icepack_shortwave_init_snicartest()
            if (icepack_warnings_aborted(subname)) return
         elseif (trim(snw_ssp_table) == 'snicar') then  ! 5 x 1471 table
            call icepack_shortwave_init_snicar()
            if (icepack_warnings_aborted(subname)) return
         else
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            call icepack_warnings_add(subname//'ERROR: snw_ssp_table = '//trim(snw_ssp_table)//' not supported')
            return
         endif

         !------------------------------
         ! Check SNICAR SSP data
         !------------------------------

         write(warnstr,'(2a,i8)') subname, ' nmbrad_snicar  = ',nmbrad_snicar
         call icepack_warnings_add(warnstr)
         write(warnstr,'(2a,i8)') subname, ' nspint         = ',nspint_5bd
         call icepack_warnings_add(warnstr)
         write(warnstr,'(2a,i8)') subname, ' nmodal1        = ',nmodal1
         call icepack_warnings_add(warnstr)
         write(warnstr,'(2a,i8)') subname, ' nmodal2        = ',nmodal2
         call icepack_warnings_add(warnstr)
         write(warnstr,'(2a,i8)') subname, ' max_aero       = ',max_aero
         call icepack_warnings_add(warnstr)
         write(warnstr,'(2a,i5,a,i5,a,g22.15)') subname, ' ssp_snwextdr(',1,         ',',1,            ') = ',ssp_snwextdr(1,1)
         call icepack_warnings_add(warnstr)
         write(warnstr,'(2a,i5,a,i5,a,g22.15)') subname, ' ssp_snwextdr(',nspint_5bd,',',1,            ') = ',ssp_snwextdr(nspint_5bd,1)
         call icepack_warnings_add(warnstr)
         write(warnstr,'(2a,i5,a,i5,a,g22.15)') subname, ' ssp_snwextdr(',1,         ',',nmbrad_snicar,') = ',ssp_snwextdr(1,nmbrad_snicar)
         call icepack_warnings_add(warnstr)
         write(warnstr,'(2a,i5,a,i5,a,g22.15)') subname, ' ssp_snwextdr(',nspint_5bd,',',nmbrad_snicar,') = ',ssp_snwextdr(nspint_5bd,nmbrad_snicar)
         call icepack_warnings_add(warnstr)

      endif

      end subroutine icepack_init_radiation

!=======================================================================
!
! Driver for basic solar radiation from CCSM3.  Albedos and absorbed solar.

      subroutine shortwave_ccsm3 (aicen,    vicen,    &
                                  vsnon,    Tsfcn,    &
                                  swvdr,    swvdf,    &
                                  swidr,    swidf,    &
                                  albedo_type,        &
                                  albicev,  albicei,  &
                                  albsnowv, albsnowi, &
                                  ahmax,              &
                                  alvdrn,   alidrn,   &
                                  alvdfn,   alidfn,   &
                                  fswsfc,   fswint,   &
                                  swuvrdr,  swuvrdf,  &
                                  swpardr,  swpardf,  &
                                  fswthrun,           &
                                  fswthrun_vdr,       &
                                  fswthrun_vdf,       &
                                  fswthrun_idr,       &
                                  fswthrun_idf,       &
                                  fswthrun_uvrdr,     &
                                  fswthrun_uvrdf,     &
                                  fswthrun_pardr,     &
                                  fswthrun_pardf,     &
                                  fswpenl,            &
                                  Iswabs,   SSwabs,   &
                                  albin,    albsn,    &
                                  coszen)

      real (kind=dbl_kind), dimension (:), intent(in) :: &
         aicen    , & ! concentration of ice per category
         vicen    , & ! volume of ice per category
         vsnon    , & ! volume of ice per category
         Tsfcn        ! surface temperature

      real (kind=dbl_kind), intent(in) :: &
         swvdr    , & ! sw down, visible, direct  (W/m^2)
         swvdf    , & ! sw down, visible, diffuse (W/m^2)
         swidr    , & ! sw down, near IR, direct  (W/m^2)
         swidf        ! sw down, near IR, diffuse (W/m^2)

      real (kind=dbl_kind), intent(in), optional :: &
         swuvrdr  , & ! sw down, vis uvr dir (W/m^2)
         swuvrdf  , & ! sw down, vis uvr dif (W/m^2)
         swpardr  , & ! sw down, vis par dir (W/m^2)
         swpardf      ! sw down, vis par dif (W/m^2)

      ! baseline albedos for ccsm3 shortwave, set in namelist
      real (kind=dbl_kind), intent(in) :: &
         albicev , & ! visible ice albedo for h > ahmax
         albicei , & ! near-ir ice albedo for h > ahmax
         albsnowv, & ! cold snow albedo, visible
         albsnowi, & ! cold snow albedo, near IR
         ahmax       ! thickness above which ice albedo is constant (m)

      character (len=char_len), intent(in) :: &
         albedo_type  ! albedo parameterization, 'ccsm3' or 'constant'

      real (kind=dbl_kind), dimension (:), intent(inout) :: &
         alvdrn   , & ! visible, direct, avg   (fraction)
         alidrn   , & ! near-ir, direct, avg   (fraction)
         alvdfn   , & ! visible, diffuse, avg  (fraction)
         alidfn   , & ! near-ir, diffuse, avg  (fraction)
         fswsfc   , & ! SW absorbed at ice/snow surface (W m-2)
         fswint   , & ! SW absorbed in ice interior, below surface (W m-2)
         fswthrun , & ! SW through ice to ocean (W m-2)
         albin    , & ! bare ice albedo
         albsn        ! snow albedo

      real (kind=dbl_kind), dimension (:), intent(out), optional :: &
         fswthrun_vdr, & ! vis dir SW through ice to ocean (W m-2)
         fswthrun_vdf, & ! vis dif SW through ice to ocean (W m-2)
         fswthrun_idr, & ! nir dir SW through ice to ocean (W m-2)
         fswthrun_idf, & ! nir dif SW through ice to ocean (W m-2)
         fswthrun_uvrdr,&! vis dir uvr SW through ice to ocean (W m-2)
         fswthrun_uvrdf,&! vis dif uvr SW through ice to ocean (W m-2)
         fswthrun_pardr,&! vis dir par SW through ice to ocean (W m-2)
         fswthrun_pardf  ! vis dif par SW through ice to ocean (W m-2)

      real (kind=dbl_kind), intent(inout) :: &
         coszen       ! cosine(zenith angle)

      real (kind=dbl_kind), dimension (:,:), intent(inout) :: &
         fswpenl  , & ! SW entering ice layers (W m-2)
         Iswabs   , & ! SW absorbed in particular layer (W m-2)
         Sswabs       ! SW absorbed in particular layer (W m-2)

      ! local variables

      integer (kind=int_kind) :: &
         n                  ! thickness category index

      ! ice and snow albedo for each category

      real (kind=dbl_kind) :: &
         alvdrni, & ! visible, direct, ice    (fraction)
         alidrni, & ! near-ir, direct, ice    (fraction)
         alvdfni, & ! visible, diffuse, ice   (fraction)
         alidfni, & ! near-ir, diffuse, ice   (fraction)
         alvdrns, & ! visible, direct, snow   (fraction)
         alidrns, & ! near-ir, direct, snow   (fraction)
         alvdfns, & ! visible, diffuse, snow  (fraction)
         alidfns    ! near-ir, diffuse, snow  (fraction)

      ! needed for optional fswthrun arrays when passed as scalars
      real (kind=dbl_kind) :: &
         l_fswthru_vdr, & ! vis dir SW through ice to ocean (W m-2)
         l_fswthru_vdf, & ! vis dif SW through ice to ocean (W m-2)
         l_fswthru_idr, & ! nir dir SW through ice to ocean (W m-2)
         l_fswthru_idf, & ! nir dif SW through ice to ocean (W m-2)
         l_fswthru_uvrdr,&! vis uvr dir SW through ice to ocean (W m-2)
         l_fswthru_uvrdf,&! vis uvr dif SW through ice to ocean (W m-2)
         l_fswthru_pardr,&! vis par dir SW through ice to ocean (W m-2)
         l_fswthru_pardf  ! vis par dif SW through ice to ocean (W m-2)
      character(len=*),parameter :: subname='(shortwave_ccsm3)'

      !-----------------------------------------------------------------
      ! Solar radiation: albedo and absorbed shortwave
      !-----------------------------------------------------------------

      ! For basic shortwave, set coszen to a constant between 0 and 1.
      coszen = p5 ! sun above the horizon

      do n = 1, ncat

      Sswabs(:,n) = c0

      alvdrni = albocn
      alidrni = albocn
      alvdfni = albocn
      alidfni = albocn

      alvdrns = albocn
      alidrns = albocn
      alvdfns = albocn
      alidfns = albocn

      alvdrn(n) = albocn
      alidrn(n) = albocn
      alvdfn(n) = albocn
      alidfn(n) = albocn

      albin(n) = c0
      albsn(n) = c0

      fswsfc(n)    = c0
      fswint(n)    = c0
      fswthrun(n)  = c0
      fswpenl(:,n) = c0
      Iswabs (:,n) = c0

      if (aicen(n) > puny) then

      !-----------------------------------------------------------------
      ! Compute albedos for ice and snow.
      !-----------------------------------------------------------------

         if (trim(albedo_type) == 'constant') then

            call constant_albedos (aicen(n),             &
                                   vsnon(n),             &
                                   Tsfcn(n),             &
                                   alvdrni,    alidrni,  &
                                   alvdfni,    alidfni,  &
                                   alvdrns,    alidrns,  &
                                   alvdfns,    alidfns,  &
                                   alvdrn(n),            &
                                   alidrn(n),            &
                                   alvdfn(n),            &
                                   alidfn(n),            &
                                   albin(n),             &
                                   albsn(n))
            if (icepack_warnings_aborted(subname)) return

         elseif (trim(albedo_type) == 'ccsm3') then

            call compute_albedos (aicen(n),             &
                                  vicen(n),             &
                                  vsnon(n),             &
                                  Tsfcn(n),             &
                                  albicev,    albicei,  &
                                  albsnowv,   albsnowi, &
                                  ahmax,                &
                                  alvdrni,    alidrni,  &
                                  alvdfni,    alidfni,  &
                                  alvdrns,    alidrns,  &
                                  alvdfns,    alidfns,  &
                                  alvdrn(n),            &
                                  alidrn(n),            &
                                  alvdfn(n),            &
                                  alidfn(n),            &
                                  albin(n),             &
                                  albsn(n))
            if (icepack_warnings_aborted(subname)) return

         else

            call icepack_warnings_add(subname//' ERROR: albedo_type '//trim(albedo_type)//' unknown')
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            return

         endif

      !-----------------------------------------------------------------
      ! Compute solar radiation absorbed in ice and penetrating to ocean.
      !-----------------------------------------------------------------

         call absorbed_solar  (aicen(n),             &
                               vicen(n),             &
                               vsnon(n),             &
                               swvdr,      swvdf,    &
                               swidr,      swidf,    &
                               alvdrni,    alvdfni,  &
                               alidrni,    alidfni,  &
                               alvdrns,    alvdfns,  &
                               alidrns,    alidfns,  &
                               swuvrdr=swuvrdr,      &
                               swuvrdf=swuvrdf,      &
                               swpardr=swpardr,      &
                               swpardf=swpardf,      &
                               fswsfc=fswsfc(n),     &
                               fswint=fswint(n),     &
                               fswthru=fswthrun(n),  &
                               fswthru_vdr=l_fswthru_vdr, &
                               fswthru_vdf=l_fswthru_vdf, &
                               fswthru_idr=l_fswthru_idr, &
                               fswthru_idf=l_fswthru_idf, &
                               fswthru_uvrdr=l_fswthru_uvrdr,&
                               fswthru_uvrdf=l_fswthru_uvrdf,&
                               fswthru_pardr=l_fswthru_pardr,&
                               fswthru_pardf=l_fswthru_pardf,&
                               fswpenl=fswpenl(:,n), &
                               Iswabs=Iswabs(:,n))

         if (icepack_warnings_aborted(subname)) return

         if (present(fswthrun_vdr)) fswthrun_vdr(n) = l_fswthru_vdr
         if (present(fswthrun_vdf)) fswthrun_vdf(n) = l_fswthru_vdf
         if (present(fswthrun_idr)) fswthrun_idr(n) = l_fswthru_idr
         if (present(fswthrun_idf)) fswthrun_idf(n) = l_fswthru_idf
         if (present(fswthrun_uvrdr)) fswthrun_uvrdr(n) = l_fswthru_uvrdr
         if (present(fswthrun_uvrdf)) fswthrun_uvrdf(n) = l_fswthru_uvrdf
         if (present(fswthrun_pardr)) fswthrun_pardr(n) = l_fswthru_pardr
         if (present(fswthrun_pardf)) fswthrun_pardf(n) = l_fswthru_pardf

      endif ! aicen > puny

      enddo                  ! ncat

      end subroutine shortwave_ccsm3

!=======================================================================
!
! Compute albedos for each thickness category

      subroutine compute_albedos (aicen,    vicen,    &
                                  vsnon,    Tsfcn,    &
                                  albicev,  albicei,  &
                                  albsnowv, albsnowi, &
                                  ahmax,              &
                                  alvdrni,  alidrni,  &
                                  alvdfni,  alidfni,  &
                                  alvdrns,  alidrns,  &
                                  alvdfns,  alidfns,  &
                                  alvdrn,   alidrn,   &
                                  alvdfn,   alidfn,   &
                                  albin,    albsn)

      real (kind=dbl_kind), intent(in) :: &
         aicen   , & ! concentration of ice per category
         vicen   , & ! volume of ice per category
         vsnon   , & ! volume of ice per category
         Tsfcn       ! surface temperature

      ! baseline albedos for ccsm3 shortwave, set in namelist
      real (kind=dbl_kind), intent(in) :: &
         albicev , & ! visible ice albedo for h > ahmax
         albicei , & ! near-ir ice albedo for h > ahmax
         albsnowv, & ! cold snow albedo, visible
         albsnowi, & ! cold snow albedo, near IR
         ahmax       ! thickness above which ice albedo is constant (m)

      real (kind=dbl_kind), intent(out) :: &
         alvdrni  , & ! visible, direct, ice   (fraction)
         alidrni  , & ! near-ir, direct, ice   (fraction)
         alvdfni  , & ! visible, diffuse, ice  (fraction)
         alidfni  , & ! near-ir, diffuse, ice  (fraction)
         alvdrns  , & ! visible, direct, snow  (fraction)
         alidrns  , & ! near-ir, direct, snow  (fraction)
         alvdfns  , & ! visible, diffuse, snow (fraction)
         alidfns  , & ! near-ir, diffuse, snow (fraction)
         alvdrn   , & ! visible, direct, avg   (fraction)
         alidrn   , & ! near-ir, direct, avg   (fraction)
         alvdfn   , & ! visible, diffuse, avg  (fraction)
         alidfn   , & ! near-ir, diffuse, avg  (fraction)
         albin    , & ! bare ice
         albsn        ! snow

      ! local variables

      real (kind=dbl_kind), parameter :: &
         dT_melt   = c1          , & ! change in temp to give dalb_mlt
                                     ! albedo change
         dalb_mlt  = -0.075_dbl_kind, & ! albedo change per dT_melt change
                                     ! in temp for ice
         dalb_mltv = -p1         , & ! albedo vis change per dT_melt change
                                     ! in temp for snow
         dalb_mlti = -p15            ! albedo nir change per dT_melt change
                                     ! in temp for snow

      real (kind=dbl_kind) :: &
         hi  , & ! ice thickness  (m)
         hs  , & ! snow thickness  (m)
         albo, & ! effective ocean albedo, function of ice thickness
         fh  , & ! piecewise linear function of thickness
         fT  , & ! piecewise linear function of surface temperature
         dTs , & ! difference of Tsfc and Timelt
         fhtan,& ! factor used in albedo dependence on ice thickness
         asnow   ! fractional area of snow cover

      character(len=*),parameter :: subname='(compute_albedos)'

      !-----------------------------------------------------------------
      ! Compute albedo for each thickness category.
      !-----------------------------------------------------------------

         hi = vicen / aicen
         hs = vsnon / aicen

         ! bare ice, thickness dependence
         fhtan = atan(ahmax*c4)
         fh = min(atan(hi*c4)/fhtan,c1)
         albo = albocn*(c1-fh)
         alvdfni = albicev*fh + albo
         alidfni = albicei*fh + albo

         ! bare ice, temperature dependence
         dTs = Timelt - Tsfcn
         fT = min(dTs/dT_melt-c1,c0)
         alvdfni = alvdfni - dalb_mlt*fT
         alidfni = alidfni - dalb_mlt*fT

         ! avoid negative albedos for thin, bare, melting ice
         alvdfni = max (alvdfni, albocn)
         alidfni = max (alidfni, albocn)

         if (hs > puny) then

            alvdfns = albsnowv
            alidfns = albsnowi

            ! snow on ice, temperature dependence
            alvdfns = alvdfns - dalb_mltv*fT
            alidfns = alidfns - dalb_mlti*fT

         endif                  ! hs > puny

         ! direct albedos (same as diffuse for now)
         alvdrni = alvdfni
         alidrni = alidfni
         alvdrns = alvdfns
         alidrns = alidfns

         ! fractional area of snow cover
         if (hs > puny) then
            asnow = hs / (hs + snowpatch)
         else
            asnow = c0
         endif

         ! combine ice and snow albedos (for coupler)
         alvdfn = alvdfni*(c1-asnow) + &
                  alvdfns*asnow
         alidfn = alidfni*(c1-asnow) + &
                  alidfns*asnow
         alvdrn = alvdrni*(c1-asnow) + &
                  alvdrns*asnow
         alidrn = alidrni*(c1-asnow) + &
                  alidrns*asnow

         ! save ice and snow albedos (for history)
         albin = awtvdr*alvdrni + awtidr*alidrni &
               + awtvdf*alvdfni + awtidf*alidfni
         albsn = awtvdr*alvdrns + awtidr*alidrns &
               + awtvdf*alvdfns + awtidf*alidfns

      end subroutine compute_albedos

!=======================================================================
!
! Compute albedos for each thickness category

      subroutine constant_albedos (aicen,              &
                                   vsnon,    Tsfcn,    &
                                   alvdrni,  alidrni,  &
                                   alvdfni,  alidfni,  &
                                   alvdrns,  alidrns,  &
                                   alvdfns,  alidfns,  &
                                   alvdrn,   alidrn,   &
                                   alvdfn,   alidfn,   &
                                   albin,    albsn)

      real (kind=dbl_kind), intent(in) :: &
         aicen   , & ! concentration of ice per category
         vsnon   , & ! volume of ice per category
         Tsfcn       ! surface temperature

      real (kind=dbl_kind), intent(out) :: &
         alvdrni  , & ! visible, direct, ice   (fraction)
         alidrni  , & ! near-ir, direct, ice   (fraction)
         alvdfni  , & ! visible, diffuse, ice  (fraction)
         alidfni  , & ! near-ir, diffuse, ice  (fraction)
         alvdrns  , & ! visible, direct, snow  (fraction)
         alidrns  , & ! near-ir, direct, snow  (fraction)
         alvdfns  , & ! visible, diffuse, snow (fraction)
         alidfns  , & ! near-ir, diffuse, snow (fraction)
         alvdrn   , & ! visible, direct, avg   (fraction)
         alidrn   , & ! near-ir, direct, avg   (fraction)
         alvdfn   , & ! visible, diffuse, avg  (fraction)
         alidfn   , & ! near-ir, diffuse, avg  (fraction)
         albin    , & ! bare ice
         albsn        ! snow

      ! local variables

      real (kind=dbl_kind), parameter :: &
         warmice  = 0.68_dbl_kind, &
         coldice  = 0.70_dbl_kind, &
         warmsnow = 0.77_dbl_kind, &
         coldsnow = 0.81_dbl_kind

      real (kind=dbl_kind) :: &
         hs      ! snow thickness  (m)

      character(len=*),parameter :: subname='(constant_albedos)'

      !-----------------------------------------------------------------
      ! Compute albedo for each thickness category.
      !-----------------------------------------------------------------

         hs = vsnon / aicen

         if (hs > puny) then
            ! snow, temperature dependence
            if (Tsfcn >= -c2*puny) then
               alvdfn = warmsnow
               alidfn = warmsnow
            else
               alvdfn = coldsnow
               alidfn = coldsnow
            endif
         else      ! hs < puny
            ! bare ice, temperature dependence
            if (Tsfcn >= -c2*puny) then
               alvdfn = warmice
               alidfn = warmice
            else
               alvdfn = coldice
               alidfn = coldice
            endif
         endif                  ! hs > puny

         ! direct albedos (same as diffuse for now)
         alvdrn  = alvdfn
         alidrn  = alidfn

         alvdrni = alvdrn
         alidrni = alidrn
         alvdrns = alvdrn
         alidrns = alidrn
         alvdfni = alvdfn
         alidfni = alidfn
         alvdfns = alvdfn
         alidfns = alidfn

         ! save ice and snow albedos (for history)
         albin = awtvdr*alvdrni + awtidr*alidrni &
               + awtvdf*alvdfni + awtidf*alidfni
         albsn = awtvdr*alvdrns + awtidr*alidrns &
               + awtvdf*alvdfns + awtidf*alidfns

      end subroutine constant_albedos

!=======================================================================
!
! Compute solar radiation absorbed in ice and penetrating to ocean
!
! authors William H. Lipscomb, LANL
!         C. M. Bitz, UW

      subroutine absorbed_solar (aicen,    &
                                 vicen,    vsnon,    &
                                 swvdr,    swvdf,    &
                                 swidr,    swidf,    &
                                 alvdrni,  alvdfni,  &
                                 alidrni,  alidfni,  &
                                 alvdrns,  alvdfns,  &
                                 alidrns,  alidfns,  &
                                 swuvrdr,  swuvrdf,  &
                                 swpardr,  swpardf,  &
                                 fswsfc,   fswint,   &
                                 fswthru,            &
                                 fswthru_vdr,        &
                                 fswthru_vdf,        &
                                 fswthru_idr,        &
                                 fswthru_idf,        &
                                 fswthru_uvrdr,      &
                                 fswthru_uvrdf,      &
                                 fswthru_pardr,      &
                                 fswthru_pardf,      &
                                 fswpenl,            &
                                 Iswabs)

      real (kind=dbl_kind), intent(in) :: &
         aicen       , & ! fractional ice area
         vicen       , & ! ice volume
         vsnon       , & ! snow volume
         swvdr       , & ! sw down, visible, direct  (W/m^2)
         swvdf       , & ! sw down, visible, diffuse (W/m^2)
         swidr       , & ! sw down, near IR, direct  (W/m^2)
         swidf       , & ! sw down, near IR, diffuse (W/m^2)
         alvdrni     , & ! visible, direct albedo,ice
         alidrni     , & ! near-ir, direct albedo,ice
         alvdfni     , & ! visible, diffuse albedo,ice
         alidfni     , & ! near-ir, diffuse albedo,ice
         alvdrns     , & ! visible, direct albedo, snow
         alidrns     , & ! near-ir, direct albedo, snow
         alvdfns     , & ! visible, diffuse albedo, snow
         alidfns         ! near-ir, diffuse albedo, snow

      real (kind=dbl_kind), intent(in), optional :: &
         swuvrdr     , & ! sw down, vis uvr dir (W/m^2)
         swuvrdf     , & ! sw down, vis uvr dif (W/m^2)
         swpardr     , & ! sw down, vis par dir (W/m^2)
         swpardf         ! sw down, vis par dif (W/m^2)

      real (kind=dbl_kind), intent(out):: &
         fswsfc      , & ! SW absorbed at ice/snow surface (W m-2)
         fswint      , & ! SW absorbed in ice interior, below surface (W m-2)
         fswthru         ! SW through ice to ocean (W m-2)

      real (kind=dbl_kind), intent(out) :: &
         fswthru_vdr  , & ! vis dir SW through ice to ocean (W m-2)
         fswthru_vdf  , & ! vis dif SW through ice to ocean (W m-2)
         fswthru_idr  , & ! nir dir SW through ice to ocean (W m-2)
         fswthru_idf  , & ! nir dif SW through ice to ocean (W m-2)
         fswthru_uvrdr, & ! vis dir uvr SW through ice to ocean (W m-2)
         fswthru_uvrdf, & ! vis dif uvr SW through ice to ocean (W m-2)
         fswthru_pardr, & ! vis dir par SW through ice to ocean (W m-2)
         fswthru_pardf    ! vis dif par SW through ice to ocean (W m-2)

      real (kind=dbl_kind), dimension (:), intent(out) :: &
         Iswabs      , & ! SW absorbed in particular layer (W m-2)
         fswpenl         ! visible SW entering ice layers (W m-2)

      ! local variables

      real (kind=dbl_kind), parameter :: &
         i0vis = 0.70_dbl_kind  ! fraction of penetrating solar rad (visible)

      integer (kind=int_kind) :: &
         k               ! ice layer index

      real (kind=dbl_kind) :: &
         fswpen      , & ! SW penetrating beneath surface (W m-2)
         trantop     , & ! transmitted frac of penetrating SW at layer top
         tranbot         ! transmitted frac of penetrating SW at layer bot

      real (kind=dbl_kind) :: &
         swabs       , & ! net SW down at surface (W m-2)
         swabsv      , & ! swabs in vis (wvlngth < 700nm)  (W/m^2)
         swabsi      , & ! swabs in nir (wvlngth > 700nm)  (W/m^2)
         fswpenvdr   , & ! penetrating SW, vis direct
         fswpenvdf   , & ! penetrating SW, vis diffuse
         hi          , & ! ice thickness (m)
         hs          , & ! snow thickness (m)
         hilyr       , & ! ice layer thickness
         asnow           ! fractional area of snow cover

      real (kind=dbl_kind) :: &
         swuvrdrpen  , & ! penetrating SW, vis uvr dir (W/m^2)
         swuvrdfpen  , & ! penetrating SW, vis uvr dif (W/m^2)
         swpardrpen  , & ! penetrating SW, vis par dir (W/m^2)
         swpardfpen      ! penetrating SW, vis par dif (W/m^2)

      character(len=*),parameter :: subname='(absorbed_solar)'

      !-----------------------------------------------------------------
      ! Initialize
      !-----------------------------------------------------------------

         trantop = c0
         tranbot = c0
         swuvrdrpen = c0
         swuvrdfpen = c0
         swpardrpen = c0
         swpardfpen = c0

         hs  = vsnon / aicen

      !-----------------------------------------------------------------
      ! Fractional snow cover
      !-----------------------------------------------------------------
         if (hs > puny) then
            asnow = hs / (hs + snowpatch)
         else
            asnow = c0
         endif

      !-----------------------------------------------------------------
      ! Shortwave flux absorbed at surface, absorbed internally,
      !  and penetrating to mixed layer.
      ! This parameterization assumes that all IR is absorbed at the
      !  surface; only visible is absorbed in the ice interior or
      !  transmitted to the ocean.
      !-----------------------------------------------------------------

         swabsv  = swvdr * ( (c1-alvdrni)*(c1-asnow) &
                           + (c1-alvdrns)*asnow ) &
                 + swvdf * ( (c1-alvdfni)*(c1-asnow) &
                           + (c1-alvdfns)*asnow )

         swabsi  = swidr * ( (c1-alidrni)*(c1-asnow) &
                           + (c1-alidrns)*asnow ) &
                 + swidf * ( (c1-alidfni)*(c1-asnow) &
                           + (c1-alidfns)*asnow )

         swabs   = swabsv + swabsi

         fswpenvdr = swvdr * (c1-alvdrni) * (c1-asnow) * i0vis
         fswpenvdf = swvdf * (c1-alvdfni) * (c1-asnow) * i0vis

          ! no penetrating radiation in near IR
!         fswpenidr = swidr * (c1-alidrni) * (c1-asnow) * i0nir
!         fswpenidf = swidf * (c1-alidfni) * (c1-asnow) * i0nir

         if (present(swuvrdr)) swuvrdrpen = swuvrdr * (c1-alvdrni) * (c1-asnow) * i0vis
         if (present(swuvrdf)) swuvrdfpen = swuvrdf * (c1-alvdfni) * (c1-asnow) * i0vis
         if (present(swpardr)) swpardrpen = swpardr * (c1-alvdrni) * (c1-asnow) * i0vis
         if (present(swpardf)) swpardfpen = swpardf * (c1-alvdfni) * (c1-asnow) * i0vis

         fswpen = fswpenvdr + fswpenvdf

         fswsfc = swabs - fswpen

         trantop = c1  ! transmittance at top of ice

      !-----------------------------------------------------------------
      ! penetrating SW absorbed in each ice layer
      !-----------------------------------------------------------------

         do k = 1, nilyr

            hi  = vicen / aicen
            hilyr = hi / real(nilyr,kind=dbl_kind)

            tranbot = exp (-kappav * hilyr * real(k,kind=dbl_kind))
            Iswabs(k) = fswpen * (trantop-tranbot)

            ! bottom of layer k = top of layer k+1
            trantop = tranbot

            ! bgc layer model
            if (k == 1) then   ! surface flux
               fswpenl(k)   = fswpen
               fswpenl(k+1) = fswpen * tranbot
            else
               fswpenl(k+1) = fswpen * tranbot
            endif
         enddo                     ! nilyr

         ! SW penetrating thru ice into ocean
         fswthru = fswpen * tranbot
         fswthru_vdr = fswpenvdr * tranbot
         fswthru_vdf = fswpenvdf * tranbot
         fswthru_idr = c0
         fswthru_idf = c0
         fswthru_uvrdr = swuvrdrpen * tranbot
         fswthru_uvrdf = swuvrdfpen * tranbot
         fswthru_pardr = swpardrpen * tranbot
         fswthru_pardf = swpardfpen * tranbot

         ! SW absorbed in ice interior
         fswint  = fswpen - fswthru

      end subroutine absorbed_solar

! End ccsm3 shortwave method
!=======================================================================
! Begin Delta-Eddington shortwave method

! Compute initial data for Delta-Eddington method, specifically,
! the approximate exponential look-up table.
!
! author:  Bruce P. Briegleb, NCAR
! 2011 ECH modified for melt pond tracers
! 2013 ECH merged with NCAR version
! 2024 DCS refactored for sealvl ponds

      subroutine run_dEdd(dt,                  &
                          aicen,    vicen,     &
                          vsnon,    Tsfcn,     &
                          alvln,    apndn,     &
                          hpndn,    ipndn,     &
                          aeron,               &
                          trcrn_bgcsw,         &
                          TLAT,     TLON,      &
                          calendar_type,       &
                          days_per_year,       &
                          nextsw_cday,   yday, &
                          sec,                 &
                          swvdr,    swvdf,     &
                          swidr,    swidf,     &
                          coszen,   fsnow,     &
                          alvdrn,   alvdfn,    &
                          alidrn,   alidfn,    &
                          fswsfcn,  fswintn,   &
                          swuvrdr,  swuvrdf,   &
                          swpardr,  swpardf,   &
                          fswthrun,            &
                          fswthrun_vdr,        &
                          fswthrun_vdf,        &
                          fswthrun_idr,        &
                          fswthrun_idf,        &
                          fswthrun_uvrdr,      &
                          fswthrun_uvrdf,      &
                          fswthrun_pardr,      &
                          fswthrun_pardf,      &
                          fswpenln,            &
                          Sswabsn,  Iswabsn,   &
                          albicen,  albsnon,   &
                          albpndn,  apeffn,    &
                          snowfracn,           &
                          dhsn,     ffracn,    &
                          rsnow,               &
                          l_print_point,       &
                          initonly)

      integer (kind=int_kind), intent(in) :: &
         sec        ! elapsed seconds into date

      real (kind=dbl_kind), intent(in), optional :: &
         yday       ! day of the year

      character (len=char_len), intent(in), optional :: &
         calendar_type       ! differentiates proleptic_gregorian from other calendars

      integer (kind=int_kind), intent(in), optional :: &
         days_per_year       ! number of days in one year

      real (kind=dbl_kind), intent(in), optional :: &
         nextsw_cday         ! julian day of next shortwave calculation

      real(kind=dbl_kind), intent(in) :: &
         dt,    & ! time step (s)
         TLAT,  & ! latitude of temp pts (radians)
         TLON,  & ! longitude of temp pts (radians)
         swvdr, & ! sw down, visible, direct  (W/m^2)
         swvdf, & ! sw down, visible, diffuse (W/m^2)
         swidr, & ! sw down, near IR, direct  (W/m^2)
         swidf, & ! sw down, near IR, diffuse (W/m^2)
         fsnow    ! snowfall rate (kg/m^2 s)

      real (kind=dbl_kind), intent(in), optional :: &
         swuvrdr, & ! sw down, vis uvr dir (W/m^2)
         swuvrdf, & ! sw down, vis uvr dif (W/m^2)
         swpardr, & ! sw down, vis par dir (W/m^2)
         swpardf    ! sw down, vis par dif (W/m^2)

      real(kind=dbl_kind), dimension(:), intent(in) :: &
         aicen, & ! concentration of ice
         vicen, & ! volume per unit area of ice (m)
         vsnon, & ! volume per unit area of snow (m)
         Tsfcn, & ! surface temperature (deg C)
         alvln, & ! level-ice area fraction
         apndn, & ! pond area fraction
         hpndn, & ! pond depth (m)
         ipndn, & ! pond refrozen lid thickness (m)
         ffracn   ! fraction of fsurfn used to melt ipond

      real(kind=dbl_kind), dimension(:,:), intent(in) :: &
         aeron,    & ! aerosols (kg/m^3)
         trcrn_bgcsw ! zaerosols (kg/m^3) + chlorophyll on shorthwave grid

      real(kind=dbl_kind), dimension(:), intent(inout) :: &
         dhsn        ! depth difference for snow on sea ice and pond ice

      real(kind=dbl_kind), intent(inout) :: &
         coszen      ! cosine solar zenith angle, < 0 for sun below horizon

      real(kind=dbl_kind), dimension(:), intent(inout) :: &
         alvdrn,   & ! visible direct albedo (fraction)
         alvdfn,   & ! near-ir direct albedo (fraction)
         alidrn,   & ! visible diffuse albedo (fraction)
         alidfn,   & ! near-ir diffuse albedo (fraction)
         fswsfcn,  & ! SW absorbed at ice/snow surface (W m-2)
         fswintn,  & ! SW absorbed in ice interior, below surface (W m-2)
         fswthrun, & ! SW through ice to ocean (W/m^2)
         albicen,  & ! albedo bare ice
         albsnon,  & ! albedo snow
         albpndn,  & ! albedo pond
         apeffn,   & ! effective pond area used for radiation calculation
         snowfracn   ! snow fraction on each category used for radiation

      real(kind=dbl_kind), dimension(:), intent(out), optional :: &
         fswthrun_vdr, & ! vis dir SW through ice to ocean (W/m^2)
         fswthrun_vdf, & ! vis dif SW through ice to ocean (W/m^2)
         fswthrun_idr, & ! nir dir SW through ice to ocean (W/m^2)
         fswthrun_idf, & ! nir dif SW through ice to ocean (W/m^2)
         fswthrun_uvrdr,&! uvr dir SW through ice to ocean (W/m^2)
         fswthrun_uvrdf,&! uvr dif SW through ice to ocean (W/m^2)
         fswthrun_pardr,&! par dir SW through ice to ocean (W/m^2)
         fswthrun_pardf  ! par dif SW through ice to ocean (W/m^2)

      real(kind=dbl_kind), dimension(:,:), intent(inout) :: &
         Sswabsn , & ! SW radiation absorbed in snow layers (W m-2)
         Iswabsn , & ! SW radiation absorbed in ice layers (W m-2)
         fswpenln    ! visible SW entering ice layers (W m-2)

      real(kind=dbl_kind), dimension(:,:), intent(inout), optional :: &
         rsnow       ! snow grain radius tracer (10^-6 m)

      logical (kind=log_kind), intent(in) :: &
         l_print_point ! print diagnostic information

      logical (kind=log_kind), optional :: &
         initonly    ! flag to indicate init only, default is false

      ! local variables
      ! snow variables for Delta-Eddington shortwave
      real (kind=dbl_kind) :: &
         fsn         , & ! snow horizontal fraction
         hsn             ! snow depth (m)

      real (kind=dbl_kind), dimension (nslyr) :: &
         rhosnwn     , & ! snow density (kg/m3)
         rsnwn           ! snow grain radius (micrometers)

      ! pond variables for Delta-Eddington shortwave
      real (kind=dbl_kind) :: &
         apondn      , & ! pond fraction of category (incl. deformed)
         fpn         , & ! pond fraction of ice cover
         hpn             ! actual pond depth (m)

      integer (kind=int_kind) :: &
         n           , & ! thickness category index
         k               ! snow layer index

      ! needed for optional fswthrun arrays when passed as scalars
      real (kind=dbl_kind) :: &
         l_fswthru_vdr  , & ! vis dir SW through ice to ocean (W m-2)
         l_fswthru_vdf  , & ! vis dif SW through ice to ocean (W m-2)
         l_fswthru_idr  , & ! nir dir SW through ice to ocean (W m-2)
         l_fswthru_idf  , & ! nir dif SW through ice to ocean (W m-2)
         l_fswthru_uvrdr, & ! vis uvr dir SW through ice to ocean (W m-2)
         l_fswthru_uvrdf, & ! vis uvr dif SW through ice to ocean (W m-2)
         l_fswthru_pardr, & ! vis par dir SW through ice to ocean (W m-2)
         l_fswthru_pardf    ! vis par dif SW through ice to ocean (W m-2)

      logical (kind=log_kind) :: &
         l_initonly      ! local initonly value

      real(kind=dbl_kind), dimension(nslyr) :: &
         l_rsnows        ! snow grain radius tracer (10^-6 m)

      character(len=*),parameter :: subname='(run_dEdd)'

      l_initonly = .false.
      if (present(initonly)) then
         l_initonly = initonly
      endif

      l_rsnows(:) = c0

      ! cosine of the zenith angle
#ifdef CESMCOUPLED
      call compute_coszen (TLAT, TLON, yday,  sec, coszen,  &
                           days_per_year, nextsw_cday, calendar_type)
#else
      if (.not.semi_implicit_Tsfc) then  ! geos sets solar angles in driver level
         call compute_coszen (TLAT, TLON, yday,  sec, coszen)
      endif
#endif
      if (icepack_warnings_aborted(subname)) return

      do n = 1, ncat

      ! note that rhosnwn, rsnw, fp, hp and Sswabs ARE NOT dimensioned with ncat
      ! BPB 19 Dec 2006

         ! set snow properties
         fsn        = c0
         hsn        = c0
         rhosnwn(:) = c0
         rsnwn(:)   = c0
         apeffn(n)    = c0 ! for history
         snowfracn(n) = c0 ! for history

         if (aicen(n) > puny) then

            if (snwgrain) then
               l_rsnows(:) = rsnow(:,n)
            endif
            call shortwave_dEdd_set_snow(R_snw,                &
                                         dT_mlt,     rsnw_mlt, &
                                         aicen(n),   vsnon(n), &
                                         Tsfcn(n),   fsn,      &
                                         hs0,        hsn,      &
                                         rhosnwn,    rsnwn,    &
                                         l_rsnows(:))
            if (icepack_warnings_aborted(subname)) return

            ! set pond properties
            if (tr_pond_lvl) then
               apondn = alvln(n)*apndn(n)
               call shortwave_dEdd_set_eff(aicen(n),  vsnon(n),   &
                                           alvln(n),  apondn,     &
                                           hpndn(n),  ipndn(n),   &
                                           ffracn(n), fsnow,      &
                                           dt,        Tsfcn(n),   &
                                           fsn,       hsn,        &
                                           dhsn(n),   fpn,        &
                                           hpn,       apeffn(n),  &
                                           l_rsnows(:), rhosnwn(:), &
                                           rsnwn(:),  l_initonly)
               if (icepack_warnings_aborted(subname)) return
            elseif (tr_pond_sealvl) then
               apondn = apndn(n)
               call shortwave_dEdd_set_eff(aicen(n),  vsnon(n),   &
                                           alvln(n),  apondn,     &
                                           hpndn(n),  ipndn(n),   &
                                           ffracn(n), fsnow,      &
                                           dt,        Tsfcn(n),   &
                                           fsn,       hsn,        &
                                           dhsn(n),   fpn,        &
                                           hpn,       apeffn(n),  &
                                           l_rsnows(:), rhosnwn(:), &
                                           rsnwn(:),  l_initonly)
               if (icepack_warnings_aborted(subname)) return
            elseif (tr_pond_topo) then
               ! Lid effective if thicker than hp1
               if (apndn(n)*aicen(n) > puny .and. ipndn(n) < hp1) then
                  fpn = apndn(n)
               else
                  fpn = c0
               endif
               if (apndn(n) > puny) then
                  hpn = hpndn(n)
               else
                  fpn = c0
                  hpn = c0
               endif

               ! Zero out fraction of thin ponds for radiation only
               if (hpn < hpmin) fpn = c0

               ! If ponds are present snow fraction reduced to
               ! non-ponded part dEdd scheme
               fsn = min(fsn, c1-fpn)

               apeffn(n) = fpn
            else
               fpn = c0
               hpn = c0
               call shortwave_dEdd_set_pond(Tsfcn(n),   &
                                            fsn, fpn,   &
                                            hpn)
               if (icepack_warnings_aborted(subname)) return

               apeffn(n) = fpn ! for history
               fpn = c0
               hpn = c0
            endif ! pond type

            snowfracn(n) = fsn ! for history

            call shortwave_dEdd(                            &
                             coszen,                        &
                             aicen(n),      vicen(n),       &
                             hsn,           fsn,            &
                             rhosnwn,       rsnwn,          &
                             fpn,           hpn,            &
                             aeron(:,n),                    &
                             swvdr,         swvdf,          &
                             swidr,         swidf,          &
                             alvdrn(n),     alvdfn(n),      &
                             alidrn(n),     alidfn(n),      &
                             fswsfcn(n),    fswintn(n),     &
                             swuvrdr=swuvrdr,               &
                             swuvrdf=swuvrdf,               &
                             swpardr=swpardr,               &
                             swpardf=swpardf,               &
                             fswthru=fswthrun(n),           &
                             fswthru_vdr=l_fswthru_vdr,     &
                             fswthru_vdf=l_fswthru_vdf,     &
                             fswthru_idr=l_fswthru_idr,     &
                             fswthru_idf=l_fswthru_idf,     &
                             fswthru_uvrdr=l_fswthru_uvrdr, &
                             fswthru_uvrdf=l_fswthru_uvrdf, &
                             fswthru_pardr=l_fswthru_pardr, &
                             fswthru_pardf=l_fswthru_pardf, &
                             Sswabs=Sswabsn(:,n),           &
                             Iswabs=Iswabsn(:,n),           &
                             albice=albicen(n),             &
                             albsno=albsnon(n),             &
                             albpnd=albpndn(n),             &
                             fswpenl=fswpenln(:,n),         &
                             zbio=trcrn_bgcsw(:,n),         &
                             l_print_point=l_print_point)

            if (icepack_warnings_aborted(subname)) return

            if(present(fswthrun_vdr)) fswthrun_vdr(n) = l_fswthru_vdr
            if(present(fswthrun_vdf)) fswthrun_vdf(n) = l_fswthru_vdf
            if(present(fswthrun_idr)) fswthrun_idr(n) = l_fswthru_idr
            if(present(fswthrun_idf)) fswthrun_idf(n) = l_fswthru_idf
            if(present(fswthrun_uvrdr)) fswthrun_uvrdr(n) = l_fswthru_uvrdr
            if(present(fswthrun_uvrdf)) fswthrun_uvrdf(n) = l_fswthru_uvrdf
            if(present(fswthrun_pardr)) fswthrun_pardr(n) = l_fswthru_pardr
            if(present(fswthrun_pardf)) fswthrun_pardf(n) = l_fswthru_pardf

            if (present(rsnow) .and. .not. snwgrain) then
               do k = 1,nslyr
                  rsnow(k,n) = rsnwn(k) ! for history
               enddo
            endif

         endif ! aicen > puny

      enddo  ! ncat

      end subroutine run_dEdd

!=======================================================================
!
!   Compute snow/bare ice/ponded ice shortwave albedos, absorbed and transmitted
!   flux using the Delta-Eddington solar radiation method as described in:
!
!   A Delta-Eddington Multiple Scattering Parameterization for Solar Radiation
!        in the Sea Ice Component of the Community Climate System Model
!            B.P.Briegleb and B.Light   NCAR/TN-472+STR  February 2007
!
!   Compute shortwave albedos and fluxes for three surface types:
!   snow over ice, bare ice and ponded ice.
!
!   Albedos and fluxes are output for later use by thermodynamic routines.
!   Invokes three calls to compute_dEdd, which sets inherent optical properties
!   appropriate for the surface type. Within compute_dEdd, a call to solution_dEdd
!   evaluates the Delta-Eddington solution. The final albedos and fluxes are then
!   evaluated in compute_dEdd. Albedos and fluxes are transferred to output in
!   this routine.
!
!   NOTE regarding albedo diagnostics:  This method yields zero albedo values
!   if there is no incoming solar and thus the albedo diagnostics are masked
!   out when the sun is below the horizon.  To estimate albedo from the history
!   output (post-processing), compute ice albedo using
!   (1 - albedo)*swdn = swabs. -ECH
!
! author:  Bruce P. Briegleb, NCAR
!   2013:  E Hunke merged with NCAR version
!
      subroutine shortwave_dEdd  (coszen,                &
                                  aice,     vice,        &
                                  hs,       fs,          &
                                  rhosnw,   rsnw,        &
                                  fp,       hp,          &
                                  aero,                  &
                                  swvdr,    swvdf,       &
                                  swidr,    swidf,       &
                                  alvdr,    alvdf,       &
                                  alidr,    alidf,       &
                                  fswsfc,   fswint,      &
                                  swuvrdr,  swuvrdf,     &
                                  swpardr,  swpardf,     &
                                  fswthru,               &
                                  fswthru_vdr,           &
                                  fswthru_vdf,           &
                                  fswthru_idr,           &
                                  fswthru_idf,           &
                                  fswthru_uvrdr,         &
                                  fswthru_uvrdf,         &
                                  fswthru_pardr,         &
                                  fswthru_pardf,         &
                                  Sswabs,                &
                                  Iswabs,   albice,      &
                                  albsno,   albpnd,      &
                                  fswpenl,  zbio,        &
                                  l_print_point )

      real (kind=dbl_kind), intent(in) :: &
         aice    , & ! concentration of ice
         vice    , & ! volume of ice
         hs      , & ! snow depth
         fs          ! horizontal coverage of snow

      real (kind=dbl_kind), dimension (:), intent(in) :: &
         rhosnw  , & ! density in snow layer (kg/m3)
         rsnw    , & ! grain radius in snow layer (m)
         aero    , & ! aerosol tracers
         zbio        ! shortwave tracers (zaero+chla)

      real (kind=dbl_kind), intent(in) :: &
         fp      , & ! pond fractional coverage (0 to 1)
         hp      , & ! pond depth (m)
         swvdr   , & ! sw down, visible, direct  (W/m^2)
         swvdf   , & ! sw down, visible, diffuse (W/m^2)
         swidr   , & ! sw down, near IR, direct  (W/m^2)
         swidf       ! sw down, near IR, diffuse (W/m^2)

      real (kind=dbl_kind), intent(in), optional :: &
         swuvrdr , & ! sw down, vis uvr dir (W/m^2)
         swuvrdf , & ! sw down, vis uvr dif (W/m^2)
         swpardr , & ! sw down, vis par dir (W/m^2)
         swpardf     ! sw down, vis par dif (W/m^2)

      real (kind=dbl_kind), intent(inout) :: &
         coszen  , & ! cosine of solar zenith angle
         alvdr   , & ! visible, direct, albedo (fraction)
         alvdf   , & ! visible, diffuse, albedo (fraction)
         alidr   , & ! near-ir, direct, albedo (fraction)
         alidf   , & ! near-ir, diffuse, albedo (fraction)
         fswsfc  , & ! SW absorbed at snow/bare ice/pondedi ice surface (W m-2)
         fswint  , & ! SW interior absorption (below surface, above ocean,W m-2)
         fswthru     ! SW through snow/bare ice/ponded ice into ocean (W m-2)

      real (kind=dbl_kind), intent(out) :: &
         fswthru_vdr , & ! vis dir SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_vdf , & ! vis dif SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_idr , & ! nir dir SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_idf , & ! nir dif SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_uvrdr,& ! vis uvr dir sw radiation through ice bot (GEOS) (W/m**2)
         fswthru_uvrdf,& ! vis uvr dif sw radiation through ice bot (GEOS) (W/m**2)
         fswthru_pardr,& ! vis par dir sw radiation through ice bot (GEOS) (W/m**2)
         fswthru_pardf   ! vis par dif sw radiation through ice bot (GEOS) (W/m**2)

      real (kind=dbl_kind), dimension (:), intent(inout) :: &
         fswpenl , & ! visible SW entering ice layers (W m-2)
         Sswabs  , & ! SW absorbed in snow layer (W m-2)
         Iswabs      ! SW absorbed in ice layer (W m-2)

      real (kind=dbl_kind), intent(out) :: &
         albice  , & ! bare ice albedo, for history
         albsno  , & ! snow albedo, for history
         albpnd      ! pond albedo, for history

      logical (kind=log_kind) , intent(in) :: &
         l_print_point

      ! local variables

      real (kind=dbl_kind) :: &
         netsw    , & ! net shortwave
         fnidr    , & ! fraction of direct to total down surface flux in nir
         hstmp    , & ! snow thickness (set to 0 for bare ice case)
         hi       , & ! ice thickness (all sea ice layers, m)
         fi           ! snow/bare ice fractional coverage (0 to 1)

      real (kind=dbl_kind), dimension (4*n_aero) :: &
         aero_mp      ! aerosol mass path in kg/m2

      integer (kind=int_kind) :: &
         srftyp       ! surface type over ice: (0=air, 1=snow, 2=pond)

      integer (kind=int_kind) :: &
         k        , & ! level index
         na       , & ! aerosol index
         klev     , & ! number of radiation layers - 1
         klevp        ! number of radiation interfaces - 1
                      ! (0 layer is included also)

      real (kind=dbl_kind) :: &
         vsno         ! volume of snow

      real (kind=dbl_kind) :: &
         swdn     , & ! swvdr(i,j)+swvdf(i,j)+swidr(i,j)+swidf(i,j)
         swab     , & ! fswsfc(i,j)+fswint(i,j)+fswthru(i,j)
         swalb        ! (1.-swab/(swdn+.0001))

      ! for history
      real (kind=dbl_kind) :: &
         avdrl    , & ! visible, direct, albedo (fraction)
         avdfl    , & ! visible, diffuse, albedo (fraction)
         aidrl    , & ! near-ir, direct, albedo (fraction)
         aidfl        ! near-ir, diffuse, albedo (fraction)

      character(len=*),parameter :: subname='(shortwave_dEdd)'

!-----------------------------------------------------------------------

      klev    = nslyr + nilyr + 1   ! number of radiation layers - 1
      klevp   = klev  + 1           ! number of radiation interfaces - 1
                                    ! (0 layer is included also)

      ! set storage albedos and fluxes to zero for accumulation over surface types
      hstmp    = c0
      hi       = c0
      fi       = c0
      alvdr    = c0
      alvdf    = c0
      alidr    = c0
      alidf    = c0
      avdrl    = c0
      avdfl    = c0
      aidrl    = c0
      aidfl    = c0
      fswsfc   = c0
      fswint   = c0
      fswthru  = c0
      fswthru_vdr  = c0
      fswthru_vdf  = c0
      fswthru_idr  = c0
      fswthru_idf  = c0
      fswthru_uvrdr  = c0
      fswthru_uvrdf  = c0
      fswthru_pardr  = c0
      fswthru_pardf  = c0
      ! compute fraction of nir down direct to total over all points:
      fnidr = c0
      if( swidr + swidf > puny ) then
         fnidr = swidr/(swidr+swidf)
      endif
      albice     = c0
      albsno     = c0
      albpnd     = c0
      fswpenl(:) = c0
      Sswabs (:) = c0
      Iswabs (:) = c0

      ! compute aerosol mass path

         aero_mp(:) = c0
         if( tr_aero ) then
            ! check 4 layers for each aerosol, a snow SSL, snow below SSL,
            ! sea ice SSL, and sea ice below SSL, in that order.
            if (size(aero) < 4*n_aero) then
               call icepack_warnings_add(subname//' ERROR: size(aero) too small')
               call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
               return
            endif
            do na = 1, 4*n_aero, 4
               vsno = hs * aice
               netsw = swvdr + swidr + swvdf + swidf
               if (netsw > puny) then ! sun above horizon
                  aero_mp(na  ) = aero(na  )*vsno
                  aero_mp(na+1) = aero(na+1)*vsno
                  aero_mp(na+2) = aero(na+2)*vice
                  aero_mp(na+3) = aero(na+3)*vice
               endif                  ! aice > 0 and netsw > 0
            enddo      ! na
         endif      ! if aerosols

         ! compute shortwave radiation accounting for snow/ice (both snow over
         ! ice and bare ice) and ponded ice (if any):

         ! sea ice points with sun above horizon
         netsw = swvdr + swidr + swvdf + swidf
         if (netsw > puny) then ! sun above horizon
            coszen = max(puny,coszen)
            ! evaluate sea ice thickness and fraction
            hi  = vice / aice
            fi  = c1 - fs - fp
            ! bare sea ice points
            if(fi > c0) then
               ! calculate bare sea ice

               srftyp = 0
               call compute_dEdd_3bd( &
                      klev,   klevp,   zbio,   fnidr,  coszen,  &
                      swvdr,  swvdf,   swidr,  swidf,  srftyp,  &
                      hstmp,  rhosnw,  rsnw,   hi,     hp,      &
                      fi,     aero_mp, avdrl,  avdfl,           &
                      aidrl,  aidfl,   fswsfc, fswint, fswthru, &
                      fswthru_vdr,     fswthru_vdf,             &
                      fswthru_idr,     fswthru_idf,             &
                      fswthru_uvrdr,   fswthru_uvrdf,           &
                      fswthru_pardr,   fswthru_pardf,           &
                      Sswabs, Iswabs,  fswpenl,                 &
                      swuvrdr,swuvrdf, swpardr,swpardf)
               if (icepack_warnings_aborted(subname)) return

               alvdr = alvdr + avdrl*fi
               alvdf = alvdf + avdfl*fi
               alidr = alidr + aidrl*fi
               alidf = alidf + aidfl*fi
               ! for history
               albice = albice &
                      + awtvdr*avdrl + awtidr*aidrl &
                      + awtvdf*avdfl + awtidf*aidfl
            endif
         endif

         ! sea ice points with sun above horizon
         netsw = swvdr + swidr + swvdf + swidf
         if (netsw > puny) then ! sun above horizon
            coszen = max(puny,coszen)
            ! snow-covered sea ice points
            if(fs > c0) then
               ! calculate snow covered sea ice

               srftyp = 1
               if (trim(shortwave) == 'dEdd_snicar_ad') then
                call compute_dEdd_5bd(                          &
                      klev,   klevp,   zbio,   fnidr,  coszen,  &
                      swvdr,  swvdf,   swidr,  swidf,  srftyp,  &
                      hs,     rhosnw,  rsnw,   hi,     hp,      &
                      fs,     aero_mp, avdrl,  avdfl,           &
                      aidrl,  aidfl,   fswsfc, fswint, fswthru, &
                      fswthru_vdr,     fswthru_vdf,             &
                      fswthru_idr,     fswthru_idf,             &
                      Sswabs, Iswabs,  fswpenl )

               else
!echmod - this can be combined with the 5bd call above, if we use module data
                  call compute_dEdd_3bd(                        &
                      klev,   klevp,   zbio,   fnidr,  coszen,  &
                      swvdr,  swvdf,   swidr,  swidf,  srftyp,  &
                      hs,     rhosnw,  rsnw,   hi,     hp,      &
                      fs,     aero_mp, avdrl,  avdfl,           &
                      aidrl,  aidfl,   fswsfc, fswint, fswthru, &
                      fswthru_vdr,     fswthru_vdf,             &
                      fswthru_idr,     fswthru_idf,             &
                      fswthru_uvrdr,   fswthru_uvrdf,           &
                      fswthru_pardr,   fswthru_pardf,           &
                      Sswabs, Iswabs,  fswpenl,                 &
                      swuvrdr,swuvrdf, swpardr,swpardf)
               endif
               if (icepack_warnings_aborted(subname)) return

               alvdr = alvdr + avdrl*fs
               alvdf = alvdf + avdfl*fs
               alidr = alidr + aidrl*fs
               alidf = alidf + aidfl*fs
               ! for history
               albsno = albsno &
                      + awtvdr*avdrl + awtidr*aidrl &
                      + awtvdf*avdfl + awtidf*aidfl
            endif
         endif

         hi = c0

         ! sea ice points with sun above horizon
         netsw = swvdr + swidr + swvdf + swidf
         if (netsw > puny) then ! sun above horizon
            coszen = max(puny,coszen)
            hi  = vice / aice
            ! if nonzero pond fraction and sufficient pond depth
            ! if( fp > puny .and. hp > hpmin ) then
            if (fp > puny) then

               ! calculate ponded ice

               srftyp = 2
               call compute_dEdd_3bd(                           &
                      klev,   klevp,   zbio,   fnidr,  coszen,  &
                      swvdr,  swvdf,   swidr,  swidf,  srftyp,  &
                      hs,     rhosnw,  rsnw,   hi,     hp,      &
                      fp,     aero_mp, avdrl,  avdfl,           &
                      aidrl,  aidfl,   fswsfc, fswint, fswthru, &
                      fswthru_vdr,     fswthru_vdf,             &
                      fswthru_idr,     fswthru_idf,             &
                      fswthru_uvrdr,   fswthru_uvrdf,           &
                      fswthru_pardr,   fswthru_pardf,           &
                      Sswabs, Iswabs,  fswpenl,                 &
                      swuvrdr,swuvrdf, swpardr,swpardf)
               if (icepack_warnings_aborted(subname)) return

               alvdr = alvdr + avdrl*fp
               alvdf = alvdf + avdfl*fp
               alidr = alidr + aidrl*fp
               alidf = alidf + aidfl*fp
               ! for history
               albpnd = albpnd &
                      + awtvdr*avdrl + awtidr*aidrl &
                      + awtvdf*avdfl + awtidf*aidfl
            endif
         endif

         ! if no incoming shortwave, set albedos to 1
         netsw = swvdr + swidr + swvdf + swidf
         if (netsw <= puny) then ! sun above horizon
            alvdr = c1
            alvdf = c1
            alidr = c1
            alidf = c1
         endif

      if (l_print_point .and. netsw > puny) then

         write(warnstr,*) subname, ' printing point'
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' coszen = ', &
                            coszen
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' swvdr  swvdf = ', &
                            swvdr,swvdf
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' swidr  swidf = ', &
                            swidr,swidf
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' aice = ', &
                            aice
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' hs = ', &
                            hs
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' hp = ', &
                            hp
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' fs = ', &
                            fs
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' fi = ', &
                            fi
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' fp = ', &
                            fp
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' hi = ', &
                            hi
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' alvdr  alvdf = ', &
                            alvdr,alvdf
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' alidr  alidf = ', &
                            alidr,alidf
         call icepack_warnings_add(warnstr)
         write(warnstr,*) subname, ' fswsfc fswint fswthru = ', &
                            fswsfc,fswint,fswthru
         call icepack_warnings_add(warnstr)
         swdn  = swvdr+swvdf+swidr+swidf
         swab  = fswsfc+fswint+fswthru
         swalb = (1.-swab/(swdn+.0001))
         write(warnstr,*) subname, ' swdn swab swalb = ',swdn,swab,swalb
         do k = 1, nslyr
            write(warnstr,*) subname, ' snow layer k    = ', k, &
                             ' rhosnw = ', &
                               rhosnw(k), &
                             ' rsnw = ', &
                               rsnw(k)
            call icepack_warnings_add(warnstr)
         enddo
         do k = 1, nslyr
            write(warnstr,*) subname, ' snow layer k    = ', k, &
                             ' Sswabs(k)       = ', Sswabs(k)
            call icepack_warnings_add(warnstr)
         enddo
         do k = 1, nilyr
            write(warnstr,*) subname, ' sea ice layer k = ', k, &
                             ' Iswabs(k)       = ', Iswabs(k)
            call icepack_warnings_add(warnstr)
         enddo

      endif  ! l_print_point .and. coszen > .01

      end subroutine shortwave_dEdd

!=======================================================================
!
! Evaluate snow/ice/ponded ice inherent optical properties (IOPs), and
! then calculate the multiple scattering solution by calling solution_dEdd.
!
! author:  Bruce P. Briegleb, NCAR
!   2013:  E Hunke merged with NCAR version
!   2022:  E Hunke, T Craig moved data (now module data)

      subroutine compute_dEdd_3bd(                           &
                      klev,   klevp,   zbio,   fnidr,  coszen,  &
                      swvdr,  swvdf,   swidr,  swidf,  srftyp,  &
                      hs,     rhosnw,  rsnw,   hi,     hp,      &
                      fi,     aero_mp, alvdr,  alvdf,           &
                      alidr,  alidf,   fswsfc, fswint, fswthru, &
                      fswthru_vdr,     fswthru_vdf,             &
                      fswthru_idr,     fswthru_idf,             &
                      fswthru_uvrdr,   fswthru_uvrdf,           &
                      fswthru_pardr,   fswthru_pardf,           &
                      Sswabs, Iswabs,  fswpenl,                 &
                      swuvrdr,swuvrdf, swpardr,swpardf)

      integer (kind=int_kind), intent(in) :: &
         klev  , & ! number of radiation layers - 1
         klevp     ! number of radiation interfaces - 1
                   ! (0 layer is included also)

      real (kind=dbl_kind), intent(in) :: &
         fnidr , & ! fraction of direct to total down flux in nir
         coszen, & ! cosine solar zenith angle
         swvdr , & ! shortwave down at surface, visible, direct  (W/m^2)
         swvdf , & ! shortwave down at surface, visible, diffuse (W/m^2)
         swidr , & ! shortwave down at surface, near IR, direct  (W/m^2)
         swidf     ! shortwave down at surface, near IR, diffuse (W/m^2)

      integer (kind=int_kind), intent(in) :: &
         srftyp    ! surface type over ice: (0=air, 1=snow, 2=pond)

      real (kind=dbl_kind), intent(in) :: &
         hs        ! snow thickness (m)

      real (kind=dbl_kind), dimension (:), intent(in) :: &
         rhosnw, & ! snow density in snow layer (kg/m3)
         rsnw  , & ! snow grain radius in snow layer (m)
         zbio  , & ! zaerosol + chla shortwave tracers kg/m^3
         aero_mp   ! aerosol mass path in kg/m2

      real (kind=dbl_kind), intent(in) :: &
         hi    , & ! ice thickness (m)
         hp    , & ! pond depth (m)
         fi        ! snow/bare ice fractional coverage (0 to 1)

      real (kind=dbl_kind), intent(inout) :: &
         alvdr , & ! visible, direct, albedo (fraction)
         alvdf , & ! visible, diffuse, albedo (fraction)
         alidr , & ! near-ir, direct, albedo (fraction)
         alidf , & ! near-ir, diffuse, albedo (fraction)
         fswsfc, & ! SW absorbed at snow/bare ice/pondedi ice surface (W m-2)
         fswint, & ! SW interior absorption (below surface, above ocean,W m-2)
         fswthru   ! SW through snow/bare ice/ponded ice into ocean (W m-2)

      real (kind=dbl_kind), intent(inout) :: &
         fswthru_vdr, & ! vis dir SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_vdf, & ! vis dif SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_idr, & ! nir dir SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_idf, & ! nir dif SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_uvrdr,&! vis uvr dir sw radiation through ice bot (GEOS) (W/m**2)
         fswthru_uvrdf,&! vis uvr dif sw radiation through ice bot (GEOS) (W/m**2)
         fswthru_pardr,&! vis par dir sw radiation through ice bot (GEOS) (W/m**2)
         fswthru_pardf  ! vis par dif sw radiation through ice bot (GEOS) (W/m**2)

      real (kind=dbl_kind), dimension (:), intent(inout) :: &
         fswpenl, & ! visible SW entering ice layers (W m-2)
         Sswabs , & ! SW absorbed in snow layer (W m-2)
         Iswabs     ! SW absorbed in ice layer (W m-2)

      real (kind=dbl_kind), intent(in), optional :: &
         swuvrdr, & ! sw down, vis uvr dir (W/m^2)
         swuvrdf, & ! sw down, vis uvr dif (W/m^2)
         swpardr, & ! sw down, vis par dir (W/m^2)
         swpardf    ! sw down, vis par dif (W/m^2)

!-----------------------------------------------------------------------
!
! Set up optical property profiles, based on snow, sea ice and ponded
! ice IOPs from:
!
! Briegleb, B. P., and B. Light (2007): A Delta-Eddington Multiple
!    Scattering Parameterization for Solar Radiation in the Sea Ice
!    Component of the Community Climate System Model, NCAR Technical
!    Note  NCAR/TN-472+STR  February 2007
!
! Computes column Delta-Eddington radiation solution for specific
! surface type: either snow over sea ice, bare sea ice, or ponded sea ice.
!
! Divides solar spectrum into 3 intervals: 0.2-0.7, 0.7-1.19, and
! 1.19-5.0 micro-meters. The latter two are added (using an assumed
! partition of incident shortwave in the 0.7-5.0 micro-meter band between
! the 0.7-1.19 and 1.19-5.0 micro-meter band) to give the final output
! of 0.2-0.7 visible and 0.7-5.0 near-infrared albedos and fluxes.
!
! Specifies vertical layer optical properties based on input snow depth,
! density and grain radius, along with ice and pond depths, then computes
! layer by layer Delta-Eddington reflectivity, transmissivity and combines
! layers (done by calling routine solution_dEdd). Finally, surface albedos
! and internal fluxes/flux divergences are evaluated.
!
!  Description of the level and layer index conventions. This is
!  for the standard case of one snow layer and four sea ice layers.
!
!  Please read the following; otherwise, there is 99.9% chance you
!  will be confused about indices at some point in time........ :)
!
!  CICE4.0 snow treatment has one snow layer above the sea ice. This
!  snow layer has finite heat capacity, so that surface absorption must
!  be distinguished from internal. The Delta-Eddington solar radiation
!  thus adds extra surface scattering layers to both snow and sea ice.
!  Note that in the following, we assume a fixed vertical layer structure
!  for the radiation calculation. In other words, we always have the
!  structure shown below for one snow and four sea ice layers, but for
!  ponded ice the pond fills "snow" layer 1 over the sea ice, and for
!  bare sea ice the top layers over sea ice are treated as transparent air.
!
!  SSL = surface scattering layer for either snow or sea ice
!  DL  = drained layer for sea ice immediately under sea ice SSL
!  INT = interior layers for sea ice below the drained layer.
!
!  Notice that the radiation level starts with 0 at the top. Thus,
!  the total number radiation layers is klev+1, where klev is the
!  sum of nslyr, the number of CCSM snow layers, and nilyr, the
!  number of CCSM sea ice layers, plus the sea ice SSL:
!  klev = 1 + nslyr + nilyr
!
!  For the standard case illustrated below, nslyr=1, nilyr=4,
!  and klev=6, with the number of layer interfaces klevp=klev+1.
!  Layer interfaces are the surfaces on which reflectivities,
!  transmissivities and fluxes are evaluated.
!
!  CCSM3 Sea Ice Model            Delta-Eddington Solar Radiation
!                                     Layers and Interfaces
!                             Layer Index             Interface Index
!    ---------------------            ---------------------  0
!                                  0  \\\   snow SSL    \\\
!       snow layer 1                  ---------------------  1
!                                  1    rest of snow layer
!    +++++++++++++++++++++            +++++++++++++++++++++  2
!                                  2  \\\ sea ice SSL   \\\
!      sea ice layer 1                ---------------------  3
!                                  3      sea ice  DL
!    ---------------------            ---------------------  4
!
!      sea ice layer 2             4      sea ice INT
!
!    ---------------------            ---------------------  5
!
!      sea ice layer 3             5      sea ice INT
!
!    ---------------------            ---------------------  6
!
!      sea ice layer 4             6      sea ice INT
!
!    ---------------------            ---------------------  7
!
! When snow lies over sea ice, the radiation absorbed in the
! snow SSL is used for surface heating, and that in the rest
! of the snow layer for its internal heating. For sea ice in
! this case, all of the radiant heat absorbed in both the
! sea ice SSL and the DL are used for sea ice layer 1 heating.
!
! When pond lies over sea ice, and for bare sea ice, all of the
! radiant heat absorbed within and above the sea ice SSL is used
! for surface heating, and that absorbed in the sea ice DL is
! used for sea ice layer 1 heating.
!
! Basically, vertical profiles of the layer extinction optical depth (tau),
! single scattering albedo (w0) and asymmetry parameter (g) are required over
! the klev+1 layers, where klev+1 = 2 + nslyr + nilyr. All of the surface type
! information and snow/ice iop properties are evaulated in this routine, so
! the tau,w0,g profiles can be passed to solution_dEdd for multiple scattering
! evaluation. Snow, bare ice and ponded ice iops are contained in data arrays
! in this routine.
!
!-----------------------------------------------------------------------

      ! local variables

      integer (kind=int_kind) :: &
         k       , & ! level index
         ns      , & ! spectral index
         nr      , & ! index for grain radius tables
         ki      , & ! index for internal absorption
         km      , & ! k starting index for snow, sea ice internal absorption
         kp      , & ! k+1 or k+2 index for snow, sea ice internal absorption
         ksrf    , & ! level index for surface absorption
         ksnow   , & ! level index for snow density and grain size
         kii         ! level starting index for sea ice (nslyr+1)

      real (kind=dbl_kind) :: &
         avdr    , & ! visible albedo, direct   (fraction)
         avdf    , & ! visible albedo, diffuse  (fraction)
         aidr    , & ! near-ir albedo, direct   (fraction)
         aidf        ! near-ir albedo, diffuse  (fraction)

      real (kind=dbl_kind) :: &
         fsfc    , & ! shortwave absorbed at snow/bare ice/ponded ice surface (W m-2)
         fint    , & ! shortwave absorbed in interior (W m-2)
         fthru   , & ! shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruvdr, & ! vis dir shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruvdf, & ! vis dif shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruidr, & ! nir dir shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruidf, & ! nir dif shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruuvrdr,&! vis uvr dir shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruuvrdf,&! vis uvr dif shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthrupardr,&! vis par dir shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthrupardf  ! vis par dif shortwave through snow/bare ice/ponded ice to ocean (W/m^2)

      real (kind=dbl_kind), dimension(nslyr) :: &
         Sabs        ! shortwave absorbed in snow layer (W m-2)

      real (kind=dbl_kind), dimension(nilyr) :: &
         Iabs        ! shortwave absorbed in ice layer (W m-2)

      real (kind=dbl_kind), dimension(nilyr+1) :: &
         fthrul      ! shortwave through to ice layers (W m-2)

      real (kind=dbl_kind), dimension (nspint_3bd) :: &
         wghtns      ! spectral weights

      real (kind=dbl_kind), parameter :: &
         cp67 = 0.67_dbl_kind, & ! nir band weight parameter
         cp78 = 0.78_dbl_kind, & ! nir band weight parameter
         cp01 = 0.01_dbl_kind    ! for ocean visible albedo

      real (kind=dbl_kind), dimension (0:klev) :: &
         tau     , & ! layer extinction optical depth
         w0      , & ! layer single scattering albedo
         g           ! layer asymmetry parameter

      ! following arrays are defined at model interfaces; 0 is the top of the
      ! layer above the sea ice; klevp is the sea ice/ocean interface.
      real (kind=dbl_kind), dimension (0:klevp) :: &
         trndir  , & ! solar beam down transmission from top
         trntdr  , & ! total transmission to direct beam for layers above
         trndif  , & ! diffuse transmission to diffuse beam for layers above
         rupdir  , & ! reflectivity to direct radiation for layers below
         rupdif  , & ! reflectivity to diffuse radiation for layers below
         rdndif      ! reflectivity to diffuse radiation for layers above

      real (kind=dbl_kind), dimension (0:klevp) :: &
         dfdir   , & ! down-up flux at interface due to direct beam at top surface
         dfdif       ! down-up flux at interface due to diffuse beam at top surface

      real (kind=dbl_kind) :: &
         refk    , & ! interface k multiple scattering term
         delr    , & ! snow grain radius interpolation parameter
      ! inherent optical properties (iop) for snow
         Qs      , & ! Snow extinction efficiency
         ks      , & ! Snow mass extinction coefficient (1/m)
         ws      , & ! Snow single scattering albedo
         gs          ! Snow asymmetry parameter

      real (kind=dbl_kind), dimension(nslyr) :: &
         frsnw       ! snow grain radius in snow layer * adjustment factor (m)

      ! ice and ponded ice IOPs, allowing for tuning
      ! modifications of the above "_mn" value
      real (kind=dbl_kind), dimension (nspint_3bd) :: &
         ki_ssl  , & ! Surface-scattering-layer ice extinction coefficient (/m)
         wi_ssl  , & ! Surface-scattering-layer ice single scattering albedo
         gi_ssl  , & ! Surface-scattering-layer ice asymmetry parameter
         ki_dl   , & ! Drained-layer ice extinction coefficient (/m)
         wi_dl   , & ! Drained-layer ice single scattering albedo
         gi_dl   , & ! Drained-layer ice asymmetry parameter
         ki_int  , & ! Interior-layer ice extinction coefficient (/m)
         wi_int  , & ! Interior-layer ice single scattering albedo
         gi_int  , & ! Interior-layer ice asymmetry parameter
         ki_p_ssl, & ! Ice under pond srf scat layer extinction coefficient (/m)
         wi_p_ssl, & ! Ice under pond srf scat layer single scattering albedo
         gi_p_ssl, & ! Ice under pond srf scat layer asymmetry parameter
         ki_p_int, & ! Ice under pond extinction coefficient (/m)
         wi_p_int, & ! Ice under pond single scattering albedo
         gi_p_int    ! Ice under pond asymmetry parameter

      real (kind=dbl_kind), dimension(0:klev) :: &
         dzk         ! layer thickness

      real (kind=dbl_kind) :: &
         dz      , & ! snow, sea ice or pond water layer thickness
         dz_ssl  , & ! snow or sea ice surface scattering layer thickness
         fs          ! scaling factor to reduce (nilyr<4) or increase (nilyr>4) DL
                     ! extinction coefficient to maintain DL optical depth constant
                     ! with changing number of sea ice layers, to approximately
                     ! conserve computed albedo for constant physical depth of sea
                     ! ice when the number of sea ice layers vary

      real (kind=dbl_kind) :: &
         sig     , & ! scattering coefficient for tuning
         kabs    , & ! absorption coefficient for tuning
         sigp        ! modified scattering coefficient for tuning

      real (kind=dbl_kind), dimension(nspint_3bd, 0:klev) :: &
         kabs_chl, & ! absorption coefficient for chlorophyll (/m)
         tzaer   , & ! total aerosol extinction optical depth
         wzaer   , & ! total aerosol single scatter albedo
         gzaer       ! total aerosol asymmetry parameter

      real (kind=dbl_kind) :: &
         albodr  , & ! spectral ocean albedo to direct rad
         albodf      ! spectral ocean albedo to diffuse rad

      ! for melt pond transition to bare sea ice for small pond depths
      real (kind=dbl_kind) :: &
         sig_i   , & ! ice scattering coefficient (/m)
         sig_p   , & ! pond scattering coefficient (/m)
         kext        ! weighted extinction coefficient (/m)

      ! aerosol optical properties from Mark Flanner, 26 June 2008
      ! order assumed: hydrophobic black carbon, hydrophilic black carbon,
      ! four dust aerosols by particle size range:
      ! dust1(.05-0.5 micron), dust2(0.5-1.25 micron),
      ! dust3(1.25-2.5 micron), dust4(2.5-5.0 micron)
      ! spectral bands same as snow/sea ice: (0.3-0.7 micron, 0.7-1.19 micron
      ! and 1.19-5.0 micron in wavelength)

      integer (kind=int_kind) :: &
         na , n                    ! aerosol index

      real (kind=dbl_kind) :: &
         taer                  , & ! total aerosol extinction optical depth
         waer                  , & ! total aerosol single scatter albedo
         gaer                  , & ! total aerosol asymmetry parameter
         swdr                  , & ! shortwave down at surface, direct  (W/m^2)
         swdf                  , & ! shortwave down at surface, diffuse (W/m^2)
         rnilyr                , & ! 1/real(nilyr)
         rnslyr                , & ! 1/real(nslyr)
         rns                   , & ! real(ns)
         tmp_0, tmp_ks, tmp_kl     ! temporary variables

      integer(kind=int_kind), dimension(0:klev) :: &
         k_bcini               , & ! index
         k_bcins               , & ! = 2 hardwired
         k_bcexs                   ! = 2 hardwired

      real(kind=dbl_kind)::  &
         tmp_gs, tmp1              ! temporary variables

      real (kind=dbl_kind), parameter :: &
         fr_max = 1.00_dbl_kind, & ! snow grain adjustment factor max
         fr_min = 0.80_dbl_kind, & ! snow grain adjustment factor min
      ! tuning parameters
      ! ice and pond scat coeff fractional change for +- one-sigma in albedo
         fp_ice = 0.15_dbl_kind, & ! ice fraction of scat coeff for + stn dev in alb
         fm_ice = 0.15_dbl_kind, & ! ice fraction of scat coeff for - stn dev in alb
         fp_pnd = 2.00_dbl_kind, & ! ponded ice fraction of scat coeff for + stn dev in alb
         fm_pnd = 0.50_dbl_kind    ! ponded ice fraction of scat coeff for - stn dev in alb

      real (kind=dbl_kind),  parameter :: &   ! chla-specific absorption coefficient
         kchl_tab = p01 ! 0.0023-0.0029 Perovich 1993, also 0.0067 m^2 (mg Chl)^-1
                        ! found values of 0.006 to 0.023 m^2/ mg  (676 nm)  Neukermans 2014
                        ! and averages over the 300-700nm of 0.0075 m^2/mg in ice Fritsen (2011)
                        ! at 440nm values as high as 0.2 m^2/mg in under ice bloom (Balch 2014)
                        ! Grenfell 1991 uses 0.004 (m^2/mg) which is (0.0078 * spectral weighting)
                        ! chlorophyll mass extinction cross section (m^2/mg chla)

      character(len=*),parameter :: subname='(compute_dEdd_3bd)'

!-----------------------------------------------------------------------
! Initialize and tune bare ice/ponded ice iops

      k_bcini(:) = 0
      k_bcins(:) = 0
      k_bcexs(:) = 0

      rnilyr = c1/real(nilyr,kind=dbl_kind)
      rnslyr = c1/real(nslyr,kind=dbl_kind)
      kii = nslyr + 1

      ! initialize albedos and fluxes to 0
      fthrul        = c0
      Iabs          = c0
      kabs_chl(:,:) = c0
      tzaer   (:,:) = c0
      wzaer   (:,:) = c0
      gzaer   (:,:) = c0

      avdr     = c0
      avdf     = c0
      aidr     = c0
      aidf     = c0
      fsfc     = c0
      fint     = c0
      fthru    = c0
      fthruvdr = c0
      fthruvdf = c0
      fthruidr = c0
      fthruidf = c0
      fthruuvrdr = c0
      fthruuvrdf = c0
      fthrupardr = c0
      fthrupardf = c0

      ! spectral weights 2 (0.7-1.19 micro-meters) and 3 (1.19-5.0 micro-meters)
      ! are chosen based on 1D calculations using ratio of direct to total
      ! near-infrared solar (0.7-5.0 micro-meter) which indicates clear/cloudy
      ! conditions: more cloud, the less 1.19-5.0 relative to the
      ! 0.7-1.19 micro-meter due to cloud absorption.
      wghtns(1) = c1
      wghtns(2) = cp67 + (cp78-cp67)*(c1-fnidr)
      wghtns(3) = c1 - wghtns(2)

      ! find snow grain adjustment factor, dependent upon clear/overcast sky
      ! estimate. comparisons with SNICAR show better agreement with DE when
      ! this factor is included (clear sky near 1 and overcast near 0.8 give
      ! best agreement).  Multiply by rnsw here for efficiency.
      do k = 1, nslyr
         frsnw(k) = (fr_max*fnidr + fr_min*(c1-fnidr))*rsnw(k)
         Sabs(k) = c0
      enddo

      ! layer thicknesses
      ! snow
      dz = hs*rnslyr
      ! for small enough snow thickness, ssl thickness half of top snow layer
!ech: note this is highly resolution dependent!
      dzk(0) = min(hs_ssl, dz/c2)
      dzk(1) = dz - dzk(0)
      if (nslyr > 1) then
         do k = 2, nslyr
            dzk(k) = dz
         enddo
      endif

      ! ice
      dz = hi*rnilyr
      ! empirical reduction in sea ice ssl thickness for ice thinner than 1.5m;
      ! factor of 30 gives best albedo comparison with limited observations
      dz_ssl = hi_ssl
!ech: note hardwired parameters
!         if( hi < 1.5_dbl_kind ) dz_ssl = hi/30._dbl_kind
      dz_ssl = min(hi_ssl, hi/30._dbl_kind)
      ! set sea ice ssl thickness to half top layer if sea ice thin enough
!ech: note this is highly resolution dependent!
      dz_ssl = min(dz_ssl, dz/c2)

      dzk(kii)   = dz_ssl
      dzk(kii+1) = dz - dz_ssl
      if (kii+2 <= klev) then
         do k = kii+2, klev
            dzk(k) = dz
         enddo
      endif

      ! adjust sea ice iops with tuning parameters; tune only the
      ! scattering coefficient by factors of R_ice, R_pnd, where
      ! R values of +1 correspond approximately to +1 sigma changes in albedo, and
      ! R values of -1 correspond approximately to -1 sigma changes in albedo
      ! Note: the albedo change becomes non-linear for R values > +1 or < -1
      if( R_ice >= c0 ) then
        do ns = 1, nspint_3bd
          sigp       = ki_ssl_mn_3bd(ns)*wi_ssl_mn_3bd(ns)*(c1+fp_ice*R_ice)
          ki_ssl(ns) = sigp+ki_ssl_mn_3bd(ns)*(c1-wi_ssl_mn_3bd(ns))
          wi_ssl(ns) = sigp/ki_ssl(ns)
          gi_ssl(ns) = gi_ssl_mn_3bd(ns)

          sigp       = ki_dl_mn_3bd(ns)*wi_dl_mn_3bd(ns)*(c1+fp_ice*R_ice)
          ki_dl(ns)  = sigp+ki_dl_mn_3bd(ns)*(c1-wi_dl_mn_3bd(ns))
          wi_dl(ns)  = sigp/ki_dl(ns)
          gi_dl(ns)  = gi_dl_mn_3bd(ns)

          sigp       = ki_int_mn_3bd(ns)*wi_int_mn_3bd(ns)*(c1+fp_ice*R_ice)
          ki_int(ns) = sigp+ki_int_mn_3bd(ns)*(c1-wi_int_mn_3bd(ns))
          wi_int(ns) = sigp/ki_int(ns)
          gi_int(ns) = gi_int_mn_3bd(ns)
        enddo
      else !if( R_ice < c0 ) then
        do ns = 1, nspint_3bd
          sigp       = ki_ssl_mn_3bd(ns)*wi_ssl_mn_3bd(ns)*(c1+fm_ice*R_ice)
          sigp       = max(sigp, c0)
          ki_ssl(ns) = sigp+ki_ssl_mn_3bd(ns)*(c1-wi_ssl_mn_3bd(ns))
          wi_ssl(ns) = sigp/ki_ssl(ns)
          gi_ssl(ns) = gi_ssl_mn_3bd(ns)

          sigp       = ki_dl_mn_3bd(ns)*wi_dl_mn_3bd(ns)*(c1+fm_ice*R_ice)
          sigp       = max(sigp, c0)
          ki_dl(ns)  = sigp+ki_dl_mn_3bd(ns)*(c1-wi_dl_mn_3bd(ns))
          wi_dl(ns)  = sigp/ki_dl(ns)
          gi_dl(ns)  = gi_dl_mn_3bd(ns)

          sigp       = ki_int_mn_3bd(ns)*wi_int_mn_3bd(ns)*(c1+fm_ice*R_ice)
          sigp       = max(sigp, c0)
          ki_int(ns) = sigp+ki_int_mn_3bd(ns)*(c1-wi_int_mn_3bd(ns))
          wi_int(ns) = sigp/ki_int(ns)
          gi_int(ns) = gi_int_mn_3bd(ns)
        enddo
      endif          ! adjust ice iops

      ! adjust ponded ice iops with tuning parameters
      if( R_pnd >= c0 ) then
        do ns = 1, nspint_3bd
          sigp         = ki_p_ssl_mn(ns)*wi_p_ssl_mn(ns)*(c1+fp_pnd*R_pnd)
          ki_p_ssl(ns) = sigp+ki_p_ssl_mn(ns)*(c1-wi_p_ssl_mn(ns))
          wi_p_ssl(ns) = sigp/ki_p_ssl(ns)
          gi_p_ssl(ns) = gi_p_ssl_mn(ns)

          sigp         = ki_p_int_mn(ns)*wi_p_int_mn(ns)*(c1+fp_pnd*R_pnd)
          ki_p_int(ns) = sigp+ki_p_int_mn(ns)*(c1-wi_p_int_mn(ns))
          wi_p_int(ns) = sigp/ki_p_int(ns)
          gi_p_int(ns) = gi_p_int_mn(ns)
        enddo
      else !if( R_pnd < c0 ) then
        do ns = 1, nspint_3bd
          sigp         = ki_p_ssl_mn(ns)*wi_p_ssl_mn(ns)*(c1+fm_pnd*R_pnd)
          sigp         = max(sigp, c0)
          ki_p_ssl(ns) = sigp+ki_p_ssl_mn(ns)*(c1-wi_p_ssl_mn(ns))
          wi_p_ssl(ns) = sigp/ki_p_ssl(ns)
          gi_p_ssl(ns) = gi_p_ssl_mn(ns)

          sigp         = ki_p_int_mn(ns)*wi_p_int_mn(ns)*(c1+fm_pnd*R_pnd)
          sigp         = max(sigp, c0)
          ki_p_int(ns) = sigp+ki_p_int_mn(ns)*(c1-wi_p_int_mn(ns))
          wi_p_int(ns) = sigp/ki_p_int(ns)
          gi_p_int(ns) = gi_p_int_mn(ns)
        enddo
      endif            ! adjust ponded ice iops

      ! use srftyp to determine interface index of surface absorption
      if (srftyp == 1) then
         ! snow covered sea ice
         ksrf = 1
      else
         ! bare sea ice or ponded ice
         ksrf = nslyr + 2
      endif

      if (tr_bgc_N .and. dEdd_algae) then ! compute kabs_chl for chlorophyll
          do k = 0, klev
             kabs_chl(1,k) = kchl_tab*zbio(nlt_chl_sw+k)
          enddo
      else
            k = klev
            kabs_chl(1,k) = kalg*(0.50_dbl_kind/dzk(k))
      endif        ! kabs_chl

      ! aerosols
      if (modal_aero) then
         do k = 0, klev
            if (k < nslyr+1) then ! define indices for snow layer
               ! use top rsnw, rhosnw for snow ssl and rest of top layer
               ! Cheng: note that aerosol IOPs are related to snow grain radius.
               ! CICE adjusted snow grain radius rsnw to frsnw in the original 3-band
               ! scheme, while for SNICAR the snow grain radius is used directly.
               ksnow = max(k,1)
               tmp_gs = frsnw(ksnow)

               ! grain size index
               if (tmp_gs < 125._dbl_kind) then
                  tmp1 = tmp_gs/50._dbl_kind
                  k_bcini(k) = nint(tmp1)
               elseif (tmp_gs < 175._dbl_kind) then
                  k_bcini(k) = 2
               else
                  tmp1 = (tmp_gs/250._dbl_kind) + c2
                  k_bcini(k) = nint(tmp1)
               endif
            else                  ! use the largest snow grain size for ice
               k_bcini(k) = 8
            endif
            ! Set index corresponding to BC effective radius.  Here,
            ! asssume constant BC effective radius of 100nm
            ! (corresponding to index 2)
            k_bcins(k) = 2 ! hardwired
            k_bcexs(k) = 2

            ! check bounds
            if (k_bcini(k) < 1)  k_bcini(k) = 1
            if (k_bcini(k) > 8)  k_bcini(k) = 8
!            if (k_bcins(k) < 1)  k_bcins(k) = 1   ! hardwired
!            if (k_bcins(k) > 10) k_bcins(k) = 10
!            if (k_bcexs(k) < 1)  k_bcexs(k) = 1
!            if (k_bcexs(k) > 10) k_bcexs(k) = 10
         enddo   ! k

         if (tr_zaero .and. dEdd_algae) then ! compute kzaero for chlorophyll
         do n = 1, n_zaero
            if (n == 1) then ! interstitial BC
               do k = 0, klev
               do ns = 1, nspint_3bd   ! not weighted by aice
                  tzaer(ns,k) = tzaer      (ns,k) &
                              + kaer_bc_3bd(ns,k_bcexs(k)) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  wzaer(ns,k) = wzaer      (ns,k) &
                              + kaer_bc_3bd(ns,k_bcexs(k)) &
                              * waer_bc_3bd(ns,k_bcexs(k)) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  gzaer(ns,k) = gzaer      (ns,k) &
                              + kaer_bc_3bd(ns,k_bcexs(k)) &
                              * waer_bc_3bd(ns,k_bcexs(k)) &
                              * gaer_bc_3bd(ns,k_bcexs(k)) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
               enddo
               enddo
            elseif (n==2) then ! within-ice BC
               do k = 0, klev
               do ns = 1, nspint_3bd
                  tzaer(ns,k) = tzaer      (ns,k) &
                              + kaer_bc_3bd(ns,k_bcins(k)) &
                              *   bcenh_3bd(ns,k_bcins(k),k_bcini(k)) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  wzaer(ns,k) = wzaer      (ns,k) &
                              + kaer_bc_3bd(ns,k_bcins(k)) &
                              * waer_bc_3bd(ns,k_bcins(k)) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  gzaer(ns,k) = gzaer      (ns,k) &
                              + kaer_bc_3bd(ns,k_bcins(k)) &
                              * waer_bc_3bd(ns,k_bcins(k)) &
                              * gaer_bc_3bd(ns,k_bcins(k)) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
               enddo
               enddo
            else                ! dust
               do k = 0, klev
               do ns = 1,nspint_3bd   ! not weighted by aice
                  tzaer(ns,k) = tzaer   (ns,k) &
                              + kaer_3bd(ns,n) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  wzaer(ns,k) = wzaer   (ns,k) &
                              + kaer_3bd(ns,n) &
                              * waer_3bd(ns,n) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  gzaer(ns,k) = gzaer   (ns,k) &
                              + kaer_3bd(ns,n) &
                              * waer_3bd(ns,n) &
                              * gaer_3bd(ns,n) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
               enddo  ! nspint
               enddo  ! k
            endif     ! n
        enddo         ! n_zaero
        endif         ! tr_zaero and dEdd_algae

      else  ! Bulk aerosol treatment
         if (tr_zaero .and. dEdd_algae) then ! compute kzaero for chlorophyll
         do n = 1, n_zaero         ! multiply by aice?
            do k = 0, klev
               do ns = 1, nspint_3bd   ! not weighted by aice
                  tzaer(ns,k) = tzaer   (ns,k) &
                              + kaer_3bd(ns,n) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  wzaer(ns,k) = wzaer   (ns,k) &
                              + kaer_3bd(ns,n) &
                              * waer_3bd(ns,n) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  gzaer(ns,k) = gzaer   (ns,k) &
                              + kaer_3bd(ns,n) &
                              * waer_3bd(ns,n) &
                              * gaer_3bd(ns,n) &
                              * zbio(nlt_zaero_sw(n)+k) * dzk(k)
               enddo ! nspint
            enddo    ! k
        enddo        ! n
        endif        ! tr_zaero
     endif           ! modal_aero

!-----------------------------------------------------------------------

      ! begin spectral loop
!echmod - split this loop for efficiency, if possible (move conditionals outside of the loop)
      do ns = 1, nspint_3bd

         ! set optical properties of air/snow/pond overlying sea ice
         ! air
         if (srftyp == 0 ) then
            do k=0,nslyr
               tau(k) = c0
               w0(k)  = c0
               g(k)   = c0
            enddo
            ! snow
         elseif (srftyp == 1 ) then
            ! interpolate snow iops using input snow grain radius,
            ! snow density and tabular data

            do k = 0, nslyr
               ! use top rsnw, rhosnw for snow ssl and rest of top layer
               ksnow = max(k,1)
               ! find snow iops using input snow density and snow grain radius:
               if (frsnw(ksnow) < rsnw_tab(1)) then
                  Qs = Qs_tab(ns,1)
                  ws = ws_tab(ns,1)
                  gs = gs_tab(ns,1)
               elseif (frsnw(ksnow) >= rsnw_tab(nmbrad_snw)) then
                  Qs = Qs_tab(ns,nmbrad_snw)
                  ws = ws_tab(ns,nmbrad_snw)
                  gs = gs_tab(ns,nmbrad_snw)
               else
                  call shortwave_search(frsnw(ksnow),rsnw_tab,nr)
                  if (icepack_warnings_aborted(subname)) return
                  delr = (frsnw(ksnow) - rsnw_tab(nr-1)) / &
                         (rsnw_tab(nr) - rsnw_tab(nr-1))
                  Qs   = Qs_tab(ns,nr-1)*(c1-delr) + &
                         Qs_tab(ns,nr  )*    delr
                  ws   = ws_tab(ns,nr-1)*(c1-delr) + &
                         ws_tab(ns,nr  )*    delr
                  gs   = gs_tab(ns,nr-1)*(c1-delr) + &
                         gs_tab(ns,nr  )*    delr
               endif
               ks = Qs*((rhosnw(ksnow)/rhoi)*3._dbl_kind / &
                       (4._dbl_kind*frsnw(ksnow)*1.0e-6_dbl_kind))

               tau(k) = (ks + kabs_chl(ns,k))*dzk(k)
               w0 (k) = ks/(ks + kabs_chl(ns,k)) * ws
               g  (k) = gs
            enddo       ! k

            ! aerosol in snow
            if (tr_zaero .and. dEdd_algae) then
               do k = 0,nslyr
                  g(k)   = (g(k)*w0(k)*tau(k) + gzaer(ns,k)) / &
                                (w0(k)*tau(k) + wzaer(ns,k))
                  w0(k)  =      (w0(k)*tau(k) + wzaer(ns,k)) / &
                                      (tau(k) + tzaer(ns,k))
                  tau(k) = tau(k) + tzaer(ns,k)
               enddo
            elseif (tr_aero) then
               k = 0  ! snow SSL
               taer = c0
               waer = c0
               gaer = c0

               do na = 1, 4*n_aero, 4
               if (modal_aero) then
                  if (na == 1) then      ! interstitial BC
                     taer = taer + aero_mp(na)*kaer_bc_3bd(ns,k_bcexs(k))
                     waer = waer + aero_mp(na)*kaer_bc_3bd(ns,k_bcexs(k)) &
                                              *waer_bc_3bd(ns,k_bcexs(k))
                     gaer = gaer + aero_mp(na)*kaer_bc_3bd(ns,k_bcexs(k)) &
                                              *waer_bc_3bd(ns,k_bcexs(k)) &
                                              *gaer_bc_3bd(ns,k_bcexs(k))
                  elseif (na == 5) then ! within-ice BC
                     taer = taer + aero_mp(na)*kaer_bc_3bd(ns,k_bcins(k)) &
                                              *  bcenh_3bd(ns,k_bcins(k),k_bcini(k))
                     waer = waer + aero_mp(na)*kaer_bc_3bd(ns,k_bcins(k)) &
                                              *waer_bc_3bd(ns,k_bcins(k))
                     gaer = gaer + aero_mp(na)*kaer_bc_3bd(ns,k_bcins(k)) &
                                              *waer_bc_3bd(ns,k_bcins(k)) &
                                              *gaer_bc_3bd(ns,k_bcins(k))
                  else                  ! other species (dust)
                     taer = taer + aero_mp(na)*kaer_3bd(ns,(1+(na-1)/4))
                     waer = waer + aero_mp(na)*kaer_3bd(ns,(1+(na-1)/4)) &
                                              *waer_3bd(ns,(1+(na-1)/4))
                     gaer = gaer + aero_mp(na)*kaer_3bd(ns,(1+(na-1)/4)) &
                                              *waer_3bd(ns,(1+(na-1)/4)) &
                                              *gaer_3bd(ns,(1+(na-1)/4))
                  endif
               else
                  taer = taer + aero_mp(na)*kaer_3bd(ns,(1+(na-1)/4))
                  waer = waer + aero_mp(na)*kaer_3bd(ns,(1+(na-1)/4)) &
                                           *waer_3bd(ns,(1+(na-1)/4))
                  gaer = gaer + aero_mp(na)*kaer_3bd(ns,(1+(na-1)/4)) &
                                           *waer_3bd(ns,(1+(na-1)/4)) &
                                           *gaer_3bd(ns,(1+(na-1)/4))
               endif ! modal_aero
               enddo ! na
               g  (k) = (g(k)*w0(k)*tau(k) + gaer) / &
                             (w0(k)*tau(k) + waer)
               w0 (k) =      (w0(k)*tau(k) + waer) / &
                                   (tau(k) + taer)
               tau(k) = tau(k) + taer

               do k = 1, nslyr
                  taer = c0
                  waer = c0
                  gaer = c0
                  do na = 1, 4*n_aero, 4
                  if (modal_aero) then
                     if (na==1) then     ! interstitial BC
                        taer = taer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_3bd(ns,k_bcexs(k))
                        waer = waer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_3bd(ns,k_bcexs(k)) &
                             * waer_bc_3bd(ns,k_bcexs(k))
                        gaer = gaer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_3bd(ns,k_bcexs(k)) &
                             * waer_bc_3bd(ns,k_bcexs(k)) &
                             * gaer_bc_3bd(ns,k_bcexs(k))
                     elseif (na==5) then ! within-ice BC
                        taer = taer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_3bd(ns,k_bcins(k)) &
                             *   bcenh_3bd(ns,k_bcins(k),k_bcini(k))
                        waer = waer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_3bd(ns,k_bcins(k)) &
                             * waer_bc_3bd(ns,k_bcins(k))
                        gaer = gaer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_3bd(ns,k_bcins(k)) &
                             * waer_bc_3bd(ns,k_bcins(k)) &
                             * gaer_bc_3bd(ns,k_bcins(k))
                     else                ! other species (dust)
                        taer = taer + (aero_mp(na+1)*rnslyr) &
                             * kaer_3bd(ns,(1+(na-1)/4))
                        waer = waer + (aero_mp(na+1)*rnslyr) &
                             * kaer_3bd(ns,(1+(na-1)/4)) &
                             * waer_3bd(ns,(1+(na-1)/4))
                        gaer = gaer + (aero_mp(na+1)*rnslyr) &
                             * kaer_3bd(ns,(1+(na-1)/4)) &
                             * waer_3bd(ns,(1+(na-1)/4)) &
                             * gaer_3bd(ns,(1+(na-1)/4))
                     endif   ! na
                  else
                     taer = taer + (aero_mp(na+1)*rnslyr) &
                          * kaer_3bd(ns,(1+(na-1)/4))
                     waer = waer + (aero_mp(na+1)*rnslyr) &
                          * kaer_3bd(ns,(1+(na-1)/4)) &
                          * waer_3bd(ns,(1+(na-1)/4))
                     gaer = gaer + (aero_mp(na+1)*rnslyr) &
                          * kaer_3bd(ns,(1+(na-1)/4)) &
                          * waer_3bd(ns,(1+(na-1)/4)) &
                          * gaer_3bd(ns,(1+(na-1)/4))
                  endif       ! modal_aero
                  enddo       ! na
                  g  (k) = (g(k)*w0(k)*tau(k) + gaer) / &
                                (w0(k)*tau(k) + waer)
                  w0 (k) =      (w0(k)*tau(k) + waer) / &
                                      (tau(k) + taer)
                  tau(k) = tau(k) + taer
               enddo       ! k
            endif     ! tr_aero

         else ! srftyp == 2
            ! pond water layers evenly spaced
            dz = hp/(real(nslyr,kind=dbl_kind)+c1)
            do k=0,nslyr
               tau(k) = kw(ns)*dz
               w0 (k) = ww(ns)
               g  (k) = gw(ns)
               ! no aerosol in pond
            enddo       ! k
         endif        ! srftyp

         ! set optical properties of sea ice

         ! bare or snow-covered sea ice layers
         if (srftyp <= 1) then
            ! ssl
            k = kii
            tau(k) =             (ki_ssl(ns) + kabs_chl(ns,k)) * dzk(k)
            w0 (k) =  ki_ssl(ns)/(ki_ssl(ns) + kabs_chl(ns,k)) * wi_ssl(ns)
            g  (k) =  gi_ssl(ns)
            ! dl
            k = kii + 1
            ! scale dz for dl relative to 4 even-layer-thickness 1.5m case
            fs = p25*real(nilyr,kind=dbl_kind)
            tau(k) =           (ki_dl(ns) + kabs_chl(ns,k)) * dzk(k) * fs
            w0 (k) = ki_dl(ns)/(ki_dl(ns) + kabs_chl(ns,k)) * wi_dl(ns)
            g  (k) = gi_dl(ns)
            ! int above lowest layer
            if (kii+2 <= klev-1) then
               do k = kii+2, klev-1
                  tau(k) =            (ki_int(ns) + kabs_chl(ns,k)) * dzk(k)
                  w0 (k) = ki_int(ns)/(ki_int(ns) + kabs_chl(ns,k)) * wi_int(ns)
                  g  (k) = gi_int(ns)
               enddo
            endif
            ! lowest layer
            k = klev
            ! add algae to lowest sea ice layer, visible only:
            kabs = ki_int(ns)*(c1-wi_int(ns))
            if (ns == 1) then
               ! total layer absorption optical depth fixed at value
               ! of kalg*0.50m, independent of actual layer thickness
               kabs = kabs + kabs_chl(ns,k)
            endif
            sig    = ki_int(ns) * wi_int(ns)
            tau(k) = (kabs+sig) * dzk(k)
            w0 (k) = sig/(sig+kabs)
            g  (k) = gi_int(ns)
            ! aerosol in sea ice
            if (tr_zaero .and. dEdd_algae) then
               do k = kii, klev
                  g(k)   = (g(k)*w0(k)*tau(k) + gzaer(ns,k)) / &
                                (w0(k)*tau(k) + wzaer(ns,k))
                  w0(k)  =      (w0(k)*tau(k) + wzaer(ns,k)) / &
                                      (tau(k) + tzaer(ns,k))
                  tau(k) = tau(k) + tzaer(ns,k)
               enddo
            elseif (tr_aero) then
               k = kii   ! sea ice SSL
               taer = c0
               waer = c0
               gaer = c0
               do na=1,4*n_aero,4
               if (modal_aero) then
                  if (na==1) then      ! interstitial BC
                     taer = taer + aero_mp(na+2) &
                          * kaer_bc_3bd(ns,k_bcexs(k))
                     waer = waer + aero_mp(na+2) &
                          * kaer_bc_3bd(ns,k_bcexs(k)) &
                          * waer_bc_3bd(ns,k_bcexs(k))
                     gaer = gaer + aero_mp(na+2) &
                          * kaer_bc_3bd(ns,k_bcexs(k)) &
                          * waer_bc_3bd(ns,k_bcexs(k)) &
                          * gaer_bc_3bd(ns,k_bcexs(k))
                  elseif (na==5) then  ! within-ice BC
                     taer = taer + aero_mp(na+2) &
                          * kaer_bc_3bd(ns,k_bcins(k)) &
                          *   bcenh_3bd(ns,k_bcins(k),k_bcini(k))
                     waer = waer + aero_mp(na+2) &
                          * kaer_bc_3bd(ns,k_bcins(k)) &
                          * waer_bc_3bd(ns,k_bcins(k))
                     gaer = gaer + aero_mp(na+2) &
                          * kaer_bc_3bd(ns,k_bcins(k)) &
                          * waer_bc_3bd(ns,k_bcins(k)) &
                          * gaer_bc_3bd(ns,k_bcins(k))
                  else                 ! other species (dust)
                     taer = taer + aero_mp(na+2) &
                          * kaer_3bd(ns,(1+(na-1)/4))
                     waer = waer + aero_mp(na+2) &
                          * kaer_3bd(ns,(1+(na-1)/4)) &
                          * waer_3bd(ns,(1+(na-1)/4))
                     gaer = gaer + aero_mp(na+2) &
                          * kaer_3bd(ns,(1+(na-1)/4)) &
                          * waer_3bd(ns,(1+(na-1)/4)) &
                          * gaer_3bd(ns,(1+(na-1)/4))
                  endif
               else      ! bulk
                  taer = taer + aero_mp(na+2) &
                       * kaer_3bd(ns,(1+(na-1)/4))
                  waer = waer + aero_mp(na+2) &
                       * kaer_3bd(ns,(1+(na-1)/4)) &
                       * waer_3bd(ns,(1+(na-1)/4))
                  gaer = gaer + aero_mp(na+2) &
                       * kaer_3bd(ns,(1+(na-1)/4)) &
                       * waer_3bd(ns,(1+(na-1)/4)) &
                       * gaer_3bd(ns,(1+(na-1)/4))
                endif     ! modal_aero
               enddo      ! na
               g  (k) = (g(k)*w0(k)*tau(k) + gaer) / &
                             (w0(k)*tau(k) + waer)
               w0 (k) =      (w0(k)*tau(k) + waer) / &
                                   (tau(k) + taer)
               tau(k) = tau(k) + taer
               do k = kii+1, klev
                  taer = c0
                  waer = c0
                  gaer = c0
                  do na = 1, 4*n_aero, 4
                  if (modal_aero) then
                     if (na==1) then     ! interstitial BC
                        taer = taer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_3bd(ns,k_bcexs(k))
                        waer = waer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_3bd(ns,k_bcexs(k)) &
                             * waer_bc_3bd(ns,k_bcexs(k))
                        gaer = gaer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_3bd(ns,k_bcexs(k)) &
                             * waer_bc_3bd(ns,k_bcexs(k)) &
                             * gaer_bc_3bd(ns,k_bcexs(k))
                     elseif (na==5) then ! within-ice BC
                        taer = taer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_3bd(ns,k_bcins(k)) &
                             *   bcenh_3bd(ns,k_bcins(k),k_bcini(k))
                        waer = waer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_3bd(ns,k_bcins(k)) &
                             * waer_bc_3bd(ns,k_bcins(k))
                        gaer = gaer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_3bd(ns,k_bcins(k)) &
                             * waer_bc_3bd(ns,k_bcins(k)) &
                             * gaer_bc_3bd(ns,k_bcins(k))
                     else                ! other species (dust)
                        taer = taer + (aero_mp(na+3)*rnilyr) &
                             * kaer_3bd(ns,(1+(na-1)/4))
                        waer = waer + (aero_mp(na+3)*rnilyr) &
                             * kaer_3bd(ns,(1+(na-1)/4)) &
                             * waer_3bd(ns,(1+(na-1)/4))
                        gaer = gaer + (aero_mp(na+3)*rnilyr) &
                             * kaer_3bd(ns,(1+(na-1)/4)) &
                             * waer_3bd(ns,(1+(na-1)/4)) &
                             * gaer_3bd(ns,(1+(na-1)/4))
                     endif
                  else       ! bulk
                     taer = taer + (aero_mp(na+3)*rnilyr) &
                          * kaer_3bd(ns,(1+(na-1)/4))
                     waer = waer + (aero_mp(na+3)*rnilyr) &
                          * kaer_3bd(ns,(1+(na-1)/4)) &
                          * waer_3bd(ns,(1+(na-1)/4))
                     gaer = gaer + (aero_mp(na+3)*rnilyr) &
                          * kaer_3bd(ns,(1+(na-1)/4)) &
                          * waer_3bd(ns,(1+(na-1)/4)) &
                          * gaer_3bd(ns,(1+(na-1)/4))
                  endif       ! modal_aero
                  enddo       ! na
                  g  (k) = (g(k)*w0(k)*tau(k) + gaer) / &
                                (w0(k)*tau(k) + waer)
                  w0 (k) =      (w0(k)*tau(k) + waer) / &
                                      (tau(k) + taer)
                  tau(k) = tau(k) + taer
               enddo ! k
            endif    ! tr_aero

         else ! srftyp == 2
            ! sea ice layers under ponds
            k = kii
            tau(k) = ki_p_ssl(ns)*dzk(k)
            w0 (k) = wi_p_ssl(ns)
            g  (k) = gi_p_ssl(ns)
            k = kii + 1
            tau(k) = ki_p_int(ns)*dzk(k)
            w0 (k) = wi_p_int(ns)
            g  (k) = gi_p_int(ns)
            if (kii+2 <= klev) then
               do k = kii+2, klev
                  tau(k) = ki_p_int(ns)*dzk(k)
                  w0 (k) = wi_p_int(ns)
                  g  (k) = gi_p_int(ns)
               enddo       ! k
            endif
            ! adjust pond iops if pond depth within specified range
            ! turn off for sea level ponds
            if(.not. tr_pond_sealvl .and. &
               hpmin <= hp .and. hp < hp0 ) then
               k = kii
               sig_i  = ki_ssl  (ns) * wi_ssl  (ns)
               sig_p  = ki_p_ssl(ns) * wi_p_ssl(ns)
               sig    = sig_i + (sig_p-sig_i) * (hp/hp0)
               kext   = sig   + ki_p_ssl(ns) * (c1-wi_p_ssl(ns))
               tau(k) = kext*dzk(k)
               w0 (k) = sig/kext
               g  (k) = gi_p_int(ns)
               k = kii + 1
               ! scale dz for dl relative to 4 even-layer-thickness 1.5m case
               fs = p25*real(nilyr,kind=dbl_kind)
               sig_i  = ki_dl   (ns) * wi_dl   (ns) * fs
               sig_p  = ki_p_int(ns) * wi_p_int(ns)
               sig    = sig_i + (sig_p-sig_i) * (hp/hp0)
               kext   = sig + ki_p_int(ns) * (c1-wi_p_int(ns))
               tau(k) = kext*dzk(k)
               w0 (k) = sig/kext
               g  (k) = gi_p_int(ns)
               if (kii+2 <= klev) then
                  do k = kii+2, klev
                     sig_i  = ki_int  (ns) * wi_int  (ns)
                     sig_p  = ki_p_int(ns) * wi_p_int(ns)
                     sig    = sig_i + (sig_p-sig_i) * (hp/hp0)
                     kext   = sig + ki_p_int(ns) * (c1-wi_p_int(ns))
                     tau(k) = kext*dzk(k)
                     w0 (k) = sig/kext
                     g  (k) = gi_p_int(ns)
                  enddo       ! k
               endif
            endif        ! small pond depth transition to bare sea ice
         endif         ! srftyp

         ! set reflectivities for ocean underlying sea ice
         rns = real(ns-1, kind=dbl_kind)
         albodr = cp01 * (c1 - min(rns, c1))
         albodf = cp01 * (c1 - min(rns, c1))

         ! layer input properties now completely specified: tau, w0, g,
         ! albodr, albodf; now compute the Delta-Eddington solution
         ! reflectivities and transmissivities for each layer; then,
         ! combine the layers going downwards accounting for multiple
         ! scattering between layers, and finally start from the
         ! underlying ocean and combine successive layers upwards to
         ! the surface; see comments in solution_dEdd for more details.

         call solution_dEdd (                                              &
                coszen,     srftyp,     klev,       klevp,                 &
                tau,        w0,         g,          albodr,     albodf,    &
                trndir,     trntdr,     trndif,     rupdir,     rupdif,    &
                rdndif)
         if (icepack_warnings_aborted(subname)) return

         ! the interface reflectivities and transmissivities required
         ! to evaluate interface fluxes are returned from solution_dEdd;
         ! now compute up and down fluxes for each interface, using the
         ! combined layer properties at each interface:
         !
         !              layers       interface
         !
         !       ---------------------  k
         !                 k
         !       ---------------------

         do k = 0, klevp
            ! interface scattering
            refk = c1/(c1 - rdndif(k)*rupdif(k))
            ! dir tran ref from below times interface scattering, plus diff
            ! tran and ref from below times interface scattering
            ! fdirup(k) = (trndir(k)*rupdir(k) + &
            !                 (trntdr(k)-trndir(k))  &
            !                 *rupdif(k))*refk
            ! dir tran plus total diff trans times interface scattering plus
            ! dir tran with up dir ref and down dif ref times interface scattering
            ! fdirdn(k) = trndir(k) + (trntdr(k) &
            !               - trndir(k) + trndir(k)  &
            !               *rupdir(k)*rdndif(k))*refk
            ! diffuse tran ref from below times interface scattering
            ! fdifup(k) = trndif(k)*rupdif(k)*refk
            ! diffuse tran times interface scattering
            ! fdifdn(k) = trndif(k)*refk

            ! dfdir = fdirdn - fdirup
            dfdir(k) = trndir(k) &
                        + (trntdr(k)-trndir(k)) * (c1 - rupdif(k)) * refk &
                        -  trndir(k)*rupdir(k)  * (c1 - rdndif(k)) * refk
            if (dfdir(k) < puny) dfdir(k) = c0 !echmod necessary?
            ! dfdif = fdifdn - fdifup
            dfdif(k) = trndif(k) * (c1 - rupdif(k)) * refk
            if (dfdif(k) < puny) dfdif(k) = c0 !echmod necessary?
         enddo       ! k

         ! calculate final surface albedos and fluxes-
         ! all absorbed flux above ksrf is included in surface absorption
         if (ns == 1) then      ! visible
            swdr   = swvdr
            swdf   = swvdf
            avdr   = rupdir(0)
            avdf   = rupdif(0)
            tmp_0  = dfdir(0    )*swdr + dfdif(0    )*swdf
            tmp_ks = dfdir(ksrf )*swdr + dfdif(ksrf )*swdf
            tmp_kl = dfdir(klevp)*swdr + dfdif(klevp)*swdf

            ! for layer biology: save visible only
            do k = nslyr+2, klevp ! Start at DL layer of ice after SSL scattering
               fthrul(k-nslyr-1) = dfdir(k)*swdr + dfdif(k)*swdf
            enddo

            fsfc  = fsfc  + tmp_0  - tmp_ks
            fint  = fint  + tmp_ks - tmp_kl
            fthru = fthru + tmp_kl
            fthruvdr = fthruvdr + dfdir(klevp)*swdr
            fthruvdf = fthruvdf + dfdif(klevp)*swdf
            if (present(swuvrdr)) fthruuvrdr = dfdir(klevp)*swuvrdr
            if (present(swuvrdf)) fthruuvrdf = dfdif(klevp)*swuvrdf
            if (present(swpardr)) fthrupardr = dfdir(klevp)*swpardr
            if (present(swpardf)) fthrupardf = dfdif(klevp)*swpardf

            ! if snow covered ice, set snow internal absorption; else, Sabs=0
            if (srftyp == 1) then
               ki = 0
               do k = 1, nslyr
                  ! skip snow SSL, since SSL absorption included in the surface
                  ! absorption fsfc above
                  km  = k
                  kp  = km + 1
                  ki  = ki + 1
                  Sabs(ki) = Sabs(ki) &
                           +  dfdir(km)*swdr + dfdif(km)*swdf &
                           - (dfdir(kp)*swdr + dfdif(kp)*swdf)
               enddo       ! k
            endif

            ! complex indexing to insure proper absorptions for sea ice
            ki = 0
            do k = nslyr+2, nslyr+1+nilyr
               ! for bare ice, DL absorption for sea ice layer 1
               km = k
               kp = km + 1
               ! modify for top sea ice layer for snow over sea ice
               if (srftyp == 1) then
                  ! must add SSL and DL absorption for sea ice layer 1
                  if (k == nslyr+2) then
                     km = k  - 1
                     kp = km + 2
                  endif
               endif
               ki = ki + 1
               Iabs(ki) = Iabs(ki) &
                        +  dfdir(km)*swdr + dfdif(km)*swdf &
                        - (dfdir(kp)*swdr + dfdif(kp)*swdf)
            enddo       ! k

         else ! ns > 1, near IR

            swdr = swidr
            swdf = swidf

            ! let fr1 = alb_1*swd*wght1 and fr2 = alb_2*swd*wght2 be the ns=2,3
            ! reflected fluxes respectively, where alb_1, alb_2 are the band
            ! albedos, swd = nir incident shortwave flux, and wght1, wght2 are
            ! the 2,3 band weights. thus, the total reflected flux is:
            ! fr = fr1 + fr2 = alb_1*swd*wght1 + alb_2*swd*wght2  hence, the
            ! 2,3 nir band albedo is alb = fr/swd = alb_1*wght1 + alb_2*wght2

            aidr   = aidr + rupdir(0)*wghtns(ns)
            aidf   = aidf + rupdif(0)*wghtns(ns)

            tmp_0  = dfdir(0    )*swdr + dfdif(0    )*swdf
            tmp_ks = dfdir(ksrf )*swdr + dfdif(ksrf )*swdf
            tmp_kl = dfdir(klevp)*swdr + dfdif(klevp)*swdf

            tmp_0  = tmp_0  * wghtns(ns)
            tmp_ks = tmp_ks * wghtns(ns)
            tmp_kl = tmp_kl * wghtns(ns)

            fsfc  = fsfc  + tmp_0  - tmp_ks
            fint  = fint  + tmp_ks - tmp_kl
            fthru = fthru + tmp_kl
            fthruidr = fthruidr + dfdir(klevp)*swdr*wghtns(ns)
            fthruidf = fthruidf + dfdif(klevp)*swdf*wghtns(ns)

            ! if snow covered ice, set snow internal absorption; else, Sabs=0
            if (srftyp == 1) then
               ki = 0
               do k = 1, nslyr
                  ! skip snow SSL, since SSL absorption included in the surface
                  ! absorption fsfc above
                  km = k
                  kp = km + 1
                  ki = ki + 1
                  Sabs(ki) = Sabs(ki) &
                           + (dfdir(km)*swdr + dfdif(km)*swdf   &
                           - (dfdir(kp)*swdr + dfdif(kp)*swdf)) &
                           * wghtns(ns)
               enddo       ! k
            endif

            ! complex indexing to insure proper absorptions for sea ice
            ki = 0
            do k = nslyr+2, nslyr+1+nilyr
               ! for bare ice, DL absorption for sea ice layer 1
               km = k
               kp = km + 1
               ! modify for top sea ice layer for snow over sea ice
               if (srftyp == 1) then
                  ! must add SSL and DL absorption for sea ice layer 1
                  if (k == nslyr+2) then
                     km = k  - 1
                     kp = km + 2
                  endif
               endif
               ki = ki + 1
               Iabs(ki) = Iabs(ki) &
                        + (dfdir(km)*swdr + dfdif(km)*swdf &
                        - (dfdir(kp)*swdr + dfdif(kp)*swdf)) &
                        * wghtns(ns)
            enddo       ! k
         endif          ! ns
      enddo             ! ns: end spectral loop

      alvdr = avdr
      alvdf = avdf
      alidr = aidr
      alidf = aidf

      ! accumulate fluxes over bare sea ice
      fswsfc  = fswsfc  + fsfc *fi
      fswint  = fswint  + fint *fi
      fswthru = fswthru + fthru*fi
      fswthru_vdr = fswthru_vdr + fthruvdr*fi
      fswthru_vdf = fswthru_vdf + fthruvdf*fi
      fswthru_idr = fswthru_idr + fthruidr*fi
      fswthru_idf = fswthru_idf + fthruidf*fi
      fswthru_uvrdr = fswthru_uvrdr + fthruuvrdr*fi
      fswthru_uvrdf = fswthru_uvrdf + fthruuvrdf*fi
      fswthru_pardr = fswthru_pardr + fthrupardr*fi
      fswthru_pardf = fswthru_pardf + fthrupardf*fi

      do k = 1, nslyr
         Sswabs(k) = Sswabs(k) + Sabs(k)*fi
      enddo

      do k = 1, nilyr
         Iswabs(k) = Iswabs(k) + Iabs(k)*fi
         ! bgc layer
         fswpenl(k) = fswpenl(k) + fthrul(k)* fi
      enddo
      fswpenl(nilyr+1) = fswpenl(nilyr+1) + fthrul(nilyr+1)*fi

      end subroutine compute_dEdd_3bd

!=======================================================================
!
! Given input vertical profiles of optical properties, evaluate the
! monochromatic Delta-Eddington solution.
!
! author:  Bruce P. Briegleb, NCAR
!   2013:  E Hunke merged with NCAR version
      subroutine solution_dEdd (                               &
             coszen,     srftyp,    klev,      klevp,          &
             tau,        w0,        g,         albodr, albodf, &
             trndir,     trntdr,    trndif,    rupdir, rupdif, &
             rdndif)

      real (kind=dbl_kind), intent(in) :: &
         coszen      ! cosine solar zenith angle

      integer (kind=int_kind), intent(in) :: &
         srftyp   , & ! surface type over ice: (0=air, 1=snow, 2=pond)
         klev     , & ! number of radiation layers - 1
         klevp        ! number of radiation interfaces - 1
                      ! (0 layer is included also)

      real (kind=dbl_kind), dimension(0:klev), intent(in) :: &
         tau     , & ! layer extinction optical depth
         w0      , & ! layer single scattering albedo
         g           ! layer asymmetry parameter

      real (kind=dbl_kind), intent(in) :: &
         albodr  , & ! ocean albedo to direct rad
         albodf      ! ocean albedo to diffuse rad

      ! following arrays are defined at model interfaces; 0 is the top of the
      ! layer above the sea ice; klevp is the sea ice/ocean interface.
      real (kind=dbl_kind), dimension (0:klevp), intent(out) :: &
         trndir  , & ! solar beam down transmission from top
         trntdr  , & ! total transmission to direct beam for layers above
         trndif  , & ! diffuse transmission to diffuse beam for layers above
         rupdir  , & ! reflectivity to direct radiation for layers below
         rupdif  , & ! reflectivity to diffuse radiation for layers below
         rdndif      ! reflectivity to diffuse radiation for layers above

!-----------------------------------------------------------------------
!
! Delta-Eddington solution for snow/air/pond over sea ice
!
! Generic solution for a snow/air/pond input column of klev+1 layers,
! with srftyp determining at what interface fresnel refraction occurs.
!
! Computes layer reflectivities and transmissivities, from the top down
! to the lowest interface using the Delta-Eddington solutions for each
! layer; combines layers from top down to lowest interface, and from the
! lowest interface (underlying ocean) up to the top of the column.
!
! Note that layer diffuse reflectivity and transmissivity are computed
! by integrating the direct over several gaussian angles. This is
! because the diffuse reflectivity expression sometimes is negative,
! but the direct reflectivity is always well-behaved. We assume isotropic
! radiation in the upward and downward hemispheres for this integration.
!
! Assumes monochromatic (spectrally uniform) properties across a band
! for the input optical parameters.
!
! If total transmission of the direct beam to the interface above a particular
! layer is less than trmin, then no further Delta-Eddington solutions are
! evaluated for layers below.
!
! The following describes how refraction is handled in the calculation.
!
! First, we assume that radiation is refracted when entering either
! sea ice at the base of the surface scattering layer, or water (i.e. melt
! pond); we assume that radiation does not refract when entering snow, nor
! upon entering sea ice from a melt pond, nor upon entering the underlying
! ocean from sea ice.
!
! To handle refraction, we define a "fresnel" layer, which physically
! is of neglible thickness and is non-absorbing, which can be combined to
! any sea ice layer or top of melt pond. The fresnel layer accounts for
! refraction of direct beam and associated reflection and transmission for
! solar radiation. A fresnel layer is combined with the top of a melt pond
! or to the surface scattering layer of sea ice if no melt pond lies over it.
!
! Some caution must be exercised for the fresnel layer, because any layer
! to which it is combined is no longer a homogeneous layer, as are all other
! individual layers. For all other layers for example, the direct and diffuse
! reflectivities/transmissivities (R/T) are the same for radiation above or
! below the layer. This is the meaning of homogeneous! But for the fresnel
! layer this is not so. Thus, the R/T for this layer must be distinguished
! for radiation above from that from radiation below. For generality, we
! treat all layers to be combined as inhomogeneous.
!
!-----------------------------------------------------------------------

      ! local variables

      integer (kind=int_kind) :: &
         kfrsnl      ! radiation interface index for fresnel layer

      ! following variables are defined for each layer; 0 refers to the top
      ! layer. In general we must distinguish directions above and below in
      ! the diffuse reflectivity and transmissivity, as layers are not assumed
      ! to be homogeneous (apart from the single layer Delta-Edd solutions);
      ! the direct is always from above.
      real (kind=dbl_kind), dimension (0:klev) :: &
         rdir    , & ! layer reflectivity to direct radiation
         rdif_a  , & ! layer reflectivity to diffuse radiation from above
         rdif_b  , & ! layer reflectivity to diffuse radiation from below
         tdir    , & ! layer transmission to direct radiation (solar beam + diffuse)
         tdif_a  , & ! layer transmission to diffuse radiation from above
         tdif_b  , & ! layer transmission to diffuse radiation from below
         trnlay      ! solar beam transm for layer (direct beam only)

      integer (kind=int_kind) :: &
         k           ! level index

      real (kind=dbl_kind), parameter :: &
         trmin = 0.001_dbl_kind   ! minimum total transmission allowed
      ! total transmission is that due to the direct beam; i.e. it includes
      ! both the directly transmitted solar beam and the diffuse downwards
      ! transmitted radiation resulting from scattering out of the direct beam
      real (kind=dbl_kind) :: &
         tautot   , & ! layer optical depth
         wtot     , & ! layer single scattering albedo
         gtot     , & ! layer asymmetry parameter
         ftot     , & ! layer forward scattering fraction
         ts       , & ! layer scaled extinction optical depth
         ws       , & ! layer scaled single scattering albedo
         gs       , & ! layer scaled asymmetry parameter
         rintfc   , & ! reflection (multiple) at an interface
         refkp1   , & ! interface multiple scattering for k+1
         refkm1   , & ! interface multiple scattering for k-1
         tdrrdir  , & ! direct tran times layer direct ref
         tdndif       ! total down diffuse = tot tran - direct tran

      ! perpendicular and parallel relative to plane of incidence and scattering
      real (kind=dbl_kind) :: &
         R1       , & ! perpendicular polarization reflection amplitude
         R2       , & ! parallel polarization reflection amplitude
         T1       , & ! perpendicular polarization transmission amplitude
         T2       , & ! parallel polarization transmission amplitude
         Rf_dir_a , & ! fresnel reflection to direct radiation
         Tf_dir_a , & ! fresnel transmission to direct radiation
         Rf_dif_a , & ! fresnel reflection to diff radiation from above
         Rf_dif_b , & ! fresnel reflection to diff radiation from below
         Tf_dif_a , & ! fresnel transmission to diff radiation from above
         Tf_dif_b     ! fresnel transmission to diff radiation from below

      ! refractive index for sea ice, water; pre-computed, band-independent,
      ! diffuse fresnel reflectivities
      real (kind=dbl_kind), parameter :: &
         refindx = 1.310_dbl_kind  , & ! refractive index of sea ice (water also)
         cp063   = 0.063_dbl_kind  , & ! diffuse fresnel reflectivity from above
         cp455   = 0.455_dbl_kind      ! diffuse fresnel reflectivity from below

      real (kind=dbl_kind) :: &
         mu0      , & ! cosine solar zenith angle incident
         mu0nij       ! cosine solar zenith angle in medium below fresnel level

      real (kind=dbl_kind) :: &
         mu0n         ! cosine solar zenith angle in medium

      real (kind=dbl_kind) :: &
         alp      , & ! temporary for alpha
         gam      , & ! temporary for agamm
         lm       , & ! temporary for el
         mu       , & ! temporary for gauspt
         ne       , & ! temporary for n
         ue       , & ! temporary for u
         extins   , & ! extinction
         amg      , & ! alp - gam
         apg          ! alp + gam

      integer (kind=int_kind), parameter :: &
         ngmax = 8    ! number of gaussian angles in hemisphere

      real (kind=dbl_kind), dimension (ngmax), parameter :: &
         gauspt     & ! gaussian angles (radians)
            = (/ .9894009_dbl_kind,  .9445750_dbl_kind, &
                 .8656312_dbl_kind,  .7554044_dbl_kind, &
                 .6178762_dbl_kind,  .4580168_dbl_kind, &
                 .2816036_dbl_kind,  .0950125_dbl_kind/), &
         gauswt     & ! gaussian weights
            = (/ .0271525_dbl_kind,  .0622535_dbl_kind, &
                 .0951585_dbl_kind,  .1246290_dbl_kind, &
                 .1495960_dbl_kind,  .1691565_dbl_kind, &
                 .1826034_dbl_kind,  .1894506_dbl_kind/)

      integer (kind=int_kind) :: &
         ng           ! gaussian integration index

      real (kind=dbl_kind) :: &
         gwt      , & ! gaussian weight
         swt      , & ! sum of weights
         trn      , & ! layer transmission
         rdr      , & ! rdir for gaussian integration
         tdr      , & ! tdir for gaussian integration
         smr      , & ! accumulator for rdif gaussian integration
         smt          ! accumulator for tdif gaussian integration

      real (kind=dbl_kind) :: &
         exp_min                    ! minimum exponential value

      character(len=*),parameter :: subname='(solution_dEdd)'

!-----------------------------------------------------------------------

      do k = 0, klevp
         trndir(k) = c0
         trntdr(k) = c0
         trndif(k) = c0
         rupdir(k) = c0
         rupdif(k) = c0
         rdndif(k) = c0
      enddo

      ! initialize top interface of top layer
      trndir(0) =   c1
      trntdr(0) =   c1
      trndif(0) =   c1
      rdndif(0) =   c0

      ! mu0 is cosine solar zenith angle above the fresnel level; make
      ! sure mu0 is large enough for stable and meaningful radiation
      ! solution: .01 is like sun just touching horizon with its lower edge
      mu0  = max(coszen,p01)

      ! mu0n is cosine solar zenith angle used to compute the layer
      ! Delta-Eddington solution; it is initially computed to be the
      ! value below the fresnel level, i.e. the cosine solar zenith
      ! angle below the fresnel level for the refracted solar beam:
      mu0nij = sqrt(c1-((c1-mu0**2)/(refindx*refindx)))

      ! compute level of fresnel refraction
      ! if ponded sea ice, fresnel level is the top of the pond.
      kfrsnl = 0
      ! if snow over sea ice or bare sea ice, fresnel level is
      ! at base of sea ice SSL (and top of the sea ice DL); the
      ! snow SSL counts for one, then the number of snow layers,
      ! then the sea ice SSL which also counts for one:
      if( srftyp < 2 ) kfrsnl = nslyr + 2

      ! proceed down one layer at a time; if the total transmission to
      ! the interface just above a given layer is less than trmin, then no
      ! Delta-Eddington computation for that layer is done.

      ! begin main level loop
      do k = 0, klev

         ! initialize all layer apparent optical properties to 0
         rdir  (k) = c0
         rdif_a(k) = c0
         rdif_b(k) = c0
         tdir  (k) = c0
         tdif_a(k) = c0
         tdif_b(k) = c0
         trnlay(k) = c0

         ! compute next layer Delta-eddington solution only if total transmission
         ! of radiation to the interface just above the layer exceeds trmin.

         if (trntdr(k) > trmin ) then

            ! calculation over layers with penetrating radiation

            tautot  = tau(k)
            wtot    = w0(k)
            gtot    = g(k)
            ftot    = gtot*gtot

            ts   = taus(wtot,ftot,tautot)
            ws   = omgs(wtot,ftot)
            gs   = asys(gtot,ftot)
            lm   = el(ws,gs)
            ue   = u(ws,gs,lm)

            mu0n = mu0nij
            ! if level k is above fresnel level and the cell is non-pond, use the
            ! non-refracted beam instead
            if( srftyp < 2 .and. k < kfrsnl ) mu0n = mu0

            exp_min = min(exp_argmax,lm*ts)
            extins = exp(-exp_min)
            ne = n(ue,extins)

            ! first calculation of rdif, tdif using Delta-Eddington formulas
!            rdif_a(k) = (ue+c1)*(ue-c1)*(c1/extins - extins)/ne
            rdif_a(k) = (ue**2-c1)*(c1/extins - extins)/ne
            tdif_a(k) = c4*ue/ne

            ! evaluate rdir,tdir for direct beam
            exp_min = min(exp_argmax,ts/mu0n)
            trnlay(k) = exp(-exp_min)
            alp = alpha(ws,mu0n,gs,lm)
            gam = agamm(ws,mu0n,gs,lm)
            apg = alp + gam
            amg = alp - gam
            rdir(k) = apg*rdif_a(k) +  amg*(tdif_a(k)*trnlay(k) - c1)
            tdir(k) = apg*tdif_a(k) + (amg* rdif_a(k)-apg+c1)*trnlay(k)

            ! recalculate rdif,tdif using direct angular integration over rdir,tdir,
            ! since Delta-Eddington rdif formula is not well-behaved (it is usually
            ! biased low and can even be negative); use ngmax angles and gaussian
            ! integration for most accuracy:
            R1 = rdif_a(k) ! use R1 as temporary
            T1 = tdif_a(k) ! use T1 as temporary
            swt = c0
            smr = c0
            smt = c0
            do ng=1,ngmax
               mu  = gauspt(ng)
               gwt = gauswt(ng)
               swt = swt + mu*gwt
               exp_min = min(exp_argmax,ts/mu)
               trn = exp(-exp_min)
               alp = alpha(ws,mu,gs,lm)
               gam = agamm(ws,mu,gs,lm)
               apg = alp + gam
               amg = alp - gam
               rdr = apg*R1 + amg*T1*trn - amg
               tdr = apg*T1 + amg*R1*trn - apg*trn + trn
               smr = smr + mu*rdr*gwt
               smt = smt + mu*tdr*gwt
            enddo      ! ng
            rdif_a(k) = smr/swt
            tdif_a(k) = smt/swt

            ! homogeneous layer
            rdif_b(k) = rdif_a(k)
            tdif_b(k) = tdif_a(k)

            ! add fresnel layer to top of desired layer if either
            ! air or snow overlies ice; we ignore refraction in ice
            ! if a melt pond overlies it:

            if( k == kfrsnl ) then
               ! compute fresnel reflection and transmission amplitudes
               ! for two polarizations: 1=perpendicular and 2=parallel to
               ! the plane containing incident, reflected and refracted rays.
               R1 = (mu0 - refindx*mu0n) / &
                    (mu0 + refindx*mu0n)
               R2 = (refindx*mu0 - mu0n) / &
                    (refindx*mu0 + mu0n)
               T1 = c2*mu0 / &
                    (mu0 + refindx*mu0n)
               T2 = c2*mu0 / &
                    (refindx*mu0 + mu0n)

               ! unpolarized light for direct beam
               Rf_dir_a = p5 * (R1*R1 + R2*R2)
               Tf_dir_a = p5 * (T1*T1 + T2*T2)*refindx*mu0n/mu0

               ! precalculated diffuse reflectivities and transmissivities
               ! for incident radiation above and below fresnel layer, using
               ! the direct albedos and accounting for complete internal
               ! reflection from below; precalculated because high order
               ! number of gaussian points (~256) is required for convergence:

               ! above
               Rf_dif_a = cp063
               Tf_dif_a = c1 - Rf_dif_a
               ! below
               Rf_dif_b = cp455
               Tf_dif_b = c1 - Rf_dif_b

               ! the k = kfrsnl layer properties are updated to combined
               ! the fresnel (refractive) layer, always taken to be above
               ! the present layer k (i.e. be the top interface):

               rintfc  = c1 / (c1-Rf_dif_b*rdif_a(k))
               tdir  (k) = Tf_dir_a*tdir(k) &
                         + Tf_dir_a*rdir(k) * Rf_dif_b*rintfc*tdif_a(k)
               rdir  (k) = Rf_dir_a &
                         + Tf_dir_a*rdir  (k) * rintfc*Tf_dif_b
               rdif_a(k) = Rf_dif_a &
                         + Tf_dif_a*rdif_a(k) * rintfc*Tf_dif_b
               rdif_b(k) = rdif_b(k) &
                         + tdif_b(k)*Rf_dif_b * rintfc*tdif_a(k)
               tdif_a(k) = tdif_a(k)*rintfc*Tf_dif_a
               tdif_b(k) = tdif_b(k)*rintfc*Tf_dif_b

               ! update trnlay to include fresnel transmission
               trnlay(k) = Tf_dir_a*trnlay(k)

            endif      ! k = kfrsnl

         endif ! trntdr(k) > trmin

         ! initialize current layer properties to zero; only if total
         ! transmission to the top interface of the current layer exceeds the
         ! minimum, will these values be computed below:
         ! Calculate the solar beam transmission, total transmission, and
         ! reflectivity for diffuse radiation from below at interface k,
         ! the top of the current layer k:
         !
         !              layers       interface
         !
         !       ---------------------  k-1
         !                k-1
         !       ---------------------  k
         !                 k
         !       ---------------------
         !       For k = klevp
         ! note that we ignore refraction between sea ice and underlying ocean:
         !
         !              layers       interface
         !
         !       ---------------------  k-1
         !                k-1
         !       ---------------------  k
         !       \\\\\\\ ocean \\\\\\\

         trndir(k+1) = trndir(k)*trnlay(k)
         refkm1      = c1/(c1 - rdndif(k)*rdif_a(k))
         tdrrdir     = trndir(k)*rdir(k)
         tdndif      = trntdr(k) - trndir(k)
         trntdr(k+1) = trndir(k)*tdir(k) &
                     + (tdndif + tdrrdir*rdndif(k))*refkm1*tdif_a(k)
         rdndif(k+1) = rdif_b(k) &
                     + (tdif_b(k)*rdndif(k)*refkm1*tdif_a(k))
         trndif(k+1) = trndif(k)*refkm1*tdif_a(k)

      enddo       ! k   end main level loop

      ! compute reflectivity to direct and diffuse radiation for layers
      ! below by adding succesive layers starting from the underlying
      ! ocean and working upwards:
      !
      !              layers       interface
      !
      !       ---------------------  k
      !                 k
      !       ---------------------  k+1
      !                k+1
      !       ---------------------

      rupdir(klevp) = albodr
      rupdif(klevp) = albodf

      do k=klev,0,-1
         ! interface scattering
         refkp1 = c1/( c1 - rdif_b(k)*rupdif(k+1))
         ! dir from top layer plus exp tran ref from lower layer, interface
         ! scattered and tran thru top layer from below, plus diff tran ref
         ! from lower layer with interface scattering tran thru top from below
         rupdir(k) = rdir(k) &
              + (        trnlay(k)  *rupdir(k+1) &
              +  (tdir(k)-trnlay(k))*rupdif(k+1))*refkp1*tdif_b(k)
         ! dif from top layer from above, plus dif tran upwards reflected and
         ! interface scattered which tran top from below
         rupdif(k) = rdif_a(k) + tdif_a(k)*rupdif(k+1)*refkp1*tdif_b(k)
      enddo       ! k

      end subroutine solution_dEdd

!=======================================================================
!
!   Set snow horizontal coverage, density and grain radius diagnostically
!   for the Delta-Eddington solar radiation method.
!
! author:  Bruce P. Briegleb, NCAR
!   2013:  E Hunke merged with NCAR version

      subroutine shortwave_dEdd_set_snow(R_snw,              &
                                         dT_mlt,   rsnw_mlt, &
                                         aice,     vsno,     &
                                         Tsfc,     fs,       &
                                         hs0,      hs,       &
                                         rhosnw,   rsnw,     &
                                         rsnow)

      real (kind=dbl_kind), intent(in) :: &
         R_snw , & ! snow tuning parameter; +1 > ~.01 change in broadband albedo
         dT_mlt, & ! change in temp for non-melt to melt snow grain radius change (C)
         rsnw_mlt  ! maximum melting snow grain radius (10^-6 m)

      real (kind=dbl_kind), intent(in) :: &
         aice   , & ! concentration of ice
         vsno   , & ! volume of snow
         Tsfc   , & ! surface temperature
         hs0        ! snow depth for transition to bare sea ice (m)

     real (kind=dbl_kind), intent(inout) :: &
         fs     , & ! horizontal coverage of snow
         hs         ! snow depth

      real (kind=dbl_kind), dimension (:), intent(in) :: &
         rsnow      ! snow grain radius tracer (micro-meters)

      real (kind=dbl_kind), dimension (:), intent(out) :: &
         rhosnw , & ! density in snow layer (kg/m3)
         rsnw       ! grain radius in snow layer (micro-meters)

      ! local variables

      integer (kind=int_kind) :: &
         ks           ! snow vertical index

      real (kind=dbl_kind) :: &
         fT  , & ! piecewise linear function of surface temperature
         dTs , & ! difference of Tsfc and Timelt
         rsnw_nm ! actual used nonmelt snow grain radius (micro-meters)

      real (kind=dbl_kind), parameter :: &
         ! units for the following are 1.e-6 m (micro-meters)
         rsnw_nonmelt = 500._dbl_kind, & ! nonmelt snow grain radius
         rsnw_sig     = 250._dbl_kind    ! assumed sigma for snow grain radius

      character(len=*),parameter :: subname='(shortwave_dEdd_set_snow)'

!-----------------------------------------------------------------------

      ! set snow horizontal fraction
      hs = vsno / aice

      if (hs >= hs_min) then
         fs = c1
         if (hs0 > puny) fs = min(hs/hs0, c1)
      endif

      if (snwgrain) then  ! use snow grain tracer

         do ks = 1, nslyr
            rsnw(ks)   = max(rsnw_fall,rsnow(ks))
            rsnw(ks)   = min(rsnw_tmax,rsnw(ks))
            rhosnw(ks) = rhos
         enddo

      else

         ! bare ice, temperature dependence
         dTs = Timelt - Tsfc
         fT  = -min(dTs/dT_mlt-c1,c0)
         ! tune nonmelt snow grain radius if desired: note that
         ! the sign is negative so that if R_snw is 1, then the
         ! snow grain radius is reduced and thus albedo increased.
         rsnw_nm = rsnw_nonmelt - R_snw*rsnw_sig
         rsnw_nm = max(rsnw_nm, rsnw_fall)
         rsnw_nm = min(rsnw_nm, rsnw_mlt)
         do ks = 1, nslyr
            ! snow density ccsm3 constant value
            rhosnw(ks) = rhos
            ! snow grain radius between rsnw_nonmelt and rsnw_mlt
            rsnw(ks) = rsnw_nm + (rsnw_mlt-rsnw_nm)*fT
            rsnw(ks) = max(rsnw(ks), rsnw_fall)
            rsnw(ks) = min(rsnw(ks), rsnw_mlt)
         enddo ! ks

      endif ! snwgrain

      end subroutine shortwave_dEdd_set_snow
!=======================================================================
!
!   Set the 'effective' snow and pond fractions and depths for dEdd
!
! author:  Bruce P. Briegleb, NCAR
!   2013:  E Hunke merged with NCAR version
!   2024:  DCS refactored for sealvl ponds

      subroutine shortwave_dEdd_set_eff(aicen,     vsnon,   alvln,   &
                                        apondn,    hpndn,   ipndn,   &
                                        ffracn,    fsnow,   dt,      &
                                        Tsfcn,     fsn,     hsn,     &
                                        dhsn,      fpn,     hpn,     &
                                        apeffn,    l_rsnows,rhosnwn, &
                                        rsnwn,     l_initonly)

      real (kind=dbl_kind), intent(in) :: &
         aicen   , & ! concentration of ice
         vsnon   , & ! volume per unit area of snow (m)
         alvln   , & ! level-ice area fraction
         apondn  , & ! pond area fraction of category (incl. deformed)
         hpndn   , & ! pond depth (m)
         ipndn   , & ! pond refrozen lid thickness (m)
         ffracn  , & ! fraction of fsurfn used to melt ipond
         fsnow   , & ! snowfall rate (kg/m^2 s)
         dt      , & ! time step (s)
         Tsfcn       ! surface temperature (deg C)

      real (kind=dbl_kind), intent(inout) :: &
         fsn     , & ! snow horizontal fraction
         hsn     , & ! snow depth (m)
         dhsn    , & ! depth difference for snow on sea ice and pond ice
         fpn     , & ! pond fraction of ice cover
         hpn         ! actual pond depth (m)

      real (kind=dbl_kind), intent(out) :: &
         apeffn      ! effective pond area used for radiation

      real(kind=dbl_kind), dimension(nslyr), intent(in) :: &
         l_rsnows    ! snow grain radius tracer (10^-6 m)

      real (kind=dbl_kind), dimension (nslyr), intent(out) :: &
         rhosnwn, &  ! snow density (kg/m3)
         rsnwn       ! snow grain radius (micrometers)

      logical (kind=log_kind), intent(in) :: &
         l_initonly  ! local initonly value

      ! local variables
      real (kind=dbl_kind) :: &
         hsnlvl  , & ! snow depth over level ice (m)
         vsn     , & ! snow volume
         alvl    , & ! area fraction of level ice
         ipn     , & ! pond lid thickness (m), mean over category
         hp      , & ! pond depth
         hs      , & ! snow depth
         asnow   , & ! fractional area of snow cover
         rp      , & ! ratio retained melt water to total liquid content
         hmx     , & ! maximum available snow infiltration equiv. depth
         dhs     , & ! local diff. in snow depth on sea ice and pond ice
         spn     , & ! snow depth on refrozen pond (m)
         tmp         ! 0 or 1

      character(len=*),parameter :: subname='(shortwave_dEdd_set_eff)'

!-----------------------------------------------------------------------
      hsnlvl = hsn ! initialize
      if (trim(snwredist) == 'bulk') then
         if (.not. tr_lvl) then
            call icepack_warnings_add(subname//' ERROR: need lvl trcr')
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            return
         endif
         hsnlvl = hsn / (c1 + snwlvlfac*(c1-alvln))
         ! snow volume over level ice
         alvl = aicen * alvln
         if (alvl > puny) then
            vsn = hsnlvl * alvl
         else
            vsn = vsnon
            alvl = aicen
         endif
         ! set snow properties over level ice
         call shortwave_dEdd_set_snow(R_snw,    &
                                       dT_mlt,     rsnw_mlt, &
                                       alvl,       vsn,      &
                                       Tsfcn,      fsn,      &
                                       hs0,        hsnlvl,   &
                                       rhosnwn(:), rsnwn(:), &
                                       l_rsnows(:))
         if (icepack_warnings_aborted(subname)) return
      endif ! snwredist

      fpn = c0  ! fraction of ice covered in pond
      hpn = c0  ! pond depth over fpn
      ! refrozen pond lid thickness avg over ice
      ! allow snow to cover pond ice
      ipn = apondn * ipndn
      dhs = dhsn ! snow depth difference, sea ice - pond
      if (.not. l_initonly .and. ipn > puny .and. &
            dhs < puny .and. fsnow*dt > hs_min) &
            dhs = hsnlvl - fsnow*dt ! initialize dhs>0
      spn = hsnlvl - dhs   ! snow depth on pond ice
      if (.not. l_initonly .and. ipn*spn < puny) dhs = c0
      dhsn = dhs ! save: constant until reset to 0

      ! not using ipn assumes that lid ice is perfectly clear
      ! if (ipn <= 0.3_dbl_kind) then

      ! fraction of ice area
      fpn = apondn
      ! pond depth over fraction fpn
      hpn = hpndn

      ! reduce effective pond area absorbing surface heat flux
      ! due to flux already having been used to melt pond ice
      fpn = (c1 - ffracn) * fpn

      ! taper pond area with snow on pond ice
      if (dhs > puny .and. spn >= puny .and. hs1 > puny) then
         asnow = min(spn/hs1, c1)
         fpn = (c1 - asnow) * fpn
      endif

      ! infiltrate snow
      hp = hpn
      if (hp > puny) then
         hs = hsnlvl
         rp = rhofresh*hp/(rhofresh*hp + rhos*hs)
         if (rp < p15) then
            fpn = c0
            hpn = c0
         else
            hmx = hs*(rhofresh - rhos)/rhofresh
            tmp = max(c0, sign(c1, hp-hmx)) ! 1 if hp>=hmx, else 0
            hp = (rhofresh*hp + rhos*hs*tmp) &
                  / (rhofresh    - rhos*(c1-tmp))
            hsn = hsn - hp*fpn*(c1-tmp)
            hpn = hp * tmp
            fpn = fpn * tmp
         endif
      endif ! hp > puny

      ! Zero out fraction of thin ponds for radiation only
      if (hpn < hpmin) fpn = c0
      fsn = min(fsn, c1-fpn)

      ! endif    ! masking by lid ice
      apeffn = fpn ! for history

      end subroutine shortwave_dEdd_set_eff

!=======================================================================
!
!   Set pond fraction and depth diagnostically for
!   the Delta-Eddington solar radiation method.
!
! author:  Bruce P. Briegleb, NCAR
!   2013:  E Hunke merged with NCAR version

      subroutine shortwave_dEdd_set_pond(Tsfc,               &
                                         fs,       fp,       &
                                         hp)

      real (kind=dbl_kind), intent(in) :: &
         Tsfc   , & ! surface temperature
         fs         ! horizontal coverage of snow

      real (kind=dbl_kind), intent(out) :: &
         fp     , & ! pond fractional coverage (0 to 1)
         hp         ! pond depth (m)

      ! local variables

      real (kind=dbl_kind) :: &
         fT  , & ! piecewise linear function of surface temperature
         dTs     ! difference of Tsfc and Timelt

      real (kind=dbl_kind), parameter :: &
         dT_pnd = c1   ! change in temp for pond fraction and depth

      character(len=*),parameter :: subname='(shortwave_dEdd_set_pond)'

!-----------------------------------------------------------------------

      ! bare ice, temperature dependence
      dTs = Timelt - Tsfc
      fT  = -min(dTs/dT_pnd-c1,c0)
      ! pond
      fp = 0.3_dbl_kind*fT*(c1-fs)
      hp = 0.3_dbl_kind*fT*(c1-fs)

      end subroutine shortwave_dEdd_set_pond

!=======================================================================
!
! authors     Nicole Jeffery, LANL

      subroutine compute_shortwave_trcr(bgcN,        zaero,     &
                                        trcrn_bgcsw, sw_grid,   &
                                        hin,         hbri,      &
                                        i_grid,      skl_bgc,   &
                                        z_tracers)

      real (kind=dbl_kind), dimension (:), intent(in) :: &
         bgcN       , & ! Nit tracer
         zaero          ! zaero tracer

      real (kind=dbl_kind), dimension (:), intent(out):: &
         trcrn_bgcsw    ! ice on shortwave grid tracers

      real (kind=dbl_kind), dimension (:), intent(in) :: &
         sw_grid    , & !
         i_grid         ! CICE bio grid

      real(kind=dbl_kind), intent(in) :: &
         hin        , & ! CICE ice thickness
         hbri           ! brine height

      logical (kind=log_kind), intent(in) :: &
         skl_bgc    , & ! skeletal layer bgc
         z_tracers      ! zbgc

      !  local variables

      integer (kind=int_kind) :: k, n, nn

      real (kind=dbl_kind), dimension (ntrcr+2) :: &
         trtmp0, &      ! temporary, remapped tracers
         trtmp

      real (kind=dbl_kind), dimension (nilyr+1):: &
         icegrid        ! correct for large ice surface layers

      real (kind=dbl_kind):: &
         top_conc       ! 1% (min_bgc) of surface concentration
                        ! when hin > hbri:  just used in sw calculation

      character(len=*),parameter :: subname='(compute_shortwave_trcr)'

      !-----------------------------------------------------------------
      ! Compute aerosols and algal chlorophyll on shortwave grid
      !-----------------------------------------------------------------

      trtmp0(:) = c0
      trtmp(:) = c0
      trcrn_bgcsw(:) = c0

      do k = 1,nilyr+1
         icegrid(k) = sw_grid(k)
      enddo
      if (sw_grid(1)*hin*c2 > hi_ssl .and. hin > puny) then
         icegrid(1) = hi_ssl/c2/hin
      endif
      icegrid(2) = c2*sw_grid(1) + (sw_grid(2) - sw_grid(1))

      if (z_tracers) then
      if (tr_bgc_N)  then
         if (size(bgcN) < n_algae*(nblyr+3)) then
            call icepack_warnings_add(subname//' ERROR: size(bgcN) too small')
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            return
         endif

         do k = 1, nblyr+1
            do n = 1, n_algae
               trtmp0(nt_bgc_N(1) + k-1) = trtmp0(nt_bgc_N(1) + k-1) &
                     + R_chl2N(n) * F_abs_chl(n) * bgcN(nt_bgc_N(n)-nt_bgc_N(1) + k)
            enddo ! n
         enddo    ! k

         top_conc = trtmp0(nt_bgc_N(1))*min_bgc
         call remap_zbgc (nilyr+1, &
                          nt_bgc_N(1),                &
                          trtmp0(1:ntrcr  ),          &
                          trtmp (1:ntrcr+2),          &
                          1,                 nblyr+1, &
                          hin,               hbri,    &
                          icegrid(1:nilyr+1),         &
                          i_grid(1:nblyr+1), top_conc )
         if (icepack_warnings_aborted(subname)) return

         do k = 1, nilyr+1
            trcrn_bgcsw(nlt_chl_sw+nslyr+k) = trtmp(nt_bgc_N(1) + k-1)
         enddo       ! k

         do n = 1, n_algae   ! snow contribution
            trcrn_bgcsw(nlt_chl_sw)= trcrn_bgcsw(nlt_chl_sw) &
                     + R_chl2N(n)*F_abs_chl(n)*bgcN(nt_bgc_N(n)-nt_bgc_N(1)+1+nblyr+1)
                              ! snow surface layer
            trcrn_bgcsw(nlt_chl_sw+1:nlt_chl_sw+nslyr) = &
                     trcrn_bgcsw(nlt_chl_sw+1:nlt_chl_sw+nslyr) &
                     + R_chl2N(n)*F_abs_chl(n)*bgcN(nt_bgc_N(n)-nt_bgc_N(1)+1+nblyr+2)
                              ! only 1 snow layer in zaero
         enddo ! n
      endif    ! tr_bgc_N

      if (tr_zaero) then
         if (size(zaero) < n_zaero*(nblyr+3)) then
            call icepack_warnings_add(subname//' ERROR: size(zaero) too small')
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            return
         endif

         do n = 1, n_zaero

            trtmp0(:) = c0
            trtmp(:) = c0

            do k = 1, nblyr+1
               trtmp0(nt_zaero(n) + k-1) = zaero(nt_zaero(n)-nt_zaero(1)+1+k-1)
            enddo

            top_conc = trtmp0(nt_zaero(n))*min_bgc
            call remap_zbgc (nilyr+1, &
                             nt_zaero(n),                &
                             trtmp0(1:ntrcr  ),          &
                             trtmp (1:ntrcr+2),          &
                             1,                 nblyr+1, &
                             hin,               hbri,    &
                             icegrid(1:nilyr+1),         &
                             i_grid(1:nblyr+1), top_conc )
            if (icepack_warnings_aborted(subname)) return

            do k = 1,nilyr+1
               trcrn_bgcsw(nlt_zaero_sw(n)+nslyr+k) = trtmp(nt_zaero(n) + k-1)
            enddo
            trcrn_bgcsw(nlt_zaero_sw(n))= zaero(nt_zaero(n)-nt_zaero(1)+1+nblyr+1) !snow ssl
            trcrn_bgcsw(nlt_zaero_sw(n)+1:nlt_zaero_sw(n)+nslyr)= zaero(nt_zaero(n)-nt_zaero(1)+1+nblyr+2)
         enddo ! n
      endif    ! tr_zaero
      elseif (skl_bgc) then

         do nn = 1,n_algae
            trcrn_bgcsw(nbtrcr_sw) = trcrn_bgcsw(nbtrcr_sw) &
                                + F_abs_chl(nn)*R_chl2N(nn) &
                                * bgcN(nt_bgc_N(nn)-nt_bgc_N(1)+1)*sk_l/hin &
                                * real(nilyr,kind=dbl_kind)
         enddo

      endif

      end subroutine compute_shortwave_trcr

!=======================================================================
!autodocument_start icepack_prep_radiation
! Scales radiation fields computed on the previous time step.
!
! authors: Elizabeth Hunke, LANL

      subroutine icepack_prep_radiation(aice,        aicen,    &
                                        swvdr,       swvdf,    &
                                        swidr,       swidf,    &
                                        alvdr_ai,    alvdf_ai, &
                                        alidr_ai,    alidf_ai, &
                                        scale_factor,          &
                                        fswsfcn,     fswintn,  &
                                        fswthrun,              &
                                        fswthrun_vdr,          &
                                        fswthrun_vdf,          &
                                        fswthrun_idr,          &
                                        fswthrun_idf,          &
                                        fswpenln,              &
                                        Sswabsn,     Iswabsn)

      real (kind=dbl_kind), intent(in) :: &
         aice        , & ! ice area fraction
         swvdr       , & ! sw down, visible, direct  (W/m^2)
         swvdf       , & ! sw down, visible, diffuse (W/m^2)
         swidr       , & ! sw down, near IR, direct  (W/m^2)
         swidf       , & ! sw down, near IR, diffuse (W/m^2)
         ! grid-box-mean albedos aggregated over categories (if calc_Tsfc)
         alvdr_ai    , & ! visible, direct   (fraction)
         alidr_ai    , & ! near-ir, direct   (fraction)
         alvdf_ai    , & ! visible, diffuse  (fraction)
         alidf_ai        ! near-ir, diffuse  (fraction)

      real (kind=dbl_kind), dimension(:), intent(in) :: &
         aicen           ! ice area fraction in each category

      real (kind=dbl_kind), intent(inout) :: &
         scale_factor    ! shortwave scaling factor, ratio new:old

      real (kind=dbl_kind), dimension(:), intent(inout) :: &
         fswsfcn     , & ! SW absorbed at ice/snow surface (W m-2)
         fswintn     , & ! SW absorbed in ice interior, below surface (W m-2)
         fswthrun        ! SW through ice to ocean (W/m^2)

      real (kind=dbl_kind), dimension(:), intent(inout), optional :: &
         fswthrun_vdr , & ! vis dir SW through ice to ocean (W/m^2)
         fswthrun_vdf , & ! vis dif SW through ice to ocean (W/m^2)
         fswthrun_idr , & ! nir dir SW through ice to ocean (W/m^2)
         fswthrun_idf     ! nir dif SW through ice to ocean (W/m^2)

      real (kind=dbl_kind), dimension(:,:), intent(inout) :: &
         fswpenln    , & ! visible SW entering ice layers (W m-2)
         Iswabsn     , & ! SW radiation absorbed in ice layers (W m-2)
         Sswabsn         ! SW radiation absorbed in snow layers (W m-2)

!autodocument_end

      ! local variables

      integer (kind=int_kind) :: &
         k           , & ! vertical index
         n               ! thickness category index

      real (kind=dbl_kind) :: netsw

      character(len=*),parameter :: subname='(icepack_prep_radiation)'

      !-----------------------------------------------------------------
      ! Compute netsw scaling factor (new netsw / old netsw)
      !-----------------------------------------------------------------

         if (aice > c0 .and. scale_factor > puny) then
            netsw = swvdr*(c1 - alvdr_ai) &
                  + swvdf*(c1 - alvdf_ai) &
                  + swidr*(c1 - alidr_ai) &
                  + swidf*(c1 - alidf_ai)
            scale_factor = netsw / scale_factor
         else
            scale_factor = c1
         endif

         do n = 1, ncat

            if (aicen(n) > puny) then

      !-----------------------------------------------------------------
      ! Scale absorbed solar radiation for change in net shortwave
      !-----------------------------------------------------------------

               fswsfcn(n)  = scale_factor * fswsfcn (n)
               fswintn(n)  = scale_factor * fswintn (n)
               fswthrun(n) = scale_factor * fswthrun(n)
               if (present(fswthrun_vdr)) fswthrun_vdr(n) = scale_factor * fswthrun_vdr(n)
               if (present(fswthrun_vdf)) fswthrun_vdf(n) = scale_factor * fswthrun_vdf(n)
               if (present(fswthrun_idr)) fswthrun_idr(n) = scale_factor * fswthrun_idr(n)
               if (present(fswthrun_idf)) fswthrun_idf(n) = scale_factor * fswthrun_idf(n)
               do k = 1,nilyr+1
                  fswpenln(k,n) = scale_factor * fswpenln(k,n)
               enddo       !k
               do k=1,nslyr
                  Sswabsn (k,n) = scale_factor * Sswabsn (k,n)
               enddo
               do k=1,nilyr
                  Iswabsn (k,n) = scale_factor * Iswabsn (k,n)
               enddo

            endif
         enddo                  ! ncat

      end subroutine icepack_prep_radiation

!=======================================================================
!autodocument_start icepack_step_radiation
! Computes radiation fields
!
! authors: William H. Lipscomb, LANL
!          David Bailey, NCAR
!          Elizabeth C. Hunke, LANL

      subroutine icepack_step_radiation (dt,                 &
                                        fbri,                &
                                        aicen,    vicen,     &
                                        vsnon,    Tsfcn,     &
                                        alvln,    apndn,     &
                                        hpndn,    ipndn,     &
                                        aeron,               &
                                        bgcNn,    zaeron,    &
                                        trcrn_bgcsw,         &
                                        TLAT,     TLON,      &
                                        calendar_type,       &
                                        days_per_year,       &
                                        nextsw_cday,         &
                                        yday,     sec,       &
                                        swvdr,    swvdf,     &
                                        swidr,    swidf,     &
                                        swuvrdr,  swuvrdf,   &
                                        swpardr,  swpardf,   &
                                        coszen,   fsnow,     &
                                        alvdrn,   alvdfn,    &
                                        alidrn,   alidfn,    &
                                        fswsfcn,  fswintn,   &
                                        fswthrun,            &
                                        fswthrun_vdr,        &
                                        fswthrun_vdf,        &
                                        fswthrun_idr,        &
                                        fswthrun_idf,        &
                                        fswthrun_uvrdr,      &
                                        fswthrun_uvrdf,      &
                                        fswthrun_pardr,      &
                                        fswthrun_pardf,      &
                                        fswpenln,            &
                                        Sswabsn,  Iswabsn,   &
                                        albicen,  albsnon,   &
                                        albpndn,  apeffn,    &
                                        snowfracn,           &
                                        dhsn,     ffracn,    &
                                        rsnow,               &
                                        l_print_point,       &
                                        initonly)

      real (kind=dbl_kind), intent(in) :: &
         dt        , & ! time step (s)
         swvdr     , & ! sw down, visible, direct  (W/m^2)
         swvdf     , & ! sw down, visible, diffuse (W/m^2)
         swidr     , & ! sw down, near IR, direct  (W/m^2)
         swidf     , & ! sw down, near IR, diffuse (W/m^2)
         fsnow     , & ! snowfall rate (kg/m^2 s)
         TLAT, TLON    ! latitude and longitude (radian)

      real (kind=dbl_kind), intent(in), optional :: &
         swuvrdr   , & ! sw down, vis uvr dir (W/m^2)
         swuvrdf   , & ! sw down, vis uvr dif (W/m^2)
         swpardr   , & ! sw down, vis par dir (W/m^2)
         swpardf       ! sw down, vis par dif (W/m^2)

      integer (kind=int_kind), intent(in) :: &
         sec           ! elapsed seconds into date

      real (kind=dbl_kind), intent(in) :: &
         yday          ! day of the year

      character (len=char_len), intent(in), optional :: &
         calendar_type ! differentiates proleptic_gregorian from other calendars

      integer (kind=int_kind), intent(in), optional :: &
         days_per_year ! number of days in one year

      real (kind=dbl_kind), intent(in), optional :: &
         nextsw_cday   ! julian day of next shortwave calculation

      real (kind=dbl_kind), intent(inout) :: &
         coszen        ! cosine solar zenith angle, < 0 for sun below horizon

      real (kind=dbl_kind), dimension(:), intent(in) :: &
         aicen     , & ! ice area fraction in each category
         vicen     , & ! ice volume in each category (m)
         vsnon     , & ! snow volume in each category (m)
         Tsfcn     , & ! surface temperature (deg C)
         alvln     , & ! level-ice area fraction
         apndn     , & ! pond area fraction
         hpndn     , & ! pond depth (m)
         ipndn     , & ! pond refrozen lid thickness (m)
         fbri           ! brine fraction

      real(kind=dbl_kind), dimension(:,:), intent(in) :: &
         aeron     , & ! aerosols (kg/m^3)
         bgcNn     , & ! bgc Nit tracers
         zaeron        ! bgcz aero tracers

      real(kind=dbl_kind), dimension(:,:), intent(inout) :: &
         trcrn_bgcsw   ! zaerosols (kg/m^3) and chla (mg/m^3)

      real (kind=dbl_kind), dimension(:), intent(inout) :: &
         alvdrn    , & ! visible, direct  albedo (fraction)
         alidrn    , & ! near-ir, direct   (fraction)
         alvdfn    , & ! visible, diffuse  (fraction)
         alidfn    , & ! near-ir, diffuse  (fraction)
         fswsfcn   , & ! SW absorbed at ice/snow surface (W m-2)
         fswintn   , & ! SW absorbed in ice interior, below surface (W m-2)
         fswthrun  , & ! SW through ice to ocean (W/m^2)
         snowfracn , & ! snow fraction on each category
         dhsn      , & ! depth difference for snow on sea ice and pond ice
         ffracn    , & ! fraction of fsurfn used to melt ipond
                       ! albedo components for history
         albicen   , & ! bare ice
         albsnon   , & ! snow
         albpndn   , & ! pond
         apeffn        ! effective pond area used for radiation calculation

      real (kind=dbl_kind), dimension(:), intent(inout), optional :: &
         fswthrun_vdr , & ! vis dir SW through ice to ocean (W/m^2)
         fswthrun_vdf , & ! vis dif SW through ice to ocean (W/m^2)
         fswthrun_idr , & ! nir dir SW through ice to ocean (W/m^2)
         fswthrun_idf , & ! nir dif SW through ice to ocean (W/m^2)
         fswthrun_uvrdr,& ! vis uvr dir SW through ice to ocean (W/m^2)
         fswthrun_uvrdf,& ! vis uvr dif SW through ice to ocean (W/m^2)
         fswthrun_pardr,& ! vis par dir SW through ice to ocean (W/m^2)
         fswthrun_pardf   ! vis par dif SW through ice to ocean (W/m^2)

      real (kind=dbl_kind), dimension(:,:), intent(inout) :: &
         fswpenln  , & ! visible SW entering ice layers (W m-2)
         Iswabsn   , & ! SW radiation absorbed in ice layers (W m-2)
         Sswabsn       ! SW radiation absorbed in snow layers (W m-2)

      logical (kind=log_kind), intent(in) :: &
         l_print_point ! flag for printing diagnostics

      real (kind=dbl_kind), dimension(:,:), intent(inout), optional :: &
         rsnow         ! snow grain radius tracer (10^-6 m)

      logical (kind=log_kind), optional :: &
         initonly      ! flag to indicate init only, default is false

!autodocument_end

      ! local variables

      integer (kind=int_kind) :: &
         n             ! thickness category index

      logical (kind=log_kind), save :: &
         first_call=.true.  ! first call logical

      real(kind=dbl_kind) :: &
         hin,         & ! Ice thickness (m)
         hbri           ! brine thickness (m)

      character(len=*),parameter :: subname='(icepack_step_radiation)'

      if ((first_call .and. argcheck == 'first') .or. (argcheck == 'always')) then
         if (snwgrain .and. .not. present(rsnow)) then
            call icepack_warnings_add(subname//' ERROR: snwgrain on, rsnow not passed')
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            return
         endif
         if (semi_implicit_Tsfc) then
            if (.not.(present(swuvrdr) .and. present(swuvrdf) .and. &
                      present(swpardr) .and. present(swpardf) .and. &
                      present(fswthrun_uvrdr) .and. present(fswthrun_uvrdf) .and. &
                      present(fswthrun_pardr) .and. present(fswthrun_pardf))) then
               call icepack_warnings_add(subname//' ERROR: semi_implicit_Tsfc=T, missing arguments')
               call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
               return
            endif
         endif
#ifdef CESMCOUPLED
         if (.not.present(days_per_year) .or. &
             .not.present(nextsw_cday) .or. &
             .not.present(calendar_type)) then
            call icepack_warnings_add(subname//' ERROR: CESMCOUPLED CPP on, need more calendar data')
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            return
         endif
#endif
      endif

      hin = c0
      hbri = c0

      ! Initialize
      do n = 1, ncat
         alvdrn  (n) = c0
         alidrn  (n) = c0
         alvdfn  (n) = c0
         alidfn  (n) = c0
         fswsfcn (n) = c0
         fswintn (n) = c0
         fswthrun(n) = c0
      enddo   ! ncat
      fswpenln (:,:) = c0
      Iswabsn  (:,:) = c0
      Sswabsn  (:,:) = c0
      trcrn_bgcsw(:,:) = c0

      ! Interpolate z-shortwave tracers to shortwave grid
      if (dEdd_algae) then
         do n = 1, ncat
              if (aicen(n) .gt. puny) then
                 hin = vicen(n)/aicen(n)
                 hbri= fbri(n)*hin
                 call compute_shortwave_trcr(                 &
                                     bgcNn(:,n),              &
                                     zaeron(:,n),             &
                                     trcrn_bgcsw(:,n),        &
                                     swgrid,       hin,       &
                                     hbri,                    &
                                     igrid,                   &
                                     skl_bgc,      z_tracers  )
                 if (icepack_warnings_aborted(subname)) return
              endif
         enddo
      endif

      if (calc_Tsfc) then
         if (trim(shortwave(1:4)) == 'dEdd') then ! delta Eddington

            call run_dEdd(dt,                           &
                          aicen,        vicen,          &
                          vsnon,        Tsfcn,          &
                          alvln,        apndn,          &
                          hpndn,        ipndn,          &
                          aeron,        &
                          trcrn_bgcsw,                  &
                          TLAT,         TLON,           &
                          calendar_type,days_per_year,  &
                          nextsw_cday,  yday,           &
                          sec,          &
                          swvdr,        swvdf,          &
                          swidr,        swidf,          &
                          coszen,       fsnow,          &
                          alvdrn,       alvdfn,         &
                          alidrn,       alidfn,         &
                          fswsfcn,      fswintn,        &
                          swuvrdr=swuvrdr,              &
                          swuvrdf=swuvrdf,              &
                          swpardr=swpardr,              &
                          swpardf=swpardf,              &
                          fswthrun=fswthrun,            &
                          fswthrun_vdr=fswthrun_vdr,    &
                          fswthrun_vdf=fswthrun_vdf,    &
                          fswthrun_idr=fswthrun_idr,    &
                          fswthrun_idf=fswthrun_idf,    &
                          fswthrun_uvrdr=fswthrun_uvrdr,&
                          fswthrun_uvrdf=fswthrun_uvrdf,&
                          fswthrun_pardr=fswthrun_pardr,&
                          fswthrun_pardf=fswthrun_pardf,&
                          fswpenln=fswpenln,            &
                          Sswabsn=Sswabsn,              &
                          Iswabsn=Iswabsn,              &
                          albicen=albicen,              &
                          albsnon=albsnon,              &
                          albpndn=albpndn,              &
                          apeffn=apeffn,                &
                          snowfracn=snowfracn,          &
                          dhsn=dhsn,                    &
                          ffracn=ffracn,                &
                          rsnow=rsnow,                  &
                          l_print_point=l_print_point,  &
                          initonly=initonly)
            if (icepack_warnings_aborted(subname)) return

         elseif (trim(shortwave(1:4)) == 'ccsm') then

            call shortwave_ccsm3(aicen,      vicen,      &
                                 vsnon,                  &
                                 Tsfcn,                  &
                                 swvdr,      swvdf,      &
                                 swidr,      swidf,      &
                                 albedo_type,            &
                                 albicev,    albicei,    &
                                 albsnowv,   albsnowi,   &
                                 ahmax,                  &
                                 alvdrn,     alidrn,     &
                                 alvdfn,     alidfn,     &
                                 fswsfcn,    fswintn,    &
                                 swuvrdr=swuvrdr,        &
                                 swuvrdf=swuvrdf,        &
                                 swpardr=swpardr,        &
                                 swpardf=swpardf,        &
                                 fswthrun=fswthrun,      &
                                 fswthrun_vdr=fswthrun_vdr,&
                                 fswthrun_vdf=fswthrun_vdf,&
                                 fswthrun_idr=fswthrun_idr,&
                                 fswthrun_idf=fswthrun_idf,&
                                 fswthrun_uvrdr=fswthrun_uvrdr,&
                                 fswthrun_uvrdf=fswthrun_uvrdf,&
                                 fswthrun_pardr=fswthrun_pardr,&
                                 fswthrun_pardf=fswthrun_pardf,&
                                 fswpenl=fswpenln,       &
                                 Iswabs=Iswabsn,         &
                                 Sswabs=Sswabsn,         &
                                 albin=albicen,          &
                                 albsn=albsnon,          &
                                 coszen=coszen)
            if (icepack_warnings_aborted(subname)) return

         else

            call icepack_warnings_add(subname//' ERROR: shortwave '//trim(shortwave)//' unknown')
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            return

         endif   ! shortwave

      else    ! .not. calc_Tsfc

      ! Calculate effective pond area for HadGEM

         if (tr_pond_topo) then
            do n = 1, ncat
               apeffn(n) = c0
               if (aicen(n) > puny) then
               ! Lid effective if thicker than hp1
                 if (apndn(n)*aicen(n) > puny .and. ipndn(n) < hp1) then
                    apeffn(n) = apndn(n)
                 else
                    apeffn(n) = c0
                 endif
                 if (apndn(n) < puny) apeffn(n) = c0
               endif
            enddo  ! ncat

         endif ! tr_pond_topo

         ! Initialize for safety
         do n = 1, ncat
            alvdrn  (n) = c0
            alidrn  (n) = c0
            alvdfn  (n) = c0
            alidfn  (n) = c0
            fswsfcn (n) = c0
            fswintn (n) = c0
            fswthrun(n) = c0
         enddo   ! ncat
         Iswabsn  (:,:) = c0
         Sswabsn  (:,:) = c0

      endif    ! calc_Tsfc

      first_call = .false.

      end subroutine icepack_step_radiation

!=======================================================================

      ! Delta-Eddington solution expressions

!=======================================================================

      real(kind=dbl_kind) function alpha(w,uu,gg,e)

      real(kind=dbl_kind), intent(in) :: w, uu, gg, e

      alpha = p75*w*uu*((c1 + gg*(c1-w))/(c1 - e*e*uu*uu))

      end function alpha

!=======================================================================

      real(kind=dbl_kind) function agamm(w,uu,gg,e)

      real(kind=dbl_kind), intent(in) :: w, uu, gg, e

      agamm = p5*w*((c1 + c3*gg*(c1-w)*uu*uu)/(c1-e*e*uu*uu))

      end function agamm

!=======================================================================

      real(kind=dbl_kind) function n(uu,et)

      real(kind=dbl_kind), intent(in) :: uu, et

      n = ((uu+c1)*(uu+c1)/et) - ((uu-c1)*(uu-c1)*et)

      end function n

!=======================================================================

      real(kind=dbl_kind) function u(w,gg,e)

      real(kind=dbl_kind), intent(in) :: w, gg, e

      u = c1p5*(c1 - w*gg)/e

      end function u

!=======================================================================

      real(kind=dbl_kind) function el(w,gg)

      real(kind=dbl_kind), intent(in) :: w, gg

      el = sqrt(c3*(c1-w)*(c1 - w*gg))

      end function el

!=======================================================================

      real(kind=dbl_kind) function taus(w,f,t)

      real(kind=dbl_kind), intent(in) :: w, f, t

      taus = (c1 - w*f)*t

      end function taus

!=======================================================================

      real(kind=dbl_kind) function omgs(w,f)

      real(kind=dbl_kind), intent(in) :: w, f

      omgs = (c1 - f)*w/(c1 - w*f)

      end function omgs

!=======================================================================

      real(kind=dbl_kind) function asys(gg,f)

      real(kind=dbl_kind), intent(in) :: gg, f

      asys = (gg - f)/(c1 - f)

      end function asys

!=======================================================================
! --- Begin 5 band dEdd subroutine ---
! Evaluate snow/ice/ponded ice inherent optical properties (IOPs), and
! then calculate the multiple scattering solution by calling solution_dEdd.
!
! author:  Bruce P. Briegleb, NCAR
!   2013:  E Hunke merged with NCAR version
!   2018:  Cheng Dang merged with SNICAR 5-band snow and aersols IOPs, UC Irvine
!
! Note by Cheng Dang 2018:
! This subroutine kept the existing delta-eddington adding-doubling (-ad)
! method, snow and sea ice layer sturcture, and most of the code structures
! of subroutine compute_dEdd_3bd, with major changes listed below to merge
! current snow treatments in SNICAR Model
! 1. The shortwave radiative transfer properties of snow-covered sea ice are
!    calculated for 5 bands (1 visible and 4 near-IR) defined in SNICAR.
! 2. The reflection/absorption/transmission of direct and diffuse shortwave
!    incidents are calculated separately to remove the snow grain adjustment
!    in subroutine compute_dEdd_3bd.
! 3. The albedo and absorption of snow-covered sea ice are adjusted when the
!    solar zenith angle is above 75 degrees.
! 4. Comments given in subroutine compute_dEdd_3bd are all kept in this subroutine
!    with modifications for the changes above.
!
! Justification and explanation of these changes can be found in
! Dang, C., Zender, C. S., and Flanner, M. G.: Intercomparison and improvement
! of two-stream shortwave radiative transfer schemes in Earth system models
! for a unified treatment of cryospheric surfaces, The Cryosphere, 13,
! 2325-2343, https://doi.org/10.5194/tc-13-2325-2019, 2019.

      subroutine compute_dEdd_5bd(                          &
                      klev,   klevp,   zbio,   fnidr,  coszen,  &
                      swvdr,  swvdf,   swidr,  swidf,  srftyp,  &
                      hs,     rhosnw,  rsnw,   hi,     hp,      &
                      fi,     aero_mp, alvdr,  alvdf,           &
                      alidr,  alidf,   fswsfc, fswint, fswthru, &
                      fswthru_vdr,     fswthru_vdf,             &
                      fswthru_idr,     fswthru_idf,             &
                      Sswabs, Iswabs,  fswpenl )

      integer (kind=int_kind), intent(in) :: &
         klev  , & ! number of radiation layers - 1
         klevp     ! number of radiation interfaces - 1
                   ! (0 layer is included also)

      real (kind=dbl_kind), intent(in) :: &
         fnidr , & ! fraction of direct to total down flux in nir
         coszen, & ! cosine solar zenith angle
         swvdr , & ! shortwave down at surface, visible, direct  (W/m^2)
         swvdf , & ! shortwave down at surface, visible, diffuse (W/m^2)
         swidr , & ! shortwave down at surface, near IR, direct  (W/m^2)
         swidf     ! shortwave down at surface, near IR, diffuse (W/m^2)

      integer (kind=int_kind), intent(in) :: &
         srftyp    ! surface type over ice: (0=air, 1=snow, 2=pond)

      real (kind=dbl_kind), intent(in) :: &
         hs        ! snow thickness (m)

      real (kind=dbl_kind), dimension (:), intent(in) :: &
         rhosnw, & ! snow density in snow layer (kg/m3)
         rsnw  , & ! snow grain radius in snow layer (m)
         zbio  , & ! zaerosol + chla shortwave tracers kg/m^3
         aero_mp   ! aerosol mass path in kg/m2

      real (kind=dbl_kind), intent(in) :: &
         hi    , & ! ice thickness (m)
         hp    , & ! pond depth (m)
         fi        ! snow/bare ice fractional coverage (0 to 1)

      real (kind=dbl_kind), intent(inout) :: &
         alvdr , & ! visible, direct, albedo (fraction)
         alvdf , & ! visible, diffuse, albedo (fraction)
         alidr , & ! near-ir, direct, albedo (fraction)
         alidf , & ! near-ir, diffuse, albedo (fraction)
         fswsfc, & ! SW absorbed at snow/bare ice/pondedi ice surface (W m-2)
         fswint, & ! SW interior absorption (below surface, above ocean,W m-2)
         fswthru   ! SW through snow/bare ice/ponded ice into ocean (W m-2)

      real (kind=dbl_kind), intent(inout) :: &
         fswthru_vdr, & ! vis dir SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_vdf, & ! vis dif SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_idr, & ! nir dir SW through snow/bare ice/ponded ice into ocean (W m-2)
         fswthru_idf    ! nir dif SW through snow/bare ice/ponded ice into ocean (W m-2)

      real (kind=dbl_kind), dimension (:), intent(inout) :: &
         fswpenl, & ! visible SW entering ice layers (W m-2)
         Sswabs , & ! SW absorbed in snow layer (W m-2)
         Iswabs     ! SW absorbed in ice layer (W m-2)

!-----------------------------------------------------------------------
! Set up optical property profiles, based on snow, sea ice and ponded
! ice IOPs from:
!
! Briegleb, B. P., and B. Light (2007): A Delta-Eddington Multiple
!    Scattering Parameterization for Solar Radiation in the Sea Ice
!    Component of the Community Climate System Model, NCAR Technical
!    Note  NCAR/TN-472+STR  February 2007
!
! Computes column Delta-Eddington radiation solution for specific
! surface type: either snow over sea ice, bare sea ice, or ponded sea ice.
!
! Divides solar spectrum into 3 intervals: 0.2-0.7, 0.7-1.19, and
! 1.19-5.0 micro-meters. The latter two are added (using an assumed
! partition of incident shortwave in the 0.7-5.0 micro-meter band between
! the 0.7-1.19 and 1.19-5.0 micro-meter band) to give the final output
! of 0.2-0.7 visible and 0.7-5.0 near-infrared albedos and fluxes.
!
! Specifies vertical layer optical properties based on input snow depth,
! density and grain radius, along with ice and pond depths, then computes
! layer by layer Delta-Eddington reflectivity, transmissivity and combines
! layers (done by calling routine solution_dEdd). Finally, surface albedos
! and internal fluxes/flux divergences are evaluated.
!
!  Description of the level and layer index conventions. This is
!  for the standard case of one snow layer and four sea ice layers.
!
!  Please read the following; otherwise, there is 99.9% chance you
!  will be confused about indices at some point in time........ :)
!
!  CICE4.0 snow treatment has one snow layer above the sea ice. This
!  snow layer has finite heat capacity, so that surface absorption must
!  be distinguished from internal. The Delta-Eddington solar radiation
!  thus adds extra surface scattering layers to both snow and sea ice.
!  Note that in the following, we assume a fixed vertical layer structure
!  for the radiation calculation. In other words, we always have the
!  structure shown below for one snow and four sea ice layers, but for
!  ponded ice the pond fills "snow" layer 1 over the sea ice, and for
!  bare sea ice the top layers over sea ice are treated as transparent air.
!
!  SSL = surface scattering layer for either snow or sea ice
!  DL  = drained layer for sea ice immediately under sea ice SSL
!  INT = interior layers for sea ice below the drained layer.
!
!  Notice that the radiation level starts with 0 at the top. Thus,
!  the total number radiation layers is klev+1, where klev is the
!  sum of nslyr, the number of CCSM snow layers, and nilyr, the
!  number of CCSM sea ice layers, plus the sea ice SSL:
!  klev = 1 + nslyr + nilyr
!
!  For the standard case illustrated below, nslyr=1, nilyr=4,
!  and klev=6, with the number of layer interfaces klevp=klev+1.
!  Layer interfaces are the surfaces on which reflectivities,
!  transmissivities and fluxes are evaluated.
!
!  CCSM3 Sea Ice Model            Delta-Eddington Solar Radiation
!                                     Layers and Interfaces
!                             Layer Index             Interface Index
!    ---------------------            ---------------------  0
!                                  0  \\\   snow SSL    \\\
!       snow layer 1                  ---------------------  1
!                                  1    rest of snow layer
!    +++++++++++++++++++++            +++++++++++++++++++++  2
!                                  2  \\\ sea ice SSL   \\\
!      sea ice layer 1                ---------------------  3
!                                  3      sea ice  DL
!    ---------------------            ---------------------  4
!
!      sea ice layer 2             4      sea ice INT
!
!    ---------------------            ---------------------  5
!
!      sea ice layer 3             5      sea ice INT
!
!    ---------------------            ---------------------  6
!
!      sea ice layer 4             6      sea ice INT
!
!    ---------------------            ---------------------  7
!
! When snow lies over sea ice, the radiation absorbed in the
! snow SSL is used for surface heating, and that in the rest
! of the snow layer for its internal heating. For sea ice in
! this case, all of the radiant heat absorbed in both the
! sea ice SSL and the DL are used for sea ice layer 1 heating.
!
! When pond lies over sea ice, and for bare sea ice, all of the
! radiant heat absorbed within and above the sea ice SSL is used
! for surface heating, and that absorbed in the sea ice DL is
! used for sea ice layer 1 heating.
!
! Basically, vertical profiles of the layer extinction optical depth (tau),
! single scattering albedo (w0) and asymmetry parameter (g) are required over
! the klev+1 layers, where klev+1 = 2 + nslyr + nilyr. All of the surface type
! information and snow/ice iop properties are evaulated in this routine, so
! the tau,w0,g profiles can be passed to solution_dEdd for multiple scattering
! evaluation. Snow, bare ice and ponded ice iops are contained in data arrays
! in this routine.
!
!-----------------------------------------------------------------------

      ! local variables

      integer (kind=int_kind) :: &
         k       , & ! level index
         ns      , & ! spectral index
         nr      , & ! index for grain radius tables
         ki      , & ! index for internal absorption
         km      , & ! k starting index for snow, sea ice internal absorption
         kp      , & ! k+1 or k+2 index for snow, sea ice internal absorption
         ksrf    , & ! level index for surface absorption
         ksnow   , & ! level index for snow density and grain size
         kii         ! level starting index for sea ice (nslyr+1)

      real (kind=dbl_kind) :: &
         avdr    , & ! visible albedo, direct   (fraction)
         avdf    , & ! visible albedo, diffuse  (fraction)
         aidr    , & ! near-ir albedo, direct   (fraction)
         aidf        ! near-ir albedo, diffuse  (fraction)

      real (kind=dbl_kind) :: &
         fsfc    , & ! shortwave absorbed at snow/bare ice/ponded ice surface (W m-2)
         fint    , & ! shortwave absorbed in interior (W m-2)
         fthru   , & ! shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruvdr, & ! vis dir shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruvdf, & ! vis dif shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruidr, & ! nir dir shortwave through snow/bare ice/ponded ice to ocean (W/m^2)
         fthruidf    ! nir dif shortwave through snow/bare ice/ponded ice to ocean (W/m^2)

      real (kind=dbl_kind), dimension(nslyr) :: &
         Sabs        ! shortwave absorbed in snow layer (W m-2)

      real (kind=dbl_kind), dimension(nilyr) :: &
         Iabs        ! shortwave absorbed in ice layer (W m-2)

      real (kind=dbl_kind), dimension(nilyr+1) :: &
         fthrul      ! shortwave through to ice layers (W m-2)

      real (kind=dbl_kind), parameter :: &
         cp67 = 0.67_dbl_kind, & ! nir band weight parameter
         cp33 = 0.33_dbl_kind, & ! nir band weight parameter
         cp78 = 0.78_dbl_kind, & ! nir band weight parameter
         cp22 = 0.22_dbl_kind, & ! nir band weight parameter
         cp01 = 0.01_dbl_kind    ! for ocean visible albedo

      real (kind=dbl_kind), dimension (0:klev) :: &
         tau     , & ! layer extinction optical depth
         w0      , & ! layer single scattering albedo
         g           ! layer asymmetry parameter

      ! following arrays are defined at model interfaces; 0 is the top of the
      ! layer above the sea ice; klevp is the sea ice/ocean interface.
      real (kind=dbl_kind), dimension (0:klevp) :: &
         trndir  , & ! solar beam down transmission from top
         trntdr  , & ! total transmission to direct beam for layers above
         trndif  , & ! diffuse transmission to diffuse beam for layers above
         rupdir  , & ! reflectivity to direct radiation for layers below
         rupdif  , & ! reflectivity to diffuse radiation for layers below
         rdndif      ! reflectivity to diffuse radiation for layers above

      real (kind=dbl_kind), dimension (0:klevp) :: &
         dfdir   , & ! down-up flux at interface due to direct beam at top surface
         dfdif       ! down-up flux at interface due to diffuse beam at top surface

      real (kind=dbl_kind) :: &
         refk    , & ! interface k multiple scattering term
         delr    , & ! snow grain radius interpolation parameter
      ! inherent optical properties (iop) for snow
         Qs      , & ! Snow extinction efficiency
         ks      , & ! Snow mass extinction coefficient (m^2/kg)
         ws      , & ! Snow single scattering albedo
         gs          ! Snow asymmetry parameter

!      real (kind=dbl_kind), dimension(nslyr) :: &
!         frsnw       ! snow grain radius in snow layer * adjustment factor (m)

      real (kind=dbl_kind), dimension(0:klev) :: &
         dzk         ! layer thickness

      real (kind=dbl_kind) :: &
         dz      , & ! snow, sea ice or pond water layer thickness
         dz_ssl  , & ! snow or sea ice surface scattering layer thickness
         fs          ! scaling factor to reduce (nilyr<4) or increase (nilyr>4) DL
                     ! extinction coefficient to maintain DL optical depth constant
                     ! with changing number of sea ice layers, to approximately
                     ! conserve computed albedo for constant physical depth of sea
                     ! ice when the number of sea ice layers vary

      real (kind=dbl_kind) :: &
         sig     , & ! scattering coefficient for tuning
         kabs    , & ! absorption coefficient for tuning
         sigp        ! modified scattering coefficient for tuning

      real (kind=dbl_kind) :: &
         albodr  , & ! spectral ocean albedo to direct rad
         albodf      ! spectral ocean albedo to diffuse rad

      ! for melt pond transition to bare sea ice for small pond depths
      real (kind=dbl_kind) :: &
         sig_i   , & ! ice scattering coefficient (/m)
         sig_p   , & ! pond scattering coefficient (/m)
         kext        ! weighted extinction coefficient (/m)

      ! aerosol optical properties from Mark Flanner, 26 June 2008
      ! order assumed: hydrophobic black carbon, hydrophilic black carbon,
      ! four dust aerosols by particle size range:
      ! dust1(.05-0.5 micron), dust2(0.5-1.25 micron),
      ! dust3(1.25-2.5 micron), dust4(2.5-5.0 micron)
      ! spectral bands same as snow/sea ice: (0.3-0.7 micron, 0.7-1.19 micron
      ! and 1.19-5.0 micron in wavelength)

      integer (kind=int_kind) :: &
         na , n                    ! aerosol index

      real (kind=dbl_kind) :: &
         taer    , & ! total aerosol extinction optical depth
         waer    , & ! total aerosol single scatter albedo
         gaer    , & ! total aerosol asymmetry parameter
         swdr    , & ! shortwave down at surface, direct  (W/m^2)
         swdf    , & ! shortwave down at surface, diffuse (W/m^2)
         rnilyr  , & ! 1/real(nilyr)
         rnslyr  , & ! 1/real(nslyr)
         rns     , & ! real(ns)
         tmp_0, tmp_ks, tmp_kl ! temporary variables

      integer(kind=int_kind), dimension(0:klev) :: &
         k_bcini , & ! index
         k_bcins , & ! = 2 hardwired
         k_bcexs     ! = 2 hardwired

      real(kind=dbl_kind)::  &
         tmp_gs, tmp1  ! temporary variables

      real (kind=dbl_kind), parameter :: &
         fr_max = 1.00_dbl_kind, & ! snow grain adjustment factor max
         fr_min = 0.80_dbl_kind, & ! snow grain adjustment factor min
      ! tuning parameters
      ! ice and pond scat coeff fractional change for +- one-sigma in albedo
         fp_ice = 0.15_dbl_kind, & ! ice fraction of scat coeff for + stn dev in alb
         fm_ice = 0.15_dbl_kind, & ! ice fraction of scat coeff for - stn dev in alb
         fp_pnd = 2.00_dbl_kind, & ! ponded ice fraction of scat coeff for + stn dev in alb
         fm_pnd = 0.50_dbl_kind    ! ponded ice fraction of scat coeff for - stn dev in alb

      real (kind=dbl_kind),  parameter :: &   ! chla-specific absorption coefficient
         kchl_tab = p01 ! 0.0023-0.0029 Perovich 1993, also 0.0067 m^2 (mg Chl)^-1
                        ! found values of 0.006 to 0.023 m^2/ mg  (676 nm)  Neukermans 2014
                        ! and averages over the 300-700nm of 0.0075 m^2/mg in ice Fritsen (2011)
                        ! at 440nm values as high as 0.2 m^2/mg in under ice bloom (Balch 2014)
                        ! Grenfell 1991 uses 0.004 (m^2/mg) which is (0.0078 * spectral weighting)
                        ! chlorophyll mass extinction cross section (m^2/mg chla)

      real (kind=dbl_kind), dimension (nspint_5bd) :: &
         wghtns_5bd_dfs        , & ! spectral weights for diffuse incident
         wghtns_5bd_drc            ! spectral weights for direct incident

      ! snow grain single-scattering properties for
      ! direct (drc) and diffuse (dfs) shortwave incidents
      ! local variable names, point to table data
      ! TODO use variable names in ice_shortwave_data directly
      real (kind=dbl_kind), pointer, dimension(:,:) :: & ! Model SNICAR snow SSP
         asm_prm_ice_drc       , & ! snow asymmetry factor (cos(theta))
         asm_prm_ice_dfs       , & ! snow asymmetry factor (cos(theta))
         ss_alb_ice_drc        , & ! snow single scatter albedo (fraction)
         ss_alb_ice_dfs        , & ! snow single scatter albedo (fraction)
         ext_cff_mss_ice_drc   , & ! snow mass extinction cross section (m2/kg)
         ext_cff_mss_ice_dfs       ! snow mass extinction cross section (m2/kg)

      ! FUTURE-WORK: update 5-band sea ice iops when avalible
      real (kind=dbl_kind), dimension (nspint_5bd) :: &  ! for ice only
         ki_ssl_5bd       , & ! Surface-scattering-layer ice extinction coefficient (/m)
         wi_ssl_5bd       , & ! Surface-scattering-layer ice single scattering albedo
         gi_ssl_5bd       , & ! Surface-scattering-layer ice asymmetry parameter
         ki_dl_5bd        , & ! Drained-layer ice extinction coefficient (/m)
         wi_dl_5bd        , & ! Drained-layer ice single scattering albedo
         gi_dl_5bd        , & ! Drained-layer ice asymmetry parameter
         ki_int_5bd       , & ! Interior-layer ice extinction coefficient (/m)
         wi_int_5bd       , & ! Interior-layer ice single scattering albedo
         gi_int_5bd           ! Interior-layer ice asymmetry parameter

      ! 5-band aersol data
      real (kind=dbl_kind), dimension(nspint_5bd, 0:klev) :: &
         kabs_chl_5bd    , & ! absorption coefficient for chlorophyll (/m)
         tzaer_5bd       , & ! total aerosol extinction optical depth
         wzaer_5bd       , & ! total aerosol single scatter albedo
         gzaer_5bd           ! total aerosol asymmetry parameter

      ! index
      integer (kind=int_kind) :: &
         nsky               ! sky = 1 (2) for direct (diffuse) downward SW incident

      ! temporary variables used to assign variables for direct/diffuse incident
      ! based on snicar 5 band IOPs
      real (kind=dbl_kind), dimension (0:klevp) :: &
         dfdir_snicar   , & ! down-up flux at interface due to direct beam at top surface
         dfdif_snicar   , & ! down-up flux at interface due to diffuse beam at top surface
         rupdir_snicar  , & ! reflectivity to direct radiation for layers below
         rupdif_snicar      ! reflectivity to diffuse radiation for layers above

      ! solar zenith angle parameters
      real (kind=dbl_kind), parameter :: &
         sza_a0 =  0.085730_dbl_kind , &
         sza_a1 = -0.630883_dbl_kind , &
         sza_a2 =  1.303723_dbl_kind , &
         sza_b0 =  1.467291_dbl_kind , &
         sza_b1 = -3.338043_dbl_kind , &
         sza_b2 =  6.807489_dbl_kind , &
         mu_75  =  0.2588_dbl_kind       ! cos(75 degrees)

      real (kind=dbl_kind) :: &
         sza_c1       , & ! parameter for high sza adjustment
         sza_c0       , & ! parameter for high sza adjustment
         sza_factor   , & ! parameter for high sza adjustment
         mu0

      character(len=*),parameter :: subname='(compute_dEdd_5bd)'

!-----------------------------------------------------------------------
! Initialize and tune bare ice/ponded ice iops

      ! copy/point to table data for local names
      asm_prm_ice_drc => ssp_sasymmdr
      asm_prm_ice_dfs => ssp_sasymmdf
      ss_alb_ice_drc => ssp_snwalbdr
      ss_alb_ice_dfs => ssp_snwalbdf
      ext_cff_mss_ice_drc => ssp_snwextdr
      ext_cff_mss_ice_dfs => ssp_snwextdf

      k_bcini(:) = c0
      k_bcins(:) = c0
      k_bcexs(:) = c0

      rnilyr = c1/real(nilyr,kind=dbl_kind)
      rnslyr = c1/real(nslyr,kind=dbl_kind)
      kii = nslyr + 1

      ! initialize albedos and fluxes to 0
      fthrul            = c0
      Iabs              = c0
      kabs_chl_5bd(:,:) = c0
      tzaer_5bd   (:,:) = c0
      wzaer_5bd   (:,:) = c0
      gzaer_5bd   (:,:) = c0

      avdr     = c0
      avdf     = c0
      aidr     = c0
      aidf     = c0
      fsfc     = c0
      fint     = c0
      fthru    = c0
      fthruvdr = c0
      fthruvdf = c0
      fthruidr = c0
      fthruidf = c0

      ! spectral weights - 3 bands
      ! this section of code is kept for future mearge between 5band and 3 band
      ! subroutines
      ! weights 2 (0.7-1.19 micro-meters) and 3 (1.19-5.0 micro-meters)
      ! are chosen based on 1D calculations using ratio of direct to total
      ! near-infrared solar (0.7-5.0 micro-meter) which indicates clear/cloudy
      ! conditions: more cloud, the less 1.19-5.0 relative to the
      ! 0.7-1.19 micro-meter due to cloud absorption.
!      wghtns(1) = c1
!      wghtns(2) = cp67 + (cp78-cp67)*(c1-fnidr)
!      wghtns(3) = cp33 + (cp22-cp33)*(c1-fnidr)
!      wghtns(3) = c1 - wghtns(2)

      ! spectral weights - 5 bands
      ! direct beam incident
      ! add-local-variable
      wghtns_5bd_drc(1) = c1
      wghtns_5bd_drc(2) = 0.49352158521175_dbl_kind
      wghtns_5bd_drc(3) = 0.18099494230665_dbl_kind
      wghtns_5bd_drc(4) = 0.12094898498813_dbl_kind
      wghtns_5bd_drc(5) = c1-(wghtns_5bd_drc(2)+wghtns_5bd_drc(3)+wghtns_5bd_drc(4))

      ! diffuse incident
      wghtns_5bd_dfs(1) = c1
      wghtns_5bd_dfs(2) = 0.58581507618433_dbl_kind
      wghtns_5bd_dfs(3) = 0.20156903770812_dbl_kind
      wghtns_5bd_dfs(4) = 0.10917889346386_dbl_kind
      wghtns_5bd_dfs(5) = c1-(wghtns_5bd_dfs(2)+wghtns_5bd_dfs(3)+wghtns_5bd_dfs(4))

      do k = 1, nslyr
        !frsnw(k) = (fr_max*fnidr + fr_min*(c1-fnidr))*rsnw(k)
         Sabs(k) = c0
      enddo

      ! layer thicknesses
      ! snow
      dz = hs*rnslyr
      ! for small enough snow thickness, ssl thickness half of top snow layer
!ech: note this is highly resolution dependent!
      dzk(0) = min(hs_ssl, dz/c2)
      dzk(1) = dz - dzk(0)
      if (nslyr > 1) then
         do k = 2, nslyr
            dzk(k) = dz
         enddo
      endif

      ! ice
      dz = hi*rnilyr
      ! empirical reduction in sea ice ssl thickness for ice thinner than 1.5m;
      ! factor of 30 gives best albedo comparison with limited observations
      dz_ssl = hi_ssl
!ech: note hardwired parameters
!         if( hi < 1.5_dbl_kind ) dz_ssl = hi/30._dbl_kind
      dz_ssl = min(hi_ssl, hi/30._dbl_kind)
      ! set sea ice ssl thickness to half top layer if sea ice thin enough
!ech: note this is highly resolution dependent!
      dz_ssl = min(dz_ssl, dz/c2)

      dzk(kii)   = dz_ssl
      dzk(kii+1) = dz - dz_ssl
      if (kii+2 <= klev) then
         do k = kii+2, klev
            dzk(k) = dz
         enddo
      endif

      ! adjust sea ice iops with tuning parameters; tune only the
      ! scattering coefficient by factors of R_ice, R_pnd, where
      ! R values of +1 correspond approximately to +1 sigma changes in albedo, and
      ! R values of -1 correspond approximately to -1 sigma changes in albedo
      ! Note: the albedo change becomes non-linear for R values > +1 or < -1
      if( R_ice >= c0 ) then
        do ns = 1, nspint_5bd
          sigp           = ki_ssl_mn_5bd(ns)*wi_ssl_mn_5bd(ns)*(c1+fp_ice*R_ice)
          ki_ssl_5bd(ns) = sigp+ki_ssl_mn_5bd(ns)*(c1-wi_ssl_mn_5bd(ns))
          wi_ssl_5bd(ns) = sigp/ki_ssl_5bd(ns)
          gi_ssl_5bd(ns) = gi_ssl_mn_5bd(ns)

          sigp           = ki_dl_mn_5bd(ns)*wi_dl_mn_5bd(ns)*(c1+fp_ice*R_ice)
          ki_dl_5bd(ns)  = sigp+ki_dl_mn_5bd(ns)*(c1-wi_dl_mn_5bd(ns))
          wi_dl_5bd(ns)  = sigp/ki_dl_5bd(ns)
          gi_dl_5bd(ns)  = gi_dl_mn_5bd(ns)

          sigp           = ki_int_mn_5bd(ns)*wi_int_mn_5bd(ns)*(c1+fp_ice*R_ice)
          ki_int_5bd(ns) = sigp+ki_int_mn_5bd(ns)*(c1-wi_int_mn_5bd(ns))
          wi_int_5bd(ns) = sigp/ki_int_5bd(ns)
          gi_int_5bd(ns) = gi_int_mn_5bd(ns)
        enddo
      else !if( R_ice < c0 ) then
        do ns = 1, nspint_5bd
          sigp           = ki_ssl_mn_5bd(ns)*wi_ssl_mn_5bd(ns)*(c1+fm_ice*R_ice)
          sigp           = max(sigp, c0)
          ki_ssl_5bd(ns) = sigp+ki_ssl_mn_5bd(ns)*(c1-wi_ssl_mn_5bd(ns))
          wi_ssl_5bd(ns) = sigp/ki_ssl_5bd(ns)
          gi_ssl_5bd(ns) = gi_ssl_mn_5bd(ns)

          sigp           = ki_dl_mn_5bd(ns)*wi_dl_mn_5bd(ns)*(c1+fm_ice*R_ice)
          sigp           = max(sigp, c0)
          ki_dl_5bd(ns)  = sigp+ki_dl_mn_5bd(ns)*(c1-wi_dl_mn_5bd(ns))
          wi_dl_5bd(ns)  = sigp/ki_dl_5bd(ns)
          gi_dl_5bd(ns)  = gi_dl_mn_5bd(ns)

          sigp           = ki_int_mn_5bd(ns)*wi_int_mn_5bd(ns)*(c1+fm_ice*R_ice)
          sigp           = max(sigp, c0)
          ki_int_5bd(ns) = sigp+ki_int_mn_5bd(ns)*(c1-wi_int_mn_5bd(ns))
          wi_int_5bd(ns) = sigp/ki_int_5bd(ns)
          gi_int_5bd(ns) = gi_int_mn_5bd(ns)
        enddo
      endif          ! adjust ice iops

      ! use srftyp to determine interface index of surface absorption
      ksrf = 1 ! snow covered sea ice

      if (tr_bgc_N .and. dEdd_algae) then ! compute kabs_chl for chlorophyll
          do k = 0, klev
             kabs_chl_5bd(1,k) = kchl_tab*zbio(nlt_chl_sw+k)
          enddo
      else
            k = klev
            kabs_chl_5bd(1,k) = kalg*(0.50_dbl_kind/dzk(k))
      endif

      if (modal_aero) then
         do k = 0, klev
            if (k < nslyr+1) then ! define indices for snow layer
               ! use top rsnw, rhosnw for snow ssl and rest of top layer
               ! Cheng: note that aerosol IOPs are related to snow grain radius.
               ! CICE adjusted snow grain radius rsnw to frsnw, while for
               ! SNICAR there is no need, the tmp_gs is therefore calculated
               ! differently from code in subroutine compute_dEdd
               ksnow = max(k,1)
               tmp_gs = rsnw(ksnow)   ! use rsnw not frsnw

               ! grain size index
               ! works for 25 < snw_rds < 1625 um:
               if (tmp_gs < 125._dbl_kind) then
                  tmp1 = tmp_gs/50._dbl_kind
                  k_bcini(k) = nint(tmp1)
               elseif (tmp_gs < 175._dbl_kind) then
                  k_bcini(k) = 2
               else
                  tmp1 = (tmp_gs/250._dbl_kind) + c2
                  k_bcini(k) = nint(tmp1)
               endif
            else                  ! use the largest snow grain size for ice
               k_bcini(k) = 8
            endif
            ! Set index corresponding to BC effective radius.  Here,
            ! asssume constant BC effective radius of 100nm
            ! (corresponding to index 2)
            k_bcins(k) = 2 ! hardwired
            k_bcexs(k) = 2 ! hardwired

            ! check bounds
            if (k_bcini(k) < 1)  k_bcini(k) = 1
            if (k_bcini(k) > 8)  k_bcini(k) = 8
!            if (k_bcins(k) < 1)  k_bcins(k) = 1   ! hardwired
!            if (k_bcins(k) > 10) k_bcins(k) = 10
!            if (k_bcexs(k) < 1)  k_bcexs(k) = 1
!            if (k_bcexs(k) > 10) k_bcexs(k) = 10
         enddo   ! k

         if (tr_zaero .and. dEdd_algae) then ! compute kzaero for chlorophyll
         do n = 1, n_zaero
            if (n == 1) then ! interstitial BC
               do k = 0, klev
               do ns = 1, nspint_5bd   ! not weighted by aice
                  tzaer_5bd(ns,k) = tzaer_5bd  (ns,k) &
                                  + kaer_bc_5bd(ns,k_bcexs(k)) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  wzaer_5bd(ns,k) = wzaer_5bd  (ns,k) &
                                  + kaer_bc_5bd(ns,k_bcexs(k)) &
                                  * waer_bc_5bd(ns,k_bcexs(k)) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  gzaer_5bd(ns,k) = gzaer_5bd  (ns,k) &
                                  + kaer_bc_5bd(ns,k_bcexs(k)) &
                                  * waer_bc_5bd(ns,k_bcexs(k)) &
                                  * gaer_bc_5bd(ns,k_bcexs(k)) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
               enddo
               enddo
            elseif (n==2) then ! within-ice BC
               do k = 0, klev
               do ns = 1, nspint_5bd
                  tzaer_5bd(ns,k) = tzaer_5bd  (ns,k) &
                                  + kaer_bc_5bd(ns,k_bcins(k)) &
                                  *   bcenh_5bd(ns,k_bcins(k),k_bcini(k)) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  wzaer_5bd(ns,k) = wzaer_5bd  (ns,k) &
                                  + kaer_bc_5bd(ns,k_bcins(k)) &
                                  * waer_bc_5bd(ns,k_bcins(k)) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  gzaer_5bd(ns,k) = gzaer_5bd  (ns,k) &
                                  + kaer_bc_5bd(ns,k_bcins(k)) &
                                  * waer_bc_5bd(ns,k_bcins(k)) &
                                  * gaer_bc_5bd(ns,k_bcins(k)) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
               enddo
               enddo
            else                ! dust
               do k = 0, klev
               do ns = 1, nspint_5bd   ! not weighted by aice
                  tzaer_5bd(ns,k) = tzaer_5bd(ns,k) &
                                  + kaer_5bd (ns,n) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  wzaer_5bd(ns,k) = wzaer_5bd(ns,k) &
                                  + kaer_5bd (ns,n) &
                                  * waer_5bd (ns,n) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  gzaer_5bd(ns,k) = gzaer_5bd(ns,k) &
                                  + kaer_5bd (ns,n) &
                                  * waer_5bd (ns,n) &
                                  * gaer_5bd (ns,n) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
               enddo  ! nspint
               enddo  ! k
            endif     ! n
         enddo        ! n_zaero
         endif        ! tr_zaero and dEdd_algae

      else  ! Bulk aerosol treatment
         if (tr_zaero .and. dEdd_algae) then ! compute kzaero for chlorophyll
         do n = 1, n_zaero          ! multiply by aice?
            do k = 0, klev
               do ns = 1, nspint_5bd   ! not weighted by aice
                  tzaer_5bd(ns,k) = tzaer_5bd(ns,k) &
                                  + kaer_5bd (ns,n) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  wzaer_5bd(ns,k) = wzaer_5bd(ns,k) &
                                  + kaer_5bd (ns,n) &
                                  * waer_5bd (ns,n) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
                  gzaer_5bd(ns,k) = gzaer_5bd(ns,k) &
                                  + kaer_5bd (ns,n) &
                                  * waer_5bd (ns,n) &
                                  * gaer_5bd (ns,n) &
                                  * zbio(nlt_zaero_sw(n)+k) * dzk(k)
               enddo  ! nspint
            enddo     ! k
         enddo        ! n
         endif        ! tr_zaero
      endif           ! modal_aero

!-----------------------------------------------------------------------

      ! begin spectral loop
      do ns = 1, nspint_5bd

         ! for snow-covered sea ice, compute 5 bands
         !if( srftyp == 1 ) then
          ! SNICAR-AD major changes
          ! 1. loop through 5bands: do ns = 1, nspint_5bd based on nsky
          ! 2. use snow grain size rsnow, not scaled frsnw
          ! 3. replace $IOPs_tab with $IOPs_snicar
          ! 4. replace wghtns with wghtns_5bd
         do nsky = 1, 2 ! loop for both direct beam and diffuse beam
         if (nsky == 1) then ! direct incident
            do k = 0, nslyr
               ! use top rsnw, rhosnw for snow ssl and rest of top layer
               ksnow = max(k,1)
               if (rsnw(ksnow) <= rsnw_snicar_min) then
                  ks = ext_cff_mss_ice_drc(ns,1)
                  ws = ss_alb_ice_drc     (ns,1)
                  gs = asm_prm_ice_drc    (ns,1)
               elseif (rsnw(ksnow) >= rsnw_snicar_max) then
                  ks = ext_cff_mss_ice_drc(ns,nmbrad_snicar)
                  ws = ss_alb_ice_drc     (ns,nmbrad_snicar)
                  gs = asm_prm_ice_drc    (ns,nmbrad_snicar)
               else
                  ! linear interpolation
                  if (trim(rsnw_datatype) == 'sorted_idelta1') then
                     ! NOTE:  Assumes delta rsnw_snicar_tab is 1 and rsnw_snicar_tab are integers
                     ! This is just for performance, could call shortwave_search
                     nr = ceiling(rsnw(ksnow)) - nint(rsnw_snicar_min) + 1
                  else
                     call shortwave_search(rsnw(ksnow),rsnw_snicar_tab,nr)
                     if (icepack_warnings_aborted(subname)) return
                  endif
                  delr = (rsnw(ksnow)         - rsnw_snicar_tab(nr-1)) &
                       / (rsnw_snicar_tab(nr) - rsnw_snicar_tab(nr-1))
                  ks = ext_cff_mss_ice_drc(ns,nr-1)*(c1-delr) &
                     + ext_cff_mss_ice_drc(ns,nr  )*    delr
                  ws = ss_alb_ice_drc     (ns,nr-1)*(c1-delr) &
                     + ss_alb_ice_drc     (ns,nr  )*    delr
                  gs = asm_prm_ice_drc    (ns,nr-1)*(c1-delr) &
                     + asm_prm_ice_drc    (ns,nr  )*    delr
               endif
               tau(k) = (ks*rhosnw(ksnow) + kabs_chl_5bd(ns,k))*dzk(k)
               w0 (k) =  ks*rhosnw(ksnow) / (ks*rhosnw(ksnow) + kabs_chl_5bd(ns,k)) * ws
               g  (k) = gs
            enddo       ! k
         elseif (nsky == 2) then ! diffuse  incident
            do k = 0, nslyr
               ! use top rsnw, rhosnw for snow ssl and rest of top layer
               ksnow = max(k,1)
               if (rsnw(ksnow) < rsnw_snicar_min) then
                  ks = ext_cff_mss_ice_dfs(ns,1)
                  ws = ss_alb_ice_dfs     (ns,1)
                  gs = asm_prm_ice_dfs    (ns,1)
               elseif (rsnw(ksnow) > rsnw_snicar_max) then
                  ks = ext_cff_mss_ice_dfs(ns,nmbrad_snicar)
                  ws = ss_alb_ice_dfs     (ns,nmbrad_snicar)
                  gs = asm_prm_ice_dfs    (ns,nmbrad_snicar)
               else
                  ! linear interpolation
                  if (trim(rsnw_datatype) == 'sorted_idelta1') then
                     ! NOTE:  delta rsnw_snicar_tab is 1 and rsnw_snicar_tab are integers
                     ! This is just for performance, could call shortwave_search
                     nr = ceiling(rsnw(ksnow)) - nint(rsnw_snicar_min) + 1
                  else
                     call shortwave_search(rsnw(ksnow),rsnw_snicar_tab,nr)
                     if (icepack_warnings_aborted(subname)) return
                  endif
                  delr = (rsnw(ksnow)         - rsnw_snicar_tab(nr-1)) &
                       / (rsnw_snicar_tab(nr) - rsnw_snicar_tab(nr-1))
                  ks = ext_cff_mss_ice_dfs(ns,nr-1)*(c1-delr) &
                     + ext_cff_mss_ice_dfs(ns,nr  )*    delr
                  ws = ss_alb_ice_dfs     (ns,nr-1)*(c1-delr) &
                     + ss_alb_ice_dfs     (ns,nr  )*    delr
                  gs = asm_prm_ice_dfs    (ns,nr-1)*(c1-delr) &
                     + asm_prm_ice_dfs    (ns,nr  )*    delr
               endif
               tau(k) = (ks*rhosnw(ksnow) + kabs_chl_5bd(ns,k))*dzk(k)
               w0 (k) =  ks*rhosnw(ksnow) / (ks*rhosnw(ksnow) + kabs_chl_5bd(ns,k)) * ws
               g  (k) = gs
            enddo       ! k
         endif ! nsky for snow IOPs

         !------------------------------------------------------------------------------

            ! aerosol in snow
            if (tr_zaero .and. dEdd_algae) then
               do k = 0,nslyr
                  g  (k) = (g(k)*w0(k)*tau(k) + gzaer_5bd(ns,k)) / &
                                (w0(k)*tau(k) + wzaer_5bd(ns,k))
                  w0 (k) =      (w0(k)*tau(k) + wzaer_5bd(ns,k)) / &
                                      (tau(k) + tzaer_5bd(ns,k))
                  tau(k) = tau(k) + tzaer_5bd(ns,k)
               enddo
            elseif (tr_aero) then
               k = 0  ! snow SSL
               taer = c0
               waer = c0
               gaer = c0

               do na = 1, 4*n_aero, 4
               if (modal_aero) then
                  if (na == 1) then      ! interstitial BC
                     taer = taer + aero_mp(na)*kaer_bc_5bd(ns,k_bcexs(k))
                     waer = waer + aero_mp(na)*kaer_bc_5bd(ns,k_bcexs(k)) &
                                              *waer_bc_5bd(ns,k_bcexs(k))
                     gaer = gaer + aero_mp(na)*kaer_bc_5bd(ns,k_bcexs(k)) &
                                              *waer_bc_5bd(ns,k_bcexs(k)) &
                                              *gaer_bc_5bd(ns,k_bcexs(k))
                  elseif (na == 5) then ! within-ice BC
                     taer = taer + aero_mp(na)*kaer_bc_5bd(ns,k_bcins(k)) &
                                              *  bcenh_5bd(ns,k_bcins(k),k_bcini(k))
                     waer = waer + aero_mp(na)*kaer_bc_5bd(ns,k_bcins(k)) &
                                              *waer_bc_5bd(ns,k_bcins(k))
                     gaer = gaer + aero_mp(na)*kaer_bc_5bd(ns,k_bcins(k)) &
                                              *waer_bc_5bd(ns,k_bcins(k)) &
                                              *gaer_bc_5bd(ns,k_bcins(k))
                  else                  ! other species (dust)
                     taer = taer + aero_mp(na)*kaer_5bd(ns,(1+(na-1)/4))
                     waer = waer + aero_mp(na)*kaer_5bd(ns,(1+(na-1)/4)) &
                                              *waer_5bd(ns,(1+(na-1)/4))
                     gaer = gaer + aero_mp(na)*kaer_5bd(ns,(1+(na-1)/4)) &
                                              *waer_5bd(ns,(1+(na-1)/4)) &
                                              *gaer_5bd(ns,(1+(na-1)/4))
                  endif
               else
                  taer = taer + aero_mp(na)*kaer_5bd(ns,(1+(na-1)/4))
                  waer = waer + aero_mp(na)*kaer_5bd(ns,(1+(na-1)/4)) &
                                           *waer_5bd(ns,(1+(na-1)/4))
                  gaer = gaer + aero_mp(na)*kaer_5bd(ns,(1+(na-1)/4)) &
                                           *waer_5bd(ns,(1+(na-1)/4)) &
                                           *gaer_5bd(ns,(1+(na-1)/4))
               endif ! modal_aero
               enddo ! na
               g  (k) = (g(k)*w0(k)*tau(k) + gaer) / &
                             (w0(k)*tau(k) + waer)
               w0 (k) =      (w0(k)*tau(k) + waer) / &
                                   (tau(k) + taer)
               tau(k) = tau(k) + taer

               do k = 1, nslyr
                  taer = c0
                  waer = c0
                  gaer = c0
                  do na = 1, 4*n_aero, 4
                  if (modal_aero) then
                     if (na==1) then     ! interstitial BC
                        taer = taer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_5bd(ns,k_bcexs(k))
                        waer = waer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_5bd(ns,k_bcexs(k)) &
                             * waer_bc_5bd(ns,k_bcexs(k))
                        gaer = gaer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_5bd(ns,k_bcexs(k)) &
                             * waer_bc_5bd(ns,k_bcexs(k)) &
                             * gaer_bc_5bd(ns,k_bcexs(k))
                     elseif (na==5) then ! within-ice BC
                        taer = taer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_5bd(ns,k_bcins(k)) &
                             *   bcenh_5bd(ns,k_bcins(k),k_bcini(k))
                        waer = waer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_5bd(ns,k_bcins(k)) &
                             * waer_bc_5bd(ns,k_bcins(k))
                        gaer = gaer + (aero_mp(na+1)*rnslyr) &
                             * kaer_bc_5bd(ns,k_bcins(k)) &
                             * waer_bc_5bd(ns,k_bcins(k)) &
                             * gaer_bc_5bd(ns,k_bcins(k))
                     else                ! other species (dust)
                        taer = taer + (aero_mp(na+1)*rnslyr) &
                             * kaer_5bd(ns,(1+(na-1)/4))
                        waer = waer + (aero_mp(na+1)*rnslyr) &
                             * kaer_5bd(ns,(1+(na-1)/4)) &
                             * waer_5bd(ns,(1+(na-1)/4))
                        gaer = gaer + (aero_mp(na+1)*rnslyr) &
                             * kaer_5bd(ns,(1+(na-1)/4)) &
                             * waer_5bd(ns,(1+(na-1)/4)) &
                             * gaer_5bd(ns,(1+(na-1)/4))
                     endif   ! na
                  else
                     taer = taer + (aero_mp(na+1)*rnslyr) &
                          * kaer_5bd(ns,(1+(na-1)/4))
                     waer = waer + (aero_mp(na+1)*rnslyr) &
                          * kaer_5bd(ns,(1+(na-1)/4)) &
                          * waer_5bd(ns,(1+(na-1)/4))
                     gaer = gaer + (aero_mp(na+1)*rnslyr) &
                          * kaer_5bd(ns,(1+(na-1)/4)) &
                          * waer_5bd(ns,(1+(na-1)/4)) &
                          * gaer_5bd(ns,(1+(na-1)/4))
                  endif       ! modal_aero
                  enddo       ! na
                  g  (k) = (g(k)*w0(k)*tau(k) + gaer) / &
                                (w0(k)*tau(k) + waer)
                  w0 (k) =      (w0(k)*tau(k) + waer) / &
                                      (tau(k) + taer)
                  tau(k) = tau(k) + taer
               enddo       ! k
            endif     ! tr_aero

         ! set optical properties of sea ice

         ! bare or snow-covered sea ice layers
         !if (srftyp <= 1) then
            ! ssl
            k = kii
            tau(k) =                (ki_ssl_5bd(ns) + kabs_chl_5bd(ns,k)) * dzk(k)
            w0 (k) = ki_ssl_5bd(ns)/(ki_ssl_5bd(ns) + kabs_chl_5bd(ns,k)) * wi_ssl_5bd(ns)
            g  (k) = gi_ssl_5bd(ns)
            ! dl
            k = kii + 1
            ! scale dz for dl relative to 4 even-layer-thickness 1.5m case
            fs = p25*real(nilyr,kind=dbl_kind)
            tau(k) =               (ki_dl_5bd(ns) + kabs_chl_5bd(ns,k)) * dzk(k) * fs
            w0 (k) = ki_dl_5bd(ns)/(ki_dl_5bd(ns) + kabs_chl_5bd(ns,k)) * wi_dl_5bd(ns)
            g  (k) = gi_dl_5bd(ns)
            ! int above lowest layer
            if (kii+2 <= klev-1) then
               do k = kii+2, klev-1
                  tau(k) =                (ki_int_5bd(ns) + kabs_chl_5bd(ns,k)) * dzk(k)
                  w0 (k) = ki_int_5bd(ns)/(ki_int_5bd(ns) + kabs_chl_5bd(ns,k)) * wi_int_5bd(ns)
                  g  (k) = gi_int_5bd(ns)
               enddo
            endif
            ! lowest layer
            k = klev
            ! add algae to lowest sea ice layer, visible only:
            kabs = ki_int_5bd(ns)*(c1-wi_int_5bd(ns))
            if (ns == 1) then
               ! total layer absorption optical depth fixed at value
               ! of kalg*0.50m, independent of actual layer thickness
               kabs = kabs + kabs_chl_5bd(ns,k)
            endif
            sig    = ki_int_5bd(ns)*wi_int_5bd(ns)
            tau(k) = (kabs+sig) * dzk(k)
            w0 (k) = sig/(sig+kabs)
            g  (k) = gi_int_5bd(ns)
            ! aerosol in sea ice
            if (tr_zaero .and. dEdd_algae) then
               do k = kii, klev
                  g  (k) = (g(k)*w0(k)*tau(k) + gzaer_5bd(ns,k)) / &
                                (w0(k)*tau(k) + wzaer_5bd(ns,k))
                  w0 (k) =      (w0(k)*tau(k) + wzaer_5bd(ns,k)) / &
                                      (tau(k) + tzaer_5bd(ns,k))
                  tau(k) = tau(k) + tzaer_5bd(ns,k)
               enddo
            elseif (tr_aero) then
               k = kii   ! sea ice SSL
               taer = c0
               waer = c0
               gaer = c0
               do na = 1, 4*n_aero, 4
               if (modal_aero) then
                  if (na==1) then      ! interstitial BC
                     taer = taer + aero_mp(na+2) &
                          * kaer_bc_5bd(ns,k_bcexs(k))
                     waer = waer + aero_mp(na+2) &
                          * kaer_bc_5bd(ns,k_bcexs(k)) &
                          * waer_bc_5bd(ns,k_bcexs(k))
                     gaer = gaer + aero_mp(na+2) &
                          * kaer_bc_5bd(ns,k_bcexs(k)) &
                          * waer_bc_5bd(ns,k_bcexs(k)) &
                          * gaer_bc_5bd(ns,k_bcexs(k))
                  elseif (na==5) then  ! within-ice BC
                     taer = taer + aero_mp(na+2) &
                          * kaer_bc_5bd(ns,k_bcins(k)) &
                          *   bcenh_5bd(ns,k_bcins(k),k_bcini(k))
                     waer = waer + aero_mp(na+2) &
                          * kaer_bc_5bd(ns,k_bcins(k)) &
                          * waer_bc_5bd(ns,k_bcins(k))
                     gaer = gaer + aero_mp(na+2) &
                          * kaer_bc_5bd(ns,k_bcins(k)) &
                          * waer_bc_5bd(ns,k_bcins(k)) &
                          * gaer_bc_5bd(ns,k_bcins(k))
                  else                 ! other species (dust)
                     taer = taer + aero_mp(na+2) &
                          * kaer_5bd(ns,(1+(na-1)/4))
                     waer = waer + aero_mp(na+2) &
                          * kaer_5bd(ns,(1+(na-1)/4)) &
                          * waer_5bd(ns,(1+(na-1)/4))
                     gaer = gaer + aero_mp(na+2) &
                          * kaer_5bd(ns,(1+(na-1)/4)) &
                          * waer_5bd(ns,(1+(na-1)/4)) &
                          * gaer_5bd(ns,(1+(na-1)/4))
                  endif
               else      ! bulk
                  taer = taer + aero_mp(na+2) &
                       * kaer_5bd(ns,(1+(na-1)/4))
                  waer = waer + aero_mp(na+2) &
                       * kaer_5bd(ns,(1+(na-1)/4)) &
                       * waer_5bd(ns,(1+(na-1)/4))
                  gaer = gaer + aero_mp(na+2) &
                       * kaer_5bd(ns,(1+(na-1)/4)) &
                       * waer_5bd(ns,(1+(na-1)/4)) &
                       * gaer_5bd(ns,(1+(na-1)/4))
                endif     ! modal_aero
               enddo      ! na
               g  (k) = (g(k)*w0(k)*tau(k) + gaer) / &
                             (w0(k)*tau(k) + waer)
               w0 (k) =      (w0(k)*tau(k) + waer) / &
                                   (tau(k) + taer)
               tau(k) = tau(k) + taer
               do k = kii+1, klev
                  taer = c0
                  waer = c0
                  gaer = c0
                  do na = 1, 4*n_aero, 4
                  if (modal_aero) then
                     if (na==1) then     ! interstitial BC
                        taer = taer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_5bd(ns,k_bcexs(k))
                        waer = waer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_5bd(ns,k_bcexs(k)) &
                             * waer_bc_5bd(ns,k_bcexs(k))
                        gaer = gaer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_5bd(ns,k_bcexs(k)) &
                             * waer_bc_5bd(ns,k_bcexs(k)) &
                             * gaer_bc_5bd(ns,k_bcexs(k))
                     elseif (na==5) then ! within-ice BC
                        taer = taer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_5bd(ns,k_bcins(k)) &
                             *   bcenh_5bd(ns,k_bcins(k),k_bcini(k))
                        waer = waer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_5bd(ns,k_bcins(k)) &
                             * waer_bc_5bd(ns,k_bcins(k))
                        gaer = gaer + (aero_mp(na+3)*rnilyr) &
                             * kaer_bc_5bd(ns,k_bcins(k)) &
                             * waer_bc_5bd(ns,k_bcins(k)) &
                             * gaer_bc_5bd(ns,k_bcins(k))
                     else                ! other species (dust)
                        taer = taer + (aero_mp(na+3)*rnilyr) &
                             * kaer_5bd(ns,(1+(na-1)/4))
                        waer = waer + (aero_mp(na+3)*rnilyr) &
                             * kaer_5bd(ns,(1+(na-1)/4)) &
                             * waer_5bd(ns,(1+(na-1)/4))
                        gaer = gaer + (aero_mp(na+3)*rnilyr) &
                             * kaer_5bd(ns,(1+(na-1)/4)) &
                             * waer_5bd(ns,(1+(na-1)/4)) &
                             * gaer_5bd(ns,(1+(na-1)/4))
                     endif
                  else       !bulk
                     taer = taer + (aero_mp(na+3)*rnilyr) &
                          * kaer_5bd(ns,(1+(na-1)/4))
                     waer = waer + (aero_mp(na+3)*rnilyr) &
                          * kaer_5bd(ns,(1+(na-1)/4)) &
                          * waer_5bd(ns,(1+(na-1)/4))
                     gaer = gaer + (aero_mp(na+3)*rnilyr) &
                          * kaer_5bd(ns,(1+(na-1)/4)) &
                          * waer_5bd(ns,(1+(na-1)/4)) &
                          * gaer_5bd(ns,(1+(na-1)/4))
                  endif       ! modal_aero
                  enddo       ! na
                  g  (k) = (g(k)*w0(k)*tau(k) + gaer) / &
                                (w0(k)*tau(k) + waer)
                  w0 (k) =      (w0(k)*tau(k) + waer) / &
                                      (tau(k) + taer)
                  tau(k) = tau(k) + taer
               enddo ! k
            endif    ! tr_aero

! ---------------------------------------------------------------------------

         ! set reflectivities for ocean underlying sea ice
         ! if ns == 1 (visible), albedo is 0.1, else, albedo is zero
         rns = real(ns-1, kind=dbl_kind)
         albodr = cp01 * (c1 - min(rns, c1))
         albodf = cp01 * (c1 - min(rns, c1))

         ! layer input properties now completely specified: tau, w0, g,
         ! albodr, albodf; now compute the Delta-Eddington solution
         ! reflectivities and transmissivities for each layer; then,
         ! combine the layers going downwards accounting for multiple
         ! scattering between layers, and finally start from the
         ! underlying ocean and combine successive layers upwards to
         ! the surface; see comments in solution_dEdd for more details.

         call solution_dEdd (                                              &
                coszen,     srftyp,     klev,       klevp,                 &
                tau,        w0,         g,          albodr,     albodf,    &
                trndir,     trntdr,     trndif,     rupdir,     rupdif,    &
                rdndif)
         if (icepack_warnings_aborted(subname)) return

         ! the interface reflectivities and transmissivities required
         ! to evaluate interface fluxes are returned from solution_dEdd;
         ! now compute up and down fluxes for each interface, using the
         ! combined layer properties at each interface:
         !
         !              layers       interface
         !
         !       ---------------------  k
         !                 k
         !       ---------------------

         do k = 0, klevp
            ! interface scattering
            refk = c1/(c1 - rdndif(k)*rupdif(k))
            ! dir tran ref from below times interface scattering, plus diff
            ! tran and ref from below times interface scattering
            ! fdirup(k) = (trndir(k)*rupdir(k) + &
            !                 (trntdr(k)-trndir(k))  &
            !                 *rupdif(k))*refk
            ! dir tran plus total diff trans times interface scattering plus
            ! dir tran with up dir ref and down dif ref times interface scattering
            ! fdirdn(k) = trndir(k) + (trntdr(k) &
            !               - trndir(k) + trndir(k)  &
            !               *rupdir(k)*rdndif(k))*refk
            ! diffuse tran ref from below times interface scattering
            ! fdifup(k) = trndif(k)*rupdif(k)*refk
            ! diffuse tran times interface scattering
            ! fdifdn(k) = trndif(k)*refk

            ! dfdir = fdirdn - fdirup
            dfdir(k) = trndir(k) &
                        + (trntdr(k)-trndir(k)) * (c1 - rupdif(k)) * refk &
                        -  trndir(k)*rupdir(k)  * (c1 - rdndif(k)) * refk
            if (dfdir(k) < puny) dfdir(k) = c0 !echmod necessary?
            ! dfdif = fdifdn - fdifup
            dfdif(k) = trndif(k) * (c1 - rupdif(k)) * refk
            if (dfdif(k) < puny) dfdif(k) = c0 !echmod necessary?
         enddo       ! k

         ! note that because the snow IOPs for diffuse and direct incidents
         ! are different, the snow albedo needs to be calculated twice for
         ! direct incident and diffuse incident respectively
         if (nsky == 1) then ! direct beam (keep the direct beam results)
            do k = 0, klevp
               dfdir_snicar(k)  = dfdir(k)
               rupdir_snicar(k) = rupdir(k)
            enddo
         elseif (nsky == 2) then ! diffuse (keep the diffuse incident results)
            do k = 0, klevp
               dfdif_snicar(k)  = dfdif(k)
               rupdif_snicar(k) = rupdif(k)
            enddo
         endif
         enddo ! end direct/diffuse nsky loop ------------------------------------

         ! calculate final surface albedos and fluxes
         ! all absorbed flux above ksrf is included in surface absorption
         if (ns == 1) then      ! visible
            swdr   = swvdr
            swdf   = swvdf
            avdr   = rupdir_snicar(0)
            avdf   = rupdif_snicar(0)
            tmp_0  = dfdir_snicar(0    )*swdr + dfdif_snicar(0    )*swdf
            tmp_ks = dfdir_snicar(ksrf )*swdr + dfdif_snicar(ksrf )*swdf
            tmp_kl = dfdir_snicar(klevp)*swdr + dfdif_snicar(klevp)*swdf

            ! for layer biology: save visible only
            do k = nslyr+2, klevp ! Start at DL layer of ice after SSL scattering
               fthrul(k-nslyr-1) = dfdir_snicar(k)*swdr + dfdif_snicar(k)*swdf
            enddo

            fsfc  = fsfc  + tmp_0  - tmp_ks
            fint  = fint  + tmp_ks - tmp_kl
            fthru = fthru + tmp_kl
            fthruvdr = fthruvdr + dfdir_snicar(klevp)*swdr
            fthruvdf = fthruvdf + dfdif_snicar(klevp)*swdf

            ! if snow covered ice, set snow internal absorption; else, Sabs=0
            if (srftyp == 1) then
               ki = 0
               do k = 1, nslyr
                  ! skip snow SSL, since SSL absorption included in the surface
                  ! absorption fsfc above
                  km  = k
                  kp  = km + 1
                  ki  = ki + 1
                  Sabs(ki) = Sabs(ki) &
                           +  dfdir_snicar(km)*swdr + dfdif_snicar(km)*swdf &
                           - (dfdir_snicar(kp)*swdr + dfdif_snicar(kp)*swdf)
               enddo       ! k
            endif

            ! complex indexing to insure proper absorptions for sea ice
            ki = 0
            do k = nslyr+2, nslyr+1+nilyr
               ! for bare ice, DL absorption for sea ice layer 1
               km = k
               kp = km + 1
               ! modify for top sea ice layer for snow over sea ice
               if (srftyp == 1) then
                  ! must add SSL and DL absorption for sea ice layer 1
                  if (k == nslyr+2) then
                     km = k  - 1
                     kp = km + 2
                  endif
               endif
               ki = ki + 1
               Iabs(ki) = Iabs(ki) &
                        +  dfdir_snicar(km)*swdr + dfdif_snicar(km)*swdf &
                        - (dfdir_snicar(kp)*swdr + dfdif_snicar(kp)*swdf)
            enddo       ! k

         else ! ns > 1, near IR

            swdr = swidr
            swdf = swidf

            ! let fr2(3,4,5) = alb_2(3,4,5)*swd*wght2(3,4,5)
            ! the ns=2(3,4,5) reflected fluxes respectively,
            ! where alb_2(3,4,5) are the band
            ! albedos, swd = nir incident shortwave flux, and wght2(3,4,5) are
            ! the 2(3,4,5) band weights. thus, the total reflected flux is:
            ! fr = fr2 + fr3 + fr4 + fr5
            !    = alb_2*swd*wght2 + alb_3*swd*wght3 + alb_4*swd*wght4 + alb_5*swd*wght5
            ! hence, the 2,3,4,5 nir band albedo is
            ! alb = fr/swd = alb_2*wght2 + alb_3*wght3 + alb_4*wght4 + alb_5*wght5

            aidr   = aidr + rupdir_snicar(0)*wghtns_5bd_drc(ns)
            aidf   = aidf + rupdif_snicar(0)*wghtns_5bd_dfs(ns)

            tmp_0  = dfdir_snicar(0    )*swdr*wghtns_5bd_drc(ns) &
                   + dfdif_snicar(0    )*swdf*wghtns_5bd_dfs(ns)
            tmp_ks = dfdir_snicar(ksrf )*swdr*wghtns_5bd_drc(ns) &
                   + dfdif_snicar(ksrf )*swdf*wghtns_5bd_dfs(ns)
            tmp_kl = dfdir_snicar(klevp)*swdr*wghtns_5bd_drc(ns) &
                   + dfdif_snicar(klevp)*swdf*wghtns_5bd_dfs(ns)

            fsfc  = fsfc  + tmp_0  - tmp_ks
            fint  = fint  + tmp_ks - tmp_kl
            fthru = fthru + tmp_kl
            fthruidr = fthruidr + dfdir_snicar(klevp)*swdr*wghtns_5bd_drc(ns)
            fthruidf = fthruidf + dfdif_snicar(klevp)*swdf*wghtns_5bd_dfs(ns)

            ! if snow covered ice, set snow internal absorption; else, Sabs=0
            if (srftyp == 1) then
               ki = 0
               do k = 1, nslyr
                  ! skip snow SSL, since SSL absorption included in the surface
                  ! absorption fsfc above
                  km = k
                  kp = km + 1
                  ki = ki + 1
                  Sabs(ki) = Sabs(ki) &
                           + dfdir_snicar(km)*swdr*wghtns_5bd_drc(ns)   &
                           + dfdif_snicar(km)*swdf*wghtns_5bd_dfs(ns)   &
                           - dfdir_snicar(kp)*swdr*wghtns_5bd_drc(ns)   &
                           - dfdif_snicar(kp)*swdf*wghtns_5bd_dfs(ns)
            enddo       ! k
         endif

            ! complex indexing to insure proper absorptions for sea ice
            ki = 0
            do k = nslyr+2, nslyr+1+nilyr
               ! for bare ice, DL absorption for sea ice layer 1
               km = k
               kp = km + 1
               ! modify for top sea ice layer for snow over sea ice
               if (srftyp == 1) then
                  ! must add SSL and DL absorption for sea ice layer 1
                  if (k == nslyr+2) then
                     km = k  - 1
                     kp = km + 2
                  endif
               endif
               ki = ki + 1
               Iabs(ki) = Iabs(ki) &
                        + dfdir_snicar(km)*swdr*wghtns_5bd_drc(ns)   &
                        + dfdif_snicar(km)*swdf*wghtns_5bd_dfs(ns)   &
                        - dfdir_snicar(kp)*swdr*wghtns_5bd_drc(ns)   &
                        - dfdif_snicar(kp)*swdf*wghtns_5bd_dfs(ns)
            enddo       ! k
         endif          ! ns
      enddo             ! ns: end spectral loop

      ! solar zenith angle parameterization
      ! calculate the scaling factor for NIR direct albedo if SZA>75 degrees
      sza_factor = c1
      if (srftyp == 1) then
         mu0 = max(coszen, p01)
         if (mu0 < mu_75) then
            sza_c1 = sza_a0 + sza_a1 * mu0 + sza_a2 * mu0**2
            sza_c0 = sza_b0 + sza_b1 * mu0 + sza_b2 * mu0**2
            sza_factor = sza_c1 * (log10(rsnw(1)) - 6.0_dbl_kind) + sza_c0
         endif
      endif

      alvdr = avdr
      alvdf = avdf
      alidr = aidr * sza_factor !sza factor is always larger than or equal to 1
      alidf = aidf

      ! accumulate fluxes over bare sea ice

      ! note that we assume the reduced NIR energy absorption by snow
      ! due to corrected snow albedo is absorbed by the snow single
      ! scattering layer only - this is generally true if snow SSL >= 2 cm
      ! by the default model set up:
      !      if snow_depth >= 8 cm, SSL = 4 cm, satisfy
      ! else if snow_depth >= 4 cm, SSL = snow_depth/2 >= 2 cm, satisfy
      ! else    snow_depth < 4 cm, SSL = snow_depth/2, may overcool SSL layer
      fswsfc  = fswsfc  + (fsfc- (sza_factor-c1)*aidr*swidr)*fi
      fswint  = fswint  + fint *fi
      fswthru = fswthru + fthru*fi
      fswthru_vdr = fswthru_vdr + fthruvdr*fi
      fswthru_vdf = fswthru_vdf + fthruvdf*fi
      fswthru_idr = fswthru_idr + fthruidr*fi
      fswthru_idf = fswthru_idf + fthruidf*fi

      do k = 1, nslyr
         Sswabs(k) = Sswabs(k) + Sabs(k)*fi
      enddo

      do k = 1, nilyr
         Iswabs(k) = Iswabs(k) + Iabs(k)*fi
         ! bgc layer
         fswpenl(k) = fswpenl(k) + fthrul(k)*fi
      enddo
      fswpenl(nilyr+1) = fswpenl(nilyr+1) + fthrul(nilyr+1)*fi

      end subroutine compute_dEdd_5bd

!=======================================================================
!     This subroutine searches array for val and returns nr such that
!       array(nr-1) < val <= array(nr)
!     If nr cannot be found, an error is thrown
!     This does NOT check that array is sorted because it would be too expensive,
!     but it must be sorted to work properly.

      subroutine shortwave_search(val,array,nr)

      real (kind=dbl_kind), intent(in) :: &
         val           ! search value

      real (kind=dbl_kind), dimension (:), intent(in) :: &
         array         ! sorted array

      integer (kind=int_kind), intent(out) :: &
         nr            ! index in array >= val

      ! local variables

      integer (kind=int_kind) :: &
         nrcnt,      & ! counter
         nrp,        & ! prior nr
         nrl, nru,   & ! lower and upper search indices
         nrsize        ! size of array

      logical (kind=log_kind) :: &
         found         ! search flag

      character (len=512) :: &
         tmpstr        ! temporary string

      character(len=*),parameter :: subname='(shortwave_search)'


      if (rsnw_datatype(1:6) /= 'sorted') then
         call icepack_warnings_add(subname//' rsnw_datatype not valid: '//trim(rsnw_datatype))
         call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
      endif

      nrsize = size(array)

!debug write(tmpstr,*) "val = ",val
!      call icepack_warnings_add(subname//trim(tmpstr))
!      write(tmpstr,*) "nrsize = ",nrsize
!      call icepack_warnings_add(subname//trim(tmpstr))
!      write(tmpstr,*) "array1 = ",array(1)
!      call icepack_warnings_add(subname//trim(tmpstr))
!      write(tmpstr,*) "arrayn = ",array(nrsize)
!      call icepack_warnings_add(subname//trim(tmpstr))

      if (nrsize > 10) then
         ! binary search
         nrl = 1
         nru = nrsize
         nr = (nrl + nru) / 2
         found = .false.
         nrcnt = 0
         do while (.not.found .and. nrcnt < nrsize)
            nrcnt = nrcnt + 1
            nrp = nr
            if (val > array(nr)) then
               if (val < array(nr+1)) then
                  found = .true.
                  nr = nr + 1
               else
                  nrl = nr + 1
                  nr = (nrl + nru) / 2
               endif
            else
               if (val > array(nr-1)) then
                  found = .true.
               else
                  nru = nr - 1
                  nr = (nrl + nru) / 2
               endif
            endif
!debug       write(tmpstr,*) "iter = ",nrcnt,nrp,nr
!            call icepack_warnings_add(subname//trim(tmpstr))
!            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
         enddo
         if (.not. found) then
            call icepack_warnings_add(subname//' ERROR: binary search failed')
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            return
         endif
      else
         ! linear search
         nr = -1
         do nrcnt = 2,nrsize
            if (val > array(nrcnt-1) .and. val < array(nrcnt)) then
               nr = nrcnt
               exit
            endif
         enddo
         if (nr < 1) then
            call icepack_warnings_add(subname//' ERROR: linear search failed')
            call icepack_warnings_setabort(.true.,__FILE__,__LINE__)
            return
         endif
      endif

      end subroutine shortwave_search

!=======================================================================

      end module icepack_shortwave

!=======================================================================
