############################################################################
#
# This script projects the dewy-pine populations using an individual-based
# model (IBM). The goal is to assess the sensitivity of population dynamics 
# to various vital rates under climate change.
#
# Author: Eva Conquet
#
###########################################################################

###########################################################################
#
# 1. House keeping and loading libraries and data ----
#
###########################################################################

## 1.1. House keeping ----
# -------------------

rm(list = ls())


## 1.2. Loading libraries ----
# -----------------------

library(lubridate)
library(mgcv)
library(crch)
library(snowfall)


## 1.3. Loading data ----
# ------------------

# Dewy-pine data
droso_natural = read.csv("Data/droso_natural.csv")
droso_full = read.csv("Data/dataDroso2022.csv")
droso_full$quadratID = paste(droso_full$transect, droso_full$subQuadrat, sep = "_")

droso_seedbank = read.csv("Data/droso_seedbank_natural.csv")

seeds_per_flower = 9.8


# Vital-rate models
load("Output/Models/Survival_GAM_Natural.RData")
load("Output/Models/Growth_GAM_Natural.RData")
load("Output/Models/FloweringProb_GAM_Natural.RData")
load("Output/Models/NbFlowers_GAM_Natural.RData")
load("Output/Models/SeedlingSize_GAM_Natural.RData")


# Average density per square for covariate standardization
nbSquares = aggregate(quadratID ~ time + site, 
                      data = droso_natural, 
                      function(x) length(unique(x)))
density_per_square = aggregate(abLarge_unscaled ~ quadratID + time + site, 
                               data = droso_natural, 
                               function(x) unique(x))
yearly_density_per_square = aggregate(abLarge_unscaled ~ time + site,
                                      data = density_per_square, 
                                      function(x) sum(x))
yearly_density_per_square$abLarge_unscaled = yearly_density_per_square$abLarge_unscaled/nbSquares$quadratID


# Year- and population-specific climatic variables for covariate standardization
summerT_timeSeries = aggregate(summerT_unscaled ~ time + site, 
                               data = droso_natural, mean)
prevwinterT_timeSeries = aggregate(prevwinterT_unscaled ~ time + site, 
                                   data = droso_natural, mean)
fallR_timeSeries = aggregate(fallR_unscaled ~ time + site, 
                             data = droso_natural, mean)
prevfallR_timeSeries = aggregate(prevfallR_unscaled ~ time + site, 
                                 data = droso_natural, mean)




###########################################################################
#
# 3. Building vital-rate functions ----
#
###########################################################################

## 3.1. Survival ----
# --------------

survival_function = function(size, abLarge, TSFcont, fallR, summerT, year, population){
  
  # Standardize covariates
  size_scaled = (size - mean(droso_natural$size_unscaled, na.rm = T)) / (2 * sd(droso_natural$size_unscaled, na.rm = T))
  abLarge_scaled = (abLarge - mean(yearly_density_per_square$abLarge_unscaled, na.rm = T)) / (2 * sd(yearly_density_per_square$abLarge_unscaled, na.rm = T))
  TSFcont_scaled = (TSFcont - mean(droso_natural$TSFcont_unscaled, na.rm = T)) / (2 * sd(droso_natural$TSFcont_unscaled, na.rm = T))
  fallR_scaled = (fallR - mean(fallR_timeSeries$fallR_unscaled, na.rm = T)) / (2 * sd(fallR_timeSeries$fallR_unscaled, na.rm = T))
  summerT_scaled = (summerT - mean(summerT_timeSeries$summerT_unscaled, na.rm = T)) / (2 * sd(summerT_timeSeries$summerT_unscaled, na.rm = T))
  
  # Calculate survival
  survival = predict(surv_natural, newdata = data.frame(size = size_scaled,
                                                   abLarge = abLarge_scaled,
                                                   TSFcont = TSFcont_scaled,
                                                   fallR = fallR_scaled,
                                                   summerT = summerT_scaled,
                                                   time = year,
                                                   site = population), type = "response")
  
  return(survival)
}


## 3.2. Growth ----
# ------------

growth_function = function(size, abLarge, TSFcont, fallR, year, population){
  
  # Standardize covariates
  size_scaled = (size - mean(droso_natural$size_unscaled, na.rm = T)) / (2 * sd(droso_natural$size_unscaled, na.rm = T))
  abLarge_scaled = (abLarge - mean(yearly_density_per_square$abLarge_unscaled, na.rm = T)) / (2 * sd(yearly_density_per_square$abLarge_unscaled, na.rm = T))
  TSFcont_scaled = (TSFcont - mean(droso_natural$TSFcont_unscaled, na.rm = T)) / (2 * sd(droso_natural$TSFcont_unscaled, na.rm = T))
  fallR_scaled = (fallR - mean(fallR_timeSeries$fallR_unscaled, na.rm = T)) / (2 * sd(fallR_timeSeries$fallR_unscaled, na.rm = T))
  
  # Get parameters from growth model (mean, sd, and degrees of freedom)
  growth_mean = predict(growth_natural, newdata = data.frame(size = size_scaled,
                                                        abLarge = abLarge_scaled,
                                                        TSFcont = TSFcont_scaled,
                                                        fallR = fallR_scaled,
                                                        time = year,
                                                        site = population), type = "response")
  
  growth_sd = family(growth_natural)$getTheta(trans = T)[2] # Using trans = T, the second value of theta is sigma, the standard deviation (https://stats.stackexchange.com/questions/550339/extracting-the-degrees-of-freedom-of-t-distribution-of-a-gam)
  
  growth_df = family(growth_natural)$getTheta(trans = T)[1] # Using trans = T, the first value of theta is nu, the degrees of freedom (https://stats.stackexchange.com/questions/550339/extracting-the-degrees-of-freedom-of-t-distribution-of-a-gam)
  
  return(list(mean = growth_mean, 
              sd = growth_sd, 
              df = growth_df))
}


## 3.3. Seedling sizes ----
# --------------------

seedling_size_function = function(abLarge, prevwinterT, TSFcont, year, population){
  
  # Standardize covariates
  abLarge_scaled = (abLarge - mean(yearly_density_per_square$abLarge_unscaled, na.rm = T)) / (2 * sd(yearly_density_per_square$abLarge_unscaled, na.rm = T))
  prevwinterT_scaled = (prevwinterT - mean(prevwinterT_timeSeries$prevwinterT_unscaled, na.rm = T)) / (2 * sd(prevwinterT_timeSeries$prevwinterT_unscaled, na.rm = T))
  TSFcont_scaled = (TSFcont - mean(droso_natural$TSFcont_unscaled, na.rm = T)) / (2 * sd(droso_natural$TSFcont_unscaled, na.rm = T))
  
  # Get parameters from seedling size model (mean, sd, and degrees of freedom)
  seedling_size_mean = as.numeric(predict(seedlingSize_natural, newdata = data.frame(abLarge = abLarge_scaled,
                                                                                prevwinterT = prevwinterT_scaled,
                                                                                TSFcont = TSFcont_scaled,
                                                                                time = year,
                                                                                site = population), type = "response"))
  
  seedling_size_sd = family(seedlingSize_natural)$getTheta(trans = T)[2] # Using trans = T, the second value of theta is sigma, the standard deviation (https://stats.stackexchange.com/questions/550339/extracting-the-degrees-of-freedom-of-t-distribution-of-a-gam)
  
  seedling_size_df = family(seedlingSize_natural)$getTheta(trans = T)[1] # Using trans = T, the first value of theta is nu, the degrees of freedom (https://stats.stackexchange.com/questions/550339/extracting-the-degrees-of-freedom-of-t-distribution-of-a-gam)
  
  return(list(mean = seedling_size_mean, 
              sd = seedling_size_sd, 
              df = seedling_size_df))
  
}


## 3.4. Flowering probability ----
# ---------------------------

flowering_function = function(size, abLarge, prevwinterT, prevfallR, TSFcont, year, population){
  
  # Standardize covariates
  size_scaled = (size - mean(droso_natural$size_unscaled, na.rm = T)) / (2 * sd(droso_natural$size_unscaled, na.rm = T))
  abLarge_scaled = (abLarge - mean(yearly_density_per_square$abLarge_unscaled, na.rm = T)) / (2 * sd(yearly_density_per_square$abLarge_unscaled, na.rm = T))
  prevwinterT_scaled = (prevwinterT - mean(prevwinterT_timeSeries$prevwinterT_unscaled, na.rm = T)) / (2 * sd(prevwinterT_timeSeries$prevwinterT_unscaled, na.rm = T))
  prevfallR_scaled = (prevfallR - mean(prevfallR_timeSeries$prevfallR_unscaled, na.rm = T)) / (2 * sd(prevfallR_timeSeries$prevfallR_unscaled, na.rm = T))
  TSFcont_scaled = (TSFcont - mean(droso_natural$TSFcont_unscaled, na.rm = T)) / (2 * sd(droso_natural$TSFcont_unscaled, na.rm = T))
  
  # Calculate flowering probability
  flowering = predict(flowering_natural, newdata = data.frame(size = size_scaled,
                                                         abLarge = abLarge_scaled,
                                                         prevwinterT = prevwinterT_scaled,
                                                         prevfallR = prevfallR_scaled,
                                                         TSFcont = TSFcont_scaled,
                                                         time = year,
                                                         site = population), type = "response")
  
  return(flowering)
}


## 3.5. Number of flowers ----
# -----------------------

nbFlowers_function = function(size, prevwinterT, TSFcont, year, population){
  
  # Standardize covariates
  size_scaled = (size - mean(droso_natural$size_unscaled, na.rm = T)) / (2 * sd(droso_natural$size_unscaled, na.rm = T))
  prevwinterT_scaled = (prevwinterT - mean(prevwinterT_timeSeries$prevwinterT_unscaled, na.rm = T)) / (2 * sd(prevwinterT_timeSeries$prevwinterT_unscaled, na.rm = T))
  TSFcont_scaled = (TSFcont - mean(droso_natural$TSFcont_unscaled, na.rm = T)) / (2 * sd(droso_natural$TSFcont_unscaled, na.rm = T))
  
  # Calculate number of flowers
  nbFlowers = predict(nbFlow_natural, newdata = data.frame(size = size_scaled,
                                                      prevwinterT = prevwinterT_scaled,
                                                      TSFcont = TSFcont_scaled,
                                                      time = year,
                                                      site = population), type = "response")
  
  return(nbFlowers)
}


## 3.6. Immediate germination (goCont) ----
# ------------------------------------

goCont_function = function(TSF){
  
  if(TSF %in% seq(0, 5)){
    
    goCont = droso_seedbank$value[which(droso_seedbank$vital_rate == "goCont" &
                                          droso_seedbank$TSF == TSF)]
  }
  
  else{
    
    goCont = droso_seedbank$value[which(droso_seedbank$vital_rate == "goCont" &
                                          droso_seedbank$TSF == 5)]
    
  }
  
  return(goCont)
}


## 3.7. Staying in the seed bank (staySB) ----
# ---------------------------------------

staySB_function = function(TSF){
  
  if(TSF %in% seq(0, 5)){
    
    staySB = droso_seedbank$value[which(droso_seedbank$vital_rate == "staySB" &
                                          droso_seedbank$TSF == TSF)]
    
  }
  
  else{
    
    staySB = droso_seedbank$value[which(droso_seedbank$vital_rate == "staySB" &
                                          droso_seedbank$TSF == 5)]
    
  }
  
  return(staySB)
}

## 3.8. Germinating out of the seed bank (outSB) ----
# ----------------------------------------------

outSB_function = function(TSF){
  
  if(TSF %in% seq(0, 5)){
    
    outSB = droso_seedbank$value[which(droso_seedbank$vital_rate == "outSB" &
                                         droso_seedbank$TSF == TSF)]
    
  }
  
  else{
    
    outSB = droso_seedbank$value[which(droso_seedbank$vital_rate == "outSB" &
                                         droso_seedbank$TSF == 5)]
  }
  
  return(outSB)
}




###########################################################################
#
# 4. Climatic projection data ----
#
###########################################################################

## 4.1. Load climate projection data ----
# ----------------------------------

CanESM5_climate = read.csv("Data/CanESM5_MonthlyClimateProjection_RCP85.csv")
EC_Earth3_climate = read.csv("Data/EC_Earth3_MonthlyClimateProjection_RCP85.csv")
FGOALS_G3_climate = read.csv("Data/FGOALS_G3_MonthlyClimateProjection_RCP85.csv")
GFDL_ESM4_climate = read.csv("Data/GFDL_ESM4_MonthlyClimateProjection_RCP85.csv")
GISS_E2_1_G_climate = read.csv("Data/GISS_E2_1_G_MonthlyClimateProjection_RCP85.csv")
INM_CM4_8_climate = read.csv("Data/INM_CM4_8_MonthlyClimateProjection_RCP85.csv")
IPSL_CM6A_LR_climate = read.csv("Data/IPSL_CM6A_LR_MonthlyClimateProjection_RCP85.csv")
MIROC6_climate = read.csv("Data/MIROC6_MonthlyClimateProjection_RCP85.csv")
MPI_ESM1_2_LR_climate = read.csv("Data/MPI_ESM1_2_LR_MonthlyClimateProjection_RCP85.csv")
MRI_ESM2_0_climate = read.csv("Data/MRI_ESM2_0_MonthlyClimateProjection_RCP85.csv")
NorESM2_MM_climate = read.csv("Data/NorESM2_MM_MonthlyClimateProjection_RCP85.csv")


## 4.2. Format date ----
# -----------------

CanESM5_climate$date = as.Date(paste(CanESM5_climate$year, 
                                     CanESM5_climate$month, "1", 
                                     sep = "-"))
EC_Earth3_climate$date = as.Date(paste(EC_Earth3_climate$year, 
                                       EC_Earth3_climate$month, "1", 
                                       sep = "-"))
FGOALS_G3_climate$date = as.Date(paste(FGOALS_G3_climate$year, 
                                       FGOALS_G3_climate$month, "1", 
                                       sep = "-"))
GFDL_ESM4_climate$date = as.Date(paste(GFDL_ESM4_climate$year, 
                                       GFDL_ESM4_climate$month, "1", 
                                       sep = "-"))
GISS_E2_1_G_climate$date = as.Date(paste(GISS_E2_1_G_climate$year, 
                                         GISS_E2_1_G_climate$month, "1", 
                                         sep = "-"))
INM_CM4_8_climate$date = as.Date(paste(INM_CM4_8_climate$year,
                                       INM_CM4_8_climate$month, "1", 
                                       sep = "-"))
IPSL_CM6A_LR_climate$date = as.Date(paste(IPSL_CM6A_LR_climate$year, 
                                          IPSL_CM6A_LR_climate$month, "1", 
                                          sep = "-"))
MIROC6_climate$date = as.Date(paste(MIROC6_climate$year,
                                    MIROC6_climate$month, "1", 
                                    sep = "-"))
MPI_ESM1_2_LR_climate$date = as.Date(paste(MPI_ESM1_2_LR_climate$year, 
                                           MPI_ESM1_2_LR_climate$month, "1",
                                           sep = "-"))
MRI_ESM2_0_climate$date = as.Date(paste(MRI_ESM2_0_climate$year,
                                        MRI_ESM2_0_climate$month, "1", 
                                        sep = "-"))
NorESM2_MM_climate$date = as.Date(paste(NorESM2_MM_climate$year, 
                                        NorESM2_MM_climate$month, "1", 
                                        sep = "-"))


## 4.3. Create climatic projection dataframe ----
# ------------------------------------------

# Empty dataframe
proj_clim = expand.grid(year = seq(2021, 2100),
                        population = unique(droso_natural$site),
                        model = c("CanESM5", "EC_Earth3", "FGOALS_G3", "GFDL_ESM4",
                                  "GISS_E2_1_G", "INM_CM4_8", "IPSL_CM6A_LR", "MIROC6",
                                  "MPI_ESM1_2_LR", "MRI_ESM2_0", "NorESM2_MM"))
proj_clim$date = as.Date(paste(proj_clim$year, "5", "1", sep = "-"))
proj_clim$summerT_unscaled = NA
proj_clim$prevwinterT_unscaled = NA
proj_clim$fallR_unscaled = NA
proj_clim$prevfallR_unscaled = NA

climate_cov = c("summerT", "prevwinterT", "fallR", "prevfallR")

for(cov in climate_cov){
  
  for(r in 1:nrow(proj_clim)){
    
    model_data = get(paste0(proj_clim$model[r], "_climate"))
    
    if(cov == "summerT"){
      
      start_date = (proj_clim$date[r]) %m+% months(0)
      end_date = (proj_clim$date[r]) %m+% months(5)
      
      proj_clim$summerT_unscaled[r] = mean(model_data$mean_max_temp[model_data$date >= start_date & 
                                                                      model_data$date < end_date & 
                                                                      model_data$pop %in% proj_clim$population[r]])
      
    }
    
    else if(cov == "prevwinterT"){
      
      start_date = (proj_clim$date[r]) %m+% months(-4)
      end_date = (proj_clim$date[r]) %m+% months(0)
      
      proj_clim$prevwinterT_unscaled[r] = mean(model_data$mean_max_temp[model_data$date >= start_date & 
                                                                        model_data$date < end_date & 
                                                                        model_data$pop %in% proj_clim$population[r]])
      
    }
    
    else if(cov == "fallR"){
      
      start_date = (proj_clim$date[r]) %m+% months(4)
      end_date = (proj_clim$date[r]) %m+% months(7)
      
      proj_clim$fallR_unscaled[r] = sum(model_data$cum_rain[model_data$date >= start_date & 
                                                            model_data$date < end_date & 
                                                            model_data$pop %in% proj_clim$population[r]])
    }
    
    else if(cov == "prevfallR"){
      
      start_date = (proj_clim$date[r]) %m+% months(-8)
      end_date = (proj_clim$date[r]) %m+% months(-5)
      
      proj_clim$prevfallR_unscaled[r] = sum(model_data$cum_rain[model_data$date >= start_date & 
                                                                model_data$date < end_date & 
                                                                model_data$pop %in% proj_clim$population[r]])
    }
  }
}


# Standardize climatic variables and remove values outside the
# observed range
proj_clim$summerT = proj_clim$prevwinterT = proj_clim$fallR = proj_clim$prevfallR = proj_clim$prevwinterR =  NA

# Next summer temperature
proj_clim$summerT_unscaled[which(proj_clim$summerT_unscaled > max(droso_natural$summerT_unscaled))] = max(droso_natural$summerT_unscaled)
proj_clim$summerT_unscaled[which(proj_clim$summerT_unscaled < min(droso_natural$summerT_unscaled))] = min(droso_natural$summerT_unscaled)

proj_clim$summerT = (proj_clim$summerT_unscaled - mean(summerT_timeSeries$summerT_unscaled, na.rm = T)) / (2 * sd(summerT_timeSeries$summerT_unscaled, na.rm = T))

# Previous winter temperature
proj_clim$prevwinterT_unscaled[which(proj_clim$prevwinterT_unscaled > max(droso_natural$prevwinterT_unscaled))] = max(droso_natural$prevwinterT_unscaled)
proj_clim$prevwinterT_unscaled[which(proj_clim$prevwinterT_unscaled < min(droso_natural$prevwinterT_unscaled))] = min(droso_natural$prevwinterT_unscaled)

proj_clim$prevwinterT = (proj_clim$prevwinterT_unscaled - mean(prevwinterT_timeSeries$prevwinterT_unscaled, na.rm = T)) / (2 * sd(prevwinterT_timeSeries$prevwinterT_unscaled, na.rm = T))

# Next fall rainfall
proj_clim$fallR_unscaled[which(proj_clim$fallR_unscaled > max(droso_natural$fallR_unscaled))] = max(droso_natural$fallR_unscaled)
proj_clim$fallR_unscaled[which(proj_clim$fallR_unscaled < min(droso_natural$fallR_unscaled))] = min(droso_natural$fallR_unscaled)

proj_clim$fallR = (proj_clim$fallR_unscaled - mean(fallR_timeSeries$fallR_unscaled, na.rm = T)) / (2 * sd(fallR_timeSeries$fallR_unscaled, na.rm = T))

# Previous fall rainfall
proj_clim$prevfallR_unscaled[which(proj_clim$prevfallR_unscaled > max(droso_natural$prevfallR_unscaled))] = max(droso_natural$prevfallR_unscaled)
proj_clim$prevfallR_unscaled[which(proj_clim$prevfallR_unscaled < min(droso_natural$prevfallR_unscaled))] = min(droso_natural$prevfallR_unscaled)

proj_clim$prevfallR = (proj_clim$prevfallR_unscaled - mean(prevfallR_timeSeries$prevfallR_unscaled, na.rm = T)) / (2 * sd(prevfallR_timeSeries$prevfallR_unscaled, na.rm = T))




###########################################################################
#
# 5. Individual-based model function ----
#
###########################################################################

## 5.1. Timestep projection function ----
# ----------------------------------

ibm_sim = function(n_sim,              # Number of simulations
                   n_years,            # Number of years per simulation
                   sim = sim,          # Current simulation number
                   first_year,         # Year of start of simulation
                   years_RE,           # Observed years available for random year effect
                   seedbank_size,      # Initial seedbank size
                   recruitCap,         # Maximum number of recruits per quadrat
                   max_nbFlowers,      # Maximum number of flowers per individual
                   population,         # Population ID
                   climate_model,      # Name of climate model
                   vital_rate,         # Name of vital rate to perturb (i.e. use climate-change values while all other vital rates use control values)
                   envS,               # Markov matrix for post-fire states sequence
                   ibm_data,           # Previously stored projection data
                   data_initial,       # Initial dataset
                   seedbank_initial){  # Initial seedbank data
  
  # Year sequence for climatic variable predictions and random effects
  
  years_obs = c(first_year, sample(years_RE, n_years-1, replace = T)) # Random effects
  
  years = c(first_year, seq(first_year + 1, first_year + n_years - 2)) # Climatic variable predictions
  # sequence of 30 years
  # for the climate-change scenario  
  
  
  # Empty files to hold results 
  
  log_meanChangeAboveground = rep(NA, n_years)
  log_lambda = rep(NA, n_years)
  seedbank_size_vec = rep(NA, n_years)
  seedbank_size_vec[1] = seedbank_size
  pop_density = vector(mode = "list", length = n_years)
  extinction = 0
  
  sim_data = NULL # Full individual data across whole simulation
  
  data_droso = data_initial[which(data_initial$time == years_obs[1]), ] # Yearly individual data
  
  # Add density data 
  data_droso$abLarge_unscaled = apply(data_droso, 1, function(x) nrow(data_droso[which(data_droso$quadratID == x[2] & data_droso$size_unscaled > 4.5), ]))
  pop_density[[1]] = aggregate(abLarge_unscaled ~ quadratID, data = data_droso, mean, na.rm = T)$abLarge_unscaled
  
  # Seedbank seeds data
  data_SB = seedbank_initial
  
  # Assign seeds in initial seedbank to quadrats
  data_SB$quadratID = sample(unique(data_droso$quadratID), size = seedbank_size, 
                             replace = T, 
                             prob = as.numeric(table(data_droso$quadratID)/sum(table(data_droso$quadratID))))
  
  # Add density data to seedbank
  data_SB$abLarge_unscaled = apply(data_SB, 1, function(x) nrow(data_droso[which(data_droso$quadratID == x[2] & data_droso$size_unscaled > 4.5), ]))
  
  # Highest seed ID (format Seed_XXX) to give names to new seeds 
  max_seed_ID = max(as.numeric(unlist(lapply(strsplit(data_SB$ID, split = "_"), function(x) x[2]))))
  
  time_sim = 1 # Timestep
  sim_data = rbind(sim_data, cbind(data_droso, time_sim)) # Merge full individual 
  # data and yearly individual 
  # data with timestep info
  
  # Building the post-fire environmental states succession vector
  states = unique(data_initial$TSFcont_unscaled)
  states[which(states > 4)] = 4
  
  # First state depends on the observed TSF in the first year of the projection
  env.at.t = ifelse(unique(data_droso$TSFcont_unscaled) < 4, 
                    unique(data_droso$TSFcont_unscaled),
                    4)
  
  states[1] = env.at.t + 1
  
  # Simulate post-fire state sequence
  for(x in 2:n_years){
    
    states[x] = env.at.t = sample(ncol(envS), 1, pr = envS[, env.at.t])
  }
  
  # Format post-fire state sequence
  states = states - 1
  TSFcont = rep(NA, length(states))
  
  # Fill in TSFcont sequence
  TSFcont[1] = unique(data_droso$TSFcont_unscaled)
  
  fires = c(which(states == 0), (length(states) + 1))
  TSFcont[fires[-length(fires)]] = 0
  
  for(f in 1:length(fires)){
    
    TSFnext = min(which(is.na(TSFcont)))
    
    TSFcont[TSFnext:(fires[f]-1)] = seq(TSFcont[TSFnext-1] + 1, 
                                        length.out = length(TSFcont[TSFnext:(fires[f]-1)]))
    
    if(fires[f] == n_years) break
    
  }
  
  
  # Project the population
  for(i in 2:n_years){
    
    # print(paste("Year", i))
    
    seed_produced = data_droso[0, ] # Data on seeds produced by reproducing plants
    seed_produced[1, ] = NA
    seed_produced$goCont = seed_produced$goSB = seed_produced$staySB = seed_produced$outSB = seed_produced$rownb = NA
    seed_produced = seed_produced[0, ]
    
    if(states[i-1] == 0){ # If fire, all aboveground individuals burn
      
      data_droso = data_droso[0, ]
    } 
    
    
    
    # Get new years for projected climatic variables and random effects
    year = years[i-1]
    year_obs = years_obs[i-1]
    
    if(nrow(data_droso) > 0) data_droso$time = (year + 1)
    
    
    ### CLIMATIC VARIABLES ###
    
    if(vital_rate == "survival"){
      
      # Climate variables for survival are future climate-change values
      summerT_survival = unique(proj_clim$summerT_unscaled[which(proj_clim$population == population &
                                                                 proj_clim$year == year &
                                                                 proj_clim$model == climate_model)])
      fallR_survival = unique(proj_clim$fallR_unscaled[which(proj_clim$population == population &
                                                             proj_clim$year == year &
                                                             proj_clim$model == climate_model)])
      
      # Climate variables for other vital rates are current (control) values
      fallR_growth = unique(droso_natural$fallR_unscaled[which(droso_natural$site == population &
                                                                     droso_natural$time == year_obs)])
      
      prevwinterT_flowering = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                    droso_natural$time == year_obs)])
      
      prevfallR_flowering = unique(droso_natural$prevfallR_unscaled[which(droso_natural$site == population &
                                                                                droso_natural$time == year_obs)])
      
      prevwinterT_nbFlowers = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                      droso_natural$time == year_obs)])
      
      prevwinterT_seedlingSize = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                       droso_natural$time == year_obs)])
      
    }
    
    else if(vital_rate == "growth"){
      
      # Climate variables for growth are future climate-change values
      fallR_growth = unique(proj_clim$fallR_unscaled[which(proj_clim$population == population &
                                                           proj_clim$year == year &
                                                           proj_clim$model == climate_model)])
      
      # Climate variables for other vital rates are current (control) values
      summerT_survival = unique(droso_natural$summerT_unscaled[which(droso_natural$site == population &
                                                                             droso_natural$time == year_obs)])
      fallR_survival = unique(droso_natural$fallR_unscaled[which(droso_natural$site == population &
                                                                         droso_natural$time == year_obs)])
      
      prevwinterT_flowering = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                      droso_natural$time == year_obs)])
      
      prevfallR_flowering = unique(droso_natural$prevfallR_unscaled[which(droso_natural$site == population &
                                                                                  droso_natural$time == year_obs)])
      
      prevwinterT_nbFlowers = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                      droso_natural$time == year_obs)])
      
      prevwinterT_seedlingSize = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                         droso_natural$time == year_obs)])
    }
    
    else if(vital_rate == "flowering"){
      
      # Climate variables for flowering probability are future climate-change values
      prevwinterT_flowering = unique(proj_clim$prevwinterT_unscaled[which(proj_clim$population == population &
                                                                          proj_clim$year == year &
                                                                          proj_clim$model == climate_model)])
      
      prevfallR_flowering = unique(proj_clim$prevfallR_unscaled[which(proj_clim$population == population &
                                                                      proj_clim$year == year &
                                                                      proj_clim$model == climate_model)])
      
      # Climate variables for other vital rates are current (control) values
      summerT_survival = unique(droso_natural$summerT_unscaled[which(droso_natural$site == population &
                                                                             droso_natural$time == year_obs)])
      fallR_survival = unique(droso_natural$fallR_unscaled[which(droso_natural$site == population &
                                                                         droso_natural$time == year_obs)])
      
      fallR_growth = unique(droso_natural$fallR_unscaled[which(droso_natural$site == population &
                                                                     droso_natural$time == year_obs)])
      
      prevwinterT_nbFlowers = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                    droso_natural$time == year_obs)])
      
      prevwinterT_seedlingSize = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                         droso_natural$time == year_obs)])
    }
    
    else if(vital_rate == "nbFlowers"){
      
      # Climate variables for number of flowers are future climate-change values
      prevwinterT_nbFlowers = unique(proj_clim$prevwinterT_unscaled[which(proj_clim$population == population &
                                                                          proj_clim$year == year &
                                                                          proj_clim$model == climate_model)])
      
      # Climate variables for other vital rates are current (control) values
      summerT_survival = unique(droso_natural$summerT_unscaled[which(droso_natural$site == population &
                                                                             droso_natural$time == year_obs)])
      fallR_survival = unique(droso_natural$fallR_unscaled[which(droso_natural$site == population &
                                                                         droso_natural$time == year_obs)])
      
      fallR_growth = unique(droso_natural$fallR_unscaled[which(droso_natural$site == population &
                                                                 droso_natural$time == year_obs)])
      
      prevwinterT_flowering = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                droso_natural$time == year_obs)])
      
      prevfallR_flowering = unique(droso_natural$prevfallR_unscaled[which(droso_natural$site == population &
                                                                            droso_natural$time == year_obs)])
      
      prevwinterT_seedlingSize = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                         droso_natural$time == year_obs)])
    }
    
    else if(vital_rate == "seedlingSize"){
      
      # Climate variables for seedling size are future climate-change values
      prevwinterT_seedlingSize = unique(proj_clim$prevwinterT_unscaled[which(proj_clim$population == population &
                                                                               proj_clim$year == year &
                                                                               proj_clim$model == climate_model)])
      
      # Climate variables for other vital rates are current (control) values
      summerT_survival = unique(droso_natural$summerT_unscaled[which(droso_natural$site == population &
                                                                             droso_natural$time == year_obs)])
      fallR_survival = unique(droso_natural$fallR_unscaled[which(droso_natural$site == population &
                                                                         droso_natural$time == year_obs)])
      
      fallR_growth = unique(droso_natural$fallR_unscaled[which(droso_natural$site == population &
                                                                       droso_natural$time == year_obs)])
      
      prevwinterT_flowering = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                      droso_natural$time == year_obs)])
      
      prevfallR_flowering = unique(droso_natural$prevfallR_unscaled[which(droso_natural$site == population &
                                                                                  droso_natural$time == year_obs)])
      
      prevwinterT_nbFlowers = unique(droso_natural$prevwinterT_unscaled[which(droso_natural$site == population &
                                                                                      droso_natural$time == year_obs)])
    }
    
    
    ### TSF AND NUMBER OF SQUARES ###
    
    TSFcat = states[i-1]
    TSF = TSFcont[i-1]
    if(nrow(data_droso) > 0) data_droso$TSFcont_unscaled = TSF
    
    if(states[i-1] != 0){
      
      if(states[i-1] > 1){
        
        if(nrow(data_droso) > 0){
          
          ### FLOWERING ###
          
          data_droso$flowering = NA
          
          data_droso$flowering = rbinom(n = nrow(data_droso), 
                                        flowering_function(size = data_droso$size_unscaled,
                                                           abLarge = data_droso$abLarge_unscaled,
                                                           prevwinterT = prevwinterT_flowering,
                                                           prevfallR = prevfallR_flowering,
                                                           TSFcont = TSF,
                                                           year = year_obs,
                                                           population = population), 
                                        size = 1)
          
          
          ### RECRUITMENT (number of flowers) ###
          
          # Number of flowers per individual reproducing
          
          data_droso$nbFlowers = NA
          
          if(length(data_droso$nbFlowers[which(data_droso$flowering == 1)]) > 0){
            
            # print("ADD NB FLOWERS")
            
            data_droso$nbFlowers[which(data_droso$flowering == 1)] = rnbinom(n = nrow(data_droso[which(data_droso$flowering == 1), ]),
                                                                             mu = nbFlowers_function(size = data_droso[which(data_droso$flowering == 1), ]$size_unscaled,
                                                                                                     prevwinterT = prevwinterT_nbFlowers,
                                                                                                     TSFcont = TSF,
                                                                                                     year = year_obs,
                                                                                                     population = population),
                                                                             size = 1)
            
            # Cap the number of flowers
            if(any(data_droso$nbFlowers[which(data_droso$flowering == 1)] > max_nbFlowers$nbFlowers[which(max_nbFlowers$TSFcont == TSFcat)], na.rm = T)) data_droso$nbFlowers[which(data_droso$flowering == 1 & data_droso$nbFlowers > max_nbFlowers$nbFlowers[which(max_nbFlowers$TSFcont == TSFcat)])] = max_nbFlowers$nbFlowers[which(max_nbFlowers$TSFcont == TSFcat)]
            
            
            # Number of seeds per individual reproducing
            
            data_droso_sub = data_droso[which(data_droso$flowering == 1 & 
                                                data_droso$nbFlowers > 0), ]
            
            if(any(data_droso_sub$nbFlowers > 0)){
              
              nb_seeds = rpois(nrow(data_droso_sub), lambda = seeds_per_flower) # Number of seeds per flower per individual
              
              seed_produced = data_droso_sub[rep(row.names(data_droso_sub), 
                                                 data_droso_sub$nbFlowers * nb_seeds), ] # Dataset of seeds produced
              
              # Format dataset to match other datasets
              seed_produced$size_unscaled = seed_produced$sizeNext = seed_produced$flowering = seed_produced$nbFlowers = seed_produced$survival = NA
              rownames(seed_produced) = seq(1, nrow(seed_produced))
              
              
              # Seeds germinating directly
              
              # print("SEEDS GERMINATING")
              
              seed_produced$goCont = rbinom(n = nrow(seed_produced),
                                            prob = goCont_function(TSF = TSF),
                                            size = 1)
              
              
              # Seeds going to the seedbank
              
              seed_produced$goSB = NA
              
              if(any(seed_produced$goCont == 0)) seed_produced$goSB[which(seed_produced$goCont == 0)] = rbinom(n = nrow(seed_produced[which(seed_produced$goCont == 0), ]),
                                                                                                               prob = (1 - goCont_function(TSF = TSF)),
                                                                                                               size = 1)
            
              
              # Size of the germinated seedlings
              
              # Seedling size parameters (mean, sd, and degrees of freedom)
              seedling_size_parameters = seedling_size_function(abLarge = seed_produced$abLarge_unscaled,
                                                                prevwinterT = prevwinterT_seedlingSize,
                                                                TSFcont = TSF,
                                                                year = year_obs, 
                                                                population = population)
              
              # Get seedling size by sampling a truncated Student-t distribution
              # to match the model family, using the corresponding mean, sd, and df
              seed_produced$rownb = seq(1, nrow(seed_produced))
              
              seed_produced$size_unscaled[which(seed_produced$goCont == 1)] = apply(seed_produced[which(seed_produced$goCont == 1), ], 
                                                                                    1, 
                                                                                    function(x) rtt(1, 
                                                                                                    location = seedling_size_parameters$mean[as.numeric(x[grep("rownb", names(x))])], 
                                                                                                    scale = seedling_size_parameters$sd, 
                                                                                                    df = seedling_size_parameters$df, 
                                                                                                    left = 0))
              
              # Assign zero to individuals with negative size
              if(any(seed_produced$size_unscaled[which(seed_produced$goCont == 1)] == Inf, na.rm = T)) seed_produced$size_unscaled[which(seed_produced$goCont == 1 & seed_produced$size_unscaled == Inf)] = max(seed_produced$size_unscaled[which(seed_produced$goCont == 1 & seed_produced$size_unscaled != Inf)], na.rm = T)
              
              if(any(is.na(seed_produced$size_unscaled[which(seed_produced$goCont == 1)]), na.rm = T)) seed_produced$size_unscaled[which(seed_produced$goCont == 1 & seed_produced$size_unscaled < 0)] = 0
              
              # Seedling ID
              
              seed_produced$ID = paste("Seed", 
                                       seq(max_seed_ID + 1, max_seed_ID + 1 + nrow(seed_produced) - 1), 
                                       sep = "_")
              
              seed_produced = seed_produced[, colnames(seed_produced)[-which(colnames(seed_produced) == "rownb")]]
              
              # Update max seed ID
              max_seed_ID = max(as.numeric(unlist(lapply(strsplit(seed_produced$ID, split = "_"), function(x) x[2]))))   
            }
          }
          
          ### SURVIVAL ###
          
          data_droso$survival = rbinom(n = nrow(data_droso),
                                       prob = survival_function(size = data_droso$size_unscaled,
                                                                abLarge = data_droso$abLarge_unscaled,
                                                                TSFcont = TSF,
                                                                fallR = fallR_survival,
                                                                summerT = summerT_survival,
                                                                year = year_obs,
                                                                population = population),
                                       size = 1)
          
          
          ### GROWTH ###
          
          data_droso$sizeNext = NA
          
          # Growth parameters (mean, sd, and degrees of freedom)
          growth_parameters = growth_function(size = data_droso$size_unscaled, 
                                              fallR = fallR_growth, 
                                              abLarge = data_droso$abLarge_unscaled,
                                              TSFcont = TSF,
                                              year = year_obs, 
                                              population = population)
          
          # Get size at next timestep by sampling a truncated Student-t distribution
          # to match the model family, using the corresponding mean, sd, and df
          data_droso$rownb = seq(1, nrow(data_droso))
          
          data_droso$sizeNext[which(data_droso$survival == 1)] = apply(data_droso[which(data_droso$survival == 1), ],
                                                                       1,
                                                                       function(x) rtt(1,
                                                                                       location = growth_parameters$mean[as.numeric(x[grep("rownb", names(x))])],
                                                                                       scale = growth_parameters$sd,
                                                                                       df = growth_parameters$df,
                                                                                       left = 0, 
                                                                                       right = max(data_droso$size_unscaled)))
          
          
          # Assign max size in current dataset to individual with infinite size
          if(any(data_droso$sizeNext[which(data_droso$survival == 1)] == Inf, na.rm = T)) data_droso$sizeNext[which(data_droso$survival == 1 & data_droso$sizeNext == Inf)] = max(data_droso$sizeNext[which(data_droso$survival == 1 & data_droso$sizeNext != Inf)], na.rm = T)
          
          if(any(is.na(data_droso$sizeNext[which(data_droso$survival == 1)]), na.rm = T)) data_droso$sizeNext[which(data_droso$survival == 1 & data_droso$sizeNext < 0)] = 0
          
          # Format dataset
          data_droso = data_droso[which(data_droso$survival == 1), colnames(data_droso)[-which(colnames(data_droso) %in% c("rownb"))]]
        }
      }
    }
    
    
    if(nrow(data_SB) > 0){
      
      ### SEEDBANK ###
      
      # Seeds germinating from the seedbank 
      
      data_SB$outSB = rbinom(n = nrow(data_SB),
                             prob = outSB_function(TSF = TSF),
                             size = 1)
      
      
      # Assign a size to the germinated seeds, add them to the dataset 
      # of aboveground individuals and remove them from the seedbank data
      
      # Seedling size parameters (mean, sd, and degrees of freedom)
      seedling_size_parameters = seedling_size_function(abLarge = data_SB$abLarge_unscaled,
                                                        prevwinterT = prevwinterT_seedlingSize,
                                                        TSFcont = TSF,
                                                        year = year_obs, 
                                                        population = population)
      
      # Get seedling size by sampling a truncated Student-t distribution
      # to match the model family, using the corresponding mean, sd, and df
      data_SB$rownb = seq(1, nrow(data_SB))
      
      if(length(which(data_SB$outSB == 1)) > 0){
        
        data_SB$size_unscaled[which(data_SB$outSB == 1)] = apply(data_SB[which(data_SB$outSB == 1), ],
                                                                 1, 
                                                                 function(x) rtt(1, 
                                                                                 location = seedling_size_parameters$mean[as.numeric(x[grep("rownb", names(x))])], 
                                                                                 scale = seedling_size_parameters$sd, 
                                                                                 df = seedling_size_parameters$df, 
                                                                                 left = 0))
        
        # Assign zero in current dataset to individual with negative size
        if(any(is.na(data_SB$size_unscaled[which(data_SB$outSB == 1)]), na.rm = T)) data_SB$size_unscaled[which(data_SB$outSB == 1 & data_SB$size_unscaled < 0)] = 0
        
        if(any(data_SB$size_unscaled[which(data_SB$outSB == 1)] == Inf, na.rm = T)) data_SB$size_unscaled[which(data_SB$outSB == 1 & data_SB$size_unscaled == Inf)] = max(data_SB$size_unscaled[which(data_SB$outSB == 1 & data_SB$size_unscaled != Inf)], na.rm = T)
        
      }
      
      
      # Preparing the seedbank data to merge with the continuous germination data
      
      data_SB$goCont = data_SB$outSB
      data_SB$goSB = NA
      
      # Merge datasets 
      seed_produced = rbind(seed_produced, data_SB[which(data_SB$outSB == 1), colnames(data_SB)[which(colnames(data_SB) %in% colnames(seed_produced))]])
      
      # Remove germinated seeds from seedbank data
      data_SB = data_SB[which(data_SB$outSB == 0), ]
      
      data_SB = data_SB[, colnames(data_SB)[-which(colnames(data_SB) %in% c("rownb", "outSB"))]]
      
      
      # Seeds staying in and going to the seedbank
      
      data_SB$staySB = rbinom(n = nrow(data_SB),
                              prob = staySB_function(TSF = TSF),
                              size = 1)
      
      # Keep only seeds staying in the seedbank
      data_SB = data_SB[which(data_SB$staySB == 1), ]
    }
    
    # Merge seeds going to SB to seedbank data
    seed_produced$staySB = seed_produced$goSB
    
    if(nrow(seed_produced[which(seed_produced$staySB == 1), ]) > 0){
      
      data_SB = rbind(data_SB, seed_produced[which(seed_produced$staySB == 1), ])
    }
    
    
    if(nrow(data_SB) > 0){
      
      # Format seedbank data to match other datasets
      data_SB = data_SB[, colnames(data_SB)[-which(colnames(data_SB) %in% c("staySB", "outSB", "rownb", "goCont", "goSB"))]]
    }
    
    seedbank_size_vec[i] = nrow(data_SB)
    
    if(nrow(seed_produced) > 0){
      
      # Keep only seeds germinating 
      seed_produced = seed_produced[which(seed_produced$goCont == 1), colnames(seed_produced)[-which(colnames(seed_produced) %in% c("goCont", "goSB", "staySB", "outSB", "rownb"))]]
      
      # Cap the number of recruits if needed
      if(!is.null(recruitCap) & nrow(seed_produced) > 0){
        
        # Get number of seedlings per quadrat
        quadratsAboveMaxSeedlings = aggregate(ID ~ quadratID, 
                                              data = seed_produced,
                                              FUN = function(x) length(x))
        
        # Keep quadrats where the number of seedlings is above the threshold
        quadratsAboveMaxSeedlings = quadratsAboveMaxSeedlings[which(quadratsAboveMaxSeedlings$ID > recruitCap$ID[which(recruitCap$TSFcont == TSFcat)]), ]
        
        # If quadrats are above the threshold, sample the seedlings that
        # will be kept
        if(nrow(quadratsAboveMaxSeedlings) > 0){
          
          for(quadrat in quadratsAboveMaxSeedlings$quadratID){
            
            recruitsKept_ID = sample(seq(1, nrow(seed_produced[which(seed_produced$quadratID == quadrat), ])), 
                                     size = recruitCap$ID[which(recruitCap$TSFcont == TSFcat)], replace = F)
            
            recruitsKept = seed_produced[which(seed_produced$quadratID == quadrat)[recruitsKept_ID], ]
            
            seed_produced = seed_produced[-which(seed_produced$quadratID == quadrat), ]
            seed_produced = rbind(seed_produced, recruitsKept)
          }
        }
      }
      
      
      # Adding new seedlings to the population
      
      if(any(is.na(data_droso$ID))) data_droso = rbind(data_droso[which(is.na(data_droso$ID)), ], seed_produced)
      
      else data_droso = rbind(data_droso, seed_produced)
    }
    
    if(nrow(data_droso) > 0){
      
      data_droso$TSFcont_unscaled[which(is.na(data_droso$TSFcont_unscaled))] = TSF
      data_droso$time = (year + 1)
    }
    
    
    log_meanChangeAboveground[i] = log(nrow(data_droso)/nrow(sim_data[which(sim_data$time_sim == i-1), ])) # Calculate mean change in aboveground population abundance
    log_lambda[i] = log((nrow(data_droso) + nrow(data_SB))/(nrow(sim_data[which(sim_data$time_sim == i-1), ]) + seedbank_size_vec[i-1])) # Calculate log lambda (with seedbank)
    
    
    ### SAVE DATA ###
    
    time_sim = i # Update timestep
    
    # Merge yearly individual data with full individual data
    if(nrow(data_droso) > 0) data_droso = cbind(data_droso, time_sim) 
    sim_data = rbind(sim_data, data_droso[, colnames(data_droso)[which(colnames(data_droso) %in% colnames(sim_data))]])
    
    # Format yearly data
    data_droso = data_droso[, -ncol(data_droso)]
    
    # Assess population quasi-extinction
    if(nrow(data_droso) < 5 & nrow(data_SB) < 50){
      
      extinction = 1
      break
      
    }
    
    
    ##### NEW  DATA FOR T + 1 #####
    
    data_droso$size_unscaled[which(!is.na(data_droso$sizeNext))] = data_droso$sizeNext[which(!is.na(data_droso$sizeNext))] # Assign new size to individuals
    
    
    ### DENSITY ###
    
    # Get quadrat-specific density from number of individuals with size > 4.5 
    data_droso$abLarge_unscaled = as.numeric(apply(data_droso, 1, FUN = function(x) nrow(data_droso[which(data_droso$quadratID == x[2] & data_droso$size_unscaled > 4.5), ])))
    if(nrow(data_droso) > 0) pop_density[[i]] = aggregate(abLarge_unscaled ~ quadratID, data = data_droso, mean, na.rm = T)$abLarge_unscaled
    
  }
  
  # Create summary data
  data_agg = aggregate(ID ~ time_sim, data = sim_data, function(x) length(x))

  data_agg$run = sim

  ibm_data = rbind(ibm_data, data_agg)

  
  return(list(pop_data = sim_data,
              pop_size = ibm_data,
              pop_density = pop_density,
              log_meanChangeAboveground = log_meanChangeAboveground,
              log_lambda = log_lambda,
              seedbank_size_vec = seedbank_size_vec,
              extinction = extinction))
}


## 5.2. Simulation function ----
# -------------------------

ibm_natural = function(n_sim = 1000,               # Number of simulations
                       n_years = 50,               # Number of years per simulation
                       first_year = 2021,          # Year of start of simulation
                       years_RE = seq(2016, 2021), # Observed years available for random year effect
                       data_initial,               # Initial dataset
                       seedbank_size = 10000,       # Initial seedbank size
                       recruitCap,                 # Maximum number of recruits per quadrat
                       max_nbFlowers,              # Maximum number of flowers per individual
                       population,                 # Population ID
                       climate_model,              # Name of climate model
                       vital_rate,                 # Vital rate to perturb
                       fire_prob = 1/30){          # Fire frequency
  
  # Initialization - Prepare storing objects
  
  extinction_vector = rep(0, n_sim)
  log_meanChangeAboveground_array = array(NA, dim = c(n_sim, n_years))
  log_lambda_array = array(NA, dim = c(n_sim, n_years))
  seedbank_size_array = array(NA, dim = c(n_sim, n_years))
  pop_size_list = vector(mode = "list", length = n_sim)
  pop_data_list = vector(mode = "list", length = n_sim)
  pop_density_list = vector(mode = "list", length = n_sim)
  
  ibm_data = NULL
  
  # Calculate number of flowers
  data_initial$nbFlowers = data_initial$fs * data_initial$fps
  
  # Format initial data
  data_initial = data_initial[, c("site", "quadratID", "ID", "TSFcont_unscaled", "size_unscaled", "sizeNext", 
                                  "fl", "nbFlowers", 
                                  "surv", "time", "abLarge_unscaled")]
  
  colnames(data_initial) = c("site", "quadratID", "ID", "TSFcont_unscaled", "size_unscaled", "sizeNext", 
                             "flowering", "nbFlowers", 
                             "survival", "time", "abLarge_unscaled")
  
  # Prepare initial seedbank data
  seedbank_initial = data.frame(site = population, quadratID = NA, ID = paste("Seed", seq(1, seedbank_size), sep = "_"),
                                TSFcont_unscaled = NA, size_unscaled = NA, sizeNext = NA, 
                                flowering = NA, nbFlowers = NA,
                                survival = NA, time = NA, abLarge_unscaled = NA)
  
  
  # Building the Markov chain to create the succession of post-fire environmental states
  envS = matrix(0, 5, 5)
  
  envS[2, 1] = envS[3, 2] = envS[4, 3] = envS[5, 4] = 1
  envS[1, 5] = fire_prob
  envS[5, 5] = 1 - fire_prob
  
  colnames(envS) = rownames(envS) = 1:5
  
  
  # Projection
  
  # For each simulation, project the population
  # If there is an error (exploding population), restart the simulation
  # to have another sequence of years
  
  for(sim in 1:n_sim){
    
    print(paste("Iteration", sim))
    
    while(TRUE){

      ibm_sim_result =
        try(ibm_sim(n_sim = n_sim,
                    n_years = n_years,
                    sim = sim,
                    first_year = first_year,
                    years_RE = years_RE,
                    seedbank_size = seedbank_size,
                    recruitCap = recruitCap,
                    max_nbFlowers,
                    population = population,
                    climate_model = climate_model,
                    vital_rate = vital_rate,
                    envS = envS,
                    ibm_data = ibm_data,
                    data_initial = data_initial,
                    seedbank_initial = seedbank_initial),
            silent = TRUE)

      if(!is(ibm_sim_result, 'try-error')) break
    }
    
    # Fill in result objects
    log_meanChangeAboveground_array[sim, ] = ibm_sim_result$log_meanChangeAboveground
    log_lambda_array[sim, ] = ibm_sim_result$log_lambda
    seedbank_size_array[sim, ] = ibm_sim_result$seedbank_size_vec
    extinction_vector[sim] = ibm_sim_result$extinction
    pop_size_list[[sim]] = ibm_sim_result$pop_size
    pop_data_list[[sim]] = ibm_sim_result$pop_data
    pop_density_list[[sim]] = ibm_sim_result$pop_density
    
  }
  
  return(list(pop_data = pop_data_list,
              pop_size = pop_size_list,
              pop_density = pop_density_list,
              log_meanChangeAboveground = log_meanChangeAboveground_array,
              log_lambda = log_lambda_array,
              seedbank_size = seedbank_size_array,
              extinction = extinction_vector))
}




###########################################################################
#
# 6. Projections ----
#
###########################################################################

n_sim = 100
n_years = 30

## 6.1. SierraCarboneraY5 ----
# -----------------------

population = "SierraCarboneraY5"

recruitCap = aggregate(ID ~ quadratID + time + TSFcont,
                       data = droso_full[which(droso_full$site == population &
                                               droso_full$stage == "SD"), ], FUN = function(x) length(x))

recruitCap$TSFcont[which(recruitCap$TSFcont > 4)] = 4

recruitCap = aggregate(ID ~ TSFcont,
                       data = recruitCap, FUN = max)

recruitCap = rbind(recruitCap, 
                   data.frame(TSFcont = 0,
                              ID = round(max(recruitCap$ID) * 1.5)))

recruitCap = recruitCap[order(recruitCap$TSFcont), ]


max_nbFlowers = aggregate(fs * fps ~ TSFcont,
                          data = droso_full[which(droso_full$site == population), ], FUN = max)
colnames(max_nbFlowers) = c("TSFcont", "nbFlowers")

max_nbFlowers$TSFcont[which(max_nbFlowers$TSFcont > 4)] = 4

max_nbFlowers = aggregate(nbFlowers ~ TSFcont,
                          data = max_nbFlowers, FUN = max)


# Function to run the projections in parallel 
ncpus <- 5

startIBM <- function(sim){
  
  if (sim > 1) {      #slave, give it other random seed,
    rm(".Random.seed", envir = .GlobalEnv); runif(1)
  }
  
  ibm_natural(n_sim = n_sim/ncpus,
              n_years = n_years,
              first_year = 2021, 
              data_initial = droso_natural[which(droso_natural$site == population), ], 
              recruitCap = recruitCap, 
              max_nbFlowers = max_nbFlowers,
              population = population, 
              climate_model = climate_model,
              vital_rate = vital_rate)
  
}


## 6.1.1. CanESM5 ----
# ---------------

climate_model = "CanESM5"

## 5.1.1.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_canesm5_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_canesm5_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_canesm5_sensitivity_survival)

## 5.1.1.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_canesm5_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_canesm5_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_canesm5_sensitivity_growth)


## 5.1.1.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_canesm5_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_canesm5_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_canesm5_sensitivity_flowering)


## 5.1.1.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_canesm5_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_canesm5_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_canesm5_sensitivity_nbFlowers)


## 5.1.1.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_canesm5_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_canesm5_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_canesm5_sensitivity_seedlingSize)


## 5.1.2. EC_Earth3 ----
# -----------------

climate_model = "EC_Earth3"

## 5.1.2.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ec_earth3_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ec_earth3_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ec_earth3_sensitivity_survival)


## 5.1.2.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ec_earth3_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ec_earth3_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ec_earth3_sensitivity_growth)


## 5.1.2.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ec_earth3_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ec_earth3_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ec_earth3_sensitivity_flowering)


## 5.1.2.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ec_earth3_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ec_earth3_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ec_earth3_sensitivity_nbFlowers)


## 5.1.2.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ec_earth3_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ec_earth3_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ec_earth3_sensitivity_seedlingSize)


## 5.1.3. FGOALS_G3 ----
# -----------------

climate_model = "FGOALS_G3"

## 5.1.3.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_fgoals_g3_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_fgoals_g3_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_fgoals_g3_sensitivity_survival)


## 5.1.3.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_fgoals_g3_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_fgoals_g3_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_fgoals_g3_sensitivity_growth)


## 5.1.3.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_fgoals_g3_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_fgoals_g3_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_fgoals_g3_sensitivity_flowering)


## 5.1.3.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_fgoals_g3_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_fgoals_g3_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_fgoals_g3_sensitivity_nbFlowers)


## 5.1.3.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_fgoals_g3_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_fgoals_g3_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_fgoals_g3_sensitivity_seedlingSize)


## 5.1.4. GFDL_ESM4 ----
# -----------------

climate_model = "GFDL_ESM4"

## 5.1.4.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_gfdl_esm4_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_gfdl_esm4_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_gfdl_esm4_sensitivity_survival)


## 5.1.4.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_gfdl_esm4_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_gfdl_esm4_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_gfdl_esm4_sensitivity_growth)


## 5.1.4.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_gfdl_esm4_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_gfdl_esm4_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_gfdl_esm4_sensitivity_flowering)


## 5.1.4.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_gfdl_esm4_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_gfdl_esm4_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_gfdl_esm4_sensitivity_nbFlowers)


## 5.1.4.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_gfdl_esm4_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_gfdl_esm4_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_gfdl_esm4_sensitivity_seedlingSize)


## 5.1.5. GISS_E2_1_G ----
# -------------------

climate_model = "GISS_E2_1_G"

## 5.1.5.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_giss_e2_1_g_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_survival)


## 5.1.5.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_giss_e2_1_g_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_growth)


## 5.1.5.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_giss_e2_1_g_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_flowering)


## 5.1.5.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_giss_e2_1_g_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_nbFlowers)


## 5.1.5.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_giss_e2_1_g_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_giss_e2_1_g_sensitivity_seedlingSize)


## 5.1.6. INM_CM4_8 ----
# -----------------

climate_model = "INM_CM4_8"

## 5.1.6.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_inm_cm4_8_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_inm_cm4_8_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_inm_cm4_8_sensitivity_survival)


## 5.1.6.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_inm_cm4_8_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_inm_cm4_8_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_inm_cm4_8_sensitivity_growth)


## 5.1.6.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_inm_cm4_8_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_inm_cm4_8_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_inm_cm4_8_sensitivity_flowering)


## 5.1.6.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_inm_cm4_8_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_inm_cm4_8_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_inm_cm4_8_sensitivity_nbFlowers)


## 5.1.6.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_inm_cm4_8_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_inm_cm4_8_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_inm_cm4_8_sensitivity_seedlingSize)


## 5.1.7. IPSL_CM6A_LR ----
# --------------------

climate_model = "IPSL_CM6A_LR"

## 5.1.7.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_survival)


## 5.1.7.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_growth)


## 5.1.7.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_flowering)


## 5.1.7.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_nbFlowers)


## 5.1.7.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_ipsl_cm6a_lr_sensitivity_seedlingSize)


## 5.1.8. MIROC6 ----
# --------------

climate_model = "MIROC6"

## 5.1.8.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_miroc6_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_miroc6_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_miroc6_sensitivity_survival)


## 5.1.8.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_miroc6_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_miroc6_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_miroc6_sensitivity_growth)


## 5.1.8.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_miroc6_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_miroc6_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_miroc6_sensitivity_flowering)


## 5.1.8.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_miroc6_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_miroc6_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_miroc6_sensitivity_nbFlowers)


## 5.1.8.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_miroc6_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_miroc6_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_miroc6_sensitivity_seedlingSize)


## 5.1.9. MPI_ESM1_2_LR ----
# ----------------------

climate_model = "MPI_ESM1_2_LR"

## 5.1.9.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_survival)


## 5.1.9.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_growth)


## 5.1.9.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_flowering)


## 5.1.9.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_nbFlowers)


## 5.1.9.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mpi_esm1_2_lr_sensitivity_seedlingSize)


## 5.1.10. MRI_ESM2_0 ----
# -------------------

climate_model = "MRI_ESM2_0"

## 5.1.10.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mri_esm2_0_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mri_esm2_0_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mri_esm2_0_sensitivity_survival)


## 5.1.10.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mri_esm2_0_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mri_esm2_0_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mri_esm2_0_sensitivity_growth)


## 5.1.10.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mri_esm2_0_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mri_esm2_0_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mri_esm2_0_sensitivity_flowering)


## 5.1.10.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mri_esm2_0_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mri_esm2_0_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mri_esm2_0_sensitivity_nbFlowers)


## 5.1.10.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_mri_esm2_0_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_mri_esm2_0_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_mri_esm2_0_sensitivity_seedlingSize)


## 5.1.11. NorESM2_MM ----
# -------------------

climate_model = "NorESM2_MM"

## 5.1.11.1. Perturbing survival ----
# -----------------------------

vital_rate = "survival"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_noresm2_mm_sensitivity_survival = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_noresm2_mm_sensitivity_survival, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_noresm2_mm_sensitivity_survival)


## 5.1.11.2. Perturbing growth ----
# ---------------------------

vital_rate = "growth"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_noresm2_mm_sensitivity_growth = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_noresm2_mm_sensitivity_growth, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_noresm2_mm_sensitivity_growth)


## 5.1.11.3. Perturbing flowering ----
# ------------------------------

vital_rate = "flowering"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_noresm2_mm_sensitivity_flowering = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_noresm2_mm_sensitivity_flowering, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_noresm2_mm_sensitivity_flowering)


## 5.1.11.4. Perturbing number of flowers ----
# --------------------------------------

vital_rate = "nbFlowers"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_noresm2_mm_sensitivity_nbFlowers = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_noresm2_mm_sensitivity_nbFlowers, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_noresm2_mm_sensitivity_nbFlowers)


## 5.1.11.5. Perturbing seedling size ----
# ----------------------------------

vital_rate = "seedlingSize"

# Set up parallel environment
sfInit(parallel = TRUE, cpus = ncpus, 
       slaveOutfile = paste0("Output/Projections/IBM_Progress_SensitivityAnalysis_", 
                             population, "_", climate_model, "_", 
                             vital_rate, ".txt"))

# Export data and libraries to each parallel core
sfExport(list = c(ls(), ".Random.seed")) 

sfLibrary("mgcv", character.only = TRUE)
sfLibrary("crch", character.only = TRUE)
sfLibrary("snowfall", character.only = TRUE)

# Run projections in parallel
ibm_sierracarboneray5_noresm2_mm_sensitivity_seedlingSize = sfClusterApplyLB(1:ncpus, startIBM)

# Save results and stop cluster
save(ibm_sierracarboneray5_noresm2_mm_sensitivity_seedlingSize, 
     file = paste0("Output/Projections/IBM_Natural_SensitivityAnalysis_", 
                   population, "_", climate_model, "_", 
                   vital_rate, ".RData"))
sfStop()

rm(ibm_sierracarboneray5_noresm2_mm_sensitivity_seedlingSize)

