# -----------------------------------------------------------
# Program Name:    Example: Signal Reconstruction with Noise
# Version:         1.0
# Author:          Olaf Margowski
# License:         MIT (https://opensource.org/licenses/MIT)
# Date:            July 2025
#
# Description:
#   This script demonstrates basic signal reconstruction using simulated data.
#   An original sine wave signal is generated and Gaussian noise is added.
#   The measured noisy signal is then reconstructed using Savitzky-Golay smoothing.
#   The original and reconstructed signals are visualized for comparison.
#
#   Dependencies:
#       numpy, matplotlib, scipy
#       (install via: pip install numpy matplotlib scipy)
#
#   Reproducibility:
#       Random seed is set to 0 for reproducibility.
#
#   Output:
#       The plot "simulation_result.png" shows the original and reconstructed signals.
# -----------------------------------------------------------

import numpy as np
import matplotlib.pyplot as plt

# Generate example signal
t = np.linspace(0, 1, 500)
f = np.sin(2 * np.pi * t)

# Add simulated Gaussian noise
np.random.seed(0)
noise = np.random.normal(0, 0.05, size=t.shape)
measured = f + noise

# Example reconstruction (here: smoothing as a placeholder)
from scipy.signal import savgol_filter
f_recon = savgol_filter(measured, window_length=31, polyorder=3)

# Plot
plt.figure(figsize=(8,4))
plt.plot(t, f, label='Original signal', color='blue')
plt.plot(t, f_recon, label='Reconstructed signal', color='orange', linestyle='--')
plt.xlabel('Time $t$')
plt.ylabel('Signal value $f(t)$')
plt.title('Simulation result: Original and reconstructed signals')
plt.legend()
plt.tight_layout()
plt.savefig('simulation-result.png', dpi=150)
plt.show()
