/* Copyright (C) 2011-2025  Egon Willighagen <egonw@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the GNU Lesser General Public License as published by the Free
 * Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
 * details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */
package io.github.egonw.nanojava.descriptor;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNotNull;
import static org.junit.jupiter.api.Assertions.assertNotSame;
import static org.junit.jupiter.api.Assertions.assertTrue;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.openscience.cdk.qsar.DescriptorValue;
import org.openscience.cdk.qsar.result.DoubleArrayResult;
import org.openscience.cdk.qsar.result.IDescriptorResult;
import org.openscience.cdk.silent.SilentChemObjectBuilder;
import org.openscience.cdk.tools.manipulator.MolecularFormulaManipulator;

import com.github.jqudt.onto.units.LengthUnit;

import io.github.egonw.nanojava.data.Material;
import io.github.egonw.nanojava.data.MaterialType;
import io.github.egonw.nanojava.data.measurement.EndPoints;
import io.github.egonw.nanojava.data.measurement.MeasurementValue;

public class EnergyBandDescriptorTest
extends MaterialDescriptorTest {

    @BeforeEach
    public void setUp() throws Exception {
        setDescriptor(EnergyBandDescriptor.class);
    }

    @Test
    public void testCalculate_ZnO() throws Exception {
        Material material = new Material(MaterialType.METALOXIDE);
        material.addAtomContainer(
            MolecularFormulaManipulator.getAtomContainer(
                "ZnO", SilentChemObjectBuilder.getInstance()
            )
        );
        DescriptorValue value = descriptor.calculate(material);
        assertNotNull(value);
        IDescriptorResult result = value.getValue();
        assertNotNull(result);
        assertTrue(result instanceof DoubleArrayResult);
        DoubleArrayResult daResult = (DoubleArrayResult)result;
        assertEquals(2, daResult.length());
        assertEquals(-3.7, daResult.get(0), 0.0001);
        assertEquals(-7.25, daResult.get(1), 0.0001);
    }

    @Test
    public void testCalculate_TestSize_TooSmall() throws Exception {
        Material material = new Material(MaterialType.METALOXIDE);
        material.addAtomContainer(
            MolecularFormulaManipulator.getAtomContainer(
                "ZnO", SilentChemObjectBuilder.getInstance()
            )
        );
        material.setSize(new MeasurementValue(EndPoints.SIZE, 10.0, 5.0, LengthUnit.NM));
        DescriptorValue value = descriptor.calculate(material);
        assertNotNull(value);
        IDescriptorResult result = value.getValue();
        assertNotNull(result);
        assertTrue(result instanceof DoubleArrayResult);
        DoubleArrayResult daResult = (DoubleArrayResult)result;
        assertEquals(2, daResult.length());
        assertEquals(Double.NaN, daResult.get(0), 0.0001);
        assertEquals(Double.NaN, daResult.get(1), 0.0001);
    }

    @Test
    public void testCalculate_TestSize_OK() throws Exception {
        Material material = new Material(MaterialType.METALOXIDE);
        material.addAtomContainer(
            MolecularFormulaManipulator.getAtomContainer(
                "ZnO", SilentChemObjectBuilder.getInstance()
            )
        );
        material.setSize(new MeasurementValue(EndPoints.SIZE, 40.0, 5.0, LengthUnit.NM));
        DescriptorValue value = descriptor.calculate(material);
        assertNotNull(value);
        IDescriptorResult result = value.getValue();
        assertNotNull(result);
        assertTrue(result instanceof DoubleArrayResult);
        DoubleArrayResult daResult = (DoubleArrayResult)result;
        assertEquals(2, daResult.length());
        assertEquals(-3.7, daResult.get(0), 0.0001);
        assertEquals(-7.25, daResult.get(1), 0.0001);
    }

    @Test
    public void testCalculate_IndiumOxide() throws Exception {
        Material material = new Material(MaterialType.METALOXIDE);
        material.addAtomContainer(
            MolecularFormulaManipulator.getAtomContainer(
                "In2O3", SilentChemObjectBuilder.getInstance()
            )
        );
        material.setSize(new MeasurementValue(EndPoints.SIZE, 40.0, 5.0, LengthUnit.NM));
        DescriptorValue value = descriptor.calculate(material);
        assertNotNull(value);
        IDescriptorResult result = value.getValue();
        assertNotNull(result);
        assertTrue(result instanceof DoubleArrayResult);
        DoubleArrayResult daResult = (DoubleArrayResult)result;
        assertEquals(2, daResult.length());
        assertEquals(-4.0, daResult.get(0), 0.0001);
        assertEquals(-6.8, daResult.get(1), 0.0001);
    }

    @Test
    public void testCalculate_IronOxides() throws Exception {
        Material oxide1 = new Material(MaterialType.METALOXIDE);
        oxide1.addAtomContainer(
            MolecularFormulaManipulator.getAtomContainer(
                "Fe2O3", SilentChemObjectBuilder.getInstance()
            )
        );
        Material oxide2 = new Material(MaterialType.METALOXIDE);
        oxide2.addAtomContainer(
            MolecularFormulaManipulator.getAtomContainer(
                "Fe3O4", SilentChemObjectBuilder.getInstance()
            )
        );
        
        DescriptorValue value1 = descriptor.calculate(oxide1);
        IDescriptorResult result1 = value1.getValue();
        DescriptorValue value2 = descriptor.calculate(oxide2);
        IDescriptorResult result2 = value2.getValue();
        DoubleArrayResult daResult1 = (DoubleArrayResult)result1;
        DoubleArrayResult daResult2 = (DoubleArrayResult)result2;
        assertNotSame(Double.NaN, daResult1.get(0));
        assertNotSame(Double.NaN, daResult1.get(1));
        assertNotSame(daResult1.get(0), daResult2.get(0));
        assertNotSame(daResult1.get(1), daResult2.get(1));
    }
}
