#!/bin/bash

SNAPSHOTS_DIR="./SnapshotInstructorUITests/__Snapshots__/BaseUITest"
DATASET_DIR="dataset"
TEST_FILES_DIR="SnapshotInstructorUITests"

mkdir -p "$DATASET_DIR"

FLAG_MAPPINGS_FILE=$(mktemp)

extract_flag_mappings() {
    find "$TEST_FILES_DIR" -name "*Tests.swift" -type f | while read test_file; do
        category=$(basename "$test_file" | sed 's/Tests\.swift//')
        
        # Extract single flag tests (runTestWithFlagVariants)
        grep -n -A 3 "runTestWithFlagVariants" "$test_file" | grep -E 'flagName:|testName:' | awk -F': ' '{print $2}' | sed 's/"//g' | sed 's/,//' | paste -d '|' - - | while read line; do
            flag_name=$(echo "$line" | cut -d'|' -f1)
            test_name=$(echo "$line" | cut -d'|' -f2)
            
            echo "$test_name|$flag_name|$category|single" >> "$FLAG_MAPPINGS_FILE"
        done
        
        # Extract multiple flag tests (runTestWithFlags)
        grep -n -A 5 "runTestWithFlags" "$test_file" | grep -E 'flags:|testName:' | sed 's/^[ \t]*//' | awk -F': ' '{print $2}' | sed 's/"//g' | sed 's/,$//' | paste -d '|' - - | while read line; do
            flags_raw=$(echo "$line" | cut -d'|' -f1)
            test_name=$(echo "$line" | cut -d'|' -f2)
            
            # Extract flag enum values from the Swift syntax
            flags=$(echo "$flags_raw" | sed 's/flags: \[//g' | sed 's/\]//g' | sed 's/\.//g' | sed 's/, /,/g')
            
            echo "$test_name|$flags|$category|multi" >> "$FLAG_MAPPINGS_FILE"
        done
        
        # Extract new multi-flag tests (runTestWithFlagSet)
        grep -n -A 5 "runTestWithFlagSet" "$test_file" | grep -n -A 1 "FeatureFlags.self" | grep -v "FeatureFlags.self" | sed 's/^[ \t]*//' | sed 's/\[//' | sed 's/\],//' | sed 's/testName:/\|/' | sed 's/^[0-9]*-//' | while read line; do
            flags_raw=$(echo "$line" | cut -d'|' -f1)
            test_name=$(echo "$line" | cut -d'|' -f2 | xargs)
            
            # Extract flag enum values and clean up
            flags=$(echo "$flags_raw" | sed 's/\.//g' | sed 's/, /,/g')
            
            echo "$test_name|$flags|$category|multi_new" >> "$FLAG_MAPPINGS_FILE"
        done
        
        # Legacy format (runTestWithMultipleFlags)
        grep -n -A 3 "runTestWithMultipleFlags" "$test_file" | grep -E 'flagNames:|testName:' | awk -F': ' '{print $2}' | sed 's/"//g' | sed 's/,//' | paste -d '|' - - | while read line; do
            flag_names=$(echo "$line" | cut -d'|' -f1)
            test_name=$(echo "$line" | cut -d'|' -f2)
            
            echo "$test_name|$flag_names|$category|legacy_multi" >> "$FLAG_MAPPINGS_FILE"
        done
    done
    
    echo "Extracted flag mappings:"
    cat "$FLAG_MAPPINGS_FILE"
}

check_imagemagick_command() {
    if command -v magick &> /dev/null; then
        MAGICK_CMD="magick"
    elif command -v convert &> /dev/null; then
        MAGICK_CMD="convert"
    else
        return 1
    fi
    return 0
}

check_and_install_imagemagick() {
    if ! check_imagemagick_command; then
        echo "ImageMagick not found. Attempting to install..."
        
        if [[ "$OSTYPE" == "darwin"* ]]; then
            if command -v brew &> /dev/null; then
                brew install imagemagick
            else
                echo "Homebrew not found. Please install Homebrew first or manually install ImageMagick."
                echo "Run: /bin/bash -c \"\$(curl -fsSL https://raw.githubusercontent.com/Homebrew/install/HEAD/install.sh)\""
                exit 1
            fi
        elif command -v apt-get &> /dev/null; then
            sudo apt-get update && sudo apt-get install -y imagemagick
        elif command -v yum &> /dev/null; then
            sudo yum install -y imagemagick
        else
            echo "Unable to automatically install ImageMagick. Please install it manually."
            exit 1
        fi
        
        if ! check_imagemagick_command; then
            echo "Failed to install ImageMagick. Please install it manually."
            exit 1
        else
            echo "ImageMagick installed successfully."
        fi
    fi
    
    echo "Using ImageMagick command: $MAGICK_CMD"
}

extract_categories_from_flags() {
    local flags="$1"
    local IFS=','
    read -ra flag_array <<< "$flags"
    
    local categories=()
    for flag in "${flag_array[@]}"; do
        flag=$(echo "$flag" | xargs | sed 's/Flag\.//')  # Trim whitespace and remove Flag. prefix
        category=$(echo "$flag" | cut -d'_' -f1,2)
        if [[ ! " ${categories[@]} " =~ " ${category} " ]]; then
            categories+=("$category")
        fi
    done
    
    echo "${categories[@]}"
}

clean_flag_name() {
    local flag_name="$1"
    echo "$flag_name" | sed 's/Flag\.//' | xargs
}

check_and_install_imagemagick
extract_flag_mappings

for dir in "$DATASET_DIR"/*; do
    if [ -d "$dir" ]; then
        base_dir=$(basename "$dir")
        if ! grep -q "^$base_dir|" "$FLAG_MAPPINGS_FILE"; then
            echo "Removing non-feature flag test directory: $dir"
            rm -rf "$dir"
        fi
    fi
done

find "$SNAPSHOTS_DIR" -name "*_Default.png" | while read default_image; do
    filename=$(basename "$default_image")
    
    test_name=$(echo "$filename" | sed 's/^Test\.//' | sed 's/_Default\.png$//')
    
    if ! grep -q "^$test_name|" "$FLAG_MAPPINGS_FILE"; then
        echo "Skipping non-feature flag test: $test_name"
        continue
    fi
    
    echo "Processing test: $test_name"
    
    flag_enabled_image="${default_image%_Default.png}_FlagEnabled.png"
    
    if [ ! -f "$flag_enabled_image" ]; then
        echo "Warning: No matching FlagEnabled image found for $default_image. Skipping."
        continue
    fi
    
    test_dir="$DATASET_DIR/$test_name"
    mkdir -p "$test_dir"
    
    cp "$default_image" "$test_dir/reference.png"
    cp "$flag_enabled_image" "$test_dir/failure.png"
    
    echo "Generating black-white diff image for $test_name"
    
    if [ "$MAGICK_CMD" = "magick" ]; then
        magick compare -metric AE -highlight-color white -lowlight-color black "$default_image" "$flag_enabled_image" "$test_dir/diff.png" 2>/dev/null
        if [ ! -s "$test_dir/diff.png" ] || [ ! -f "$test_dir/diff.png" ]; then
            magick "$default_image" "$flag_enabled_image" -compose difference -composite -threshold 5% "$test_dir/diff.png"
        fi
    else
        convert "$default_image" "$flag_enabled_image" -compose difference -composite -threshold 5% "$test_dir/diff.png"
        if [ ! -s "$test_dir/diff.png" ] || [ ! -f "$test_dir/diff.png" ]; then
            convert "$default_image" "$flag_enabled_image" -compose difference -composite -auto-level "$test_dir/diff.png"
        fi
    fi
    
    flag_info=$(grep "^$test_name|" "$FLAG_MAPPINGS_FILE")
    test_type=$(echo "$flag_info" | cut -d'|' -f4)
    flags_raw=$(echo "$flag_info" | cut -d'|' -f2)
    
    if [ "$test_type" = "single" ]; then
        # Handle single flag test
        flag_name=$(clean_flag_name "$flags_raw")
        main_category=$(echo "$flag_name" | cut -d'_' -f1,2)
        categories="[\"$main_category\"]"
        
        cat > "$test_dir/metadata.json" << EOF
{
  "flagName": "$flag_name",
  "categories": $categories,
  "description": "Feature flag test for $flag_name"
}
EOF
        echo "Created dataset for $test_name with flag $flag_name, category $main_category"
    else
        # Handle multi-flag tests (both new and legacy format)
        if [ "$test_type" = "legacy_multi" ]; then
            # Legacy format with string array - need to handle Flag. prefix
            flags_raw=$(echo "$flags_raw" | sed 's/\[//g' | sed 's/\]//g')
            
            # Split by spaces and Flag. to get a comma-separated list
            flags=$(echo "$flags_raw" | sed 's/Flag\./,/g' | sed 's/^ *,//' | sed 's/ //g')
        else
            # New format with enum values - clean up any remaining brackets
            flags=$(echo "$flags_raw" | sed 's/\[//g' | sed 's/\]//g' | sed 's/Flag\.//' | tr -d '[:space:]')
        fi
        
        # Extract categories from flags
        read -ra categories_array <<< "$(extract_categories_from_flags "$flags")"
        
        # Create a clean array of flags
        IFS=',' read -ra flag_array <<< "$flags"
        clean_flags=()
        for flag in "${flag_array[@]}"; do
            clean_flag=$(echo "$flag" | tr -d '[:space:]')
            if [ ! -z "$clean_flag" ]; then
                clean_flags+=("\"$clean_flag\"")
            fi
        done
        
        # Construct the categories JSON array
        categories="["
        for i in "${!categories_array[@]}"; do
            if [ $i -gt 0 ]; then
                categories+=", "
            fi
            categories+="\"${categories_array[$i]}\""
        done
        categories+="]"
        
        # Join the clean flags with commas for the metadata
        joined_flags=$(IFS=', '; echo "${clean_flags[*]}")
        flag_display=$(IFS=', '; echo "${flag_array[*]}")
        
        # Special handling for multi-category tests - create a better flag name
        if [ "${#categories_array[@]}" -gt 1 ]; then
            custom_flag_name="${test_name}_EXPERIENCE"
            
            cat > "$test_dir/metadata.json" << EOF
{
  "flagName": "$custom_flag_name",
  "categories": $categories,
  "relatedFlags": [$joined_flags],
  "description": "Multi-category test combining ${flag_display}"
}
EOF
            echo "Created multi-category dataset for $test_name with flag name $custom_flag_name"
        else
            cat > "$test_dir/metadata.json" << EOF
{
  "flagName": "${test_name}_EXPERIENCE",
  "categories": $categories,
  "relatedFlags": [$joined_flags],
  "description": "Feature flag test with multiple flags: $flag_display"
}
EOF
            echo "Created dataset for $test_name with multiple flags: $flag_display"
        fi
        echo "  Categories: ${categories_array[*]}"
    fi
done

rm "$FLAG_MAPPINGS_FILE"

echo "Processing complete. Dataset directory populated." 