//
//  SnapshotInstructorUITests.swift
//  SnapshotInstructorUITests
//
//  Created by Batuhan Kaynak on 4.03.2025.
//

import XCTest
import SnapshotTestingCore
import SwiftUI
import Combine
@testable import SnapshotInstructor

class WaitUtility {
    static let shared = WaitUtility()
    
    private init() {}
    
    func waitForAppear(_ elementString: String, timeout: TimeInterval = 10) {
        return waitForViewState(elementString: elementString, predicate: "TRUE", timeout: timeout)
    }
    
    func waitForDisappear(_ elementString: String, timeout: TimeInterval = 10) {
        return waitForViewState(elementString: elementString, predicate: "FALSE", timeout: timeout)
    }
    
    func waitForViewState(elementString: String, predicate: String, timeout: TimeInterval = 10) {
        let element = XCUIApplication().elementWith(identifier: elementString)
        let existsPredicate = NSPredicate(format: "exists == \(predicate)")
        let expectation = XCTNSPredicateExpectation(predicate: existsPredicate, object: element)
        let result = XCTWaiter.wait(for: [expectation], timeout: timeout)
        
        if result == .completed {
            return
        } else {
            print("\(elementString) took too long")
        }
    }
}


class BaseUITest: XCTestCase {
    struct TestSettings {
        var takeSnapshot: Bool
        var snapshotPrecision: Float
        var testName: String
        var suffix: String
    }
    var phone: String!
    var app: XCUIApplication!

    var currentTestSettings: TestSettings = TestSettings(takeSnapshot: true, snapshotPrecision: 0.999, testName: "Test.", suffix: "")
    var cancellables = Set<AnyCancellable>()
    
    var additionalLaunchArgs: [String]? = nil
    
    override func setUp() {
        super.setUp()
    }
    
    override func tearDown() {
        app = nil
        super.tearDown()
    }
    
    func goToHomeTab() {
        app.elementWith(identifier: "homeTab").forceTapElement()
    }
    
    func goToProfileTab() {
        app.elementWith(identifier: "ProfileButton").forceTapElement()
    }
    
    func goToSettingsTab() {
        app.elementWith(identifier: "settingsTab").forceTapElement()
    }
    
    func waitForKeyboard() -> Bool {
        WaitUtility.shared.waitForDisappear("LoadingProgress")
        return app.keyboards.element.waitForExistence(timeout: 5)
    }
    
    func performWithoutSnapshot(actions: () -> Void) {
        currentTestSettings.takeSnapshot = false
        let tempTestName = currentTestSettings.testName
        actions()
        currentTestSettings.testName = tempTestName
        currentTestSettings.takeSnapshot = true
    }
    
    // Helper to run the test twice - once with flag disabled, once with flag enabled
    func runTestWithFlagVariants(flagName: String, testName: String, setupAction: () -> Void, snapshotAction: () -> Void) {
        // First run: flag disabled (default state)
        app = XCUIApplication()
        app.launchArguments = ["UI-TESTING", "DISABLE-ANIMATIONS"]
        app.launch()
        
        setupAction()
        snapshotAction()
        self.assertSnapshotWithName(named: "\(testName)_Default")
        
        // Second run: flag enabled
        app.terminate()
        app = XCUIApplication()
        
        let isAnimationTest = flagName.contains("ANIMATION")
        
        if isAnimationTest {
            app.launchArguments = ["UI-TESTING", "ENABLE_FLAG_\(flagName)"]
        } else {
            app.launchArguments = ["UI-TESTING", "DISABLE-ANIMATIONS", "ENABLE_FLAG_\(flagName)"]
        }
        
        app.launch()
        
        setupAction()
        snapshotAction()
        self.assertSnapshotWithName(named: "\(testName)_FlagEnabled")
    }
    
    // Helper to run test with multiple flags
    func runTestWithMultipleFlags(flagNames: [String], testName: String, setupAction: () -> Void, snapshotAction: () -> Void) {
        // First run: all flags disabled (default state)
        app = XCUIApplication()
        app.launchArguments = ["UI-TESTING", "DISABLE-ANIMATIONS"]
        app.launch()
        
        setupAction()
        snapshotAction()
        self.assertSnapshotWithName(named: "\(testName)_Default")
        
        // Second run: all flags enabled
        app.terminate()
        app = XCUIApplication()
        
        let hasAnimationFlag = flagNames.contains { $0.contains("ANIMATION") }
        
        var launchArgs = ["UI-TESTING"]
        
        if !hasAnimationFlag {
            launchArgs.append("DISABLE-ANIMATIONS")
        }
        
        for flagName in flagNames {
            launchArgs.append("ENABLE_FLAG_\(flagName)")
        }
        
        app.launchArguments = launchArgs
        app.launch()
        
        setupAction()
        snapshotAction()
        self.assertSnapshotWithName(named: "\(testName)_FlagEnabled")
    }
    
    // A cleaner approach that still uses strings but with categorization
    enum FlagCategory: String {
        case PADDING_CHANGE
        case COLOR_CHANGE
        case TEXT_CHANGE
        case LAYOUT_CHANGE
        case ANIMATION_CHANGE
        case CONTENT_CHANGE
        case ANIMATION_PHASE
    }
    
    // Define common flags as static properties for convenience
    struct Flag {
        // PADDING_CHANGE flags
        static let PADDING_CHANGE_PREMIUM_BANNER = "PADDING_CHANGE_PREMIUM_BANNER"
        static let PADDING_CHANGE_PROFILE_INFO = "PADDING_CHANGE_PROFILE_INFO"
        static let PADDING_CHANGE_FEATURE_CARDS = "PADDING_CHANGE_FEATURE_CARDS"
        
        // COLOR_CHANGE flags
        static let COLOR_CHANGE_PREMIUM_BADGE = "COLOR_CHANGE_PREMIUM_BADGE"
        static let COLOR_CHANGE_TOGGLE_SWITCH = "COLOR_CHANGE_TOGGLE_SWITCH"
        static let COLOR_CHANGE_PROFILE_STATS = "COLOR_CHANGE_PROFILE_STATS"
        
        // TEXT_CHANGE flags
        static let TEXT_CHANGE_PREMIUM_BUTTON = "TEXT_CHANGE_PREMIUM_BUTTON"
        static let TEXT_CHANGE_PROFILE_HEADING = "TEXT_CHANGE_PROFILE_HEADING"
        
        // LAYOUT_CHANGE flags
        static let LAYOUT_CHANGE_SETTINGS_SECTIONS = "LAYOUT_CHANGE_SETTINGS_SECTIONS"
        static let LAYOUT_CHANGE_PROFILE_AVATAR = "LAYOUT_CHANGE_PROFILE_AVATAR"
        
        // ANIMATION_CHANGE flags
        static let ANIMATION_CHANGE_LOADING_SPINNER = "ANIMATION_CHANGE_LOADING_SPINNER"
        static let ANIMATION_CHANGE_BUTTON_PRESS = "ANIMATION_CHANGE_BUTTON_PRESS"
        
        // CONTENT_CHANGE flags
        static let CONTENT_CHANGE_PREMIUM_FEATURES = "CONTENT_CHANGE_PREMIUM_FEATURES"
        static let CONTENT_CHANGE_WELCOME_MESSAGE = "CONTENT_CHANGE_WELCOME_MESSAGE"
        
        // ANIMATION_PHASE flags
        static let ANIMATION_PHASE_PREMIUM_PULSE = "ANIMATION_PHASE_PREMIUM_PULSE"
        static let ANIMATION_PHASE_PROFILE_PHOTO_ROTATE = "ANIMATION_PHASE_PROFILE_PHOTO_ROTATE"
        
        // Helper to get the category of a flag
        static func getCategory(forFlag flag: String) -> FlagCategory? {
            if let firstPart = flag.split(separator: "_").dropLast().joined(separator: "_") as String?,
               let category = FlagCategory(rawValue: firstPart) {
                return category
            }
            return nil
        }
    }
    
    public func assertSnapshot<Value, Format>(
        matching value: @autoclosure () throws -> Value,
        as snapshotting: Snapshotting<Value, Format>,
        named name: String? = nil,
        record recording: Bool = false,
        timeout: TimeInterval = 5,
        file: StaticString = #file,
        testName: String = #function,
        line: UInt = #line
        ) {

        let failure = verifySnapshot(
        matching: try value(),
        as: snapshotting,
        named: name,
        record: recording,
        timeout: timeout,
        file: file,
        testName: testName,
        line: line
        )
                          
        if let message = failure {
            XCTFail(message, file: file, line: line)
#if !DEBUG
            XCTAssertTrue(false)
#endif
        }
                    
    }
    
    func assertSnapshotWithName(named: String, suffix: String? = nil, screenshot: UIImage? = nil) {
        var snapshotName = named
        if let suffix {
            if !suffix.isEmpty {
                snapshotName += "_\(suffix)"
            }
        } else if !currentTestSettings.suffix.isEmpty {
            snapshotName += "_\(currentTestSettings.suffix)"
        }
        if currentTestSettings.takeSnapshot {
            WaitUtility.shared.waitForDisappear("LoadingProgress")
            
            let finalScreenshot = screenshot ?? app.getScreen()
            assertSnapshot(matching: finalScreenshot, as: .image(precision: currentTestSettings.snapshotPrecision), named: snapshotName, testName: currentTestSettings.testName)
        }
    }
}
