import SwiftUI

struct SettingsScreen: View {
    @ObservedObject var stateManager = UserStateManager.shared
    @State private var darkMode = false
    @State private var notificationsEnabled = true
    @State private var autoUpdateEnabled = true
    @State private var selectedLocale = "en_US"
    
    private let locales = [
        ("en_US", "English (US)"),
        ("en_GB", "English (UK)"),
        ("es_ES", "Spanish"),
        ("fr_FR", "French"),
        ("de_DE", "German"),
        ("ja_JP", "Japanese")
    ]
    
    var body: some View {
        List {
            // If layout change flag is enabled, we change the order of sections
            if FeatureFlags.isEnabled(.LAYOUT_CHANGE_SETTINGS_SECTIONS) {
                appearanceSection
                appSettingsSection
                notificationsSection
                accountSection
                privacySection
                debugSection
            } else {
                accountSection
                appearanceSection
                notificationsSection
                appSettingsSection
                privacySection
                debugSection
            }
        }
        .navigationTitle("Settings")
        .accessibilityIdentifier("SettingsScreen")
    }
    
    private var accountSection: some View {
            Section(header: Text("Account")) {
                HStack {
                    Text("Account Type")
                    Spacer()
                    Text(stateManager.userType == .premium ? "Premium" : "Free")
                        .foregroundColor(.secondary)
                        .accessibilityIdentifier("AccountTypeValue")
                }
                
                if stateManager.userType == .free {
                    Button(action: {
                        stateManager.toggleUserType()
                    }) {
                        Text("Upgrade to Premium")
                            .foregroundColor(.blue)
                    }
                    .accessibilityIdentifier("UpgradeButton")
                } else {
                    Button(action: {
                        stateManager.toggleUserType()
                    }) {
                        Text("Manage Subscription")
                            .foregroundColor(.blue)
                    }
                    .accessibilityIdentifier("ManageSubscriptionButton")
            }
                }
            }
            
    private var appearanceSection: some View {
            Section(header: Text("Appearance")) {
                Toggle("Dark Mode", isOn: $darkMode)
                    .accessibilityIdentifier("DarkModeToggle")
                .toggleStyle(SwitchToggleStyle(tint: FeatureFlags.isEnabled(.COLOR_CHANGE_TOGGLE_SWITCH) ? .red : .blue))
                
                Picker("Language", selection: $selectedLocale) {
                    ForEach(locales, id: \.0) { locale in
                        Text(locale.1).tag(locale.0)
                    }
                }
                .accessibilityIdentifier("LanguagePicker")
                .onChange(of: selectedLocale) { newValue in
                    stateManager.setLocale(newValue)
            }
                }
            }
            
    private var notificationsSection: some View {
            Section(header: Text("Notifications")) {
                Toggle("Enable Notifications", isOn: $notificationsEnabled)
                .accessibilityIdentifier("NotificationsToggle")
                .toggleStyle(SwitchToggleStyle(tint: FeatureFlags.isEnabled(.COLOR_CHANGE_TOGGLE_SWITCH) ? .red : .blue))
                
                if notificationsEnabled {
                    Toggle("Promotional Emails", isOn: .constant(stateManager.userType == .free))
                        .disabled(stateManager.userType == .free)
                    .accessibilityIdentifier("PromoEmailsToggle")
                    .toggleStyle(SwitchToggleStyle(tint: FeatureFlags.isEnabled(.COLOR_CHANGE_TOGGLE_SWITCH) ? .red : .blue))
                    
                    if stateManager.userType == .free {
                        Text("Upgrade to Premium to disable promotional emails")
                            .font(.caption)
                            .foregroundColor(.secondary)
                }
                    }
                }
            }
            
    private var appSettingsSection: some View {
            Section(header: Text("App Settings")) {
                Toggle("Automatic Updates", isOn: $autoUpdateEnabled)
                .accessibilityIdentifier("AutoUpdatesToggle")
                .toggleStyle(SwitchToggleStyle(tint: FeatureFlags.isEnabled(.COLOR_CHANGE_TOGGLE_SWITCH) ? .red : .blue))
                
                Button(action: {

                }) {
                    Text("Clear Cache")
                        .foregroundColor(.blue)
                }
            .accessibilityIdentifier("ClearCacheButton")
                
                HStack {
                    Text("Version")
                    Spacer()
                    Text("1.0.0")
                        .foregroundColor(.secondary)
            }
                }
            }
            
    private var privacySection: some View {
            Section(header: Text("Privacy")) {
                NavigationLink(destination: Text("Privacy Policy would be shown here")) {
                    Text("Privacy Policy")
                }
            .accessibilityIdentifier("PrivacyPolicyLink")
                
                NavigationLink(destination: Text("Terms of Service would be shown here")) {
                    Text("Terms of Service")
                }
            .accessibilityIdentifier("TermsLink")
                
                if stateManager.userType == .premium {
                    NavigationLink(destination: Text("Data Export would be shown here")) {
                        Text("Export My Data")
                    }
                .accessibilityIdentifier("ExportDataLink")
                } else {
                    HStack {
                        Text("Export My Data")
                        Spacer()
                        Text("Premium Only")
                            .foregroundColor(.secondary)
                    }
                .accessibilityIdentifier("ExportDataDisabled")
            }
                }
            }
            
    private var debugSection: some View {
            Section(header: Text("Debug")) {
                Button(action: {
                    stateManager.updateRandomGreeting()
                }) {
                    Text("Randomize Greeting")
                        .foregroundColor(.blue)
                }
                .accessibilityIdentifier("RandomizeGreetingButton")
                
                Button(action: {
                    stateManager.toggleUserType()
                }) {
                    Text("Toggle User Type")
                        .foregroundColor(.blue)
                }
                .accessibilityIdentifier("ToggleUserTypeButton")
                
                Button(action: {
                    stateManager.resetToDefault()
                }) {
                    Text("Reset All Settings")
                        .foregroundColor(.red)
                }
                .accessibilityIdentifier("ResetSettingsButton")
            }
    }
}

#Preview {
    NavigationView {
        SettingsScreen()
    }
} 
