import SwiftUI

struct ProfileScreen: View {
    @ObservedObject var stateManager = UserStateManager.shared
    @State private var isEditing = false
    @State private var name = "John Doe"
    @State private var email = "john.doe@example.com"
    @State private var rotation: Double = 0
    @State private var animationsEnabled = TestingHelper.shouldEnableAnimations
    
    var body: some View {
        ScrollView {
            VStack(spacing: 24) {
                VStack {
                    if FeatureFlags.isEnabled(.LAYOUT_CHANGE_PROFILE_AVATAR) {
                        HStack(spacing: 16) {
                            Image(systemName: "person.crop.circle.fill")
                                .font(.system(size: 60))
                                .foregroundColor(stateManager.userType == .premium ? 
                                               (FeatureFlags.isEnabled(.COLOR_CHANGE_PREMIUM_BADGE) ? .purple : .yellow) : 
                                               .blue)
                                .rotationEffect(.degrees(rotation))
                                .accessibilityIdentifier("ProfileAvatar")
                            
                            VStack(alignment: .leading, spacing: 4) {
                                Text(name)
                                    .font(.title)
                                    .fontWeight(.bold)
                                
                                Text(email)
                                    .font(.subheadline)
                                    .foregroundColor(.secondary)
                                
                                Text(stateManager.userType == .premium ? "Premium Member" : "Free User")
                                    .font(.caption)
                                    .fontWeight(.semibold)
                                    .foregroundColor(stateManager.userType == .premium ? .white : .blue)
                                    .padding(.horizontal, 12)
                                    .padding(.vertical, 6)
                                    .background(
                                        stateManager.userType == .premium ? 
                                        (FeatureFlags.isEnabled(.COLOR_CHANGE_PREMIUM_BADGE) ? Color.purple : Color.yellow) : 
                                        Color.blue.opacity(0.1)
                                    )
                                    .cornerRadius(12)
                                    .accessibilityIdentifier("PremiumBadge")
                            }
                        }
                        .padding()
                    } else {
                    Image(systemName: "person.crop.circle.fill")
                        .font(.system(size: 80))
                            .foregroundColor(stateManager.userType == .premium ? 
                                          (FeatureFlags.isEnabled(.COLOR_CHANGE_PREMIUM_BADGE) ? .purple : .yellow) : 
                                          .blue)
                            .rotationEffect(.degrees(rotation))
                            .accessibilityIdentifier("ProfileAvatar")
                    
                    Text(name)
                        .font(.title)
                        .fontWeight(.bold)
                    
                    Text(email)
                        .font(.subheadline)
                        .foregroundColor(.secondary)
                    
                    Text(stateManager.userType == .premium ? "Premium Member" : "Free User")
                        .font(.caption)
                        .fontWeight(.semibold)
                        .foregroundColor(stateManager.userType == .premium ? .white : .blue)
                        .padding(.horizontal, 12)
                        .padding(.vertical, 6)
                            .background(
                                stateManager.userType == .premium ? 
                                (FeatureFlags.isEnabled(.COLOR_CHANGE_PREMIUM_BADGE) ? Color.purple : Color.yellow) : 
                                Color.blue.opacity(0.1)
                            )
                        .cornerRadius(12)
                        .padding(.top, 8)
                            .accessibilityIdentifier("PremiumBadge")
                    }
                }
                .padding(.bottom, FeatureFlags.isEnabled(.LAYOUT_CHANGE_PROFILE_AVATAR) ? 8 : 0)
                .onAppear {
                    if animationsEnabled && FeatureFlags.isEnabled(.ANIMATION_PHASE_PROFILE_PHOTO_ROTATE) {
                        rotation = 15
                        
                        DispatchQueue.main.asyncAfter(deadline: .now() + 0.1) {
                            withAnimation(Animation.linear(duration: 1.5).repeatForever(autoreverses: false)) {
                                rotation = 360
                            }
                        }
                    }
                }
                
                VStack(alignment: .leading, spacing: 0) {
                    Text(FeatureFlags.isEnabled(.TEXT_CHANGE_PROFILE_HEADING) ? "Your Information" : "Profile Information")
                        .font(.headline)
                        .padding()
                        .accessibilityIdentifier("ProfileInfoHeading")
                    
                    Divider()
                    
                    if isEditing {
                        editableProfileInfo
                    } else {
                        profileInfoRow(title: "Name", value: name)
                        profileInfoRow(title: "Email", value: email)
                        profileInfoRow(title: "Account Type", value: stateManager.userType == .premium ? "Premium" : "Free")
                        if stateManager.userType == .premium {
                            profileInfoRow(title: "Member Since", value: "January 1, 2023")
                            profileInfoRow(title: "Renewal Date", value: "January 1, 2024")
                        }
                    }
                }
                .background(Color(.systemBackground))
                .cornerRadius(12)
                .shadow(color: Color.black.opacity(0.1), radius: 5, x: 0, y: 2)
                .padding(.horizontal)
                .padding(.top, FeatureFlags.isEnabled(.PADDING_CHANGE_PROFILE_INFO) ? 16 : 0)
                
                if stateManager.userType == .premium {
                    VStack(alignment: .leading, spacing: 16) {
                        Text("Your Premium Statistics")
                            .font(.headline)
                            .padding(.horizontal)
                        
                        HStack {
                            statCard(value: "35", label: "Downloads")
                            statCard(value: "128", label: "Favorites")
                            statCard(value: "4.8", label: "Rating")
                        }
                        .padding(.horizontal)
                    }
                    .padding(.vertical)
                    .background(Color(.systemBackground))
                    .cornerRadius(12)
                    .shadow(color: Color.black.opacity(0.1), radius: 5, x: 0, y: 2)
                    .padding(.horizontal)
                } else {
                    VStack(alignment: .leading, spacing: 12) {
                        Text("Upgrade to Premium to see your detailed statistics and unlock more features!")
                            .font(.subheadline)
                            .foregroundColor(.secondary)
                            .multilineTextAlignment(.center)
                            .padding()
                    }
                    .frame(maxWidth: .infinity)
                    .padding(.vertical)
                    .background(Color(.systemBackground))
                    .cornerRadius(12)
                    .shadow(color: Color.black.opacity(0.1), radius: 5, x: 0, y: 2)
                    .padding(.horizontal)
                }
                
                Button(action: {
                    withAnimation {
                        isEditing.toggle()
                    }
                }) {
                    Text(isEditing ? "Save Profile" : "Edit Profile")
                        .font(.headline)
                        .foregroundColor(.white)
                        .padding()
                        .frame(maxWidth: .infinity)
                        .background(Color.blue)
                        .cornerRadius(10)
                        .padding(.horizontal)
                }
                .accessibilityIdentifier("EditProfileButton")
            }
            .padding(.vertical)
        }
        .navigationTitle("Profile")
    }
    
    private var editableProfileInfo: some View {
        VStack(alignment: .leading, spacing: 0) {
            editField(title: "Name", binding: $name)
            editField(title: "Email", binding: $email)
            profileInfoRow(title: "Account Type", value: stateManager.userType == .premium ? "Premium" : "Free")
            if stateManager.userType == .premium {
                profileInfoRow(title: "Member Since", value: "January 1, 2023")
                profileInfoRow(title: "Renewal Date", value: "January 1, 2024")
            }
        }
    }
    
    private func editField(title: String, binding: Binding<String>) -> some View {
        VStack(alignment: .leading, spacing: 8) {
            Text(title)
                .font(.subheadline)
                .foregroundColor(.secondary)
            
            TextField("", text: binding)
                .font(.body)
                .padding(12)
                .background(Color(.systemGray6))
                .cornerRadius(8)
                .padding(.bottom, 16)
        }
        .padding(.horizontal)
        .padding(.top, 16)
    }
    
    private func profileInfoRow(title: String, value: String) -> some View {
        VStack(alignment: .leading, spacing: 4) {
            Text(title)
                .font(.subheadline)
                .foregroundColor(.secondary)
            
            Text(value)
                .font(.body)
            
            Divider()
                .padding(.top, 12)
        }
        .padding(.horizontal)
        .padding(.top, 16)
    }
    
    private func statCard(value: String, label: String) -> some View {
        VStack(spacing: 4) {
            Text(value)
                .font(.system(size: 24, weight: .bold))
                .foregroundColor(.primary)
            
            Text(label)
                .font(.caption)
                .foregroundColor(.secondary)
        }
        .frame(maxWidth: .infinity)
        .padding(.vertical, 16)
        .background(FeatureFlags.isEnabled(.COLOR_CHANGE_PROFILE_STATS) ? Color.green.opacity(0.05) : Color.blue.opacity(0.05))
        .cornerRadius(8)
    }
}

#Preview {
    NavigationView {
        ProfileScreen()
    }
} 
