import SwiftUI

struct HomeScreen: View {
    @ObservedObject var stateManager = UserStateManager.shared
    @State private var isLoading = false
    var animationsEnabled: Bool = TestingHelper.shouldEnableAnimations
    
    var body: some View {
        ScrollView {
            VStack(spacing: 24) {
                VStack(spacing: 8) {
                    Text(FeatureFlags.isEnabled(.CONTENT_CHANGE_WELCOME_MESSAGE) ? 
                        "Welcome back, \(stateManager.userType == .premium ? "Premium User" : "User")!" : 
                        stateManager.greeting)
                        .font(.largeTitle)
                        .fontWeight(.bold)
                        .foregroundColor(.primary)
                        .frame(maxWidth: .infinity, alignment: .leading)
                    
                    Text(FeatureFlags.isEnabled(.CONTENT_CHANGE_WELCOME_MESSAGE) ? 
                        "Here's what's new today:" : 
                        "What would you like to explore today?")
                        .font(.headline)
                        .foregroundColor(.secondary)
                        .frame(maxWidth: .infinity, alignment: .leading)
                }
                .padding(.horizontal)
                
                LazyVGrid(columns: [GridItem(.flexible()), GridItem(.flexible())], spacing: 16) {
                    featureCard(icon: "photo.fill", title: "Photos")
                    featureCard(icon: "video.fill", title: "Videos")
                    featureCard(icon: "music.note", title: "Music")
                    featureCard(icon: "doc.fill", title: "Documents")
                }
                .padding(.horizontal)
                
                if stateManager.userType == .free {
                    PremiumBanner(animationsEnabled: animationsEnabled)
                } else {
                    PremiumContent()
                }
                
                if FeatureFlags.isEnabled(.CONTENT_CHANGE_PREMIUM_FEATURES) && stateManager.userType == .premium {
                    VStack(spacing: 16) {
                        Text("Premium Features")
                            .font(.headline)
                            .frame(maxWidth: .infinity, alignment: .leading)
                        
                        HStack(spacing: 16) {
                            premiumFeatureCard(icon: "shield.fill", title: "Advanced Security")
                            premiumFeatureCard(icon: "arrow.up.bin.fill", title: "Unlimited Storage")
                        }
                        
                        HStack(spacing: 16) {
                            premiumFeatureCard(icon: "person.2.fill", title: "Family Sharing")
                            premiumFeatureCard(icon: "bolt.fill", title: "Priority Support")
                        }
                    }
                    .padding()
                    .background(Color(.systemBackground))
                    .cornerRadius(12)
                    .shadow(color: Color.black.opacity(0.1), radius: 5, x: 0, y: 2)
                    .padding(.horizontal)
                    .accessibilityIdentifier("PremiumFeaturesCard")
                }
                
                VStack(alignment: .leading, spacing: 12) {
                    Text("Recent Activity")
                        .font(.title2)
                        .fontWeight(.bold)
                    
                    ForEach(1...3, id: \.self) { index in
                        recentActivityRow(
                            title: "Activity \(index)",
                            subtitle: "Last accessed: Today",
                            icon: ["star.fill", "heart.fill", "flag.fill"][index - 1]
                        )
                    }
                }
                .padding()
                .background(Color(.systemBackground))
                .cornerRadius(12)
                .shadow(color: Color.black.opacity(0.1), radius: 5, x: 0, y: 2)
                .padding(.horizontal)
                
                Button(action: {
                    isLoading = true
                    DispatchQueue.main.asyncAfter(deadline: .now() + 3) {
                        stateManager.updateRandomGreeting()
                        isLoading = false
                    }
                }) {
                    Text("Refresh Content")
                        .font(.headline)
                        .foregroundColor(.blue)
                        .padding()
                        .frame(maxWidth: .infinity)
                        .background(Color.blue.opacity(0.1))
                        .cornerRadius(10)
                        .padding(.horizontal)
                }
                .accessibilityIdentifier("RefreshButton")
                
                Button(action: {
                    stateManager.toggleUserType()
                }) {
                    Text("Change User Type")
                        .font(.headline)
                        .foregroundColor(.blue)
                        .padding()
                        .frame(maxWidth: .infinity)
                        .background(Color.blue.opacity(0.1))
                        .cornerRadius(10)
                        .padding(.horizontal)
                }
                .accessibilityIdentifier("ChangeUserTypeButton")
                
                Button(action: {
                    stateManager.updateRandomGreeting()
                }) {
                    Text("Change Greeting")
                        .font(.headline)
                        .foregroundColor(.blue)
                        .padding()
                        .frame(maxWidth: .infinity)
                        .background(Color.blue.opacity(0.1))
                        .cornerRadius(10)
                        .padding(.horizontal)
                }
            }
            .padding(.vertical)
        }
        .navigationTitle("Home")
        .loading(
            isLoading: isLoading,
            animationsEnabled: animationsEnabled,
            frozenRotationDegrees: animationsEnabled ? nil : TestingHelper.fixedRotationDegrees
        )
    }
    
    private func featureCard(icon: String, title: String) -> some View {
        VStack(spacing: 12) {
            Image(systemName: icon)
                .font(.system(size: 24))
                .foregroundColor(.blue)
            
            Text(title)
                .font(.headline)
                .foregroundColor(.primary)
        }
        .frame(maxWidth: .infinity)
        .frame(height: 100)
        .padding(FeatureFlags.isEnabled(.PADDING_CHANGE_FEATURE_CARDS) ? 12 : 0)
        .background(Color(.systemBackground))
        .cornerRadius(12)
        .shadow(color: Color.black.opacity(0.1), radius: 3, x: 0, y: 2)
        .accessibilityIdentifier("FeatureCard-\(title)")
    }
    
    private func premiumFeatureCard(icon: String, title: String) -> some View {
        VStack(spacing: 10) {
            Image(systemName: icon)
                .font(.system(size: 20))
                .foregroundColor(.yellow)
            
            Text(title)
                .font(.subheadline)
                .foregroundColor(.primary)
                .multilineTextAlignment(.center)
        }
        .frame(maxWidth: .infinity)
        .frame(height: 80)
        .padding(8)
        .background(Color(.secondarySystemBackground))
        .cornerRadius(10)
        .shadow(color: Color.black.opacity(0.05), radius: 2, x: 0, y: 1)
        .accessibilityIdentifier("PremiumFeature-\(title)")
    }
    
    private func recentActivityRow(title: String, subtitle: String, icon: String) -> some View {
        HStack(spacing: 16) {
            Image(systemName: icon)
                .font(.title2)
                .foregroundColor(.blue)
                .frame(width: 40, height: 40)
                .background(Color.blue.opacity(0.1))
                .cornerRadius(8)
            
            VStack(alignment: .leading, spacing: 4) {
                Text(title)
                    .font(.headline)
                    .foregroundColor(.primary)
                
                Text(subtitle)
                    .font(.subheadline)
                    .foregroundColor(.secondary)
            }
            
            Spacer()
            
            Image(systemName: "chevron.right")
                .foregroundColor(.gray)
        }
        .padding(.vertical, 8)
    }
}

#Preview {
    NavigationView {
        HomeScreen()
    }
} 
