import SwiftUI

struct LoadingSpinner: View {
    @State private var isRotating = false
    @State private var rotationDegrees: Double
    
    var animationsEnabled: Bool
    
    init(animationsEnabled: Bool = true, initialRotationDegrees: Double? = nil) {
        self.animationsEnabled = animationsEnabled
        self._rotationDegrees = State(initialValue: initialRotationDegrees ?? 0)
    }
    
    var body: some View {
        ZStack {
            Circle()
                .stroke(lineWidth: 4)
                .frame(width: 50, height: 50)
                .foregroundColor(Color.gray.opacity(0.3))
            
            Circle()
                .trim(from: 0, to: 0.7)
                .stroke(
                    FeatureFlags.isEnabled(.ANIMATION_CHANGE_LOADING_SPINNER) ? 
                        LinearGradient(gradient: Gradient(colors: [.blue, .purple]), startPoint: .leading, endPoint: .trailing) :
                        LinearGradient(gradient: Gradient(colors: [.blue, .blue.opacity(0.5)]), startPoint: .leading, endPoint: .trailing),
                    style: StrokeStyle(lineWidth: 4, lineCap: .round)
                )
                .frame(width: 50, height: 50)
                .rotationEffect(Angle(degrees: rotationDegrees))
                .animation(
                    animationsEnabled ? 
                    Animation.linear(duration: FeatureFlags.isEnabled(.ANIMATION_CHANGE_LOADING_SPINNER) ? 1.5 : 1.0)
                        .repeatForever(autoreverses: false) : .none,
                    value: isRotating
                )
        }
        .onAppear {
            isRotating = true
            
            if animationsEnabled && rotationDegrees == 0 {
                withAnimation(Animation.linear(duration: FeatureFlags.isEnabled(.ANIMATION_CHANGE_LOADING_SPINNER) ? 1.5 : 1.0).repeatForever(autoreverses: false)) {
                    rotationDegrees = 360
                }
            } else {
                rotationDegrees = 360
            }
        }
        .accessibilityIdentifier("LoadingSpinner")
    }
}

struct LoadingSpinnerModifier: ViewModifier {
    var isLoading: Bool
    var animationsEnabled: Bool = true
    var frozenRotationDegrees: Double? = nil
    
    func body(content: Content) -> some View {
        ZStack {
            content
                .disabled(isLoading)
                .blur(radius: isLoading ? 2 : 0)
            
            if isLoading {
                Color.black.opacity(0.2)
                    .ignoresSafeArea()
                
                LoadingSpinner(
                    animationsEnabled: animationsEnabled,
                    initialRotationDegrees: frozenRotationDegrees
                )
            }
        }
    }
}

extension View {
    func loading(
        isLoading: Bool, 
        animationsEnabled: Bool = true,
        frozenRotationDegrees: Double? = nil
    ) -> some View {
        self.modifier(
            LoadingSpinnerModifier(
                isLoading: isLoading,
                animationsEnabled: animationsEnabled,
                frozenRotationDegrees: frozenRotationDegrees
            )
        )
    }
}

#Preview {
    VStack {
        LoadingSpinner()
        
        Text("Stuff")
            .padding()
            .loading(isLoading: true, frozenRotationDegrees: 180)
    }
} 
