package jupyter

import (
	"encoding/json"
	"errors"
	"os"
	"sync"
	"time"

	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
)

var (
	ErrAlreadyConnectedToKernel = errors.New("session is already connected to its kernel")
	ErrRequestTimedOut          = errors.New("the request timed out")
)

// The difference between this and `SessionModel` is that this struct has a different type for the `JupyterNotebook` field so that it isn't null in the request.
type jupyterSessionReq struct {
	LocalSessionId   string                 `json:"-"`
	JupyterSessionId string                 `json:"id"`
	Path             string                 `json:"path"`
	Name             string                 `json:"name"`
	SessionType      string                 `json:"type"`
	JupyterKernel    *jupyterKernel         `json:"kernel"`
	JupyterNotebook  map[string]interface{} `json:"notebook"`
	ResourceSpec     *ResourceSpec          `json:"resource_spec"`
	WorkloadId       string                 `json:"workload_id"`

	SessionConnection *SessionConnection `json:"-"`
}

func newJupyterSessionForRequest(sessionName string, sessionPath string, sessionType string, kernelSpecName string,
	resourceSpec *ResourceSpec, workloadId string) *jupyterSessionReq {
	jupyterKernel := newJupyterKernel(sessionName, kernelSpecName)

	req := &jupyterSessionReq{
		JupyterSessionId: sessionName,
		LocalSessionId:   sessionName,
		Path:             sessionPath,
		Name:             sessionName,
		SessionType:      sessionType,
		JupyterKernel:    jupyterKernel,
		JupyterNotebook:  make(map[string]interface{}),
		WorkloadId:       workloadId,
	}

	if resourceSpec != nil {
		req.ResourceSpec = resourceSpec
	}

	return req
}

func (s *jupyterSessionReq) String() string {
	out, err := json.Marshal(s)
	if err != nil {
		panic(err)
	}

	return string(out)
}

type SessionModel struct {
	LocalSessionId   string           `json:"-"`
	JupyterSessionId string           `json:"id"`
	Path             string           `json:"path"`
	Name             string           `json:"name"`
	SessionType      string           `json:"type"`
	JupyterKernel    *jupyterKernel   `json:"kernel"`
	JupyterNotebook  *jupyterNotebook `json:"notebook"`

	SessionConnection *SessionConnection `json:"-"`
}

func (s *SessionModel) String() string {
	out, err := json.Marshal(s)
	if err != nil {
		panic(err)
	}

	return string(out)
}

type jupyterKernel struct {
	Id             string `json:"id"`
	Name           string `json:"name"`
	LastActivity   string `json:"last_activity"`
	ExecutionState string `json:"execution_state"`
	Connections    int    `json:"connections"`
}

func (k *jupyterKernel) String() string {
	out, err := json.Marshal(k)
	if err != nil {
		panic(err)
	}

	return string(out)
}

func newJupyterKernel(id string, name string) *jupyterKernel {
	return &jupyterKernel{
		Id:   id,
		Name: name,
	}
}

type jupyterNotebook struct {
	Path string `json:"path"`
	Name string `json:"name"`
}

type Session struct {
	id        string // The ID of the session that the user/trace data supplied.
	jupyterId string // The session ID generated by Jupyter.
	kernelId  string // The ID of the kernel associated with the Session.
}

func NewSession(id string, jupyterId string, kernelId string) *Session {
	sess := &Session{
		id:        id,
		jupyterId: jupyterId,
		kernelId:  kernelId,
	}

	return sess
}

type SessionConnection struct {
	model  *SessionModel
	Kernel KernelConnection

	jupyterServerAddress string

	// createdAt is the time at which the SessionConnection was created.
	createdAt time.Time

	// metadata is a map containing basic metadata used for labeling kernelMetricsManager.
	metadata      map[string]interface{}
	metadataMutex sync.Mutex

	logger        *zap.Logger
	sugaredLogger *zap.SugaredLogger
	atom          *zap.AtomicLevel

	// Used to record Prometheus metrics.
	metricsConsumer MetricsConsumer

	onError func(err error)
}

// NewSessionConnection creates a new SessionConnection.
//
// We do not return until we've successfully connected to the kernel.
func NewSessionConnection(model *SessionModel, username string, jupyterServerAddress string, atom *zap.AtomicLevel,
	metricsConsumer MetricsConsumer, onError func(err error)) (*SessionConnection, error) {

	conn := &SessionConnection{
		model:                model,
		jupyterServerAddress: jupyterServerAddress,
		atom:                 atom,
		createdAt:            time.Now(),
		metadata:             make(map[string]interface{}),
		metricsConsumer:      metricsConsumer,
		onError:              onError,
	}

	core := zapcore.NewCore(zapcore.NewConsoleEncoder(zap.NewDevelopmentEncoderConfig()), os.Stdout, atom)
	conn.logger = zap.New(core, zap.Development())

	conn.sugaredLogger = conn.logger.Sugar()

	err := conn.connectToKernel(username)
	if err != nil && !errors.Is(err, ErrAlreadyConnectedToKernel) {
		return nil, err
	}

	conn.logger.Debug("Successfully connected to kernel (as part of a SessionConnection).",
		zap.String("kernel_id", model.JupyterKernel.Id),
		zap.String("session_id", model.JupyterSessionId))

	return conn, err
}

// AddMetadata attaches some metadata to the SessionConnection and optionally to the associated KernelConnection,
// if one exists and if the 'addToKernelConnection' parameter is passed as true.
//
// If the KernelConnection exists and already has an entry in its metadata dictionary for the given key, then
// that metadata will be overwritten.
//
// This particular implementation of AddMetadata is thread-safe.
func (conn *SessionConnection) AddMetadata(key string, value interface{}, addToKernelConnection bool) error {
	conn.metadataMutex.Lock()
	defer conn.metadataMutex.Unlock()

	conn.metadata[key] = value

	if addToKernelConnection && conn.Kernel != nil {
		conn.logger.Debug("Adding metadata to kernel.", zap.String("kernel_id", conn.model.JupyterKernel.Id),
			zap.String("metadata_key", key), zap.Any("metadata_value", value))
		return conn.Kernel.AddMetadata(key, value)
	}

	return nil
}

// GetMetadata retrieves a piece of metadata that may be attached to the SessionConnection.
//
// This particular implementation of GetMetadata is thread-safe.
func (conn *SessionConnection) GetMetadata(key string) (interface{}, bool) {
	conn.metadataMutex.Lock()
	defer conn.metadataMutex.Unlock()

	value, ok := conn.metadata[key]
	if !ok {
		conn.logger.Warn("Could not find metadata with specified key attached to BasicKernelConnection.", zap.String("key", key))
	}
	return value, ok
}

// connectToKernel creates a new WebSocket-backed connection to the Kernel associated with this session.
// Side-effect: set the `kernel` field of the SessionConnection.
func (conn *SessionConnection) connectToKernel(username string) error {
	if conn.Kernel != nil {
		return ErrAlreadyConnectedToKernel
	}

	kernel, err := NewKernelConnection(conn.model.JupyterKernel.Id, conn.model.JupyterSessionId, username,
		conn.jupyterServerAddress, conn.atom, conn.metricsConsumer, conn.onError)

	if err != nil {
		return err
	}

	conn.Kernel = kernel

	// Add all the SessionConnection's metadata to the new KernelConnection.
	conn.metadataMutex.Lock()
	defer conn.metadataMutex.Unlock()

	var metadataErrors []error
	for key, value := range conn.metadata {
		conn.logger.Debug("Adding metadata to kernel.", zap.String("kernel_id", conn.model.JupyterKernel.Id),
			zap.String("metadata_key", key), zap.Any("metadata_value", value))
		metadataError := conn.Kernel.AddMetadata(key, value)

		if metadataError != nil {
			conn.logger.Error("Could not add metadata to kernel.",
				zap.String("kernel_id", conn.model.JupyterKernel.Id),
				zap.String("metadata_key", key),
				zap.Any("metadata_value", value))
			metadataErrors = append(metadataErrors, metadataError)
		}
	}

	return err // Will be nil if everything went OK.
}

func (conn *SessionConnection) RegisterIoPubHandler(id string, handler IOPubMessageHandler) error {
	return conn.Kernel.RegisterIoPubHandler(id, handler)
}

// UnregisterIoPubHandler unregisters a handler/consumer of IOPub messages that was registered under the specified ID.
func (conn *SessionConnection) UnregisterIoPubHandler(id string) error {
	return conn.Kernel.UnregisterIoPubHandler(id)
}
