"""
Author: Dominika Malinowska

This script defines functions for assessing vulnerability of bridges based on
the data from the LSB Database. The methodology is based on the Italian guidelines
for bridge vulnerability assessment.
"""

import sys
import numpy as np


def check_issues(row_issues, list_issues):
    """
    Check how many structural issues from the list are found for a given bridge
    If none, assign 0, if only 1 assign 1, if more than 1 assign 2

    Arguments:
        row_issues (set): A set of unique issues for a given bridge
        list_issues (set): List of potential issues for all bridges

    Returns:
        int: The degradation class of the bridge
    """
    # Find the intersection of the two sets
    common_issues = row_issues.intersection(list_issues)

    # Check the number of common issues and return the appropriate value
    if len(common_issues) == 0:
        return 0
    elif len(common_issues) == 1:
        return 1
    else:
        return 2


def assign_degradation_class(df, list_issues):
    """
    Assign a degradation class to each bridge based on the issues found

    Arguments:
        df (DataFrame): The input DataFrame with the bridge data
        list_issues (set): List of potential issues for all bridges

    Returns:
        DataFrame: The input DataFrame with the degradation class assigned
    """
    # Iterate over each row in the dataframe
    for index, row in df.iterrows():
        # Check if the health of the structure is not 'Deficient'
        if row["Health"] != "Deficient":
            # Create a set of unique issues for the current row
            row_issues = set(
                list(row[["Issues (1)", "Issues (2)", "Issues (3)"]].unique())
            )
            # Assign a degradation class based on the issues
            df.at[index, "Degradation_class"] = (
                check_issues(row_issues, list_issues) + 1
            )
    return df


def assign_degradation_speed_class(df):
    """
    Assign a degradation speed class to each bridge based on
    the degradation class and year of construction

    Arguments:
        df (DataFrame): The input DataFrame with the bridge data

    Returns:
        DataFrame: The input DataFrame with the degradation speed class assigned
    """
    # Iterate over each row in the dataframe
    for index, row in df.iterrows():
        # Get the year of construction or reconstruction, whichever is later
        year_construction = max(row["Construction Finished"], row["Year Reconstructed"])
        # Assign a degradation speed class based on the degradation class and year of construction
        if row["Degradation_class"] == 1:
            df.at[index, "Degradation_speed_class"] = (
                1 if year_construction < 1980 else 2
            )
        elif row["Degradation_class"] == 2:
            df.at[index, "Degradation_speed_class"] = (
                1
                if year_construction <= 1945
                else (2 if year_construction < 1980 else 3)
            )
        elif row["Degradation_class"] == 3:
            df.at[index, "Degradation_speed_class"] = (
                2
                if year_construction <= 1945
                else (3 if year_construction < 1980 else 4)
            )
        elif row["Degradation_class"] == 4:
            df.at[index, "Degradation_speed_class"] = (
                4 if year_construction < 1980 else 5
            )
    return df


def assign_design_code(df):
    """
    Assign a design code to each bridge based on the year of construction start

    Arguments:
        df (DataFrame): The input DataFrame with the bridge data

    Returns:
        DataFrame: The input DataFrame with the design code assigned
    """
    # Iterate over each row in the dataframe
    for index, row in df.iterrows():
        # Assign a design code based on the year of construction start
        if row["Construction Start"] < 1990:
            df.at[index, "Design_code"] = 1  # Class A
        elif row["Construction Start"] >= 1990 and row["Construction Start"] < 2008:
            df.at[index, "Design_code"] = 2  # Class B
        else:
            df.at[index, "Design_code"] = 3  # Class C
    return df


def assign_design_code_class(df):
    """
    Assign a design code class to each bridge based on the degradation speed class
    and design code

    Arguments:
        df (DataFrame): The input DataFrame with the bridge data

    Returns:
        DataFrame: The input DataFrame with the design code class assigned
    """
    # Iterate over each row in the dataframe
    for index, row in df.iterrows():
        # Assign a design code class based on the degradation speed class and design code
        if row["Degradation_speed_class"] == 1:
            df.at[index, "Design_code_class"] = 1
        elif row["Degradation_speed_class"] == 2:
            df.at[index, "Design_code_class"] = (
                3 if row["Design_code"] == 1 else (2 if row["Design_code"] == 2 else 1)
            )
        elif row["Degradation_speed_class"] == 3:
            df.at[index, "Design_code_class"] = (
                4 if row["Design_code"] == 1 else (3 if row["Design_code"] == 2 else 2)
            )
        elif row["Degradation_speed_class"] == 4:
            df.at[index, "Design_code_class"] = (
                5 if row["Design_code"] == 1 else (4 if row["Design_code"] == 2 else 3)
            )
        else:
            df.at[index, "Design_code_class"] = 5
    return df


def assign_material_class(df):
    """
    Assign a material class to each bridge based on the type of structure and materials used

    Arguments:
        df (DataFrame): The input DataFrame with the bridge data

    Returns:
        DataFrame: The input DataFrame with the material class assigned
    """
    # Iterate over each row in the dataframe
    for index, row in df.iterrows():
        # Assign a material class based on the type of structure and the materials used
        if row["Type"] == "Truss":
            df.at[index, "Material_class"] = (
                3
                if row["Material: Deck"] == "Steel"
                and row["Material: Piers/Pylons"] == "Steel"
                else 4
            )
        elif row["Type"] == "Arch":
            df.at[index, "Material_class"] = (
                5
                if row["Material: Deck"] == "Wrought Iron"
                and row["Material: Piers/Pylons"] == "Wrought Iron"
                else 4
            )
        elif row["Type"] == "Cable Stayed":
            df.at[index, "Material_class"] = (
                4
                if row["Material: Deck"] == "Steel"
                and row["Material: Piers/Pylons"] == "Steel"
                else 5
            )
        else:
            df.at[index, "Material_class"] = 5
    return df


def assign_vulnerability_class(df):
    """
    Assign a vulnerability class to each bridge based on the design code class
    and material class

    Arguments:
        df (DataFrame): The input DataFrame with the bridge data

    Returns:
        DataFrame: The input DataFrame with the vulnerability class assigned
    """

    # Iterate over each row in the dataframe
    for index, row in df.iterrows():
        # Assign a vulnerability class based on the design code class and material class
        if row["Design_code_class"] == 1:
            df.at[index, "Vulnerability_class"] = 1 if row["Material_class"] <= 3 else 2
        elif row["Design_code_class"] == 2:
            df.at[index, "Vulnerability_class"] = (
                1
                if row["Material_class"] == 1
                else (2 if row["Material_class"] <= 3 else 3)
            )
        elif row["Design_code_class"] == 3:
            df.at[index, "Vulnerability_class"] = (
                2
                if row["Material_class"] == 1
                else (3 if row["Material_class"] <= 3 else 4)
            )
        elif row["Design_code_class"] == 4:
            df.at[index, "Vulnerability_class"] = 3 if row["Material_class"] <= 2 else 4
        else:
            df.at[index, "Vulnerability_class"] = (
                3
                if row["Material_class"] == 1
                else (4 if row["Material_class"] <= 3 else 5)
            )
    return df


def get_vulnerability(df):
    """
    Get the vulnerability class for each bridge based on the data from the LSB Database

    Arguments:
        df (DataFrame): The input DataFrame with the bridge data

    Returns:
        DataFrame: The input DataFrame with the vulnerability class assigned
    """
    # Check if the minimum value of 'Maximum Span' is less than 25
    # In this case the code is not applicable, so stop the execution
    if np.min(df["Maximum Span"]) < 25:
        print("This code is for bridges with maximum span length greater than 25m")
        sys.exit("Stopping script execution.")

    # If the bridge's health is deficient, assign 5 as vulnerability score
    df.loc[
        df["Health"] == "Deficient",
        [
            "Degradation_class",
            "Degradation_speed_class",
            "Design_code_class",
            "Vulnerability_class",
        ],
    ] = 5

    # Fill nan values for further processing
    df[["Issues (1)", "Issues (2)", "Issues (3)"]] = df[
        ["Issues (1)", "Issues (2)", "Issues (3)"]
    ].fillna("nan")

    # Define a list of issues
    list_issues = set(["Moisture", "Wearing", "Corrosion", "Erosion", "Cracking"])

    # Assign degradation class
    # If no issues, class is 1, if only one issue, class is 2, if more than one issue, class is 3
    df = assign_degradation_class(df, list_issues)

    # Assign degradation speed class
    df = assign_degradation_speed_class(df)

    # Assign design code
    df = assign_design_code(df)

    # Assign design code class
    df = assign_design_code_class(df)

    # Assign material class
    df = assign_material_class(df)

    # Assign vulnerability class
    df = assign_vulnerability_class(df)

    return df
