%% Result calculation for embodied re and ff
    % A database for identifying and tracking renewable energy embodied in global trade
    % Copyright (c) 2025 Yan Cheng
% Input: IO variables, energy account, labels
% Output: embodied energy flow

%% Part 1 Basic output of MRIO
%% 1.1 Set
clc;
clear;
% 1) Parameters Set
c = 189;
s = 163;
n = c*s;
years = [2011:2022];

% 2) Dir Set
IOFolder = 'D:\DataCollection\REX3\01InputEmbodied';
EnergyFolder = 'D:\nutSync\我的坚果云\SciStudy\09paperWork\REAccount\Data\DataForCode\08REX3_Region_REX3_Sector_Final_V7';
outputFolder = 'D:\nutSync\我的坚果云\SciStudy\09paperWork\REAccount\Data\ResultOutput\01BasicMRIO_V7';  
if ~exist(outputFolder, 'dir')
    mkdir(outputFolder);
end


% 3) Label for results
% region and sector label
regionFile = 'D:\nutSync\我的坚果云\SciStudy\09paperWork\REAccount\Data\BasicData\01Map rule\REX3RegionLabel.xlsx';
sectorFile = 'D:\nutSync\我的坚果云\SciStudy\09paperWork\REAccount\Data\BasicData\01Map rule\REX3SectorLabel.xlsx';
sectorAggFile = 'D:\nutSync\我的坚果云\SciStudy\09paperWork\REAccount\Data\BasicData\01Map rule\REX3SectorAgg.xlsx';

regionTable = readtable(regionFile);  
sectorTable = readtable(sectorFile);  

regions = regionTable.region;  
sectors = sectorTable.sector; 

% label used for ET(cs2c)
ET_row_region = repelem(regions, s, 1);  % repeat each region s times
ET_row_sector = repmat(sectors, c, 1);  % repeat all sectors c times
ET_row_names = [ET_row_region, ET_row_sector]; 

% Sector Aggregation
aggData = readtable(sectorAggFile, 'PreserveVariableNames', true);
aggMatrix = table2array(aggData(:, 3:end)); % 0-1 matrix
aggSectors = aggData.Properties.VariableNames(3:end); % Aggregated sector name


%% 1.2 Yearly MRIO calculation
for i = 1:length(years)
    year = years(i);
    
    %% -- 
    % get IO data
    X_file = fullfile(IOFolder, sprintf('%d', year), 'X_REX3.mat');
    L_file = fullfile(IOFolder, sprintf('%d', year), 'L.mat');
    Ftotal_file = fullfile(IOFolder, sprintf('%d', year), 'Y_REX3.mat');
    
    load(X_file, 'X_REX3');
    load(L_file, 'L');
    load(Ftotal_file, 'Y_REX3');
    
    % get energy data
    ff_col_file = fullfile(EnergyFolder, 'ff_col', sprintf('ff_col_%d.csv', year));
    re_col_file = fullfile(EnergyFolder, 're_col', sprintf('re_col_%d.csv', year));
    
    E_ff = readmatrix(ff_col_file);  
    E_re = readmatrix(re_col_file);  
    
    % create output subfolder (year)
    yearOutputFolder = fullfile(outputFolder, sprintf('%d', year));
    if ~exist(yearOutputFolder, 'dir')
        mkdir(yearOutputFolder);
    end
    
    % create output subfolder (year-ff,year-re)
    ffOutputFolder = fullfile(yearOutputFolder, 'ff');
    reOutputFolder = fullfile(yearOutputFolder, 're');
    
    if ~exist(ffOutputFolder, 'dir')
        mkdir(ffOutputFolder);
    end
    if ~exist(reOutputFolder, 'dir')
        mkdir(reOutputFolder);
    end
    
    fprintf('Year %d - Folder loaded\n', year);


    %% --
    % MRIO calculation
    fprintf('Year %d - MRIO starts\n', year);

    addpath(genpath('D:\nutSync\我的坚果云\SciStudy\09paperWork\REAccount\CodeEora')); % MRIO function path

    [ET_ff, Ec2c_ff, Ecs_ff, country_aggre_ff] = MRIO_02(n, c, s, E_ff, X_REX3, L, Y_REX3);
    fprintf('Year %d - ff MRIO Done\n', year);
    [ET_re, Ec2c_re, Ecs_re, country_aggre_re] = MRIO_02(n, c, s, E_re, X_REX3, L, Y_REX3);
    fprintf('Year %d - re MRIO Done\n', year);


    %% -- 
    % save result
    %% 1) ET table with grouped c&s labels: cs2c [n x c]
    ET_ff_table = array2table(ET_ff, 'RowNames', cellstr(strcat(ET_row_region, '_', ET_row_sector)), 'VariableNames', regions);
    ET_re_table = array2table(ET_re, 'RowNames', cellstr(strcat(ET_row_region, '_', ET_row_sector)), 'VariableNames', regions);
    
    % split c&s labels
    ET_ff_table_with_csCol = [table(ET_row_region, ET_row_sector, 'VariableNames', {'Region', 'Sector'}), ET_ff_table];
    ET_re_table_with_csCol = [table(ET_row_region, ET_row_sector, 'VariableNames', {'Region', 'Sector'}), ET_re_table];
    
    writetable(ET_ff_table_with_csCol, fullfile(ffOutputFolder, 'ET.xlsx'), 'WriteRowNames', true);
    writetable(ET_re_table_with_csCol, fullfile(reOutputFolder, 'ET.xlsx'), 'WriteRowNames', true);
    
    %% 2) Ec2c [c x c]
    Ec2c_ff_table = array2table(Ec2c_ff, 'RowNames', regions, 'VariableNames', regions);
    Ec2c_re_table = array2table(Ec2c_re, 'RowNames', regions, 'VariableNames', regions);
    
    writetable(Ec2c_ff_table, fullfile(ffOutputFolder, 'Ec2c.xlsx'), 'WriteRowNames', true);
    writetable(Ec2c_re_table, fullfile(reOutputFolder, 'Ec2c.xlsx'), 'WriteRowNames', true);
    
    %% 3) Ecs [s x n]: Volume & Share
    % export volume
    Ecs_ff_table = array2table(Ecs_ff, 'RowNames', sectors, 'VariableNames', regions);
    Ecs_re_table = array2table(Ecs_re, 'RowNames', sectors, 'VariableNames', regions);
    
    writetable(Ecs_ff_table, fullfile(ffOutputFolder, 'Ecs_volume.xlsx'), 'WriteRowNames', true);
    writetable(Ecs_re_table, fullfile(reOutputFolder, 'Ecs_volume.xlsx'), 'WriteRowNames', true);

    % export share
    Ecs_ff_Ctotal = sum(Ecs_ff, 1); % total export of each region
    Ecs_ff_share = Ecs_ff./ Ecs_ff_Ctotal;
    Ecs_ff_share(isnan(Ecs_ff_share)) = 0; 

    Ecs_re_Ctotal = sum(Ecs_re, 1); % total export of each region
    Ecs_re_share = Ecs_re./ Ecs_re_Ctotal;
    Ecs_re_share(isnan(Ecs_re_share)) = 0; 

    Ecs_ff_share_table = array2table(Ecs_ff_share, 'RowNames', sectors, 'VariableNames', regions);
    Ecs_re_share_table = array2table(Ecs_re_share, 'RowNames', sectors, 'VariableNames', regions);
    
    writetable(Ecs_ff_share_table, fullfile(ffOutputFolder, 'Ecs_share.xlsx'), 'WriteRowNames', true);
    writetable(Ecs_re_share_table, fullfile(reOutputFolder, 'Ecs_share.xlsx'), 'WriteRowNames', true);

    %% 4) Ecs_aggregation
    % Sector Map
    ff_aggregatedVolume = aggMatrix' * Ecs_ff;         
    ff_aggregatedShare = aggMatrix' * Ecs_ff_share;  
    
    re_aggregatedVolume = aggMatrix' * Ecs_re;         
    re_aggregatedShare = aggMatrix' * Ecs_re_share;   
    
    % col name:ID+region
    AggVariableNames = ['ID', cellstr(regions)'];  
    aggSectorIDs = (1:length(aggSectors))';        
    
    % save ff Ecs volume
    aggregatedVolumeWithID_ff = [aggSectorIDs, ff_aggregatedVolume];  
    aggregatedShareWithID_ff = [aggSectorIDs, ff_aggregatedShare];   
    
    ff_aggVolumeTable = array2table(aggregatedVolumeWithID_ff, 'VariableNames', AggVariableNames, 'RowNames', aggSectors);
    aggVolumeFileName_ff = fullfile(ffOutputFolder, sprintf('ff_%d_sector_region_volume_agg.xlsx', year));
    writetable(ff_aggVolumeTable, aggVolumeFileName_ff, 'WriteRowNames', true);
    
    % save ff Ecs share
    ff_aggShareTable = array2table(aggregatedShareWithID_ff, 'VariableNames', AggVariableNames, 'RowNames', aggSectors);
    aggShareFileName_ff = fullfile(ffOutputFolder, sprintf('ff_%d_sector_region_share_agg.xlsx', year));
    writetable(ff_aggShareTable, aggShareFileName_ff, 'WriteRowNames', true);
    
    % save re Ecs volume
    aggregatedVolumeWithID_re = [aggSectorIDs, re_aggregatedVolume]; 
    aggregatedShareWithID_re = [aggSectorIDs, re_aggregatedShare];    
    
    re_aggVolumeTable = array2table(aggregatedVolumeWithID_re, 'VariableNames', AggVariableNames, 'RowNames', aggSectors);
    aggVolumeFileName_re = fullfile(reOutputFolder, sprintf('re_%d_sector_region_volume_agg.xlsx', year));
    writetable(re_aggVolumeTable, aggVolumeFileName_re, 'WriteRowNames', true);
    
    % save re Ecs share
    re_aggShareTable = array2table(aggregatedShareWithID_re, 'VariableNames', AggVariableNames, 'RowNames', aggSectors);
    aggShareFileName_re = fullfile(reOutputFolder, sprintf('re_%d_sector_region_share_agg.xlsx', year));
    writetable(re_aggShareTable, aggShareFileName_re, 'WriteRowNames', true);


    %% 5) country_aggre [c x 8]
    country_aggre_columns = {'ID','Production-based', 'Consumption-based', 'Domestic', ...
                             'Export', 'Import', 'Net Export', 'Net Import'};
    
    country_aggre_ff_table = array2table(country_aggre_ff, 'RowNames', regions, 'VariableNames', country_aggre_columns);
    country_aggre_re_table = array2table(country_aggre_re, 'RowNames', regions, 'VariableNames', country_aggre_columns);
    
    writetable(country_aggre_ff_table, fullfile(ffOutputFolder, 'country_aggre.xlsx'), 'WriteRowNames', true);
    writetable(country_aggre_re_table, fullfile(reOutputFolder, 'country_aggre.xlsx'), 'WriteRowNames', true);
    
    fprintf('Year %d - saved as .xlsx\n', year);
end



