%% MRIO calculation function
    % A database for identifying and tracking renewable energy embodied in global trade
    % Copyright (c) 2025 Yan Cheng

% Input：
    % n=region*sector
    % c:region
    % s:sector
    % E：energy account (n*1)
    % X：total output (n*1)
    % L：leontif (inv(I-A)) (n*n)
    % Ftotal：Final demand (n*c)
    % Unit: ktoe
% Output：
    % Unit: Mtoe 
    % ET：embodied flow cs to c (n*c)
    % Ec2c:embodied flow c to c (n*n)
    % Ecs: embodied flow exported from sector in c(s*n)
    % country_aggre: (c*8) Aggregation Result: [1 ID 2 production-based 3 consumption-based 4 Domestic 
        % 5 Export 6 Import 7 Net Export 8 Net import]

function [ET,Ec2c,Ecs,country_aggre]=MRIO_02(n,c,s,E,X,L,Ftotal)

%% 1 embodied energy calculation
% Unit change: ktoe to Mtoe
E = E/1000; 
E(isnan(E))=0;
E(isinf(E))=0;
E(X == 0) = 0;

sum(E(:))

f = E./X;
f(isnan(f))=0;
f(isinf(f))=0;

Ftotal(Ftotal<0)=0;

ET = diag(f) * L * Ftotal;

% check
ET(isnan(ET))=0;
ET(isinf(ET))=0;
ET(ET < 0) = 0;

sum(ET(:))


%% 2 result aggregation
% Ec2c
Ec2c = zeros(c,c);
j = 1;
range = n;

for i = 1:s:range; 
    temp1 = ET(i:i+s-1,:); 
    Ec2c(j,:) = sum(temp1,1); 
    j = j+1;
end

% Ecs(s*c)

ETnoDom = ET; 
% set domestic as zero
for i = 1:c
    sectorStart = s * (i - 1) + 1;
    sectorEnd = s * i;
    ETnoDom(sectorStart:sectorEnd, i) = 0; 
end

Ecs = reshape(sum(ETnoDom, 2), s,c); 

% pro_based
pro_E = sum(Ec2c,2);

% con_based
con_E = sum(Ec2c,1);

% domestic
domestic = diag(Ec2c); 

% export
export = pro_E-domestic;
[m,n] = find(export <0); 
export(export<0)=0;

% import
import = con_E'-domestic; 
[m,n] = find(import <0); 
import(import<0)=0;

% net
net_export = export - import;
net_import = import - export;

% aggregate
country_aggre = zeros(c,8); 
country_aggre(:,1) = 1:c;
country_aggre(:,2) = pro_E;
country_aggre(:,3) = con_E';
country_aggre(:,4) = domestic;
country_aggre(:,5) = export;
country_aggre(:,6) = import;
country_aggre(:,7) = net_export;
country_aggre(:,8) = net_import;

