#! /usr/bin/env python
# ==========================================================================
# Show models
#
# Copyright (C) 2024-2025 Juergen Knoedlseder
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.
#
# ==========================================================================
import os
import math
import matplotlib.pyplot as plt
import gammalib
import cscripts


# ================ #
# Plot table model #
# ================ #
def plot_table_model(ax, filename, flux=1.0e-3, modelnorm=2e-11, label=None,
                     ep=None, gamma=None, emax=None, ecut=None,
                     linestyle_positrons=(0,(3,1,1,1)), linewidth_positrons=1.5,
                     ye=None, resfile=None, fontsize=5.5, gc=True, alpha_gc=0.4):
    """
    Plot table model

    Parameters
    ----------
    ax : pyplot
        Plotting frame
    flux : float
        511 keV line flux normalisation
    gc : boolean, optional
        Show GC spectrum
    alpha_gc : float, optional
        Alpha value for GC model
    """
    # Initialise TS values of Bulge and core components
    ts_ia = None
    ts_gc = None

    # Set table component component filenames
    file_total = filename
    file_ia    = filename.replace('.fits', '_ia.fits')
    file_ib    = filename.replace('.fits', '_ib.fits')
    file_eb    = filename.replace('.fits', '_eb.fits')
    file_ic    = filename.replace('.fits', '_ic.fits')

    # Load table model FITS files
    if os.path.isfile(file_total):
        model_total = gammalib.GModelSpectralTable(file_total, 1.0)
    else:
        model_total = None
    if os.path.isfile(file_ia):
        model_ia = gammalib.GModelSpectralTable(file_ia, 1.0)
    else:
        model_ia = None
    if os.path.isfile(file_ib):
        model_ib = gammalib.GModelSpectralTable(file_ib, 1.0)
    else:
        model_ib = None
    if os.path.isfile(file_eb):
        model_eb = gammalib.GModelSpectralTable(file_eb, 1.0)
    else:
        model_eb = None
    if os.path.isfile(file_ic):
        model_ic = gammalib.GModelSpectralTable(file_ic, 1.0)
    else:
        model_ic = None

    # Plot SEDs
    plot_sed(ax, source='68_IA', color='red')
    if gc:
        plot_sed(ax, source='68_GC', color='blue', zorder=-3, alpha=alpha_gc)

    # Continue only if there is a total table model
    if model_total != None:

        # Extract energy axis
        ebounds      = model_total.ebounds()
        energies     = [ebounds.elogmean(i) for i in range(ebounds.size())]
        energies_MeV = [ebounds.elogmean(i).MeV() for i in range(ebounds.size())]

        # Optionally extract TS value, flux and model parameters
        if resfile != None:
            if os.path.isfile(resfile):
                models  = gammalib.GModels(resfile)
                ts_ia   = models['IA'].ts()
                ts_gc   = models['GC'].ts()
                flux_ia = models['IA']['Normalization'].value()
                spec_gc = [models['GC'].spectral().eval(e)*e.MeV()*e.MeV()*gammalib.MeV2erg for e in energies]
                if flux == None:
                    flux = flux_ia
                if ep == None:
                    if models['IA'].has_par('Energy'):
                        ep = models['IA']['Energy'].value()
                    if emax == None:
                        if models['IA'].has_par('MaxEnergy'):
                            emax = models['IA']['MaxEnergy'].value()
                if gamma == None:
                    if models['IA'].has_par('Index'):
                        gamma = -models['IA']['Index'].value()
            else:
                print('--- %s' % resfile)

        # Return if flux is None
        if flux == None:
            return

        # Set conversion factor
        conv = flux * gammalib.MeV2erg

        # Plot total
        if ye != None:
            model_total['Ionization'].value(ye)
        values = [conv * model_total.eval(e) * e.MeV()*e.MeV() for e in energies]
        ax.plot(energies_MeV, values, linestyle='solid', color='red', linewidth=2,
                label=r'Total $\gamma$-ray emission')

        # Plot components
        if model_ia != None:
            if ye != None:
                model_ia['Ionization'].value(ye)
            values = [conv * model_ia.eval(e) * e.MeV()*e.MeV() for e in energies]
            ax.plot(energies_MeV, values, linestyle='dashed', color='#ff7f0e', linewidth=1,
                    label='In-flight annihilation')
        if model_ib != None:
            if ye != None:
                model_ib['Ionization'].value(ye)
            values = [conv * model_ib.eval(e) * e.MeV()*e.MeV() for e in energies]
            ax.plot(energies_MeV, values, linestyle='dashdot', color='#279e68', linewidth=1,
                    label='Internal Bremsstrahlung')
        if model_eb != None:
            if ye != None:
                model_eb['Ionization'].value(ye)
            values = [conv * model_eb.eval(e) * e.MeV()*e.MeV() for e in energies]
            ax.plot(energies_MeV, values, linestyle='dotted', color='#d62728', linewidth=1,
                    label='Bremsstrahlung')
        if model_ic != None:
            if ye != None:
                model_ic['Ionization'].value(ye)
            values = [conv * model_ic.eval(e) * e.MeV()*e.MeV() for e in energies]
            ax.plot(energies_MeV, values, linestyle='dotted', color='#aa40fc', linewidth=1)

        # Plot input spectrum
        if ep != None:
            if type(ep) == type('hello'):
                csv     = gammalib.GCsv(ep)
                eng     = [float(csv[i,0]) for i in range(csv.nrows())]
                spec    = [float(csv[i,1])*float(csv[i,0])*float(csv[i,0]) for i in range(csv.nrows())]
                maxspec = max(spec)
                spec    = [s/maxspec * modelnorm for s in spec]
                ax.plot(eng, spec, linestyle=linestyle_positrons, color='black',
                        linewidth=linewidth_positrons, label='Injected positrons')
            else:
                ax.plot([ep,ep], [0.0,modelnorm], linestyle=linestyle_positrons, color='black',
                        linewidth=linewidth_positrons, label='Injected positrons')
        elif gamma != None and emax != None:
            engs      = gammalib.GEnergies(100, gammalib.GEnergy(0.511, 'MeV'),
                                                gammalib.GEnergy(emax, 'MeV'))
            engs_MeV  = [e.MeV() for e in engs]
            model     = [math.pow(e,-gamma)*e*e for e in engs_MeV]
            maxmod    = max(model)
            model     = [m/maxmod * modelnorm for m in model]
            model[0]  = 0.0
            model[99] = 0.0
            ax.plot(engs_MeV, model, linestyle=linestyle_positrons, color='black',
                    linewidth=linewidth_positrons, label='Injected positrons')
        elif gamma != None and ecut != None:
            engs     = gammalib.GEnergies(100, gammalib.GEnergy(0.511, 'MeV'),
                                               gammalib.GEnergy(3*ecut, 'MeV'))
            engs_MeV = [e.MeV() for e in engs]
            model    = [math.pow(e,-gamma)*math.exp(-e/ecut)*e*e for e in engs_MeV]
            maxmod   = max(model)
            model    = [m/maxmod * modelnorm for m in model]
            ax.plot(engs_MeV, model, linestyle=linestyle_positrons, color='black',
                    linewidth=linewidth_positrons, label='Injected positrons')

        # Optionally plot GC model
        if gc:
            ax.plot(energies_MeV, spec_gc, linestyle='-', color='blue', zorder=-3, alpha=alpha_gc)

        # Show fitted parameters
        xpos = 0.015
        sflux = r'$\Phi_{511}$ = %.1f 10$^{-3}$ ph cm$^{-2}$ s$^{-1}$' % (flux*1000.0)
        ax.text(xpos, 0.98, sflux, color='black', fontsize=6,
                 horizontalalignment='left', verticalalignment='top',
                 transform=ax.transAxes)
        dy = 0.08
        y  = 0.98 - dy
        if label != None:
            ax.text(xpos, y, label, color='black', fontsize=fontsize,
                     horizontalalignment='left', verticalalignment='top',
                     transform=ax.transAxes)
            y -= dy
        elif ep != None:
            sep = r'E$_p$ = %.1f MeV' % (ep)
            ax.text(xpos, y, sep, color='black', fontsize=fontsize,
                     horizontalalignment='left', verticalalignment='top',
                     transform=ax.transAxes)
            y -= dy
        if ecut != None:
            secut = r'E$_c$ = %.1f MeV' % (ecut)
            ax.text(xpos, y, secut, color='black', fontsize=fontsize,
                     horizontalalignment='left', verticalalignment='top',
                     transform=ax.transAxes)
            y -= dy
        if emax != None:
            semax = r'E$_{max}$ = %.1f MeV' % (emax)
            ax.text(xpos, y, semax, color='black', fontsize=fontsize,
                     horizontalalignment='left', verticalalignment='top',
                     transform=ax.transAxes)
            y -= dy
        if gamma != None:
            sgamma = r'$\Gamma$ = %.1f' % (gamma)
            ax.text(xpos, y, sgamma, color='black', fontsize=fontsize,
                     horizontalalignment='left', verticalalignment='top',
                     transform=ax.transAxes)
            y -= dy
        if ts_ia != None:
            ax.text(xpos, y, r'TS$_{\rm Bulge}$=%.1f' % (ts_ia), color='red', fontsize=fontsize,
            horizontalalignment='left', verticalalignment='top',
            transform=ax.transAxes)
            y -= dy
        if ts_gc != None:
            ax.text(xpos, y, r'TS$_{\rm Core}$=%.1f' % (ts_gc), color='blue', fontsize=fontsize,
            horizontalalignment='left', verticalalignment='top',
            transform=ax.transAxes)
            y -= dy

    # Set attributes
    ax.loglog()
    ax.set_xlim([0.1,100.0])
    ax.set_ylim([1e-12,1e-9])
    ax.tick_params(axis='both', which='major', labelsize=6)
    ax.tick_params(axis='both', which='minor', labelsize=5)

    # Return
    return


# ============ #
# Get filename #
# ============ #
def get_filename(band, brems, ic, prefix='spectrum_bins', ia='ia-g-plaw', dge='logps',
                 prefitted=False, source='ia', filetype='.fits'):
    """
    Get result filename

    Parameters
    ----------
    band : str
        Energy band ('low', 'high', 'full')
    brems : str
        Bremsstrahlungs component ('none','map','gmap')
    ic : str
        Inverse Compton component ('none','map','gmap',[2D name])
    prefix : str, optional
        Filename prefix
    ia : str, optional
        In-flight annihilation component string
    dge : str, optional
        DGE spectral fitting string
    prefitted : boolean, optional
        Use prefitted results?
    source : str, optional
        Resource string
    filetype : str, optional
        File type

    Returns
    -------
    filename : str
        Result XML filename
    """
    # Set filename
    if band == 'low':
        energies = '00750-03000keV_4bins_wo26Al'
    elif band == 'high':
        energies = '03000-30000keV_4bins'
    elif band == 'full':
        energies = '16bins_wo26Al'
    if brems == 'map' or ic == 'map':
        map = 'map'
    elif brems == 'gmap' or ic == 'gmap':
        map = 'gmap'
    else:
        map = ic
    suffix = map
    if brems != 'none':
        suffix += '-brems-map'
    if ic != 'none':
        suffix += '-ic-map'
    if prefitted:
        fitmode = '_prefitted'
    else:
        fitmode = ''
    if source != None:
        fitmode += '_%s' % (source)
    filename = '%s_com_240x140_drwnorm2_%s_conv-%s_%s-' \
               'g14-3c9-3c3-p18-p05-cr-ve-x1-ca-ls-%s-' \
               'isofix_%s_bgdlixf_ni11%s%s' % (prefix, energies, map, ia, suffix, dge, fitmode, filetype)

    # If file does not exist then try prepending results/
    if not os.path.isfile(filename):
        alt_filename = 'results/%s' % (filename)
        if os.path.isfile(alt_filename):
            filename = alt_filename

    # Return filename
    return filename


# ======== #
# Plot SED #
# ======== #
def plot_sed(ax, color='red', source='IA', marker='o', markersize=3, linewidth=None, label=None, zorder=None, alpha=None):
    """
    Plot SED

    Parameters
    ----------
    ax : pyplot
        Plotting frame
    color : str, optional
        Color of SED
    source : str, optional
        Source name
    markersize : float, optional
        Marker size
    linewidth : float, optional
        Line width
    label : str, optional
        Label
    """
    # Set SED string
    sed = '%s-%s-%s-%s' % ('gc-l-0.12+b0.66', 'bins3', 'ia-g-l1.4-b0.2r3.3', 'bins8')

    # Get SED filename
    filename = get_filename('full', 'none', 'icmaps358003-507006', ia=sed, source=gammalib.tolower(source))

    # Continue only if file exists
    if os.path.isfile(filename):

        # Read SED
        spec = get_spectrum_file(filename, source=source.lstrip('68_'))

        # Plot error bars
        ax.errorbar(spec['energies'], spec['flux'],
                    yerr=spec['e_flux'], xerr=[spec['ed_engs'], spec['eu_engs']],
                    marker=marker, color=color, linestyle='None',
                    markersize=markersize, linewidth=linewidth, label=label,
                    zorder=zorder, alpha=alpha)

        # Plot upper limits
        if len(spec['ul_energies']) > 0:
            ax.errorbar(spec['ul_energies'], spec['ul_flux'], yerr=spec['yerr'],
                        xerr=[spec['ul_ed_engs'], spec['ul_eu_engs']],
                        uplims=True, marker=marker, color=color, linestyle='None',
                        markersize=markersize, linewidth=linewidth,
                        zorder=zorder, alpha=alpha)

    # ... otherwise indicate that file is missing
    else:
        print('--- %s' % filename)

    # Return
    return


# ===================================== #
# Extract the spectrum info from a file #
# ===================================== #
def get_spectrum_file(filename, source='IA'):
    """
    Extract the spectrum info from a file for plotting

    Parameters
    ----------
    filename : str
        Name of spectrum FITS file
    source : str, optional
        Source name

    Returns
    -------
    spec : dict
        Python dictionary defining spectral plot parameters
    """
    # Initialise dictionary
    spec = None

    # Get filename
    fname = gammalib.GFilename(filename)

    # If filename is a FITS file then extract spectrum from FITS file
    if fname.is_fits():
        fits = gammalib.GFits(filename)
        if fits.table(1).string('OBJECT') == source:
            spec = get_spectrum_fits(fits)

    # ... otherwise extract spectrum from model
    else:
        models = gammalib.GModels(filename)
        if models.contains(source):
            model  = models[source].spectral()
            spec   = get_spectrum_model(model)

    # Return dictionary
    return spec


# ============================================= #
# Extract the spectrum info from a GFits object #
# ============================================= #
def get_spectrum_fits(fits):
    """
    Extract the spectrum info from a GFits object

    Parameters
    ----------
    fits : `~gammalib.GFits`
        Spectral GFits object
    
    Returns
    -------
    spec : dict
        Python dictionary defining spectral plot parameters
    """
    # Read spectrum objects
    table    = fits.table(1)
    c_energy = table['e_ref']
    c_ed     = table['e_min']
    c_eu     = table['e_max']
    c_flux   = table['ref_e2dnde']
    c_norm   = table['norm']
    c_eflux  = table['norm_err']
    c_upper  = table['norm_ul']
    c_ts     = table['ts']

    # Initialise arrays to be filled
    spec = {
        'energies'    : [],
        'flux'        : [],
        'ed_engs'     : [],
        'eu_engs'     : [],
        'e_flux'      : [],
        'ul_energies' : [],
        'ul_ed_engs'  : [],
        'ul_eu_engs'  : [],
        'ul_flux'     : [],
        'yerr'        : [],
    }

    # Determine if we can load the delta-log-likelihood profiles
    has_sedtype = table.has_card('SED_TYPE')
    load_dll    = False
    if has_sedtype:
        seds = table.card('SED_TYPE').string().split(',')

    # Loop over rows of the file
    nrows = table.nrows()
    for row in range(nrows):

        # Get Test Statistic, flux and flux error
        ts    = c_ts.real(row)
        norm  = c_norm.real(row)
        flx   = norm * c_flux.real(row)
        e_flx = flx * c_eflux.real(row)

        # If Test Statistic is larger than 1 and flux error is smaller than
        # flux then append flux plots ...
        if ts > 1.0 and e_flx < flx:
            spec['energies'].append(c_energy.real(row)*1.0e6)
            spec['flux'].append(flx)
            spec['ed_engs'].append(c_ed.real(row)*1.0e6)
            spec['eu_engs'].append(c_eu.real(row)*1.0e6)
            spec['e_flux'].append(e_flx)

        # ... otherwise append upper limit
        else:
            spec['ul_energies'].append(c_energy.real(row)*1.0e6)
            spec['ul_flux'].append(flx*c_upper.real(row))
            spec['ul_ed_engs'].append(c_ed.real(row)*1.0e6)
            spec['ul_eu_engs'].append(c_eu.real(row)*1.0e6)

    # Set upper limit errors
    spec['yerr'] = [0.6 * x for x in spec['ul_flux']]

    # Return dictionary
    return spec


# ============================================ #
# Extract the spectrum from GModelSpectralBins #
# ============================================ #
def get_spectrum_model(model):
    """
    Extract the spectrum info from a GFits object

    Parameters
    ----------
    model : `~gammalib.GModelSpectralBins`
        Spectral bins model
    
    Returns
    -------
    spec : dict
        Python dictionary defining spectral plot parameters
    """
    # Initialise arrays to be filled
    spec = {
        'energies'    : [],
        'flux'        : [],
        'ed_engs'     : [],
        'eu_engs'     : [],
        'e_flux'      : [],
        'ul_energies' : [],
        'ul_ed_engs'  : [],
        'ul_eu_engs'  : [],
        'ul_flux'     : [],
        'yerr'        : [],
    }

    # Loop over bins
    nbins = model.bins()
    for bin in range(nbins):
        emin   = model.emin(bin).MeV()
        emax   = model.emax(bin).MeV()
        flux   = model.intensity(bin)
        error  = model.error(bin)
        emean  = math.sqrt(emin*emax)
        ed_eng = emean - emin
        eu_eng = emax - emean
        norm   = emean * emean * gammalib.MeV2erg
        flux  *= norm
        error *= norm
        spec['energies'].append(emean)
        spec['flux'].append(flux)
        spec['ed_engs'].append(ed_eng)
        spec['eu_engs'].append(eu_eng)
        spec['e_flux'].append(error)

    # Return dictionary
    return spec


# =========== #
# Show models #
# =========== #
def show_models():
    """
    Show models
    """
    # Set size
    figwidth  = 183.0 / 25.4
    figheight = 163.0 / 25.4

    # Create figure
    fig = plt.figure(figsize=(figwidth,figheight))
    fig.subplots_adjust(left=0.075, bottom=0.053, right=0.985, top=0.99, wspace=0.05, hspace=0.07)

    # Create frames
    ax1 = fig.add_subplot(4,2,1)
    ax2 = fig.add_subplot(4,2,2)
    ax3 = fig.add_subplot(4,2,3)
    ax4 = fig.add_subplot(4,2,4)
    ax5 = fig.add_subplot(4,2,5)
    ax6 = fig.add_subplot(4,2,6)
    ax7 = fig.add_subplot(4,2,7)
    ax8 = fig.add_subplot(4,2,8)

    # Set filenames (for spatial models fixed to parameters of energy band analysis)
    filemono = get_filename('full', 'none', 'icmaps358003-507006',
                            ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-mono0.10',
                            dge='logps', source=None, prefix='results', filetype='.xml')
    filedm   = get_filename('full', 'none', 'icmaps358003-507006',
                            ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-dm0.10',
                            dge='logps', source=None, prefix='results', filetype='.xml')
    filecont = get_filename('full', 'none', 'icmaps358003-507006',
                            ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-cont10.10',
                            dge='logps', source=None, prefix='results', filetype='.xml')
    filec50  = get_filename('full', 'none', 'icmaps358003-507006',
                            ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-cont500.10',
                            dge='logps', source=None, prefix='results', filetype='.xml')
    file22Na = get_filename('full', 'none', 'icmaps358003-507006',
                            ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-22Na0.10',
                            dge='logps', source=None, prefix='results', filetype='.xml')
    file26Al = get_filename('full', 'none', 'icmaps358003-507006',
                            ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-26Al0.10',
                            dge='logps', source=None, prefix='results', filetype='.xml')
    file44Sc = get_filename('full', 'none', 'icmaps358003-507006',
                            ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-44Sc0.10',
                            dge='logps', source=None, prefix='results', filetype='.xml')
    file56Co = get_filename('full', 'none', 'icmaps358003-507006',
                            ia='gc-l-0.12+b0.66-plaw-ia-g-l1.4-b0.2r3.3-56Co0.10',
                            dge='logps', source=None, prefix='results', filetype='.xml')

    # Show model for mono-energetic injection
    plot_table_model(ax1, 'models/table_model_mono.fits', flux=None, resfile=filemono)

    # Show model for injection from dark matter annihilation
    plot_table_model(ax2, 'models/table_model_dm.fits', flux=None, ep=2.05179683510036,
                     resfile=filedm)

    # Show model for injection of continuum spectrum
    #plot_table_model(ax3, 'models/table_model_cont.fits', flux=None,
    #                 gamma=-2.71086231045421, emax=2.39996843966139,
    #                 resfile=filecont)
    plot_table_model(ax3, 'models/table_model_cont.fits', flux=None, resfile=filecont)
    
    # Show model for injection of 50 MeV continuum spectrum
    plot_table_model(ax4, 'models/table_model_cont50.fits', flux=None, resfile=filec50)

    # Show model for beta+ injection from 22Na
    plot_table_model(ax5, 'models/table_model_22Na.fits', flux=None,
                     ep='models/beta-22Na.csv', label=r'$^{22}$Na($\beta^+$)', ye=0.1,
                     resfile=file22Na)

    # Show model for beta+ injection from 26Al
    plot_table_model(ax6, 'models/table_model_26Al.fits', flux=None,
                     ep='models/beta-26Al.csv', label=r'$^{26}$Al($\beta^+$)', ye=0.1,
                     resfile=file26Al)

    # Show model for beta+ injection from 44Sc
    plot_table_model(ax7, 'models/table_model_44Sc.fits', flux=None,
                     ep='models/beta-44Sc.csv', label=r'$^{44}$Sc($\beta^+$)', ye=0.1,
                     resfile=file44Sc)

    # Show model for beta+ injection from 56Co
    plot_table_model(ax8, 'models/table_model_56Co.fits', flux=None,
                     ep='models/beta-56Co.csv', label=r'$^{56}$Co($\beta^+$)', ye=0.1,
                     resfile=file56Co)

    # Add figure labels
    ax1.text(0.99, 0.98, 'a', color='black', fontsize=8, weight='bold',
             horizontalalignment='right', verticalalignment='top',
             transform=ax1.transAxes)
    ax2.text(0.99, 0.98, 'b', color='black', fontsize=8, weight='bold',
             horizontalalignment='right', verticalalignment='top',
             transform=ax2.transAxes)
    ax3.text(0.99, 0.98, 'c', color='black', fontsize=8, weight='bold',
             horizontalalignment='right', verticalalignment='top',
             transform=ax3.transAxes)
    ax4.text(0.99, 0.98, 'd', color='black', fontsize=8, weight='bold',
             horizontalalignment='right', verticalalignment='top',
             transform=ax4.transAxes)
    ax5.text(0.99, 0.98, 'e', color='black', fontsize=8, weight='bold',
             horizontalalignment='right', verticalalignment='top',
             transform=ax5.transAxes)
    ax6.text(0.99, 0.98, 'f', color='black', fontsize=8, weight='bold',
             horizontalalignment='right', verticalalignment='top',
             transform=ax6.transAxes)
    ax7.text(0.99, 0.98, 'g', color='black', fontsize=8, weight='bold',
             horizontalalignment='right', verticalalignment='top',
             transform=ax7.transAxes)
    ax8.text(0.99, 0.98, 'h', color='black', fontsize=8, weight='bold',
             horizontalalignment='right', verticalalignment='top',
             transform=ax8.transAxes)

    # Set attributes
    ax1.set_ylabel(r'E$^2$ $\times$ dN/dE (erg cm$^{-2}$ s$^{-1}$)', fontsize=6)
    ax1.axes.xaxis.set_ticklabels([])
    #
    ax2.tick_params(labelleft=False, left=False)
    ax2.axes.xaxis.set_ticklabels([])
    ax2.legend(fontsize=6, loc='lower right', labelspacing=0.1)
    #
    ax3.set_xlabel(r'Energy (MeV)', fontsize=6)
    ax3.set_ylabel(r'E$^2$ $\times$ dN/dE (erg cm$^{-2}$ s$^{-1}$)', fontsize=6)
    ax3.axes.xaxis.set_ticklabels([])
    #
    ax4.set_xlabel(r'Energy (MeV)', fontsize=6)
    ax4.tick_params(labelleft=False, left=False)
    ax4.axes.xaxis.set_ticklabels([])
    #
    ax5.set_xlabel(r'Energy (MeV)', fontsize=6)
    ax5.set_ylabel(r'E$^2$ $\times$ dN/dE (erg cm$^{-2}$ s$^{-1}$)', fontsize=6)
    ax5.axes.xaxis.set_ticklabels([])
    #
    ax6.set_xlabel(r'Energy (MeV)', fontsize=6)
    ax6.tick_params(labelleft=False, left=False)
    ax6.axes.xaxis.set_ticklabels([])
    #
    ax7.set_xlabel(r'Energy (MeV)', fontsize=6)
    ax7.set_ylabel(r'E$^2$ $\times$ dN/dE (erg cm$^{-2}$ s$^{-1}$)', fontsize=6)
    ax7.axes.xaxis.set_ticklabels(['','0.1','1','10','100'])
    #
    ax8.set_xlabel(r'Energy (MeV)', fontsize=6)
    ax8.tick_params(labelleft=False, left=False)
    ax8.axes.xaxis.set_ticklabels(['','','1','10','100'])

    # Show plot
    plt.show()

    # Save figure
    fig.savefig('fig4.pdf', dpi=300)

    # Return
    return



# ======================== #
# Main routine entry point #
# ======================== #
if __name__ == '__main__':

    # Set usage string
    usage = 'show_models.py'

    # Set default options
    options = []

    # Get arguments and options from command line arguments
    args, options = cscripts.ioutils.get_args_options(options, usage)

    # Show table models
    show_models()
